
/**********************************************************************
 * $Id: groupDisplay.c,v 1.3 92/11/30 11:28:47 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Viewport.h>

#include <xerion/MenuPane.h>
#include <xerion/Selection.h>

#include <xerion/display.h>
#include <xerion/Controller.h>
#include <xerion/displayUtils.h>
#include <xerion/display2Itf.h>
#include <xerion/display2Sim.h>

#ifdef getValues
#undef getValues
#endif
#ifdef setValues
#undef setValues
#endif

#include <xerion/useful.h>
#include <xerion/simulator.h>
#include <xerion/commands.h>
#include <xerion/groupCom.h>
#include <xerion/costCom.h>

#include "groupDisplay.h"
#include "check.h"

#define GD_NET		((int)0)
#define GD_GROUP	((int)1)

/********************************************************************/
static void	    registerGroupDisplay	ARGS((GroupDisplay)) ;
static void	    deregisterGroupDisplay	ARGS((GroupDisplay)) ;
static GroupDisplay nameToGroupDisplay		ARGS((String)) ;
/********************************************************************/
static void	createGroupDisplayP ARGS((Widget)) ;
static void	showHighlighted	    ARGS((GroupMaskSet)) ;
/********************************************************************/
static void	gmsDestroy	ARGS((GroupMaskSet)) ;
static Mask	gmsHighlight	ARGS((GroupMaskSet, Mask)) ;
static Mask	gmsUnhighlight	ARGS((GroupMaskSet, Mask)) ;
static Mask	gmsSensitize	ARGS((GroupMaskSet, Mask)) ;
static Mask	gmsInsensitize	ARGS((GroupMaskSet, Mask)) ;
static void	gmsSetCallback	ARGS((GroupMaskSet, GMSCallbackType, 
				      GMSCallback, void *)) ;
static void	gdSetCallback	ARGS((GroupDisplay, GDCallbackType, 
				      GDCallback, void *)) ;
/********************************************************************/


/*********************************************************************
 *	Name:		createGroupMaskSet
 *	Description:	creates a group mask set for inclusion in a
 *			groupDisplay object.
 *	Parameters:
 *	  String	name - the name for the mask set	
 *	  GroupDisplay	parent - the groupDisplay it's going into
 *	  Mask		mask - the mask to display. NOTE: all masks
 *			must already be registered as GROUP_CLASS masks.
 *	Return Value:
 *	  GroupMaskSet	createGroupMaskSet - the new mask set
 *********************************************************************/
GroupMaskSet	createGroupMaskSet(name, parent, mask, type)
  String	name ;
  GroupDisplay	parent ;
  Mask		mask ;
  GMSSetType	type ;
{
  GroupMaskSet	this ;
  
  this = XtNew(GroupMaskSetRec) ;
  if (this == NULL)
    return NULL ;

  this->name	= XtNewString(name) ;
  this->parent	= parent ;
  this->mask	= mask ;
  this->type	= type ;

  this->sensitiveMask	= this->mask ;
  this->highlightedMask	= 0 ;

  this->callback[GMSSelect]	  = NULL ;
  this->callback[GMSUnselect]	  = NULL ;
  this->callbackData[GMSSelect]	  = NULL ;
  this->callbackData[GMSUnselect] = NULL ;

  this->toggle		= NULL ;
  this->toggleMask	= NULL ;

  this->destroy		= gmsDestroy ;
  this->highlight 	= gmsHighlight ;
  this->unhighlight	= gmsUnhighlight ;
  this->sensitize	= gmsSensitize ;
  this->insensitize	= gmsInsensitize ;
  this->setCallback	= gmsSetCallback ;

  MGDaddMaskSet(parent, this) ;

  return this ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gmsDestroy
 *	Description:	the destroy method for a group mask set	
 *	Parameters:
 *	  GroupMaskSet	this - the mask set to destroy
 *	Return Value:
 *	  static void	gmsDestroy - NONE
 *********************************************************************/
static void	gmsDestroy(this)
  GroupMaskSet	this ;
{
  if (this == NULL)
    return ;

  if (this->name)
    XtFree((void *)this->name) ;

  if (this->toggle)
    XtFree((void *)this->toggle) ;

  if (this->toggleMask)
    XtFree((void *)this->toggleMask) ;

  free((void *)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gmsHighlight
 *	Description:	adds a mask to the set of highlighted ones
 *	Parameters:
 *	  GroupMaskSet	this - the mask set
 *	  Mask		newMasks - the mask to ADD to the highlighted set
 *	Return Value:
 *	 static Mask	gmsHighlight - the complete set of
 *			highlighted masks.
 *********************************************************************/
static Mask	gmsHighlight(this, newMasks)
  GroupMaskSet	this ;
  Mask		newMasks ;
{
  int		idx ;

  for(idx = 0 ; MGDtoggle(this, idx) ; ++idx)
    this->highlightedMask |= MGDtoggleMask(this, idx) & newMasks ;

  showHighlighted(this) ;

  return MGDhighlighted(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gmsUnhighlight
 *	Description:	adds a mask to the set of unhighlighted ones
 *	Parameters:
 *	  GroupMaskSet	this - the mask set
 *	  Mask		newMask - the mask to ADD to the unhighlighted set
 *	Return Value:
 *	 static Mask	gmsUnhighlight - the complete set of
 *			unhighlighted masks.
 *********************************************************************/
static Mask	gmsUnhighlight(this, newMasks)
  GroupMaskSet	this ;
  Mask		newMasks ;
{
  int		idx ;

  for(idx = 0 ; MGDtoggle(this, idx) ; ++idx)
      this->highlightedMask &= ~(MGDtoggleMask(this, idx) & newMasks);

  showHighlighted(this) ;
    
  return MGDunhighlighted(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gmsSensitize
 *	Description:	adds a mask to the set of sensitive ones
 *			(the ones that can be selected)
 *	Parameters:
 *	  GroupMaskSet	this - the mask set
 *	  Mask		mask - the mask to ADD to the sensitive set
 *	Return Value:
 *	 static Mask	gmsSensitize - the complete set of
 *			sensitive masks.
 *********************************************************************/
static Mask	gmsSensitize(this, mask)
  GroupMaskSet	this ;
  Mask		mask ;
{
  int		idx ;

  for(idx = 0 ; MGDtoggle(this, idx) ; ++idx) {
    if (MGDtoggleMask(this, idx) & mask) {
      XtVaSetValues(MGDtoggle(this, idx), XtNsensitive, TRUE, NULL) ;
      this->sensitiveMask |=  MGDtoggleMask(this, idx) ;
    }
  }
  if (MGDsensitive(this))
    XtVaSetValues(MGDmenuButton(this), XtNsensitive, TRUE, NULL) ;

  return MGDsensitive(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gmsInsensitize
 *	Description:	adds a mask to the set of insensitive ones
 *			(the ones that can not be selected)
 *	Parameters:
 *	  GroupMaskSet	this - the mask set
 *	  Mask		mask - the mask to ADD to the insensitive set
 *	Return Value:
 *	 static Mask	gmsInsensitize - the complete set of
 *			insensitive masks.
 *********************************************************************/
static Mask	gmsInsensitize(this, mask)
  GroupMaskSet	this ;
  Mask		mask ;
{
  int		idx ;

  for(idx = 0 ; MGDtoggle(this, idx) ; ++idx) {
    if (MGDtoggleMask(this, idx) & mask) {
      XtVaSetValues(MGDtoggle(this, idx), XtNsensitive, FALSE, NULL) ;
      this->sensitiveMask &= ~MGDtoggleMask(this, idx) ;
    }
  }
  if (MGDinsensitive(this) == MGDmask(this))
    XtVaSetValues(MGDmenuButton(this), XtNsensitive, FALSE, NULL) ;

  return MGDinsensitive(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gmsSetCallback
 *	Description:	sets the callback to be called whenever a mask
 *			is changed
 *	Parameters:
 *	  GroupMaskSet	  this - the mask set
 *	  GMSCallbackType type - the type of callback
 *	  GMSCallback	  callback - the callback
 *	  void		  *data - user data to pass to the callback
 *	Return Value:
 *	  static void	gmsSetSelectCallback - NONE
 *********************************************************************/
static void	gmsSetCallback(this, type, callback, data)
  GroupMaskSet		this ;
  GMSCallbackType	type ;
  GMSCallback		callback ;
  void			*data ;
{
  this->callback[type]	   = callback ;
  this->callbackData[type] = data ;
}
/********************************************************************/


/********************************************************************/
static void	gdDestroy     ARGS((GroupDisplay)) ;
static void	gdAddMaskSet  ARGS((GroupDisplay, GroupMaskSet)) ;
/********************************************************************/


/*********************************************************************
 *	Name:		createGroupDisplay
 *	Description:	creates a group display object
 *	Parameters:
 *	  String	name - the name for the display
 *	Return Value:
 *	  GroupDisplay	createGroupDisplay - the new display
 *********************************************************************/
GroupDisplay	createGroupDisplay(name)
  String	name ;
{
  GroupDisplay	this ;
  
  this = (GroupDisplay)malloc(sizeof(GroupDisplayRec)) ;
  if (this == NULL)
    return NULL ;

  this->name	= strdup(name) ;
  this->group	= NULL ;

  this->maskSet = NULL ;
  this->callback[GDSelect]     = NULL ;
  this->callbackData[GDSelect] = NULL ;


  this->destroy	    = gdDestroy ;
  this->addMaskSet  = gdAddMaskSet ;
  this->setCallback = gdSetCallback ;

  addUserDisplay(MGDname(this), createGroupDisplayP) ;

  /* add to global list of displays */
  registerGroupDisplay(this) ;

  return this ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gdDestroy
 *	Description:	the destroy method for a group display object
 *	Parameters:
 *	  GroupDisplay	this - the display to destroy
 *	Return Value:
 *	  static void	gdDestroy - NONE
 *********************************************************************/
static void	gdDestroy(this)
  GroupDisplay	this ;
{
  int		idx ;
  String	*list ;

  if (this == NULL)
    return ;

  /* remove from global list of displays */
  deregisterGroupDisplay(this) ;

  if (MGDshell(this))
    XtDestroyWidget(MGDshell(this)) ;

  if (MGDtitleFormat(this))
    XtFree((void *)MGDtitleFormat(this)) ;

  list = MGDgroupList(this) ;
  for (idx = 0 ; list && list[idx] ; ++idx)
    XtFree((void *)list[idx]) ;

  if (list)
    XtFree((void *)list) ;

  for (idx = 0 ; MGDmaskSet(this, idx) ; ++idx)
    MGDdestroy(MGDmaskSet(this, idx)) ;
  
  if (this->maskSet)
    free((void *)this->maskSet) ;
  
  free((void *)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gdAddMaskSet
 *	Description:	adds a mask set to the group display
 *	Parameters:
 *	  GroupDisplay	this    - the group display
 *	  GroupMaskSet	maskSet - the mask set to add
 *	Return Value:
 *	 static void	gdAddMaskSet - NONE
 *********************************************************************/
static void	gdAddMaskSet(this, maskSet)
  GroupDisplay	this ;
  GroupMaskSet	maskSet ;
{
  int	idx ;
  
  for (idx = 0 ; this->maskSet && this->maskSet[idx] ; ++idx)
    ;
  
  this->maskSet = (GroupMaskSet *)XtRealloc((void *)this->maskSet,
					    (idx + 2)*sizeof(GroupMaskSet)) ;
  this->maskSet[idx] 	 = maskSet ;
  this->maskSet[idx + 1] = NULL ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		gdSetCallback
 *	Description:	sets the callback to be called whenever a mask
 *			is changed
 *	Parameters:
 *	  GroupMaskSet	  this - the mask set
 *	  GDCallbackType type - the type of callback
 *	  GDCallback	  callback - the callback
 *	  void		  *data - user data to pass to the callback
 *	Return Value:
 *	  static void	gdSetSelectCallback - NONE
 *********************************************************************/
static void	gdSetCallback(this, type, callback, data)
  GroupDisplay		this ;
  GDCallbackType	type ;
  GDCallback		callback ;
  void			*data ;
{
  this->callback[type]	   = callback ;
  this->callbackData[type] = data ;
}
/********************************************************************/


/********************************************************************/
static void  controllerCB   	ARGS((Widget, XtPointer, XtPointer)) ;
static void  setGroupListCB   	ARGS((Widget, XtPointer, XtPointer)) ;
static void  setGroupCB	   	ARGS((Widget, XtPointer, XtPointer)) ;
static void  showHighlightedCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void  toggleCB	   	ARGS((Widget, XtPointer, XtPointer)) ;
static void  showCostModelCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void  setNetCostModelCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void  setGroupCostModelCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void  toggleNetCostModelCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void  toggleGroupCostModelCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void  setNetNumGaussiansCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void  setGroupNumGaussiansCB	ARGS((Widget, XtPointer, XtPointer)) ;
/********************************************************************/
static Widget	createCostModelDisplay ARGS((CostModelDisplay, 
					     String, Widget)) ;
static String	*listGroups ARGS((String *)) ;
/********************************************************************/
static Widget	createMaskSetP	ARGS((GroupMaskSet, Widget)) ;
static void	setGroupList	ARGS((GroupDisplay)) ;
static void	setGroup	ARGS((GroupDisplay)) ;
static void	setHighlighted	ARGS((GroupDisplay)) ;
static void	doLabel		ARGS((GroupMaskSet)) ;
static void	setCostModel	ARGS((GroupDisplay, int)) ;
static void	applyCostModel	ARGS((GroupDisplay, int)) ;
static void	showCostModel	ARGS((CostModelDisplay)) ;
static void	setNumGaussians ARGS((GroupDisplay, int, int)) ;
static void	toggleCostModel ARGS((GroupDisplay, Widget, int)) ;
static void	gmsCallCallback	ARGS((GroupMaskSet, GMSCallbackType, Mask)) ;
static void	gdCallCallback	ARGS((GroupDisplay, GDCallbackType)) ;
/********************************************************************/


/*********************************************************************
 *	Name:		createGroupDisplayP
 *	Description:	
 *	Parameters:
 *	  Widget	shell - 
 *	Return Value:
 *	  export void	createGroupDisplayP - 
 *********************************************************************/
static void	createGroupDisplayP(shell)
  Widget	shell ;
{
  GroupDisplay		this ;
  CostModelDisplay	costModel ;

  Widget		menuPane, menu, form, dialog ;
  Widget		widget, lastWidget, menuButton ;
  String		name, ptr ;
  int			idx ;

  if (topLevel == NULL)
    return ;

  name = XtName(shell) ;
  ptr = strstr(name, "Display") ;
  if (ptr == NULL)
    return ;
  else
    *ptr = '\0' ;

  this = nameToGroupDisplay(name) ;
  if (this == NULL)
    return ;

  MGDsetShell(this, shell) ;

  MGDsetTitleFormat(this, getTitle(shell)) ;
  getNetValue(NAME, &name) ;
  widgetPrintf(shell, MGDtitleFormat(this), name) ;
  
  /**********************
   * Form to hold everything
   */
  form = XtVaCreateManagedWidget("groupDisplayForm", formWidgetClass, shell,
				 NULL) ;

  /**********************
   * Pulldown menu pane 
   */
  menuPane = XtVaCreateManagedWidget("menuPane", menuPaneWidgetClass, form,
				     NULL) ;

  /**********************
   * The list of groups
   */
  MGDsetGroupList(this, listGroups(NULL)) ;
  widget =  XtVaCreateManagedWidget("list", selectionWidgetClass, form,
				    XtNlist,	MGDgroupList(this),
				    NULL) ;
  XtAddCallback(widget, XtNcallback, setGroupCB, (XtPointer)this) ;
  MGDsetGroupSelection(this, widget) ;

  /**********************
   * form to hold the mask sets
   */
  form = XtVaCreateManagedWidget("form", formWidgetClass, form,
				 NULL) ;

  /**********************
   * Network cost model menu
   */
  lastWidget = XtVaCreateManagedWidget("costLabel", labelWidgetClass, form, 
				       NULL) ;

  costModel  = MGDnetCostModel(this) ;
  widget     = createCostModelDisplay(costModel, "netCostModel", form) ;
  MGDCMsetHighlighted(costModel, GDUnknown) ;
  XtVaSetValues(widget, XtNfromVert, lastWidget, NULL) ;
  lastWidget = widget ;

  XtAddCallback(MGDCMoption(costModel, GDNone), 
		XtNcallback, toggleNetCostModelCB, (XtPointer)this) ;
  XtAddCallback(MGDCMoption(costModel, GDSumSquare),
		XtNcallback, toggleNetCostModelCB, (XtPointer)this) ;
  XtAddCallback(MGDCMoption(costModel, GDMixture),
		XtNcallback, toggleNetCostModelCB, (XtPointer)this) ;
  widget = XtNameToWidget(MGDCMquery(costModel), "dialog.ok") ;
  XtAddCallback(widget,
		XtNcallback, setNetNumGaussiansCB, (XtPointer)this) ;
  /**********************
   * Group cost model menu
   */
  costModel  = MGDgroupCostModel(this) ;
  widget     = createCostModelDisplay(costModel, "groupCostModel", form) ;
  MGDCMsetHighlighted(costModel, GDUnknown) ;
  XtVaSetValues(widget, XtNfromVert, lastWidget, NULL) ;
  lastWidget = widget ;

  XtAddCallback(MGDCMoption(costModel, GDNone), 
		XtNcallback, toggleGroupCostModelCB, (XtPointer)this) ;
  XtAddCallback(MGDCMoption(costModel, GDSumSquare),
		XtNcallback, toggleGroupCostModelCB, (XtPointer)this) ;
  XtAddCallback(MGDCMoption(costModel, GDMixture),
		XtNcallback, toggleGroupCostModelCB, (XtPointer)this) ;
  widget = XtNameToWidget(MGDCMquery(costModel), "dialog.ok") ;
  XtAddCallback(widget,
		XtNcallback, setGroupNumGaussiansCB, (XtPointer)this) ;
  /**********************
   * Type masks
   */
  lastWidget = XtVaCreateManagedWidget("typeLabel", labelWidgetClass, form,
				       XtNfromVert,	lastWidget,
				       NULL) ;
  for (idx = 0 ; MGDmaskSet(this, idx) ; ++idx) {
    widget = createMaskSetP(MGDmaskSet(this, idx), form) ;

    XtVaSetValues(widget, XtNfromVert, lastWidget, NULL) ;
    lastWidget = widget ;
  }

  /**********************
   * Options pulldown menu 
   */
  menuButton = XtVaCreateManagedWidget("options",
				       menuButtonWidgetClass, menuPane,
				       XtNmenuName,	"optionsMenu",
				       NULL) ;
  
  menu = XtVaCreatePopupShell("optionsMenu", simpleMenuWidgetClass, menuButton,
			      NULL) ;
  
  widget = XtVaCreateManagedWidget("reset", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, setGroupListCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("line", smeLineObjectClass, menu, NULL) ;

  widget = XtVaCreateManagedWidget("close", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, popdownWidgetCB, (XtPointer)shell);

  /**********************
   * Final stuff for the shell
   */
  XtAddCallback(controller, XtNcallback, controllerCB, (XtPointer)this) ;
  XtVaSetValues(shell, XtNcreatePopupChildProc, NULL, NULL) ;

  XtAddCallback(shell, XtNpopupCallback, setGroupListCB, (XtPointer)this) ;
  setGroupList(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		createCostModelDisplay
 *	Description:	creates a cost model display WITH NO CALLBACKS
 *			ON SELECTION OR THE QUERY.
 *	Parameters:
 *	  CostModelDisplay	this - pointer to the costModel 
 *	  Widget		parent - parent widget
 *	Return Value:
 *	  static Widget	createCostModelDisplay - the form surrounding 
 *			everything
 *********************************************************************/
static Widget	createCostModelDisplay(this, name, parent)
  CostModelDisplay	this ;
  String		name ;
  Widget		parent ;
{
  Widget	widget, lastWidget ;
  Widget	form, menu, menuButton ;

  form       = XtVaCreateManagedWidget("maskSetForm", formWidgetClass, parent, 
				       NULL) ;
  menuButton = XtVaCreateManagedWidget(name,
				       menuButtonWidgetClass, form,
				       XtNmenuName,	"costModelMenu",
				       NULL) ;
  widget     = XtVaCreateManagedWidget("costModelLabel",
				       labelWidgetClass, form, 
				       XtNfromHoriz,	menuButton, 
				       NULL) ;
  MGDCMsetLabel(this, widget) ;

  menu = XtVaCreatePopupShell("costModelMenu", 
			      simpleMenuWidgetClass, menuButton, NULL) ;
  XtAddCallback(menu, XtNpopupCallback, showCostModelCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("none", smeBSBObjectClass, menu, 
				   XtNleftMargin,	check_width, 
				   NULL);
  MGDCMsetOption(this, GDNone,  widget) ;

  widget = XtVaCreateManagedWidget("sumSquare", smeBSBObjectClass, menu, 
				   XtNleftMargin,	check_width, 
				   NULL);
  MGDCMsetOption(this, GDSumSquare,  widget) ;

  widget = XtVaCreateManagedWidget("mixture", smeBSBObjectClass, menu, 
				   XtNleftMargin,	check_width, 
				   NULL);
  MGDCMsetOption(this, GDMixture,  widget) ;

  /* find the shell */
  while (!XtIsSubclass(parent, shellWidgetClass))
    parent = XtParent(parent) ;
  
  widget = XtVaCreatePopupShell("query", transientShellWidgetClass, parent,
				NULL) ;
  MGDCMsetQuery(this, widget) ;
  widget = XtVaCreateManagedWidget("dialog", dialogWidgetClass, widget,
				   XtNvalue,	"",
				   NULL) ;
  XawDialogAddButton(widget, "ok", popdownWidgetCB, 
		     (XtPointer)MGDCMquery(this)) ;
  XtInstallAllAccelerators(widget, widget) ;

  return form ;
}
/********************************************************************/

/*********************************************************************
 *	Name:		createMaskSetP
 *	Description:	
 *	Parameters:
 *	  GroupMaskSet	this - 
 *	  Widget	parent -
 *	Return Value:
 *	  static void	createMaskSetP - 
 *********************************************************************/
static Widget	createMaskSetP(this, parent)
  GroupMaskSet	this ;
  Widget	parent ;
{
  Mask		mask ;
  int		idx, toggleIdx, numToggles ;
  Widget	widget, form, menuButton, menu ;
  char		labelName[64], menuName[64], buttonName[64] ;
  
  /**********************
   * Form that holds everything
   */
  form = XtVaCreateManagedWidget("maskSetForm", formWidgetClass, parent,
				 NULL) ;

  /**********************
   * generate all the names
   */
  sprintf(buttonName, "%s:",	 MGDname(this)) ;
  sprintf(menuName,   "%sMenu",	 MGDname(this)) ;
  sprintf(labelName,  "%sLabel", MGDname(this)) ;

  /**********************
   * menu button, menu and label
   */
  widget = XtVaCreateManagedWidget(buttonName, menuButtonWidgetClass, form,
				   XtNmenuName,	strdup(menuName),
				   NULL) ;
  MGDsetMenuButton(this, widget) ;

  menu = XtVaCreatePopupShell(menuName, simpleMenuWidgetClass, 
			      MGDmenuButton(this), 
			      NULL) ;
  XtAddCallback(menu, XtNpopupCallback, showHighlightedCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget(labelName, labelWidgetClass, form, 
				   XtNfromHoriz,	MGDmenuButton(this),
				   NULL) ;
  MGDsetLabel(this, widget) ;

  /**********************
   * count the masks and allocate memory
   */
  mask = 1 ;
  toggleIdx = 0 ;
  for (idx = 0; idx < 8*sizeof(Mask) ; idx++) {
    if (this->mask & mask)
      ++toggleIdx ;
    mask <<= 1;
  }
  numToggles = toggleIdx ;

  MGDsetToggleArray (this, (Widget  *)XtCalloc(numToggles+1, sizeof(Widget)));
  MGDsetToggleMaskArray(this, (Mask *)XtCalloc(numToggles,   sizeof(Mask)));


  /**********************
   * set the toggle masks
   */
  mask = 1 ;
  toggleIdx = 0 ;
  for (idx = 0; idx < 8*sizeof(Mask) ; idx++) {
    if (MGDmask(this) & mask) {
      MGDsetToggleMask(this, toggleIdx, mask) ;
      ++toggleIdx ;
    }
    mask <<= 1;
  }

  /**********************
   * create the toggles
   */
  for (idx = 0; idx < numToggles ; ++idx) {
    String	name   = classMaskName(GROUP_CLASS, MGDtoggleMask(this, idx)) ;

    widget = XtVaCreateManagedWidget(name, smeBSBObjectClass, menu,
				     XtNleftMargin, check_width, 
				     NULL) ;
    XtAddCallback(widget, XtNcallback, toggleCB, (XtPointer)this) ;
    MGDsetToggle (this, idx, widget) ; 
  }
  MGDsetToggle(this, idx, NULL) ;

  return form ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		toggleCB
 *	Description:	callback called when a toggle is (un)selected.
 *			It sets the selected mask properly (which may
 *			cause callbacks to be called).
 *	Parameters:
 *	  Widget	widget - the toggle widget
 *	  XtPointer	clientData - the GroupMaskSet
 *	  XtPointer	callData - UNUSED
 *	Return Value:
 *	  static void	toggleCB - NONE
 *********************************************************************/
static void	toggleCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  GroupMaskSet	this = (GroupMaskSet)clientData ;
  Boolean	selected ;
  Mask		mask ;
  int		idx ;

  /* if read only, don't do anything */
  if (MGDtype(this) == GMSReadOnly)
    return ;

  /* find the mask */
  for(idx = 0 ; MGDtoggle(this, idx) && MGDtoggle(this, idx) != widget ; ++idx)
    ;

  if (MGDtoggle(this, idx) == NULL)
    return ;

  mask = MGDtoggleMask(this, idx) ;

  /* set the selected mask properly */
  selected = !(MGDhighlighted(this) & mask) ;

  if (selected) {
    /* if one of many, unselect everything else */
    if (MGDtype(this) == GMSOneOfMany) {
      MGDunhighlight(this, ~mask) ;
      gmsCallCallback(this, GMSUnselect, mask) ;
    }
    MGDhighlight(this, mask) ;
    gmsCallCallback(this, GMSSelect, mask) ;
  } else {
    MGDunhighlight(this, mask) ;
    gmsCallCallback(this, GMSUnselect, mask) ;
  }
}
/********************************************************************/
static void	toggleNetCostModelCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  toggleCostModel((GroupDisplay)clientData, widget, GD_NET) ;
}
/********************************************************************/
static void	toggleGroupCostModelCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  toggleCostModel((GroupDisplay)clientData, widget, GD_GROUP) ;
}
/********************************************************************/
static void	toggleCostModel(this, widget, which)
  GroupDisplay	this ;
  Widget	widget ;
  int		which ;
{
  CostModelDisplay	costModel ;

  if (which == GD_GROUP)
    costModel = MGDgroupCostModel(this) ;
  else if (which == GD_NET)
    costModel = MGDnetCostModel(this) ;
  else
    return ;

  if (widget == MGDCMoption(costModel, GDNone))
    MGDCMsetHighlighted(costModel, GDNone) ;
  else if (widget == MGDCMoption(costModel, GDSumSquare))
    MGDCMsetHighlighted(costModel, GDSumSquare) ;
  else if (widget == MGDCMoption(costModel, GDMixture))
    MGDCMsetHighlighted(costModel, GDMixture) ;
  else
    MGDCMsetHighlighted(costModel, GDNone) ;

  if (MGDCMhighlighted(costModel) == GDMixture) {
    popupCentered(MGDCMquery(costModel), XtGrabExclusive) ;
  } else {
    applyCostModel(this, which) ;
    showCostModel(costModel) ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		gmsCallCallback
 *	Description:	calls the desired callback passing in the
 *			mask and the data
 *	Parameters:
 *	  GroupMaskSet		this - the mask set
 *	  GMSCallbackType	callbackType - which type of callback
 *	  Mask			mask - the masks to call it for
 *	Return Value:
 *	  static void	gmsCallCallback - 
 *********************************************************************/
static void	gmsCallCallback(this, callbackType, mask)
  GroupMaskSet		this ;
  GMSCallbackType	callbackType ;
  Mask			mask ;
{
  GMSCallback	callback      = this->callback[callbackType] ;
  void		*callbackData = this->callbackData[callbackType] ;
  int		idx ;

  if (callback == NULL)
    return ;

  for(idx = 0 ; MGDtoggle(this, idx) ; ++idx) {
    Mask	bit = MGDtoggleMask(this, idx) ;
    if (bit & mask)
	callback(this, bit, callbackData) ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		gdCallCallback
 *	Description:	calls the desired callback passing in the
 *			mask and the data
 *	Parameters:
 *	  GroupDisplay		this - the group display
 *	  GDCallbackType	callbackType - which type of callback
 *	Return Value:
 *	  static void	gdCallCallback - NONE
 *********************************************************************/
static void	gdCallCallback(this, callbackType)
  GroupDisplay		this ;
  GDCallbackType	callbackType ;
{
  if (this->callback[callbackType])
    this->callback[callbackType](this, this->group, 
				 this->callbackData[callbackType]) ;
}
/********************************************************************/


/********************************************************************/
static void	setNetCostModelCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  setCostModel ((GroupDisplay)clientData, GD_NET) ;
}
/********************************************************************/
static void	setGroupCostModelCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  setCostModel ((GroupDisplay)clientData, GD_GROUP) ;
}
/********************************************************************/
static void	setCostModel(this, which)
  GroupDisplay	this ;
  int		which ;
{
  CostModelDisplay	costModel ;
  Group			group = MGDgroup(this) ;
  Net			net ;

  if (which == GD_GROUP) {
    costModel = MGDgroupCostModel(this) ;
    if (group == NULL)
      MGDCMsetHighlighted(costModel, GDUnknown) ;
    else if (McostModel(group) == NULL)
      MGDCMsetHighlighted(costModel, GDNone) ;
    else if (McostModel(group)->type == CM_SUMSQUARE)
      MGDCMsetHighlighted(costModel, GDSumSquare) ;
    else if (McostModel(group)->type == CM_MIXTURE)
      MGDCMsetHighlighted(costModel, GDMixture) ;
    else
      MGDCMsetHighlighted(costModel, GDUnknown) ;
  } else if (which == GD_NET) {
    costModel = MGDnetCostModel(this) ;
    net = group ? group->net : NULL ;

    if (net == NULL)
      MGDCMsetHighlighted(costModel, GDUnknown) ;
    else if (McostModel(net) == NULL)
      MGDCMsetHighlighted(costModel, GDNone) ;
    else if (McostModel(net)->type == CM_SUMSQUARE)
      MGDCMsetHighlighted(costModel, GDSumSquare) ;
    else if (McostModel(net)->type == CM_MIXTURE)
      MGDCMsetHighlighted(costModel, GDMixture) ;
    else
      MGDCMsetHighlighted(costModel, GDUnknown) ;
  } else
    return ;

  showCostModel(costModel) ;
}
/********************************************************************/
static void	setNetNumGaussiansCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  GroupDisplay		this = (GroupDisplay)clientData ;
  CostModelDisplay	costModel = MGDgroupCostModel(this) ;
  String		string = XawDialogGetValueString(XtParent(widget)) ;
  
  XtPopdown(MGDCMquery(costModel)) ;
  setNumGaussians(this, atoi(string), GD_NET) ;
}
/********************************************************************/
static void	setGroupNumGaussiansCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  GroupDisplay		this = (GroupDisplay)clientData ;
  CostModelDisplay	costModel = MGDgroupCostModel(this) ;
  String		string = XawDialogGetValueString(XtParent(widget)) ;
  
  XtPopdown(MGDCMquery(costModel)) ;
  setNumGaussians(this, atoi(string), GD_GROUP) ;
}
/********************************************************************/
static void	setNumGaussians(this, num, which)
  GroupDisplay	this ;
  int		num ;
  int		which ;
{
  CostModelDisplay	costModel ;
  if (which == GD_GROUP)
    costModel = MGDgroupCostModel(this) ;
  else if (which == GD_NET)
    costModel = MGDnetCostModel(this) ;
  else
    return ;

  if (num <= 0)
    num = 1 ;
  MGDCMsetNumGaussians(costModel, num) ;
  applyCostModel(this, which) ;
  showCostModel(costModel) ;
}
/********************************************************************/
static void	applyCostModel(this, which)
  GroupDisplay	this ;
  int		which ;
{
  Group			group = MGDgroup(this) ;
  CostModelDisplay	costModel ;
  Net			net ;

  if (group == NULL)
    return ;

  net = group->net ;

  if (which == GD_NET) {
    costModel = MGDnetCostModel(this) ;
    group     = NULL ;
  } else if (which == GD_GROUP){
    costModel = MGDgroupCostModel(this) ;
  } else
    return ;

  switch(MGDCMhighlighted(costModel)) {
  case GDNone:
    deleteCost(net, group) ;
    break ;
  case GDSumSquare:
    addSumSquareCost(net, group) ;
    break ;
  case GDMixture:
    addMixtureCost(net, group, MGDCMnumGaussians(costModel), -0.3, 0.3);
  case GDUnknown:
  default:
    break ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		showCostModelCB
 *	Description:	shows the currently set cost model for a 
 *			GroupDisplay
 *	Parameters:
 *	  Widget	widget     - UNUSED
 *	  XtPointer	clientData - the group display
 *	  XtPointer	callData   - UNUSED
 *	  GroupDisplay	this	   - the group display
 *	Return Value:
 *	  static void	showCostModelCB - NONE 
 *********************************************************************/
static void	showCostModelCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  showCostModel((CostModelDisplay)clientData) ;
}
/********************************************************************/
static void		showCostModel(this)
  CostModelDisplay	this ;
{
  Widget		choice ;
  String		label ;
  int			idx ;

  if (MGDCMhighlighted(this) == GDUnknown)
    choice = NULL ;
  else
    choice = MGDCMoption(this, MGDCMhighlighted(this)) ;

  for (idx = 0 ; idx < MGDCMnumOptions(this) ; ++idx) {
    if (choice == MGDCMoption(this, idx))
      checkMenuItem(MGDCMoption(this, idx), TRUE) ;
    else
      checkMenuItem(MGDCMoption(this, idx), FALSE) ;
  }

  if (choice) {
    label = getLabel(choice) ;
    if (MGDCMhighlighted(this) == GDMixture)
      widgetPrintf(MGDCMlabel(this), "%s (%d)", 
		  label, MGDCMnumGaussians(this));
    else
      widgetPrintf(MGDCMlabel(this), "%s", label) ;
    XtFree(label) ;
  } else {
    widgetPrintf(MGDCMlabel(this), "") ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		setGroupList(CB)
 *	Description:	sets the list of groups (and then the group
 *			if one is selected) in a GroupDisplay.
 *	Parameters:
 *	  Widget	widget 	   - UNUSED
 *	  XtPointer	clientData - the GroupDisplay
 *	  XtPointer	callData   - UNUSED
 *	  GroupDisplay	this	   - the GroupDisplay
 *	Return Value:
 *	  static void	setGroupList(CB)- NONE
 *********************************************************************/
static void	controllerCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  if ((DisplayNotification)callData == DNStructureChange)
    setGroupList((GroupDisplay)clientData) ;
}
/********************************************************************/
static void	setGroupListCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  setGroupList((GroupDisplay)clientData) ;
}
/********************************************************************/
static void	setGroupList(this)
  GroupDisplay	this ;
{
  MGDsetGroupList(this, listGroups(MGDgroupList(this))) ;
  XtVaSetValues(MGDgroupSelection(this), XtNlist, MGDgroupList(this), NULL) ;
  setGroup(this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setGroup(CB)
 *	Description:	sets the selected group in a GroupDisplay (NULL
 *			if none selected), and then the selected masks
 *			from that group.
 *	Parameters:
 *	  Widget	widget 	   - UNUSED
 *	  XtPointer	clientData - the GroupDisplay
 *	  XtPointer	callData   - UNUSED
 *	  GroupDisplay	this	   - the GroupDisplay
 *	Return Value:
 *	  static void	setGroup(CB)- NONE
 *********************************************************************/
static void	setGroupCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  setGroup((GroupDisplay)clientData) ;
}
/********************************************************************/
static void	setGroup(this)
  GroupDisplay	this ;
{
  Widget		selection = MGDgroupSelection(this) ;
  XawListReturnStruct	*rs	  = XtSelectionGetSelected(selection) ;

  this->group = NULL ;
  if (currentNet && rs && rs->list_index != XAW_LIST_NONE)
    this->group = groupFromName(currentNet, rs->string) ;

  setCostModel(this, GD_NET) ;
  setCostModel(this, GD_GROUP) ;
  setHighlighted(this) ;
  gdCallCallback(this, GDSelect) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setHighlighted
 *	Description:	sets the highlighted masks in all the MaskSets
 *			of the GroupDisplay from the highlighted group.
 *	Parameters:
 *	  GroupDisplay	this	   - the GroupDisplay
 *	Return Value:
 *	  static void	setHighlighted- NONE
 *********************************************************************/
static void	setHighlighted(this)
  GroupDisplay	this ;
{
  int		idx ;
  Group		group	= MGDgroup(this) ;
  Mask		newMask	= 0 ;
  GroupMaskSet	maskSet ;

  if (group)
    newMask = group->type ;

  for (idx = 0 ; maskSet = MGDmaskSet(this, idx) ; ++idx) {
    MGDunhighlight(maskSet, MGDmask(maskSet)) ;
    MGDhighlight  (maskSet, newMask) ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		showHighlighted(CB)
 *	Description:	shows the selected masks in all of the MaskSets
 *			of a GroupDisplay (DOES NOT GET THEM FROM THE
 *			GROUP).
 *	Parameters:
 *	  Widget	widget 	   - UNUSED
 *	  XtPointer	clientData - the GroupDisplay
 *	  XtPointer	callData   - UNUSED
 *	  GroupDisplay	this	   - the GroupDisplay
 *	Return Value:
 *	  static void	showHighlighted(CB)- NONE
 *********************************************************************/
static void	showHighlightedCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  showHighlighted((GroupMaskSet)clientData) ;
}
/********************************************************************/
static void	showHighlighted(this)
  GroupMaskSet	this ;
{
  int		idx ;

  for(idx = 0 ; MGDtoggle(this, idx) ; ++idx)
    checkMenuItem(MGDtoggle(this, idx), 
		  MGDtoggleMask(this, idx) & MGDhighlighted(this) 
		  ? TRUE : FALSE) ;

  doLabel(this) ;
}
/********************************************************************/




/*********************************************************************
 *	Name:		listGroups
 *	Description:	
 *	Parameters:
 *	Return Value:
 *	  static String	*listGroups - 
 *********************************************************************/
static String	*listGroups(list) 
  String	*list ;
{
  int	idx, listSize ;

  for (listSize = 0 ; list && list[listSize] ; ++listSize)
    ;

  if (currentNet == NULL || listSize != currentNet->numGroups) {
    if (list) {
      for (idx = 0 ; idx < listSize ; ++idx)
	XtFree((void *)list[idx]) ;
      XtFree((void *)list) ;
    }
  }

  if (currentNet == NULL) {
    listSize = 0 ;
    list = (String *)XtCalloc(listSize + 1, sizeof(String)) ;

  } else if (listSize != currentNet->numGroups) {
    listSize = currentNet->numGroups ;
    list = (String *)XtCalloc(listSize + 1, sizeof(String)) ;
    for (idx = 0 ; idx < currentNet->numGroups ; ++idx)
      list[idx] = strdup(currentNet->group[idx]->name) ;
  } else {
    for (idx = 0 ; idx < currentNet->numGroups ; ++idx) {
      if (strcmp(currentNet->group[idx]->name, list[idx])) {
	XtFree((void *)list[idx]) ;
	list[idx] = strdup(currentNet->group[idx]->name) ;
      }
    }
  }

  list[listSize] = NULL ;
  return list ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		doLabel
 *	Description:	prints all the highlighted mask names in the label 
 *			of a GroupMaskSet (or clears it if no highlightions)
 *	Parameters:
 *	  GroupMaskSet	this	- the mask set
 *	Return Value:
 *	  static void	doLabel - NONE
 *********************************************************************/
static void	doLabel(this)
  GroupMaskSet	this ;
{
  char		string[256] ;
  int		idx ;
  
  string[0] = '\0' ;
  for (idx = 0 ; MGDtoggle(this, idx) ; ++idx) {
    if (MGDtoggleMask(this, idx) & MGDhighlighted(this)) {
      if (string[0] != '\0')
	strcat(string, ", ") ;
      strcat(string, XtName(MGDtoggle(this, idx))) ;
    }
  }
  
  widgetPrintf(MGDlabel(this), "%s", string) ;
}
/********************************************************************/


/*******************************************************************
 *	Name:		registerGroupDisplay
 *	Description:	registers a groupDisplay for later retrieval
 *			by name
 *	Parameters:
 *	  GroupDisplay	this - the groupDisplay
 *	Return Value:
 *	  static void	registerGroupDisplay - NONE 
 *******************************************************************/
static GroupDisplay    *groupDisplayList ;
static Cardinal		listSize ;
/******************************************************************/
static void	registerGroupDisplay(this)
  GroupDisplay	this ;
{
  groupDisplayList
    = (GroupDisplay *)XtRealloc((XtPointer)groupDisplayList, 
				(listSize + 1)*sizeof(GroupDisplay)) ;
  groupDisplayList[listSize++] = this ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		deregisterGroupDisplay
 *	Description:	removes a groupDisplay from the registered list
 *	Parameters:
 *	  GroupDisplay	this - the GroupDisplay
 *	Return Value:
 *	  static void	deregisterGroupDisplay - NONE
 *******************************************************************/
static void	deregisterGroupDisplay(this)
  GroupDisplay	this ;
{
  int	idx ;
  for (idx = 0 ; idx < listSize && this != groupDisplayList[idx] ; ++idx)
    ;
  if (idx < listSize) {
    for ( ; idx < listSize ; ++idx)
      groupDisplayList[idx] = groupDisplayList[idx + 1] ;
    groupDisplayList
      = (GroupDisplay *)XtRealloc((XtPointer)groupDisplayList,
				  (--listSize)*sizeof(GroupDisplay)) ;
  }
}
/******************************************************************/


/*******************************************************************
 *	Name:		nameToGroupDisplay
 *	Description:	looks up a groupDisplay given its name
 *	Parameters:
 *	  String	name - the name to look for
 *	Return Value:
 *	  static GroupDisplay	nameToGroupDisplay - the GroupDisplay
 *******************************************************************/
static GroupDisplay	nameToGroupDisplay(name)
  String		name ;
{
  int	idx ;

  for (idx = 0 ; 
       idx < listSize && strcmp(name, MGDname(groupDisplayList[idx])) ; ++idx)
    ;

  if (idx < listSize)
    return groupDisplayList[idx] ;
  else
    return NULL ;
}
/******************************************************************/

