
/**********************************************************************
 * $Id: graphDisplay.c,v 1.12 93/01/11 11:19:42 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>

#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Text.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>

#include <xerion/Graph.h>
#include <xerion/MenuPane.h>
#include <xerion/display.h>

#include "display2Itf.h"
#include "displayUtils.h"
#include "Trace.h"
#include "graphDisplay.h"

typedef struct {
  Widget	shell ;
  Widget	fixedMin ;
  Widget	fixedMax ;
  Widget	maxValue ;
  Widget	minValue ;
  Widget	maxFormat ;
  Widget	minFormat ;
  Widget	numColumns ;
  Widget	jumpScroll ;
  Widget	graph ;
} *MiscGraphDialog ;

typedef struct GraphDisplay {
  Widget	shell ;		/* the shell containing the display	 */
  String	objectName ;	/* the name of the object being plotted	 */
  String	titleFormat ;	/* the format string to print the title	 */
  Widget	graphWidget ;	/* the graph widget			 */
  Widget	graphMenu ;	/* the graphOptions menuButton		 */
  Widget	plotButton ;	/* the button for plotting the object	 */
  MiscGraphDialog miscDialog ;	/* holds all the stuff for the misc menu */
} GraphDisplayObj, *GraphDisplay ;

/* private functions */
static void	createGraphCB   ARGS((Widget, XtPointer, XtPointer)) ;
static void	changeGraphCB   ARGS((Widget, XtPointer, XtPointer)) ;
static void	destroyGraphCB  ARGS((Widget, XtPointer, XtPointer)) ;
static void	destroyDialogCB ARGS((Widget, XtPointer, XtPointer)) ;
static void	plotValueCB     ARGS((Widget, XtPointer, XtPointer)) ;
static void	rescaleGraphCB  ARGS((Widget, XtPointer, XtPointer)) ;
static void	shiftGraphCB    ARGS((Widget, XtPointer, XtPointer)) ;
static void	clearGraphCB    ARGS((Widget, XtPointer, XtPointer)) ;
static void 	setSensitiveCB  ARGS((Widget, XtPointer, XtPointer)) ;
static void 	miscellaneousCB ARGS((Widget, XtPointer, XtPointer)) ;

static void	setSensitive	ARGS((GraphDisplay)) ;

#define MAXGRAPHS	64
static GraphDisplayObj	graphDisplay[MAXGRAPHS] ;

/*********************************************************************
 *	Name:		createGraphQuery
 *	Description:	creates the graph query popup (asks for variable
 *			to graph, and adds, buttons to make the graph)
 *	Parameters:
 *	  Widget	popup - the popup shell that is the parent
 *	Return Value:
 *	  export void	createGraphQuery - NONE
 *********************************************************************/
export void	createGraphQuery(popup)
  Widget	popup ;
{
  Widget	dialog, value, button ;
  
  dialog = XtVaCreateManagedWidget("graphDialog", dialogWidgetClass, popup,
				   NULL) ;

  XtVaSetValues(dialog, XtNvalue, "currentNet.error", NULL) ;

  button = XtVaCreateManagedWidget("ok",     commandWidgetClass, dialog, NULL);
  XtAddCallback(button, XtNcallback, popdownWidgetCB, (XtPointer)popup) ;
  XtAddCallback(button, XtNcallback, createGraphCB,   (XtPointer)dialog) ;

  button = XtVaCreateManagedWidget("cancel", commandWidgetClass, dialog, NULL);
  XtAddCallback(button, XtNcallback, popdownWidgetCB, (XtPointer)popup) ;

  value = XtNameToWidget(dialog, "value") ;
  XtSetKeyboardFocus(dialog, value) ;
  XtInstallAllAccelerators(value, dialog) ;
  
  XtVaSetValues(popup, XtNcreatePopupChildProc, NULL, NULL) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		createPopupGraph
 *	Description:	pops up a graph (creates it if necessary) given the
 *			name of the object being graphed
 *	Parameters:
 *	  String objectName - the name of the object being graphed
 *	Return Value:
 *	  Widget	popupGraph - NULL on failure, popup on success.
 *********************************************************************/
Widget		createPopupGraph(objectName, geometry)
  String	objectName ;
  String	geometry ;
{
  GraphDisplay	this ;
  int		idx ;

  for (idx = 0 ; 
       idx < MAXGRAPHS 
       && (graphDisplay[idx].objectName == NULL
	   || strcmp(objectName, graphDisplay[idx].objectName)) ; ++idx)
    ;
  
  if (idx == MAXGRAPHS) {
    createGraph(objectName) ;
    for (idx = 0 ; 
	 idx < MAXGRAPHS 
	 && (graphDisplay[idx].objectName == NULL
	     || strcmp(objectName, graphDisplay[idx].objectName)) ; ++idx)
      ;
    if (idx == MAXGRAPHS)
      return NULL ;
  } 

  this = &graphDisplay[idx] ;

  if (geometry)
    XtVaSetValues(this->shell, XtNgeometry, geometry, NULL) ;

  XtPopup(this->shell, XtGrabNone) ;
  XSetWMProtocols(XtDisplay(this->shell), 
		  XtWindow(this->shell), &wm_delete_window, 1);

  return this->shell ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		createGraph
 *	Description:	creates a graph display and returns a pointer
 *			to the popup shell surrounding it.
 *	Parameters:
 *	  char	*objectName - the name of the object to graph
 *	Return Value:
 *	  Widget 	createGraph - the popup shell around the graph
 *********************************************************************/
Widget 	createGraph(objectName)
  char	*objectName ;
{
  GraphDisplay	this ;
  Widget	menuPane, menu, widget, form, trace, changeShell ;
  char		name[128] ;
  String	ptr ;
  int		idx ;

  if (topLevel == NULL)
    return NULL ;

  for (idx = 0 ;
       idx < MAXGRAPHS && graphDisplay[idx].graphWidget != NULL ; ++idx)
    ;
  
  if (idx == MAXGRAPHS)
    return NULL ;
  this = &graphDisplay[idx] ;

  this->objectName = XtNewString(objectName) ;

  /********************************************
   * the shell and top level form
   */
  widget = XtVaCreatePopupShell("graphShell",
				topLevelShellWidgetClass, topLevel, NULL) ;
  this->shell       = widget ;
  this->titleFormat = getTitle(this->shell) ;
  widgetPrintf(this->shell, this->titleFormat, this->objectName) ;

  form = XtVaCreateManagedWidget("graphForm", formWidgetClass, this->shell,
				 NULL) ;
  redirectKeyboardInput(form) ;

  /********************************************
   * Display Updates 
   */
  sprintf(name, "graph %d", idx) ;
  trace = XtVaCreatePopupShell(name, traceWidgetClass, this->shell, 
			       NULL) ;
  XtAddCallback(trace, XtNcallback, plotValueCB, (XtPointer)this) ;

  /********************************************
   * Pulldown menu pane 
   */
  menuPane = XtVaCreateManagedWidget("menuPane", menuPaneWidgetClass, form,
				     NULL) ;

  /**********************
   * Options menu 
   */
  widget = XtVaCreateManagedWidget("options", menuButtonWidgetClass, menuPane, 
				   XtNmenuName, "optionsMenu",
				   NULL);

  menu = XtVaCreatePopupShell("optionsMenu", simpleMenuWidgetClass, widget,
			      NULL) ;
  
  widget = XtVaCreateManagedWidget("updates", smeBSBObjectClass,  menu, NULL) ;
  XtAddCallback(widget, XtNcallback, popupCenteredCB, (XtPointer)trace) ;

  widget = XtVaCreateManagedWidget("line",    smeLineObjectClass, menu, NULL) ;

  widget = XtVaCreateManagedWidget("close",   smeBSBObjectClass,  menu, NULL) ;
  XtAddCallback(widget, XtNcallback, popdownWidgetCB, (XtPointer)this->shell) ;

  /**********************
   * Graph menu 
   */
  widget = XtVaCreateManagedWidget("graph", menuButtonWidgetClass, menuPane,
				   XtNmenuName, "graphMenu",
				   NULL) ;

  this->graphMenu = widget ;
  menu = XtVaCreatePopupShell("graphMenu", simpleMenuWidgetClass, widget,
			      NULL) ;
  
  widget = XtVaCreateManagedWidget("rescale", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, rescaleGraphCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("shift",   smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, shiftGraphCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("reset",   smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, clearGraphCB,   (XtPointer)this) ;
  XtAddCallback(widget, XtNcallback, setSensitiveCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("misc",    smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(widget, XtNcallback, miscellaneousCB, (XtPointer)this) ;

  /**********************
   * Object menu 
   */
  widget = XtVaCreateManagedWidget("object", menuButtonWidgetClass, menuPane,
				   XtNmenuName, "objectMenu",
				   NULL) ;
  this->plotButton = widget ;

  menu = XtVaCreatePopupShell("objectMenu", simpleMenuWidgetClass, widget, 
			      NULL) ;
  
  widget = XtVaCreateManagedWidget("plotValue", smeBSBObjectClass, menu,
				   NULL) ;
  XtAddCallback(widget, XtNcallback, plotValueCB, (XtPointer)this) ;

  widget = XtVaCreateManagedWidget("changeObject", smeBSBObjectClass, menu,
				   NULL) ;
  changeShell = XtVaCreatePopupShell("changeShell",
				     transientShellWidgetClass, this->shell,
				     NULL) ;
  XtAddCallback(widget, XtNcallback, popupCenteredCB, (XtPointer)changeShell) ;

  createGraphQuery(changeShell) ;
  widget = XtNameToWidget(changeShell, "graphDialog.ok") ;
  XtRemoveAllCallbacks(widget, XtNcallback) ;
  XtAddCallback(widget, XtNcallback, popdownWidgetCB, (XtPointer)changeShell) ;
  XtAddCallback(widget, XtNcallback, changeGraphCB,  (XtPointer)this) ;
  XtAddCallback(widget, XtNcallback, setSensitiveCB, (XtPointer)this) ;

  /********************************************
   * The graph itself
   */

  /**********************
   * name the graph the object name with '.' replaced by '_'
   */
  strncpy(name, this->objectName, 128) ;
  for (ptr = name ; *ptr && ptr - name < sizeof(name) ; ++ptr) {
    if (*ptr == '.')
      *ptr = '_' ;
  }

  /**********************
   * make the graph
   */
  widget = XtVaCreateManagedWidget(name, graphWidgetClass, form,
				   XtNlabel, "",
				   NULL) ;
  this->graphWidget = widget ;
  redirectKeyboardInput(widget) ;

  /********************************************
   * att callbacks to the display itself
   */
  XtAddCallback(this->shell, 
		XtNpopdownCallback, destroyGraphCB, (XtPointer)this) ;
  XtAddCallback(controller, 
		XtNcallback,	    setSensitiveCB, (XtPointer)this) ;
  setSensitive(this) ;
  XtInstallAllAccelerators(this->graphWidget, form) ;

  return this->shell ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		createGraphCB
 *	Description:	actually builds a graph display, given the name
 *			of the object to graph
 *	Parameters:
 *	  Widget	widget - the widget the callback is attached to
 *	  XtPointer	clientData - the dialog widget to get the object
 *				name from
 *	  XtPointer	callData - UNUSED
 *	Return Value:
 *	  static void	createGraphCB - NONE
 *********************************************************************/
static void	createGraphCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget	popup, dialog = (Widget)clientData ;
  String	objectName ;

  objectName = XawDialogGetValueString(dialog) ;
  if (objectName == NULL || objectName[0] == '\0')
    return ;
  
  popup = createGraph(objectName) ;

  XtPopup(popup, XtGrabNone) ;
  XSetWMProtocols(XtDisplay(popup), XtWindow(popup), &wm_delete_window, 1);
}
/********************************************************************/


/*********************************************************************
 *	Name:		changeGraphCB
 *	Description:	changes the object being plotted int the graph
 *	Parameters:
 *	  Widget	widget     - a button in a dialog widget
 *	  XtPointer	clientData - the graphDisplay
 *	  XtPointer	callData   - UNUSED
 *	Return Value:
 *	  static void	changeGraphCB - NONE
 *********************************************************************/
static void	changeGraphCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  GraphDisplay	this = (GraphDisplay)clientData ;
  String	objectName ;
  Widget	parent = XtParent(widget) ;

  objectName = XawDialogGetValueString(parent) ;
  if (objectName == NULL || objectName[0] == '\0')
    return ;

  if (this->objectName)
    XtFree(this->objectName) ;
  this->objectName = XtNewString(objectName) ;

  widgetPrintf(this->shell, this->titleFormat, this->objectName) ;

  graphReset(this->graphWidget) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setSensitiveCB
 *	Description:	sets the sensitive field to true or false for
 *			all of the widgets that care.
 *	Parameters:
 *	  Widget	widget -  the widget the callback is attached to
 *	  XtPointer	clientData - a pointer to the graphDisplay object
 *	  XtPointer	callData   - UNUSED
 *	Return Value:
 *	  static void	setSensitiveCB - NONE
 *********************************************************************/
static void	setSensitiveCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  setSensitive((GraphDisplay)clientData) ;
}
/********************************************************************/
static void	setSensitive(this)
  GraphDisplay	this ;
{
  char		*valueString ;
  Boolean	sensitive ;

  valueString = getSimpleObjectValue(this->objectName, "%.8g") ;

  if (valueString == NULL || strcmp(valueString, "") == 0)
    sensitive = FALSE ;
  else
    sensitive = TRUE ;

  XtSetSensitive(this->graphMenu,	sensitive) ;
  XtSetSensitive(this->plotButton,	sensitive) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		destroyGraphCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	destroyGraphCB - 
 *********************************************************************/
static void	destroyGraphCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  GraphDisplay	this  = (GraphDisplay)clientData ;

  XtRemoveCallback(controller, XtNcallback, setSensitiveCB, (XtPointer)this) ;
  XtDestroyWidget(this->shell) ;
  if (this->objectName)
    XtFree((char *)this->objectName) ;
  if (this->miscDialog)
    XtFree((char *)this->miscDialog) ;

  memset(this, 0, sizeof(*this)) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		clearGraphCB
 *	Description:	clears a graph (passed in as client data)
 *	Parameters:	
 *		Widget	widget 	   - the widget the callback is attached to
 *		XtPointer	clientData - the graph Widget
 *		XtPointer	callData   - the call data (unused)
 *	Return Value:	NONE
 ***********************************************************************/
static void	clearGraphCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  graphReset(((GraphDisplay)clientData)->graphWidget) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		rescaleGraphCB
 *	Description:	rescales a graph (passed in as client data)
 *	Parameters:	
 *		Widget	widget 	   - the widget the callback is attached to
 *		XtPointer	clientData - the graph Widget
 *		XtPointer	callData   - the call data (unused)
 *	Return Value:	NONE
 ***********************************************************************/
static void	rescaleGraphCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  graphRescale(((GraphDisplay)clientData)->graphWidget) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		shiftGraphCB
 *	Description:	shifts a graph (passed in as client data)
 *	Parameters:	
 *		Widget	widget 	   - the widget the callback is attached to
 *		XtPointer	clientData - the graph Widget
 *		XtPointer	callData   - the call data (unused)
 *	Return Value:	NONE
 ***********************************************************************/
static void	shiftGraphCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  graphShift(((GraphDisplay)clientData)->graphWidget) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		plotValueCB
 *	Description:	adds values to the graph widget using 
 *			updateGraphDisplay from above
 *	Parameters:
 *		Widget	widget 	   - the widget the callback is attached to
 *		XtPointer	clientData - the graph widget
 *		XtPointer	callData   - the call data (unused)
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	plotValueCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  GraphDisplay	this = (GraphDisplay)clientData ;
  char		*valueString ;
  float		value ;

  valueString = getSimpleObjectValue(this->objectName, "%.8g") ;
  if (valueString == NULL)
    return ;

  value = atof(valueString) ;
  graphAddValue(this->graphWidget, value) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		setMiscellaneousCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData -
 *	Return Value:
 *	  static void	setMiscellaneousCB - 
 *********************************************************************/
static void	setMiscellaneousCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MiscGraphDialog	dialog = (MiscGraphDialog)clientData ;
  Boolean		fixedMin, fixedMax ;
  float			minValue, maxValue ;
  int			numColumns, jumpScroll ;
  String		maxFormat, minFormat, string ;

  minValue   = 0.0 ;
  maxValue   = 0.0 ;
  numColumns = 0 ;
  jumpScroll = 0 ;
  minFormat  = NULL ;
  maxFormat  = NULL ;

  string = getText(dialog->minValue) ;
  if (string != NULL)
    minValue = atof(string) ;

  string = getText(dialog->maxValue) ;
  if (string != NULL)
    maxValue = atof(string) ;

  string = getText(dialog->maxFormat) ;
  if (string != NULL)
    maxFormat = string ;

  string = getText(dialog->minFormat) ;
  if (string != NULL)
    minFormat = string ;

  string = getText(dialog->numColumns) ;
  if (string != NULL)
    numColumns = atof(string) ;

  string = getText(dialog->jumpScroll) ;
  if (string != NULL)
    jumpScroll = atof(string) ;
    
  XtVaGetValues(dialog->fixedMin, XtNstate, &fixedMin, NULL) ;
  XtVaGetValues(dialog->fixedMax, XtNstate, &fixedMax, NULL) ;

  XtVaSetValues(dialog->graph,
		XtNminValue,	&minValue,
		XtNmaxValue,	&maxValue,
		XtNmaxFormat,	maxFormat,
		XtNminFormat,	minFormat,
		XtNjumpScroll,	jumpScroll,
		XtNnumColumns,	numColumns,
		XtNfixedMin,	fixedMin,
		XtNfixedMax,	fixedMax,
		NULL) ;
}

/*********************************************************************
 *	Name:		miscellaneousCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	miscellaneousCB - 
 *********************************************************************/
static void	miscellaneousCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  GraphDisplay	this = (GraphDisplay)clientData ;
  float		*min, *max ;
  Boolean	fixedMin, fixedMax ;
  int		numColumns, jumpScroll ;
  String	minFormat, maxFormat ;
  MiscGraphDialog	dialog ;

  if (this->miscDialog == NULL) {
    Widget	form, button, table ;
    Position	x, y ;
    
    dialog = (MiscGraphDialog)XtMalloc(sizeof(*dialog)) ;
    this->miscDialog = dialog ;

    dialog->shell = XtVaCreatePopupShell("miscShell",
					 transientShellWidgetClass,
					 this->shell, 
					 NULL) ;
    form = XtVaCreateManagedWidget("form", 
				   formWidgetClass, dialog->shell, NULL) ;

    table = startTable(form) ;
    dialog->minValue   = addTableItem("minValue",   &table, TRUE) ;
    dialog->maxValue   = addTableItem("maxValue",   &table, TRUE) ;
    dialog->maxFormat  = addTableItem("maxFormat",  &table, FALSE) ;
    dialog->minFormat  = addTableItem("minFormat",  &table, FALSE) ;
    dialog->numColumns = addTableItem("numColumns", &table, FALSE) ;
    dialog->jumpScroll = addTableItem("jumpScroll", &table, FALSE) ;

    dialog->fixedMax   = XtNameToWidget(form, "maxValueToggle") ;
    dialog->fixedMin   = XtNameToWidget(form, "minValueToggle") ;

    button = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
				     XtNfromVert,	dialog->jumpScroll,
				     NULL) ;
    XtAddCallback(button, XtNcallback, setMiscellaneousCB, (XtPointer)dialog) ;
    XtAddCallback(button, XtNcallback, popdownWidgetCB, 
		  (XtPointer)dialog->shell) ;

    button = XtVaCreateManagedWidget("cancel", commandWidgetClass, form,
				     XtNfromVert,	dialog->jumpScroll,
				     XtNfromHoriz,	button,
				     NULL) ;
    XtAddCallback(button, XtNcallback, popdownWidgetCB, 
		  (XtPointer)dialog->shell) ;
    XtAddCallback(button, XtNdestroyCallback, destroyDialogCB,
		  (XtPointer)dialog) ;

    XtInstallAllAccelerators(form, form) ;
  }
  dialog = this->miscDialog ;

  dialog->graph = this->graphWidget ;

  XtVaGetValues(dialog->graph,
		XtNfixedMin,	&fixedMin,
		XtNfixedMax,	&fixedMax,
		XtNminValue,	&min,
		XtNmaxValue,	&max,
		XtNmaxFormat,	&maxFormat,
		XtNminFormat,	&minFormat,
		XtNnumColumns,	&numColumns,
		XtNjumpScroll,	&jumpScroll,
		NULL) ;

  XtVaSetValues(dialog->fixedMin, XtNstate, fixedMin, NULL) ;
  XtVaSetValues(dialog->fixedMax, XtNstate, fixedMax, NULL) ;

  widgetPrintf(dialog->minValue,   "%g", *min) ;
  widgetPrintf(dialog->maxValue,   "%g", *max) ;
  widgetPrintf(dialog->maxFormat,  "%s", maxFormat) ;
  widgetPrintf(dialog->minFormat,  "%s", minFormat) ;
  widgetPrintf(dialog->numColumns, "%d", numColumns) ;
  widgetPrintf(dialog->jumpScroll, "%d", jumpScroll) ;

  popupCentered(dialog->shell, XtGrabExclusive) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		destroyDialogCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	destroyDialogCB - 
 *********************************************************************/
static void	destroyDialogCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  MiscGraphDialog	this  = (MiscGraphDialog)clientData ;

  if (this)
    XtFree((char *)this) ;
}
/********************************************************************/
