
/**********************************************************************
 * $Id: displayUtils.c,v 1.15 93/02/09 16:45:53 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 *
 *			 All Rights Reserved
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/
#define __displayUtils_c_

#include <stdio.h>
#include <varargs.h>
#include <sys/types.h>
#include <signal.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/SmeBSB.h>

#include <xerion/useful.h>
#include "displayUtils.h"
#include "check.h"

static void	keyboardCB ARGS((Widget, XtPointer, XEvent *, Boolean *));

/***********************************************************************
 *	Name:		widgetPrintf
 *	Description:	same as printf, but writes to a widget. If the
 *			widget is a subclass of label it writes to the
 *			label, if it is a subclass of text it writes
 *			to the text resource, if it is a subclass of
 *			shell it writes to the title, otherwise it
 *			writes nothing and returns EOF. It doesn't
 *			update the field if the string is unchanged.
 *	Parameters:	va_alist - this uses varargs, so be careful
 *				that the format string is correct.
 *	Return Value:	whatever vsprintf returns.
 ***********************************************************************/
int	widgetPrintf(va_alist)
  va_dcl
{
  va_list 	args ;
  Widget	widget ;
  int		returnValue ;
  char		*format, *oldString, newString[BUFSIZ] ;

  va_start(args) ;
  widget = va_arg(args, Widget) ;
  format = va_arg(args, char *) ;

  if (XtIsSubclass(widget, labelWidgetClass)
      || XtIsSubclass(widget, smeBSBObjectClass))
    XtVaGetValues(widget, XtNlabel, &oldString, NULL) ;
  else if (XtIsSubclass(widget, textWidgetClass))
    XtVaGetValues(widget, XtNstring, &oldString, NULL) ;
  else if (XtIsSubclass(widget, shellWidgetClass))
    XtVaGetValues(widget, XtNtitle, &oldString, NULL) ;
  else
    return EOF ;

  returnValue = vsprintf(newString, format, args) ;

  if (strcmp(newString, oldString)) {
    if (XtIsSubclass(widget, labelWidgetClass)
	|| XtIsSubclass(widget, smeBSBObjectClass))
      XtVaSetValues(widget, XtNlabel, newString, NULL) ;
    else if (XtIsSubclass(widget, textWidgetClass))
      XtVaSetValues(widget, XtNstring, newString, NULL) ;
    else if (XtIsSubclass(widget, shellWidgetClass))
      XtVaSetValues(widget, XtNtitle, newString, NULL) ;
  }
  va_end(args) ;
  return returnValue ;
}
/**********************************************************************/



/***********************************************************************
 *	Name:		widgetGetText
 *	Description:	copies the text of a widget into a user supplied
 *			buffer. If the widget is a subclass of label
 *			it gets the label, if it is a subclass of text
 *			gets the text resource, if it is a subclass of
 *			shell it gets the title, otherwise it gets
 *			nothing and and returns NULL.
 *	Parameters:	
 *		  Widget	widget - the widget
 *		  char		buffer[] - the buffer to copy to
 *		  int		size - maximum number of chars to copy
 *	Return Value:	
 *		 String		widgetGetText - the string, NULL if error
 ***********************************************************************/
String		widgetGetText(widget, buffer, size)
  Widget	widget ;
  char		buffer[] ;
  int		size ;
{
  int		returnValue ;
  char		*ptr ;

  if (XtIsSubclass(widget, labelWidgetClass)
      || XtIsSubclass(widget, smeBSBObjectClass))
    XtVaGetValues(widget, XtNlabel, &ptr, NULL) ;
  else if (XtIsSubclass(widget, textWidgetClass))
    XtVaGetValues(widget, XtNstring, &ptr, NULL) ;
  else if (XtIsSubclass(widget, shellWidgetClass))
    XtVaGetValues(widget, XtNtitle,  &ptr, NULL) ;
  else
    ptr == NULL ;

  if (ptr) 
    strncpy(buffer, ptr, size) ;

  if (XtIsSubclass(widget, labelWidgetClass)
      || XtIsSubclass(widget, shellWidgetClass)
      || XtIsSubclass(widget, smeBSBObjectClass))
    XtFree(ptr) ;
        
  return ptr ? buffer : NULL ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		getTitle
 *	Description:	returns a copy of the title resource from a shell
 *			widget
 *	Parameters:
 *	  Widget	widget - the widget to get the title from
 *	Return Value:
 *	  char		*getTitle - a copy of the string
 *********************************************************************/
char		*getTitle(widget)
  Widget	widget ;
{
  char		*ptr ;

  XtVaGetValues(widget, XtNtitle,  &ptr, NULL) ;
  if (ptr != NULL)
    return XtNewString(ptr) ;
  else
    return NULL ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		getLabel
 *	Description:	returns a copy of the label in a widget
 *	Parameters:
 *	  Widget	widget - the widget to get the label from
 *	Return Value:
 *	  char		*getLabel - a copy of the string
 *********************************************************************/
char		*getLabel(widget)
  Widget	widget ;
{
  char		*ptr ;

  XtVaGetValues(widget, XtNlabel,  &ptr, NULL) ;
  if (ptr != NULL)
    return XtNewString(ptr) ;
  else
    return NULL ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		getText
 *	Description:	returns a copy of the string in a text widget
 *	Parameters:
 *	  Widget	widget - the widget to get the text from
 *	Return Value:
 *	  char		*getText - the string, *not* a copy
 *********************************************************************/
char		*getText(widget)
  Widget	widget ;
{
  char		*ptr ;

  XtVaGetValues(widget, XtNstring,  &ptr, NULL) ;
  return ptr ;
}
/********************************************************************/


static Widget	tableParent ;

/*********************************************************************
 *	Name:		startTable
 *	Description:	initializes a table of options
 *	Parameters:
 *	  Widget	form - the form to be the parent
 *	Return Value:
 *	  Widget	startTable - NULL
 *********************************************************************/
Widget		startTable(form)
  Widget	form ;
{
  tableParent = form ;
  return NULL ;
}
/********************************************************************/

/*********************************************************************
 *	Name:		addTableItem
 *	Description:	creates a labeled edit widget with
 *	Parameters:
 *	  String	name - the name for the edit widget (Label will
 *				be called name"Label"
 *	  Widget	*table - the value returned from startTable
 *	  Boolean	toggleToo - if TRUE add a toggle to the right
 *				called name"Toggle"
 *	Return Value:
 *	  static Widget	addTableItem - the edit widget
 *********************************************************************/
Widget		addTableItem(name, table, toggleToo)
  String	name ;
  Widget	*table ;
  Boolean	toggleToo ;
{
  char		labelName[BUFSIZ] ;
  Widget	label, edit, toggle ;
  XtVarArgsList	fromVert ;

  sprintf(labelName, "%sLabel", name) ;

  if (*table)
    fromVert = XtVaCreateArgsList(NULL, XtNfromVert, *table, NULL) ;
  else
    fromVert = XtVaCreateArgsList(NULL, NULL) ;
  

  label = XtVaCreateManagedWidget(labelName, labelWidgetClass, tableParent,
				  XtVaNestedList, fromVert,
				  NULL) ;

  edit = XtVaCreateManagedWidget(name, asciiTextWidgetClass, tableParent,
				 XtNfromHoriz,	 label,
				 XtNtype,	 XawAsciiString,
				 XtNeditType,	 XawtextEdit,
				 XtVaNestedList, fromVert,
				 NULL) ;
				 
  if (toggleToo) {
    sprintf(labelName, "%sToggle", name) ;
    toggle = XtVaCreateManagedWidget(labelName, toggleWidgetClass, tableParent,
				     XtNfromHoriz,	edit,
				     XtVaNestedList,	fromVert,
				     NULL) ;
  }
  XtFree(fromVert) ;

  *table = edit ;
  return edit ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		popupInterruptDialog
 *	Description:	pops up a dialog that will interrupt the process.
 *	Parameters:
 *	  Widget	widget - the widget to center the dialog in.
 *	Return Value:
 *	  static int	popupInterruptDialog - the id to pop it down with
 *********************************************************************/
int	popupInterruptDialog(widget)
 Widget	widget ;
{
  int		pid ;
  Position	x, y ;
  Dimension	width, height ;
  char		geometryString[32] ;
  char		mypidString[32] ;

  XtVaGetValues(widget,
		XtNx, &x, XtNy, &y,
		XtNwidth, &width, XtNheight, &height,
		NULL) ;
  x += width/2 - 64 ;
  y += height/2 - 24 ;
  sprintf(geometryString, "*popup.geometry: +%d+%d", x, y) ;

  sprintf(mypidString, "%d", getpid()) ;
  pid = fork() ;
  if (pid == 0) {
    /* ingore SIGINT (kick will be sending it), exec kick process and ... */
    signal(SIGINT, SIG_IGN) ;
    execlp("xkick",
	   "xkick", "-xrm", geometryString, mypidString, NULL) ;
    exit(1);
  }
  return pid ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		popdownInterruptDialog
 *	Description:	pops down the interrupt dialog (actually, it
 *			kills the process)
 *	Parameters:
 *	 int		pid - the id identifying the process
 *	Return Value:
 *	  static void	popdownInterruptDialog - NONE
 *********************************************************************/
void	popdownInterruptDialog(pid)
 int	pid ;
{
  if (pid > 0) {
   kill(pid, SIGKILL) ;
   wait(0) ;
 }
}
/********************************************************************/


/*********************************************************************
 *	Name:		keyboardCB
 *	Description:	redirects keyboard input another window
 *	Parameters:
 *	  Widget	w - the widget getting input
 *	  XtPointer	client_data - pointer to the window id
 *	  XKeyEvent	*event - the keyboard event
 *	Return Value:
 *	  static void	keyboardCB  - NONE
 *********************************************************************/
static void	keyboardCB (w, client_data, event, continue_to_dispatch)
  Widget	w;
  XtPointer	client_data;
  XEvent	*event;
  Boolean	*continue_to_dispatch ;
{
  XKeyEvent	*keyEvent = (XKeyEvent *)event ;

  keyEvent->window = *(Window *)client_data ;
  XSendEvent(XtDisplay(w), keyEvent->window, True, KeyPressMask, event) ;

  return ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		redirectKeyboardInput
 *	Description:	redirects keyboard input to the parent XTerm
 *			(using the environment variable WINDOWID).
 *			If it can't get WINDOWID, it does nothing.
 *	Parameters:
 *	  Widget	widget - the widget to redirect from
 *	Return Value:
 *	  void		redirectKeyboardInput - NONE
 *********************************************************************/
void		redirectKeyboardInput(widget)
  Widget	widget ;
{
  static Window xtermWindowId = (Window)0 ;
  if (xtermWindowId == 0) {
    String	idString = getenv("WINDOWID") ;
    if (idString)
      xtermWindowId = (Window) atol(idString) ;
  }
  if (xtermWindowId)
    XtAddEventHandler(widget, KeyPressMask, False, 
		      keyboardCB, (XtPointer)&xtermWindowId);
}
/********************************************************************/


/*********************************************************************
 *	Name:		popupCenteredCB
 *	Description:	pops up the "Display Updates" shell in the
 *			center of its parent
 *	Parameters:
 *	  Widget	widget - the widget the cb is attached  to
 *	  XtPointer	clientData - the popup shell to pop up
 *	  XtPointer	callData - UNUSED
 *	Return Value:
 *	  export void	popupCenteredCB - NONE
 *********************************************************************/
void		popupCenteredCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget	popup  = (Widget)clientData ;

  if (popup == NULL)
    return ;

  popupCentered(popup, XtGrabExclusive) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		popupCentered
 *	Description:	pops up a shell centered in it's parent 
 *	Parameters:
 *	  Widget	popup - the popup shell
 *	  XtGrabKind	grabKind - the grab kind.
 *	Return Value:
 *	  export void	popupCentered - NONE
 *********************************************************************/
void		popupCentered(popup, grabKind)
  Widget	popup ;
  XtGrabKind	grabKind ;
{
  Position	x, y ;
  Dimension	width, height ;
  Widget	parent ;

  for (parent = XtParent(popup) ; 
       !XtIsSubclass(parent, shellWidgetClass) ; parent = XtParent(parent)) 
    ;

  if (parent == NULL)
    return ;

  XtVaGetValues(parent,
		XtNx, &x, XtNy, &y, XtNwidth, &width, XtNheight, &height,
		NULL) ;
  
  /* center the popup in the center of its parent */
  x += width/2  ;
  y += height/2 ;
  
  /* make sure the popup doesn't get managed, then realize it,
   * so we can get its dimensions */
  XtVaSetValues(popup, XtNmappedWhenManaged, FALSE, NULL) ;
  XtRealizeWidget(popup) ;

  XtVaGetValues(popup, XtNwidth, &width, XtNheight, &height, NULL) ;

  x -= width/2 ;
  y -= height/2 ;
  XtVaSetValues(popup, XtNx, x, XtNy, y, NULL) ;
  XtPopup(popup, grabKind) ;
}
/********************************************************************/


/**********************************************************************
 *	Name:		popdownWidgetCB
 *	Description:	pops down a popup shell which is passed in as
 *			the client data
 *	Parameters:
 *	  Widget	widget 	   - the widget the callback is attached to
 *	  XtPointer	clientData - the shell to pop down
 *	  XtPointer	callData   - the call data (unused)
 *	Return Value:
 *		NONE
 **********************************************************************/
void		popdownWidgetCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  Widget	shell ;

  for (shell = (Widget)clientData ;
       !XtIsSubclass(shell, shellWidgetClass) ; shell = XtParent(shell)) 
    ;

  if (shell == NULL)
    return ;

  XtPopdown(shell) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		checkMenuItem
 *	Description:	puts a check beside a smeBSB (or clears it)
 *			depending on the flag passed in
 *	Parameters:
 *	  Widget	widget - the widget to put the check in
 *	  Boolean	flag   - TRUE - put check, FALSE - clear it
 *	Return Value:
 *	  static void	checkMenuItem - NONE
 *********************************************************************/
void		checkMenuItem(widget, flag)
  Widget	widget ;
  int		flag ;
{
  static Pixmap	check = NULL ;

  if (check == NULL && XtIsRealized(widget)) {
    check = XCreateBitmapFromData(XtDisplayOfObject(widget),
				  XtWindowOfObject(widget), 
				  check_bits, check_width, check_height) ;
  }
  XtVaSetValues(widget, 
		XtNleftMargin, check_width,
		XtNleftBitmap, flag ? check : NULL, 
		NULL) ;
}
/********************************************************************/

/********************************************************************/
static Boolean	firstTime = TRUE ;
static XContext	context ;
static void 	saveContextCB	 ARGS((Widget, XtPointer, XtPointer)) ;
/********************************************************************/

/*********************************************************************
 *	Name:		storeData
 *	Description:	stores data on the shell surrounding the widget
 *			DON'T TRY TO RETRIEVE THE DATA BEFORE THE SHELL
 *			IS REALIZED.
 *	Parameters:
 *	  Widget	w - the widget
 *	  XtPointer 	data - the data
 *	Return Value:
 *	  void		storeData - NONE
 *********************************************************************/
void		storeData(w, data) 
  Widget	w ;
  XtPointer 	data ;
{
  for ( ; w && !XtIsSubclass(w, shellWidgetClass) ; w = XtParent(w)) 
    ;

  if (w == NULL)
    return ;

  if (firstTime) {
    context = XUniqueContext() ; 
    firstTime = FALSE ;
  }

  if (XtIsRealized(w)) {
    XSaveContext(XtDisplay(w), XtWindow(w), context, data) ;
  } else {
    XtAddCallback(w, XtNpopupCallback, saveContextCB, data) ;
  }
}
/********************************************************************/

/*********************************************************************
 *	Name:		retrieveData
 *	Description:	retrieves data stored on the shell surrounding
 *			a widget (with storeData)
 *	Parameters:
 *	  Widget	w - the widget
 *	Return Value:
 *	  XtPointer	retrieveData - the data, NULL if any problem
 *********************************************************************/
XtPointer	retrieveData(w)
  Widget	w ;
{
#if XtSpecificationRelease <= 4
  caddr_t	*data ;
#else
  XPointer	data ;
#endif

  if (firstTime)
    return NULL ;

  for ( ; w && !XtIsSubclass(w, shellWidgetClass) ; w = XtParent(w)) 
    ;

  if (w == NULL)
    return ;

  if (XFindContext(XtDisplay(w), XtWindow(w), context, &data))
    return NULL ;
  else
    return (XtPointer)data ;
}
/********************************************************************/

/*********************************************************************
 *	Name:		saveContextCB
 *	Description:	internal callback used to store the data when
 *			the shell is realized
 *	Parameters:
 *	  Widget	widget - the shell
 *	  XtPointer	clientData - the data to store
 *	  XtPointer	callData - UNUSED
 *	Return Value:
 *	  static void	saveContextCB - NONE
 *********************************************************************/
static void	saveContextCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  if (!firstTime && XtIsSubclass(widget, shellWidgetClass)) {
    XtSetMappedWhenManaged(widget, FALSE) ;
    XtRealizeWidget(widget) ;
    XSaveContext(XtDisplay(widget), XtWindow(widget), context, clientData) ;
    XtSetMappedWhenManaged(widget, TRUE);
  }
  XtRemoveCallback(widget, XtNpopupCallback, saveContextCB, clientData) ;
}
/********************************************************************/
