
/**********************************************************************
 * $Id: display2Sim.c,v 1.9 93/04/14 09:42:24 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>

#include <xerion/useful.h>
#include <xerion/simulator.h>
#include <xerion/major.h>
#include <xerion/minimize.h>
#include <xerion/commands.h>
#include <xerion/minimizeCom.h>

#include "display2Itf.h"
#include "display2Sim.h"

#define MAX(x,y)	(((x) > (y)) ? (x) : (y))
#define MIN(x,y)	(((x) < (y)) ? (x) : (y))
#define EPSILON		(Real)(1e-20)

static void	countUnits        ARGS((Group	group, void	*data)) ;
static void	getUnitActivation ARGS((Unit	unit,  void	*data)) ;
static void	getUnitTarget     ARGS((Unit	unit,  void	*data)) ;
static String	*unitFields	  ARGS(()) ;
static Real	unitValue	  ARGS((Unit, int, int)) ;
static void	setUnitValue	  ARGS((Unit, int, int, double)) ;
static String	*linkFields	  ARGS(()) ;
static Real	linkValue	  ARGS((Link, int)) ;
static void	setLinkValue	  ARGS((Link, int, double)) ;

String	*(*listUnitFields)  ARGS((void))		   = unitFields ;
Real	(*unitValueProc)    ARGS((Unit, int, int))	   = unitValue ;
void	(*setUnitValueProc) ARGS((Unit, int, int, double)) = setUnitValue ;
String	*(*listLinkFields)  ARGS((void))	 	   = linkFields ;
Real	(*linkValueProc)    ARGS((Link, int))		   = linkValue ;
void	(*setLinkValueProc) ARGS((Link, int, double))	   = setLinkValue ;

Minimize	currentMZ ;

/***********************************************************************
 *
 *	NET ROUTINES
 *
 ***********************************************************************/


/***********************************************************************
 *	Name:		listNetParams
 *	Description:	returns an array of the network learning
 *			parameters.
 *	Parameters:	NONE
 *	Return Value:	
 *		char	**listNetParams - a STATIC array of the names.
 ***********************************************************************/
char	**listNetParams() {
  return IListNetParams() ;
}
/**********************************************************************/
static char	**shortNames ;
static int	numParams ;
char	**listNetParamsShort() {
  int	newNumParams, idx ;
  char	**fieldNames = IListNetParams() ;

  for (newNumParams = 0 ; fieldNames[newNumParams] != NULL ; ++newNumParams)
    ;

  if (newNumParams >= numParams) {
    if (shortNames == NULL)
      shortNames = (char **)calloc(newNumParams+1, sizeof(char *)) ;
    else
      shortNames = (char **)realloc(shortNames, 
				    (newNumParams+1)*sizeof(char *)) ;
    if (shortNames == NULL)
      numParams = 0 ;
    else
      numParams = newNumParams ;
  }
  
  for (idx = 0 ; idx < numParams ; ++idx) {
    char	*string ;
    string = strrchr(fieldNames[idx], '.') ;
    if (string == NULL)
      shortNames[idx] = fieldNames[idx] ;
    else
      shortNames[idx] = string + 1 ;
  }
  shortNames[idx] = NULL ;

  return shortNames ;
}
/**********************************************************************/
char	*netParamLongName(shortName)
  char	*shortName ;
{
  int	idx ;
  static char	paramName[BUFSIZ] ;
  char		**longNames = IListNetParams() ;

  for (idx = 0 ; 
       idx < numParams && strcmp(shortName, shortNames[idx]) ; ++idx)
    ;

  if (idx == numParams)
    return NULL ;
  
  if (strchr(longNames[idx], '.') == NULL)
    return NULL ;

  if (strncmp(longNames[idx], 
	      "NetExtensionRec.", strlen("NetExtensionRec.")) == 0)
    sprintf(paramName, "%s%s", "currentNet.extension.", 
	    longNames[idx] + strlen("NetExtensionRec.")) ;
  else if (strncmp(longNames[idx], "NetRec.", strlen("NetRec.")) == 0)
    sprintf(paramName, "%s%s", "currentNet.", 
	    longNames[idx] + strlen("NetRec.")) ;
  else
    sprintf(paramName, "") ;
  
  return paramName ;
}
/**********************************************************************/
  

/***********************************************************************
 *	Name:		listMinimizeParams
 *	Description:	returns an array of the network learning
 *			parameters.
 *	Parameters:	NONE
 *	Return Value:	
 *		char	**listMinimizeParams - a STATIC array of the names.
 ***********************************************************************/
char	**listMinimizeParams() {
  return IListMinimizeParams() ;
}
/**********************************************************************/
static char	**shortMiniParamNames ;
static int	numMiniParams ;
char	**listMinimizeParamsShort() {
  int	newNumParams, idx ;
  char	**fieldNames = IListMinimizeParams() ;

  for (newNumParams = 0 ; fieldNames[newNumParams] != NULL ; ++newNumParams)
    ;

  if (newNumParams >= numMiniParams) {
    if (shortMiniParamNames == NULL)
      shortMiniParamNames = (char **)calloc(newNumParams+1, sizeof(char *)) ;
    else
      shortMiniParamNames = (char **)realloc(shortMiniParamNames, 
					     (newNumParams+1)*sizeof(char *)) ;
    if (shortMiniParamNames == NULL)
      numMiniParams = 0 ;
    else
      numMiniParams = newNumParams ;
  }
  
  for (idx = 0 ; idx < numMiniParams ; ++idx) {
    char	*string ;
    string = strrchr(fieldNames[idx], '.') ;
    if (string == NULL)
      shortMiniParamNames[idx] = fieldNames[idx] ;
    else
      shortMiniParamNames[idx] = string + 1 ;
  }
  shortMiniParamNames[idx] = NULL ;

  return shortMiniParamNames ;
}
/**********************************************************************/
char	*minimizeParamLongName(shortName)
  char	*shortName ;
{
  static char	paramName[BUFSIZ] ;
#ifdef OLD
  char		**longNames = IListMinimizeParams() ;
  int	idx ;

  for (idx = 0 ; 
       idx < numMiniParams && strcmp(shortName, shortMiniParamNames[idx]) ; 
       ++idx)
    ;

  if (idx == numMiniParams)
    return NULL ;
  
  if (strchr(longNames[idx], '.') == NULL)
    return NULL ;

  if (strncmp(longNames[idx], "minimizeRec.", strlen("minimizeRec.")) == 0)
    sprintf(paramName, "%s%s", "currentNet.mz.", 
	    longNames[idx] + strlen("minimizeRec.")) ;
  else
    sprintf(paramName, "") ;
#else
  static bound = FALSE ;
  if (bound == FALSE) {
    IBindStruct("currentMZ", "minimizeRec", (char *)&currentMZ, PP_OBJECT, "");
    bound = TRUE ;
  }
  sprintf(paramName, "%s%s", "currentMZ.", shortName) ;
#endif
  return paramName ;
}
/**********************************************************************/
  

/***********************************************************************
 *	Utils for getting the 'major' and 'minor' parameters from the net
 ***********************************************************************/
#define NET_FIELD(f)	(currentNet ? currentNet->f : (int)0)
/**********************************************************************/
int	netExists()		{ return (int)currentNet ; }
/**********************************************************************/
Boolean	getNetValue(which, where)
  int	which ;
  void	*where ;
{
  Boolean	found = TRUE ;

  if (where == NULL)
    return 0 ;

  switch(which) {
  case SELF:
    if (currentNet != NULL)
      *(Net *)where = currentNet ;
    else
      *(Net *)where = NULL ;
    break ;
  case ACTIVATION_WIDTH:
    *(int *)where = NET_FIELD(activityWidth) ;
    break ;
  case ACTIVATION_HEIGHT:
    *(int *)where = NET_FIELD(activityHeight) ;
    break ;
  case ACTIVATION_MARGIN:
    *(int *)where = NET_FIELD(activityMargin) ;
    break ;
  case MINOR_WIDTH:
    *(int *)where = NET_FIELD(minorWidth) ;
    break ;
  case MINOR_HEIGHT:
    *(int *)where = NET_FIELD(minorHeight) ;
    break ;
  case MINOR_MARGIN:
    *(int *)where = NET_FIELD(minorMargin) ;
    break ;
  case MAJOR_WIDTH:
    *(int *)where = NET_FIELD(majorWidth) ;
    break ;
  case MAJOR_HEIGHT:
    *(int *)where = NET_FIELD(majorHeight) ;
    break ;
  case MAJOR_MARGIN:
    *(int *)where = NET_FIELD(majorMargin) ;
    break ;
  case CURRENT_TIME:
    *(int *)where = NET_FIELD(currentTime) ;
    break ;
  case TIME_SLICES:
    *(int *)where = NET_FIELD(timeSlices) ;
    break ;
  case BATCH_SIZE:
    *(int *)where = NET_FIELD(batchSize) ;
    break ;
  case NAME:
    if (currentNet == NULL)
      *(char **)where = "Empty Net" ;
    else if (currentNet->name == NULL)
      *(char **)where = "No Name" ;
    else
      *(char **)where = currentNet->name ;
    break ;
  default:
    *(int *)where = 0 ;
    found = FALSE ;
    break ;
  }
  return found ;
}
/**********************************************************************/
Boolean	setNetValue(which, where)
  int	which ;
  void	*where ;
{
  Boolean	found = TRUE ;

  if (where == NULL)
    return FALSE ;
  if (currentNet == NULL)
    return FALSE ;

  switch(which) {
  case ACTIVATION_WIDTH:
    currentNet->activityWidth = *(int *)where ;
    break ;
  case ACTIVATION_HEIGHT:
    currentNet->activityHeight = *(int *)where ;
    break ;
  case ACTIVATION_MARGIN:
    currentNet->activityMargin = *(int *)where ;
    break ;
  case MINOR_WIDTH:
    currentNet->minorWidth = *(int *)where ;
    break ;
  case MINOR_HEIGHT:
    currentNet->minorHeight = *(int *)where ;
    break ;
  case MINOR_MARGIN:
    currentNet->minorMargin = *(int *)where ;
    break ;
  case MAJOR_WIDTH:
    currentNet->majorWidth = *(int *)where ;
    break ;
  case MAJOR_HEIGHT:
    currentNet->majorHeight = *(int *)where ;
    break ;
  case MAJOR_MARGIN:
    currentNet->majorMargin = *(int *)where ;
    break ;
  case CURRENT_TIME:
    currentNet->currentTime = *(int *)where ;
    break ;
  case TIME_SLICES:
    currentNet->timeSlices = *(int *)where ;
    break ;
  case BATCH_SIZE:
    currentNet->batchSize = *(int *)where ;
    break ;
  case NAME:
    if (currentNet != NULL)
      currentNet->name = *(char **)where ;
    break ;
  default:
    found = FALSE ;
    break ;
  }
  return found ;
}
/**********************************************************************/
int	getNumberOfUnits () {
  int	numUnits ;

  numUnits = 0 ;
  if (currentNet != NULL)
    netForAllGroups(currentNet, ALL, countUnits, &numUnits) ;

  return numUnits ;
}
/**********************************************************************/
Minimize	currentMinimizeRecord()
{
  if (getMinimizeRecord)
    currentMZ = getMinimizeRecord() ;
  else
    currentMZ = NULL ;
  return currentMZ ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		listExamples
 *	Description:	makes a list of the current example names
 *	Parameters:	
 *		int	index - an integer saying which example set to use
 *	Return Value:	
 *		char	**listExamples - a STATIC list of the example
 *				names, OVERWRITTEN AT THE NEXT CALL.
 ***********************************************************************/
static ExampleSet	exampleSetFromIdx(setIdx)
  int	setIdx ;
{
  Mask	mask ;

  if (setIdx == 0)
    mask = TRAINING ;
  else if (setIdx == 1)
    mask = TESTING ;
  else if (setIdx == 2)
    mask = VALIDATION ;
  else
    mask = TRAINING ;
  
  return currentNet ? netGetExampleSet(currentNet, mask) : NULL ;
}
/**********************************************************************/
char	**listExamples(setIdx)
  int	setIdx ;
{
  ExampleSet	exampleSet  = exampleSetFromIdx(setIdx) ;
  int		numExamples = exampleSet ? exampleSet->numExamples : 0 ;
  int		idx ;
  char		buffer[32] ;
  
  static int	maxExamples = 0 ;
  static char	**list = NULL;

  if (numExamples + 1 > maxExamples) {
    if (extendList((void *)&list, 
		   sizeof(char *), maxExamples, numExamples + 1) == 0)
      return NULL ;
    maxExamples = numExamples + 1 ;
  }

  for (idx = 0 ; idx < numExamples ; ++idx) {
    if (exampleSet && exampleSet->example && exampleSet->example[idx].name)
      sprintf(buffer, "%s", exampleSet->example[idx].name) ;
    else
      sprintf(buffer, "Example %d", idx) ;
    if (list[idx] != NULL)
      free(list[idx]) ;

    list[idx] = strdup(buffer) ;
  }

  if (list[idx] != NULL)
    free(list[idx]) ;
  list[idx] = NULL ;

  return list ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		doExample
 *	Description:	calls the current networks updateNetActivities
 *			procedure for a given example
 *	Parameters:	
 *		int	setIndex - the index of the example set to use
 *		int	exampleIdx - the index of the example in the set
 *	Return Value:	1
 ***********************************************************************/
int	doExample(setIdx, exampleIdx)
  int	setIdx ;
  int	exampleIdx ;
{
  ExampleSet	set = exampleSetFromIdx(setIdx) ;

  if (set)
    exampleIdx = MgetNumber(set, exampleIdx) ;

  if (exampleIdx < 0)
    return -1 ;

  updateNetActivities() ;

  return exampleIdx ;
}
/**********************************************************************/
int	deleteExample(setIdx, exampleIdx)
  int	setIdx ;
  int	exampleIdx ;
{
  ExampleSet	set = exampleSetFromIdx(setIdx) ;

  if (set)
    exampleSetDeleteExample(set, exampleIdx) ;

  return 0 ;
}
/**********************************************************************/
int	moveExample(fromSetIdx, fromExampleIdx, toSetIdx, toExampleIdx)
  int	fromSetIdx ;
  int	fromExampleIdx ;
  int	toSetIdx ;
  int	toExampleIdx ;
{
  ExampleSet	fromSet = exampleSetFromIdx(fromSetIdx) ;
  ExampleSet	toSet   = exampleSetFromIdx(toSetIdx) ;

  if (fromSet && toSet)
    exampleSetMoveExample(fromSet, fromExampleIdx, toSet, toExampleIdx) ;

  return 0 ;
}
/**********************************************************************/
int	copyExample(fromSetIdx, fromExampleIdx, toSetIdx, toExampleIdx)
  int	fromSetIdx ;
  int	fromExampleIdx ;
  int	toSetIdx ;
  int	toExampleIdx ;
{
  ExampleSet	fromSet = exampleSetFromIdx(fromSetIdx) ;
  ExampleSet	toSet   = exampleSetFromIdx(toSetIdx) ;

  if (fromSet && toSet)
    exampleSetCopyExample(fromSet, fromExampleIdx, toSet, toExampleIdx) ;
 
  return 0 ;
}
/**********************************************************************/
int	getExampleIndex(setIdx)
  int	setIdx ;
{
  ExampleSet	set = exampleSetFromIdx(setIdx) ;

  return set ? set->currentIdx : -1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		updateNetActivities
 *	Description:	updates the network activities (using whatever
 *			state the network is currently in)
 *	Parameters:	
 *	Return Value:	1
 ***********************************************************************/
int	updateNetActivities()
{
  if (currentNet && currentNet->activityUpdateProc) {
    MupdateNetActivities(currentNet) ;
  }
  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		unitFields
 *	Description:	returns an array of all the names of the fields
 *			in a unit.
 *	Parameters:	NONE
 *	Return Value:	as above
 ***********************************************************************/
static String	*unitFields () {
  static char	*fields[] =  { 
  "activity",
  "totalInput",
  "target",
  "inputDeriv",
  "outputDeriv",
  NULL } ;

  return fields ;
}
/**********************************************************************/
static Real	unitValue(unit, fieldIdx, time)
  Unit		unit ;
  int		fieldIdx ;
  int		time ;
{
  Real	value ;

  switch (fieldIdx) {
  case 0:
    if (unit->net->type & RECURRENT)
      value = unit->outputHistory[time] ;
    else
      value = unit->output ;
    break ;
  case 1:
    value = unit->totalInput ;
    break ;
  case 2:
    if (unit->net->type & RECURRENT)
      value = unit->targetHistory[time] ;
    else
      value = unit->target ;
    break ;
  case 3:
    value = unit->inputDeriv ;
    break ;
  case 4:
    value = unit->outputDeriv ;
    break ;
  default:
    value = 0.0 ;
    break ;
  }
  return value ;
}
/**********************************************************************/
static void	setUnitValue(unit, fieldIdx, time, value)
  Unit		unit ;
  int		fieldIdx ;
  int		time ;
  double	value ;
{
  switch (fieldIdx) {
  case 0:
    if (unit->net->type & RECURRENT)
      unit->outputHistory[time] = value ;
    else
      unit->output = value ;
    break ;
  case 1:
    unit->totalInput = value ;
    break ;
  case 2:
    if (unit->net->type & RECURRENT)
      unit->targetHistory[time] = value ;
    else
      unit->target = value ;
    break ;
  case 3:
    unit->inputDeriv = value ;
    break ;
  case 4:
    unit->outputDeriv = value ;
    break ;
  default:
    break ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		getNetworkActivations
 *	Description:	returns an array of UnitDisplayRecs with all their
 *			values set properly for an activation display
 *	Parameters:	
 *		UnitDisplayRec - an existing pointer to UnitDisplayRec
 *				of size *maxPtr which may be reallocated.
 *				this will hold the array.
 *		int	*numPtr - ptr to the number of unitRecs (return)
 *		int	*maxPtr - ptr to the size of arrayPtr (may be
 *				changed)
 *	Return Value:	
 *		UnitDisplayRec - the modified array of units
 ***********************************************************************/
static int	staticFieldIdx ;
static int	unitIdx ;
static int	maxColumn ;
/**********************************************************************/
static void	getMaxColumn(unit, data)
  Unit		unit ;
  void		*data ;
{
  int	*max = (int *)data ;

  if (unit->activityColumn > *max)
    *max = unit->activityColumn ;
}
/**********************************************************************/
UnitDisplayRec		*getUnitValues(idx, arrayPtr, numPtr, maxPtr)
  int			idx ;
  UnitDisplayRec	*arrayPtr ;
  int			*numPtr ;
  int			*maxPtr ;
{
  int	numUnits ;
  numUnits = 0 ;
  netForAllGroups(currentNet, ALL, countUnits, &numUnits) ;

  if (currentNet->type & RECURRENT) {
    numUnits *= currentNet->timeSlices ;
    maxColumn = 0 ;
    netForAllUnits(currentNet, ALL, getMaxColumn, &maxColumn) ;
    maxColumn += 2 ;		/* add in a blank space */
  }

  if (numUnits > *maxPtr) {
    extendList((void *)&arrayPtr, sizeof(UnitDisplayRec), *maxPtr, numUnits) ;
    *maxPtr = numUnits ;
  }
  unitIdx  = 0 ;
  staticFieldIdx = idx ;
  netForAllUnits(currentNet, ALL, getUnitActivation, arrayPtr) ;
  *numPtr = unitIdx ;

  return arrayPtr ;
}
/**********************************************************************/
static void	getUnitActivation(unit, data)
  Unit		unit ;
  void		*data ;
{
  UnitDisplayRec	*displayRec = ((UnitDisplayRec *)data) + unitIdx ;
  
  if (unit->activityRow < 0 || unit->activityColumn < 0)
    return ;

  if (unit->net->type & RECURRENT) {
    int	idx ;
    for (idx = 0 ; idx < unit->timeSlices ; ++idx) {
      displayRec[idx].row    = unit->activityRow ;
      displayRec[idx].column = unit->activityColumn + idx * maxColumn ;
      displayRec[idx].value  = unitValueProc(unit, staticFieldIdx, idx) ;
      displayRec[idx].userData = unit->name ;
    }
    unitIdx += idx ;
  } else {
    displayRec[0].row      = unit->activityRow ;
    displayRec[0].column   = unit->activityColumn ;
    displayRec[0].value    = unitValueProc(unit, staticFieldIdx, 0) ;
    displayRec[0].userData = unit->name ;
    ++unitIdx ;
  }
}
/**********************************************************************/
void		changeUnitValue(unitDisplay, fieldIdx)
  UnitDisplayRec	*unitDisplay ;
  int			fieldIdx ;
{
  if (currentNet) {
    Unit unit = unitFromName(currentNet, (char *)unitDisplay->userData) ;
    if (currentNet->type & RECURRENT)
      fprintf(stderr, "Unable to change values in a recurrent net.\n") ;
    else
      setUnitValueProc(unit, fieldIdx, 0, unitDisplay->value) ;
  }
}
/**********************************************************************/
void		changeTarget(unitDisplay)
  UnitDisplayRec	*unitDisplay ;
{
  if (currentNet) {
    Unit unit = unitFromName(currentNet, (char *)unitDisplay->userData) ;
    if (unit)
      unit->target = unitDisplay->value ;
  }
}
/**********************************************************************/
void		moveUnit(unitDisplay)
  UnitDisplayRec	*unitDisplay ;
{
  if (currentNet) {
    Unit unit = unitFromName(currentNet, (char *)unitDisplay->userData) ;
    if (unit)
      unit->activityRow    = unitDisplay->row ;
      unit->activityColumn = unitDisplay->column ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		getNetworkTargets
 *	Description:	returns an array of UnitDisplayRecs with all their
 *			values set properly for an activation display
 *	Parameters:	
 *		UnitDisplayRec - an existing pointer to UnitDisplayRec
 *				of size *maxPtr which may be reallocated.
 *				this will hold the array.
 *		int	*numPtr - ptr to the number of unitRecs (return)
 *		int	*maxPtr - ptr to the size of arrayPtr (may be
 *				changed)
 *	Return Value:	
 *		UnitDisplayRec - the modified array of units
 ***********************************************************************/
#ifndef I_HUGE
#define I_HUGE	((int)1<<(sizeof(int)-1))
#endif
static int	minRow ;
/**********************************************************************/
UnitDisplayRec	*getNetworkTargets(arrayPtr, numPtr, maxPtr)
  UnitDisplayRec	*arrayPtr ;
  int			*numPtr ;
  int			*maxPtr ;
{
  int			numUnits ;

  numUnits = 0 ;
  netForAllGroups(currentNet, OUTPUT, countUnits, &numUnits) ;

  if (currentNet->type & RECURRENT) {
    numUnits *= currentNet->timeSlices ;
    maxColumn = 0 ;
    netForAllUnits(currentNet, ALL, getMaxColumn, &maxColumn) ;
    maxColumn += 2 ;		/* add in a blank space */
  }

  if (numUnits > *maxPtr) {
    extendList((void *)&arrayPtr, sizeof(UnitDisplayRec), *maxPtr, numUnits) ;
    *maxPtr = numUnits ;
  }
  unitIdx = 0 ;
  minRow = I_HUGE ;
  netForAllUnits(currentNet, OUTPUT, getUnitTarget, arrayPtr) ;
  *numPtr = unitIdx ;

  for (unitIdx = 0 ; unitIdx < numUnits ; ++unitIdx)
    arrayPtr[unitIdx].row -= minRow ;

  return arrayPtr ;
}
/**********************************************************************/
static void	getUnitTarget(unit, data)
  Unit		unit ;
  void		*data ;
{
  UnitDisplayRec	*displayRec = (UnitDisplayRec *)data + unitIdx ;
  
  if (unit->activityRow < 0 || unit->activityColumn < 0)
    return ;

  if (unit->net->type & RECURRENT) {
    int	idx ;
    for (idx = 0 ; idx < unit->timeSlices ; ++idx) {
      displayRec[idx].row    = unit->activityRow ;
      displayRec[idx].column = unit->activityColumn + idx * maxColumn ;
      displayRec[idx].value  = unit->targetHistory[idx] ;
      displayRec[idx].userData = unit->name ;
    }
    unitIdx += idx ;
  } else {
    displayRec[0].row      = unit->activityRow ;
    displayRec[0].column   = unit->activityColumn ;
    displayRec[0].value	   = unit->target ;
    displayRec[0].userData = unit->name ;
    ++unitIdx ;
  }
  minRow = MIN(minRow, MAX(0, displayRec[0].row)) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		displayForAllUnits
 *	Description:	  !!!HACK ALERT!!!  
 *			DO NOT USE IN NESTED CALLS
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static int	currentUnitIdx ;
static Unit	currentUnit ;
static void	(*displayProc) ARGS((int, void *)) ;
static void	callDisplayProc ARGS((Unit	unit, void	*data)) ;
/**********************************************************************/
void	displayForAllUnits (proc, data)
  void	(*proc)() ;
  void	*data ;
{
  currentUnitIdx = -1 ;
  currentUnit    = NULL ;
  displayProc    = proc ;
  netForAllUnits(currentNet, ALL, callDisplayProc, data) ;
  currentUnit    = NULL ;
}
/**********************************************************************/
static void	callDisplayProc (unit, data)
  Unit		unit ;
  void		*data ;
{
  ++currentUnitIdx ;
  currentUnit = unit ;
  (*displayProc) (currentUnitIdx, data) ;
}
/**********************************************************************/
int	getUnitMajorRow() {
  if (currentUnit == NULL)
    return -1 ;
  else
    return currentUnit->weightRow ;
}
/**********************************************************************/
int	getUnitMajorColumn() {
  if (currentUnit == NULL)
    return -1 ;
  else
    return currentUnit->weightColumn ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		listExampleSets
 *	Description:	returns an array of all the names of the example
 *			sets in the current network
 *	Parameters:	NONE
 *	Return Value:	as above
 ***********************************************************************/
static char	*exampleSets[] =  { 
  "trainingSet",
  "testingSet",
  "validationSet",
  NULL } ;

char	**listExampleSets () {
  return exampleSets ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		linkFields
 *	Description:	returns an array of all the names of the fields
 *			in a link.
 *	Parameters:	NONE
 *	Return Value:	as above
 ***********************************************************************/
static String	*linkFields () {
  static char	*fields[] =  { 
  "weight",
  "deltaWeight",
  "deriv",
  NULL } ;

  return fields ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		linkValue
 *	Description:	returns the value of the requested field in a link
 *	Parameters:
 *	  Link		link - the link
 *	  int		fieldIdx - the index of the field (from "linkFields")
 *	Return Value:
 *	  static Real	linkValue - the value
 *********************************************************************/
static Real	linkValue(link, fieldIdx)
  Link		link ;
  int		fieldIdx ;
{
  Real	value ;

  switch (fieldIdx) {
  case 0:
    value = link->weight ;
    break ;
  case 1:
    value = link->deltaWeight ;
    break ;
  case 2:
    value = link->deriv ;
    break ;
  default:
    value = 0 ;
    break ;
  }
  return value ;
}
/********************************************************************/
static void	setLinkValue(link, fieldIdx, value)
  Link		link ;
  int		fieldIdx ;
  double	value ;
{
  switch (fieldIdx) {
  case 0:
    linkSetWeight(link, value) ;
    break ;
  case 1:
    link->deltaWeight = value ;
    break ;
  case 2:
    link->deriv = value ;
    break ;
  default:
    break ;
  }
}
/********************************************************************/




/***********************************************************************
 *	Name:		getFanIn
 *	Description:	gets the fan in weights to a given unit and returns
 *			them in an array of UnitDisplayRecs.
 *	Parameters:    
 *		UnitDisplayRec	**arrayPtr - address of the current array
 *					of UnitRecs.
 *		int	*numPtr  - address to return number of \
 *					UnitDisplayRecs in
 *	Return Value:	
 *		UnitDisplayRec *getFanIn - the new array
 ***********************************************************************/
UnitDisplayRec		*getFanIn(fieldIdx, arrayPtr, numPtr)
  int			fieldIdx ;
  UnitDisplayRec	**arrayPtr ;
  int			*numPtr ;
{
  static UnitDisplayRec	*linkArray = NULL ;
  static int		maxLinks = 0 ;
  
  int		idx, numLinks, linkIdx ;
  Unit		unit ;

  if (currentUnit == NULL)
    return NULL ;

  unit = currentUnit ;

  if (currentNet->linkLayoutMode == SHOW_ALL)
    numLinks = unit->numIncoming + unit->numOutgoing ;
  else if (currentNet->linkLayoutMode == SHOW_OUTGOING)
    numLinks = unit->numOutgoing ;
  else
    numLinks = unit->numIncoming ;

  if (numLinks > maxLinks) {
    extendList((void *)&linkArray, sizeof(UnitDisplayRec), maxLinks, numLinks);
    maxLinks = numLinks ;
  }

  idx = 0 ;
  if (currentNet->linkLayoutMode == SHOW_INCOMING 
      || currentNet->linkLayoutMode == SHOW_ALL) {
    for (linkIdx  = 0 ; linkIdx < unit->numIncoming ; ++idx, ++linkIdx) {
      Link	link = unit->incomingLink[linkIdx] ;

      linkArray[idx].row    = link->weightRow[1] ;
      linkArray[idx].column = link->weightColumn[1] ;

      if (linkArray[idx].row < 0 || linkArray[idx].column < 0) {
	--idx ;
	continue ;
      }

      linkArray[idx].value    = linkValueProc(link, fieldIdx) ;
      linkArray[idx].userData = link->name ;
    }
  }

  if (currentNet->linkLayoutMode == SHOW_OUTGOING 
      || currentNet->linkLayoutMode == SHOW_ALL){
    for (linkIdx = 0 ; linkIdx < unit->numOutgoing ; ++idx, ++linkIdx) {
      Link	link = unit->outgoingLink[linkIdx] ;

      linkArray[idx].row    = link->weightRow[0] ;
      linkArray[idx].column = link->weightColumn[0] ;

      if (linkArray[idx].row < 0 || linkArray[idx].column < 0) {
	--idx ;
	continue ;
      }

      linkArray[idx].value    = linkValueProc(link, fieldIdx) ;
      linkArray[idx].userData = link->name ;
    }
  }
 
  *arrayPtr = linkArray ;
  *numPtr   = idx ;

  return linkArray ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		getWeightParams
 *	Description:	calculates the mean, variance and scale for
 *			all the weights in the net.
 *	Parameters:	
 *		double	*meanPtr  - a pointer to storage for the mean
 *		double	*varPtr   - a pointer to storage for the variance
 *		double	*scalePtr - a pointer to storage for the scale
 *	Return Value:
 *		1
 ***********************************************************************/
static	Real	s, ss, maxAbs ;
static	int	numLinks ;
/***********************************************************************/
static void	updateParams(unit, data)
  Unit		unit ;
  void		*data ;
{
  int	fieldIdx = (int)data ;
  int	idx ;
  Real	value ;

  for (idx = 0 ; idx < unit->numIncoming ; ++idx) {
    value  = linkValueProc(unit->incomingLink[idx], fieldIdx) ;
    s     += value ;
    ss    += value*value ;
    maxAbs = MAX(maxAbs, fabs(value)) ;
    numLinks++;
  }
}
/***********************************************************************/
int		getWeightParams(fieldIdx, meanPtr, varPtr, scalePtr)
  int		fieldIdx ;
  double	*meanPtr ;
  double	*varPtr ;
  double	*scalePtr ;
 {
  Real	mean, var, scale ;

  numLinks = 0 ;
  s = ss = maxAbs = 0.0 ;
  netForAllUnits(currentNet, ALL, updateParams, (void *)fieldIdx) ;

  mean  = s/numLinks;
  var   = (ss  - s*s/numLinks)/(numLinks - 1);
  scale = MAX(EPSILON, maxAbs) ;

  *meanPtr  = mean ;
  *varPtr   = var ;
  *scalePtr = scale ;
  return 1 ;
}
/**********************************************************************/
static void	countUnits(group, data)
  Group		group ;
  void		*data ;
{
  *((int *)data) += group->numUnits ;
}
/**********************************************************************/
void		changeLinkField(unitDisplay, fieldIdx)
  UnitDisplayRec *unitDisplay ;
  int		 fieldIdx ;
{
  if (currentNet) {
    Link link = linkFromName(currentNet, (char *)unitDisplay->userData) ;
    if (link)
      setLinkValueProc(link, fieldIdx, unitDisplay->value) ;
  }
}
/***********************************************************************/


/*********************************************************************
 *	Name:		setNetTime/incrementNetTime
 *	Description:	routines for changing the time slice of the
 *			current net.
 *	Parameters:
 *	  int	time - the time to set it to
 *	Return Value:
 *	  int	setNetTime/incrementNetTime - the new time
 *********************************************************************/
int	setNetTime(time)
  int	time ;
{
  if (currentNet) {
    currentNet->currentTime = MIN(time, currentNet->timeSlices-1) ;
    return currentNet->currentTime ;
  } else
    return 0 ;
}
/********************************************************************/
int	incrementNetTime() 
{
  if (currentNet) {
    currentNet->currentTime 
      = (currentNet->currentTime + 1) % currentNet->timeSlices ;
    return currentNet->currentTime ;
  } else
    return 0 ;
}
/********************************************************************/

void	displayFiddleLayout(fiddleWidth, fiddleHeight)
  int	fiddleWidth ;
  int	fiddleHeight ;
{
  if (currentNet)
    touchUpMajorDimensions(currentNet, fiddleWidth, fiddleHeight) ;
}
