
/**********************************************************************
 * $Id: display2Itf.c,v 1.5 92/11/30 11:28:36 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <xerion/config.h>

#include <stdio.h>
#include <signal.h>

#include <xerion/useful.h>
#include <xerion/commands.h>

#include "display2Itf.h"

extern int	buildGraphics() ;
extern int	destroyGraphics() ;


extern int	popupShell	  ARGS((String name, String geometry)) ;
extern void	*createGraph	  ARGS((String name)) ;
extern void	*createPopupGraph ARGS((String name, String geometry)) ;

#define	TI_name(p)	(char *)((TBL_ITEM *)(p)->name)
extern TRACE	*first_trace ;

static int	compare ARGS((const void *s1, const void  *s2)) ;
static int	catchInterrupt ARGS((int	sig)) ;

/***********************************************************************
 *
 *	COMMANDS
 *
 *	NOTE: we are very careful about setting the signal handler
 *		around the graphics calls in the commands. X is NOT
 *		re-entrant, so you cannot do a long jump inside it.
 *		We, instead, save up the interrupts, then do a long
 *		jump once all of the X code is done.
 *
 ***********************************************************************/

/*******************************************************************
 *	Name:		popup
 *	Description:	pop up one of the displays
 *	Parameters:
 *	  int	tokc - As usual
 *	  char	*tokv -
 *	Return Value:
 *	  int	command_popup - as usual
 *******************************************************************/
int	command_popup(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  char		*name, *geometry ;
  int		returnStatus ;

  IUsage("[geometry] <display>");
  if (GiveHelp(tokc)) {
    ISynopsis("popup one of the graphical displays.");
    IHelp
      (IHelpArgs,
       "This command pops up the specifed display. Valid displays are:",
       "  activationDisplay",
       "  weightDisplay",
       "  miniDisplay		(Learning Methods)",
       "  variableDisplay	(System Variables)",
       "",
       "It is most useful when saved in a file and  read in to configure the",
       "display in the way you want it.",
       "",
       "SEE ALSO",
       "createGraph, addTrace",
       NULL);
    return 0;
  }

  if (tokc > 3 || tokc < 2)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  setSignalHandler() ;

  ++tokv, --tokc ;
  if (tokc == 2) {
    geometry = *tokv ;
    ++tokv, --tokc ;
  } else {
    geometry = NULL ;
  }
  name = *tokv ;

  returnStatus = popupShell(name, geometry) ;

  if (resetSignalHandler())
    return 0 ;

  if (returnStatus == 0)
    warn("popup", "Cannot pop up display \"%s\"", *tokv) ;

  return 1 ;
}
/******************************************************************/

int	command_createGraph(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  char		*name, *geometry ;
  int		returnStatus ;

  IUsage("[geometry] <object>");
  if (GiveHelp(tokc)) {
    ISynopsis("create and pop up a graph for <object>.");
    IHelp
      (IHelpArgs,
       "This command creates a graph for an object and pops it up.",
       "",
       "It is most useful when saved in a file and  read in to configure the",
       "display in the way you want it.",
       "",
       "SEE ALSO",
       "popup, addTrace",
       NULL);
    return 0;
  }

  if (tokc > 3)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  setSignalHandler() ;

  ++tokv, --tokc ;
  if (tokc == 2) {
    geometry = *tokv ;
    ++tokv, --tokc ;
  } else {
    geometry = NULL ;
  }
  name = *tokv ;

  if (returnStatus)
    returnStatus = (int)createPopupGraph(name, geometry) ;

  if (resetSignalHandler())
    return 0 ;

  if (returnStatus == 0)
    warn("createGraph", "Cannot create graph for object \"%s\"", *tokv) ;

  return 1 ;
}


/*********************************************************************
 *	Name:		createMixtureDisplay
 *	Description:	creates a mixture display for a given name
 *	Parameters:
 *	  int	tokc - 
 *	  char	*tokv[] -
 *	Return Value:
 *	  int	command_createMixtureDisplay - 
 *********************************************************************/
int	command_createMixtureDisplay(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  char		*name, *geometry ;
  int		returnStatus ;

  IUsage("[geometry] <name>");
  if (GiveHelp(tokc)) {
    ISynopsis("create and pop up a mixture display for mixture <name>.");
    IHelp
      (IHelpArgs,
       "This command creates a mixture display for a  given mixture and pops",
       "it up.",
       "",
       "It is most useful when saved in a file and  read in to configure the",
       "display in the way you want it.",
       "",
       "SEE ALSO",
       "popup, createGraph, triggerCallbacks, addTrace",
       NULL);
    return 0;
  }

  if (tokc > 3)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  setSignalHandler() ;

  ++tokv, --tokc ;
  if (tokc == 2) {
    geometry = *tokv ;
    ++tokv, --tokc ;
  } else {
    geometry = NULL ;
  }
  name = *tokv ;

  returnStatus = (int)createMixtureDisplay(name, geometry) ;

  if (resetSignalHandler())
    return 0 ;

  if (returnStatus == 0)
    warn("createMixtureDisplay", 
	 "Cannot create display for mixture \"%s\"", *tokv) ;

  return 1 ;
}
/********************************************************************/


/*******************************************************************
 *	Name:		command_buildDisplays
 *	Description:	builds the graphical display
 *	Parameters:
 *	  int	tokc    - the number of command line args
 *	  char	*tokv[] - the command line args
 *	Return Value:
 *	  int	command_buildDisplay - 
 *******************************************************************/
int	command_buildDisplays(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  IUsage(" ");
  if (GiveHelp(tokc)) {
    ISynopsis("Builds the graphical display.");
    IHelp
      (IHelpArgs,
       "This command builds the  graphical displays used  to display network",
       "activations, connections, and error plots.  If  you are not  running",
       "under X windows, this command will not work.",
       "",
       "See also destroyDisplays.",
       NULL);
    return 1;
  }

  if (buildGraphics() == 0)
    IErrorAbort("Unable to build displays") ;

  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		command_destroyDisplays
 *	Description:	destroys the graphical display
 *	Parameters:
 *	  int	tokc    - the number of command line args
 *	  char	*tokv[] - the command line args
 *	Return Value:
 *	  int	command_destroyDisplay - 
 *******************************************************************/
int	command_destroyDisplays(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  IUsage(" ");
  if (GiveHelp(tokc)) {
    ISynopsis("Destroys the graphical display.");
    IHelp
      (IHelpArgs,
       "This command destroys the graphical displays used to display network",
       "activations, connections, and error plots.   If you  are not running",
       "under X windows, this command will not work.",
#ifdef LATER
       "",
       "See also buildDisplays.",
       "",
#endif
       NULL);
    return 1;
  }

  if (destroyGraphics() == 0)
    IErrorAbort("Unable to destroy displays") ;

  while(IDoSingleCommand(din, dout) != 0)
    ;

  return 0 ;
}
/******************************************************************/


/*********************************************************************
 *	Name:		(re)setSignalHandler
 *	Description:	signal handler for the interrupt signal.
 *			setSignalHandler resets the current number of
 *			interrupts detected, then sets the signal handler
 *			to a routine that just keeps track of the number
 *			of signals detected.
 *			resetSignalHandler restores the old handler, 
 *			then checks for any detected interrups, and if
 *			they occured it does a lonjump (ICatchInterrupt)
 *	Parameters:	NONE
 *********************************************************************/
static int	numInterrupts = 0 ;
static int	catchInterrupt(sig)
  int		sig ;
{
  char	*prompt = IGetValue("prompt", NULL) ;

#ifdef SYSV_SIGNALS
  (void) signal(SIGINT, catchInterrupt);
#endif
  
  ++numInterrupts ;
  fprintf(stderr, "\n%s", prompt ? prompt : "") ;
  return -1 ;
}
/********************************************************************/
static SignalHandler	oldSignalHandler = (SignalHandler)catchInterrupt ;
/********************************************************************/
int	setSignalHandler()
{
  numInterrupts = 0 ;
  oldSignalHandler = (SignalHandler)signal(SIGINT, catchInterrupt);

  return numInterrupts ;
}
/********************************************************************/
int	resetSignalHandler()
{
  if (oldSignalHandler != (SignalHandler)catchInterrupt) {
    oldSignalHandler = (SignalHandler)signal(SIGINT, oldSignalHandler);
  }

  if (numInterrupts && !itf_errno) {
    itf_errno     = INTERRUPT;
    caught_signal = SIGINT;
  }

  return numInterrupts ;
}
/********************************************************************/


/***********************************************************************
 *
 *	TRACE ROUTINES
 *
 ***********************************************************************/


/***********************************************************************
 *	Name:		listTraces
 *	Description:	returns a list of all the trace names
 *	Parameters:	NONE
 *	Return Value:	
 *			a NULL terminated, static array of strings.
 ***********************************************************************/
static int	compare(s1, s2)
  const void	*s1 ;
  const void	*s2 ;
{
  return strcmp(*(char **)s1, *(char **)s2) ;
}
/**********************************************************************/
char	**listTraces() {
  static char	**traceName = NULL ;

  TRACE		*trace;

  int		numTraces   = 0 ;
  int		idx ;

  if (traceName != NULL)
    free(traceName) ;

  for (trace = first_trace ; trace != NULL ; trace = trace->next)
    ++numTraces ;

  traceName = (char **) calloc((numTraces + 1), sizeof(*traceName)) ;
  if (traceName == NULL)
    return NULL ;

  idx = 0 ;
  for (trace = first_trace ; trace != NULL ; trace = trace->next)
    traceName[idx++] = trace->name ;

  qsort((void *)traceName, numTraces, sizeof(*traceName), compare) ;
  return traceName ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		traceOn
 *	Description:	Boolean test to see whether a command is already
 *			on a trace
 *	Parameters:	
 *		char	*traceName   - the name of the trace to add to
 *		char	*commandName - the name of the command to add
 *			to the trace.
 *	Return Value:	
 *		Boolean	traceOn - TRUE if the command is on the trace,
 *				  FALSE if not, or on any error.
 ***********************************************************************/
Boolean	traceOn(traceName, commandName)
  char	*traceName ;
  char	*commandName ;
{
  TRACE		*trace;
  int		idx ;

  /* find the trace */
  for (trace = first_trace ; 
       trace != NULL && strcmp(traceName, trace->name) != 0 ;
       trace = trace->next)
    ;

  if (trace == NULL || commandName == NULL)
    return FALSE ;

  /* If it's already there, return TRUE */
  for (idx = 0; idx < trace->n_traces ; ++idx) {
    if (strcmp(trace->trace[idx].command, commandName) == 0)
      return TRUE ;
  }

  return FALSE ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		addTrace
 *	Description:	adds a command to a trace
 *	Parameters:	
 *		char	*traceName   - the name of the trace to add to
 *		char	*commandName - the name of the command to add
 *			to the trace.
 *	Return Value:	NONE
 ***********************************************************************/
void	addTrace(traceName, commandName)
  char	*traceName ;
  char	*commandName ;
{
  TRACE		*trace;
  int		idx ;

  for (trace = first_trace ; 
       trace != NULL && strcmp(traceName, trace->name) != 0 ;
       trace = trace->next)
    ;

  if (trace == NULL || commandName == NULL)
    return ;

  /* If it's already there, return */
  for (idx = 0; idx < trace->n_traces ; ++idx) {
    if (strcmp(trace->trace[idx].command, commandName) == 0)
      return ;
  }

  /* The following is copied from commandTrace (with a few rephrases)
   * in itf/trace.c */
  /* add the new trace */
  idx = trace->n_traces;
  if (trace->n_traces > 0)
    trace->trace = (struct TRACE_RECORD *)
      realloc(trace->trace,
	      (unsigned)(sizeof(struct TRACE_RECORD)*(trace->n_traces+1))) ;
  else
    trace->trace = (struct TRACE_RECORD *)
      calloc(sizeof(struct TRACE_RECORD), 1);

  if (trace->trace == NULL) {
    IOutOfMemory("addTrace", "allocating trace record");
    return;
  }

  trace->n_traces++;
  trace->trace[idx].command = strdup(commandName) ;
  if (trace->trace[idx].command == NULL) {
    IOutOfMemory("addTrace", "allocating command string");
    return;
  }
  trace->trace[idx].active = 1;
  trace->trace[idx].freq = 0;
  trace->trace[idx].count = 0;
  trace->trace[idx].stream_not_open = 0;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		deleteTrace
 *	Description:	deletes a command from a trace
 *		char	*traceName   - the name of the trace to add to
 *		char	*commandName - the name of the command to add
 *			to the trace.
 *	Return Value:	NONE
 ***********************************************************************/
void	deleteTrace(traceName, commandName)
  char	*traceName ;
  char	*commandName ;
{
  TRACE		*trace ;
  int		idx ;

  for (trace = first_trace ; 
       trace != NULL && strcmp(traceName, trace->name) != 0 ;
       trace = trace->next)
    ;

  if (trace == NULL || commandName == NULL)
    return ;

  /* find the command and free its memory */
  for (idx = 0 ; idx < trace->n_traces ; ++idx) {
    if (strcmp(trace->trace[idx].command, commandName) == 0) {
      free(trace->trace[idx].command);
      break ;
    }
  }
  /* if we found the command, move all the others forward */
  if (idx < trace->n_traces) {
    for ( ; idx < trace->n_traces ; ++idx)
      trace->trace[idx] = trace->trace[idx + 1] ;
    --trace->n_traces ;
    trace->trace = (struct TRACE_RECORD *)
      realloc(trace->trace,
	      (unsigned)(sizeof(struct TRACE_RECORD)*(trace->n_traces+1))) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *
 *	OBJECT LIST ROUTINES
 *
 ***********************************************************************/


/**********************************************************************
 *	Name:		extendList
 *	Description:	increases the size of a malloced array of 
 *			items of a given size.
 *	Parameters:
 *		void	**list - a  pointer to the start of the array
 *		int	size   - the size of a single item in the list
 *		int	oldMax - the old size of the array
 *		int	newMax - the desired size of the array
 *	Return Value:
 *		int	extendList - 0 on failure, 1 on success
 **********************************************************************/
int 	extendList(list, size, oldMax, newMax)
  void	**list ;
  int	size ;
  int	oldMax ;
  int	newMax ;
 {
  char	*tmpPtr ;

  if (*list == NULL)
    tmpPtr = (char *)malloc(newMax*size) ;
  else
    tmpPtr = (char *)realloc((*list),  newMax*size) ;
  if (tmpPtr == NULL) {
    return 0 ;
  } else {
    memset((tmpPtr + oldMax*size), 0, (newMax - oldMax)*size) ;
  }
  *list = (void *)tmpPtr ;

  return 1 ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		setAddressEnv
 *	Description:	sets the address environment so that you
 *			can evaluate object types and values
 *	Parameters:
 *		char		*name - the name of the object to be 
 *				evaluated
 *	Return Value:
 *		ADDRESS_ENV	*setAddressEnv - the current address 
 *				environment
 **********************************************************************/
static ADDRESS_ENV	*setAddressEnv(name)
  char			*name ;
{
  static ADDRESS_ENV	addressEnv ;

  int		takeAddress ;
  char 		*itfName = itf_name_buffer;

  MASSAGE_VAR_NAME(name, itfName);

  if (!IParseObject(itfName, &addressEnv, &takeAddress))
    return 0 ;

  IMoveToPointer(&addressEnv) ;

  return &addressEnv ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		isObject
 *	Description:	boolean test to see if the object corresponding
 *			to a given name is a simple object
 *	Parameters:
 *		char	*name - the name of the object to test
 *	Return Value:
 *		int	isSimpleObject - 0 if not simple, nonzero if it is
 **********************************************************************/
static int	isObject(name)
  char		*name ;
{
  return 1 ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		isSimpleObject
 *	Description:	boolean test to see if the object corresponding
 *			to a given name is a simple object
 *	Parameters:
 *		char	*name - the name of the object to test
 *	Return Value:
 *		int	isSimpleObject - 0 if not simple, nonzero if it is
 **********************************************************************/
static int	isSimpleObject(name)
  char		*name ;
{
  ADDRESS_ENV	*addressEnvPtr ;

  addressEnvPtr = setAddressEnv(name) ;

  return IEnvIsSimple(addressEnvPtr) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		listObjectNames
 *	Description:	returns a pointer to a list of object
 *			names obtained from the simulator. All objects
 *			must pass a test function passed in.
 *			the test function must be of form
 *				test(char *name)
 *			and return 1 if passed, 0 if failed.
 *	Parameters:	
 *		int	(*test)() - the test function to see if the name
 *			should be included.
 *		char	*(*listPtr[]) - pointer to the list
 *		int	*maxSizePtr - pointer to the maximum list size
 *	Return Value:
 *		char	*listObjectNames[] - an array of strings.
 *				Each string is an object name.
 *				The list is terminated by a NULL pointer.
 *			NOTE - the list may be made larger and *maxSizePtr
 *				updated to reflect this new size.
 **********************************************************************/
static char	**listObjectNames(test, listPtr, maxSizePtr)
  int		(*test)() ;
  char		*(*listPtr[]) ;
  int		*maxSizePtr ;
{
  int		idx ;
  int		id = 0 ;
  TBL_ITEM	*item ;

  for(idx = 0 ; 
      item = ITableNextByContext(itf_table, &id, itf_obj_context) ;
      ++idx) {

    if (!(*test)(TI_name(item))) {
      --idx ;
      continue ;
    }

    if (idx >= *maxSizePtr) {
      int	newMaxObjects = 1.5*(*maxSizePtr + 1) ;

      if (extendList((void *)listPtr, 
		     sizeof(char *), *maxSizePtr, newMaxObjects) == 0)
	return NULL ;
      *maxSizePtr = newMaxObjects ;
    }

    if ((*listPtr)[idx] != NULL)
      free((*listPtr)[idx]) ;

    (*listPtr)[idx] = strdup(TI_name(item)) ;
  }

  qsort((void *)*listPtr, idx, sizeof(char *), compare) ;

  if (idx >= *maxSizePtr) {
    int	newMaxObjects = *maxSizePtr + 1 ;

    if (extendList((void *)listPtr, 
		   sizeof(char *), *maxSizePtr, newMaxObjects) == 0)
      return NULL ;
    *maxSizePtr = newMaxObjects ;
  }
  (*listPtr)[idx] = NULL ;

  return *listPtr ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		listAllObjectNames
 *	Description:	returns a pointer to a STATIC list of object
 *			names obtained from the simulator
 *	Parameters:	
 *		NONE
 *	Return Value:
 *		char	*listAllObjectnames[] - a STATIC array of strings.
 *				Each string is an object name.
 *				The list is terminated by a NULL pointer.
 *				THE LIST IS OVERWRITTEN AT EACH CALL
 **********************************************************************/
char	**listAllObjectNames()
{
  static char	**name ;
  static int	maxObjects = 0 ;

  return listObjectNames(isObject, &name, &maxObjects) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		listSimpleObjectNames
 *	Description:	returns a pointer to a STATIC list of object
 *			names obtained from the simulator. All objects
 *			are simple objects. i.e. strings, integers, floats;
 *			NOT arrays or struct.
 *	Parameters:	
 *		NONE
 *	Return Value:
 *		char	*listSimpleObjectnames[] - a STATIC array of strings.
 *				Each string is an object name.
 *				The list is terminated by a NULL pointer.
 **********************************************************************/
char	**listSimpleObjectNames()
{
  static char	**name ;
  static int	maxObjects = 0 ;

  return listObjectNames(isSimpleObject, &name, &maxObjects) ;
}
/**********************************************************************/


/**********************************************************************
 *    	Name:		getSimpleObjectValue
 *	Description:	returns a pointer to astring representing the
 *			current value of a simple object (string, int, 
 *			float)
 *	Parameters:
 *		char	*name - the name of the object
 *		char	*name - the format to return it in
 *				eg. %.8f or %s, use NULL if you don't
 *				know or care
 *	Return Value:
 *		char	*getSimpleObjectValue - a STATIC string of the
 *				object's value. Overwritten at each call.
 **********************************************************************/
char	*getSimpleObjectValue(name, format)
  char	*name ;
  char	*format ;
{
  ADDRESS_ENV	*addressEnvPtr ;

  addressEnvPtr = setAddressEnv(name) ;
  if (addressEnvPtr == NULL) {
    IResetError() ;
    return "" ;
  }
  else
    return IGetSimpleValue(addressEnvPtr, 0, format, 1) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		saveSimpleObjectValue
 *	Description:	saves the string value inValue to the object
 *			with name 'name'.
 *			NOTE: this strips off surrounding double quotes
 *				from the value if they are there.
 *	Parameters:	
 *		char	*name - the name of the object
 *		char	*inValue - a string representing the value
 *	Return Value:	
 *		char	*saveSimpleObjectValue - the new value of the 
 *				object
 ***********************************************************************/
char	*saveSimpleObjectValue(name, inValue)
  char	*name ;
  char	*inValue ;
{
  char		*value = inValue ;

  value = (char *)strtok(inValue, "\"") ;
  if (value == NULL)
    value = inValue ;
  if (strcmp(value, "\"\"") == 0)
    value[0] = '\0' ;

  ISetValue(name, value) ;
  
  return getSimpleObjectValue(name, NULL) ;
}
/**********************************************************************/
