
/**********************************************************************
 * $Id: display.c,v 1.14 92/11/30 11:28:34 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <stdio.h>

#include <X11/ShellP.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Command.h>
#include <X11/cursorfont.h>

#include <X11/Xaw/Box.h>
#include <X11/Xaw/Toggle.h>

/* necessary to circumvent taps private assert routines */
#define _ASSERT_
#include <xerion/useful.h>
#include <xerion/display.h>

#include "Controller.h"
#include "displayUtils.h"
#include "display2Itf.h"
#include "actDisplay.h"
#include "weightDisplay.h"
#include "graphDisplay.h"
#include "mixtureDisplay.h"
#include "xshow.h"
#include "miniDisplay.h"

extern int	IDoSingleCommand ARGS((FILE *inStream, FILE *outStream)) ;

/* Private functions */
static int	enableCommandLineInput ARGS((FILE *in, FILE *out)) ;

static Widget	createMainWindow  ARGS((Widget)) ;
static void	commandLineCB     ARGS((XtPointer, int *, XtInputId *)) ;
static void	quitCB		  ARGS((Widget, XtPointer, XtPointer)) ;

static void	confirmQuit	  ARGS(()) ;
static void	confirmQuitCB	  ARGS((Widget, XtPointer, XtPointer)) ;
static void	confirmQuitAction ARGS((Widget, XEvent *,
					String *, Cardinal *));

static XtActionsRec actions[] = {
  {"quit",		confirmQuitAction},
};

/* Top Level widget for the application */
Widget		topLevel ;
Widget		controller ;

/* input id for command line input */
Atom			wm_delete_window;
static XtInputId	commandId ;

/* 
 * these are passed in from the itf routines 
 * I'd rather not have them here, but there is no easy
 * way around it right now.
 */
static FILE		*InStream, *OutStream ;

typedef struct {
  Boolean	noGraphics ;
} OptionsRec ;

static OptionsRec	options ;

static XtResource	resources[] = {
  {"noGraphics", "NoGraphics", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(OptionsRec, noGraphics), XtRImmediate, (XtPointer)FALSE }
} ;

static XrmOptionDescRec	optionDesc[] = {
  {"-noGraphics", "*noGraphics", XrmoptionNoArg, (XtPointer)"true"}
} ;

/**********************************************************************
 *	Name:		initGraphics
 *	Description:	initializes the Xt toolkit and opens the 
 *			display. We can't use XtAppInitialize
 *			because it exits when it can't open the display
 *			and we want to continue, but without graphics.
 *	Parameters:
 *		int	*argc  - pointer to command line argc
 *		char	**argv - command line argv
 *	Return Value:
 *		int	initGraphics - 0 on failure, 1 on success
 **********************************************************************/
int	initGraphics(argc, argv)
  int		*argc ;
  char		**argv ;
{
  XtAppContext	appContext ;
  Display	*display ;

  XtToolkitInitialize() ;
  appContext = XtCreateApplicationContext() ;
  display    = XtOpenDisplay(appContext, NULL,
			     argv[0], "Xerion", 
			     optionDesc, XtNumber(optionDesc), argc, argv) ;

  if (display == NULL) {
    XtDestroyApplicationContext(appContext) ;
    fprintf (stderr, "Unable to open display, graphics disabled.\n") ;
    return 0 ;
  }

  topLevel = XtVaAppCreateShell(argv[0], "Xerion", 
				applicationShellWidgetClass, display, NULL) ;
  XtVaSetValues(topLevel, XtNallowShellResize, TRUE, NULL) ;

  XtGetApplicationResources(topLevel, (XtPointer)&options,
			    resources, XtNumber(resources), NULL, 0) ;
  if (options.noGraphics) {
    XtDestroyWidget(topLevel) ;
    topLevel = NULL ;
    XtDestroyApplicationContext(appContext) ;
    fprintf (stderr, "Graphics disabled.\n") ;
    return 0 ;
  }

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		buildGraphics
 *	Description:	builds the graphical displays for the simulator.
 *			IInit and initGraphics MUST be called before this 
 *			is called. NOT safe to call multiple times unless
 *			destroyGraphics has been called.
 *	Parameters:	NONE
 *	Return Value:	
 *		int	buildGraphics - 0 on failure, 1 on success
 ***********************************************************************/
int	buildGraphics() 
{
  if (topLevel == NULL)
    return 0 ;

  XtAppAddActions(XtWidgetToApplicationContext(topLevel),
		  actions, XtNumber(actions));

  createMainWindow(topLevel) ;

  XtSetMappedWhenManaged(topLevel, FALSE);
  XtRealizeWidget(topLevel);

  /* do WM_DELETE_WINDOW before map */
  wm_delete_window = XInternAtom(XtDisplay(topLevel),
				 "WM_DELETE_WINDOW", False);
  XSetWMProtocols(XtDisplay(topLevel), XtWindow(topLevel), 
		  &wm_delete_window, 1);
  XtMapWidget(topLevel) ;

  return 1 ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		destroyGraphics
 *	Description:	unmanages and destroys the topLevel widget and
 *			all its children, then sets topLevel to NULL
 *			and closes the display.
 *	Parameters:	NONE
 *	Return Value:
 *	  int	destroyGraphics - 0 if topLevel == NULL, 1 otherwise
 *********************************************************************/
int	destroyGraphics() 
{
  Display	*display ;

  if (topLevel == NULL)
    return 0 ;

  /* get the display and save it, we'll need it later */
  display = XtDisplay(topLevel) ;

  /* Now destroy the top levelest and close the display */
  XtDestroyWidget(topLevel) ;
  XFlush(display) ;
  XtCloseDisplay(display) ;
  topLevel   = NULL ;
  controller = NULL ;

  return 1 ;
}
/********************************************************************/


/**********************************************************************
 *	Name:		graphicsLoop
 *	Description:	the main event loop for using with the graphics.
 *			It also initializes the alternate input stream.
 *	Parameters:
 *		FILE	*inStream  - the stream for input (NULL if you
 *					don't want one)
 *		FILE	*outStream - the stream for output
 *	Return Value:
 *			0 on error, otherwise NEVER RETURNS.
 **********************************************************************/
int	graphicsLoop(inStream, outStream)
  FILE	*inStream, *outStream;
{
  XtAppContext	appContext ;

  appContext =  XtWidgetToApplicationContext(topLevel) ;

  if (inStream != NULL) {
    if (outStream == NULL)
      outStream = stdout ;
    if (enableCommandLineInput(inStream, outStream) == 0)
      return 0 ;
  }

  setSignalHandler() ;
  XtAppMainLoop(appContext) ;
  resetSignalHandler() ;

  return 0 ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		delayDisplayReset
 *	Description:	sets a flag saying to save the display resets
 *			till later
 *	Name:		catchupOnDisplayResets
 *	Description:	unsets a flag saying to save the display resets
 *			till later, then does any resets.
 *	Parameters:	NONE
 *	Return Value:
 *	  int	delayDisplayReset      - TRUE
 *	  int	catchupOnDisplayResets - FALSE
 *********************************************************************/
static	int			delayDisplayResetFlag = FALSE ;
static DisplayNotification	networkState = DNNoChange ;
/**********************************************************************/
int	delayDisplayResets() 
{
  delayDisplayResetFlag = TRUE ;
  return delayDisplayResetFlag ;
}
/********************************************************************/
int			notifyChange(state)
  DisplayNotification	state ;
{
  if (topLevel == NULL)
    return 0 ;

  delayDisplayResetFlag = FALSE ;

  if (controller)
    controllerTriggerCallbacks(controller, (XtPointer)state) ;

  return delayDisplayResetFlag ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		resetDisplay
 *	Description:	sets flags saying that the network widgets
 *			must be rebuilt (should be called after major
 *			changes to the network or whenever the display
 *			params are changed), and if resets are not being
 *			delays
 *	Parameters:	
 *		int	which - a mask set saying which displays should
 *				be redrawn. May contain one or more of:
 *				ACTIVITY_DISPLAY
 *				EXAMPLE_DISPLAY
 *				CONNECTION_DISPLAY
 *				ALL_DISPLAYS
 *	Return Value:	1
 ***********************************************************************/
int	catchupOnDisplayResets() 
{
  notifyChange(networkState) ;
  networkState = DNNoChange ;
  return 1 ;
}
/********************************************************************/
int	markToRebuildDisplay(which) 
  int	which ;
{
  if (topLevel == NULL)
    return 0 ;

  if (which & ALL_DISPLAYS)
    networkState = DNStructureChange ;

  if (delayDisplayResetFlag == FALSE) {
    notifyChange(networkState) ;
    networkState = DNNoChange ;
  }

  return (int)1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		dispatchExposures
 *	Description:	flushes the output buffers and dispatches all
 *			exposure events in the input queue. This should
 *			be called intermittently by commands that take
 *			a VERY long time.
 *	Parameters:	NONE
 *	Return Value:	1
 ***********************************************************************/
int	dispatchExposures() {
  XEvent	event ;

  if (topLevel == NULL)
    return 0 ;

  while (XPending(XtDisplay(topLevel))) {
    XNextEvent(XtDisplay(topLevel), &event) ;
    XtDispatchEvent(&event) ;
  }

  return 1 ;
}
/**********************************************************************/
  

/*********************************************************************
 *	Name:		addUserDisplay
 *	Description:	allows a user to register a name and creation 
 *			procedure for a top level display
 *	Parameters:
 *	  char	*name - the base name for the display 
 *			(eg. "usr" -> "usrDisplay")
 *	  void	(*createProc)() - the creation procedure, will be passed
 *			the shell to create children of and to pop up.
 *	Return Value:
 *	  int	addUserDisplay - 0 on failure, 1 on success
 *********************************************************************/
typedef struct UserDisplay {
  char	*name ;
  void	(*createProc)() ;
} UserDisplay ;
/********************************************************************/
static UserDisplay	*userDisplayList = NULL ;
static int		numUserDisplays = 0 ;
/********************************************************************/
int	addUserDisplay(name, createProc)
  char	*name ;
  void	(*createProc)() ;
{
  ++numUserDisplays ;
  userDisplayList 
    = (UserDisplay *)XtRealloc((char *)userDisplayList,
			       numUserDisplays*sizeof(UserDisplay)) ;

  if (userDisplayList == NULL) {
    numUserDisplays = 0 ;
    return 0 ;
  }
  userDisplayList[numUserDisplays-1].name       = XtNewString(name) ;
  userDisplayList[numUserDisplays-1].createProc = createProc ;

  return 1 ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		createMainWindow
 *	Description:	creates a window with push buttons for popping
 *			up the display  windows for the simulator
 *	Parameters:
 *		Widget	parent - the parent widget to this widget
 *	Return Value:
 *		Widget	createMainWindow - the widget created, NULL on 
 *				error.
 **********************************************************************/
static Widget	createMainWindow(parent)
  Widget	parent ;
{
  int		idx ;

  controller = XtVaCreateManagedWidget("displayBox", 
				       controllerWidgetClass, parent, NULL) ;
  controllerAddDisplay(controller, "activation", 
		       topLevelShellWidgetClass, createActivationDisplay) ;
  controllerAddDisplay(controller, "weight", 
		       topLevelShellWidgetClass, createWeightDisplay) ;
  controllerAddDisplay(controller, "mini", 
		       topLevelShellWidgetClass, createMinimizeDisplay) ;

  /* User defined displays (added by addUserDisplay) */
  for (idx = 0 ; idx < numUserDisplays ; ++idx)
    controllerAddDisplay(controller, userDisplayList[idx].name, 
			 topLevelShellWidgetClass, 
			 userDisplayList[idx].createProc) ;

  controllerAddDisplay(controller, "mixture", 
		       transientShellWidgetClass, createMixtureQuery) ;

  /* The graph displays */
  controllerAddDisplay(controller, "graph", 
		       transientShellWidgetClass, createGraphQuery) ;

  /* The variable etc. lists */
  controllerAddDisplay(controller, "xshow", 
		       topLevelShellWidgetClass, createXshowDisplay) ;

  /* The quit button */
  XtAddCallback(controller, 
		XtNquitCallback, confirmQuitCB, (XtPointer)topLevel) ;

  XtInstallAllAccelerators(controller, controller) ;

  return controller ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		popupShell
 *	Description:	pops up one of the displays (in GrabNone mode)
 *	Parameters:
 *	  char	*name - the name of the shell (display)
 *	Return Value:
 *	  int	popupShell - 1 on success, 0 on failure.
 *********************************************************************/
int	popupShell(name, geometry)
  char	*name ;
  char	*geometry ;
{
  Widget	widget ;

  if (topLevel == NULL)
    return 0 ;

  widget = XtNameToWidget(topLevel, name) ;
  if (widget == NULL)
    return 0 ;

  if (geometry)
    XtVaSetValues(widget, XtNgeometry, geometry, NULL) ;

  XtPopup(widget, XtGrabNone);
  return 1 ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		confirmQuitCB
 *	Description:	quits the application BUT CONFIRMS FIRST
 *	Parameters:	
 *		Widget	widget 	   - the widget the callback is attached to
 *		XtPointer	clientData - the clientData (unused)
 *		XtPointer	callData   - the call data (unused)
 *	Return Value:
 *		NONE
 ***********************************************************************/
static void	confirmQuitCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  confirmQuit() ;
}
/**********************************************************************/
static void	confirmQuitAction(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
  if(event->type == ClientMessage
     && event->xclient.data.l[0] != wm_delete_window)
    return;
    
  confirmQuit() ;
}
/**********************************************************************/
static void	confirmQuit(widget, clientData, callData)
{
  static Widget	shell ;

  if (shell == NULL) {
    Widget	dialog, button ;
    shell = XtVaCreatePopupShell("quitDialog", 
				 transientShellWidgetClass, topLevel, NULL) ;
    
    dialog = XtVaCreateManagedWidget("quitDialog", 
				     dialogWidgetClass, shell, NULL) ;

    button = XtVaCreateManagedWidget("ok", commandWidgetClass, dialog, NULL) ;
    XtAddCallback(button, XtNcallback, popdownWidgetCB, (XtPointer)shell) ;
    XtAddCallback(button, XtNcallback, quitCB	      , (XtPointer)topLevel) ;

    button = XtVaCreateManagedWidget("cancel", 
				     commandWidgetClass, dialog, NULL) ;
    XtAddCallback(button, XtNcallback, popdownWidgetCB, (XtPointer)shell) ;
  }
  popupCentered(shell, XtGrabExclusive) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		quitCB
 *	Description:	quits the entire application
 *	Parameters:
 *		Widget	widget 	   - the widget the callback is attached to
 *		XtPointer	clientData - the client data (unused)
 *		XtPointer	callData   - the call data (unused)
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	quitCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  XtDestroyWidget(topLevel) ;
  fprintf(stderr, "Bye!\n") ;
  fflush(stderr) ;
  exit(0) ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		enableCommandLineInput
 *	Description:	adds command line input to the graphics events
 *			and adds the callback to handle the event.
 *	Parameters:
 *		FILE	*inStream - the stream for input
 *		FILE	*outStream - the stream for output
 *	Return Value:
 *		int	enableCommandLineInput - +1 on success, 0 on 
 *			failure.
 **********************************************************************/
static int	enableCommandLineInput(inStream, outStream)
  FILE	*inStream, *outStream;
{
  int			fileDescriptor ;
  XtAppContext		appContext ;

  if (topLevel == NULL)
    return 0 ;

  InStream  = inStream ;
  OutStream = outStream ;
 
  fileDescriptor = fileno(InStream) ;
  if (fileDescriptor < 0)
    return 0 ;

  appContext =  XtWidgetToApplicationContext(topLevel) ;

  commandId = XtAppAddInput(appContext, fileDescriptor, 
			    (XtPointer)XtInputReadMask, 
			    commandLineCB, (XtPointer)NULL) ;
  if (commandId == 0) 
    return 0 ;

  return 1 ;
}
/**********************************************************************/


/**********************************************************************
 *	Name:		commandLineCB
 *	Description:	the callback for when a command is entered
 *			to the commandLine. This calls the routines
 *			from the itf module which actually reads the
 *			command. It must be added to the event list
 *			by XtAppAddInput.
 *	Parameters:
 *		XtPointer		clientData - unused
 *		int		sourceFile - the file descriptor of
 *				the input file
 *		XtInputId	id - the id returned by XtAppAddInput
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	commandLineCB(clientData, sourceFile, id)
  XtPointer	clientData ;
  int		*sourceFile ;
  XtInputId	*id ;
{
  blockDeviceEvents() ;
  resetSignalHandler() ;
  if (IDoSingleCommand(InStream, OutStream) == 0)
    exit (0) ;
  setSignalHandler() ;
  unblockDeviceEvents() ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		blockDeviceEvents / unblockDeviceEvents
 *	Description:	blocks device events to the application and 
 *			shows a busy cursor, then unblocks it (if it 
 *			is blocked)
 *	Parameters:	NONE
 *	Return Value:	NONE
 ***********************************************************************/
void	blockDeviceEvents() {
  if (controller)
    controllerBlockInput(controller, NULL) ;
}
/**********************************************************************/
void	unblockDeviceEvents() {
  if (controller)
    controllerUnblockInput(controller, NULL) ;
}
/**********************************************************************/
void	blockShellDeviceEvents(shell)
  Widget	shell ;
{
  if (controller)
    controllerBlockInput(controller, shell) ;
}
/**********************************************************************/
void	unblockShellDeviceEvents(shell)
  Widget	shell ;
{
  if (controller)
    controllerUnblockInput(controller, shell) ;
}
/**********************************************************************/
