
/**********************************************************************
 * $Id: actDisplay.c,v 1.11 93/03/03 12:48:12 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>

#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Viewport.h>

#include <xerion/Network.h>
#include <xerion/RowColumn.h>
#include <xerion/Selection.h>
#include <xerion/MenuPane.h>
#include <xerion/display.h>
#include <xerion/Trace.h>

#include "display2Itf.h"
#include "display2Sim.h"
#include "displayUtils.h"
#include "check.h"

typedef struct {
  Widget	shell ;
  Widget	fixedMax ;
  Widget	maxValue ;
  Widget	size ;
  Widget	border ;
  Widget	margin ;
} *MiscNetworkDialog ;

typedef struct {
  Widget	shell ;		/* the popup shell for the display 	*/
  String	titleFormat ;	/* the format for the title string	*/
  Widget	*back ;		/* the widget backgrounds available	*/
  XtStyle	*style ;	/* the widget backgrounds available	*/
  int		styleIdx ;	/* which style to use 			*/
  Widget	*set ;		/* the example sets available		*/
  Widget	setLabelWidget ;/* the label for the example list	*/
  int		setIdx ;	/* the index for the set in use		*/
  Widget	*field ;	/* the widget backgrounds available	*/
  int		fieldIdx ;	/* the index of the link field to display */
  Widget	fieldLabel ;	/* the label for the hinton diagram	*/
  Widget	listWidget ;	/* the example list			*/
  Widget	activationForm ;/* form around the activation network	*/
  Widget	activationNet ;	/* the activation network		*/
  Widget	targetForm ;	/* form around the target network	*/
  Widget	targetNet ;	/* the target network			*/
  Widget	showTargets ;	/* option to show/hide target network	*/

  MiscNetworkDialog	dialog ;
  DisplayNotification	state ;
} ActivationDisplayRec, *ActivationDisplay ;

typedef struct {
  String	showExampleSet ;
  String	showField ;
  Boolean	showTargets ;
} OptionsRec ;

static OptionsRec	options ;

static XtResource	resources[] = {
  {"showExampleSet", "ShowExampleSet", XtRString, sizeof(String),
     XtOffsetOf(OptionsRec, showExampleSet), 
     XtRImmediate, (XtPointer)"trainingSet" },
  {"showField", "ShowField", XtRString, sizeof(String),
     XtOffsetOf(OptionsRec, showField), XtRImmediate, (XtPointer)"activity" },
  {"showTargets", "ShowTargets", XtRBoolean, sizeof(Boolean),
     XtOffsetOf(OptionsRec, showTargets), XtRString, (XtPointer)"true" }
} ;

#if 1
static void 	markExampleAction ARGS((Widget, 
					XEvent *, String *, Cardinal *)) ;
static void 	deleteExampleAction ARGS((Widget, 
					  XEvent *, String *, Cardinal *)) ;
static void 	copyExampleAction ARGS((Widget, 
					XEvent *, String *, Cardinal *)) ;
static void 	moveExampleAction ARGS((Widget, 
					XEvent *, String *, Cardinal *)) ;
static XtActionsRec	actions[] = {
  {"MarkExample", 	markExampleAction},
  {"DeleteExample", 	deleteExampleAction},
  {"CopyExample", 	copyExampleAction},
  {"MoveExample", 	moveExampleAction}
} ;
#endif

/* Routines for redisplaying the whole display */
static void	resetDisplay	 ARGS((ActivationDisplay)) ;
static void 	resetCB		 ARGS((Widget, XtPointer, XtPointer)) ;
static void 	stateResetCB	 ARGS((Widget, XtPointer, XtPointer)) ;
static void 	displayResetCB	 ARGS((Widget, XtPointer, XtPointer)) ;
static void 	structureResetCB ARGS((Widget, XtPointer, XtPointer)) ;
static void 	setSensitive	 ARGS((ActivationDisplay)) ;

/* routines for redisplaying networks */
static void	resetNets       ARGS((ActivationDisplay)) ;
static void	rebuildNet	ARGS((ActivationDisplay,
				      Widget, UnitDisplayRec *, int)) ;

/* routines for handling the network appearance */
static void 	miscellaneousCB    ARGS((Widget, XtPointer, XtPointer)) ;
static void 	changeBackgroundCB ARGS((Widget, XtPointer, XtPointer)) ;
static void 	checkBackgroundCB  ARGS((Widget, XtPointer, XtPointer)) ;
static void	changeFieldCB	   ARGS((Widget, XtPointer, XtPointer)) ;
static void 	checkFieldCB	   ARGS((Widget, XtPointer, XtPointer)) ;
static void 	toggleManagedCB	   ARGS((Widget, XtPointer, XtPointer)) ;
static void 	toggleManaged	   ARGS((ActivationDisplay)) ;

/* routines for handling the examples */
static void	changeExampleSetCB ARGS((Widget, XtPointer, XtPointer)) ;
static void 	checkSetCB	   ARGS((Widget, XtPointer, XtPointer)) ;
static void	doExampleCB        ARGS((Widget, XtPointer, XtPointer)) ;
static void	resetExamples	   ARGS((ActivationDisplay)) ;

/* miscellaneous routines */
static void 	destroyCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	destroyMiscCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	moveUnitCB	ARGS((Widget, XtPointer, XtPointer)) ;
static void 	changeValueCB   ARGS((Widget, XtPointer, XtPointer)) ;
static void 	printValueCB	ARGS((Widget, XtPointer, XtPointer)) ;

static Cardinal	numDisplays = 0 ;

static XtStyle	styleOption[] = { XtShadow, XtLines, XtLines2, XtWeave,
				    XtPlain, XtSolid } ;
static String	styleName[] = { "shadow", "lines", "lines2", "weave",
				  "plain", "solid" } ;

/***********************************************************************
 *	Name:		createActivationDisplay
 *	Description:	creates the window for the network display
 *			and all of the sub windows and necessary methods
 *	Parameters:	
 *		Widget	popup - the popup shell we are going in
 *	Return Value:	
 *		NONE
 ***********************************************************************/
export void	createActivationDisplay(popup)
  Widget	popup ;
{
  ActivationDisplay	this ;
  Widget	menuPane, menuButton, menu, line ;
  Widget	selection, net ;
  Widget	form, vpane, subForm, viewport, updateShell ;
  Widget	widget, label, button ;
  String	*list, name ;
  char		buffer[BUFSIZ] ;
  int		idx ;

  this = (ActivationDisplay)XtMalloc(sizeof(ActivationDisplayRec)) ;

  this->shell = popup ;
  XtAddCallback(this->shell, XtNdestroyCallback, destroyCB, (XtPointer)this) ;

  XtGetApplicationResources(this->shell, (XtPointer)&options,
			    resources, XtNumber(resources), NULL, 0) ;
#if 1
  storeData(this->shell, (XtPointer)this) ;
  XtAppAddActions(XtWidgetToApplicationContext(this->shell),
		  actions, XtNumber(actions)) ;
#endif

  /********************************************
   * Network Name label 
   */
  this->titleFormat = getTitle(this->shell) ;
  getNetValue(NAME, &name) ;
  widgetPrintf(this->shell, this->titleFormat, name) ;

  form = XtVaCreateManagedWidget("form", formWidgetClass, popup, NULL) ;
  redirectKeyboardInput(form) ;

  /********************************************
   * Display Updates 
   */
  sprintf(buffer, "actDisplay %d", numDisplays++) ;
  updateShell = XtVaCreatePopupShell(buffer, traceWidgetClass, popup, 
				     NULL) ;
  XtAddCallback(updateShell, XtNcallback, stateResetCB, (XtPointer)this) ;

  /********************************************
   * Pulldown menu pane 
   */
  menuPane = XtVaCreateManagedWidget("menuPane", menuPaneWidgetClass, form, 
				     NULL) ;
  /**********************
   * Options menu 
   */
  menuButton = XtVaCreateManagedWidget("options",
				       menuButtonWidgetClass, menuPane,
				       XtNmenuName, "optionsMenu",
				       NULL);

  menu = XtVaCreatePopupShell("optionsMenu", simpleMenuWidgetClass, menuButton,
			      NULL) ;

  button = XtVaCreateManagedWidget("updates", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, popupCenteredCB, (XtPointer)updateShell) ;

  button = XtVaCreateManagedWidget("reset", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, structureResetCB, (XtPointer)this) ;

  button = XtVaCreateManagedWidget("targets", smeBSBObjectClass, menu, 
				   XtNlabel, "Hide Targets",
				   NULL) ;
  XtAddCallback(button, XtNcallback, toggleManagedCB, (XtPointer)this) ;
  this->showTargets = button ;

  line = XtVaCreateManagedWidget("line", smeLineObjectClass, menu, NULL) ;

  button = XtVaCreateManagedWidget("close", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, popdownWidgetCB, (XtPointer)popup) ;


  /**********************
   * "Example Set" menu 
   */
  menuButton = XtVaCreateManagedWidget("set", menuButtonWidgetClass, menuPane,
				       XtNmenuName, "setMenu",
				       NULL) ;

  menu = XtVaCreatePopupShell("setMenu", simpleMenuWidgetClass, menuButton, 
			      NULL) ;
  XtAddCallback(menu, XtNpopupCallback, checkSetCB, (XtPointer)this) ;
  
  list = listExampleSets() ;
  for (idx = 0 ; list && list[idx] ; ++idx)
    ;
  this->set    = (Widget  *)XtCalloc(idx + 1, sizeof(Widget)) ;
  this->setIdx = 0 ;
  for (idx = 0 ; list && list[idx] ; ++idx) {
    widget = XtVaCreateManagedWidget(list[idx], smeBSBObjectClass, menu,
				     XtNleftMargin, check_width, 
				     NULL) ;
    XtAddCallback(widget, XtNcallback, changeExampleSetCB, (XtPointer)this) ;
    this->set[idx] = widget ;
    if (strcmp(options.showExampleSet, list[idx]) == 0)
      this->setIdx = idx ;
  }
  this->set[idx] = NULL ;

  /**********************
   * "Field" menu 
   */
  menuButton = XtVaCreateManagedWidget("field",
				       menuButtonWidgetClass, menuPane,
				       XtNmenuName, "fieldMenu",
				       NULL) ;

  menu = XtVaCreatePopupShell("fieldMenu", simpleMenuWidgetClass, menuButton, 
			      NULL) ;
  XtAddCallback(menu, XtNpopupCallback, checkFieldCB, (XtPointer)this) ;
  
  list = listUnitFields() ;
  for (idx = 0 ; list && list[idx] ; ++idx)
    ;
  this->field = (Widget *)XtCalloc(idx + 1, sizeof(Widget)) ;

  this->fieldIdx = 0 ;
  for (idx = 0 ; list && list[idx] != NULL ; ++idx) {
    widget = XtVaCreateManagedWidget(list[idx], smeBSBObjectClass, menu,
				     XtNleftMargin, check_width, 
				     NULL) ;
    XtAddCallback(widget, XtNcallback, changeFieldCB, (XtPointer)this) ;
    this->field[idx] = widget ;
    if (strcmp(list[idx], options.showField) == 0)
      this->fieldIdx = idx ;
  }
  this->field[idx] = NULL ;

  /**********************
   * Network menu 
   */
  menuButton = XtVaCreateManagedWidget("network",
				       menuButtonWidgetClass, menuPane,
				       XtNmenuName, "networkMenu",
				       NULL) ;
  menu = XtVaCreatePopupShell("networkMenu", simpleMenuWidgetClass, menuButton,
			      NULL);
  XtAddCallback(menu, XtNpopupCallback,
		checkBackgroundCB, (XtPointer)this) ;
  
  this->back  = (Widget  *)XtCalloc(XtNumber(styleOption)+1, sizeof(Widget)) ;
  this->style = (XtStyle *)XtCalloc(XtNumber(styleOption)+1, sizeof(XtStyle)) ;
  for (idx = 0 ; idx < XtNumber(styleOption) ; ++idx) {
    Widget	back ;
    back = XtVaCreateManagedWidget(styleName[idx], smeBSBObjectClass, menu,
				   XtNleftMargin, check_width,
				   NULL) ;
    XtAddCallback(back, XtNcallback, changeBackgroundCB, (XtPointer)this) ;
    this->back[idx]  = back ;
    this->style[idx] = styleOption[idx] ;
  }
  this->styleIdx   = -1 ;
  this->back[idx]  = NULL ;
  this->style[idx] = (XtStyle)-1 ;

  line = XtVaCreateManagedWidget("line", smeLineObjectClass, menu, NULL) ;

  button = XtVaCreateManagedWidget("misc", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, miscellaneousCB, (XtPointer)this) ;

  /********************************************
   * Example lists
   */
  /* Example Set list of examples */
  selection = XtVaCreateManagedWidget("exampleList", 
				      selectionWidgetClass, form,
				      XtNlist,listExamples(this->setIdx),
				      NULL) ;
  XtAddCallback(selection, XtNcallback, doExampleCB, (XtPointer)this) ;
  this->listWidget = selection ;

  this->setLabelWidget = XtNameToWidget(selection, "listLabel") ;
  XtVaGetValues(this->set[this->setIdx], XtNlabel, &name, NULL) ;
  widgetPrintf(this->setLabelWidget, "%-10s", name) ;

  /********************************************
   * The Networks 
   */
  vpane = XtVaCreateManagedWidget("vpane", panedWidgetClass, form,
				  XtNtop, 	XtChainTop,
				  XtNbottom, 	XtChainBottom,
				  XtNright,  	XtChainRight,
				  NULL) ;
  /**********************
   * Target Outputs 
   */
  subForm  = XtVaCreateManagedWidget("targetForm", formWidgetClass, vpane, 
				     NULL) ;
  this->targetForm = subForm ;
  label    = XtVaCreateManagedWidget("targetLabel", labelWidgetClass, subForm, 
				     NULL) ;
  viewport = XtVaCreateManagedWidget("targetViewport", 
				     viewportWidgetClass, subForm, NULL) ;
  net      = XtVaCreateManagedWidget("targets", networkWidgetClass,  viewport, 
				     NULL);
  XtAddCallback(net, XtNqueryCallback,	     printValueCB,  (XtPointer)this) ;
  XtAddCallback(net, XtNchangeValueCallback, changeValueCB, (XtPointer)this) ;
  this->targetNet  = net ;

  if (!options.showTargets)
    toggleManaged(this) ;

  /**********************
   * Network Activation 
   */
  subForm  = XtVaCreateManagedWidget("activityForm", formWidgetClass, vpane, 
				     NULL) ;
  this->activationForm = subForm ;
  label    = XtVaCreateManagedWidget("activityLabel", 
				     labelWidgetClass, subForm, NULL) ;
  XtVaGetValues(this->field[this->fieldIdx], XtNlabel, &name, NULL) ;
  widgetPrintf(label, name) ;
  this->fieldLabel = label ;

  viewport = XtVaCreateManagedWidget("activityViewport",
				     viewportWidgetClass, subForm, NULL) ;
  net 	   = XtVaCreateManagedWidget("activities",
				     networkWidgetClass, viewport, NULL);
  XtAddCallback(net, XtNqueryCallback,  printValueCB,	(XtPointer)this) ;
  XtAddCallback(net, XtNmotionCallback, moveUnitCB,     (XtPointer)net) ;
  XtAddCallback(net, XtNchangeValueCallback, changeValueCB, (XtPointer)this);
  this->activationNet = net ;

  XtInstallAllAccelerators(form, form) ;

  /********************************************
   * Do all the popup CB's for the first time 
   */
  XtAddCallback(popup, XtNpopupCallback, structureResetCB, (XtPointer)this) ;
#if 1
#endif
  XtAddCallback(controller, XtNcallback, resetCB, 	   (XtPointer)this) ;

  XtVaSetValues(popup, XtNcreatePopupChildProc, NULL, NULL) ;

  this->dialog = NULL ;
  this->state  = DNStructureChange ;
  resetDisplay(this) ;
}
/**********************************************************************/

#if 1
#endif

/*********************************************************************
 *	Name:		destroyCB
 *	Description:	destroys the ActivationDisplay
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - the ActivationDisplay
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	destroyCB - 
 *********************************************************************/
static void	destroyCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  if (this == NULL)
    return ;

  if (this->titleFormat)
    XtFree((void *)this->titleFormat) ;
  if (this->set)
    XtFree((void *)this->set) ;
  if (this->field)
    XtFree((void *)this->field) ;
  if (this->back)
    XtFree((void *)this->back) ;
  if (this->style)
    XtFree((void *)this->style) ;

  XtRemoveCallback(controller, XtNcallback, resetCB, (XtPointer)this) ;

  XtFree((void *)this) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		changeBackgroundCB
 *	Description:	changes the unit background used in the networks
 *	Parameters:
 *	  Widget	widget     - the menu option
 *	  XtPointer	clientData - the activationDisplay
 *	  XtPointer	callData   - UNUSED
 *	Return Value:
 *	  static void	changeBackgroundCB - 
 *********************************************************************/
static void	changeBackgroundCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  int			idx ;

  if (this == NULL || this->back == NULL)
    return ;

  for (idx = 0 ; this->back[idx] && this->back[idx] != widget ; ++idx)
    ;

  if (this->back[idx]) {
    this->styleIdx    = idx ;
    this->state = DNDisplayChange ;
    resetDisplay(this) ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		checkBackgroundCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Background:
 *	  static void	checkBackgroundCB - 
 *********************************************************************/
static void	checkBackgroundCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  int			idx ;

  for (idx = 0 ; this->back[idx] ; ++idx)
    checkMenuItem(this->back[idx], idx == this->styleIdx ? TRUE : FALSE) ;
}
/********************************************************************/


/**********************************************************************
 *	Name:		changeFieldCB
 *	Description:	sets the mask of what connection field to
 *			display
 *	Parameters:
 *	  Widget	widget 	   - the widget the callback is attached to
 *	  XtPointer	clientData - the index to set it to
 *	  XtPointer	callData   - the call data (XtListReturnStruct *)
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	changeFieldCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  int			idx ;

  if (this == NULL || this->field == NULL)
    return ;

  for (idx = 0 ; this->field[idx] && this->field[idx] != widget ; ++idx)
    ;

  if (this->field[idx]) {
    String	name ;
    this->fieldIdx = idx ;
    XtVaGetValues(widget, XtNlabel, &name, NULL) ;
    widgetPrintf(this->fieldLabel, name) ;
    this->state = DNDisplayChange ;
    resetDisplay(this) ;
  }
}
/**********************************************************************/


/*********************************************************************
 *	Name:		checkFieldCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	checkFieldCB - 
 *********************************************************************/
static void	checkFieldCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  int			idx ;

  for (idx = 0 ; this->field[idx] ; ++idx)
    checkMenuItem(this->field[idx], idx == this->fieldIdx ? TRUE : FALSE) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		resetNets
 *	Description:	rebuilds the network widgets showing the 
 *			network outputs and targets from the current net
 *	  ActivationDisplay	this - the display
 *	Return Value:
 *		NONE
 ***********************************************************************/
static void		resetNets(this)
  ActivationDisplay	this ;
{
  static UnitDisplayRec	*activeUnit = NULL, *targetUnit = NULL ;
  static int		maxActiveUnits = 0, maxTargetUnits = 0 ;

  DisplayNotification	state = this->state ; 
  int			numActiveUnits, numTargetUnits ;

  if (netExists()) {
    activeUnit = getUnitValues(this->fieldIdx, 
			       activeUnit, &numActiveUnits, &maxActiveUnits) ;
    targetUnit = getNetworkTargets(targetUnit, 
				   &numTargetUnits, &maxTargetUnits) ;
  } else {
    numActiveUnits = 0 ;
    numTargetUnits = 0 ;
  }

  if (state == DNStructureChange || state == DNDisplayChange) {
    rebuildNet(this, this->activationNet, activeUnit, numActiveUnits) ;
    rebuildNet(this, this->targetNet,     targetUnit, numTargetUnits) ;
  } else {
    if (numActiveUnits == 0) /* No network, blank out the widget */
      rebuildNet(this, this->activationNet, NULL, 0) ;
    else
      XtNetworkSetUnits(this->activationNet, activeUnit) ;

    if (numTargetUnits == 0) /* No network, blank out the widget */
      rebuildNet(this, this->targetNet, NULL, 0) ;
    else
      XtNetworkSetUnits(this->targetNet, targetUnit) ;
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		rebuildNet
 *	Description:	rebuilds the units and size of a network widget
 *	  Widget		network    - the network widget to update
 *	  UnitDisplayRec	*unitArray - the array of units to  update with
 *	  int		numUnits   - the number of units
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	rebuildNet(this, network, unitArray, numUnits)
  ActivationDisplay	this ;
  Widget		network ;
  UnitDisplayRec	*unitArray ;
  int			numUnits ;
{
  int		height, width, margin ;

  getNetValue(ACTIVATION_HEIGHT, &height) ;
  getNetValue(ACTIVATION_WIDTH,  &width) ;
  getNetValue(ACTIVATION_MARGIN, &margin) ;

  if (this->styleIdx < 0) {
    XtStyle	style ;
    int		idx ;

    XtVaGetValues(network, XtNunitBackgroundStyle, &style, NULL) ;
    for (idx = 0 ; this->style[idx] >= 0 && style != this->style[idx] ; ++idx)
      ;
    this->styleIdx = this->style[idx] >= 0 ? idx : 0 ;
  }

  XtUnmanageChild(network) ;
  XtVaSetValues(network,
		XtNunitBackgroundStyle, this->style[this->styleIdx],
		XtNunitHeight,	height,
		XtNunitWidth, 	width,
		XtNunitSep,	margin,
		XtNnumUnits,	numUnits,
		XtNunitArray,	numUnits ? unitArray : NULL,
		NULL) ;
  XtManageChild(network) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		checkSetCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	checkFieldCB - 
 *********************************************************************/
static void	checkSetCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  int			idx ;

  for (idx = 0 ; this->set[idx] != NULL ; ++idx)
    checkMenuItem(this->set[idx], idx == this->setIdx ? TRUE : FALSE) ;
}
/********************************************************************/


/**********************************************************************
 *	Name:		changeExampleSetCB
 *	Description:	sets the index of what ExampleSet to get examples
 *			from
 *	Parameters:
 *	  Widget	widget 	   - the widget the callback is attached to
 *	  XtPointer	clientData - the index of the example set to use
 *	  XtPointer	callData   - the call data (XawListReturnStruct *)
 *	Return Value:
 *		NONE
 **********************************************************************/
static void	changeExampleSetCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  String		label ;
  int			idx ;

  if (this == NULL || this->set == NULL)
    return ;

  for (idx = 0 ; this->set[idx] && this->set[idx] != widget ; ++idx)
    ;

  if (this->set[idx]) {
    XtVaGetValues(widget, XtNlabel, &label, NULL) ;
    widgetPrintf(this->setLabelWidget, "%-10s", label) ;
    this->setIdx = idx ;
    this->state = DNDisplayChange ;
    resetDisplay(this) ;
  }
}
/**********************************************************************/


/*********************************************************************
 *	Name:		setSensitive
 *	Description:	sets the sensitive field to true or false for
 *			all of the widgets that care.
 *	Parameters:
 *	  ActivationDisplay	display - the display
 *	Return Value:
 *	  static void	setSensitive - NONE
 *********************************************************************/
static void	setSensitive(display)
  ActivationDisplay	display ;
{
  Widget	menuPane = XtNameToWidget(display->shell, "form.menuPane") ;

  /* if no network, can't set the background of units */
  XtVaSetValues(XtNameToWidget(menuPane, "network"), 
		XtNsensitive, netExists() ? TRUE : FALSE,
		NULL) ;
  XtVaSetValues(XtNameToWidget(menuPane, "set"),
		XtNsensitive, netExists() ? TRUE : FALSE,
		NULL) ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		resetExamples
 *	Description:	rebuilds the list of examples
 *		Widget	listWidget - the list (or selection) widget to update
 *		int	index - the index of the example set to use
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	resetExamples(this)
  ActivationDisplay	this ;
{
  int		exampleIdx ;

  if (this->state == DNStructureChange || this->state == DNDisplayChange)
    XtVaSetValues(this->listWidget, XtNlist, listExamples(this->setIdx), NULL);

  exampleIdx = getExampleIndex(this->setIdx) ;
  if (exampleIdx >= 0)
    XtSelectionHighlight(this->listWidget, exampleIdx) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		doExampleCB
 *	Description:	performs an example on the current net. It
 *			gets the index of the example to do from
 *			the callData which must be XawListReturnStruct *
 *	Parameters:	
 *	  Widget	widget 	   - the list the callback is attached to
 *	  XtPointer	clientData - the activation display
 *	  XtPointer	callData   - the call data (XawListReturnStruct *)
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	doExampleCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  XawListReturnStruct	*returnStruct = (XawListReturnStruct *)callData ;

  if (returnStruct == NULL || returnStruct->list_index == XAW_LIST_NONE)
    return ;

  doExample(this->setIdx, returnStruct->list_index) ;

  this->state = DNStateChange ;
  resetDisplay(this) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		printValueCB
 *	Description:	prints out the value (target/activity) of a unit
 *	Parameters:	
 *		Widget		widget     - the network widget
 *		XtPointer	clientData - the activationDisplay
 *		XtPointer	callData   - NetworkReturnStruct *
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	printValueCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this	      = (ActivationDisplay)clientData ;
  NetworkReturnStruct	*returnStruct = (NetworkReturnStruct *)callData ;
  UnitDisplayRec	*unitPtr ;

  if (!XtIsSubclass(widget, networkWidgetClass))
    return ;

  unitPtr = XtNetworkGetUnit(widget, returnStruct->index) ;
  if (unitPtr) {
    String	fieldName ;
    if (widget == this->targetNet) 
      fieldName = strdup("Target") ;
    else 
      fieldName = getLabel(this->field[this->fieldIdx]) ;
    
    fprintf(stdout, "\n%s = %.4g for unit \"%s\"\n",
	    fieldName, unitPtr->value, (char *)unitPtr->userData) ;
    XtFree((char *)unitPtr) ;
    XtFree((char *)fieldName) ;
  }
}
/**********************************************************************/
static void	changeValueCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this	      = (ActivationDisplay)clientData ;
  NetworkReturnStruct	*returnStruct = (NetworkReturnStruct *)callData ;
  UnitDisplayRec	*unitPtr ;

  if (!XtIsSubclass(widget, networkWidgetClass))
    return ;

  unitPtr = XtNetworkGetUnit(widget, returnStruct->index) ;
  if (unitPtr) {
    if (widget == this->targetNet)
      changeTarget(unitPtr) ;
    else if (widget == this->activationNet)
      changeUnitValue(unitPtr, this->fieldIdx) ;
    updateNetActivities() ;
    XtFree((char *)unitPtr) ;
  }

  this->state = DNStateChange ;
  resetDisplay(this) ;
  
}
/**********************************************************************/
static void	moveUnitCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this	      = (ActivationDisplay)clientData ;
  NetworkReturnStruct	*returnStruct = (NetworkReturnStruct *)callData ;
  UnitDisplayRec	*unitPtr ;

  unitPtr = XtNetworkGetUnit(widget, returnStruct->index) ;
  moveUnit(unitPtr) ;
  this->state = DNDisplayChange ;
  resetDisplay(this) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		...ResetCB
 *	Description:	resets the display (i.e. sets the sensitivity
 *			of the widgets and rebuilds nets and example
 *			lists)
 *	Parameters:
 *	  Widget	widget     - the widget the cb is attached to
 *	  XtPointer	clientData - the activationDisplay
 *	  XtPointer	callData   - UNUSED
 *	Return Value:
 *	  static void resetDisplayCB - NONE
 *********************************************************************/
static void		resetDisplay(this)
  ActivationDisplay	this ;
{
  if (this == NULL)
    return ;

  if (this->state == DNStructureChange) {
    String		name ;
    getNetValue(NAME, &name) ;
    widgetPrintf(this->shell, this->titleFormat, name) ;
    setSensitive(this) ;
  }

  /* reset the examples and the nets */
  resetExamples(this) ;
  resetNets(this) ;

  this->state = DNNoChange ;

  dispatchExposures() ;
}
/********************************************************************/
static void resetCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;

  this->state = (DisplayNotification)callData ;
  resetDisplay(this) ;
}
/********************************************************************/
static void stateResetCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;

  this->state = DNStateChange ;
  resetDisplay(this) ;
}
/********************************************************************/
static void structureResetCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;

  this->state = DNStructureChange ;
  resetDisplay(this) ;
}
/********************************************************************/
static void displayResetCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;

  this->state = DNDisplayChange ;
  resetDisplay(this) ;
}
/********************************************************************/

/*********************************************************************
 *	Name:		setMiscellaneousCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData -
 *	Return Value:
 *	  static void	setMiscellaneousCB - 
 *********************************************************************/
static void	setMiscellaneousCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  MiscNetworkDialog	dialog = this->dialog ;
  Boolean		fixedMax ;
  int			size, margin, border ;
  float			maxValue ;
  String		string ;
  XtVarArgsList		argList ;

  string = getText(dialog->maxValue) ;
  if (string != NULL)
    maxValue = atof(string) ;
  else
    maxValue = 0.0 ;

  XtVaGetValues(dialog->fixedMax, XtNstate, &fixedMax, NULL) ;

  string = getText(dialog->size) ;
  if (string != NULL) {
    size = atol(string) ;
    setNetValue(ACTIVATION_WIDTH,  &size) ;
    setNetValue(ACTIVATION_HEIGHT, &size) ;
  } else {
    getNetValue(ACTIVATION_WIDTH,  &size) ;
    getNetValue(ACTIVATION_HEIGHT, &size) ;
  }

  string = getText(dialog->border) ;
  if (string != NULL)
    border = atol(string) ;
  else
    XtVaGetValues(this->targetNet, XtNunitBorderWidth, &border, NULL) ;

  string = getText(dialog->margin) ;
  if (string != NULL) {
    margin = atol(string) ;
    setNetValue(ACTIVATION_MARGIN,  &margin) ;
  } else {
    getNetValue(ACTIVATION_MARGIN,  &margin) ;
  }

  argList = XtVaCreateArgsList(NULL,
			       XtNfixedMax,		fixedMax,
			       XtNmaxValue,		&maxValue,
			       XtNunitHeight,		size,
			       XtNunitWidth,		size,
			       XtNunitSep,		margin,
			       XtNunitBorderWidth,	border,
			       NULL) ;
  XtVaSetValues(this->activationNet, XtVaNestedList, argList, NULL) ;
  XtVaSetValues(this->targetNet,     XtVaNestedList, argList, NULL) ;
  XtFree(argList) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		miscellaneousCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	miscellaneousCB - 
 *********************************************************************/
static void	miscellaneousCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  float			*max ;
  Dimension		size, margin, border ;
  Boolean		fixedMax ;

  if (this->dialog == NULL) {
    MiscNetworkDialog	dialog ;
    Widget		form, button, table ;

    dialog = (MiscNetworkDialog)XtMalloc(sizeof(*dialog)) ;

    dialog->shell = XtVaCreatePopupShell("miscShell", 
					 transientShellWidgetClass, 
					 this->shell, NULL) ;
    XtAddCallback(dialog->shell, 
		  XtNdestroyCallback, destroyMiscCB, (XtPointer)this) ;
    
    form = XtVaCreateManagedWidget("form", 
				   formWidgetClass, dialog->shell, NULL) ;

    table = startTable(form) ;
    dialog->maxValue = addTableItem("maxValue", &table, TRUE) ;
    dialog->size     = addTableItem("size",   	&table, FALSE) ;
    dialog->border   = addTableItem("border",	&table, FALSE) ;
    dialog->margin   = addTableItem("margin",	&table, FALSE) ;

    dialog->fixedMax = XtNameToWidget(form, "maxValueToggle") ;

    button = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
				     XtNfromVert,	dialog->margin,
				     NULL) ;
    XtAddCallback(button, XtNcallback, setMiscellaneousCB, (XtPointer)this) ;
    XtAddCallback(button, XtNcallback,
		  popdownWidgetCB, (XtPointer)dialog->shell) ;

    button = XtVaCreateManagedWidget("cancel", commandWidgetClass, form,
				     XtNfromVert,	dialog->margin,
				     XtNfromHoriz,	button,
				     NULL) ;
    XtAddCallback(button, XtNcallback, 
		  popdownWidgetCB, (XtPointer)dialog->shell) ;

    XtInstallAllAccelerators(form, form) ;
    this->dialog = dialog ;
  }

  XtVaGetValues(this->activationNet,
		XtNfixedMax,		&fixedMax,
		XtNmaxValue,		&max,
		XtNunitHeight,		&size,
		XtNunitSep,		&margin,
		XtNunitBorderWidth,	&border,
		NULL) ;

  XtVaSetValues(this->dialog->fixedMax, XtNstate, fixedMax, NULL) ;
  widgetPrintf(this->dialog->maxValue,	"%g", *max) ;
  widgetPrintf(this->dialog->size,	"%d", size) ;
  widgetPrintf(this->dialog->border,	"%d", border) ;
  widgetPrintf(this->dialog->margin,	"%d", margin) ;

  popupCentered(this->dialog->shell, XtGrabExclusive) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		destroyMiscCB
 *	Description:	destroys the ActivationDisplay
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - the ActivationDisplay
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	destroyMiscCB - 
 *********************************************************************/
static void	destroyMiscCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  ActivationDisplay	this = (ActivationDisplay)clientData ;
  if (this == NULL)
    return ;

  if (this->dialog)
    XtFree((void *)this->dialog) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		toggleManagedCB
 *	Description:	
 *	Parameters:
 *	  Widget	widget - 
 *	  XtPointer	clientData - 
 *	  XtPointer	callData - 
 *	Return Value:
 *	  static void	toggleManagedCB - 
 *********************************************************************/
static void		toggleManaged(this)
  ActivationDisplay	this ;
{
  String		label = getLabel(this->showTargets) ;

  if (strcmp(label, "Show Targets") == 0) { 
    /* order of managing is important */
    XtUnmanageChild(this->activationForm) ;
    XtManageChild  (this->targetForm) ;
    XtManageChild  (this->activationForm) ;
    widgetPrintf(this->showTargets, "Hide Targets") ;
  } else {
    XtUnmanageChild(this->targetForm) ;
    widgetPrintf(this->showTargets, "Show Targets") ;
  }
  XtFree(label) ;
}
/********************************************************************/
static void	toggleManagedCB(widget, clientData, callData)
  Widget	widget ;
  XtPointer	clientData ;
  XtPointer	callData ;
{
  toggleManaged((ActivationDisplay)clientData) ;
}
/********************************************************************/

#if 1
/**********************************************************************/
#define	findActivationDisplay(w) ((ActivationDisplay)retrieveData(w))
/**********************************************************************/
typedef struct	MarkedExample {
  ActivationDisplay	display ;
  int		set ;
  int		idx ;
} MarkedExample ;
static MarkedExample markedExample = { NULL, -1, -1 } ;
/**********************************************************************/
static void 	markExampleAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  XawListReturnStruct	*returnStruct ;
  ActivationDisplay	this = findActivationDisplay(w) ;

  if (this == NULL || this->listWidget == NULL)
    return ;

  returnStruct = XtSelectionGetSelected(this->listWidget) ;
  
  if (returnStruct == NULL || returnStruct->list_index == XAW_LIST_NONE)
    return ;

  markedExample.display = this ;
  markedExample.set = this->setIdx ;
  markedExample.idx = returnStruct->list_index ;
}
/**********************************************************************/
static void 	deleteExampleAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  if (markedExample.display == NULL
      || markedExample.set < 0 || markedExample.idx < 0)
    return ;

  if (markedExample.set >= 0 && markedExample.idx >= 0) {
    deleteExample(markedExample.set, markedExample.idx) ;
    markedExample.set = -1 ;
    markedExample.idx = -1 ;
  }

  markedExample.display->state = DNStructureChange ;
  resetDisplay(markedExample.display) ;
}
/**********************************************************************/
static void 	copyExampleAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  XawListReturnStruct	*returnStruct ;
  int			toIdx ;
  ActivationDisplay	this ;

  if (markedExample.display == NULL
      || markedExample.set < 0 || markedExample.idx < 0)
    return ;

  this = findActivationDisplay(w) ;
  if (this == NULL || this->listWidget == NULL)
    return ;

  returnStruct = XtSelectionGetSelected(this->listWidget) ;
  
  if (returnStruct == NULL || returnStruct->list_index == XAW_LIST_NONE)
    toIdx = 0 ;
  else
    toIdx = returnStruct->list_index ;

  copyExample(markedExample.set, markedExample.idx, this->setIdx, toIdx) ;
  markedExample.display = this ;
  markedExample.set = this->setIdx ;
  markedExample.idx = toIdx ;

  this->state = DNStructureChange ;
  resetDisplay(this) ;
}
/**********************************************************************/
static void 	moveExampleAction(w, event, params, numParams)
  Widget	w ;
  XEvent	*event ;
  String	*params ;
  Cardinal	*numParams ;
{
  XawListReturnStruct	*returnStruct ;
  int			toIdx ;
  ActivationDisplay	this ;

  if (markedExample.display == NULL
      || markedExample.set < 0 || markedExample.idx < 0)
    return ;

  this = findActivationDisplay(w) ;
  if (this == NULL || this->listWidget == NULL)
    return ;

  returnStruct = XtSelectionGetSelected(this->listWidget) ;
  
  if (returnStruct == NULL || returnStruct->list_index == XAW_LIST_NONE)
    toIdx = 0 ;
  else
    toIdx = returnStruct->list_index ;

  moveExample(markedExample.set, markedExample.idx, this->setIdx, toIdx) ;
  markedExample.display = this ;
  markedExample.set = this->setIdx ;
  markedExample.idx = toIdx ;
  
  this->state = DNStructureChange ;
  resetDisplay(this) ;
  if (this != markedExample.display) {
    markedExample.display->state = DNStructureChange ;
    resetDisplay(markedExample.display) ;
  }
}
/**********************************************************************/
#endif
