
/**********************************************************************
 * $Id: unitCom.c,v 1.4 92/11/30 11:27:46 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <xerion/commands.h>
#include "groupCom.h"
#include "unitCom.h"

/***********************************************************************
 *	Name:		command_addUnit
 *	Description:	adds a unit to a group in the current network
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_addUnit - 0 on failure, 1 on success
 ***********************************************************************/
int	command_addUnit (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  char	*name, *groupName ;
  Group	group ;
  Unit	unit, bias ;

  IUsage("<name> <group>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("add a unit to an existing group in the current network") ;
    IHelp
      (IHelpArgs,
       "Creates a unit with name  <name> and adds  it to the group with name",
       "<group> in the current network.  The name must  be unique.  The bias",
       "unit will be automatically connected to the unit.",
       "EXAMPLES",
       "\txerion-> addUnit BiasUnit BiasGroup",
       "SEE ALSO",
       "deleteUnits, connectUnits, disconnectUnits",
       NULL);
    return 0;
  }

  if (tokc != 3)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  name      = tokv[1] ;
  groupName = tokv[2] ;

  /* Check for all the possible errors */
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  unit = unitFromName(currentNet, name) ;
  if (unit != NULL)
    IErrorAbort("Unit \"%s\" already exists.", unit->name) ;
  
  group = groupFromName(currentNet, groupName) ;
  if (group == NULL)
    IErrorAbort("Group \"%s\" is not in the current net.", groupName) ;

  /* Now create the unit int the requested group */
  unit = createUnit(name, group) ;

  if (unit == NULL)
    IErrorAbort("Cannot create unit \"%s\"", name) ;

  if (!(group->type & BIAS)) {
    bias = unitFromName(currentNet, "Bias") ;
    if (bias != NULL)
      if (connectUnits(bias, unit, 0) == NULL)
	IErrorAbort("Cannot connect \"%s\" to bias", unit->name) ;
  }

  markToRebuildDisplay(ACTIVITY_DISPLAY | CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_deleteUnits
 *	Description:	deletes units from the current network
 *			uses regex to match namess
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_deleteUnit - 0 on failure, 1 on success
 ***********************************************************************/
int	command_deleteUnits (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int	idx ;
  Unit	*unitArray, *unit ;

  IUsage("<unit1> [ <unit2> ... ]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("deletes units from the current network") ;
    IHelp
      (IHelpArgs,
       "Deletes the units with names <unit1>  etc. from the current network.",
       "Names may contain pattern  matching expressions of  the form used by",
       "grep.",
       "EXAMPLES",
       "To delete the unit with name \"InputUnit\",",
       "",
       "\txerion-> deleteUnits InputUnit",
       "",
       "To delete all units whose names begin with \"Hidden\",",
       "",
       "\txerion-> deleteUnits Hidden.*",
       "SEE ALSO",
       "addUnit, connectUnits, disconnectUnits",
       NULL);
    return 0;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  for (idx = 1 ; idx < tokc ; ++idx) {
    unitArray = unitsFromRegex(currentNet, tokv[idx]) ;
    if (unitArray == NULL)
      IErrorAbort("Bad string: \"%s\"", tokv[idx]) ;
    if (unitArray[0] == NULL)
      IError("No match: \"%s\"", tokv[idx]) ;

    for (unit = unitArray ; *unit != NULL ; ++unit) {
      destroyUnit (*unit) ;
    }

    free(unitArray) ;
  }
  markToRebuildDisplay(ACTIVITY_DISPLAY | CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_connectUnits
 *	Description:	connects two units togther
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_connectUnit - 0 on failure, 1 on success
 ***********************************************************************/
int	command_connectUnits (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int	linkMask, newMask ;
  Unit	*preUnitArray,  *preUnit ;
  Unit	*postUnitArray, *postUnit ;
  String name ;
  IUsage("[ -type TYPE [ -type TYPE ...] ] <unit1> <unit2> ...") ;
  if (GiveHelp(tokc)) {
    ISynopsis("connects two sets of units to each other") ;
    IHelp
      (IHelpArgs,
       "Connects <unit1> to <unit2>, where <unit1> and <unit2> are the names",
       "of two  units in the  current network.  Order, is important. <unit1>",
       "is the preceding unit  to <unit2>.   After  <unit1>  is connected to",
       "<unit2>, <unit2> is connected to <unit3> etc.",
       "",
       "Unit names may  use regular expressions  similar  to grep to connect",
       "more than two units.   Optionally, a type mask may be  specified for",
       "the type  of link to create.  At present there are no standard  link",
       "types.  ",
       "EXAMPLES",
       "\tconnectUnits \"Input\" \"Hidden.*\"",
       "",
       "will   connect the unit with  name  \"Input\" to all  units  whose name",
       "begins with the string \"Hidden\".",
       "SEE ALSO",
       "connectGroups, disconnectGroups, disconnectUnits",
       NULL);
    return 0;
  }

  name = tokv[0] ;

  if (tokc < 3)
    IErrorAbort(IPrintUsage(name, usage)) ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  linkMask = 0 ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-type", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      newMask = findClassMask(LINK_CLASS, *tokv) ;
      if (newMask == 0) {
	IErrorAbort("Unknown type: \"%s\".", *tokv) ;
	return 0 ;
      }
      linkMask |= newMask ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  /* get the list of pre groups assuming *tokv is a regular expression */
  preUnitArray = unitsFromRegex(currentNet, *tokv) ;
  if (preUnitArray == NULL)
    IErrorAbort("Bad string: \"%s\"", *tokv) ;
  if (preUnitArray[0] == NULL)
    IErrorAbort("No match: \"%s\"",   *tokv) ;

  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    /* get the list of post units assuming *tokv is a regular expression */
    postUnitArray = unitsFromRegex(currentNet, *tokv) ;
    if (postUnitArray == NULL)
      IErrorAbort("Bad string: \"%s\"", *tokv) ;
    if (postUnitArray[0] == NULL)
      IErrorAbort("No match: \"%s\"",   *tokv) ;

    for (postUnit = postUnitArray ; *postUnit != NULL ; ++postUnit)
      for (preUnit = preUnitArray ; *preUnit !=  NULL ; ++preUnit)
	if (NULL == connectUnits(*preUnit, *postUnit, linkMask))
	  IErrorAbort("Cannot connect units \"%s\" and \"%s\"",
		      (*preUnit)->name, (*postUnit)->name) ;

    free(preUnitArray) ;
    preUnitArray = postUnitArray ;
  }
  free(postUnitArray) ;

  markToRebuildDisplay(CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_disconnectUnits
 *	Description:	disconnects two units
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_disconnectUnit - 0 on failure, 1 on success
 ***********************************************************************/
int	command_disconnectUnits (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  Unit	*preUnitArray,  *preUnit ;
  Unit	*postUnitArray, *postUnit ;
  String name ;

  IUsage("<unit1> <unit2> ...") ;
  if (GiveHelp(tokc)) {
    ISynopsis("disconnects two sets of units from each other") ;
    IHelp
      (IHelpArgs,
       "Disconnects <unit1> from <unit2>, where <unit1>  and <unit2> are the",
       "names of two  units  in  the current  network.   Order is important.",
       "<unit1>  is  the  preceding unit to  <unit2>.   After  disconnecting",
       "<unit1>  from <unit2>,  disconnect  <unit2> from <unit3>, etc.  Unit",
       "names may use regular expressions similar to grep to disconnect more",
       "than two units.",
       "",
       "EXAMPLES",
       "\txerion-> disconnectUnits \"bias\" \"hidden.*\"",
       "",
       "will disconnect the unit with name  \"bias\" from all units whose name",
       "begins with the string \"hidden\".",
       "SEE ALSO",
       "connectGroups, connectUnits, disconnectGroups",
       NULL);
    return 0;
  }

  name = tokv[0] ;

  if (tokc < 3)
    IErrorAbort(IPrintUsage(name, usage)) ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  ++tokv, --tokc ;
  /* get the list of pre groups assuming *tokv is a regular expression */
  preUnitArray = unitsFromRegex(currentNet, *tokv) ;
  if (preUnitArray == NULL)
    IErrorAbort("Bad string: \"%s\"", *tokv) ;
  if (preUnitArray[0] == NULL)
    IErrorAbort("No match: \"%s\"",   *tokv) ;

  /* get the list of post units assuming *tokv is a regular expression */
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    postUnitArray = unitsFromRegex(currentNet, *tokv) ;
    if (postUnitArray == NULL)
      IErrorAbort("Bad string: \"%s\"", *tokv) ;
    if (postUnitArray[0] == NULL)
      IErrorAbort("No match: \"%s\"",   *tokv) ;

    for (postUnit = postUnitArray ; *postUnit != NULL ; ++postUnit)
      for (preUnit = preUnitArray ; *preUnit != NULL ; ++preUnit)
	disconnectUnits(*preUnit, *postUnit) ;

    free(preUnitArray) ;
    preUnitArray = postUnitArray ;
  }
  free(postUnitArray) ;

  markToRebuildDisplay(CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_unitLongName
 *	Description:	returns the long name of a unit (given short
 *			name (e.g. "Bias" -> "currentNet.group[0].unit[0].")
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_createUnit - 0 on failure, 1 on success
 ***********************************************************************/
int	command_unitLongName (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  char	*name, *unitName, *longName ;

  IUsage("<name>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("return the long name of a unit.") ;
    IHelp
      (IHelpArgs,
       "Returns the  long  name of a unit, given the unit's real name. For",
       "example,",
       "",
       "\txerion-> unitLongName \"Input.0\"",
       "",
       "would return the string:",
       "\tcurrentNet.group[1].unit[0]",
       "SEE ALSO",
       "groupLongName",
       NULL);
    return 0;
  }
  
  name = tokv[0] ;
  if (tokc != 2)
    IErrorAbort(IPrintUsage(name, usage)) ;
  unitName = tokv[1] ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  longName = unitLongName(unitName) ;

  if (longName) {
    fprintf(dout, "%s\n", longName) ;
    free((void *)longName) ;
  } else {
    IErrorAbort("Unknown unit: \"%s\"", unitName) ;
  }
  return 1 ;
}
/**********************************************************************/
String		unitLongName(name)
  String	name ;
{
  Group	group ;
  Unit	unit ;
  char	longName[128] ;
  char	*groupName ;
  int	idx ;

  if (currentNet == NULL || name == NULL)
    return NULL ;

  unit  = unitFromName(currentNet, name) ;
  if (unit == NULL || unit->group == NULL)
    return NULL ;
  group = unit->group ;
  
  groupName = groupLongName(group->name) ;
  if (groupName == NULL)
    return NULL ;

  for (idx = 0 ; idx < group->numUnits ; ++idx)
    if (strcmp(name, group->unit[idx]->name) == 0)
      break ;

  if (idx == group->numUnits)
    return NULL ;
  
  sprintf(longName, "%s.unit[%d]", groupName, idx) ;
  free((void *)groupName) ;

  return strdup(longName) ;
}
/**********************************************************************/
