
/**********************************************************************
 * $Id: minimizeCom.c,v 1.9 93/03/29 14:14:48 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

/**********************************************************************
 * Written by Tony Plate in June 1991, based on code written by
 * Rudi Mathon.
 **********************************************************************/

#include <stdio.h>
#include <math.h>
#include <errno.h>

#include <xerion/commands.h>
#include <xerion/option.h>
#include <xerion/minimize.h>
#include <xerion/checkgrad.h>
#include "minimizeCom.h"
#include "minimizeHelp.h"

#include <signal.h>

#define BAD_MZ_RETURN(mz) (   mz->mzResultCode == MZINTERRUPTED\
			   || mz->mzResultCode == MZSYSTEMERROR\
			   || mz->mzResultCode == MZBADMETHOD\
			   || mz->mzResultCode == MZBADRETURN)

static Minimize	getCurrentNetMinimize ARGS(()) ;
static String	getCurrentNetMzName   ARGS(()) ;

struct CHECK_GRAD check_grad;

Minimize (*getMinimizeRecord) ARGS((void)) = getCurrentNetMinimize ;

static int	isLineSearchMethod(method)
  int		method;
{
  return method==MZRAYSLS ||method==MZRUDISLS ||method==MZSLOPESEARCH;
}

/* ARGSUSED */
int command_initMinimize(tokc, tokv)
int tokc;
char *tokv[];
{
  Minimize	mz;
  struct ARG_DESC *argd;
  argd = StartArgs(tokv[0]);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("initialize the minimize record");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Allocate the minimize  record  and  set parameter  values to   their",
       "default values.",
       NULL);
    return 1;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  mz = getMinimizeRecord() ;

  if (mz == NULL)
    IErrorAbort("There is no minimize record.") ;

  checkParameters(mz);
  return 1 ;
}

/***********************************************************************
 *	Name:		command_minimize
 *	Description:	trains the current network on its training example
 *			set
 *	Parameters:	
 *		int	tokc    - number of command line tokens
 *		char	*tokv[] - vector of tokens
 *	Return Value:	
 *		int	command_mz - 0 on failure, 1 on success
 ***********************************************************************/
int	command_minimize(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  /*
   * modify the following assignments of functions to use with different
   * network functions
   */
  Minimize mz = NULL;
  int epochs = 0;
  int check = -1;
  int steepest = 0;
  int fixed = 0;
  int quickprop = 0;
  int deltabardelta = 0;
  int cg = 0;
  int cgRudi = 0;
  int cgRestart = 0;
  int momentum = 0;
  Real epsilon = 0.0;
  Real alpha = -1.0;
  Real qpEpsilon = -1.0;
  Real maxGrowthFactor = 0.0;
  int lsRay = 0;
  int lsTap = 0;
  int lsRudi = 0;
  int lsPlot = -1;
  int lsPlotPoints = -1;
  int slopeRatio = 0;
  int shoot = 0;
  int isSLen = 0;
  int isAsk = 0;
  int isNew = 0;
  int lsSummarize = -1;
  int lsVerbosity = -1;	/* -1 is a valid level for these */
  int mzVerbosity = -1;
  int lsDebug = -1;
  int mzDebug = -1;
  int maxFuncEvals = -1;
  int iterations = 0;
  int lsMaxFuncEvals = -1;
  int lsFlexFuncEvals = -1;
  int continueDirection = -1;
  Real acceptableFmin = MAXREAL;
  Real expectedFmin = MAXREAL;
  Real tolerance = 0.0;
  Real minFuncReduction = 0.0;
  Real maxSlopeRatio = 0.0;
  Real initialStepFactor = 0.0;
  Real backUpFactor = 0.0;
  Real stepBound = 0.0;
  Real extrapLimits = 0.0;
  Real interpLimits = 0.0;
  Real extensor  = 0.0;
  Real maxExtrapol = 0.0;
  int quasiNewton = -1;
  int localMinima = -1;
  int wolfeTest = -1;
  int reset = -1;
  Real gainIncrement = 0.0;
  Real gainDecrement = 0.0;
  int wobbleTest = -1;
  String lsLogFile = NULL;
  struct ARG_DESC *argd;

  argd = StartArgs(tokv[0]);
  Args(argd, "[<n iterations>%i]", &iterations);
  Args(argd, "[-PmaxFuncEvals<n>%i]", &maxFuncEvals);
  Args(argd, "[-Piterations <n>%i]", &iterations);
  Args(argd, "[-Prepetitions <n batches>%i]", &epochs);
  Args(argd, "[-Scheck %S%?]", &check,
       "check the consistency of the network functions");
  Args(argd, "[-Psteepest %P%?]", &steepest, "use steepest descent");
  Args(argd, "[-Pmomentum %P%? [<momentum-factor>%r]]", &momentum,
       "use steepest descent with momentum, optionally specify the momentum factor", &alpha);
  Args(argd, "[-PquickProp %P%? [<epsilon>%r <max-update-factor>%r]]",
       &quickprop,
       "use the quickprop direction method, optionally specify the epsilon and max-update-factor to use with it", &qpEpsilon, &maxGrowthFactor);
  Args(argd, "[-PdeltaBarDelta %P%? [<gain-increment>%r <gain-decrement>%r]]",
       &deltabardelta, "use the delta-bar-delta step size rule",
       &gainIncrement, &gainDecrement);
  Args(argd, "[-Pcg %P%?]", &cg,
       "use conjugate gradient directions (default)");
  Args(argd, "[-PcgRudi %P%?]", &cgRudi,
       "use Rudi's conjugate gradient directions");
  Args(argd, "[-PcgRestart %P%?]", &cgRestart,
       "use Rudi's conjugate gradient directions with restarts");
  Args(argd, "[-PfixedStep %P%? [<step-size>%r]]", &fixed,
       "use fixed step size, optionally specify the step-size", &epsilon);
  Args(argd, "[-PlsRay %P%?]", &lsRay, "use Ray's line search");
  Args(argd, "[-PlsRudi %P%?]", &lsRudi, "use Rudi's line search (default)");
  Args(argd, "[-PlsTap %P%?]", &lsTap, "use Tap's slope search");
  Args(argd, "[-SlsPlot %S%? [<n-points>%i]]", &lsPlot,
       "plot the function along the search direction", &lsPlotPoints);
  Args(argd, "[-Palpha <momentum-factor>%r %?]", &alpha,
       "can also be specified after the momentum option");
  Args(argd, "[-Pepsilon <step-size>%r %?]", &epsilon,
       "can also be specified after the fixedstep option");
  Args(argd, "[-PqpEpsilon <quick-prop-descent-component>%r %?]", &qpEpsilon,
       "can also be specified after the quickprop option");
  Args(argd, "[-PmaxGrowthFactor <quick-prop growth factor>%r %?]",
       &maxGrowthFactor, "can also be specified after the quickprop option");
  Args(argd, "[-Pgain <increment>%r <decrement>%r %?]",
       &gainIncrement, &gainDecrement,
       "can also be specified after the deltabardelta option");
  Args(argd, "[-PisDRatio %P%?]", &slopeRatio,
       "use slope-ratio initial steps for the linesearch");
  Args(argd, "[-PisShoot %P%?]", &shoot,
       "use \"shooting\" initial steps for the linesearch");
  Args(argd, "[-PisSLen %P%?]", &isSLen,
       "use \"search length\" initial steps for the linesearch");
  Args(argd, "[-PisAsk %P%?]", &isAsk,
       "ask the user for the initial steps for the linesearch");
  Args(argd, "[-PisNew %P%?]", &isNew,
       "customizable routine for the initial steps for the linesearch");
  Args(argd, "[-PlsSummarize %P%?]", &lsSummarize,
       "print summaries of the line searches");
  Args(argd, "[-PlsVerbosity %P%? [<level>%i]]", &lsVerbosity,
       "control the verbosity of the line search", &lsVerbosity);
  Args(argd, "[-PmzVerbosity %P%? [<level>%i]]", &mzVerbosity,
       "control the verbosity of the minimization", &mzVerbosity);
  Args(argd, "[-PlsDebug %P%? [<level>%i]]", &lsDebug,
       "control the debugging output of the line search", &lsDebug);
  Args(argd, "[-PmzDebug %P%? [<level>%i]]", &mzDebug,
       "control the debugging output of the minimization", &mzDebug);
  Args(argd, "[-PlsMaxFuncEvals <n>%i]", &lsMaxFuncEvals);
  Args(argd, "[-PlsFlexFuncEvals <n>%i]", &lsFlexFuncEvals);
  Args(argd, "[-Ptolerance <v>%r]", &tolerance);
  Args(argd, "[-PminFuncReduction <v>%r]", &minFuncReduction);
  Args(argd, "[-PmaxSlopeRatio <v>%r]", &maxSlopeRatio);
  Args(argd, "[-PacceptableFuncMin <v>%r]", &acceptableFmin);
  Args(argd, "[-PexpectedFuncMin <v>%r]", &expectedFmin);
  Args(argd, "[-PinitialStepFactor <v>%r]", &initialStepFactor);
  Args(argd, "[-PbackUpFactor <v>%r]", &backUpFactor);
  Args(argd, "[-PstepBound <v>%r]", &stepBound);
  Args(argd, "[-PextrapLimits <v>%r]", &extrapLimits);
  Args(argd, "[-PinterpLimits <v>%r]", &interpLimits);
  Args(argd, "[-Pextensor <v>%r]", &extensor);
  Args(argd, "[-PmaxExtrapol <v>%r]", &maxExtrapol);
  Args(argd, "[-PgainIncrement <v>%r]", &gainIncrement);
  Args(argd, "[-PgainDecrement <v>%r]", &gainDecrement);
  Args(argd, "[-PlogFile <filename>%s]", &lsLogFile);
  Args(argd, "[-ScontinueDirection %S]", &continueDirection);
  Args(argd, "[-SquasiNewton %S]", &quasiNewton);
  Args(argd, "[-SlocalMinima %S]", &localMinima);
  Args(argd, "[-SwolfeTest %S]", &wolfeTest);
  Args(argd, "[-Sreset %S%?]", &reset, "reset the iteration counter");
  Args(argd, "[-SwobbleTest %S [<level>%i]]", &wobbleTest, &wobbleTest);

  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("train a network on the examples using a minimization method");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Train the network with a maximum of n function evaluations.",
       "DIRECTION METHODS",
       "steepest",
       "momentum",
       "deltabardelta",
       "quickprop",
       "cg (conjugate gradient)",
       "cgRudi (Rudi's conjugate gradient)",
       "cgRestart (Rudi's conjugate gradient with restarts)",
       "STEP METHODS",
       "fixed",
       "lsRudi (Rudi's line search)",
       "lsRay (Ray's line search)",
       "lsTap (Tap's line search - not fully finished, but working)",
       "INITIAL STEP METHODS",
       "These methods are for choosing the initial step of each line search.",
       "(They have no effect with non-line search methods.",
       "isDRatio",
       "isShoot",
       "isSLen",
       "isAsk",
       "isNew",
       "",
       "If both methods are specified the shoot method  is used on the first",
       "line search, and the dratio method on subsequent ones.",
       "SEE ALSO",
       "steepest, momentum, deltaBarDelta, quickProp, cg, train",
       "",
       "For a complete description of the minimize command see the sman page",
       "(unix-> sman minimize, xerion-> sh sman minimize).",
       NULL);
    return 1 ;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  mz = getMinimizeRecord() ;

  if (mz == NULL)
    IErrorAbort("There is no minimize record.") ;

  if (epochs>0)
    mz->repetitionCount = epochs;

  if (check==0)
    mz->doNotCheckFunctions = 1;
  else if (check==1)
    mz->doNotCheckFunctions = 0;
  
  if (steepest && momentum) steepest = 0;
  if (deltabardelta && momentum) momentum = 0;

  if (steepest+momentum+cg+cgRudi+cgRestart+quickprop+deltabardelta == 0
      && mz->directionMethod == 0)
    cgRudi = 1;
  else
    if (steepest+momentum+cg+cgRudi+cgRestart+quickprop+deltabardelta>1)
      IErrorAbort("can use only one direction method");

  if (fixed + lsRay + lsRudi + lsTap > 1)
    IErrorAbort("can use only one step method");

  if (steepest) {
    mz->directionMethod = MZSTEEPEST;
    mz->stepMethod = MZFIXEDSTEP;
  } else if (momentum) {
    mz->directionMethod = MZMOMENTUM;
    mz->stepMethod = MZFIXEDSTEP;
  } else if (deltabardelta) {
    mz->directionMethod = MZDELTABARDELTA;
    mz->stepMethod = MZFIXEDSTEP;
  } else if (cg) {
    mz->directionMethod = MZCONJGRAD;
    if (!isLineSearchMethod(mz->stepMethod))
      mz->stepMethod = MZRAYSLS;
  } else if (cgRudi) {
    mz->directionMethod = MZRUDICG;
    if (!isLineSearchMethod(mz->stepMethod))
      mz->stepMethod = MZRAYSLS;
  } else if (cgRestart) {
    mz->directionMethod = MZCONJGRADRST;
    if (!isLineSearchMethod(mz->stepMethod))
      mz->stepMethod = MZRAYSLS;
  } else if (quickprop) {
    mz->directionMethod = MZQUICKPROP;
    mz->stepMethod = MZFIXEDSTEP;
    mz->epsilon    = 1.0;
  }

  if (fixed)
    mz->stepMethod = MZFIXEDSTEP;
  else if (lsRudi)
    mz->stepMethod = MZRUDISLS;
  else if (lsRay)
    mz->stepMethod = MZRAYSLS;
  else if (lsTap)
    mz->stepMethod = MZSLOPESEARCH;

  if (slopeRatio && shoot)
    mz->initialStepMethod = MZDRATIOSHOOT;
  else if (slopeRatio)
    mz->initialStepMethod = MZDRATIO;
  else if (shoot)
    mz->initialStepMethod = MZSHOOT;
  else if (isSLen)
    mz->initialStepMethod = MZISSLEN;
  else if (isAsk)
    mz->initialStepMethod = MZISASK;
  else if (isNew)
    mz->initialStepMethod = MZISNEW;

  if (lsPlot!=-1)
    mz->lsPlot = lsPlot;
  if (lsPlotPoints!=-1)
    mz->lsPlotPoints = lsPlotPoints;

  if (wobbleTest!=-1)
    mz->wobbleTest = wobbleTest;

  if (lsSummarize!=-1)
    mz->lsSummarize = lsSummarize;
  if (lsVerbosity!=-1)
    mz->lsVerbosity = lsVerbosity;
  if (mzVerbosity!=-1)
    mz->mzVerbosity = mzVerbosity;
  if (lsDebug!=-1)
    mz->lsDebug = lsDebug;
  if (mzDebug!=-1)
    mz->mzDebug = mzDebug;

  if (lsMaxFuncEvals>-1)
    mz->lsMaxFuncEvals = lsMaxFuncEvals;
  if (lsFlexFuncEvals>-1)
    mz->lsFlexFuncEvals = lsFlexFuncEvals;

  if (continueDirection != -1)
    mz->continueDirection = continueDirection;
  if (quasiNewton != -1)
    mz->quasiNewton = quasiNewton;
  if (localMinima != -1)
    mz->localMinima = localMinima;
  if (wolfeTest != -1)
    mz->wolfeTest = wolfeTest;
  if (acceptableFmin < (Real)(MAXREAL - MAXREAL/100.0))
    mz->acceptableFuncMin = acceptableFmin;
  if (expectedFmin < (Real)(MAXREAL - MAXREAL/100.0))
    mz->expectedFuncMin = expectedFmin;
  if (epsilon != 0.0)
    mz->epsilon = epsilon;
  if (alpha != -1.0)
    mz->alpha = alpha;
  if (qpEpsilon != -1.0)
    mz->qpEpsilon = qpEpsilon;
  if (maxGrowthFactor != 0.0)
    mz->maxGrowthFactor = maxGrowthFactor;
  if (gainIncrement != 0.0)
    mz->gainIncrement = gainIncrement;
  if (gainDecrement != 0.0)
    mz->gainDecrement = gainDecrement;
  if (tolerance != 0.0)
    mz->tolerance = tolerance;
  if (minFuncReduction != 0.0)
    mz->minFuncReduction = minFuncReduction;
  if (maxSlopeRatio != 0.0)
    mz->maxSlopeRatio = maxSlopeRatio;
  if (initialStepFactor != 0.0)
    mz->initialStepFactor = initialStepFactor;
  if (backUpFactor != 0.0)
    mz->backUpFactor = backUpFactor;
  if (stepBound != 0.0)
    mz->stepBound = stepBound;
  if (extrapLimits != 0.0)
    mz->extrapLimits = extrapLimits;
  if (interpLimits != 0.0)
    mz->interpLimits = interpLimits;
  if (extensor != 0.0)
    mz->extensor = extensor;
  if (maxExtrapol != 0.0)
    mz->maxExtrapol = maxExtrapol;
  if (gainIncrement != 0.0)
    mz->gainIncrement = gainIncrement;
  if (gainDecrement != 0.0)
    mz->gainDecrement = gainDecrement;
  if (lsLogFile != NULL)
    mz->lsLogFile = strdup(lsLogFile);
  if (reset==1)
    mz->xLen = 0.0;
  else if (reset==0)
    mz->xLen = -1.0;

  /* 
   * sensible pairings of direction & stepsize rules are checked by
   * checkParameters() (we have to fiddle with the blocking and
   * unblocking of X device events here, (by default everything is
   * blocked, but ours are handled in the traces, so no need to block
   * anything but those for the minimize display))
   */
  unblockDeviceEvents() ;
  blockMiniDisplayDeviceEvents() ;
  minimize(mz, maxFuncEvals, iterations) ;
  unblockMiniDisplayDeviceEvents() ;
  blockDeviceEvents() ;

  fprintf(dout, "minimize: code= %d nFE= %d f= %.8g\n",
	  mz->mzResultCode, mz->nFuncEvals, mz->E);
  fprintf(dout, "minimize: %s\n", mz->mzResult);

  if (BAD_MZ_RETURN(mz))
    return 0;
  else
    return 1;
}

/*********************************************************************
 *	Name:		getCurrentNetMinimize
 *	Description:	return the minimize record of the currentNet
 *	Parameters:	NONE
 *	Return Value:
 *	  static Minimize	getCurrentNetMinimize - ibid
 *********************************************************************/
static Minimize	getCurrentNetMinimize() 
{
  if (currentNet == NULL)
    return NULL ;

  if (currentNet->mz == NULL)
    currentNet->mz = initMinimize(currentNet->mz, 0);

  syncValues(currentNet, VectorFromWeights) ;

  return currentNet->mz ;
}
/********************************************************************/


/***********************************************************************
 *	Name:		command_steepest
 *	Description:	trains the current network on its training example
 *			set using steepest descent minimization
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_steepest - 0 on failure, 1 on success
 ***********************************************************************/
int command_steepest (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  ExampleSet	exampleSet ;
  int		repSize   = -1 ;
  int		numEpochs = -1 ;
  Real		epsilon   = 0.0 ;
  char		command[256] ;
  struct ARG_DESC *argd;

  argd = StartArgs(tokv[0]);
  Args(argd, "[-Pepsilon <e>%r]", &epsilon) ;
  Args(argd, "[-Prepetitions <m>%i]", &repSize);
  Args(argd, "[<n>%i]", &numEpochs);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("train a network on the examples using steepest descent");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Train  the  network  using the   training  example set and  steepest",
       "descent optimization, updating the weights  (after each presentation",
       "of a  batch of examples) <n>  times.  Steepest descent training sets",
       "the search  direction to be the   negative of  the  gradient.  Fixed",
       "steps of size <e> are taken along this direction.",
       "",
       "The -rep option causes training statistics to be printed after every",
       "<m> iterations. Its default value is 1.",
       "",
       "The size  of each  example  batch is determined by   the \"batchSize\"",
       "field in the currentNet.",
       "",
       "This command is equivalent to:",
       "  minimize -steepest -fixedStep -epsilon <e> -iterations <n> -rep <m>",
       "SEE ALSO",
       "minimize, momentum, deltaBarDelta, quickProp, cg, test, validate",
       NULL);
    return 0;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  if (currentNet->calculateErrorDerivProc == NULL)
    IErrorAbort("There is no train procedure for net \"%s\".", 
		currentNet->name) ;

  exampleSet = netGetExampleSet(currentNet, TRAINING) ;
  if (exampleSet == NULL)
    IErrorAbort("There is no training example set for net \"%s\".", 
		currentNet->name) ;

  strcpy(command, "minimize -steepest -fixedStep") ;
  if (epsilon > 0.0)
    sprintf(command + strlen(command), " -epsilon %f", epsilon) ;
  if (numEpochs >= 0)
    sprintf(command + strlen(command), " -iterations %d", numEpochs) ;
  if (repSize >= 0)
    sprintf(command + strlen(command), " -repetitions %d", repSize) ;

  if (IDoCommandLine(command) == 0)
    return 0 ;

  fprintf(dout,"epoch = %d,\terror = %-8g",
	  currentNet->currentEpoch, currentNet->error);
  if (currentNet->cost != 0.0)
    fprintf(dout,"\textra cost = %-8g", currentNet->cost);
  putc('\n', dout) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_momentum
 *	Description:	trains the current network on its training example
 *			set using momentum descent minimization
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_momentum - 0 on failure, 1 on success
 ***********************************************************************/
int command_momentum (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  ExampleSet	exampleSet ;
  int		repSize   = -1 ;
  int		numEpochs = -1 ;
  Real		epsilon   = 0.0 ;
  Real		alpha	  = -1.0 ;
  char		command[256] ;
  struct ARG_DESC *argd;

  argd = StartArgs(tokv[0]);
  Args(argd, "[-Pepsilon <e>%r]", &epsilon) ;
  Args(argd, "[-Palpha <a>%r]", &alpha) ;
  Args(argd, "[-Prepetitions <m>%i]", &repSize);
  Args(argd, "[<n>%i]", &numEpochs);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("train a network on the examples using momentum descent");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Train the network   using  the  training  example set  and  momentum",
       "descent optimization, updating the  weights (after each presentation",
       "of a batch of  examples) <n> times.  Momentum  descent training sets",
       "the  search direction to   be  the  steepest descent plus a decaying",
       "average of the previous descent directions.  Fixed steps of size <e>",
       "are taken along this direction.  This strategy is supposed to reduce",
       "sloshing  in the  ravines of the  error surface,  and is often quite",
       "effective at speeding the rate of convergence.",
       "",
       "The -alpha option specifies the rate of  decay. Its default value is",
       "0.9.  The  sensible range of  values for alpha is  from zero to one.",
       "Typical values are 0.9 for networks  with only a few connections and",
       "up to 0.999 for networks with many connections.",
       "",
       "The -rep option causes training statistics to be printed after every",
       "<m> iterations. Its default value is 1.",
       "",
       "The size  of each  example  batch is determined by   the \"batchSize\"",
       "field in the currentNet.",
       "",
       "This command is equivalent to:",
       "  minimize -momentum -fixedStep -epsilon <e> -alpha <a>",
       "           -iterations <n> -rep <m>",
       "SEE ALSO",
       "minimize, steepest, deltaBarDelta, quickProp, cg, test, validate",
       NULL);
    return 0;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  if (currentNet->calculateErrorDerivProc == NULL)
    IErrorAbort("There is no train procedure for net \"%s\".", 
		currentNet->name) ;

  exampleSet = netGetExampleSet(currentNet, TRAINING) ;
  if (exampleSet == NULL)
    IErrorAbort("There is no training example set for net \"%s\".", 
		currentNet->name) ;

  strcpy(command, "minimize -momentum -fixedStep") ;
  if (epsilon > 0.0)
    sprintf(command + strlen(command), " -epsilon %f", epsilon) ;
  if (alpha >= 0.0)
    sprintf(command + strlen(command), " -alpha %f", alpha) ;
  if (numEpochs >= 0)
    sprintf(command + strlen(command), " -iterations %d", numEpochs) ;
  if (repSize >= 0)
    sprintf(command + strlen(command), " -repetitions %d", repSize) ;

  if (IDoCommandLine(command) == 0)
    return 0 ;

  fprintf(dout,"epoch = %d,\terror = %-8g",
	  currentNet->currentEpoch, currentNet->error);
  if (currentNet->cost != 0.0)
    fprintf(dout,"\textra cost = %-8g", currentNet->cost);
  putc('\n', dout) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_deltaBarDelta
 *	Description:	trains the current network on its training example
 *			set using deltaBarDelta descent minimization
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_deltaBarDelta - 0 on failure, 1 on success
 ***********************************************************************/
int command_deltaBarDelta (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  ExampleSet	exampleSet ;
  int		repSize   = -1 ;
  int		numEpochs = -1 ;
  Real		epsilon   = 0.0 ;
  Real		alpha	  = -1.0 ;
  Real		gainIncrement = -1.0 ;
  Real		gainDecrement = -1.0 ;
  char		command[256] ;
  struct ARG_DESC *argd;

  argd = StartArgs(tokv[0]);
  Args(argd, "[-Pepsilon <e>%r]", &epsilon) ;
  Args(argd, "[-Palpha <a>%r]", &alpha) ;
  Args(argd, "[-Pgain <i>%r <d>%r]",
       &gainIncrement, &gainDecrement) ;
  Args(argd, "[-Prepetitions <m>%i]", &repSize);
  Args(argd, "[<n>%i]", &numEpochs);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("train a network on the examples using deltaBarDelta");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Train the network using the training  example set  and deltaBarDelta",
       "descent optimization, updating the weights  (after each presentation",
       "of a  batch of examples)  <n>  times.  The  deltaBarDelta  direction",
       "method is  an attempt to  improve  on   straight momentum by  taking",
       "larger steps (gain) for those components of the gradient whose signs",
       "are unchanging.   Fixed  steps  of size <e>  are  taken  along  this",
       "direction.",
       "",
       "The  -alpha option specifies  the rate  of decay <a> (as in momentum",
       "descent).  Its default value is 0.9.   The  sensible range of values",
       "for alpha is from zero to one.  Typical  values are 0.9 for networks",
       "with only a few connections  and up to 0.999  for networks with many",
       "connections.",
       "",
       "The -gain option specifies the  additive gain increment  <i> and the",
       "multiplicative gain decrement <d> .  The increment should be a small",
       "positive value (default 0.1), and the decrement should be a positive",
       "value less than one (default 0.9).  The gain increment and decrement",
       "should be balanced so that the  decrement is  approximately equal to",
       "1.0 - increment.   This balance ensures that the  gain stays  around",
       "one when gradients are changing randomly.",
       "",
       "The -rep option causes training statistics to be printed after every",
       "<m> iterations. Its default value is 1.",
       "",
       "The size  of each  example  batch is determined by   the \"batchSize\"",
       "field in the currentNet.",
       "",
       "This command is equivalent to:",
       "  minimize -deltaBarDelta -fixedStep -epsilon <e> -alpha <a>",
       "           -gain i d -iterations <n> -rep <m>",
       "NOTES",
       "This method was  introduced by R.   Jacobs  in \"Increased  Rates  of",
       "Convergence Through   Learning  Rate   Adaptation\", COINS-TR-87-117,",
       "1987, University of Massachusetts.",
       "SEE ALSO",
       "minimize, steepest, momentum, quickProp, cg, test, validate",
       NULL);
    return 0;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  if (currentNet->calculateErrorDerivProc == NULL)
    IErrorAbort("There is no train procedure for net \"%s\".", 
		currentNet->name) ;

  exampleSet = netGetExampleSet(currentNet, TRAINING) ;
  if (exampleSet == NULL)
    IErrorAbort("There is no training example set for net \"%s\".", 
		currentNet->name) ;

  strcpy(command, "minimize -deltaBarDelta -fixedStep") ;
  if (epsilon > 0.0)
    sprintf(command + strlen(command), " -epsilon %f", epsilon) ;
  if (alpha >= 0.0)
    sprintf(command + strlen(command), " -alpha %f", alpha) ;
  if (gainIncrement >= 0.0)
    sprintf(command + strlen(command), " -gainIncrement %f", gainIncrement) ;
  if (gainDecrement >= 0.0)
    sprintf(command + strlen(command), " -gainDecrement %f", gainDecrement) ;
  if (numEpochs >= 0)
    sprintf(command + strlen(command), " -iterations %d", numEpochs) ;
  if (repSize >= 0)
    sprintf(command + strlen(command), " -repetitions %d", repSize) ;

  if (IDoCommandLine(command) == 0)
    return 0 ;

  fprintf(dout,"epoch = %d,\terror = %-8g",
	  currentNet->currentEpoch, currentNet->error);
  if (currentNet->cost != 0.0)
    fprintf(dout,"\textra cost = %-8g", currentNet->cost);
  putc('\n', dout) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_quickProp
 *	Description:	trains the current network on its training example
 *			set using quickProp descent minimization
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_quickProp - 0 on failure, 1 on success
 ***********************************************************************/
int command_quickProp (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  ExampleSet	exampleSet ;
  int		repSize   = -1 ;
  int		numEpochs = -1 ;
  Real		epsilon   = 0.0 ;
  Real		qpEpsilon = -1.0 ;
  Real		maxGrowth = -1.0 ;
  char		command[256] ;
  struct ARG_DESC *argd;

  argd = StartArgs(tokv[0]);
  Args(argd, "[-Pepsilon <e>%r]", &epsilon) ;
  Args(argd, "[-PqpEpsilon <q>%r]", &qpEpsilon) ;
  Args(argd, "[-PmaxGrowth <x>%r]", &maxGrowth) ;
  Args(argd, "[-Prepetitions <m>%i]", &repSize);
  Args(argd, "[<n>%i]", &numEpochs);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("train a network on the examples using quickProp descent");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Train the network using the  training example set  and the quickProp",
       "descent method, updating the weights  (after  each presentation of a",
       "batch of examples) <n> times.  Quickprop is an attempt to get higher",
       "rates of  convergence than are  possible with gradient  descent.  In",
       "this  method   the  current  and     previous  component values  and",
       "derivatives are modeled as independent quadratics and a jump is made",
       "to the projected minimum of each quadratic.  Fixed steps of size <e>",
       "are taken along this direction (default value 1.0).",
       "",
       "The  -qpEpsilon option specifies  the steepest  descent component to",
       "the direction <q>.    This  kick-starts  the  method and    prevents",
       "stalling.  Sensible  values  are small   positive  numbers.  Fahlman",
       "suggests a value of 0.1   for this parameter  for  small nets.   The",
       "default value is  0.3/sqrt(n)  (where  n  is the number   of network",
       "parameters).",
       "",
       "The -maxGrowth option specifies maximum permissible  increase in the",
       "step taken on   a particular  component  (default  1.75).   Sensible",
       "values are positive.   A zero value is  not possible because the max",
       "growth    factor  is used  as   the   growth   factor when quadratic",
       "interpolation fails.",
       "",
       "The -rep option causes training statistics to be printed after every",
       "<m> iterations. Its default value is 1.",
       "",
       "The size  of each  example  batch is determined by   the \"batchSize\"",
       "field in the currentNet.",
       "",
       "This command is equivalent to:",
       "  minimize -quickProp -fixedStep -epsilon <e> -qpEpsilon <q> ",
       "           -maxGrowth <x> -iterations <n> -rep <m>",
       "NOTES",
       "This method was  developed by  S.  Fahlman and  is  discussed in  S.",
       "Fahlman \"An Empirical Study of  Learning Speed in  Back- Propagation",
       "Networks\", 1988, CMU-CS-88-162, Carnegie Mellon University.",
       "SEE ALSO",
       "minimize, steepest, momentum, deltaBarDelta, cg, test, validate",
       NULL);
    return 0;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  if (currentNet->calculateErrorDerivProc == NULL)
    IErrorAbort("There is no train procedure for net \"%s\".", 
		currentNet->name) ;

  exampleSet = netGetExampleSet(currentNet, TRAINING) ;
  if (exampleSet == NULL)
    IErrorAbort("There is no training example set for net \"%s\".", 
		currentNet->name) ;

  strcpy(command, "minimize -quickProp -fixedStep") ;
  if (epsilon > 0.0)
    sprintf(command + strlen(command), " -epsilon %f", epsilon) ;
  if (qpEpsilon > 0.0)
    sprintf(command + strlen(command), " -qpEpsilon %f", qpEpsilon) ;
  if (maxGrowth >= 0.0)
    sprintf(command + strlen(command), " -maxGrowth %f", maxGrowth) ;
  if (numEpochs >= 0)
    sprintf(command + strlen(command), " -iterations %d", numEpochs) ;
  if (repSize >= 0)
    sprintf(command + strlen(command), " -repetitions %d", repSize) ;

  if (IDoCommandLine(command) == 0)
    return 0 ;

  fprintf(dout,"epoch = %d,\terror = %-8g",
	  currentNet->currentEpoch, currentNet->error);
  if (currentNet->cost != 0.0)
    fprintf(dout,"\textra cost = %-8g", currentNet->cost);
  putc('\n', dout) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_cg
 *	Description:	trains the current network on its training example
 *			set using conjugate gradient optimization
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_cg - 0 on failure, 1 on success
 ***********************************************************************/
int command_cg (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  ExampleSet	exampleSet ;
  int		cg = 0, cgRudi = 0, cgRestart = 0 ;
  int		repSize   = -1 ;
  int		numEpochs = -1 ;
  char		command[256] ;
  struct ARG_DESC *argd;

  argd = StartArgs(tokv[0]);
  Args(argd, "[-Pcg %P%?]", &cg,
       "use conjugate gradient directions (default)");
  Args(argd, "[-Prudi %P%?]", &cgRudi,
       "use Rudi's conjugate gradient directions");
  Args(argd, "[-Prestart %P%?]", &cgRestart,
       "use Rudi's conjugate gradient directions with restarts");
  Args(argd, "[-Prepetitions <m>%i]", &repSize);
  Args(argd, "[<n>%i]", &numEpochs);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("train a network using conjugate gradient descent");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Train the  network  using the training   example set  and  conjugate",
       "gradient optimization,  updating  the weights  <n> times.  Conjugate",
       "gradient optimization chooses directions  that   are  conjugate   to",
       "previous directions.  A line search is used with this method because",
       "directions change radically, and consequently so do appropriate step",
       "sizes.  The conjugate gradient direction methods have no parameters.",
       "",
       "The -cg option  (default)  causes the standard   conjugate  gradient",
       "method to be used.  The method computes the following:",
       "",
       "          s' = -g' + s * (g'-g).g' / g.g",
       "",
       "where",
       "",
       "          s' is the new search direction,",
       "          s is the previous search direction,",
       "          g' is the current gradient, and",
       "          g is the previous gradient.",
       "",
       "The conjugate gradient method works by finding the  minimum value of",
       "the function along a  series  of conjugate search  directions.   The",
       "line  search is necessary  to find the   minimum value, and stricter",
       "line searches  will result   in shorter series of search  directions",
       "(but stricter line searches take more function evaluations.)",
       "",
       "Rudi's conjugate gradient direction method (-rudi) is a modification",
       "of the standard  method.  This  original code  for this  method  was",
       "written by Rudi Mathon.  This method often gives faster convergence.",
       "",
       "The method computes the following:",
       "",
       "          u1 = s.g' / (g'-g).(g'-g)",
       "          u2 = (g'-g).g' / (g'-g).(g'-g) - 2s.g' / (g'-g).s",
       "          u3 = s.g' / (g'-g).(g'-g)",
       "          s' = u1 * (g'-g) + u2 * s - u3 * g'",
       "",
       "where",
       "",
       "          s' is the new search direction,",
       "          s is the step taken in the previous search direction,",
       "          g' is the current gradient, and",
       "          g is the previous gradient.",
       "",
       "The  conjugate gradient  with  restarts direction method  (-restart)",
       "chooses directions that   are conjugate to  previous directions, and",
       "does a  form of restarting.  This original code  for this method was",
       "written by Rudi Mathon.",
       "",
       "The -rep option causes training statistics to be printed after every",
       "<m> iterations. Its default value is 1.",
       "",
       "The size  of each  example  batch is determined by   the \"batchSize\"",
       "field in the currentNet.",
       "",
       "This command is equivalent to:",
       "  minimize [ -cg | -cgRudi | -cgRestarg ] -iterations <n> -rep <m>",
       "NOTES",
       "For more complex  minimization or control of  the line searches, use",
       "the \"minimize\" command.",
       "",
       "The code for the -cg method was written to follow the description in",
       "\"Numerical Recipes in  C\", by W.  Press, B. Flannery, S.  Teukolsky,",
       "and W.  Vetterling, Cambridge University Press, 1988.",
       "SEE ALSO",
       "minimize, steepest, momentum, deltaBarDelta, quickProp, test, validate",
       NULL);
    return 0;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (cg + cgRestart + cgRudi > 1)
    IErrorAbort("Only one of cg, rudi, restart may be specified") ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  if (currentNet->calculateErrorDerivProc == NULL)
    IErrorAbort("There is no train procedure for net \"%s\".", 
		currentNet->name) ;

  exampleSet = netGetExampleSet(currentNet, TRAINING) ;
  if (exampleSet == NULL)
    IErrorAbort("There is no training example set for net \"%s\".", 
		currentNet->name) ;

  strcpy(command, "minimize") ;
  if (cgRudi)
    sprintf(command + strlen(command), " -cgRudi") ;
  else if (cgRestart)
    sprintf(command + strlen(command), " -cgRestart") ;
  else
    sprintf(command + strlen(command), " -cg") ;

  if (numEpochs >= 0)
    sprintf(command + strlen(command), " -iterations %d", numEpochs) ;
  if (repSize >= 0)
    sprintf(command + strlen(command), " -repetitions %d", repSize) ;

  if (IDoCommandLine(command) == 0)
    return 0 ;

  fprintf(dout,"epoch = %d,\terror = %-8g",
	  currentNet->currentEpoch, currentNet->error);
  if (currentNet->cost != 0.0)
    fprintf(dout,"\textra cost = %-8g", currentNet->cost);
  putc('\n', dout) ;

  return 1 ;
}
/**********************************************************************/


/* ARGSUSED */
int command_checkGrad(tokc, tokv)
int tokc;
char *tokv[];
{
  static Net	lastNet ;

  int i, n;
  int check_verbosity = 1;
  int printall = 1;
  Real allow = -1.0;
  Real epsilon = -1.0;
  Real epswarn = -1.0;
  Real gwarn = -1.0;
  int check = -1;
  int verbosity = 1;
  int method = -1;
  int gradients = 1;
  Real *x;
  struct ARG_DESC *argd;
  Minimize	mz ;

  argd = StartArgs(tokv[0]);
  Args(argd, "[-Pverbosity %P [<level>%i]]", &verbosity, &verbosity);
  Args(argd, "[-Pmethod <n>%i]", &method);
  Args(argd, "[-Sgradients %S]", &gradients);
  Args(argd, "[-Scheck %S [<verbosity-level>%i]]", &check, &check_verbosity);
  Args(argd, "[-Sprintall %S]", &printall);
  Args(argd, "[-Pallow <% difference>%r]", &allow);
  Args(argd, "[-Pepsilon <eps>%r]", &epsilon);
  Args(argd, "[-PwarnEpsilon <level>%r]", &epswarn);
  Args(argd, "[-PwarnGradient <level>%r]", &gwarn);
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("checks partial derivatives for accuracy");
    IHelp
      (tokc, tokv[0], NULL, synopsis,
       "Calls the appropriate  gradient  calculation routine from  the net's",
       "minimize record to get all gradients, then tries changing parameters",
       "one at  a   time, recalculating  the error  after  each change.  The",
       "observed  error  is   compared with  the   error  expected  from the",
       "gradient.",
       "",
       "The methods for choosing the change are:",
       "    0: dx = eps                         (fixed step)",
       "    1: dx = eps * dEdx                  (steepest descent)",
       "    2: dx = eps * x                     (eps change in x)",
       "    3: dx = eps / dEdx                  (eps change in E)",
       "    4: dx = maximum of method 1 and 2   (smallest change that",
       "                                        is at least eps in",
       "                                        both x and E",
       "",
       "The  default is  method 0,   because  it produces the  most reliable",
       "change - one  that is large enough  to be  accurately represented in",
       "both  x and  E and   small enough that   E changes  linearly with  x",
       "according to dEdx",
       "SEE ALSO",
       "minimize",
       "",
       "For a  complete  description of the  checkGrad command  see the sman",
       "page (unix-> sman checkGrad, xerion-> sh sman checkGrad).",
       NULL);
    return 1;
  }	
  (void) ParseArgs(argd, tokc, tokv, 0);

  mz = getMinimizeRecord() ;

  if (mz == NULL) 
    IErrorAbort("There is no minimize record.") ;

  n = mz->getNVars(mz);
  x = ITempCalloc(Real, n);

  if (lastNet != currentNet && check!=0 || check==1) {
    checkOutFunctions(mz, mz->getNVars, mz->getValues, mz->setValues,
		      mz->fEval, mz->gEval, mz->fgEval, mz->valueName,
		      "checkGrad", check_verbosity);
    lastNet = currentNet;
  }
  
  if (gradients) {
    mz->getValues(mz, n, x);

    if (epsilon!=-1.0)
      check_grad.epsilon = epsilon;
    if (epswarn!=-1.0)
      check_grad.epswarn = epswarn;
    if (gwarn!=-1.0)
      check_grad.gwarn = gwarn;
    if (allow!=-1.0)
      check_grad.criterion = 1.0-allow/100.0;

    if (verbosity!=-1)
      check_grad.verbosity = verbosity;
    if (method!=-1)
      check_grad.method = method;
    check_grad.printall = printall;
    
    i = checkGradients(&check_grad, n, x, NULL,
		       mz, mz->fEval, mz->gEval, mz->fgEval, mz->valueName);
    if (i>0) {
      fprintf(dout, "Failed the gradient check for %d parameters.\n", i);
      if (verbosity < 3)
	fprintf(dout, "Run checkGrad again with flags \"%cverbosity 3\" for complete information.\n\t(current verbosity: %d)\n",
		true(plus_minus_options) ? '+' : '-', verbosity);
    } else if (printall || verbosity)
      fprintf(dout, "The gradient checks succeeded for all parameters.\n");
    /* restore the values */
    mz->setValues(mz, n, x);
  }
  return 1 ;
}
/********************************************************************/

    
/********************************************************************/
typedef struct _ErrorCheck {
  Real	maxError ;
  Real	allowableError ;
  int	numOutside ;
} ErrorCheck ;
/********************************************************************/
static void	findMaxError ARGS((Unit, void *)) ;
/********************************************************************/

/***********************************************************************
 *	Name:		command_rangeTerminate
 *	Description:	stop training if all outputs are within range of
 *			their targets.
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_rangeTerminate - 0 on failure, 1 on success
 ***********************************************************************/
int	command_rangeTerminate(tokc, tokv)
  int	tokc ;
  char	**tokv ;
{
  ErrorCheck	errorCheck ;
  String	name ;
  Mask		setMask ;
  int		setIdx, exampleIdx ;
  Boolean	print = FALSE ;

  IUsage("[-type <example set mask>] [-print] <range>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("stop minimizing if all outputs are within range of their targets") ;
    IHelp
      (IHelpArgs,
       "RangeTerminate  stops  the  minimize  routine  if  all  output  unit",
       "activations  are within <range> of their targets.   By  default  the",
       "command uses  the TRAINING example set  when calculating the errors,",
       "but the TESTING, or VALIDATION sets  may also be used by giving  the",
       "\"-type\" option.",
       "",
       "The \"-print\" option causes the command to print out the total number",
       "of times any unit's  activation is outside the range (across all the",
       "examples), followed by the maximum error.",
       "",
       "EXAMPLE",
       "To  use  the command,  put  it on  one  of the  minimize traces. For",
       "example, to stop minimizing when the maximum unit error is less than",
       "0.2, and to  print the maximum  error  after  every progress report,",
       "use:",
       "",
       "xerion-> addTrace minimizeEndIteration  \"rangeTerminate 0.2\"",
       "xerion-> addTrace minimizeEndRepetition \"rangeTerminate -p 0.2\"",
       NULL) ;
    return 1 ;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  /* Parse any command line options */
  setMask = 0 ;
  name = *tokv ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-type", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      setMask = findClassMask(EXAMPLESET_CLASS, *tokv) ;
      if (setMask == 0)
	IErrorAbort("Unknown example set: \"%s\".", *tokv) ;
    } else if (strncmp(*tokv, "-print", strlen(*tokv)) == 0) {
      print = TRUE ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }
  if (setMask == 0)
    setMask = TRAINING ;

  switch(setMask) {
  case TRAINING:
    setIdx = 0 ;
    break ;
  case TESTING:
    setIdx = 1 ;
    break ;
  case VALIDATION:
    setIdx = 2 ;
    break ;
  default:
    IErrorAbort("Unknown example set type: \"0x%0x\".", setMask) ;
    break ;
  }

  if (tokc == 0)
    errorCheck.allowableError = 0.2 ;
  else if (tokc == 1)
    errorCheck.allowableError = atof(*tokv) ;
  else
    IErrorAbort(IPrintUsage(name, usage)) ;

  /***********************
   * Now, do the examples
   */
  errorCheck.maxError   = 0.0 ;
  errorCheck.numOutside = 0 ;
  for (exampleIdx = 0 ; doExample(setIdx, exampleIdx) >= 0 ; ++exampleIdx) 
    netForAllUnits(currentNet, OUTPUT, findMaxError, &errorCheck) ;

  if (exampleIdx == 0)
    IErrorAbort("Example Set is empty") ;

  if (print)
    fprintf(dout, "%d (%f)\n", errorCheck.numOutside, errorCheck.maxError) ;

  if (errorCheck.numOutside == 0) {
    fprintf(dout, "All outputs are within %g of their targets.\n", 
	    errorCheck.allowableError) ;
    ++(currentNet->mz->stopFlag) ;
  }

  return 1 ;
}  
/********************************************************************/


/***********************************************************************/
static void	findMaxError(unit, data)
  Unit		unit ;
  void		*data ;
{
  ErrorCheck	*errorCheck = (ErrorCheck *)data ;
  Real		error = fabs(unit->target - unit->output) ;

  if (error > errorCheck->maxError)
    errorCheck->maxError = error ;

  if (error > errorCheck->allowableError)
    ++errorCheck->numOutside ;
}
/********************************************************************/
