
/**********************************************************************
 * $Id: layoutCom.c,v 1.7 93/03/09 12:16:10 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <xerion/commands.h>
#include <xerion/layout.h>
#include <xerion/lex.h>
#include <xerion/varsub.h>

static int	layoutCommandLoop ARGS((FILE	*inStream, FILE	 *outStream)) ;
static void	setLayout ARGS((Unit	unit, void	*data)) ;

static void	resetInput     ARGS((FILE  *in)) ;
static char	*readInputLine ARGS((FILE  *in)) ;
static int	getInputLine   ARGS((FILE  *in, int  *argc, char  ***argv)) ;
static void	parseInputLine ARGS((char  *in, int  *argc, char  ***argv)) ;

/***********************************************************************
 *	Name:		command_doLayout
 *	Description:	reads in a layout description file and
 *			parses it. It then actually sets the values
 *			used to display the network.
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_doLayout - 0 on failure, 1 on success
 ***********************************************************************/
int	command_doLayout (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  FILE	*inStream ;

  IUsage("[file]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("read an input file containing layout commands");
    IHelp
      (IHelpArgs,
       "doLayout  takes the name of  a file containing  layout  commands and",
       "parses it to set the  necessary   fields  in the  Net and Unit  data",
       "structures. If no file name is given, it reads from standard input.",
       "",
       "A layout file is of the following form:",
       "",
       "activity-layout  cell-size <c>  margin <m>  per-row <n>",
       "[repeat-count] <layout commands>",
       "[repeat-count] <layout commands>",
       "...",
       "minor-layout  cell-size <c> margin <m>  per-row <n> show-links <o>",
       "[repeat-count] <layout commands>",
       "[repeat-count] <layout commands>",
       "...",
       "minor-layout  for-unit <unit>",
       "[repeat-count] <layout commands>",
       "[repeat-count] <layout commands>",
       "...",
       "minor-layout  for-group <group>",
       "[repeat-count] <layout commands>",
       "[repeat-count] <layout commands>",
       "...",
       "major-layout  block-width <w>  block-height <h>  margin <m> per-row <n>",
       "[repeat-count] <layout commands>",
       "[repeat-count] <layout commands>",
       "...",
       "",
       "The activity-layout  command specifies the size of each  unit in the",
       "activity display,  as  well as  the margin between  units,  and  the",
       "number of units in  each row. The values <c>, <m>, and <n> are given",
       "in pixels.",
       "",
       "The   minor-layout   command   specifies  the  same  values  as  the",
       "activity-layout, but for the connection  display.   A  default minor",
       "layout (used for all connections) must be specified.  As well, minor",
       "layouts for the incoming connections to specific units may be given.",
       "These layouts are identified by one  of the  key words \"for-unit\" or",
       "\"for-group\" followed by  the  name of  the  unit or group  of units.",
       "These layouts  must be  given  after  the default minor-layout,  and",
       "simply over-ride the default values.",
       "",
       "The \"show-links\"  option controls which links are shown in the minor",
       "layouts. The  string <o> may be  one of: \"incoming\" - show only  the",
       "incoming  links to the units; \"outgoing\"  - show  only  the outgoing",
       "links from the units; \"all\" - show all connections of the units.",
       "",
       "The   major-layout command specifies  the  dimensions of  the  large",
       "blocks containing the minor layouts in the  connection display.  The",
       "block-width  and   block-height commands are  optional.  If omitted,",
       "they are automatically calculated from the maximum minor dimensions.",
       "",
       "valid layout commands are: ",
       "",
       "all <group name>: places all of the  units in the specified group on",
       "                       a line.",
       "next <group name> <n>: places the  next n  units  in  the  specified",
       "                       group on a line.",
       "section <group name> <m> <n>:  places units m  through n (inclusive)",
       "                       of the specified group  on  a line. Units are",
       "                       numbered base 0.",
       "blank <n>: places n blank spaces on a line.",
       "fill: places enough blank spaces on a line  to evenly fill it out to",
       "                       the per-row number.",
       "",
       "The  repeat  count is  the number  of   times to repeat   the layout",
       "commands for separate  lines.   If omitted, it  is assumed  to be 1.",
       "",
       "EXAMPLE",
       "To lay out an 8-3-8 encoder, you could use the following commands:",
       "",
       "activity-layout  cell-size 15  margin 3  per-row 8",
       "all Input",
       "fill, all Hidden, fill",
       "all Output",
       "",
       "minor-layout  cell-size 15  margin 3  per-row 5 show-links incoming",
       "all Hidden, fill, all Bias",
       "",
       "minor-layout for-unit Output.0",
       "all Bias, fill, all Hidden",
       "",
       "major-layout  margin 5  per-row 1",
       "[8] next Output 1",
       "",
       NULL);
    return 0;
  }

  if (currentNet == NULL)
    IErrorAbort("No current network") ;

  if (tokc == 1) {
    inStream = din ;
  } else if (tokc == 2) {
    inStream = IOpenFileOrAbort(tokv[1], "r", NULL) ;
  } else {
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;
    return 0 ;
  }

  layoutCommandLoop(inStream, dout) ;

  if (inStream != din)
    ICloseFile(inStream, NULL) ;

  markToRebuildDisplay(ACTIVITY_DISPLAY | CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		layoutCommandLoop
 *	Description:	loops through the input file collecting commands
 *	Parameters:	
 *		FILE	*inStream  - the stream to read input from
 *		FILE	*outStream - the stream to write output to
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static int	layoutCommandLoop(inStream, outStream)
  FILE		*inStream ;
  FILE		*outStream ;
{
  static Mode	mode ;
  Boolean	majorDone = FALSE, minorDone = FALSE, activityDone = FALSE ;

  int	argc ;
  char	**argv ;

  resetInput(inStream) ;
  while (getInputLine(inStream, &argc, &argv) != 0) {
    if (strcmp(argv[0], "major-layout") == 0) {
      majorDone = TRUE ;
      setMajorOptions(currentNet, argv, argc) ;
      mode = Major ;
    } else if (strcmp(argv[0], "minor-layout") == 0) {
      minorDone = TRUE ;
      setMinorOptions(currentNet, argv, argc) ;
      mode = Minor ;
    } else if (strcmp(argv[0], "activity-layout") == 0) {
      activityDone = TRUE ;
      setActivityOptions(currentNet, argv, argc) ;
      mode = Activity ;
    } else if (mode == Major) {
      setMajorLine(currentNet, argv, argc) ;
      incrementCurrentRow() ;
    } else if (mode == Minor) {
      setMinorLine(currentNet, argv, argc) ;
      incrementCurrentRow() ;
    } else if (mode == Activity) {
      setActivityLine(currentNet, argv, argc) ;
      incrementCurrentRow() ;
    } else {
      IErrorAbort("Line %d: unknown command \"%s\"\n",
		  layoutInputLine, argv[0]) ;
    }
  }
  if (minorDone == FALSE && activityDone == TRUE) {
    netForAllUnits(currentNet, ALL, setLayout, (void*)Minor) ;
    currentNet->minorWidth  = currentNet->activityWidth ;
    currentNet->minorHeight = currentNet->activityHeight ;
    currentNet->minorMargin = currentNet->activityMargin ;
  }

  if (activityDone == FALSE && minorDone == TRUE) {
    netForAllUnits(currentNet, ALL, setLayout, (void*)Activity) ;
    currentNet->activityWidth  = currentNet->minorWidth ;
    currentNet->activityHeight = currentNet->minorHeight ;
    currentNet->activityMargin = currentNet->minorMargin ;
  }

  touchUpMajorDimensions(currentNet, FALSE, FALSE) ;
  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		setLayout
 *	Description:	copies the layout fields from minor to activity
 *			or vice versa 
 *	Parameters:	
 *		Unit	unit - the unit tho set the layout for
 *		void	*data - (Mode) either Activity or Minor, depending
 *				on what you want to set
 *	Return Value:	NONE
 ***********************************************************************/
static void	setLayout(unit, data)
  Unit		unit ;
  void		*data ;
{
  Link	*link ;
  int	numLinks, idx ;

  if ((Mode)data == Minor) {
    numLinks = unit->numIncoming ;
    link     = unit->incomingLink ;
    for (idx = 0 ; idx < numLinks ; ++idx) {
      link[idx]->weightRow[0]    = -1 ;
      link[idx]->weightColumn[0] = -1 ;
      link[idx]->weightRow[1]    = link[idx]->preUnit->activityRow ;
      link[idx]->weightColumn[1] = link[idx]->preUnit->activityColumn ;
    }
  } else if ((Mode)data == Activity) {
    LinkLayoutMode	mode = unit->net->linkLayoutMode ;
    if (mode == SHOW_ALL || mode == SHOW_INCOMING) {
      numLinks = unit->numIncoming ;
      link     = unit->incomingLink ;
      idx      = 0 ;
    } else {
      numLinks = unit->numOutgoing ;
      link     = unit->outgoingLink ;
      idx      = 1 ;
    }
    if (numLinks > 0) {
      unit->activityRow	   = link[0]->weightRow[idx] ;
      unit->activityColumn = link[0]->weightColumn[idx] ;
    } else {
      unit->activityRow	   = -1 ;
      unit->activityColumn = -1 ;
    }
  }
}
/**********************************************************************/


/***********************************************************************
 *	Name:		readInputLine
 *	Description:	EVERYTHING ASSUMES THAT THE INPUT LINE ARRAY
 *			IS BUFSIZ
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static char	*readInputLine(in)
  FILE		*in ;
{
  static char	line[BUFSIZ] ;

  char	*p   = line ;
  char	*newLine ;
  int	length ;

  *p = '\0' ;
  while (1) {
    length = line + BUFSIZ - p ;
    newLine = fgets(p, length, in);
    ++layoutInputLine ;

    if (newLine == NULL)
      return (p == line ? NULL : line) ;

    while (*p) p++ ;

    if ((p - 1) > line && p[-2] == '\\') {
      p[-2] = p[-1];
      p[-1] = '\0';
      p--;
    } else
      break;
  }
  return line ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		parseInputLine
 *	Description:	
 *	Parameters:	
 *	Return Value:	
 ***********************************************************************/
static void	parseInputLine (line, argc, argv)
  char		*line ;
  int		*argc ;
  char		***argv ;
{
  char		*newLine, expandedLine[BUFSIZ] ;
  static Boolean	firstTime = TRUE ;
  static LEX_STACK	stack ;
  
  if (firstTime == TRUE)
    firstTime = FALSE ;
  else
    LexAnalysePop(&stack) ;

  if (line && (newLine = VarSubstitute(line, expandedLine, BUFSIZ, FALSE))) {
    *argv = LexAnalysePush(command_syntax, newLine, argc, &stack);
    if (*argv == NULL)
      *argc = 0 ;
  } else {
    *argv = NULL ;
    *argc = 0 ;
  }

  commandSubstitute(*argc, *argv, FALSE) ;

  if (itf_errno) {
    IReportError(stderr);
    IAbort() ;
  }
}
/**********************************************************************/

/********************************************************************/
static int	repeat ;
static int	oldArgc ;
static char	**oldArgv ;
/********************************************************************/


/*********************************************************************
 *	Name:		resetInput
 *	Description:	resets variables for getInputLIne
 *	Parameters:
 *	  FILE		*in - the input file (unused)
 *	Return Value:
 *	  static void	resetInput - NONE
 *********************************************************************/
static void	resetInput(in)
  FILE		*in ;
{
  extern int	layoutInputLine ;

  layoutInputLine = 0 ;
  oldArgc = 0 ;
  oldArgv = NULL ;
  repeat  = 0 ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		getInputLine
 *	Description:	
 *	Parameters:
 *	  FILE		*in - 
 *	  int		*argc - 
 *	  char		***argv - 
 *	Return Value:
 *	  static int	getInputLine  - 
 *********************************************************************/
static int	getInputLine (in, argc, argv)
  FILE		*in ;
  int		*argc ;
  char		***argv ;
{
  if (!repeat) {
    char	*line ;
    do {
      line = readInputLine(in) ;
      parseInputLine(line, argc, argv) ;
    } while (*argc == 0 && *argv != NULL) ;

    /* check for a repeat arg */
    if (*argc > 0) {
      String	ptr, string = (*argv)[0] ;
      if (string[0] == '[' && string[strlen(string) - 1] == ']') {
	repeat = strtol(string+1, &ptr, 0) ;
	if (repeat > 0 && ptr != string) {
	  oldArgc = --(*argc) ;
	  oldArgv = ++(*argv) ;
	  --repeat ;
	} else {
	  repeat = 0 ;
	}
      }
    }
  } else {
    *argc = oldArgc ;
    *argv = oldArgv ;
    --repeat ;
  }

  if (*argv == NULL)
    return 0 ;
  else
    return 1 ;
}
/********************************************************************/
