
/**********************************************************************
 * $Id: groupCom.c,v 1.5 93/01/15 13:05:35 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <xerion/commands.h>
#include "groupCom.h"

/***********************************************************************
 *	Name:		command_addGroup
 *	Description:	adds a group to the current network, automatically
 *			connects bias unit to all units
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_createGroup - 0 on failure, 1 on success
 ***********************************************************************/
int	command_addGroup (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  char	*name ;
  int	numUnits, idx ;
  int	newMask, groupMask ;
  Group	group, bias ;

  IUsage("[ -type TYPE [ -type TYPE ...] ] <name> n") ;
  if (GiveHelp(tokc)) {
    ISynopsis("create a group in the current network") ;
    IHelp
      (IHelpArgs,
       "Creates a group with the given <name>,  and  with the specified type",
       "masks  set.  TYPE may be  INPUT, BIAS,  OUTPUT, HIDDEN, or  any user",
       "created mask.   Multiple types may be specified,  each preceded with",
       "the keyword \"-type\".",
       "",
       "The group is  created in the  current network and contains n  units.",
       "The   name   of the group  must  be   unique. The    bias  unit will",
       "automatically be connected to all units in the group.",
       "EXAMPLE",
       "To add an input group with name \"InputGroup\" and 24 units,",
       "",
       "\txerion-> addGroup -type INPUT InputGroup 24",
       "SEE ALSO",
       "deleteGroups, connectGroups, disconnectGroups",
       NULL);
    return 0;
  }

  if (tokc < 3 || tokc % 2 != 1)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  /* Check for all the possible errors */
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  groupMask = 0 ;
  for (idx = 1 ; 
       strncmp(tokv[idx], "-type", strlen(tokv[idx])) == 0 ; idx += 2) {
    newMask = findClassMask(GROUP_CLASS, tokv[idx+1]) ;
    if (newMask == 0) {
      IErrorAbort("Unknown group type: \"%s\".", tokv[idx+1]) ;
      return 0 ;
    }
    groupMask |= newMask ;
  }
  name = tokv[idx] ;
  numUnits = atoi(tokv[idx+1]) ;

  if (numUnits < 0)
    IErrorAbort("The group cannot have a negative number of units.") ;

  group = groupFromName(currentNet, name) ;
  if (group != NULL)
    IErrorAbort("Group \"%s\" already exists.", group->name) ;
  
  /* Now create the group and add the units */
  group = createGroup(name, groupMask, currentNet) ;

  if (group == NULL)
    IErrorAbort("Cannot create group \"%s\"", name) ;

  for (idx = 0 ; idx < numUnits ; ++idx) {
    char	unitName[BUFSIZ] ;
    sprintf(unitName, "%s.%d", name, idx) ;
    if (0 == createUnit(unitName, group))
      IErrorAbort("Cannot create unit \"%s\"", unitName) ;
  }    

  if (!(group->type & BIAS)) {
    bias = groupFromName(currentNet, "Bias") ;
    if (bias != NULL)
      connectGroups(bias, group, 0) ;
  }
  markToRebuildDisplay(ACTIVITY_DISPLAY | CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_deleteGroups
 *	Description:	deletes groups from the current network
 *			uses regex to match names
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_deleteGroup - 0 on failure, 1 on success
 ***********************************************************************/
int	command_deleteGroups (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int	idx ;
  Group	*groupArray, *group ;

  IUsage("<group1> [ <group2> ... ]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("deletes groups from the current network") ;
    IHelp
      (IHelpArgs,
       "Deletes  the groups  with names   <group1>   etc. from the   current",
       "network. Names may contain pattern matching  expressions of the form",
       "used by grep.",
       "EXAMPLES",
       "To delete the group with name \"InputGroup\",",
       "",
       "\txerion-> deleteGroups InputGroup",
       "",
       "To delete all groups whose names begin with \"Hidden\",",
       "",
       "\txerion-> deleteGroups Hidden.*",
       "SEE ALSO",
       "addGroup, connectGroups, disconnectGroups",
       NULL);
    return 0;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  for (idx = 1 ; idx < tokc ; ++idx) {
    groupArray = groupsFromRegex(currentNet, tokv[idx]) ;
    if (groupArray == NULL)
      IErrorAbort("Bad string: \"%s\"", tokv[idx]) ;
    if (groupArray[0] == NULL)
      IError("No match: \"%s\"", tokv[idx]) ;

    for (group = groupArray ; *group != NULL ; ++group) {
      destroyGroup (*group) ;
    }

    free(groupArray) ;
  }

  markToRebuildDisplay(ACTIVITY_DISPLAY | CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_connectGroups
 *	Description:	connects two groups togther completely
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_connectGroup - 0 on failure, 1 on success
 ***********************************************************************/
int	command_connectGroups (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int	linkMask, newMask ;
  Group	*preGroupArray,  *preGroup ;
  Group	*postGroupArray, *postGroup ;
  String name ;

  IUsage("[ -type TYPE [ -type TYPE ...] ] <group1> <group2> ...") ;
  if (GiveHelp(tokc)) {
    ISynopsis("connects sets of groups to each other") ;
    IHelp
      (IHelpArgs,
       "Connects each  unit  in <group1> to  all of the  units in  <group2>,",
       "where  <group1>  and  <group2> are  the names of  two  groups in the",
       "current  network.   Order  is important.   <group1> is the preceding",
       "group  to  <group2>.   After  <group1>  is  connected  to  <group2>,",
       "<group2> is connected to <group3> etc.",
       "",
       "Group  names  may use regular expressions  similar to those used  in",
       "grep to connect more  than two groups.   Optionally, a type mask may",
       "be specified for the type of  link to create.   At present there are",
       "no standard link types.",
       "EXAMPLE",
       "\txerion-> connectGroups \"Input\" \"Hidden.*\" \"Output\"",
       "",
       "will connect the group named \"Input\" to all groups whose names begin",
       "with the  string \"Hidden\",  then all  the \"Hidden.*\"  groups to  the",
       "group named \"Output\".",
       "SEE ALSO",
       "connectUnits, disconnectGroups, disconnectUnits",
       NULL);
    return 0;
  }

  name = tokv[0] ;

  if (tokc < 3)
    IErrorAbort(IPrintUsage(name, usage)) ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  linkMask = 0 ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-type", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      newMask = findClassMask(LINK_CLASS, *tokv) ;
      if (newMask == 0) {
	IErrorAbort("Unknown type: \"%s\".", *tokv) ;
	return 0 ;
      }
      linkMask |= newMask ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  if (tokc < 2)
    IErrorAbort(IPrintUsage(name, usage)) ;

  /* get the list of pre groups assuming *tokv is a regular expression */
  preGroupArray = groupsFromRegex(currentNet, *tokv) ;
  if (preGroupArray == NULL)
    IErrorAbort("Bad string: \"%s\"", *tokv) ;
  if (preGroupArray[0] == NULL)
    IErrorAbort("No match: \"%s\"",   *tokv) ;
  
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    /* get the list of post groups assuming *tokv is a regular expression */
    postGroupArray = groupsFromRegex(currentNet, *tokv) ;
    if (postGroupArray == NULL)
      IErrorAbort("Bad string: \"%s\"", *tokv) ;
    if (postGroupArray[0] == NULL)
      IErrorAbort("No match: \"%s\"",   *tokv) ;

    for (postGroup = postGroupArray ; *postGroup != NULL ; ++postGroup) {
      for (preGroup = preGroupArray ; *preGroup  != NULL ; ++preGroup) {
	if (0 == connectGroups(*preGroup, *postGroup, linkMask))
	  IErrorAbort("Cannot connect groups \"%s\" and \"%s\"",
		      (*preGroup)->name, (*postGroup)->name) ;
      }
    }
    free(preGroupArray) ;
    preGroupArray = postGroupArray ;
  }
  free(postGroupArray) ;

  markToRebuildDisplay(CONNECTION_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_disconnectGroups
 *	Description:	disconnects two groups
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_disconnectGroup - 0 on failure, 1 on success
 ***********************************************************************/
int	command_disconnectGroups (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  Group	*preGroupArray,  *preGroup ;
  Group	*postGroupArray, *postGroup ;
  String name ;
  IUsage("<group1> <group2> ...") ;
  if (GiveHelp(tokc)) {
    ISynopsis("disconnects sets of groups from each other") ;
    IHelp
      (IHelpArgs,
       "Disconnects each unit in <group1> from all of the units in <group2>,",
       "where <group1>  and <group2> are  the  names of two  groups  in  the",
       "current  network.  Order is important.   <group1> is  the  preceding",
       "group to  <group2>.   After  disconnecting  <group1>  from <group2>,",
       "disconnect <group2> from <group3>, etc. Group  names may use regular",
       "expressions  similar to those used in  grep to disconnect more  than",
       "two groups.",
       "EXAMPLE",
       "\txerion-> disconnectGroups \"bias\" \"hidden.*\"",
       "",
       "will  disconnect the group with  name \"bias\" from all   groups whose",
       "name begins with the string \"hidden\".",
       "SEE ALSO",
       "connectGroups, connectUnits, disconnectUnits",
       NULL);
    return 0;
  }

  name = tokv[0] ;

  if (tokc < 3)
    IErrorAbort(IPrintUsage(name, usage)) ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  ++tokv, --tokc ;
  /* get the list of pre groups assuming *tokv is a regular expression */
  preGroupArray = groupsFromRegex(currentNet, *tokv) ;
  if (preGroupArray == NULL)
    IErrorAbort("Bad string: \"%s\"", *tokv) ;
  if (preGroupArray[0] == NULL)
    IErrorAbort("No match: \"%s\"",   *tokv) ;

  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    /* get the list of post groups assuming *tokv is a regular expression */
    postGroupArray = groupsFromRegex(currentNet, *tokv) ;
    if (postGroupArray == NULL)
      IErrorAbort("Bad string: \"%s\"", *tokv) ;
    if (postGroupArray[0] == NULL)
      IErrorAbort("No match: \"%s\"", *tokv) ;

    for (postGroup = postGroupArray ; *postGroup != NULL ; ++postGroup)
      for (preGroup = preGroupArray ; *preGroup  != NULL ; ++preGroup)
	disconnectGroups(*preGroup, *postGroup) ;

    free(preGroupArray) ;
    preGroupArray = postGroupArray ;
  }
  free(postGroupArray) ;
  
  markToRebuildDisplay(CONNECTION_DISPLAY) ;
  
  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_orderGroups
 *	Description:	orders the array of groups in the current net.
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_orderGroups - 0 on failure, 1 on success
 ***********************************************************************/
static int	compare ARGS((const void	*p1, const void	*p2)) ;
static Group	*groupArray ;
static int	numGroups ;
int	command_orderGroups (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int	idx ;

  IUsage("<group1> [ <group2> ... ]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("orders the groups in the current net") ;
    IHelp
      (IHelpArgs,
       "Sets the  order for the forward  pass in the  current  network.  The",
       "backward  pass uses the  reverse order to the forward  pass.  If any",
       "groups  are not specified, their  position  in the  ordering is  not",
       "certain.",
       "SEE ALSO",
       "connectGroups",
       NULL);
    return 0;
  }

  /* Check for all the possible errors */
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;
  if (currentNet->numGroups == 0)
    IErrorAbort("There are no groups in the current net.") ;


  ++tokv ;
  --tokc ;
  groupArray = (Group *) callocOrAbort(currentNet->numGroups, sizeof(Group)) ;
  numGroups  = currentNet->numGroups ;
  for (idx = 0 ; idx < tokc ; ++idx) {
    if ((groupArray[idx] = groupFromName(currentNet, tokv[idx])) == NULL)
      IErrorAbort("Unknown group: \"%s\"", tokv[idx]) ;
  }

  qsort(currentNet->group, currentNet->numGroups, sizeof(Group), compare) ;
  free(groupArray) ;
  return 1 ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		groupTypes
 *	Description:	prints the types of the groups in the net
 *	Parameters:
 *	int tokc - 
 *	char *tokv[] -
 *	Return Value:
 *	  int command_groupTypes - 
 *********************************************************************/
static void	printGroupTypes(group, data)
  Group		group ;
  void		*data ;
{
  FILE		*fp = (FILE *)data ;
  Mask		mask;
  int		j;

  fprintf(fp, "Group %s:\t", group->name);
  mask = 1;
  for (j=0; j<8*sizeof(Mask); j++) {
    if (group->type & mask)
      fprintf(fp, " %s", classMaskName(GROUP_CLASS, mask));
    mask <<= 1;
  }
  fprintf(fp, "\n");
}
/**********************************************************************/
int command_groupTypes(tokc, tokv)
int tokc;
char *tokv[];
{
  struct ARG_DESC *argd;
  argd = StartArgs(tokv[0]);
  /* Args(argd, "<name>%s", &name); */ /* for example */
  EndArgs(argd);
  if (GiveHelp(tokc)) {
    ISynopsis("prints out the types for all groups");
    IHelp(tokc, tokv[0], NULL, synopsis,
	  "\"groupTypes\" prints  out  the types of  the groups   in the  current",
	  "network. It gets these types from the type mask field of the groups.",
	  "",
	  NULL);
    return 0;
  }
  (void) ParseArgs(argd, tokc, tokv, 0);

  if (currentNet==NULL)
    IErrorAbort("current net is NULL");

  netForAllGroups(currentNet, ALL, printGroupTypes, dout) ;

  return 0;
}
/********************************************************************/



/***********************************************************************
 *	Name:		compare
 *	Description:	compares to pointers to Groups and figures out
 *			which one comes first in groupArray
 *	Parameters:	the pointers to the group
 *	Return Value:	-1, 0, 1 depending on which comes first.
 ***********************************************************************/
static int	compare(p1, p2)
  const void	*p1 ;
  const void	*p2 ;
{
  int	i1, i2 ;
  
  for (i1 = 0 ; i1 < numGroups && *(Group *)p1 != groupArray[i1] ; ++i1) 
    ;

  for (i2 = 0 ; i2 < numGroups && *(Group *)p2 != groupArray[i2] ; ++i2) 
    ;

  return i1 - i2 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_groupLongName
 *	Description:	returns the long name of a group (given short
 *			name (e.g. "Bias" -> "currentNet.group[0]")
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_createGroup - 0 on failure, 1 on success
 ***********************************************************************/
int	command_groupLongName (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  char	*name, *groupName, *longName ;

  IUsage("<name>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("return the long name of a group.") ;
    IHelp
      (IHelpArgs,
       "Returns the  long  name of a group, given the group's real name. For",
       "example,",
       "",
       "\txerion-> groupLongName \"Bias\"",
       "",
       "would return the string:",
       "\tcurrentNet.group[0]",
       "SEE ALSO",
       "unitLongName",
       NULL);
    return 0;
  }
  
  name = tokv[0] ;
  if (tokc != 2)
    IErrorAbort(IPrintUsage(name, usage)) ;
  groupName = tokv[1] ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  longName = groupLongName(groupName) ;

  if (longName) {
    fprintf(dout, "%s\n", longName) ;
    free((void *)longName) ;
  } else {
    IErrorAbort("Unknown group: \"%s\"", groupName) ;
  }
  return 1 ;
}
/**********************************************************************/
String		groupLongName(name)
  String	name ;
{
  int	idx ;
  char	longName[128] ;

  if (currentNet == NULL || name == NULL)
    return NULL ;

  for (idx = 0 ; idx < currentNet->numGroups ; ++idx)
    if (strcmp(name, currentNet->group[idx]->name) == 0)
      break ;

  if (idx == currentNet->numGroups)
    return NULL ;
  
  sprintf(longName, "currentNet.group[%d]", idx) ;
  return strdup(longName) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_connectBlocks
 *	Description:	connects blocks of units in one group to 
 *			each unit in another group. See the help
 *			message for the full description.
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_template - 0 on failure, 1 on success
 ***********************************************************************/
static void	connectBlocks ARGS((Group, int, Group, int, int, int)) ;
/**********************************************************************/
int	command_connectBlocks(tokc, tokv)
  int	tokc ;
  char	**tokv ;
{
  Group	group1, group2 ;
  char	*name ;
  int	width1, width2, xShift, yShift ;

  IUsage("-x <n> -y <n> <group1> <width1> <group2> <width2>");
  if (GiveHelp(tokc)) {
    ISynopsis("connect blocks of units in one group to units in another");
    IHelp
      (IHelpArgs,
       "This  command connects blocks of units in one group to  each unit in",
       "another group.  Once the two  groups are connected,  the links  from",
       "corresponding units  in  each block  are  constrained.  The argument",
       "<width1> is the  width  of  the rectangle  the first  group is to be",
       "layed out in. Similarly width2 for group2.  The  '-x'  option allows",
       "you to  specify the  x-offset for each  block of units  in the first",
       "group, and the '-y' option allows you to specify the y-offset.",
       "",
       "An example will probably make this more clear.",
       "",
       "EXAMPLES",
       "Consider a net with an input group of 100 units, an  output group of",
       "9 units  and  no hidden  layer.   The groups  can be  layed  out  in",
       "corresponding rectangles of 10x10 and 3x3. The command:",
       "",
       "xerion-> connectBlocks input 10 output 3",
       "",
       "would  connect to  each  unit  in the output layer a block of units,",
       "8x8, from the input layer.  Each block of  units would be  offset by",
       "one unit in both the x  and y directions.   Note that  there is  one",
       "block  for  each of the 9 output units, and that  they have the same",
       "layout (8x8); however, the blocks of units overlap each other.",
       "",
       "The constraints would cause the corresponding links from  each  unit",
       "in each block to be constrained  together.  For example, the 9 links",
       "to the output units coming from the  unit in the bottom right corner",
       "of their corresponding blocks would all be equal.",
       NULL) ;
    return 1 ;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  /* Parse any command line options */
  name = *tokv ;
  xShift = 1 ;
  yShift = 1 ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-x", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      xShift = atol(*tokv) ;
      if (xShift <= 0)
	IErrorAbort("x shift must be greater than 0 (read: \"%s\").", *tokv) ;
    } else if (strncmp(*tokv, "-y", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      yShift = atol(*tokv) ;
      if (yShift <= 0)
	IErrorAbort("y shift must be greater than 0 (read: \"%s\").", *tokv) ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }
  
  if (tokc != 4)
    IErrorAbort(IPrintUsage(name, usage)) ;

  group1 = groupFromName(currentNet, *tokv) ;
  if (group1 == NULL)
    IErrorAbort("Unknown group: \"%s\".", *tokv) ;
  ++tokv, --tokc ;

  width1 = atol(*tokv) ;
  if (width1 <= 0)
    IErrorAbort("Width must be greater than 0: \"%s\".", *tokv) ;
  ++tokv, --tokc ;

  group2 = groupFromName(currentNet, *tokv) ;
  if (group2 == NULL)
    IErrorAbort("Unknown group: \"%s\".", *tokv) ;
  ++tokv, --tokc ;

  width2 = atol(*tokv) ;
  if (width2 <= 0)
    IErrorAbort("Width must be greater than 0: \"%s\".", *tokv) ;
  ++tokv, --tokc ;

  /***
   * Do whatever you want in here. 
   * Note that all writing should be to external FILE * 'dout'.
   */
  connectBlocks(group1, width1, group2, width2, xShift, yShift) ;

  return 1 ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		connectBlocks
 *	Description:	See the help message for the full description.
 *	Parameters:
 *	  Group		group1 - the group the links come from
 *	  int		width1 - the width of the group (height calculated
 *				 from this and group1->numUnits)
 *	  Group		group2 - the group the links go to
 *	  int		width2 - as for width1
 *	  int		xShift - the x offset for each block of units
 *				 in the first group
 *	  int		yShift - the y offset for each block of units
 *	Return Value:
 *	  export void	connectBlock - NONE
 *********************************************************************/
static void	connectBlocks(group1, width1, group2, width2, xShift, yShift)
  Group		group1 ;
  int		width1 ;
  Group		group2 ;
  int		width2 ;
  int		xShift ;
  int		yShift ;
{
  int		n1, n2, height1, height2 ;
  int		blockHeight, blockWidth ;
  int		idx, xIdx, yIdx ;

  /* calculate all the necessary values */
  n1 = group1->numUnits ;
  n2 = group2->numUnits ;
  height1     = n1/width1 ;
  height2     = n2/width2 ;
  blockWidth  = width1  - xShift*(width2  - 1) ;
  blockHeight = height1 - yShift*(height2 - 1) ;

  /* all the possible problems, just abort */
  if (n2 > n1)
    IErrorAbort("\"%s\" has more units than \"%s\"", group2->name, group1->name) ;
  if (width1 > n1)
    IErrorAbort("\"%s\" is too wide", group1->name) ;
  if (width2 > n2)
    IErrorAbort("\"%s\" is too wide", group2->name) ;
  if (width2 > width1)
    IErrorAbort("\"%s\" is wider than \"%s\"", group2->name, group1->name) ;
  if (n1 % width1)
    IErrorAbort("\"%s\" is not rectangular", group1->name) ;
  if (n2 % width2)
    IErrorAbort("\"%s\" is not rectangular", group2->name) ;
  if (height2 > height1)
    IErrorAbort("\"%s\" is higher than \"%s\"", group2->name, group1->name) ;
  if (blockWidth <= 0)
    IErrorAbort("X Shift of %d is too large", xShift) ;
  if (blockHeight <= 0)
    IErrorAbort("Y Shift of %d is too large", yShift) ;

  /****************************************
   * Do the looping this way so that we can do constraints 
   * at the same time.
   */
  for (yIdx = 0 ; yIdx < blockHeight ; ++yIdx) {
    for (xIdx = 0 ; xIdx < blockWidth ; ++xIdx) {
      Link	lastLink = NULL ;
      for (idx = 0 ; idx < n2 ; ++idx) {
	int	yOffset  = (idx / width2)*yShift + yIdx ;
	int	xOffset  = (idx % width2)*xShift + xIdx ;
	Unit	fromUnit = group1->unit[yOffset*width1 + xOffset] ;
	Unit	toUnit   = group2->unit[idx] ;
	Link	link	 = connectUnits(fromUnit, toUnit, UNKNOWN) ;
	if (lastLink)
	  netConstrainLink(group1->net, link, lastLink, 1.0) ;
	lastLink = link ;
      }
    }
  }
}
/********************************************************************/
