
/**********************************************************************
 * $Id: freezeCom.c,v 1.6 92/11/30 13:16:51 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <xerion/commands.h>

/*******************************************************************
 *	Name:		freezeLink
 *	Description:	
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_freezeLink - 
 *******************************************************************/
int	command_freezeLink(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Link		*link ;
  Real		weight ;
  int		idx ;
  Boolean	weightSet = FALSE ;

  IUsage("[ -weight <w> ] <link1> ...") ;
  if (GiveHelp(tokc)) {
    ISynopsis("freeze the weight of a link") ;
    IHelp
      (IHelpArgs,
       "\"freezeLink\" freezes the weight of a  link at a specified value.  If",
       "no  value  is  specified, it is frozen at  its current weight.  Link",
       "names may contain regular expressions.",
       "",
       "If the link is constrained  with other  links, these other links are",
       "also frozen.",
       "EXAMPLES",
       "To  freeze  link  \"Hidden.0 -> Output.0\" at value 0.5 use the command:",
       "",
       "xerion-> freezeLink -w 0.5 \"Hidden.0 -> Output.0\"",
       "SEE ALSO",
       "freezeUnit,   freezeGroup,   unfreezeLink, unfreezeUnit,",
       "unfreezeGroup",
       NULL);
    return 1;
  }
  
  name = *tokv ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-weight", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      if (!IIsNumber(*tokv))
	IErrorAbort(IPrintUsage(name, usage)) ;
      weight = atof(*tokv) ;
      weightSet = TRUE ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  while (tokc > 0) {
    link = linksFromRegex(currentNet, *tokv) ;
    if (link == NULL || link[0] == NULL)
      IErrorAbort("Unknown link: \"%s\".", *tokv) ;
    for (idx = 0 ; link[idx] != NULL ; ++idx) {
      netFreezeLink(currentNet, link[idx]) ;
      if (weightSet)
	linkSetWeight(link[idx], weight) ;
    }
    free(link) ;
    ++tokv, --tokc ;
  }
  if (weightSet)
    markToRebuildDisplay(CONNECTION_DISPLAY) ;
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		unfreezeLink
 *	Description:	
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_unfreezeLink - 
 *******************************************************************/
int	command_unfreezeLink(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Link		*link ;
  int		idx ;

  IUsage("<link1> ...") ;
  if (GiveHelp(tokc)) {
    ISynopsis("unfreeze the weight of a link") ;
    IHelp
      (IHelpArgs,
       "\"unfreezeLink\"  unfreezes a  previously  frozen link  and any  other",
       "links it is constrained with.",
       "EXAMPLES",
       "To  unfreeze  link  \"Hidden.0 -> Output.0\" use the command:",
       "",
       "xerion-> unfreezeLink \"Hidden.0 -> Output.0\"",
       "SEE ALSO",
       "freezeLink, freezeUnit, freezeGroup, unfreezeUnit, unfreezeGroup",
       "",
       NULL);
    return 1;
  }
  
  name = *tokv ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  while (tokc > 0) {
    link = linksFromRegex(currentNet, *tokv) ;
    if (link == NULL || link[0] == NULL)
      IErrorAbort("Unknown link: \"%s\".", *tokv) ;
    for (idx = 0 ; link[idx] != NULL ; ++idx) {
      netUnfreezeLink(currentNet, link[idx]) ;
    }
    free(link) ;
    ++tokv, --tokc ;
  }
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		freezeUnit
 *	Description:	freezes all incoming links of a unit
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_freezeUnit - 
 *******************************************************************/
int command_freezeUnit(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Unit		*unit ;
  Real		weight ;
  Boolean	weightSet = FALSE ;
  int		idx, linkIdx ;
  
  IUsage("[-weight <weight>] [<unit> ...]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("freeze weights of a unit's incoming links") ;
    IHelp
      (IHelpArgs,
       "\"freezeUnit\" freezes the weights of all incoming  links to a unit at",
       "some value.   <weight> is the weight the  links  are frozen at.  Its",
       "default value is the links' current weights.",
       "",
       "If any  of the links are  constrained with other links,  these other",
       "links are also frozen.",
       "EXAMPLE",
       "To freeze all incoming links of \"Output.0\" use the command:",
       "",
       "	xerion-> freezeUnit Output.0",
       "SEE ALSO ",
       "freezeGroup,  freezeLink,   unfreezeLink,  unfreezeUnit,",
       "unfreezeGroup",
       NULL);
    return 1;
  }
  
  /* Parse the command line */
  name = *tokv ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-weight", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      if (!IIsNumber(*tokv))
	IErrorAbort(IPrintUsage(name, usage)) ;
      weight = atof(*tokv) ;
      weightSet = TRUE ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  while (tokc > 0) {
    unit = unitsFromRegex(currentNet, *tokv) ;
    if (unit == NULL || unit[0] == NULL)
      IErrorAbort("Unknown unit: \"%s\".", *tokv) ;
    for (idx = 0 ; unit[idx] != NULL ; ++idx) {
      for (linkIdx = 0 ; linkIdx < unit[idx]->numIncoming ; ++linkIdx) {
	Link	link = unit[idx]->incomingLink[linkIdx] ;
	netFreezeLink(currentNet, link) ;
	if (weightSet)
	  linkSetWeight(link, weight) ;
      }
    }
    free(unit) ;
    ++tokv, --tokc ;
  }

  if (weightSet)
    markToRebuildDisplay(CONNECTION_DISPLAY) ;
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		unfreezeUnit
 *	Description:	unfreezes all incoming links to a unit
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_unfreezeUnit - 
 *******************************************************************/
int command_unfreezeUnit(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Unit		*unit ;
  int		idx, linkIdx ;

  IUsage("<unit1> [ <unit2> ...]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("unfreeze weights of a unit's incoming links") ;
    IHelp
      (IHelpArgs,
       "\"unfreezeUnit\" unfreezes all previously  frozen weights of a  unit's",
       "incoming links and any other links constrained with them.",
       "EXAMPLES",
       "To unfreeze all incoming links of \"Output.0\", use the command:",
       "",
       "	xerion-> unfreezeUnit Output.0",
       "",
       "SEE ALSO",
       "unfreezeLink, unfreezeGroup, freezeUnit, freezeGroup, freezeLink",
       NULL);
    return 1;
  }
  
  name = *tokv ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  while (tokc > 0) {
    unit = unitsFromRegex(currentNet, *tokv) ;
    if (unit == NULL || unit[0] == NULL)
      IErrorAbort("Unknown unit: \"%s\".", *tokv) ;
    for (idx = 0 ; unit[idx] != NULL ; ++idx) {
      for (linkIdx = 0 ; linkIdx < unit[idx]->numIncoming ; ++linkIdx)
	netUnfreezeLink(currentNet, unit[idx]->incomingLink[linkIdx]) ;
    }
    free(unit) ;
    ++tokv, --tokc ;
  }
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		freezeGroup
 *	Description:	freeze all incoming links of a group 
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_freezeGroup - 
 *******************************************************************/
int command_freezeGroup(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Group		*group ;
  Real		weight ;
  Boolean	weightSet = FALSE ;
  int		idx, unitIdx, linkIdx ;
  
  IUsage("[-weight <weight>] [<group> ...]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("freeze weights of a group's incoming links") ;
    IHelp
      (IHelpArgs,
       "\"freezeGroup\" freezes the weights of all incoming links to all units",
       "in  a group at some  value.   <weight> is  the  weight the links are",
       "frozen at.  Its default value is the links' current weights.",
       "",
       "If any  of the links are  constrained with other links,  these other",
       "links are also frozen.",
       "EXAMPLE",
       "To freeze all incoming links of group \"Output\" use the command:",
       "",
       "	xerion-> freezeGroup Output",
       "SEE ALSO ",
       "freezeUnit, freezeLink, unfreezeLink, unfreezeUnit, unfreezeGroup",
       "",
       NULL);
    return 1;
  }
  
  /* Parse the command line */
  name = *tokv ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-weight", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      if (!IIsNumber(*tokv))
	IErrorAbort(IPrintUsage(name, usage)) ;
      weight = atof(*tokv) ;
      weightSet = TRUE ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  while (tokc > 0) {
    group = groupsFromRegex(currentNet, *tokv) ;
    if (group == NULL || group[0] == NULL)
      IErrorAbort("Unknown group: \"%s\".", *tokv) ;

    for (idx = 0 ; group[idx] != NULL ; ++idx) {
      for (unitIdx = 0 ; unitIdx < group[idx]->numUnits ; ++unitIdx) {
	Unit	unit = group[idx]->unit[unitIdx] ;
	for (linkIdx = 0 ; linkIdx < unit->numIncoming ; ++linkIdx) {
	  Link	link = unit->incomingLink[linkIdx] ;
	  netFreezeLink(currentNet, link) ;
	  if (weightSet)
	    linkSetWeight(link, weight) ;
	}
      }
    }
    free(group) ;
    ++tokv, --tokc ;
  }

  if (weightSet)
    markToRebuildDisplay(CONNECTION_DISPLAY) ;
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		unfreezeGroup
 *	Description:	removes freezets on all incoming links to
 *			a group.
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_unfreezeGroup - 
 *******************************************************************/
int command_unfreezeGroup(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Group		*group ;
  int		idx, unitIdx, linkIdx ;
  
  IUsage("<group1> [ <group2> ...]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("unfreeze weights of a group's incoming links") ;
    IHelp
      (IHelpArgs,
       "\"unfreezeGroup\"  unfreezes  all previously   frozen  weights  of all",
       "incoming links of   all units  in  a group, and    any other   links",
       "constrained with them.",
       "EXAMPLES",
       "To  unfreeze  all  incoming  links of the  group   \"Output\", use the",
       "command:",
       "",
       "	xerion-> unfreezeGroup Output",
       "SEE ALSO",
       "unfreezeLink, unfreezeUnit,    freezeUnit,  freezeGroup,",
       "freezeLink",
       NULL);
    return 1;
  }
  
  name = *tokv ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  while (tokc > 0) {
    group = groupsFromRegex(currentNet, *tokv) ;
    if (group == NULL || group[0] == NULL)
      IErrorAbort("Unknown group: \"%s\".", *tokv) ;

    for (idx = 0 ; group[idx] != NULL ; ++idx) {
      for (unitIdx = 0 ; unitIdx < group[idx]->numUnits ; ++unitIdx) {
	Unit	unit = group[idx]->unit[unitIdx] ;
	for (linkIdx = 0 ; linkIdx < unit->numIncoming ; ++linkIdx)
	  netUnfreezeLink(currentNet, unit->incomingLink[linkIdx]) ;
      }
    }
    free(group) ;
    ++tokv, --tokc ;
  }

  return 1 ;
}
/******************************************************************/
