%{
/**********************************************************************
 * $Id: exampleParse.y,v 1.5 93/02/26 12:03:05 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include "exampleParse.h"

extern FILE	*yyin ;
extern int	yyparse() ;

static char	*tag ;
static int	nextIndex ;
static int	numSlices ;
static void	addRange() ;
static void	addValue() ;

static void	finishInputs() ;
static void	finishTargets() ;
static void	startNewExample() ;
static void	startNextVector() ;
static void	addExample() ;

%}
%union {
  float	realValue ;
  char	stringValue[128] ;
}
%token	<realValue>	INTEGER
%token	<realValue>	REAL
%token	<stringValue>	TAG
%type	<stringValue>	exampleTag
%type	<realValue>	real
%token	ALL
%token	LAST
%token	NEXT
%token	'[' ']'
%left	','
%token	'-'
%left	UNARYMINUS
%%

exampleSet: 
  | exampleSet 'EOF'
  | exampleSet inputList targetList	        { tag = NULL ;
						  startNewExample() ; }
  | exampleSet exampleTag inputList targetList { tag = NULL ; 
						 startNewExample() ; }
  ;

exampleTag: TAG			{ tag = $1 ; }
  ;


inputList: ','			{ confirmVectorSize() ; finishInputs() ; }
  | realVector ','		{ confirmVectorSize() ; finishInputs() ; }
  | vectorList ','		{ finishInputs() ; }
  ;

targetList: ';'			{  confirmVectorSize() ; 
				   finishTargets() ; addExample(tag) ; }
  | realVector ';'		{  confirmVectorSize() ; 
				   finishTargets() ; addExample(tag) ; }
  | vectorList ';'		{ finishTargets() ; addExample(tag) ; }
  ;

vectorList: '[' range ']' realVector	 {  confirmVectorSize() ; 
					    startNextVector() ; }
  | vectorList '[' range ']' realVector	 {  confirmVectorSize() ; 
					    startNextVector() ; }
  ;

range: INTEGER			{ addRange((int)$1, (int)$1) ; }
  | INTEGER '-' INTEGER		{ addRange((int)$1, (int)$3) ; }
  | ALL				{ addRange(0, numSlices - 1) ; }
  | NEXT INTEGER		{ addRange(nextIndex,
					   nextIndex + (int)$2 - 1) ; }
  | LAST INTEGER		{ addRange(numSlices - (int)$2, 
					   numSlices - 1) ; }
  | range ',' range
  ;

realVector: real		{ addValue((double)$1) ; }
  | realVector real		{ addValue((double)$2) ; }
  ;

real: REAL
  | INTEGER
  | '-' real %prec UNARYMINUS	{ $$ = -$2 ; }
  ;

%%
#ifdef ALL
#undef ALL
#endif  
#include <stdio.h>
#include <ctype.h>
#include <xerion/useful.h>
#include <xerion/simulator.h>

typedef enum Mode {Input, Target} Mode ;

static Mode	mode ;
static Real	**inputArrays ;
static Real	**targetArrays ;
static Real	*nanInputArray ;
static Real	*nanTargetArray ;
static Real	*valueArray ;
static int	numValues, maxValues ;
static int	numInputs  ;
static int	numTargets ;

static ExampleSet	exampleSet ;

static Real	*getValueArray() ;
static void	allocateArrays() ;

int exampleFileLineNumber = 1 ;

/*********************************************************************
 *	Name:		yyerror
 *	Description:	Error mechanism for the parser
 *	Parameters:
 *	va_dcl		similar to printf (format, args ...)
 *	Return Value:
 *	  int	yyerror - actually, it aborts.
 *********************************************************************/
int	yyerror(va_alist)
va_dcl
{
  va_list	 ap;

#ifndef hpux
  yyerrok ;
#endif

  va_start(ap);
  IErrorV(&ap) ;
  IError("Near line %d\n", exampleFileLineNumber) ;
  IAbort() ;
  return 1 ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		readExampleFile
 *	Description:	reads in a file of examples and adds them to a
 *			set
 *	Parameters:
 *	  FILE		*fp - the file to read from
 *	  ExampleSet	set - the example set to add to
 *	  int		expectedSlices - the number of time slices in
 *				the net
 *	  int		expectedInputs - the number of input units in
 *				the net
 *	  int		expectedTargets - the number of target units
 *				in the net
 *	Return Value:
 *	  void	readExampleFile - NONE
 *********************************************************************/
void	readExampleFile(fp, set, 
			expectedSlices, expectedInputs, expectedTargets)
  FILE	*fp ;
  ExampleSet	set ;
  int	expectedSlices ;
  int	expectedInputs ;
  int	expectedTargets ;
{
  numSlices  = expectedSlices ;
  numInputs  = expectedInputs ;
  numTargets = expectedTargets ;
  exampleSet = set ;
  exampleFileLineNumber = 1 ;
  yyin = fp ;

  startNewExample() ;
  yyparse() ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		addRange
 *	Description:	adds the current vector to the range of time
 *			slices
 *	Parameters:
 *	  int	low, high - the lowest and highest time slices to add
 *			the vector to (inclusive range)
 *	Return Value:
 *	  void	addRange - NONE
 *********************************************************************/
static void	addRange(low, high) 
  int	low, high ;
{
  Real	*value ;

  if (low < nextIndex)
    yyerror("range is not increasing: %d,%d", nextIndex, low) ;
  else if (low > high)
    yyerror("range is not increasing: %d-%d", low, high) ;
  else if (low > numSlices)
    yyerror("value too high: %d", low) ;
  else if (high > numSlices)
    yyerror("value too high: %d", high) ;

  value = getValueArray(mode) ;
  if (mode == Input) {
    while(low <= high) {
      if (inputArrays[low] != nanInputArray)
	yyerror("vector already specified for slice %d", low) ;
      inputArrays[low++] = value ;
    }
  } else {
    while(low <= high) {
      if (targetArrays[low] != nanTargetArray)
	yyerror("vector already specified for slice %d", low) ;
      targetArrays[low++] = value ;
    }
  }
  nextIndex = low ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		addValue
 *	Description:	adds a value to the current vector
 *	Parameters:
 *	  float	value - the value to add
 *	Return Value:
 *	  void	addValue - NONE
 *********************************************************************/
static void	addValue(value) 
  double	value ;
{
  Real	*array ;

  array = getValueArray(mode) ;

  if (numValues >= maxValues)
    yyerror("too many values: %d", numValues) ;

  array[numValues++] = value ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		startNewExample
 *	Description:	resets the example arrays, making it ready
 *			for a new list of examples
 *	Parameters:	NONE
 *	Return Value:
 *	  void	startNewExample - NONE
 *********************************************************************/
static void	startNewExample() {
  mode = Input ;
  allocateArrays() ;
  nextIndex = 0 ;
  startNextVector() ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		finishInputs
 *	Description:	cleans up the input vectors before going
 *			on to targets
 *	Parameters:	NONE
 *	Return Value:
 *	  int	finishInputs - NONE
 *********************************************************************/
static void	finishInputs() {
  int	idx ;

  for (idx = 0 ; idx < numSlices ; ++idx)
    if (inputArrays[idx] != nanInputArray)
      break ;
  if (idx == numSlices)
    inputArrays[0] = getValueArray(mode) ;

  for (idx = 0 ; idx < numSlices ; ++idx)
    if (inputArrays[idx] == nanInputArray)
      break ;
  if (idx == numSlices)
    free(nanInputArray) ;

  mode = Target ;
  nextIndex = 0 ;
  startNextVector() ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		finishTargets
 *	Description:	cleans up the target vectors before going
 *			on to the next example list
 *	Parameters:
 *	Return Value:
 *	  int	finishTargets - 
 *********************************************************************/
static void	finishTargets() 
{
  int	idx ;
  for (idx = 0 ; idx < numSlices ; ++idx)
    if (targetArrays[idx] != nanTargetArray)
      break ;
  if (idx == numSlices)
    targetArrays[numSlices - 1] = getValueArray(mode) ;

  for (idx = 0 ; idx < numSlices ; ++idx)
    if (targetArrays[idx] == nanTargetArray)
      break ;
  if (idx == numSlices)
    free(nanTargetArray) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		confirmVectorSize
 *	Description:	makes sure that the vector has enough values
 *	Parameters:	NONE
 *	Return Value:
 *	  static void	confirmVectorSize - NONE
 *********************************************************************/
static void	confirmVectorSize()
{
  if (mode == Input && numValues != numInputs)
    yyerror("Wrong number of input values") ;

  if (mode == Target && numValues != numTargets)
    yyerror("Wrong number of target values") ;
}
/********************************************************************/

    
/*********************************************************************
 *	Name:		startNextVector
 *	Description:	sets values so that we will be ready to read
 *			in a new vector.
 *	Parameters:	NONE
 *	Return Value:
 *	  void	startNextVector - NONE
 *********************************************************************/
static void	startNextVector() 
{
  valueArray = NULL ;
  numValues = 0 ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		allocateArrays
 *	Description:	allocates the arrays for the example vectors
 *	Parameters:	NONE
 *	Return Value:
 *	  static void	allocateArrays - NONE
 *********************************************************************/
static void	allocateArrays() {
  int		idx ;

  inputArrays    = (Real **)calloc(numSlices,  sizeof(*inputArrays)) ;
  targetArrays   = (Real **)calloc(numSlices,  sizeof(*targetArrays)) ;
  
  nanInputArray  = (Real *)calloc(numInputs,  sizeof(*nanInputArray)) ;
  for (idx = 0 ; idx < numInputs ; ++idx)
    setNaN(nanInputArray[idx]) ;

  nanTargetArray = (Real *)calloc(numTargets, sizeof(*nanTargetArray)) ;
  for (idx = 0 ; idx < numTargets ; ++idx)
    setNaN(nanTargetArray[idx]) ;

  for (idx = 0 ; idx < numSlices ; ++idx) {
    inputArrays[idx]  = nanInputArray ;
    targetArrays[idx] = nanTargetArray ;
  }
}
/********************************************************************/


/*********************************************************************
 *	Name:		getValueArray
 *	Description:	returns the array of values, allocating it if
 *			necessary.
 *	Parameters:
 *	  Mode		mode - Input or Target
 *	Return Value:
 *	  static Real	*getValueArray - the array.
 *********************************************************************/
static Real	*getValueArray(arrayMode) 
  Mode		arrayMode ;
{
  int	idx, numElements ;

  if (valueArray == NULL) {
    if (arrayMode == Input)
      numElements = numInputs ;
    else
      numElements = numTargets ;

    maxValues = numElements ;
    valueArray = (Real *)calloc(numElements, sizeof(*valueArray)) ;
    for (idx = 0 ; idx < numElements ; ++idx)
      setNaN(valueArray[idx]) ;
  }
  
  return valueArray ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		addExample
 *	Description:	adds an example to the already set exampleSet
 *	Parameters:	NONE
 *	Return Value:
 *	  static void	addExample - NONE
 *********************************************************************/
static void	addExample(name) 
  char		*name ;
{
  int	idx ;
  idx = exampleSetAddExampleList(exampleSet, numSlices,
				 inputArrays, numInputs,
				 targetArrays, numTargets) ;
  if (idx >= 0 && tag)
    exampleSet->example[idx].name = strdup(tag) ;
}
/********************************************************************/
