
/**********************************************************************
 * $Id: exampleCom.c,v 1.8 93/03/29 14:14:01 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <stdio.h>

#include <xerion/display.h>
#include <xerion/commands.h>
#include "exampleParse.h"

struct TRACE doExamples ;

/***********************************************************************
 *	Name:		sumUnits
 *	Description:	increments an integer (address passed in as data)
 *			by the number of units in the group
 *	Parameters:	
 *		Group	group - the group to sum
 *		void	*data - a pointer to an int to increment.
 *	Return Value:	NONE
 ***********************************************************************/
static void	sumUnits (group, data)
  Group		group ;
  void		*data ;
{
  *(int *)data += group->numUnits ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_addExamples
 *	Description:	adds examples to a specified example set in
 *			the currentNet.
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_addExamples - 0 on failure, 1 on success
 ***********************************************************************/
int	command_addExamples (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  ExampleSet	exampleSet ;
  int		idx, newMask, setMask ;
  char		*fileName ;
  FILE		*fp ;
  int		numClamped, numTargets, numSlices  ;


  IUsage("-type <type> [ <file> ]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("adds examples to the specified example set of a net") ;
    IHelp
      (IHelpArgs,
       "addExamples reads an input file of example records and  adds them to",
       "an example set. The example set is specified by <type>  which should",
       "be one of   TESTING, TRAINING, or  VALIDATION.  Each  record  in the",
       "input file should be of the form:",
       "",
       "\tinput_1 ... input_n , target_1 ... target_m ;",
       "",
       "where n is  the number of inputs  and m the number of targets.  Each",
       "example is added to the  end  of  the specified example list in  the",
       "current net.  If no input file is specified, standard input is read.",
       "The data file is free format.  Examples may also have \"tags\".  A tag",
       "is  a name given to an example, and  shown in the Activation Display",
       "example  list.  Tags are specified  by preceding the example  with a",
       "line starting with the string \"tag:\" followed by the desired tag.",
       "",
       "If the network to which the examples are being added is a time delay",
       "network each example  consists  of vectors  of  inputs followed  by",
       "vectors  of targets.  Each vector   must be  preceded  by the range,",
       "surrounded by  square brackets, of  time slices it applies  to (e.g.",
       "[1-4,6]). Special keywords are also allowed in the range:",
       "",
       "\tn-m    - the range of numbers n through m (m >= n)",
       "\tall    - specifies that all time slices use this vector.",
       "\tlast n - specifies that the last n time slices use this vector.",
       "\tnext n - specifies that the next n time slices use this vector.",
       "",
       "These keywords  may be combined, separated by   commas, in a  single",
       "range of units.",
       "",
       "There need not be one input and one target for every time  slice. If",
       "an  input vector  is  not specified for   a  particular  time, it is",
       "assumed that all input units are unclamped at that time.  Similarly,",
       "for  targets it  is assumed that  the units have no  target value at",
       "that time. To specify that an individual unit has  no input (target)",
       "at a given time, set its value to the string \"NaN\" (Not a Number).",
       "EXAMPLES",
       "",
       "\txerion-> addExamples -type TESTING file.ex",
       "",
       "For a normal (not time delay) network with 5 inputs  and  2 targets,",
       "file.ex may contain:",
       "",
       "\ttag: First example",
       "\t0.0 1.0 0.0 0.0 1.0, 1.0 1.0 ;",
       "\ttag: Second example",
       "\t0.0 0.0 0.0 0.0 0.0, 1.0 0.0 ;",
       "",
       "For the same network, but with 10 time slices:",
       "",
       "\t[all]\t\t0.0 1.0 0.0 0.0 1.0,",
       "\t[last 3]\t1.0 1.0 ;",
       "",
       "\t[0]\t\t0.0 0.0 0.0 0.0 0.0",
       "\t[1]\t\t0.2 0.0 0.0 0.0 0.0",
       "\t[next 2]\t0.5 0.0 0.0 0.0 0.0",
       "\t[next 2]\t1.0 0.0 0.0 0.0 0.0,",
       "\t[0,last 2]\t1.0 0.0 ;",
       "",
       "SEE ALSO",
       "deleteExamples",
       NULL);
    return 0;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;
  
  if (tokc < 3 || tokc > 4)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  setMask = 0 ;
  for (idx = 1 ; 
       idx < tokc && 
       strncmp(tokv[idx], "-type", strlen(tokv[idx])) == 0 ; idx += 2) {
    newMask = findClassMask(EXAMPLESET_CLASS, tokv[idx+1]) ;
    if (newMask == 0) {
      IErrorAbort("Unknown example set type: \"%s\".", tokv[idx+1]) ;
      return 0 ;
    }    
    setMask |= newMask ;
  }
  
  if (setMask != TRAINING && setMask != VALIDATION && setMask != TESTING)
    IErrorAbort("Unknown example set type: \"0x%0x\".", setMask) ;
  
  exampleSet = netGetExampleSet(currentNet, setMask) ;

  fileName = NULL ;
  if (idx = tokc - 1)
    fileName = tokv[idx] ;
  else if (idx != tokc)
    IErrorAbort(IPrintUsage(tokv[0], usage)) ;

  /* set the proper input source */
  if (fileName == NULL) {
    fp = din ;
  } else {
    fp = IOpenFileOrAbort(fileName, "r", NULL) ;
  }

  if (exampleSet == NULL)
    IErrorAbort("No example set specified for net \"%s\".", currentNet->name) ;

  numClamped = 0 ;
  netForAllGroups(exampleSet->net, 
		  exampleSet->clampMask, sumUnits, &numClamped) ;

  numTargets = 0 ;
  netForAllGroups(exampleSet->net, 
		  exampleSet->targetMask, sumUnits, &numTargets) ;

  numSlices = exampleSet->net->timeSlices ;

  /* read the records, which are of the form:
   * input_1 input_2 ... input_n, target_1 target_2 ... target_m ;
   */
  readExampleFile(fp, exampleSet, numSlices, numClamped, numTargets) ;

  if (fp != din)
    ICloseFile(fp, NULL) ;

  markToRebuildDisplay(EXAMPLE_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_deleteExamples
 *	Description:	clears all examples from the specified set in
 *			the currentNet.
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_deleteExamples - 0 on failure, 1 on success
 ***********************************************************************/
int	command_deleteExamples (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int		idx, newMask, setMask ;
  ExampleSet	exampleSet ;

  IUsage("-type <type>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("deletes examples from the specified example set") ;
    IHelp
      (IHelpArgs,
       "deleteExamples deletes all examples from the  specified example set.",
       "<type>  specifies which  example set to  use. It   should be  one of",
       "TRAINING, TESTING, or VALIDATION.",
       "EXAMPLE",
       "\txerion-> deleteExamples -type TRAINING",
       "",
       "removes all training examples from the current net.",
       "SEE ALSO",
       "addExamples",
       NULL);
    return 1;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  if (tokc != 3)
    IErrorAbort(IPrintUsage(tokv[0],usage));

  setMask = 0 ;
  for (idx = 1 ; 
       idx < tokc && 
       strncmp(tokv[idx], "-type", strlen(tokv[idx])) == 0 ; idx += 2) {
    newMask = findClassMask(EXAMPLESET_CLASS, tokv[idx+1]) ;
    if (newMask == 0) {
      IErrorAbort("Unknown example set type: \"%s\".", tokv[idx+1]) ;
      return 0 ;
    }    
    setMask |= newMask ;
  }

  exampleSet = netGetExampleSet(currentNet, setMask) ;

  if (exampleSet == NULL)
    IErrorAbort("Could not get example set: \"0x%0x\".", setMask) ;

  while (exampleSet->numExamples > 0)
    exampleSetDeleteExample(exampleSet, exampleSet->numExamples - 1) ;

  markToRebuildDisplay(EXAMPLE_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_doExample
 *	Description:	A command to process a single example through 
 *			the network.
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_template - 0 on failure, 1 on success
 ***********************************************************************/
int	command_doExample(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  String	name ;
  Mask		setMask ;
  int		setIdx, exampleIdx ;

  
  /***********************
   * Handle all the help stuff
   */
  IUsage("[ -type <type>] <n>");
  if (GiveHelp(tokc)) {
    ISynopsis("process a specific example through the net");
    IHelp
      (IHelpArgs,
       "This  command  processes  a  single  example  through  the  network,",
       "updating  the network activations. The optional argument \"-type\" can",
       "be used  to select the example  set  from which  to get the  example",
       "(<type> should be one of TRAINING, TESTING, or  VALIDATION).  If  no",
       "type  is specified, TRAINING  is used. You  must also give the index",
       "(starting at 0) of the example in the example set.",
       "",
       "This command  is  most useful  when used  in  conjunction  with  the",
       "\"print\" or \"show\" command.",
       "",
       "EXAMPLE ",
       "To process the fourth example from the testing set, use the command:",
       "",
       "  xerion-> doExample -t TESTING 3",
       "SEE ALSO",
       "doExamples, print, show",
       NULL) ;
    return 1 ;
  }

  /***********************
   * Parse the command line options 
   */
  name = *tokv ;
  setMask = 0 ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-type", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      setMask = findClassMask(EXAMPLESET_CLASS, *tokv) ;
      if (setMask == 0)
	IErrorAbort("Unknown example set: \"%s\".", *tokv) ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }
  if (setMask == 0)
    setMask = TRAINING ;

  if (tokc != 1 || !IIsInteger(*tokv)) 
    IErrorAbort(IPrintUsage(name, usage)) ;
  else
    exampleIdx = atol(*tokv) ;

  switch(setMask) {
  case TRAINING:
    setIdx = 0 ;
    break ;
  case TESTING:
    setIdx = 1 ;
    break ;
  case VALIDATION:
    setIdx = 2 ;
    break ;
  default:
    IErrorAbort("Unknown example set type: \"0x%0x\".", setMask) ;
    break ;
  }

  /***********************
   * Now, do the example
   */
  if (doExample(setIdx, exampleIdx) < 0)
    IErrorAbort("No such example") ;
  
  markToRebuildDisplay(ACTIVITY_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_doExamples
 *	Description:	A command to process all the examples in a set
 *			through the network.
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_template - 0 on failure, 1 on success
 ***********************************************************************/
int	command_doExamples(tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  String	name ;
  Mask		setMask ;
  int		setIdx, exampleIdx ;

  
  /***********************
   * Handle all the help stuff
   */
  IUsage("[ -type <type>]");
  if (GiveHelp(tokc)) {
    ISynopsis("process an example set through the net");
    IHelp
      (IHelpArgs,
       "This  command  processes  a complete  example set through  the  network,",
       "updating  the network activations. The optional argument \"-type\" can",
       "be used  to select the example  set  from which  to get the  example",
       "(<type> should be one of TRAINING, TESTING, or  VALIDATION).  If  no",
       "type  is specified, TRAINING  is used. ",
       "",
       "This  command  registers   a  trace  called  \"doExamples\"  which  is",
       "triggered after each example is processed. This trace can be used to",
       "print network activations, etc.",
       "",
       "EXAMPLES",
       "To process the testing set, use the command:",
       "",
       "  xerion-> doExamples -t TESTING",
       "",
       "To  print  the  ouput of a  specific  unit  after  each  example  is",
       "processed, use:",
       "",
       "  xerion-> addTrace doExamples \"print currentNet.group[2].unit[0].output\"",
       "",
       "SEE ALSO",
       "doExample, addTrace, print, show",
       NULL) ;
    return 1 ;
  }

  /***********************
   * Parse the command line options 
   */
  name = *tokv ;
  setMask = 0 ;
  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-type", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      setMask = findClassMask(EXAMPLESET_CLASS, *tokv) ;
      if (setMask == 0)
	IErrorAbort("Unknown example set: \"%s\".", *tokv) ;
    } else if (*tokv[0] == '-') {
      IErrorAbort(IPrintUsage(name, usage)) ;
    } else {
      break ;
    }
  }
  if (setMask == 0)
    setMask = TRAINING ;

  if (tokc != 0)
    IErrorAbort(IPrintUsage(name, usage)) ;

  switch(setMask) {
  case TRAINING:
    setIdx = 0 ;
    break ;
  case TESTING:
    setIdx = 1 ;
    break ;
  case VALIDATION:
    setIdx = 2 ;
    break ;
  default:
    IErrorAbort("Unknown example set type: \"0x%0x\".", setMask) ;
    break ;
  }

  /***********************
   * Now, do the example
   */
  for (exampleIdx = 0 ; doExample(setIdx, exampleIdx) >= 0 ; ++exampleIdx)
    IDoTrace(&doExamples) ;

  if (exampleIdx == 0)
    IErrorAbort("Example Set is empty") ;
  
  markToRebuildDisplay(ACTIVITY_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/

/***********************************************************************
 *      Name:         command_permuteExamples
 *      Description:  permutes the chosen set of examples.
 *                    int   tokc   - the number of command line tokens
 *                    char *tokv[] - the vector of tokens
 *      Return Value: int command_permuteExamples - 0 failure, 1 success
 ***********************************************************************/
int	command_permuteExamples(tokc, tokv)
  int   tokc;
  char *tokv[];
{
  int		newMask;
  ExampleSet	exampleSet;

  IUsage("[-type <type>]");
  if (GiveHelp(tokc)) {
    ISynopsis("permute a set of examples");
    IHelp(IHelpArgs,
	  "When batchSize != trainingExampleSetSize, it is desirable to shuffle",
	  "the trainingExampleSet  after each pass in order to  break  spurious",
	  "temporal correlations between training examples.",
	  "",
	  "permuteExamples shuffles  the  specified example  set  in  order  to",
	  "destroy  the spurious correlations between successive examples.  The",
	  "optional <type>  should  be  TRAINING, TESTING  or  VALIDATION;  the",
	  "default is TRAINING.",
	  "",
	  "permuteExamples is  invoked automatically after all  the examples in",
	  "an example  set have  been  processed  for  those sets  that have  a",
	  "non-zero ``permute'' flag in their record.",
	  "",
	  "AUTHOR",
	  "Nici Schraudolph (schraudo@helmholtz.sdsc.edu)",
	  NULL);
    return(1);
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.");
   
  if (tokc > 3 || tokc == 2)
    IErrorAbort(IPrintUsage(tokv[0], usage));

  if (tokc > 1) {
    if (strncmp(tokv[1], "-type", strlen(tokv[1])))
      IErrorAbort(IPrintUsage(tokv[0], usage));
    if ((newMask = findMask(tokv[2])) == 0)
      IErrorAbort("Unknown example set type: \"%s\".", tokv[2]);
  } else
    newMask = TRAINING;

  exampleSet = netGetExampleSet(currentNet, newMask) ;

  if (exampleSet == NULL)
    IErrorAbort("Current net has no \"%s\" example set.", tokv[2]);
   
  exampleSetPermute(exampleSet);

  markToRebuildDisplay(ACTIVITY_DISPLAY) ;
  return(1);
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_moveExample
 *	Description:	adds examples to a specified example set in
 *			the currentNet.
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_addExamples - 0 on failure, 1 on success
 ***********************************************************************/
int	command_moveExample (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int		mask ;
  ExampleSet	fromSet, toSet ;
  int		fromIdx, toIdx ;
  char		*name ;

  IUsage("-type <fromType> <n> -type <toType> <n>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("move an example from one set to another") ;
    IHelp
      (IHelpArgs,
       "moveExample  moves  an  example  from  one  example  set  to another",
       "(possibly the same) example set. The arguments passed to the command",
       "are: ",
       "  <fromSet> -  one  of TRAINING, TESTING, or VALIDATION. This is the",
       "  	set to move the example from.",
       "  <fromIdx> - the index of the example  within the fromSet (starting",
       "              at 0).",
       "  <toSet>   -  one  of TRAINING, TESTING, or VALIDATION. This is the",
       "  	set to move the example to.",
       "  <toIdx> -  the  index within the toSet (starting  at  0) where you",
       "  	wish to put the example.",
       "",
       "It is an error to  specify a from index outside the range of already",
       "existing examples, or  a  to index  more than  one greater  than the",
       "number of existing examples.",
       NULL);
    return 0;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;
  
  if (tokc != 7)
    IErrorAbort(IPrintUsage(name, usage)) ;

  name = *tokv ;
  ++tokv, --tokc ;

  /* get everything about the "from" example */
  if (strncmp(*tokv, "-type", strlen(*tokv)))
    IErrorAbort(IPrintUsage(name, usage)) ;
  ++tokv, --tokc ;

  fromSet = netGetExampleSet(currentNet,
			     findClassMask(EXAMPLESET_CLASS, *tokv)) ;
  if (fromSet == NULL)
    IErrorAbort("Unable to find example set %s", *tokv) ;
  ++tokv, --tokc ;

  if (!IIsInteger(*tokv))
    IErrorAbort(IPrintUsage(name, usage)) ;
  fromIdx = atoi(*tokv) ;
  if (fromIdx < 0 || fromIdx >= fromSet->numExamples)
    IErrorAbort("Unable to find example %d", fromIdx) ;
  ++tokv, --tokc ;

  /* get everything about the "to" example */
  if (strncmp(*tokv, "-type", strlen(*tokv)))
    IErrorAbort(IPrintUsage(name, usage)) ;
  ++tokv, --tokc ;

  toSet = netGetExampleSet(currentNet,
			   findClassMask(EXAMPLESET_CLASS, *tokv)) ;
  if (toSet == NULL)
    IErrorAbort("Unable to find example set %s", *tokv) ;
  ++tokv, --tokc ;

  if (!IIsInteger(*tokv))
    IErrorAbort(IPrintUsage(name, usage)) ;
  toIdx = atoi(*tokv) ;
  if (toIdx < 0 || toIdx > toSet->numExamples)
    IErrorAbort("Unable to find example %d", toIdx) ;
  ++tokv, --tokc ;

  exampleSetMoveExample(fromSet, fromIdx, toSet, toIdx) ;

  markToRebuildDisplay(EXAMPLE_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		command_moveExamples
 *	Description:	adds examples to a specified example set in
 *			the currentNet.
 *	Parameters:	
 *		int	tokc    - the number of command line tokens
 *		char	*tokv[] - the vector of tokens
 *	Return Value:	
 *		int	command_addExamples - 0 on failure, 1 on success
 ***********************************************************************/
int	command_moveExamples (tokc, tokv)
  int	tokc ;
  char	*tokv[] ;
{
  int		mask ;
  ExampleSet	fromSet, toSet ;
  char		*name ;

  IUsage("-type <fromType> -type <toType>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("move all examples from one set to another") ;
    IHelp
      (IHelpArgs,
       "moveExample  moves all examples  from one  example  set  to  another",
       "(possibly the same) example set. The arguments passed to the command",
       "are:",
       "  <fromSet> -  one  of TRAINING, TESTING, or VALIDATION. This is the",
       "  	set to move the example from.",
       "  <toSet>   -  one  of TRAINING, TESTING, or VALIDATION. This is the",
       "  	set to move the example to.",
       "",
       "All examples are moved to the end of the example set.",
       NULL);
    return 0;
  }

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;
  
  if (tokc != 5)
    IErrorAbort(IPrintUsage(name, usage)) ;

  name = *tokv ;
  ++tokv, --tokc ;

  /* get everything about the "from" example */
  if (strncmp(*tokv, "-type", strlen(*tokv)))
    IErrorAbort(IPrintUsage(name, usage)) ;
  ++tokv, --tokc ;

  fromSet = netGetExampleSet(currentNet,
			     findClassMask(EXAMPLESET_CLASS, *tokv)) ;
  if (fromSet == NULL)
    IErrorAbort("Unable to find example set %s", *tokv) ;
  ++tokv, --tokc ;

  /* get everything about the "to" example */
  if (strncmp(*tokv, "-type", strlen(*tokv)))
    IErrorAbort(IPrintUsage(name, usage)) ;
  ++tokv, --tokc ;

  toSet = netGetExampleSet(currentNet,
			   findClassMask(EXAMPLESET_CLASS, *tokv)) ;
  if (toSet == NULL)
    IErrorAbort("Unable to find example set %s", *tokv) ;
  ++tokv, --tokc ;

  while (fromSet->numExamples > 0)
    exampleSetMoveExample(fromSet, 0, toSet, toSet->numExamples) ;

  markToRebuildDisplay(EXAMPLE_DISPLAY) ;

  return 1 ;
}
/**********************************************************************/
