
/**********************************************************************
 * $Id: constraintCom.c,v 1.4 92/11/30 13:16:49 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		       Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute,  and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided  that the above copyright notice  appears in all copies and
 * that both the copyright notice and this permission notice  appear in
 * supporting documentation, and  that  the  name of The University  of
 * Toronto  not  be used  in advertising   or publicity pertaining   to
 * distribution  of   the software   without  specific, written   prior
 * permission.  The  University  of Toronto  makes   no representations
 * about the  suitability  of  this software  for  any purpose.   It is
 * provided "as is" without express or implied warranty.
 *
 * THE  UNIVERSITY OF  TORONTO DISCLAIMS ALL WARRANTIES  WITH REGARD TO
 * THIS SOFTWARE,  INCLUDING ALL  IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT  SHALL THE UNIVERSITY  OF TORONTO BE LIABLE
 * FOR ANY SPECIAL,  INDIRECT OR CONSEQUENTIAL  DAMAGES  OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF  USE, DATA OR PROFITS,  WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE  OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF OR  IN  CONNECTION   WITH  THE  USE OR  PERFORMANCE  OF THIS
 * SOFTWARE.
 **********************************************************************/

#include <xerion/commands.h>

/*******************************************************************
 *	Private functions
 *******************************************************************/
static int	constrainLinks ARGS((Link *link, Link *linkTo, 
				     int numLinks, double scale)) ;
static int	constrainBiLinks ARGS((Link *link, Link *linkTo, 
				       int numLinks, int numToLinks,
				       double scale)) ;
/*******************************************************************/


/*******************************************************************
 *	Name:		constrainLink
 *	Description:	
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_constrainLink - 
 *******************************************************************/
int	command_constrainLink(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Link		link[2] ;
  Real		scale ;

  IUsage("[ -scale <scale> ] <link1> <link2>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("constrain one link to be a linear multiple of another") ;
    IHelp
      (IHelpArgs,
       "\"constrainLink\"  constrains  one link to  be   a linear multiple  of",
       "another.  <scale> is the linear constraint applied  to <link1>.  Its",
       "default value is 1.0.  ",
       "EXAMPLES",
       "To  constrain  link  \"Hidden.0->Output.0\"  to be   the   negative of",
       "\"Output.0->Hidden.0\", use the command:",
       "",
       "xerion-> constrainLink -scale -1.0 \"Hidden.0->Output.0\" \"Output.0->Hidden.0\"",
       "WARNING",
       "The order of  the arguments for this command is important. The first",
       "link  is constrained  to be the same as the second. If the first one",
       "is already constrained, the existing constraint is broken before the",
       "new  one  is  made.  The second  link  is  not  modified  (i.e.  its",
       "constraints are not deleted).",
       "SEE ALSO",
       "constrainUnit,   constrainGroup,   unconstrainLink, unconstrainUnit,",
       "unconstrainGroup",
       NULL);
    return 1;
  }
  
  name = *tokv ;
  ++tokv, --tokc ;

  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  /* Parse command line */
  if (tokc == 4) {
    if (strncmp(*tokv, "-scale", strlen(*tokv)))
      IErrorAbort(IPrintUsage(name, usage)) ;

    ++tokv, --tokc ;
    if (!IIsNumber(*tokv))
      IErrorAbort(IPrintUsage(name, usage)) ;
    
    scale = atof(*tokv) ;
    ++tokv, --tokc ;
  } else {
    scale = 1.0 ;
  }

  if (tokc != 2)
    IErrorAbort(IPrintUsage(name, usage)) ;

  name  = *tokv ;
  link[0] = linkFromName(currentNet, *tokv) ;
  if (link[0] == NULL)
    IErrorAbort("Unknown link: \"%s\".", *tokv) ;
  ++tokv, --tokc ;
  
  link[1] = linkFromName(currentNet, *tokv) ;
  if (link[1] == NULL)
    IErrorAbort("Unknown link: \"%s\".", *tokv) ;

  /* constrain the links */
  constrainLinks(&link[0], &link[1], (int)1, scale) ;

  markToRebuildDisplay(CONNECTION_DISPLAY) ;
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		constrainUnit
 *	Description:	constrains all incoming links of one unit
 *			with those of another
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_constrainUnit - 
 *******************************************************************/
int command_constrainUnit(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Unit		unit[2] ;
  Real		scale ;
  Boolean	bidirectional ;
  
  IUsage("[-bidirectional] [-scale <scale>] <unit1> <unit2>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("constrain incoming links of one unit with those of another") ;
    IHelp
      (IHelpArgs,
       "\"constrainUnit\" forces all incoming links  of <unit1> to be linearly",
       "constrained with  the  incoming links  of <unit2>.   Both units must",
       "have the same number of incoming links.",
       "",
       "<scale> is  the linear  constraint applied to  the links of <unit1>.",
       "Its default value is 1.0.",
       "",
       "If the \"-bidirectional\" flag is given, then only bidirectional links",
       "are constrained.   That  is, the  link  from  <unit1>  to <unit2> is",
       "constrained with  the corresponding  link  from <unit2>  to <unit1>,",
       "(assuming that both  exist). In this case,  the units need  not have",
       "the same number of incoming links.",
       "EXAMPLES",
       "To  constrain all  incoming links  of  \"Output.0\" to be  the same as",
       "those of \"Output.1\", use the command:",
       "",
       "	xerion-> constrainUnit Output.0 Output.1",
       "",
       "To  constrain the same  units, but  to make  the incoming links   of",
       "\"Output.0\" be the negative to those of \"Output.1\" use:",
       "",
       "	xerion-> constrainUnit -scale -1.0 Output.0 Output.1",
       "",
       "To constrain only the  bidirectional  links between  \"Output.1\"  and",
       "\"Output.0\", use:",
       "",
       "	xerion-> constrainUnit -bidirectional Output.0 Output.1",
       "WARNING",
       "The order of the arguments for this command is important. The  links",
       "of  the first unit are constrained to be the  same  as those of  the",
       "second.  If the  first ones  are  already constrained, the  existing",
       "constraints are broken  before the  new ones are made.  The links of",
       "the second  unit are not modified  (i.e.  their constraints  are not",
       "deleted).",
       "SEE ALSO ",
       "constrainGroup,  constrainLink,   unconstrainLink,  unconstrainUnit,",
       "unconstrainGroup",
       NULL);
    return 1;
  }
  
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  /* Parse the command line */
  name  = *tokv ;
  scale = 1.0 ;
  bidirectional = FALSE ;
  for (++tokv, --tokc ; tokc > 2 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-scale", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      if (!IIsNumber(*tokv))
	IErrorAbort(IPrintUsage(name, usage)) ;
      scale = atof(*tokv) ;
    } else if (strncmp(*tokv, "-bidirectional", strlen(*tokv)) == 0) {
      bidirectional = TRUE ;
    } else {
      IErrorAbort(IPrintUsage(name, usage)) ;
    }
  }

  if (tokc != 2)
    IErrorAbort(IPrintUsage(name, usage)) ;

  unit[0] = unitFromName(currentNet, *tokv) ;
  if (unit[0] == NULL)
    IErrorAbort("Unknown unit: \"%s\".", *tokv) ;
  ++tokv, --tokc ;
  
  unit[1] = unitFromName(currentNet, *tokv) ;
  if (unit[1] == NULL)
    IErrorAbort("Unknown unit: \"%s\".", *tokv) ;

  if (bidirectional) {
    /* constrain only bidirectional links */
    constrainBiLinks(unit[0]->incomingLink, unit[1]->incomingLink,
		     unit[0]->numIncoming, unit[1]->numIncoming, scale) ;
  } else {
    /* make sure there are the same number of incoming then constrain them */
    if (unit[0]->numIncoming != unit[1]->numIncoming)
      IErrorAbort("\"%s\" and \"%s\" have different number of incoming links",
		  unit[0]->name, unit[1]->name) ;

    constrainLinks(unit[0]->incomingLink, unit[1]->incomingLink,
		   unit[0]->numIncoming, scale) ;
  }

  markToRebuildDisplay(CONNECTION_DISPLAY) ;
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		constrainGroup
 *	Description:	constrain all incoming links of one group 
 *			with those of another
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_constrainGroup - 
 *******************************************************************/
int command_constrainGroup(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  Group		group[2] ;
  Unit		*unit, *unitTo ;
  int		idx ;
  Real		scale ;
  Boolean	bidirectional ;

  IUsage("[-bidirectional] [-scale <scale>] <group1> <group2>") ;
  if (GiveHelp(tokc)) {
    ISynopsis("constrain incoming links of one group with those of another") ;
    IHelp
      (IHelpArgs,
       "\"constrainGroup\" forces the incoming links of each unit  in <group1>",
       "to  be linearly   constrained   with  the   incoming links  of   the",
       "corresponding  unit in  <group2>.  Both  groups  must  have the same",
       "number of units and both units in a pair  must have  the same number",
       "of incoming links.",
       "",
       "<scale> is the  linear constraint  applied to the links of <group1>.",
       "Its default value is 1.0. ",
       "",
       "If the \"-bidirectional\" flag is given, then only bidirectional links",
       "are constrained.  That  is, the link  from  <unit1>  in <group1>  to",
       "<unit2> in <group2> is constrained with the corresponding  link from",
       "<unit2> to  <unit1>, (assuming that both  exist). In  this case, the",
       "groups  need not be the  same size, and  the units need not have the",
       "same number of incoming links.",
       "EXAMPLES",
       "To constrain all incoming links of  the units in the group \"Hidden1\"",
       "to be the same as those of \"Hidden2\", use the command:",
       "",
       "	xerion-> constrainGroup Hidden1 Hidden2",
       "",
       "To  constrain the same  groups, but  to make  the incoming links   of",
       "\"Hidden1\" be the negative to those of \"Hidden2\" use:",
       "",
       "	xerion-> constrainGroup -scale -1.0 Hidden1 Hidden2",
       "WARNING",
       "The order of the arguments for this  command is important. The links",
       "of the first group are constrained to be  the same  as  those of the",
       "second.   If  the  first ones are  already constrained, the existing",
       "constraints are broken before  the  new ones are made.  The links of",
       "the second group are  not modified  (i.e.  their constraints are not",
       "deleted).",
       "SEE ALSO",
       "constrainUnit,  constrainLink,    unconstrainLink,  unconstrainUnit,",
       "unconstrainGroup",
       NULL);
    return 1;
  }
  
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  /* Parse the command line */
  scale = 1.0 ;
  name  = *tokv ;
  bidirectional = FALSE ;
  for (++tokv, --tokc ; tokc > 2 ; ++tokv, --tokc) {
    if (strncmp(*tokv, "-scale", strlen(*tokv)) == 0) {
      ++tokv, --tokc ;
      if (!IIsNumber(*tokv))
	IErrorAbort(IPrintUsage(name, usage)) ;
      scale = atof(*tokv) ;
    } else if (strncmp(*tokv, "-bidirectional", strlen(*tokv)) == 0) {
      bidirectional = TRUE ;
    } else {
      IErrorAbort(IPrintUsage(name, usage)) ;
    }
  }

  if (tokc != 2)
    IErrorAbort(IPrintUsage(name, usage)) ;

  name  = *tokv ;
  group[0] = groupFromName(currentNet, *tokv) ;
  if (group[0] == NULL)
    IErrorAbort("Unknown group: \"%s\".", *tokv) ;
  ++tokv, --tokc ;
  
  group[1] = groupFromName(currentNet, *tokv) ;
  if (group[1] == NULL)
    IErrorAbort("Unknown group: \"%s\".", *tokv) ;

  unit   = group[0]->unit ;
  unitTo = group[1]->unit ;
  if (bidirectional) {
    /* traverse all the units and find all bidirectional links */
    for (idx = 0 ; idx < group[0]->numUnits ; ++idx) {
      int	idxTo ;
      for (idxTo = 0 ; idxTo < group[1]->numUnits ; ++idxTo) {
	constrainBiLinks(unit[idx]->incomingLink, 
			 unitTo[idxTo]->incomingLink,
			 unit[idx]->numIncoming, 
			 unitTo[idxTo]->numIncoming, scale) ;
      }
    }
  } else {
    if (group[0]->numUnits != group[1]->numUnits)
      IErrorAbort("\"%s\" and \"%s\" have different number of units",
		  group[0]->name, group[1]->name) ;
    /* constrain pairwise */
    for (idx = 0 ; idx < group[0]->numUnits ; ++idx) {
      if (unit[idx]->numIncoming != unitTo[idx]->numIncoming)
	IErrorAbort ("\"%s\" and \"%s\" have different number of incoming links",
		     unit[idx]->name, unitTo[idx]->name) ;
      constrainLinks(unit[idx]->incomingLink, unitTo[idx]->incomingLink,
		     unit[idx]->numIncoming, scale) ;
    }
  }

  markToRebuildDisplay(CONNECTION_DISPLAY) ;
  return 1 ;
}
/******************************************************************/
static int	constrainLinks(link, linkTo, numLinks, scale) 
  Link		*link ;
  Link		*linkTo ;
  int		numLinks ;
  double	scale ;
{
  int		idx, status = 1 ;

  for (idx = 0 ; idx < numLinks ; ++idx) {
    if ((link[idx]->type & LOG_TRANSFORM)
	!= (linkTo[idx]->type & LOG_TRANSFORM))
      IErrorAbort("\"%s\" and \"%s\" are not both %s",
		  link[idx]->name, linkTo[idx]->name, 
		  classMaskName(LINK_CLASS, LOG_TRANSFORM)) ;

    status = netConstrainLink(currentNet, link[idx], linkTo[idx], scale) ;
    if (status != 1)
      break ;
  }

  return status ;
}
/******************************************************************/
static int	constrainBiLinks(link, linkTo, numLinks, numToLinks, scale) 
  Link		*link ;
  Link		*linkTo ;
  int		numLinks ;
  int		numToLinks ;
  double	scale ;
{
  int		idx, idxTo, status = 1 ;

  for (idx = 0 ; idx < numLinks ; ++idx) {
    Unit	preUnit  = link[idx]->preUnit ;
    Unit	postUnit = link[idx]->postUnit ;

    for (idxTo = 0 ; idxTo < numToLinks ; ++idxTo) {
      if (preUnit == linkTo[idxTo]->postUnit
	  && postUnit == linkTo[idxTo]->preUnit) {
	status = netConstrainLink(currentNet, link[idx], linkTo[idxTo], scale);
	if (status != 1)
	  break ;
      }
    }
    if (status != 1) break ;
  }

  return status ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		unconstrainLink
 *	Description:	removes all constraints on a link
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_unconstrainLink - 
 *******************************************************************/
int command_unconstrainLink(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  
  IUsage("<link1> [ <link2> ...]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("remove any constraints on a link") ;
    IHelp
      (IHelpArgs,
       "\"unconstrainLink\" removes any constraints on a link.",
       "SEE ALSO",
       "unconstrainUnit,  unconstrainGroup,  constrainUnit,  constrainGroup,",
       "constrainLink",
       NULL);
    return 1;
  }
  
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  name  = *tokv ;
  if (tokc <= 1)
    IErrorAbort(IPrintUsage(name, usage)) ;

  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    Link	*link = linksFromRegex(currentNet, *tokv) ;
    int		idx ;
    if (link == NULL || *link == NULL)
      IErrorAbort("Unknown link: \"%s\".", *tokv) ;
    
    for (idx = 0 ; link[idx] != NULL ; ++idx)
      netUnconstrainLink(currentNet, link[idx]) ;

    free(link) ;
  }
  
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		unconstrainUnit
 *	Description:	removes all constraints on the incoming links
 *			to a unit
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_unconstrainUnit - 
 *******************************************************************/
int command_unconstrainUnit(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  
  IUsage("<unit1> [ <unit2> ...]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("remove constraints from incoming links to a unit") ;
    IHelp
      (IHelpArgs,
       "\"unconstrainUnit\"  removes constraints from   the  incoming links to",
       "<unit>.",
       "",
       "EXAMPLES",
       "To remove constraints from all incoming links of \"Output.0\", use the",
       "command:",
       "",
       "	xerion-> unconstrainUnit Output.0",
       "",
       "SEE ALSO",
       "unconstrainLink, unconstrainGroup,    constrainUnit, constrainGroup,",
       "constrainLink",
       NULL);
    return 1;
  }
  
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  name  = *tokv ;
  if (tokc <= 1)
    IErrorAbort(IPrintUsage(name, usage)) ;

  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    Unit	*unit = unitsFromRegex(currentNet, *tokv) ;
    int		numLinks, idx, unitIdx ;
    Link	*link ;

    if (unit == NULL || *unit == NULL)
      IErrorAbort("Unknown unit: \"%s\".", *tokv) ;

    for (unitIdx = 0 ; unit[unitIdx] != NULL ; ++unitIdx) {
      numLinks = unit[unitIdx]->numIncoming ;
      link     = unit[unitIdx]->incomingLink ;
      for (idx = 0 ; idx < numLinks ; ++idx)
	netUnconstrainLink(currentNet, link[idx]) ;
    }
    free(unit) ;
  }
  
  return 1 ;
}
/******************************************************************/


/*******************************************************************
 *	Name:		unconstrainGroup
 *	Description:	removes constraints on all incoming links to
 *			a group.
 *	Parameters:
 *	  int   tokc - 
 *	  char *tokv
 *	Return Value:
 *	  int command_unconstrainGroup - 
 *******************************************************************/
int command_unconstrainGroup(tokc,tokv)
  int   tokc;
  char *tokv[];
{
  String	name ;
  int		unitIdx ;

  IUsage("<group1> [ <group2> ...]") ;
  if (GiveHelp(tokc)) {
    ISynopsis("remove constraints from incoming links to units in a group") ;
    IHelp
      (IHelpArgs,
       "\"unconstrainGroup\"  removes constraints from   the  incoming links to",
       "all units in <group>.",
       "",
       "EXAMPLES",
       "To remove constraints from all the incoming links  of units in group",
       "\"Output\", use the command:",
       "",
       "	xerion-> unconstrainGroup Output",
       "",
       "SEE ALSO",
       "unconstrainLink, unconstrainUnit,    constrainUnit,  constrainGroup,",
       "constrainLink",
       NULL);
    return 1;
  }
  
  if (currentNet == NULL)
    IErrorAbort("There is no current net.") ;

  name  = *tokv ;
  if (tokc <= 1)
    IErrorAbort(IPrintUsage(name, usage)) ;

  for (++tokv, --tokc ; tokc > 0 ; ++tokv, --tokc) {
    Group	*group = groupsFromRegex(currentNet, *tokv) ;
    int		groupIdx ;

    if (group == NULL || *group == NULL)
      IErrorAbort("Unknown group: \"%s\".", *tokv) ;

    for (groupIdx = 0 ; group[groupIdx] != NULL ; ++groupIdx) {
      for (unitIdx = 0 ; unitIdx < group[groupIdx]->numUnits ; ++unitIdx) {
	Unit	unit     = group[groupIdx]->unit[unitIdx] ;
	int	numLinks = unit->numIncoming ;
	Link	*link    = unit->incomingLink ;
	int	idx ;
      
	for (idx = 0 ; idx < numLinks ; ++idx)
	  netUnconstrainLink(currentNet, link[idx]) ;
      }
    }
    free(group) ;
  }
  
  return 1 ;
}
/******************************************************************/
