/**********************************************************************
 * $Id: display.c,v 1.4 93/01/13 21:08:48 xerion Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

/**********************************************************************
 * A whole mess o' include files.
 **********************************************************************/
#include <math.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>

#include <xerion/MenuPane.h>
#include <xerion/Plot.h>

#include <xerion/display.h>
#include <xerion/displayUtils.h>
#include <xerion/Trace.h>

#include <xerion/commands.h>
#include "hcl.h"

/*******************************************************************/
static Proc  loadPattern          ARGS((Group group, CanvasPoint *point)) ;

static void  redrawCB             ARGS((Widget, caddr_t, caddr_t)) ;
static void  clearCB	          ARGS((Widget, caddr_t, caddr_t)) ;
static void  popdownCB            ARGS((Widget, caddr_t, caddr_t)) ;
static void  redrawScatterPlot	  ARGS((Widget)) ;

static void  loadDisplayPatterns  ARGS((Net currentNet,ExampleSet exampleSet));
static void  loadDisplayMeans     ARGS((Net currentNet));
static Proc  loadPattern          ARGS((Group group, CanvasPoint *point));
static Proc  loadMean             ARGS((Unit unit, Real *data));
static Proc  incNetUnitCounter    ARGS((Unit  unit, void *data)) ;

static int	numPoints = 0;
static int	numCircles = 0;
static int	currentCircle = 0;

static CanvasPoint 	*points ;
static CanvasCircle	*circles ;

/*******************************************************************
 *	Name:		loadDisplayPatterns
 *	Description:	loads the 1st 2 components of each test pattern into
 *                      points[patternnum]->x and ->y.
 *			Called by redrawScatterPlot
 *	Parameters:
 *               Net        currentNet
 *               ExampleSet exampleSet
 *	Return Value:
 *	  static void	loadDisplayPatterns - NONE
 *******************************************************************/
static void	loadDisplayPatterns(currentNet,exampleSet)
  ExampleSet	exampleSet;
  Net		currentNet;  
{
  int exNum,
      numExamples = exampleSet->numExamples;
  if (numPoints != numExamples) {
    if (numPoints > 0) 
      free(points);
    points = (CanvasPoint*)calloc(numExamples, sizeof(CanvasPoint));
  }
  numPoints = numExamples;
  
  for (exNum = 0 ; exNum < numExamples ; ++exNum) {
    MgetNext(exampleSet) ;
    netForAllGroups(currentNet, INPUT, loadPattern, points+exNum) ;
    
  }
}

/*******************************************************************/

/***********************************************************************
 *	Name:		loadPattern
 *	Description:
 *               set point.x and point.y to the activations of the first 
 *               two input units.
 *      Parameters:
 *		Group	group     - the group object 
 *              CanvasPoint *point - the current point
 *                                  to be displayed on the widget
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static Proc  loadPattern (group, point)
  Group       group;
  CanvasPoint *point;
{
  if (group->type & ~INPUT) return;

  point->x = group->unit[0]->output;
  point->y = group->unit[1]->output;
}
/**********************************************************************/


/*******************************************************************
 *	Name:		loadDisplayMeans
 *	Description:	loads the 1st 2 weights of each unit into
 *                      circles[].x and .y and radius into .radius
 *			Called by redrawScatterPlot
 *	Parameters:
 *               Net        currentNet
 *	Return Value:
 *	  static void	loadDisplayMeans - NONE
  *******************************************************************/
static void	loadDisplayMeans(net)
  Net		net;  
{
  int	numUnits = MunitCounter(net);
  float	radius;

  if (numUnits == 0) {
    netForAllUnits(net, ~(INPUT | BIAS), incNetUnitCounter, net);
    numUnits = MunitCounter(net);
  }

  if (numCircles != numUnits) {
    if (numCircles > 0) 
      free(circles);
    circles = (CanvasCircle*)calloc(numUnits, sizeof(CanvasCircle));
  }
  numCircles = numUnits;
  currentCircle = 0;
  radius = Mradius(net) ;
  netForAllUnits(net, ~(INPUT | BIAS), loadMean, &radius) ;
    
}

/*******************************************************************/

/***********************************************************************
 *	Name:		loadMean
 *	Description:
 *               set circle.x and circle.y to the weights of the first 
 *               two links to unit. set circle.radius to radius.
 *      Parameters:
 *		Unit unit     - the unit object 
 *              Real *radius
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static Proc  loadMean (unit, radius)
  Unit        unit;
  Real        *radius;
{
  Link	*incoming   = unit->incomingLink ;

  circles[currentCircle].x = incoming[0]->weight;
  circles[currentCircle].y = incoming[1]->weight;
  circles[currentCircle].radius = *radius;
  ++currentCircle;
}
/**********************************************************************/
static Proc	incNetUnitCounter(unit, data)
  Unit	unit ;
  void	*data;
{
  Net   net = (Net)data ;
  ++(MunitCounter(net));
}

/*******************************************************************
 *	Name:		redrawScatterPlot
 *	Description:	Draws into the plot widget.
 *			Called by the "updateScatterDisplay" command and
 *			the "redraw" button.
 *	Parameters:
 *	  Widget	plot - the plot widget
 *	Return Value:
 *	  static void	redrawScatterPlot - NONE
 *******************************************************************/
static void	redrawScatterPlot(plot)
  Widget	plot ;
{
  if (currentNet == NULL) return;

  XtPlotClear(plot, TRUE);

  if (numPoints != currentNet->testingExampleSet->numExamples)
    loadDisplayPatterns(currentNet, currentNet->testingExampleSet);
  XtPlotAddPoints(plot, points, (int)numPoints, TRUE) ;

  loadDisplayMeans(currentNet);
  XtPlotAddCircles(plot, circles, (int)numCircles, TRUE) ;

  dispatchExposures() ;
}
/******************************************************************/


/*********************************************************************
 *	Name:		createScatterDisplay
 *	Description:	procedure to create the scatter display
 *	Parameters:
 *	  Widget	popup - the popup shell that will contain the
 *				display
 *	Return Value:
 *	  export void	createScatterDisplay - NONE
 *********************************************************************/
export void	createScatterDisplay(popup)
  Widget	popup ;
{
  Widget	shell, form, menu, menuPane ;
  Widget	menuButton, button, line ;
  Widget	canvas, plot ;

  form = XtVaCreateManagedWidget("form", formWidgetClass, popup, NULL) ;

  /********************************************
   * Pulldown menu pane 
   */
  menuPane = XtVaCreateManagedWidget("menuPane", menuPaneWidgetClass, form,
				     NULL) ;

  /********************************************
   * Make the plot widget
   */
  plot   = XtVaCreateManagedWidget("plot", plotWidgetClass, form, NULL) ;
  canvas = XtNameToWidget(plot, "canvas") ;

  /********************************************
   * Display Updates popup shell
   */
  shell = XtVaCreatePopupShell("scatterDisplay", traceWidgetClass, popup, 
			       NULL) ;
  XtAddCallback(shell, XtNcallback, redrawCB, (XtPointer)plot) ;

  /**********************
   * Options pulldown menu 
   */
  menuButton = XtVaCreateManagedWidget("options",
				       menuButtonWidgetClass, menuPane,
				       XtNmenuName, "optionsMenu",
				       NULL) ;

  menu = XtVaCreatePopupShell("optionsMenu", simpleMenuWidgetClass, menuButton,
			      NULL) ;
  
  button = XtVaCreateManagedWidget("updates", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, popupCenteredCB, (XtPointer)shell) ;

  button = XtVaCreateManagedWidget("reset", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, redrawCB, (XtPointer)plot) ;

  button = XtVaCreateManagedWidget("clear", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, clearCB, (XtPointer)canvas) ;

  line = XtVaCreateManagedWidget("line", smeLineObjectClass, menu, NULL) ;

  button = XtVaCreateManagedWidget("close", smeBSBObjectClass, menu, NULL) ;
  XtAddCallback(button, XtNcallback, popdownCB, (XtPointer)popup) ;

  /* clear the create popup resource (since we just created it) */
  XtVaSetValues(popup, XtNcreatePopupChildProc, NULL, NULL) ;
}
/******************************************************************/


/******************************************************************/
static void	redrawCB(widget, clientData, callData)
  Widget	widget ;
  caddr_t	clientData ;
  caddr_t	callData ;
{
  redrawScatterPlot((Widget)clientData) ;
}
/******************************************************************/
static void	clearCB(widget, clientData, callData)
  Widget	widget ;
  caddr_t	clientData ;
  caddr_t	callData ;
{
  XtCanvasClear((Widget)clientData, TRUE) ;
}
/******************************************************************/
static void	popdownCB(widget, clientData, callData)
  Widget	widget ;
  caddr_t	clientData ;
  caddr_t	callData ;
{
  XtPopdown((Widget)clientData) ;
}
/******************************************************************/
