/**********************************************************************
 * $Id: bp.c,v 1.8 92/12/01 14:48:41 drew Exp $
 **********************************************************************/

/**********************************************************************
 *   Copyright 1990,1991,1992,1993 by The University of Toronto,
 *		      Toronto, Ontario, Canada.
 * 
 *			 All Rights Reserved
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its  documentation for  any purpose is  hereby granted  without
 * fee, provided that the above copyright notice appears in all copies
 * and  that both the  copyright notice  and   this  permission notice
 * appear in   supporting documentation,  and  that the  name  of  The
 * University  of Toronto  not  be  used in  advertising or  publicity
 * pertaining   to  distribution   of  the  software without specific,
 * written prior  permission.   The  University of   Toronto makes  no
 * representations  about  the  suitability of  this software  for any
 * purpose.  It  is    provided   "as is"  without express or  implied
 * warranty.
 *
 * THE UNIVERSITY OF TORONTO DISCLAIMS  ALL WARRANTIES WITH REGARD  TO
 * THIS SOFTWARE,  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TORONTO  BE LIABLE
 * FOR ANY  SPECIAL, INDIRECT OR CONSEQUENTIAL  DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR  PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT  OF  OR  IN  CONNECTION WITH  THE  USE  OR PERFORMANCE  OF THIS
 * SOFTWARE.
 *
 **********************************************************************/

#include <stdio.h>
#include <math.h>

#include <xerion/useful.h>
#include <xerion/simulator.h>
#include <xerion/groupDisplay.h>
#include "unit.h"
#include "bp.h"

#include "help.h"

static void	calculateNetErrorDeriv ARGS((Net net, ExampleSet exampleSet)) ;
static void	calculateNetError      ARGS((Net net, ExampleSet exampleSet)) ;

static void	initNet		ARGS((Net	net)) ;
static void	deinitNet	ARGS((Net	net)) ;
static void	initGroup	ARGS((Group	group)) ;
static void	deinitGroup	ARGS((Group	group)) ;
static void	initUnit	ARGS((Unit	unit)) ;
static void	deinitUnit	ARGS((Unit	unit)) ;
static void	initLink	ARGS((Link	link)) ;
static void	deinitLink	ARGS((Link	link)) ;

static void	makeGroupDisplay ARGS(()) ;

static void	zeroUnitDerivs	ARGS((Unit unit, void *data)) ;
static Real	square       	ARGS((double  x)) ;
static Boolean	multipleBits 	ARGS((Mask)) ;

/***********************************************************************
 *	Name:		main 
 *	Description:	the main function, used for the xerion simulator
 *	Parameters:	
 *		int	argc	- the number of input args
 *		char	**argv  - array of argument strings from command 
 *				  line
 *	Return Value:	
 *		int	main	- 0
 ***********************************************************************/
int main (argc, argv)
  int	argc ;
  char	**argv ;
{
  /* Register all of the Unit types */
  registerClassMask(DISTANCE,	GROUP_CLASS, "DISTANCE") ;
  registerClassMask(DPROD,	GROUP_CLASS, "DPROD") ;
  registerClassMask(PI,		GROUP_CLASS, "PI") ;

  registerClassMask(EXPONENTIAL,GROUP_CLASS, "EXPONENTIAL") ;
  registerClassMask(NEGEXPONENTIAL,GROUP_CLASS, "NEGEXPONENTIAL") ;
  registerClassMask(TANH,	GROUP_CLASS, "TANH") ;
  registerClassMask(LINEAR,	GROUP_CLASS, "LINEAR") ;
  registerClassMask(LOGISTIC,	GROUP_CLASS, "LOGISTIC") ;
  registerClassMask(SOFTMAX,	GROUP_CLASS, "SOFTMAX") ;

  registerClassMask(CROSSENTROPY,GROUP_CLASS,"CROSSENTROPY") ;
  registerClassMask(SUMSQUARE,	GROUP_CLASS, "SUMSQUARE") ;

  registerClassMask(RBF,	GROUP_CLASS, "RBF") ;
  registerClassMask(IPI,	GROUP_CLASS, "IPI") ;
  registerClassMask(GAIN,	GROUP_CLASS, "GAIN") ;
  registerClassMask(ADAPTIVESUMSQUARE,	GROUP_CLASS, "ADAPTIVESUMSQUARE") ;

  registerClassMask(GATE,	GROUP_CLASS, "GATE") ;
  registerClassMask(GATED,	GROUP_CLASS, "GATED") ;

  makeGroupDisplay() ;

  /* Register the create and destroy hooks */
  setCreateNetHook(initNet) ;
  setCreateGroupHook(initGroup) ;
  setCreateUnitHook(initUnit) ;
  setCreateLinkHook(initLink) ;

  setDestroyNetHook(deinitNet) ;
  setDestroyGroupHook(deinitGroup) ;
  setDestroyUnitHook(deinitUnit) ;
  setDestroyLinkHook(deinitLink) ;

  /* Perform initialization of the simulator */
  IStandardInit(&argc, argv);

  /* Enter loop that reads commands and handles graphics */
  ICommandLoop(stdin, stdout, NULL);

  return 0 ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		initNet 
 *	Description:	sets the error calculation procedures for
 *			a network as well as allocating the memory
 *			for the extension record
 *	Parameters:	
 *		Net	net - the net to act on
 *	Return Value:	NONE
 ***********************************************************************/
static void	initNet (net)
  Net	net ;
{
  net->calculateErrorDerivProc = calculateNetErrorDeriv ;
  net->calculateErrorProc      = calculateNetError ;

  net->extension = (NetExtension)calloc(1, sizeof(NetExtensionRec)) ;

  MzeroErrorRadius(net) = 0.2 ;
}
/**********************************************************************/
static void	deinitNet (net)
  Net	net ;
{
  if (net->extension != NULL)
    free(net->extension) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		initGroup 
 *	Description:	sets the update procedures for the units in
 *			a group. (forward, backward)
 *	Parameters:	
 *		Group	group - the group to set the procedures for
 *	Return Value:	NONE
 ***********************************************************************/
static Boolean	multipleBits(mask)
  Mask		mask ;
{
  if (mask == 0)
    return FALSE ;

  while (!(mask & (Mask)(1))) 
    mask >>= 1 ;

  if (mask == (Mask)1)
    return FALSE ;
  else
    return TRUE ;
}
/**********************************************************************/


/**********************************************************************/
static void	confirmTypes(group)
  Group		group ;
{
  /**************************************************
   * gated groups must be output 
   */
  if (group->type & GATED && !(group->type & OUTPUT))
    IErrorAbort("Group \"%s\" can only be %s if it is also %s", 
		group->name, classMaskName(GROUP_CLASS, GATED), 
		classMaskName(GROUP_CLASS, OUTPUT)) ;

  /************************************************
   * gate groups must be softmax
   */
  if (group->type & GATE)
    group->type |= SOFTMAX ;
  
  /************************************************
   * force softmax outputs to use cross entropy error
   */
  if (   (group->type & SOFTMAX)
      && (group->type & OUTPUT)
      && (group->type & (ERROR_MASKS & ~CROSSENTROPY)))
    IErrorAbort("Group \"%s\" is %s, but not %s",
		group->name, classMaskName(GROUP_CLASS, SOFTMAX),
		classMaskName(GROUP_CLASS, CROSSENTROPY)) ;

  /**************************************************
   * RBF units must also be DISTANCE and NEGEXP
   */
  if (group->type & RBF) {
    if (!(group->type & DISTANCE))
      IErrorAbort("Group \"%s\" is %s, but not %s",
		  group->name, classMaskName(GROUP_CLASS, RBF),
		  classMaskName(GROUP_CLASS, DISTANCE)) ;
    group->type |= DISTANCE ;

    if (!(group->type & NEGEXPONENTIAL))
      IErrorAbort("Group \"%s\" is %s, but not %s",
		  group->name, classMaskName(GROUP_CLASS, RBF),
		  classMaskName(GROUP_CLASS, NEGEXPONENTIAL)) ;
    group->type |= NEGEXPONENTIAL ;
  }

  /**************************************************
   * input and bias units don't have combin or transfer
   */
  if (group->type & (INPUT | BIAS)) {
    if (group->type & COMBIN_MASKS)
      IErrorAbort("Group \"%s\" is %s, but has a combin function",
		  group->name, classMaskName(GROUP_CLASS, INPUT)) ;
    if (group->type & TRANSFER_MASKS)
      IErrorAbort("Group \"%s\" is %s, but has an transfer function",
		  group->name, classMaskName(GROUP_CLASS, INPUT)) ;
  } else if (!(group->type & COMBIN_MASKS)) {
    group->type |= DPROD ;
  }

  /**************************************************
   * output units must have an error measure,
   * all others must not
   */
  if (group->type & OUTPUT) {
    if (!(group->type & ERROR_MASKS))
      group->type |= (group->type & SOFTMAX) ? CROSSENTROPY : SUMSQUARE ;
  } else if (group->type & ERROR_MASKS) {
    IErrorAbort("Group \"%s\" is not %s, but has an error function",
		group->name, classMaskName(GROUP_CLASS, OUTPUT)) ;
  }

  /**************************************************
   * everything but input and bias needs an transfer
   */
  if (!(group->type & (INPUT | BIAS)) && !(group->type & TRANSFER_MASKS))
      group->type |= LOGISTIC ;
}
/**********************************************************************/


/**********************************************************************/
static void	setCombIn(group, mask)
  Group		group ;
  Mask		mask ;
{
  if (multipleBits(mask))
    IErrorAbort("Group \"%s\" can only have one combin type", group->name) ;
  
  group->type &= ~COMBIN_MASKS ;
  group->type |= mask ;
  switch(mask) {
  case PI:
    setMultiply(group) ;
    break ;
  case DISTANCE:
    setDistance(group) ;
    break ;
  case DPROD:
    setDotProduct(group) ;
    break ;
  default:
    break ;
  }
}
/**********************************************************************/
static void	setTransfer(group, mask)
  Group		group ;
  Mask		mask ;
{
  if (multipleBits(mask))
    IErrorAbort("Group \"%s\" can only have one transfer type", group->name);

  group->type &= ~TRANSFER_MASKS ;
  group->type |=  mask ;
  switch(mask) {
  case LINEAR:
    setLinear(group) ;
    break ;
  case TANH:
    setTanh(group) ;
    break ;
  case SOFTMAX:
    setSoftMax(group) ;
    break ;
  case EXPONENTIAL:
    setExponential(group) ;
    break ;
  case NEGEXPONENTIAL:
    setNegExponential(group) ;
    break ;
  case LOGISTIC:
    setLogistic(group) ;
    break ;
  default:
    break ;
  }
}
/**********************************************************************/
static void	setError(group, mask)
  Group		group ;
  Mask		mask ;
{
  if (multipleBits(mask))
    IErrorAbort("Group \"%s\" can only have one error type", group->name) ;

  group->type &= ~ERROR_MASKS ;
  group->type |= mask ;
  switch(mask) {
  case CROSSENTROPY:
    setCrossEntropyError(group) ;
    break ;
  case ADAPTIVESUMSQUARE:
    setAdaptiveSumSquareError(group) ;
    break ;
  case SUMSQUARE:
    setSumSquareError(group) ;
    break ;
  default:
    break ;
  }
}
/**********************************************************************/


/**********************************************************************/
static void	initGroup (group)
  Group	group ;
{
  group->extension = (GroupExtension)calloc(1, sizeof(GroupExtensionRec)) ;

  confirmTypes(group) ;

  /* set the combIn procedures based on type */
  setCombIn(group, group->type & COMBIN_MASKS) ;

  /* set the transfer functions based on type */
  setTransfer(group, group->type & TRANSFER_MASKS) ;
  
  /* now set the error functions based on type */
  setError(group, group->type & ERROR_MASKS) ;

  /*
   * Set gated here because it doesn't really fall under any other
   * category, it should over-ride everything else, and it can't be 
   * changed once set.
   */
  if (group->type & GATED)
    setGated(group) ;
}
/**********************************************************************/
static void	deinitGroup (group)
  Group	group ;
{
  if (group->extension)
    free((void *)group->extension) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		initUnit 
 *	Description:	allocates the memory for the extension record
 *	Parameters:	
 *		Unit	unit - the unit to act on
 *	Return Value:	NONE
 ***********************************************************************/
static void	initUnit (unit)
  Unit	unit ;
{
  unit->extension = (UnitExtension)calloc(1, sizeof(UnitExtensionRec)) ;
}
/**********************************************************************/
static void	deinitUnit (unit)
  Unit	unit ;
{
  if (unit->extension)
    free((void *)unit->extension) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		initLink 
 *	Description:	If the link is connected to (not from) a PI
 *			unit, then constrain the weight to be the same
 *			as the first link incoming to the unit (if it
 *			exists).
 *	Parameters:	
 *		Link	link - the link to act on
 *	Return Value:	NONE
 ***********************************************************************/
static void	initLink (link)
  Link	link ;
{
  Unit	unit = link->postUnit ;

  /* If we are incoming to a PI unit, then constrain all links together */
  if (unit && (unit->group->type & PI) && (unit->numIncoming > 1)) {
    Net	net = unit->net ;
    if (net) netConstrainLink(net, link, unit->incomingLink[0], 1.0) ;
  }
}
/**********************************************************************/
static void	deinitLink (link)
  Link	link ;
{
  return ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		calculateNetErrorDeriv
 *	Description:	gradient calculation procedure for backprop net
 *			It processes 'MbatchSize(net)' examples
 *	Parameters:	
 *		Net		net - the net to use
 *		ExampleSet	exampleSet - the examples to use
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	calculateNetErrorDeriv(net, exampleSet)
  Net		net ;
  ExampleSet	exampleSet ;
{
  int		numExamples ;
  int		idx ;

  /* zero the net error and all derivative fields in the links */
  net->error = 0.0 ;
  for (idx = 0 ; idx < net->numLinks ; ++idx)
    net->links[idx]->deriv = 0.0 ;
    
  /* For each example	- zero the derivative fields in the units
   *			- do a forward pass updating the activities
   *			- do a backward pass updating the derivatives
   */
  for (numExamples = 0 ; numExamples < MbatchSize(net) ; ++numExamples) {
    MgetNext(exampleSet) ;

    netForAllUnits(net, ALL, zeroUnitDerivs, NULL) ;
    MupdateNetActivities(net) ;
    MupdateNetGradients(net) ;
  }

  if (numExamples <= 0)
    IErrorAbort("calculateNetErrorDeriv: no examples processed") ;

  /* update the cost after everything else is done */
  MevaluateCostAndDerivs(net) ;
}
/**********************************************************************/


/***********************************************************************
 *	Name:		calculateNetError
 *	Description:	error calculation procedure for backprop net
 *			It processes 'MbatchSize(net)' examples
 *	Parameters:	
 *		Net		net - the net to use
 *		ExampleSet	exampleSet - the examples to use
 *	Return Value:	
 *		NONE
 ***********************************************************************/
static void	calculateNetError(net, exampleSet)
  Net		net ;
  ExampleSet	exampleSet ;
{
  int		numExamples ;

  net->error = 0.0 ;
  for (numExamples = 0 ; numExamples < MbatchSize(net) ; ++numExamples) {
    MgetNext(exampleSet) ;
    MupdateNetActivities(net) ;
  }
  if (numExamples <= 0)
    IErrorAbort("calculateNetError: no examples processed") ;

  /* update the cost after everything else is done */
  MevaluateCost(net) ;
}
/**********************************************************************/


/*********************************************************************
 *	Name:		zeroUnitDerivs
 *	Description:	zeroes the deriv fields in a unit
 *	Parameters:
 *	  Unit		unit - the unit to act on
 *	  void		*data - UNUSED
 *	Return Value:
 *	  static void	zeroUnitDerivs - NONE
 *********************************************************************/
static void	zeroUnitDerivs(unit, data)
  Unit		unit ;
  void		*data ;
{
  unit->inputDeriv  = 0.0 ;
  unit->outputDeriv = 0.0 ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		square
 *	Description:	squares a real valued number
 *	Parameters:
 *	  double	x - the number to square
 *	Return Value:
 *	  static Real	square - x^2
 *********************************************************************/
static Real	square(x)
  double	x ;
{
  return (Real) (x * x) ;
}
/********************************************************************/

typedef struct GroupDataRec {
  GroupDisplay	display ;
  GroupMaskSet	combInSet ;
  GroupMaskSet	transferSet ;
  GroupMaskSet	errorSet ;
  GroupMaskSet	miscSet ;
  GroupMaskSet	fixedSet ;
} GroupDataRec ;

/*********************************************************************
 *	Name:		miscCB
 *	Description:	callback to ensure that all the proper transfer
 *			and error masks are chosen when the user
 *			makes a group softmax
 *	Parameters:
 *	  GroupMaskSet	maskSet	- the maskSet
 *	  Mask		mask	- the mask (un)selecting softmax
 *	  void		*data	- the GroupDataRec
 *	Return Value:
 *	  static void	miscCB - NONE
 *********************************************************************/
static void	miscCB(maskSet, mask, data)
  GroupMaskSet	maskSet ;
  Mask		mask ;
  void		*data ;
{
  GroupDataRec	*groupData = (GroupDataRec *)data ;
  Group		group	   = MGDgroup(MGDparent(maskSet)) ;

  if (MGDhighlighted(maskSet) & RBF) {
    group->type |= RBF ;
    MGDunhighlight(groupData->transferSet, TRANSFER_MASKS & ~NEGEXPONENTIAL) ;
    MGDinsensitize(groupData->transferSet, TRANSFER_MASKS & ~NEGEXPONENTIAL) ;
    MGDhighlight  (groupData->transferSet, NEGEXPONENTIAL) ;
    setTransfer(group, NEGEXPONENTIAL) ;

    MGDunhighlight(groupData->combInSet, COMBIN_MASKS & ~DISTANCE) ;
    MGDinsensitize(groupData->combInSet, COMBIN_MASKS & ~DISTANCE) ;
    MGDhighlight  (groupData->combInSet, DISTANCE) ;
    setCombIn(group, DISTANCE) ;
  } else {
    group->type &= ~RBF ;
    MGDsensitize(groupData->combInSet, COMBIN_MASKS) ;
  }

  if (!((MGDhighlighted(maskSet) & RBF)))
    MGDsensitize(groupData->transferSet, TRANSFER_MASKS) ;

  if (MGDhighlighted(maskSet) & IPI)
    group->type |=  IPI ;
  else
    group->type &= ~IPI ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		combInCB/transferCB/errorCB
 *	Description:	callbacks for setting unit methods
 *	Parameters:
 *	  GroupMaskSet	maskSet - the mask set
 *	  Mask		mask	- mask describing how to set method
 *	  void		*data	- unused ;
 *	Return Value:
 *	  static void	combInCB - NONE
 *********************************************************************/
static void	combInCB(maskSet, mask, data)
  GroupMaskSet	maskSet ;
  Mask		mask ;
  void		*data ;
{
  setCombIn(MGDgroup(MGDparent(maskSet)), mask) ;
}
/********************************************************************/
static void	transferCB(maskSet, mask, data)
  GroupMaskSet	maskSet ;
  Mask		mask ;
  void		*data ;
{
  GroupDataRec	*groupData = (GroupDataRec *)data ;
  Group		group = MGDgroup(MGDparent(maskSet)) ;

  setTransfer(group, mask) ;

  /* force SOFTMAX outputs to use cross entropy error */
  if (group->type & OUTPUT) {
    if (group->type & SOFTMAX) {
      MGDinsensitize(groupData->errorSet, ERROR_MASKS & ~CROSSENTROPY) ;
      MGDunhighlight(groupData->errorSet, ERROR_MASKS & ~CROSSENTROPY) ;
      MGDhighlight  (groupData->errorSet, CROSSENTROPY) ;
      setError(group, CROSSENTROPY) ;
    } else {
      MGDsensitize  (groupData->errorSet,  ERROR_MASKS) ;
    }
  }
}
/********************************************************************/
static void	errorCB(maskSet, mask, data)
  GroupMaskSet	maskSet ;
  Mask		mask ;
  void		*data ;
{
  setError(MGDgroup(MGDparent(maskSet)), mask) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		setGroupCB
 *	Description:	callback to make sure the display reflects
 *			the state of the selected group
 *	Parameters:
 *	  GroupDisplay	this	- the group display
 *	  Group		group	- the group
 *	  void		*data	- the groupData record
 *	Return Value:
 *	  static void	setGroupCB - NONE
 *********************************************************************/
static void	setGroupCB(this, group, data)
  GroupDisplay	this ;
  Group		group ;
  void		*data ;
{
  GroupDataRec	*groupData = (GroupDataRec *)data ;
  Mask		mask ;

  if (group == NULL) {
    MGDinsensitize(groupData->fixedSet, 	FIXED_MASKS) ;
    MGDinsensitize(groupData->miscSet,	 	MISC_MASKS) ;
    MGDinsensitize(groupData->combInSet, 	COMBIN_MASKS) ;
    MGDinsensitize(groupData->transferSet, 	TRANSFER_MASKS) ;
    MGDinsensitize(groupData->errorSet, 	ERROR_MASKS) ;
    return ;
  }
  
  mask = group->type ;

  MGDsensitize(groupData->fixedSet, 		FIXED_MASKS) ;
    
  if (mask & OUTPUT)
    MGDsensitize  (groupData->errorSet, 	ERROR_MASKS) ;
  else
    MGDinsensitize(groupData->errorSet, 	ERROR_MASKS) ;

  if (mask & (BIAS | INPUT)) {
    MGDinsensitize(groupData->transferSet,	TRANSFER_MASKS) ;
    MGDinsensitize(groupData->combInSet,	COMBIN_MASKS) ;
    MGDinsensitize(groupData->miscSet,	 	MISC_MASKS) ;
  } else {
    MGDsensitize  (groupData->transferSet,	TRANSFER_MASKS) ;
    MGDsensitize  (groupData->combInSet,	COMBIN_MASKS) ;
    MGDsensitize  (groupData->miscSet,	 	MISC_MASKS) ;
  }

  if (mask & PI)
    MGDinsensitize(groupData->combInSet,	COMBIN_MASKS) ;
  else
    MGDinsensitize(groupData->combInSet,	PI) ;

  if (mask & SOFTMAX)
    MGDinsensitize(groupData->errorSet, 	ERROR_MASKS & ~CROSSENTROPY) ;

  if (mask & RBF) {
    MGDinsensitize(groupData->transferSet, 	
		   TRANSFER_MASKS & ~NEGEXPONENTIAL);
    MGDinsensitize(groupData->combInSet, 	COMBIN_MASKS & ~DISTANCE) ;
  }

  if (mask & GATE) {
    MGDinsensitize(groupData->transferSet,	TRANSFER_MASKS & ~SOFTMAX) ;
    MGDinsensitize(groupData->miscSet,	 	RBF) ;
  }

  if (mask & GATED) {
    MGDinsensitize(groupData->transferSet, 	TRANSFER_MASKS) ;
    MGDunhighlight(groupData->transferSet, 	TRANSFER_MASKS) ;
    MGDinsensitize(groupData->errorSet,		ERROR_MASKS) ;
    MGDunhighlight(groupData->errorSet, 	ERROR_MASKS) ;
    MGDinsensitize(groupData->combInSet, 	COMBIN_MASKS) ;
    MGDunhighlight(groupData->combInSet, 	COMBIN_MASKS) ;
    MGDinsensitize(groupData->miscSet,	 	MISC_MASKS) ;
  }

  MGDinsensitize(groupData->miscSet,	 	GAIN) ;
}
/********************************************************************/


/*********************************************************************
 *	Name:		makeGroupDisplay
 *	Description:	builds the groupDisplay
 *	Parameters:
 *	Return Value:
 *	  static void	makeGroupDisplay - NONE
 *********************************************************************/
static void	makeGroupDisplay()
{
  static GroupDataRec	groupData ;
  GroupDisplay		display ;
  GroupMaskSet		maskSet ;
  
  display = createGroupDisplay("Group Types") ;
  MGDsetCallback(display, GDSelect, setGroupCB, (void *)&groupData) ;
  groupData.display = display ;

  maskSet = createGroupMaskSet("Fixed Types", display, 
			       FIXED_MASKS, GMSReadOnly) ;
  groupData.fixedSet = maskSet ;

  maskSet = createGroupMaskSet("Input Combining Function", display, 
			       COMBIN_MASKS, GMSOneOfMany) ;
  MGDsetCallback(maskSet, GMSSelect,   combInCB, 	NULL) ;
  groupData.combInSet = maskSet ;

  maskSet = createGroupMaskSet("Transfer Function", display,
			       TRANSFER_MASKS, GMSOneOfMany) ;
  MGDsetCallback(maskSet, GMSSelect,   transferCB,	(void *)&groupData) ;
  groupData.transferSet = maskSet ;

  maskSet = createGroupMaskSet("Error Measure", display,
			       ERROR_MASKS, GMSOneOfMany) ;
  MGDsetCallback(maskSet, GMSSelect,   errorCB, 	NULL) ;
  groupData.errorSet = maskSet ;


  maskSet = createGroupMaskSet("Miscellaneous Types", display,
			       MISC_MASKS, GMSNOfMany) ;
  MGDsetCallback(maskSet, GMSSelect,   miscCB, (void *)&groupData) ;
  MGDsetCallback(maskSet, GMSUnselect, miscCB, (void *)&groupData) ;
  groupData.miscSet = maskSet ;
}
/********************************************************************/
