/**************************************************************************/
/* allocation.c - dynamic allocation and deallocation routines  /\/\      */
/* Version 1.1.0 -- June 1991                                   \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/
/**************************************************************************/
/* Note: this software has originally been written and conceived by       */
/* neighorn (revisions 1.1--1.5).                                         */
/**************************************************************************/

#include <stdio.h>
#include <malloc.h>

#include "allocation.h"

/*
 * Matrix allocation routines.  To allocate a 10 by 10 matrix of floats use:
 * 
 * char **alloc2d(); float **x;
 * 
 * x = (float **) alloc2d(10, 10, sizeof(float));
 * 
 * To free this matrix use:
 * 
 * free2d(x);
 */



/*--Allocate & free 2 arrays--*/

char  **alloc2d(dim1, dim2, size)
int  dim1, dim2, size;
{
  int         i;
  unsigned    nelem;
  char        *p, **pp;

  nelem = (unsigned) dim1 *dim2;

  p = calloc(nelem, (unsigned) size);
  if (p == NULL)
	return NULL;

  pp = (char **) calloc((unsigned) dim1, (unsigned) sizeof (char *));
  if (pp == NULL)
  {
	cfree(p);
	return NULL;
  }
  for (i = 0; i < dim1; i++)
	pp[i] = p + i * dim2 * size;

  return pp;
}



void free2d(mat)
char  **mat;
{
  if (mat != NULL && *mat != NULL)
	cfree((char *) *mat);
  if (mat != NULL)
	cfree((char *) mat);
}




/*--Allocate 3 dimensional array----------*/

char  ***alloc3d(dim1, dim2, dim3, size)
int dim1, dim2, dim3, size;
{
  int  i;
  char **pp, ***ppp;

  pp = (char **) alloc2d(dim1 * dim2, dim3, size);
  if (pp == NULL)
	return NULL;

  ppp = (char ***) calloc((unsigned) dim1, (unsigned) sizeof (char **));

  if (ppp == NULL)
  {
	free2d(pp);
	return NULL;
  }

  for (i = 0; i < dim1; i++)
	ppp[i] = pp + i * dim2;
  return ppp;
}



void free3d(mat)
char ***mat;
{
  free2d(*mat);
  if (mat != NULL)
	cfree((char *) mat);
}




/*---Allocate 4 dimensional array--------*/

char ****alloc4d(dim1, dim2, dim3, dim4, size)
int dim1, dim2, dim3, dim4, size;
{
  int   i;
  char  ***ppp, ****pppp;

  ppp = (char ***) alloc3d(dim1 * dim2, dim3, dim4, size);
  if (ppp == NULL)
	return NULL;

  pppp = (char ****) calloc(dim1, sizeof (char ****));
  if (pppp == NULL)
  {
	free3d(ppp);
	return NULL;
  }

  for (i = 0; i < dim1; i++)
	pppp[i] = ppp + i * dim2;
  return pppp;
}



void free4d(mat)
char ****mat;
{
  free3d((char *) *mat);
  if (mat != NULL)
	cfree((char *) mat);
}




/*---Allocate 5 dimensional array--------*/

char *****alloc5d(dim1, dim2, dim3, dim4, dim5, size)
int dim1, dim2, dim3, dim4, dim5, size;
{
  int   i;
  char ****pppp, *****ppppp;

  pppp = (char ****) alloc4d(dim1 * dim2, dim3, dim4, dim5, size);
  if (pppp == NULL)
	return NULL;

  ppppp = (char *****) calloc(dim1, sizeof (char *****));
  if (ppppp == NULL)
  {
	free4d(pppp);
	return NULL;
  }

  for (i = 0; i < dim1; i++)
	ppppp[i] = pppp + i * dim2;
  return ppppp;
}



void free5d(mat)
char *****mat;
{
  free4d((char *) *mat);
  if (mat != NULL)
	cfree((char *) mat);
}




/*--allocate lower triangule of square matrix------------*/

char **alloclt(dim, size)
int dim, size;
{
  int             i;
  char           *p, **pp;
  unsigned        nelem;

  nelem = (dim * (dim + 1)) / 2;
  p = calloc(nelem, (unsigned) size);

  if (p == NULL)
	return NULL;

  pp = (char **) calloc((unsigned) dim, (unsigned) sizeof (char *));
  if (pp == NULL)
  {
	cfree(p);
	return NULL;
  }

  pp[0] = p;
  for (i = 1; i < dim; i++)
	pp[i] = pp[i - 1] + i * size;
  return pp;
}




/*--allocate upper triangule of square matrix------------*/

char **allocut(dim, size)
int dim, size;
{
  int             i;
  char           *p, **pp;
  unsigned        nelem;

  nelem = (dim * (dim + 1)) / 2;
  p = calloc(nelem, (unsigned) size);
  if (p == NULL)
	return NULL;
  pp = (char **) calloc((unsigned) dim, (unsigned) sizeof (char *));
  if (pp == NULL)
  {
	cfree(p);
	return NULL;
  }
  pp[0] = p;
  for (i = 1; i < dim; i++)
	pp[i] = pp[i - 1] + (dim - i) * size;
  return pp;
}




/*---------------------------------------------------------------
 *  ALLOC2D2 :
 *	Allocates a vector of one dimensional arrays
 *	like a two dimensional array, however the sizes
 *	of each vector can be different.
 *
 *	To free the space use :
 *
 *		free2d(x) ;
 *-------------------------------------------------------------*/

char **alloc2d2(numclass, nvals, size)
int numclass, *nvals, size;
{
  register int    i;
  int             sum = 0;
  unsigned        nelem = 0;
  char           *p, **pp;

  for (i = 0; i < numclass; i++)
	nelem += (unsigned) nvals[i];

  p = calloc(nelem, (unsigned) size);

  if (p == NULL)
	return NULL;

  pp = (char **) calloc((unsigned) numclass, (unsigned) sizeof (char *));

  if (pp == NULL)
  {
	cfree(p);
	return NULL;
  }
  for (i = 0; i < numclass; i++)
  {
	pp[i] = p + sum * size;
	sum += nvals[i];
  }

  return pp;

}


char ***alloc3d2(dim1, dim2, nvals, numnvals, size)
int dim1, dim2, size, *nvals, numnvals;
{
  register int    i, j, numarr2;

  register int    sum = 0;
  unsigned        nelem = 0;
  char           *p, **pp, ***ppp;

  if (numnvals != dim2)
  {
	fprintf(stderr, 
		"error on alloc3d2 dim2 %d != numnvals %d returning NULL!\n",
		dim2, numnvals);
	return NULL;
  }

  for (i = 0; i < numnvals; i++)
	nelem += (unsigned) nvals[i];

  nelem = nelem * dim1;

  p = calloc(nelem, (unsigned) size);

  if (p == NULL)
	return NULL;

  pp = (char **) calloc(dim1 * dim2, sizeof (char *));
  if (pp == NULL)
  {
	cfree(p);
	return NULL;
  }

  numarr2 = dim1 * dim2;
  sum = 0;
  for (i = 0, j = 0; i < numarr2; i++, j++)
  {
	if (j == numnvals)
		j = 0;
	pp[i] = p + (sum * size);
	sum += nvals[j];
  }

  ppp = (char ***) calloc(dim1, sizeof (char **));

  /* now set the pointers */

  sum = 0;
  for (i = 0; i < dim1; i++)
	ppp[i] = pp + (i * dim2);

  return ppp;
}


/*
 * alloc3 array where second dimension is variable i.e.
 * labinclass[class][numlabinclass][numchar]
 */
char ***alloc3d2ndvar(dim1, nvals, dim2, size)
int dim1, dim2, size, *nvals;
{
  register int    i, numarr2;
  register int    sum = 0;
  unsigned        nelem = 0;
  char           *p, **pp, ***ppp;

  for (i = 0; i < dim1; i++)
	nelem += (unsigned) nvals[i];

  numarr2 = nelem;
  nelem = nelem * dim2;

  p = calloc(nelem, (unsigned) size);

  if (p == NULL)
	return NULL;

  pp = (char **) calloc(numarr2, sizeof (char *));
  if (pp == NULL)
  {
	cfree(p);
	return NULL;
  }
  sum = 0;
  for (i = 0; i < numarr2; i++)
	pp[i] = p + (i * dim2 * size);

  ppp = (char ***) calloc(dim1, sizeof (char **));

  /* now set the pointers */

  sum = 0;
  for (i = 0; i < dim1; i++)
  {
	ppp[i] = pp + sum;
	sum += nvals[i];
  }
  return ppp;
}
