/* Simiulation of ART-1 Neural network (Ref: Lippman 87)
   Written by Jagath Samarabandu <bandu@cs.buffalo.edu> 
   To compile, type cc art1.c -o art1  */

#include <stdio.h>

#define N      25   /* No. of nodes in F1 */
#define M      20   /* No. of nodes in F2 */

initialize(top_down,bot_up,n,m) /* initialize the LTM traces */
     double top_down[],bot_up[]; /* top down and bot. up LTM traces */
     int n,m;                   /* No. of nodes in F1 and F2 */
{
  int i;
  double t;

  t = 1.0/(1.0+n);
  for (i=0;i<n*m;i++) {
    top_down[i] = 1.0; bot_up[i] = t;
  }
}

compute_matching_scores(mu,bot_up,input,n,m) /* calculate matching scores */
     double mu[],bot_up[],input[];            /* mu - matching score */
     int n,m;                                /* No. of F1 and F2 nodes */
{
  int i,j;

  for (j=0; j<m; j++) 
    for (i=0, mu[j]=0.0; i<n; i++)
      mu[j] += bot_up[i*m+j]*input[i];
}

double vigilance(top_down,input,jmax,n,m) /* returns |T.X|/|X| */
     double top_down[],input[];
     int n,m,jmax;
{
  int i;
  double x,t;

  for (i=0,x=0.0; i<n; i++)
    x += input[i];
  for (i=0,t=0.0; i<n; i++)
    t += top_down[i*m+jmax]*input[i];
  return(t/x);
}

int find_max(array,len) /* find the max of array and return the index */
     double array[];
     int len;
{
  int j,jmax;

  for (j=0,jmax=0; j<len; j++)
    if (array[j]>array[jmax])
      jmax = j;
  return (jmax);
}

adapt_LTM(top_down,bot_up, input,jmax,n,m) /* change top down and bot.up LTM */
     double top_down[],bot_up[],input[];
     int n,m,jmax;
{
  int i,ij;
  double sum,t;
  
  for (i=0,sum=0.5; i<n; i++)
    sum += top_down[i*m+jmax]*input[i];

  for (i=0,ij=jmax; i<n; i++,ij += m) {
    t = top_down[ij]*input[i];
    bot_up[ij] = t/sum;
    top_down[ij] = t;
  }
}

load_data(d,max,n,fname) /* load data from file */
     int d[],max,n;
     char *fname[];
{
  FILE *fp;
  int n_pat,n_var,i,j;
  
  if (!(fp=fopen(fname,"r"))) exit(perror(fname));
  fscanf(fp,"%d %d\n",&n_pat,&n_var);
  if (n_pat>max) {
    printf("Warning! only %2d patterns out of %d are read\n",max,n_pat);
    n_pat = max;
  }
  if (n_var!=n)
    exit(printf("wrong pattern size: should be %2d. was %2d\n",n,n_var)); 
    
  for (i=0;j<n_pat;i++)
    for (j=0;j<n_var;j++)
      fscanf(fp,"%d",&d[i*n+j]);
  fclose(fp);
}

display(in,top_down,x,y,m) /* display input and top down weights */
     double in[],top_down[];
     int x,y,m;
{
  int i,ix,iy,j;

  for (iy=0,i=0; iy<y; iy++){
    for (ix=0,i=iy*y; ix<x; ix++,i++)
      printf("%c",(in[i]>0.5)?'#':' ');
    printf(" | ");
    for (j=0; j<m; j++) {
      for (ix=0,i=iy*y; ix<x; ix++,i++)
	printf("%c",(top_down[i*m+j]>0.5)?'#':' ');
      printf(" ");
    }
    printf("\n");
  }
  printf("\n");
}
/*****************  main routine starts here  *******************************/

int data[20][N]={
#include "art1.data"
};

main(argc,argv)
     int argc;
     char *argv[];
{
char ch;
  double t[N][M],b[N][M],x[N],mu[M],rho;
  int max_j,i,j,n_pat,ok,seq[M];

  if (argc>1)
    n_pat = load_data(data,20,N,argv[1]);
  else n_pat=20;
  initialize(t,b,N,M);
  printf("Vigilance threshold: "); scanf("%lf",&rho);
  printf("\nSimulation of ART1 network with vigilance Threshold = %3.3lf\n\n",rho);

  do {
    for (i=0; i<n_pat; i++) {
      for (j=0; j<N; j++)
	x[j] = (double)data[i][j];
      compute_matching_scores(mu,b,x,N,M);
      bzero((char *)seq,M*sizeof(int)); j=1;
      do {
	max_j = find_max(mu,M); seq[max_j] = j++;
	if (vigilance(t,x,max_j,N,M)>rho) {
	  adapt_LTM(t,b,x,max_j,N,M);
	  seq[max_j] = -1;
	  break;
	}
	else 
	  mu[max_j] = 0.0;
      } while (1);
      printf("IN:%2d    ",i);
      for (j=0;j<M; j++) {
	if (seq[j]>0)
	  printf("%1d     ",seq[j]);
	else if (seq[j]==0)
	  printf("      ");
	else {
	  printf("R\n"); break;
	}
      }
      display(x,t,5,5,M);
    }
    printf("Another round? (1-yes): "); scanf("%d",&ok);
  } while (ok);
  ok =1;

do {
   
   printf("\n[A-T] or ESC to quit :"); scanf("%s",&ch);
   printf("\n");
   if ((ch == 27) || ((ch < 'A') || (ch > 'T'))) ok =0;
   
   else
      {
      for (j=0; j<N; j++)
	x[j] = (double)data[ch -'A'][j];
      compute_matching_scores(mu,b,x,N,M);
      bzero((char *)seq,M*sizeof(int)); j=1;
      do {
	max_j = find_max(mu,M); seq[max_j] = j++;
	if (vigilance(t,x,max_j,N,M)>rho) {
	  adapt_LTM(t,b,x,max_j,N,M);
	  seq[max_j] = -1;
	  break;
	}
	else 
	  mu[max_j] = 0.0;
      } while (1);
      printf("IN:%2d    ",i++);
      for (j=0;j<M; j++) {
	if (seq[j]>0)
	  printf("%1d     ",seq[j]);
	else if (seq[j]==0)
	  printf("      ");
	else {
	  printf("R\n"); break;
	}
      }
      display(x,t,5,5,M);

 }
 } while (ok);

  
}
