/*
         Ŀ
                     Copyright (c) 1991 by Bruce W. Colletti            
                                                                        
          You may copy and distribute intact the contents of this disk. 
          Monetary contributions for PDNNET are not solicited.          
          Although PDNNET may not be sold for profit, fees may be       
          charged to cover minimal distribution costs such as disks,    
          postage, etc.                                                 
                                                                        
          The author does not guarantee the accuracy of the code and    
          documentation, but does believe that both are correct.        
          PDNNET is used at one's own risk.  Users may modify the       
          source code to suit a nonprofit need, and may not "alter and  
          distribute" this document file (MANUAL.DOC) without the       
          author's permission.                                          
         
         
                          Public Domain Neural Network 1.0

 This code implements a backpropagation neural network (NNet) whose
 architecture is determined by the user.  Training doesn't consider a 
 momentum factor.  Following the definitions below, the matrix algebraic
 specification of backpropagation (and thus of this program) is given.
 
 This program was created during JUL-AUG 90, written in TURBO C 2.0 (small
 model) on a 286 PC (640K, MSDOS 3.3, with math-chip), and entered into 
 the public domain.  

 Reader familiarity with the rudiments of backpropagation and of neural 
 network terminology is assumed.  Introductions to the backpropagation 
 NNet (as well as to many other NNets) can be found in these texts:
 
      - "Artificial Neural Systems" by Patrick K. Simpson (Pergamon Press, 
         NY, NY; 800-257-5755; ISBN 0-08-037894-3; $19.95) 

      - "Adaptive Pattern Recognition and Neural Networks" by Yoh-Han Pao
        (Addison Wesley, Reading MA; ISBN 0-201-12584-6)

 Also, the monthly magazine AI-Expert (415-397-1881) has published 
 articles about neural networks, notably the "Neural Network Primer" 
 series by Maureen Caudill.  This program's creation was entirely the 
 author's undertaking.

         AUTHOR:  Bruce W. Colletti
                  10 FEB 91
                  Richmond VA

                  GEnie     :  b.colletti
                  CompuServe:  71121,1452

         OTHER BBS':

         ShadeTree                (Pittsburgh PA):  412-244-9416 2400N81
         Instant Recall           (Bethesda MD)  :  301-530-2890 2400N81
         Blue Ridge Express       (Richmond VA)  :  804-790-1675 2400N81
                 
            ********** SPECIFICATION OF BACKPROPAGATION **********
 
 DEFINITIONS:

 N        the number of layers in the NNet.  Layer 1 is the input layer
          and Layer N is the output layer. 

         the learning rate

         the error tolerance between computed and desired outputs

 nodes_in_layer(i) is the number of neurons in layer i

 W(i)     is the matrix of synaptic weights between layers i and i+1.
          W(i) has dimensions "nodes_in_layer(i+1) X nodes_in_layer(i)".
          The j'th row of W(i) are the synaptic weights extending from the
          j'th neuron of layer i+1.  Thus, W(i,j,k) is the synaptic weight
          from neuron j (in layer i+1) to neuron k (in layer i).

 (i,j)   the transfer function of node j in layer i.
          
 '(i,j)  the derivative of (i,j).  
 
 NET(i)   is the columnar matrix (dimensions nodes_in_layer(i) X 1) of net
          inputs in the neurons in layer i.  Thus, NET(i,j) is the net 
          input into neuron j of layer i.

 X(i)     is the columnar matrix (dimensions nodes_in_layer(i) X 1)
          obtained by passing NET(i) through the transfer functions of
          layer i.  Thus, X(i,j) is the output of node j in layer i, i.e.,
          X(i,j) = (i,j)(NET(i,j)).

 DELTA(i) is the columnar matrix (dimensions nodes_in_layer(i) X 1) 
          containing the "delta" values of the neurons in layer i.  All
          delta vectors are computed prior to backpropagation. 
 
 A @ B    represents the matrix obtained by elementwise multiplication of
          the entries in matrices A and B.

 
 MATRIX ALGEBRAIC FORMULATION OF BACKPROPAGATION:

 - Define  (i)(NET(i)) = [ (i,j)(NET(i,j)) ],  a columnar matrix

 - Define '(i)(NET(i)) = [ '(i,j)(NET(i,j)) ], a columnar matrix
 
 - For each input/output pair (X(1),) in the training set:

                        NET(i+1) = W(i) * X(i), i < N            (1)

              X(i) = (i)(NET(i)) = [ (i,j)(NET(i,j)) ], i  N  (2)

                    DELTA(N) = '(N)(NET(N)) @ ( - X(N))        (3)

 - For 2  i < N (provided that the absolute value of each entry in
    - X(N) is  ):                                             (3.5)

               DELTA(i) = [W'(i) * DELTA(i+1)] @ '(i)(NET(i))   (4)

   where W'(i) is the transpose of W(i).

 - Weights are adjusted according to the formula

           W(i) = W(i) +  DELTA(i+1) * 1R * diag(X(i)), i < N   (5)

   where 1R is a row matrix of ones (of appropriate size) and diag(X(i))
   is the square matrix whose diagonal is X(i) and whose other entries are
   zero.

 - If necessary, recycle the training set until every computed output within
   a cycle is within the established error tolerance  of the desired
   output.                                                       (6)

 **************************************************************************
 **************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <conio.h>
#include <alloc.h>
#include <signal.h>
#include <string.h>

#define xtype float
#define xtypef "%f "

typedef struct {xtype **Matrix; int Rows,Cols;} matrix;
typedef xtype (*sigmoid)(xtype);

/***************************************************************/
/**************** Functions used in this program ***************/
/***************************************************************/

void      backpropagate(void);
short int calculate_error(void);
void      compute_deltas(void);
void      create_architecture(void);
matrix   *create_matrix(int Rows,int Cols);
void      create_transfers_derivatives(void);
void      forward_propagate(void);
void      FPEhandler(void);
void      Ftransfer(sigmoid *function,matrix *INPUT, matrix *OUTPUT);
void      load_trainingSET(void);
void      manually_create_architecture(void);
void      matrix_multiply(matrix *Left,matrix *Right,matrix *Product);
void      multiply(matrix *Left, matrix *Right, matrix *Result);
void      print_matrix(matrix *Matrix);
void      propagate(void);
void      readint(int *x);
void      readreal(xtype *x);
void      read_architecture(void);
void      read_beta_epsilon(void);
void      read_IOpair(void);
void      remove_network(void);
void      save_weights(void);
void      train_network(void);

/***************************************************************/
/***********       Error Capturing Routines          ***********/
/***************************************************************/

void FPEhandler()
  {printf("\nHave encountered a floating point error.  Will cease operation.");
   exit(-1);
  }

int matherr(struct exception *a)
  {if ((a->type == 3) && (strcmp(a->name,"exp")==0))
    {printf("\n\nThe argument being passed to the exponential function e(x) is too large.");
     printf("\nThe argument passed is %lf",a->arg1);
     printf("\n\nInspect the training set for abnormally large values.");
     printf("\nThese may cause the net input into the neuron to become too large.");
     printf("\nIf this is the case, rescale the training set to smaller values");
     printf("\nand start over.");
     exit(-1);
    }
   else
    if (a->type != 4)
    {printf("\nHave encountered math error type %d %s", a->type, a->name);
     printf("\nPassed argument is %lf",a->arg1);
     printf("\nWill cease operation.");
     exit(-1);
    }
  }


/***************************************************************/
/*********** Neural Network Architectural Parameters ***********/
/***************************************************************/

int N=0,             /* number of layers */
    *nodes_in_layer, /* nodes_in_layer[i] # neurons in layer i */
    obs;             /* number of I/O pairs in training set */

matrix **W,          /* W[i] synaptic weights matrix btw layers i and i+1 */
       **NET,        /*   NET[i] column vector of net inputs into layer i */
       **X,          /*     X[i] column vector of outputs from layer i */
       **DELTA,      /* DELTA[i] column vector of delta values for layer i */
       *ERROR;       /* ERROR is columnar error vector */

sigmoid **transfer;   /* transfer[i][j] transfer function for node j, layer i */
sigmoid **derivative; /* derivative[i][j] derivative of transfer[i][j] */

FILE *TrainingSet, *outfile;

xtype huge *tset1,   /* start of training set in memory */
      huge *tset2;   /* a utility variable */

xtype epsilon=0,     /* error tolerance */
      beta=0,        /* learning rate */
      MAXERR,        /* Maximum absolute error within a cycle */
    LastCycleMAXERR; /* MAXERR of last cycle */

int long cycle;      /* the number of passes through training set */

/***************************************************************/
/* FIRST  USER INPUT:                                          */
/*                                                             */
/*             Define Transfer Functions Here                  */
/***************************************************************/

xtype sigmoid00(xtype x) {return(1.0);}
xtype sigmoid01(xtype x) {return(1.0 / (1.0 + exp(-x)));}

/***************************************************************/
/* SECOND USER INPUT                                           */
/*                                                             */
/*      Define Transfer Function Derivatives Here              */
/***************************************************************/

xtype sigmoid00D(xtype x) {return(0.0);}
xtype sigmoid01D(xtype x) {xtype y; y = sigmoid01(x); return(y*(1-y));}

/***************************************************************/
/* LAST USER INPUT                                             */
/*                                                             */
/*    establish neuronal transfer functions and derivatives:

transfer[i][j]   is the transfer function for neuron j in layer i
derivative[i][j] is the derivative of transfer[i][j]

NOTE:  this program originally assumed a bias node in each layer (except 
at the output) and that this bias node is the first one in each layer.  
That is, the first neuron in each layer (except the output) ALWAYS outputs 
the value 1 by assigning the transfer function sigmoid00 to the first 
neuron in each layer.  To assign different transfer functions to neurons, 
establish the appropriate transfers and derivatives in the FIRST and 
SECOND USER INPUT areas above, modify the following function, and 
recompile the code.  
****************************************************************/ 

void create_transfers_derivatives(void)
{int i,j,k;

 for(i=1; i<N; i++)
   {  transfer[i][0] = sigmoid00;
    derivative[i][0] = sigmoid00D;

    k = (i<N-1) ? 1 : 0;
    for(j=k; j<nodes_in_layer[i]; j++)
      {  transfer[i][j] = sigmoid01;
       derivative[i][j] = sigmoid01D;
      }}

 return;
}

/***************************************************************/
/***************** Effect backpropagation (5) ******************/
/***************************************************************/

void backpropagate(void)
{int i,j,k;

 for(i=0; i<N-1; i++)
   {for(j=0; j<W[i]->Rows; j++)
      {for(k=0; k<W[i]->Cols; k++)
         W[i]->Matrix[j][k] += beta *
                               DELTA[i+1]->Matrix[j][0] * X[i]->Matrix[k][0];
    }
  }

 return;
}

/***************************************************************/
/******** Calculate the error vector at the output layer *******/
/******* 1 means that error vector is within tolerance (3.5) ***/
/***************************************************************/

short int calculate_error(void)
{int i;
 short int j=1;
 xtype Z;

/* 
   NOTE:  ERROR was established in the read_IOpair subroutine as the
   output member of the training I/O pair 
*/

 for(i=0; i<nodes_in_layer[N-1]; i++)
   {Z = fabs(ERROR->Matrix[i][0] -= X[N-1]->Matrix[i][0]);

    if (Z > epsilon) j=0;
    if (Z > MAXERR) MAXERR = Z;
   }

 return(j);
}

/***************************************************************/
/************* Compute the Delta Vectors (3) *******************/
/***************************************************************/

void compute_deltas(void)
{int i,j,k;
 xtype sum;

 /* calculate DELTA vector at output layer */

 Ftransfer(derivative[N-1],NET[N-1],DELTA[N-1]);
 multiply(DELTA[N-1],ERROR,DELTA[N-1]);

 /* calculate DELTA vectors for hidden layers */

 for(i=N-2; i>0; i--)
   {for(j=0; j<W[i]->Cols; j++)
      {sum = 0.0;

       for(k=0; k<W[i]->Rows; k++)
         sum += W[i]->Matrix[k][j] * DELTA[i+1]->Matrix[k][0];

       DELTA[i]->Matrix[j][0] = sum;
      }

    Ftransfer(derivative[i],NET[i],NET[i]);
    multiply(DELTA[i],NET[i],DELTA[i]);
   }

 return;
}

/***************************************************************/
/********************* Establish architecture ******************/
/***************************************************************/

void create_architecture(void)
{int i,j;

    derivative = farmalloc(N*sizeof(sigmoid *));
      transfer = farmalloc(N*sizeof(sigmoid *));
         DELTA = farmalloc(N*sizeof(matrix *));
         ERROR = create_matrix(nodes_in_layer[N-1],1);
           NET = farmalloc(N*sizeof(matrix *));
             X = farmalloc(N*sizeof(matrix *));
             W = farmalloc((N-1)*sizeof(matrix *));

 for(i=0; i<N; i++) /* establish layer specific parameters */
   {j = nodes_in_layer[i];

    derivative[i] = farmalloc(j * sizeof(sigmoid));
      transfer[i] = farmalloc(j * sizeof(sigmoid));
         DELTA[i] = create_matrix(j,1);
           NET[i] = create_matrix(j,1);
             X[i] = create_matrix(j,1);

    if (i < N-1) /* create weight matrices; N layers -> N-1 weight matrices */
       W[i] = create_matrix(nodes_in_layer[i+1],nodes_in_layer[i]);
   }

  return;
}

/***************************************************************/
/***************** create a Rows X Cols matrix *****************/
/***************************************************************/

matrix *create_matrix(int Rows,int Cols)
  {xtype *w;
   matrix *y;
   int i;

   y = farmalloc(sizeof(matrix));
   y->Matrix = farmalloc(Rows * sizeof(xtype *));
   y->Rows   = Rows;
   y->Cols   = Cols;

   if ((w = farmalloc((unsigned long) Rows * Cols * sizeof(xtype))) == NULL)
      {printf("\n\nNot enought memory to hold this %d X %d matrix",
       Rows,Cols);
       exit(-1);
      }

   for(i=0; i<Rows; i++)  y->Matrix[i] = &w[i*Cols];

   return(y);
  }

/***************************************************************/
/**** forward propagate from the input to the output layers ****/
/*                       (1) and (2)                           */
/***************************************************************/

void forward_propagate(void)
{int i;

 for(i=0; i<N-1; i++)
   {matrix_multiply(W[i],X[i],NET[i+1]);       /*     WX = NET */
    Ftransfer(transfer[i+1],NET[i+1],X[i+1]);  /* (NET) = X   */
   }

 return;
}

/***************************************************************/
/*** Send Net input through neuronal transfer functions (2) ****/
/***************************************************************/

void Ftransfer(sigmoid *function,matrix *INPUT, matrix *OUTPUT)
{int i;

 for(i=0; i<INPUT->Rows; i++)
    OUTPUT->Matrix[i][0] = (function[i])(INPUT->Matrix[i][0]);
 return;
}

/***************************************************************/
/********************* Load Training Set ***********************/
/***************************************************************/

void load_trainingSET(void)
{char buffer[50];
 long int i;

 printf("\nTraining Set");
 while (gets(buffer) && ((TrainingSet=fopen(buffer,"r"))==NULL))
   printf("\n%s not found.  Please reenter.",buffer);

 printf("Number of observations in training set"); readint(&obs);
 i = (long) obs * (nodes_in_layer[0]-1+nodes_in_layer[N-1]) * sizeof(xtype);

 if ((tset2 = tset1 = farmalloc((long) i)) == NULL)
    {printf("\nInsufficient memory to hold training set");
     exit(-1);
    }

 i = (long) obs * (nodes_in_layer[0]-1+nodes_in_layer[N-1]);

 while ((fscanf(TrainingSet,xtypef,tset2++) > 0) && i--); 

 if (i>0)
   {
    printf("\nThere's a problem reading training set %s.",buffer);
    printf("\nEither it has too few observations (%d were specified,),",obs);
    printf("\nI/O pair sizes (%d and %d respectively) are wrong,",
           nodes_in_layer[0]-1,nodes_in_layer[N-1]);
    printf("\nor there is a non-numeric entry in training set.");
    exit(-1);
   }

 fclose(TrainingSet);
 return;
}

/***************************************************************/
/*************** Manually establish architecture ***************/
/***************************************************************/

void manually_create_architecture(void)
{int i,j;
 xtype *k;

 printf("\nNumber of Layers "); readint(&N);
 nodes_in_layer = farmalloc(N*sizeof(int));

 printf("\nNumber of entries in the input member of an I/O training pair");
 readint(&i);
 nodes_in_layer[0] = i+1;

 for(i=1; i<N-1; i++)
   {printf("Number of neurons in hidden layer %d",i);
    readint(&nodes_in_layer[i]);
   }

 printf("Number of entries in the output member of an I/O training pair");
 readint(&nodes_in_layer[N-1]);

 printf("\nRandom # seed used to initially establish synaptic weights");
 readint(&i);
 srand(i);

 read_beta_epsilon();
 create_architecture();
 create_transfers_derivatives(); /* each node has a transfer/derivative */

 for(i=0; i<N-1; i++)       /* initial weights  (-1,1) */
   {k = W[i]->Matrix[0];
    for(j=0; j<(W[i]->Rows * W[i]->Cols); j++)
      *k++ = (rand() < 16384) ? -rand()/32767.0 : rand()/32767.0;
   }

 return;
}

/***************************************************************/
/** matrix multiplication (assumes dimensional compatability) **/
/***************************************************************/

void matrix_multiply(matrix *Left,matrix *Right,matrix *Product)
{int i,j,k;
 xtype sum;

 for(i=0; i<Left->Rows; i++)
   {for(j=0; j<Right->Cols; j++)
      {sum=0;
       for(k=0; k<Left->Cols; k++)
          sum += Left->Matrix[i][k] * Right->Matrix[k][j];

       Product->Matrix[i][j] = sum;
   }}
 return;
}

/***************************************************************************/
/** matrix elementwise multiplication (assumes dimensional compatability) */
/***************************************************************************/

void multiply(matrix *Left, matrix *Right, matrix *Result)
{int i,j;

 for(i=0; i<Left->Rows; i++)
   {for(j=0; j<Left->Cols; j++)
      Result->Matrix[i][j] = Left->Matrix[i][j] * Right->Matrix[i][j];
   }

 return;
}

/***************************************************************/
/*********************** print a matrix ************************/
/***************************************************************/

void print_matrix(matrix *Matrix)
{int i,j;

 fprintf(outfile,"\n");
 for(i=0; i<Matrix->Rows; i++)
   {for(j=0; j<Matrix->Cols; j++)
      fprintf(outfile,"%11.6f ",Matrix->Matrix[i][j]);
      fprintf(outfile,"\n");
   }

 return;
}

/***************************************************************/
/************** PROPAGATE A SET OF OBSERVATIONS ****************/
/***************************************************************/

void propagate(void)
{FILE *infile;
 char buffer[50];
 int i,flag=9;

 read_architecture();

 printf("\nFile containing observations to be propagated ");

 while (gets(buffer) && ((infile=fopen(buffer,"rt"))==NULL))
   printf("\n%s not found.  Please reenter",buffer);

 printf("\nFile to write results of propagation ");

 while (gets(buffer) && ((outfile=fopen(buffer,"wt"))==NULL))
   printf("\nProblem opening %s.  Reenter a different name.",buffer);

 fprintf(outfile,"%d %d",nodes_in_layer[0]-1,nodes_in_layer[N-1]);

LOOP:
 fprintf(outfile,"\n");
 X[0]->Matrix[0][0] = 1.0; /* first node of input layer is a bias node */
 i = 1;

 while ((i < nodes_in_layer[0]) &&
   ((flag=fscanf(infile,xtypef,&X[0]->Matrix[i][0])) > 0))

   fprintf(outfile,xtypef,X[0]->Matrix[i++][0]);

 if (flag > 0)
   {forward_propagate();
    fprintf(outfile,"         ");

    for(i=0; i<nodes_in_layer[N-1]; i++)
       fprintf(outfile,xtypef,X[N-1]->Matrix[i][0]);

    goto LOOP;
   }

 fclose(infile);
 fclose(outfile);
 return;
}

/***************************************************************/
/*********************** read an integer ***********************/
/***************************************************************/

void readint(int *x)
 {char buffer[10];

  while (gets(buffer) && (sscanf(buffer,"%d",x)==0))
    printf("Numeric inputs only.  Correct value ");

  return;
 }

/***************************************************************/
/*********************** read a real ***************************/
/***************************************************************/

void readreal(xtype *x)
 {char buffer[10];

  while (gets(buffer) && (sscanf(buffer,xtypef,x)==0))
    printf("Numeric inputs only.  Correct value ");

  return;
 }

/***************************************************************/
/*************** Establish architecture from file **************

 Format for architectural file is:
              @ beta N nodes_in_layer[0]...nodes_in_layer[N-1]
                                   weights

 Whitespace delimits elements.
 ***************************************************************/

void read_architecture(void)
{FILE *infile;
 char buffer[50];
 int i,j;
 xtype *k;

 if (N) remove_network();

LOOP1:
 printf("\nFile containing architecture/weights ");

 while (gets(buffer) && ((infile=fopen(buffer,"rt"))==NULL))
    printf("\n%s not found.  Please reenter.",buffer);

 fscanf(infile,"%c",buffer);

 if (buffer[0] != '@')
   {printf("This is not an architectural file.");
    fclose(infile);
    goto LOOP1;
   }

 fscanf(infile,"%f %d",&beta,&N);
 nodes_in_layer = farmalloc(N*sizeof(int));
 for(i=0; i<N; i++) fscanf(infile,"%d",&nodes_in_layer[i]);

 create_architecture();
 create_transfers_derivatives();

 /* read weights */
 for(i=0; i<N-1; i++)
   {k = W[i]->Matrix[0];
    for(j=0; j<(W[i]->Rows * W[i]->Cols); j++) fscanf(infile,xtypef,k++);
   }

 fclose(infile);
 return;
}

/***************************************************************/
/***************** Read beta and epsilon ***********************/
/***************************************************************/

void read_beta_epsilon(void)
{
 printf("\nLearning rate   (default is %f)",beta);      readreal(&beta);
 printf("Error tolerance > 0 (default is %f)",epsilon); readreal(&epsilon);
 epsilon = fabs(epsilon);

 return;
}

/***************************************************************/
/****** Read in an Input/Output Pair from the training set *****/
/***************************************************************/

void read_IOpair(void)
{int i;

 /* NOTE:  This assumes that the first node in the input layer is a bias 
    node, automatically created in manually_create_architecture().
 */

 X[0]->Matrix[0][0] = 1.0; /* first node of input layer is a bias node */

 /* start at i=1 because the bias node is at i=0 */
 for(i=1; i<nodes_in_layer[0]; i++)    X[0]->Matrix[i][0] = *tset2++;
 for(i=0; i<nodes_in_layer[N-1]; i++) ERROR->Matrix[i][0] = *tset2++;

 return;
}

/***************************************************************/
/***************** Remove current Network **********************/
/***************************************************************/

void remove_network(void)
{int i;

 farfree(nodes_in_layer);    
 farfree(ERROR);

 for(i=0; i<N-1; i++) farfree(W[i]);

 for(i=0; i<N;   i++)
   {farfree(NET[i]);
    farfree(X[i]);
    farfree(DELTA[i]);
    farfree(transfer[i]);
    farfree(derivative[i]);
   }

 farfree(W);     farfree(NET);      farfree(X);
 farfree(DELTA); farfree(transfer); farfree(derivative);

 return;
}

/***************************************************************/
/************** Save the weights of trained network ************/
/***************************************************************/

void save_weights(void)
{char buffer[50];
 int i;

 printf("\nFile to save weights (return if no save)");

 if (gets(buffer) && (sscanf(buffer,"%s",buffer) != EOF))
   {outfile = fopen(buffer,"wt");
    fprintf(outfile,"@%f %d\n",beta,N);
    for(i=0; i<N; i++) fprintf(outfile,"%d ",nodes_in_layer[i]);
    fprintf(outfile,"\n");
    for(i=0; i<N-1; i++) print_matrix(W[i]);
    fclose(outfile);
   }

 return;
}

/***************************************************************/
/***************** Train Neural Network ************************/
/***************************************************************/

void train_network(void)
 {int i;
  char trained,c;
  
  printf("\n");
  printf("\n       (M)anually create architecture");
  printf("\n       (R)ead architecture/weights from external file");
  printf("\n       (Q)uit to main menu");
  if (N) printf("\n       (U)se current architecture/weights");
  printf("\n");

LAB2:
  switch (getch())
    {case 'm' :
     case 'M' : if (N) remove_network();
                manually_create_architecture();
                break;

     case 'r' :
     case 'R' : read_architecture();
                read_beta_epsilon();
                break;

     case 'q' :
     case 'Q' : return;

     case 'u' :
     case 'U' : read_beta_epsilon();
                break;

     default  : printf("\nValid options are M,R,Q,U.  Please reenter.");
                goto LAB2;
    }

  load_trainingSET();

  tset2 = tset1;
  LastCycleMAXERR = MAXERR = cycle = 0;
  i = obs;
  trained = 'a';

LBL5:
  printf("\n\n****** Network is currently training ******");
  printf("\nB change learning rate");
  printf("\nE change error tolerance");
  printf("\nW save weights to file");
  printf("\nQ abort and exit");
  printf("\nA abort to main menu");
  printf("\nAny other key gives status of network");
  printf("\n\n*******************************************");

  while ((!kbhit()) && (trained != 'y'))
    {read_IOpair();
     forward_propagate();

     if (calculate_error()==0) /* 1 is within tolerance */
       {trained = 'n';
        compute_deltas();
	backpropagate();
       }

     if (--i == 0)
       {cycle++;
        LastCycleMAXERR = MAXERR;

        if (trained=='a') trained = 'y';
        else {i = obs; tset2 = tset1; trained = 'a'; MAXERR=0;}
       }
    } /* end while */

  if (trained=='y')
     {printf("\n***** Training Completed in %ld cycles *****",cycle);
      save_weights();

      printf("\nContinue training?");
      c = getch();
      if ((c == 'y') || (c == 'Y'))
         {printf("Old tolerance = %f   New tolerance ",epsilon);
          readreal(&epsilon); epsilon = fabs(epsilon);
          tset2 = tset1;
          trained = 'a';
          i = obs;
          goto LBL5;
         }
      else farfree((xtype far *) tset1);
     }

LAB4:
  if (trained != 'y')
     {switch(getch()) /*switch2*/
        {case 'b' :
         case 'B' : printf("\nOld  = %f   New  = ",beta); 
                    readreal(&beta);
                    goto LBL5;

         case 'e' :
         case 'E' : printf("\nOld  = %f   New  = ",epsilon);
                    readreal(&epsilon);
                    goto LBL5;

         case 'w' :
         case 'W' : save_weights(); break;

         case 'q' :
         case 'Q' : exit(-1);

         case 'a' :
         case 'A' : break;

         default  : printf("\nLastCycleMAXERR=%f cycle=%ld =%f =%f",
                    LastCycleMAXERR,cycle,beta,epsilon);
                    printf("\nContinue?");

                    switch(getch())  /* switch 1 */
                      {case 'n' :
                       case 'N' : printf("Option?"); goto LAB4;
                       default  : goto LBL5;
                      }
        } /* switch */
       } /* if trained not equals y */

  return;
}

/***************************************************************/
/************************ MAIN PROGRAM *************************/
/***************************************************************/

main()
{float xxx=sin(1); /* Borland's patch to overcome the "scanf:  floating
                      point formats not linked" message.  This can occur
                      with the fscanf statements */

 signal(SIGFPE,FPEhandler);

 printf("\n         Ŀ");
 printf("\n                     Public Domain Neural Network 1.0 (PDNNET)       ");
 printf("\n                                                                     ");
 printf("\n               Free, elementary software for building and using      ");
 printf("\n                   backpropagation neural networks on the PC         ");
 printf("\n                                                                     ");
 printf("\n                          User's Guide is MANUAL.DOC                 ");
 printf("\n                                                                     ");
 printf("\n                       copyright 1991 by Bruce Colletti              ");
 printf("\n                              Richmond VA, MAR91                     ");
 printf("\n                           CompuServe:  [71121,1452]                 ");
 printf("\n                              GEnie:  b.colletti                     ");
 printf("\n                                                                     ");
 printf("\n          DISCLAIMER:  Author doesn't guarantee PDNNET's suitability ");
 printf("\n          for an application. It's \"real world\" use is at one's own  ");
 printf("\n          risk.                                                      ");
 printf("\n         ");

LAB1:
 printf("\n\n(T)rain Neural Network\n(P)ropagate observations");
 printf("\n(Q)uit\n");

 switch (getch())
   {case 't' :
    case 'T' : cycle=0; train_network(); break;

    case 'p' :
    case 'P' : propagate(); break;

    case 'q' :
    case 'Q' : exit(-1);

    default  : printf("\nOnly T,P,Q are allowed.  Try again.");
               break;
   }

 goto LAB1;
}
