;;; -*- syntax: common-lisp; package: cmn; base: 10; mode: lisp -*-

(in-package :cmn)

;;; this file provides the wedged beams used by folks like Bartok
;;; there are two versions, a simple beam addition, and a whole new cmn object type
;;;
;;; Finale calls these "feathered" beams.


;;; the simplest version just adds the slanted beams to a normal beam:

(defun display-wedge-beams (mark note beamxy-note score start-beams end-beams &rest rest)
  (declare (ignore mark))
  (when (and (not (member :just-fooling rest))
	     (not (member :just-do-it rest)))
    ;; now find the beam location and draw the other beams
    (let* ((beamxy (outer-beam beamxy-note))
	   (bptr (beams note))
	   (beam-above (eq (beam-direction bptr) :up))
	   (xb (+ (first beamxy) (half-stem-width score)))
	   (xe (third beamxy))
	   (yb (second beamxy))
	   (ye (fourth beamxy))
	   (faster (< start-beams end-beams))
	   (beam-sep1 (beam-spacing score))
	   (beam-wid1 (beam-width score))
	   (beam-sep (if beam-above (- beam-sep1) beam-sep1))
	   (beam-thickness (if beam-above beam-wid1 (- beam-wid1)))
	   (dyb (if faster 0 beam-sep))
	   (dye (if faster beam-sep 0))
	   (beams (abs (- start-beams end-beams))))
      (when (eq (beam-direction bptr) :between)
	(let* ((binfo (beam-data bptr))
	       (bnotes (beam-notes bptr))
	       (need-draw nil))
	  ;; stem-x0 = true location of stem of each note and stem-end = true end.
	  ;; if beam-above and note-above agree, then we have to extend the stem to the current wedge width
	  (setf (line-width score) (stem-width score))
	  (loop for n in bnotes and desc in binfo do
	    (if (and (audible-p n)
		     (not (beams n))
		     (bdat-above desc))
		(progn
		  (setf need-draw t)
		  (moveto score (stem-x0 n) (stem-end n))
		  (rlineto score 0 (* beams
				      beam-sep1
				      (/ (- xe (stem-x0 n)) (- xe xb)))))))
	  (if need-draw (draw score))))
      (when (not beam-above)
	(progn
	  (incf yb beam-wid1)
	  (incf ye beam-wid1)))
      (if (> (min start-beams end-beams) 1)
	  (let ((base-beam-sep (* (1- (min start-beams end-beams)) beam-sep)))
	    (incf ye base-beam-sep)
	    (incf yb base-beam-sep)))

      (let ((left-out-beams (- (min start-beams end-beams) (bdat-fb (first (beam-data bptr))))))
	(when (plusp left-out-beams)
	  (let ((fixup (if (not beam-above) beam-wid1 0.0)))
	    (draw-beams score xb (- yb fixup) xe (- ye fixup) left-out-beams (beam-direction bptr)))))
	    
      (loop for i from 1 to beams and y0 from (+ yb dyb) by dyb and y1 from (+ ye dye) by dye do
	;; now draw the beam with a sharp point at the end with fewer beams
	(moveto score xb y0)
	(fill-in score
		 :path (list xb y0 
			     xe y1
			     xe (+ y1 (if faster beam-thickness 0))
			     xb (+ y0 (if faster 0 beam-thickness))))))))

(defun ur-begin-wedge-beam (which-beam start-beams end-beams &rest args)
  (engorge
   (list
    (apply which-beam
	   no-beam-break
	   (justification (list
			   (* (+ .5 (max start-beams (1- end-beams))) (beam-width *cmn-score*))
			   (* (+ .5 (max end-beams (1- start-beams))) (beam-width *cmn-score*))))
	   args)
    (mark #'(lambda (mark note score &rest rest)
	      (if (member :return-limits rest)
		  (list start-beams end-beams)
		(apply #'display-wedge-beams mark note note score start-beams end-beams rest)))
	  :wedge-beam))))

(defun begin-wedge-beam (start-beams end-beams &rest args)
  (apply #'ur-begin-wedge-beam #'begin-beam start-beams end-beams args))

(defun wedge-beam- (start-beams end-beams &rest args)
  (apply #'ur-begin-wedge-beam #'beam- start-beams end-beams args))

(defun end-wedge-beam (&optional ur-beam)
  (if (and ur-beam
	   (or (not (self-acting-p ur-beam))
	       (not (beam-p (arguments ur-beam)))))
      (cmn-warn "odd argument to end-wedge-beam"))
  (end-beam ur-beam))


(defun fixup-wedge-beam (beam)
  (let ((true-order (reverse (tag-note beam))))
    (if (not (beams (first true-order)))
      ;; we crossed a staff
	(let* ((beamed-note (find-if #'beam-p (tag-note beam) :key #'beams))
	       (old-mark (find :wedge-beam (marks (first true-order)) :key #'(lambda (n) (and (sundry-p n) (sundry-name n)))))
	       (limits (funcall (sundry-mark old-mark) old-mark nil nil :return-limits))
	       (start-beams (first limits))
	       (end-beams (second limits)))
	  ;; :return-limits is a kludge to pass back to us the arguments to the lambda form in the original
	  (add-to-marks beamed-note (list
				     (mark #'(lambda (mark note score &rest rest)
					       (apply #'display-wedge-beams 
						      mark note 
						      (find-if #'outer-beam true-order) 
						      score start-beams end-beams rest))
					   :wedge-beam)))
	  (setf (marks (first true-order)) (remove old-mark (marks (first true-order))))))))


(defun -wedge-beam (beam)
  (make-self-acting
   :action #'(lambda (note old-beam)
	       (let ((true-beam (arguments old-beam)))
		 (push note (tag-note true-beam))
		 (let ((beams (reverse (tag-note true-beam))))
		   (if (every #'onset beams)
		       (progn
			 (setf beams (sort beams #'< :key #'onset))
			 (setf (beamed (first beams))
			   #'(lambda (score staff stf-y0)
			       (annotate-beam score true-beam stf-y0 beams 1.0)
			       (fixup-wedge-beam true-beam))))
		     (setf (beamed (first beams))
		       #'(lambda (score staff stf-y0) 
			   (if (every #'onset beams)
			       (setf beams (sort beams #'< :key #'onset)))
			   (annotate-beam score true-beam stf-y0 beams 1.0)
			   (fixup-wedge-beam true-beam))))
		   nil)))
   :arguments beam))


(defun -wedge-beam- (&optional ur-beam)
  (-beam- ur-beam))

#|
(cmn staff treble (c4 e (begin-wedge-beam 4 1)) (c4 e) (c4 e) (c4 e (end-wedge-beam)))
(cmn staff treble (c5 e (begin-wedge-beam 4 1)) (c5 e) (c5 e) (c5 e (end-wedge-beam)))
(cmn staff treble (c4 e (begin-wedge-beam 1 4)) (c4 e) (c4 e) (c4 e (end-wedge-beam)))
(cmn staff treble (c4 e (begin-wedge-beam 1 4)) (c4 e) (d4 e) (e4 e) (f4 e) (d4 e) (e4 e) (g4 e (end-wedge-beam)))
(cmn (staff treble 
	    (c4 e (let ((wbl (wedge-beam- 4 1 (dy -.5)))) (setf wb (first (data wbl))) wbl)) (c4 e (-wedge-beam- wb)) quarter-rest)
     (staff treble quarter-rest (c4 e (-wedge-beam- wb)) (c4 e (-wedge-beam wb))))
(cmn (size 40) 
     (staff treble 
	    (c4 e (let ((wbl (wedge-beam- 3 1 (dy -.5)))) (setf wb (first (data wbl))) wbl)) (c4 e (-wedge-beam- wb)) quarter-rest)
     (staff (dy -.5) treble quarter-rest (c4 e (-wedge-beam- wb)) (c4 e (-wedge-beam- wb)) 
	    (c4 e (-wedge-beam- wb)) (c4 e (-wedge-beam- wb)) (c4 e (-wedge-beam wb)) ))
(cmn (size 40) 
     (staff treble eighth-rest
	    (c4 e (let ((wbl (wedge-beam- 3 1 (dy -.5)))) (setf wb (first (data wbl))) wbl)) (c4 e (-wedge-beam- wb)) quarter-rest)
     (staff (dy -.5) treble (c4 e (-wedge-beam- wb)) quarter-rest (c4 e (-wedge-beam- wb)) 
	    (c4 e (-wedge-beam- wb)) (c4 e (-wedge-beam- wb)) (c4 e (-wedge-beam wb)) ))

;;; the nominal note duration for every note under the wedge beam should be the duration that corresponds to the
;;; minimum number of beams -- i.e. if we go from 2 to 6 beams, every note should have a duration of s (sixteenth=2 beams).
;;; (I subsequently added a check for this).

(cmn staff treble (c4 s (begin-wedge-beam 2 4)) (c4 s) (c4 s) (c4 s (end-wedge-beam)))
|#



;;; but that version doesn't know what the duration of the overall group is (for metering purposes and
;;; so on), nor does it automatically provide the changed spacing of the notes within the group), so
;;; the following version treats the group under the beam as a sort of chord spread out horizontally.
;;; To keep other simultaneous voices lined up right, we insert invisible spacers into the current staff
;;; and fix up their apparent x0 and x1 later.


(defvar spacer-space 0)      ;a bit of a kludge (else need pointers into current staff data)

(defclass spacer (wrest)
  ((ws :initform 0 :initarg :ws :accessor ws)))

(defmethod house ((sp spacer) score)
  (declare (ignore score))
  (setf (box-x1 sp) (ws sp)))

(defmethod display ((sp spacer) container score &rest rest)
  (declare (ignore container score rest))
  nil)

(defun invisible-spacer (dur white-space)
  (make-instance 'spacer :quarters dur :ws white-space))


(defclass true-space (sundry) ())

(defmethod identify ((sp true-space)) "")

(defmethod backpatch ((tsp true-space)) t)

(defmethod backpatch-time ((tsp true-space) obj)
  (+ (odb-onset obj) (* 4 (quarters obj))))


(defclass bartok-bounce (chord)
  ((data :initarg :data :initform nil :accessor bartok-bounce-data)
   (spacing :initarg :spacing :initform .7 :accessor spacing)
   (start-beams :initarg :start-beams :initform 1 :accessor start-beams)
   (end-beams :initarg :end-beams :initform 1 :accessor end-beams)
   (dxs :initarg :dxs :initform nil :accessor dxs)
   (ws :initarg :ws :initform 0 :accessor ws)))

(self-action start-beams setf-start-beams)
(self-action end-beams setf-end-beams)
(self-action spacing setf-spacing)

(defmethod bartok-bounce-p ((obj t)) nil)
(defmethod bartok-bounce-p ((obj bartok-bounce)) t)

(defmethod maximum-line ((bb bartok-bounce)) (loop for note in (bartok-bounce-data bb) maximize (maximum-line note)))
(defmethod minimum-line ((bb bartok-bounce)) (loop for note in (bartok-bounce-data bb) minimize (minimum-line note)))

(defun bartok-bounce (&rest objects)
  (let ((notes nil)
	(bb (make-instance 'bartok-bounce)))
    (setf *cmn-owning-object* bb)
    (setf (center bb) .15)
    (loop for object in objects do
      (setf *cmn-object* object)
      (when object
	(if (self-acting-p object)
	    (funcall (action object) bb (arguments object))
	  (if (or (note-p object) (chord-p object) (rest-p object))
	      (push object notes)
	    (if (rhythm-p object)
		(rcopy object bb)
	      (if (score-object-p object)
		  (if (or (sundry-p object) (pause-p object) (dynamics-p object) (glyph-list-p object))
		      (add-to-marks bb (list (if (write-protected object) (copy object) object)))
		    (copy object bb))
		(if (score-object-list-p object)
		    (loop for note in (data object) do
		      (push (if (write-protected note) (copy note) note) notes))
		  (if (text-p object)
		      (add-to-marks bb (list object))
		    (cmn-warn "odd argument to bartok-bounce: ~A" object)))))))))
    (setf (bartok-bounce-data bb) (nreverse notes))
    (loop for note in (bartok-bounce-data bb) do
      (setf (beat note) 0)
      (setf (flags note) 1)
      (setf (quarters note) 1/2))
    (cmn-tick-pipe bb)
    (setf (quarters bb) (/ (quarters bb) 4))
    
    (let* ((notes (length (bartok-bounce-data bb)))
	   (spaces (1- notes))
	   (note-linear-dxs (loop for note in (bartok-bounce-data bb)
			     collect (+ .5 (if (note-p note)
					       (if (note-sign note) (+ (g-rx (note-sign note)) .05) 0)
					     (if (chord-p note) 
						 (if (find-if #'sign (chord-data note)) .3 0)
					       0)))))
	   (linear-x-space (+ .1 (apply #'+ note-linear-dxs)))
	   (faster (> (end-beams bb) (start-beams bb)))
	   (note-dxs (append (list 0)
			     (let* ((space (* .5 linear-x-space))
				    (spacex (loop for n from 1 to spaces 
					     collect (* space (spacing bb))
					     do (decf space (* space (spacing bb))))))
			       (if (not faster) (setf spacex (nreverse spacex)))
			       (loop for x in note-linear-dxs and dx in spacex 
				collect (+ x dx)))))
	   (x-space (apply #'+ note-dxs)))
      (setf (dxs bb) note-dxs)
      (let ((qb (quarters bb))
	    (qs (* .25 x-space)))
	(setf (ws bb) qs)
	(add-to-marks bb (list (make-instance 'true-space 
				   :name :true-space
				   :mark #'(lambda (mark note score &rest rest)
					     (declare (ignore mark note score rest))
					     nil))))
	(engorge (list
		  bb 
		  (invisible-spacer qb qs)
		  (invisible-spacer qb qs)
		  (invisible-spacer qb qs)))))))

(defun house-hidden-chord (chord score)
  (declare (ignore score))
  (setf *old-cmn-score-size* *cmn-score-size*)
  (let ((new-score (make-score)))
    (setf (scr-size new-score) 1.0)
    (setf *cmn-score-size* 1.0)
    (let ((nc (display chord nil new-score :just-fooling)))
      (setf *cmn-score-size* *old-cmn-score-size*)
      (setf *old-cmn-score-size* nil)
      nc)))

(defmethod display ((bb bartok-bounce) container score &rest rest)
  (declare (ignore container))
  ;; set position of each note, beam the bb, etc
  ;; if this is a housing pass, just figure out how much room we'll need at a minimum
  (if (member :just-fooling rest)
      (let ((x0 (box-x0 bb))
	    (dxtotal (apply #'+ (dxs bb)))
	    (dxcur 0.0))
	(moveto score (+ (box-x0 bb) (ws bb)) (box-y0 bb))
	(loop for note in (bartok-bounce-data bb) and dx in (dxs bb) do
	  (setf (box-x0 note) (+ x0 dx))
	  (setf (duration note) 0)
	  (setf (onset note) (+ (onset bb) (* 4 (duration bb) (/ dxcur dxtotal))))
	  (incf dxcur dx)
	  (setf (box-x1 note) (box-x0 note))
	  (incf x0 dx))
	.15)
    (let* ((x0 (+ (box-x0 bb) (dxy-dx bb)))
	   (stfy0 (%staff-y0 bb))
	   (bx1 (box-x1 (find :true-space (marks bb) :key #'sundry-name)))
	   (extra-space (max 0 (- bx1 (+ (box-x0 bb) 1 (* 4 (ws bb))))))
	   (note-dxs (dxs bb))
	   (pad (/ extra-space (1+ (length note-dxs)))))
      (loop for note in (bartok-bounce-data bb) and dx in note-dxs do
	(setf (box-x0 note) (+ x0 dx))
	(incf x0 (+ dx pad))
	(setf (box-y0 note) (* (minimum-line note) (staff-line-separation score)))
	(if (not (rest-p note))
	    (setf (center note) (+ .15 
				   (if (note-p note)
				       (if (and (note-sign note) (zerop (audible-collision note)))
					   (+ (g-rx (note-sign note))
					      (if (or (<= (minimum-line note) -2) (<= 10 (maximum-line note)))
						  (leger-line-length note)
						0)
					      .075)
					 0)
				     (if (chord-p note)
					 (if (some #'note-sign (chord-data note))
					     (let ((nc (house-hidden-chord note score)))
					       (- nc x0))
					   0)
				       0)))))
	(let ((slop (- (center note) .15)))
	  (decf (box-x0 note) slop))
	(if (chord-p note) (loop for n in (chord-data note) do (setf (%staff-y0 n) stfy0)))
	(setf (%staff-y0 note) stfy0))
      (annotate-beam score nil stfy0 (bartok-bounce-data bb) 1.0)
      (setf (visible-justification (beams (first (bartok-bounce-data bb)))) 
	(list (* (+ .5 (max (start-beams bb) (1- (end-beams bb)))) (beam-width score))
	      (* (+ .5 (max (end-beams bb) (1- (start-beams bb)))) (beam-width score))))
      (add-to-marks (first (bartok-bounce-data bb))
		    (list (mark #'(lambda (mark note score &rest rest)
				    (apply #'display-wedge-beams mark note note score 
					   (start-beams bb) 
					   (end-beams bb)
					   rest))
				:wedge-beam)))
      (loop for note in (bartok-bounce-data bb) do
	(display note nil score)))))

;;; now set up the wedged beam stuff as before
;;; (cmn staff treble (bartok-bounce q (notes c4 d4 e4 f4) (start-beams 1) (end-beams 3)))
#|
(cmn (staff treble (c4 q) (bartok-bounce q (notes cs4 d4 ef4 f4 e4 d4 c4) (spacing .7) (start-beams 1) (end-beams 3)) (c4 q)) 
     (staff bass c3 e c3 e c3 e c3 e c3 e c3 e))
(cmn (staff treble (c4 q) (bartok-bounce q (notes (chord c4 g4) d4 eighth-rest f4 e4 d4 c4) (spacing .7) (start-beams 1) (end-beams 3)) (c4 q)) 
     (staff bass c3 e c3 e c3 e c3 e c3 e c3 e))
(cmn (staff treble (c4 q) 
       (bartok-bounce q (notes (chord cs4 g4) d4 eighth-rest f4 (chord e4 fs4) d4 c4) (spacing .7) (start-beams 1) (end-beams 3)) (c4 q)) 
     (staff bass c3 e c3 e c3 e c3 e c3 e c3 e))

;;; the arguments to bartok-bounce are the duration (and onset time if desired)
;;;    the list of notes under the wedged beams (in the "notes" statement) -- each of these is a full fledged note
;;;    start-beams and end-beams (as above)
;;;    spacing = how to space the notes -- between 0 and 1, default is .7 -- clumpier as you approach 1

 (cmn (staff treble (meter 2 4) 
        (c4 q) 
        (bartok-bounce q (notes (c4 staccato) (d4 marcato) 
                                (e4 upside-down-fermata) (f4 (bartok-pizzicato (dy .5))) 
                                (e4 (fingering 7 9)) (d4 ppp) c4) 
          (spacing .5) (start-beams 1) (end-beams 3)) 
        (c4 q)) 
      (staff bass (meter 2 4) c3 e c3 e c3 e c3 e c3 e c3 e))


(cmn (size 24) (AUTOMATIC-LINE-BREAKS nil) 
  (staff treble       
    (note fs5 s (begin-wedge-beam 2 4))(note fn5 s)(note e5 s)(note fs5 s) 
    (note fn5 s) (note e5 s) (note fs5 s) (note fn5 s)
    (note e5 s) (note fs5 s) (note fn5 s) (note e5 s)
    (note fs5 s) (note fn5 s) (note e5 s) (note fs5 s)
    (note fn5 s) (note e5 s) (note fs5 s)(note fn5 s)
    (note e5 s) (note fs5 s) (note fn5 s) (note e5 s (end-wedge-beam))
    (bar)(line-mark)))

|#
