;;; -*- syntax: common-lisp; package: cmn; base: 10; mode: lisp -*-
;;;
;;; continuation of cmn0.lisp

(in-package :cmn)

(defclass staff-relative-mixin ()
  ((staff-y0 :initarg :staff-y0 :initform 0 :accessor staff-y0 :accessor %staff-y0)))

(defmethod staff-relative-mixin-p ((obj t)) nil)
(defmethod staff-relative-mixin-p ((obj staff-relative-mixin)) t)

(defmethod descry ((stf-y0 staff-relative-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(staff-relative-mixin" "")
	  (format nil " :staff-y0 ~A" (staff-y0 stf-y0))
	  (if (next-method-p) (call-next-method stf-y0 stream (or controller stf-y0)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((stf-y0 staff-relative-mixin) &optional object)
  (let ((new-stf (if (not object) (make-staff-relative-mixin)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (%staff-y0 new-stf) (%staff-y0 stf-y0))
    (if (next-method-p) (call-next-method stf-y0 new-stf))
    new-stf))




(defclass thick-mixin ()
  ((thickness :initarg :thickness :initform nil :reader thickness)))

(defclass thick (thick-mixin)
  ((thickness :accessor thickness)))

(defmethod descry ((thk thick-mixin) &optional stream controller)
  (format stream "~A~A"
	  (if (thickness thk) (format nil " :thickness ~A" (thickness thk)) "")
	  (if (next-method-p) (call-next-method thk stream (or controller thk)) "")))

(defmethod copy ((thk thick-mixin) &optional object)
  (setf (thickness object) (thickness thk))
  (if (next-method-p) (call-next-method thk object))
  object)

(self-action thickness setf-thickness)
    


(defclass breathing-space-mixin ()
  ((breathing-space :initarg :breathing-space :initform 0 :reader breathing-space)))

(defclass breathing-space ()
  ((breathing-space :initarg :breathing-space :initform 0 :accessor breathing-space)))

(defmethod descry ((thk breathing-space-mixin) &optional stream controller)
  (format stream "~A~A"
	  (if (not (zerop (breathing-space thk))) (format nil " :breathing-space ~1,3F" (breathing-space thk)) "")
	  (if (next-method-p) (call-next-method thk stream (or controller thk)) "")))

(defmethod copy ((thk breathing-space-mixin) &optional object)
  (setf (breathing-space object) (breathing-space thk))
  (if (next-method-p) (call-next-method thk object))
  object)

(self-action breathing-space setf-breathing-space)
    



;;;
;;; ----------------    bar-lines, double-bars etc
;;;

(defclass bar-mixin (staff-relative-mixin score-object-mixin thick-mixin breathing-space)
  ((double :initarg :double :initform nil :reader double)
   (terminal :initarg :terminal :initform nil :reader terminal)
   (pause :initarg :pause :initform nil :reader bar-pause)
   (within-measure :initarg :within-measure :initform nil :reader within-measure)
   (dashed :initarg :dashed :initform nil :reader dashed)
   (dots-left :initarg :dots-left :initform nil :reader dots-left)
   (dots-right :initarg :dots-right :initform nil :reader dots-right)
   (inserted :initarg :inserted :initform nil :reader inserted)
   (broken :initarg :broken :initform t :reader broken)
   (thick-lines :initarg :thick-lines :initform nil :reader thick-lines)
   (marks :reader bar-marks)
   (thickness :initform .1)
   (breathing-space :initform .1)))

;;; bar can be single line, double line, either can be light or heavy or dashed, either side may have repeat dots
;;; marks can be dal-segno fine ds-al-coda dc dc-al-fine etc
;;; repeat bar can occur within a measure and has no metric significance
;;; double-repeat bar can be of three (or more?) forms: [thin-line thick line], [thin thick thick thin], [thick thick]


(defclass bar (bar-mixin staff-relative-mixin score-object thick breathing-space)
  ((double :accessor double)
   (terminal :accessor terminal)
   (pause :accessor bar-pause)
   (within-measure :accessor within-measure)
   (dashed :accessor dashed)
   (dots-left :accessor dots-left)
   (dots-right :accessor dots-right)
   (inserted :accessor inserted)
   (broken :accessor broken)
   (thick-lines :accessor thick-lines)
   (marks :accessor bar-marks)))

(defclass write-protected-bar (write-protect bar-mixin)
  ())

(self-action double setf-double)
(self-action terminal setf-terminal)
(self-action justification setf-justification)
(self-action dashed setf-dashed)
(self-action within-measure setf-within-measure)
(self-action dots-left setf-dots-left)
(self-action dots-right setf-dots-right)
(self-action broken setf-broken)
(self-action thick-lines setf-thick-lines)

(defvar bar (make-instance 'write-protected-bar))
(defvar double-bar (make-instance 'write-protected-bar :double t :terminal t))
(defvar terminal-bar (make-instance 'write-protected-bar :terminal t))
(defvar interior-double-bar (make-instance 'write-protected-bar :double t))
(defvar dashed-bar (make-instance 'write-protected-bar :dashed t))
(defvar begin-repeat-bar (make-instance 'write-protected-bar :double t :dots-right t))
(defvar end-repeat-bar (make-instance 'write-protected-bar :double t :dots-left t))
(defvar final-repeat-bar (make-instance 'write-protected-bar :double t :dots-left t :terminal t))
(defvar begin-and-end-repeat-bar (make-instance 'write-protected-bar :double t :dots-right t :dots-left t))
(defvar full-bar (make-instance 'write-protected-bar :broken nil))
(defvar full-double-bar (make-instance 'write-protected-bar :double t :terminal t :broken nil))
(defvar full-interior-double-bar (make-instance 'write-protected-bar :double t :broken nil))
(defvar begin-and-end-repeat-bar-without-thin-lines (make-instance 'write-protected-bar :double t :dots-right t :dots-left t :thick-lines 2))
(defvar begin-and-end-repeat-bar-with-one-thick-line (make-instance 'write-protected-bar :double t :dots-right t :dots-left t :thick-lines 1))

(defmethod bar-p ((obj t)) nil)
(defmethod bar-p ((obj bar-mixin)) t)

(defun bar-print-name (bar)
  (if (double bar) 
      (if (terminal bar) 
	  (if (dots-left bar)
	      "final-repeat-bar"
	    (if (broken bar)
		"double-bar"
	      "full-double-bar"))
	(if (dots-left bar)
	    (if (dots-right bar)
		(if (not (thick-lines bar))
		    "begin-and-end-repeat-bar"
		  (if (= (thick-lines bar) 1)
		      "begin-and-end-repeat-bar-with-one-thick-line"
		    "begin-and-end-repeat-bar-without-thin-lines"))
	      "end-repeat-bar")
	  (if (dots-right bar)
	      "begin-repeat-bar"
	    (if (broken bar)
		"interior-double-bar"
	      "full-interior-double-bar"))))
    (if (terminal bar)
	"terminal-bar"
      (if (dashed bar)
	  "dashed-bar"
	(if (broken bar)
	    "bar"
	  "full-bar")))))

(defmethod descry ((bar bar-mixin) &optional stream controller)
  (format stream "(~A~A~A~A~A~A~A~A~A~A"
	  (if (not controller) (format nil "~A" (bar-print-name bar)) "")
	  (if (bar-pause bar) (format nil " :pause ~A" (bar-pause bar)) "")
	  (if (within-measure bar) (format nil " :within-measure ~A" (within-measure bar)) "")
	  (if (dashed bar) (format nil " :dashed ~A" (dashed bar)) "")
	  (if (dots-left bar) (format nil " :dots-left ~A" (dots-left bar)) "")
	  (if (dots-right bar) (format nil " :dots-right ~A" (dots-right bar)) "")
	  (if (inserted bar) " (created by cmn)" "")
	  (if (thick-lines bar) (format nil " :thick-lines ~D" (thick-lines bar)) "")
	  (if (next-method-p) (call-next-method bar stream (or controller bar)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((bar bar-mixin) &optional object)
  (let ((new-bar (if (not object) (make-bar)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (double new-bar) (double bar))
    (setf (terminal new-bar) (terminal bar))
    (setf (broken new-bar) (broken bar))
    (setf (within-measure new-bar) (within-measure bar))
    (setf (dots-left new-bar) (dots-left bar))
    (setf (dots-right new-bar) (dots-right bar))
    (setf (dashed new-bar) (dashed bar))
    (setf (inserted new-bar) (inserted bar))
    (setf (thick-lines new-bar) (thick-lines bar))
    (if (bar-pause bar) (setf (bar-pause new-bar) (copy (bar-pause bar))))
    (if (next-method-p) (call-next-method bar new-bar))
    new-bar))

(defmethod identify ((bar bar-mixin))
  (format nil "(~A~A~A~A~A)" 
	  (bar-print-name bar)
	  (check-for-cmn-store-tag bar)
	  (identify-marks bar)
	  (identify-dxy bar)
	  (if (bar-pause bar) (identify (bar-pause bar)) "")))

(defun bar (&rest objects)
  (apply #'ur-bar (make-bar) objects))

(defun %bar (&key onset beat duration hidden)
  (let ((new-bar (make-bar)))
    (setf (odb-onset new-bar) onset)
    (setf (odb-beat new-bar) beat)
    (setf (odb-duration new-bar) duration)
    (setf (inserted new-bar) t)
    (if hidden (setf (matrix new-bar) (list 0 0 0 0 0 0)))
    new-bar))

(defun ur-bar (new-bar &rest objects)
  (loop for act in objects do
    (when act
      (if (self-acting-p act)
	  (funcall (action act) new-bar (arguments act))
	(if (bar-p act)
	    (copy act new-bar)
	  (if (pause-p act)
	      (setf (bar-pause new-bar) (copy act))
	    (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
		(push (copy act) (bar-marks new-bar))
	      (cmn-warn "odd argument to bar: ~A" act)))))))
  new-bar)

(defmethod minimum-line ((bar bar-mixin)) 0)

(defun terminal-bar (&rest args) (apply #'bar (terminal t) args))
(defun double-bar (&rest args) (apply #'bar (double t) (terminal t) args))
(defun interior-double-bar (&rest args) (apply #'bar (double t) args))
(defun dashed-bar (&rest args) (apply #'bar (dashed t) args))
(defun begin-repeat-bar (&rest args) (apply #'bar (double t) (dots-right t) args))
(defun end-repeat-bar (&rest args) (apply #'bar (double t) (dots-left t) args))
(defun final-repeat-bar (&rest args) (apply #'bar (double t) (dots-left t) (terminal t) args))
(defun begin-and-end-repeat-bar (&rest args) (apply #'bar (double t) (dots-right t) (dots-left t) args))
(defun begin-and-end-repeat-bar-without-thin-lines (&rest args) (apply #'bar (double t) (dots-right t) (dots-left t) (thick-lines 2) args))
(defun begin-and-end-repeat-bar-with-one-thick-line (&rest args) (apply #'bar (double t) (dots-right t) (dots-left t) (thick-lines 1) args))

(defun full-double-bar (&rest args) (apply #'bar (double t) (terminal t) (broken nil) args))
(defun full-interior-double-bar (&rest args) (apply #'bar (double t) (broken nil) args))
(defun full-bar (&rest args) (apply #'bar (broken nil) args))

(defmethod notify ((bar bar) &optional objects)
  (apply #'ur-bar bar objects))

#+Petrucci (defun Petrucci-repeat-dots (score x0 y0)
		    (circle score (+ x0 .1) (+ y0 .375) .0625 0 360 t)
		    (circle score (+ x0 .1) (+ y0 .375 .25) .0625 0 360 t))

(defmethod display ((bar bar-mixin) container score &rest rest)
  (declare (ignore container))
  (let* ((x0 (+ (box-x0 bar) (dxy-dx bar)))
	 (y0 (+ (box-y0 bar) (dxy-dy bar)))
	 (y1 (+ (box-y1 bar) (dxy-dy bar)))
	 (double-bar-space (breathing-space bar))
	 (double-bar-thickness (thickness bar))
	 (dot-space .25)
	 (size (scr-size score))
	 (px0 (- x0 (if (double bar)
			(if (or (terminal bar) (dots-left bar))
			    (+ double-bar-thickness
			       (if (or (not (thick-lines bar)) (/= (thick-lines bar) 2)) double-bar-space 0)		       
			       (if (or (and (dots-left bar)
					    (eq (justification bar) :right))
				       (dots-right bar))
				   .2 0)
			       )
			  double-bar-space)
			0))))
    (when (bar-pause bar) (display (bar-pause bar) bar score))
    (if (bar-marks bar) (apply #'display-marks bar score rest))
    (when (not (invisible-matrix-p bar))
      ;; now the zillion possible bars -- dashed single, single, double, dashed double
      ;;   terminal, double both, left and right dots, probably more I've forgotten.
      (moveto score px0 y0)
      (if (dots-left bar)
	  (progn
	    #+Sonata (simple-show score %repeatdots)
	    #+Petrucci (Petrucci-repeat-dots score px0 y0)
	    (incf px0 dot-space)
	    (moveto score px0 y0)))
      (if (and (or (not (dots-right bar))
		   (dots-left bar))
	       (or (not (thick-lines bar))
		   (/= (thick-lines bar) 2)))
	  (progn
	    (setf (line-width score) 0)
	    (if (dashed bar)
		(lineto score px0 y1 :pattern (list (floor (* 5 (/ size 40))) (floor (* 7 (/ size 40)))))
	      (progn
		(lineto score px0 y1)
		(draw score)))
	    (incf px0 double-bar-space)))
      (if (and (thick-lines bar) (= (thick-lines bar) 2)) (decf px0 .05))
      (if (double bar)
	  (progn
	    (if (or (terminal bar) (dots-left bar) (dots-right bar))
		(progn
		  (fill-in score :rectangle t :path (list px0
							  y0
							  double-bar-thickness
							  (- y1 y0)))
		  (incf px0 (+ double-bar-thickness double-bar-space))
		  (if (and (dots-left bar) 
			   (dots-right bar)
			   (or (not (thick-lines bar))
			       (/= (thick-lines bar) 1)))
		      (progn
			(fill-in score :rectangle t :path (list px0
								y0
								double-bar-thickness
								(- y1 y0)))
			(incf px0 (+ double-bar-thickness double-bar-space))))))
	    (if (and (or (dots-right bar)
			 (not (or (dots-left bar) (dots-right bar) (terminal bar))))
		     (or (not (thick-lines bar))
			 (/= (thick-lines bar) 2)))
		(progn
		  (moveto score px0 y0)
		  (lineto score px0 y1)
		  (draw score)
		  (incf px0 double-bar-space)))
	    (if (dots-right bar)
		(progn
		  (moveto score px0 y0)
		  #+Sonata (simple-show score %repeatdots)
		  #+Petrucci (Petrucci-repeat-dots score px0 y0)
		  )))))))


(defparameter bar-walls '(.1 .1))	;was .1 0 but that presses against right side too tightly
;(defparameter bar-fences '(.05 .025))
;(defparameter bar-expanders '(1 1))
(defparameter bar-fences '(.1 .05))
(defparameter bar-expanders '(0 1))

(defmethod house ((bar bar-mixin) score)
  (declare (ignore score))
  (let ((dot-space .25))
    (setf (box-x1 bar) (+ (if (dots-left bar) dot-space 0)
			  (if (dots-right bar) dot-space 0)
			  (if (double bar) (breathing-space bar) 0)
			  (if (terminal bar) (thickness bar) 0)
			  (if (and (dots-right bar) (dots-left bar)) 
			      (+ (breathing-space bar) (thickness bar)) 
			    0)))
    (setf (box-y1 bar) 1.0)
    (setf (center bar) (* .5 (box-x1 bar)))
    (if (not (walls bar)) (setf (walls bar) bar-walls))
    (if (not (fences bar)) (setf (fences bar) bar-fences))
    (if (not (expanders bar)) (setf (expanders bar) bar-expanders))))

;;; (cmn staff treble c4 w begin-and-end-repeat-bar c4 w begin-and-end-repeat-bar-without-thin-lines 
;;;    c4 w begin-and-end-repeat-bar-with-one-thick-line c4 w end-repeat-bar)



;;;
;;; ----------------    braces, brackets
;;;

(defclass bracket-mixin (score-object-mixin)
  ())

(defclass bracket (bracket-mixin score-object) ())

(defclass write-protected-bracket (write-protect bracket-mixin) ())

(defvar bracket (make-instance 'write-protected-bracket))

(defmethod bracket-p ((obj t)) nil)
(defmethod bracket-p ((obj bracket-mixin)) t)

(defmethod descry ((bracket bracket-mixin) &optional stream controller)
  (format stream "~A~A~A"
	  (if (not controller) "(bracket" "")
	  (if (next-method-p) (call-next-method bracket stream (or controller bracket)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((bracket bracket-mixin))
  (format nil "(bracket~A~A)" (identify-dxy bracket) (identify-marks bracket)))

(defmethod copy ((bracket bracket-mixin) &optional object)
  (let ((new-bracket (if (not object) (make-bracket)
		       (if (write-protected object) (copy object)
			 object))))
    (if (next-method-p) (call-next-method bracket new-bracket))
    new-bracket))

(defun bracket (&rest objects)
  (let ((new-bracket (make-bracket)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-bracket (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (push act (marks new-bracket))
	    (cmn-warn "odd argument to bracket: ~A" act)))))
    new-bracket))

(defmethod notify ((bracket bracket) &optional objects)
  (apply #'bracket bracket objects))

#+Petrucci (defun Petrucci-bracket (score bx by up)
		    (let* ((xfactor .002)
			   (yfactor (if up xfactor (- xfactor))))
		      (moveto score bx by)
		      (g-begin-filled-polygon score)
		      (rlineto score 0 (* -145 yfactor))
		      (rlineto score (* 75 xfactor) 0)
		      (rlineto score 0 (* 110 yfactor))
		      (curveto score (+ bx (* 130 xfactor)) (- by (* 15 yfactor))
			             (+ bx (* 170 xfactor)) (+ by (* 5 yfactor))
			             (+ bx (* 210 xfactor)) (+ by (* 65 yfactor)))
		      (rlineto score (* -12 xfactor) (* 3 yfactor))
		      (curveto score (+ bx (* 170 xfactor)) (+ by (* 35 yfactor))
			             (+ bx (* 130 xfactor)) (+ by (* 5 yfactor))
			             bx by)
		      (fill-in score)))


#|
 100 365 moveto
 0 -145  rlineto  
 75 0 rlineto
 0 110 rlineto % 178 330
 230 350 270 370 310 430 curveto
 -12 3 rlineto
  270 400 230 370 100 365 curveto
 fill
 stroke
|#


(defmethod display ((bracket bracket-mixin) container score &rest rest)
  (declare (ignore container))
  (let* ((x0 (+ (box-x0 bracket) (dxy-dx bracket)))
	 (y0 (+ (box-y0 bracket) (dxy-dy bracket)))
	 (y1 (+ (box-y1 bracket) (dxy-dy bracket))) ;y1=(+ y0 1 (* 3 (1- staves)))
	 (bx (/ (floor (* (- (or x0 0.0) .25) (scr-size score))) (scr-size score)))
	 ;; the bracket glyph gets positioned at the pixel boundary, so we have to fix up our box to follow suit
	 (by y0))
    (if (marks bracket) (apply #'display-marks bracket score rest))
    (comment score "bracket")
    #+Sonata (moveto score bx by)
    #+Sonata (simple-show score %bottombracket)
    #+Petrucci (Petrucci-bracket score bx (- by .2) nil)
    (moveto score bx y0)
    (g-begin-filled-polygon score)
    (lineto score bx y1)
    (lineto score (+ bx .15) y1)
    (lineto score (+ bx .15) y0)
    (fill-in score :closepath t)
    #+Sonata (moveto score bx y1)
    #+Sonata (simple-show score %topbracket)
    #+Petrucci (Petrucci-bracket score bx (+ y1 .2) t)
    (when (and rest (member :with-line rest))
      (moveto score (+ x0 (box-x0 %staff)) (+ by (box-y0 %staff)))
      (lineto score (+ x0 (box-x0 %staff)) y1)
      (draw score))))

(defparameter bracket-walls '(.05 .05))

(defmethod house ((bracket bracket-mixin) score)
  (declare (ignore score))
  (setf (box-y1 bracket) 1.0)
  (if (not (walls bracket)) (setf (walls bracket) bracket-walls)))



(defclass brace-mixin (score-object-mixin)
  ((staves :initarg :staves :initform 2 :reader brace-staves)))

(defclass brace (brace-mixin score-object)
  ((staves :accessor brace-staves)))

(defclass write-protected-brace (write-protect brace-mixin) ())

(defvar brace (make-instance 'write-protected-brace))

(defmethod brace-p ((obj t)) nil)
(defmethod brace-p ((obj brace-mixin)) t)

(defmethod descry ((brace brace-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(brace" "")
	  (if (/= (brace-staves brace) 2) (format " :staves ~D" (brace-staves brace)) "")
	  (if (next-method-p) (call-next-method brace stream (or controller brace)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((brace brace-mixin))
  (format nil "(brace~A~A~A)" 
	  (if (/= (brace-staves brace) 2) (format " (brace-staves ~D)" (brace-staves brace)) "")
	  (identify-dxy brace) 
	  (identify-marks brace)))

(defmethod copy ((brace brace-mixin) &optional object)
  (let ((new-brace (if (not object) (make-brace)
		     (if (write-protected object) (copy object)
		       object))))
    (setf (brace-staves new-brace) (brace-staves brace))
    (if (next-method-p) (call-next-method brace new-brace))
    new-brace))

(self-action brace-staves setf-brace-staves)

(defun brace (&rest objects)
  (apply #'ur-brace (make-brace) objects))

(defun ur-brace (new-brace &rest objects)
  (loop for act in objects do
    (when act
      (if (self-acting-p act)
	  (funcall (action act) new-brace (arguments act))
	(if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	    (push act (marks new-brace))
	  (cmn-warn "odd argument to brace: ~A" act)))))
  new-brace)

(defmethod notify ((brace brace) &optional objects)
  (apply #'ur-brace brace objects))

#|
  ;;; here's a brace drawn using 4 bezier curves (for Petrucci)
  100 200 moveto 
  50 300 130 300 80 360 curveto
  145 290 60 290 100 200 curveto
  fill
  100 200 moveto
  50 100 130 100 80 40 curveto
  145 110 60 110 100 200 curveto
  fill
|#
#+Petrucci 
(defmethod display ((brace brace-mixin) container score &rest rest)
  (declare (ignore container))
  (let* ((tx0 (+ (box-x0 brace) (dxy-dx brace)))
	 (x0 (- tx0 (+ .1 (brace-space score))))
	 (y0 (+ (box-y0 brace) (dxy-dy brace)))
	 (y1 (+ (box-y1 brace) (dxy-dy brace)))
	 (midy (* .5 (+ y0 y1)))
	 (xfactor (/ (- y1 y0) 3.2))
	 (yfactor (/ (- y1 y0) 3.2))
	 )
    (if (marks brace) (apply #'display-marks brace score rest))
    (moveto score x0 midy)
    (g-begin-filled-polygon score)
    (curveto score (+ x0 (* .5 xfactor)) (+ midy yfactor)
	           (- x0 (* .3 xfactor)) (+ midy yfactor)
		   (+ x0 (* .2 xfactor)) (+ midy (* 1.6 yfactor)))
    (curveto score (- x0 (* .45 xfactor)) (+ midy (* .9 yfactor))
	           (+ x0 (* .4 xfactor)) (+ midy (* .9 yfactor))
		   x0 midy)
    (fill-in score)
    (moveto score x0 midy)
    (g-begin-filled-polygon score)
    (curveto score (+ x0 (* .5 xfactor)) (- midy yfactor)
	           (- x0 (* .3 xfactor)) (- midy yfactor)
		   (+ x0 (* .2 xfactor)) (- midy (* 1.6 yfactor)))
    (curveto score (- x0 (* .45 xfactor)) (- midy (* .9 yfactor))
	           (+ x0 (* .4 xfactor)) (- midy (* .9 yfactor))
		   x0 midy)
    (fill-in score)
    (when (and rest (member :with-line rest))
      (moveto score (+ tx0 (box-x0 %staff)) (+ y0 (box-y0 %staff)))
      (lineto score (+ tx0 (box-x0 %staff)) y1)
      (draw score))))


#+Sonata 
(defmethod display ((brace brace-mixin) container score &rest rest)
  (declare (ignore container))
  (let* ((x0 (+ (box-x0 brace) (dxy-dx brace)))
	 (y0 (+ (box-y0 brace) (dxy-dy brace)))
	 (y1 (+ (box-y1 brace) (dxy-dy brace))))
    (if (marks brace) (apply #'display-marks brace score rest))
    (if (= (- y1 y0) 4)
	(let* ((pb-x (- (or x0 0.0) (brace-space score) .2))
	       (pb-y (+ y0 1.0))
	       (pt-x pb-x)
	       (pt-y (+ pb-y 2)))
	  (comment score "brace")
	  (moveto score pb-x pb-y)
	  (simple-show score %bracedown)
	  (moveto score pt-x pt-y)
	  (simple-show score %braceup)
	  (when (and rest (member :with-line rest))
	    (moveto score (+ x0 (box-x0 %staff)) (+ y0 (box-y0 %staff)))
	    (lineto score (+ x0 (box-x0 %staff)) (+ y0 4))
	    (draw score)))
      (let* ((scl (/ (- y1 y0) 4.0))
	     (old-size (scr-size score))
	     (new-size (round (* (scr-size score) scl)))
	     (fscl (/ new-size old-size))
	     (pt-x (- x0 (* fscl (+ .2 (brace-space score)))))
	     (pt-y0 (+ y0 fscl))
	     (pt-y1 (+ pt-y0 (* 2 fscl))))
	(comment score "scaled brace")
	(moveto score pt-x pt-y0)
	(show score %bracedown :size new-size)
	(moveto score pt-x pt-y1)
	(show score %braceup :size new-size)
	(when (and rest (member :with-line rest))
	  (moveto score (+ x0 (box-x0 %staff)) (+ y0 (box-y0 %staff)))
	  (lineto score (+ x0 (box-x0 %staff)) y1)
	  (draw score))))))

(defparameter brace-walls '(.1 .05))

(defmethod house ((brace brace-mixin) score)
  (declare (ignore score))
  (setf (box-y1 brace) 1.0)
  (if (not (walls brace)) (setf (walls brace) brace-walls)))

;;; (cmn (system (brace (brace-staves 3)) (staff (staff-size .7) treble c4 q) (staff treble c4 q) (staff bass c3 q)))


(defun the-usual-suspects (object)
  (format nil "~A~A~A~A~A" 
	  (identify-dxy object) 
	  (identify-marks object) 
	  (check-for-cmn-store-tag object)
	  (identify-matrix object)
	  (identify-visible object)))


;;;
;;; ----------------    clefs
;;;

;;; clefs are french-violin treble soprano mezzo-soprano alto tenor baritone baritone-C baritone-F bass sub-bass percussion

(defclass clef-mixin (staff-relative-mixin score-object-mixin glyph-mixin breathing-space-mixin) 
  ((position :initarg :position :initform nil :reader clef-position)
   (base-pitch :initarg :base-pitch :initform nil :reader clef-base-pitch)
   (base-line-note :initarg :base-line-note :initform nil :reader clef-base-line-note)
   (base-space-note :initarg :base-space-note :initform nil :reader clef-base-space-note)
   (top-space-note :initarg :top-space-note :initform nil :reader clef-top-space-note)
   (sharp-offset :initarg :sharp-offset :initform nil :reader clef-sharp-offset)
   (flat-offset :initarg :flat-offset :initform nil :reader clef-flat-offset)
   (letter :initarg :letter :initform nil :reader clef-letter)
   (glyph :initarg :glyph :initform nil :reader clef-glyph)
   (name :initarg :name :initform nil :reader clef-name)
   (breathing-space :initform .04)))

(defclass clef (clef-mixin score-object glyph breathing-space) 
  ((position :accessor clef-position)
   (base-pitch :accessor clef-base-pitch)
   (base-line-note :accessor clef-base-line-note)
   (base-space-note :accessor clef-base-space-note)
   (top-space-note :accessor clef-top-space-note)
   (sharp-offset :accessor clef-sharp-offset)
   (flat-offset :accessor clef-flat-offset)
   (letter :accessor clef-letter)
   (glyph :accessor clef-glyph)
   (name :accessor clef-name)
   (breathing-space :initform .04)))

(defclass write-protected-clef (write-protect clef-mixin) ())

(defmethod clef-p ((obj t)) nil)
(defmethod clef-p ((obj clef-mixin)) (not (key-p obj)))

(defmethod descry ((clef clef-mixin) &optional stream controller)
  (format stream "~A :position ~D :base-pitch ~D :base-line-note ~D :base-space-note ~D :top-space-note ~D :sharp-offset ~D~%~A~
                  :flat-offset ~D :letter :~(~A~) :name :~(~A~) :glyph ~A~A~A"
	  (if (not controller) "(clef" "")
	  (clef-position clef)
	  (clef-base-pitch clef)
	  (clef-base-line-note clef)
	  (clef-base-space-note clef)
	  (clef-top-space-note clef)
	  (clef-sharp-offset clef)
	  prewhitespace
	  (clef-flat-offset clef)
	  (clef-letter clef)
	  (clef-name clef)
	  (clef-glyph clef)
	  (if (next-method-p) (call-next-method clef stream (or controller clef)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((clef clef-mixin) &optional object)
  (let ((new-clef (if (not object) (make-clef)
		    (if (write-protected object) (copy object)
		      object))))
    (setf (clef-position new-clef) (clef-position clef))
    (setf (clef-base-pitch new-clef)(clef-base-pitch clef))
    (setf (clef-base-line-note new-clef)(clef-base-line-note clef))
    (setf (clef-base-space-note new-clef)(clef-base-space-note clef))
    (setf (clef-top-space-note new-clef)(clef-top-space-note clef))
    (setf (clef-sharp-offset new-clef)(clef-sharp-offset clef))
    (setf (clef-flat-offset new-clef)(clef-flat-offset clef))
    (setf (clef-letter new-clef)(clef-letter clef))
    (setf (clef-glyph new-clef) (clef-glyph clef))
    (setf (clef-name new-clef) (clef-name clef))
    (if (next-method-p) (call-next-method clef new-clef))
    (if (marks new-clef) (setf (marks new-clef) nil))
    new-clef))

(defmethod dcopy ((clef clef-mixin) key)
  (setf (clef-position key) (clef-position clef))
  (setf (clef-base-pitch key)(clef-base-pitch clef))
  (setf (clef-base-line-note key)(clef-base-line-note clef))
  (setf (clef-base-space-note key)(clef-base-space-note clef))
  (setf (clef-top-space-note key)(clef-top-space-note clef))
  (setf (clef-sharp-offset key)(clef-sharp-offset clef))
  (setf (clef-flat-offset key)(clef-flat-offset clef))
  (setf (clef-letter key)(clef-letter clef))
  (setf (clef-glyph key) (clef-glyph clef))
  (setf (clef-name key) (clef-name clef))
  key)

(defun clef (&rest objects)
  (apply #'ur-clef (make-clef) objects))

(defun ur-clef (new-clef &rest objects)
  (loop for object in objects do
    (when object
      (if (clef-p object)
	  (copy object new-clef)
	(if (self-acting-p object)
	    (funcall (action object) new-clef (arguments object))
	  (if (or (sundry-p object) (text-p object) (glyph-list-p object) (dynamics-p object))
	      (push object (marks new-clef))
	    (cmn-warn "odd argument to clef: ~A" object))))))
  new-clef)

(defun %clef (old-clef &key x0 center staff-y0 justification dx dy)
  (let ((new-clef (make-clef)))
    (copy old-clef new-clef)
    (setf (box-x0 new-clef) x0)
    (setf (box-x1 new-clef) (+ x0 (- (box-x1 old-clef) (box-x0 old-clef))))
    (setf (center new-clef) center)
    (setf (%staff-y0 new-clef) staff-y0)
    (setf (visible-justification new-clef) justification)
    (setf (dxy-dx new-clef) dx)
    (setf (dxy-dy new-clef) dy)
    new-clef))

(defmethod notify ((clef clef) &optional objects)
  (apply #'ur-clef clef objects))

(defmacro define-clef (name base-pitch base-line-note position base-space-note top-space-note sharp-offset flat-offset letter cname clef-glyph)
  `(progn
     (defvar ,name (make-instance 'write-protected-clef 
		       :position ,position
		       :base-pitch ,base-pitch
		       :base-line-note ,base-line-note
		       :base-space-note ,base-space-note
		       :top-space-note ,top-space-note
		       :sharp-offset ,sharp-offset
		       :flat-offset ,flat-offset
		       :letter ,letter
		       :name ,cname
		       :glyph ,clef-glyph
		       :index (index ,clef-glyph) 
		       :rx (g-rx ,clef-glyph)
		       :x0 (box-x0 ,clef-glyph)
		       :x1 (box-x1 ,clef-glyph)
		       :y0 (box-y0 ,clef-glyph)
		       :y1 (box-y1 ,clef-glyph)))
     (defun ,name (&rest args) 
       (apply #'ur-clef (make-clef
			 :position ,position
			 :base-pitch ,base-pitch
			 :base-line-note ,base-line-note
			 :base-space-note ,base-space-note
			 :top-space-note ,top-space-note
			 :sharp-offset ,sharp-offset
			 :flat-offset ,flat-offset
			 :letter ,letter
			 :name ,cname
			 :glyph ,clef-glyph
			 :index (index ,clef-glyph) 
			 :rx (g-rx ,clef-glyph)
			 :x0 (box-x0 ,clef-glyph)
			 :x1 (box-x1 ,clef-glyph)
			 :y0 (box-y0 ,clef-glyph)
			 :y1 (box-y1 ,clef-glyph))
	      args))))

(define-clef sub-bass 28 2 .25 1 4 8 4 :F-clef :sub-bass %bassclef)
(define-clef bass 31 4 0 3 6 6 2 :F-clef :bass %bassclef)		
(define-clef double-bass 19 4 0 3 6 6 2 :F-clef :double-bass %bass8clef)		
(define-clef baritone-F 35 6 -.25 5 1 4 8 :F-clef :baritone-F %bassclef)  
(define-clef baritone-C 35 6 .5 5 1 4 7 :C-clef :baritone %cclef)    
(define-clef baritone 35 6 .5 5 1 4 7 :C-clef :baritone %cclef)      
(define-clef tenor 38 1 .25 0 3 2 5 :C-clef :tenor %cclef)	
(define-clef alto 41 3 0 2 5 7 3 :C-clef :alto %cclef)		
(define-clef mezzo-soprano 45 5 -.25 4 0 5 1 :C-clef :mezzo-soprano %cclef)
(define-clef soprano 48 0 -.5 6 2 3 6 :C-clef :soprano %cclef)	 
(define-clef treble 52 2 0 1 4 8 4 :G-clef :treble %trebleclef)	         
(define-clef tenor-treble 40 2 0 1 4 8 4 :G-clef :tenor-treble %treble8clef)	         
(define-clef french-violin 55 4 -.25 3 6 2 6 :G-clef :french-violin %trebleclef)
(define-clef percussion 52 2 .25 1 4 8 4 :no-clef :percussion %percclef1)

#+Petrucci (defun Petrucci-clef-offset (clf)
		    (if (eq (clef-glyph clf) %trebleclef) .25
		      (if (eq (clef-glyph clf) %bassclef) .75
			(if (eq (clef-glyph clf) %cclef) .5
			  0))))

(defmethod display ((clef clef-mixin) container score &rest rest)
  (declare (ignore container))
  (let ((y0 (+ (%staff-y0 clef) 
	       #+Petrucci (Petrucci-clef-offset clef)
	       (dxy-dy clef)))
	(x0 (+ (box-x0 clef) (dxy-dx clef))))
    (comment score (format nil "~A" (clef-name clef)))
    (if (marks clef) (apply #'display-marks clef score rest))
    #+Sonata (when (and (member (clef-name clef) '(:tenor-treble :double-bass))
			(not (invisible-matrix-p clef)))
	       (let ((xscl (if (not (matrix clef)) 1.0 (first (matrix clef))))
		     (yscl (if (not (matrix clef)) 1.0 (fourth (matrix clef))))
		     (xoff (if (eq (clef-name clef) :tenor-treble) .25 .125))
		     (yoff (if (eq (clef-name clef) :tenor-treble) -.8 -.2)))
		 (moveto score (+ x0 (breathing-space clef) (* xscl xoff)) (+ y0 (clef-position clef) (* yscl yoff)))
		 (show score (text "8" (font-name "Times-Roman") (font-scaler (* xscl .5))))))
    (moveto score (+ x0 (breathing-space clef)) (+ y0 (clef-position clef)))
    (show score clef)))

(defparameter clef-walls '(.05 .05))
(defparameter clef-fences '(.05 .05))

(defmethod house ((clef clef-mixin) score)
  (declare (ignore score))
  (setf (center clef) 0)
  (if (not (invisible-matrix-p clef))
      (progn
	(if (not (walls clef)) (setf (walls clef) clef-walls))
	(if (not (fences clef)) (setf (fences clef) clef-fences)))
    (progn
      (setf (box-x0 clef) 0)
      (setf (box-x1 clef) 0)
      (setf (rx clef) 0)))
  clef)

;;; (cmn staff (treble (scale 0 0)) (ef-major (scale 0 0)) (meter 3 4) c4 q c4 h c4 h c4 q unmetered c4 w c4 w c4 w)

(defmethod identify ((clef clef-mixin))
  (format nil "(~(~A~)~A)" 
	  (clef-name clef) 
	  (the-usual-suspects clef)))


;;;
;;; ----------------    accidentals
;;;

(defclass accidental-mixin (score-object-mixin glyph-mixin) 
  ())

(defclass accidental (accidental-mixin score-object glyph) 
  ())

(defclass write-protected-accidental (write-protect accidental-mixin) ())

(defmethod accidental-p ((obj t)) nil)
(defmethod accidental-p ((obj accidental-mixin)) t)

(defmethod descry ((accidental accidental-mixin) &optional stream controller)
  (format stream "~A~A~A"
	  (if (not controller) "(accidental" "")
	  (if (next-method-p) (call-next-method accidental stream (or controller accidental)) "")
	  (if (not controller) ")" "")))

(defun %sign-name (sign)
  (if (= sign (index %sharp)) 'sharp
    (if (= sign (index %flat)) 'flat
      (if (= sign (index %natural)) 'natural
	(if (= sign (index %dblsharp)) 'double-sharp
	  (if (= sign (index %dblflat)) 'double-flat
	    (if (= sign (index %smallsharp)) 'small-sharp
	      (if (= sign (index %smallflat)) 'small-flat
		(if (= sign (index %smallnatural)) 'small-natural)))))))))
		  
(defmethod identify ((accidental accidental-mixin))
  (format nil "(sign ~(~A~)~A)" (%sign-name (index accidental)) (the-usual-suspects accidental)))

(defmethod copy ((accidental accidental-mixin) &optional object)
  (let ((new-accidental (if (not object) (make-accidental)
			  (if (write-protected object) (copy object)
			    object))))
    (if (next-method-p) (call-next-method accidental new-accidental))
    new-accidental))

(defun accidental (&rest objects)
  (apply #'ur-accidental (make-accidental) objects))

(defun ur-accidental (new-accidental &rest objects)
  (loop for object in objects do
    (when object
      (if (self-acting-p object)
	  (funcall (action object) new-accidental (arguments object))
	(if (or (sundry-p object) (text-p object) (glyph-list-p object) (dynamics-p object))
	    (push object (marks new-accidental))
	  (cmn-warn "odd argument to accidental: ~A" object)))))
  new-accidental)

(defmethod notify ((accidental accidental) &optional objects)
  (apply #'ur-accidental accidental objects))

(defmacro define-accidental (name accidental-glyph)
  `(progn
     (defvar ,name 
	 (make-instance 'write-protected-accidental 
	     :index (index ,accidental-glyph) 
	     :x0 (box-x0 ,accidental-glyph)
	     :x1 (box-x1 ,accidental-glyph)
	     :y0 (box-y0 ,accidental-glyph)
	     :y1 (box-y1 ,accidental-glyph)
	     :rx (g-rx ,accidental-glyph)))
     (defun ,name (&rest objects) 
       (apply #'ur-accidental (make-accidental 
			       :index (index ,accidental-glyph) 
			       :x0 (box-x0 ,accidental-glyph)
			       :x1 (box-x1 ,accidental-glyph)
			       :y0 (box-y0 ,accidental-glyph)
			       :y1 (box-y1 ,accidental-glyph)
			       :rx (g-rx ,accidental-glyph))
	      objects))))


(define-accidental sharp %sharp)
(define-accidental small-sharp %smallsharp)
(define-accidental flat %flat)
(define-accidental small-flat %smallflat)
(define-accidental natural %natural)
(define-accidental small-natural %smallnatural)
(define-accidental double-sharp %dblsharp)
(define-accidental double-flat %dblflat)

(defun sign-name (sign)
  (if sign
      (if (eq sign sharp) 'sharp
	(if (eq sign flat) 'flat
	  (if (eq sign natural) 'natural
	    (if (eq sign double-sharp) 'double-sharp
	      (if (eq sign double-flat) 'double-flat
		(if (eq sign small-sharp) 'small-sharp
		  (if (eq sign small-flat) 'small-flat
		    (if (eq sign small-natural) 'small-natural
		      sign))))))))))


(defmethod display ((accidental accidental-mixin) container score &rest rest)
  (declare (ignore container))
  (let ((dxy-time (or (and (dxy-dx accidental) (not (zerop (dxy-dx accidental))))
		      (and (dxy-dy accidental) (not (zerop (dxy-dy accidental)))))))
    (if dxy-time
      (rmoveto score (or (dxy-dx accidental) 0) (or (dxy-dy accidental) 0)))
    (if (marks accidental) (apply #'display-marks accidental score rest))
    (show score accidental)
    (if dxy-time
	(rmoveto score (or (- (dxy-dx accidental)) 0) (or (- (dxy-dy accidental)) 0)))))
  

(defparameter accidental-walls '(.025 .025))
(defparameter accidental-fences nil)

(defmethod house ((accidental accidental) score)
  (declare (ignore score))
  (setf (center accidental) (* .5 (box-x1 accidental)))
  (if (not (walls accidental)) (setf (walls accidental) accidental-walls))
  (if (not (fences accidental)) (setf (fences accidental) accidental-fences)))

;;; natural-sharp and natural-flat are found under the sundry class definitions
;;; (loader gets confused otherwise)

(defmethod g-rx ((acc accidental-mixin)) ;needed mainly for invisible accidentals
  (if (identity-matrix-p acc)		 ; since we use show, any other matrix involves gsave, so apparent size is 0
      (rx acc)				 ; -- not g-rx!
    0))




;;;
;;; ----------------    keys
;;;

(defclass key-mixin (clef-mixin score-object-mixin) 
  ((signature :initarg :signature :initform nil :reader signature)))

(self-action signature setf-signature)

(defclass key (key-mixin clef score-object) 
  ((signature :accessor signature)))

(defclass write-protected-key (write-protect key-mixin) ())

(defmethod key-p ((obj t)) nil)
(defmethod key-p ((obj key-mixin)) t)

(defmethod canceled-key-p ((obj t)) nil)
(defmethod canceled-key-p ((obj key-mixin))
  (and (signature obj)
       (third (signature obj))
       (eq (third (signature obj)) :cancel)))

(defmethod descry ((key key-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(key" "")
	  (if (signature key) (format nil " :signature (list ~(~A~) ~D~A)" 
				      (sign-name (first (signature key))) 
				      (second (signature key))
				      (if (third (signature key))
					  (format nil " :~(~A~)" (third (signature key)))
					""))
	    "")
	  (if (next-method-p) (call-next-method key stream (or controller key)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((key key-mixin))
  (format nil "~A(key (signature~A)~A)~A"
	  (if (and (signature key) (eq (third (signature key)) :cancel)) "(cancel " "")
	  (if (signature key) (format nil " (list ~(~A~) ~D)"
				      (sign-name (first (signature key))) 
				      (second (signature key)))
	    " nil")
	  (the-usual-suspects key)
	  (if (and (signature key) (eq (third (signature key)) :cancel)) ")" "")))

(defmethod copy ((key key-mixin) &optional object)
  (let ((new-key (if (not object) (make-key)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (signature new-key) (signature key))
    (if (next-method-p) (call-next-method key new-key))
    new-key))

(defun key (&rest objects)
  (apply #'ur-key (make-key) objects))

(defun ur-key (new-key &rest objects)
  (loop for object in objects do
    (when object
      (if (key-p object)
	  (setf (signature new-key) (signature object))
	(if (listp object)		;might be '(sharp 5) explicitly
	    (setf (signature new-key) object)
	  (if (self-acting-p object)
	      (funcall (action object) new-key (arguments object))
	    (if (or (sundry-p object) (text-p object) (glyph-list-p object) (dynamics-p object))
		(push object (marks new-key))
	      (cmn-warn "odd argument to key: ~A" object)))))))
  new-key)

(defun %key (old-key &key x0 center staff-y0)
  (let ((new-key (make-key)))
    (copy old-key new-key)
    (setf (box-x0 new-key) x0)
    (setf (box-x1 new-key) (+ x0 (- (box-x1 old-key) (box-x0 old-key))))
    (setf (center new-key) center)
    (setf (%staff-y0 new-key) staff-y0)
    new-key))

(defmethod notify ((key key) &optional objects)
  (apply #'ur-key key objects))

(defmacro define-key (name data)
  `(progn
     (defparameter ,name (make-instance 'write-protected-key :signature ,data))
     (defun ,name (&rest objects) 
       (apply #'ur-key (make-key :signature ,data) objects))))

(define-key no-key nil)
(define-key c-major  nil)
(define-key a-minor  nil)
(define-key cs-major (list sharp 7))
(define-key as-minor (list sharp 7))
(define-key df-major (list flat 5))
(define-key bf-minor (list flat 5))
(define-key d-major  (list sharp 2))
(define-key b-minor  (list sharp 2))
(define-key ef-major (list flat 3))
(define-key c-minor  (list flat 3))
(define-key e-major  (list sharp 4))
(define-key cs-minor (list sharp 4))
(define-key f-major  (list flat 1))
(define-key d-minor  (list flat 1))
(define-key fs-major (list sharp 6))
(define-key ds-minor (list sharp 6))
(define-key gf-major (list flat 6))
(define-key ef-minor (list flat 6))
(define-key g-major  (list sharp 1))
(define-key e-minor  (list sharp 1))
(define-key af-major (list flat 4))
(define-key f-minor  (list flat 4))
(define-key a-major  (list sharp 3))
(define-key fs-minor (list sharp 3))
(define-key bf-major (list flat 2))
(define-key g-minor  (list flat 2))
(define-key b-major  (list sharp 5))
(define-key gs-minor (list sharp 5))
(define-key cf-major (list flat 7))
(define-key af-minor (list flat 7))

(defmacro def-key (name &rest accs)
  `(define-key ,name (list :special (list ,@accs))))
;;; (def-key foo fs4 cs4 bf4) -> special case key signature
;;; accidentals are displayed in the order listed. 


(defun cancel (key &rest args)
  (let ((new-key (if (write-protected key)
		     (copy key)
		   key)))
    (setf (signature new-key) (append (signature key) (list :cancel)))
    (apply #'ur-key new-key args)))

(defmethod display ((key key-mixin) container score &rest rest)
  (declare (ignore container))
  (when (and (signature key)
	     (not (invisible-matrix-p key)))
    (let* ((key-sig (signature key))
	   (cancel (and (third key-sig) (eq (third key-sig) :cancel)))
	   (fancy (eq (first key-sig) :special))
	   (x0 (+ (box-x0 key) (dxy-dx key)))
	   (y0 (+ (%staff-y0 key) (dxy-dy key)))
	   (num (if fancy (length (second key-sig)) (second key-sig)))
	   (sharps (eq (first key-sig) sharp))
	   (glf (if cancel 
		    %natural 
		  (if (not fancy)
		      (if sharps 
			  %sharp 
			%flat)
		    (loop for n0 in (second key-sig) collect (note-sign n0)))))
	   (dx (if cancel .2 (if fancy .22 (g-rx glf))))
	   (sls (staff-line-separation score))
	   (y0-offset (if sharps 
			  (clef-sharp-offset key) 
			(clef-flat-offset key)))
	   (jumps (if (not fancy)
		      (if sharps	;the pattern is different if it won't fit on the staff (tenor with sharps for example)
			  (if (> y0-offset 3)
			      '(-3 4 -3 -3 4 -3 0) 
			    '(4 -3 4 -3 4 -3 0))
			(if (< y0-offset 6)
			    '(3 -4 3 -4 3 -4 0)
			  '(-4 3 -4 3 -4 3 0)))
		    (let* ((lines (loop for n0 in (second key-sig) 
				   collect (place-of-note-on-clef n0 treble)))
			   (diffs (loop for n0 in lines by #'cdr and n1 in (cdr lines) by #'cdr 
				   collect (- n1 n0))))
		      (setf y0-offset (- (first lines) (- (clef-sharp-offset treble) (clef-sharp-offset key))))
		      (append diffs (list 0))))))
      ;; the "3" and "6" may be wrong
      (if (marks key) (apply #'display-marks key score rest))
      #-KCL (loop for i from 0 below num and x from x0 by dx do
	      (let ((y (+ y0 (* sls y0-offset))))
		(incf y0-offset (nth i jumps))
		(moveto score x y)
		(if cancel
		    (if (plusp (nth i jumps))
			(decf x .075) 
		      (incf x .075)))
		(simple-show score (if (listp glf) (pop glf) glf))))
      #+KCL (do ((i 0 (1+ i))
		 (x x0 (+ x dx)))
		((>= i num))
	      (let ((y (+ y0 (* sls y0-offset))))
		(incf y0-offset (nth i jumps))
		(moveto score x y)
		(if cancel
		    (if (plusp (nth i jumps))
			(decf x .075) 
		      (incf x .075)))
		(simple-show score (if (listp glf) (pop glf) glf))))
      )))

(defparameter key-walls '(0 .05))
(defparameter key-fences '(.05 .05))

(defmethod house ((key key-mixin) score)
  (declare (ignore score))
  (let* ((sig (and (not (invisible-matrix-p key)) (signature key)))
	 (cancel (and (third sig) (eq (third sig) :cancel)))
	 (special (eq (first sig) :special))
	 (dx (if sig (* (if cancel
			    .2 
			  (if special
			      .22
			    (g-rx (if (eq (first sig) sharp) %sharp %flat))) )
			(if (not special)
			    (second sig)
			  (length (second sig))))
	       0)))
    (setf (box-x1 key) (+ (box-x0 key) dx))
    (setf (center key) 0.0)
    (if (not (walls key)) (setf (walls key) (if sig key-walls)))
    (if (not (fences key)) (setf (fences key) (if sig key-fences))))
  key)




;;;
;;; ----------------    meter
;;;


(defclass meter-mixin (staff-relative-mixin score-object-mixin) 
  ((num :initarg :num :initform nil :reader num)
   (den :initarg :den :initform nil :reader den)
   (name :initarg :name :initform nil :reader meter-name)
   (size :initarg :size :initform nil :reader meter-size)
   (style :initarg :style :initform nil :reader meter-style)
   (beaming :initarg :beaming :initform nil :reader meter-beaming)))

(self-action num setf-num)
(self-action den setf-den)
(self-action meter-size setf-meter-size)
(self-action meter-style setf-meter-style)
(self-action beaming setf-beaming)

(defclass meter (meter-mixin score-object) 
  ((num :accessor num)
   (den :accessor den)
   (name :accessor meter-name)
   (size :accessor meter-size)
   (style :accessor meter-style)
   (beaming :accessor beaming)))

(defclass write-protected-meter (write-protect meter-mixin) ())

(defmethod meter-p ((obj t)) nil)
(defmethod meter-p ((obj meter-mixin)) t)

(defmethod descry ((meter meter-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A~A~A~A"
	  (if (not controller) "(meter" "")
	  (if (or (num meter) (den meter)) (format nil " :num ~A :den ~A" (num meter) (den meter)) "")
	  (if (meter-name meter) (format nil " :name ~A" (meter-name meter)) "")
	  (if (meter-size meter) (format nil " :size ~1,3F" (meter-size meter)) "")
	  (if (meter-style meter) (format nil " :style ~(~A~)" (meter-style meter)) "")
	  (if (meter-beaming meter) (format nil " :beaming ~A" (meter-beaming meter)) "")
	  (if (next-method-p) (call-next-method meter stream (or controller meter)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((meter meter-mixin))
  (format nil "(~A~A~A~A)"
	  (if (meter-name meter) 
	      (format nil "~(~A~)" (meter-name meter))
	    (format nil "meter ~A ~A" (num meter) (den meter)))
	  (if (meter-size meter) (format nil " (meter-size ~1,3F)" (meter-size meter)) "")
	  (if (meter-style meter) (format nil " (meter-style :~(~A~))" (meter-style meter)) "")
	  (if (meter-beaming meter) 
	      (if (listp (meter-beaming meter)) 
		  (format nil " (beaming '~A)" (meter-beaming meter))
		(format nil " (beaming ~A)" (meter-beaming meter)))
	    "")
	  (the-usual-suspects meter)))

(defmethod copy ((meter meter-mixin) &optional object)
  (let ((new-meter (if (not object) (make-meter)
		     (if (write-protected object) (copy object)
		       object))))
    (setf (num new-meter) (num meter))
    (setf (den new-meter) (den meter))
    (setf (meter-name new-meter) (meter-name meter))
    (setf (meter-size new-meter) (meter-size meter))
    (setf (meter-style new-meter) (meter-style meter))
    (setf (beaming new-meter) (if (listp (meter-beaming meter)) (copy-list (meter-beaming meter)) (meter-beaming meter)))
    (if (next-method-p) (call-next-method meter new-meter))
    new-meter))

(defun meter (&rest objects)
  (apply #'ur-meter (make-meter) objects))

(defun ur-meter (new-meter &rest objects)
  (loop for object in objects do
    (when object
      (if (meter-p object)
	  (copy object new-meter)
	(if (self-acting-p object)
	    (funcall (action object) new-meter (arguments object))
	  (if (or (sundry-p object) (pause-p object) (dynamics-p object) (text-p object) (glyph-list-p object))
	      (add-to-marks new-meter (list (if (write-protected object) (copy object) object)))
	    (if (listp object)
		(progn
		  (setf (num new-meter) (first object))
		  (setf (den new-meter) (second object)))
	      (if (not (num new-meter))
		  (setf (num new-meter) object)
		(if (not (den new-meter))
		    (setf (den new-meter) object)
		  (cmn-warn "odd argument to meter: ~A" object)))))))))
  new-meter)

(defmethod notify ((meter meter) &optional objects)
  (apply #'ur-meter meter objects))

(defmacro define-meter (name num den meter-name)
  `(progn
     (defvar ,name (make-instance 'write-protected-meter :num ,num :den ,den :name ,meter-name))
     (defun ,name (&rest objects) 
       (apply #'ur-meter (make-meter :num ,num :den ,den :name ,meter-name) objects))))

(define-meter alla-breve 2 2 :alla-breve) 
(define-meter common-time 4 4 :common-time)
(define-meter cut-time 2 2 :cut-time)

(defvar unmetered (make-instance 'write-protected-meter :num 0 :den 4 :name :unmetered :matrix (list 0 0 0 0 0 0) :justification :none))
(defun unmetered () (make-instance 'meter :num 0 :den 4 :name :unmetered :matrix (list 0 0 0 0 0 0) :justification :none))
(defvar suppressed-denominator
    (make-self-acting :action #'(lambda (nm arg) 
				  (declare (ignore arg)) 
				  (setf (meter-style nm) :suppressed)) 
		      :arguments nil))

;;; (cmn staff treble (meter 3 4 (meter-size 2) suppressed-denominator) c4 q)
;;; (cmn staff treble (meter 3 4 note-head-denominator) c4 q)

(defvar note-head-denominator
    (make-self-acting :action #'(lambda (nm arg) 
				  (declare (ignore arg)) 
				  (setf (meter-style nm) :note-head))
		      :arguments nil))

(defmethod display ((meter meter-mixin) container score &rest rest)
  (when container
    (setf (box-x0 meter) (box-x0 container))
    (setf (staff-y0 meter) (staff-y0 container)))
  (when (and (or (not (member :just-fooling rest))
		 (not (eq (visible-justification meter) :none)))
	     (not (invisible-matrix-p meter)))
    (let* ((parens (find-if #'(lambda (n) (and (sundry-p n) (eq :in-parentheses (sundry-name n)))) (marks meter)))
	   (x0 (+ (box-x0 meter) (dxy-dx meter) (if parens .1 0)))
	   (y0 (+ (%staff-y0 meter) (dxy-dy meter))))
      (if (marks meter) (apply #'display-marks meter score rest))
      (when (or (not (meter-size meter))
		(not (zerop (meter-size meter))))
	(if (or (meter-name meter) (eq (meter-style meter) :suppressed))
	    (progn
	      (comment score (format nil "~A" (meter-name meter)))
	      (moveto score x0 (+ y0 (- (half-staff-dy score) .025)))
	      (if (meter-name meter)
		  (show score (if (eq (meter-name meter) :common-time) %commontime %cuttime))
		(let ((num-text (format nil "~S" (num meter))))
		  (if (or (not (meter-size meter))
			  (<= (meter-size meter) 1.0))
		      (show score (%%text :letters num-text))
		    (show score (%%text :letters num-text :font-name Music-Font :font-scaler (meter-size meter)))))))
	  (let* ((num-text (format nil "~S" (num meter)))
		 (den-text (format nil "~S" (den meter)))
		 (num-len (length num-text))
		 (den-len (length den-text))
		 (num-offset (if (>= num-len den-len) 
				 (if (and (numberp (num meter))
					  (= (num meter) 6))
				     -.0125 0)
			       (* .15 (- den-len num-len))))
		 (den-offset (if (>= den-len num-len)
				 (if (and (numberp (den meter))
					  (= (den meter) 8))
				     .0125 0)
			       (* .15 (- num-len den-len)))))
	    (if (or (not (meter-size meter))
		    (<= (meter-size meter) 1.0))
		(progn
		  (moveto score (+ x0 num-offset) (+ y0 .75))
		  (show score (%%text :letters num-text))
		  (if (not (eq (meter-style meter) :note-head))
		      (progn
			(moveto score 
				(+ x0 den-offset) 
				#+Sonata (+ y0 (- .25 .025)) #+Petrucci (+ y0 .25))
			(show score (%%text :letters den-text)))
		    (let ((ob (quarters-to-text (/ 4 (den meter)) t))) ; 2nd arg -> stem-down
		      (moveto score x0 (+ y0 .25))
		      (show score (text ob (font-name Music-Font))))))
	      (if (plusp (meter-size meter))
		  (let ((yup (* (meter-size meter) #+Sonata .215 #+Petrucci .25))
			(ydown (* (meter-size meter) #+Sonata .215 #+Petrucci .23))
			(added-num-offset (* num-offset (meter-size meter)))
			(added-den-offset (* den-offset (meter-size meter))))
		    (moveto score (+ x0 added-num-offset) (+ y0 .5 yup))
		    (show score (%%text :letters num-text :font-name Music-Font :font-scaler (meter-size meter)))
		    (moveto score (+ x0 added-den-offset) (- (+ y0 .5 -.025) ydown))
		    (show score (%%text :letters den-text :font-name Music-Font :font-scaler (meter-size meter))))))))))))

(defun text-dx (txt)
  (loop for i from 0 below (length txt) sum (g-rx (character-to-glyph (elt txt i)))))

(defparameter meter-walls '(.05 .05))
(defparameter meter-fences '(.1 .1))
(defparameter meter-expanders '(0 2))

(defmethod house ((meter meter) score)
  (declare (ignore score))
  (if (not (invisible-matrix-p meter))
      (let* ((num-text (format nil "~S" (num meter)))
	     (den-text (format nil "~S" (den meter)))
	     (dx-num (text-dx num-text))
	     (dx-den (text-dx den-text))
	     (parens (find-if #'(lambda (n) (and (sundry-p n) (eq :in-parentheses (sundry-name n)))) (marks meter))))
	(setf (box-x1 meter) (+ (* (max dx-num dx-den) (or (meter-size meter) 1.0)) (if parens .1 0)))
	(setf (center meter) 0)
	(if (not (walls meter)) (setf (walls meter) meter-walls))
	(if (not (fences meter)) (setf (fences meter) meter-fences))
	(if (not (expanders meter)) (setf (expanders meter) meter-expanders)))
    (progn
      (setf (box-x1 meter) 0)
      (setf (center meter) 0))))

(defun algol+ (num)
  (let* ((nums (format nil "~S" num))
	 (plus (position #\+ nums)))
    (if (not plus)
	(read-from-string nums)
      (+ (read-from-string (subseq nums 0 plus)) 
	 (read-from-string (subseq nums (1+ plus)))))))

(defun beats-per-measure (meter)
  (if (numberp (num meter))
      (num meter)
    (algol+ (num meter))))

(defun beat-duration (meter)
  (/ 4.0 (den meter)))




;;;
;;; ----------------    text
;;;
;;; see cmn0.lisp for class declarations and so on

(defmethod display ((text text) container score &rest rest)
  (if (and (or (not (member :just-fooling rest))
	       (not (member (visible-justification text) '(:none :none-left :none-right :none-center))))
	   (not (invisible-matrix-p text)))
      (let* ((audp container)		; (audible-p container) -- can't remember why I did this
	     (just (visible-justification text))
	     (letter-size (if (font-size text) 
			      (/ (font-size text) (if (member :just-fooling rest)
						      *old-cmn-score-size*
						    *cmn-score-size*))
			    (font-scaler text)))
	     (letter-length (* .4 letter-size (length (letters text))))
	     (x-off (if (text-x text)
			(apply (text-x text) text container score rest)
		      (+ (box-x0 text) 
			 (dxy-dx text) 
			 (if audp 
			     (if (or (not just) (member just '(:left :none :none-left)))
				 (box-x0 container) 
			       (if (member just '(:right :none-right))
				   (- (box-x0 container) letter-length)
				 (if (member just '(:center :none-center))
				     (- (box-x0 container) (* .5 letter-length))
				   (cmn-error "unknown justification: ~A" just))))
			   0))))
	     (y-off (if (text-y text)
			(apply (text-y text) text container score rest)
		      (+ (box-y0 text) (dxy-dy text) (if audp (box-y0 container) 0)))))
	(if (marks text) (apply #'display-marks text score rest))
	(when (not (font-name text)) (setf (font-name text) "Times-Roman"))
	(moveto score x-off y-off)
	(if (and audp (member :just-fooling rest))
	    (moveto score (+ x-off 
			     (if (or (not just) (member just '(:left :none :none-left)))
				 letter-length
			       (if (member just '(:right :none-right))
				   0.0
				 (if (member just '(:center :none-center))
				     (* .5 letter-length)))))
		    y-off)
	  (show score text)))))

;;;(self-action font-name setf-font-name)  ; in cmn0
;;;(self-action font-size setf-font-size)
(self-action x setf-x)
(self-action y setf-y)

(defun text (&rest objects)
  (apply #'ur-text (make-text) objects))

(defun %text (&rest objects)
  (apply #'ur-text (make-instance 'text) objects))

(defun %%text (&key letters font-name font-size font-scaler)
  (let ((new-text (make-text)))
    (if letters (setf (letters new-text) letters))
    (if font-name (setf (font-name new-text) font-name))
    (if font-size (setf (font-size new-text) font-size))
    (if font-scaler (setf (font-scaler new-text) font-scaler))
    new-text))

(defun ur-text (new-text &rest objects)
  (loop for act in objects do
    (when act
      (if (self-acting-p act)
	  (funcall (action act) new-text (arguments act))
	(if (stringp act)
	    (setf (letters new-text) act)
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (push act (marks new-text))
	    (cmn-warn "odd argument to text: ~A" act))))))
  new-text)

(defparameter text-walls '(.05 .05))
(defparameter text-fences '(.2 .2))
(defparameter text-expanders '(5 5))

(defmethod identify ((text text))
  (format nil "(text ~S~A~A~A~A)"
	  (letters text)
#|
	  #+excl (if (and (x text) (functionp (x text))) (format nil " (x #'~A)" (x text)) "") #-excl ""
	  #+excl (if (and (y text) (functionp (y text))) (format nil " (y #'~A)" (y text)) "") #-excl ""
          ;; this doesn't quite work, unfortunately
|#
	  (if (font-name text) (format nil " (font-name ~S)" (font-name text)) "")
	  (if (font-size text) (format nil " (font-size ~1,3F)" (font-size text)) "")
	  (if (/= (font-scaler text) 1.0) (format nil " (font-scaler ~1,3F)" (font-scaler text)) "")
	  (the-usual-suspects text)))

(defvar unjustified (make-self-acting 
		     :action #'(lambda (obj &rest rest)
				 (declare (ignore rest))
				 (setf (visible-justification obj) :none)
				 nil)
		     :arguments nil))



;;;
;;; ----------------    pauses -- fermata general-pause (gp, g.p., and grand-pause) caesura breath-mark
;;;
;;; the Sonata font calls caesura "grandpause" and breath-mark "pause"

(defclass pause-mixin (score-object-mixin)
  ((name :initform nil :initarg :name :reader pause-name)
   (mark :initform nil :initarg :mark :reader pause-mark)))

(defclass pause (pause-mixin score-object)
  ((name :accessor pause-name)
   (mark :accessor pause-mark)))

(defclass write-protected-pause (write-protect pause-mixin) ())

(defparameter pause-font "Times-Bold")

(defun pause-1 (&rest objects)
  (apply #'ur-pause (make-pause) objects))

(defun ur-pause (new-pause &rest objects)
  (loop for object in objects do
    (when object
      (if (pause-p object)
	  (copy object new-pause)
	(if (self-acting-p object)
	    (funcall (action object) new-pause (arguments object))
	  (if (or (sundry-p object) (text-p object) (glyph-list-p object) (dynamics-p object))
	      (push object (marks new-pause))
	    (cmn-warn "odd argument to pause: ~A" object))))))
  new-pause)

(defmacro define-pause (pname mrk)
  `(progn
     (defvar ,pname (make-instance 'write-protected-pause :name ',pname :mark ,mrk))
     (defun ,pname (&rest objects) 
       (apply #'ur-pause (make-pause :name ',pname :mark ,mrk) objects))))

(define-pause breath-mark %pause)
(define-pause general-pause (%text "G.P." (font-name pause-font) (font-scaler 0.5)))
(define-pause g.p. (%text "G.P." (font-name pause-font) (font-scaler 0.5)))
(define-pause grand-pause (%text "G.P." (font-name pause-font) (font-scaler 0.5)))
(define-pause fermata %fermata)
(define-pause upside-down-fermata %fermataup)
(define-pause hold %fermata)
(define-pause caesura %grandpause)	;weird!
(define-pause pause %grandpause)

(defmethod pause-p ((obj t)) nil)
(defmethod pause-p ((obj pause-mixin)) t)

(defmethod descry ((pause pause-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A"
	  (if (not controller) "(pause" "")
	  (if (pause-name pause) (format nil " :name '~(~A~)" (pause-name pause)) "")
	  (if (pause-mark pause) (format nil " :mark ~A" (descry (pause-mark pause))) "")
	  (if (next-method-p) (call-next-method pause stream (or controller pause)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((pause pause-mixin))
  (format nil "(~(~A~)~A)" (pause-name pause) (the-usual-suspects pause)))

(defmethod copy ((pause pause-mixin) &optional object)
  (let ((new-pause (if (not object) (make-pause)
		     (if (write-protected object) (copy object)
		       object))))
    (setf (pause-name new-pause) (pause-name pause))
    (setf (pause-mark new-pause) (pause-mark pause))
    (if (next-method-p) (call-next-method pause new-pause))
    new-pause))

(defmethod display ((pause pause-mixin) subject score &rest rest)
  ;; subject can be note chord bar or rest
  ;; pause can be any of those given above
  ;; here we sort out where the damn thing goes relative to the subject
  (when (and (or (not (member :just-fooling rest))
		 (not (eq (visible-justification pause) :none)))
	     (not (invisible-matrix-p pause)))
    (let* ((mark (pause-mark pause))
	   (glf (not (text-p mark)))
	   (ind (and glf (index mark)))
	   (wid (if glf (g-rx mark) (* .4 (length (letters mark)))))
	   (sx0 (+ (box-x0 subject) 
		   (if (not (bar-p subject)) 
		       (if (and (rest-p subject)
				(eq (rest-mark subject) %wholerest)
				(not (member :unmetered (store-data subject))))
			   (* .5 (- (box-x1 subject) (box-x0 subject)))
			 (center subject))
		     0)))
	   (sy0 (box-y0 subject))
	   (sx1 (max sx0 (box-x1 subject)))
	   (x0 (or (and (not (zerop (box-x0 pause))) (box-x0 pause))
		   (if (bar-p subject)
		       (or (and ind 
				(or (= ind (index %fermata)) 
				    (= ind (index %fermataup)))
				(- sx0 (* .5 wid)))
			   (- sx0 wid))
		     (or (and (or (not ind) 
				  (= ind (index %fermata))
				  (= ind (index %fermataup)))
			      (- sx0 (* .5 wid)))
			 sx1))))
	   (y0 (or (and (not (zerop (box-y0 pause))) (box-y0 pause))
		   (and ind 
			(= ind (index %fermataup)) 
			(or (and (bar-p subject) (- (box-y0 subject) .3))
			    (+ (staff-y0 subject) (* (min -2 (- (minimum-line subject) 3)) (staff-line-separation score)))))
		   (+ (max sy0 (if (bar-p subject) 
				   (box-y1 subject) 
				 (+ (staff-y0 subject) 
				    (* (max 8 
					    (+ 2 (if (and (or (note-p subject) (chord-p subject))
							  (stem-is-up subject))
						     3
						   0)
					       (maximum-line subject))) 
				       (staff-line-separation score)))))
		      (if (not ind) .05
			(if (= ind (index %fermata)) .25
			  (if (= ind (index %grandpause)) -.25
			    (if (= ind (index %pause)) .4
			      .25))))))))
      (when (not (member :just-fooling rest))
	(setf (box-x0 pause) x0)
	(setf (box-y0 pause) y0))
      (comment score (format nil "~S" (pause-name pause)))
      (if (marks pause) (apply #'display-marks pause score rest))
      (moveto score (+ x0 (dxy-dx pause)) (+ y0 (dxy-dy pause)))
      (if (identity-matrix-p pause)
	  (show score mark)
	(show score (let ((pm (copy mark)))
		      (setf (matrix pm) (matrix pause))
		      pm))))))

;;; (cmn (staff (mm 60 (gray-scale .5) (scale 2 1)) treble (c4 q (fermata (gray-scale .5) (scale 2 1)))))

(defparameter pause-walls '(.1 .1))
(defparameter pause-fences '(.25 .25))
(defparameter pause-expanders '(8 8))

(defmethod house ((pause pause) score)
  (declare (ignore score))
  (if (not (invisible-matrix-p pause))
      (progn
	(setf (box-x1 pause) (box-x1 (pause-mark pause)))
	(setf (box-y0 pause) (box-y0 (pause-mark pause)))
	(setf (box-y1 pause) (box-y1 (pause-mark pause)))
	(setf (center pause) (* .5 (box-x1 pause)))
	(if (not (walls pause)) (setf (walls pause) pause-walls))
	(if (not (fences pause)) (setf (fences pause) pause-fences))
	(if (not (expanders pause)) (setf (expanders pause) pause-expanders)))
    (setf (box-x1 pause) 0)))




;;;
;;; ----------------    dynamics
;;;

(defclass dynamics-mixin (score-object-mixin)
  ((name :initform nil :initarg :name :reader dynamics-name)
   (mark :initform nil :initarg :mark :reader dynamics-mark)))

(defclass dynamics (dynamics-mixin score-object)
  ((name :accessor dynamics-name)
   (mark :accessor dynamics-mark)))

(defclass write-protected-dynamics (write-protect dynamics-mixin) ())

(defun glyphify-dynamics (n)
  (let* ((len (length n))
	 (glfs (loop for i from 0 below len 
		collect
		 (let ((char (elt n i)))
		   (if (member char '(#\s #\S)) %s
		     (if (member char '(#\f #\F)) %forte
		       (if (member char '(#\r #\R)) #+Sonata %r #+Petrucci (warn "there's no 'r' in the Petrucci font.")
			 (if (member char '(#\z #\Z)) %z
			   (if (member char '(#\p #\P)) %piano
			     (if (member char '(#\m #\M)) %m))))))))))
    (apply #'%glyphs glfs)))

(defun unglyphify-dynamics (n)
  (format nil "~{~A~}" (loop for glf in (data (dynamics-mark n))
			collect (if (eq (index glf) (index %piano)) "p"
				  (if (eq (index glf) (index %s)) "s"
				    (if (eq (index glf) (index %forte)) "f"
				      (if (eq (index glf) (index %z)) "z"
					(if (eq (index glf) (index %r)) "r"
					  (if (eq (index glf) (index %m)) "m")))))))))

(defun dynamics (&rest objects)
  (apply #'ur-dynamics (make-dynamics) objects))

(defun ur-dynamics (new-dynamics &rest objects)
  (loop for act in objects do
    (when act
      (if (dynamics-p act)
	  (copy act new-dynamics)
	(if (self-acting-p act)
	    (funcall (action act) new-dynamics (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act))
	      (push act (marks new-dynamics))
	    (cmn-warn "odd argument to dynamics: ~A" act))))))
  new-dynamics)


(defmacro define-dynamics (name true-name mark) 
  `(progn
     (defvar ,name (make-instance 'write-protected-dynamics :name ,true-name :mark ,mark))
     (defun ,name (&rest objects) 
       (apply #'ur-dynamics (make-dynamics :name ,true-name :mark ,mark) objects))))

(define-dynamics ppp :ppp %ppp)
(define-dynamics pianississimo :ppp %ppp)
(define-dynamics pp :pp %pp)
(define-dynamics pianissimo :pp %pp)
(define-dynamics p :p %piano)
(define-dynamics piano :p %piano)
(define-dynamics mezzopiano :mp %mezzopiano)
(define-dynamics mp :mp %mezzopiano)
(define-dynamics fff :fff %fff)
(define-dynamics fortississimo :fff %fff)
(define-dynamics ff :ff %ff)
(define-dynamics fortissimo :ff %ff)
(define-dynamics f :f %forte)
(define-dynamics forte :f %forte)
(define-dynamics mezzoforte :mf %mezzoforte)
(define-dynamics mf :mf %mezzoforte)
(define-dynamics sf :sf %sforzando)
(define-dynamics sforzato :sf %sforzando)
(define-dynamics sforzando :sf %sforzando)
(define-dynamics forzando :fz %forzando)
(define-dynamics fz :fz %forzando)

#+Sonata (define-dynamics fp :fp (glyphify-dynamics "fp"))
#+Petrucci (define-dynamics fp :fp %fp)

#-Petrucci (define-dynamics rfz :rfz (glyphify-dynamics "rfz"))

#+Sonata (define-dynamics sfp :sfp (glyphify-dynamics "sfp"))
#+Petrucci (define-dynamics sfp :sfp %sfp)

#+Sonata (define-dynamics pppp :pppp (glyphify-dynamics "pppp"))
#+Petrucci (define-dynamics pppp :pppp %pppp)

#+Sonata (define-dynamics ffff :ffff (glyphify-dynamics "ffff"))
#+Petrucci (define-dynamics ffff :ffff %ffff)

#+Sonata (define-dynamics sfz :sfz (glyphify-dynamics "sfz"))
#+Petrucci (define-dynamics sfz :sfz %sfz)

(define-dynamics sff :sff (%glyphs %s %ff))
(define-dynamics spp :spp (%glyphs %s %pp))
(define-dynamics sp :sp (%glyphs %s %piano))
;; these two look bad because of the weird spacing of the "s" (it's set up to work with "f" not "p") 

#-Petrucci (define-dynamics rinforzando :rfz (glyphify-dynamics "rfz"))

;;; others will have to use the dynamic function

(self-action dynamics-mark setf-dynamics-mark)

(defun dynamic (n) 
  (if (dynamics-p n)
      n
    (if (stringp n)
	(make-dynamics :mark (glyphify-dynamics n))
      (if (symbolp n)
	  (make-dynamics :mark (glyphify-dynamics (symbol-name n)))))))

(defmethod dynamics-p ((obj t)) nil)
(defmethod dynamics-p ((obj dynamics-mixin)) t)

(defmethod descry ((dynamics dynamics-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A"
	  (if (not controller) "(dynamics" "")
	  (if (dynamics-name dynamics) (format nil " :name ~A" (dynamics-name dynamics)) "")
	  (if (dynamics-mark dynamics) (format nil " :mark ~A" (descry (dynamics-mark dynamics))) "")
	  (if (next-method-p) (call-next-method dynamics stream (or controller dynamics)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((dynamics dynamics-mixin))
  (format nil "(~(~A~)~A)" 
	  (or (dynamics-name dynamics) (format nil "dynamic ~S" (unglyphify-dynamics dynamics)))
	  (the-usual-suspects dynamics)))

(defmethod copy ((dynamics dynamics-mixin) &optional object)
  (let ((new-dynamics (if (not object) (make-dynamics)
			(if (write-protected object) (copy object)
			  object))))
    (setf (dynamics-name new-dynamics) (dynamics-name dynamics))
    (setf (dynamics-mark new-dynamics) (dynamics-mark dynamics))
    (if (next-method-p) (call-next-method dynamics new-dynamics))
    new-dynamics))

(defparameter dynamics-walls '(.05 .05))
(defparameter dynamics-fences '(.1 .1))
(defparameter dynamics-expanders '(3 3))

(defmethod house ((dynamics dynamics) score)
  (declare (ignore score))
  (if (not (invisible-matrix-p dynamics))
      (progn
	(setf (box-x1 dynamics) (box-x1 (dynamics-mark dynamics)))
	(setf (box-y0 dynamics) (box-y0 (dynamics-mark dynamics)))
	(setf (box-y1 dynamics) (box-y1 (dynamics-mark dynamics)))
	(setf (center dynamics) (* .5 (box-x1 dynamics)))
	(if (not (walls dynamics)) (setf (walls dynamics) dynamics-walls))
	(if (not (fences dynamics)) (setf (fences dynamics) dynamics-fences))
	(if (not (expanders dynamics)) (setf (expanders dynamics) dynamics-expanders)))
    (setf (box-x1 dynamics) 0)))

(defmethod display ((dynamics dynamics) container score &rest rest) ;see note case in cmn2.lisp and sundry case under sundry-mixin
  (declare (ignore container))
  (when (and (or (not (member :just-fooling rest))
		 (not (eq (visible-justification dynamics) :none)))
	     (not (invisible-matrix-p dynamics)))
    (let ((x0 (+ (box-x0 dynamics) (dxy-dx dynamics)))
	  (y0 (+ (box-y0 dynamics) (dxy-dy dynamics))))
      (comment score (format nil "~S" (dynamics-name dynamics)))
      (if (marks dynamics) (apply #'display-marks dynamics score rest))
      (moveto score x0 y0)
      (if (identity-matrix-p dynamics)
	  (if (member (dynamics-name dynamics) '(:sp :spp))
	      (progn
		(rmoveto score 0 -.05)
		(simple-show score %s)
		(rmoveto score -.05 .05)
		(simple-show score (if (eq (dynamics-name dynamics) :sp) %piano %pp)))
	    (show score (dynamics-mark dynamics) :dx -.1))
	(show score (let ((dm (copy (dynamics-mark dynamics))))
		      (setf (matrix dm) (matrix dynamics))
		      dm)
	      :dx -.1)))))

;;; (cmn (staff treble (mf (dy 3.0) (gray-scale .5) (scale 3 1)) (c4 q (ppp (gray-scale .5) (scale 2 1)))))

(defmethod display :before ((dynamics dynamics) (bar bar-mixin) score &rest rest)
  (declare (ignore score rest))
  (if (zerop (box-x0 dynamics)) (setf (box-x0 dynamics) (box-x0 bar)))
  (if (zerop (box-y0 dynamics)) (setf (box-y0 dynamics) (box-y0 bar))))

;;; (cmn staff treble (meter 3 4) (c4 q breath-mark) c4 h p c4 h c4 q unmetered (c4 w (breath-mark (scale 0 0))) (c4 w (p (scale 0 0))) c4 w)



;;;
;;; ----------------    articulation and so on
;;;
;;; staccato accent (wedge >) tenuto marcato (wedge v) staccatissimo (wedge filled)

(defclass sundry-mixin (score-object-mixin)
  ((name :initform nil :initarg :name :reader sundry-name)
   (mark :initform nil :initarg :mark :reader sundry-mark)))

(defclass sundry (sundry-mixin score-object)
  ((name :accessor sundry-name)
   (mark :accessor sundry-mark)))

(defclass write-protected-sundry (write-protect sundry-mixin) ())

(defmethod sundry-p ((obj t)) nil)
(defmethod sundry-p ((obj sundry-mixin)) t)

(defmethod display ((sundry sundry-mixin) note score &rest rest)
  (when (or (not (member :just-fooling rest))
	    (not (eq (visible-justification sundry) :none)))
    (apply (sundry-mark sundry) sundry note score rest)
    (if (marks sundry) (apply #'display-marks sundry score rest))))

(defmethod display ((dynamics dynamics) (mark sundry-mixin) score &rest rest) (declare (ignore score rest)))     
;; why is this disabled??

(defmethod descry ((sundry sundry-mixin) &optional stream controller)
  (format stream "~A~A~A"
	  (if (tag-p sundry) "" (format stream "(~(~A~)"(sundry-name sundry)))
	  (if (next-method-p) (call-next-method sundry stream (or controller sundry)) "")
	  (if (tag-p sundry) "" ")")))

(defmethod identify ((sundry sundry-mixin))
  (if (sundry-name sundry)
      (if (eq (sundry-name sundry) :graphics)
	  (format nil "(graphics (file ~S)~A)" (funcall (sundry-mark sundry) sundry nil nil :file-name) (the-usual-suspects sundry))
	(if (eq (sundry-name sundry) :auxiliary-note)
	    (format nil "(auxiliary-note ~A)" (identify (funcall (sundry-mark sundry) sundry nil nil :note)))
	  (if (eq (sundry-name sundry) :mm)
	      (let ((data (funcall (sundry-mark sundry) sundry nil nil :metronome)))
		(format nil "(mm ~D~A)" 
			(second data) 
			(if (/= (quarters (first data)) 1) 
			    (format nil " ~A" (rhythm-name (quarters (first data))))
			  "")))
	    (if (not (member (sundry-name sundry) 
			     '(:slur :set-up-slur :beat :beat-subdivision :title :staff-name :beam-between-staves :stem-tie)))
		(format nil "(~(~A~)~A)" (sundry-name sundry) (the-usual-suspects sundry))
	      ""))))
    ""))

(defmethod copy ((sundry sundry-mixin) &optional object)
  (let ((new-sundry (if (not object) (make-sundry)
		      (if (write-protected object) (copy object)
			object))))
    (setf (sundry-name new-sundry) (sundry-name sundry))
    (setf (sundry-mark new-sundry) (sundry-mark sundry))
    (if (next-method-p) (call-next-method sundry new-sundry))
    new-sundry))

(defgeneric backpatch (mark) )
(defgeneric backpatch-time (mark obj) )
(defmethod backpatch (anything) (declare (ignore anything)) nil)
(defmethod backpatch ((sundry sundry-mixin)) nil)
(defmethod backpatch-time ((sundry sundry-mixin) obj) (declare (ignore obj)) nil)


(defun direction-from-note (mark our-note) 
  (if (member (visible-justification mark) '(:up :above :down :below))
      (if (member (visible-justification mark) '(:up :above)) :up :down)
    (if (stem-is-up? our-note) :down :up)))

(defun mark (func name &rest objects)
  (let ((new-mark (make-sundry :name name :mark func)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-mark (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (if (write-protected act)
		  (push (copy act) (marks new-mark))
		(push act (marks new-mark)))))))
    new-mark))

(defmethod notify ((sundry sundry-mixin) &optional objects)
  (let ((new-mark (copy sundry)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-mark (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (push act (marks new-mark))))))
    new-mark))



(defun display-natural-sharp (mark note score &rest rest)
  (declare (ignore note rest))
  (rmoveto score (+ (dx mark) -.2) (dy mark))
  (simple-show score %natural)
  (rmoveto score (- .025 (dx mark)) (- (dy mark))))

(defun natural-sharp (&rest objects) 
  (sharp (apply #'mark #'display-natural-sharp :natural-sharp objects)))

(defvar natural-sharp (make-instance 'write-protected-accidental 
			  :index (index sharp) :x0 (x0 sharp) :x1 (x1 sharp) :y0 (y0 sharp) :y1 (y1 sharp) :rx (g-rx sharp) 
			  :marks (list (make-instance 'write-protected-sundry :mark #'display-natural-sharp :name :natural-sharp))))

;;; (cmn staff treble (g4 q (natural-sharp (dx -.1))))

(defun display-natural-flat (mark note score &rest rest) 
  (declare (ignore note rest))
  (rmoveto score (+ (dx mark) -.225) (dy mark))
  (simple-show score %natural)
  (rmoveto score (- .05 (dx mark)) (- (dy mark))))

(defun natural-flat (&rest objects) 
  (flat (apply #'mark #'display-natural-flat :natural-flat objects)))

(defvar natural-flat (make-instance 'write-protected-accidental 
			 :index (index flat) :x0 (x0 flat) :x1 (x1 flat) :y0 (y0 flat) :y1 (y1 flat) :rx (g-rx flat) 
			 :marks (list (make-instance 'write-protected-sundry :mark #'display-natural-flat :name :natural-flat))))

;;; is natural-natural actually ever used?  what about natural-natural-flat? natural-double-flat?


(defun title-space (title)
  (if title 
      (+ 1.0 (if (text-p title) (+ (font-scaler title) (dxy-dy title)) 1.0))
    0.0))

(defun make-title (title)
  (make-sundry 
   :name :title
   :mark #'(lambda (mark score ignored &rest rest)
	     (declare (ignore mark ignored))
	     (when (not (member :just-fooling rest))
	       (let* ((txt (if (text-p title) title
			     (make-text
			      :letters title 
			      :font-name "Times-Bold" 
			      :font-scaler 1.0)))
		      (x0 (box-x0 score))
		      (x1 (box-x1 score))
		      (y0 (let ((first-staff (first (staff-data (first (staves (first (systems score))))))))
			    (if (not (staff-p first-staff))
				(setf first-staff (first (staves (first (systems score))))))
			    (+ (or (staff-size first-staff) 1.0)
			       1.0
			       (box-y0 first-staff))))
		      (cx (* .5 (+ x1 x0)))
		      (ltx (- cx (* .2 (length (letters txt))))))
		 (moveto score (+ ltx (dxy-dx txt)) (+ y0 (dxy-dy txt)))
		 (show score txt))))))
  
(defun make-staff-name ()
  (make-sundry 
   :name :staff-name 
   :mark #'(lambda (mark staff score &rest rest)
	     (declare (ignore rest))
	     (when (staff-name staff)
	       (let* ((nam (staff-name staff))
		      (txt (text-p nam)))
		 (when txt
		   (if (not (font-name nam)) (setf (font-name nam) (staff-name-font score)))
		   (if (and (= (font-scaler nam) 1.0)
			    (not (font-size nam)))
		       (setf (font-scaler nam) (staff-name-font-scaler score))))
		 (let* ((letters (if txt (letters nam) nam))
			(len (length letters))
			(flat (and (> len 7) (string-equal "-flat" (subseq letters 1 6))))
			(sharp (and (> len 8) (string-equal "-sharp" (subseq letters 1 7))))
			(fname (or (and txt (font-name nam)) (staff-name-font score)))
			(dx-mark (if txt (dxy-dx nam) (dxy-dx mark)))
			(dy-mark (if txt (dxy-dy nam) (dxy-dy mark)))
			(fsize (or (and txt (or (font-size nam)
						(and (font-scaler nam)
						     (floor (* (font-scaler nam) (scr-size score))))))
				   (max (staff-name-font-minimum-size score) 
					(floor (* (size score) (staff-name-font-scaler score))))))
			(fscale (/ fsize (scr-size score))))
		   (if (not (staff-name-x0 staff))
		       (setf (staff-name-x0 staff) -1.0))
		   (if (and (not flat) (not sharp))
		       (let* ((y-up (* -.4 fscale)))
			 (moveto score 
				 (+ (box-x0 staff) (staff-name-x0 staff) dx-mark)
				 (+ (box-y0 staff) dy-mark .5 y-up))
			 (if txt
			     (show score nam)
			   (show score (%%text :letters letters 
					       :font-name fname
					       :font-size fsize))))
		     (let* ((l0 (subseq letters 0 1))
			    (l1 (subseq letters (if flat 6 7)))
			    (y-up (* -.4 fscale)))
		       (moveto score 
			       (+ (box-x0 staff) (staff-name-x0 staff) dx-mark) 
			       (+ (box-y0 staff) dy-mark .5 y-up))
		       (show score (%%text :letters l0 :font-name fname :font-size fsize))
		       (rmoveto score (* (if flat .5 .6) fscale) (* (if flat .125 .4) fscale))
		       (show score (%%text :letters (if flat "b" "#") :font-name Music-Font :font-size (* 1.25 fsize)))
		       (rmoveto score (* .3 fscale) (* (if flat -.125 -.4) fscale))
		       (show score (%%text :letters l1 :font-name fname :font-size fsize))))))))))



(defun display-staccato (mark note score &rest rest)
  (declare (ignore rest))		;Ross says this should be over the stem if not over the notehead -- contradicted by actual scores
  (let* ((hl (head-line note))
	 (dir (direction-from-note mark note))
	 (y-off (+ .25 (if (and (evenp hl) 
				(or (< 1 hl 9)
				    (and (> hl 9) (eq dir :down))
				    (and (< hl 1) (eq dir :up))))
			   .125 0)))
	 (y0 (+ (%staff-y0 note) (* hl (staff-line-separation score))))
	 (new-dot (copy %dot)))
    (setf (matrix new-dot) (matrix mark))
    (moveto score 
	    (+ (box-x0 note) (box-x0 mark) (dxy-dx mark) (center note) -.05)
	    (+ y0 (box-y0 mark) (dxy-dy mark) (if (eq dir :up) y-off (- y-off))))
    (show score new-dot)))

(defvar staccato (make-instance 'write-protected-sundry :name :staccato :mark #'display-staccato))
(defun staccato (&rest objects) (apply #'mark #'display-staccato :staccato objects))
(defun staccato-p (object) (and object (sundry-p object) (eq (sundry-name object) :staccato)))


(defun display-accent (mark note score &rest rest)
  (declare (ignore rest))
  (let ((y-off (+ (box-y0 mark) (dxy-dy mark) (max (+ (or (stem-end note) 0) .125)
						   (+ (%staff-y0 note) 
						      (* (max 10 (+ 3 (head-line note))) 
							 (staff-line-separation score))))))
	(x-off (+ (box-x0 note) (dxy-dx mark) (center note) -.2 (x0 mark)))
	(new-glf (copy %accent)))
    (moveto score x-off y-off)
    (setf (matrix new-glf) (matrix mark))
    (show score new-glf)))

(defvar accent (make-instance 'write-protected-sundry :name :accent :mark #'display-accent))
(defun accent (&rest objects) (apply #'mark #'display-accent :accent objects))


(defun display-little-swell (mark note score &rest rest)
  (let* ((dyn (and (marks mark) (find-if #'dynamics-p (marks mark))))
	 (y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		   (if (member (visible-justification mark) '(:down :below))
		       (min (* (- (head-line note) 2) (staff-line-separation score)) -.5)
		     (* (max 10 (+ 3 (head-line note))) (staff-line-separation score)))))
	 (x-off (+ (box-x0 note) (dxy-dx mark) (center note) -.1 (if dyn -.3 0) (x0 mark)))
	 (new-glf (copy %accent))
	 (mirror-glf (copy %accent)))
    (setf (matrix new-glf) (matrix mark))
    (setf (matrix mirror-glf) (copy-list (matrix mark)))
    (moveto score x-off y-off)
    (show score (mirror mirror-glf))
    (when dyn
      (setf (box-x0 dyn) x-off)
      (setf (box-y0 dyn) y-off)
      (apply #'display dyn nil score rest)
      (if (matrix dyn) 
	  (rmoveto score (* (first (matrix dyn)) .5) 0) 
	(rmoveto score -.05 0)))
    (rmoveto score .1 0)
    (show score new-glf)))

(defvar little-swell (make-instance 'write-protected-sundry :name :little-swell :mark #'display-little-swell))
(defun little-swell (&rest objects) (apply #'mark #'display-little-swell :little-swell objects))


(defun display-wedge (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((hl (head-line note))
	 (dir (direction-from-note mark note))
	 (y-off (+ (box-y0 mark) (dxy-dy mark) 
		   (%staff-y0 note) 
		   (* (+ (if (eq dir :up) 3 -3) 
			 hl 
			 (if (= hl 3) 1 (if (= hl 5) -1 0)))
		      (staff-line-separation score))))
	 (x-off (+ (box-x0 note) (dxy-dx mark) (if (and (eq dir :down) 
							(not (whole-note-p note))) 
						   .05 
						 0) 
		   (center note) (box-x0 mark)))
	 (dy (if (eq dir :up) .25 -.25)))
    (moveto score x-off y-off)
    (matrix-front score (matrix mark))
    (g-begin-filled-polygon score)
    (rlineto score -.075 dy)
    (rlineto score .15 0)
    (fill-in score :closepath t)
    (matrix-back score (matrix mark))))

(defvar wedge (make-instance 'write-protected-sundry :name :wedge :mark #'display-wedge))
(defun wedge (&rest objects) (apply #'mark #'display-wedge :wedge objects))


(defun display-tenuto (mark note score &rest rest)
  (declare (ignore rest))
  ;; tenuto gives way only to staccato
  (let* ((hl (head-line note))
	 (staccato (find-if #'staccato-p (marks note)))
	 (mark-length (+ .3 (if (whole-note-p note) .1 0)))
	 (y-off (+ .25 (if staccato .2 0) (if (and (evenp hl) (< 1 hl 8)) .125 0)))
	 (y0 (+ (%staff-y0 note) (dxy-dy mark) (* hl (staff-line-separation score)))))
    (moveto score 
	    (+ (box-x0 note) (center note) (* -.5 mark-length) (dxy-dx mark) (x0 mark))
	    (+ y0 (box-y0 mark) (if (eq (direction-from-note mark note) :up) y-off (- y-off))))
    (matrix-front score (matrix mark))
    (setf (line-width score) .05)
    (rlineto score mark-length 0)
    (draw score)
    (setf (line-width score) 0)
    (matrix-back score (matrix mark))))

(defvar tenuto (make-instance 'write-protected-sundry :name :tenuto :mark #'display-tenuto))
(defun tenuto (&rest objects) (apply #'mark #'display-tenuto :tenuto objects))
(defun tenuto-p (object) (and object (sundry-p object) (eq (sundry-name object) :tenuto)))


(defun display-down-bow (mark note score &rest rest)
  (when (not (member :just-fooling rest))
    (let* ((y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		     (max (if (and (< 1 (head-line note) 5)
				   (not (whole-note-p note)))
			      (* (+ 9 (head-line note)) (staff-line-separation score))
			    0.0)
			  (* (max 10 (+ 3 (head-line note))) (staff-line-separation score)))))
	   (dy -.15)
	   (x-off (+ (box-x0 note) (dxy-dx mark) (if (not (whole-note-p note)) -.05 -.1) (center note) (box-x0 mark))))
      (moveto score (- x-off .1) (+ y-off dy))
      (lineto score (- x-off .1) y-off)
      (draw score)
      (setf (line-width score) .075)
      (moveto score (- x-off .1) y-off)
      (lineto score (+ x-off .2) y-off)
      (draw score)
      (setf (line-width score) 0)
      (moveto score (+ x-off .2) y-off)
      (lineto score (+ x-off .2) (+ y-off dy))
      (draw score))))

(defvar down-bow (make-instance 'write-protected-sundry :name :down-bow :mark #'display-down-bow))
(defun down-bow (&rest objects) (apply #'mark #'display-down-bow :down-bow objects))

(defun display-up-bow (mark note score &rest rest)
  (when (not (member :just-fooling rest))
    (let* ((y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		     (max (if (and (< 1 (head-line note) 5)
				   (not (whole-note-p note)))
			      (* (+ 8 (head-line note)) (staff-line-separation score))
			    0.0)
			  (* (max 9 (+ 3 (head-line note))) (staff-line-separation score)))))
	   (dy .25)
	   (x-off (+ (box-x0 note) (dxy-dx mark) (if (not (whole-note-p note)) -.05 -.1) (center note) (box-x0 mark))))
      (moveto score x-off y-off)
      (rlineto score -.075 dy)
      (moveto score x-off y-off)
      (rlineto score .075 dy)
      (draw score))))

(defvar up-bow (make-instance 'write-protected-sundry :name :up-bow :mark #'display-up-bow))
(defun up-bow (&rest objects) (apply #'mark #'display-up-bow :up-bow objects))


(defun display-detache (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((hl (head-line note))
	 (mark-length (+ .3 (if (whole-note-p note) .1 0)))
	 (y-off (+ .25 (if (and (evenp hl) (< 1 hl 8)) .125 0)))
	 (y0 (+ (%staff-y0 note) (* hl (staff-line-separation score)))))
    (moveto score 
	    (+ (box-x0 note) (dxy-dx mark) (center note) (* -.5 mark-length) (box-x0 mark))
	    (+ y0 (box-y0 mark) (dxy-dy mark) (if (eq (direction-from-note mark note) :up) y-off (- y-off))))
    (matrix-front score (matrix mark))
    (rlineto score mark-length 0)
    (draw score)
    (matrix-back score (matrix mark))))

(defvar detache (make-instance 'write-protected-sundry :name :detache :mark #'display-detache))
(defun detache (&rest objects) (apply #'mark #'display-detache :detache objects))


(defun display-martele (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((dir (direction-from-note mark note))
	 (y-off (+ (box-y0 mark) (dxy-dy mark) 
		   (%staff-y0 note) 
		   (if (eq dir :up) 
		       (* (max 11 (+ 3 (head-line note))) (staff-line-separation score))
		     (* (min -3 (- (minimum-line note) 3)) (staff-line-separation score)))))
	 (x-off (+ (box-x0 note) (dxy-dx mark) (center note) (box-x0 mark)))
	 (dy (if (eq dir :up) .2 -.2)))
    (moveto score x-off y-off)
    (matrix-front score (matrix mark))
    (rmoveto score -.1 dy)
    (rlineto score .1 (- dy))
    (rlineto score .1 dy)
    (draw score)
    (matrix-back score (matrix mark))))

(defvar martele (make-instance 'write-protected-sundry :name :martele :mark #'display-martele))
(defun martele (&rest objects) (apply #'mark #'display-martele :martele objects))


(defclass thick-sundry-mixin (sundry-mixin thick-mixin) ())
(defclass thick-sundry (thick-sundry-mixin sundry thick) ())
(defclass write-protected-thick-sundry (write-protect thick-sundry-mixin) ())

(defmethod identify ((sundry thick-sundry))
  (format nil "(~(~A~)~A~A)" 
	  (sundry-name sundry) 
	  (if (thickness sundry) (format nil " (thickness ~1,3F)" (thickness sundry)))
	  (the-usual-suspects sundry)))

(defmethod copy ((sundry thick-sundry-mixin) &optional object)
  (let ((new-sundry (if (not object) (make-thick-sundry)
		      (if (write-protected object) (copy object)
			object))))
    (setf (sundry-name new-sundry) (sundry-name sundry))
    (setf (sundry-mark new-sundry) (sundry-mark sundry))
    (setf (thickness new-sundry) (thickness sundry))
    (if (next-method-p) (call-next-method sundry new-sundry))
    new-sundry))

(defun thick-mark (func name &rest objects)
  (let ((new-mark (make-thick-sundry :name name :mark func)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-mark (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (if (write-protected act)
		  (push (copy act) (marks new-mark))
		(push act (marks new-mark)))))))
    new-mark))


(defun display-marcato (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((dir (direction-from-note mark note))
	 (y-off (+ (box-y0 mark) (dxy-dy mark) 
		   (%staff-y0 note) 
		   (if (eq dir :up) 
		       (* (max 11 (+ 3 (head-line note))) (staff-line-separation score))
		     (* (min -3 (- (minimum-line note) 3)) (staff-line-separation score)))))
	 (x-off (+ (box-x0 note) (dxy-dx mark) (center note) (box-x0 mark)))
	 (dy (if (eq dir :up) -.2 .2)))
    (moveto score x-off y-off)
    (matrix-front score (matrix mark))
    (setf (line-width score) (or (thickness mark) .025))
    (rmoveto score -.1 dy)
    (rlineto score .1 (- dy))
    (rlineto score .1 dy)
    (draw score)
    (setf (line-width score) 0)
    (matrix-back score (matrix mark))))

(defvar marcato (make-instance 'write-protected-thick-sundry :name :marcato :mark #'display-marcato))
(defun marcato (&rest objects) (apply #'thick-mark #'display-marcato :marcato objects))


(defun display-bartok-pizzicato (mark note score &optional upsidedown)
  (let* ((y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		   (if upsidedown .1 0)
		   (max (if (and (< 1 (head-line note) 5)
				 (not (whole-note-p note)))
			    (* (+ 8 (head-line note)) (staff-line-separation score))
			  0.0)
			(* (max 10 (+ 3 (head-line note))) (staff-line-separation score)))))
	 (r (/ 6 40))
	 (lx (/ 10 40))
	 (x-off (+ (box-x0 note) -.05 (dxy-dx mark) (center note) (box-x0 mark))))
    (setf (line-width score) (or (thickness mark) .025))
    (circle score x-off y-off r)
    (moveto score x-off y-off)
    (rlineto score 0 (if upsidedown (- lx) lx))
    (draw score)
    (setf (line-width score) 0)))

(defvar bartok-pizzicato (make-instance 'write-protected-thick-sundry :name :bartok-pizzicato :mark #'display-bartok-pizzicato))
(defun bartok-pizzicato (&rest objects) (apply #'thick-mark #'display-bartok-pizzicato :bartok-pizzicato objects))

(defun snap-pizzicato (&rest args) (apply #'bartok-pizzicato args))
(defvar snap-pizzicato bartok-pizzicato)


(defun display-thumb (mark note score &rest rest) 
  (declare (ignore rest))
  (display-bartok-pizzicato mark note score t))

(defvar thumb (make-instance 'write-protected-thick-sundry :name :thumb :mark #'display-thumb))
(defun thumb (&rest objects) (apply #'thick-mark #'display-thumb :thumb objects))


(defgeneric display-in-parentheses (mark note score &rest rest) )

(defmethod display-in-parentheses ((mark sundry-mixin) (note accidental-mixin) score &rest rest)
  (declare (ignore rest))
  (rmoveto score -.2 -.2)
  (simple-show score %parenleft)
  (rmoveto score .25 0)
  (simple-show score %parenright)
  (rmoveto score (- (+ .05 (g-rx %parenleft) (g-rx %parenright))) .2))
  

(defmethod display-in-parentheses ((mark sundry-mixin) (note meter-mixin) score &rest rest)
  (declare (ignore rest))
  (let* ((x0 (- (+ (box-x0 note) (dxy-dx note) .1) (dxy-dx mark) .25))
	 (x1 (+ (box-x1 note) (dxy-dx mark) (dxy-dx note)))
	 (y (+ (dxy-dy mark) (staff-y0 note))))
    (moveto score x0 y)
    (show score %parenleft :size (* 2 (size score)))
    (moveto score x1 y)
    (show score %parenright :size (* 2 (size score)))))

;;; see cmn2.lisp for audible-mixin version of this method

(defmethod display-in-parentheses ((mark sundry-mixin) (note t) score &rest rest)
  (declare (ignore rest))
  (let* ((x0 (- (+ (box-x0 note) (dxy-dx mark)) .25))
	 (x1 (+ (box-x1 note) .1 (dxy-dy mark)))
	 (y (+ (dxy-dy mark) (if (plusp (box-y0 note)) (box-y0 note) (staff-y0 note))))
	 (size (if (plusp (box-y1 note)) (* 2 (box-y1 note) (size score)) (size score))))
    ;; the factor of 2 reflects the height of the paren (.5)
    (moveto score x0 y)
    (show score %parenleft :size size)
    (moveto score x1 y)
    (show score %parenright :size size)))

(defmethod display-in-parentheses ((mark sundry-mixin) (note dynamics-mixin) score &rest rest)
  (declare (ignore rest))
  (let* ((x0 (- (+ (box-x0 note) (dxy-dx note) (dxy-dx mark)) .15))
	 (x1 (+ x0 (if (glyph-list-p (dynamics-mark note))
		       (+ (apply #'+ (loop for g in (data (dynamics-mark note)) collect (g-rx g)))
			  (* -.1 (length (data (dynamics-mark note)))))
		     (+ -.05 (g-rx (dynamics-mark note))))))
	 (y (+ -.125 (dxy-dy mark) (box-y0 note) (dxy-dy note))))
    (moveto score x0 y)
    (simple-show score %parenleft)
    (moveto score x1 y)
    (simple-show score %parenright)))

#|
(cmn staff treble (c4 q (p in-parentheses)) (c4 q (pp in-parentheses)) (c4 q (sp in-parentheses)) 
  (c4 q (spp in-parentheses)) (c4 q (f in-parentheses)) (c4 q (ff in-parentheses)) 
  (c4 q (fp in-parentheses)) (c4 q (rfz in-parentheses)) (c4 q (sff in-parentheses))) 
|#

(defun in-parentheses (&rest objects)
  (let ((new-paren (make-sundry :name :in-parentheses :mark #'display-in-parentheses)))
    (loop for act in objects do
      (if (self-acting-p act)
	  (funcall (action act) new-paren (arguments act))))
    new-paren))

(defvar in-parentheses (make-instance 'write-protected-sundry :name :in-parentheses :mark #'display-in-parentheses))


(defun display-left-hand-pizzicato (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((hl (head-line note))
	 (dir (direction-from-note mark note))
	 (y0 (+ (box-y0 mark) (dxy-dy mark) 
		(%staff-y0 note) 
		(if (eq dir :up) 
		    (* (max 10 (+ 3 hl)) (staff-line-separation score))
		  (* (min -2 (- (minimum-line note) 3)) (staff-line-separation score)))))
	 (new-glf (copy %cross)))
    (setf (matrix new-glf) (matrix mark))
    (moveto score 
	    (+ (box-x0 note) (dxy-dx mark) (box-x0 mark) (center note) -.1)
	    y0)
    (show score new-glf)))

(defvar left-hand-pizzicato (make-instance 'write-protected-sundry :name :left-hand-pizzicato :mark #'display-left-hand-pizzicato))
(defun left-hand-pizzicato (&rest objects) (apply #'mark #'display-left-hand-pizzicato :left-hand-pizzicato objects))


(defun display-natural-harmonic (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((hl (head-line note))
	 (y-off (+ .3 
		   (if (find-if #'tenuto-p (marks note)) .15 0)
		   (if (and (evenp hl) (< 1 hl 9)) .075 0))) ;was .125
	 (y0 (+ (%staff-y0 note) (* hl (staff-line-separation score))))
	 (new-glf (copy %naturalharmonic)))
    (setf (matrix new-glf) (matrix mark))
    (moveto score 
	    (+ (box-x0 note) (box-x0 mark) (dxy-dx mark) (center note) -.05)
	    (+ y0 (box-y0 mark) (dxy-dy mark) (if (eq (direction-from-note mark note) :up) y-off (- y-off))))
    (show score new-glf)))

(defvar natural-harmonic (make-instance 'write-protected-sundry :name :natural-harmonic :mark #'display-natural-harmonic))
(defun natural-harmonic (&rest objects) (apply #'mark #'display-natural-harmonic :natural-harmonic objects))


(defvar stopped-note (make-instance 'write-protected-sundry :name :stopped-note :mark #'display-left-hand-pizzicato))
(defun stopped-note (&rest objects) (apply #'left-hand-pizzicato objects))

(defvar open-note (make-instance 'write-protected-sundry :name :open-note :mark #'display-natural-harmonic))
(defun open-note (&rest objects) (apply #'natural-harmonic objects))
  
#|
;;; test cases
(cmn staff treble 
  a3 q staccato d4 h staccato b4 w staccato e5 q staccato d6 q staccato as3 q staccato df4 q staccato fs5 w staccato
  a3 q accent d4 h accent  b4 w accent e5 q accent  d6 q accent as3 q accent df4 q accent fs5 w accent
  a3 q little-swell d4 h little-swell  b4 w little-swell e5 q little-swell  d6 q little-swell 
    as3 q little-swell df4 q little-swell fs5 w little-swell
  a3 q wedge d4 h wedge  b4 w wedge e5 q wedge  d6 q wedge as3 q wedge df4 q wedge fs5 w wedge
  a3 q tenuto d4 h tenuto  b4 w tenuto e5 q tenuto  d6 q tenuto as3 q tenuto df4 q tenuto fs5 w tenuto
  a3 q marcato d4 h marcato  b4 w marcato e5 q marcato  d6 q marcato as3 q marcato df4 q marcato fs5 w marcato
  a3 q down-bow d4 h down-bow  b4 w down-bow e5 q down-bow  d6 q down-bow as3 q down-bow df4 q down-bow fs5 w down-bow
  a3 q up-bow d4 h up-bow  b4 w up-bow e5 q up-bow  d6 q up-bow as3 q up-bow df4 q up-bow fs5 w up-bow
  a3 q detache d4 h detache  b4 w detache e5 q detache  d6 q detache as3 q detache df4 q detache fs5 w detache
  a3 q martele d4 h martele  b4 w martele e5 q martele  d6 q martele as3 q martele df4 q martele fs5 w martele
  a3 q thumb d4 h thumb  b4 w thumb e5 q thumb  d6 q thumb as3 q thumb df4 q thumb fs5 w thumb
  a3 q natural-harmonic d4 h natural-harmonic  b4 w natural-harmonic e5 q natural-harmonic  d6 q natural-harmonic 
    as3 q natural-harmonic df4 q natural-harmonic fs5 w natural-harmonic
  a3 q bartok-pizzicato d4 h bartok-pizzicato  b4 w bartok-pizzicato e5 q bartok-pizzicato  d6 q bartok-pizzicato 
    as3 q bartok-pizzicato df4 q bartok-pizzicato fs5 w bartok-pizzicato
  a3 q stopped-note d4 h stopped-note  b4 w stopped-note e5 q stopped-note  d6 q stopped-note 
    as3 q stopped-note df4 q stopped-note fs5 w stopped-note
  a3 q open-note d4 h open-note  b4 w open-note e5 q open-note  d6 q open-note as3 q open-note df4 q open-note fs5 w open-note
  a3 q left-hand-pizzicato d4 h left-hand-pizzicato  b4 w left-hand-pizzicato e5 q left-hand-pizzicato  d6 q left-hand-pizzicato 
    as3 q left-hand-pizzicato df4 q left-hand-pizzicato fs5 w left-hand-pizzicato
  )
|#


(defun display-pedal (mark note score &rest rest)
  (declare (ignore rest))
  (let ((y-off (+ (min (- (%staff-y0 note) .75) 
		       (- (box-y0 note) (if (stem-is-down? note) 1.5 .75)))
		  (dxy-dy mark)))
	(x-off (+ (box-x0 note) -.25 (dxy-dx mark) (box-x0 mark)))
	(new-glf (copy %pedal)))
    (setf (matrix new-glf) (matrix mark))
    (moveto score x-off y-off)
    (show score new-glf)))

(defvar pedal (make-instance 'write-protected-sundry :name :pedal :mark #'display-pedal))
(defun pedal (&rest objects) (apply #'mark #'display-pedal :pedal objects))


(defun display-segno (mark bar score &rest rest)
  (declare (ignore rest))
  (let ((y-off (+ (box-y0 bar) (dxy-dy mark) (* 12 (staff-line-separation score))))
	(x-off (+ (box-x0 bar) (dxy-dx mark) (center bar) -.25 (box-x0 mark)))
	(new-glf (copy %segno)))
    (setf (matrix new-glf) (matrix mark))
    (moveto score x-off y-off)
    (show score new-glf)))

(defvar segno (make-instance 'write-protected-sundry :name :segno :mark #'display-segno))
(defun segno (&rest objects) (apply #'mark #'display-segno :segno objects))


(defun display-coda (mark bar score &rest rest)
  (declare (ignore rest))
  (let ((y-off (+ (box-y0 bar) (dxy-dy mark) (* 12 (staff-line-separation score)))) 
	(x-off (+ (box-x0 bar) (dxy-dx mark) (center bar) -.25 (box-x0 mark)))
	(new-glf (copy %coda)))
    (setf (matrix new-glf) (matrix mark))
    (moveto score x-off y-off)
    (show score new-glf)))

(defvar coda (make-instance 'write-protected-sundry :name :coda :mark #'display-coda))
(defun coda (&rest objects) (apply #'mark #'display-coda :coda objects))



(defun quarters-to-text (dur &optional down-stem)
  (if (= dur 1/8) (if down-stem "R" "r")
    (if (= dur 1/4) (if down-stem "X" "x")
      (if (= dur 3/8) (if down-stem "X ." "x .")
	(if (= dur 1/2) (if down-stem "E" "e")
	  (if (= dur 3/4) (if down-stem "E ." "e .")
	    (if (= dur 1) (if down-stem "Q" "q")
	      (if (= dur 3/2) (if down-stem "Q ." "q .")
		(if (= dur 2) (if down-stem "H" "h")
		  (if (= dur 3) (if down-stem "H ." "h .")
		    (if (< dur 6) "w"
		      (if (< dur 8) "w ."
			"W"))))))))))))


(defun change-beat (old-beat new-beat &rest objects)
  (let* ((oq (quarters old-beat))
	 (nq (quarters new-beat))
	 (oqt (quarters-to-text oq))
	 (nqt (quarters-to-text nq))
	 (new-cb (make-sundry 
	      :name :change-beat
	      :mark #'(lambda (mark note score &rest rest)
			(let ((x0 (+ (box-x0 note) (dxy-dx mark) (center note) -.5 
				     (if (> (length oqt) 1) -.25 0) 
				     (if (< oq 1.0) -.25 0)))
			      (y0 (+ (box-y0 note) (dxy-dy mark) (* 10 (staff-line-separation score)))))
			  (when (and (marks mark) 
				     (not (member :just-fooling rest)))
			    (setf (box-x0 mark) x0)
			    (setf (box-x1 mark) (+ x0 
						   (if (char-equal (elt oqt 0) #\e) .15 
						     (if (char-equal (elt oqt 0) #\x) .25 0))
						   (if (> (length oqt) 1) .15 0)
						   1.2 ; " = " in .6 times-roman font
						   (if (char-equal (elt nqt 0) #\e) .15 
						     (if (char-equal (elt nqt 0) #\x) .25 0))
						   (if (> (length nqt) 1) .15 0)))
			    (setf (box-y0 mark) y0))
			  (moveto score x0 y0)
			  (show score (text oqt (font-name Music-Font) (font-scaler .75)) :save :start)
			  (show score (text " = " (font-name "Times-Roman") (font-scaler .6)) :save nil)
			  (show score (text nqt (font-name Music-Font) (font-scaler .75)) :save :finish))))))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-cb (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act))
	      (push act (marks new-cb))))))
    new-cb))


(defun mm (num &rest objects)
  (let ((new-mm (make-sundry :name :mm :mark nil))
	(beat nil))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-mm (arguments act))
	  (if (rhythm-p act)
	      (setf beat act)
	    (if (or (sundry-p act) (text-p act) (glyph-list-p act))
		(push act (marks new-mm)))))))
    (let ((ob (or (and beat (quarters-to-text (quarters beat))) "q")))
      (setf (sundry-mark new-mm)
	#'(lambda (mark note score &rest rest)
	    (if (member :metronome rest) ;for cmn-store's benefit
		(list (or beat (rq 1)) num)
	      (let* ((x0 (+ (dxy-dx mark) (if note (box-x0 note) (box-x0 mark))))
		     (y0 (+ (if note (box-y0 note) (%staff-y0 mark))
			    (dxy-dy mark)
			    (* 12 (staff-line-separation score)))))
		(when (and (marks mark) 
			   (not (member :just-fooling rest)))
		  (setf (box-x0 mark) x0)
		  (setf (box-x1 mark) (+ x0 
					 (if (char-equal (elt ob 0) #\e) .15 
					   (if (char-equal (elt ob 0) #\x) .25 0))
					 (if (> (length ob) 1) .15 0) ;dotted
					 (* .2 (1+ (length (format nil " = ~D" num))))))
		  (setf (box-y0 new-mm) y0))
		(moveto score x0 y0)
		(show score (text ob (font-name Music-Font) (font-scaler .75) (matrix (matrix new-mm))) :save :start)
		(show score (text (format nil " = ~D" num) (font-name "Times-Roman") (font-scaler .5)) :save :finish))))))
    new-mm))






;;;
;;; ----------------    ornamentation
;;;
;;; trill tr, tr wavy line, tr+sign, tr+grace, tr+little note-head in parens, 
;;  mordent, arpeggio, turn, etc
;;; these are like the "articulation" signs in that they are tied to a note

(defclass ornament-mixin (sundry-mixin)
  ((sign :initarg :sign :initform nil :reader ornament-sign))) ;sign is sharp flat and so on

(defclass write-protected-ornament (write-protect ornament-mixin) () )

(defclass ornament (ornament-mixin sundry)
  ((sign :accessor ornament-sign)))

(self-action ornament-sign setf-ornament-sign)

(defmethod descry ((ornament ornament-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) (format nil "(~(~A~)" (sundry-name ornament)) "")
	  (if (ornament-sign ornament) (format nil " :sign ~A" (descry (ornament-sign ornament))) "")
	  (if (next-method-p) (call-next-method ornament stream (or controller ornament)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((ornament ornament-mixin))
  (format nil "(~(~A~)~A~A)" 
	  (sundry-name ornament) 
	  (if (ornament-sign ornament) (format nil " (ornament-sign ~A)" (sign-name (ornament-sign ornament))) "")
	  (the-usual-suspects ornament)))

(defmethod copy ((ornament ornament-mixin) &optional object)
  (let ((new-ornament (if (not object) (make-ornament)
			(if (write-protected object) (copy object)
			  object))))
    (setf (sundry-name new-ornament) (sundry-name ornament))
    (setf (sundry-mark new-ornament) (sundry-mark ornament))
    (if (next-method-p) (call-next-method ornament new-ornament))
    new-ornament))

(defun display-ornament (mark glf note score &optional (count 1) (other-glf nil))
  (let ((y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) (* (max 10 (+ 3 (head-line note))) (staff-line-separation score))))
	(x-off (+ (box-x0 note) (dxy-dx mark) (center note) (box-x0 mark)))
	(dy (box-y1 glf)))
    (moveto score x-off y-off)
    (show score glf :count count)
    (when other-glf
      (moveto score x-off (+ y-off dy (- (box-y0 other-glf)) .05))
      (incf dy (+ (- (box-y1 other-glf) (box-y0 other-glf)) .05))
      (show score other-glf))
    (when (ornament-sign mark)
      (moveto score (+ x-off .2) (+ y-off dy .05 (- (box-y0 (ornament-sign mark)))))
      (show score (ornament-sign mark)))))

(defun ornament (func name &rest objects)
  (let ((new-ornament (make-ornament :name name :mark func)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-ornament (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (push act (marks new-ornament))))))
    new-ornament))


(defun display-mordent (mark note score &rest rest)
  (declare (ignore rest))
  (decf (box-x0 mark) .125)
  (display-ornament mark %mordentslash note score))

;;; mordent with the slash -- the Harvard dictionary calls this a mordent, and the symbol
;;; without a slash is called "inverted-mordent" -- Gardner Read says these names are used
;;; in either way by various authors.

(defvar mordent (make-instance 'write-protected-ornament :name :mordent :mark #'display-mordent))
(defun mordent (&rest objects) (apply #'ornament #'display-mordent :mordent objects))

(defun display-inverted-mordent (mark note score &rest rest)
  (declare (ignore rest))
  (decf (box-x0 mark) .125)
  (display-ornament mark %mordent note score))

(defvar inverted-mordent (make-instance 'write-protected-ornament :name :inverted-mordent :mark #'display-inverted-mordent))
(defun inverted-mordent (&rest objects) (apply #'ornament #'display-inverted-mordent :inverted-mordent objects))

(defun display-double-mordent (mark note score &rest rest)
  (declare (ignore rest))
  (decf (box-x0 mark) .125)
  (display-ornament mark %mordentthree note score))

(defvar double-mordent (make-instance 'write-protected-ornament :name :double-mordent :mark #'display-double-mordent))
(defun double-mordent (&rest objects) (apply #'ornament #'display-double-mordent :double-mordent objects))


(defun display-turn (mark note score &rest rest)
  (declare (ignore rest))
  (decf (box-x0 mark) (if (stem-is-up? note) .2 .15))
  (display-ornament mark %turn note score))

(defvar turn (make-instance 'write-protected-ornament :name :turn :mark #'display-turn))
(defun turn (&rest objects) (apply #'ornament #'display-turn :turn objects))

(defun display-short-trill (mark note score &rest rest)
  (declare (ignore rest))
  (decf (box-x0 mark) (if (stem-is-up? note) .2 .15))
  (display-ornament mark %trillsection note score 2))

(defvar short-trill (make-instance 'write-protected-ornament :name :short-trill :mark #'display-short-trill))
(defun short-trill (&rest objects) (apply #'ornament #'display-short-trill :short-trill objects))

(defun display-trilled-turn (mark note score &rest rest)
  (declare (ignore rest))
  (decf (box-x0 mark) (if (stem-is-up? note) .2 .15))
  (display-ornament mark %trillsection note score 2 %turn))

(defvar trilled-turn (make-instance 'write-protected-ornament :name :trilled-turn :mark #'display-trilled-turn))
(defun trilled-turn (&rest objects) (apply #'ornament #'display-trilled-turn :trilled-turn objects))


;;; trills are relatively complicated -- there can be all kinds of additional information attached to the trill

(defclass trill-mixin (ornament-mixin)
  ((sign-position :initarg :sign-position :initform nil :reader sign-position)
   (other-note :initarg :other-note :initform nil :reader other-note)
   (wavy-line :initarg :wavy-line :initform nil :reader wavy-line)
   (wavy-time :initarg :wavy-time :initform nil :reader wavy-time)))

(defclass write-protected-trill (write-protect trill-mixin)
  ())

(defclass trill (trill-mixin ornament)
  ((sign-position :accessor sign-position)
   (other-note :accessor other-note)
   (wavy-line :accessor wavy-line)
   (wavy-time :accessor wavy-time)))

(defmethod trill-p ((obj t)) nil)
(defmethod trill-p ((obj trill-mixin)) t)

(defmethod descry ((trill trill-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A~A~A"
	  (if (not controller) "(trill" "")
	  (if (sign-position trill) (format nil " :sign-position :~(~A~)" (sign-position trill)) "")
	  (if (other-note trill) (format nil " :other-note ~A" (other-note trill)) "")
	  (if (wavy-line trill) (format nil " :wavy-line ~A~A" 
					(if (listp (wavy-line trill)) "'" "")
					(wavy-line trill)) 
	    "")
	  (if (wavy-time trill) (format nil " :wavy-time ~A" (wavy-time trill)) "")
	  (if (next-method-p) (call-next-method trill stream (or controller trill)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((trill trill-mixin))
  (format nil "(trill~A~A~A~A~A)"
	  (if (ornament-sign trill) (format nil " (ornament-sign ~A)" (sign-name (ornament-sign trill))) "")
	  (if (sign-position trill) (format nil " (sign-position :~(~A~))" (sign-position trill)) "")
	  (if (wavy-line trill) 
	      (format nil " (wavy-line ~A~A)" 
		      (if (listp (wavy-line trill)) "'" "")
		      (wavy-line trill)) 
	    "")
	  (if (wavy-time trill) (format nil " (wavy-time ~1,3F)" (wavy-time trill)) "")
	  (the-usual-suspects trill)))

(defmethod copy ((trill trill-mixin) &optional object)
  (let ((new-trill (if (not object) (make-trill)
		     (if (write-protected object) (copy object)
		       object))))
    (if (other-note trill) (setf (other-note new-trill) (copy (other-note trill))))
    (setf (sign-position new-trill) (sign-position trill))
    (setf (wavy-line new-trill) (wavy-line trill))
    (setf (wavy-time new-trill) (wavy-time trill))
    (if (next-method-p) (call-next-method trill new-trill))
    new-trill))

(defmethod backpatch ((trill trill-mixin)) 
  (wavy-line trill))

(defmethod backpatch-time ((trill trill-mixin) obj)
  (declare (ignore obj))
  (wavy-time trill))

(self-action sign-position setf-sign-position)
(self-action wavy-line setf-wavy-line)
(self-action wavy-time setf-wavy-time)
(self-action other-note setf-other-note)

(defun display-trill (trill note score &rest rest)
  (let ((y-off (+ (box-y0 trill) (dxy-dy trill) (%staff-y0 note) (* (max 9 (+ 2 (head-line note))) (staff-line-separation score))))
	(x-off (+ (box-x0 note) (center note) (dxy-dx trill) (box-x0 trill) -.2))
	(use-parens (eq (sign-position trill) :in-parentheses)))
    (moveto score x-off y-off)
    (simple-show score %trillofinno)
    (if (or (ornament-sign trill) (other-note trill))
	(if (and (ornament-sign trill)
		 (or (not (sign-position trill))
		     (eq (sign-position trill) :right)))
	    (progn
	      (rmoveto score .1 .15)
	      (show score (ornament-sign trill)))
	  (if (and (ornament-sign trill)
		   (eq (sign-position trill) :up))
	      (progn
		(moveto score (+ x-off .1) (+ y-off .75))
		(show score (ornament-sign trill)))
	    (if (or (and (other-note trill) 
			 (not (sign-position trill)))
		    use-parens)
		(let ((other-note-line (and (other-note trill)
					    (place-of-note-given-note note (other-note trill)))))
		  (moveto score (+ x-off (g-rx %trillofinno))
			  (if (other-note trill)
			      (+ (%staff-y0 note) (* (staff-line-separation score) other-note-line))
			    (+ (box-y0 note) (staff-line-separation score))))
		  (rmoveto score (if (and (numberp (dots note)) (plusp (dots note))) .15 .05) -.15)
		  (if use-parens
		      (show score %parenleft :size (* .8 (scr-size score))))
		  (if (or (sign (other-note trill))
			  (ornament-sign trill))
		      (rmoveto score (if use-parens .15 .05) .15))
		  (if (ornament-sign trill)
		      (show score (ornament-sign trill))
		    (if (sign (other-note trill))
			(progn
			  (show score (sign (other-note trill)) :size (* .5 (scr-size score)))
			  (rmoveto score .1 0))))
		  (if (other-note trill)
		      (let ((line other-note-line)
			    (sls (staff-line-separation score)))
			(rmoveto score .05 0)

			(if (or (< line -1) (> line 9))
			    (let* ((yline (if (oddp line) 
					      (if (minusp line) 
						  sls
						(- sls))
					    0))
				   (factor (if (minusp line) 2 -2))
				   (lines (if (oddp line) 
					      (if (plusp line) 
						  (floor (- line 9) 2)
						(floor (- (abs line) 1) 2))
					    (if (plusp line)
						(floor (- line 8) 2)
					      (floor (abs line) 2)))))
			      (rmoveto score .05 0)
			      (show score %quarternotehead :size (* .5 (scr-size score)))
			      (rmoveto score -.05 yline)
			      (do ((i 0 (1+ i)))
				  ((>= i lines))
				(rlineto score .25 0)
				(rmoveto score -.25 (* factor sls)))
			      (rmoveto score .1 (- (+ yline (* sls factor lines)))))
			  (show score %quarternotehead :size (* .5 (scr-size score))))))

		  (rmoveto score (if (other-note trill) .15 .05) -.15)
		  (if use-parens (show score %parenright :size (* .8 (scr-size score)))))))))
    (if (wavy-line trill)
	(let* ((sign-dx (if (or (other-note trill) use-parens)
			    .8
			  (if (and (ornament-sign trill)
				   (or (not (sign-position trill))
				       (member (sign-position trill) '(:right :in-parentheses))))
			      .4
			    0)))
	       (wavy-dx (and (not (member :just-fooling rest))
			     (listp (wavy-line trill))))
	       (trill-x0 (+ x-off .5 sign-dx (if wavy-dx (first (wavy-line trill)) 0)))
	       (trill-length (- (+ (box-x1 trill) 
				   (if wavy-dx (third (wavy-line trill)) 0))
				trill-x0 .25))
	       (count (round trill-length (g-rx %trillsection))))
	  (when (plusp count)
	    (moveto score trill-x0 y-off)
	    (show score %trillsection :count count))))))

#|
(cmn (staff treble 
       (note gf5 w. (trill (wavy-line t) (ornament-sign small-natural) (sign-position :in-parentheses) (other-note f5)))) 
     (staff bass c3 q c3 q c3 q c3 q c3 q c3 q ))
|#


(defvar trill (make-instance 'write-protected-trill :name :trill :mark #'display-trill))

(defun trill (&rest objects)
  (let ((new-trill (make-trill :mark #'display-trill)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-trill (arguments act)))))
    new-trill))



(defclass arpeggio-mark-mixin (sundry-mixin)
  ((arrow-direction :initarg :arrow-direction :initform nil :reader arrow-direction)))

(defclass write-protected-arpeggio-mark (write-protect arpeggio-mark-mixin) ())

(defclass arpeggio-mark (arpeggio-mark-mixin sundry)
  ((arrow-direction :accessor arrow-direction)))

(defmethod arpeggio-p ((obj t)) nil)
(defmethod arpeggio-p ((obj arpeggio-mark-mixin)) t)

(self-action arrow-direction setf-arrow-direction)

(defmethod descry ((arp arpeggio-mark-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) (format nil "(~(~A~)" (sundry-name arp)) "")
	  (if (arrow-direction arp) (format nil " :arrow-direction :~(~A~)" (arrow-direction arp)) "")
	  (if (next-method-p) (call-next-method arp stream (or controller arp)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((arpeggio arpeggio-mark-mixin))
  (format nil "(~(~A~)~A~A)"
	  (sundry-name arpeggio)
	  (if (and (eq (sundry-name arpeggio) :arpeggio)
		   (arrow-direction arpeggio))
	      (if (eq (arrow-direction arpeggio) :up)
		  " arrow-up" " arrow-down")
	    "")
	  (the-usual-suspects arpeggio)))

(defmethod copy ((arp arpeggio-mark-mixin) &optional object)
  (let ((new-arp (if (not object) (make-arpeggio-mark)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (sundry-name new-arp) (sundry-name arp))
    (setf (sundry-mark new-arp) (sundry-mark arp))
    (setf (arrow-direction new-arp) (arrow-direction arp))
    (if (next-method-p) (call-next-method arp new-arp))
    new-arp))

(defun display-arpeggio (mark chord score &optional just-arrow)
  (let* ((arrow-info (arrow-direction mark))
	 (maxl (maximum-line chord))
	 (minl (minimum-line chord))
	 (dist (- maxl minl))
	 (bottom (+ (%staff-y0 chord) -.125 (* minl (staff-line-separation score)) (dxy-dy mark) (box-y0 mark)))
	 (y-loc 0)
	 (x-loc (+ (box-x0 chord) (dxy-dx mark) (box-x0 mark) (if (or (< minl -1) (> maxl 9)) -.05 0))))
    (moveto score x-loc bottom)
    (if just-arrow
	(progn
	  (rlineto score 0 (setf y-loc (* dist (staff-line-separation score))))
	  (incf y-loc bottom))
      (progn
	(loop for i from 0 below dist by 4 do
	  (simple-show score %arpeggio)
	  (moveto score x-loc (setf y-loc (+ bottom (* (+ i 4) (staff-line-separation score))))))
	(when (zerop (mod dist 4))
	  (rmoveto score 0 (* -2 (staff-line-separation score)))
	  (simple-show score %arpeggio))))
    (when arrow-info
      (let* ((above (eq arrow-info :up))
	     (dy (if above -.2 .2))
	     (x1-loc (+ x-loc (if (not just-arrow) (* .5 (g-rx %arpeggio)) 0)))
	     (y1-loc (if above (+ y-loc (if (not just-arrow)
					    (if (zerop (mod dist 4)) 
						(* 2 (staff-line-separation score)) 
					      0)
					  0))
		       bottom)))
	(moveto score x1-loc y1-loc)
	(rlineto score 0 (- dy))
	(draw score)
	(moveto score x1-loc (- y1-loc dy))
	(setf (line-width score) .05)
	(rlineto score .1 dy)
	(rmoveto score -.1 (- dy))
	(rlineto score -.1 dy)
	(draw score)
	(setf (line-width score) 0)))))

(defvar arpeggio (make-instance 'write-protected-arpeggio-mark :name :arpeggio :mark #'display-arpeggio))

(defun arpeggio (&rest objects)
  (let ((new-arp (make-arpeggio-mark :name :arpeggio :mark #'display-arpeggio)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-arp (arguments act))
	  (if (arrow-p act)
	      (if (eq (sundry-name act) :arrow-up)
		  (setf (arrow-direction new-arp) :up)
		(if (eq (sundry-name act) :arrow-down)
		    (setf (arrow-direction new-arp) :down)))
	    (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
		(push act (marks new-arp)))))))
    new-arp))



(defun display-no-arpeggio (mark chord score &rest rest)
  (declare (ignore rest))
  (let* ((maxl (maximum-line chord))
	 (minl (minimum-line chord))
	 (dist (- maxl minl))
	 (bottom (+ (%staff-y0 chord) (* minl (staff-line-separation score)) (dxy-dy mark)))
	 (x-loc (+ (box-x0 chord) (dxy-dx mark) (if (or (< minl -1) (> maxl 9)) -.15 -.05))))
    (moveto score x-loc bottom)
    (rlineto score -.1 0)
    (rlineto score 0 (* dist (staff-line-separation score)))
    (rlineto score .1 0)
    (draw score)))

(defvar no-arpeggio (make-instance 'write-protected-sundry :name :no-arpeggio :mark #'display-no-arpeggio))
(defun no-arpeggio (&rest objects) (apply #'mark #'display-no-arpeggio :no-arpeggio objects))



(defun arrow-p (object)
  (and object (arpeggio-p object) (member (sundry-name object) '(:arrow-up :arrow-down))))

(defun display-arrow (mark chord score &rest rest) 
  (declare (ignore rest))
  (display-arpeggio mark chord score t))

(defvar arrow-up (make-instance 'write-protected-arpeggio-mark :name :arrow-up :mark #'display-arrow :arrow-direction :up))

(defun arrow-up (&rest objects) 
  (let ((new-arrow (apply #'arpeggio objects)))
    (setf (arrow-direction new-arrow) :up)
    (setf (sundry-name new-arrow) :arrow-up)
    (setf (sundry-mark new-arrow) #'display-arrow)
    new-arrow))

(defvar arrow-down (make-instance 'write-protected-arpeggio-mark :name :arrow-down :mark #'display-arrow :arrow-direction :down))

(defun arrow-down (&rest objects) 
  (let ((new-arrow (apply #'arpeggio objects)))
    (setf (arrow-direction new-arrow) :down)
    (setf (sundry-name new-arrow) :arrow-down)
    (setf (sundry-mark new-arrow) #'display-arrow)
    new-arrow))

(defun arrow (&rest objects)
  (let ((new-arrow (make-arpeggio-mark :name :arrow-up :mark #'display-arrow :arrow-direction :up)))
    (loop for act in objects do
      (if (self-acting-p act)
	  (funcall (action act) new-arrow (arguments act))
	(if (eq act :down)
	    (progn
	      (setf (sundry-name new-arrow) :arrow-down)
	      (setf (arrow-direction new-arrow) :down))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (push act (marks new-arrow))))))
    new-arrow))



;;;
;;; ----------------    rehearsal numbers and letters
;;;
;;; these can be boxed or circled or left alone, normally attached to a bar line
;;; auto-incrementing is needed so we don't have to count them by hand
;;; letters increment past Z as AA BB CC and so on.

(defun next-rehearsal-letter (current-letter)
  (if (not current-letter)
      "A"
    (let ((count (length current-letter))
	  (base-letter (char current-letter 0)))
      (if (char= base-letter #\Z) 
	  (progn
	    (setf base-letter #\A)
	    (incf count))
	(if (char= base-letter #\H)	;skip "I" -- looks too much like "1" and "J"
	    (setf base-letter #\J)
	  (setf base-letter (code-char (1+ (char-code base-letter))))))
      (make-string count :initial-element base-letter))))

(defun next-rehearsal-number (current-number)
  (1+ current-number))


(defclass rehearsal-mark (sundry font-mixin)
  ((remark :initarg :remark :initform nil :accessor remark)
   (frame :initarg :frame :initform nil :accessor frame)
   (frame-width :initarg :rehearsal-frame-width :initform .075 :accessor rehearsal-frame-width)
   (frame-white-space :initarg :rehearsal-frame-white-space :initform .05 :accessor rehearsal-frame-white-space)
   (font-name :initform "Times-Bold")
   (font-scaler :initform 1.0)))

(self-action frame setf-frame)		;:box, :circle, :none, or a function
(self-action rehearsal-frame-width setf-rehearsal-frame-width)
(self-action rehearsal-frame-white-space setf-rehearsal-frame-white-space)


(defmethod copy ((rn rehearsal-mark) &optional object)
  (let ((new-rn (if (not object) (make-rehearsal-mark)
		  (if (write-protected object) (copy object)
		    object))))
    (setf (remark new-rn) (remark rn))
    (setf (frame new-rn) (frame rn))
    (setf (rehearsal-frame-width new-rn) (rehearsal-frame-width rn))
    (setf (rehearsal-frame-white-space new-rn) (rehearsal-frame-white-space rn))
    (if (next-method-p) (call-next-method rn new-rn))
    new-rn))

(defmethod rehearsal-mark-p ((obj t)) nil)
(defmethod rehearsal-mark-p ((obj rehearsal-mark)) t)

(defmethod descry ((rm rehearsal-mark) &optional stream controller)
  (format stream "~A :remark ~A :frame ~A~A~A~A~A"
	  (if (not controller) 
	      (if (numberp (remark rm)) 
		  "(rehearsal-number" 
		"(rehearsal-letter") 
	    "")
	  (remark rm)
	  (if (frame rm) (format nil ":~(~A~)" (frame rm)))
	  (format nil " :width ~1,3F" (rehearsal-frame-width rm))
	  (format nil " :white-space ~1,3F" (rehearsal-frame-white-space rm))
	  (if (next-method-p) (call-next-method rm stream (or controller rm)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((rm rehearsal-mark))
  (format nil "(~A~A~A~A~A~A)"
	  (if (numberp (remark rm)) "rehearsal-number" "rehearsal-letter")
	  (if (numberp (remark rm)) (format nil " ~D" (remark rm)) (format " ~S" (remark rm)))
	  (if (frame rm) (format nil "(frame :~(~A~))" (frame rm)) "")
	  (if (/= (rehearsal-frame-width rm) .075) (format nil " (rehearsal-frame-width ~1,3F)" (rehearsal-frame-width rm)) "")
	  (if (/= (rehearsal-frame-white-space rm) .05) (format nil " (rehearsal-frame-white-space ~1,3F)" (rehearsal-frame-white-space rm)) "")
	  (the-usual-suspects rm)))

(defvar rehearsal-letter (make-self-acting 
			  :action #'(lambda (bar &rest rest)
				      (declare (ignore rest))
				      (add-to-marks bar (list (ur-rehearsal-mark nil))))
			  :arguments nil))

(defvar rehearsal-number (make-self-acting 
			  :action #'(lambda (bar &rest rest)
				      (declare (ignore rest))
				      (add-to-marks bar (list (ur-rehearsal-mark t))))
			  :arguments nil))

(defun display-rehearsal-mark (mark bar score &rest rest)
  (declare (ignore rest))
  (let ((x0 (+ (box-x0 bar) (dxy-dx mark)))
	(y0 (+ (box-y1 bar) 0.5 (dxy-dy mark)))
	(txt (%%text :letters (if (numberp (remark mark)) (format nil "~D" (remark mark)) (remark mark))
		     :font-name (font-name mark)
		     :font-scaler (font-scaler mark))))
    (moveto score x0 y0)
    (show score txt)
    (when (and (frame mark) (not (eq (frame mark) :none)))
      (let* ((width (rehearsal-frame-width mark))
	     (ws (rehearsal-frame-white-space mark))
	     (fx0 (- x0 width ws))
	     (fy0 (- y0 width ws))
	     (fx1 (+ x0 ws width (* (length (letters txt)) (font-scaler mark) .7)))
	     ;; the "right" thing would be to go out, find the appropriate .afm file,
	     ;; read in the glyph widths, and use them here rather than .7.
	     (fy1 (+ y0 width ws (* .7 (font-scaler mark)))))
	(setf (line-width score) width)
	(case (frame mark)
	  (:box 
	   (moveto score fx0 fy0)
	   (lineto score fx1 fy0)
	   (lineto score fx1 fy1)
	   (lineto score fx0 fy1)
	   (lineto score fx0 (- fy0 (* .5 width)))
	   (draw score))
	  (:circle 
	   (let ((rx (* .5 (+ fx0 fx1)))
		 (ry (* .5 (+ fy0 fy1)))
		 (r (+ width (* .5 (max (- fx1 fx0) (- fy1 fy0))))))
	     (circle score rx ry r)
	     (draw score)))
	  (otherwise (cmn-error "unknown rehearsal mark frame: ~A" (frame mark))))
	(setf (line-width score) 0)))))

(defun ur-rehearsal-mark (num &rest args)
  (let ((new-rehearsal-mark (make-rehearsal-mark :name :rehearsal-mark :mark #'display-rehearsal-mark)))
    (loop for act in args do
      (when act
	(if (and num (numberp act))
	    (setf (remark new-rehearsal-mark) act)
	  (if (and (not num) (stringp act))
	      (setf (remark new-rehearsal-mark) act)
	    (if (and (not num) (characterp act))
		(setf (remark new-rehearsal-mark) (make-string 1 :initial-element act))
	      (if (self-acting-p act)
		  (funcall (action act) new-rehearsal-mark (arguments act))
		(if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
		    (push act (marks new-rehearsal-mark)))))))))
    (if (not (remark new-rehearsal-mark))
	(setf (remark new-rehearsal-mark) 
	  (if num 
	      (if rehearsal-stack 
		  (next-rehearsal-number (remark rehearsal-stack))
		1)
	    (if rehearsal-stack
		(next-rehearsal-letter (remark rehearsal-stack))
	      "A"))))
    (if (and rehearsal-stack
	     (not (frame new-rehearsal-mark))
	     (frame rehearsal-stack))
	(setf (frame new-rehearsal-mark) (frame rehearsal-stack)))
    (setf rehearsal-stack new-rehearsal-mark)
    new-rehearsal-mark))

(defun rehearsal-number (&rest args) (apply #'ur-rehearsal-mark t args))
(defun rehearsal-letter (&rest args) (apply #'ur-rehearsal-mark nil args))




;;;
;;; ----------------    measure numbers
;;;

(defclass measure-mark (sundry font-mixin)
  ((remark :initarg :remark :initform nil :accessor remark)
   (font-name :initform "Times-Italic")	;Ross says this should be the standard music font numerals (as used in meters)
   (font-scaler :initform .5)))

(defmethod copy ((rn measure-mark) &optional object)
  (let ((new-rn (if (not object) (make-measure-mark)
		  (if (write-protected object) (copy object)
		    object))))
    (setf (remark new-rn) (remark rn))
    (if (next-method-p) (call-next-method rn new-rn))
    new-rn))

(defmethod measure-mark-p ((obj t)) nil)
(defmethod measure-mark-p ((obj measure-mark)) t)

(defmethod descry ((rm measure-mark) &optional stream controller)
  (format stream "(measure-number :remark ~A~A~A"
	  (remark rm)
	  (if (next-method-p) (call-next-method rm stream (or controller rm)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((rm measure-mark))
  (format nil "(measure-number ~A~A)" (remark rm) (the-usual-suspects rm)))

(defun display-measure-mark (mark bar score)
  (when (not (invisible-matrix-p mark))
    (let ((x0 (+ (box-x0 bar) (dxy-dx mark)))
	  (y0 (+ (box-y1 bar) 0.5 (dxy-dy mark)))
	  (txt (%%text :letters (if (numberp (remark mark)) (format nil "~D" (remark mark)) (remark mark))
		       :font-name (font-name mark)
		       :font-scaler (font-scaler mark))))
      (moveto score x0 y0)
      (show score txt))))

(defun measure-number (&rest args)
  (let ((new-measure-mark (make-measure-mark :name :measure-mark :mark #'display-measure-mark)))
    (loop for act in args do
      (when act
	(if (or (numberp act) (stringp act))
	    (setf (remark new-measure-mark) act)
	  (if (self-acting-p act)
	      (funcall (action act) new-measure-mark (arguments act))
	    (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
		(push act (marks new-measure-mark)))))))
    new-measure-mark))




;;;
;;; ----------------    fingering
;;;

(defclass finger-mark (sundry font-mixin)
  ((fingers :initarg :fingers :initform nil :accessor fingers)
   (size :initarg :fingering-size :initform 1.0 :accessor fingering-size)
   (font-name :initform "Times-Roman")
   (font-scaler :initform .33)))

(defmethod finger-mark-p ((obj t)) nil)
(defmethod finger-mark-p ((obj finger-mark)) t)

(self-action fingering-size setf-fingering-size)
(self-action fingers setf-fingers)

(defmethod copy ((fing finger-mark) &optional object)
  (let ((new-fing (if (not object) (make-finger-mark)
		    (if (write-protected object) (copy object)
		      object))))
    (setf (fingers new-fing) (fingers fing))
    (if (next-method-p) (call-next-method fing new-fing))
    new-fing))

(defmethod descry ((fing finger-mark) &optional stream controller)
  (format stream "~A :fingers '~A~A~A"
	  (if (not controller) "(fingering" "")
	  (fingers fing)
	  (if (next-method-p) (call-next-method fing stream (or controller fing)) "")
	  (if (not controller) ")" "")))

(defmethod identify ((fng finger-mark))
  (format nil "(fingering~{ ~D~}~A)" (fingers fng) (the-usual-suspects fng)))

(defun display-fingering (mark note score &rest rest)
  (let* ((dir (direction-from-note mark note))
	 (ffs (* (font-scaler mark) (fingering-size mark)))
	 (nums (fingers mark))
	 (len (length nums))
	 (dy (* (1- len) ffs))
	 (y-off (+ (box-y0 mark) (dxy-dy mark)  
		   (%staff-y0 note) 
		   (if (eq dir :up) 
		       (+ dy
			  (* (max 9 (+ 2 (head-line note))) (staff-line-separation score)))
		     (min (- (+ ffs .125)) 
			  (* (- (minimum-line note) 3) (staff-line-separation score))))))
	 (x-off (+ (box-x0 note) (dxy-dx mark) -.1 (center note) (box-x0 mark))))
#-Kcl (loop for num in nums and new-y from y-off by (- ffs) do
	(moveto score x-off new-y)
	(if (not (text-p num))
	    (show score (%%text :letters (if (numberp num) (format nil "~D" num) num)
				:font-name (font-name mark) :font-scaler ffs))
	  (progn
	    (setf (box-x0 num) x-off)
	    (setf (box-y0 num) new-y)
	    (display num mark score rest))))
#+Kcl (let ((new-y y-off))
	(loop for num in nums do
	  (moveto score x-off new-y)
	  (if (not (text-p num))
	      (show score (%%text :letters (if (numberp num) (format nil "~D" num) num)
				  :font-name (font-name mark) :font-scaler ffs))
	    (progn
	      (setf (box-x0 num) x-off)
	      (setf (box-y0 num) new-y)
	      (display num mark score rest)))
	  (decf new-y ffs)))
   ))

(defun fingering (&rest objects)
  (let ((new-mark (make-finger-mark :name :fingering :mark #'display-fingering))
	(fingers nil))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-mark (arguments act))
	  (if (or (sundry-p act) (text-p act) (glyph-list-p act) (dynamics-p act))
	      (push act (marks new-mark))
	    (push act fingers)))))
    (if (not (fingers new-mark)) (setf (fingers new-mark) (reverse fingers)))
    new-mark))




;;;
;;; ----------------    octave signs
;;;

(defclass octave-sign (sundry font-mixin)
  ((octave :initarg :octave :initform nil :accessor octave)
   (note0 :initarg :note0 :initform nil :accessor note0)
   (note1 :initarg :note1 :initform nil :accessor note1)
   (max-line :initarg :max-line :initform nil :accessor max-line)
   (connecting-pattern :initarg :connecting-pattern :initform '(10 20) :accessor octave-sign-pattern)
   (vertical-separation :initarg :vertical-separation :initform 3 :accessor vertical-separation)
   (font-name :initform "Times-Roman")
   (font-scaler :initform .5)))

(self-action vertical-separation setf-vertical-separation)
(self-action octave-sign-pattern setf-octave-sign-pattern)

(defmethod descry ((octave octave-sign) &optional stream controller)
  (format stream "~A~A~A~A~A~A~A~A~A"
	  (if (not controller) (format nil "(octave-sign") "")
	  (if (octave octave) (format nil " :octave ~D" (octave octave)) "")
	  (if (note0 octave) (format nil " :note0 ~A" (note0 octave)) "")
	  (if (note1 octave) (format nil " :note1 ~A" (note1 octave)) "")
	  (if (max-line octave) (format nil " :max-line ~D" (max-line octave)) "")
	  (format nil " :vertical-separation ~A" (vertical-separation octave))
	  (format nil " :connecting-pattern '~A" (octave-sign-pattern octave))
	  (if (next-method-p) (call-next-method octave stream (or controller octave)) "")
	  (if (not controller) ")" "")))

(defvar cmn-store-tags nil)		;a list of (obj str) pairs

(defmethod identify ((octave octave-sign))
  ;; like beam or slur -- have to mark end note (note1) and deal with possible tags / line breaks
  ;; first problem is handled with add-to-cmn-store-tags for the ending note
  ;; there are also the special cases of single note octaves and no octaves
  ;; if zerop dy and dx, probably safe to use the variable form
  ;; in general, no need for explicit tags here
  (let* ((var-time (and (zerop (dxy-dx octave)) (zerop (dxy-dy octave))))
	 (oct (octave octave))
	 (fun-name (concatenate 'string 
		     (if (= oct 0) "no-"
		       (if (= (abs oct) 2) "two-"
			 ""))
		     "octave"
		     (if (= oct 0) "-sign"
		       (if (= (abs oct) 2) "s"
			 ""))
		     (if (= oct 0) ""
		       (if (plusp oct) "-up"
			 "-down"))))
	 (begin-and-end (not (eq (note0 octave) (note1 octave)))))
    (if begin-and-end
	(if (not (pitch (note0 octave)))
	    (let ((pair (find :octave cmn-store-tags :key #'third :test #'eq)))
	      (if pair
		  (progn
		    (setf cmn-store-tags (remove pair cmn-store-tags))
		    (format nil "~A" (second pair)))))
	  (progn
	    (push (list (note1 octave)
			(format nil "~Aend-~A~A"
				(if var-time "" "(")
				fun-name
				(if var-time "" ")"))
			:octave)
		  cmn-store-tags)
	    (format nil "~Abegin-~A~A~A"
		    (if var-time "" "(")
		    fun-name
		    (if var-time "" (the-usual-suspects octave))
		    (if var-time "" ")"))))
      (if var-time
	  fun-name
	(format nil "(~A~A)" fun-name (the-usual-suspects octave))))))

(defmethod copy ((octave octave-sign) &optional object)
  (let ((new-octave (if (not object) (make-octave-sign)
		      (if (write-protected object) (copy object)
			object))))
    (setf (octave new-octave) (octave octave))
    (setf (max-line new-octave) (max-line octave))
    (setf (vertical-separation new-octave) (vertical-separation octave))
    (setf (octave-sign-pattern new-octave) (copy-list (octave-sign-pattern octave)))
    (if (next-method-p) (call-next-method octave new-octave))
    new-octave))

(defun start-octave (num &rest args)
  (make-self-acting
   :action #'(lambda (note &rest rest)
	       (declare (ignore rest))
	       (let ((new-8 (make-octave-sign
			     :note0 note
			     :octave num
			     :mark #'display-octave
			     :name :octave)))
		 (setf octave-stack new-8)
		 (if args (loop for arg in args do
			    (if (self-acting-p arg)
				(funcall (action arg) new-8 (arguments arg)))))
		 nil))
   :arguments nil))

(defun end-octave (num &rest args)
  (declare (ignore args))
  (make-self-acting 
   :action #'(lambda (note &rest rest)
	       (declare (ignore rest))
	       (when (not (zerop num))
		 (if (not octave-stack) 
		     (funcall (action (start-octave num)) note nil)))
	       (add-to-marks (note0 octave-stack) (list octave-stack))
	       (setf (note1 octave-stack) note)
	       (setf octave-stack nil))
   :arguments nil))

(defgeneric (setf octaved) (val obj))
(defgeneric (setf store-data) (val obj))

(defun start-and-end-octave (num &rest args)
  (make-self-acting
   :action #'(lambda (note &rest rest)
	       (declare (ignore rest))
	       (setf (octaved note) t)
	       (add-to-marks note (list (let ((new-8 (make-octave-sign
						      :note0 note
						      :note1 note
						      :mark #'display-octave
						      :octave num
						      :name :octave)))
					  (if args (loop for arg in args do
						     (if (self-acting-p arg)
							 (funcall (action arg) new-8 (arguments arg)))))
					  new-8))))
   :arguments nil))

(defvar begin-octave-up         (start-octave 1))
(defvar octave-up               (start-and-end-octave 1))
(defvar end-octave-up           (end-octave 1))
(defvar begin-octave-down       (start-octave -1))
(defvar octave-down             (start-and-end-octave -1))
(defvar end-octave-down         (end-octave -1))
(defvar two-octaves-up          (start-and-end-octave 2))
(defvar two-octaves-down        (start-and-end-octave -2))
(defvar begin-two-octaves-up    (start-octave 2))
(defvar begin-two-octaves-down  (start-octave -2))
(defvar end-two-octaves-up      (end-octave 2))
(defvar end-two-octaves-down    (end-octave -2))
(defvar no-octave-sign          (start-and-end-octave 0))
(defvar begin-no-octave-sign    (start-octave 0))
(defvar end-no-octave-sign      (end-octave 0))

(defun begin-octave-up (&rest args)      (apply #'start-octave 1 args))
(defun octave-up (&rest args)            (apply #'start-and-end-octave 1 args))
(defun end-octave-up (&rest args)        (apply #'end-octave 1 args))
(defun begin-octave-down (&rest args)    (apply #'start-octave -1 args))
(defun octave-down (&rest args)          (apply #'start-and-end-octave -1 args))
(defun end-octave-down (&rest args)      (apply #'end-octave -1 args))
(defun two-octaves-up (&rest args)       (apply #'start-and-end-octave 2 args))
(defun two-octaves-down (&rest args)     (apply #'start-and-end-octave -2 args))
(defun begin-two-octaves-up (&rest args) (apply #'start-octave 2 args))
(defun begin-two-octaves-down (&rest args) (apply #'start-octave -2 args))
(defun end-two-octaves-up (&rest args)   (apply #'end-octave 2 args))
(defun end-two-octaves-down (&rest args) (apply #'end-octave -2 args))
(defun no-octave-sign (&rest args)       (apply #'start-and-end-octave 0 args))
(defun begin-no-octave-sign (&rest args) (apply #'start-octave 0 args))
(defun end-no-octave-sign (&rest args)   (apply #'end-octave 0 args))

(defun prepare-octave-signs (octize oct) ;used only by automatic octave signs
  (list
   (make-octave-sign 
    :name :octave
    :mark #'display-octave
    :note0 (first octize) 
    :note1 (first (last octize)) 
    :octave oct 
    :max-line (if (plusp oct)
		  (loop for note in octize maximize (maximum-line note))
		(loop for note in octize minimize (minimum-line note))))))

(defun display-octave (sundry note score &rest rest)

  ;; Sonata font octave signs are crazy -- what in hell were they thinking of?
  ;; direction is whether we are above of below (:up or :down)
  ;; octaves is how many octaves (8 or 15 is the number used)
  ;; use-italian (if t) means use either "8va" or "15ma" rather than "8" or "15" (i much prefer the latter)
  ;; add-bassa (if t) means "add the word bassa to the 8 or 8va or whatever.
  ;; I suppose a prettier version would put the "va" or "ma" in a smaller font than the "8"
  (if (and (not (member :just-fooling rest)) (not (zerop (octave sundry))))
      (let ((no-start (not (note0 sundry)))
	    (no-end (not (note1 sundry))))
	(if (and (note0 sundry)
		 (note1 sundry)
		 (/= (%staff-y0 (note0 sundry)) (%staff-y0 (note1 sundry))))
	    (let* ((all-data (staff-data *cmn-staff*))
		   (n0 (note0 sundry))
		   (n1 (note1 sundry))
		   (s0 (%staff-y0 n0))
		   (s1 (%staff-y0 n1))
		   (curs s0))
	      ;; loop through the staves in the data list looking for the end staff and inserting no-start-or-end
	      ;;  markers for any in-between -- first find overall staff (*cmn-staff* used here -- bad form)
	      (loop while (not (eq n0 (first all-data))) do (pop all-data))
	      (let ((first-notes nil))
		(loop while (not (eq n1 (first all-data))) do
		  (let* ((curn (pop all-data))
			 (curns (if (audible-p curn) (%staff-y0 curn) curs)))
		    (if (and (/= s1 curns) (/= curs curns))
			(progn
			  (push curn first-notes)
			  (setf curs curns)))))
		(when first-notes
		  (loop for firno in first-notes do
		    (add-to-marks firno
				  (list (make-octave-sign
					 :name :octave
					 :mark #'display-octave
					 :note0 nil
					 :dx (dxy-dx sundry)
					 :dy (dxy-dy sundry)
					 :octave (octave sundry)
					 :max-line nil))))))
	      (add-to-marks (note1 sundry) 
			    (list (make-octave-sign
				   :name :octave
				   :mark #'display-octave
				   :note0 nil
				   :dx (dxy-dx sundry)
				   :dy (dxy-dy sundry)
				   :note1 (note1 sundry)
				   :octave (octave sundry)
				   :max-line nil)))
	      (setf no-end t)
	      (setf (note1 sundry) nil)))
	
	(if  no-start
	    (setf (note0 sundry) (make-note 
				  :line 0 
				  :x0 (if *cmn-staff* (box-x0 *cmn-staff*) (box-x0 score)))))
	(if no-end
	    (setf (note1 sundry) (make-note 
				  :line 0 
				  :x1 (if *cmn-staff* (box-x1 *cmn-staff*) (box-x1 score)))))
	
	(let* ((use-italian (use-italian-octave-signs score))
	       (direction (if (plusp (octave sundry)) :up :down))
	       (octaves (octave sundry))
	       (add-bassa (add-bassa-to-octave-signs score))
	       (x0 (- (x0 (note0 sundry)) (if no-start 0 .2)))
	       (x1 (x1 (note1 sundry)))
	       (y0 (+ (%staff-y0 note) 
		      (if (eq direction :up)
			  (* (max 12
				  (+ (vertical-separation sundry)
				     (or (max-line sundry) 0)))
			     (staff-line-separation score))
			(* (min -4 
				(- (or (max-line sundry) (minimum-line note) 0)
				   (vertical-separation sundry)))
			   (staff-line-separation score)))))
	       (octmsg (if (= octaves 1)
			   (if use-italian
			       "8va"	;why not ottava
			     "8")
			 (if (= octaves 2)
			     (if use-italian
				 "15ma"	;why not quindicesima
			       "15")
			   (if (= octaves -1)
			       (if use-italian
				   (if add-bassa
				       "8va bassa"
				     "8va")
				 "8")
			     (if (= octaves -2)
				 (if use-italian
				     (if add-bassa
					 "15ma bassa"
				       "15ma")
				   "15")
			       "")))))
	       (siz (scr-size score))
	       (mid-y0 (+ y0 (if (eq direction :up) .2 -.2)))
	       (px1 (* (+ x0 (* .3 (length octmsg))))))
	  
	  (incf x0 (dxy-dx sundry))
	  (incf y0 (dxy-dy sundry))
	  (incf mid-y0 (dxy-dy sundry))
	  (incf px1 (dxy-dx sundry))
	  (moveto score x0 (- y0 (if (eq direction :up) 0 .4)))
	  (if (not no-start)
	      (show score (%%text :font-name (font-name sundry)
				  :font-size (floor (* siz (font-scaler sundry)))
				  :letters octmsg)))
	  (when (< (+ px1 .5) x1)	; i.e. when there's room for a dashed line given our 15ma bassa silliness
	    (moveto score (+ px1 .2) mid-y0)
	    (lineto score (- x1 .25) mid-y0 :pattern (map 'list #'(lambda (pt) 
								    (* pt (/ (scr-size score) 40))) 
							  (octave-sign-pattern sundry))))
	  (when (not no-end)
	    (moveto score (- x1 .2) mid-y0)
	    (rlineto score .2 0)
	    (lineto score x1 y0)
	    (draw score))))))

;;; (cmn staff treble (c4 q begin-octave-up) c4 q line-break c4 q c4 q line-break c4 q (c4 q end-octave-up))



;;;
;;; ----------------    graphics
;;;
;;; graphics function (i.e. (c4 q (graphics (file "hi.eps") (rotate 90))) places hi.eps at the middle c rotated 90 degress)

(defun file (name) (make-sundry :name :file :mark name))

(defun display-graphics (mark note score file matrix)
  (let* ((y-off (+ (box-y0 mark) (dxy-dy mark) (staff-y0 note) (* (line note) (staff-line-separation score))))
	 (x-off (+ (box-x0 note) (dxy-dx mark) (box-x0 mark)))
	 (gbb (if (not matrix)
		  (include-eps score file x-off y-off)
		(include-eps-with-matrix-and-pattern score file x-off y-off matrix nil))))
    ;; gbb is the graphics bounding box -- we have to track it because other programs (i.e. WriteNow)
    ;; clip the pasted-in graphics image to the reported bounding box, and including an entire .eps
    ;; file can be arbitrarily outside the bounds of the other graphics.
    (setf (bounds score) gbb)))


(defun graphics (&rest objects)
  (let ((new-graph (make-sundry :name :graphics))
	(file nil))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-graph (arguments act))
	  (if (sundry-p act)
	      (if (eq (sundry-name act) :file)
		  (setf file (sundry-mark act))
		(push act (marks new-graph)))))))
    (setf (sundry-mark new-graph) #'(lambda (mark note score &rest rest)
				      (if (member :file-name rest)
					  ;; this is for cmn-store's benefit
					  file
					(if (not (member :just-fooling rest))
					    ;; at housing time server size is 1.0 but output bounds were true, not scaled,
					    ;; and we have no clean way to see the scaling, so we'll put off the bounds check.
					    (display-graphics mark note score file (matrix new-graph))))))
    new-graph))

