;;; -*- syntax: common-lisp; package: cmn; base: 10; mode: lisp -*-
;;;
;;; CMN=Common Music Notation
;;;
;;; cmn-init sets up the cmn package, cmn-excl, cmn-mcl, cmn-kcl, cmn-clisp set up special case code,
;;; cmn-grfx translates between the various graphics systems, cmn0..4 constitute the main program.

(in-package :cmn)

(defvar *cmn-score* nil)		;top of current score structure (needed globally for debugging and error messages)
(defparameter prewhitespace "                    ")


;;;
;;; ----------------    protect against some lisp actions that aren't what we want
;;;

(defun divide (a b) 
  ;; we don't want user-level divides dropping the poor user into the debugger!
  (if (zerop b) 
      (progn 
	(warn "attempt to divide by 0") 
	a)				;SAIL!
    (/ a b)))


(defun not-rational (x)			;flush lisp rationals which server can't handle and prettify print-out
					;~,3F is needed because kcl prints out endless bogus digits, thereby confusing postscript
					;not-rational should only be used with ~A (sigh -- this is kcl's fault)
  #-(and excl Allegro-v3.1) (if (integerp x) x (format nil "~,3F" (float x)))
  #+(and excl Allegro-v3.1) (if (ratiop x) (float x) x)
  )



;;;
;;; ----------------    main methods
;;;

(defgeneric copy (read-object &optional write-object))
(defgeneric descry (object &optional stream controller))
(defgeneric display (object container score &rest rest))
(defgeneric house (object score))
(defgeneric notify (controller &optional objects))
(defgeneric identify (object))

;;; these are the primary generic functions that every score object responds to
;;; copy returns a copy (unwrite-protected) of its argument (at top level)
;;; descry is our describe method intended for debugging (would be nice if lisp's describe were specializable)
;;; display displays an object
;;; house puts the three layers of white space boxes around an object (during justification)
;;; notify is the syntax checker -- it reads all the arguments collected into an object and regularizes everything
;;; identify is used by the error reporting mechanism to try to give an understandable reference to the current object

(defmethod descry (anything &optional stream controller)
  (declare (ignore controller))
  (format stream " ~A" anything))

(defmethod identify (anything)
  anything)



;;;
;;; ----------------    read-only structures
;;;
;;; our basic score syntax includes such things as (cmn staff treble c4 q) and we have
;;; to make sure we never write the fields of the "staff" or "treble" (etc) variables
;;; so all such special variables are "write-protected".
;;;
;;; This is actually a kludge to get around a PCL bug that (defconstant hi (make-instance...))
;;;   does not return a true constant (i.e. (constantp hi) is NIL) -- in ACL 4.1, and I assume
;;;   any "real" CLOS lisp, the write-protected class is unneeded, but at this point a rewrite
;;;   would be a big undertaking.

(defclass write-protect ()
  ())

;;; using typep as in:
;;;(defun write-protected (object) (typep object 'write-protect))
;;; is about 10 times slower than methods such as:

(defmethod write-protected ((obj t)) nil)
(defmethod write-protected ((obj write-protect)) t)



;;;
;;; ----------------    bounds (the "bounding-box")
;;;
;;; x0 y0 is lower left corner, normally -- some objects slop over one way or the other

(defclass box-mixin ()
  ((x0 :initarg :x0 :initform 0 :reader x0 :reader box-x0 :type number)
   (y0 :initarg :y0 :initform 0 :reader y0 :reader box-y0 :type number)
   (x1 :initarg :x1 :initform 0 :reader x1 :reader box-x1 :type number)
   (y1 :initarg :y1 :initform 0 :reader y1 :reader box-y1 :type number)))

(defclass box (box-mixin)
  ((x0 :accessor x0 :accessor box-x0 :type number)
   (y0 :accessor y0 :accessor box-y0 :type number)
   (x1 :accessor x1 :accessor box-x1 :type number)
   (y1 :accessor y1 :accessor box-y1 :type number)))

(defun copy-bounding-box (obj1 obj2)
  (setf (box-x0 obj2) (box-x0 obj1))
  (setf (box-x1 obj2) (box-x1 obj1))
  (setf (box-y0 obj2) (box-y0 obj1))
  (setf (box-y1 obj2) (box-y1 obj1)))

(defmethod staff-y0 ((box box-mixin)) (box-y0 box))

(defmethod copy ((box box-mixin) &optional object)
  (let ((new-box (if (not object) (make-box)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (box-x0 new-box) (box-x0 box))
    (setf (box-y0 new-box) (box-y0 box))
    (setf (box-x1 new-box) (box-x1 box))
    (setf (box-y1 new-box) (box-y1 box))
    (if (next-method-p) (call-next-method box new-box))
    new-box))

(defmethod descry ((box box-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A~A~A"
	  (if (not controller) "(box" 
	    (if (plusp (x0 box)) (format nil "~%~A" prewhitespace)
	      ""))
	  (if (not (zerop (x0 box))) (format nil " :x0 ~A" (not-rational (x0 box))) "")
	  (if (not (zerop (y0 box))) (format nil " :y0 ~A" (not-rational (y0 box))) "")
	  (if (not (zerop (x1 box))) (format nil " :x1 ~A" (not-rational (x1 box))) "")
	  (if (not (zerop (y1 box))) (format nil " :y1 ~A" (not-rational (y1 box))) "")
	  (if (next-method-p) (call-next-method box stream (or controller box)) "")
	  (if (not controller) ")" "")))




;;; 
;;; ---------------     patterns and coordinate transformations
;;;
;;; this is not directly a part of the CMN world (it's just graphics fooling around)
;;; so it is not supported in the fanciest possible manner -- upside down bass clef for Bach.
;;; (see transform-box for code to do this silliness correctly)


(defclass matrix-mixin ()
  ((matrix :initarg :matrix 
	   :initform nil
	   :reader matrix)))

(defclass matrix (matrix-mixin)
  ((matrix :accessor matrix)))

(defmethod matrix-p ((obj t)) nil)
(defmethod matrix-p ((obj matrix-mixin)) t)

(defmethod copy ((matrix matrix-mixin) &optional object)
  (let ((new-matrix (if (not object) (make-matrix)
		      (if (write-protected object) (copy object)
			object))))
    (if (matrix matrix) (setf (matrix new-matrix) (copy-list (matrix matrix))))
    (if (next-method-p) (call-next-method matrix new-matrix))
    new-matrix))

(defun identity-matrix-p (object)
  (let ((mat (matrix object)))
    (or (not mat)
	(equal mat '(1 0 0 1 0 0)))))

(defun invisible-matrix-p (object)
  (equal (matrix object) '(0 0 0 0 0 0)))

(defmethod descry ((matrix matrix-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(matrix" 
	    (if (not (identity-matrix-p matrix))
		(format nil "~%~A" prewhitespace)
	      ""))
	  (if (not (identity-matrix-p matrix))
	      (format nil " :matrix '~A"
		      (map 'list #'not-rational (matrix matrix)))
	    "")
	  (if (next-method-p) (call-next-method matrix stream (or controller matrix)) "")
	  (if (not controller) ")" "")))

(defun identify-matrix (object)
  (if (not (identity-matrix-p object))
      (if (invisible-matrix-p object)
	  " (scale 0 0)"
	(if (equal (matrix object) '(-1 0 0 1 0 0))
	    " mirrored"
	  (if (let ((mobj (matrix object)))
		(and (zerop (second mobj)) (zerop (third mobj)) (zerop (fifth mobj)) (zerop (sixth mobj))))
	      (format nil " (scale ~,3F ~,3F)" (first (matrix object)) (fourth (matrix object)))
	    (format nil " (matrix '~A)" (map 'list #'not-rational (matrix object))))))
    ""))



;;; patterns (at this level) are functions that are wrapped around the thing being displayed in place of the "show" or "stroke" command

(defclass pattern-mixin ()
  ((pattern :initarg :pattern :initform nil :reader pattern)))

(defclass pattern (pattern-mixin)
  ((pattern :accessor pattern)))

(defmethod copy ((pattern pattern-mixin) &optional object)
  (let ((new-pattern (if (not object) (make-pattern)
		       (if (write-protected object) (copy object)
			 object))))
    (if (pattern pattern) (setf (pattern new-pattern) (copy-list (pattern pattern))))
    (if (next-method-p) (call-next-method pattern new-pattern))
    new-pattern))

(defmethod descry ((pattern pattern-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(pattern" 
	    (if (pattern pattern) (format nil "~%~A" prewhitespace)
	      ""))
	  (if (pattern pattern)
	      (format nil " :pattern ~A" (pattern pattern))
	    "")
	  (if (next-method-p) (call-next-method pattern stream (or controller pattern)) "")
	  (if (not controller) ")" "")))

(defvar pends nil)

(defmethod display :before ((any-object pattern-mixin) container score &rest rest)
  (declare (ignore container))
  (when (and (pattern any-object) (not (member :just-fooling rest)))
    (multiple-value-bind
	(pstart tpend pend)
	(funcall (pattern any-object) score any-object)
      (declare (ignore tpend))
      (push pend pends)
      (if (not (c-output-in-progress score))
	  (g-begin-pattern score pstart t)
	(c-print (format nil " gsave ~A" pstart))))))

(defmethod display :after ((any-object pattern-mixin) container score &rest rest)
  (declare (ignore container))
  (when (and (pattern any-object) (not (member :just-fooling rest)))
    (if (not (c-output-in-progress score))
	(g-end-pattern score (pop pends))
      (c-print (format nil " ~A grestore" (pop pends))))))

(defun pattern-front (server object)
  ;; just for staff patterns in cmn4
  (if (pattern object)
      (if (c-output-in-progress server)
	  (c-print (format nil (funcall (pattern object) server object)))
	(g-begin-pattern server (funcall (pattern object) server object) nil))))




;;; maxtrix multiplication done by hand (no need for full multiply) 
;;; (third column is assumed to be 0 0 1 since we're in flatland)
;;; rather than optimize using rotate scale concat etc, I'll just use concat.

(defun %rotate-matrix (matrix angle)	;degrees
  (flet ((cosd (n) (cos (* n (/ pi 180))))
	 (sind (n) (sin (* n (/ pi 180)))))
    (let ((cs (coerce (cosd angle) 'single-float)) ;not short-float = trouble in MCL
	  (sn (coerce (sind angle) 'single-float)))
      (if (null matrix)
	  (list cs sn (- sn) cs 0 0)
	(let ((a (first matrix))
	      (b (second matrix))
	      (c (third matrix))
	      (d (fourth matrix))
	      (e (fifth matrix))
	      (f (sixth matrix)))
	  (list (+ (* a cs) (* c sn))
		(+ (* b cs) (* d sn))
		(- (* c cs) (* a sn))
		(- (* d cs) (* b sn))
		e f))))))

(defun %scale-matrix (matrix x-scale y-scale)
  (if (not matrix)
      (list x-scale 0 0 y-scale 0 0)
    (list (* x-scale (first matrix))
	  (* x-scale (second matrix))
	  (* y-scale (third matrix))
	  (* y-scale (fourth matrix))
	  (fifth matrix)
	  (sixth matrix))))

(defun %mirror-matrix (matrix)		;could be scale with -x, of course
  (if matrix
      (append (list (- (first matrix))
		    (- (second matrix)))
	      (cddr matrix))
    (list -1 0 0 1 0 0)))

(defun %truncated-matrix-multiply (m1 m2)
  (let ((a1 (first m1))  (a2 (first m2))
	(b1 (second m1)) (b2 (second m2))
	(c1 (third m1))  (c2 (third m2))
	(d1 (fourth m1)) (d2 (fourth m2))
	(e1 (fifth m1))  (e2 (fifth m2))
	(f1 (sixth m1))  (f2 (sixth m2)))
    (list (+ (* a1 a2) (* b1 c2))
	  (+ (* a1 b2) (* b1 d2))
	  (+ (* c1 a2) (* d1 c2))
	  (+ (* c1 b2) (* d1 d2))
	  (+ (* e1 a2) (* f1 c2) e2)
	  (+ (* e1 b2) (* f1 d2) f2))))


(defun transform-box (mat tv bb)
  
  ;;Transforming Axis-Aligned Bounding Boxes
  ;;by Jim Arvo
  ;;from "Graphics Gems", Academic Press, 1990
  ;;
  ;;Transforms a 3D axis-aligned box via a 3x3 matrix and a translation
  ;;vector and returns an axis-aligned box enclosing the result.
  ;;
  ;;reduced here to 2-D and reversed sense of rotation angle
  ;;(original was clockwise, but PostScript is counterclockwise) 
  
  (let* ((amin (make-array 2 :element-type 'short-float))
	 (amax (make-array 2 :element-type 'short-float))
	 (bmin (make-array 2 :element-type 'short-float))
	 (bmax (make-array 2 :element-type 'short-float))
	 (a 0.0) (b 0.0))
    
    (setf (aref amin 0) (float (first bb)))
    (setf (aref amin 1) (float (second bb)))
    
    (setf (aref amax 0) (float (third bb)))
    (setf (aref amax 1) (float (fourth bb)))
    
    (setf (aref bmin 0) (float (first tv)))
    (setf (aref bmin 1) (float (second tv)))
    
    (setf (aref bmax 0) (float (first tv)))
    (setf (aref bmax 1) (float (second tv)))
    
    (loop for i from 0 to 1 do
      (loop for j from 0 to 1 do
	(setf a (* (aref amin j) (nth (+ (* j 2) i) mat)))
	(setf b (* (aref amax j) (nth (+ (* j 2) i) mat)))
	(if (< a b)
	    (progn
	      (incf (aref bmin i) a)
	      (incf (aref bmax i) b))
	  (progn
	    (incf (aref bmin i) b)
	    (incf (aref bmax i) a)))))
    (list (aref bmin 0) (aref bmin 1)
	  (aref bmax 0) (aref bmax 1))))




;;;
;;; ----------------    glyphs (font characters)
;;;

(defclass write-protected-marks-mixin ()
  ((marks :initarg :marks :initform nil :reader marks)))

(defclass marks-mixin (write-protected-marks-mixin)
  ((marks :accessor marks)))

(defmethod marks-p ((obj t)) nil)
(defmethod marks-p ((obj marks-mixin)) t)

(defmethod descry ((vis write-protected-marks-mixin) &optional stream controller)
  (format stream "~A~A"
	  (if (marks vis) 
	      (format stream "~%~A :marks (list~{ ~A~})"
		      prewhitespace
		      (loop for mark in (marks vis) collect (descry mark stream nil)))
	    "")
	  (if (next-method-p) (call-next-method vis stream (or controller vis)) "")))

(defmethod copy ((vis write-protected-marks-mixin) &optional object)
  (let ((new-vis (if (not object) (make-marks-mixin)
		   (if (write-protected object) (copy object)
		     object))))
    (if (marks vis) (setf (marks new-vis) (loop for mark in (marks vis) collect (copy mark))))
    (if (next-method-p) (call-next-method vis new-vis))
    new-vis))

(defun display-marks (object score &rest rest)
  (if (marks object) 
      (loop for mark in (marks object) do 
	(if (or (not (sundry-p mark))
		(not (eq (sundry-name mark) :slur)))
	    (apply #'display mark object score rest)))))

(defmethod add-to-marks ((anything write-protected-marks-mixin) objects) 
  (setf (marks anything) (append (marks anything) objects)))

(defun edit-mark (anything mark-identifier mark-editor)
  (loop for m in (marks anything) do
    (if (funcall mark-identifier m anything)
	(funcall mark-editor m anything))))

(defun identify-marks (object &optional file)
  (if (marks object)
      (format file "~{ ~A~}" 
	      (loop for mark in (marks object)
	       collect (identify mark)))
    ""))




(defclass visible-mixin (write-protected-marks-mixin box-mixin pattern-mixin matrix-mixin)
  ((justification :initform nil :initarg :justification :reader visible-justification)
   (visible-section :initform nil :initarg :visible-section :reader visible-section)))

(defclass visible (visible-mixin marks-mixin box pattern matrix)
  ((justification :accessor visible-justification :accessor justification)
   (visible-section :accessor visible-section)))

(defmethod visible-p ((obj t)) nil)
(defmethod visible-p ((obj visible-mixin)) t)

(defmethod descry ((vis visible-mixin) &optional stream controller)
  (format stream "~A~A~A"
	  (if (visible-justification vis) 
	      (format stream " :justification :~(~A~)" (visible-justification vis))
	    "")
 	  (if (visible-section vis) 
 	      (format stream " :section ~A" (visible-section vis))
 	    "")
	  (if (next-method-p) (call-next-method vis stream (or controller vis)) "")))

(defmethod copy ((vis visible-mixin) &optional object)
  (let ((new-vis (if (not object) 
		     (cmn-error "attempt to copy a bare instance of the visible-mixin class")
		   (if (write-protected object) (copy object)
		     object))))
    (setf (visible-justification new-vis) (visible-justification vis) )
    (setf (visible-section new-vis) (visible-section vis))
    (if (next-method-p) (call-next-method vis new-vis))
    new-vis))

(defun identify-visible (object &optional file)
  (if (visible-justification object)
      (if (eq (visible-justification object) :none)
	  (format file " unjustified")
	(format file " (justification :~(~A~))" (visible-justification object)))
    ""))



(defclass glyph-mixin (visible-mixin)
  ((rx :initarg :rx :initform 0 :reader rx :reader g-rx)
   (index :initarg :index :initform nil :reader index)))

(defclass glyph (glyph-mixin visible)
  ((rx :accessor rx)
   (index :accessor index)))

(defclass write-protected-glyph (write-protect glyph-mixin)
  ())

(defmethod glyph-p ((obj t)) nil)
(defmethod glyph-p ((obj glyph-mixin)) t)

(defmethod copy ((glyph glyph-mixin) &optional object)
  (let ((new-glyph (if (not object) (make-glyph)
		     (if (write-protected object) (copy object)
		       object))))
    (setf (rx new-glyph) (g-rx glyph))
    (setf (index new-glyph) (index glyph))
    (if (next-method-p) (call-next-method glyph new-glyph))
    new-glyph))

(defmethod descry ((glyph glyph-mixin) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(glyph" 
	    (if (index glyph) (format nil "~%~A" prewhitespace)
	      ""))
	  (if (index glyph)
	      (format nil " :index ~A :rx ~A" (index glyph) (not-rational (rx glyph)))
	    "")
	  (if (next-method-p) (call-next-method glyph stream (or controller glyph)) "")
	  (if (not controller) ")" "")))

(defun glyph (&rest args)
  (let ((glf (make-glyph))) 
    (loop for act in args do
      (when act
	(if (glyph-p act)
	    (progn
	      (setf (index glf) (index act))
	      (setf (rx glf) (rx act)))
	  (if (self-acting-p act)
	      (funcall (action act) glf (arguments act))
	    (if (integerp act)
		(setf (index glf) act)
	      (if (characterp act)
		  (setf (index glf) (decimal-to-octal (char-code act)))))))))
    glf))



(defclass font-mixin ()
  ((font-size :initarg :font-size :initform nil :accessor font-size)
   (font-name :initarg :font-name :initform nil :accessor font-name)
   (font-scaler :initarg :font-scaler :initform 1.0 :accessor font-scaler)))

(defmethod descry ((fnt font-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A~A"
	  (if (not controller) "(font" "")
	  (if (font-name fnt) (format nil " :font-name ~A" (font-name fnt)) "")
	  (if (font-size fnt) (format nil " :font-size ~A" (font-size fnt)) "")
	  (if (/= (font-scaler fnt) 1.0) (format nil " :font-scaler ~A" (font-scaler fnt)) "")
	  (if (next-method-p) (call-next-method fnt stream (or controller fnt)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((fnt font-mixin) &optional object)
  (let ((new-fnt (if (not object) (make-font-mixin)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (font-size new-fnt) (font-size fnt))
    (setf (font-name new-fnt) (font-name fnt))
    (setf (font-scaler new-fnt) (font-scaler fnt))
    (if (next-method-p) (call-next-method fnt new-fnt))
    new-fnt))




(defclass dxy-mixin ()
  ((dx :initarg :dx :initform 0 :reader dx :reader dxy-dx)
   (dy :initarg :dy :initform 0 :reader dy :reader dxy-dy)))

(defclass dxy (dxy-mixin)
  ((dx :accessor dx :accessor dxy-dx)
   (dy :accessor dy :accessor dxy-dy)))

(defmethod dxy-p ((obj t)) nil)
(defmethod dxy-p ((obj dxy-mixin)) t)

(defmethod descry ((dxy dxy-mixin) &optional stream controller)
  (format stream "~A~A~A" 
	  (if (not (zerop (dx dxy))) (format nil " :dx ~,3F" (dx dxy)) "")
	  (if (dy dxy)
	      (if (listp (dy dxy))
		  (format nil " :dy '~A " (dy dxy))
		(if (not (zerop (dy dxy))) 
		    (format nil " :dy ~,3F" (dy dxy)) 
		  ""))
	    "")
	  (if (next-method-p) (call-next-method dxy stream (or controller dxy)) "")))

(defmethod copy ((dxy dxy-mixin) &optional object)
  (let ((new-dxy (if (not object) (make-dxy-mixin)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (dxy-dx new-dxy) (dxy-dx dxy))
    (setf (dxy-dy new-dxy) (dxy-dy dxy))
    (if (next-method-p) (call-next-method dxy new-dxy))
    new-dxy))

(defun identify-dxy (object)
  (if (or (not (zerop (dx object)))
	  (not (zerop (dy object))))
      (format nil "~A~A" 
	      (if (not (zerop (dx object))) (format nil " (dx ~,3F)" (dx object)) "")
	      (if (not (zerop (dy object))) (format nil " (dy ~,3F)" (dy object)) ""))
    ""))


(defclass text (dxy font-mixin visible)
  ((letters :initarg :letters :initform nil :accessor letters)
   (y :initarg :y :initform nil :accessor y :accessor text-y)
   (x :initarg :x :initform nil :accessor x :accessor text-x)
   (font-name :initform "Times-Roman")))

(defmethod text-p ((obj t)) nil)
(defmethod text-p ((obj text)) t)

(defmethod descry ((text text) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(text" "")
	  (if (letters text) (format nil " :letters ~A" (letters text)) "")
	  (if (next-method-p) (call-next-method text stream (or controller text)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((text text) &optional object)
  (let ((new-text (if (not object) (make-text)
		    (if (write-protected object) (copy object)
		      object))))
    (setf (letters new-text) (letters text))
    (setf (text-x new-text) (text-x text))
    (setf (text-y new-text) (text-y text))
    (if (next-method-p) (call-next-method text new-text))
    new-text))




(defgeneric %rotate (object angle))
(defgeneric %scale (object xscl yscl))
(defgeneric %mirror (object &rest rest) )
(defgeneric %transform (object matrix))

(defmethod %rotate ((object matrix) angle)
  (setf (matrix object) (%rotate-matrix (or (matrix object) '(1 0 0 1 0 0)) angle))
  object)

(defmethod %scale ((object matrix) xscl yscl)
  (setf (matrix object) (%scale-matrix (matrix object) (or xscl 1) (or yscl 1)))
  object)

(defmethod %scale-1 ((object matrix) scls)
  (setf (matrix object) (%scale-matrix (matrix object) (or (first scls) 1) (or (second scls) 1)))
  object)

(defmethod %mirror ((object matrix) &rest rest)
  (declare (ignore rest))
  (setf (matrix object) (%mirror-matrix (matrix object)))
  object)

(defmethod %transform ((object matrix) matrix)
  (setf (matrix object) (%truncated-matrix-multiply (or (matrix object) '(1 0 0 1 0 0)) matrix))
  object)

(defmethod %rotate ((object write-protect) angle)
  (let ((new-object (copy object)))
    (setf (matrix new-object) (%rotate-matrix (or (matrix new-object) '(1 0 0 1 0 0)) angle))
    new-object))

(defmethod %scale ((object write-protect) xscl yscl)
  (let ((new-object (copy object)))
    (setf (matrix new-object) (%scale-matrix (matrix new-object) (or xscl 1) (or yscl 1)))
    new-object))

(defmethod %mirror ((object write-protect) &rest rest)
  (declare (ignore rest))
  (let ((new-object (copy object)))
    (setf (matrix new-object) (%mirror-matrix (matrix new-object)))
    new-object))

(defmethod %transform ((object write-protect) matrix)
  (let ((new-object (copy object)))
    (%truncated-matrix-multiply (or (matrix new-object) '(1 0 0 1 0 0))  matrix)
    new-object))



;;;
;;; ----------------    the graphics server
;;;
;;;   lineto moveto rlineto rmoveto show
;;;   initialize header footer finalize
;;;   comment line-width draw fill-in
;;;   rotate scale mirror 
;;;   x y (current drawing position)

(defgeneric lineto (server x y &key))	;draw a line to (x y)
(defgeneric rlineto (server x y &key))	;draw a line to (current-x + x, current-y + y)
(defgeneric moveto (server x y))	;hop to (x y)
(defgeneric rmoveto (server x y))	;hop to (current-x + x, current-y + y)
(defgeneric curveto (server x0 y0 x1 y1 x2 y2)) ;draw a curve 
(defgeneric show (server x &key))	;cause "x" to be displayed
(defgeneric initialize (server &key))	;prepare a clean slate
(defgeneric header (server &key))	;whatever setup commands are needed by the server
(defgeneric footer (server &key))	;whatever shut-down commands are needed
(defgeneric finalize (server &key))	;finish graphics
(defgeneric comment (server x))		;put a comment in the server command stream (optional)
(defgeneric draw (server))		;draw the current path
(defgeneric fill-in (server &key))	;fill the current object

;;; it is my belief that all the system-specific graphics code is in cmn-grfx, and the other files
;;; build on whatever is here, so only that file needs to be changed to move to a different 
;;; graphics environment.  See rueful comment in cmn-grfx about big plans gone awry...

(defvar *cmn-score-size* nil)
(defvar *old-cmn-score-size* nil)

(defclass server (box-mixin matrix)
  ((x :initarg :x :initform nil :accessor x :accessor scr-x) ;current drawing position
   (y :initarg :y :initform nil :accessor y :accessor scr-y)
   (size :initarg :size :initarg size :accessor size :accessor scr-size :initform 1)
   (scaling-matrix :initarg :scaling-matrix :accessor scaling-matrix :initform nil)
   (output :initform nil :initarg :output :accessor output) ;output command stream
   (c-output-in-progress :initform nil :accessor c-output-in-progress)
   ;; this slot is t if C format calls are in progress (much faster than Lisp's format)
   (output-file :initform nil :initarg :output-file :initarg output-file :accessor output-file)
   (line-width :initform 0 :initarg :line-width :accessor line-width)
   (font :initform nil :initarg :font :accessor font)
   (prologue :initarg :prologue :initarg prologue :initform nil :accessor prologue)
   (bounded :initform nil :initarg :bounded :accessor bounded) ;is server supposed to track the excursion religiously.
   (output-type :initform :postscript :initarg :output-type :accessor output-type)
   (output-state :initform nil :initarg nil :accessor output-state)
   (x0 :accessor scr-x0 :type number)	; in ACL, by giving a unique name to the accessor, slot accesses are sped up by a factor of 10
   (y0 :accessor scr-y0 :type number)
   (x1 :accessor scr-x1 :type number)
   (y1 :accessor scr-y1 :type number)))


(defmethod copy ((server server) &optional object)
  (declare (ignore object))
  (error "can't copy ~A" server))

(defmethod descry ((server server) &optional stream controller)
  (format stream "~A~A~A~A~A~A~A~A~A~A~A"
	  (if (not controller) "(server" 
	    (format nil "~%    "))
	  (if (x server) (format nil " :x ~A" (x server)) "")
	  (if (y server) (format nil " :y ~A" (y server)) "")
	  (if (size server) (format nil " :size ~A" (size server)) "")
	  (if (output server) (format nil " :output-file ~A" (output-file server)) "")
	  (format nil " (~(:~A~))" (output-type server))
	  (if (not (zerop (line-width server))) (format nil " :line-width ~A" (line-width server)) "")
	  (if (font server) (format nil " :font ~A" (font server)) "")
	  (if (bounded server) (format nil " :bounded ~A" (bounded server)) "")
	  (if (next-method-p) (call-next-method server stream (or controller server)) "")
	  (if (not controller) ")" "")))

(defmethod clear-bounding-box ((server server))
  (setf (scr-x0 server) 1000)
  (setf (scr-x1 server) 0)
  (setf (scr-y0 server) 1000)
  (setf (scr-y1 server) 0))

(defgeneric bounds (server))
(defgeneric (setf bounds) (box server))

(defmethod %update-bounds ((server server))
  (declare (optimize (speed 3) (safety 0)))
  (setf (scr-x0 server) (min (scr-x0 server) (scr-x server)))
  (setf (scr-y0 server) (min (scr-y0 server) (scr-y server)))
  (setf (scr-x1 server) (max (scr-x1 server) (scr-x server)))
  (setf (scr-y1 server) (max (scr-y1 server) (scr-y server))))

(defmethod %xy ((server server) x y) 
  (setf (scr-x server) x)
  (setf (scr-y server) y))

(defmethod %dxy ((server server) dx dy)
  (incf (scr-x server) dx)
  (incf (scr-y server) dy))

(defmethod bounds ((server server))
  (list (scr-x0 server) (scr-y0 server) (scr-x1 server) (scr-y1 server)))

(defmethod (setf bounds) (box (server server)) ;default here is a list passed with box coordinates
  (declare (optimize (speed 3) (safety 0)))
  (when (not (bounded server))
    (setf (scr-x0 server) (min (scr-x0 server) (first box)))
    (setf (scr-y0 server) (min (scr-y0 server) (second box)))
    (setf (scr-x1 server) (max (scr-x1 server) (third box)))
    (setf (scr-y1 server) (max (scr-y1 server) (fourth box)))))

(defmethod (setf bounds) ((box box) (server server))
  (setf (bounds server) (list (box-x0 box) (box-y0 box) (box-x1 box) (box-y1 box))))

(defvar c-args (make-array 6 :element-type 'short-float :initial-element 0.0))

(defmethod lineto ((server server) x y &key pattern)
  (declare (optimize (speed 3) (safety 0)))
  (let* ((size (or *cmn-score-size* (scr-size server)))
	 (xs (float (* x size)))
	 (ys (float (* y size))))
    (if (output server)	
	(if (not (c-output-in-progress server))
	    (if pattern
		(g-dashed-line server xs ys pattern nil)
	      (g-lineto server xs ys))
	  (progn 
	    #-clisp (setf (aref c-args 0) xs) 
	    #-clisp (setf (aref c-args 1) ys)
	    #-clisp (c-lineto c-args)
	    #+clisp (c-lineto xs ys)
	    (if pattern (c-print (format nil " [~{~A ~}] 0 setdash stroke [] 0 setdash" (map 'list #'not-rational pattern)))))))
    (setf (scr-x server) xs)
    (setf (scr-y server) ys)
    (setf (scr-x0 server) (min (scr-x0 server) xs))
    (setf (scr-y0 server) (min (scr-y0 server) ys))
    (setf (scr-x1 server) (max (scr-x1 server) xs))
    (setf (scr-y1 server) (max (scr-y1 server) ys))))

(defmethod rlineto ((server server) dx dy &key pattern)
  (declare (optimize (speed 3) (safety 0)))
  (let* ((size (or *cmn-score-size* (scr-size server)))
	 (dxs (float (* dx size)))
	 (dys (float (* dy size))))
    (if (output server)	
	(if (not (c-output-in-progress server))
	    (if pattern
		(g-dashed-line server dxs dys pattern t)
	      (g-rlineto server dxs dys))
	  (progn 
	    #-clisp (setf (aref c-args 0) dxs) 
	    #-clisp (setf (aref c-args 1) dys)
	    #-clisp (c-rlineto c-args)
	    #+clisp (c-rlineto dxs dys)
	    (if pattern (c-print (format nil " [~{~A ~}] 0 setdash stroke [] 0 setdash" (map 'list #'not-rational pattern)))))))
    (setf dxs (incf (scr-x server) dxs))
    (setf dys (incf (scr-y server) dys))
    (setf (scr-x0 server) (min (scr-x0 server) dxs))
    (setf (scr-y0 server) (min (scr-y0 server) dys))
    (setf (scr-x1 server) (max (scr-x1 server) dxs))
    (setf (scr-y1 server) (max (scr-y1 server) dys))))

(defmethod moveto ((server server) x y)
  ;; most of the run time is spent here -- all in Lisp's format function
  (declare (optimize (speed 3) (safety 0)))
  (let* ((size (or *cmn-score-size* (scr-size server)))
	 (xs (float (* x size)))
	 (ys (float (* y size))))
    (if (output server)	
	(if (not (c-output-in-progress server))
	    (g-moveto server xs ys)
	  (progn
	    #+clisp (c-moveto xs ys)
	    #-clisp (setf (aref c-args 0) xs) 
	    #-clisp (setf (aref c-args 1) ys)
	    #-clisp (c-moveto c-args)
	    )))
    (setf (scr-x server) xs)
    (setf (scr-y server) ys)
    (setf (scr-x0 server) (min (scr-x0 server) xs))
    (setf (scr-y0 server) (min (scr-y0 server) ys))
    (setf (scr-x1 server) (max (scr-x1 server) xs))
    (setf (scr-y1 server) (max (scr-y1 server) ys))))

(defmethod rmoveto ((server server) dx dy)
  (declare (optimize (speed 3) (safety 0)))
  (let* ((size (or *cmn-score-size* (scr-size server)))
	 (dxs (float (* dx size)))
	 (dys (float (* dy size))))
    (if (output server) 
	(if (not (c-output-in-progress server))
	    (g-rmoveto server dxs dys)
	  (progn 
	    #+clisp (c-rmoveto dxs dys)
	    #-clisp (setf (aref c-args 0) dxs) 
	    #-clisp (setf (aref c-args 1) dys)
	    #-clisp (c-rmoveto c-args)
	    )))
    (setf dxs (incf (scr-x server) dxs))
    (setf dys (incf (scr-y server) dys))
    (setf (scr-x0 server) (min (scr-x0 server) dxs))
    (setf (scr-y0 server) (min (scr-y0 server) dys))
    (setf (scr-x1 server) (max (scr-x1 server) dxs))
    (setf (scr-y1 server) (max (scr-y1 server) dys))))

(defmethod comment ((server server) x)
  (if (output server) 
      (if (not (c-output-in-progress server))
	  (g-comment server x)
	(c-print (format nil "% ~A" x)))))

(defmethod (setf line-width) :after (x (server server))
  (if (output server) 
      (if (not (c-output-in-progress server))
	  (g-set-line-width server (* x (or *cmn-score-size* (scr-size server))))
	(if (zerop x)
	    (c-print " 0 setlinewidth")
	  (c-print (format nil " ~A setlinewidth" (not-rational (* x (or *cmn-score-size* (scr-size server))))))))))

(defmethod draw ((server server))
  (if (output server)
      (if (not (c-output-in-progress server))
	  (g-draw server)
	(c-print " stroke"))))

(defgeneric %rx (server object &key) )

(defmethod %rx ((server server) (glyph glyph-mixin) &key (count 1) size &allow-other-keys)
  (if (identity-matrix-p glyph)
      (* (g-rx glyph) count (or size *cmn-score-size* (scr-size server)))
    0))					;if matrix then we used gsave/grestore and the outer point did not move

(defmethod simple-show ((server server) glf)
  ;; this is only used when we know what the glf is (as write-protected) and nothing is happening
  (when (output server)
    (if (not (c-output-in-progress server))
	(g-show server (index glf))
      (c-glyph (index glf))))
  (let ((dxs (* (g-rx glf) (or *cmn-score-size* (scr-size server)))))
    (setf dxs (incf (scr-x server) dxs))
    (setf (scr-x0 server) (min (scr-x0 server) dxs))
    (setf (scr-x1 server) (max (scr-x1 server) dxs))))

(defmethod show ((server server) (glyph glyph-mixin) &key (count 1) size &allow-other-keys)
  (when (not (invisible-matrix-p glyph))
    (let ((all-done nil))
      (when (output server)
	(let ((c-out (c-output-in-progress server))
	      (pat (pattern glyph))
	      (no-mat (identity-matrix-p glyph)))
	  (if (and c-out no-mat (= count 1) (not pat)) ;optimize the most common cases
	      (if size
		  (c-font-glyph (font server) (floor size) (index glyph))
		(progn
		  (setf all-done t)
		  (simple-show server glyph)))
	    (let ((glyphs (loop for i from 1 to count collect (index glyph)))
		  (pstart "")
		  (pend "show"))
	      (when size 
		(if (not c-out)
		    (g-new-font server (font server) size t)
		  (c-print (format nil " gsave~% /~A findfont ~D scalefont setfont" (font server) size))))
	      (if pat
		  (multiple-value-bind
		      (qpstart qpend) 
		      (funcall pat server glyph)
		    (if qpstart (setf pstart qpstart))
		    (if qpend (setf pend qpend))))
	      (if no-mat
		  (if (not c-out)
		      (g-glyphs server pstart glyphs pend size)
		    (c-print (format nil " ~A(~{\\~D~}) ~A~A" pstart glyphs pend (if size " grestore" ""))))
		(if (not c-out)
		    (g-scaled-glyphs server pstart glyphs pend (not size) (matrix glyph))
		  (c-print (format nil " ~A~A(~{\\~D~}) [ ~{~A ~}] concat ~A grestore"
				   (if (not size) "gsave " "")
				   pstart glyphs (map 'list #'not-rational (matrix glyph)) pend))))))))
      (if (not all-done)
	  (let* ((size-server (or *cmn-score-size* (scr-size server)))
		 (x0g (box-x0 glyph))
		 (x1g (box-x1 glyph))
		 (rxg (g-rx glyph))
		 (xs (scr-x server))
		 (ys (scr-y server))
		 (orig-box (list 0
				 (* size-server (box-y0 glyph))
				 (* (or size size-server) 
				    (* count (if (zerop rxg)
						 (if (< x0g x1g)
						     (- x1g x0g)
						   x1g)
					       rxg)))
				 (* (or size size-server) (box-y1 glyph))))
		 (new-box (if (identity-matrix-p glyph)
			      (list (+ xs (first orig-box))
				    (+ ys (second orig-box))
				    (+ xs (third orig-box))
				    (+ ys (fourth orig-box)))
			    (transform-box (matrix glyph) (list xs ys) orig-box))))
	    (setf (bounds server) new-box)
	    (if (identity-matrix-p glyph) 
		(%dxy server (%rx server glyph :count count :size size) 0)))))))

(defmacro with-scaling (server size x0 y0 &body body)
  `(let ((current-box (bounds ,server))
	 (current-scaling-matrix (scaling-matrix ,server))
	 (new-matrix (list ,size 0 0 ,size ,x0 ,y0)))
     ;; bounding-box for duration of this scaling is scaled both ways by size and translated by x0 y0
     (clear-bounding-box ,server)
     (setf (scaling-matrix ,server) 
       (%truncated-matrix-multiply (or current-scaling-matrix 
				       (list 1 0 0 1 0 0))
				   new-matrix))
     (matrix-front ,server new-matrix)
     ,@body
     ;; I originally had ,.body here, but in ACL that sometimes gets confused and repeats the last statement over and over
     (g-save-server-bounds ,server)
     (setf (x0 ,server) (min (first current-box) (+ (* ,size (x0 ,server)) ,x0)))
     (setf (y0 ,server) (min (second current-box) (+ (* ,size (y0 ,server)) ,y0)))
     (setf (x1 ,server) (max (third current-box) (+ (* ,size (x1 ,server)) ,x0)))
     (setf (y1 ,server) (max (fourth current-box) (+ (* ,size (y1 ,server)) ,y0)))
     ;; should we fix up (x/y server) here?
     (setf (scaling-matrix ,server) current-scaling-matrix)
     (matrix-back ,server new-matrix (list ,x0 ,y0))))

(defmacro with-pattern (server pattern &body body)
  `(multiple-value-bind
      (pstart pend)
    (funcall (pattern ,pattern) ,server ,pattern)
    (if (not (c-output-in-progress ,server))
	(g-begin-pattern ,server pstart t)
      (c-print (format nil " gsave~% ~A~%" pstart)))
    ,@body
    (if (not (c-output-in-progress ,server))
	(g-end-pattern ,server pend)
      (c-print (format nil " ~A grestore" pend)))))

(defun matrix-front (server matrix &optional (saved t))
  (if (and matrix
	   (not (equal matrix '(1 0 0 1 0 0))))
      (if (not (c-output-in-progress server))
	  (g-begin-coordinate-transformation server matrix saved)
	(c-print (format nil " ~A[ ~{~A ~}] concat~%" (if saved " gsave " "") (map 'list #'not-rational matrix))))))

(defun matrix-back (server matrix &optional vector)
  (if (and matrix
	   (not (equal matrix '(1 0 0 1 0 0))))
      (if (not (c-output-in-progress server))
	  (g-end-coordinate-transformation server matrix t vector)
	(c-print " grestore"))))

(defmacro with-transformation (server matrix x0 y0 &body body)
  `(let ((matr (copy-list (if (matrix-p ,matrix) (matrix ,matrix) ,matrix)))
	 (current-box (bounds ,server))
	 (current-scaling-matrix (scaling-matrix ,server)))
     ;; here the problem is similar to the with-scaling case, but we need to transform the
     ;; resultant box by the transforming matrix before accumulating the edges into the box
     (clear-bounding-box ,server)
     (setf (fifth matr) ,x0)
     (setf (sixth matr) ,y0)
     (setf (scaling-matrix ,server) 
       (%truncated-matrix-multiply (or current-scaling-matrix 
				       (list 1 0 0 1 0 0))
				   matr))
     (matrix-front ,server matr)
     ,@body
     (let ((new-box (transform-box matr (list ,x0 ,y0) (bounds ,server))))
       (g-save-server-bounds ,server)
       (setf (x0 ,server) (min (first current-box) (first new-box)))
       (setf (y0 ,server) (min (second current-box) (second new-box)))
       (setf (x1 ,server) (max (third current-box) (third new-box)))
       (setf (y1 ,server) (max (fourth current-box) (fourth new-box))))
     (setf (scaling-matrix ,server) current-scaling-matrix)
     (matrix-back ,server matr (list ,x0 ,y0))))

;;; in Postscript, if the string to be printed contains "(" or ")" the
;;;   only safe way to deal with them is to prepend the escape character "\"
(defun ps-letters (letters)
  (let ((trouble (or (find #\( letters)
		     (find #\) letters))))
    (if trouble
	(let* ((baddies (+ (count #\( letters) (count #\) letters)))
	       (inlen (length letters))
	       (i -1)
	       (strlen (+ inlen baddies))
	       (newstr (make-string strlen)))
	  (loop for k from 0 below inlen do
	    (let ((kc (elt letters k)))
	      (if (or (char-equal #\( kc)
		      (char-equal #\) kc))
		  (setf (elt newstr (incf i)) #\\))
	      (setf (elt newstr (incf i)) kc)))
	  newstr)
      letters)))

(defmethod show ((server server) (text text) &key (dx 0) (dy 0) font size scaler (save :both) &allow-other-keys)
  (when (not (invisible-matrix-p text))
    (let* ((lfont (or font (font-name text)))
	   (ssize (or *cmn-score-size* (scr-size server)))
	   (lsize (or size (font-size text) (* (or scaler (font-scaler text)) ssize)))
	   (pstart "")
	   (update-xy t)
	   (zxy (and (zerop dx) (zerop dy)))
	   (pend (if zxy "show" "ashow")))

      (when (output server)
	(let ((c-out (c-output-in-progress server)))
	  (if lfont
	      (if (< lsize 1.0)
		  (if (and (font-size text)
			   (< (font-size text) 1.0))
		      (warn "(text ~S: font-size = ~,3F which will be invisible)" 
			    (letters text) (font-size text))
		    (warn "(text ~S: font-scaler = ~,3F so font-size = ~,3F which will be invisible)" 
			  (letters text) (font-scaler text) lsize))))
	  (if (and c-out zxy lfont (not (matrix text)) (not (pattern text)) (eq save :both))
	      ;; optimize most common case
	      (c-text lfont (round lsize) (ps-letters (letters text)))
	    (progn
	      (when (or lfont (matrix text)) 
		(if (member save '(:both :start)) 
		    (if c-out 
			(c-print " gsave") 
		      (g-save server)))
		(if lfont 
		    (if c-out
			(c-print (format nil " /~A findfont ~D scalefont setfont" lfont lsize))
		      (g-new-font server lfont lsize nil)))
		(matrix-front server (matrix text) nil))
	      (if (pattern text)
		  (multiple-value-bind
		      (qpstart qpend) 
		      (funcall (pattern text) server text)
		    (if qpstart (setf pstart qpstart))
		    (if qpend (setf pend qpend))))
	      (if (or zxy
		      (and pstart (not (string-equal pend "ashow"))))
		  (if c-out
		      (c-print (format nil " ~A(~A) ~A" pstart (ps-letters (letters text)) pend))
		    (g-patterned-text server pstart (letters text) pend))
		(if c-out
		    (c-print (format nil " ~A ~,3F ~,3F (~A) ashow" pstart (* dx ssize) (* dy ssize) (ps-letters (letters text))))
		  (g-patterned-text-with-spacing server pstart (letters text) (* dx ssize) (* dy ssize))))
	      (if (and (member save '(:both :finish)) 
		       (or lfont (matrix text))) 
		  (progn
		    (setf update-xy nil)
		    (if c-out
			(c-print " grestore")
		      (g-restore server))))))))
      (if update-xy (%dxy server (%rx server text :dx dx :size lsize :scaler scaler) 0)) ;was lsize for Y?? 17-Feb-94
      (%update-bounds server))))


(defmethod %rx ((server server) (text text) &key (dx 0) size scaler &allow-other-keys)
  (if (identity-matrix-p text)
      (* (length (letters text)) (+ .4 dx) (or scaler 1.0) (or size *cmn-score-size* (scr-size server)))
    0))


(defmethod curveto ((server server) x0 y0 x1 y1 x2 y2)
  (declare (optimize (speed 3) (safety 0)))
  (let* ((size (or *cmn-score-size* (scr-size server)))
	 (size-x0 (float (* size x0)))
	 (size-y0 (float (* size y0)))
	 (size-x1 (float (* size x1)))
	 (size-y1 (float (* size y1)))
	 (size-x2 (float (* size x2)))
	 (size-y2 (float (* size y2)))
	 (wild-guess-y (* .5 (+ size-y0 size-y1))))
    ;; this is just a stab at where the y min/max occurred, given that this is a slur or something like one.
    (if (output server)
	(if (c-output-in-progress server)
	    (progn
	      #+clisp (c-curveto size-x0 size-y0 size-x1 size-y1 size-x2 size-y2)
	      #-clisp (setf (aref c-args 0) size-x0)
	      #-clisp (setf (aref c-args 1) size-y0)
	      #-clisp (setf (aref c-args 2) size-x1)
	      #-clisp (setf (aref c-args 3) size-y1)
	      #-clisp (setf (aref c-args 4) size-x2)
	      #-clisp (setf (aref c-args 5) size-y2)
	      #-clisp (c-curveto c-args)
	      )
	  (g-curveto server size-x0 size-y0 size-x1 size-y1 size-x2 size-y2)))
    (setf (scr-x server) size-x2)
    (setf (scr-y server) size-y2)
    (setf (scr-x0 server) (min (scr-x0 server) size-x2))
    (setf (scr-y0 server) (min (scr-y0 server) size-y2 wild-guess-y))
    (setf (scr-x1 server) (max (scr-x1 server) size-x2))
    (setf (scr-y1 server) (max (scr-y1 server) size-y2 wild-guess-y))))



;;; here is the PostScript Bezier curve (curveto above) done by hand:
;;; (n is how accurate a curve we draw)

(defun bezier (score x0 y0 x1 y1 x2 y2 x3 y3 n)
  (let* ((cx (* 3 (- x1 x0)))
	 (cy (* 3 (- y1 y0)))
	 (bx (- (* 3 (- x2 x1)) cx))
	 (by (- (* 3 (- y2 y1)) cy))
	 (ax (- x3 x0 cx bx))
	 (ay (- y3 y0 cy by))
	 (incr (/ 1.0 n)))
    (moveto score x0 y0)
    (loop for i from 0 to 1 by incr do
      (lineto score (+ x0 (* i (+ cx (* i (+ bx (* i ax))))))
	      (+ y0 (* i (+ cy (* i (+ by (* i ay))))))))
    (draw score)))

(defun solve-bezier (x x0 y0 y1 y2 x3 y3)
  (let* ((cy (* 3 (- y1 y0)))
	 (by (- (* 3 (- y2 y1)) cy))
	 (ay (- y3 y0 cy by))
	 (i (divide (- x x0) (- x3 x0))))
    (+ y0 (* i (+ cy (* i (+ by (* i ay))))))))



(defmethod fill-in ((server server) &key rectangle closepath path)
  (if (output server)
      (let ((cmd (if rectangle (if (= (length path) 4) 
				   (if (= PS-level 1) "RF" "rectfill")
				 (error "internal cmn error -- rectangle of ~D corners?" (length path)))
		   (if closepath "closepath fill"
		     "fill")))
	    (c-out (c-output-in-progress server)))
	(if path
	    (let ((size (or *cmn-score-size* (scr-size server))))
	      (if c-out
		  (if (= (length path) 4)
		      (progn
			#+clisp (c-fill-in (float (* size (first path))) 
					   (float (* size (second path)))
					   (float (* size (third path)))
					   (float (* size (fourth path)))
					   cmd)
			#-clisp (setf (aref c-args 0) (float (* size (first path))))
			#-clisp (setf (aref c-args 1) (float (* size (second path))))
			#-clisp (setf (aref c-args 2) (float (* size (third path))))
			#-clisp (setf (aref c-args 3) (float (* size (fourth path))))
			#-clisp (c-fill-in c-args cmd)
			)
		    (c-print (format nil " ~{~,2F ~,2F lineto ~}~A" (map 'list #'(lambda (n) (float (* n size))) path) cmd)))
		(if rectangle
		    (g-filled-rectangle server 
					(float (* size (first path))) (float (* size (second path)))
					(float (* size (third path))) (float (* size (fourth path))))
		  (g-filled-polygon server path size cmd closepath))))
	  (if c-out
	      (c-print cmd)
	    (g-fill server cmd closepath))))))

(defmethod circle ((score server) x y r &optional (ang1 0) (ang2 360) fill)
  (let* ((size (scr-size score)))
    (if (output score)
	(if (c-output-in-progress score)
	    (c-print (format nil " ~,3F ~,3F ~D ~D ~D newpath arc~A" (float (* size x)) (float (* size y))
			     (float (* size r))	ang1 ang2
			     (if fill " fill" "")))
	  (g-arc score (float (* size x)) (float (* size y)) (float (* size r)) ang1 ang2 fill)))
    (setf (bounds score) (list (* (- x r) size)
			       (* (- y r) size)
			       (* (+ x r) size)
			       (* (+ y r) size)))))

(defmethod optimize-stem ((score server) x0 y0 y1 width)
  (let* ((ssize (scr-size score))
	 (x0s (float (* ssize x0)))
	 (y0s (float (* ssize y0)))
	 (y1s (float (* ssize y1)))
	 (sws (float (* ssize width))))
    (if (output score)
	(if (c-output-in-progress score)
	    (progn
	      #+clisp (c-stem x0s y0s y1s sws)
	      #-clisp (setf (aref c-args 0) x0s)
	      #-clisp (setf (aref c-args 1) y0s)
	      #-clisp (setf (aref c-args 2) y1s)
	      #-clisp (setf (aref c-args 3) sws)
	      #-clisp (c-stem c-args)
	      )
	  (g-thick-vertical-line score sws x0s y0s y1s)))
    (%xy score x0s y1s)
    (%update-bounds score)))

(defmethod just-stem ((score server) x0 y0 y1)
  (let* ((ssize (scr-size score))
	 (x0s (float (* ssize x0)))
	 (y0s (float (* ssize y0)))
	 (y1s (float (* ssize y1))))
    (if (output score)
	(if (c-output-in-progress score)
	    (progn
	      #+clisp (c-just-stem x0s y0s y1s)
	      #-clisp (setf (aref c-args 0) x0s)
	      #-clisp (setf (aref c-args 1) y0s)
	      #-clisp (setf (aref c-args 2) y1s)
	      #-clisp (c-just-stem c-args)
	      )
	  (g-vertical-line score x0s y0s y1s)))))

(defmethod just-line ((score server) x0 y0 x1)
  (let* ((ssize (scr-size score))
	 (x0s (float (* ssize x0)))
	 (y0s (float (* ssize y0)))
	 (x1s (float (* ssize x1))))
    (if (output score)
	(if (c-output-in-progress score)
	    (progn
	      #+clisp (c-line x0s y0s x1s)
	      #-clisp (setf (aref c-args 0) x0s)
	      #-clisp (setf (aref c-args 1) y0s)
	      #-clisp (setf (aref c-args 2) x1s)
	      #-clisp (c-line c-args)
	      )
	  (g-horizontal-line score x0s y0s x1s)))))

(defmethod slanted-fill-in ((score server) x0 y0 x1 y1 thickness)
  (let* ((ssize (scr-size score))
	 (x0s (float (* ssize x0)))
	 (y0s (float (* ssize y0)))
	 (x1s (float (* ssize x1)))
	 (y1s (float (* ssize y1)))
	 (thk (float (* ssize thickness))))
    (if (output score)
	(if (c-output-in-progress score)
	    (progn
	      #+clisp (c-slanted-line x0s y0s x1s y1s thk)
	      #-clisp (setf (aref c-args 0) x0s)
	      #-clisp (setf (aref c-args 1) y0s)
	      #-clisp (setf (aref c-args 2) x1s)
	      #-clisp (setf (aref c-args 3) y1s)
	      #-clisp (setf (aref c-args 4) thk)
	      #-clisp (c-slanted-line c-args)
	      )
	  (g-slanted-filled-rectangle score x0s y0s x1s y1s thk)))
    (setf (scr-x score) x0s)
    (setf (scr-y score) y0s)
    (setf (scr-x0 score) (min (scr-x0 score) x0s x1s))
    (setf (scr-y0 score) (min (scr-y0 score) y0s y1s))
    (setf (scr-x1 score) (max (scr-x1 score) x0s x1s))
    (setf (scr-y1 score) (max (scr-y1 score) (+ (max y0s y1s) thk)))))


(defun creation-date ()
  (flet ((month-name (month) (nth (- month 1) '("Jan" "Feb" "Mar" "Apr" "May" "Jun" "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")))
	 (day-name (day) (nth day '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))))
    (multiple-value-bind (second minute hour date month year day daylight-saving-p time-zone)
	(get-decoded-time)
      (declare (ignore second daylight-saving-p time-zone))
      (format nil "~A ~D-~A-~D at ~D:~2,'0D"
	      (day-name day) date (month-name month) (- year 1900) hour minute))))

(defmethod header ((server server) &key name &allow-other-keys)
  (let ((user-name #+(or excl KCL) (first (last (pathname-directory (user-homedir-pathname))))
		   #+clisp nil
		   ;; mcl code suggested by Tobias Kunze
		   #+mcl (ccl::with-macptrs ((h (mcl-get-string -16096)))
			   (if (or (ccl::%null-ptr-p h) 
				   (ccl::%null-ptr-p (ccl::%get-ptr h)) 
				   (eql 0 (ccl::%get-byte (ccl::%get-ptr h))))
			       "unspecified"
			     (ccl::%get-string h)))
		   ))
    (if (not (c-output-in-progress server))
	(g-header server name user-name)
      (progn
	(c-print (format nil "%!PS-Adobe-2.0 EPSF-2.0"))
	(if name (c-print (format nil "%%Title: ~A" name)))
	(if user-name (c-print (format nil "%%Creator: ~A" user-name)))
	(c-print (format nil "%%CreationDate: ~A" (creation-date)))
	(if (bounded server)
	    (progn
	      (setf (bounds server) (bounded server))
	      (c-print (format nil "%%BoundingBox:~{ ~D~}" (map 'list #'ceiling (bounds server)))))
	  (c-print (format nil "%%BoundingBox:(atend)")))
	(c-print (format nil "%%EndComments"))
	(include-eps-defs-and-whatever server)
	(c-print (format nil "%%EndProlog"))))))


(defmethod footer ((server server) &key)
  (if (c-output-in-progress server)
      (c-print (format nil " showpage~%%%Trailer"))
    (g-footer server))
  (if (not (bounded server))
      (progn
	;; should we pad x1 out to the page-width here? what about header-margin?
	(setf (box-y1 server) (max (box-y1 server)
				   (min (+ (box-y1 server) (inches-to-ps (header-margin *cmn-score*)))
					(inches-to-ps (page-height *cmn-score*)))))
	(setf (box-y0 server) (min (box-y0 server)
				   (max 0 (- (box-y0 server) (inches-to-ps (footer-margin *cmn-score*))))))
	(setf (box-x0 server) 0)
	(if (and (lines server) (> (lines server) 1))
	    (setf (box-x1 server) (max (box-x1 server) ;for /NextApps/Preview's benefit
				       (inches-to-ps (page-width *cmn-score*))))
	  (if (< (box-x1 server) (inches-to-ps (- (page-width *cmn-score*) (right-margin *cmn-score*))))
	      (incf (box-x1 server) (inches-to-ps (right-margin *cmn-score*)))))
	(if (c-output-in-progress server)
	    (c-print (format nil "%%BoundingBox:~{ ~D~}~%" (map 'list #'(lambda (n) 
									  (round (1+ n)))
								(bounds server))))
	  (g-bounding-box server)))))

(defvar *cmn-output-pathname* "aaa.eps")
(defvar *cmn-initialized* nil)


;;; removed args font size title 10-Feb-94 -- never used and causing confusion

(defmethod initialize ((server server) &key file bounded with-pages)
  (if (not *cmn-initialized*)
      (progn
	#+mcl (mcl-initialize)
	(if (not (output-file server)) (setf (output-file server) (or file *cmn-output-pathname*)))
	(if (not (pathname-type (output-file server))) (setf (output-file server) (concatenate 'string (output-file server) ".eps")))
	(if (c-output-in-progress server)
	    (let ((ok 0))
	      (setf (output server) t)
	      #+mcl (setf (output-file server) (namestring (translate-logical-pathname (output-file server))))
	      (setf ok (c-open (output-file server)))
	      (if (= ok 1) (throw 'cannot-open-file nil)))
	  (setf (output server) (open (output-file server)
				      :direction :output
				      :if-exists :supersede
				      :if-does-not-exist :create)))
	#+mcl (ccl:set-mac-file-type (output-file server) "TEXT")
	(setf (bounded server) bounded)
	(if (not (scr-size server)) (setf (scr-size server) 40))
	(if (not *cmn-score-size*) (setf *cmn-score-size* (scr-size server)))
	(if (not (font server)) (setf (font server) Music-Font))
	(let ((ltitle (title *cmn-score*)))
	  (if (and ltitle (text-p ltitle))
	      (header server :name (letters ltitle))
	    (header server :name ltitle)))
	(if (c-output-in-progress server)
	    (progn
	      (if with-pages (c-print (format nil "%%Page: 1 1")))
	      (c-print (format nil " /~A findfont ~D scalefont setfont" (font server) (scr-size server)))
	      (c-print (format nil " 0 setlinewidth")))
	  (progn
	    (if with-pages (g-page-number server 1))
	    (g-new-font server (font server) (scr-size server) nil)
	    (g-set-line-width server 0)))
	(setf *cmn-initialized* server)
	server)
    (progn
      (setf (size server) (size *cmn-initialized*))
      (setf (scaling-matrix server) (scaling-matrix *cmn-initialized*))
      (setf (output server) (output *cmn-initialized*))
      (setf (c-output-in-progress server) (c-output-in-progress *cmn-initialized*))
      (setf (output-file server) (output-file *cmn-initialized*))
      (setf (font server) (font *cmn-initialized*))
      (setf (bounded server) (bounded *cmn-initialized*))
      server)))

(defmethod clear-server ((server server))
  (declare (optimize (speed 3) (safety 0)))
  (g-clear server)
  (clear-bounding-box server)
  (setf (scr-x server) 0)
  (setf (scr-y server) 0))

(defvar *cmn-preview* nil)
(defvar *max-screen-X* 1024)		;actually 1120 on the NeXT
(defvar *max-screen-Y* 780)		;actually 830
(defvar *preview-X0* 40)
(defvar *preview-Y0* 40)

(defmethod finalize ((server server) &key)
  (let ((outf (output-file server)))
    #+mcl (declare (ignore outf))
    (footer server)
    (setf *cmn-initialized* nil)
    (if (c-output-in-progress server)
	(progn
	  #-clisp (c-close)
	  #+clisp (c-close 0)
	  (setf (c-output-in-progress server) nil)
	  #+NeXT (if *cmn-preview* (dpy-ps outf
				    (ceiling (max 0 *preview-X0*)) (ceiling (max 0 *preview-Y0*))
				    (floor (min *max-screen-X* (+ *preview-X0* (- (third (bounds server)) (first (bounds server))))))
				    (floor (min *max-screen-Y* (+ *preview-Y0* (- (fourth (bounds server)) (second (bounds server))))))))
	  )
      (g-close server 
	       #-mcl nil
	       #+mcl (= (pages *cmn-score*) 1)
	       )))
  (setf (output-file server) nil)
  (clear-server server)
  #+mcl (mcl-finalize)
  (setf *cmn-score-size* nil))



;;;
;;; --------------    Font info
;;; 

;;; now infinite data about Sonata font -- all in Sonata.afm (or whatever).
;;; The first constants below are OCTAL mascarading as decimal -- my mistake.
;;; Adobe font metric info = octal char code, rx dx dy w h
;;;
;;; a few of these numbers have been hand-tweaked (%staff for example) -- 
;;; don't read directly from the .afm file!


(defun make-constant-glyph (info)
  (if info
      (make-instance 'write-protected-glyph 
	  :index (first info) 
	  :rx (* .001 (second info))
	  :x0 (* .001 (third info))
	  :y0 (* .001 (fourth info))
	  :x1 (* .001 (fifth info))
	  :y1 (* .001 (sixth info)))
    (warn "nil argument to make-constant-glyph")))

(defparameter %zero (make-constant-glyph #+Sonata '(60 360 0 -217 360 218) #+Petrucci '(60 244 1 -253 230 253)))
(defparameter %one (make-constant-glyph #+Sonata '(61 250 0 -209 250 209) #+Petrucci '(61 378 0 -253 364 253)))
(defparameter %two (make-constant-glyph #+Sonata '(62 340 0 -217 340 218) #+Petrucci '(62 330 0 -253 316 253)))
(defparameter %three (make-constant-glyph #+Sonata '(63 324 0 -217 324 218) #+Petrucci '(63 348 0 -253 335 253)))
(defparameter %four (make-constant-glyph #+Sonata '(64 329 0 -209 329 209) #+Petrucci '(64 316 2 -253 303 253)))
(defparameter %five (make-constant-glyph #+Sonata '(65 300 0 -209 300 209) #+Petrucci '(65 344 2 -253 330 253)))
(defparameter %six (make-constant-glyph #+Sonata '(66 334 0 -217 334 218) #+Petrucci '(66 347 4 -253 335 253)))
(defparameter %seven (make-constant-glyph #+Sonata '(67 341 0 -215 341 216) #+Petrucci '(67 327 2 -253 314 253)))
(defparameter %eight (make-constant-glyph #+Sonata '(70 328 0 -217 328 218) #+Petrucci '(70 342 0 -253 328 253)))
(defparameter %nine (make-constant-glyph #+Sonata '(71 334 0 -217 334 218) #+Petrucci '(71 287 10 -445 165 96)))

(defparameter %sharp (make-constant-glyph #+Sonata '(43 224 0 -379 224 375) #+Petrucci '(43 237 1 -380 211 380)))
(defparameter %smallsharp (make-constant-glyph #+Sonata '(111 112 0 -193 112 188) #+Petrucci '(111 215 30 -185 136 214)))
(defparameter %flat (make-constant-glyph #+Sonata '(142 198 0 -178 198 491) #+Petrucci '(142 224 2 -195 219 485)))
(defparameter %smallflat (make-constant-glyph #+Sonata '(151 95 0 -89 95 246) #+Petrucci '(151 224 12 -123 122 216)))
(defparameter %natural (make-constant-glyph #+Sonata '(156 176 0 -350 176 350) #+Petrucci '(156 195 1 -349 170 349)))
(defparameter %dblflat (make-constant-glyph #+Sonata '(272 367 0 -178 367 491) #+Petrucci '(272 403 2 -195 402 485)))
(defparameter %dblsharp (make-constant-glyph #+Sonata '(334 266 0 -131 266 131) #+Petrucci '(334 278 0 -126 253 126)))
(defparameter %smallnatural (make-constant-glyph #+Sonata '(351 88 0 -192 88 186) #+Petrucci '(351 169 0 -169 84 177)))

(defparameter %pedal (make-constant-glyph #+Sonata '(241 792 0 -2 792 458) #+Petrucci '(241 824 0 -3 808 507)))
(defparameter %pedaloff (make-constant-glyph #+Sonata '(52 361 0 0 361 350) #+Petrucci '(52 406 -4 4 374 376)))
(defparameter %mordentslash (make-constant-glyph #+Sonata '(115 586 0 -90 586 310) #+Petrucci '(115 634 0 -84 607 338)))
(defparameter %harmonic (make-constant-glyph #+Sonata '(117 300 0 -131 300 132) #+Petrucci '(117 301 0 -126 269 127)))
(defparameter %turn (make-constant-glyph #+Sonata '(124 618 0 -133 618 133) #+Petrucci '(124 621 -25 -167 605 130)))
(defparameter %arpeggio (make-constant-glyph #+Sonata '(147 138 0 0 138 534) #+Petrucci '(147 152 9 5 136 571)))
(defparameter %mordent (make-constant-glyph #+Sonata '(155 606 0 -1 606 240) #+Petrucci '(155 528 0 7 607 257)))
(defparameter %naturalharmonic (make-constant-glyph #+Sonata '(157 120 0 -61 120 60) #+Petrucci '(157 203 0 -84 169 84)))
(defparameter %fermataup (make-constant-glyph #+Sonata '(165 658 0 -352 658 0) #+Petrucci '(165 710 -2 -383 678 -2)))
(defparameter %trillsection (make-constant-glyph #+Sonata '(176 337 0 0 413 165) #+Petrucci '(176 338 -45 -7 383 173)))
(defparameter %mordentthree (make-constant-glyph #+Sonata '(265 852 0 -1 852 238) #+Petrucci '(265 790 0 5 870 257)))
(defparameter %tremeloshort (make-constant-glyph #+Sonata '(276 300 0 0 300 479) #+Petrucci '(276 379 1 0 335 497)))
(defparameter %trillofinno (make-constant-glyph #+Sonata '(331 468 0 -13 468 369) #+Petrucci '(331 444 0 -22 423 380)))

(defparameter %coda (make-constant-glyph #+Sonata '(336 443 0 -254 443 255) #+Petrucci '(336 482 19 -66 442 429)))
(defparameter %repeatdots (make-constant-glyph #+Sonata '(173 126 0 307 126 696) #+Petrucci '(40 150 0 0 0 0)))
(defparameter %segno (make-constant-glyph #+Sonata '(45 515 0 -343 515 336) #+Petrucci '(45 570 12 0 532 694)))
(defparameter %repsign (make-constant-glyph #+Sonata '(324 610 0 0 610 509) #+Petrucci '(324 676 0 0 633 507)))

(defparameter %trebleclef (make-constant-glyph #+Sonata '(46 611 0 -402 611 1334) #+Petrucci '(46 676 0 -684 670 1077)))
(defparameter %treble8clef (make-constant-glyph #+Sonata '(46 611 0 -402 611 1334) #+Petrucci '(126 687 12 -895 683 1077)))
(defparameter %percclef1 (make-constant-glyph #+Sonata '(57 174 0 0 174 500) #+Petrucci '(57 190 0 0 169 507)))
(defparameter %bassclef (make-constant-glyph #+Sonata '(77 691 0 192 691 991) #+Petrucci '(77 723 4 -525 712 248)))
(defparameter %bass8clef (make-constant-glyph #+Sonata '(77 691 0 192 691 991) #+Petrucci '(164 761 4 -756 712 248)))
(defparameter %cclef (make-constant-glyph #+Sonata '(102 674 0 -9 674 1009) #+Petrucci '(102 676 0 -507 694 507)))

(defparameter %space (make-constant-glyph #+Sonata '(40 150 0 0 0 0) #+Petrucci '(40 150 0 0 0 0)))
(defparameter %parenleft (make-constant-glyph #+Sonata '(50 165 0 -50 165 500) #+Petrucci '(50 241 0 21 190 570)))
(defparameter %parenright (make-constant-glyph #+Sonata '(51 165 0 -50 165 500) #+Petrucci '(51 241 5 21 195 570)))
(defparameter %cross (make-constant-glyph #+Sonata '(53 300 0 -150 300 150) #+Petrucci '(53 330 -42 -186 338 189)))

(defparameter %fermata (make-constant-glyph #+Sonata '(125 658 0 0 658 352) #+Petrucci '(125 697 -2 0 678 380)))
(defparameter %grandpause (make-constant-glyph #+Sonata '(42 511 0 0 511 509) #+Petrucci '(42 494 14 0 450 507)))
(defparameter %pause (make-constant-glyph #+Sonata '(54 182 0 -231 182 68) #+Petrucci '(54 211 23 -218 185 53)))

(defparameter %wedgefill (make-constant-glyph #+Sonata '(340 228 0 0 228 199) #+Petrucci '(340 330 0 0 295 211)))
(defparameter %dot (make-constant-glyph #+Sonata '(56 94 0 -46 94 48) #+Petrucci '(56 104 22 -16 101 59)))
(defparameter %accent (make-constant-glyph #+Sonata '(76 417 0 4 417 257) #+Petrucci '(76 418 0 0 390 169)))
(defparameter %dotaccentup (make-constant-glyph #+Sonata '(254 278 0 0 278 416) #+Petrucci '(254 317 0 0 295 422)))
(defparameter %dotaccentdown (make-constant-glyph #+Sonata '(350 278 0 0 278 416) #+Petrucci '(350 317 0 11 295 433)))
(defparameter %wedgedown (make-constant-glyph #+Sonata '(166 263 0 0 263 417) #+Petrucci '(166 304 -5 -8 290 414)))
(defparameter %wedgeup (make-constant-glyph #+Sonata '(136 263 0 0 263 417) #+Petrucci '(136 317 0 0 295 422)))

(defparameter %eighthflagup (make-constant-glyph #+Sonata '(152 0 275 124 546 874) #+Petrucci '(152 331 0 -543 304 269)))
(defparameter %eighthflagdown (make-constant-glyph #+Sonata '(112 0 0 -874 271 -124) #+Petrucci '(112 331 0 -269 304 543)))
(defparameter %dblwholenote (make-constant-glyph #+Sonata '(127 614 0 -166 614 166) #+Petrucci '(127 637 7 -183 636 183)))
(defparameter %wholenote (make-constant-glyph #+Sonata '(167 412 0 -126 412 126) #+Petrucci '(167 406 0 -126 405 126)))
(defparameter %quarternotehead (make-constant-glyph #+Sonata '(317 300 0 -125 300 125) #+Petrucci '(317 287 1 -131 283 124)))
(defparameter %extendflagdown (make-constant-glyph #+Sonata '(360 0 0 -159 272 375) #+Petrucci '(360 312 0 -249 296 466)))
(defparameter %halfnotehead (make-constant-glyph #+Sonata '(372 300 0 -134 300 134) #+Petrucci '(372 283 1 -131 283 124)))
(defparameter %extendflagup (make-constant-glyph #+Sonata '(373 0 275 -360 546 184) #+Petrucci '(373 315 0 -466 296 249)))
(defparameter %rhythmX (make-constant-glyph #+Sonata '(300 300 0 -149 300 150) #+Petrucci '(300 292 -42 -175 301 175)))

(defparameter %staff (make-constant-glyph #+Sonata '(75 1000 -11 -9 1011 1009) #+Petrucci '(75 1014 -42 0 1056 1056)))
(defparameter %bottombracket (make-constant-glyph #+Sonata '(114 408 0 -296 408 155) #+Petrucci '(40 150 0 0 0 0)))
(defparameter %bracedown (make-constant-glyph #+Sonata '(352 300 0 -1001 300 1009) #+Petrucci '(40 150 0 0 0 0)))
(defparameter %topbracket (make-constant-glyph #+Sonata '(302 408 0 -155 408 296) #+Petrucci '(40 150 0 0 0 0)))
(defparameter %braceup (make-constant-glyph #+Sonata '(247 300 0 -1001 300 1009) #+Petrucci '(40 150 0 0 0 0)))

(defparameter %cuttime (make-constant-glyph #+Sonata '(103 421 0 -424 421 425) #+Petrucci '(103 423 4 -414 418 376)))
(defparameter %commontime (make-constant-glyph #+Sonata '(143 421 0 -254 421 255) #+Petrucci '(143 423 4 -248 418 247)))

(defparameter %mezzoforte (make-constant-glyph #+Sonata '(106 834 0 -172 834 413) #+Petrucci '(106 875 5 -177 859 419)))
(defparameter %mezzopiano (make-constant-glyph #+Sonata '(120 800 0 -172 800 266) #+Petrucci '(120 837 5 -147 805 295)))
(defparameter %sforzando (make-constant-glyph #+Sonata '(123 599 0 -172 599 413) #+Petrucci '(123 659 11 -148 614 448)))
(defparameter %forzando (make-constant-glyph #+Sonata '(132 651 0 -172 651 413) #+Petrucci '(132 685 1 -177 669 419)))
(defparameter %forte (make-constant-glyph #+Sonata '(146 515 0 -172 515 413) #+Petrucci '(146 518 1 -177 516 419)))
(defparameter %piano (make-constant-glyph #+Sonata '(160 445 0 -172 445 266) #+Petrucci '(160 437 0 -147 396 295)))
(defparameter %s (make-constant-glyph #+Sonata '(163 226 0 -16 226 266) #+Petrucci '(163 259 11 -9 236 292)))
(defparameter %z (make-constant-glyph #+Sonata '(172 291 0 -16 291 250) #+Petrucci '(172 355 -2 -12 290 287)))
(defparameter %ppp (make-constant-glyph #+Sonata '(270 1197 0 -172 1197 266) #+Petrucci '(270 1206 0 -147 1100 295)))
(defparameter %pp (make-constant-glyph #+Sonata '(271 821 0 -172 821 266) #+Petrucci '(271 823 0 -147 748 295)))
(defparameter %m (make-constant-glyph #+Sonata '(275 479 0 -16 479 266) #+Petrucci '(275 520 5 -13 471 294)))
(defparameter %ff (make-constant-glyph #+Sonata '(304 781 0 -172 781 413) #+Petrucci '(304 792 1 -177 788 419)))
(defparameter %fff (make-constant-glyph #+Sonata '(354 1047 0 -172 1047 413) #+Petrucci '(354 1063 1 -177 1060 419)))
(defparameter %r (make-constant-glyph #+Sonata '(363 280 0 0 280 266) #+Petrucci '(40 150 0 0 0 0)))

#+Petrucci (defparameter %fp (make-constant-glyph '(352 676 1 -249 671 419)))
#+Petrucci (defparameter %sfp (make-constant-glyph '(202 834 11 -164 815 448)))
#+Petrucci (defparameter %pppp (make-constant-glyph '(257 1499 0 -147 1453 295)))
#+Petrucci (defparameter %ffff (make-constant-glyph '(353 1432 1 -177 1340 419)))
#+Petrucci (defparameter %sfz (make-constant-glyph '(247 834 11 -147 767 449)))

(defparameter %thirtyrest (make-constant-glyph #+Sonata '(250 207 0 -384 400 578) #+Petrucci '(250 423 0 -511 377 456)))
(defparameter %wholerest (make-constant-glyph #+Sonata '(267 313 0 125 313 250) #+Petrucci '(267 423 0 126 422 255)))
(defparameter %sixteenthrest (make-constant-glyph #+Sonata '(305 138 0 -384 331 322) #+Petrucci '(305 304 0 -511 318 202)))
(defparameter %measurerest (make-constant-glyph #+Sonata '(335 500 0 0 500 500) #+Petrucci '(335 515 0 -253 507 253)))
(defparameter %dblwholerest (make-constant-glyph #+Sonata '(343 150 0 0 150 250) #+Petrucci '(40 150 0 0 0 0)))
(defparameter %eighthrest (make-constant-glyph #+Sonata '(344 69 0 -384 262 78) #+Petrucci '(344 304 28 -257 287 202)))
(defparameter %one28rest (make-constant-glyph #+Sonata '(345 345 0 -384 538 1078) #+Petrucci '(345 608 0 -511 495 957)))
(defparameter %halfrest (make-constant-glyph #+Sonata '(356 313 0 0 313 125) #+Petrucci '(356 423 0 -2 422 126)))
(defparameter %sixtyfourrest (make-constant-glyph #+Sonata '(364 276 0 -384 469 828) #+Petrucci '(364 608 0 -511 436 709)))
(defparameter %quarterrest (make-constant-glyph #+Sonata '(316 257 0 -355 257 400) #+Petrucci '(316 313 12 -404 290 388)))

(defparameter %square (make-constant-glyph #+Sonata '(255 300 0 -150 300 150) #+Petrucci '(255 279 -42 -169 295 169)))
(defparameter %squarefill (make-constant-glyph #+Sonata '(320 300 0 -150 300 150) #+Petrucci '(320 308 0 -118 278 160)))
(defparameter %trianglefill (make-constant-glyph #+Sonata '(321 300 0 -116 300 155) #+Petrucci '(321 389 0 -135 371 143)))
(defparameter %diamond (make-constant-glyph #+Sonata '(341 300 0 -150 300 150) #+Petrucci '(341 275 -42 -173 304 173)))
(defparameter %diamondfill (make-constant-glyph #+Sonata '(342 300 0 -150 300 150) #+Petrucci '(342 299 0 -142 283 142)))



(defun number-to-glyph (num)
  (if (<= 0 num 9) 
      (case num (0 %zero) (1 %one) (2 %two) (3 %three) (4 %four) (5 %five) (6 %six) (7 %seven) (8 %eight) (9 %nine))
    (text (format nil "~D" num))))	;font happens to line up right -- probably not the "right" thing in general.

(defun character-to-glyph (char)
  (case char
    (#\0 %zero) (#\1 %one) (#\2 %two) (#\3 %three) (#\4 %four)
    (#\5 %five) (#\6 %six) (#\7 %seven) (#\8 %eight) (#\9 %nine)
    (#\+ %cross) (#\. %dot) (#\C %commontime) (#\f %forte)
    (#\s %s) (#\r %r) (#\z %z) (#\p %piano) (#\m %m)
    (otherwise %space)))



;;;
;;; --------------    "messages" (see cmn docs)
;;; 

;;; as a convenience for callers of clm, we like to package up all the information about
;;; an entity within the parentheses of that entity -- that is:
;;;    (c4 q staccato (onset 1.5))
;;; which requires that the "onset" function pass its argument on to the "c4" function.
;;; In normal CLOS syntax this would be (setf (onset (c4 q staccato)) 1.5) or something,
;;; and the real entity rapidly gets swallowed up in onion-like layers of setfs with the
;;; associated setf-d data further and further from the field it is setting.  The
;;; functions that act like "onset" above return a "self-acting" instance and the
;;; "self-action" macro wraps up the rest.

(defstruct (self-acting (:conc-name nil)) action arguments)

(eval-when (compile load eval)
  (defmacro self-action (name setf-name)
    `(progn
       (defgeneric ,name (obj))
       (defgeneric ,setf-name (obj val))
       (defgeneric (setf ,name) (val obj))
       (defmethod ,setf-name ((inst t) val) (setf (,name inst) val))
       (defmethod ,name ((val t)) (make-self-acting :action #',setf-name :arguments val))))
  )

(defun rotate (object &optional angle)
  (if (or angle
	  (not (numberp object)))
      (%rotate object angle)
    (make-self-acting :action #'%rotate :arguments object)))

(defun scale (object &optional xscl yscl)
  (if (or yscl
	  (not (numberp object)))
      (%scale object xscl yscl)
    (make-self-acting :action #'%scale-1 :arguments (list object xscl))))

(defun mirror (&optional object)
  (if (or object
	  (not (numberp object)))
      (%mirror object)
    (make-self-acting :action #'%mirror :arguments nil)))

(defvar mirrored (make-self-acting :action #'%mirror :arguments nil))

(defun transform (object &optional matrix)
  (if (or matrix
	  (not (listp object)))
      (%transform object matrix)
    (make-self-acting :action #'%transform :arguments object)))

(defun invisible () (scale 0 0))
(defvar invisible (scale 0 0))



;;;
;;; --------------    basic justification fields
;;; 

(defclass bounds-mixin ()		;used for alignment and justification
  ((center :initarg :center		;center point
	   :initform 0
	   :reader center)
   (walls :initarg :walls		;absolute "don't draw inside these bounds" 
	  :initform nil
	  :reader walls)
   (fences :initarg :fences		;ideal bounds if there is room for them
	   :initform nil
	   :reader fences)
   (expanders :initform nil		;expansion behaviour when there's lots of free room to parcel out
	      :initarg :expanders
	      :reader expanders)))

(defclass bounds (bounds-mixin)
  ((center :accessor center)
   (walls :accessor walls)
   (fences :accessor fences)
   (expanders :accessor expanders)))

(defmethod descry ((bounds bounds-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A~A~A"
	  (if (not controller) "(bounds" 
	    (if (or (walls bounds) (fences bounds) (expanders bounds)) 
		(format nil "~%~A" prewhitespace)
	      ""))
	  (if (and (center bounds) (not (zerop (center bounds)))) (format nil " :center ~A" (center bounds)) "")
	  (if (walls bounds) (format nil " :walls '~A" (walls bounds)) "")
	  (if (fences bounds) (format nil " :fences '~A" (fences bounds)) "")
	  (if (expanders bounds) (format nil " :expanders '~A" (expanders bounds)) "")
	  (if (next-method-p) (call-next-method bounds stream (or controller bounds)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((bounds bounds-mixin) &optional object)
  (let ((new-object (if (not object) (make-bounds)
		      (if (write-protected object) 
			  (copy object) 
			object))))
    (setf (center new-object) (center bounds))
    (setf (walls new-object) (copy-list (walls bounds)))
    (setf (fences new-object) (copy-list (fences bounds)))
    (setf (expanders new-object) (copy-list (expanders bounds)))
    (if (next-method-p) (call-next-method bounds new-object))
    new-object))

(self-action center setf-center)
(self-action fences setf-fences)  
(self-action walls setf-walls)
(self-action expanders setf-expanders)  

(self-action x0 setf-x0) 
(self-action y0 setf-y0) 
(self-action x1 setf-x1) 
(self-action y1 setf-y1) 

(self-action dx setf-dx)
(self-action dy setf-dy)

(self-action staff-y0 setf-staff-y0)

(self-action matrix setf-matrix) 
(self-action pattern setf-pattern) 

(self-action font-name setf-font-name) 
(self-action font-size setf-font-size) 
(self-action font-scaler setf-font-scaler) 



;;;
;;; --------------    basic musical time fields (onset/duration/beat=>odb)
;;; 

(defclass odb-mixin ()
  ((onset :initarg :onset :initform nil :reader onset :reader odb-onset)
   (duration :initarg :duration :initform nil :reader duration :reader odb-duration)
   (beat :initarg :beat :initform nil :reader beat :reader odb-beat)))

(defclass odb (odb-mixin)
  ((onset :accessor onset :accessor odb-onset)
   (duration :accessor duration :accessor odb-duration)
   (beat :accessor beat :accessor odb-beat)))

(defmethod descry ((odb odb-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A~A"
	  (if (not controller) "(odb" 
	    (if (or (onset odb) (duration odb) (beat odb))
		(format nil "~%~A" prewhitespace)
	      ""))
	  (if (onset odb) (format nil " :onset ~A" (onset odb)) "")
	  (if (duration odb) (format nil " :duration ~A" (duration odb)) "")
	  (if (beat odb) (format nil " :beat ~A" (beat odb)) "")
	  (if (next-method-p) (call-next-method odb stream (or controller odb)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((odb odb-mixin) &optional object)
  (let ((new-object (if (not object) (make-odb) 
		      (if (write-protected object) (copy object)
			object))))
    (setf (odb-onset new-object) (odb-onset odb))
    (setf (odb-duration new-object) (odb-duration odb))
    (setf (odb-beat new-object) (odb-beat odb))
    (if (next-method-p) (call-next-method odb new-object))
    new-object))

;;;(self-action onset setf-onset)
;;;(self-action duration setf-duration)
;;;(self-action beat setf-beat)
;;; these three have to be careful -- avoid later round-off errors

(defvar smallest-note .015625)		;256-th note (1/64)

#-KCL
(defun ratify (num)			;rational returns gigantic useless factors
  (if (floatp num)
      (loop for i from 1 to 64 and val from num by num do 
	(if (< (abs (- val (round val))) smallest-note) 
	    (return-from ratify (list (round val) i))))
    (if (ratiop num)
	(list (numerator num) (denominator num))
      (list num 1))))

#+KCL
(defun ratify (num)			;uh oh... KCL gets confused about intermediate variable types in the loop above
  (if (floatp num)
      (do ((i 1 (1+ i))
	   (val num (+ val num)))
	  ((< (abs (- val (round val))) smallest-note)
	   (list (round val) i))
	())
    (if (ratiop num)
	(list (numerator num) (denominator num))
      (list num 1))))



(defun fratify (num)
  (if (floatp num)
      (apply #'/ (ratify num))
    num))

;;; expand self-action by hand and make these guys use ratify
(defgeneric onset (val) )
(defgeneric duration (val) )
(defgeneric beat (val) )
(defgeneric setf-onset (obj val) )
(defgeneric setf-duration (obj val) )
(defgeneric setf-beat (obj val) )
(defgeneric (setf onset) (val obj) )
(defgeneric (setf duration) (val obj) )
(defgeneric (setf beat) (val obj) )

(defmethod onset (val) (make-self-acting :action #'setf-onset :arguments val))
;;;(defmethod onset ((val number)) (make-self-acting :action #'setf-onset :arguments val))
(defmethod duration ((val number)) (make-self-acting :action #'setf-duration :arguments val))
(defmethod beat ((val number)) (make-self-acting :action #'setf-beat :arguments val))
(defmethod setf-onset (obj val) (if (numberp val) (setf (onset obj) (fratify val)) (setf (onset obj) val)))
(defmethod setf-duration (obj val) (setf (duration obj) (fratify val)))
(defmethod setf-beat (obj val) (setf (beat obj) (fratify val)))

(defun prolog (func) (make-self-acting :action #'setf-prolog :arguments func))
(defun setf-prolog (obj val) (push val (prologue obj)))


;;;
;;; ----------------    score (top node of entire data structure)
;;;

(defclass score (server dxy bounds visible)
   ;; first group of slots are for cmn's private use
  ((systems :initarg :systems :accessor systems :initform nil)
   (title :initarg :title :initarg title :accessor title :initform nil)
   (time-line :initarg :time-line :initform nil :accessor time-line)
   (time-lines :initarg :time-lines :initform nil :accessor time-lines)
   (line-data :initarg :line-data :initform nil :accessor line-data)
   (lines :initarg :lines :initform nil :accessor lines)
   (pages :initarg :pages :initform nil :accessor pages)
   (staves :initarg :staves :initform nil :accessor staves)
   (staff-line-separation :initarg :staff-line-separation :initarg staff-line-separation :accessor staff-line-separation :initform .126)
   (staff-dy :initarg :staff-dy :initarg staff-dy :accessor staff-dy :initform 1.018)

   ;; now user-settable slots
   (page-height :initarg :page-height :initarg page-height :accessor page-height :initform 11.0)
   (page-width :initarg :page-width :initarg page-width :accessor page-width :initform 8.5)
   (left-margin :initarg :left-margin :initarg left-margin :accessor left-margin :initform 0.5)
   (right-margin :initarg :right-margin :initarg right-margin :accessor right-margin :initform 0.5)
   (header-margin :initarg :header-margin :initarg header-margin :accessor header-margin :initform 1.0)
   (footer-margin :initarg :footer-margin :initarg footer-margin :accessor footer-margin :initform 1.0)
   (free-expansion-factor :initarg :free-expansion-factor :initarg free-expansion-factor :accessor free-expansion-factor :initform 1.25)
   (line-separation :initarg :line-separation :initarg line-separation :accessor line-separation :initform 2.0)
   (staff-separation :initarg :staff-separation :initarg staff-separation :accessor staff-separation :initform 1.5)
   (system-separation :initarg :system-separation :initarg system-separation :accessor system-separation :initform 1.5)
   (size :initarg :size :initarg size :accessor size :initform 40)
   (regularize :initarg :regularize :initarg regularize :accessor regularize :initform nil)
   (beats-per-quarter-note :initarg :beats-per-quarter-note :accessor beats-per-quarter-note :initform 1)
   (maximum-slur-slope :initarg :maximum-slur-slope :initarg maximum-slur-slope 
		       :accessor maximum-slur-slope :initform .75 :accessor maximum-slur-slope)
   (maximum-beam-tilt :initarg :maximum-beam-tilt :initarg maximum-beam-tilt :accessor maximum-beam-tilt :initform .4)
   (maximum-subdivision-bracket-tilt :initarg :maximum-subdivision-bracket-tilt :initarg maximum-subdivision-bracket-tilt 
				     :accessor maximum-subdivision-bracket-tilt :initform 0)
   (partial-beam-length :initarg :partial-beam-length :initarg partial-beam-length :accessor partial-beam-length :initform .4)
   (partial-stem-length :initarg :partial-stem-length :initarg partial-stem-length :accessor partial-stem-length :initform .10)
   (initial-octave :initarg :initial-octave :accessor initial-octave :initform 4)
   (metronome :initarg :metronome :initarg metronome :accessor metronome :initform 60)
   (beam-spacing :initarg :beam-spacing :initarg beam-spacing :accessor beam-spacing :initform (* 2.125 .126)) ;was 2.25
   (beam-width :initarg :beam-width :initarg beam-width :accessor beam-width :initform (* 1.125 .126)) ;was 1.25
   (beam-slope-trigger :initarg :beam-slope-trigger :initarg beam-slope-trigger :accessor beam-slope-trigger :initform .5)
   (staff-line-width :initarg :staff-line-width :initarg staff-line-width :accessor staff-line-width :initform .02) ;was .01
   (note-head-size :initarg :note-head-size :initarg note-head-size :accessor note-head-size :initform 1.0)
   (stem-width :initarg :stem-width :initarg stem-width :accessor stem-width :initform .025)
   ;; Ross says the stem should be narrower than the staff-line, but that looks ugly to me (I like narrow staff lines)
   (ideal-stem-length :initarg :ideal-stem-length :initarg ideal-stem-length :accessor ideal-stem-length :initform .5)
   (curvy-flags :initarg :curvy-flags :initarg curvy-flags :initform t :accessor curvy-flags)
   (old-style-beams :initarg :old-style-beams :initarg old-style-beams :initform nil :accessor old-style-beams)
   (automatic-page-numbers :initarg :automatic-page-numbers :initarg automatic-page-numbers :initform nil :accessor automatic-page-numbers)
   (automatic-measure-numbers :initarg :automatic-measure-numbers :initarg automatic-measure-numbers 
			      :initform nil :accessor automatic-measure-numbers)
   (first-measure-number :initarg :first-measure-number :initarg first-measure-number 
			      :initform nil :accessor first-measure-number)
   (use-italian-octave-signs :initarg :use-italian-octave-signs :initarg use-italian-octave-signs 
			     :initform nil :accessor use-italian-octave-signs)
   (add-bassa-to-octave-signs :initarg :add-bassa-to-octave-signs :initarg add-bassa-to-octave-signs 
			      :initform nil :accessor add-bassa-to-octave-signs)
   (always-show-staff-names :initarg :always-show-staff-names :initarg always-show-staff-names :initform t :accessor always-show-staff-names)
   (all-output-in-one-file :initarg :all-output-in-one-file :initarg all-output-in-one-file :initform nil :accessor all-output-in-one-file)
   (automatic-line-breaks :initarg :automatic-line-breaks :initarg automatic-line-breaks :initform t :accessor automatic-line-breaks)
   (automatic-octave-signs :initarg :automatic-octave-signs :initarg automatic-octave-signs :initform t :accessor automatic-octave-signs)
   (automatic-beams :initarg :automatic-beams :initarg automatic-beams :initform t :accessor automatic-beams)
   (liberal-automatic-beams :initarg :liberal-automatic-beams :initarg liberal-automatic-beams :initform t :accessor liberal-automatic-beams)
   (automatic-ties :initarg :automatic-ties :initarg automatic-ties :initform t :accessor automatic-ties)
   (automatic-bars :initarg :automatic-bars :initarg automatic-bars :initform t :accessor automatic-bars)
   (automatic-rests :initarg :automatic-rests :initarg automatic-rests :initform t :accessor automatic-rests)
   (automatic-naturals :initarg :automatic-naturals :initarg automatic-naturals :initform nil :accessor automatic-naturals)
   (implicit-accidental-style :initarg :implicit-accidental-style :initarg implicit-accidental-style 
			      :initform :new-style :accessor implicit-accidental-style)
   (implicit-accidental-duration :initarg :implicit-accidental-duration :initarg implicit-accidental-duration 
				 :initform nil :accessor implicit-accidental-duration)
   (redundant-accidentals :initarg :redundant-accidentals :initarg redundant-accidentals :initform t :accessor redundant-accidentals)
   (automatic-beat-subdivision-numbers :initarg :automatic-beat-subdivision-numbers :initarg automatic-beat-subdivision-numbers 
				       :initform t :accessor automatic-beat-subdivision-numbers)
   (show-rulers :initarg :show-rulers :initarg show-rulers :initform nil :accessor show-rulers)
   (initial-onset :initarg :initial-onset :initarg initial-onset :initform 0 :accessor initial-onset)
   (section-onset :initarg :section-onset :initarg section-onset :initform 0 :accessor section-onset)
   (full-last-line :initarg :full-last-line :initarg full-last-line :initform nil :accessor full-last-line)
   (line-hook :initarg :line-hook :initarg line-hook :initform nil :accessor line-hook)
   (page-hook :initarg :page-hook :initarg page-hook :initform nil :accessor page-hook)
   (spacing-hook :initarg :spacing-hook :initarg spacing-hook :initform nil :accessor spacing-hook)
   ;; we use these "hooks", analogous to lisp's eval-hook because we want to make local
   ;;  modifications to a particular evaluation of the cmn function, and the equivalent mechanism within CLOS
   ;;  (using generic-flet) is not implemented in pcl.
   (use-abbreviated-staff-names :initarg :use-abbreviated-staff-names :initarg use-abbreviated-staff-names 
				:initform t :accessor use-abbreviated-staff-names)

   ;; the next group are useful enough to be worth retaining
   (tie-thickness :initarg :tie-thickness :initarg tie-thickness :accessor tie-thickness :initform .04)
   (tie-curvature :initarg :tie-curvature :initarg tie-curvature :accessor tie-curvature :initform .125 :accessor tie-curvature)
   (slur-thickness :initarg :slur-thickness :initarg slur-thickness :accessor slur-thickness :initform .03)
   (slur-curvature :initarg :slur-curvature :initarg slur-curvature :accessor slur-curvature :initform .25 :accessor slur-curvature)
   (dynamics-size :initarg :dynamics-size :initarg dynamics-size :initform .8 :accessor dynamics-size)
   (grace-note-size :initarg :grace-note-size :initarg grace-note-size :initform .5 :accessor grace-note-size)
   (text-connecting-pattern :initarg :text-connecting-pattern :initarg text-connecting-pattern 
			    :initform '(10 0) :accessor text-connecting-pattern)

   ;; these affect line layout and are most easily handled here (could be moved)
   (brace-space :initarg :brace-space :initarg brace-space :initform .125 :accessor brace-space)
   (staff-name-font :initarg :staff-name-font :initarg staff-name-font :initform "Times-Roman" :accessor staff-name-font)
   (staff-name-font-scaler :initarg :staff-name-font-scaler :initarg staff-name-font-scaler :initform .4 :accessor staff-name-font-scaler)
   (staff-name-font-minimum-size :initarg :staff-name-font-minimum-size :initarg staff-name-font-minimum-size 
				 :initform 8 :accessor staff-name-font-minimum-size)
   ))


(defmethod half-stem-width ((score score)) (* .5 (stem-width score)))
(defmethod half-staff-line-separation ((score score)) (* .5 (staff-line-separation score)))
(defmethod half-staff-dy ((score score)) (* .5 (staff-dy score)))
(defmethod octave-width ((score score)) (* 7 (staff-line-separation score)))
(defmethod sixth-width ((score score)) (* 5 (staff-line-separation score)))

(self-action title setf-title)
(self-action page-height setf-page-height)
(self-action page-width setf-page-width)
(self-action left-margin setf-left-margin)
(self-action right-margin setf-right-margin)
(self-action header-margin setf-header-margin)
(self-action footer-margin setf-footer-margin)
(self-action free-expansion-factor setf-free-expansion-factor)
(self-action line-separation setf-line-separation)
(self-action staff-separation setf-staff-separation)
(self-action system-separation setf-system-separation)
(self-action regularize setf-regularize)
					
;;;(self-action size setf-size)
;;;  here we need to know as fast as possible what the score size is
(defgeneric size (val) )
(defmethod size ((val number)) 
  (if (not *cmn-score-size*) (setf *cmn-score-size* val))
  (make-self-acting :action #'setf-size :arguments val))
(defmethod setf-size ((obj score) val) (setf (size obj) val))


;;;(self-action output-file setf-output-file)
;;;  in very large scores, cmn can take a long time before trying to open the output file,
;;;  which is annoying if it dies in the attempt -- output-file therefore makes an
;;;  immediate attempt to open the file so that cmn dies right away if it's going to anyway.
(defgeneric output-file (name) )
(defmethod output-file (name)
  (let ((full-name name)
	(fil nil))
    (if (not (pathname-type full-name)) 
	(setf full-name (concatenate 'string full-name ".eps")))
    (setf fil (open full-name :direction :output :if-exists :supersede))
    (close fil)
    (make-self-acting :action #'setf-output-file :arguments name)))
(defmethod setf-output-file ((obj score) name) (setf (output-file obj) name))

(self-action output-type setf-output-type)

(self-action beats-per-quarter-note setf-beats-per-quarter-note)
(self-action stem-width setf-stem-width)
(self-action ideal-stem-length setf-ideal-stem-length)
(self-action tie-thickness setf-tie-thickness)
(self-action tie-curvature setf-tie-curvature)
(self-action slur-thickness setf-slur-thickness)
(self-action slur-curvature setf-slur-curvature)
(self-action maximum-slur-slope setf-maximum-slur-slope)
(self-action maximum-beam-tilt setf-maximum-beam-tilt)
(self-action maximum-subdivision-bracket-tilt setf-maximum-subdivision-bracket-tilt)
(self-action partial-beam-length setf-partial-beam-length)
(self-action partial-stem-length setf-partial-stem-length)
(self-action initial-octave setf-initial-octave)
(self-action metronome setf-metronome)
(self-action beam-spacing setf-beam-spacing)
(self-action beam-width setf-beam-width)
(self-action beam-slope-trigger setf-beam-slope-trigger)
(self-action staff-line-width setf-staff-line-width)
(self-action note-head-size setf-note-head-size)
(self-action staff-line-separation setf-staff-line-separation)
(self-action staff-dy setf-staff-dy)
(self-action brace-space setf-brace-space)
(self-action curvy-flags setf-curvy-flags)
(self-action old-style-beams setf-old-style-beams)
(self-action automatic-page-numbers setf-automatic-page-numbers)
(self-action automatic-measure-numbers setf-automatic-measure-numbers)
(self-action first-measure-number setf-first-measure-number)
(self-action use-italian-octave-signs setf-use-italian-octave-signs)
(self-action add-bassa-to-octave-signs setf-add-bassa-to-octave-signs)
(self-action dynamics-size setf-dynamics-size)
(self-action staff-name-font setf-staff-name-font)
(self-action staff-name-font-scaler setf-staff-name-font-scaler)
(self-action staff-name-font-minimum-size setf-staff-name-font-minimum-size)
(self-action use-abbreviated-staff-names setf-use-abbreviated-staff-names)
(self-action grace-note-size setf-grace-note-size)
(self-action text-connecting-pattern setf-text-connecting-pattern)
(self-action always-show-staff-names setf-always-show-staff-names)
(self-action all-output-in-one-file setf-all-output-in-one-file)
(self-action automatic-line-breaks setf-automatic-line-breaks)
(self-action automatic-octave-signs setf-automatic-octave-signs)
(self-action automatic-beams setf-automatic-beams)
(self-action liberal-automatic-beams setf-liberal-automatic-beams)
(self-action automatic-ties setf-automatic-ties)
(self-action automatic-bars setf-automatic-bars)
(self-action automatic-rests setf-automatic-rests)
(self-action automatic-naturals setf-automatic-naturals)
(self-action implicit-accidental-style setf-implicit-accidental-style)
(self-action implicit-accidental-duration setf-implicit-accidental-duration)
(self-action redundant-accidentals setf-redundant-accidentals)
(self-action automatic-beat-subdivision-numbers setf-automatic-beat-subdivision-numbers)
(self-action show-rulers setf-show-rulers)
(self-action initial-onset setf-initial-onset)
(self-action section-onset setf-section-onset)
(self-action full-last-line setf-full-last-line)
(self-action line-hook setf-line-hook)
(self-action page-hook setf-page-hook)
(self-action spacing-hook setf-spacing-hook)


(defgeneric layout (val) )
;; purely user-level, as a cleaner handle on justification of score, system, and staff
(defmethod layout (val) (make-self-acting :action #'(lambda (obj val) (setf (visible-justification obj) val)) :arguments val))


(defmethod copy ((score score) &optional object)
  (if (not object)
      (make-instance 'score 
	  :systems (loop for system in (systems score) collect (copy system))
	  :title (title score) 
	  :time-line (copy-tree (time-line score))
	  :time-lines (copy-tree (time-lines score))
	  :line-data (copy-tree (line-data score))
	  :lines (lines score)
	  :pages (pages score)
	  :staves (staves score)
	  :size (size score) 
	  :page-height (page-height score)
	  :page-width (page-width score)
	  :left-margin (left-margin score)
	  :right-margin (right-margin score)
	  :header-margin (header-margin score)
	  :footer-margin (footer-margin score)
	  :free-expansion-factor (free-expansion-factor score)
	  :line-separation (line-separation score)
	  :staff-separation (staff-separation score)
	  :system-separation (system-separation score)
	  :size (size score)
	  :regularize (regularize score)
	  :beats-per-quarter-note (beats-per-quarter-note score)
	  :stem-width (stem-width score)
	  :ideal-stem-length (ideal-stem-length score)
	  :tie-thickness (tie-thickness score)
	  :tie-curvature (tie-curvature score)
	  :slur-thickness (slur-thickness score)
	  :slur-curvature (slur-curvature score)
	  :maximum-slur-slope (maximum-slur-slope score)
	  :maximum-beam-tilt (maximum-beam-tilt score)
	  :maximum-subdivision-bracket-tilt (maximum-subdivision-bracket-tilt score)
	  :partial-beam-length (partial-beam-length score)
	  :partial-stem-length (partial-stem-length score)
	  :initial-octave (initial-octave score)
	  :metronome (metronome score)
	  :beam-spacing (beam-spacing score)
	  :beam-width (beam-width score)
	  :beam-slope-trigger (beam-slope-trigger)
	  :staff-line-width (staff-line-width score)
	  :note-head-size (note-head-size score)
	  :staff-line-separation (staff-line-separation score)
	  :staff-dy (staff-dy score)
	  :brace-space (brace-space score)
	  :old-style-beams (old-style-beams score)
	  :automatic-page-numbers (automatic-page-numbers score)
	  :automatic-measure-numbers (automatic-measure-numbers score)
	  :first-measure-number (first-measure-number score)
	  :curvy-flags (curvy-flags score)
	  :use-italian-octave-signs (use-italian-octave-signs score)
	  :add-bassa-to-octave-signs (add-bassa-to-octave-signs score)
	  :dynamics-size (dynamics-size score)
	  :staff-name-font (staff-name-font score)
	  :staff-name-font-scaler (staff-name-font-scaler score)
	  :staff-name-font-minimum-size (staff-name-font-minimum-size score)
	  :use-abbreviated-staff-names (use-abbreviated-staff-names score)
	  :grace-note-size (grace-note-size score)
	  :text-connecting-pattern (text-connecting-pattern score)
	  :always-show-staff-names (always-show-staff-names score)
	  :all-output-in-one-file (all-output-in-one-file score)
	  :automatic-line-breaks (automatic-line-breaks score)
	  :automatic-octave-signs (automatic-octave-signs score)
	  :automatic-beams (automatic-beams score)
	  :liberal-automatic-beams (liberal-automatic-beams score)
	  :automatic-ties (automatic-ties score)
	  :automatic-bars (automatic-bars score)
	  :automatic-rests (automatic-rests score)
	  :automatic-naturals (automatic-naturals score)
	  :implicit-accidental-style (implicit-accidental-style score)
	  :implicit-accidental-duration (implicit-accidental-duration score)
	  :redundant-accidentals (redundant-accidentals score)
	  :automatic-beat-subdivision-numbers (automatic-beat-subdivision-numbers score)
	  :show-rulers (show-rulers score)
	  :initial-onset (initial-onset score)
	  :section-onset (section-onset score)
	  :full-last-line (full-last-line score))
    (error "can't copy embedded scores yet")))


(defun print-if-changed (field field-name init-value &optional store-form keyword)
  (let ((init-char (if store-form "(" ":"))
	(end-char (if store-form ")" "")))
    (or (and (not init-value)
	     field
	     (format nil "~%  ~A~A ~A~A" init-char field-name (if keyword (format nil ":~(~A~)" field) field) end-char))
	(and init-value
	     (or (and (eq init-value t)
		      (not field)
		      (format nil "~%  ~A~A nil~A" init-char field-name end-char))
		 (and (numberp init-value)
		      (/= init-value field)
		      (format nil "~%  ~A~A ~A~A" init-char field-name field end-char))
		 (and (listp init-value)
		      (not (equal init-value field))
		      (format nil "~%  ~A~A '~A~A" init-char field-name field end-char))
		 (and (stringp init-value)
		      (not (string-equal init-value field))
		      (format nil "~%  ~A~A ~S~A" init-char field-name field end-char))
		 ""))
	"")))

(defun style-list (score &optional (all t) (sf nil))
  (list 
   (if all (print-if-changed (lines score) "lines" nil sf) "")
   (if all (print-if-changed (pages score) "pages" nil sf) "")
   (if all (print-if-changed (staves score) "staves" nil sf) "")
   (print-if-changed (page-height score) "page-height" 11.0 sf)
   (print-if-changed (page-width score) "page-width" 8.5 sf)
   (print-if-changed (left-margin score) "left-margin" 0.5 sf)
   (print-if-changed (right-margin score) "right-margin" 0.5 sf)
   (print-if-changed (header-margin score) "header-margin" 1.0 sf)
   (print-if-changed (footer-margin score) "footer-margin" 1.0 sf)
   (print-if-changed (free-expansion-factor score) "free-expansion-factor" 1.25 sf)
   (print-if-changed (line-separation score) "line-separation" 2.0 sf)
   (print-if-changed (staff-separation score) "staff-separation" 1.5 sf)
   (print-if-changed (system-separation score) "system-separation" 1.5 sf)
   (print-if-changed (size score) "size" 40 sf)
   (print-if-changed (regularize score) "regularize" nil sf)
   (print-if-changed (beats-per-quarter-note score) "beats-per-quarter-note" 1 sf)
   (print-if-changed (stem-width score) "stem-width" .025 sf)
   (print-if-changed (ideal-stem-length score) "ideal-stem-length" .5 sf)
   (print-if-changed (tie-thickness score) "tie-thickness" .04 sf)
   (print-if-changed (tie-curvature score) "tie-curvature" .125 sf)
   (print-if-changed (slur-thickness score) "slur-thickness" .03 sf)
   (print-if-changed (slur-curvature score) "slur-curvature" .25 sf)
   (print-if-changed (maximum-slur-slope score) "maximum-slur-slope" .75 sf)
   (print-if-changed (maximum-beam-tilt score) "maximum-beam-tilt" .4 sf)
   (print-if-changed (maximum-subdivision-bracket-tilt score) "maximum-subdivision-bracket-tilt" 0 sf)
   (print-if-changed (partial-beam-length score) "partial-beam-length" .4 sf)
   (print-if-changed (partial-stem-length score) "partial-stem-length" .10 sf)
   (print-if-changed (initial-octave score) "initial-octave" 4 sf)
   (print-if-changed (metronome score) "metronome" 60 sf)
   (print-if-changed (beam-spacing score) "beam-spacing" (* 2.125 .126) sf)
   (print-if-changed (beam-width score) "beam-width" (* 1.125 .126) sf)
   (print-if-changed (beam-slope-trigger score) "beam-slope-trigger" .5 sf)
   (print-if-changed (staff-line-width score) "staff-line-width" .02 sf)
   (print-if-changed (note-head-size score) "note-head-size" 1.0 sf)
   (print-if-changed (staff-line-separation score) "staff-line-separation" .126 sf)
   (print-if-changed (staff-dy score) "staff-dy" 1.018 sf)
   (print-if-changed (brace-space score) "brace-space" .125 sf)
   (print-if-changed (curvy-flags score) "curvy-flags" t sf)
   (print-if-changed (old-style-beams score) "old-style-beams" nil sf)
   (print-if-changed (automatic-page-numbers score) "automatic-page-numbers" nil sf)
   (print-if-changed (automatic-measure-numbers score) "automatic-measure-numbers" nil sf 
		     (and (automatic-measure-numbers score) (member (automatic-measure-numbers score) '(:by-line :by-page))))
   (print-if-changed (first-measure-number score) "first-measure-number" nil sf)
   (print-if-changed (use-italian-octave-signs score) "use-italian-octave-signs" nil sf)
   (print-if-changed (add-bassa-to-octave-signs score) "add-bassa-to-octave-signs" nil sf)
   (print-if-changed (dynamics-size score) "dynamics-size" .8 sf)
   (print-if-changed (staff-name-font score) "staff-name-font" "Times-Roman" sf)
   (print-if-changed (staff-name-font-scaler score) "staff-name-font-scaler" .4 sf)
   (print-if-changed (staff-name-font-minimum-size score) "staff-name-font-minimum-size" 8 sf)
   (print-if-changed (use-abbreviated-staff-names score) "use-abbreviated-staff-names" t sf)
   (print-if-changed (grace-note-size score) "grace-note-size" .5 sf)
   (print-if-changed (text-connecting-pattern score) "text-connecting-pattern" '(10 0) sf)
   (print-if-changed (always-show-staff-names score) "always-show-staff-names" t sf)
   (print-if-changed (all-output-in-one-file score) "all-output-in-one-file" nil sf)
   (print-if-changed (automatic-line-breaks score) "automatic-line-breaks" t sf)
   (print-if-changed (automatic-octave-signs score) "automatic-octave-signs" t sf)
   (print-if-changed (automatic-beams score) "automatic-beams" t sf)
   (print-if-changed (liberal-automatic-beams score) "liberal-automatic-beams" t sf)
   (print-if-changed (automatic-ties score) "automatic-ties" t sf)
   (print-if-changed (automatic-bars score) "automatic-bars" t sf)
   (print-if-changed (automatic-rests score) "automatic-rests" t sf)
   (print-if-changed (automatic-naturals score) "automatic-naturals" nil sf)
   (print-if-changed (implicit-accidental-style score) "implicit-accidental-style" :new-style sf)
   (print-if-changed (implicit-accidental-duration score) "implicit-accidental-duration" nil sf)
   (print-if-changed (redundant-accidentals score) "redundant-accidentals" t sf)
   (print-if-changed (automatic-beat-subdivision-numbers score) "automatic-beat-subdivision-numbers" t sf)
   (print-if-changed (show-rulers score) "show-rulers" nil sf)
   (print-if-changed (initial-onset score) "initial-onset" 0 sf)
   (print-if-changed (section-onset score) "section-onset" 0 sf)
   (print-if-changed (full-last-line score) "full-last-line" nil sf)))

(defmethod descry ((score score) &optional stream controller)
  (format stream "(score ~A~A~A~A~A~{~A~}~%  ~A)"
	  (if (title score) (format nil " :title ~A" (title score)) "")
	  (if (next-method-p) (call-next-method score stream (or controller score)) "")
	  (if (time-line score)
	      (format nil "~%   :time-line (list ~{~A~})" (time-line score))
	    "")
	  (if (time-lines score)
	      (format nil "~%   :time-lines (list ~{~%          ~A~})" (loop for ln in (time-lines score) collect (descry ln)))
	    "")
	  (if (line-data score)
	      (format nil "~%   :line-data (list ~{~%          ~A~})" (loop for ln in (line-data score) collect (descry ln)))
	    "")
	  (style-list score)
	  (if (systems score) (format nil " :systems (list ~{~%     ~A~})" 
				      (loop for system in (systems score) collect (descry system stream score)))
	    "")))

(defmethod score-p ((obj t)) nil)
(defmethod score-p ((obj score)) t)

(defmethod identify ((score score))
  (with-output-to-string (s)
    (format s "(with-cmn ")
    (format s "~A~A~A" 
	  (identify-dxy score) 
	  (identify-matrix score)
	  (identify-visible score))
    (identify-score score s nil)
    (format s ")")))



(defvar *cmn-system* nil)		;for better error reporting
(defvar *cmn-staff* nil)
(defvar *cmn-staff-data* nil)
(defvar *cmn-object* nil)
(defvar *cmn-owning-object* nil)
(defvar *cmn-page* nil)
(defvar cmn-pipe-0 nil)
(defvar cmn-pipe-1 nil)
(defvar cmn-pipe-2 nil)



;;; method display and house score are at very end -- same for systems?

(defvar no-initialization nil)

(defun scorify (objects)
  (let ((sections nil)
	(actions nil)
	(marks nil)
	(systems nil))
    (if (and (= (length objects) 1) (score-p (first objects)))
	(setf *cmn-score* (first objects))
      (let ((new-score (make-score)))
	(setf *cmn-score* new-score)
	(multiple-value-setq
	    (sections systems actions marks)
	  (notify new-score objects))
	(setf (systems new-score) systems)
	(setf (marks new-score) marks)
	(loop for act in actions do
	  (funcall (action act) new-score (arguments act)))))
    (if (title *cmn-score*) 
	(add-to-marks *cmn-score* (list (make-title (title *cmn-score*)))))
    (if (eq (output-type *cmn-score*) :postscript)
	#+(or Excl mcl kcl (and clisp foreign unix))  (setf (c-output-in-progress *cmn-score*) t)
	#-(or Excl mcl kcl (and clisp foreign unix))  (setf (c-output-in-progress *cmn-score*) nil)
	)
    (if (not no-initialization) 
	(if (output-file *cmn-score*)
	    (initialize *cmn-score* :with-pages (all-output-in-one-file *cmn-score*))
	  (initialize *cmn-score* :with-pages t)))
    (if sections
	
	(let ((new-score *cmn-score*)
	      (scores (loop for sct in sections collect (scorify (append actions marks (data sct))))))
	  (if systems
	      (push new-score scores)
	    (progn
	      (setf *cmn-score* (first scores))
	      scores)))
      *cmn-score*)))

(defmethod notify ((score score) &optional objects)
  ;; make sure everything in the objects list is parcelled out to the correct portion of the score
  (let ((sections nil)
	(systems nil)
	(actions nil)
	(marks nil)
	(object-list objects)
	(current-default-system nil)
	(current-default-system-data nil))
    (loop while object-list do
      (let ((object (pop object-list)))
	(when object
	  (if (and (score-object-list-p object) (not (glyph-list-p object)))
	      (setf object-list (append (disgorge object) object-list))
	    (if (section-p object)
		(push object sections)
	      (if (system-p object)
		  (progn
		    (if current-default-system
			(push (apply #'system (nreverse current-default-system-data)) systems))
		    (setf current-default-system-data nil)
		    (if (write-protected object)
			(setf current-default-system object)
		      (progn
			(setf current-default-system nil)
			(if (member object systems) (cmn-error "attempt to load the same system twice"))
			(push object systems))))
		(if (not current-default-system)
		    (if (self-acting-p object)
			(push object actions)
		      (if (or (text-p object) (sundry-p object) (glyph-list-p object))
			  (push objects marks)
			(progn
			  (setf current-default-system (make-write-protected-system))
			  (push object current-default-system-data))))
		  (progn
		    (if (not current-default-system)
			(setf current-default-system (make-write-protected-system)))
		    (push object current-default-system-data)))))))))
    (if current-default-system-data (push (apply #'system (nreverse current-default-system-data)) systems))
    (values (nreverse sections) (nreverse systems) (nreverse actions) marks)))


;;;
;;; --------------    overall organization of cmn:
;;;

(defvar tie-stack nil)
(defvar slur-stack nil)
(defvar glissando-stack nil)
(defvar tremolo-stack nil)
(defvar crescendo-stack nil)
(defvar text-stack nil)
(defvar octave-stack nil)
(defvar rehearsal-stack nil)
(defvar beam-stack nil)
(defvar *cmn-hidden-sections* nil)

(defun cmn (&rest objects)
  (when objects				; if no args, cmn is a no-op
    (unwind-protect
	(catch 'cannot-open-file	; open file error aborts entire process
	  (cmn-finalize			; cleanup, return complete score structure (for subsequent editing)
	   (drawify			; drawify draws the result (normally creates a separate .eps file for each page)
	    (filterify			; filterify filters out "uninteresting" portions of the final score (for page extraction)
	     (slurify			; slurify adds slurs
	      (markify			; markify adds most marks like fingerings, staccato, sF, etc
	       (tieify			; tieify adds ties
		(beamify		; beamify adds beams
		 (justify		; justify adds in white space to make each line and page look pretty
		  (pagify		; pagify breaks the linified form into pages
		   (linify		; linify breaks the compacted form into lines
		    (compactify		; compactify aligns boxes across all systems
		     (boxify		; boxify finds bounding boxes and center points
		      (fillify		; fillify does the first level of filling in missing data (bar lines, rests etc)
		       (scorify		; scorify (via notify) regularizes the input stream of objects
			(edify		; edify filters out all but "interesting" data (i.e. for part extraction, transposition, etc)
			 (cmn-initialize objects)))))))))))))))))
      (progn
	(if (and *cmn-score* 
		 (c-output-in-progress *cmn-score*))
	    (progn
	      (setf (c-output-in-progress *cmn-score*) nil)
	      (setf *cmn-initialized* nil)
	      #-clisp (c-close)
	      #+clisp (c-close 0)
	      ))
	(if (not no-initialization)
	    (setf tie-stack nil
		  slur-stack nil
		  glissando-stack nil
		  crescendo-stack nil
		  tremolo-stack nil
		  text-stack nil
		  rehearsal-stack nil
		  beam-stack nil
		  octave-stack nil
		  cmn-pipe-0 nil
		  cmn-pipe-1 nil
		  cmn-pipe-2 nil
		  *cmn-hidden-sections* nil
		  *cmn-score-size* nil
		  *old-cmn-score-size* nil
		  *cmn-system* nil
		  *cmn-staff* nil
		  *cmn-staff-data* nil
		  *cmn-page* nil
		  *cmn-object* nil
		  *cmn-owning-object* nil))))))

(defun check-for-left-overs ()
  (if tie-stack 
      (format t "~%begin-tie without matching end-tie:~{~%    ~A~}"
	      (loop for tie in (reverse tie-stack) collect (descry tie))))
  (if slur-stack 
      (format t "~%begin-slur without matching end-slur:~{~%    ~A~}" 
	      (loop for slur in (reverse slur-stack) collect (descry slur))))
  (if glissando-stack 
      (format t "~%begin-glissando without matching end-glissando:~{~%    ~A~}" 
	      (loop for gliss in (reverse glissando-stack) collect (descry gliss))))
  (if tremolo-stack 
      (format t "~%begin-tremolo without matching end-tremolo:~{~%    ~A~}" 
	      (loop for trem in (reverse tremolo-stack) collect (descry trem))))
  (if crescendo-stack 
      (format t "~%begin-crescendo or diminuendo without matching end:~{~%    ~A~}"
	      (loop for cresc in (reverse crescendo-stack) collect (descry cresc))))
  (if octave-stack 
      (if (listp octave-stack)
	  (format t "~%begin-octave without matching end-octave:~{~%    ~A~}" 
		  (loop for oct in (reverse octave-stack) collect (descry oct)))
	(format t "~%begin-octave, but no end-octave")))
  (if beam-stack 
      (format t "~%begin-beam without matching end-beam:~{~%    ~A~}" 
	      (loop for beam in (reverse beam-stack) collect (descry beam)))))

(defvar editor-hook nil)
(defvar filter-hook nil)
(defvar page-hook nil)
(defvar line-hook nil)

(defmethod cmn-finalize ((score score)) 
  (if (not no-initialization)		;might want to save/restore "hooks" across with-cmn
      (progn
	(check-for-left-overs)
	(setf editor-hook nil)
	(setf filter-hook nil)
	(finalize score)))
  score)

(defmethod cmn-initialize (objects) 
  objects)

;;; these two are methods so that callers can customize set-up and shut-down behavior.


;;; this is no longer needed -- cmn-initialize is now a method, so edify is just a sort of :after method

;;; edify is the bottom level editor
;;; it is passed the score (a list of lists of objects) and does what it wants

(defun edify (objects)
  (if editor-hook			;if an editor is present, let it munge the un-regularized data list
      (funcall editor-hook objects)
    objects))


;;; filterify should be done by making drawify a method

;;; filterify takes the finished score description (ready for the printer) and
;;; applies one last filtering pass

(defun filterify (score)
  (if filter-hook
      (funcall filter-hook score)
    score))



(defclass score-object-mixin (dxy-mixin bounds-mixin odb-mixin visible-mixin)
  ())

(defmethod score-object-p ((obj t)) nil)
(defmethod score-object-p ((obj score-object-mixin)) t)

(defclass score-object (score-object-mixin dxy bounds odb visible)
  ())

;;; anything that the score needs to keep track of (during alignment for example) is a score-object



;;;
;;; ----------------    section
;;;
;;; a section is a group of systems (staves) with auxiliary local score actions

(defclass section () ((data :accessor data :initform nil :initarg :data)))
(defmethod section-p ((obj t)) nil)
(defmethod section-p ((obj section)) t)

(defun section (&rest args)
  (make-instance 'section :data args))

(defun handle-hidden-sections ()
  (when *cmn-hidden-sections*
    (loop for hidden-section in *cmn-hidden-sections* do
      (initialize hidden-section) 
      (setf (ideal-stem-length hidden-section) .3)
      (setf (beam-width hidden-section) .1)
      (setf (beam-spacing hidden-section) .175)
      (setf (note-head-size hidden-section) .8)
      (setf (size hidden-section) (size *cmn-score*)))
    (setf *cmn-hidden-sections* nil)))

#|
(cmn (section (redundant-accidentals nil) (staff treble (meter 4 4) cs4 q cs4 q cs4 q cs4 q)) 
     (section (staff treble cs4 q cs4 q cs4 q cs4 q)) )
|#


;;;
;;; ----------------    system
;;;
;;; a system is a group of staves with auxiliary annotations (measure numbers, overall tempo indications, etc)
;;;   It has been pointed out to me that most others use "system" to mean the entire group of staves that make
;;;   up what I call a "line", and that the word "section" is used where I am using "system".  I was thinking
;;;   of "section" as "portion of a piece" (see above), not "portion of an ensemble".  Hopefully this use of
;;;   the wrong word won't cause too much confusion.


(defclass system-mixin (score-object-mixin)
  ((staves :initarg :staves :initform nil :reader staves)
   (bracketing :initarg :bracketing :initform nil :reader bracketing)))

(defclass system (system-mixin score-object)
  ((staves :accessor staves)
   (bracketing :accessor bracketing)))

(defclass write-protected-system (write-protect system-mixin)
  ())

(defmethod system-p ((obj t)) nil)
(defmethod system-p ((obj system-mixin)) t)

(defvar system (make-instance 'write-protected-system))

(defmethod descry ((system system-mixin) &optional stream controller)
  (format stream "~A~A~A~A~A"
	  "(system"
	  (if (bracketing system) 
	      (format nil " ~A:bracketing ~A"
 		      (if controller (format nil "~%       ") "")
		      (bracketing system))
	    "")
	  (if (staves system)
	      (format nil " ~A:staves (list ~{~%~A~})" 
		      (if controller (format nil "~%       ") "")
		      (loop for staff in (staves system) collect (descry staff stream (or controller system))))
	    "")
	  (if (next-method-p) (call-next-method system stream (or controller system)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((system system-mixin) &optional object)
  (let ((new-system (if (not object) (make-system)
		      (if (write-protected object) (copy object)
			object))))
    (setf (bracketing new-system) (bracketing system))
    (setf (staves new-system) (loop for staff in (staves system) collect (copy staff)))
    (if (next-method-p) (call-next-method system new-system))
    new-system))

(defun system (&rest objects)
  (let ((new-system (make-system)))
    (setf *cmn-system* new-system)
    (multiple-value-bind
	(staves actions marks)
	(notify new-system objects)
      (setf (staves new-system) staves)
      (setf (marks new-system) marks)
      (loop for act in actions do
	(funcall (action act) new-system (arguments act))))
    (setf *cmn-system* nil)
    new-system))

(defmethod notify ((system system) &optional objects)
  (let ((default-staff nil)
	(staves nil)
	(actions nil)
	(marks nil)
	(object-list objects)
	(default-staff-data nil))
    (loop while object-list do
      (let ((object (pop object-list)))
	(when object
	  (if (and (score-object-list-p object) (not (glyph-list-p object)))
	      (setf object-list (append (disgorge object) object-list))
	    (if (and (not default-staff-data)
		     (or (bracket-p object) (brace-p object)))
		(setf (bracketing system) object)
	      (if (staff-p object)
		  (progn
		    (if default-staff
			(push (apply #'staff (nreverse default-staff-data)) staves))
		    (setf default-staff-data nil)
		    (if (write-protected object)
			(setf default-staff object)
		      (progn
			(setf default-staff nil)
			(if (member object staves) (cmn-error "attempt to load the same staff twice"))
			(push object staves))))
		(if (not default-staff)
		    (if (self-acting-p object)
			(push object actions)
		      (if (or (text-p object) (sundry-p object) (glyph-list-p object))
			  (push object marks)
			(progn
			  (setf default-staff (make-write-protected-staff))
			  (push object default-staff-data))))
		  (progn
		    (if (not default-staff)
			(setf default-staff (make-write-protected-staff)))
		    (push object default-staff-data)))))))))
    (if default-staff (push (apply #'staff (nreverse default-staff-data)) staves))
    (values (nreverse staves) (nreverse actions) marks)))



;;;
;;; ----------------    staff 
;;;

(defclass staff-mixin (score-object-mixin)
  ((name :initarg :name :initform nil :reader %staff-name)
   (data :initarg :data :initform nil :reader staff-data)
   (size :initarg :size :initform nil :reader staff-size)
   (lines :initarg :lines :initform nil :reader staff-lines)
   (times :initarg :times :initform nil :reader times)
   (start-line :initarg :start-line :initform 0 :reader staff-start-line)
   (local-brace :initarg :local-brace :initform nil :reader staff-local-brace)
   (staff-name-x0 :initarg :staff-name-x0 :initform nil :reader staff-name-x0)
   (true-staff :initarg :true-staff :initform nil :reader true-staff)
   (back-staff :initarg :back-staff :initform nil :reader back-staff)
   (staff-user-data :initarg :staff-user-data :initform nil :reader staff-user-data)))

(defclass write-protected-staff (write-protect staff-mixin)
  ())

(defclass staff (staff-mixin score-object)
  ((name :accessor %staff-name)
   (data :accessor staff-data)
   (size :accessor staff-size)
   (lines :accessor staff-lines)
   (times :accessor times)
   (start-line :accessor start-line)
   (local-brace :accessor staff-local-brace)
   (staff-name-x0 :accessor staff-name-x0)
   (true-staff :accessor true-staff)
   (back-staff :accessor back-staff)
   (staff-user-data :accessor staff-user-data)))

(self-action staff-lines setf-lines)
(self-action staff-size setf-staff-size)
(self-action staff-name-x0 setf-staff-name-x0)
(self-action start-line setf-start-line)

(defmethod staff-y0 ((staff staff-mixin)) (box-y0 staff))
(defmethod (setf staff-y0) (val (staff staff-mixin)) (setf (box-y0 staff) val))

(defgeneric staff-name (obj &rest args))
(defgeneric setf-staff-name (obj val))
(defgeneric (setf staff-name) (val obj))

(defmethod setf-staff-name ((inst t) val) 
  (setf (%staff-name inst) val))

(defmethod staff-name ((val t) &rest args)
  (make-self-acting :action #'setf-staff-name 
		    :arguments (if (not args)
				   val
				 (apply #'ur-text (%%text :letters val 
							  :font-name (staff-name-font *cmn-score*)
							  :font-scaler (staff-name-font-scaler *cmn-score*))
					args))))
;;; see also make-staff-name in cmn1.lisp

(defmethod staff-name ((inst staff-mixin) &rest args)
  (declare (ignore args))
  (%staff-name inst))

(defmethod (setf staff-name) (val (inst staff-mixin))
  (setf (%staff-name inst) val))

(defmethod dsud ((stf staff-mixin) num-i)
  (if (not (staff-user-data stf)) 
      (setf (staff-user-data stf) (list 0 0 0 0)))
  (if (> (second num-i) 3) 
      (cmn-error "staff can't handle d~A~D" 
		 (if (evenp (second num-i)) "x" "y")
		 (floor (second num-i) 2)))
  (setf (nth (second num-i) (staff-user-data stf)) (first num-i)))

(defun identify-user-data (data stream)
  (if data
      (format stream "~{~A~}"
	  (loop for dat in data and i from 0 collect
	    (if (not (zerop dat))
		(format nil " (d~A~D ~,3F)" (if (evenp i) "x" "y") (floor i 2) dat)
	      "")))
    ""))	


(defvar staff (make-instance 'write-protected-staff))

(defun identify-staff (score staff)
  (loop for system in (systems score) and sys-num from 1 do
    (loop for stf in (staves system) and staff-num from 1 do
      (if (eq stf staff)
	  (return-from identify-staff 
	    (format nil "~A in system ~D (staff ~D)" 
		    (or (staff-name staff) "un-named staff")
		    sys-num staff-num))))))

(defmethod descry ((staff staff-mixin) &optional stream controller)
  (let ((prespace (if controller  "        " "")))
    (format stream "~A(staff~A~A~A~A~A~A~A~A~A~A~A)"
	    prespace
	    (if (staff-name staff) (format nil " :name ~S" (staff-name staff)) "")
	    (if (staff-name-x0 staff) (format nil " :staff-name-x0 ~,3F" (staff-name-x0 staff)) "")
	    (if (staff-size staff) (format nil " :size ~A" (staff-size staff)) "")
	    (if (staff-lines staff) (format nil " :lines ~A" (staff-lines staff)) "")
	    (if (times staff) (format nil " :times '~A" (times staff)) "")
	    (if (and (numberp (staff-start-line staff))
		     (> (staff-start-line staff) 0))
		(format nil " :start-line ~D" (staff-start-line staff)) "")
	    (if (staff-local-brace staff) (format nil " :local-brace ~A" (staff-local-brace staff)) "")
	    (if (true-staff staff) (format nil " :true-staff ~A" (true-staff staff)) "")
	    (if (staff-user-data staff) (format nil " :staff-user-data '~A" (staff-user-data staff)) "")
	    (if (next-method-p) (call-next-method staff stream (or controller staff)) "")
	    (if (staff-data staff) 
		(format nil "~%~A  :data (list~{~%            ~A~})" 
			prespace
			(loop for object in (staff-data staff) 
			 collect (descry object)))
	      ""))))

(defmethod copy ((staff staff-mixin) &optional object)
  (let ((new-staff (if (not object) (make-staff)
		     (if (write-protected object) (copy object)
		       object))))
    (setf (staff-data new-staff) (loop for obj in (staff-data staff) collect (copy obj)))
    (setf (staff-name new-staff) (staff-name staff))
    (setf (staff-name-x0 new-staff) (staff-name-x0 staff))
    (setf (staff-size new-staff) (staff-size staff))
    (setf (staff-lines new-staff) (staff-lines staff))
    (setf (times new-staff) (times staff))
    (setf (start-line new-staff) (staff-start-line staff))
    (setf (true-staff new-staff) (true-staff staff))
    (setf (back-staff new-staff) (back-staff staff))
    (setf (staff-user-data new-staff) (staff-user-data staff))
    (if (staff-local-brace staff) (setf (staff-local-brace new-staff) (copy (staff-local-brace staff))))
    (if (next-method-p) (call-next-method staff new-staff))
    new-staff))

(defmethod staff-p ((obj t)) nil)
(defmethod staff-p ((obj staff-mixin)) t)

(defun staff (&rest objects)
  (let ((new-staff (make-staff)))
    (multiple-value-bind
	(data actions local-brace)
	(notify new-staff objects)
      (setf *cmn-staff* new-staff)
      (setf (staff-data new-staff) data)
      (setf *cmn-staff-data* nil)
      (cmn-clear-pipe)
      (setf (staff-local-brace new-staff) local-brace)
      (loop for act in actions do
	(funcall (action act) new-staff (arguments act)))
      (if (staff-name new-staff)
	  (add-to-marks new-staff (list (make-staff-name))))
      (setf *cmn-staff* nil)
      new-staff)))

(defvar all-instrument-names 
    '("piccolo" "flute" "alto flute" "bass flute" 
      "oboe" "oboe d'amore" "english horn"
      "e-flat clarinet" "b-flat clarinet" "a clarinet" "bass clarinet" "e-flat contrabass clarinet" "b-flat contrabass clarinet"
      "bassoon" "contrabassoon"
      "soprano saxophone" "alto saxophone" "tenor saxophone" "baritone saxophone" "bass saxophone"
      "french horn" "b-flat piccolo trumpet" "e-flat trumpet" "d trumpet" "c trumpet" "b-flat trumpet"
      "cornet" "fluegelhorn" "bass trumpet"
      "alto trombone" "tenor trombone" "bass trombone" 
      "baritone horn" "euphonium" "tuba"
      "timpani" "vibraphone" "xylophone" "marimba" "glockenspiel" "chimes"
      "celesta" "piano"
      "violin" "viola" "cello" "contrabass"))

(defvar all-abbreviated-instrument-names 
    '("picc." "fl." "alt. fl." "bs. fl." 
      "ob." "ob. d'am." "e.h."
      "e-flat cl." "b-flat cl." "a cl." "bs. cl" "e-flat cbs. cl." "b-flat cbs. cl."
      "bsn." "cbsn."
      "sop. sax." "alt. sax." "ten. sax." "bar. sax." "bs. sax."
      "hn." "b-flat picc. tpt." "e-flat tpt." "d tpt." "c tpt." "b-flat tpt." "cor." "flhn." "bs. tpt."
      "alt. tbn." "ten. tbn." "bs. tbn."
      "bar." "euph." "tba." 
      "timp." "vibe." "xyl." "mar." "glock." "chm."
      "cel." "pno."
      "vn." "vla." "vc." "cb."))

(defmethod abbreviation-of-instrument-name (name)
  (let ((pos (position name all-instrument-names :test #'string-equal)))
    (or (and pos
	     (nth pos all-abbreviated-instrument-names))
	name)))

(defmethod abbreviation-of-instrument-name ((text text))
  (abbreviation-of-instrument-name (letters text)))

(defmethod tied-to ((staff staff-mixin))
  (make-self-acting
   :action #'(lambda (new-staff old-staff)
	       (setf (true-staff new-staff) old-staff)
	       (setf (staff-size new-staff) (staff-size old-staff))
	       nil)
   :arguments staff))

(defun notifiable (obj1 obj2)
  (or (and (audible-p obj1)
	   (or (rhythm-p obj2)
	       (sundry-p obj2)
	       (pause-p obj2)
	       (text-p obj2)
	       (glyph-list-p obj2)
	       (self-acting-p obj2)
	       (tag-p obj2)
	       (accidental-p obj2)
	       (dynamics-p obj2)))
      (and (score-object-p obj1)
	   (sundry-p obj2))))

(defmethod notify ((staff staff-mixin) &optional objects)
  ;; just make sure all unattached (write-protected) objects are repackaged -- for the
  ;; initial cases, place clef key meter name and size onto the current staff object (these can change)
  (let ((last-wp-object nil)
	(last-wp-data nil)
	(actions nil)
	(local-brace nil)
	(object-list objects))
    (setf *cmn-staff-data* nil)
    (setf *cmn-staff* nil)
    (cmn-clear-pipe)
    (loop while object-list do
      (let* ((pobj (pop object-list)))
	(when pobj
	  (let* ((cons-obj (write-protected pobj))
		 (ok-obj (if cons-obj (copy pobj) pobj)))
	    (if (and (score-object-list-p ok-obj) (not (glyph-list-p ok-obj)))
		(setf object-list (append (disgorge ok-obj) object-list))
	      (if (or (bracket-p ok-obj) (brace-p ok-obj))
		  (setf local-brace ok-obj)
		(let ((object nil))
		  (if last-wp-object
		      (if (not (notifiable last-wp-object ok-obj))
			  (progn
			    (push (funcall #'notify last-wp-object (nreverse last-wp-data)) *cmn-staff-data*)
			    (setf last-wp-data nil)
			    (setf last-wp-object nil)
			    (if cons-obj 
				(setf last-wp-object ok-obj)
			      (setf object ok-obj)))
			(push ok-obj last-wp-data))
		    (if cons-obj
			(setf last-wp-object ok-obj)
		      (setf object ok-obj)))
		  (when object
		    (if (self-acting-p object)
			(push object actions)
		      (if (or (text-p object) (glyph-list-p object) (sundry-p object))
			  (push (make-self-acting :action #'add-to-marks :arguments (list object)) actions)
			(push object *cmn-staff-data*)))))))))))
    (if last-wp-object (push (funcall #'notify last-wp-object (nreverse last-wp-data)) *cmn-staff-data*))
    (if (or (line-p (first *cmn-staff-data*)) (page-p (first *cmn-staff-data*))) (pop *cmn-staff-data*))
    (values (nreverse *cmn-staff-data*) (nreverse actions) local-brace)))

(defmethod display ((staff staff-mixin) container score &rest rest)
  (declare (ignore container))
  (let ((px0 (+ (box-x0 staff) (dxy-dx staff) (if (staff-user-data staff) (first (staff-user-data staff)) 0)))
	(py0 (+ (box-y0 staff) (dxy-dy staff)))
	(x1 (+ (box-x1 staff) (if (staff-user-data staff) (third (staff-user-data staff)) 0))))
    (when (or px0 py0 x1)
      ;; we are at x y and want the staff to go to x1.
      ;; we don't use %staff because the scale factor causes the dx-left amount to be nearly unpredictable
      ;; that is, the staff "character"'s origin is not the lower staff line's beginning
      (if (marks staff) (apply #'display-marks staff score rest))
      (let* ((x-back (box-x0 %staff))	;mimic font's staff character
	     (y-down (box-y0 %staff))
	     (line-width (staff-line-width score))
	     (lines (or (staff-lines staff) 5))
	     (sls (* 2 (staff-line-separation score)))
	     (x0 (+ px0 x-back))
	     (y0 (+ py0 y-down (* sls (staff-start-line staff)))))
	(comment score "staff lines")
	(setf (line-width score) line-width)
	(loop for i from 0 below lines do
	  (just-line score x0 y0 x1)
	  (incf y0 sls))		;.2545 -- this is to fit the clef signs -- 1018 total distance
	(draw score)
	(setf (line-width score) 0)
	(let* ((ssize (or *cmn-score-size* (scr-size score)))
	       (x0s (* ssize x0))
	       (x1s (* ssize x1))
	       (y0s (* ssize (+ py0 y-down)))
	       (y1s (* ssize (- y0 sls))))
	  (setf (scr-x score) x1s)
	  (setf (scr-y score) y1s)
	  (setf (scr-x0 score) (min (scr-x0 score) x0s x1s))
	  (setf (scr-y0 score) (min (scr-y0 score) y0s y1s))
	  (setf (scr-x1 score) (max (scr-x1 score) x0s x1s))
	  (setf (scr-y1 score) (max (scr-y1 score) y0s y1s)))))))

(defmethod house ((staff staff-mixin) score)
  (declare (ignore score))
  (setf (box-y1 staff) 1.0)
  (setf (center staff) 0.0))




(defclass score-object-list ()
  ((data :initarg :data :initform nil :accessor data)))

(defmethod score-object-list-p ((obj t)) nil)
(defmethod score-object-list-p ((obj score-object-list)) t)

(defmethod disgorge ((objlist score-object-list))
  (prog1
      (data objlist)
    (setf (data objlist) nil)))

(defmethod engorge (objects)
  (make-score-object-list :data objects))

(defmethod descry ((sob score-object-list) &optional stream controller)
  (format stream "~A~A~A~A"
	  (if (not controller) "(score-object-list" 
	    (if (data sob) (format nil "~%~A" prewhitespace)
	      ""))
	  (if (data sob)
	      (format nil " :data (list ~{ ~A~})" (loop for datum in (data sob) collect (descry datum)))
	    "")
	  (if (next-method-p) (call-next-method sob stream (or controller sob)) "")
	  (if (not controller) ")" "")))

(defmethod copy ((sob score-object-list) &optional object)
  (let ((new-sob (if (not object) (make-score-object-list)
		   (if (write-protected object) (copy object)
		     object))))
    (setf (data new-sob) (loop for datum in (data sob) collect (copy datum)))
    (if (next-method-p) (call-next-method sob new-sob))
    new-sob))

(defmethod display ((score-obj score-object-list) container score &rest rest)
  (when (not (invisible-matrix-p score-obj))
    (let ((sobs (data score-obj)))
      (when (output score)
	(matrix-front score (matrix score-obj))
	(loop for sob in sobs do (apply #'display sob container score rest))
	(if (c-output-in-progress score)
	    (c-print "show")
	  (g-just-show score))
	(matrix-back score (matrix score-obj))))))

(defmethod %rx ((server server) (score-obj score-object-list) &key)
  (if (identity-matrix-p score-obj)
      (loop for object in (data score-obj) sum (%rx server object))
    0))



(defclass glyph-list (score-object score-object-list font-mixin) ())

(defmethod glyph-list-p ((obj t)) nil)
(defmethod glyph-list-p ((obj glyph-list)) t)

(defmethod show ((server server) (glyph-obj glyph-list) &key (dx 0) (dy 0) size &allow-other-keys)
  (when (not (invisible-matrix-p glyph-obj))
    (let ((glyphs (loop for glyph in (data glyph-obj) collect (index glyph)))
	  (pstart "")
	  (update-xy t)
	  (ssize (or *cmn-score-size* (scr-size server)))
	  (pend (if (and (zerop dx) (zerop dy)) "show" "ashow")))
      (when (output server)
	(let ((font-case (or size (font-name glyph-obj)))
	      (c-out (c-output-in-progress server)))
	  (when font-case
	    (setf update-xy nil)
	    (if c-out (c-print (format nil " gsave~% /~A findfont ~D scalefont setfont" 
				       (if size (font server) (font-name glyph-obj))
				       (or size (font-size glyph-obj) (* (font-scaler glyph-obj) ssize))))
	      (g-new-font server 
			  (if size (font server) (font-name glyph-obj))
			  (or size (font-size glyph-obj) (* (font-scaler glyph-obj) ssize)) t)))
	  (if (pattern glyph-obj)
	      (multiple-value-bind
		  (qpstart qpend)
		  (funcall (pattern glyph-obj) server glyph-obj)
		(if qpstart (setf pstart qpstart))
		(if qpend (setf pend qpend))))
	  (if (identity-matrix-p glyph-obj)
	      (if (or (not (zerop dx)) (not (zerop dy)))
		  (if c-out
		      (c-print (format nil " ~A ~,3F ~,3F (~{\\~D~}) ~A~A" pstart (* dx ssize) (* dy ssize) glyphs pend
				       (if font-case " grestore" "")))
		    (g-patterned-glyphs-with-spacing server pstart glyphs (* dx ssize) (* dy ssize) pend font-case))
		(if c-out
		    (c-print (format nil " ~A (~{\\~D~}) ~A~A" pstart glyphs pend (if font-case " grestore" "")))
		  (g-patterned-glyphs server pstart glyphs pend font-case)))
	    (progn
	      (setf update-xy nil)
	      (if c-out
		  (c-print (format nil " ~A ~A(~{\\~D~}) [ ~{~A ~}] concat ~A grestore"
				   (if (not font-case) "gsave " "") pstart glyphs (map 'list #'not-rational (matrix glyph-obj)) pend))
		(g-transformed-patterned-glyphs server pstart glyphs (matrix glyph-obj) pend t))))))
      (if update-xy (%dxy server (%rx server glyph-obj :dx dx :size size) 0))
      (%update-bounds server))))

(defmethod %rx ((server server) (score-obj glyph-list) &key (dx 0) size &allow-other-keys)
  (if (identity-matrix-p score-obj)
      (let ((tsize (or size *cmn-score-size* (scr-size server))))
	(* tsize (loop for object in (data score-obj) sum (+ (g-rx object) dx))))
    0))

(defun decimal-to-octal (n &optional (nn 0) (pow 1)) 
  (if (< n 8) 
      (+ (* n pow) nn) 
    (multiple-value-bind 
	(int frac) 
	(floor n 8) 
      (decimal-to-octal int (+ nn (* frac pow)) (* pow 10)))))

(defun glyphs (&rest objects)
  (apply #'ur-glyphs (make-glyph-list) objects))

(defun %glyphs (&rest objects)
  (apply #'ur-glyphs (make-instance 'glyph-list) objects))

(defun ur-glyphs (obj &rest objects)
  (let ((glfs nil))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) obj (arguments act))
	  (if (integerp act)
	      (push (make-glyph :index act) glfs)
	    (if (characterp act)
		(push (make-glyph :index (decimal-to-octal (char-code act))) glfs)
	      (push act glfs))))))
    (setf (data obj) (nreverse glfs))
    obj))

(defmethod display ((text glyph-list) container score &rest rest)
  (if (not (member :just-fooling rest))
      (let* ((audp (and container (audible-p container)))
	     (x-off (+ (box-x0 text) (dxy-dx text) (if audp (box-x0 container) 0)))
	     (y-off (+ (box-y0 text) (dxy-dy text) (if audp (box-y0 container) 0))))
	(if (marks text) (apply #'display-marks text score rest))
	(moveto score x-off y-off)
	(show score text))))

;;; need copy, identify, descry here

(defmethod identify ((glf glyph-list))
  (format nil "(glyphs ~{~A ~}~A)"
	  (loop for dat in (data glf) collect (index dat))
	  (the-usual-suspects glf)))

(defmethod copy ((glf glyph-list) &optional object)
  (let ((new-glf (or object (make-glyph-list))))
    (if (next-method-p) (call-next-method glf new-glf))
    new-glf))

(defmethod descry ((glf glyph-list) &optional stream controller)
  (format stream "(glyph-list ~A)" 
   (if (next-method-p) (call-next-method glf stream (or controller glf)) "")))


;;; some help in error handing

(defun cmn-tick-pipe (obj)
  (setf cmn-pipe-2 cmn-pipe-1)
  (setf cmn-pipe-1 cmn-pipe-0)
  (setf cmn-pipe-0 obj))

(defun cmn-clear-pipe ()
  (setf cmn-pipe-0 nil)
  (setf cmn-pipe-1 nil)
  (setf cmn-pipe-2 nil))

(defun cmn-pipe-context ()
  (if cmn-pipe-0
      (let ((cmn0 (identify cmn-pipe-0)))
	(format nil " recently evaluated note~P:~A~A~A~A~A~A~%~VT"
		(if cmn-pipe-1 2 1)
		(if cmn-pipe-2 (format nil "~%        ") " ")
		(if cmn-pipe-2 (identify cmn-pipe-2) "")
		(if cmn-pipe-1 (format nil "~%        ") " ")
		(if cmn-pipe-1 (identify cmn-pipe-1) "")
		(if cmn-pipe-1 (format nil "~%        ") " ")
		cmn0
		(length cmn0)))
    ""))

(defun system-context (sys)
  (if sys
      (if (and (systems *cmn-score*) 
	       (> (length (systems *cmn-score*)) 1))
	  (let ((pos (position sys (systems *cmn-score*))))
	    (if pos (format nil "in system ~D " pos)
	      "in an unattached system "))
	"")
    ""))

(defun staff-context (stf)
  (if stf
      (let ((pos (and *cmn-system* 
		      (staves *cmn-system*) 
		      (> (length (staves *cmn-system*)) 1)
		      (position stf (staves *cmn-system*)))))
	(format nil "~A(staff~A~A~A "
		(if pos (format nil "in staff ~D " pos) "")
		(if (staff-size stf) (format nil " (staff-size ~,3F)" (staff-size stf)) "")
		(if (staff-lines stf) (format nil " (staff-lines ~D)" (staff-lines stf)) "")
		(if (staff-name stf) (format nil " (staff-name ~S)" (staff-name stf)) "")))
    ""))

(defun staff-data-start (stf obj owner)
  (let ((pos (and stf obj (staff-data stf)
		  (or (and owner (position owner (staff-data stf)))
		      (position obj (staff-data stf))))))
    (if (and stf (staff-data stf))
	(format nil "~{~A ~}~A"
		(loop for i from 0 below (min (or pos 5) 5) and note in (staff-data stf)  
		 if (and (not (staff-p note)) (not (bar-p note)) (not (eq obj note)) (not (eq owner note)))
		 collect (identify note))
		(if (or (and pos (> pos 8)) 
			(and (not pos) (> (length (staff-data stf)) 8)))
		    " ... " ""))
      "")))

(defun staff-data-continuation (stf obj owner sys stf0)
  (let ((pos (and stf obj (staff-data stf)
		  (or (and owner (position owner (staff-data stf)))
		      (position obj (staff-data stf))
		      (length (staff-data stf))))))
    (if (and pos (> pos 4))
	(let ((stf1 (format nil "~{~A ~}" (loop for n from (max 5 (- pos 3)) below pos
					   collect (identify (nth n (staff-data stf)))))))
	  (format nil "~A~%~VT"		;~V on p582 CLtL
		  stf1 
		  (+ 2 (length sys) (length stf0) (length stf1)))) 
					;used to include (length stf) which doesn't make any sense
					;but neither does the rest of this calculation -- has something fallen out somewhere?
      (cmn-pipe-context))))

(defun owner-context (obj owner)
  (format nil "~A~A~A"
	  (if obj (identify obj))
	  (if owner " in " "")
	  (if owner (identify owner) "")))

(defun cmn-error-1 (format-string &rest args)
  (let* ((sys (system-context *cmn-system*))
	 (stf (staff-context *cmn-staff*))
	 (stf0 (staff-data-start (or *cmn-staff*
				     (and *cmn-staff-data*
					  (make-staff :data (reverse *cmn-staff-data*))))
				 *cmn-object* *cmn-owning-object*))
	 (stf1 (staff-data-continuation (or *cmn-staff* 
					    (and *cmn-staff-data*
						 (make-staff :data (reverse *cmn-staff-data*))))
					*cmn-object* *cmn-owning-object*
					sys stf0))
	 (obj (identify *cmn-object*))
	 (owner (if *cmn-owning-object* (identify *cmn-owning-object*) "")))
    (format nil "~A:~%  ~A~A~A~A~A"
	   (if args
	       (apply #'format nil format-string args)
	     format-string)
	   sys stf stf0 stf1 (owner-context obj owner))))

(defun cmn-error (format-string &rest args)
  (error (apply #'cmn-error-1 format-string args)))

(defun cmn-warn (format-string &rest args)
  (warn (apply #'cmn-error-1 format-string args)))

(defmethod notify :before (any-object &optional objects)
  (declare (ignore objects))
  (setf *cmn-object* any-object))

(defmethod display :before (any-object container score &rest rest)
  (declare (ignore score rest))
  (setf *cmn-object* any-object)
  (setf *cmn-owning-object* container))

(defmethod copy :before (any-object &optional other-object)
  (setf *cmn-object* any-object)
  (setf *cmn-owning-object* other-object))

(defmethod house :before (any-object score)
  (declare (ignore score))
  (setf *cmn-object* any-object))




;;;
;;; ----------------    included .eps files
;;;
;;; see graphics function in cmn1.lisp

(defun include-eps-defs (server)
  (let ((cmd
	 "/BEGINEPSFILE { %def
    /EPSFsave save def
    0 setgray 0 setlinecap 1 setlinewidth 0 setlinejoin 10 setmiterlimit [] 0 setdash
    newpath
    /showpage {} def
} bind def
/ENDEPSFILE { %def
    EPSFsave restore
} bind def
"))
    (if (c-output-in-progress server)
	(c-print cmd)
      (format (output server) "~A" cmd))))

(defun make-ISO-encoded-version (server font-name new-font-name)
  (let ((cmd (format nil
		     " /~A findfont
 dup length dict begin
   {1 index /FID ne {def} {pop pop} ifelse} forall
   /Encoding ISOLatin1Encoding def
   currentdict
 end
 /~A exch definefont pop
" 
		     font-name new-font-name)))
    (if (c-output-in-progress server)
	(c-print cmd)
      (format (output server) "~A" cmd))))


(defun include-rectfill-definition (server)
  (let ((cmd				;taken from Postscript Language Reference Manual, D.3.2 p587
" /RF {
 gsave newpath
 4 -2 roll moveto
 dup 0 exch rlineto exch 0 rlineto neg 0 exch rlineto closepath
 fill grestore
 } bind def
"))
    (if (c-output-in-progress server)
	(c-print cmd)
      (format (output server) "~A" cmd))))
    

(defun include-eps-defs-and-whatever (server)
  (include-eps-defs server)
  (if (c-output-in-progress server) (c-print "/cmncolor 0 def") (format (output server) " /cmncolor 0 def~%"))
  (if (= PS-level 1) (include-rectfill-definition server))
  (if (prologue server)
      (loop for head in (reverse (prologue server)) do
	(funcall head server))))

(defun read-4-numbers (gbb)
  (let ((gbb-as-list nil))
    (loop for i from 0 to 3 do
      (multiple-value-bind 
	  (int len)
	  (read-from-string gbb)
	(push int gbb-as-list)
	(if (< len (length gbb)) (setf gbb (subseq gbb len)))))
    (nreverse gbb-as-list)))

(defun include-eps (server file x0 y0 &optional (xscl 1.0) (yscl 1.0))
  (let ((gbb nil)
	(size (scr-size server))
	(c-out (c-output-in-progress server)))
    (if c-out
	(c-print (format nil "BEGINEPSFILE~%  ~A ~A translate~% ~,3F ~,3F scale" 
			 (not-rational (* x0 size))
			 (not-rational (* y0 size))
			 xscl yscl))
      (format (output server) "BEGINEPSFILE~%  ~A ~A translate~% ~,3F ~,3F scale~%" 
	      (not-rational (* x0 size))
	      (not-rational (* y0 size))
	      xscl yscl))
    (setf gbb (read-4-numbers (insert-eps-file server file)))
    (if c-out
	(c-print "ENDEPSFILE")
      (format (output server) "ENDEPSFILE~%"))
    (list (* x0 size)
	  (* y0 size) 
	  (+ (* x0 size) (* xscl (third gbb))) 
	  (+ (* y0 size) (* yscl (fourth gbb))))))

(defun include-eps-with-matrix-and-pattern (server file x0 y0 matrix pattern)
  (let ((gbb nil)
	(local-matrix (copy-list matrix))
	(pstart (if pattern (funcall pattern server pattern) ""))
	(c-out (c-output-in-progress server)))
    (setf (fifth local-matrix) (* x0 (scr-size server)))
    (setf (sixth local-matrix) (* y0 (scr-size server)))
    (if c-out
	(c-print (format nil "BEGINEPSFILE~% ~A [ ~{~A ~}] concat~%" pstart (map 'list #'not-rational local-matrix)))
      (format (output server) "BEGINEPSFILE~% ~A [ ~{~A ~}] concat~%" pstart (map 'list #'not-rational local-matrix)))
    (setf gbb (read-4-numbers (insert-eps-file server file)))
    (if c-out
	(c-print "ENDEPSFILE")
      (format (output server) "ENDEPSFILE~%"))
    (transform-box matrix (list (* x0 (scr-size server)) (* y0 (scr-size server))) gbb)))

(defun insert-eps-file (server file)
  ;; ideally we would leave the PostScript comments in the inserted file alone, but
  ;; that confused WriteNow which believes the first %%BoundingBox it sees and it 
  ;; quits upon encountering %%Trailer.  So, we'll strip out those portions...
  (let ((gbb nil))
    (with-open-file (s file)
      (loop do
	(let ((text (read-line s nil :at-end)))
	  (if (not (eq text :at-end))
	      (progn
		(if (and (> (length text) 15) 
			 (string= "%%BoundingBox" (subseq text 0 13)))
		    (setf gbb (subseq text 14))
		  (if (or (< (length text) 9) 
			  (not (string= "%%Trailer" (subseq text 0 9))))
		      (if (c-output-in-progress server)
			  (c-print text)
			(format (output server) "~A~%" text)))))
	    (return-from insert-eps-file gbb)))))))

(defun get-eps-bounding-box (file)
  (with-open-file (s file)
    (loop do
      (let ((text (read-line s nil :at-end)))
	(if (and (not (eq text :at-end))
		 (> (length text) 25) 
		 (string= "%%BoundingBox" (subseq text 0 13)))
	    (return-from get-eps-bounding-box (read-4-numbers (subseq text 14))))))))

(defun output-page-break (score page)
  ;; output a page break to the .ps file
  (if (c-output-in-progress score)
      (progn
	(c-print " showpage")
	(c-print (format nil "%%Page: ~D ~D" (1+ page) (1+ page)))
	(c-print (format nil " /~A findfont ~D scalefont setfont~% 0 setlinewidth" Music-Font (scr-size score))))
    (progn 
      (g-show-page score)
      (g-page-number score (1+ page))
      (g-new-font score Music-Font (scr-size score) nil)
      (g-set-line-width score 0))))

(defun set-gray-scale (score num)
  (if (c-output-in-progress score)
      (let ((snum (if (integerp num) (format nil "~D" num) (format nil "~,2F" num))))
	(c-print (format nil " /cmncolor ~A def  ~A setgray~%" snum snum)))
    (g-begin-gray-scale score num t)))

(defun bind-gray-scale (score num)
  (if (c-output-in-progress score)
      (let ((snum (if (integerp num) (format nil "~D" num) (format nil "~,2F" num))))
	(c-print (format nil " ~A setgray~%" snum)))
    (g-begin-gray-scale score num nil)))

(defun unbind-gray-scale (score)
  (if (c-output-in-progress score)
      (c-print (format nil " cmncolor setgray~%"))
    (g-end-gray-scale score)))



#|
;;; tests
(cmn (size 100) (staff (staff-lines 0) (treble (gray-scale .75)) 
     (c4 w (text "hi" (gray-scale .5)) (begin-tie (gray-scale .5)) (marcato (dy .125) (gray-scale .5) (dx .75))) 
     (c4 w end-tie) (bass (outlined 1)) g2 w))
|#