;;; -*- syntax: common-lisp; package: cmn; base: 10; mode: lisp -*-
;;;
;;; Postscript or Quickdraw output
;;;
;;; The default in CMN is to produce a file containing a Postscript program.
;;; This is awkward for previewing (and eventually editing), so this file
;;; provides Quickdraw equivalents, and eventually (?) common-graphics, clim, X...
;;;
;;; For the NeXT, we need a remote procedure call package to communicate with
;;; NeXT's Display PostScript window server -- objective C is apparently
;;; incompatible with Lisp, and you can't get access to all the OS events
;;; in raw DPS code.
;;;
;;; The field output-type is used only in this file.  I added output-state
;;; for Quickdraw's benefit.  It also occurs only here, so other systems
;;; can use it in any way they like.

(in-package :cmn)

;;; we use PostScript's coordinate system here.  Many of these functions are
;;; optimizations of simpler sequences.


;;; -------- Quickdraw stuff ------------------------
;;;
;;; REMAINING BUGS:
;;;    nested coordinate transformations ignored
;;;    many text transformations (scaling, rotations etc) ignored
;;;    need easier way to call up separate pages and move quickly around in the scrolled view
;;;    not all font names are translatable -- should we set up a table, or come up with a better font naming convention?
;;;    "objects" are not mouse active
;;;    flags on scaled staves are not drawn -- cannot figure out why not!
;;;
;;; QUICKDRAW SUGGESTIONS: 
;;;    use the Sonata screen fonts (sizes like 14,18,20,24,36) -- these look very much better
;;;    increase the page-width, page-height, and size fields by the same scale factor -- the bigger the better
;;;    set the slur-thickness and tie-thickness to .1 or even more
;;;    since we're just rounding whatever Postscript sends in here, many scores, especially small ones, can
;;;       look really ragged -- they'll be ok when sent to the printer.         

(defvar *cmn-top-y* 0)			; for QuickDraw y axis inversion (durned window assumes (0,0) is top left corner)
					; this is set in cmn4.lisp when we know where the top left corner is in Postscript coordinates
(defvar *cmn-space* "  ")		; for prettier auto-generated lisp code
(defvar *cmn-quickdraw-bounds* nil)

;;; now try to kludge around the Mac function size limit -- we hit this even in a simple case like moz.cmn!
(defvar *cmn-quickdraw-functions* 0)
(defvar *cmn-quickdraw-lines* 0)
(defvar *cmn-quickdraw-split-safe* t)
(defvar *cmn-quickdraw-max-lines 400)
(defvar *cmn-quickdraw-picture-in-progress* nil)
(defvar *cmn-quickdraw-not-top-level-functions* nil)
(defvar *cmn-quickdraw-picture-first-function* nil)
(defvar *cmn-quickdraw-saved-top-y* 0)
(defvar *cmn-quickdraw-last-pen-size* 0) ;optimize out repeated pen sizes (rounded numbers are often the same)
(defvar *cmn-quickdraw-slur-accuracy* 5)
(defvar *cmn-quickdraw-saved-font* nil)
(defvar *cmn-quickdraw-current-font* nil)
(defvar *cmn-quickdraw-pending-font* nil)
(defvar *cmn-quickdraw-actual-font* nil)
(defvar *cmn-quickdraw-current-font-size* nil)
(defvar *cmn-quickdraw-text-style* 0)

(defun check-function-size (server &optional forced)
  (incf *cmn-quickdraw-lines*)
  (when (or forced
	    (and *cmn-quickdraw-split-safe*
		 (> *cmn-quickdraw-lines* *cmn-quickdraw-max-lines)))
    (setf *cmn-quickdraw-lines* 0)
    (if *cmn-quickdraw-picture-in-progress* (push *cmn-quickdraw-functions* *cmn-quickdraw-not-top-level-functions*))
    (incf *cmn-quickdraw-functions*)
    (setf *cmn-quickdraw-last-pen-size* 0)
    (format (output server) "  )~%~
                             #'(lambda (*cmn-view*)~%~
                             ~2T(declare (ignore-if-unused *cmn-view*))~%")))

;;; Quickdraw font specs are slightly different from NeXT's version of Postscript -- we have to split off
;;; the family name, change the style to a keyword and include the size.  This naming convention is not
;;; that of Adobe, or really Postscript, so maybe some more neutral naming convention should be adopted.

(defun Ps-to-Qd-font-name (name)
  ;; we'll take the portion up to the "-" as the family name, and anything trailing as
  ;; the style name -- turned into a symbol
  (let* ((dashloc (position #\- name))
	 (family-name (if dashloc (subseq name 0 dashloc) name))
	 (type (if dashloc (subseq name (1+ dashloc)))))
    (list family-name
	  (if type
	      (if (or (string-equal type "italic") (string-equal type "oblique")) :italic
		(if (string-equal type "bold") :bold
		  ;; bolditalic occurs on NeXT
		  :plain))
	    :plain))))

(defun Post-to-Quick-gray-scale (num)
  (if (or (not num) (zerop num)) "*black-pattern*"
    (if (< num .33) "*dark-gray-pattern*"
      (if (< num .67) "*gray-pattern*"
	(if (< num 1.0) "*light-gray-pattern*"
	  "*white-pattern*")))))

(defun encode-text-face (n)		;Inside Mac I-210 -- these bits can obviously by OR'd
  (if (eq n :italic) 2 
    (if (eq n :bold) 1
      (if (eq n :outline) 8
	(if (eq n :shadow) 16
	  0)))))

;;; -------------------------------------------------


;;; originally I intended to use specializations of the server class to handle non-PostScript
;;; output, but as years went by with no such output possibilities, the code involved became
;;; harder and harder to handle in that way; case statements to the rescue...

(defun g-lineto (server xs ys)
  ;; draw a line (hopefully in the current line-width) from the current point to (xs ys)
  ;; the current point in cmn is [(x server),(y server)]
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F lineto~%" xs ys))
    (:quickdraw  
     (progn 
       (check-function-size server)
       (format (output server) "~A(#_LineTo ~D ~D)~%" *cmn-space* (round xs) (round (- *cmn-top-y* ys)))))
    ;; common-graphics (draw-to (output server) (make-position xs (- *cmn-top-y* ys)))
    ))

(defun g-rlineto (server dxs dys)
  ;; draw a line from the current point by (dxs dys)
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F rlineto~%" dxs dys))    
    (:quickdraw  
     (progn
       (check-function-size server)
       (format (output server) "~A(#_Line ~D ~D)~%" *cmn-space* (round dxs) (round (- dys)))))
    ;; common-graphics (draw-by (output server) (make-position dxs (- dys)))
    ))

(defun g-moveto (server xs ys)
  ;; move (without drawing) the current point to (xs ys)
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F moveto~%" xs ys))
    (:quickdraw  
     (progn 
       (check-function-size server)
       (format (output server) "~A(#_MoveTo ~D ~D)~%" *cmn-space* (round xs) (round (- *cmn-top-y* ys)))))
    ;; common-graphics (draw-to (output server) (make-position xs (- *cmn-top-y* ys)))
    ))

(defun g-rmoveto (server dxs dys)
  ;; move the current point by (dxs dys)
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F rmoveto~%" dxs dys))    
    (:quickdraw  
     (progn 
       (check-function-size server)
       (format (output server) "~A(#_Move ~D ~D)~%" *cmn-space* (round dxs) (round (- dys)))))
    ;; common-graphics (move-by (output server) (make-position dxs (- dys)))
    ))

(defun g-comment (server x)
  ;; place a comment in the graphics output stream 
  (case (output-type server)
    (:postscript (format (output server) "% ~A~%" x))
    (:quickdraw  (format (output server) "~A;; ~A~%" *cmn-space* x))
    ;; this is purely a debugging convenience, so it can be ignored
    ))

(defun g-set-line-width (server xs)
  ;; set line-width for subsequent drawing to xs (can be 0 = thinnest possible line)
  (case (output-type server)
    (:postscript (format (output server) " ~A setlinewidth~%" (not-rational xs)))
    (:quickdraw  
     (let ((ps (max 1 (ceiling xs))))
       (when (/= ps *cmn-quickdraw-last-pen-size*)
	 (check-function-size server)
	 (format (output server) "~A(#_PenSize ~D ~D)~%" *cmn-space* ps ps)
	 (setf *cmn-quickdraw-last-pen-size* ps))))
    ))

(defun g-begin-pattern (server pstart &optional (saved t))
  ;; prepare to draw in some pattern -- pstart (returned by the pattern function) is the set-up sequence
  (case (output-type server)
    (:postscript (if saved (format (output server) " gsave ~A~%" pstart) 
		   (format (output server) " ~A~%" pstart)))
    (:quickdraw 
     (progn 
       (check-function-size server)
       (if saved (g-save server))
       (if (and pstart (numberp pstart))
	   (format (output server) "~A(#_PenPat ~(~A~))~%~A" 
		   *cmn-space* 
		   (Post-to-Quick-gray-scale pstart)
		   ;; this TextMode stuff is not quite right -- appears to mess up subsequent lines
		   (if (zerop pstart) ""
		     (if (= pstart 1) (format nil "~A(#_TextMode 3)~%" *cmn-space*)
		       (format nil "~A(#_TextMode 49)~%" *cmn-space*))))
	 (funcall pstart server))))
    ))

(defun g-end-pattern (server pend)
  ;; stop current pattern, return to previous (or some reasonable default)
  (case (output-type server)
    (:postscript (format (output server) " ~A grestore~%" pend))
    (:quickdraw  
     (progn 
       (check-function-size server)
       (g-restore server)
       (if pend 
	   (funcall pend server)
	 (format (output server) "~A(#_PenPat *black-pattern*) (#_TextMode 1)~%" *cmn-space*))))
    ))

(defun g-draw (server)
  ;; if some object is awaiting a draw decision, send it out
  (case (output-type server)
    (:postscript (format (output server) " stroke~%"))
    (:quickdraw )			;drawing is immediate in quickdraw
    ))

(defun g-dashed-line (server x y pattern rl)
  ;; can be horizontal or vertical (dashed bar line for example)
  (case (output-type server)
    (:postscript			
     (format (output server) " ~,2F ~,2F ~A [~{~A ~}] 0 setdash stroke~% [] 0 setdash~%" 
	     x y 
	     (if rl "rlineto" "lineto")
	     (map 'list #'not-rational pattern)))
    (:quickdraw 			; dashed lines in quickdraw are done by hand via line/move
     (let* ((dline (first pattern))
	    (dspace (second pattern))
	    (dlen (+ dspace dline)))
       (if (or (zerop dspace) (zerop dline))
	   (g-rlineto server x y)
	 (let* ((dx (if rl x (- x (scr-x server))))
		(dy (if rl y (- y (scr-y server))))
		(dashes (ceiling (sqrt (+ (* dx dx) (* dy dy))) dlen)))
	   (when (plusp dashes)
	     (let* ((ddx (if (zerop dy) dlen (/ dx dashes)))
		    (ddy (if (zerop dx) dlen (/ dy dashes)))
		    (dashx (* ddx (/ dline dlen)))
		    (dashy (* ddy (/ dline dlen)))
		    (spacex (- ddx dashx))
		    (spacey (- ddy dashy))
		    (lastdx (if (plusp dx) (- dx (* (1- dashes) dlen)) 0))
		    (lastdy (if (plusp dy) (- dy (* (1- dashes) dlen)) 0)))
	       (loop for i from 0 below (1- dashes) and 
	                 x0 from (scr-x server) by ddx and 
	                 y0 from (scr-y server) by ddy do
		 (g-rlineto server dashx dashy)
		 (g-rmoveto server spacex spacey))
	       (if (or (plusp lastdx) (plusp lastdy))
		   (g-rlineto server lastdx lastdy))))))))
    ))

(defun g-just-show (server)
  (case (output-type server)
    (:postscript (format (output server) " show~%"))
    (:quickdraw )			; text display is immediate in Quickdraw
    ))

(defun g-show (server glf-index)
  ;; simplest text case -- display the glf at the current point
  ;; the index is passed as an octal integer (that is, printed directly as is but read as octal by Postscript)
  (case (output-type server)
    (:postscript (princ (format nil  " (\\~D) show~%" glf-index) (output server)))
    (:quickdraw  
     (progn 
       (check-function-size server)
       ;; can't use stream-tyo because it is buggy inside a picture
       (establish-pending-font server)
       (format (output server) "~A(#_DrawChar (code-char #o~D))~%" *cmn-space* glf-index)))
    ))

(defun g-glyphs (server pstart glyphs pend backup)
  ;; pstart and pend can be patterns, but there's no matrix transformation involved here
  (case (output-type server)
    (:postscript (princ (format nil " ~A(~{\\~D~}) ~A~A~%" pstart glyphs pend (if backup " grestore" "")) (output server)))
    (:quickdraw  
     (progn
       (loop for g in glyphs do (g-show server g))
       (if backup (g-restore server))))
    ))

(defun g-scaled-glyphs (server pstart glyphs pend need-save matr)
  ;; here we can have a matrix
  (case (output-type server)
    (:postscript (princ (format nil " ~A~A(~{\\~D~}) [ ~{~A ~}] concat ~A grestore~%"
				(if need-save "gsave " "")
				pstart glyphs (map 'list #'not-rational matr) pend)
			(output server)))
    (:quickdraw
     (progn
       (if need-save (g-save server))
       ;(g-comment server (format nil "g-scaled-glyphs at ~D ~D" (round (scr-x server)) (round (- *cmn-top-y* (scr-y server)))))
       (if (/= (first matr) 1) 
	   (g-qd-new-font server *cmn-quickdraw-current-font* (first matr)))
       (g-glyphs server pstart glyphs pend t)
       (g-restore server)))
    ))

(defun g-patterned-text (server pstart text pend)
  (case (output-type server)
    (:postscript (format (output server) " ~A(~A) ~A~%" pstart (ps-letters text) pend))
    (:quickdraw  
     (progn 
       (check-function-size server)
       (establish-pending-font server)
       ;; use of %put-byte here (instead of with-pstrs) didn't speed redisplay at all
       (format (output server) "~A(with-pstrs ((*str* ~S)) (#_DrawString *str*))~%" *cmn-space* text)))
    ))

(defun g-patterned-text-with-spacing (server pstart text dx dy)
  (case (output-type server)
    (:postscript (format (output server) " ~A ~,3F ~,3F (~A) ashow~%" pstart dx dy (ps-letters text)))
    (:quickdraw 
     (loop for char in text do
       (g-patterned-text server pstart char nil)
       (g-rmoveto server dx dy)))
    ))

(defun g-patterned-glyphs (server pstart glyphs pend &optional (saved t))
  (case (output-type server)
    (:postscript (princ (format nil " ~A (~{\\~D~}) ~A~A~%" pstart glyphs pend (if saved " grestore" "")) (output server)))
    (:quickdraw  
     (progn
       (g-glyphs server pstart glyphs pend saved)))
    ))

(defun g-patterned-glyphs-with-spacing (server pstart glyphs dx dy pend &optional (saved t))
  (case (output-type server)
    (:postscript (princ (format nil " ~A ~,3F ~,3F (~{\\~D~}) ~A~A~%" 
				pstart dx dy glyphs pend (if saved " grestore" ""))
			(output server)))
    (:quickdraw
     (progn
       (loop for g in glyphs do
	 (g-patterned-glyphs server pstart (list g) pend nil)
	 (g-rmoveto server dx dy))
       (if saved (g-restore server))))
    ))

(defun g-transformed-patterned-glyphs (server pstart glyphs matrix pend &optional (saved t))
  (case (output-type server)
    (:postscript (princ (format nil " ~A ~A(~{\\~D~}) [ ~{~A ~}] concat ~A grestore~%"
				(if (not saved) "gsave " "") pstart glyphs (map 'list #'not-rational matrix) pend)
			(output server)))
    (:quickdraw				; saved=restore *cmn-saved-font*
     (progn
       (if (not saved) (g-save server))
       (g-comment server "g-transformed-patterned-glyphs which is unimplemented...")
       (g-restore server)))
    ))


(defun g-new-font (server font size &optional (saved t))
  ;; establish font as new default -- save old if saved
  (case (output-type server)
    (:postscript (format (output server) " ~A/~A findfont ~D scalefont setfont~%" 
			 (if saved (format nil " gsave~% ") "")
			 font (round size)))
    (:quickdraw 
     ;; use of set-view-font-codes did not speed this up at all, so I'll leave it the old way since it's easier to read
     (let ((new-font-name (Ps-to-Qd-font-name font)))
       (check-function-size server)
       (if saved (g-save server))
       (when (and saved (not *cmn-quickdraw-saved-font*))
	 (setf *cmn-quickdraw-saved-font* (list (first new-font-name) (round size) (second new-font-name))))
       (setf *cmn-quickdraw-current-font* (list (first new-font-name) (round size) (second new-font-name)))
       (setf *cmn-quickdraw-current-font-size* (round size))
       (g-qd-new-font server *cmn-quickdraw-current-font*)))
    ))

(defun g-qd-new-font (server font &optional (scaler 1))
  (declare (ignore server))
  (let*	((new-size (abs (round (* scaler (second font))))) ;abs because we can't transform characters yet
	 (new-face (encode-text-face (third font)))
	 (font-number #+mcl (ash (ccl:font-codes (list (first font) new-size (third font))) -16)
		      #-mcl 0))
    ;(g-comment (output server) (format nil "pending-font '(~S ~D :~(~A~))~%" (first font) new-size (third font)))
    (setf *cmn-quickdraw-pending-font* (list new-size new-face font-number))
    ))

(defun establish-pending-font (server)
  (when *cmn-quickdraw-pending-font*
    (when (not (equal *cmn-quickdraw-pending-font* *cmn-quickdraw-actual-font*))
      (format (output server) "~A(#_TextSize ~D) (#_TextFace ~D) (#_TextFont ~D)~%"
	      *cmn-space* 
	      (first *cmn-quickdraw-pending-font*) 
	      (+ (second *cmn-quickdraw-pending-font*)  *cmn-quickdraw-text-style*)
	      (third *cmn-quickdraw-pending-font*))
      (setf *cmn-quickdraw-actual-font* (copy-list *cmn-quickdraw-pending-font*)))
    (setf *cmn-quickdraw-pending-font* nil)))

(defun output-bezier (server x0 y0 x1 y1 x2 y2 x3 y3 n)
  (let* ((cx (* 3 (- x1 x0)))
	 (cy (* 3 (- y1 y0)))
	 (bx (- (* 3 (- x2 x1)) cx))
	 (by (- (* 3 (- y2 y1)) cy))
	 (ax (- x3 x0 cx bx))
	 (ay (- y3 y0 cy by))
	 (incr (/ 1.0 n)))
    (if (or (/= x0 (scr-x0 server))
	    (/= y0 (scr-y0 server)))
	(g-moveto server x0 y0))
    (loop for i from 0 to 1 by incr do
      (g-lineto server 
		(+ x0 (* i (+ cx (* i (+ bx (* i ax))))))
		(+ y0 (* i (+ cy (* i (+ by (* i ay))))))))))

(defun g-curveto (server x0 y0 x1 y1 x2 y2)
  ;; draw Bezier curve
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F ~,2F ~,2F ~,2F ~,2F curveto~%" x0 y0 x1 y1 x2 y2))
    (:quickdraw  (output-bezier server (scr-x server) (scr-y server) x0 y0 x1 y1 x2 y2 *cmn-quickdraw-slur-accuracy*))
    ;; Postscript uses a cubic Bezier curve, as opposed to the quadratic Quickdraw font-manager version
    ))

(defun g-thick-vertical-line (server sws x0s y0s y1s)
  ;; draw a vertical line of thickness sws at (x0s y0s) to (x0s y1s), reset line width to 0
  ;; called to draw stems in cmn
  (case (output-type server)
    (:postscript (format (output server) 
			 " ~,3F setlinewidth ~,2F ~,2F moveto ~,2F ~,2F lineto stroke 0 setlinewidth~%" 
			 sws x0s y0s x0s y1s))
    (:quickdraw (progn 
		  (g-set-line-width server sws) 
		  (g-moveto server x0s y0s) 
		  (g-lineto server x0s y1s) 
		  (g-set-line-width server 0)))
    ))

(defun g-vertical-line (server x0s y0s y1s)
  ;; draw a vertical line at (x0s y0s) to (x0s y1s) (assume current width)
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F moveto ~,2F ~,2F lineto~%" x0s y0s x0s y1s))
    (:quickdraw (progn 
		  (g-moveto server x0s y0s) 
		  (g-lineto server x0s y1s)))
    ))

(defun g-horizontal-line (server x0s y0s x1s)
  ;; draw a horizontal line from (x0s y0s) to (x1s y0s) (assume current width)
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F moveto ~,2F ~,2F lineto~%" x0s y0s x1s y0s))
    (:quickdraw (progn 
		  (g-moveto server x0s y0s) 
		  (g-lineto server x1s y0s)))
    ))


(defun g-begin-filled-polygon (server)
  (case (output-type server)
    (:postscript )			;Postscript paths handle this automatically
    (:quickdraw 
     (progn
       (setf *cmn-quickdraw-split-safe* nil)
       (push :polygon (output-state server))
       (format (output server) "~A(setf (view-get *cmn-view* 'my-poly) (#_OpenPoly))~%" *cmn-space*)))
    ))
     
(defun g-filled-rectangle (server x0 y0 dx dy)
  (case (output-type server)
    (:postscript (format (output server) 
			 (if (= PS-level 1)
			     " ~,2F ~,2F ~,2F ~,2F RF~%"
			   " ~,2F ~,2F ~,2F ~,2F rectfill~%")
			 x0 y0 dx dy))
    (:quickdraw 
     (progn 
       (check-function-size server)
       (format (output server) "~A(ccl::with-rectangle-arg (r ~D ~D ~D ~D)~%~A  (#_PaintRect r))~%" 
	       *cmn-space* 
	       (round x0) (round (- *cmn-top-y* (+ y0 dy))) (round (+ x0 dx)) (round (- *cmn-top-y* y0))
	       *cmn-space*)))
    ))

(defun g-filled-polygon (server path size cmd closepath)
  ;; actual path is from current-point via linetos, and if closepath back to start
  (case (output-type server)
    (:postscript (format (output server) " ~{~,2F ~,2F lineto ~}~A~%" (map 'list #'(lambda (n) (float (* n size))) path) cmd))
    (:quickdraw (let ((x0 (scr-x server))
		      (y0 (scr-y server)))
		  (check-function-size server)
		  (format (output server) "~A(setf (view-get *cmn-view* 'my-poly) (#_OpenPoly))~%" *cmn-space*)
		  (loop for x in path by #'cddr and y in (cdr path) by #'cddr do
		    (g-lineto server (float (* x size)) (float (* y size))))
		  (if closepath (g-lineto server x0 y0))
		  (format (output server) "~A(let ((new-poly (view-get *cmn-view* 'my-poly)))~%~A  ~
		                                   (#_ClosePoly)~%~A  ~
                                                   (#_PaintPoly new-poly)~%~A  ~
                                                   (#_KillPoly new-poly))~%"
			  *cmn-space* *cmn-space* *cmn-space* *cmn-space*)))
    ))

(defun g-fill (server cmd closepath)
  (declare (ignore closepath))
  (case (output-type server)
    (:postscript (format (output server) " ~A~%" cmd))
    (:quickdraw				;only needed in cmn for closing a polygon -- others handled with keyword args etc
     (if (and (output-state server)
	      (eq (first (output-state server)) :polygon))
	 (progn
	   (pop (output-state server))
	   (format (output server) "~A(let ((new-poly (view-get *cmn-view* 'my-poly)))~%~A  ~
		                             (#_ClosePoly)~%~A  ~
                                             (#_PaintPoly new-poly)~%~A  ~
                                             (#_KillPoly new-poly))~%"
		   *cmn-space* *cmn-space* *cmn-space* *cmn-space*)
	   (setf *cmn-quickdraw-split-safe* t))))
    ))

(defun g-arc (server x y r ang1 ang2 &optional fill)
  ;; arc of a circle centered at (x y) radius r in current line-width or (if fill) filled, partial arc if ang1 ang2
  ;; ang1=0 and ang2=360 is a full circle (very common special case)
  ;; Postscript defines this as angle increases counterclockwise, 0=>x axis
  (case (output-type server)
    (:postscript (format (output server) " ~,3F ~,3F ~D ~D ~D newpath arc~%" x y r ang1 ang2))
    (:quickdraw  
     ;; here we need the implied circle's bounding box, with 0=>y axis upward (the negative y axis!!)
     ;; with the angle increasing clockwise
     (progn 
       (check-function-size server)
       (format (output server) "~A(ccl::with-rectangle-arg (r ~D ~D ~D ~D)~%~A  ~
                                      (~A r~A))~%"
	       *cmn-space*
	       (round (- x r))
	       (round (- *cmn-top-y* (+ y r)))
	       (round (+ x r))
	       (round (- *cmn-top-y* (- y r)))
	       *cmn-space*
	       (if (and (= ang1 0) (= ang2 360))
		   (if fill "#_PaintOval" "#_FrameOval")
		 (if fill "#_PaintArc" "#_FrameArc"))
	       (if (and (= ang1 0) (= ang2 360))
		   ""
		 (format nil " ~D ~D"	;these are the start and angle traversed (different from Postscript)
			 (round (- 90 ang1))
			 (round (- ang1 ang2)))))))
    ))

(defun g-slanted-filled-rectangle (server x0s y0s x1s y1s thk)
  (case (output-type server)
    (:postscript (format (output server) " ~,2F ~,2F moveto ~,2F ~,2F lineto ~,2F ~,2F lineto ~,2F ~,2F lineto closepath fill~%"
			 x0s y0s x0s (+ y0s thk) x1s (+ y1s thk) x1s y1s))
    (:quickdraw 
     (progn 
       (g-moveto server x0s y0s) 
       (g-filled-polygon server (list x0s (+ y0s thk) x1s (+ y1s thk) x1s y1s x0s y0s) 1.0 nil nil)))
    ))

(defun g-save-server-bounds (server)
  (case (output-type server)
    (:postscript )
    (:quickdraw (setf *cmn-quickdraw-bounds* (list (x0 server) (y0 server) (x1 server) (y1 server))))
    ))

(defun g-begin-coordinate-transformation (server matrix &optional (saved t))
  ;; matrix is the transformation matrix from the current coordinate system to the new one
  (case (output-type server)
    (:postscript (format (output server) " ~A[ ~{~A ~}] concat~%" (if saved "gsave " "") (map 'list #'not-rational matrix)))
    (:quickdraw				;major problem here is that Quickdraw pictures can't be nested --
					;  some complicated kludge will probably be needed eventually --
					;  probably a list of views
					;  also, we don't know the transformed picture's bound at the start of the picture
     ;; this is trickier than it ought to be because the quickdraw.lisp support routines set the clip region
     ;; of the picture view to the picture bounds, then stupidly forget to re-establish the old region.
     ;; Also, the included code in the function we're building can overflow the MCL compiled function size limit,
     ;; so we have to redirect the function builders to place the calls here without crossing the function boundary.
     (if *cmn-quickdraw-picture-in-progress*
	 ;; Quickdraw cannot handle nested coordinate transformations, a major drag since cmn is based on that idea,
	 ;; but this is just a previewer, so we'll ignore any nested coordinates.
	 (push :drop-matrix-back (output-state server))
       (progn
	 (if saved (g-save server))
	 (setf *cmn-quickdraw-saved-top-y* *cmn-top-y*)
	 (setf *cmn-top-y* (* *cmn-top-y* (/ 1.0 (fourth matrix))))
	 ;; weird!! negative Y indices simply get tossed?!!$%@
	 (check-function-size server t)	;force output of current function
	 (setf *cmn-quickdraw-picture-in-progress* t)
	 (setf *cmn-quickdraw-picture-first-function* *cmn-quickdraw-functions*))))
    ))

(defun g-end-coordinate-transformation (server matrix &optional (saved t) vector)
  (case (output-type server)
    (:postscript (if saved (format (output server) " grestore~%")))
    (:quickdraw 
     (if (and (output-state server)
	      (eq (first (output-state server)) :drop-matrix-back))
	 (pop (output-state server))
       (let ((last-included-function *cmn-quickdraw-functions*))
	 (check-function-size server t)	;force completion of local function
	 (setf *cmn-quickdraw-picture-in-progress* nil)
	 (if (not *cmn-quickdraw-bounds*) (g-save-server-bounds server))
	 (if (not vector) (setf vector (list (fifth matrix) (sixth matrix))))
	 (let ((new-box (transform-box matrix (list (first vector) (second vector)) *cmn-quickdraw-bounds*)))
	       ;; matrix is the original transformation matrix, *cmn-quickdraw-bounds* is either nil or a list
	       ;;  giving the current picture's bounds -- all these bounds have to be turned upside down for Quickdraw, of course.
#|
	   (format (output server) "~A;; offset ~D: '(~{~D ~}) by '(~{~D ~}) to '(~{~D ~})~%~
                                    ~A;; =>     ~D: '~D ~D and ~D ~D~%"
		   *cmn-space* (round *cmn-top-y*)
		   (map 'list #'round *cmn-quickdraw-bounds*)
		   (map 'list #'round vector)
		   (map 'list #'round new-box)
		   *cmn-space* *cmn-quickdraw-saved-top-y*
		   (round (- *cmn-top-y* (second *cmn-quickdraw-bounds*)))
		   (round (- *cmn-top-y* (fourth *cmn-quickdraw-bounds*)))
		   (round (- *cmn-quickdraw-saved-top-y* (second new-box)))
		   (round (- *cmn-quickdraw-saved-top-y* (fourth new-box))))
|#
	   (format (output server) "~A(let ((*cmn-region* (#_NewRgn)))~%~A  ~
                                        (#_GetClip *cmn-region*)~%~A  ~
                                        (start-picture *cmn-view* ~D ~D ~D ~D)~%"
		   *cmn-space* *cmn-space* *cmn-space*
		   ;; these bounds have to be off by one because the Quickdraw clipper is too enthusiastic
		   (max 0 (1- (floor (first *cmn-quickdraw-bounds*))))
		   ;; bounds of -1 = complete Quickdraw confusion
		   (max 0 (1- (floor (- *cmn-top-y* (fourth *cmn-quickdraw-bounds*)))))
		   (1+ (ceiling (third *cmn-quickdraw-bounds*)))
		   (+ 2 (ceiling (- *cmn-top-y* (second *cmn-quickdraw-bounds*)))))
	   (setf *cmn-space* "    ")
	   (loop for i from *cmn-quickdraw-picture-first-function* to last-included-function do
	     (format (output server) "~A(cmn-qd-~D *cmn-view*)~%" *cmn-space* i))
	   (format (output server) "~A(let ((*cmn-picture* (get-picture *cmn-view*)))~%~A  ~
                                        (#_SetClip *cmn-region*)~%~A  ~
                                        (#_DisposeRgn *cmn-region*)~%~A  ~
                                        (with-rectangle-arg (*r* ~D ~D ~D ~D)~%~A    ~
                                          (#_DrawPicture *cmn-picture* *r*))~%~A  ~
                                        (#_KillPicture *cmn-picture*)))~%"
		   *cmn-space* *cmn-space* *cmn-space* *cmn-space* 
		   (max 0 (1- (floor (first new-box))))
		   (max 0 (1- (floor (- *cmn-quickdraw-saved-top-y* (fourth new-box)))))
		   (1+ (ceiling (third new-box)))
		   (+ 2 (ceiling (- *cmn-quickdraw-saved-top-y* (second new-box))))
		   *cmn-space* *cmn-space*)
	   (setf *cmn-space* "  ")
	   (setf *cmn-top-y* *cmn-quickdraw-saved-top-y*)
	   (g-restore server)
	   (setf *cmn-quickdraw-bounds* nil)))))
    ))

(defvar *cmn-quickdraw-saved-x* 0)
(defvar *cmn-quickdraw-saved-y* 0)

(defun g-save (server)
  (case (output-type server)
    (:postscript (format (output server) " gsave~%"))
    (:quickdraw
     (progn 
       ;(g-comment server (format nil "g-save at ~D ~D" (round (scr-x server)) (round (- *cmn-top-y* (scr-y server)))))
       (setf *cmn-quickdraw-saved-x* (scr-x server))
       (setf *cmn-quickdraw-saved-y* (scr-y server))
       (setf *cmn-quickdraw-saved-font* *cmn-quickdraw-current-font*)))
    ))

(defun g-restore (server)
  (case (output-type server)
    (:postscript (format (output server) " grestore~%"))
    (:quickdraw
     (progn 
       (g-moveto server *cmn-quickdraw-saved-x* *cmn-quickdraw-saved-y*)
       (when (not (equal *cmn-quickdraw-current-font* *cmn-quickdraw-saved-font*))
	 (setf *cmn-quickdraw-current-font* *cmn-quickdraw-saved-font*))
       (g-qd-new-font server *cmn-quickdraw-saved-font*)))
    ))

(defun g-begin-gray-scale (server num &optional (saved t))
  (case (output-type server)
    (:postscript (let ((snum (if (integerp num) (format nil "~D" num) (format nil "~,2F" num))))
		   (if saved (format (output server) " /cmncolor ~A def" snum))
		   (format (output server) " ~A setgray~%" snum)))
    (:quickdraw 
     (progn 
       (check-function-size server)
       (format (output server) "~A(#_PenPat ~(~A~)) (#_TextMode 49)~%" *cmn-space* (Post-to-Quick-gray-scale num))))
    ))

(defun g-end-gray-scale (server)
  (case (output-type server)
    (:postscript (format (output server) " cmncolor setgray~%"))
    (:quickdraw  
     (progn 
       (check-function-size server)
       (format (output server) "~A(#_PenPat *black-pattern*) (#_TextMode 1)" *cmn-space*)))
    ))

(defun g-show-page (server)
  ;; in Postscript this causes the stored display program to be displayed -- not needed in Quickdraw
  (case (output-type server)
    (:postscript (format (output server) " showpage~%"))
    (:quickdraw )
    ))

(defun g-page-number (server page)
  ;; purely for Postscript's benefit (multi-page output comment)
  (case (output-type server)
    (:postscript (format (output server) "%%Page: ~D ~D~%" page page))
    (otherwise )
    ))


(defun gray-scale (val)
  (pattern #'(lambda (score obj)
	       (declare (ignore obj))
	       (case (output-type score)
		 (:postscript
		  (values 
		   (format nil " /cmncolor ~,3F def  ~,3F setgray " val val)
		   " show /cmncolor 0 def 0 setgray "
		   " /cmncolor 0 def 0 setgray "))
		 (:quickdraw 
		  (values val nil nil))
		 ))))

(defun outlined (num)
  (pattern #'(lambda (score obj)
	       (declare (ignore obj))
	       (case (output-type score)
		 (:postscript
		  (values
		   ""
		   (format nil " false charpath ~D setlinewidth stroke 0 setlinewidth " num)
		   ""))
		 (:quickdraw 
		  (let ((current-face (second (or *cmn-quickdraw-pending-font* *cmn-quickdraw-actual-font*))))
		  (values 
		   #'(lambda (server) (declare (ignore server)) (setf *cmn-quickdraw-text-style* 8))
		   nil
		   #'(lambda (server) 
		       (setf *cmn-quickdraw-text-style* 0) 
		       (format (output server) "~A(#_TextFace ~D)~%" *cmn-space* current-face))
		   )))
		 ))))

(defun g-close (server &optional load-it)
  (case (output-type server)
    (:postscript (close (output server)))
    (:quickdraw
     (progn
       (close (output server))
       (if load-it (load (output-file server)))
       ))
    ))

(defun g-clear (server)
  (case (output-type server)
    (:postscript (setf (output server) nil))
    (:quickdraw  (setf (output server) nil))
    ))

(defun g-header (server title user-name)
  (case (output-type server)
    (:postscript 
      (progn 
	(format (output server) "%!PS-Adobe-2.0 EPSF-2.0~%") ;this is what our Next's seem to have -- Adobe recommends 3.0
	;; exact caps matter here --if you had %!Ps-Adobe, WriteNow would throw up its hands and refuse to paste it in.
	(if title (format (output server) "%%Title: ~A~%" title))
	(if user-name (format (output server) "%%Creator: ~A~%" user-name))
	(format (output server) "%%CreationDate: ~A~%" (creation-date))
	(if (bounded server)
	    (progn
	      (setf (bounds server) (bounded server))
	      (format (output server) "%%BoundingBox:~{ ~D~}~%" (map 'list #'ceiling (bounds server))))
	  (format (output server) "%%BoundingBox:(atend)~%"))
	(format (output server) "%%EndComments~%")
	(include-eps-defs-and-whatever server)
	(format (output server) "%%EndProlog~%")))
    (:quickdraw 
     (progn
       (setf *cmn-space* "  ")
       (setf *cmn-quickdraw-functions* 0)
       (setf *cmn-quickdraw-lines* 0)
       (setf *cmn-quickdraw-split-safe* t)
       (setf *cmn-quickdraw-picture-in-progress* nil)
       (setf *cmn-quickdraw-not-top-level-functions* nil)
       (setf *cmn-quickdraw-picture-first-function* nil)
       (setf *cmn-quickdraw-last-pen-size* 0)
       (setf *cmn-quickdraw-saved-font* nil)
       (setf *cmn-quickdraw-current-font* nil)
       (setf *cmn-quickdraw-pending-font* nil)
       (setf *cmn-quickdraw-actual-font* nil)
       (format (output server) ";;; quickdraw program ~Acreated ~A by ~A from cmn~%" 
	       (if title (format nil "~A " title) "") (creation-date) user-name)
       (format (output server) "(in-package :cl-user)~%")
       (format (output server) "(require :quickdraw)~%(require :scrollers)~%~%")
       (format (output server) "(preview-score (list~%")
       (format (output server) "#'(lambda (*cmn-view*)~%~
                                ~2T(declare (ignore-if-unused *cmn-view*))~%")))
    ))
	
(defun g-footer (server)
  (case (output-type server)
    (:postscript (format (output server) " showpage~%%%Trailer~%"))
    (:quickdraw 
     (let ((cmn-title (or (title *cmn-score*) "cmn")))
       (if (text-p cmn-title) (setf cmn-title (letters cmn-title)))
       (format (output server) "  )~%~%)  ~s *force-new-preview-windows*)"
               cmn-title)
       ))))

(defun g-bounding-box (server)
  ;; this sets the actual graphics boundary -- in Postscript it is sent out at the end of the file.
  (case (output-type server)
    (:postscript (format (output server) "%%BoundingBox:~{ ~D~}~%" (map 'list #'(lambda (n) (round (1+ n))) (bounds server))))
    (:quickdraw )
    ))
