;;; -*- syntax: common-lisp; package: cmn; base: 10; mode: lisp -*-
;;;
;;; various articulation marks -- an extension of the stuff in cmn1.lisp
;;;
;;; included here are:
;;;   
;;;   sul G, sul tasto, col legno and so on (connected text with bracket at end)
;;;   fingernail articulation mark (harp)
;;;   double-tongue and triple-tongue (slur+two or three dots)
;;;   nail-pizzicato (circle+dot)
;;;   martellato (wedge+dot)
;;;   heavy-accent (big wedge)
;;;   hauptstimme and nebenstimme
;;;   no-accent (an underlined "u")
;;;   doink, rip, and smear (Finale appears to use the name "doit" for these marks)
;;;   sprechstimme (i.e. x'd-stem) and circled-stem
;;;   organ-heel and toe
;;;   vibrato
;;;   inverted-turn


(in-package :cmn)



;;; ------------------------------------------------
;;; sul G and other such indications
;;;
;;;   default is to go above the staff with a bracket down at the end

(defun sul- (&rest args) 
  (apply #'text- 
	 (connecting-pattern '(10 10))
	 (font-name "Times-Italic")
	 unjustified
	 (y #'(lambda (mark note score) 
		(+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		   (max (if (and (< 1 (head-line note) 5)
				 (not (whole-note-p note)))
			    (* (+ 8 (head-line note)) (staff-line-separation score))
			  0.0)
			(* (max 10 (+ 3 (head-line note))) (staff-line-separation score))))))
	 args))

(defun -sul (&rest args) (apply #'-text (end-pattern :bracket-down) args))


(defun sul-tasto- (&rest args) (apply #'sul- "sul tasto" args))
(defun -sul-tasto (&rest args) (apply #'-sul args))

;;; and so on for sul-pont, sul G, col legno, non vib and the rest of that clan
;;; unfortunately, the defvar versions of the same entities are slightly more complex...

(defvar sul-tasto- (make-self-acting 
		    :action #'(lambda (new-note &rest rest)
				(declare (ignore rest))
				(let ((st (sul-tasto-)))
				  (funcall (action st) new-note (arguments st))))
		    :arguments nil))

(defvar -sul-tasto (make-self-acting
		    :action #'(lambda (new-note &rest rest)
				(declare (ignore rest))
				(let ((st (-sul-tasto)))
				  (funcall (action st) new-note (arguments st))))
		    :arguments nil))


;;; ------------------------------------------------
;;; fingernail articulation mark
;;;
;;; this mark is apparently used in harp music

(defun display-fingernail (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		   (max (if (and (< 1 (head-line note) 5)
				 (not (whole-note-p note)))
			    (* (+ 8 (head-line note)) (staff-line-separation score))
			  0.0)
			(* (max 10 (+ 3 (head-line note))) (staff-line-separation score)))))
	 (x-off (+ (box-x0 note) -.05 (dxy-dx mark) (center note) (box-x0 mark))))
    (setf (line-width score) .025)
    (circle score x-off y-off .3 10 170)
    (draw score)
    (circle score x-off (- y-off .2) .4 45 135)
    (draw score)
    (setf (line-width score) 0)))

(defvar fingernail (make-instance 'write-protected-sundry :name :fingernail :mark #'display-fingernail))
(defun fingernail (&rest objects) (apply #'mark #'display-fingernail :fingernail objects))


;;; ------------------------------------------------
;;; double and triple tongue marks
;;;

(defun display-tongue (mark note score dots)
  (let ((y0 (+ (max (+ 1.25 (%staff-y0 note))
		    (* (+ 2 (head-line note)) (staff-line-separation score)))
	       (dxy-dy mark)))
	(x0 (+ (box-x0 note) (dxy-dx mark))))
    (display-tie (make-tie :x0 (- x0 .125) 
			   :x1 (+ x0 .375) 
			   :y0 y0
			   :direction :up)
		 score)
  (if (= dots 2)
      (progn
	(moveto score (- x0 .025) (- y0 .1))
	(show score %dot)
	(rmoveto score .1 0)
	(show score %dot))
    (progn
      (moveto score (- x0 .075) (- y0 .1))
      (show score %dot)
      (rmoveto score .075 0)
      (show score %dot)
      (rmoveto score .075 0)
      (show score %dot)))))

(defvar double-tongue (make-instance 'write-protected-sundry 
			  :name :double-tongue 
			  :mark #'(lambda (mark note score &rest rest)
				    (declare (ignore rest))
				    (display-tongue mark note score 2))))
(defun double-tongue (&rest objects) 
  (apply #'mark #'(lambda (mark note score &rest rest)
		    (declare (ignore rest))
		    (display-tongue mark note score 2))
	 :double-tongue
	 objects))

(defvar triple-tongue (make-instance 'write-protected-sundry 
			  :name :triple-tongue 
			  :mark #'(lambda (mark note score &rest rest)
				    (declare (ignore rest))
				    (display-tongue mark note score 3))))
(defun triple-tongue (&rest objects) 
  (apply #'mark #'(lambda (mark note score &rest rest)
		    (declare (ignore rest))
		    (display-tongue mark note score 3))
	 :triple-tongue
	 objects))


;;; ------------------------------------------------
;;; nail pizzicato (Bartok style)
;;;

(defun display-nail-pizzicato (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		   (max (if (and (< 1 (head-line note) 5)
				 (not (whole-note-p note)))
			    (* (+ 8 (head-line note)) (staff-line-separation score))
			  0.0)
			(* (max 10 (+ 3 (head-line note))) (staff-line-separation score)))))
	 (r (/ 6 40))
	 (x-off (+ (box-x0 note) -.05 (dxy-dx mark) (center note) (box-x0 mark))))
    (setf (line-width score) .025)
    (circle score x-off y-off r)
    (draw score)
    (circle score x-off y-off .025 0 360 t)
    (setf (line-width score) 0)))

(defvar nail-pizzicato (make-instance 'write-protected-sundry :name :nail-pizzicato :mark #'display-nail-pizzicato))
(defun nail-pizzicato (&rest objects) (apply #'mark #'display-nail-pizzicato :nail-pizzicato objects))



;;; ------------------------------------------------
;;; martellato
;;;

(defun display-martellato (mark note score no-dot)
  (let* ((upper (not (member (visible-justification mark) '(:down :below))))
	 (y0 (+ (dxy-dy mark) (%staff-y0 note) 
		(if upper
		    (max (if (and (< 1 (head-line note) 5)
				  (not (whole-note-p note)))
			     (* (+ 8 (head-line note)) (staff-line-separation score))
			   0.0)
			 (* (max 10 (+ 3 (head-line note))) (staff-line-separation score)))
		  (- (min 0.0 (box-y0 note))
		     1.0))))
	 (x0 (+ (box-x0 note) (dxy-dx mark) -.05 (center note) (box-x0 mark))))
    (moveto score x0 y0)
    (show score (if upper (if no-dot %wedgeup %dotaccentup) (if no-dot %wedgedown %dotaccentdown)))))

(defvar martellato (make-instance 'write-protected-sundry 
		     :name :martellato 
		     :mark #'(lambda (mark note score &rest rest)
			       (declare (ignore rest))
			       (display-martellato mark note score nil))))
(defun martellato (&rest objects) (apply #'mark #'(lambda (mark note score &rest rest)
						    (declare (ignore rest))
						    (display-martellato mark note score nil))
					 :martellato objects))


;;; ------------------------------------------------
;;; heavy-accent

(defvar heavy-accent (make-instance 'write-protected-sundry 
		     :name :heavy-accent 
		     :mark #'(lambda (mark note score &rest rest)
			       (declare (ignore rest))
			       (display-martellato mark note score t))))

(defun heavy-accent (&rest objects) (apply #'mark #'(lambda (mark note score &rest rest)
						    (declare (ignore rest))
						    (display-martellato mark note score t))
					 :heavy-accent objects))


;;; ------------------------------------------------
;;; Hauptstimme

(defun display-hauptstimme (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y-off (+ (dxy-dy mark) (box-y0 note)))
	 (x-off (+ (box-x0 note) -.125 (dxy-dx mark))))
    (moveto score x-off y-off)
    (matrix-front score (matrix mark))
    (setf (line-width score) .025)
    (rlineto score 0 .5)
    (rlineto score .2 0)
    (rmoveto score -.2 -.25)
    (rlineto score -.325 0)
    (rmoveto score 0 -.25)
    (rlineto score 0 .5)
    (draw score)
    (setf (line-width score) 0)
    (matrix-back score (matrix mark))))

(defvar hauptstimme (make-instance 'write-protected-sundry :name :hauptstimme :mark #'display-hauptstimme))
(defun hauptstimme (&rest objects) (apply #'mark #'display-hauptstimme :hauptstimme objects))


;;; ------------------------------------------------
;;; Nebenstimme

(defun display-nebenstimme (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y-off (+ (dxy-dy mark) (box-y0 note)))
	 (x-off (+ (box-x0 note) -.125 (dxy-dx mark))))
    (moveto score x-off y-off)
    (matrix-front score (matrix mark))
    (setf (line-width score) .025)
    (rlineto score 0 .5)
    (rlineto score .2 0)
    (rmoveto score -.2 -.5)
    (rlineto score -.3 .5)
    (rlineto score 0 -.5)
    (draw score)
    (setf (line-width score) 0)
    (matrix-back score (matrix mark))))

(defvar nebenstimme (make-instance 'write-protected-sundry :name :nebenstimme :mark #'display-nebenstimme))
(defun nebenstimme (&rest objects) (apply #'mark #'display-nebenstimme :nebenstimme objects))


;;; ------------------------------------------------
;;; no-accent

(defun display-no-accent (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y-off (+ (box-y0 mark) (dxy-dy mark) (%staff-y0 note) 
		   (max (if (and (< 1 (head-line note) 5)
				 (not (whole-note-p note)))
			    (* (+ 8 (head-line note)) (staff-line-separation score))
			  0.0)
			(* (max 10 (+ 3 (head-line note))) (staff-line-separation score)))))
	 (x-off (+ (box-x0 note) -.05 (dxy-dx mark) (center note) (box-x0 mark))))
    (setf (line-width score) .025)
    (circle score x-off (+ y-off .2) .15 180 360)
    (draw score)
    (moveto score (- x-off .15) (- y-off .025))
    (rlineto score .325 0)
    (moveto score (- x-off .15) (+ y-off .2))
    (rlineto score 0 .1)
    (moveto score (+ x-off .15) (+ y-off .2))
    (rlineto score 0 .1)
    (draw score)
    (setf (line-width score) 0)))

(defvar no-accent (make-instance 'write-protected-sundry :name :no-accent :mark #'display-no-accent))
(defun no-accent (&rest objects) (apply #'mark #'display-no-accent :no-accent objects))


;;; ------------------------------------------------
;;; doink

(defun display-doink (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y0 (+ (dxy-dy mark) (box-y0 note))) 
	 (x0 (+ (box-x0 note) .5 (dxy-dx mark))))
    (moveto score x0 y0)
    (g-begin-filled-polygon score)
    (curveto score 
	     (+ x0 (* .2 .5)) y0
	     (+ x0 (* .8 .5)) (+ y0 .2)
	     (+ x0 .5) (+ y0 .5))
    (curveto score
	     (+ x0 (* .8 .5)) (+ y0 .15)
	     (+ x0 (* .2 .5)) (- y0 .05)
	     x0 y0)
    (fill-in score)))
	 

(defvar doink (make-instance 'write-protected-sundry :name :doink :mark #'display-doink))
(defun doink (&rest objects) (apply #'mark #'display-doink :doink objects))


;;; ------------------------------------------------
;;; rip

(defun display-rip (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y0 (+ (dxy-dy mark) (box-y0 note))) 
	 (x0 (+ (box-x0 note) -.125 (dxy-dx mark))))
    (moveto score x0 y0)
    (g-begin-filled-polygon score)
    (curveto score 
	     (- x0 (* .2 .5)) y0
	     (- x0 (* .8 .5)) (- y0 .2)
	     (- x0 .5) (- y0 .5))
    (curveto score
	     (- x0 (* .8 .5)) (- y0 .15)
	     (- x0 (* .2 .5)) (+ y0 .05)
	     x0 y0)
    (fill-in score)))
	 

(defvar rip (make-instance 'write-protected-sundry :name :rip :mark #'display-rip))
(defun rip (&rest objects) (apply #'mark #'display-rip :rip objects))


;;; ------------------------------------------------
;;; smear

(defun display-smear (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((hl (head-line note))
	 (y0 (+ (%staff-y0 note) (* hl (staff-line-separation score)) .25 (if (and (evenp hl) (< 1 hl 9)) .125 0)))
	 (x0 (+ (box-x0 note) -.05 (dxy-dx mark))))
    (moveto score x0 (- y0 .05))
    (g-begin-filled-polygon score)
    (curveto score 
	     (+ x0 .2) (+ y0 .3)
	     (+ x0 .3) (- y0 .3)
	     (+ x0 .45) (+ y0 .05))
    (curveto score 
	     (+ x0 .3) (- y0 .2)
	     (+ x0 .2) (+ y0 .2)
	     x0 (- y0 .05))
    (fill-in score)))

(defvar smear (make-instance 'write-protected-sundry :name :smear :mark #'display-smear))
(defun smear (&rest objects) (apply #'mark #'display-smear :smear objects))


;;; ------------------------------------------------
;;; sprechstimme
;;;
;;;   any note or chord's stem can have any arbitrary mark placed on its stem.
;;;   The stem-mark field is a function funcall'd with the args score x0 y0 y1 (stem-wise)

(defun sprechstimme (&rest objects)
  (let ((nm (apply #'mark nil nil objects)))
    (stem-mark #'(lambda (score x0 y0 y1)
		   (let ((line-width (line-width score)))
		     (setf (line-width score) .01)
		     (moveto score (- (+ x0 (dxy-dx nm)) .125 .01) (- (+ (dxy-dy nm) (* .5 (+ y0 y1))) .125))
		     (rlineto score .25 .25)
		     (rmoveto score 0 -.25)
		     (rlineto score -.25 .25)
		     (draw score)
		     (setf (line-width score) line-width))))))

(defvar sprechstimme (sprechstimme))


;;; ------------------------------------------------
;;; circled-stem

(defun circled-stem (&rest objects)
  (let ((nm (apply #'mark nil nil objects)))
    (stem-mark #'(lambda (score x0 y0 y1)
		   (let ((line-width (line-width score)))
		     (setf (line-width score) .01)
		     (circle score (+ x0 (dxy-dx nm)) (+ (dxy-dy nm) (* .5 (+ y0 y1))) .125)
		     (draw score)
		     (setf (line-width score) line-width))))))

(defvar circled-stem (circled-stem))


;;; ------------------------------------------------
;;; organ-heel
;;;
;;; my organ music does not use the same marking as recommended by Read, "Music Notation"
;;; This function implements the latter.

(defun display-organ-heel (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y-off (+ (dxy-dy mark) (staff-y0 note) -.5))
	 (x-off (+ (box-x0 note) (center note) (dxy-dx mark))))
    (setf (line-width score) .0125)
    (circle score x-off y-off .15)
    (draw score)
    (setf (line-width score) 0)))

(defvar organ-heel (make-instance 'write-protected-sundry :name :organ-heel :mark #'display-organ-heel))
(defun organ-heel (&rest objects) (apply #'mark #'display-organ-heel :organ-heel objects))


;;; ------------------------------------------------
;;; organ-toe

(defun display-organ-toe (mark note score &rest rest)
  (declare (ignore rest))
  (let* ((y-off (+ (dxy-dy mark) (staff-y0 note) -.5))
	 (x-off (+ (box-x0 note) (center note) (dxy-dx mark))))
    (setf (line-width score) .0125)
    (moveto score (- x-off .1) (- y-off .3))
    (rlineto score .1 .4)
    (rlineto score .1 -.4)
    (draw score)
    (setf (line-width score) 0)))

(defvar organ-toe (make-instance 'write-protected-sundry :name :organ-toe :mark #'display-organ-toe))
(defun organ-toe (&rest objects) (apply #'mark #'display-organ-toe :organ-toe objects))


;;; ------------------------------------------------
;;; vibrato
;;;
;;; (this uses the wavy line of the trill ornament)

(defclass vib (trill)
  ((wavy-line :initform t)))

(defun display-vibrato (vib note score &rest rest)
  (declare (ignore rest))
  (let ((x0 (+ (box-x0 note) (dxy-dx vib) .5))
	(y0 (+ (box-y0 note) (dxy-dy vib) -.0625 
	       (if (and (evenp (head-line note)) 
			(< (head-line note) 9)) .125 0))))
    (moveto score x0 y0)
    (show score %trillsection :count (round (- (box-x1 vib) x0 .75) 
					    (g-rx %trillsection)))))

(defun vibrato (&rest objects)
  (let ((new-vib (make-instance 'vib :mark #'display-vibrato)))
    (loop for act in objects do
      (when act
	(if (self-acting-p act)
	    (funcall (action act) new-vib (arguments act)))))
    new-vib))

(defvar vibrato (make-instance 'vib :name :vibrato :mark #'display-vibrato))


;;; ------------------------------------------------
;;; inverted-turn
;;;

(defun display-inverted-turn (mark note score &rest rest)
  (declare (ignore rest))
  (incf (box-x0 mark) (if (stem-is-up? note) .1 .15))
  (display-ornament mark (%mirror %turn) note score))

(defvar inverted-turn (make-instance 'write-protected-ornament :name :inverted-turn :mark #'display-inverted-turn))
(defun inverted-turn (&rest objects) (apply #'ornament #'display-inverted-turn :inverted-turn objects))
