;;; -*- Lisp -*-

;;; translate from Sambox packing mode 4 (2 16 right adjusted in 36) to Next mode 3 (linear 16)
;;; assume original is full bit stream (i.e. all 36 bits are stored).

(defvar bytes 0)
(defvar byte-counter 0)
(defvar shift-counter 0)
(defvar waitingB 0)

;;; toss every 9-th 4-bit nibble (starting with first), assemble the remaining bit stream into 16-bit chunks.

(defun next4-bits (curB)
  (let ((outB nil))
    (when (not (zerop byte-counter))
      (if (not (zerop shift-counter))
	  (setf outB (logior waitingB curB))
	(setf waitingB (ash curB 4)))
      (incf shift-counter)
      (if (= shift-counter 2) (setf shift-counter 0)))
    (incf byte-counter)
    (if (= byte-counter 9) (setf byte-counter 0))
    outB))

(defun next8-bits (curB)
  (let* ((in4h (logand (ash curB -4) #xF))
	 (in4l (logand curB #xF))
	 (outB (next4-bits in4h))
	 (outC (next4-bits in4l)))
    (or outB outC)))

#|
;;; byte order is the same on the two machines, at least when using binary transfers.
(defvar lastB nil)

(defun write-swapped-byte (outB outf)
  (if lastB
      (progn
	(write-byte outB outf)
	(write-byte lastB outf)
	(setf lastB nil))
    (setf lastB outB)))
	 	
(defvar inp nil)
|#

(defun translate (in-file out-file &optional words)
  (let* ((inf (open in-file 
		    :direction :input 
		    :element-type '(unsigned-byte 8)))
	 (inlen (or words (- (file-length inf) 576)))
	 (outlen (floor (* inlen (/ 8 9))))
	 (outB nil)
	 (outf (open out-file 
		     :direction :output 
		     :element-type '(unsigned-byte 8) 
		     :if-exists :supersede
		     :if-does-not-exist :create)))
    (unwind-protect 
	(let ((arr (make-array 18 :element-type '(unsigned-byte 8))))
	  
	  (setf bytes 0)
	  (setf shift-counter 0)
	  (setf byte-counter 0)
	  (setf waitingB 0)
					;	  (setf lastB nil)
	  ;; old sound file header was:
	  ;;                           #o525252,,525252
	  ;;                           srate as PDP-10 float
	  ;;                           samples-per-word,,packing-mode
	  ;;                           channels
	  ;; rest of header not interesting.

	  (loop for i from 0 below 18 do
	    (setf (aref arr i) (read-byte inf)))

	  (let* ((snd-header-H (logior (ash (aref arr 0) 10)
				       (ash (aref arr 1) 2)
				       (ash (aref arr 2) -6)))
		 (snd-header-L (logior (ash (logand (aref arr 2) #x3F) 12)
				       (ash (aref arr 3) 4)
				       (ash (aref arr 4) -4)))
		 (srate-H (logior (ash (logand (aref arr 4) #xF) 14)
				  (ash (aref arr 5) 6)
				  (ash (aref arr 6) -2)))
		 (srate-L (logior (ash (logand (aref arr 6) #x3) 16)
				  (ash (aref arr 7) 8)
				  (aref arr 8)))
		 ;; PDP-10 floating point format was sign in bit 0 , excess 128 exponent in 1-8, fraction in 9-35
		 (sign (if (= 1 (logand srate-H #o400000)) -1 1))
		 (exponent (- (ash (logand srate-H #o377000) -9) 128))
		 (fraction (float (/ (logior (ash (logand srate-H #o777) 18) srate-L) (expt 2 27))))
		 (packing-mode-L (logior (ash (logand (aref arr 11) #x3F) 12)
					 (ash (aref arr 12) 4)
					 (ash (aref arr 13) -4)))
		 (channels-L (logior (ash (logand (aref arr 15) #x3) 16)
				     (ash (aref arr 16) 8)
				     (aref arr 17))))
	    
	    (print (format nil "old header: ~A, sampling-rate: ~A, mode: ~A, channels: ~A"
			   (if (and (= snd-header-H #o525252)
				    (= snd-header-L #o525252))
			       "sound" "unknown file type!")
			   (* sign (expt 2 exponent) fraction)
			   packing-mode-L
			   channels-L))

	    (loop for i from 18 below 576 do (read-byte inf)) 
	    ;; flush rest of old header (576 = '200 * 4.5 = number of 8 bit bytes left over after we've
	    ;; read the first 4 36-bit words.

	    ;; write new Next header:
	    ;;                       #x2e736e64 (i.e. ".snd")
	    ;;                       data location in bytes (assume no comment, so 28)
	    ;;                       data size (in 8-bit bytes), don't include header size
	    ;;                       data format (assume packing mode 3 for now = 16-bit linear)
	    ;;                       sampling rate (as 32-bit integer).
	    ;;                       channels
	    ;;                       comment (nil for now)

	    (write-byte #x2e outf) (write-byte #x73 outf) (write-byte #x6e outf) (write-byte #x64 outf)
	    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte 28 outf)
	    (loop for i from 0 below 4 do (write-byte (logand (ash outlen (- (* 8 i) 24)) #xFF) outf))
	    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte 3 outf)
	    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte #x56 outf) (write-byte #x22 outf)
	    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte (or channels-L 1) outf)
	    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte 0 outf)    (write-byte 0 outf))

	  (loop for i from 0 below inlen do
	    (setf outB (next8-bits (read-byte inf)))
	    (if outB (write-byte outB outf))))
      (progn
	(close inf)
	(close outf)))))
