;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-

(in-package :clm)

(defvar last-dac-file-name nil)

(defvar *open-input-explicit-output* nil)
(defvar *open-input-explicit-reverb* nil)
(defvar *open-input-verbose* nil)
(defvar *open-input-print* nil)
(defvar *open-input-pathname* nil)
(defvar *open-input-truename* nil)
(defvar *clm-force-recomputation* nil)

(defvar *last-play-options* nil)
(defvar *reverb-decay-time* 0.0)

(defvar *clm-instruments* nil)

(defun clu-reset ()
  (c56-reset-cleanup)
  (clm-end-run)
  (clm-cleanup t))

(defun printf (s) (princ s) (force-output))

(defvar *clm-default-channels* 1)
(defvar *clm-default-srate* 22050)
(defvar *clm-default-play* #-mcl t #+mcl nil)

(defmacro with-sound ((&key (output default-sound-file-name)
			    sndfile
			    (channels *clm-default-channels*)
			    (srate *clm-default-srate*)
			    sampling-rate
			    continue-old-file
			    (reverb nil revthere)
			    reverb-data
			    (reverb-channels 1)
			    revfile
			    (play *clm-default-play*)
			    play-options
			    (cleanup-first t)
			    wait
			    notehook
			    statistics
			    (decay-time 1.0)
			    output-buffer-size
			    info
			    type
			    force-recomputation
			    save-body
			    (verbose *clm-verbose*)
			    commentary
			    comment)
		      &body body)
  `(let ((*clm-verbose* ,verbose))
     (unwind-protect
	 (let ((out-file (or *open-input-explicit-output*
			     (namestring (full-merge-pathnames (or ,sndfile ,output) default-sound-file-name))))
	       (out-file-p (or *open-input-explicit-output* ,sndfile ,output))
	       (revf (and (or ,revthere ,revfile) (or ,revfile *open-input-explicit-reverb* default-reverb-file-name))))
	   (if (/= 0 (dac-is-in-use)) (wait-for-dac))
	   ;; on the 68040's a collision here can cause a machine crash!
	   (when ,cleanup-first
	     (c56-reset-cleanup)	;in case called recursively in error handler
	     (clm-cleanup))
	   (set-srate (or ,sampling-rate ,srate))
	   (setf last-dac-file-name out-file)
	   (setf *last-play-options* ,play-options)
	   (setf *reverb-decay-time* ,decay-time)
	   (setf *clm-interrupted* 0)
	   (if ,statistics (clm-initialize-statistics ,statistics out-file revf))
	   (clm-initialize-notehook ,notehook)
	   (setf *clm-force-recomputation* ,force-recomputation)
	   ,@(if continue-old-file
		 `((reopen-output out-file))
	       `((open-output out-file 
			      (make-header :channels ,channels
					   :sampling-rate (or ,sampling-rate ,srate)
					   :type (or ,type default-sound-file-type)
					   :info (format nil "~A~A" 
							 ,(or info
							      (if (or comment commentary)
								  (format nil "#| ~A |#" (or comment commentary))
								(if *open-input-explicit-output*
								    (concatenate 'string 
								      (clm-get-default-header)
								      (format nil " (from ~A via open-input)" 
									      (namestring *open-input-truename*)))
								  (clm-get-default-header))))
							 (if ,save-body (format nil " #| ~A |#" ,(write-to-string body)) "")))
			      out-file-p 
			      ,output-buffer-size)))
	   ,@(if (or revthere revfile)
		 `((setf *reverb* (open-output revf
					       (make-header :channels ,reverb-channels
							    :sampling-rate (or ,sampling-rate ,srate)
							    :info ";temporary reverb stream")))))
	   (catch :FINISH
	     ,.body)
	   ,@(when (or revthere revfile)
	       `((close-output *reverb*)))
	   ,@(when revthere
	       `((open-input revf)
		 (,reverb 0 (+ ,decay-time (clm-get-duration *current-input-file*)) ,@reverb-data)
		 (close-input)))
	   (close-output)
	   (if ,statistics (clm-print-statistics ,statistics))
	   (clm-cleanup)
	   #+mcl (if (and ,type (= ,type AIFF-sound-file)) (ccl:set-mac-file-type out-file "AIFF"))
	   ,@(when (and play (not *open-input-explicit-output*)) 
	       `((dac-n :file out-file :times ,play)))
	   ,@(when (and wait (not *open-input-explicit-output*)) 
	       `((wait-for-dac))))
       (progn
	 (setf *clm-force-recomputation* nil)
	 (clu-reset)))))


;;; an example: say we have a reverberator named NREV
;;;
;;; (with-sound (:reverb nrev) (fm-violin 0 .25 440 .1))   ;mono output, one reverberated note
;;;
;;; (with-sound () (fm-violin 0 1 440 .1) (fm-violin .2 .4 330 .1))  ;mono out, 2 notes, no reverb
;;;
;;; (with-sound (:reverb nrev :channels 2)
;;;    (fm-violin 0 .25 440 .1 :location 45.0)
;;;    (fm-violin .5 .125 660 .1 :location 90.0 :reverb-amount .2))
;;;                  ;2 notes, stereo, reverberated, lots of reverb on second note
;;;
;;; see ins.lisp for more documentation
;;;
;;; Other possibly useful procedures that might be added here are SET-DAC-FILTER and SET-VOLUME

(defun clm-load (pathname &key (sndfile default-sound-file-name)
			       output
			       (channels 1)
			       (srate 22050)
			       sampling-rate
			       reverb
			       reverb-data
			       revfile
			       (reverb-channels 1)
			       (play t)
			       wait
			       comment
			       commentary
			       (verbose *clm-verbose*)
			       info
			       type
			       force-recomputation
			       save-body
			       statistics
			       notehook
			       (decay-time 1.0)
			       continue-old-file
			       (load-package *package*)
			       &allow-other-keys)
  (let ((*clm-verbose* verbose)
	(old-recompute *clm-force-recomputation*))
    (unwind-protect
	(let ((file (full-merge-pathnames (or *open-input-explicit-output* sndfile output) default-sound-file-name))
	      (*reverb* nil)
	      (revf (and (or reverb revfile) (or revfile *open-input-explicit-reverb* default-reverb-file-name))))
	  (if (/= 0 (dac-is-in-use)) (wait-for-dac))
	  (if save-body (warn "clm-load can't save its contents -- save-body applies only to with-sound."))
	  (c56-reset-cleanup)
	  (clm-cleanup)
	  (set-srate (or sampling-rate srate))
	  (setf last-dac-file-name (namestring file))
	  (setf *reverb-decay-time* decay-time)
	  (setf *clm-interrupted* 0)
	  (if statistics (clm-initialize-statistics statistics file revf))
	  (if notehook (clm-initialize-notehook notehook))
	  (setf *clm-force-recomputation* force-recomputation)
	  (if continue-old-file
	      (reopen-output file)
	    (open-output file (make-header :channels channels
					   :sampling-rate (or sampling-rate srate)
					   :type (or type default-sound-file-type)
					   :info (or info
						     (if (or comment commentary)
							 (format nil "#| ~A |#" (or comment commentary))
						       (clm-get-default-header))))))
	  (when revf
	    (setf *reverb*
	      (open-output revf
			   (make-header :channels reverb-channels :sampling-rate (or sampling-rate srate)
					:info ";temporary reverb stream"))))
	  
	  (catch :FINISH
	    (let ((*package* (if (packagep load-package)
				 load-package
			       (find-package load-package))))
	      (load pathname)))
	  
	  (when revf
	    (close-output *reverb*))
	  (when reverb
	    (open-input (or *open-input-explicit-reverb* default-reverb-file-name))
	    (apply reverb 0 (+ decay-time (clm-get-duration *current-input-file*))
		   reverb-data)
	    (close-input *reverb*))
	  
	  (close-output)
	  (if statistics (clm-print-statistics statistics))
	  (clm-cleanup)
	  (when (and play (not *open-input-explicit-output*))
	    (unless (numberp play)
	      (setf play 1))
	    (dac-n :file file :times play))
	  (when (and wait (not *open-input-explicit-output*))
	    (wait-for-dac)))
      (progn
	(setf *clm-force-recomputation* old-recompute)
	(clu-reset)))))



(defun start-a-dac (nam &optional (start 0.0) (end 0.0) srate)
  #-NeXT (declare (ignore start end))
  (if *sound-player*
      (funcall *sound-player* nam (or srate *last-play-options*))
    (let ((err #+NeXT (start-dac nam 0 start end)
	       #-NeXT (start-dac nam)))
      (if (/= err 0)
	  (warn "~A: ~A" nam (sound-error err))))))

;;; might also want *sound-player-ignore-error* and *sound-player-stop* (for abort-dac)
;;;
;;; so to have the dac function call Jean Laroche's play program instead of our
;;; built-in dac function (to get srate changes and so on), put "play" where
;;; it can be found as an executable (i.e. on the current directory, for example),
;;; then, (jrdac name) plays name at either its srate or the last explicit srate,
;;; (jrdac name srate) plays name at srate, (jrdac nil srate) plays the last file
;;; played at srate.

#+NeXT 
  (defun init-jrdac ()
    (setf *sound-player* #'(lambda (nam &rest rest) 
			     (declare (ignore rest))
			     (let ((str (format nil "play ~A" nam)))
			       #+excl (excl:shell str)
			       #+kcl (lisp:system str)
			       #+Clisp (shell str)
			       ))))

#+NeXT (defvar *last-jrdac-name* default-sound-file-name)
#+NeXT (defvar *last-jrdac-srate* sampling-rate)

#+NeXT
  (defun jrdac (&optional name sr)
    (let* ((srate (or sr *last-jrdac-srate*))
	   (str (format nil "play~A ~A" (if srate (format nil " -S~D" (floor srate)) "") (or name *last-jrdac-name*))))
      #+excl (excl:shell str)
      #+kcl (lisp:system str)
      #+Clisp (shell str)
      (if name (setf *last-jrdac-name* name))
      (if sr (setf *last-jrdac-srate* sr)))
    )

(defun sound-file-error-checks (nam) (next-sound-file-error-checks nam))

(defun dac (&optional (name nil) (worry-about-error t) (start 0.0) (end 0.0))
  (let ((nam (probe-file
	      (if (not name) 
		  (if (not last-dac-file-name)
		      default-sound-file-name
		    last-dac-file-name)
		(full-merge-pathnames name default-sound-file-name)))))
    (when (not nam)
      (setf nam (probe-file (full-merge-pathnames name "test.snd")))
      (when (not nam)
	(setf nam (probe-file (or name last-dac-file-name default-sound-file-name)))))
    (if nam
	(progn
	  (setf nam (namestring (truename nam)))
	  (setf last-dac-file-name nam)
	  (if (or (not worry-about-error) 
		  (Sound-File-Error-Checks nam))
	      (if (not (zerop (dsp-is-open)))
		  (if (y-or-n-p "DSP is busy: ~A -- use it anyway? " (dsp-who))
		      (progn
			(dsp-close)
			(start-a-dac nam start end)))
		(if (not (zerop (dac-is-in-use)))
		    (if (or *grab-dac* (y-or-n-p "DAC is running -- grab it? "))
			(progn
			  (abort-dac)
			  (start-a-dac nam start end))
		      (if (y-or-n-p "wait for it? ")
			  (progn
			    (wait-for-dac)
			    (start-a-dac nam start end))))
		  (if worry-about-error 
		      (start-a-dac nam start end) 
		    #+NeXT (start-dac nam 1 start end)
		    #-NeXT (start-dac nam)
		    )))))
      (if name 
	  (print (format nil "can't find ~A" name))
	(if last-dac-file-name
	    (print (format nil "what happened to ~A?" last-dac-file-name))
	  (print "can't figure out what to play"))))
    nam))

(defun dac-n (&key (file default-sound-file-name)
		   (times 1)
		   (worry t)
		   (start 0.0)
		   (end 0.0))			;doesn't work -- dac eventually hangs
  (if (or (not (numberp times)) (= times 1))
      (dac file worry start end)
    (loop for i from 1 to times do
      (dac file worry start end)
      (wait-for-dac))))

	

(defun close-output (&optional (o-stream *current-output-file*))
  (declare (optimize (speed 3) (safety 0)))
  (if (and o-stream
	   (not (io-p o-stream)))
      (error "the arguments to close-output must be an IO structure: ~A" o-stream))
  (c56-cleanup)
  (clm-close-output o-stream))

(defun close-input (&optional (i-stream *current-input-file*))
  (declare (optimize (speed 3) (safety 0)))
  (if (and i-stream
	   (not (io-p i-stream)))
      (error "the arguments to close-input must be an IO structure: ~A" i-stream))
  (c56-cleanup)
  (clm-close-input i-stream))

;;; OPEN-OUTPUT was in io.lisp also, but we need to do qp-initialization (if any) before
;;;   trying to start any instruments (to avoid stomping on external-memory primarily)

(defvar last-open-input-file-name nil)

(defun open-output (&optional (name default-sound-file-name) (header nil) (actual-file t) (buffer-size nil))
  (declare (optimize (speed 3) (safety 0)))
  #+56-mus-gens (c56-initialization-check nil)
  (clm-open-output name header actual-file buffer-size))

(defun reopen-output (&optional (name default-sound-file-name))
  (declare (optimize (speed 3) (safety 0)))
  #+56-mus-gens (c56-initialization-check nil)
  (clm-reopen-output name))

			

;;; MIX-SOUND
;;;
;;; for a "make" facility for sound file pieces.   Here we tie into the
;;; sound file headers etc.  Mix-Sound is called if the file-to-be-merged
;;; is up-to-date.  We need the name of the output file, input file,
;;; sample number in the output to begin at, sample number in the input
;;; to start at, number of channel-independent samples to merge (i.e. seconds*srate).
;;; Output header may be changed.

(defun mix-sound-wrapper (samps output-file output-sample input-file)
  (if (= samps -1) (error "cannot open output file ~A" output-file)
    (if (= samps -2) (error "cannot open input file ~A" input-file)
      (if (= samps -3) (warn "very strange -- mix completed but then headers unwritable!?!"))))
  (when clm-statistics
    (setf clm-last-begin-time (max clm-last-begin-time output-sample))
    ;; samples-to-merge can be -1=entire file
    (incf clm-total-duration samps))
  samps)

#-mcl (defvar dumbarr (make-array 1 :element-type 'fixnum :initial-element 0))
#+mcl (defvar dumbarr (c-make-array 1))

(defun mix-sound (output-file output-sample input-file input-sample samples-to-merge)
  (declare (optimize (speed 3) (safety 0)))
  (mix-sound-wrapper (c-mix-compatible-sounds output-file output-sample input-file input-sample samples-to-merge 0 0 dumbarr)
		     output-file output-sample input-file))

(defun make-typed-file-name (name ext)	;same name, directory, etc, but change "type" = extension
  (declare (optimize (speed 3) (safety 0)))
  (make-pathname :type ext :defaults name))



;;; OPEN-INPUT
;;; 
;;; this version is like lisp's load function to some extent in that
;;; if it gets an incomplete file name, or a cm/clm file name, it
;;; checks to see if the associated sound file is either not present
;;; or out of date and recomputes it if so.  In any case, open-input
;;; opens the sound file and returns an IO structure for it.  If it
;;; has to recompute the file, it must also close the current computation,
;;; cleanup all running dsps, then open the new computation, run it to
;;; completion, then reopen the previous computation where it left off.

(defun open-input (&optional name
		   &key (verbose nil verbose-p) 
			(print nil print-p) 
			(element-type nil element-p) ;can be :sound :clm :cm :lisp 
			(if-does-not-exist :error)
			(mix-at nil)
			(mix-duration nil)
			(force-recomputation *clm-force-recomputation*))
  (let* ((fname (or name last-open-input-file-name default-sound-file-name))
	 (sound-file-extension (or (and element-p
					(eq element-type :sound)
					(pathname-type name))
				   (pathname-type default-sound-file-name)))
	 (file-extension (pathname-type fname))
	 (sound-file-name fname))
    (if (or (not (string-equal file-extension sound-file-extension))
	    (and element-p (not (eq element-type :sound))))
	;; not an obvious sound file, so start checking for nested computations
	;; mimic the Load function in Lisp
	(let* ((*open-input-verbose* (or (and verbose-p verbose) *open-input-verbose*))
	       (*open-input-print* (or (and print-p print) *open-input-print*))
	       (*open-input-pathname* name)
	       (cm-file (probe-file (make-typed-file-name fname "cm")))
	       (clm-file (probe-file (make-typed-file-name fname "clm")))
	       (snd-file (probe-file (make-typed-file-name fname sound-file-extension)))
	       (cm-date (and cm-file (file-write-date cm-file)))
	       (clm-date (and clm-file (file-write-date clm-file)))
	       (snd-date (and snd-file (file-write-date snd-file))))
	  (if (or force-recomputation
		  (and (not snd-file)
		       (or cm-file clm-file))
		  (and cm-file snd-file
		       (> cm-date snd-date))
		  (and clm-file snd-file
		       (> clm-date snd-date)))

	      ;; close current computation, if any
	      
	      (let* ((old-output *current-output-file*)
		     (old-reverb *reverb*))
		(if (and *current-output-file*
			 (string-equal (namestring (make-typed-file-name fname sound-file-extension)) 
				       (io-nam *current-output-file*)))
		    (warn "we're about to overwrite ~A..." (io-nam *current-output-file*)))
		(if old-output (close-output old-output))
		(if old-reverb (close-output old-reverb))
		(if old-output (clm-cleanup))
		(let* ((*open-input-explicit-output* (namestring (make-typed-file-name fname sound-file-extension)))
		       (default-sound-file-name *open-input-explicit-output*)
		       ;; this is to turn off the directory fill-in in clm-open-input
		       ;; we can't use truename because it dies if it's passed a non-existent file
		       (*open-input-explicit-reverb* (namestring 
						      (make-pathname
						       :defaults fname
						       :name (concatenate 'string (pathname-name fname) "-reverb")
						       :type sound-file-extension)))
		       (*open-input-truename* (namestring (if (and cm-file clm-file)
							      (if (> cm-date clm-date)
								  cm-file
								clm-file)
							    (or cm-file clm-file)))))
		  (if *open-input-verbose* (printf (format nil "update ~A via ~A~% " *open-input-explicit-output* *open-input-truename*)))
		  (load *open-input-truename*)

		  (if (and cm-file (eq cm-file *open-input-truename*))
		      ;; now make sure cm actually made the new sound file
		      (let* ((clm-file (probe-file (make-typed-file-name fname "clm")))
			     (snd-file (probe-file (make-typed-file-name fname sound-file-extension)))
			     (clm-date (and clm-file (file-write-date clm-file)))
			     (snd-date (and snd-file (file-write-date snd-file))))
			(if (or (and (not snd-file) clm-file)
				(and clm-file (> clm-date snd-date)))
			    (let ((*open-input-truename* (namestring clm-file)))
			      (load *open-input-truename*)))))

		;; reopen old computation, if any
		
		  (if old-output (setf *current-output-file* (reopen-output (io-nam old-output))))
		  (if old-reverb (setf *reverb* (reopen-output (io-nam old-reverb))) (setf *reverb* nil))
		  (setf sound-file-name (truename *open-input-explicit-output*))))
	    (if snd-file (setf sound-file-name snd-file)))))

    (if (and sound-file-name (probe-file sound-file-name))
	(progn
	  (setf last-open-input-file-name sound-file-name)
	  (if (not mix-at)
	      (progn
		(c56-initialization-check nil)
		(clm-open-input sound-file-name))
	    (let* ((beg (floor (* mix-at sampling-rate))))
	      (flush-snd *current-output-file*)
	      (let ((samples (mix-sound (io-nam *current-output-file*) 
					beg 
					(if (pathnamep sound-file-name)
					    (namestring sound-file-name) 
					  (namestring (truename sound-file-name)))
					0 (or (and mix-duration (floor (* sampling-rate mix-duration))) -1))))
		;; the 0 -1 => start at start (beg=0) and go to the end (end=-1)
		;; mix-sound in C updates the header size field if needed
		(setf (io-siz *current-output-file*) (max (io-siz *current-output-file*) (+ beg samples)))

		(if (or (<= (io-beg *current-output-file*) beg (io-end *current-output-file*))
			(<= (io-beg *current-output-file*) (+ beg samples) (io-end *current-output-file*))
			(<= beg (io-beg *current-output-file*) (io-end *current-output-file*) samples))
		    (progn
		      ;; if we wrote new data into what would have been the in-core output buffer,
		      ;;   then clobber those pointers to force clm to re-read the output file if
		      ;;   any normal computation takes place.  It is possible to run without ever
		      ;;   actually accessing these buffers.
		      (setf (io-beg *current-output-file*) -1)
		      (setf (io-end *current-output-file*) -1)))))))

      (if (eq if-does-not-exist :error)
	  (error "cannot find ~A~A" name (if (not (eq name sound-file-name)) (format nil " (~A)" sound-file-name) ""))))))

(defun mix-in (source-file begin-time &optional duration)
  (declare (optimize (speed 3) (safety 0)))
  (open-input source-file :mix-at begin-time :mix-duration duration))

(defun evaluate-header (fname)
  ;; read all the forms in the header of the file fname and evaluate them as lisp code
  (let* ((f (open-input fname)))
    (unwind-protect
	(let* ((header (snd-header-comment (io-hdr f)))
	       (form nil)
	       (start 0)
	       (end (length header)))
	  (if header
	      (loop until (>= start end) do
		(multiple-value-setq 
		    (form start)
		  (read-from-string header nil nil :start start))
		(eval form))))
      (close-input f))))

(defvar *mix-calls* nil)
(defvar *mix-options* nil)

(defun get-mix-calls (f)
  (setf *mix-calls* nil)
  (setf *mix-options* nil)
  (evaluate-header f)
  (write-to-string *mix-calls*))

(defun get-mix-options (f)
  (declare (ignore f))
  (write-to-string *mix-options*))



;;; MIX
;;;
;;; this macro is wrapped around a relatively stable section of a piece and
;;; saves the computation in a separate file.  This output can be easily
;;; mixed in and moved around as a block as long as the forms going into
;;; its definition are not changed (a string comparison is done to check each time).
;;;
;;; (with-sound () 
;;;   (mix "section-1" 0 (fm-violin 0 1 440 .1)
;;;                      (fm-violin 1 2 660 .1))
;;;   (mix "section-2" ...)
;;;   )
;;;
;;; We need some way to recognize an interrupted section computation, but can't use
;;;   any header inconsistency (even when interrupted, clm produces a correct header),
;;;   and an extra .error file is kinda ugly, and probably won't work, but...


(defmacro mix (ur-chkpt-file ur-beg &body body)
  (let ((chkpt-file (eval ur-chkpt-file))
	(beg (eval ur-beg))
	(old-recompute *clm-force-recomputation*))
    (if (not (numberp beg))
	(error "begin time for ~S = ~A??" chkpt-file beg))
    (if (null body)
	(mix-in chkpt-file beg)
      (let ((call-str (write-to-string body))
	    (sndf (full-merge-pathnames (make-typed-file-name chkpt-file "snd") default-sound-file-name))
	    (errf (full-merge-pathnames (make-typed-file-name chkpt-file "error") default-sound-file-name))
	    (revf (full-merge-pathnames (make-typed-file-name chkpt-file "rev.snd") default-sound-file-name)))
	(if (and (not *clm-force-recomputation*)
		 (probe-file sndf)
		 (or (not *reverb*)
		     (probe-file revf))
		 (not (probe-file errf))
		 (string-equal (get-mix-calls sndf) call-str))
	    (progn
	      (if *clm-verbose* (printf (format nil "mixing ~A " (namestring sndf))))
	      (mix-in (namestring sndf) beg)
	      (if *reverb* (mix-sound (io-nam *reverb*) (floor (* beg sampling-rate)) (namestring revf) 0 -1)))
	  (let ((finished-ok nil))
	    (unwind-protect
		(let ((clmf (full-merge-pathnames (make-typed-file-name chkpt-file "clm") default-sound-file-name)))
		  (if *clm-verbose* 
		      (if (probe-file errf) 
			  (printf (format nil "~A was interrupted during previous computation -- will recompute it~%" (namestring sndf)))
			(printf (format nil "computing ~A " (namestring sndf)))))
		  (with-open-file (fil clmf 
				   :direction :output :if-does-not-exist :create :if-exists :supersede)
		    (format fil ";Temporary notelist for ~A~%~A~%" chkpt-file (clm-get-default-header))
		    (format fil "(with-sound (:srate ~D~%             :channels ~D~%             ~
                                              :sndfile ~S~%             :revfile ~A~%             ~
                                              :play nil~%             :info ~S)~%~{  ~S~%~})"
			    sampling-rate 
			    (if (quad *current-output-file*) 4 (if (stereo *current-output-file*) 2 1))
			    (namestring sndf)
			    (and *reverb* (format nil "~S" (namestring revf)))
			    (format nil "~A~%  (setf *mix-calls* '~A)~%" (clm-get-default-header) call-str)
			    body))
		  (mix-in clmf beg)
		  (if *reverb* (mix-sound (io-nam *reverb*) (floor (* beg sampling-rate)) (namestring revf) 0 -1))
		  (if (probe-file errf) (delete-file errf))
		  (setf finished-ok t))
	      (if (not finished-ok)
		  (let ((err (open errf :direction :output :if-exists :supersede)))
		    (close err))))))))
    (setf *clm-force-recomputation* old-recompute)
    nil))


(defvar temp-sound-ctr 0)

(defmacro sound-let (sounds &body body)
  ;; the syntax of each local sound is like with-sound -- a list of options, then the body
  ;; here, if any of the needed internal options are omitted, we append them (especially the file name)
  ;; the result of the let variable form is to return the temp snd file name as the value of the variable
  ;; so that in the sound-let body a reference to that variable is a reference to the associated file.

  ;; because these sounds are viewed as temporary computations, there's no effort made to save and reuse
  ;; them as in mix and friends -- this might be a nice addition someday.

  ;; first close the current with-sound computation, if any
  `(let* ((old-output *current-output-file*)
	  (old-reverb *reverb*)
	  (sound-file-list nil)
	  (old-recompute *clm-force-recomputation*))
     (if old-output (close-output old-output))
     (if old-reverb (close-output old-reverb))
     (if old-output (clm-cleanup))

     (loop for (snd opts calls) in ',sounds do
       (let ((name-loc (position :output opts)))
	 (push (namestring (full-merge-pathnames (if name-loc
						(nth (1+ name-loc) opts)
					      (concatenate 'string "snd" (format nil "~D" (incf temp-sound-ctr))))
					    default-sound-file-name))
	       sound-file-list)))

     (let* (,@(loop for (snd opts calls) in sounds and i from 0 and all-calls in sounds 
	       collect `(,snd (progn
				(with-sound (,@opts
					     ,@(if (not (find :output opts))
						   (list :output `(nth ,i sound-file-list)))
					     ,@(if (not (find :channels opts))
						   (list :channels `(clm-get-channels old-output)))
					     ;; the old way (no backquote, use of *current-output-file*) picked up
					     ;; the default with-sound keyword values, not those in effect at the
					     ;; point of the sound-let call.
					     ,@(if (not (find :srate opts))
						   (list :srate `(clm-get-sampling-rate old-output)))
					     ,@(if (find :play opts)
						   (list :wait t)
						 (list :play nil)))
				   ,@(cddr all-calls))
				(nth ,i sound-file-list)))))
       
       ;; now re-open the original with-sound computation before running the body of the sound-let
       (if old-output (setf *current-output-file* (reopen-output (io-nam old-output))))
       (if old-reverb (setf *reverb* (reopen-output (io-nam old-reverb))))

       (progn ,@body))

     (setf *clm-force-recomputation* old-recompute)
     ;; now clean up the temp output files
     (loop for snd in sound-file-list do (delete-file snd))))



;;; WITH-MIX
;;;
;;; like MIX except that local with-sound options are supported (I should have
;;; done this in MIX, but it's too late now).
;;;
;;; (with-sound () 
;;;   (with-mix () "section-1" 0 (fm-violin 0 1 440 .1)
;;;                              (fm-violin 1 2 660 .1))
;;;   (with-mix (:reverb nrev) "section-2" ...)
;;;   )


(defmacro with-mix (options ur-chkpt-file ur-beg &body body)
  (let ((chkpt-file (eval ur-chkpt-file))
	(beg (eval ur-beg))
	(old-recompute *clm-force-recomputation*))
    (if (not (numberp beg))
	(error "begin time for ~S = ~A??" chkpt-file beg))
    (if (null body)
	(mix-in chkpt-file beg)
      (let ((call-str (write-to-string body))
	    (option-str (write-to-string options))
	    (sndf (full-merge-pathnames (make-typed-file-name chkpt-file "snd") default-sound-file-name))
	    (errf (full-merge-pathnames (make-typed-file-name chkpt-file "error") default-sound-file-name))
	    (revf (full-merge-pathnames (make-typed-file-name chkpt-file "rev.snd") default-sound-file-name)))
	(if (and (not *clm-force-recomputation*)
		 (probe-file sndf)
		 (or (not *reverb*)
		     (probe-file revf))
		 (not (probe-file errf))
		 (string-equal (get-mix-calls sndf) call-str)
		 (string-equal (get-mix-options sndf) option-str))
	    (progn
	      (if *clm-verbose* (printf (format nil "mixing ~A " (namestring sndf))))
	      (mix-in (namestring sndf) beg)
	      (if *reverb* (mix-sound (io-nam *reverb*) (floor (* beg sampling-rate)) (namestring revf) 0 -1)))
	  (let ((finished-ok nil))
	    (unwind-protect
		(let ((clmf (full-merge-pathnames (make-typed-file-name chkpt-file "clm") default-sound-file-name)))
		  (if *clm-verbose* 
		      (if (probe-file errf) 
			  (printf (format nil "~A was interrupted during previous computation -- will recompute it~%" (namestring sndf)))
			(printf (format nil "computing ~A " (namestring sndf)))))
		  (with-open-file (fil clmf 
				   :direction :output :if-does-not-exist :create :if-exists :supersede)
		    (format fil ";Temporary notelist for ~A~%~A~%" chkpt-file (clm-get-default-header))
		    (format fil "(with-sound (~{~S ~A ~}~%             ~
                                              :play nil~%            ~
                                              ~A~A~A~A ~%             ~
                                              :info ~S)~%~
                                              ~{  ~S~%~})"
			    options                         
			    (if (not (find :output options))
				(format nil " :sndfile ~S" (namestring sndf))
			      "")
			    (if (not (find :channels options))
				(format nil " :channels ~D" (clm-get-channels *current-output-file*))
			      "")
			    (if (not (find :srate options))
				(format nil " :srate ~D" sampling-rate)
			      "")
			    (if (and *reverb* (not (find :reverb options)))
				(format nil " :revfile ~S" (namestring revf))
			      "")
			    (format nil "~A~%  (setf *mix-calls* '~A)~%  (setf *mix-options* '~A)~%" 
				    (clm-get-default-header) call-str option-str)
			    body))
		  (mix-in clmf beg)
		  (if *reverb* (mix-sound (io-nam *reverb*) (floor (* beg sampling-rate)) (namestring revf) 0 -1))
		  (if (probe-file errf) (delete-file errf))
		  (setf finished-ok t))
	      (if (not finished-ok)
		  (let ((err (open errf :direction :output :if-exists :supersede)))
		    (close err))))))))
    (setf *clm-force-recomputation* old-recompute)
    nil))



;;; very simple sound file processing is slower on the 56000 than in straight C
;;; because IO bandwidth to the chip is relatively low.
;;; Here is an example of a relatively fast mixing function using C.  All sound-level
;;; arithmetic is done with integers, and so on.  The name is a bit of nostalgia...
;;;
;;; in timing tests, this function runs about 4 to 5 times faster than the instrument add-sound.
;;;
;;; the underlying c functions (in merge.c) are:
;;;   c-mix-compatible-sounds
;;;   c-mix-mono-to-stereo-sounds
;;;   c-mix-stereo-to-mono-sounds
;;;   c-mix-stereo-to-stereo-sounds
;;;   c-mix-quad-to-quad-sounds
;;;
;;; the SHIFT constant in merge.c has to agree with the shift applied here via scale2 and scale3

(defun fasmix (in-file &key (start-time 0.0) start
			    (input-file-start-time 0.0) input-file-start
			    duration
			    amplitude
			    ampA ampB ampAB ampBA ampC ampD
			    amp-env
			    safe)
  (declare (ignore safe))
  (let* ((in-beg (or input-file-start (floor (* input-file-start-time sampling-rate))))
	 (out-beg (or start (floor (* start-time sampling-rate))))
	 (scale2 (expt 2 10))
	 (scale3 (expt 2 24))
	 (infile (namestring (full-merge-pathnames in-file default-sound-file-name)))
	 (outchans (clm-get-channels *current-output-file*))
	 (outfile-name (io-nam *current-output-file*)))
    (multiple-value-bind
	(inhead inhead-end infile-size inchans)
	(read-header infile)
      (if (not (and inhead inhead-end infile-size inchans))
	  (error "(read-header ~S) failed!" infile)
	(progn
	  #-mcl (flush-snd *current-output-file*)
	  #+mcl (close-output *current-output-file*) ;this also sets *current-output-file* to nil
					; Mac apparently doesn't allow more than one writer at a time, unlike Unix.
	  (if *clm-verbose* (printf (format nil "fasmix ~A ~1,3F " infile (float (/ out-beg sampling-rate)))))
	  (flet ((cify-data (ur-data new-samples) ;comes in as a list of floats (pass rate pass rate...)
		   (let* ((data (fix-envelope ur-data new-samples))
			  (new-data #-mcl (make-array (1+ (length data)) :element-type 'fixnum :initial-element 0)
				    #+mcl (c-make-array (1+ (length data)))
				    ))
		     (loop for dat on data by #'cddr and i from 0 by 2 do
		       #-mcl (progn
			       (setf (aref new-data i) (round (first dat)))
			       (setf (aref new-data (1+ i)) (round (* scale3 (second dat)))))
		       #+mcl (progn
			       (saref new-data i (round (first dat)))
			       (saref new-data (1+ i) (round (* scale3 (second dat)))))
		       )
		     #-mcl (setf (aref new-data (length data)) (floor (1+ new-samples)))
		     #+mcl (saref new-data (length data) (floor (1+ new-samples)))
		     new-data)))
	    (let ((samples (if (and (= outchans inchans)
				    (/= outchans 4)
				    (or (= outchans 1)
					(not (and ampA ampB))
					(and ampA ampB (= ampA ampB) (not ampAB) (not ampBA))))
			       (if (and (or (not amplitude) (= amplitude 1.0))
					(not amp-env))
				   (mix-sound-wrapper
				    (c-mix-compatible-sounds outfile-name out-beg infile in-beg 
								  (if duration (floor (* duration sampling-rate)) -1)
								  0 0 dumbarr)
				    outfile-name out-beg infile)
				 (if (not amp-env)
				     (let ((true-amp (round (* (or amplitude ampA ampB 1.0) scale2))))
				       (mix-sound-wrapper 
					(c-mix-compatible-sounds outfile-name
								      out-beg infile in-beg 
								      (if duration (floor (* duration sampling-rate)) -1) 
								      (if (zerop true-amp) 1 true-amp)
								      0 dumbarr)
					outfile-name out-beg infile))
				   (let* ((new-dur (or duration (float (/ infile-size sampling-rate))))
					  (new-samples (floor (* new-dur sampling-rate)))
					  (ampf (make-env :envelope amp-env 
							  :scaler (or amplitude ampA ampB 1.0)
							  :start-time 0
							  :duration (* inchans new-dur))))
				     (mix-sound-wrapper
				      (c-mix-compatible-sounds outfile-name
								    out-beg infile in-beg new-samples
								    (round (* scale3 (envelope-current-value ampf)))
								    1 (cify-data (envelope-data ampf) (* inchans new-samples)))
				      outfile-name out-beg infile))))
			     ;; else mono to stereo or stereo to mono or quad to quad
			     (if (and (= outchans 2)
				      (= inchans 1))
				 (if (not amp-env)
				     (mix-sound-wrapper
				      (c-mix-mono-to-stereo-sounds outfile-name
									out-beg infile in-beg 
									(if duration (floor (* duration sampling-rate)) -1)
									(if (and (or ampA amplitude)
										 (= (or ampA amplitude) 1.0))
									    0
									  (let ((true-amp (round (* (or ampA amplitude 0.0) scale2))))
									    (if (and (or ampA amplitude) (zerop true-amp)) 1 true-amp)))
									(if (and ampB (= ampB 1.0))
									    0
									  (round (* (or ampB 0.0) scale2)))
									(if (and ampA ampB
										 (not (zerop ampA))
										 (not (zerop ampB)))
									    -1
									  (if (or (not ampB) (zerop ampB)) 0 1))
									0 dumbarr)
				      outfile-name out-beg infile)
				   (let* ((new-dur (or duration (float (/ infile-size sampling-rate))))
					  (new-samples (floor (* new-dur sampling-rate)))
					  (ampf (make-env :envelope amp-env 
							  :scaler (or ampA ampB amplitude 1.0)
							  :start-time 0
							  :duration new-dur)))
				     (mix-sound-wrapper
				      (c-mix-mono-to-stereo-sounds outfile-name
									out-beg infile in-beg new-samples
									(round (* scale3 (envelope-current-value ampf)))
									(if (and ampB (not (zerop ampB))
										 ampA (not (zerop ampA)))
									    (round (* (/ ampA ampB) scale2))
									  0)
									(if (and ampA ampB
										 (not (zerop ampA))
										 (not (zerop ampB)))
									    -1
									  (if (or (not ampB) (zerop ampB)) 0 1))
									1 (cify-data (envelope-data ampf) new-samples))
				      outfile-name out-beg infile)))
			       ;; now the stereo to mono cases
			       (if (and (= outchans 1)
					(= inchans 2))
				   (if (not amp-env)
				       (mix-sound-wrapper
					(c-mix-stereo-to-mono-sounds outfile-name
									  out-beg infile in-beg 
									  (if duration (floor (* duration sampling-rate)) -1)
									  (if (and (or ampA amplitude)
										   (= (or ampA amplitude) 1.0))
									      0
									    (let ((true-amp (round (* (or ampA amplitude 0.0) scale2))))
									      (if (and (or ampA amplitude) (zerop true-amp)) 1 true-amp)))
									  (if (and ampB (= ampB 1.0))
									      0
									    (round (* (or ampB 0.0) scale2)))
									  (if (and ampA ampB
										   (not (zerop ampA))
										   (not (zerop ampB)))
									      -1
									    (if (or (not ampB) (zerop ampB)) 0 1))
									  0 0 dumbarr)
					outfile-name out-beg infile)
				     (let* ((new-dur (or duration (float (/ infile-size sampling-rate))))
					    (new-samples (floor (* new-dur sampling-rate)))
					    (ampf (make-env :envelope amp-env 
							    :scaler 1.0
							    :start-time 0
							    :duration new-dur)))
				       (mix-sound-wrapper
					(c-mix-stereo-to-mono-sounds outfile-name
									  out-beg infile in-beg new-samples
									  (round (* (or ampA amplitude 0.0) scale2))
									  (round (* (or ampB 0.0) scale2))
									  (if (and ampA ampB
										   (not (zerop ampA))
										   (not (zerop ampB)))
									      -1
									    (if (or (not ampB) (zerop ampB)) 0 1))
									  1 (round (* scale3 (envelope-current-value ampf)))
									  (cify-data (envelope-data ampf) new-samples))
					outfile-name out-beg infile)))
				 (if (= outchans inchans 2)
				     (mix-sound-wrapper
				      (c-mix-stereo-to-stereo-sounds outfile-name
									  out-beg infile in-beg 
									  (if duration (floor (* duration sampling-rate)) -1)
									  (round (* (or ampA 0.0) scale2))
									  (round (* (or ampB 0.0) scale2))
									  (round (* (or ampAB 0.0) scale2))
									  (round (* (or ampBA 0.0) scale2)))
				      outfile-name out-beg infile)
				   (if (= outchans inchans 4)
				       (let* ((new-dur (and amp-env (or duration (float (/ infile-size sampling-rate)))))
					      (new-samples (and amp-env (floor (* new-dur sampling-rate))))
					      (ampf (and amp-env
							 (make-env :envelope amp-env 
								   :scaler (or ampA amplitude 1.0)
								   :start-time 0
								   :duration new-dur))))
					 (mix-sound-wrapper
					  (c-mix-quad-to-quad-sounds outfile-name
									  out-beg infile in-beg 
									  (if duration (floor (* duration sampling-rate)) -1)
									  (round (* (or ampA 0.0) scale2))
									  (round (* (or ampB 0.0) scale2))
									  (round (* (or ampC 0.0) scale2))
									  (round (* (or ampD 0.0) scale2))
									  (+ (if (or ampA ampB ampC ampD) 1 0) 
									     (if amp-env 2 0))
									  (if amp-env 
									      (round (* scale3 (envelope-current-value ampf))) 
									    0)
									  (if amp-env 
									      (cify-data (envelope-data ampf) new-samples)
									    dumbarr))
					  outfile-name out-beg infile))
				     (error "fasmix can't do whatever you just asked it to do"))
				   ))))))
	      #+mcl (declare (ignore samples))
	      #+mcl (reopen-output outfile-name)
	      #-mcl (progn
		      (setf (io-siz *current-output-file*) (max (io-siz *current-output-file*) (+ out-beg samples)))
		      (if (or (<= (io-beg *current-output-file*) out-beg (io-end *current-output-file*))
			      (<= (io-beg *current-output-file*) (+ out-beg samples) (io-end *current-output-file*))
			      (<= out-beg (io-beg *current-output-file*) (io-end *current-output-file*) samples))
			  (progn
			    (setf (io-beg *current-output-file*) -1)
			    (setf (io-end *current-output-file*) -1))))
	      )))))))



#|
  ;; say we have these two sounds:
  (with-sound (:output "/zap/1.snd") (loop for i from 0 to 9 do (fm-violin i 1 (* (1+ i) 100) .1)))
  (with-sound (:output "/zap/2.snd" :channels 2) (loop for i from 0 to 9 do (fm-violin i 1 (* (- 10 i) 120) .1 :degree 60)))

   
   (with-sound ()
     (fasmix "1")                                    ;add "1.snd" to current output
     (fasmix "1" :duration 1.0 :amplitude .5)        ;scale first 1 sec of "1" by .5 and mix into current output
     (fasmix "1" :amplitude .5 :amp-env '(0 0 100 1));scale and envelope sound
     (fasmix "1" :duration 1.0 :start-time 1.5 :input-file-start-time 3.0))
                                                     ;take section in "1.snd" from 3.0 to 4.0 and mix into output starting at 1.5
   ;; and the same if stereo to stereo:

  (with-sound (:channels 2) (fasmix "2" :amp-env '(0 0 100 1)))

   ;; now mono-to-stereo:

  (with-sound (:channels 2) 
    (fasmix "1")                                      ;add "1" into channel 1 of output
    (fasmix "1" :ampA .5 :ampB .5)                    ;add .5 of "1" into each channel
    (fasmix "1" :ampB 1.0 :duration 2.0 :input-file-start-time 2.0 :amp-env '(0 0 1 1 2 1 3 0)))
                                                      ;put envelope portion (2.0:4.0) only into channel 2 (B)

  ;; and stereo-to-mono:

  (with-sound () 
    (fasmix "2")                                      ;channel 1 of "2" added into output (i.e. extract channel 1)
    (fasmix "2" :ampB .75)                            ;just channel 2 (scaled) of "2" into output
    (fasmix "2" :ampA .5 :ampB .75)                   ;mix chanA*.5+chanB*.75 into output
    (fasmix "2" :ampA 1.0 :duration 3.0 :input-file-start-time 2.0 :amp-env '(0 0 1 1 2 1 3 0)))
                                                      ;chanA from 2.0 to 5.0, enveloped

  ;; and stereo to stereo where we want to mix the separate channels as well as scale them

  (with-sound (:channels 2)
    (fasmix "2" :ampB 1.0 :ampA 1.0 :ampBA .1 :ampAB .2))
                                                      ;outA<-(inA*1.0 + inB*.2), outB<-(inA*.1 + inB*1.0)

  ;; fasmix can perform header and data type translations:

  (with-sound (:srate 8012) (fasmix "/me/cl/mulaw.snd"))  ;from mulaw to 16-bit linear
  (with-sound () (fasmix "/me/cl/aiff1.snd"))             ;from 8-bit to 16-bit linear (AIFF->NeXT)
  (with-sound () (fasmix "/me/cl/esps.snd"))              ;from ESPS to NeXT (16-bit linear)

|#

#|
(defun maxamp (file) 
  (let ((hi (open-input (namestring (full-merge-pathnames file default-sound-file-name))))) 
    (multiple-value-bind (a b) (clm-get-max-amp hi) 
      (close-input hi) 
      (list a b))))
|#

;;; and now a sound display function that actually works, as opposed to the
;;; silly one supplied by NeXT and seemingly embedded in all NeXT programs.

(defun dpysnd (file start-1)
  (let* ((fil (clm-open-input file))
	 (arr (make-array 128 :element-type 'float))
	 (start (floor start-1)))
    (loop for i from 0 to 127 and k from start by 1 do
      (setf (aref arr i) (ina k)))
    (close-input fil)
    (show-data arr)))


;;; (with-sound (:verbose t :force-recomputation t) 
;;;   (mix "x1" 0 (fm-violin 0 1 440 .1) (fm-violin 1 1 660 .1)) 
;;;   (with-mix () "x2" .75 (fm-violin 0 .25 550 .05) (fm-violin .5 .25 550 .1)))



;;; these are for Common Music's benefit

(defstruct wsdat revfun revdat revdecay outtype play stats wait)

(defun init-with-sound (&key (output default-sound-file-name)
			    sndfile
			    (channels 1)
			    (srate 22050)
			    sampling-rate
			    continue-old-file
			    (reverb nil revthere)
			    reverb-data
			    (reverb-channels 1)
			    revfile
			    (play t)
			    play-options
			    (cleanup-first t)
			    wait
			    notehook
			    statistics
			    (decay-time 1.0)
			    output-buffer-size
			    info
			    type
			    force-recomputation
			    (verbose *clm-verbose*)
			    commentary
			    comment)
  (let ((*clm-verbose* verbose)
	(out-file (or *open-input-explicit-output*
		      (namestring (full-merge-pathnames (or sndfile output) default-sound-file-name))))
	(out-file-p (or *open-input-explicit-output* sndfile output))
	(revf (and (or revthere revfile) (or revfile *open-input-explicit-reverb* default-reverb-file-name))))
    (if (/= 0 (dac-is-in-use)) (wait-for-dac))
    ;; on the 68040's a collision here can cause a machine crash!
    (when cleanup-first
      (c56-reset-cleanup)		;in case called recursively in error handler
      (clm-cleanup))
    (set-srate (or sampling-rate srate))
    (setf last-dac-file-name out-file)
    (setf *last-play-options* play-options)
    (setf *reverb-decay-time* decay-time)
    (setf *clm-interrupted* 0)
    (if statistics (clm-initialize-statistics statistics out-file revf))
    (clm-initialize-notehook notehook)
    (setf *clm-force-recomputation* force-recomputation)
    (if continue-old-file
	(reopen-output out-file)
      (open-output out-file 
		   (make-header :channels channels
				:sampling-rate (or sampling-rate srate)
				:type (or type default-sound-file-type)
				:info (format nil "~A" 
					      (or info
						  (if (or comment commentary)
						      (format nil "#| ~A |#" (or comment commentary))
						    (if *open-input-explicit-output*
							(concatenate 'string 
							  (clm-get-default-header)
							  (format nil " (from ~A via open-input)" 
								  (namestring *open-input-truename*)))
						      (clm-get-default-header))))))
		   out-file-p 
		   output-buffer-size))
    (if revf
	(setf *reverb* (open-output revf
				    (make-header :channels reverb-channels
						 :sampling-rate (or sampling-rate srate)
						 :info ";temporary reverb stream"))))
    (make-wsdat :revfun reverb :revdat reverb-data :revdecay decay-time :outtype type :play play :stats statistics :wait wait)))


(defun finish-with-sound (wsd)
  (when *reverb* 
    (let ((revname (io-nam *reverb*)))
      (close-output *reverb*)
      (open-input revname)
      (apply (wsdat-revfun wsd) 0 (+ (wsdat-revdecay wsd) (clm-get-duration *current-input-file*)) (wsdat-revdat wsd))
      (close-input)))
  (let ((outname (io-nam *current-output-file*)))
    (close-output)
    (if (wsdat-stats wsd) (clm-print-statistics (wsdat-stats wsd)))
    (clm-cleanup)
    #+mcl (if (and (wsdat-outtype wsd)
		   (= (wsdat-outtype wsd) AIFF-sound-file)) 
	      (ccl:set-mac-file-type outname "AIFF"))
    (when (wsdat-play wsd)
      (dac-n :file outname :times (wsdat-play wsd))
      (when (wsdat-wait wsd)
	(wait-for-dac)))))
