;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-

(defconstant sms-magic 767)
(defconstant sms-format-harmonic 1)
(defconstant sms-format-inharmonic 2)
(defconstant sms-stoc-filter-type 1)
(defconstant sms-stoc-mag-env-type 2)
(defconstant sms-stoc-none 3)

(defconstant *max-dB* 96.3296)

(defstruct SMSHeader
  (iSmsMagic 0 :type fixnum)
  (iHeadBSize 0 :type fixnum)
  (nRecords 0 :type fixnum)
  (iRecordBSize 0 :type fixnum)
  (iFormat 0 :type fixnum)
  (iFrameRate 0 :type fixnum)
  (iStochasticType 0 :type fixnum)
  (nTrajectories 0 :type fixnum)
  (nStochasticCoeff 0 :type fixnum)
  (fAmplitude 0 :type short-float)
  (fFrequency 0 :type short-float)
  (iBegSteadyState 0 :type fixnum)
  (iEndSteadyState 0 :type fixnum)
  (nLoopRecords 0 :type fixnum)
  (nSpecEnvelopePoints 0 :type fixnum)
  (nTextCharacters 0 :type fixnum)
  LoopRecords
  SpectralEnvelope
  TextCharacters)

(defstruct smsTraj
  freq
  mag)

(defstruct smsStoc
  stocGain
  stocCoeff)

(defun un-db (x)
  (if (zerop x)
      0.0
    (expt 10 (/ (- x 96.3296) 20))))

(defun convert-c-float (i) 
  (if (zerop i)
      0.0
    (* (if (zerop (ash i -31)) 1.0 -1.0) 
       (expt 2 (- (logand (ash i -23) #xff) 127)) 
       (logior #x800000 (logand i #x7fffff)) 
       (expt 2 -23))))
;; i.e. sign bit, 8 bits exponent, 23 bits mantissa with assumed 1 bit in 24-th bit.

(defvar detTraj nil)
(defvar stocData)
(defvar header (make-SMSHeader))

(defun readSmsHeader (file)
  (let ((smsfile (open file :element-type '(unsigned-byte 32))))
    (if (/= (read-byte smsfile) sms-magic)
	(print (format nil "~A is not an smsAnal output file." file)))
    (setf (SMSHeader-iHeadBSize header) (read-byte smsfile)) ;in bytes
    (setf (SMSHeader-nRecords header) (read-byte smsfile))
    (setf (SMSHeader-iRecordBSize header) (read-byte smsfile)) ;in bytes
    (setf (SMSHeader-iFormat header) (read-byte smsfile))
    (if (not (plusp (SMSHeader-nRecords header)))
	(print (format nil "~A has ~A records?" file
		       (SMSHeader-nRecords header))))
    (setf (SMSHeader-iFrameRate header)(read-byte smsfile))
    (setf (SMSHeader-iStochasticType header) (read-byte smsfile))
    (setf (SMSHeader-nTrajectories header) (read-byte smsfile))
    (setf (SMSHeader-nStochasticCoeff header)(read-byte smsfile))
    (setf (SMSHeader-fAmplitude header) (read-byte smsfile))
    (setf (SMSHeader-fFrequency header) (read-byte smsfile))
    (setf (SMSHeader-iBegSteadyState header) (read-byte smsfile))
    (setf (SMSHeader-iEndSteadyState header) (read-byte smsfile))
    (setf (SMSHeader-nLoopRecords header) (read-byte smsfile))
    (setf (SMSHeader-nSpecEnvelopePoints header) (read-byte smsfile))
    (setf (SMSHeader-nTextCharacters header) (read-byte smsfile))
					;jump over header
    (file-position smsfile (floor (SMSHeader-iHeadBSize header) 4))     		
    smsfile))

(defun goToRecord (smsfile number)
  (file-position smsfile 
		 (floor (+ (* number (SMSHeader-iRecordBSize header))
			   (SMSHeader-iHeadBSize header)) 4)))

(defun readTrajRecord (smsfile rec traj)
  (let ((pos-rec (floor (+ (* rec (SMSHeader-iRecordBSize header))
			   (SMSHeader-iHeadBSize header)) 4))
	(nTraj (SMSHeader-nTrajectories header))
	(eof-value -1)
	(value 0.0))
    (file-position smsfile (+ pos-rec traj))
    (if ( = (setf value (read-byte smsfile nil eof-value)) eof-value)
	(return-from readTrajRecord eof-value))
    (push rec (smsTraj-freq detTraj))
    (push (in-Hz (convert-c-float value)) (smsTraj-freq detTraj))
    (file-position smsfile (+ pos-rec traj nTraj))
    (if ( = (setf value (read-byte smsfile nil eof-value)) eof-value)
	(return-from readTrajRecord eof-value))
    (push rec (smsTraj-mag detTraj))
    (push (un-db (convert-c-float value)) (smsTraj-mag detTraj))))

(defun readStocRecord (smsfile rec sizeDet)
  (let ((pos-rec (floor (+ (* rec (SMSHeader-iRecordBSize header))
			   (SMSHeader-iHeadBSize header)) 4))
	(nCoeff (SMSHeader-nStochasticCoeff header))
	(eof-value -1)
	(value 0.0))
    (file-position smsfile (+ pos-rec sizeDet))
    (loop for coeff from 0 below nCoeff do
	  (if ( = (setf value (read-byte smsfile nil eof-value)) eof-value)
	      (return-from readStocRecord eof-value))
	  (push rec (aref (smsStoc-stocCoeff stocData) coeff))
	  (push (convert-c-float value) (aref (smsStoc-stocCoeff stocData) coeff)))
    (if ( = (setf value (read-byte smsfile nil eof-value)) eof-value)
	(return-from readStocRecord eof-value))
    (push rec (smsStoc-stocGain stocData))
    (push (un-db (convert-c-float value)) (smsStoc-stocGain stocData))))

(defun shorten (e)
  (let ((e1 nil)
	(y0 nil)
	(y1 nil)
	(x nil)
	(y nil))
    (loop while e do
      (setf x (pop e))
      (setf y (pop e))
      (when (or (not y0) (not y1) (/= y0 y1) (/= y y1))
	(push x e1)
	(push y e1)
	(setf y0 y1)
	(setf y1 y)))
    (when (/= (second e1) x)
      (push x e1)
      (push y e1))
    (nreverse e1)))

(defun readSmsTraj (smsfile traj first-record last-record)
  (setf detTraj (make-smsTraj))
  (loop for rec from first-record to last-record do (readTrajRecord smsfile rec traj))
  (setf (smsTraj-freq detTraj) (shorten (nreverse (smsTraj-freq detTraj))))
  (setf (smsTraj-mag detTraj) (shorten (nreverse (smsTraj-mag detTraj))))
  dettraj)

(defun readSmsStoc (smsfile first-record last-record)
  (let ((nCoeff (SMSHeader-nStochasticCoeff header))
	(sizeDet (+ (* 2 (SMSHeader-nTrajectories header)) 
		    (if (= (SMSHeader-iFormat header) sms-format-inharmonic) 
			(SMSHeader-nTrajectories header)
		      0))))
    (setf stocData (make-smsStoc :stocCoeff (make-array nCoeff :element-type 'list :initial-element nil)))
    (loop for rec from first-record to last-record do
	  (readStocRecord smsfile rec sizeDet))
    (setf (smsStoc-stocGain stocData) (shorten (nreverse (smsStoc-stocGain stocData))))
    (loop for coeff from 0 below nCoeff do
      (setf (aref (smsStoc-stocCoeff stocData) coeff) (shorten (nreverse (aref (smsStoc-stocCoeff stocData) coeff))))))
  stocData)

(definstrument filter-noise (beg dur amp ampf coeffs)
  (let* ((st (floor (* beg sampling-rate)))
	 (noi (make-randh :frequency (* .5 sampling-rate) 
			  :amplitude (/ amp 5)))
	 (order (length coeffs))
	 (amp-env (make-env :envelope ampf :scaler 1.0 :start-time beg 
			    :duration dur))
	 (flA (make-lattice-filter :m order
				   :k nil))
	 (envs (make-array order :element-type 'envelope))
	 (nd (+ st (floor (* sampling-rate dur))))
	 (value 0)
	 (last-value 0))
    
    (loop for i from 0 below order do
	  (setf (aref (flt-b flA) i) 1.0)
	  (setf (aref envs i) (make-env :envelope (aref coeffs i) 
					:scaler 1.0 :start-time beg 
					:duration dur :base 0)))
     (Run					
     (loop for i from st to nd do
	   (dotimes (k order)
		    (setf (aref (flt-a flA) k) (env (aref envs k))))
	   (setf value (+ (lattice-filter flA (* (env amp-env)(randh noi)))))
	   (outa i (+ value (* .9 last-value)))
	   (setf last-value value)))))

(defcinstrument sms (file beg &optional (orig-beg 0.0)
		 (orig-dur -1)
		 (max-traj -1)
		 (freq-mult 1.0)
		 (freq-env '(0 1 100 1))
		 (amp-mult 1.0)
		 (amp-env '(0 1 100 1))
		 (stretch-factor 1.0)
		 (stretch-fun '(0 1 100 1))
		 (spectral-compression 1.0)
		 (spectral-comp-env '(0 1 100 1)))
  (let ((smsfile (readSmsHeader file)))
    (unwind-protect
	(let* ((nRecords (SMSHeader-nRecords header))
	       (nTraj (if (= max-traj -1) (SMSHeader-nTrajectories header)
			(min (SMSHeader-nTrajectories header) max-traj)))
	       (first-record (min (1- nRecords) 
				  (floor (* orig-beg (SMSHeader-iFrameRate header)))))
	       (last-record (if (= orig-dur -1) 
				(1- nRecords)
			      (min (1- nRecords)
				   (floor (* (+ orig-beg orig-dur) (SMSHeader-iFrameRate header))))))
	       (read-records (1+ (- last-record first-record)))
	       (traj-frq (make-array nTraj))
	       (traj-amp (make-array nTraj))
	       (duration (/ read-records (SMSHeader-iFrameRate header)))
	       (st (floor (* sampling-rate beg)))
	       (nd (+ st (floor (* sampling-rate duration))))
	       (amp-mod (make-env :envelope amp-env :scaler 1.0 :start-time beg :duration duration))
	       (frq-mod (make-env :envelope freq-env :start-time beg :duration duration :scaler 1.0))
	       (traj-osc (make-array nTraj))
	       (j -1)
	       (last-value 0.0))

	  (readSmsStoc smsfile first-record last-record)
          (if (= (SMSHeader-iStochasticType header) sms-stoc-filter-type)
	      (filter-noise beg duration amp-mult (smsStoc-stocGain stocData)
			    (smsStoc-stocCoeff stocData))
	    (warn "can't handle fft style noise section yet"))

	  (dotimes (k nTraj)
		   (readSmsTraj smsfile k first-record last-record)
	    (when (> (max-envelope (smsTraj-mag detTraj)) 0.0)
	      (incf j)
	      (setf (aref traj-osc j) (make-oscil :frequency 0))
	      (setf (aref traj-frq j) (make-env :envelope (smsTraj-freq detTraj) :start-time beg :duration duration :scaler freq-mult))
	      (setf (aref traj-amp j) (make-env :envelope (smsTraj-mag detTraj) :start-time beg :duration duration :scaler amp-mult))))

	  (Run
	   (loop for i from st to nd do
	     (let ((sum 0.0))
	       (dotimes (k j)
		 (incf sum (* (env (aref traj-amp k)) (oscil (aref traj-osc k) (* (env frq-mod) (env (aref traj-frq k)))))))
	       (setf sum (* (env amp-mod) sum))
	       (outa i (+ (* .9 last-value) sum)) ;a "de-emphasis filter"
	       (setf last-value sum)))))

      (close smsfile))))

