;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-

(in-package :clm)

;;; MULTIPLE DSP SUPPORT

(defvar dsp-pool nil)			;list (or array) of available dsps (dsp-data structs)
(defvar built-in-dsp nil)		;optimization for bare NeXTs
(defvar max-dsp 0)			;max dsp used so far

#+QP (defvar QP-master nil)		;Ariel QP board master dsp (DSPE in their nomenclature)
					;QP-master is an array of pointers to the master chips
#+QP (defvar qp-dram-as-cache t)	;t if dram used as cache
#+QP (defvar main-QP-master nil)	;optimization for reverb
#+QP (defvar max-QPs-possible 3)

(defstruct ld dsp beg end curbeg curend bufsiz len 
	      sigops sigadr sigptr sigbas sigtop sigsiz 
	      dly-ops in-ops (end-run nil) (SRAM-locs nil) (x-or-ys nil) (end-run-values nil))
					;the structure used to manage an active dsp
					;ld-dsp is a pointer to a structure describing the given dsp

(defconstant cpu-dsp 0)
#+QP (defconstant qp-manager 1)
#+QP (defconstant qp-worker 2)
(defconstant dsp-drone 3)		;a drone is a dsp program that does not respond to "normal" dsp communications
					;the dsp-data-status field is used normally only to notice qp-manager -- other
					;  types are all the same as long as they aren't qp-manager. 

(defstruct (dsp-data 
	    (:print-function
	     (lambda (d s k)
	       (declare (ignore k))
	       (format s "#<Dsp-Data: ~Aslot: ~A, dsp: ~A, open: ~A, in-use: ~A, ~
                           ext-mem-siz: ~A, mem-map: ~X, ins-name: ~A, state: ~A, ~
                           status: ~A, pool: ~A, hub: ~A, :x-off ~A, :y-off ~A, :P-overlaid ~A >"
		       (if (dsp-data-machine d)
			   (format nil "machine: ~A, " (dsp-data-machine d))
			 "")
		       (if (integerp (dsp-data-slot d)) 
			   (if (zerop (dsp-data-slot d))
			       "CPU" 
			     (dsp-data-slot d))
			 nil)
		       (if (integerp (dsp-data-dsp d)) 
			   (if (and (integerp (dsp-data-slot d))
				    (zerop (dsp-data-slot d)))
			       "Built-in DSP"
			     #+QP (if (qp-master-p d)
				      "QP master"
				    (dsp-data-dsp d))
			     #-QP (dsp-data-dsp d)
			     )
			 nil)
		       (if (dsp-data-open d) "yes" "no")
		       (if (dsp-data-in-use-by d) "yes" "no")
		       (dsp-data-ext-mem-siz d)
		       (dsp-data-mem-map d)
		       (dsp-data-ins-name d)
		       (if (integerp (dsp-data-state d))
			   (case (dsp-data-state d) 
			     (0 "dsp normal") 
			     (1 "dsp ready to merge")
			     #+QP (2 "qp merging")
			     (3 "dsp merging")
			     #+QP (4 "qp uninitialized")
			     #+QP (5 "qp is reverb")
			     (6 "dsp manually disabled")
			     #+QP (7 "qp initializing")
			     #+QP (8 "qp normal")
			     (9 "dsp uninitialized")
			     (10 "dsp not currently in use")
			     #+QP (11 "qp shifting")
			     (t (format nil "unknown state: ~D" (dsp-data-state d))))
			 (dsp-data-state d))
		       (case (dsp-data-status d)
			 (0 "cpu") 
			 (1 "manager") 
			 (2 "worker") 
			 (3 "drone")
			 (t (format nil "unknown status: ~D" (dsp-data-status d))))
		       (dsp-data-pool d)
		       (if (dsp-data-hub d) "yup" "nope")
		       (dsp-data-ext-X-offset d)
		       (dsp-data-ext-Y-offset d)
		       (dsp-data-P-overlaid d)))))
  (dsp 0) 
  (slot 0) 
  (machine nil)
  (open nil) 
  (in-use-by nil) 
  (ext-mem-siz 0)
  (mem-map 0) 
  (ins-name nil) 
  (state dsp-not-currently-in-use)
  (hub nil)
  (status cpu-dsp)
  (pool 0)
  (ext-x-offset #xA000)
  (ext-y-offset 4096)			;!Actually y-from-X-offset!
  (p-overlaid t))

(defun c56-reinitialize ()
  (setf (aref dsp-pool 0) (make-dsp-data :ext-mem-siz external-L-size 
					 :ext-X-offset ext-X-offset
					 :ext-Y-offset external-Y-from-X-offset
					 :P-overlaid ext-P-overlaid))
  (setf built-in-dsp (aref dsp-pool 0)))


(defun c56-initialize ()		;find out what chips are available, get descriptions of them
  (clear-host-interfaces)
  (set-up-dsp-memory-map)
  (let ((dspnum (+ 1 
		   #+QP (qp-initialize))))
					;look for QP boards (sets up qp-info)
					;  assume built-in dsp is there.
    (setf dsp-pool (make-array dspnum :element-type 'dsp-data))
    (setf (aref dsp-pool 0) (make-dsp-data :ext-mem-siz external-L-size 
					   :ext-X-offset ext-X-offset
					   :ext-Y-offset external-Y-from-X-offset
					   :P-overlaid ext-P-overlaid))
    ;; this is the built-in dsp
    (setf built-in-dsp (aref dsp-pool 0))
    #+QP (when (> dspnum 1)			;there are others
	   (let ((qpi 0)
		 (qpn 0))
	     (if (/= 4096 external-Y-from-X-offset)
		 (warn "QP monitor is not compatible with ~A" (memory-map-name external-Y-from-X-offset)))
	     (setf qp-master (make-array max-QPs-possible :element-type 'dsp-data))
	     (loop for i from 0 below 3 do ;check each slot (via qp-info)
	       (when (zerop (aref qp-info i)) ;0 => there's a QP board present
		 (loop for k from 0 to 3 do ;set up the slaves
		   (incf qpi)
		   (setf (aref dsp-pool qpi) 
		     (make-dsp-data :dsp k 
				    :slot (* 2 (+ i 1))
				    :ext-mem-siz (floor (* .5 (aref qp-info (+ 6 i))))
				    :state dsp-not-currently-in-use
				    :status qp-worker
				    :pool qpi
				    :hub (if (/= -1 (aref qp-info (+ 9 i))) (aref qps i) nil)
				    :P-overlaid t
				    :ext-X-offset #xA000
				    :ext-y-offset 4096 ;(floor (* .5 (aref qp-info (+ 6 i))))
				    :mem-map (if qp-dram-as-cache #xe00000 #x600000))))
		 (incf qpi)
		 (setf (aref QP-master qpn)
		   (make-dsp-data :dsp 4
				  :slot (* 2 (+ i 1))
				  :ext-mem-siz 4096 ;L size wanted here
				  :state qp-uninitialized
				  :status qp-manager
				  :pool qpi
				  :hub (if (/= -1 (aref qp-info (+ 9 i))) (aref qps i) nil)
				  :P-overlaid t
				  :ext-X-offset #xA000
				  :ext-Y-offset 4096
				  :mem-map 0))
		 (if (null main-QP-master) (setf main-QP-master (aref QP-master qpn)))
		 (if (dsp-data-hub (aref QP-master qpn))
		     (setf (QP-data-DRAM-size (aref qps i)) (aref qp-info (+ i 3))))
		 (setf (qp-data-master (aref qps i)) (aref QP-master qpn))
		 (setf (aref dsp-pool qpi) (aref QP-master qpn))
		 (incf qpn)))
	     ))
    dspnum))

(defvar first-note t)			;this is a desperate kludge (QP board hangs in boot if CPU DSP not booted!?)
(defvar global-beg 0)			;optimization when flushing QP DRAM cache (doesn't need to be a global variable...)

#+QP (defun qp-master-p (m) (and m (dsp-data-p m) (= (dsp-data-status m) qp-manager)))

(defun c56-initialization-check (&optional check-qp)
  #-QP (declare (ignore check-qp))
  (if (zerop dsps) (setf dsps (c56-initialize)))
  #+QP (if (and check-qp
		qp-dram-as-cache
		(> dsps 1)
		qp-master
		(not first-note)
		(= qp-uninitialized (dsp-data-state (aref qp-master 0))))
	   (loop for i from 0 to 2 do
	     (if (aref qp-master i)
		 (qp-fire-up-merging-monitor (aref qp-master i) t))))
  )

(defun find-dsp (slot dsp)		;given slot and dsp number, find associated dsp-data record
  (if (or (= dsps 1) 
	  (zerop slot)) 
      built-in-dsp
    (do ((i 0 (1+ i)))
	((or (= i dsps)
	     (and (= (dsp-data-slot (aref dsp-pool i)) slot)
		  (= (dsp-data-dsp (aref dsp-pool i)) dsp)))
	 (if (= i dsps)
	     built-in-dsp
	   (aref dsp-pool i))))))

(defun find-free-dsp ()
  (if (= dsps 1)	;just the built-in 56000 available
      (if (= dsp-not-currently-in-use (dsp-data-state built-in-dsp))
	  built-in-dsp
	nil)
    (if (and first-note 
	     (not (setf first-note nil))
	     (/= dsp-off (dsp-data-state built-in-dsp)))
	built-in-dsp
      (do ((i 1 (1+ i)))
	  ((or (= i dsps)
	       (and #+QP (not (qp-master-p (aref dsp-pool i)))
		    (= dsp-not-currently-in-use (dsp-data-state (aref dsp-pool i)))))
	   (if (= i dsps)
	       (if (= dsp-not-currently-in-use (dsp-data-state built-in-dsp))
		   built-in-dsp
		 nil)
	     (progn
	       (setf max-dsp (max max-dsp i))
	       (aref dsp-pool i))))))))

(defvar currently-active-dsps 0)

(defun allocate-dsp (dsp ldp)
  (incf currently-active-dsps)
  (setf (ld-dsp ldp) dsp)
  (setf (dsp-data-state dsp) dsp-normal)
  (setf (dsp-data-in-use-by dsp) ldp))

(defun deallocate-dsp (dsp)
  (decf currently-active-dsps)
  (if (= dsp-merging (dsp-data-state dsp))
      (error "attempt to deallocate unmerged dsp!"))
  (setf (dsp-data-state dsp) dsp-not-currently-in-use)
  (setf (dsp-data-in-use-by dsp) nil))
  
(defun who-am-i ()			;who is currently active (just debug proc)
  (if (zerop current-slot)
      (format nil "CPU dsp (~A):~A" (if (/= 0 (dsp-is-open)) "open" "closed") built-in-dsp)
    #+QP (if (or (= 2 current-slot) (= 4 current-slot) (= 6 current-slot))
	     (format nil 
		     "QP dsp: ~D, slot: ~D (~A):~A" 
		     current-dsp 
		     current-slot 
		     (if (/= 0 (qp-is-open)) "open" "closed")
		     (find-dsp current-slot current-dsp))
	   (format nil "slot confusion"))
    ))

(defun some-dsp-is-open () 
  (or (/= 0 (dsp-is-open))
      (and (> dsps 1)
	   #+QP (/= 0 (qp-is-open))
	   )))

(defun dsp-at-work (dsp) 
  ;;  (member (dsp-data-state dsp) (list dsp-normal dsp-merging dsp-ready-to-merge #+QP qp-as-reverb)))
  (let ((state (dsp-data-state dsp)))
    (and 
     (or (= state dsp-normal)
	 (= state dsp-merging)
	 (= state dsp-ready-to-merge)
	 #+QP (= state qp-as-reverb)
	 )
     (/= dsp-drone (dsp-data-status dsp)))))
  
(defun some-dsp-is-running ()
  ;; this is called a zillion times, so it's worth a global variable
  (plusp currently-active-dsps))

(defun make-active-dsp (dsp &optional (check-state t))
  (when dsp
    ;;    (if (member (dsp-data-state dsp) 
    ;;		(list dsp-off dsp-merging dsp-uninitialized qp-uninitialized dsp-not-currently-in-use))
    (if check-state
	(let ((state (dsp-data-state dsp)))
	  (if (or (= state dsp-merging)
		  (= state dsp-off)
		  (= state dsp-uninitialized)
		  #+QP (= state qp-uninitialized)
		  (= state dsp-not-currently-in-use))
	      (error "attempt to make ~A active" dsp))))
    (dsp-set (dsp-data-slot dsp) (dsp-data-dsp dsp)))
  dsp)

(defun find-named-dsp (name)
  ;; as we get more dsps, this might want to become a hash table lookup
  (loop for i from 0 to max-dsp do
    (if (eq (dsp-data-ins-name (aref dsp-pool i)) name)
	(return (make-active-dsp (aref dsp-pool i) nil)))))

(defun delay-ops (ld-ptr)
  (and ld-ptr
       (integerp (ld-dly-ops ld-ptr))
       (/= 0 (ld-dly-ops ld-ptr))))

(defun data-ready (dsp)
  (when dsp
    (make-active-dsp dsp)
    (and (or (and (eq dsp built-in-dsp)
		  (/= 0 (dsp-is-open)))
	     (and (> dsps 1) 
		  #+QP (/= 0 (qp-is-open))
		  ))
	 (/= 0 (dsp-data-ready)))))

(defun in-frame (ld-ptr)
  (and (>= (ld-curbeg ld-ptr) (io-beg (or *current-output-file* (error "attempt to write output with no output file open"))))
       (<= (ld-curend ld-ptr) (io-end *current-output-file*))
       (or (zerop (ld-in-ops ld-ptr))
	   (<= (+ (ld-curend ld-ptr) (ld-bufsiz ld-ptr)) (io-end *current-output-file*)))))
  ;; input using INA is one buffer ahead of the game (READIN is handled separately)
  ;; we move all the file buffers at once (for simplicity?), so we need two buffers worth
  ;; if in-ops is not 0 (it is an indication that IN-N was called).

#|
(defun ready (dsp)
  (or (data-ready dsp)
      (delay-ops (dsp-data-in-use-by dsp))))
|#
(defun ready (dsp)
  (or (delay-ops (dsp-data-in-use-by dsp))
      (data-ready dsp)))

(defun find-best-dsp (dsp1 dsp2)
  (if (null dsp2) dsp1
    (if (null dsp1) dsp2
      (let* ((ld1 (dsp-data-in-use-by dsp1))
	     (ld2 (dsp-data-in-use-by dsp2))
	     (dsp1-beg (ld-curbeg ld1))
	     (dsp2-beg (ld-curbeg ld2))
	     (dsp1-time (- (ld-end ld1) dsp1-beg))
	     (dsp2-time (- (ld-end ld2) dsp2-beg)))
	(if (/= dsp1-beg dsp2-beg)
	    (if (or (= dsp1-time dsp2-time)
		    (< (abs (- dsp1-time dsp2-time)) file-buffer-size))
		(if (< dsp1-beg dsp2-beg)
		    dsp1
		  dsp2)
	      (if (< dsp1-time dsp2-time)
		  dsp1
		dsp2))
	  (if (<= dsp1-time dsp2-time)
	      dsp1
	    dsp2))))))

(defvar retries 0)
(defvar max-retry 100)
(defvar max-wait max-retry)

(defun dsp-ready ()			;here we try to decide who should run and in what way
  (if (zerop max-dsp)			;i.e. there's only the built-in dsp running
      (if (and (dsp-at-work built-in-dsp)
	       (ready built-in-dsp))	;does he have data ready?
	  built-in-dsp
	nil)
    #+QP (if (> dsps 1)
	     (if (and main-QP-master 
		      (= qp-as-reverb (dsp-data-state main-Qp-master)))
		 (if (ready main-qp-master)
		     main-QP-master
		   nil)
	       (let ((qpn nil)
		     (init-on nil) 
		     (merge-on nil) 
		     (shift-on nil) 
		     (not-on nil) 
		     (qp-busy nil)
		     (found-merger nil) 
		     (found-in-frame nil)
		     (found-in-DRAM-frame nil) 
		     (found-running 0)
		     (io-dsp nil) 
		     (best-dsp nil) 
		     (any-dsp nil))
		 (loop for i from 0 to max-dsp do
		   (let* ((curdsp (aref dsp-pool i))
			  (cur-qp-hub (dsp-data-hub curdsp))
			  (cur-qp-master (and cur-qp-hub (qp-data-master cur-qp-hub))))
		     (if (and cur-qp-master
			      (not (eq qpn cur-qp-master)))
			 (let ((qp-state (dsp-data-state cur-qp-master)))
			   (setf qpn cur-qp-master)
			   (setf init-on (and (= qp-initializing qp-state)
					      (not (qp-master-is-done-initializing cur-qp-master))))
			   (setf merge-on (and (= qp-merging qp-state)
					       (not (qp-master-is-done-merging cur-qp-master))))
			   (setf shift-on (and (= qp-shifting qp-state)
					       (not (qp-master-is-done-shifting cur-qp-master))))
			   (setf not-on (= qp-uninitialized qp-state))
			   (setf qp-busy (or merge-on init-on not-on shift-on))
			   (setf found-merger nil)))
		     (let* ((state (if curdsp (dsp-data-state curdsp) dsp-uninitialized))
			    (state-ok (or (= state dsp-normal)
					  (= state dsp-ready-to-merge)))
			    (running (and state-ok (dsp-data-in-use-by curdsp)))
			    (d-ready (and running 
					  (or (= state dsp-ready-to-merge)
					      (ready curdsp)))))
		       (if running (incf found-running))
		       (when (and running
				  (plusp i) 
				  (not not-on))
			 (if (and d-ready (= state dsp-normal))
			     (setf (dsp-data-state curdsp) dsp-ready-to-merge))
			 (let ((DRAM-ok (QP-in-current-DRAM-frame running cur-qp-master)))
			   (setf found-in-DRAM-frame (or found-in-DRAM-frame DRAM-ok))
			   (if (and d-ready (not qp-busy) DRAM-ok)
			       (return-from dsp-ready curdsp)))) ;if any ready to merge on QP, and it's ready, handle it first
		       
		       (when (= state dsp-merging)
			 (if (not merge-on)
			     (error "confusion in merger during QP initialization"))
			 (if (and merge-on found-merger)
			     (error "two dsp's think they are merging!"))
			 (setf found-merger t))
		       
		       (setf found-in-frame (or found-in-frame (and running (in-frame running))))
		       
		       (if d-ready
			   (setf any-dsp (find-best-dsp any-dsp curdsp)))
		       (if (and d-ready (in-frame running))
			   (setf io-dsp (find-best-dsp io-dsp curdsp)))
		       (if (= state dsp-ready-to-merge)
			   (setf best-dsp (find-best-dsp best-dsp curdsp))))))
		 ;; if we get here, we didn't send off to QP merge (i.e. either QP busy, or no one in current frame etc)

		 (if (not found-in-DRAM-frame)
					;if no one is awaiting merge into DRAM (which we handle if at all possible)
		     (if io-dsp
			 io-dsp		;if in current IO frame, take least first
		       (if (not found-in-frame)
					;if anyone is in the current frame, wait for him
			   (progn	;else if we've waited awhile already, choose best of currently ready dsps
			     (incf retries)
			     (when (or (= 1 found-running) ; i.e. there's only one active dsp
				       (> retries max-wait))
			       (setf retries 0)
			       (if best-dsp best-dsp any-dsp))))))))
	   nil)
    ))



;;; HF3="at a breakpoint" flag (i.e. in the monitor awaiting attention of user or lisp).
;;; HF2="data ready" flag (i.e. in .flush-buffers ready to send blocks of output).
;;; HF2&HF3="at .input" flag (for random access IO).


(defun c56-reset-all-files (act-dsp beg)
  ;; this just forces us to have all files starting at beg (no data is being sent)
  (clm-reset-all-files beg 0 t)
  (if (/= 0 (ld-in-ops act-dsp))
      (loop for i from 0 below (ld-len act-dsp) do
	(when (= 1 (aref (ld-sigops act-dsp) i))
	  (setf (aref (ld-sigbas act-dsp) i) beg)
	  (setf (aref (ld-sigtop act-dsp) i) 
	    (min 
	     (+ beg file-buffer-size -1)
	     (- (aref (ld-sigsiz act-dsp) i) 1)))
	  (if (< beg (aref (ld-sigsiz act-dsp) i))
	      (setf (aref (ld-sigptr act-dsp) i) (ld-bufsiz act-dsp))
	    ;; we're reading one buffer ahead if INA, so next read starts at bufsiz
	    (setf (aref (ld-sigptr act-dsp) i) file-buffer-size))))))

(defun check-for-flushable-data ()
  #+QP (if (or (= max-dsp 0)
	       (not qp-master)
	       (not (aref qp-master 0))
	       (not (qp-flush (aref qp-master 0) global-beg))
	       (not (aref qp-master 1))
	       (not (qp-flush (aref qp-master 1) global-beg))
	       (not (aref qp-master 2))
	       (not (qp-flush (aref qp-master 2) global-beg)))
	   (clm-flush-all-files))
  nil)

(defun grab-dsp-data (dsp)		;get files in proper position, merge in next block of data from dsp, etc
  (if dsp
      (progn
	;; dsp is DSP-DATA-data pointer -- its IN-USE-BY field points to the LD structure that
	;;  is keeping track of where this dsp is currently in the various open files.
	;;  We currently just track the basic output file on the assumption that if he's happy, so are the others.
	
	(let ((act-dsp (dsp-data-in-use-by (make-active-dsp dsp))))
	  
	  ;; act-dsp has the current state of the dsp program
	  ;; grab sends bufsiz, examines list, gets next block, resets all pointers,
	  ;;   lets dsp go on, then merges buffers into output files
	  ;;   if end reached, close dsp, remove struct, else just return
	  
	  (let ((cursiz (1+ (- (ld-curend act-dsp) (ld-curbeg act-dsp)))))
            (if (or (= dsp-normal (dsp-data-state dsp))
		    (null (dsp-data-hub dsp))
		    #+QP (/= qp-normal (dsp-data-state (qp-data-master (dsp-data-hub dsp))))
		    #+QP (and (not (QP-merge dsp act-dsp cursiz global-beg))
			      (make-active-dsp dsp)) ;master made active in QP-merge and sometimes left that way
		    )
	      (progn
		(if (= (dsp-data-state dsp) dsp-merging)
		    (error "attempt to merge while already merging!"))
                (if (not (in-frame act-dsp))
		    (c56-reset-all-files act-dsp (ld-curbeg act-dsp)))
		(let* ((curbeg (- (ld-curbeg act-dsp) (io-beg *current-output-file*)))
		       (curend (+ curbeg cursiz -1)))
		  (if (or (minusp curbeg)
			  (> curend file-buffer-size))
		      (error "attempt to write outside buffer bounds: ~D to ~D" curbeg curend))
		  
		  ;; file buffers should now be ready for merging current dsp's data
		  (c-read-dsp-block curbeg
				    curend
				    (ld-len act-dsp)
				    (ld-dly-ops act-dsp)
				    (ld-in-ops act-dsp)
				    (ld-sigops act-dsp)
				    (ld-sigadr act-dsp)
				    (ld-sigptr act-dsp)
				    (ld-sigbas act-dsp)
				    (ld-sigtop act-dsp)
				    (ld-sigsiz act-dsp))
		  (clm-flush-all-files curbeg curend)
		  (when (= dsp-ready-to-merge (dsp-data-state dsp))
		    (setf (dsp-data-state dsp) dsp-normal))
		  (update-dsp act-dsp dsp)))))))
    (check-for-flushable-data))
  nil)
  
(defun update-dsp (act-dsp dsp)
  (incf (ld-curbeg act-dsp) (ld-bufsiz act-dsp))
  (setf (ld-curend act-dsp) (min (ld-end act-dsp) 
				 (+ (ld-curend act-dsp) (ld-bufsiz act-dsp))))
  (if (>= (ld-curend act-dsp) (ld-end act-dsp)) (setf (ld-in-ops act-dsp) 0))
  (if (and (/= dsp-merging (dsp-data-state dsp))
	   (>= (ld-curbeg act-dsp) (ld-end act-dsp)))
      (finish-dsp act-dsp dsp))
  nil)

(defun finish-dsp (act-dsp dsp)
  (if (ld-end-run act-dsp) (funcall (ld-end-run act-dsp)))
  (if *clm-locsigs* (forget-locsig act-dsp))
  (deallocate-dsp dsp)
  nil)

(defun run-until-dsp-free ()
  (do* ((dsp (dsp-ready) (dsp-ready))
	(tmp (grab-dsp-data dsp) (grab-dsp-data dsp))
	;; grab-dsp-data flushes finished data whenever there's nothing else to do
	;; so it is not a waste of time to call it a lot -- however, it might help
	;; to leave the find-free-dsp call until dsp is not nil
	(freed (find-free-dsp) (find-free-dsp)))
      (freed freed)))

(defun run-until-all-dsps-free ()
  (do* ((dsp (dsp-ready) (dsp-ready))
	(tmp (grab-dsp-data dsp) (grab-dsp-data dsp)))
      ((not (some-dsp-is-running)))))

(defun wait-for-free-dsp ()
  (if (or (= dsps 1)			;no choice (i.e. no QP board in this case)
	  (null DRAM-list))
      (or (find-free-dsp)
	  (run-until-dsp-free))
    #+QP (if (not (dsp-at-work main-QP-master))
	     (progn			;DRAM-list and QP present, so use QP DRAM (reverb optimization)
	       (setf max-dsp (max max-dsp (dsp-data-pool main-QP-master)))
	       main-QP-master)
	   (run-until-dsp-free))
    #-QP (run-until-dsp-free)
    ))

(defvar *clm-current-open-file-data* nil)

(defun c56-reset-cleanup ()
  (if dsp-pool
      (progn
	(loop for i from 0 below dsps do 
	  (let ((c-dsp (aref dsp-pool i)))
	    (setf (dsp-data-open c-dsp) nil)
	    (if (/= (dsp-data-state c-dsp) dsp-off)
		(setf (dsp-data-state c-dsp) dsp-not-currently-in-use))
	    (setf (dsp-data-in-use-by c-dsp) nil)))
	#+QP (when qp-master
	       (loop for i from 0 to 2 do
		 (when (and (aref qp-master i) 
			    (/= qp-uninitialized (dsp-data-state (aref qp-master i))))
		   (setf (dsp-data-in-use-by (aref qp-master i)) nil)
		   (qp-reset-cleanup (aref qp-master i) (dsp-data-hub (aref qp-master i))))))
	#+QP (if (> dsps 1)
		 (if (/= 0 (qp-is-open)) 
		     (qp-close)))
	)
    (setf dsps (c56-initialize)))
  (setf *clm-current-open-file-data* nil)
  (setf max-dsp 0)
  (setf external-envelopes nil)
  (setf global-beg 0)
  (setf max-wait max-retry)
  (setf first-note t)
  (setf currently-active-dsps 0)
  (if (/= 0 (dsp-is-open)) (dsp-close))
  (setf in-dsp-break nil))

#+QP (defun qp-cleanup ()
       (loop for i from 0 to 2 do
	 (if (aref qp-master i)
	     (qp-cleanup-1 (aref qp-master i)))))

(defun c56-cleanup ()
  (setf global-beg (floor (max (* 2 global-beg) (* sampling-rate 1000))))
  (setf max-wait 10)
  (run-until-all-dsps-free)
  #+QP (if (and qp-master 
		(> dsps 1) 
		(/= 0 (qp-is-open)))
	   (qp-cleanup))
  (c56-reset-cleanup))


(defun allocate-dsp-IO-and-process-data (beg end ex ey sigs dsp &optional no-output-here)
  (let* ((len (length sigs))
	 (sigadr (make-array len :element-type 'fixnum))
	 (sigops (make-array len :element-type 'fixnum))
	 (sigbas (make-array len :element-type 'fixnum :initial-element 0))
	 (sigtop (make-array len :element-type 'fixnum :initial-element 0))
	 (sigsiz (make-array len :element-type 'fixnum :initial-element 0))
	 (sigptr (make-array len :element-type 'fixnum :initial-element 0))
	 (srams  (make-array len :element-type 'fixnum :initial-element 0))
	 (xs     (make-array len :element-type 'fixnum :initial-element 0))
	 (signum (get-addresses sigs sigadr))
	 (bufsiz (if (dsp-data-P-overlaid dsp)
		     (set-buf-siz ex ey signum (dsp-data-ext-mem-siz dsp) no-output-here)
		   (set-buf-siz ex ey signum Internal-L-size no-output-here)))
         ) 
    (multiple-value-bind
	(dly-ops in-ops)
	(get-ops sigs sigops sigsiz sigbas sigtop sigptr bufsiz ex ey srams xs)
      #+QP (if (QP-master-p dsp) (setf dly-ops 0))
      (setf global-beg beg)
      (allocate-dsp
       dsp
       (add-dsp-data dsp beg end bufsiz len sigops sigadr sigptr sigbas sigtop sigsiz dly-ops in-ops srams xs)))))

(defun current-dsp-start-again (dsp tag tag-loc)
  (and dsp
       (dsp-data-open dsp)
       (or (and (zerop current-slot)
		(/= 0 (dsp-is-open)))
	   (and (> dsps 1)
		#+QP (/= 0 (qp-is-open))
		))
       (/= 0 (dsp-start-again tag tag-loc))))


(defun set-up-program-and-data (name
				*ix-pc*
				*iy-pc*
				*ex-pc*
				*ey-pc*
				*run-beg*
				*run-end*
				*sigarr*
				tag
				tag-loc
				monitor-end
				&optional no-output-here)
  (let ((dsp (wait-for-free-dsp)))
    (if (dsp-data-P-overlaid dsp)
	(allocate-dsp-IO-and-process-data *run-beg* *run-end* *ex-pc* *ey-pc* *sigarr* dsp no-output-here)
      (allocate-dsp-IO-and-process-data *run-beg* *run-end* *ix-pc* *iy-pc* *sigarr* dsp no-output-here))
    
    (when *clm-verbose*
      (if (and clm-last-begin-time 
	       *run-beg* 
	       (> (- *run-beg* clm-last-begin-time) (* sampling-rate 10000)))
	  (warn "begin time = ~,3F, but last begin time = ~,3F?" (/ *run-beg* sampling-rate) (/ clm-last-begin-time sampling-rate)))
      (setf clm-last-begin-time (max (or clm-last-begin-time 0) *run-beg*)))
    (when clm-statistics
      (if (not *clm-verbose*) (setf clm-last-begin-time (max clm-last-begin-time *run-beg*)))
      (incf clm-total-duration (- *run-end* *run-beg*)))
    (if clm-notehook (eval clm-notehook))
    (when *clm-verbose* (princ (format nil "~(~A~) ~,3F " name (/ *run-beg* sampling-rate))) (force-output))
    
    (make-active-dsp dsp)
    #+QP (if (> dsps 1) (setf-y-mem (qp-size-location) (qp-size-code dsp)))
    #+QP (when DRAM-list		;there are requests for external delays
	   (when (and (> dsps 1)	;there is a QP board
		      (QP-master-p dsp)) ;and we are running on the master 
	     ;; can't use QP-DRAM for presets
	     (let ((preset nil))
	       (loop for i in DRAM-list while (not preset) do 
		 (if (>= (getf-y-mem i) 8) (setf preset t)))
	       (when (not preset)
		 (setf (dsp-data-state dsp) qp-as-reverb)
		 (loop for i in DRAM-list do 
		   (setf-y-mem i 4)))))
	   (setf DRAM-list nil))
    (let ((dsp-err 
	   (if (not (current-dsp-start-again dsp tag tag-loc))
	       (let ((proglen (get name :dsp-compiled-program-length)))
		 (if (< proglen 60)
		     (error "~A must have loaded incorrectly" name))
		 (setf (dsp-data-open dsp) t)
		 (if (dsp-data-P-overlaid dsp)
		     (dsp-open (1- proglen)
			       (get name :dsp-compiled-program)
			       (dsp-data-slot dsp)
			       (dsp-data-dsp dsp)
			       monitor-end
			       (dsp-data-mem-map dsp))
		   (dsp-open (min proglen 500)
			     (get name :dsp-compiled-program)
			     (dsp-data-slot dsp)
			     (dsp-data-dsp dsp)
			     monitor-end
			     (dsp-data-mem-map dsp))))
	     0)))
      (if (= 0 dsp-err)
	  (progn
	    (if (dsp-data-P-overlaid dsp)
		(setf dsp-err (dsp-set-up-program *ix-pc* *iy-pc* *ex-pc* *ey-pc* 
						  0 (dsp-data-ext-x-offset dsp) (dsp-data-ext-y-offset dsp) 0 
						  internal-x-memory 
						  internal-y-memory 
						  external-memory))
	      
	      (let* ((len (get name :dsp-compiled-program-length))   
		     (poff (if (> len 500) 501 0))
		     (pctr (if (> len 500) (- len 500) 0)))
		(setf dsp-err (dsp-set-up-program *ix-pc* *iy-pc* 0 0
						  pctr 0 0 poff
						  internal-x-memory
						  internal-y-memory
						  (get name :dsp-compiled-program)))))
	    (if (/= dsp-err 0)
		(error (case dsp-err 
			 (1 "Can't start data load")
			 (2 "Can't load internal x data")
			 (3 "Can't start Y data load")
			 (4 "Can't load internal y data")
			 (5 "Can't start external x data load")
			 (6 "Can't load external x data")
			 (7 "Can't start external y data load")
			 (8 "Can't load external y data")
			 (9 "Can't start external P load")
			 (10 "Can't load external P memory")
			 (t "Unknown set up error")))
	      (let ((dsp-l (dsp-data-in-use-by dsp)))
		(setf last-loaded-ld dsp-l)
		(setf *current-instrument-name* name)
		(setf (dsp-data-ins-name dsp) name)
		(if (/= 0 (ld-in-ops dsp-l))
		    (dsp-send-input (ld-sigbas dsp-l)
				    (min (ld-bufsiz dsp-l) (- *run-end* *run-beg* -1))
				    (ld-sigops dsp-l)
				    (ld-sigptr dsp-l)
				    (ld-sigadr dsp-l)
				    (ld-sigsiz dsp-l)
				    (ld-len dsp-l))))))
	
	(if (= -1 dsp-err) 
	    (error "could not get host interface!")
	  (if (= -2 dsp-err)
	      (error "could not boot DSP! -- some other process is probably using it: ~A" (DSP-who))
	    (error "unknown DSP error: ~D" dsp-err)))))
    dsp))

  

;;;	  MAKE-PHRASE WAIT-FOR-PHRASE PHRASE-VALUE PHRASE
;;;
;;; phrasing -- here we provide a mechanism to wait for a given dsp to finish, and
;;; then pickup arbitrary values from its memory.  The user has to pass around a variable
;;; to keep data in for each separate phrase.
  
(defstruct phr dsp (ready t) vars rest)
  
(defun make-phrase_56 (&optional arg) (make-phr :rest arg))
  
(defun end-run_56 (&optional phrase)
  (when (active-dsp)
    (if (not phrase)
	(setf (ld-end-run (active-dsp)) 'clm-end-run) 
      (progn
	(setf (ld-end-run-values (active-dsp)) phrase)
	(setf (ld-end-run (active-dsp)) #'(lambda ()
					    (clm-end-run)
					    (get-final-values)))))
    (ld-dsp (active-dsp))))

(defun get-final-values ()
  (let* ((act-ld (active-dsp))
	 (act-dsp (ld-dsp act-ld))
	 (ins-name (dsp-data-ins-name act-dsp))
	 (ins-vars (get ins-name :ins-vars))
	 (ld-phrase (ld-end-run-values act-ld)))
    (when ld-phrase
      (let ((alist (phr-vars ld-phrase)))
	(loop for (var . val) in alist do
	  (if (eq val :unbound)
	      (rplacd (assoc var (phr-vars ld-phrase)) 
		      (on-chip-value (gethash (symbol-name var) ins-vars))))))
      (setf (phr-ready ld-phrase) t))
    act-dsp))

(defun on-chip-value (hash-val)
  (get-dsp-value (second (first hash-val)) (second hash-val) (third hash-val)))

(defun run-until-desired-dsp-free (desired-dsp)
  (when desired-dsp
    (do* ((dsp (dsp-ready) (dsp-ready))
	  (tmp (grab-dsp-data dsp) (grab-dsp-data dsp)))
	((= dsp-not-currently-in-use (dsp-data-state desired-dsp))))))

(defun wait-for-phrase_56 (&rest phrases)
  (loop for phrase in phrases do
    (when (phr-p phrase)
      (if (not (phr-ready phrase))
	  (if (and (dsp-data-p (phr-dsp phrase))
		   (/= dsp-not-currently-in-use (dsp-data-state (phr-dsp phrase))))
	      (run-until-desired-dsp-free (phr-dsp phrase))
	    (progn
	      ;; user started anew with a left over phrase marker, I guess
	      (setf (phr-ready phrase) t)
	      (setf (phr-dsp phrase) nil)
	      (loop for (var . val) in (phr-vars phrase) do
		(if val
		    (rplacd (assoc var (phr-vars phrase)) :unbound)
		  (setf (phr-vars phrase) (acons var :unbound (phr-vars phrase))))))))
      (if (and (phr-dsp phrase) 
	       (not (phr-ready phrase)))
	  (print "phrase finished but without setting values")))))

(defun phrase-value_56 (phrase var)
  (if phrase
      (if var
	  (and (phr-vars phrase)
	       (let ((val (cdr (assoc var (phr-vars phrase)))))
		 (and (not (eq val :unbound))
		      val)))
	(phr-rest phrase))))

(defun setf-phrase-value_56 (phrase var val)
  (if var (print "bad setf phrase-value"))
  (setf (phr-rest phrase) val))

(defsetf phrase-value_56 setf-phrase-value_56)

(defun on-chip (var)
  (let ((vars (get *current-instrument-name* :ins-vars)))
    (not (eq (gethash var vars :does-not-exist) :does-not-exist))))

(defun phrase_56 (p &rest vars)
  (when p 
    (if (active-dsp)
	(progn
	  (setf (phr-dsp p) (ld-dsp (active-dsp)))
	  (setf (phr-ready p) nil))
      (setf (phr-ready p) t))
    (setf (phr-vars p) 
      (pairlis vars (loop for var in vars
		     collect (if (and (active-dsp) 
				      (on-chip (symbol-name var)))
				 :unbound 
			       var)))))
  p)


#+QP (defvar old-dsps 0)
#+QP (defvar old-master nil)

#+QP (defun QP-off () 
       (c56-initialization-check)
       (setf old-dsps dsps)
       (setf dsps 1)
       (if (null built-in-dsp) 
	   (setf built-in-dsp (make-dsp-data :ext-mem-siz external-L-size 
					     :ext-X-offset ext-X-offset
					     :ext-y-offset external-Y-from-X-offset
					     :P-overlaid ext-P-overlaid)))
       ;; this is the built-in dsp
       (setf old-master QP-master)
       (setf QP-master nil))

#+QP (defun QP-on ()
       (if old-master
	   (progn
	     (setf dsps old-dsps)
	     (setf qp-master old-master)
	     (setf old-master nil))))

#+QP (defun DRAM-off () 
       (c56-initialization-check)
       (setf qp-dram-as-cache nil)
       (loop for i from 1 to 4 do 
	 (setf (dsp-data-ins-name (aref dsp-pool i)) nil)
	 (setf (dsp-data-mem-map (aref dsp-pool i)) #x600000)))

#+QP (defun DRAM-on () 
       (c56-initialization-check)
       (setf qp-dram-as-cache t)
       (loop for i from 1 to 4 do 
	 (setf (dsp-data-ins-name (aref dsp-pool i)) nil)
	 (setf (dsp-data-mem-map (aref dsp-pool i)) #xe00000)))

(defun cpu-off ()
  (c56-initialization-check)
  (setf (dsp-data-state built-in-dsp) dsp-off)
  (setf (dsp-data-in-use-by built-in-dsp) nil))

(defun cpu-on ()
  (setf (dsp-data-state built-in-dsp) dsp-normal))
