;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;

(in-package :clm)

;;;
;;; intermediate code generator using walk-form
;;;
;;; Unless it runs into something it can't handle, Run turns its argument into I-code that should
;;; be relatively easy to translate into DSP code (that comment obviously preceded a close look at
;;; the 56000...) 

(defun nopfun (x y z) (declare (ignore y z)) x)

(defmacro fully-expand (x) 
  (pprint (walk-form x nil 'nopfun)))

(defvar unhappy nil)
(defvar old-prog nil)
(defvar loop-beg nil)
(defvar loop-end nil)
(defvar loop-prog nil)
(defvar setf-structure-ref nil)
(defvar loop-label-stack nil)

(defvar t-lab-stack nil)		;jumping code stacks (simulate procedure call stack)
(defvar f-lab-stack nil)
(defvar j-lab-stack nil)
(defvar return-stack nil)
(defvar tables nil)
(defvar addresses nil)
(defvar last-var nil)

(defconstant boole-ops '(= /= < > <= >= or and not minusp zerop plusp logbitp logtest oddp evenp null))
(defun boole-op (op) (member op boole-ops))

(defvar dsp-functions nil)

(defun dsp-function (x) 
  (or (and dsp-functions (gethash x dsp-functions))
      (and user-structs (gethash x user-structs))))

(defun def-dsp-fun (x y) 
  (if (null dsp-functions) (setf dsp-functions (make-hash-table)))
  (setf (gethash x dsp-functions) y))

(defun associated-function (x) 
  (or (gethash x dsp-functions)
      ;; now look for a user-struct field name and if found, conjure up the appropriate response
      (let* ((desc (gethash x user-structs)))
	(if desc                        ;accessor-name {real|(array el-typ)} struct-name field-name
	    #'(lambda (var px) 
		(package-address x var px (first (second desc)) (third desc) (or (second (second desc)) 'real) desc))))))
   
;;; the dsp name needs to be different from the actual name so that we can trap
;;; built-in functions in Run without shadowing them.

#-mcl
(defmacro Run (run-time-code &optional no-output-here)
  (if (and no-output-here (not (equal no-output-here t)))
      (warn "Run got '~A as it's second argument -- is this really what you want??" no-output-here))
  `(macroexpand-1 (run-1 ,run-time-code ,no-output-here)))

#+mcl
(defmacro Run (run-time-code &optional no-output-here)
  (declare (ignore no-output-here))
  run-time-code)

(defvar outer-environment nil)

(defmacro Run-1 (run-time-code &optional no-output-here &environment env) 
  ;;Lisp expands this twice unless we take other action (i.e. run and run-1 two-step)
  (setf new-prog nil
	unhappy nil
	loop-beg nil
	loop-end nil
	loop-var nil
	loop-label-stack nil
	new-sig nil
	user-var nil
	true-user-var nil
	typed-user-var nil
	typed-user-sig nil
	full-array-names nil
	tables nil
	addresses nil
	setf-first-var nil
	local-vars nil
	t-lab-stack nil
	f-lab-stack nil
	j-lab-stack nil
	return-stack nil
	run-safety *clm-safety*			;default SAFETY setting
	run-speed *clm-speed*
	setf-structure-ref nil
	checked-variables nil
	inited-arrays nil
	dsp-pc 0			;in case no 56k code generated
	outer-environment env		;needed by macrolet (see macroexpand in dsp-ize)
	loop-prog (walk-form run-time-code nil 'dsp-ize))
  #-(and excl (not Allegro-v3.1)) (DEBUGGING (pprint (append (list 'progn) (reverse new-prog))))
  #+(and excl (not Allegro-v3.1)) (DEBUGGING (print (format nil "~{  ~A~%~}" (reverse new-prog))))
  (if unhappy				;we hit something we can't deal with
      `(macroexpand-1 (run-3 ,run-time-code))
					;try to optimize the Lisp using the C unit generator library
    `(progn
       (<start> ,loop-beg ,loop-end) 
       ,@(nreverse new-prog) 
       (<end> ,no-output-here))))


(defun give-up (x)
  (setf unhappy t)
  (if *ask-before-falling-back-on-C-and-Lisp-in-run*
      (if (eq *ask-before-falling-back-on-C-and-Lisp-in-run* :warn)
	  (warn "Attempt to compile into ~(~A~) failed -- falling back on ~(~A~)" 
		(if (eq *clm-language* :c) "optimized C" "56000 code")
		(if (eq *clm-language* :c) "Lisp primarily" "optimized C"))
	(cerror "Use C and Lisp instead of the DSP"
		"CLM's DSP library cannot handle ~S in ~S" (if (listp x) (car x) x) x)))
  (values nil t))			;second arg = t causes code walker to abandon ship


(defvar array-type nil)

(defun package-op (dsp-name var x &optional (presumptive-type nil) (save-var t) (issue-check nil) (check-id 2))
  (let ((args nil))  
    (if var
	(let ((happy presumptive-type))
	  (if save-var (push var new-sig))
	  (setf array-type presumptive-type)
	  (loop for i in (cdr x) do
	    (if (and (not (listp i))
		     (symbolp i)
		     (or (not (known-var i))
			 (and happy presumptive-type
			      (not (car-member i local-vars)) ;22-Jun-92
			      (not (car-member i typed-user-var)))))
		(if (not happy)		;this is to keep from declaring the fm term in (oscil osc fm) to be of type 'osc
		    (push i user-var)
		  (push (list i presumptive-type) typed-user-var)))
	    (when (and happy presumptive-type
		       (not (member presumptive-type '(integer real fraction))))
	      (setf happy nil)))))
    (setf args (loop for whatsit in (cdr x) 
		collect (walk-form whatsit nil 'dsp-ize)
		do (setf array-type nil))) ;19-Aug-91
    (if dsp-name
	(if var
	    (push `(,dsp-name ,var ,@args) new-prog)
	  (push `(,dsp-name ,@args) new-prog)))
    (if issue-check (push `(<check> ,(cadr x) ,check-id) new-prog))
    (if dsp-name var args)))

(defun package-args (var arg-form) (package-op nil var arg-form))

(defun rem-var (var) 
  (when (not (numberp var))
    (if (not (known-var var)) (push var new-sig))
    var))

(defun chkv (var typ &optional (el-typ nil))
  (if (not (eq typ 'array)) (setf array-type (or el-typ typ)))
  (when (not (listp var))
    (if (not (member var user-var)) (push var user-var))
    (if (not (car-member var typed-user-var))
	(if el-typ
	    (push (list var typ el-typ) typed-user-var)
	  (push (list var typ) typed-user-var)))))

(defun assure-type (x typ)
  (let ((name (second x)))
    (if (and name (not (listp name)) (symbolp name))
	(let* ((current-data (find name typed-user-var :key #'first))
	       (current-type (and current-data (second current-data))))
	  (if (and current-data
		   (not (eq current-type typ)))
	      (setf (second current-data) typ))))))

(defun find-loop-control-info (var x)
  (declare (ignore var))
  
  ;; We save the name of the loop control variable and check it against the OUTA or OUTB call, if any.

  (if (not loop-end)
      (do* ((i 1 (1+ i))
	    (loop-e nil)
	    (loop-b 0)
	    (loop-v nil)
	    (loop-s 1)			;loop step amount -- currently ignored
	    (all-done nil)
	    (val (nth i x) (nth i x)))
	  ((or all-done
	       (= i (length x)))
	   (if (not loop-e)
	       (error "Can't find end point of outer loop")
	     (progn 
	       (setf loop-end loop-e)
	       (setf loop-beg loop-b)
	       (setf loop-var loop-v)
	       (push (list loop-v 'long-int) typed-user-var)
	       `(progn ,@(nthcdr i x)))))
	(if (symbolp val)
	    (if (or (string= val "DO")
		    (string= val "AND"))
		(setf all-done (string= val "DO"))
	      (if (or (string= val "FOR")
		      (string= val "AS"))
		  (setf loop-v (nth (+ i 1) x))
		(if (or (string= val "FROM")
			(string= val "UPFROM")
			(string= val "="))
		    (setf loop-b (nth (+ i 1) x))
		  (if (or (string= val "TO")
			  (string= val "UPTO"))
		      (setf loop-e (nth (+ i 1) x))
		    (if (string= val "BY")
			(setf loop-s (nth (+ i 1) x))
		      (if (string= val "BELOW")
			  (setf loop-e (list '- (nth (+ i 1) x) 1))))))))))
    (progn
      ;; the problem here is that we might have nested loops within the outer loop, and all of
      ;; them have the identical label names LOOP::NEXT-LOOP and LOOP::END-LOOP.  It is only
      ;; at this level that we have block structure info, so we can't put off the label disambiguation
      ;; until later -- END-LOOPs are all at the end, but references within the I-code triples
      ;; to END-LOOP have no way of telling which END-LOOP.  So we expand the loop block, then
      ;; run through it, making sure all occurences of END-LOOP and NEXT-LOOP are fixed up so
      ;; no confusion can happen later.  The next problem is that in ACL 4.1, which has loop built-in,
      ;; these labels have different names (EXCL::END-LOOP).
      (push (gentemp "L_") loop-label-stack)
      (setf last-var (walk-form (macroexpand x) nil 'dsp-ize))
      (pop loop-label-stack)
      (if last-var (rem-var last-var)))))
	  

;;; dealing with DO and DO* will be harder...


(defun push-args (tlabel flabel jump)
  (push tlabel t-lab-stack)
  (push flabel f-lab-stack)
  (push jump j-lab-stack))

(defun pop-args ()
  (pop t-lab-stack)
  (pop f-lab-stack)
  (pop j-lab-stack))

(defun current-t-label () (car t-lab-stack))
(defun current-f-label () (car f-lab-stack))
(defun current-j-label () (car j-lab-stack))
(defun nojump () (eq (car j-lab-stack) 'true))

(defun save-bool-var (var)
  (rem-var var)
  (if (not (car-member var typed-user-var)) 
      (push (list var 'real) typed-user-var))) ;was 'integer

(defun bool-var (var val1 jtrue &optional (f-lab nil))
  (let ((ok (new-label)))
    (save-bool-var var)
    (push `(<setf> ,var ,val1) new-prog)
    (push `(,jtrue ,ok) new-prog)
    (if f-lab (push `(<local-label> ,f-lab) new-prog))
    (push `(<setf> ,var ,(if (= val1 <nil>) <t> <nil>)) new-prog)
    (push `(<local-label> ,ok) new-prog)
    var))


(defun package-bool-op (dsp-name var x &optional (presumptive-type nil)
						 (jfalse '<jump-false>)
						 (jtrue '<jump-true>))
  (package-op dsp-name var x presumptive-type nil)
  (if t-lab-stack
      (progn
	(if (nojump) 
	    (push `(,jfalse ,(current-f-label)) new-prog)
	  (push `(,jtrue ,(current-t-label)) new-prog))
	nil)
    (bool-var var <t> jtrue)))


(defun package-rel-op (var x jfalse jtrue &optional dont-need-C)
  (let ((opl (new-signal))
	(opr (new-signal))
	(t-lab nil)
	(first-time t)
	(f-lab (or (current-f-label) (new-label))))
    (if (listp (cadr x))
	(progn
	  (push `(<setf> ,opl ,(walk-form (cadr x) nil 'dsp-ize)) new-prog)
	  (push opl new-sig))
      (setf opl (local-var-alias (cadr x))))
    (if (and (not (constantp opl))
	     (not (known-var opl)))
	(push opl user-var))
    (loop for i in (cddr x) do
      (if (not first-time)
	  (progn
	    (push `(,jfalse ,f-lab) new-prog)
	    (push `(<local-label> ,t-lab) new-prog)
	    (setf opl opr)
	    (setf opr (new-signal))))
      (setf first-time nil)
      (setf t-lab (new-label))
      (push-args t-lab f-lab 'true)
      (if (listp i)
	  (progn
	    (push `(<setf> ,opr ,(walk-form i nil 'dsp-ize)) new-prog)
	    (push opr new-sig))
	(setf opr (local-var-alias i)))
      (if (and (not (constantp opr))
	       (not (known-var opr)))
	  (push opr user-var))
      (push `(<compare> ,opl ,opr ,dont-need-C) new-prog)
      (pop-args))
    (if t-lab-stack
	(progn
	  (if (nojump)
	      (push `(,jfalse ,f-lab) new-prog)
	    (push `(,jtrue ,(current-t-label)) new-prog))
	  nil)
      (bool-var var <t> jtrue f-lab))))

(defvar save-full-name nil)

(defun package-address (name var x &optional (typ 'real) (a-typ 'real) (el-typ nil) user-def)
  ;; typ=field type, a-typ=structure type, el-typ=element type of array within structure
  (setf array-type (or el-typ a-typ))
  (let ((old-ref setf-structure-ref))
    (setf setf-structure-ref nil)
    (let ((nvar (walk-form (cadr x) nil 'dsp-ize)))
      (setf setf-structure-ref old-ref)
      (if (not user-def)
	  (push `(,name ,var ,nvar ,typ ,setf-structure-ref) new-prog)
	;; here we have to figure out the on-chip offset for dsp-address
	;; user-def is the field descriptor
	(if (eq *clm-language* :56k)
	    (let ((offset #+56-mus-gens (user-struct-offset_56 (fourth user-def) (gethash (third user-def) user-structs))
			  #-56-mus-gens nil
			  ))
	      (push `(<dsp-address_56> ,var ,nvar L ,offset ,typ 1 ,setf-structure-ref) new-prog))
	  (let ((offset (user-struct-offset_c (fourth user-def) (gethash (third user-def) user-structs))))
	    (push `(package-c-address ,var ,nvar ,name
				      ,(if (member typ '(real fraction)) :float
					 (if (member typ '(integer long-int)) :int
					   (if (member typ pure-float-structs) :float*
					     :int*)))
				      ,(if (or (member typ '(real fraction)) (member typ pure-float-structs)) 0 (second offset))
				      ,setf-structure-ref
				      ,(if (or (member typ '(real fraction)) (member typ pure-float-structs)) (first offset)))
		  new-prog))))
      (if (member typ '(table tbl x-table y-table)) (push var tables))
      (if setf-structure-ref (push (list var typ a-typ el-typ) addresses))
      (if (not (member nvar new-sig)) (chkv nvar a-typ el-typ))
      (if (or save-full-name el-typ a-typ) (push (list var x) full-array-names))
      (when (and (not (eq a-typ 'real)) 
		 (eq *clm-language* :c) 
		 (not (car-member var typed-user-var))) 
	    (pushnew var new-sig)
	    (push (list var typ) typed-user-var))
      (rem-var var))))

(defun fpackage-op (dsp-name var x presumptive-type &optional (c-type 'io))
  (setf save-full-name t)
  ;; this saves the original stream name on full-array-names for outsig-n in code56.lisp (where we need the original name)
  ;; the actual name is saved during the package-op walk-form (I think)
  (let ((val (package-op dsp-name var x presumptive-type t nil 2)))
    (when (eq *clm-language* :c) 
      (push (list var c-type) typed-user-sig)
      (if (eq c-type 'fraction) (push (list (new-signal) 'io var) typed-user-sig)))
    (setf save-full-name nil)
    val))
	
(defun get-array-type (x)
  (loop for i in typed-user-var if (eq x (car i)) return (third i)))

(defun table-p (x)
  (or (member x tables)
      (and (member (car-member x typed-user-var) '(table x-table y-table)))))


(defun dsp-ize (x y env)		;this is the heart of Run -- the tree walker calls this on nearly every branch or node
  (declare (ignore y env))		;env is the local environment that the tree walker keeps track of
  (if (and x (listp x))
      (let ((var (new-signal)))
	(if (dsp-function (car x))
	    (apply (associated-function (car x)) (list var x))
	  (if (and (listp (car x)) (eq (caar x) 'LAMBDA))
	      (lambda-branch var x)
	    (let ((xx (macroexpand x outer-environment)))
	      (if (eq x xx) 
		  (give-up x) 
		xx)))))
    (if (symbolp x)
	(if (constantp x)
	    (if (eq x nil) 
		<nil>
	      (if (eq x t)
		  <t>
		(eval x)))
	  (if (not (known-var x))
	      (progn 
		(push x user-var)
		x)
	    (or (car-member x local-vars)
		x)))
      x)))

;;; return has to thread its way through block lists and get its value returned
;;; from the correct place, so it's not enough to just dump the value in A and
;;; jump to the end.  The return stack values is a list of lists, pushed anew
;;; as blocks are opened, and popped as they close.  The inner list is of the 
;;; form (block-name block-variable block-end-label) -- the block-variable is
;;; what we return (either via return or by falling through the bottom), block-label
;;; is the location of the end of the block, and the block-name is either nil
;;; for an unnamed block or the name provided by the caller.

(defun Return-branch (var x)		;return [result]
  (declare (ignore var))
  (if return-stack
      (let ((our-block-data (find nil return-stack :key #'first)))
	(if (not our-block-data) 
	    (error "RETURN not inside a block named NIL")
	  (let ((block-lab (third our-block-data))
		(block-var (second our-block-data)))
	    (if (cdr x) 
		(push `(<setf> ,block-var ,(walk-form (cadr x) nil 'dsp-ize)) new-prog)
	      (push `(<setf> ,block-var ,<nil>) new-prog))
	    (push `(<jump> ,block-lab) new-prog)
	    block-var)))
    (error "RETURN outside any block")))

(defun Return-from-branch (var x)
  (declare (ignore var))
  (if return-stack
      (let ((our-block-data (find (cadr x) return-stack :key #'first)))
	(if (not our-block-data) 
	    (error "RETURN-FROM ~S not inside the specified block" (cadr x))
	  (let ((block-lab (third our-block-data))
		(block-var (second our-block-data)))
	    (if (cddr x) 
		(push `(<setf> ,block-var ,(walk-form (caddr x) nil 'dsp-ize)) new-prog)
	      (push `(<setf> ,block-var ,<nil>) new-prog))
	    (push `(<jump> ,block-lab) new-prog)
	    block-var)))
    (error "RETURN-FROM outside any block")))

(defun Block-branch (var x)
  (let ((end-label (new-label))
	(old-return-stack return-stack))
    (push var new-sig)
    (push (list (cadr x) var end-label) return-stack)
    (loop for i in (cddr x) do (setf last-var (walk-form i nil 'dsp-ize)))
    (if last-var (push `(<setf> ,var ,last-var) new-prog))
    (push `(<local-label> ,end-label) new-prog)
    (setf return-stack old-return-stack)
    (if last-var (rem-var last-var))
    var))

(defun Tagbody-branch (var x)
  ;; need check for duplicate tag within one scope, and shadowing
  (declare (ignore var))
  (let ((inner-labs nil))
    (loop for i in (cdr x) do
      (if (listp i) 
	  (setf last-var (walk-form i nil 'dsp-ize))
	(let ((lab (if (and loop-label-stack
			    #-(or mcl (and excl cltl2)) (member i '(LOOP::NEXT-LOOP LOOP::END-LOOP))
			    #+(and excl cltl2) (member i '(EXCL::NEXT-LOOP EXCL::END-LOOP))
			    #+mcl (member i '(ANSI-LOOP::NEXT-LOOP ANSI-LOOP::END-LOOP))
			    )
		       (intern (concatenate 'string (symbol-name i) "-" (symbol-name (first loop-label-stack))))
		     i)))
	  (push lab inner-labs)
	  (push `(<label> ,lab) new-prog))))
    (if inner-labs (push `(<hide> ,@inner-labs) new-prog)))
  (if last-var (rem-var last-var)))

(defun Or-branch (var x)
  (let ((f-lab nil)
	(old-f-lab (current-f-label))
	(opl nil)
	(old-j-lab (current-j-label))
	(j-lab 'false)
	(t-lab (or (current-t-label) (new-label))))
    (do* ((fr (cdr x) (cdr fr))
	  (i (car fr) (car fr)))
	((null i))
      (if (or (cdr fr) (null j-lab-stack)) 
	  (progn
	    (setf j-lab 'false)
	    (setf f-lab (new-label)))
	(progn
	  (setf f-lab old-f-lab)
	  (setf j-lab old-j-lab)))
      (push-args t-lab f-lab j-lab)
      (if (and (listp i) (boole-op (car i)))
	  (walk-form i nil 'dsp-ize)
	(progn
	  (if (listp i)
	      (progn
		(setf opl (new-signal))
		(push `(<setf> ,opl ,(walk-form i nil 'dsp-ize)) new-prog)
		(save-bool-var opl))
	    (setf opl (walk-form i nil 'dsp-ize)))
	  (push `(<not-null> ,opl) new-prog)
	  (if (or (cdr fr) (null t-lab-stack))
	      (push `(<jump-true> ,t-lab) new-prog)
	    (if t-lab-stack
		(if (nojump)
		    (push `(<jump-false> ,(current-f-label)) new-prog)
		  (push `(<jump-true> ,(current-t-label)) new-prog))))))
      (if (or (cdr fr) (null t-lab-stack))
	  (push `(<local-label> ,f-lab) new-prog))
      (pop-args))
    (if (null t-lab-stack)
	(bool-var var <nil> '<jump> t-lab))))

(defun And-branch (var x)
  (let ((t-lab nil)
	(old-t-lab (current-t-label))
	(opl nil)
	(old-j-lab (current-j-label))
	(j-lab 'true)
	(f-lab (or (current-f-label) (new-label))))
    (do* ((fr (cdr x) (cdr fr))
	  (i (car fr) (car fr)))
	((null i))
      (if (or (cdr fr) (null j-lab-stack)) 
	  (progn
	    (setf j-lab 'true)
	    (setf t-lab (new-label)))
	(progn
	  (setf t-lab old-t-lab)
	  (setf j-lab old-j-lab)))
      (push-args t-lab f-lab j-lab)
      (if (and (listp i) (boole-op (car i)))
	  (walk-form i nil 'dsp-ize)
	(progn
	  (if (listp i)
	      (progn
		(setf opl (new-signal))
		(push `(<setf> ,opl ,(walk-form i nil 'dsp-ize)) new-prog)
		(save-bool-var opl))
	    (setf opl (walk-form i nil 'dsp-ize)))
	  (push `(<not-null> ,opl) new-prog)
	  (if (or (cdr fr) (null t-lab-stack))
	      (push `(<jump-false> ,f-lab) new-prog)
	    (if t-lab-stack
		(if (nojump)
		    (push `(<jump-false> ,(current-f-label)) new-prog)
		  (push `(<jump-true> ,(current-t-label)) new-prog))))))
      (if (or (cdr fr) (null t-lab-stack))
	  (push `(<local-label> ,t-lab) new-prog))
      (pop-args))
    (if (null t-lab-stack)
	(bool-var var <t> '<jump> f-lab))))

(defun If-branch (var x)
  (let ((tlabel (new-label))
	(flabel (new-label))
	(end-label (new-label)))
    (push-args tlabel flabel 'true)
    (if (and (listp (cadr x)) (boole-op (caadr x)))
	(walk-form (cadr x) nil 'dsp-ize)
      (progn
	(push `(<not-null> ,(walk-form (cadr x) nil 'dsp-ize)) new-prog)
	(push `(<jump-false> ,flabel) new-prog)))
    (pop-args)
    (save-bool-var var)
    (push `(<local-label> ,tlabel) new-prog)
    (if (caddr x)
	(push `(<setf> ,var ,(walk-form (caddr x) nil 'dsp-ize)) new-prog)
      (push `(<setf> ,var ,<t>) new-prog))
    (push `(<jump> ,end-label) new-prog)
    (push `(<local-label> ,flabel) new-prog)
    (if (cadddr x) 
	(push `(<setf> ,var ,(walk-form (cadddr x) nil 'dsp-ize)) new-prog)
      (push `(<setf> ,var ,<nil>) new-prog))
    (push `(<local-label> ,end-label) new-prog)
    var))

(defun When-branch (var x)
  (let ((tlabel (new-label))
	(flabel (new-label))
	(end-label (new-label)))
    (save-bool-var var)
    (push-args tlabel flabel 'true)
    (if (and (listp (cadr x)) (boole-op (caadr x)))
	(walk-form (cadr x) nil 'dsp-ize)
      (progn
	(push `(<not-null> ,(walk-form (cadr x) nil 'dsp-ize)) new-prog)
	(push `(<jump-false> ,flabel) new-prog)))
    (pop-args)
    (push `(<local-label> ,tlabel) new-prog)
    (loop for i in (cddr x) do (setf last-var (walk-form i nil 'dsp-ize)))
    (push `(<setf> ,var ,last-var) new-prog)
    (push `(<jump> ,end-label) new-prog)
    (push `(<local-label> ,flabel) new-prog)
    (push `(<setf> ,var ,<nil>) new-prog)
    (push `(<local-label> ,end-label) new-prog)
    var))

(defun Unless-branch (var x)
  (let ((flabel (new-label))
	(tlabel (new-label))
	(end-label (new-label)))
    (push-args tlabel flabel 'false)
    (if (and (listp (cadr x)) (boole-op (caadr x)))
	(walk-form (cadr x) nil 'dsp-ize)
      (progn
	(push `(<not-null> ,(walk-form (cadr x) nil 'dsp-ize)) new-prog)
	(push `(<jump-true> ,tlabel) new-prog)))
    (pop-args)
    (save-bool-var var)
    (push `(<local-label> ,flabel) new-prog)
    (loop for i in (cddr x) do (setf last-var (walk-form i nil 'dsp-ize)))
    (push `(<setf> ,var ,last-var) new-prog)
    (push `(<jump> ,end-label) new-prog)
    (push `(<local-label> ,tlabel) new-prog)
    (push `(<setf> ,var ,<nil>) new-prog)
    (push `(<local-label> ,end-label) new-prog)
    var))

(defun Cond-branch (var x)
  (let ((tlabel nil)
	(flabel nil)
	(end-label (new-label)))
    (loop for i in (cdr x) do
      (setf tlabel (new-label))
      (setf flabel (new-label))
      (push-args tlabel flabel 'true)
      (if (and (listp (car i)) (boole-op (caar i)))
	  (walk-form (car i) nil 'dsp-ize)
	(progn
	  (push `(<not-null> ,(walk-form (car i) nil 'dsp-ize)) new-prog)
	  (push `(<jump-false> ,flabel) new-prog)))
      (pop-args)
      (push `(<local-label> ,tlabel) new-prog)
      (if (cdr i)
	  (loop for j in (cdr i) do
	    (setf last-var (walk-form j nil 'dsp-ize)))
	(setf last-var '<nil>))
      (push `(<setf> ,var ,last-var) new-prog)
      (push `(<jump> ,end-label) new-prog)
      (push `(<local-label> ,flabel) new-prog))
    (push `(<local-label> ,end-label) new-prog))

  (when (not (known-var var)) 
    (push var new-sig)
    (push (list var 'real) typed-user-var))
  var)


(defun Case-branch (var x)
  (declare (ignore var))
  ;; CASE is of this form:
  ;;     eval index expr => t
  ;;     goto test
  ;; L1: code for S1
  ;;     goto next
  ;; L2 ...
  ;; ...
  ;; test: case V1 L1 (V1=index of that branch)
  ;;       case V2 L2 (i.e. if t=V2 goto L2 etc)
  ;; ...
  ;;     goto Ln (i.e. case-else branch, if any)
  ;; next: 
  
  (let ((test-lab (new-label))
	(next-lab (new-label))
	(nvar (new-signal))
	(selectors nil)
	(labels nil)
	(v-lab nil)
	(var1 (walk-form (cadr x) nil 'dsp-ize)))
    ;; can this return an integer constant? (case 1 ...)
    (if (and (not (member var1 new-sig))
	     (not (car-member var1 local-vars))
	     (not (car-member var1 typed-user-var))) 
	(push (list var1 'integer) typed-user-var))
    (push `(<jump> ,test-lab) new-prog)
    (loop for i in (cddr x) do
      (setf v-lab (new-label))
      (push `(<label> ,v-lab) new-prog)
      (push (local-var-alias (car i)) selectors)
      (push v-lab labels)
      (setf last-var '<nil>)
      (loop for j in (cdr i) do
	(setf last-var (walk-form j nil 'dsp-ize)))
      (if last-var (push `(<setf> ,nvar ,last-var) new-prog))
      (push `(<jump> ,next-lab) new-prog))
    (push `(<local-label> ,test-lab) new-prog)
    (push `(<case> ,var1 ,(nreverse selectors) ,(nreverse labels)) new-prog)
    (push `(<setf> ,nvar ,<nil>) new-prog)
    (push `(<local-label> ,next-lab) new-prog)
    (rem-var nvar)))

(defun new-local-signal (var)
  (intern (concatenate 'string (symbol-name (new-signal)) "[" (symbol-name var) "]")))

(defun make-local-var (var &optional (type 'real))
  (let ((sig (new-local-signal var)))
    (push (list sig type) typed-user-sig)
    (push (list var sig type) local-vars)
    sig))

(defun make-parallel-local-var (var &optional (type 'real))
  (let ((sig (new-local-signal var)))
    (push (list sig type) typed-user-sig)
    (push (list var sig type) parallel-local-vars)
    sig))

(defun merge-locals-lists (locals) 
  (when (plusp locals) 
    (if (<= locals (length parallel-local-vars))
	(loop for i from 0 below locals do
	  (push (pop parallel-local-vars) local-vars))
      (warn "error in local variable handling: ~D is not <= (length '~A)" locals parallel-local-vars))))

(defun unmake-local-vars (num)
  (let ((all-gone (loop for i from 0 below num collect (second (pop local-vars)))))
    (if all-gone
	(push `(<undefine> ',all-gone) new-prog))))

(defun Dotimes-branch (var x)
  (let ((cmp-lab (new-label))
	(res-lab (new-label))
	(body-lab (new-label))
	(end-lab (new-label))
	(var1 (new-signal))
	(end-var (new-signal))
	(inner-labs nil)
	(endnum (second (second x)))
	(do-var (make-local-var (first (second x)) 'integer)))
    (push (list nil end-var end-lab) return-stack)
    (when (and (symbolp endnum)
	       (or (not local-vars) (not (car-member endnum local-vars)))
	       (not (car-member endnum typed-user-var)))
      (push (list endnum 'integer) typed-user-var))
    (push `(<setf> ,do-var 0) new-prog)
    (push `(<setf> ,var ,(walk-form endnum nil 'dsp-ize)) new-prog)
    (if (eq *clm-language* :c) (push (list var 'integer) typed-user-sig))
    (push `(<label> ,cmp-lab) new-prog)
    (push `(<compare> ,var ,do-var) new-prog)
    (push `(<jump-g> ,body-lab) new-prog) ;i.e. if leq drop through into result block
    (push `(<label> ,res-lab) new-prog)
    (if (third (second x)) 
	(push `(<setf> ,end-var ,(walk-form (third (second x)) nil 'dsp-ize)) new-prog)
      (push `(<setf> ,end-var ,<nil>) new-prog))
    (push `(<jump> ,end-lab) new-prog)
    (push `(<label> ,body-lab) new-prog)
    (let ((old-return-stack return-stack))
      (push (list (cadr x) end-lab) return-stack)
      (loop for i in (cddr x) do
	(if (listp i) 
	    (walk-form i nil 'dsp-ize)
	  (progn
	    (push i inner-labs)
	    (push `(<label> ,i) new-prog))))
      (push `(<incf> ,var1 ,do-var) new-prog)
      (push var1 new-sig)
      (push `(<jump> ,cmp-lab) new-prog)
      (setf return-stack old-return-stack))
    (push `(<label> ,end-lab) new-prog)
    (if inner-labs (push `(<hide> ,@inner-labs) new-prog))
    (pop local-vars)			;delete DO counter variable
    (push `(<hide> ,cmp-lab ,res-lab ,body-lab ,end-lab) new-prog)
    (rem-var end-var)))

(defun Do-branch (var x starred)
  (declare (ignore var))
  (let* ((body-lab (new-label))		;start of DO body (i.e. tagbody)
	 (test-lab (new-label))		;start of end test expr
	 (res-lab (new-label))		;result expr
	 (step-lab nil)			;start of stepping exprs (are threaded through init code) (keep original intact)
	 (next-step-lab (new-label))
	 (next-init-lab nil)		;these two are for threading through initialization and stepping segments
	 (end-lab (new-label))		;end of DO statement (for RETURN, etc)
	 (end-var (new-signal))		;result (if any) of DO
	 (lvars (cadr x))		;list of (var init step) lists
	 (end-test (caaddr x))		;car of caddr I hope
	 (result (cadr (caddr x)))	;cadr of caddr?
	 (body (cdddr x))
	 (locals (length lvars))
	 (inner-labs nil))		;keep track of labels created on the fly
    (let ((need-next-init nil)
	  (need-next-step nil))
      ;; variables (list of (var init step) lists)
      (loop for j on lvars do
	(let* ((i (car j))
	       (var-i (if starred 
			  (make-local-var (car i))
			(let ((good-grief (make-parallel-local-var (car i))))
			  (push (list (car i) good-grief) local-vars)
			  good-grief))))
	  ;; I knew there was something really peculiar about DO -- the local variable is not declared
	  ;; within the block of variables, yet it is declared within the step statement in its declaration!!!
	  ;; No wonder I can't remember how DO works -- it's a semantic nightmare (and not the same as LET).
	  (when next-init-lab 
	    (push next-init-lab inner-labs)
	    (push `(<label> ,next-init-lab) new-prog))
	  (setf next-init-lab (new-label))
	  (setf need-next-init nil)
	  (if (cdr i)
	      (push `(<setf> ,var-i ,(walk-form (cadr i) nil 'dsp-ize)) new-prog)
	    (push `(<setf> ,var-i ,<nil>) new-prog))
	  (when (cddr i)		;is there a step expr?
	    (push `(<jump> ,next-init-lab) new-prog)
	    (setf need-next-init t)
	    (if (null step-lab) (setf step-lab next-step-lab))
	    (push `(<label> ,next-step-lab) new-prog)
	    (push next-step-lab inner-labs)
	    (setf next-step-lab (new-label))
	    (setf need-next-step nil)
	    (push `(<setf> ,var-i ,(walk-form (caddr i) nil 'dsp-ize)) new-prog)
	    (when (cdr j) 
	      (setf need-next-step t)
	      (push `(<jump> ,next-step-lab) new-prog)))
	  (if (not starred) (pop local-vars))))
      (if (not starred) (merge-locals-lists locals))
      (if (null step-lab)		;no step exprs found
	  (progn
	    (setf step-lab (new-label))
	    (push step-lab inner-labs)
	    (push `(<label> ,step-lab) new-prog))
	(when need-next-step 
	  (push next-step-lab inner-labs)
	  (push `(<label> ,next-step-lab) new-prog)))
      (when need-next-init 
	(push next-init-lab inner-labs)
	(push `(<label> ,next-init-lab) new-prog))
      ;; end test
      (push `(<label> ,test-lab) new-prog)
      (if end-test
	  (progn
	    (push-args res-lab body-lab 'true)
	    (if (and (listp end-test) (boole-op (car end-test)))
		(walk-form end-test nil 'dsp-ize)
	      (progn
		(push `(<not-null> ,(walk-form end-test nil 'dsp-ize)) new-prog)
		(push `(<jump-false> ,body-lab) new-prog)))
	    (pop-args))
	(if result (push `(<jump> ,body-lab) new-prog)))
      ;; result
      (push `(<label> ,res-lab) new-prog)
      (if result
	  (push `(<setf> ,end-var ,(walk-form result nil 'dsp-ize)) new-prog)
	(push `(<setf> ,end-var ,<nil>) new-prog))
      (push `(<jump> ,end-lab) new-prog)
      ;; DO body (a tagbody)
      (push `(<label> ,body-lab) new-prog)
      (push (list nil end-var end-lab) return-stack)
      (loop for i in body do
	(if (listp i) 
	    (walk-form i nil 'dsp-ize)
	  (progn
	    (push i inner-labs)
	    (push `(<label> ,i) new-prog))))
      (pop return-stack)
      (push `(<jump> ,step-lab) new-prog)
      (push `(<label> ,end-lab) new-prog)
      (if inner-labs (push `(<hide> ,@inner-labs) new-prog))
      (unmake-local-vars locals)
      (push `(<hide> ,res-lab ,end-lab ,body-lab ,test-lab) new-prog)
      (rem-var end-var))))

(defun structure-reference (n)
  (or (member (first n) '(<aref> <setf-tref>))
      (find (second n) addresses :key #'first)))

(defun setf-branch (var x)
  (declare (ignore var))
  (let ((last-svar nil))
    (setf array-type nil)
    (if (evenp (length x)) (error "~D args to ~A?" (1- (length x)) (car x)))
    (loop for svar in (cdr x) by #'cddr and
              sval in (cddr x) by #'cddr do
      (when (listp svar)		;i.e. structure or array ref
	(setf setf-structure-ref t)
	(let ((lvar (walk-form svar nil 'dsp-ize)))
	  (if (not (structure-reference (first new-prog)))
	      (warn "odd location to setf: ~A in ~A" svar x))
	  (setf svar lvar))
	(setf setf-structure-ref nil))
      
      (push `(<setf> ,(local-var-alias svar) ,(setf last-var (walk-form sval nil 'dsp-ize))) new-prog)
      ;; has to be done first for cases like (setf val (+ 1 val)) when val not yet seen
      (if (and (not (member svar true-user-var))
	       (not (car-member svar local-vars))
	       (not (member svar user-var)))
	  (push svar setf-first-var))
      (setf last-svar svar)
      (if (not (car-member svar local-vars)) (pushnew svar true-user-var)))
    (if last-var (rem-var last-var))
    last-svar))

(defun aref-branch (var x)
  ;; here we try to catch 'table or 'tbl (they are compatible) as a type, and push <tref> rather than <aref>
  ;; (tbl has an internal index it keeps track of whereas table is just an array of fractional numbers)
  ;; the array type actually matters (off by 1 indexing for dlys and so on) -- this part of this code is a mess.
  (let ((op '<aref>)
	(its-a-table nil)
	(arr (cadr x))
	(typ 'array)
	(el-typ nil))

    (if (listp arr)			;it's an array or table within a structure (we hope)
	(let* ((addr (walk-form (cadr x) nil 'dsp-ize)))
	  (setf its-a-table (table-p addr))
	  (setf arr addr))
					; this used to be (if its-a-table (setf arr addr))
      (setf its-a-table (table-p arr)))
    
    (if setf-structure-ref (push (list var typ el-typ) addresses))
    (if its-a-table 
	(progn
	  (setf op (if setf-structure-ref '<setf-tref> '<tref>))
	  (setf typ 'table)
	  (setf el-typ 'fraction))
      (setf el-typ (or el-typ (get-array-type arr) array-type 'real)))
    ;; need to save el-typ before going off to walk the indices
    
    (if (not (listp (cadr x))) (chkv arr typ el-typ))
    (if (and (= (length x) 3) 
	     (not (listp (third x))) 
	     (not (numberp (third x))) 
	     (not (car-member (third x) typed-user-var))
	     (not (car-member (third x) local-vars)))
	(push (list (third x) 'integer) typed-user-var))
    
    (let ((old-ref setf-structure-ref))
      (setf setf-structure-ref nil)
      (let ((indices (loop for form in (cddr x) by #'cdr 
			   collect (walk-form form nil 'dsp-ize))))
	;; here we should save the array-type, and then cancel that type in the index collecting loop
	;; then restore the original type (e.g. array of zdelays indexed through an integer array)
	
	(push `(,op 
		,var 
		,(local-var-alias arr)
		,el-typ
		,@indices)
	      new-prog))
      (setf setf-structure-ref old-ref))

    (when (and (eq *clm-language* :c) (not (car-member var typed-user-var)))
      (push (list var el-typ) typed-user-var)
      (pushnew var new-sig))
      
    (setf array-type nil)
    (rem-var var)))

(defun let-branch (var x starred)
  (declare (ignore var))
  (let ((locals (length (cadr x))))
    (when (cadr x)			;(LET NIL ...) ?
      (loop for svar in (cadr x) by #'cdr do
	;; the following is legal lisp: (let* ((hi 3) (hi (+ hi 1))) ...) so we have to walk the expression first
	(if (listp svar)		;(LET ((A 1) ...))
	    (let* ((form (walk-form (cadr svar) nil 'dsp-ize))
		   (sig (if starred
			   (make-local-var (car svar))
			 (make-parallel-local-var (car svar)))))
	      (push `(<setf> ,sig ,form) new-prog))
	  (if starred
	      (make-local-var svar)
	    (make-parallel-local-var svar))))) ;(LET (A B C) ...)
    (if (not starred) (merge-locals-lists locals))
    (loop for i in (cddr x) do (setf last-var (walk-form i nil 'dsp-ize)))
    (unmake-local-vars locals)
    (if last-var (rem-var last-var))))

(defun lambda-branch (var x)
  ;; x is the outer (enclosing) list ((LAMBDA (args) body) args)
  (declare (ignore var))
  (let* ((passed-args (cdr x))
	 (lambda-args (second (first x)))
	 (lambda-body (cddr (first x))))
    `(let (,@(loop for xx in lambda-args and yy in passed-args collect (list xx yy)))
       ,@lambda-body)))
    


(def-dsp-fun 'SW-CURRENT-VALUE #'(lambda (var x) (package-address '<SW-CURRENT-VALUE> var x 'real 'sw)))
(def-dsp-fun 'SW-PHASE         #'(lambda (var x) (package-address '<SW-PHASE> var x 'real 'sw)))
(def-dsp-fun 'SW-FREQ          #'(lambda (var x) (package-address '<SW-FREQ> var x 'real 'sw)))
(def-dsp-fun 'SW-BASE          #'(lambda (var x) (package-address '<SW-BASE> var x 'real 'sw)))

(def-dsp-fun 'OSC-PHASE        #'(lambda (var x) (package-address '<OSC-PHASE> var x 'real 'osc)))
(def-dsp-fun 'OSC-FREQ         #'(lambda (var x) (package-address '<OSC-FREQ> var x 'real 'osc)))

(def-dsp-fun 'NOI-PHASE        #'(lambda (var x) (package-address '<NOI-PHASE> var x 'real 'noi)))
(def-dsp-fun 'NOI-FREQ         #'(lambda (var x) (package-address '<NOI-FREQ> var x 'real 'noi)))
(def-dsp-fun 'NOI-OUTPUT       #'(lambda (var x) (package-address '<NOI-OUTPUT> var x 'real 'noi)))
(def-dsp-fun 'NOI-BASE         #'(lambda (var x) (package-address '<NOI-BASE> var x 'real 'noi)))
(def-dsp-fun 'NOI-INCR         #'(lambda (var x) (package-address '<NOI-INCR> var x 'fraction 'noi)))

(def-dsp-fun 'DLY-SIZE         #'(lambda (var x) (package-address '<DLY-SIZE> var x 'integer 'dly)))
(def-dsp-fun 'DLY-PLINE        #'(lambda (var x) (package-address '<DLY-PLINE> var x 'integer 'dly 'fraction)))
(def-dsp-fun 'DLY-LOC          #'(lambda (var x) (package-address '<DLY-LOC> var x 'integer 'dly)))

(def-dsp-fun 'TBL-PHASE        #'(lambda (var x) (package-address '<TBL-PHASE> var x 'real 'tbl)))
(def-dsp-fun 'TBL-FREQ         #'(lambda (var x) (package-address '<TBL-FREQ> var x 'real 'tbl)))
(def-dsp-fun 'TBL-TABLE        #'(lambda (var x) (package-address '<TBL-TABLE> var x 'integer 'tbl 'fraction)))
(def-dsp-fun 'TBL-TABLE-SIZE   #'(lambda (var x) (package-address '<TBL-TABLE-SIZE> var x 'integer 'tbl)))
(def-dsp-fun 'TBL-INTERNAL-MAG #'(lambda (var x) (package-address '<TBL-INTERNAL-MAG> var x 'real 'tbl)))

(def-dsp-fun 'CMBFLT-SCALER    #'(lambda (var x) (package-address '<CMBFLT-SCALER> var x 'fraction 'cmbflt)))
(def-dsp-fun 'CMBFLT-DLY-UNIT  #'(lambda (var x) (package-address '<CMBFLT-DLY-UNIT> var x 'dly 'cmbflt)))

(def-dsp-fun 'ALLPASSFLT-FEEDBACK #'(lambda (var x) (package-address '<ALLPASSFLT-FEEDBACK> var x 'fraction 'allpassflt)))
(def-dsp-fun 'ALLPASSFLT-FEEDFORWARD #'(lambda (var x) (package-address '<ALLPASSFLT-FEEDFORWARD> var x 'fraction 'allpassflt)))
(def-dsp-fun 'ALLPASSFLT-DLY-UNIT #'(lambda (var x) (package-address '<ALLPASSFLT-DLY-UNIT> var x 'dly 'allpassflt)))

(def-dsp-fun 'SMPFLT-A0         #'(lambda (var x) (package-address '<SMPFLT-A0> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-A1         #'(lambda (var x) (package-address '<SMPFLT-A1> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-A2         #'(lambda (var x) (package-address '<SMPFLT-A2> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-B1         #'(lambda (var x) (package-address '<SMPFLT-B1> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-B2         #'(lambda (var x) (package-address '<SMPFLT-B2> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-X1         #'(lambda (var x) (package-address '<SMPFLT-X1> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-X2         #'(lambda (var x) (package-address '<SMPFLT-X2> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-Y1         #'(lambda (var x) (package-address '<SMPFLT-Y1> var x 'fraction 'smpflt)))
(def-dsp-fun 'SMPFLT-Y2         #'(lambda (var x) (package-address '<SMPFLT-Y2> var x 'fraction 'smpflt)))

(def-dsp-fun 'FRMNT-G           #'(lambda (var x) (package-address '<FRMNT-G> var x 'real 'frmnt)))
(def-dsp-fun 'FRMNT-TZ          #'(lambda (var x) (package-address '<FRMNT-TZ> var x 'flt-two-zero 'frmnt)))
(def-dsp-fun 'FRMNT-TP          #'(lambda (var x) (package-address '<FRMNT-TP> var x 'flt-two-pole 'frmnt)))

(def-dsp-fun 'RDIN-I            #'(lambda (var x) (package-address '<RDIN-I> var x 'long-int 'rdin)))
(def-dsp-fun 'RDIN-INC          #'(lambda (var x) (package-address '<RDIN-INC> var x 'long-int 'rdin)))
(def-dsp-fun 'RDIN-CHN          #'(lambda (var x) (package-address '<RDIN-CHN> var x 'integer 'rdin)))
(def-dsp-fun 'RDIN-FIL          #'(lambda (var x) (package-address '<RDIN-FIL> var x 'io 'rdin)))

(def-dsp-fun 'WT-WAVE           #'(lambda (var x) (package-address '<WT-WAVE> var x 'table 'wt 'fraction)))
(def-dsp-fun 'WT-WSIZ           #'(lambda (var x) (package-address '<WT-WSIZ> var x 'integer 'wt)))
(def-dsp-fun 'WT-FREQ           #'(lambda (var x) (package-address '<WT-FREQ> var x 'real 'wt)))
(def-dsp-fun 'WT-B              #'(lambda (var x) (package-address '<WT-B> var x 'rblk 'wt 'fraction)))
(def-dsp-fun 'WT-INTERNAL-MAG   #'(lambda (var x) (package-address '<WT-INTERNAL-MAG> var x  'real 'wt)))
(def-dsp-fun 'WT-PHASE          #'(lambda (var x) (package-address '<WT-PHASE> var x 'real 'wt)))

(def-dsp-fun 'RBLK-SIZ          #'(lambda (var x) (package-address '<RBLK-SIZ> var x 'integer 'rblk)))
(def-dsp-fun 'RBLK-BUF          #'(lambda (var x) (package-address '<RBLK-BUF> var x 'table 'rblk 'fraction)))
(def-dsp-fun 'RBLK-LOC          #'(lambda (var x) (package-address '<RBLK-LOC> var x 'integer 'rblk)))
(def-dsp-fun 'RBLK-CTR          #'(lambda (var x) (package-address '<RBLK-CTR> var x 'real 'rblk)))

(def-dsp-fun 'FFTFLT-ENV        #'(lambda (var x) (package-address '<FFTFLT-ENV> var x  'table 'fftflt 'fraction)))
(def-dsp-fun 'FFTFLT-SIZ        #'(lambda (var x) (package-address '<FFTFLT-SIZ> var x 'integer 'fftflt)))
(def-dsp-fun 'FFTFLT-HOP        #'(lambda (var x) (package-address '<FFTFLT-HOP> var x 'integer 'fftflt)))
(def-dsp-fun 'FFTFLT-RD         #'(lambda (var x) (package-address '<FFTFLT-RD> var x 'rdin 'fftflt)))
(def-dsp-fun 'FFTFLT-B          #'(lambda (var x) (package-address '<FFTFLT-B> var x 'rblk 'fftflt)))
(def-dsp-fun 'FFTFLT-DATAR      #'(lambda (var x) (package-address '<FFTFLT-DATAR> var x 'x-table 'fftflt 'fraction)))
(def-dsp-fun 'FFTFLT-DATAI      #'(lambda (var x) (package-address '<FFTFLT-DATAI> var x 'y-table 'fftflt 'fraction)))
(def-dsp-fun 'FFTFLT-HALF-SIZ   #'(lambda (var x) (package-address '<FFTFLT-HALF-SIZ> var x 'integer 'fftflt)))

(def-dsp-fun 'FFT-DATA-REAL     #'(lambda (var x) (package-address '<FFT-DATA-REAL> var x 'x-table 'fft-data 'fraction)))
(def-dsp-fun 'FFT-DATA-IMAGINARY #'(lambda (var x) (package-address '<FFT-DATA-IMAGINARY> var x 'y-table 'fft-data 'fraction)))
(def-dsp-fun 'FFT-DATA-SIZE     #'(lambda (var x) (package-address '<FFT-DATA-SIZE> var x 'integer 'fft-data)))

(def-dsp-fun 'FLT-M             #'(lambda (var x) (package-address '<FLT-M> var x 'integer 'flt)))
(def-dsp-fun 'FLT-TYP           #'(lambda (var x) (package-address '<FLT-TYP> var x 'integer 'flt)))
(def-dsp-fun 'FLT-SO            #'(lambda (var x) (package-address '<FLT-SO> var x 'fraction 'flt)))
(def-dsp-fun 'FLT-A             #'(lambda (var x) (package-address '<FLT-A> var x 'y-table 'flt 'fraction)))
(def-dsp-fun 'FLT-B             #'(lambda (var x) (package-address '<FLT-B> var x 'y-table 'flt 'fraction)))
(def-dsp-fun 'FLT-C             #'(lambda (var x) (package-address '<FLT-C> var x 'x-table 'flt 'fraction)))
(def-dsp-fun 'FLT-D             #'(lambda (var x) (package-address '<FLT-D> var x  'x-table 'flt 'fraction)))

(def-dsp-fun 'SMP-SR            #'(lambda (var x) (package-address '<SMP-SR> var x 'real 'smp)))
(def-dsp-fun 'SMP-LST           #'(lambda (var x) (package-address '<SMP-LST> var x 'fraction 'smp)))
(def-dsp-fun 'SMP-NXT           #'(lambda (var x) (package-address '<SMP-NXT> var x 'fraction 'smp)))
(def-dsp-fun 'SMP-I             #'(lambda (var x) (package-address '<SMP-I> var x 'long-int 'smp)))

(def-dsp-fun 'SR-INCR           #'(lambda (var x) (package-address '<SR-INCR> var x 'real 'sr)))
(def-dsp-fun 'SR-RD             #'(lambda (var x) (package-address '<SR-RD> var x 'rdin 'sr)))
(def-dsp-fun 'SR-X              #'(lambda (var x) (package-address '<SR-X> var x 'real 'sr)))
(def-dsp-fun 'SR-DATA           #'(lambda (var x) (package-address '<SR-DATA> var x 'table 'sr 'fraction)))
(def-dsp-fun 'SR-FILT           #'(lambda (var x) (package-address '<SR-FILT> var x  'table 'sr 'fraction)))
(def-dsp-fun 'SR-LEFT           #'(lambda (var x) (package-address '<SR-LEFT> var x 'integer 'sr)))
(def-dsp-fun 'SR-RIGHT          #'(lambda (var x) (package-address '<SR-RIGHT> var x 'integer 'sr)))
(def-dsp-fun 'SR-WIDTH          #'(lambda (var x) (package-address '<SR-WIDTH> var x 'integer 'sr)))

(def-dsp-fun 'WS-TAB            #'(lambda (var x) (package-address '<WS-TAB> var x 'table 'ws 'fraction)))
(def-dsp-fun 'WS-OS             #'(lambda (var x) (package-address '<WS-OS> var x 'osc 'ws)))
(def-dsp-fun 'WS-OFFSET         #'(lambda (var x) (package-address '<WS-OFFSET> var x 'integer 'ws)))

(def-dsp-fun 'ZDLY-PHASE        #'(lambda (var x) (package-address '<ZDLY-PHASE> var x 'real 'zdly)))
(def-dsp-fun 'ZDLY-DEL          #'(lambda (var x) (package-address '<ZDLY-DEL> var x 'dly 'zdly)))

(def-dsp-fun 'COSP-PHASE        #'(lambda (var x) (package-address '<COSP-PHASE> var x 'real 'cosp)))
(def-dsp-fun 'COSP-FREQ         #'(lambda (var x) (package-address '<COSP-FREQ> var x 'real 'cosp)))
(def-dsp-fun 'COSP-SCALER       #'(lambda (var x) (package-address '<COSP-SCALER> var x 'real 'cosp)))
(def-dsp-fun 'COSP-COSINES      #'(lambda (var x) (package-address '<COSP-COSINES> var x 'real 'cosp)))

(def-dsp-fun 'LOCS-ASCL         #'(lambda (var x) (package-address '<LOCS-ASCL> var x 'fraction 'locs)))
(def-dsp-fun 'LOCS-BSCL         #'(lambda (var x) (package-address '<LOCS-BSCL> var x 'fraction 'locs)))
(def-dsp-fun 'LOCS-CSCL         #'(lambda (var x) (package-address '<LOCS-CSCL> var x 'fraction 'locs)))
(def-dsp-fun 'LOCS-DSCL         #'(lambda (var x) (package-address '<LOCS-DSCL> var x 'fraction 'locs)))
(def-dsp-fun 'LOCS-RSCL         #'(lambda (var x) (package-address '<LOCS-RSCL> var x 'fraction 'locs)))
(def-dsp-fun 'LOCS-DEG          #'(lambda (var x) (package-address '<LOCS-DEG> var x 'real 'locs)))
(def-dsp-fun 'LOCS-DIS          #'(lambda (var x) (package-address '<LOCS-DIS> var x 'real 'locs)))
(def-dsp-fun 'LOCS-PC-REV       #'(lambda (var x) (package-address '<LOCS-PC-REV> var x 'real 'locs)))
(def-dsp-fun 'LOCS-REVNAME      #'(lambda (var x) (package-address '<LOCS-REVNAME> var x 'integer 'locs)))

(def-dsp-fun 'SPD-RD            #'(lambda (var x) (package-address '<SPD-RD> var x 'rdin 'spd)))
(def-dsp-fun 'SPD-LEN           #'(lambda (var x) (package-address '<SPD-LEN> var x 'integer 'spd)))
(def-dsp-fun 'SPD-RMP           #'(lambda (var x) (package-address '<SPD-RMP> var x 'integer 'spd)))
(def-dsp-fun 'SPD-AMP           #'(lambda (var x) (package-address '<SPD-AMP> var x 'fraction 'spd)))
(def-dsp-fun 'SPD-IN-SPD        #'(lambda (var x) (package-address '<SPD-IN-SPD> var x 'integer 'spd)))
(def-dsp-fun 'SPD-OUT-SPD       #'(lambda (var x) (package-address '<SPD-OUT-SPD> var x 'integer 'spd)))
(def-dsp-fun 'SPD-CUR-IN        #'(lambda (var x) (package-address '<SPD-CUR-IN> var x 'long-int 'spd)))
(def-dsp-fun 'SPD-CUR-OUT       #'(lambda (var x) (package-address '<SPD-CUR-OUT> var x 'integer 'spd)))
(def-dsp-fun 'SPD-S20           #'(lambda (var x) (package-address '<SPD-S20> var x 'integer 'spd)))
(def-dsp-fun 'SPD-S50           #'(lambda (var x) (package-address '<SPD-S50> var x 'integer 'spd)))
(def-dsp-fun 'SPD-CTR           #'(lambda (var x) (package-address '<SPD-CTR> var x 'integer 'spd)))

(def-dsp-fun 'IO-FIL            #'(lambda (var x) (package-address '<IO-FIL> var x 'integer 'io)))
(def-dsp-fun 'IO-NAM            #'(lambda (var x) (package-address '<IO-NAM> var x 'string 'io)))
(def-dsp-fun 'IO-DAT-A          #'(lambda (var x) (package-address '<IO-DAT-A> var x 'integer 'io)))
(def-dsp-fun 'IO-DAT-B          #'(lambda (var x) (package-address '<IO-DAT-B> var x 'integer 'io)))
(def-dsp-fun 'IO-DAT-C          #'(lambda (var x) (package-address '<IO-DAT-C> var x 'integer 'io)))
(def-dsp-fun 'IO-DAT-D          #'(lambda (var x) (package-address '<IO-DAT-D> var x 'integer 'io)))
(def-dsp-fun 'IO-BEG            #'(lambda (var x) (package-address '<IO-BEG> var x 'integer 'io)))
(def-dsp-fun 'IO-END            #'(lambda (var x) (package-address '<IO-END> var x 'integer 'io)))
(def-dsp-fun 'IO-DATA-END       #'(lambda (var x) (package-address '<IO-DATA-END> var x 'integer 'io)))
(def-dsp-fun 'IO-DATA-START     #'(lambda (var x) (package-address '<IO-DATA-START> var x 'integer 'io)))
(def-dsp-fun 'IO-OPEN-INDEX     #'(lambda (var x) (package-address '<IO-OPEN-INDEX> var x 'integer 'io)))
(def-dsp-fun 'IO-SIZ            #'(lambda (var x) (package-address '<IO-SIZ> var x 'integer 'io)))
(def-dsp-fun 'IO-DIR            #'(lambda (var x) (package-address '<IO-DIR> var x 'integer 'io)))
(def-dsp-fun 'IO-BUFSIZ         #'(lambda (var x) (package-address '<IO-BUFSIZ> var x 'integer 'io)))
;;; io-hdr, io-hdr-end, and io-external not implemented

(def-dsp-fun 'LOGIOR            #'(lambda (var x) (package-op '<logior> var x 'integer)))
(def-dsp-fun 'LOGXOR            #'(lambda (var x) (package-op '<logxor> var x 'integer)))
(def-dsp-fun 'LOGAND            #'(lambda (var x) (package-op '<logand> var x 'integer)))
(def-dsp-fun 'LOGNOT            #'(lambda (var x) (package-op '<lognot> var x 'integer)))
(def-dsp-fun 'LOGEQV            #'(lambda (var x) (package-op '<logeqv> var x 'integer)))
(def-dsp-fun 'LOGNAND           #'(lambda (var x) (package-op '<lognand> var x 'integer)))
(def-dsp-fun 'LOGNOR            #'(lambda (var x) (package-op '<lognor> var x 'integer)))
(def-dsp-fun 'LOGANDC1          #'(lambda (var x) (package-op '<logandc1> var x 'integer)))
(def-dsp-fun 'LOGANDC2          #'(lambda (var x) (package-op '<logandc2> var x 'integer)))
(def-dsp-fun 'LOGORC1           #'(lambda (var x) (package-op '<logorc1> var x 'integer)))
(def-dsp-fun 'LOGORC2           #'(lambda (var x) (package-op '<logorc2> var x 'integer)))

(def-dsp-fun 'REM               #'(lambda (var x) (package-op '<rem> var x)))
(def-dsp-fun 'GCD               #'(lambda (var x) (package-op '<gcd> var x 'integer)))
(def-dsp-fun 'LCM               #'(lambda (var x) (package-op '<lcm> var x 'integer)))
(def-dsp-fun 'NUMERATOR         #'(lambda (var x) (package-op '<numerator> var x)))
(def-dsp-fun 'DENOMINATOR       #'(lambda (var x) (package-op '<denominator> var x)))
	
(def-dsp-fun 'MAX               #'(lambda (var x) (package-op '<max> var x)))
(def-dsp-fun 'MIN               #'(lambda (var x) (package-op '<min> var x)))
(def-dsp-fun 'ABS               #'(lambda (var x) (package-op '<abs> var x)))
(def-dsp-fun 'MOD               #'(lambda (var x) (package-op '<mod> var x)))

(def-dsp-fun 'FLOOR             #'(lambda (var x) (package-op '<floor> var x)))        ;toward -inf (largest int not larger)
(def-dsp-fun 'CEILING           #'(lambda (var x) (package-op '<ceiling> var x)))      ;toward +inf (smallest int not smaller)
(def-dsp-fun 'ROUND             #'(lambda (var x) (package-op '<round> var x)))        ;+.5 (round to even if .5)
(def-dsp-fun 'TRUNCATE          #'(lambda (var x) (package-op '<truncate> var x)))     ;toward 0
(def-dsp-fun 'SIGNUM            #'(lambda (var x) (package-op '<signum> var x)))
(def-dsp-fun 'SQRT              #'(lambda (var x) (package-op '<sqrt> var x)))
(def-dsp-fun 'RANDOM            #'(lambda (var x) (package-op '<random> var x)))

(def-dsp-fun 'SIN               #'(lambda (var x) (package-op '<sin> var x)))
(def-dsp-fun 'COS               #'(lambda (var x) (package-op '<cos> var x)))
(def-dsp-fun 'TAN               #'(lambda (var x) (package-op '<tan> var x)))
(def-dsp-fun 'LOG               #'(lambda (var x) (package-op '<log> var x)))
(def-dsp-fun 'EXPT              #'(lambda (var x) (package-op '<expt> var x)))
(def-dsp-fun 'EXP               #'(lambda (var x) (package-op '<exp> var x)))
(def-dsp-fun 'ASIN              #'(lambda (var x) (package-op '<asin> var x)))
(def-dsp-fun 'ACOS              #'(lambda (var x) (package-op '<acos> var x)))
(def-dsp-fun 'ATAN              #'(lambda (var x) (package-op '<atan> var x)))
(def-dsp-fun 'COSH              #'(lambda (var x) (package-op '<cosh> var x)))
(def-dsp-fun 'SINH              #'(lambda (var x) (package-op '<sinh> var x)))
(def-dsp-fun 'TANH              #'(lambda (var x) (package-op '<tanh> var x)))
(def-dsp-fun 'ASINH             #'(lambda (var x) (package-op '<asinh> var x)))
(def-dsp-fun 'ACOSH             #'(lambda (var x) (package-op '<acosh> var x)))
(def-dsp-fun 'ATANH             #'(lambda (var x) (package-op '<atanh> var x)))
(def-dsp-fun 'FLOAT             #'(lambda (var x) (package-op '<float> var x)))
(def-dsp-fun 'ASH               #'(lambda (var x) (package-op '<ash> var x)))
	  
(def-dsp-fun 'OSCIL             #'(lambda (var x) (package-op '<oscil> var x 'osc)))
(def-dsp-fun 'ENV               #'(lambda (var x) (package-op '<env> var x 'envelope)))
(def-dsp-fun 'RESTART-ENV       #'(lambda (var x) (package-op '<restart-env> var x 'envelope)))
(def-dsp-fun 'OUTA              #'(lambda (var x) (fpackage-op '<outa> var x 'long-int) 0))
(def-dsp-fun 'OUTB              #'(lambda (var x) (fpackage-op '<outb> var x 'long-int) 0))
(def-dsp-fun 'OUTC              #'(lambda (var x) (fpackage-op '<outc> var x 'long-int) 0))
(def-dsp-fun 'OUTD              #'(lambda (var x) (fpackage-op '<outd> var x 'long-int) 0))
					;(def-dsp-fun 'OUT-N                #'(lambda (var x) (package-op '<outn> var x 'long-int)))
(def-dsp-fun 'INA               #'(lambda (var x) (fpackage-op '<in-a> var x 'long-int 'fraction)))
(def-dsp-fun 'INB               #'(lambda (var x) (fpackage-op '<in-b> var x 'long-int 'fraction)))
(def-dsp-fun 'IN-A              #'(lambda (var x) (fpackage-op '<in-a> var x 'long-int 'fraction)))
(def-dsp-fun 'IN-B              #'(lambda (var x) (fpackage-op '<in-b> var x 'long-int 'fraction)))
(def-dsp-fun 'IN-C              #'(lambda (var x) (fpackage-op '<in-c> var x 'long-int 'fraction)))
(def-dsp-fun 'IN-D              #'(lambda (var x) (fpackage-op '<in-d> var x 'long-int 'fraction)))
					;(def-dsp-fun 'IN-N                 #'(lambda (var x) (package-op '<inn> var x 'long-int)))
(def-dsp-fun 'QUAD              #'(lambda (var x) (fpackage-op '<quad> var x 'file 'integer)))
(def-dsp-fun 'STEREO            #'(lambda (var x) (fpackage-op '<stereo> var x 'file 'integer)))
(def-dsp-fun 'MONO              #'(lambda (var x) (fpackage-op '<mono> var x 'file 'integer)))
					;revin and revout are macros
(def-dsp-fun 'LOCSIG            #'(lambda (var x) 
				    (when (eq *clm-language* :c) 
				      (if (listp (cadr x)) (push `(<check-locsig> ,(cadr x)) new-prog))
				      (push (list var 'hloc) typed-user-sig))
				    (package-op '<locsig> var x 'locs nil)
				    (if (eq *clm-language* :c) var)))
					;In-Hz is a macro
(def-dsp-fun 'RANDH             #'(lambda (var x) (package-op '<randh> var x 'noi)))
(def-dsp-fun 'RANDI             #'(lambda (var x) (package-op '<randi> var x 'randi)))
(def-dsp-fun 'SAWTOOTH-WAVE     #'(lambda (var x) (package-op '<sawtooth-wave> var x 'sw)))
(def-dsp-fun 'TRIANGLE-WAVE     #'(lambda (var x) (package-op '<triangle-wave> var x 'sw)))
(def-dsp-fun 'PULSE-TRAIN       #'(lambda (var x) (package-op '<pulse-train> var x 'sw)))
(def-dsp-fun 'SQUARE-WAVE       #'(lambda (var x) (package-op '<square-wave> var x 'sw)))
					;table-lookup is a macro in mus.lisp -- calls ur-table-lookup
(def-dsp-fun 'UR-TABLE-LOOKUP   #'(lambda (var x) (package-op '<ur-table-lookup> var x 'tbl)))
					;ring-modulate is a macro
(def-dsp-fun 'AMPLITUDE-MODULATE #'(lambda (var x) (package-op '<amplitude-modulate> var x)))
(def-dsp-fun 'DELAY             #'(lambda (var x) (package-op '<delay> var x 'dly t t)))
(def-dsp-fun 'TAP               #'(lambda (var x) (package-op '<tap> var x 'dly t t)))
(def-dsp-fun 'ZDELAY            #'(lambda (var x) (package-op '<zdelay> var x 'zdly t t 4)))
(def-dsp-fun 'ZTAP              #'(lambda (var x) (package-op '<ztap> var x 'zdly t t 4)))
(def-dsp-fun 'COMB              #'(lambda (var x) (package-op '<comb> var x 'cmbflt t t)))
					;notch is a macro
(def-dsp-fun 'ALL-PASS          #'(lambda (var x) (package-op '<all-pass> var x 'allpassflt t t)))
(def-dsp-fun 'DOT-PRODUCT       #'(lambda (var x) 
				    (chkv (caddr x) 'table)
				    (package-op '<dot-product> var x 'table)))
(def-dsp-fun 'ONE-POLE          #'(lambda (var x) (package-op '<one-pole> var x 'flt-one-pole) (assure-type x 'flt-one-pole) var))
(def-dsp-fun 'ONE-ZERO          #'(lambda (var x) (package-op '<one-zero> var x 'flt-one-zero) (assure-type x 'flt-one-zero) var))
(def-dsp-fun 'TWO-POLE          #'(lambda (var x) (package-op '<two-pole> var x 'flt-two-pole) (assure-type x 'flt-two-pole) var))
(def-dsp-fun 'TWO-ZERO          #'(lambda (var x) (package-op '<two-zero> var x 'flt-two-zero) (assure-type x 'flt-two-zero) var))
					;ppolar, zpolar, and formnt are macros in mus.lisp
(def-dsp-fun 'FILTER            #'(lambda (var x) (package-op '<filter> var x 'flt)))
(def-dsp-fun 'LATTICE-FILTER    #'(lambda (var x) (package-op '<lattice-filter> var x 'flt)))
(def-dsp-fun 'LADDER-FILTER     #'(lambda (var x) (package-op '<ladder-filter> var x 'flt)))
(def-dsp-fun 'DIRECT-FILTER     #'(lambda (var x) (package-op '<direct-filter> var x 'flt)))
(def-dsp-fun 'FFT-FILTER        #'(lambda (var x) 
				    (package-op '<fft-filter> var x 'fftflt)
				    (if (listp (cadr x)) (push `(<check-fftflt> ,(cadr x)) new-prog))
				    var))
(def-dsp-fun 'CONVOLVE          #'(lambda (var x) 
				    (package-op '<convolve> var x 'conv)
				    (if (listp (cadr x)) (push `(<check-convolve> ,(cadr x)) new-prog))
				    var))
(def-dsp-fun 'FFT               #'(lambda (var x) (package-op '<fft> var x)))
(def-dsp-fun 'INVERSE-FFT       #'(lambda (var x) (package-op '<inverse-fft> var x)))
;(def-dsp-fun 'SPECTRUM          #'(lambda (var x) (package-op '<spectrum> var x)))
(def-dsp-fun 'FFT-WINDOW        #'(lambda (var x) (package-op '<fft-window> var x)))
(def-dsp-fun 'CLEAR-BLOCK       #'(lambda (var x) (package-op '<clear-block> var x)))

(def-dsp-fun 'RESAMPLE          #'(lambda (var x) 
				    (package-op '<resample> var x 'smp)
				    (if (listp (cadr x)) (push `(<check-resample> ,(cadr x)) new-prog))
				    var))
(def-dsp-fun 'SRC               #'(lambda (var x) 
				    (package-op '<src> var x 'sr)
				    (if (listp (cadr x)) (push `(<check-src> ,(cadr x)) new-prog))
				    var))
					;contrast-enhancement is handled as a macro in mus.lisp
(def-dsp-fun 'READIN            #'(lambda (var x) 
				    (package-op '<readin> var x 'rdin)
				    (if (listp (cadr x)) (push `(<check-readin> ,(cadr x)) new-prog))
				    var))
(def-dsp-fun 'READIN-REVERSE    #'(lambda (var x) 
				    (package-op '<readin> var x 'rdin)
				    (if (listp (cadr x)) (push `(<check-readin> ,(cadr x)) new-prog))
				    var))
(def-dsp-fun 'READ-POSITION     #'(lambda (var x) (package-op '<read-position> var x)))
(def-dsp-fun 'READ-FORWARD      #'(lambda (var x) (package-op '<read-forward> var x)))
(def-dsp-fun 'READ-BACKWARD     #'(lambda (var x) (package-op '<read-backward> var x)))

(def-dsp-fun 'WAVE-TRAIN        #'(lambda (var x) (package-op '<wave-train> var x 'wt)))
(def-dsp-fun 'RUN-BLOCK         #'(lambda (var x) (package-op '<run-block> var x 'rblk)))
(def-dsp-fun 'EXPAND            #'(lambda (var x) 
				    (package-op '<expand> var x 'spd)
				    (if (listp (cadr x)) (push `(<check-expand> ,(cadr x)) new-prog))
				    var))
					;called SpeedFile in Mixer.
					;waveshape is a macro -- calls table-interp and ws struct accessors
(def-dsp-fun 'TABLE-INTERP      #'(lambda (var x) (package-op '<table-interp> var x 'table)))
(def-dsp-fun 'ARRAY-INTERP      #'(lambda (var x) 
				    (chkv (cadr x) 'array 'real)
				    (package-op '<array-interp> var x 'array)))
(def-dsp-fun 'POLYNOMIAL        #'(lambda (var x) 
				    (chkv (cadr x) 'array 'fraction)
				    (if (listp (cadr x)) (chkv (second (second x)) 'array 'array))
				    (package-op '<polynomial> var x)))
;;; here if run-safety = 3, we could expand into the guts of the polynomial function in mus.lisp

(def-dsp-fun 'SUM-OF-COSINES    #'(lambda (var x) (package-op '<sum-of-cosines> var x 'cosp)))

;;; here some context is useful -- if we're in a conditional, then there's no need to return a value
;;; from these guys, just jump to the correct place (i.e. the value is implicit in the program execution).
;;; So, we'll have three stacks of labels to direct traffic.  Decided not to use the "y" (context) argument
;;; in the tree walker, since I'm not sure I can step on it with impunity.

(def-dsp-fun '=                #'(lambda (var x) (package-rel-op var x '<jump-neq> '<jump-eq> t)))
(def-dsp-fun '/=               #'(lambda (var x) (package-rel-op var x '<jump-eq> '<jump-neq> t)))
(def-dsp-fun '<                #'(lambda (var x) (package-rel-op var x '<jump-geq> '<jump-l>)))
(def-dsp-fun '>                #'(lambda (var x) (package-rel-op var x '<jump-leq> '<jump-g>)))
(def-dsp-fun '<=               #'(lambda (var x) (package-rel-op var x '<jump-g> '<jump-leq>)))
(def-dsp-fun '>=               #'(lambda (var x) (package-rel-op var x '<jump-l> '<jump-geq>)))

(def-dsp-fun 'ZEROP            #'(lambda (var x) (package-bool-op '<test> var x nil '<jump-neq> '<jump-eq>)))
(def-dsp-fun 'PLUSP            #'(lambda (var x) (package-bool-op '<test> var x nil '<jump-leq> '<jump-pl>)))
(def-dsp-fun 'MINUSP           #'(lambda (var x) (package-bool-op '<test> var x nil '<jump-geq> '<jump-mi>)))

(def-dsp-fun 'ODDP             #'(lambda (var x) (package-bool-op '<test-bit-0> var x 'integer '<jump-eq> '<jump-neq>)))
(def-dsp-fun 'EVENP            #'(lambda (var x) (package-bool-op '<test-bit-0> var x 'integer '<jump-neq> '<jump-eq>)))

(def-dsp-fun 'LOGBITP          #'(lambda (var x) (package-bool-op '<logbitp> var x 'integer '<jump-eq> '<jump-neq>)))
(def-dsp-fun 'LOGTEST          #'(lambda (var x) (package-bool-op '<logtest> var x 'integer '<jump-eq> '<jump-neq>)))


;;; integerp numberp rationalp floatp realp typep

#|
;;; NOT and NULL are trickier than I anticipated.

;;; after 25-Feb-92
(def-dsp-fun 'NULL             #'(lambda (var x) 
				   (declare (ignore var))
				   (package-bool-op '<not-null> nil x nil '<jump-true> '<jump-false>)))
(def-dsp-fun 'NOT              #'(lambda (var x) 
				   (declare (ignore var))
				   (package-bool-op '<not-null> nil x nil '<jump-true> '<jump-false>)))
(clu:definstrument simp ()
  (let ((x 0)
	(y 1)
	(z nil))
    (run
     (loop for i from 0 to 10 do
       (if (not (zerop x)) (print "yow"))
       (if (/= 0 x) (print "yowza"))
       (if (/= x 0) (print "oops"))
       (if (zerop y) (print "zow"))
       (if (= 0 y) (print "zowza"))
       (if (= y 0) (print "zoops"))
       (if z (print "z oops"))
       (if (not (null z)) (print "z yow"))
       (if (not (not z)) (print "z yowza"))
       (if (not (and (zerop x) (not (zerop y)) (not z))) (print "1 bug"))
       (if (or (minusp y) (and (not (zerop x)) (not z))) (print "2 bug"))
       (if (not (and (/= x 1) (/= y 0) (not z) (null z))) (print "3 bug"))
       (outa i x)))))
|#
;;; before 25-Feb-92 and again after 10-Feb-93
(defun Not-branch (var x)
  (if t-lab-stack
      (let ((old-j (nojump)))
	(push-args (current-f-label) (current-t-label) (if (nojump) 'false 'true))
	(if (and (listp (cadr x)) (boole-op (caadr x)))
	    (walk-form (cadr x) nil 'dsp-ize)
	  (progn
	    (push `(<not-null> ,(walk-form (cadr x) nil 'dsp-ize)) new-prog)
	    (if old-j
		(push `(<jump-true> ,(current-t-label)) new-prog) ;these jumps were reversed 25-Feb-92
	      (push `(<jump-false> ,(current-f-label)) new-prog))))
	(pop-args)
	nil)
    (progn
      (push `(<not-null> ,(walk-form (cadr x) nil 'dsp-ize)) new-prog)
      (bool-var var <nil> '<jump-true> nil))))

(def-dsp-fun 'NOT    'Not-branch)
(def-dsp-fun 'NULL   'Not-branch)



;;; these have to be a little smart because in expressions like
;;;   (incf (aref hi (incf i)))
;;; we can't repeat the incf i 

(defun incf-arguments (x)
  (let* ((variable (cadr x))
	 (change (or (caddr x) 1)))
    (if (and (listp variable)
	     (eq (first variable) 'AREF))
	(setf variable `(AREF 
			 ,(second variable)
			 ,@(loop for arg in (cddr variable)
			    collect (if (listp arg)
					(walk-form arg)
				      arg)))))
    (values variable change)))


(def-dsp-fun 'INCF             #'(lambda (var x)
				   (declare (ignore var))
				   (multiple-value-bind 
				       (variable change)
				       (incf-arguments x)
				     (walk-form `(setf ,variable (+ ,variable ,change)) nil 'dsp-ize))))

(def-dsp-fun 'DECF             #'(lambda (var x)
				   (declare (ignore var))
				   (multiple-value-bind 
				       (variable change)
				       (incf-arguments x)
				     (walk-form `(setf ,variable (- ,variable ,change)) nil 'dsp-ize))))

(def-dsp-fun '-                #'(lambda (var x) 
				   (if (cddr x)
				       (package-op '<subtract> var x)
				     (package-op '<negate> var x))))
(def-dsp-fun '*                #'(lambda (var x) (package-op '<multiply> var x)))
(def-dsp-fun '+                #'(lambda (var x) (package-op '<add> var x)))
(def-dsp-fun '/                #'(lambda (var x) (package-op '<divide> var x)))

(def-dsp-fun '1+               #'(lambda (var x) (package-op '<add-1> var x)))
(def-dsp-fun '1-               #'(lambda (var x) (package-op '<subtract-1> var x)))

(def-dsp-fun 'OR     'Or-branch)
(def-dsp-fun 'AND    'And-branch)
(def-dsp-fun 'IF     'If-branch)
(def-dsp-fun 'WHEN   'When-branch)
(def-dsp-fun 'UNLESS 'Unless-branch)
(def-dsp-fun 'COND   'Cond-branch)
(def-dsp-fun 'CASE   'Case-branch)
(def-dsp-fun 'DOTIMES 'Dotimes-branch)
(def-dsp-fun 'DO     #'(lambda (var x) (Do-branch var x nil)))
(def-dsp-fun 'DO*    #'(lambda (var x) (Do-branch var x t)))
(def-dsp-fun 'SETF   'setf-branch)
(def-dsp-fun 'SETQ   'setf-branch)

(def-dsp-fun 'TAGBODY 'Tagbody-branch)
(def-dsp-fun 'BLOCK   'Block-branch)
(def-dsp-fun 'RETURN  'Return-branch)
(def-dsp-fun 'RETURN-FROM 'Return-from-branch)

(def-dsp-fun 'LOOP   'find-loop-control-info)
;;;(def-dsp-fun 'BREAK  #'(lambda (var x) (declare (ignore var x)) (push '(<break>) new-prog) nil))
(def-dsp-fun 'BREAK #'(lambda (var x)     (package-op '<break> var x)))
(def-dsp-fun 'WARN #'(lambda (var x)      (package-op '<warn> var x)))
(def-dsp-fun 'ERROR #'(lambda (var x)     (package-op '<error> var x)))
(def-dsp-fun 'PRINT #'(lambda (var x)     (package-op '<print> var x)))
(def-dsp-fun 'CLM-PRINT #'(lambda (var x) (package-op '<clm-print> var x)))
(def-dsp-fun 'PRINC #'(lambda (var x)     (package-op '<princ> var x)))
(def-dsp-fun 'Y-OR-N-P #'(lambda (var x)  (package-op '<y-or-n-p> var x)))
(def-dsp-fun 'YES-OR-NO-P #'(lambda (var x) (package-op '<yes-or-no-p> var x)))
(def-dsp-fun 'TERPRI #'(lambda (var x)    (package-op '<terpri> var x)))
(def-dsp-fun 'APPLY #'(lambda (var x)     (package-op '<apply> var x)))
(def-dsp-fun 'FUNCALL #'(lambda (var x)   (package-op '<funcall> var x)))
(def-dsp-fun 'FUNCTION #'(lambda (var x)  (declare (ignore var)) (symbol-name (cadr x))))


;;; array-dimension, array-element-type, array-rank, array-dimensions, array-total-size, array-in-bounds-p
;;; for tables these are easy: Y[loc] has table size
;;; for arrays size=(/ (1+ X[loc+1]) Y[loc]) and element type is unknown (size thereof is known)
;;; run time disambiguation might involve Y[loc+1]=0 or 1 for table, >1 for array
;;; (ARRAY-DIMENSIONS 1)

(def-dsp-fun 'AREF   'aref-branch)
(def-dsp-fun 'ELT    'aref-branch)
(def-dsp-fun 'SVREF  'aref-branch)
	   
           ;; no progv prog prog*
(def-dsp-fun 'PROGN  #'(lambda (var x) 
			 (declare (ignore var))
			 (loop for i in (cdr x) do (setf last-var (walk-form i nil 'dsp-ize)))
			 (if last-var (rem-var last-var))))

(def-dsp-fun 'PROG1  #'(lambda (var x) 
			 (declare (ignore var))
			 (let ((nvar (walk-form (cadr x) nil 'dsp-ize)))
			   (loop for i in (cddr x) do (walk-form i nil 'dsp-ize))
			   (if nvar (rem-var nvar)))))

(def-dsp-fun 'PROG2  #'(lambda (var x) 
			 (declare (ignore var))
			 (walk-form (cadr x) nil 'dsp-ize)
			 (let ((nvar (walk-form (caddr x) nil 'dsp-ize)))
			   (loop for i in (cdddr x) do (walk-form i nil 'dsp-ize))
			   (if nvar (rem-var nvar)))))

(def-dsp-fun 'GO     #'(lambda (var x) 
			 (declare (ignore var))
			 (let ((lab (if (and loop-label-stack
					     #-(or mcl (and excl cltl2)) (member (cadr x) '(LOOP::NEXT-LOOP LOOP::END-LOOP))
					     #+(and excl cltl2) (member (cadr x) '(EXCL::NEXT-LOOP EXCL::END-LOOP))
					     #+mcl (member (cadr x) '(ANSI-LOOP::NEXT-LOOP ANSI-LOOP::END-LOOP))
					     )
					(intern (concatenate 'string (symbol-name (cadr x)) "-" (symbol-name (first loop-label-stack))))
				      (cadr x))))
			   (push `(<jump> ,lab) new-prog))
			 nil))

(def-dsp-fun 'LOOP-FINISH
                     #'(lambda (var x)
			 (declare (ignore var x))
			 (let ((lab (if loop-label-stack
					(intern (concatenate 'string (symbol-name 
								      #-(or mcl (and excl cltl2)) 'LOOP::END-LOOP
								      #+(and excl cltl2) 'EXCL::END-LOOP
								      #+mcl 'ANSI-LOOP::END-LOOP)
							     "-" (symbol-name (first loop-label-stack)))))))
			   (if loop-label-stack
			       (push `(<jump> ,lab) new-prog)
			     (push `(<loop-finish>) new-prog)))
			 nil))

(def-dsp-fun 'LET    #'(lambda (var x) (let-branch var x nil)))
(def-dsp-fun 'LET*   #'(lambda (var x) (let-branch var x t)))

#|
(def-dsp-fun 'DECLARE #'(lambda (var x) (declare (ignore var x)) nil))
;;; everything in the declare form is simply ignored for now
;;; one of the mit-loops declares types incorrectly, so it's not safe to use these yet.
|#

(def-dsp-fun 'DECLARE			;this would work in general in RUN, but mit-loop is buggy (it generates bogus TYPE declarations)
  #'(lambda (var x)
      (declare (ignore var))
      (loop for decl in (cdr x) by #'cdr do
	(if (eq (car decl) 'OPTIMIZE)
	    (loop for opt in (cdr decl) by #'cdr do
	      (if (eq (car opt) 'SAFETY)
		  (setf run-safety (cadr opt))
		(if (eq (car opt) 'SPEED)
		    (setf run-speed (cadr opt))))))
	(if (eq (car decl) 'TYPE)
	    (let* ((ur-typ (second decl))
		   (typ (if (eq ur-typ 'float) 'real
			  (if (eq ur-typ 'fixnum) 'integer
			    (if (eq ur-typ 'bignum) 'long-int)))))
	      (if typ
		  (loop for var-decl in (cddr decl) by #'cdr do
		    (let ((vr (find var-decl typed-user-var :key #'first)))
		      (if vr (setf (second vr) typ)
			(let ((vr (find var-decl user-var)))
			  (if vr
			      (progn
				(setf user-var (remove var-decl user-var))
				(push (list var-decl typ) typed-user-var))
			    (push (list var-decl typ) typed-user-var))))))))))
      nil))


;;; the following is just a stop-gap for users of cmu-loop -- it generates
;;; macrolets within the lambda form that loop expands into, but these
;;; macrolets appear to me to be very simple.

(defun submac (name body code) 
  (if (listp code) 
      (if (eq (first code) name) 
	  body 
	(loop for x in code collect (submac name body x))) 
    code))

(def-dsp-fun 'MACROLET #'(lambda (var x) 
			   (declare (ignore var)) 
			   (submac (first (second x)) (third (second x)) (third x))))

;;; I suppose we could make a list of macrolet macros with args, then form
;;; the LETs that are equivalent thereto, then do the substitution with 
;;; args in submac -- surely someone has already written such a function.
