;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;

(in-package :clm)

;;;
;;; This is an adjunct to lib56.lisp -- a library of 56000 code intended to run on the Ariel QP board.
;;; This code either uses the main memory on the board, or uses the master-slave commnuication channels.
;;; All other straightforward 56000 code (that would run on any 56000) is in lib56.lisp.

(defconstant QP-merge-done #o25252525)	;nostalgia -- SND file flag from long long ago
(defconstant QP-size-loc 4)

(defun QP-Initialize-names ()
  (emit `(DEFINE QP-DRAM-size-location ,QP-size-loc))
  (emit '(DEFINE DMC-MCR        #xFFFA));DMC master control register
  (emit '(DEFINE DMC-read-addr  #xFFE0));where to write DMC address for reading data
  (emit '(DEFINE DMC-read-data  #xFFE2));where to read DMC data
  (emit '(DEFINE DMC-write-addr #xFFE1));where to write DMC address for writing data
  (emit '(DEFINE DMC-write-data #xFFE3));where to write DMC data
  (emit '(DEFINE DMC-config     #xFFE6));DMC configuration register (written while held reset)
  (emit '(DEFINE DMC-enable 1))
  (emit '(DEFINE DMC-select 2))
  (emit '(DEFINE DMC-reset 4))		;mode bit in DMC master control register -- on for reset, off normally
  (emit '(DEFINE DMC-refresh 7))
  (emit '(DEFINE set-256K #x8000))	;magic number to configure for 256K DRAM
  (emit '(DEFINE set-1M   #x10000))	;ditto for 1M
  (emit '(DEFINE set-4M   #x18000))	;ditto for 4M
  (emit '(DEFINE M-CmdA #xFFC0))	;DSPA command port from Master's point of view
  (emit '(DEFINE M-CmdB #xFFC1))
  (emit '(DEFINE M-CmdC #xFFC2))
  (emit '(DEFINE M-CmdD #xFFC3))
  (emit '(DEFINE M-DatA #xFFC4))	;DSPA data port from Master's point of view
  (emit '(DEFINE M-DatB #xFFC5))
  (emit '(DEFINE M-DatC #xFFC6))
  (emit '(DEFINE M-DatD #xFFC7))
  (emit '(DEFINE M-slave-interrupt-reg #xFFF8))
  (emit '(DEFINE M-Misc-interrupt-reg  #xFFF9))
  (emit '(DEFINE M-slave-status-reg    #xFFFB))
  (emit '(DEFINE M-slave-priority-reg  #xFFFD))
  (emit '(DEFINE M-Misc-status-reg     #xFFFF))
  (emit '(DEFINE S-Dat  #xFFC0))
  (emit '(DEFINE S-Cmd  #xFFC1))	;status and command ports (status in bits 20..23)
  (emit '(DEFINE S-Ctrl #xFFC2)))	;control port (memory map mode primarily -- handled in qp.c)
  ;; the slaves DAT and CMD locations are actually two locations each -- what the slave last wrote,
  ;; and what the master last wrote -- the slave reads what the master wrote, and the master reads
  ;; what the slave wrote.  The status bits 22 and 23 can be used as follows: set bit 23 in FFC2 to
  ;; enable the handshake flags, then the default state is 4xxxxx (22 on, 23 off).  Write a word to
  ;; FFC0, when master reads it, bit 22 goes off; and upon next write (from slave), it is on again.
  ;; Similarly for bit 23, but here from master-write point of view (i.e. it is off until master
  ;; writes to data port, is then on until slave reads data port).  The main source of confusion here
  ;; is that the data read from the status port does not change (to its "current" value) until it
  ;; has been read.

(defun QP-size-location () 
  (or QP-size-loc 0))

(defun QP-size-code (dsp) 
  (if (and dsp (dsp-data-hub dsp))
      (let ((DRAM-size (qp-data-DRAM-size (dsp-data-hub dsp))))
	(if (= DRAM-size 256) QP-256K (if (= DRAM-size 1000) QP-1M QP-4M)))
    0))

(defun QP-initialize-temps ()
  (emit (list 'DEFINE 'qp-temp-loc (get-L-memory)))
  (emit (list 'DEFINE 'QP-bufs (get-X-memory)))
  (emit (list 'DEFINE 'QP-size (get-Y-memory)))
  (emit (list 'DEFINE 'QP-dsp  (get-X-memory)))
  (emit (list 'DEFINE 'QP-data (get-L-memory 16)))
  (emit (list 'DEFINE 'QP-shift-data (get-L-memory 32)))
  (emit (list 'DEFINE 'QP-shifts (get-X-memory)))
  (emit (list 'DEFINE 'QP-sbufs (get-X-memory)))
  (emit (list 'DEFINE 'QP-ssize (get-Y-memory)))
  (emit (list 'DEFINE 'QP-sdata (get-L-memory 16)))
  (emit (list 'DEFINE 'QP-save-R (get-L-memory)))
  (emit (list 'DEFINE 'QP-save-N (get-L-memory))))

(defun QP-write-DRAM (addr data)
  (emit `(STORE ,addr Y-IO DMC-write-addr))
  (emit `(STORE ,data Y-IO DMC-write-data)))

(defun QP-read-DRAM (addr data)		;data here is register of some sort
  (emit `(STORE ,addr Y-IO DMC-read-addr))
  (emit `(LOAD ,data Y-IO DMC-read-data)))

(defun QP-refresh-on ()
  (emit '(BSET DMC-refresh Y-IO DMC-MCR)))

(defun QP-refresh-off ()
  (emit '(BCLR DMC-refresh Y-IO DMC-MCR)) ;turn off refresh
  (emit '(refoff))			  ;wait for refresh de-assert
  (emit '(JSET DMC-refresh Y-IO DMC-MCR refoff))
  (emit '(BCLR DMC-select Y-IO DMC-MCR))  ;refresh bit messes with select bit
  (emit '(HIDE refoff)))

(libdecl .QP-initialize-DRAM 'QP-initialize-DRAM-load '() nil)
(defun QP-Initialize-DRAM-load ()
					;see Ariel QP technical manual p9
  (emit '(.QP-Initialize-DRAM))
  (emit '(STORE #x1C Y-IO DMC-MCR))	;master control reg set to reset state (reset DMC)
  (emit '(STORE 1 X-IO M-BCR))		;set IO wait states to 1 (unnecessary? -- master's monitor does this)
  (QP-read-DRAM 0 'X0)			;init DMC by doing a dummy read -- toggles RAS input causing internal init
  (QP-refresh-on)
  (emit '(REP 4095))			;do DRAM wake up cycles
  (emit '(NOP))
  (QP-refresh-off)
  (emit '(BCLR DMC-select Y-IO DMC-MCR))
  (emit '(NOP))				;pipeline settling, I suppose
;  (emit `(STORE ,QP-DRAM-size-code Y-IO DMC-config))
  ;; this ^ worked as long as we had only one QP board -- with two, they might have different DRAM sizes, so
  ;; the decision as to DRAM size has to be deferred until run-time (we might be called by a reverberator, for
  ;; example, so every instrument (that uses delay lines) needs to be able to find the DRAM size somewhere).
  (emit `(MOVE Y QP-DRAM-size-location Y-IO DMC-config))
					;configuration register -- while DMC is in reset state, it
					;expects us to write 3 values to DMC-config, hence the odd
					;looking code here
  (emit '(STORE #xFFFFFF Y-IO DMC-config)) ;mask register
  (emit '(STORE #xFFFFFF Y-IO DMC-config)) ;burst counter
  (emit '(BCLR DMC-reset Y-IO DMC-MCR))	   ;start DMC running (1=reset)
  (if (> dsp-pc internal-P-size) (error "QP-Initialize-DRAM disables DSPRAM, but is being loaded into DSPRAM!"))
  (emit '(BSET DMC-enable Y-IO DMC-MCR))   ;enable DSPRAM P access
  (emit '(NOP))
  (emit '(RTS)))

(defun QP-clear-DRAM-load ()		;clobbers A, X0 and Y0
					;A=data start reg, B=data-end-reg
  (emit '(.QP-clear-DRAM))
  (emit '(LOAD X0 1))
  (emit '(LOAD Y0 0))
  (QP-refresh-off)
  (emit '(init-DO))			;can't use hardware DO because it is 16 bit max
  (emit '(CMP A B))
  (emil '(JLE clear-DRAM))		;not JLT (B=A+size)
  (QP-write-DRAM 'A 'Y0)		;DRAM[reg] <- 0
  (emit '(ADD X0 A))			;reg <- reg + 1
  (emil '(JMP init-DO))
  (emit '(clear-DRAM LOCAL))
  (emit '(HIDE init-DO))
  (QP-refresh-on)
  (emit '(RTS)))

(defun QP-flush-and-clear-DRAM-load () ;clobbers A, X and Y
					;A=start, B=end
  (emit '(.QP-flush-and-clear-DRAM))
  (emit '(LOAD X0 1))
  (emit '(LOAD X1 #x8000))
  (QP-refresh-off)
  (emit '(init-DO))
  (emit '(CMP A B)      '(COPY A Y1))
  (emil '(JLE read-DRAM))
  (QP-read-DRAM 'Y1 'Y0)		;Y0 <- DRAM[Y1]
  (emit '(MPY X1 Y0 A)  '(LOAD X1 0))
  (QP-write-DRAM 'Y1 'X1)		;DRAM[Y1] <- 0
  ;; now we need to send the datum to the host interface.
  ;; since we may have to wait an arbitrary amount of time here,
  ;; we eventually turn on refresh.
  (emit '(JSET M-HTDE X-IO M-HSR sendA))
  (emit '(DO 25 wait-for-host))
  (emit     '(JCLR M-HTDE X-IO M-HSR keep-going))
  (emit     '(ENDDO))
  (emit     '(JMP sendA))		;ENDDO is weird (not equivalent to DONE)
  (emit     '(keep-going LOCAL))
  (emit     '(NOP))
  (emit     '(NOP))
  (emit '(wait-for-host LOCAL))
  (QP-refresh-on)			;if we get here, we've waited n times and still no host
  ;; this slows us down (the refresh on/off shuffle), but is necessary because
  ;; we can be swapped out for any length of time while waiting for the caller
  ;; to be allowed to read what we just wrote.  Without this wait, we occasionally
  ;; get garbage read out of the DRAM.
  (emit '(wait-forever-if-necessary))
  (emit '(JCLR M-HTDE X-IO M-HSR wait-forever-if-necessary))
  (emit '(HIDE wait-forever-if-necessary))
  (QP-refresh-off)
  (emit '(NOP))
  (emit '(sendA LOCAL))
  (emit '(STORE A X-IO M-HTX))
  (emit '(ok LOCAL))
  (emit '(COPY Y1 A))
  (emit '(ADD X0 A)     '(LOAD X1 #x8000))
  (emil '(JMP init-DO))
  (emit '(read-DRAM LOCAL))
  (emit '(HIDE init-DO))
  (QP-refresh-on)
  (emit '(NOP))
  (emit '(RTS)))

(defun QP-shift-DRAM-load ()		;clobbers A, B, X and Y (including X1)
					;A=new start, B=old start, X1=old end
  (emit '(.QP-shift-DRAM))
  (emit '(LOAD X0 1))
  (emit '(LOAD Y1 0))
  (QP-refresh-off)
  (emit '(init-DO))
  (QP-read-DRAM 'B 'Y0)			;Y0<-DRAM[B]
  (QP-write-DRAM 'A 'Y0)		;DRAM[A]<-Y0
  (emit '(ADD X0 A))			;A<-A+1
  (QP-write-DRAM 'B 'Y1)		;DRAM[B]<-0
  (emit '(ADD X0 B))			;B<-B+1
  (emit '(CMP X1 B))			;B<=end?
  (emil '(JLE init-DO))
  (emit '(HIDE init-DO))
  (QP-refresh-on)
  (emit '(NOP))
  (emit '(RTS)))

(defun QP-merge-monitor-load ()

  ;; this is the QP master (DSP) side of the merging communication (for the slave dsp's benefit).
  ;; We are called via a Host command (#x13 I think) = jump here (long interrupt)
  ;; we expect to get the following data via the host interface:
  ;;    DSP-id (0..3=slave)
  ;;    buffer size (all assumed to be the same size)
  ;;    number of buffers
  ;;    for each buffer: 
  ;;        DRAM loc
  ;; All this is packaged up in an array that the master stores in its SRAM, and goes off to merge the buffers.
  ;; Since we have already used all possible HF2 and HF3 combinations, we have to use a kludge to
  ;; tell lisp that the merge is complete -- we will write #o25252525 to the host interface -- this
  ;; means that the 1 bit in ISR coupled with this bit pattern in HTX is our "all done" flag.

  (emit '(.QP-merge-monitor))		;via HC
  ;; that's room for 16 IO channels -- ought to be enough for now.
  (get-datum '(X QP-bufs))
  (emit '(LOAD N0 X QP-bufs))
  (get-datum '(Y QP-size))
  (get-datum '(X QP-dsp))
  (emit '(LOAD R7 QP-Data))
  (emit '(DO N0 load-locs))
  (           get-datum 'A)
  (emit      '(STORE A X R7 R+1))	;DRAM loc
  (emit '(load-locs LOCAL))
  (emit '(LOAD Y0 Y QP-size))		;buffer size
  (emit '(LOAD R0 QP-Data))
  (QP-refresh-off)
  (emit '(LOAD Y1 X QP-dsp))		;dsp-id 
  (emit '(LOAD A #x94e04))		;LOAD A Y-IO M-DatA assembled
  (emit '(ADD Y1 A))			;  add in offset for this slave (i.e. 0 = A and so on)
  (emit '(STORE A P dat-loc))		;  plunk it down in read loop
  (emit '(DO N0 merge-bufs))
  (emit     '(LOAD B X R0 R+1))
					; Y1=dsp-reg (0=0..4=error)
					; B=DRAM location for first word of merge
					; Y0=size of merged buffer
  (emit     '(LOAD X0 1))
  (emit     `(DO Y0 merge-buffer))	
  (emit         `(dat-loc))
  (emit         `(LOAD A Y-IO M-datA))	;this command just a place holder (could be NOP)
  (emit         '(HIDE dat-loc))
  (              QP-read-DRAM 'B 'X1)
  (emit         `(ADD X1 A))
  (              QP-write-DRAM 'B 'A)
  (emit         `(ADD X0 B))
  (emit         '(REP 5))		;see note in QP-instrument-monitor-load!
					;here we have 2+2+2+2+10+10 say 28 without the REP delay
					;  and we drop samples sometimes -- so far 5 cycles is enough.
					;  Each NOP=2 and REP=4, so we're currently running say 40 cycles per word merged.
					;(by the way, the symptom of a dropped sample is that the entire computation
					;hangs because from dsp-ready's point of view no one is ever ready to merge --
					;all the slaves are still in their merge loop waiting for the master to read)
					;REP 3 hangs
  (emit         '(NOP))
  (emit         '(NOP))
  (emit     '(merge-buffer LOCAL))
  (emit     '(NOP))
  (emit '(merge-bufs LOCAL))
  (emit '(NOP))
  (QP-refresh-on)
  ;; now the "all done" signal.
  (emit `(LOAD A ,QP-merge-done))
  (emit '(STORE A X-IO M-HTX))
  (emit '(wait-for-host))
  (emit '(JCLR M-HTDE X-IO M-HSR wait-for-host))
  (emit '(HIDE wait-for-host))
  (emit '(NOP))
  (emit '(RTI)))			;SR is stacked upon interrupt -- contains CCR

(defun QP-DRAM-merge-load ()
  (emit '(.QP-master-flush-and-clear-DRAM))
  (emit '(deal-with-DRAM))
  (get-datum 'B)			;what to do
  (emit '(TST B))			;0=quit, n=end of buffer
  (emil '(JEQ end-frame-move))
  (get-datum 'A)			;start of buffer to flush
  (emit '(JSR .QP-flush-and-clear-DRAM))
  (emil '(JMP deal-with-DRAM))
  (emit '(HIDE deal-with-DRAM))
  (emit '(end-frame-move LOCAL))

  (get-datum 'B)			;now for shift info, if any (0=done)
  (emit '(TST B)    '(COPY B N0))
  (emit '(JEQ all-done))
  (emit '(STORE B X QP-shifts))		;number of buffers to shift
  (emit '(LOAD R0 QP-shift-data))
  (emit '(DO N0 get-shift-data))
  (          get-datum '(X R0 R))	;start of buffer to be shifted 
  (          get-datum '(Y R0 R+1))	;start of destination of block being shifted
  (          get-datum '(X R0 R+1))	;end of block being shifted (current bounds -- B..X1 goes to A..?)
  (emit     '(NOP))			;68000 goes on its way now while we do the shifts
  (emit     '(get-shift-data LOCAL))
  (emit '(LOAD R0 QP-shift-data))
  (emit '(DO N0 do-shifts))
  (emit     '(LOAD B X R0 R))
  (emit     '(LOAD A Y R0 R+1))
  (emit     '(LOAD X1 X R0 R+1))
  (emit     '(JSR .QP-shift-DRAM))
  (emit     '(NOP))
  (emit     '(NOP))
  (emit     '(do-shifts LOCAL))
  (emit `(LOAD A ,QP-merge-done))	;if shifting, need to send "all done" signal (like merging et al)
  (emit '(STORE A X-IO M-HTX))
  (emit '(wait-for-host))
  (emit '(JCLR M-HTDE X-IO M-HSR wait-for-host))
  (emit '(HIDE wait-for-host))

  (emit '(all-done LOCAL))

  (emit '(NOP))
  (emit '(RTI)))

(defun QP-instrument-monitor-load (output)
  (emit '(.QP-DRAM-merge))
  (emit '(STORE R7 X QP-save-R))
  (emit '(STORE N0 X QP-save-N))
  (emit '(BCLR M-HF2 X-IO M-HCR))
  (emit '(STORE R0 Y QP-save-R))
  (emit '(MOVE Y-IO S-Cmd X qp-temp-loc))
  (get-datum '(X QP-sbufs))		;how many buffers to merge?
  (emit '(LOAD N0 X QP-sbufs))
  (get-datum '(Y QP-ssize))		;how big are they>
  (emit '(LOAD R7 QP-sData))		;for each buffer, get SRAM loc and x-or-y bit
  (emit '(DO N0 load-locs))
  (           get-datum 'A)
  (emit      '(STORE A X R7 R))		;SRAM loc
  (           get-datum 'A)
  (emit      '(STORE A Y R7 R+1))	;x-or-y
  (emit      '(NOP))
  (emit '(load-locs LOCAL))
  (emit '(LOAD R0 QP-sData))
  (emit '(LOAD Y0 Y QP-ssize))		;buffer size -- Y0 and Y1 are not changed during the merge
  (emit '(DO N0 merge-bufs))
  (emit     '(LOAD R7 X R0 R))		;SRAM loc
  (emit     '(LOAD A Y R0 R+1))		;x-or-y
					; Y0=size, R7=SRAM loc, A=x-or-y
  (emit     '(TST A))			;this has to be as fast as possible, hence split in two
					;I believe this loop time is about 5+5+6+2=say 20 cycles
					;due to the lack of a handshake flag for the master to read
					;saying "slave has written data", this loop better be a lot
					;faster than the corresponding one in absorb-slave-buffer.
  (emil     '(JEQ x-mem))
  (emit     '(DO Y0 cache-y-buffer))
  (emit         '(MOVE Y R7 R+1 Y-IO S-Dat))
  (emit         '(MOVE Y-IO S-Cmd X qp-temp-loc))
  (emit         '(wait-for-master))
  (emit         '(JCLR 22 Y-IO S-Cmd wait-for-master))
  (emit         '(HIDE wait-for-master))
  (emit         '(NOP))
  (emit         '(NOP))
  (emit     '(cache-y-buffer LOCAL))
  (emit     '(NOP))
  (emit     '(JMP next-buffer))
  (emit     '(x-mem LOCAL))
  (emit     '(DO Y0 cache-x-buffer))
  (emit         '(MOVE X R7 R+1 Y-IO S-Dat))
  (emit         '(MOVE Y-IO S-Cmd X qp-temp-loc))
  (emit         '(wait-for-master))
  (emit         '(JCLR 22 Y-IO S-Cmd wait-for-master))
  (emit         '(HIDE wait-for-master))
  (emit         '(NOP))
  (emit         '(NOP))
  (emit     '(cache-x-buffer LOCAL))
  (emit     '(NOP))
  (emit     '(next-buffer LOCAL))
  (emit     '(NOP))
  (emit '(merge-bufs LOCAL))
  (emit '(LOAD R7 X QP-save-R))
  (emit '(LOAD R0 Y QP-save-R))
  (emit '(LOAD N0 X QP-save-N))
  ;; now reset all buffer pointers to start
  (emit '(CLR B)  `(LOAD R5 X ,output))	;R5=>address of first out-n structure on the list
  (if (> output 63) (error "output location in QP-merge has to fit in 6-bits: ~D" output))
  (emit `(DO Y ,output all-bufs))	;Y <output> = number of output streams open
  (emit     '(CLR A)  '(LOAD R4 X R5 R+1))	;R4->out-n structure base, R5->next out-n struct
  (emit     '(NOP))			;let pipeline settle on R4
  (emit     '(LOAD A X R4 R+1))		;check for inactive streams
  (emit     '(TST A))
  (emil     '(JEQ no-data))		;can't jump to LA (DO restriction)
  (emit     '(UPDATE R4 R-1))
  (emit     '(LOAD A X R4 R+1))		;get buf start (to reset curptr)
  (emit     '(no-data LOCAL))
  (emit     '(STORE A X R4 R))		;now buffer is ready to go on (writes 0 to bufstart in dummy case of no data)
  (emit '(all-bufs LOCAL))
  (emit '(NOP))
  (emit '(RTS)))


(defun QP-monitor-load ()		;for QP-MONITOR in qpins.lisp
  (if (or (null (gethash '.QP-Initialize-DRAM labels))
	  (eq 'UNDEFINED (car (gethash '.QP-Initialize-DRAM labels))) )
      (QP-Initialize-DRAM-load))	;sometimes loaded by delay lines et al
					;required below, so whether or not required up to here, we will need it
					;need to load first because it must execute out of internal P mem 
					; (it turns off external SRAM while initializing DRAM)
  (QP-merge-monitor-load)		;destination of HC vector
  (QP-DRAM-merge-load)
  (QP-clear-DRAM-load)
  (QP-shift-DRAM-load)
  (QP-flush-and-clear-DRAM-load))


;;; now the lisp side of the transaction

#|
  "DRAM protocol" (vote for names!) is:
      any dsp runs until output buffer fills up. 
         waits at .flush-buffers for: 
           size of buffer
           if size>#x10000, go to DRAM merge
           else get buffer locations from <output> struct pointer on chip and send to host
  
  DRAM-merge:
    
  .QP-DRAM-merge
      op code: 0=just merge, n=buffer end to flush and clear, -n=buffer end just to clear
               if n or -n: start of buffer to flush-and-clear and loop back to op code
      (so this sequence has to have a final 0 as a stop signal)

  .QP-merge-monitor (master side):
      number of buffers
      buffer size
      which dsp (0..3)
      loop for each buffer: 
           DRAM loc
      ... merges
      sends back all-done to HTX

  .QP-slave-merge-monitor (if not master):
      number of buffers
      buffer size
      loop for each buffer:
          SRAM loc
          x-or-y

  .QP-shift-DRAM wants A=old beg, B=new beg, X1=size
|#

(defun QP-in-current-DRAM-frame (ld-ptr QP-ptr)
  (if (not (dsp-data-p qp-ptr))
      (error "incorrect pointer type"))
  (let ((m (dsp-data-hub qp-ptr)))
    (and (>= (ld-curbeg ld-ptr) (QP-data-fbeg m))
	 ;; these are sample numbers, not seconds
	 (<= (ld-curend ld-ptr) (QP-data-fend m)))))

(defun QP-master-is-done-merging (master)
  ;; if master has written  QP-merge-done to HTX, we reset his flag to qp-normal
  (if (/= qp-merging (dsp-data-state master))
      (error "why check for merge done when not merging?"))
  (make-active-dsp master)
  (let ((val 0))
    (when (and (logbitp 0 (dsp-read-ISR))
	       (or (= (setf val (dsp-get-one-word)) QP-merge-done)
		   (error "We just read something unexpected (~A) from the QP master" val)))
      (let ((dsp (qp-data-current-dsp (dsp-data-hub master))))
	(if (not dsp) (error "QP merge done, but no dsp was merging!"))
	(let ((act-dsp (dsp-data-in-use-by dsp)))
	  (setf (dsp-data-state dsp) dsp-normal)
	  (if (>= (ld-curbeg act-dsp) (ld-end act-dsp))
	      (finish-dsp act-dsp dsp))
	  (setf (qp-data-current-dsp (dsp-data-hub master)) nil)))
      (setf (dsp-data-state master) qp-normal)))
  (= (dsp-data-state master) qp-normal))

(defun QP-master-is-done-initializing (master)
  ;; if master has written  QP-merge-done to HTX, we reset his flag to qp-normal
  (make-active-dsp master)
  (let ((val 0))
    (when (and (logbitp 0 (dsp-read-ISR))
	       (or (= (setf val (dsp-get-one-word)) QP-merge-done)
		   (error "We just read something unexpected (~A) from the QP master" val)))
      (setf (dsp-data-state master) qp-normal)))
  (= (dsp-data-state master) qp-normal))

(defun QP-master-is-done-shifting (master) 
  (QP-master-is-done-initializing master))

(defun QP-no-one-can-use-current-frame (master)
  ;; if dsp-data-hub (aref dsp-pool i) (up to max-dsp) = QP-ptr
  ;; then if running (dsp-data-in-use-by is not nil)
  ;;         then if in current bounds, return false
  ;;         set current-least-beg to min 
  ;;  return (at least one QP dsp is running, and none of those running can use the current DRAM frame)
  (let* ((dspA (- (dsp-data-pool master) 4))
	 (dspD (+ dspA 3)))
    (do ((i dspA (1+ i))
	 (someone-is-happy nil))
	((or someone-is-happy 
	     (> i dspD))
	 (not someone-is-happy))
      (let* ((dsp-ptr (aref dsp-pool i))
	     (ld-ptr (and (dsp-at-work dsp-ptr) (dsp-data-in-use-by dsp-ptr))))
	(when ld-ptr
	  (setf someone-is-happy 
	    (or (= dsp-ready-to-merge (dsp-data-state dsp-ptr))
		(QP-in-current-DRAM-frame ld-ptr master))))))))
#|
  (do ((i 1 (1+ i))
       (someone-is-happy nil))
      ((or someone-is-happy 
	   (> i max-dsp))
       (not someone-is-happy))
    (let* ((dsp-ptr (aref dsp-pool i))
	   (qp-data-ptr (dsp-data-hub dsp-ptr))
	   (qp-data-m (dsp-data-hub master))
	   (ld-ptr (and (dsp-at-work dsp-ptr) (dsp-data-in-use-by dsp-ptr))))
      (if (or (not (qp-data-p qp-data-ptr))
	      (not (qp-data-p qp-data-m))
	      (not (dsp-data-p dsp-ptr))
	      (not (dsp-data-p master)))
	  (error "incorrect pointer type"))
      (when (and ld-ptr			;it's running
		 (eq qp-data-m qp-data-ptr)) ;it's on this QP board
	(setf someone-is-happy 
	  (or (= dsp-ready-to-merge (dsp-data-state dsp-ptr))
	      (QP-in-current-DRAM-frame ld-ptr master)))))))
|#

(defun QP-get-least-beg (master &optional (beg 0.0))
  (let* ((least-beg beg)
	 (dspA (- (dsp-data-pool master) 4))
	 (dspD (+ dspA 3)))
    (do ((i dspA (1+ i)))
	((> i dspD)
	 least-beg)
      (let* ((dsp-ptr (aref dsp-pool i))
	     (ld-ptr (and (dsp-at-work dsp-ptr) (dsp-data-in-use-by dsp-ptr))))
	(when ld-ptr
	  (setf least-beg (min least-beg (ld-curbeg ld-ptr))))))))
#|
    (do ((i 1 (1+ i)))
	((> i max-dsp))
      (let* ((dsp-ptr (aref dsp-pool i))
	     (qp-data-ptr (dsp-data-hub dsp-ptr))
	     (qp-data-m (dsp-data-hub master))
	     (ld-ptr (and (dsp-at-work dsp-ptr) (dsp-data-in-use-by dsp-ptr))))
	(if (or (not (qp-data-p qp-data-ptr))
		(not (qp-data-p qp-data-m))
		(not (dsp-data-p dsp-ptr))
		(not (dsp-data-p master)))
	    (error "incorrect pointer type"))
	(when (and ld-ptr		;it's running
		   (eq qp-data-m qp-data-ptr)) ;it's on this QP board
	  (setf least-beg (min least-beg (ld-curbeg ld-ptr))))))
    least-beg))
|#


;;; The run-time index to the associated output IO buffer is in sigadrs
;;; SRAM location is in ld-SRAM-locs paralleling sigops et al
;;; X-or-Y is in ld-x-or-ys parallel to sram-locs
;;; DRAM locs need to be set up by fire-up-merging-monitor (iolocs in qp-data)
;;; map between DRAM and SRAM data arrays is via sigadrs and DRAM-adrs

(defun QP-save-IO-data (ptr x-or-y j srams xs)
  (setf (aref srams j) ptr)
  (setf (aref xs j) x-or-y))

(defun QP-find-DRAM-loc (adr QP-ptr)
  (loop for i from 0 below (QP-data-active-bufs QP-ptr) do
    (if (= (aref (QP-data-iolocs QP-ptr) i) adr) 
	(return-from QP-find-DRAM-loc (aref (QP-data-DRAM-adrs QP-ptr) i)))))

(defun QP-send-slave-IO-info (master slave cursiz)
  (let* ((QP-ptr (dsp-data-hub master))
	 (ld-ptr (dsp-data-in-use-by slave))
	 (signum (ld-len ld-ptr))
	 (bufnum (QP-data-active-bufs QP-ptr)))
    (if (eq slave built-in-dsp) (error "attempt to merge built-in dsp into QP DRAM."))
    (if (eq slave master) (error "attempt to treat master dsp as a slave during merge."))
    (make-active-dsp slave)
    (dsp-put-one-word (+ #x10000 (dsp-data-dsp slave)))
					;    (dsp-put-one-word bufnum)
    (dsp-put-one-word (min signum bufnum)) 
					;otherwise we hang attempting to read a non-existent output buffer
					;if (for example) some instruments have reverb output and others don't.
					;We can't use signum because it is a max of possible active streams.
					;(i.e. LEN=3 if stereo/rev possible, but only one in use BUFNUM=1.
					;but, LEN=2 if no rev, but BUFNUM=3 if overall is stereo/rev)
    (dsp-put-one-word cursiz)
    (loop for i from 0 below signum do
      (when (= (aref (ld-sigops ld-ptr) i) 0)
	(dsp-put-one-word (aref (ld-SRAM-locs ld-ptr) i))
	(dsp-put-one-word (aref (ld-x-or-ys ld-ptr) i))))
    (setf (dsp-data-state slave) dsp-merging)))

(defun QP-send-master-IO-info (master slave cursiz)
  (let* ((QP-ptr (dsp-data-hub master))
	 (ld-ptr (dsp-data-in-use-by slave))
	 (dsp-number (dsp-data-dsp slave))
	 (offset (floor (- (ld-curbeg ld-ptr) (qp-data-fbeg qp-ptr)))) ; where are we within each DRAM buffer
	 (signum (ld-len ld-ptr))
	 (bufnum (QP-data-active-bufs QP-ptr)))
    (if (minusp offset) 
	(error "DRAM buffer start = ~D, but current beg = ~D?" (ld-curbeg ld-ptr) (qp-data-fbeg qp-ptr)))
    (if (or (> offset (qp-data-buf-size qp-ptr))
	    (> (ld-curend ld-ptr) (qp-data-fend qp-ptr)))
	(error "DRAM buffer end: ~D, but merge request goes to: ~D (~D)?" (qp-data-fend qp-ptr) (ld-curend ld-ptr) offset))
    (setf (qp-data-fdata-start qp-ptr) (min (qp-data-fdata-start qp-ptr) offset))
    (if (/= qp-normal (dsp-data-state master))
	(error "QP master is not ready to merge"))
    (if (qp-data-current-dsp qp-ptr)
	(error "QP master apparently never finished its previous merge."))
    (setf (qp-data-current-dsp qp-ptr) slave)
    (make-active-dsp master)
    (dsp-write-CVR #x93)		;#x93=turn on HC bit and set #x26 as vector location (JSR .QP-merge-monitor)
					;    (dsp-put-one-word bufnum)
    (dsp-put-one-word (min bufnum signum)) ;see note above.
    (dsp-put-one-word cursiz)
    (dsp-put-one-word dsp-number)
    (loop for i from 0 below signum do
      (when (= (aref (ld-sigops ld-ptr) i) 0)
	(dsp-put-one-word (floor (+ (QP-find-DRAM-loc (aref (ld-sigadr ld-ptr) i) QP-ptr) offset)))))
    (setf (dsp-data-state master) qp-merging)))
 
(defvar qp-merger-data nil)
(defvar qp-trigger .25)			;determines when we decide to move the current DRAM boundaries

(defun QP-fire-up-merging-monitor (master &optional (no-output-buffers nil))
  ;; start up a monitor on the master, allocate and clear buffers in DRAM, set up initial frame bounds
  ;; allocate qp-data arrays and set qp-data-active-bufs
  (let* ((m (dsp-data-hub master))
	 (proglen (get 'QP-MONITOR :dsp-compiled-program-length))
	 (DRAM-size (QP-data-DRAM-size m)))
    (if (or (not (numberp proglen)) (< proglen 60))
	(error "QP monitor is messed up"))
    (if (< monitor-end 60) (error "QP-MONITOR not loaded?"))
    (setf (dsp-data-in-use-by master) (make-ld :dsp master))
    (setf (dsp-data-state master) qp-initializing)
    (make-active-dsp master)
    (setf (dsp-data-ins-name master) 'QP-MONITOR)
    (setf (dsp-data-open master) t)
    (set-initial-contents 'QP-MONITOR
			  (get 'QP-MONITOR :dsp-ix)
			  (get 'QP-MONITOR :dsp-iy)
			  (get 'QP-MONITOR :dsp-ex)
			  (get 'QP-MONITOR :dsp-ey) 0)
    (setf (aref internal-x-memory 0) (* 1024 DRAM-size))
    (setf (aref internal-y-memory QP-size-loc) (if (= DRAM-size 256) QP-256K (if (= DRAM-size 1000) QP-1M QP-4M)))
    (if (zerop (dsp-open (1- (get 'QP-MONITOR :dsp-compiled-program-length))
			 (get 'QP-MONITOR :dsp-compiled-program)
			 (dsp-data-slot master) 
			 (dsp-data-dsp master) 
			 monitor-end 
			 (dsp-data-mem-map master)))
	(if (not (zerop (dsp-set-up-program ix-ptr iy-ptr ex-ptr ey-ptr 0 ext-x-offset external-y-from-x-offset 0
					    internal-x-memory
					    internal-y-memory
					    external-memory)))
	    (error "can't set up QP merge monitor"))
      (error "can't boot QP merge monitor"))
    (let* ((buffers (clm-count-active-output-buffers))
	   (DRAM-size (* 1024 DRAM-size))
	   (output-sigadrs (clm-collect-output-buffer-addresses buffers no-output-buffers)))
      (setf (QP-data-active-bufs m) buffers)
      (setf (QP-data-fbeg m) 0)
      (setf (QP-data-buf-size m) (if (plusp buffers) (floor DRAM-size buffers) 0))
      (setf (QP-data-trigger-point m) (floor (* qp-trigger (qp-data-buf-size m))))
      (setf (QP-data-fend m) (1- (QP-data-buf-size m)))
      (setf (QP-data-fdata-end m) 0)
      (setf (QP-data-fdata-start m) (QP-data-buf-size m))
      (setf (QP-data-current-dsp m) nil)
      (setf (QP-data-iolocs m) (make-array buffers :element-type 'fixnum :initial-element 0))
      (setf (QP-data-DRAM-adrs m) (make-array buffers :element-type 'fixnum :initial-element 0))
      (loop for i from 0 below buffers do
	(setf (aref (QP-data-iolocs m) i) (aref output-sigadrs i))
	(setf (aref (QP-data-DRAM-adrs m) i) (* i (QP-data-buf-size m)))))))

(defun QP-merge (curdsp curld cursiz &optional (beg 0))
  ;; don't assume curdsp's data buffers fit its associated master's DRAM frame
  ;; this is the lisp side of the affair (i.e. sending data to master chip)
  (let* ((qp-ptr (or (dsp-data-hub curdsp) 
		     (error "attempt to merge, but no merge monitor!")))
	 (master (or (qp-data-master qp-ptr) 
		     (error "can't find master dsp!")))
	 (ldbeg (ld-curbeg curld))
	 (ldend (ld-curend curld))
	 (ldsiz (- ldend ldbeg)))
    (if (or (/= qp-normal (dsp-data-state master))
	    (/= dsp-ready-to-merge (dsp-data-state curdsp)))
	(error "odd states in merge ~A ~A" curdsp master))
    (if (or (< ldbeg (qp-data-fbeg qp-ptr))
	    (> ldend (qp-data-fend qp-ptr)))
	(if (<= ldsiz (qp-data-buf-size qp-ptr))
	    (let ((cbeg (qp-get-least-beg master beg)))
;(print (format nil "c: ~D, l: ~D" cbeg ldbeg))
	      (if (> ldend (+ cbeg (qp-data-buf-size qp-ptr)))
		  (qp-merge-new-dram-frame master ldbeg)
		(qp-merge-new-dram-frame master cbeg)))
	  (error "Merge request is too big: ~D to ~D (~D), but DRAM buffer size = ~D" 
		 ldbeg ldend (1+ ldsiz) (qp-data-buf-size qp-ptr))))
    (when (/= qp-shifting (dsp-data-state master))
;      (incf qp-grabs)
      (QP-send-slave-IO-info master curdsp cursiz)
      (QP-send-master-IO-info master curdsp cursiz)
      ;; fdata-start set in QP-send-master-IO-info
      (setf (qp-data-fdata-end qp-ptr) (max (qp-data-fdata-end qp-ptr) 
					    (- (ld-curend curld) (qp-data-fbeg qp-ptr))))
      (update-dsp curld curdsp)
      curld)))				;grab-dsp-data wants non-nil for success 

(defun QP-merge-one-buffer (start stop arr)
  (dsp-merge-one-buffer start stop arr))

(defun QP-merge-all-buffers (master qp-ptr &optional (new-beg 0))
  ;; if new-beg is not 0, we try to send out only the data below new-beg in DRAM, and shift the rest.
  ;; I believe the shift process is about 5 times as fast as passing the stuff through the host interface.
  (if (and (= qp-shifting (dsp-data-state master))
	   (not (qp-master-is-done-shifting master)))
      (loop until (or (= qp-normal (dsp-data-state master))
		      (qp-master-is-done-shifting master))))
  (if (/= qp-normal (dsp-data-state master))
      (error "attempt to dump DRAM while master is trying to merge?"))
  (let* ((data-end (qp-data-fdata-end qp-ptr))
	 (data-start (qp-data-fdata-start qp-ptr)))
    ;; four possible paths here -- flush data and shift data in any combination (including neither)
    (if (> data-end data-start)
      ;; there is data in DRAM (could have been flushed via QP-flush and therefore DRAM is empty)
      ;; if there is no data, we should return 0 => no data left in DRAM
	(let ((flush-end (if (and (plusp new-beg) 
				  (> new-beg (qp-data-fbeg qp-ptr)))
			     ;; we don't shift forwards yet (assume we normally move forwards in score)
			     (min data-end (- new-beg (qp-data-fbeg qp-ptr)))
			   data-end)))
					; (setf flush-end data-end) turns off shifting
	  (make-active-dsp master)
	  (dsp-write-CVR #x94)
	  (if (> flush-end data-start)
	      ;; is there data to flush?
	      (let ((curbeg (+ (qp-data-fbeg qp-ptr) data-start)))
		(if (/= curbeg (io-beg *current-output-file*))
		    (clm-reset-all-files curbeg 0 t))
		(loop for i from data-start below flush-end by file-buffer-size do
		  (let ((cursize (min file-buffer-size (1+ (- flush-end i)))))
		    (loop for j from 0 below (qp-data-active-bufs qp-ptr) do
		      (let ((curadr (aref (qp-data-DRAM-adrs qp-ptr) j)))
			(dsp-put-one-word (+ curadr i cursize))
			(dsp-put-one-word (+ curadr i))
			(qp-merge-one-buffer 0 (1- cursize) (aref (qp-data-iolocs qp-ptr) j))))
		    (incf curbeg cursize)
		    (clm-flush-all-files 0 (1- cursize))
		    (if (= cursize file-buffer-size)
			(clm-reset-all-files curbeg 0 t))))))
	  (dsp-put-one-word 0)		;no more buffers to merge
	  ;; now check for shiftable data
	  (if (> data-end flush-end)
	      (progn
		(dsp-put-one-word (qp-data-active-bufs qp-ptr))
		(loop for j from 0 below (qp-data-active-bufs qp-ptr) do
		  (let ((curadr (aref (qp-data-DRAM-adrs qp-ptr) j)))
		    (dsp-put-one-word (+ curadr flush-end))
		    (dsp-put-one-word curadr)
		    (dsp-put-one-word (+ curadr data-end))))
		(setf (dsp-data-state master) qp-shifting))
	    (dsp-put-one-word 0))	;end of shift
	  (return-from QP-merge-all-buffers (- data-end flush-end)))
      0)))				;new data end

(defvar min-flush-size 128)		;minimum amount we'll go to the trouble of flushing
(defvar max-flush-size 512)		;maximum for otherwise idle time flush

(defun qp-flush (m beg)
  (when (= qp-normal (dsp-data-state m))

    ;; if we're close to the current data end (i.e. fdata-end - fdata-start < file-buffer-size)
    ;; and (min current-least-beg global-beg) > fbeg (so there is wasted space in the current DRAM buffer)
    ;; then we should just flush the current contents, and move the entire DRAM frame.
    ;; Otherwise, if there's at least min-flush-size ready to go out (i.e. global-beg and current-least-min
    ;; have passed fdata-start by that amount), then flush it out (up to max-flush-size) and move 
    ;; fdata-start forward (if notes are ordered by begin time, which is the normal case, this will
    ;; get the file IO happening in parallel with everything else, rather than stopping for a huge
    ;; flush when no one can use the current DRAM frame (flushing all of DRAM can take about 10 to 15 secs!)

    ;; Of course, none of this can happen if the qp master is currently doing something else. 

    (let* ((q (dsp-data-hub m))
	   (cbeg (qp-get-least-beg m (if beg beg 0)))
	   (ds (+ (qp-data-fdata-start q) min-flush-size))
	   (lbeg (- cbeg (qp-data-fbeg q)))
	   (data (- lbeg (qp-data-fdata-start q))))
      (when (and (>= data min-flush-size)
		 (> (qp-data-fdata-end q) ds))
	;; we have enough data to be interesting, and we actually have data (sigh)

	(if (> lbeg (qp-data-trigger-point q))
	    (qp-trigger-new-dram-frame m cbeg)

	  (let* ((curbeg (+ (qp-data-fbeg q) (qp-data-fdata-start q)))
		 (curend (+ curbeg max-flush-size))
		 (cpu-ld (dsp-data-in-use-by built-in-dsp))
		 (filbeg (io-beg *current-output-file*))
		 (filend (io-end *current-output-file*)))
	    ;; now do we grab one little buffer and update fdata-start, or do we get greedy and grab everything,
	    ;; resetting all DRAM boundaries as we exit?
	    (when (or (and (>= curbeg filbeg)
			   (<= curend filend))
		      (not cpu-ld)	;maybe built-in dsp not in use
		      (not (in-frame cpu-ld)))
	      (if (or (< curbeg filbeg)
		      (> curend filend))
		  (clm-reset-all-files curbeg 0 t))
	      (let* ((DRAM-start (qp-data-fdata-start q))
		     (newfilbeg (io-beg *current-output-file*))
		     (file-start (- curbeg newfilbeg))
		     (cursize (min data
				   max-flush-size
				   (- file-buffer-size file-start)))
		     (file-stop (+ file-start cursize -1)))
		(make-active-dsp m)
		(dsp-write-CVR #x94)
		(loop for j from 0 below (qp-data-active-bufs q) do
		  (let ((curadr (aref (qp-data-DRAM-adrs q) j)))
		    (dsp-put-one-word (+ curadr DRAM-start cursize))
		    (dsp-put-one-word (+ curadr DRAM-start))
		    (qp-merge-one-buffer file-start file-stop (aref (qp-data-iolocs q) j))))
		(dsp-put-one-word 0)	;end merge
		(dsp-put-one-word 0)	;no shifts here
		(clm-flush-all-files file-start file-stop)
		(incf (qp-data-fdata-start q) cursize)))))))))
    
(defun QP-cleanup-1 (master)
  (let ((qp-ptr (dsp-data-hub master)))
  ;; flush out last buffers (up to current fdata-end)
    (when (= (dsp-data-state master) qp-merging) ;gotta wait for merge to finish
      (print "hmm...QP master is still trying to merge -- will humor it for awhile")
      (do ((i 0 (1+ i))) 
	  ((QP-master-is-done-merging master))
	(when (> i 10000)
	  (error "waiting for QP merge..."))))
    (when (plusp (qp-data-fdata-end qp-ptr))
      (make-active-dsp master)
      (QP-merge-all-buffers master qp-ptr))
    (qp-reset-cleanup master qp-ptr)))

(defun qp-reset-cleanup (master qp-ptr)
  (setf (qp-data-iolocs qp-ptr) nil)
  (setf (qp-data-fend qp-ptr) -1)
  (setf (qp-data-fdata-end qp-ptr) 0)
  (setf (qp-data-fdata-start qp-ptr) 0)
  (setf (qp-data-current-dsp qp-ptr) nil)
  (setf (dsp-data-state master) qp-uninitialized))

(defun QP-new-DRAM-frame (master new-beg)
  ;; flush and clear current DRAM (to fdata-end), use new-beg to choose new boundaries
  ;; shift and clear top would be better (call .QP-shift-DRAM then .QP-clear-DRAM ?)
  ;; new-beg is a sample number, not time in seconds
  (if (= qp-merging (dsp-data-state master)) 
      (error "attempt to move DRAM boundaries while merge in progress"))
  (let* ((m (dsp-data-hub master))
	 (new-end (QP-merge-all-buffers master m new-beg)))
    (setf (QP-data-fbeg m) new-beg)
    (setf (QP-data-fdata-start m) (if (zerop new-end) (QP-data-buf-size m) 0))
    (setf (QP-data-fend m) (1- (+ new-beg (QP-data-buf-size m))))
    (setf (QP-data-fdata-end m) new-end)))

(defun qp-merge-new-dram-frame (master new-beg) 
  (qp-new-dram-frame master new-beg))

(defun qp-trigger-new-dram-frame (master new-beg) 
  (qp-new-dram-frame master new-beg))

(defun QP-Make-Monitor ()
  (setf label-hook 'save-labels)
  (setf label-list nil)
  (initialize-everything_56)
  (setf monitor-end (- (get-dsp-monitor t) 4))
  (Qp-initialize-temps)
  (emit '(.User-Start))
  (emit '(CLR A)   '(LOAD B 0 SHORT))
  (load-chip-data)			;possibly monitor bigger than 512 words
  (emit '(JSR .QP-initialize-DRAM))
  ;; while here we should mask off interrupts, or whatever to keep the host from trying
  ;; to start a merge prematurely (with 4MW this clear-DRAM might take 4 seconds (?!))
  (emit '(CLR A)   `(LOAD B X 0))	;1024*DRAM-Size into X 0 at run-time
  (emit '(JSR .QP-clear-DRAM))
  ;; now the "all done" signal.
  (emit `(LOAD A ,QP-merge-done))
  (emit '(STORE A X-IO M-HTX))
  (emit '(wait-for-host))		;don't jump to .break until host reads QP-merge-done signal
  (emit '(JCLR M-HTDE X-IO M-HSR wait-for-host))
  (emit '(HIDE wait-for-host))
  (emit '(.new-break))
  (emit '(JMP .new-break))	
  (emit '(JMP .new-break))
  (QP-monitor-load)
  (load-library-routines library-load-list)
  (check-for-undefined-labels)
  (DEBUGGING (setf emit-prog (nreverse emit-prog)))
  (DEBUGGING (pprint (append '(case) emit-prog)))
  (DEBUGGING (pprint (append (list 'progn) (reverse pup))))
  (setf qp-merger-data (qp-io-names)))



;;; QP DEBUGGING STUFF

(defun qp-io-names ()
  (if (gethash 'QP-bufs names)
      (list (or (gethash 'QP-bufs names) 0) 
	    (or (gethash 'QP-size names) 0)
	    (or (gethash 'QP-dsp names) 0)
	    (or (gethash 'QP-data names) 0)
	    (or (gethash 'QP-sbufs names) 0)
	    (or (gethash 'QP-ssize names) 0)
	    (or (gethash 'QP-sdata names) 0))))

(defun qp-io-stuff (&optional bufs size dsp data sbufs ssize sdata)
  (if bufs
      (let ((mbufs (dm-x bufs))
	    (msize (dm-y size))
	    (mdsp (dm-x dsp))
	    (sbufs (dm-x sbufs))
	    (ssize (dm-y ssize)))
	(princ (format nil "~%QP master: bufs: ~A, size: ~A, last dsp: ~A, ~
                            ~%    data: ~{~A:~A to ~A in DRAM ~} ~
                            ~%QP slave:  bufs: ~A, size: ~A ~
                            ~%    data: ~{~A:~A ~}"
		       mbufs msize mdsp
		       (if (and (< mbufs 16)
				(< msize #x10000))
			   (loop for i from 0 below mbufs and j from data by 2
			    append (list (if (plusp (dm-y j)) "Y" "X") (dm-x j) (dm-x (+ j 1))))
			 (list "ignored" (if (< mbufs 16) " bad buffer size" " bad buffer number") "no place"))
		       sbufs ssize
		       (if (and (< sbufs 16)
				(< ssize #x10000))
			   (loop for i from 0 below sbufs and j from sdata
			    append (list (if (plusp (dm-y j)) "Y" "X") (dm-x j)))
			 (list "ignored" (if (< sbufs 16) " bad buffer size" " bad buffer number"))))))
    (princ "odd -- no QP data")))
  
(defun qp-io-master ()
  (dsp-debug 2 4)
  (apply 'qp-io-stuff qp-merger-data)
  (dsp-close))


;;; QP stuff

(defun QP-MCR (&optional MCR-1)
  (let ((MCR (if MCR-1 MCR-1 (dsp-mem 'Y-IO #xFFFA))))
    (format nil "MCR: ~B, ~A~A~A~A~A~A~A~A"
	    MCR
	    (if (= MCR #x1C) "(reset state) " "")
	    (if (logbitp 0 MCR) "0: assert host interrupt!?! " "")
	    (if (logbitp 1 MCR) "" "1: DSPRAM P access off! ")
	    (if (logbitp 2 MCR) "" "2: DMC selected!?! ")
	    (if (logbitp 3 MCR) "3: SCSI reset " "3: SCSI not reset! ")
	    (if (logbitp 4 MCR) "4: DMC reset " "4: DMC running ")
	    (if (logbitp 6 MCR) "6: Timer running! " "")
	    (if (logbitp 7 MCR) "7: refresh on" "7: refresh off!"))))

(defun QP-Slave-Interrupt ()
  (let ((SIR (dsp-mem 'Y-IO #xFFF8))
	(SSR (dsp-mem 'Y-IO #xFFFB))
	(SIP (dsp-mem 'Y-IO #xFFFD)))
    (format nil "Slave Interrupt: ~B~A ~A~A~A~A~A~A~A~A  Interrupt Status: ~B  Interrupt Priority: ~B"
	    SIR
	    (if (/= 0 (logand SIR #xFF)) ":" "")
	    (if (logbitp 0 SIR) "0: A transmit " "")
	    (if (logbitp 1 SIR) "1: A receive " "")
	    (if (logbitp 2 SIR) "2: B transmit " "")
	    (if (logbitp 3 SIR) "3: B receive " "")
	    (if (logbitp 4 SIR) "4: C transmit " "")
	    (if (logbitp 5 SIR) "5: C receive " "")
	    (if (logbitp 6 SIR) "6: D transmit " "")
	    (if (logbitp 7 SIR) "7: D receive " "")
	    SSR
	    SIP)))

(defun QP-Master-Interrupt ()
  (let ((MIR (logand (dsp-mem 'Y-IO #xFFF9) #xF))
	(MSR (dsp-mem 'Y-IO #xFFFC))
	(MIP (dsp-mem 'Y-IO #xFFFE)))
    (format nil "Master Interrupt: ~B~A ~A~A~A~A  Misc Interrupt Status: ~B  Misc Interrupt Priority: ~B"
	    MIR
	    (if (/= 0 MIR) ":" "")
	    (if (logbitp 0 MIR) "SCSI INT " "")
	    (if (logbitp 1 MIR) "SCSI DRQ " "")
	    (if (logbitp 2 MIR) "DMC-EBM " "")
	    (if (logbitp 3 MIR) "TIMER" "")
	    MSR
	    MIP)))

(defun QP-MISC ()
  (let ((MISC (logand (dsp-mem 'Y-IO #xFFFF) #x3F)))
    (format nil "Misc: ~B~A ~A~A~A~A~A"
	    MISC
	    (if (/= 0 MISC) ":" "")
	    (if (logbitp 0 MISC) "DRAM cycle in progress " "")
	    (if (logbitp 0 MISC) (if (logbitp 1 MISC) "read " "write ") "")
	    (if (logbitp 2 MISC) "SIMULT " "")
	    (if (logbitp 3 MISC) "RESERVED " "")
	    (if (logbitp 4 MISC)
		(if (logbitp 5 MISC) "256K" "1M")
	      (if (logbitp 5 MISC) "4M" "unused (i.e. bogus) DRAM size")))))

(defun QP-M-S ()
  (format nil 
	  "Slave ports (CMD read/write, DATA read/write): ~&    A: ~X  ~X~&    B: ~X  ~X~&    C: ~X  ~X~&    D: ~X  ~X"
	  (dsp-mem 'Y-IO #xFFC0)
	  (dsp-mem 'Y-IO #xFFC4)
	  (dsp-mem 'Y-IO #xFFC1)
	  (dsp-mem 'Y-IO #xFFC5)
	  (dsp-mem 'Y-IO #xFFC2)
	  (dsp-mem 'Y-IO #xFFC6)
	  (dsp-mem 'Y-IO #xFFC3)
	  (dsp-mem 'Y-IO #xFFC7)))

(defun QP-S-M ()
  (let ((cmd (dsp-mem 'Y-IO #xFFC1)))
    (format nil "Data port: ~X, Command port: ~X, Status port: ~X (~A~A~A~A)" 
	    (dsp-mem 'Y-IO #xFFC0) 
	    (logand cmd #xF)
	    (logand cmd #xF00000)
	    (if (logbitp 20 cmd) "RESERVED, " "")
	    (if (logbitp 21 cmd) "SIMULT, " "")
	    (if (logbitp 22 cmd) "ok to write, " "don't write, ")
	    (if (logbitp 23 cmd) "ok to read" "nothing to read"))))
		  

(defun QPRD (x) 
  (if (eq x 'A) (print "A clobbered by dsp-mem -- can't be used as argument to QPRD"))
  (let ((MCR (dsp-mem 'Y-IO #xFFFA)))
    (when (logbitp 4 MCR) 
      (ex '(JSR .QP-initialize-dram))
      (print "had to initialize DRAM"))
    (ex '(BCLR DMC-refresh Y-IO DMC-MCR))
    (ex '(BCLR DMC-select Y-IO DMC-MCR))
    (ex `(STORE ,x Y-IO #xFFE0))
    (ex `(LOAD A Y-IO #xFFE2))
    (ex '(BSET DMC-refresh Y-IO DMC-MCR))
    (dsp-reg 'A1)))

(defun QPWD (x y) 
  (if (or (eq x 'A) (eq y 'A)) (print "A clobbered by dsp-mem -- can't be used as argument to QPWD"))
  (let ((MCR (dsp-mem 'Y-IO #xFFFA)))
    (when (logbitp 4 MCR) 
      (ex '(JSR .QP-initialize-dram))
      (print "had to initialize DRAM"))
    (ex '(BCLR DMC-refresh Y-IO DMC-MCR))
    (ex '(BCLR DMC-select Y-IO DMC-MCR))
    (ex `(STORE ,x Y-IO #xFFE1))
    (ex `(STORE ,y Y-IO #xFFE3))
    (ex '(BSET DMC-refresh Y-IO DMC-MCR))))

(defun QPWT (x y) (QPWD x y))

(defun Interrupt-name (i)
  ;; from /usr/include/sys/signal.h
  (nth  i '("none" "SIGHUP" "SIGINT" "SIGQUIT" "SIGILL" "SIGTRAP" "SIGIOT" "SIGEMT"
	    "SIGFPE" "SIGKILL" "SIGBUS" "SIGSEGV" "SIGSYS" SIGPIPE" SIGALRM" "SIGTERM"
	    "SIGURG" "SIGSTOP" "SIGTSTP" "SIGCONT" "SIGHCLD" "SIGTTIN" "SIGTTOU"
	    "SIGIO" "SIGXCPU" "SIGXFSZ" "SIGVTALRM" "SIGPROF" "SIGWINCH" "SIGLOST"
	    "SIGUSR1" "SIGUSR2")))

(defun QP-status (&optional (slot 2))
  (if (/= 0 (dsp-check-host-interface))
      (progn
	(print "no host interface")
	(c56-initialization-check))
    (if (= 0 (qp-is-open))
	(print "qp is closed")
      (progn
	(loop for i from 0 to 4 do
	  (dsp-set slot i)
	  (print (format nil "~&dsp: ~D, HI: ~A" i (hi-status))))
	(dsp-set 2 4)
	(dsp-write-CVR #x92)
	(print "
QP master: ")
	(if (logbitp 7 (dsp-read-cvr))
	    (progn
	      (print "QP master not running (ignored HC) -- will boot it")
	      (dsp-debug slot 4)))
	(print (dsp-dpy-all))
	(print (QP-MCR))
	(print (QP-Slave-Interrupt))
	(print (QP-Master-interrupt))
	(print (QP-MISC))
	(print (QP-M-S))
	(if (/= 0 (qp-get-interrupt))
	    (print (format nil "Got ~D (~A) interrupt!!" (qp-get-interrupt) (interrupt-name (qp-get-interrupt)))))
	(loop for i from 0 to 3 do
	  (print (format nil "~&~&QP dsp: ~D " i))
	  (dsp-set slot i)
	  (dsp-write-CVR #x92)
	  (if (logbitp 7 (dsp-read-cvr))
	      (progn
		(print "not running (ignored HC) -- will boot it")
		(dsp-debug slot i)))
	  (print (dsp-dpy-all))
	  (print (QP-S-M)))))))
    