;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;
;;;  Support for the Ariel QuintProcessor board (see qp.c)
;;;
;;;     qp.c checks the slots, finds any QP boards, and provides host interfaces
;;;     to whatever 56000's are out there.  All communication after booting is
;;;     handled by the procedures in next56.c -- we just set the global hostInterface
;;;     pointer to point to the currently active 56000.  This means, for example,
;;;     that the debugger in lib56.lisp need not be changed at all to deal with the
;;;     QP board.
;;;

(in-package :clm)

(defun qp-close () (qp-all-done))

(defun qp-hi ()
  (let ((arr (make-array 16 :element-type 'integer :initial-element 0)))
    (qp-hi-1 arr)
    arr))

;;; now the code used by the run-time system (sched.lisp) to keep track of these guys.
;;; This code could be in sched.lisp, but it seemed nicer to put it all here.

(defconstant QP-256K #x8000)
(defconstant QP-1M #x10000)
(defconstant QP-4M #x18000)
;(defvar QP-DRAM-size-code QP-256K)

(defun qp-dram-size (slot)		;master's DRAM size in Kwords
  (dsp-debug slot 4)
  (let ((size (dsp-mem 'Y #xFFFF)))
    (qp-close)
    (case (logand (ash size -4) 3)
      (0				;(setf QP-DRAM-size-code 0)
         0)
      (1				;(setf QP-DRAM-size-code QP-1M)
         1000)
      (2				;(setf QP-DRAM-size-code QP-4M)
         4000)
      (3				;(setf QP-DRAM-size-code QP-256K)
         256))))


(defconstant 64K (* 64 1024))
(defconstant 16K (* 16 1024))

(defun qp-sram-size (slot)		;we'll assume they're all the same on a given board
  (dsp-debug slot 0 0)			;any slave dsp (0), memory map 0
  (ex '(LOAD A #x123456))
  (ex '(STORE A X 56000))
  (let ((sram (if (= #x123456 (dsp-mem 'X 56000)) 64K 16K)))
    (qp-close)
    sram))

(defun get-qp-info ()			;find which slots, if any, have QP boards, get SRAM/DRAM sizes.
  (let ((data (make-array 12 :element-type 'fixnum :initial-element -1)))
    (qp-check-all-slots data)
    (loop for i from 0 to 2 do
      (when (zerop (aref data i))
	(setf (aref data (+ i 3)) (qp-dram-size (* (+ i 1) 2)))
	(setf (aref data (+ i 6)) (qp-sram-size (* (+ i 1) 2)))))
    data))


(defvar qp-info nil)
(defvar qp-present 0)

(defstruct QP-data 
    (fbeg 0) 
    (fend -1) 
    (fdata-end 0)
    (active-bufs 0) 
    (iolocs nil) 
    (DRAM-adrs nil) 
    (master nil)
    (buf-size 0)
    (DRAM-size 0)
    (current-dsp nil)
    (fdata-start 0)
    (trigger-point 0))

(defvar qps nil)

(defun qp-initialize ()
  (if (null qp-info) 
      (progn
	(setf qp-present 0)
	(setf qp-info (get-qp-info))
	(when (and qp-info 
		 (or (zerop (aref qp-info 0)) 
		     (zerop (aref qp-info 1)) 
		     (zerop (aref qp-info 2))))
	  (setf qps (make-array 3 :element-type 'QP-data))
	  (loop for i from 0 to 2 do 
	    (when (zerop (aref qp-info i)) 
	      (setf (aref qp-info (+ i 9)) 0)
	      (setf (aref qps i) (make-QP-data))
	      (incf qp-present 5))))))
  qp-present)

