;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-

(in-package :clm)

;;; set up the memory map for the built-in dsp on the NeXT -- this comes
;;; (currently) in 3 different styles: the 8K built-in RAM, the 32K expansion
;;; module available from NeXT, both of these using P overlaid on external X,
;;; and the 192K SFSU DSP expansion module, with separate external P memory.
;;; Since all addresses are legal and address lines appear to be grounded,
;;; to distinguish these, we have to write to two locations and look back
;;; at the first to decide where the image repeats.

;;; on NeXTStep/Intel we currently have two choices (given the 56000 at all),
;;; the Ariel PC56-D with the 64K memory option, and the Ilink i56 with 8K.
;;; The latter looks like the NeXT built-in dsp in all regards.

(defvar Internal-P-size 512)
(defvar Internal-L-size 256)

(defvar External-Mem-size 8192)
(defvar External-L-size 4096)
(defvar External-Y-from-X-offset 4096)

(defvar ext-P-offset #xA000)
(defvar ext-L-offset #xA000)
(defvar ext-X-offset #xA000)
(defvar ext-Y-offset #xA000)
(defvar top-P-loc (+ ext-P-offset external-L-size))
(defvar top-L-loc (+ ext-L-offset external-L-size))

(defvar ext-P-overlaid t)

(defun set-up-dsp-memory-map ()
#+(and Arielpc56d (not ILink) (not MultiSound))
      (progn				;there is also a 16K map (Ariel PC56D)
	(setf external-mem-size 0)
	(setf external-L-size 0)
	(setf external-Y-from-X-offset 0)
	(setf ext-P-offset 0)
	(setf ext-L-offset 0)
	(setf ext-X-offset 0)
	(setf ext-Y-offset 0)
	(setf ext-P-overlaid nil)
	(setf internal-P-size 32768)	;8192
	(setf internal-L-size 16384)	;4096
	(setf top-P-loc 32767)		;8191
	(setf top-L-loc 16383)		;4095
	(initialize-chip-memory))
#+Ilink 
     (progn
       (setf external-mem-size 8192)
       (setf external-L-size (floor external-mem-size 2))
       (setf external-Y-from-X-offset external-L-size)
       (setf ext-P-offset #xA000)
       (setf ext-L-offset #xA000)
       (setf ext-X-offset #xA000)
       (setf ext-Y-offset #xA000)
       (setf ext-P-overlaid t)
       (setf internal-P-size 512)
       (setf internal-L-size 256)
       (setf top-P-loc (+ ext-P-offset external-L-size))
       (setf top-L-loc (+ ext-L-offset external-L-size)))

#+MultiSound
     (progn
       (setf external-mem-size 32768)
       (setf external-L-size (floor external-mem-size 2))
       (setf external-Y-from-X-offset #x4000)
       (setf ext-P-offset #x4000)
       (setf ext-L-offset #x4000)
       (setf ext-X-offset #x4000)
       (setf ext-Y-offset #x4000)
       (setf ext-P-overlaid t)
       (setf internal-P-size 512)
       (setf internal-L-size 256)
       (setf top-P-loc (+ ext-P-offset external-L-size))
       (setf top-L-loc (+ ext-L-offset external-L-size)))

#-Arielpc56d
  (let ((mem-type nil))
    ;; first set up a dummy memory map so we can ask the chip what its actual state is
    (setf internal-P-size 512)
    (setf internal-L-size 256)
    (setf external-mem-size 0)
    (setf ext-P-offset 0)
    (setf top-P-loc 511)
    (initialize-chip-memory)
    (dsp-debug)
    (ex '(LOAD A1 #x123456))
    (ex '(LOAD A0 #x654321))
    (ex '(STORE A1 X #xA001))
    (ex '(STORE A0 X #xB001))		;if 8K this will overwrite A1's store
    (ex '(LOAD B1 X #xA001))
    (if (= (bit24 (dsp-reg 'B1)) #x654321)
	(setf mem-type :8K)
      (progn
	(ex '(STORE A0 X #xE001))	;if 32K this will overwrite A1's store
	(ex '(LOAD B1 X #xA001))
	(if (= (bit24 (dsp-reg 'B1)) #x654321)
	    (setf mem-type :32K)
	  (setf mem-type :192K))))
    (dsp-close)
    (if (or (eq mem-type :8K) (eq mem-type :32K))
	(progn
	  (setf external-mem-size (if (eq mem-type :8K) 8192 32768))
	  (setf external-L-size (floor external-mem-size 2))
	  (setf external-Y-from-X-offset external-L-size)
	  (setf ext-P-offset #xA000)
	  (setf ext-L-offset #xA000)
	  (setf ext-X-offset #xA000)
	  (setf ext-Y-offset #xA000)
	  (setf ext-P-overlaid t)
	  (setf internal-P-size 512)
	  (setf internal-L-size 256)
	  (setf top-P-loc (+ ext-P-offset external-L-size))
	  (setf top-L-loc (+ ext-L-offset external-L-size)))
      (progn
	(setf external-mem-size 0)
	(setf external-L-size 0)
	(setf external-Y-from-X-offset 0)
	(setf ext-P-offset 0)
	(setf ext-L-offset 0)
	(setf ext-X-offset 0)
	(setf ext-Y-offset 0)
	(setf ext-P-overlaid nil)
	(setf internal-P-size #xFF00)
	(setf internal-L-size #xFF00)
	(setf top-P-loc (1- #xFF00))
	(setf top-L-loc (1- #xFF00))))
    t))


;;; By some lucky miracle, we can use these same offsets with the Ariel QP board.
;;; only change is that external-L-size is 8192 (for 16K SRAM like we have now).
;;;
;;; Here's why I believe the above map will work:
;;;
;;; Next dsp in image 2 (default in lib56):
;;;    Internal:
;;;      P:0 to P:1FF - internal P memory
;;;      X:0 to X:FF  - internal X memory
;;;      Y:0 to Y:FF  - internal Y memory
;;;      L:0 to L:FF  - vector (i.e. XY together) at addr (i.e. X addr = Y addr)
;;;    Image 2 external RAM: (any address over #xA000 is mapped into external RAM)
;;;      P:A000 to P:AFFF - overlays X
;;;      X:A000 to X:AFFF and Y:A000 to Y:AFFF separate banks
;;;      L:A000 t0 L:AFFF vector (X addr = Y addr)
;;;
;;; QP Master dsp (only one memory map available):
;;;    Internal (i.e. any address P below 512, X or Y below 256):
;;;      mapped as Next.
;;;    External (8K as Next):
;;;      appears to be same as Next if A000 offset used (separate 4K banks, vector, X overlays P)
;;;
;;; QP Slave dsp (in memory map 3 (i.e. AMODE=3, Y:FFC2 = #x600000).
;;;    Internal (ditto)
;;;    External (16K on our version of the board), same as above only bigger
;;;
;;; all the sizes are correct except for slave External-L-Size and External-Mem-Size, thus also top-L-loc
;;; These unfortunately need to be variables, since we don't know until qp-get-info called what sizes we have.
