;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;

(in-package :clm)

;;;
;;; Common Lisp Music 
;;;
;;;
;;; A large portion of this Lisp code is shadowed by 56000 code in lib56 or C code in cmus.c.
;;; It is here mostly as a form of documentation, and as a fall-back for instruments that
;;; (for whatever reason) cannot use the Run macro.


(defconstant one-pi (coerce pi 'single-float))
;;; NOT short-float!  In MCL that causes any computation involving one-pi and friends to be bogus.
;;; In KCL and MCL, single-float => long/double-float, in ACL single-float=>short-float.

(defconstant two-pi (* one-pi 2))	;may need eval-when here
(defconstant half-pi (/ one-pi 2))
(defconstant three-half-pi (* 3 half-pi))
(defconstant lotsa-pi (* two-pi 1000))

;;; if sampling rate is changed, we should update "mag" too
(defvar frequency-mag (/ two-pi sampling-rate))

(defun set-srate (new-srate)
  (setf frequency-mag (/ two-pi new-srate))
  (setf sampling-rate new-srate))
  
(defmacro in-Hz (val) `(* ,val frequency-mag))


;;; sine-wave oscillator 
;;;
;;;   timing tests indicate that SIN is three times as fast as an interpolated table lookup 
;;;  (apparently uses the 68882) -- this was on the 68030 Nexts.
;;; DAJ tells me:
;;;    Although it's mostly irrelevant, since you usually use the DSP, you might
;;;    be interested to know that on the 040, things changed.  On the 030, the sin
;;;    function was done on the 68882, but there was no room for that on the 040
;;;    so it got moved into software simulation.  It turns out that the sin on
;;;    the 040 is about the same speed as it was on the 030 or a little slower, while
;;;    interpolated table lookup got much faster, making it (table lookup) about
;;;    3 times faster than the sin function on the 040.


(defstruct (Osc
	    (:print-function
	     (lambda (d s k)
	       (declare (ignore k))
	       (format s "#<Osc: freq: ~A (~A Hz), phase: ~A (~A deg)>"
		       (Osc-freq d) (round (/ (Osc-freq d) frequency-mag))
		       (Osc-phase d) (round (* (Osc-phase d) (/ 360.0 two-pi)))))))
  freq phase)

(defun degrees-to-radians (x) 
  (* two-pi (/ x 360)))

(defun radians-to-degrees (x)
  (* x (/ 360.0 two-pi)))

(defun check-initial-phase (x)
  (if (and x 
	   (numberp x) 
	   (> (abs x) two-pi))
      (progn
	(cerror "convert ~F degrees to ~F radians"
		"initial-phase of ~F doesn't make much sense (in radians)"
		x (degrees-to-radians x))
	(degrees-to-radians x))
    x))

(defun make-oscil (&key (frequency 440.0)
			(initial-phase 0.0))
  (make-osc :freq (* frequency frequency-mag)
	    :phase (check-initial-phase initial-phase)))

(defun oscil (o &optional (fm-input 0.0) (pm-input 0.0))
  (declare (optimize (speed 3) (safety 0)))
  (prog1 
      (sin (+ (Osc-phase O) pm-input))
    (incf (Osc-phase O) (+ (Osc-freq O) fm-input))
    ;; if we were being extremely careful, we'd add the fm-input into the sin call at the start too.
    (if (> (Osc-phase O) lotsa-pi) (decf (Osc-phase O) lotsa-pi))
    (if (< (Osc-phase O) (- lotsa-pi)) (incf (Osc-phase O) lotsa-pi))))
    ;; do we need to worry about keeping Osc-phase in more or less the same range as Osc-freq?
    ;; If we have a sampling-rate of around 40000, and a frequency of around 1000Hz, we're accumulating
    ;; two-pi * 1000 every second, but the increment stays around two-pi / 40.  The former might
    ;; easily run upwards for 20 seconds, getting us up to about 2^19 or 2^20.  The increment, meanwhile,
    ;; stays at around 2^(-3).  Our short-floats may be 32 bits, 23 of mantissa.  So, by the end
    ;; of the note, we are losing all the significant bits of the increment, and the phase hangs.
    ;; This problem should be real obvious!  Anyway, we added the DECF line just to sleep well at night.
    ;; (When you get hit this problem the effect is that the carrier wobbles and the spectrum 
    ;; starts to split up).


#|
;;; optimized version (about twice as fast, in some tests):

(defun oscil (o &optional (fm-input 0.0) (pm-input 0.0))
  (declare (optimize (speed 3) (safety 0)))
  (declare (type single-float fm-input pm-input))
  (declare (type osc o))
  (prog1 
      (sin (the single-float (+ (the single-float (Osc-phase O)) pm-input)))
    ;; table lookup here was slower than the call on sin even on the 68040
    (incf (the single-float (Osc-phase O)) (the single-float (+ (the single-float (Osc-freq O)) fm-input)))
    (if (> (the single-float (Osc-phase o)) lotsa-pi) (decf (Osc-phase o) lotsa-pi))
    (if (< (the single-float (Osc-phase o)) (- lotsa-pi)) (incf (Osc-phase o) lotsa-pi))))
|#



;;; Additive synthesis
;;; 
;;;   we may want an arbitrary number of partials in the basic wave, so here we load up the waveform
;;;   into a table and read from it using interpolated table-lookup

(defun max-amp (table)			;return max (abs val) found in table
  (do ((i 0 (+ i 1))
       (maxamp 0.0 (max maxamp (abs (aref table i))))
       (lim (array-dimension table 0)))
      ((>= i lim) maxamp)))

(defun normalize (table &optional (maxamp 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (let ((maxval (if (/= 0.0 maxamp) maxamp (max-amp table)))
	(lim (array-dimension table 0)))
    (if (and (/= maxval 1.0)		;if 1.0 by some miracle, save us a million no-ops
	     (/= maxval 0.0))		;sigh -- an empty array?
	(do ((i 0 (+ i 1))		;otherwise divide through by maxamp
	     (inv-max (/ 1.0 maxval)))	;* probably much faster than /
	    ((>= i lim) maxval)
	  (setf (aref table i) (* (aref table i) inv-max)))
      maxval)))

;;; Array Interpolation (interpolated table lookup)

(defun array-interp (fn x &optional size) ;order of args is like AREF
  (declare (optimize (speed 3) (safety 1)))
  (let ((len (if (or (not size) (zerop size)) (length fn) size)))
    (multiple-value-bind
	(int-part frac-part) 
	(truncate x)
      (if (>= int-part len)
	  (setf int-part (mod int-part len)))
      (if (zerop frac-part) 
	  (aref fn int-part)
	(+ (aref fn int-part)
	   (* frac-part (- (aref fn (if (< (1+ int-part) len) (1+ int-part) 0))
			   (aref fn int-part))))))))

(defun make-table (&optional (size default-table-size))
  (make-array size :element-type 'short-float 
	           :initial-element 0.0))

(defun get-increment (frq tblsiz srate)	;(/ tblsiz srate) was called MAG in Mus10
  (* frq (/ tblsiz srate)))


;;; Interpolating Table lookup

(defstruct Tbl
  freq					;as a table increment (mag * freq in Mus10 terms)
  (internal-mag 1.0)
  (phase 0.0)				;current table location
  table					;which table to use as basic wave
  table-size)

(defun make-table-lookup (&key (frequency 440.0)   ;in Hz
			       (initial-phase 0.0) ;in radians
			       wave-table)         ;what basic wave-form we are producing
  (let ((tblsiz (array-dimension wave-table 0)))
    (make-tbl :freq (get-increment frequency tblsiz sampling-rate)
	      :internal-mag (/ tblsiz two-pi)      ;allow caller to think in terms of 0..two-pi
	      :phase (/ (* (check-initial-phase initial-phase) tblsiz) two-pi)
	      :table wave-table
	      :table-size tblsiz)))
	      
(defun Ur-Table-Lookup (tl &optional (fm-input 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1				           ;return lookup, post-increment everything
      (array-interp (Tbl-table tl) (Tbl-phase tl))
    (incf (Tbl-phase tl) (+ (Tbl-freq tl) fm-input))

    ;;now elaborate checks for phase out of bounds (FM can make it negative)
    (loop while (>= (Tbl-phase tl) (Tbl-table-size tl)) do
      (decf (Tbl-phase tl) (Tbl-table-size tl)))
    (loop while (minusp (Tbl-phase tl)) do
      (incf (Tbl-phase tl) (Tbl-table-size tl)))))

(defmacro Table-Lookup (tl &optional fm-input)
  (if fm-input
      `(Ur-Table-Lookup ,tl (* (Tbl-internal-mag ,tl) ,fm-input))
    `(ur-table-lookup ,tl)))

;;; Additive Synthesis -- data comes in "synth table", a list of partial--amp pairs    
			   
(defun load-one-sine-wave (partial partial-amp table table-size &optional (partial-phase 0.0))
  #-(or mcl (and excl cltl2)) 
      (c-load-one-sine-wave (float partial) (float partial-amp) table table-size (float partial-phase))
  #+(or mcl (and excl cltl2))
      (progn
	(when (/= 0.0 partial-amp)
	  (do ((angle partial-phase (+ angle freq))
	       (freq (* partial (/ two-pi table-size)))
	       (i 0 (+ i 1)))
	      ((>= i table-size))
	    (incf (aref table i) (* partial-amp (sin angle))))))
      )

#+mcl (defun c-normalize (lim table) (declare (ignore lim)) (normalize table))

(defun load-synthesis-table (synth-data table &optional (norm t))
  (declare (optimize (speed 3) (safety 1)))
  (if (not (listp synth-data)) (error "weird argument to load-synthesis-table: ~A" synth-data))
  (let ((lim (length table)))
    (loop for partial in synth-data by #'cddr and amp in (cdr synth-data) by #'cddr do
      (load-one-sine-wave partial amp table lim))
    (if norm (c-normalize lim table))
    table))
			   
(defun load-synthesis-table-with-phases (synth-data table &optional (norm t))
  (declare (optimize (speed 3) (safety 1)))
  (if (not (listp synth-data)) (error "weird argument to load-synthesis-table: ~A" synth-data))
  (let ((lim (length table)))
    (loop for partial in synth-data by #'cdddr and amp in (cdr synth-data) by #'cdddr and angle in (cddr synth-data) by #'cdddr do
      (load-one-sine-wave partial amp table lim angle))
    (if norm (c-normalize lim table))
    table))


;;; Ring Modulation (one of computer music's dumber possibilities).  Each pair of incoming 
;;; sine wave components gets split in accordance with the "Modulation Theorem": since multiplying 
;;; a signal by a phasor (e ^ (j w t)) translates its spectrum by w / two-pi Hz, multiplying by 
;;; a sinusoid splits its spectrum into two equal parts translated up and down by w/two-pi Hz.
;;; The simplest case is: cos f1 * cos f2 = (cos (f1 + f2) + cos (f1 - f2)) / 2.

(defmacro ring-modulate (in1 in2) `(* ,in1 ,in2))



;;; Amplitude modulation (one often seen definition is in1 * (k + in2))

(defun amplitude-modulate (am-carrier input1 input2)
  (* input1 (+ am-carrier input2)))



;;; Delay Line -- delay a signal by n samples

(defstruct (Dly 
	    (:print-function
	     (lambda (d s k)
	       (declare (ignore k))
	       (format s "<Dly: siz: ~A, loc: ~A, id: ~A (~a), ext-id: ~A, ~A>"
		       (dly-size d) (dly-loc d) (dly-id d) (dly-pline d) (dly-ext-id d)
		       (if (dly-pline d) 
			   (if (not (arrayp (dly-pline d)))
			       (array-contents (dly-pline d) (dly-size d)) 
			     (dly-pline d))
			 "nil")))))
  size pline (loc 0) (id nil) (ext-id nil))

(defvar *clm-delay-lines* nil)
(defvar *clm-zdelay-lines* nil)
(defvar *clm-max-delay* -1)

(defun remember-delay (dl)
  (if (null *clm-delay-lines*)
      (setf *clm-delay-lines* (make-array *clm-max-delay-lines* :initial-element nil)))
  (let ((ind (find-open-slot *clm-delay-lines*)))
    (setf (aref *clm-delay-lines* ind) dl)
    (setf (dly-id dl) ind)
    (setf *clm-max-delay* (max *clm-max-delay* ind))
    dl))

(defun make-delay (length &key initial-contents initial-element)
  (let* ((flen (floor length))
	 #-mcl (parr (if (not (eq *clm-language* :c)) (c-make-array flen)))
	 #+mcl (parr (c-make-array flen))
	 (dl (make-dly :size flen
		       :pline parr
		       :ext-id (or initial-contents 
				   (and initial-element
					(/= initial-element 0.0))))))
    (if (not (plusp flen)) (error "make-delay length: ~A?~A" length (if (plusp length) " (use zdelay)" "")))
    (if (not (eq *clm-language* :c))
	(if initial-contents
	    (loop for i in initial-contents and j from 0 below flen do
	      (saref parr j (real-to-fix i)))
	  (if (and initial-element
		   (/= initial-element 0.0))
	      (let ((fixed-val (real-to-fix initial-element)))
		(loop for i from 0 below flen do
		  (saref parr i fixed-val))))))
    (remember-delay dl)))

(defun forget-delay (dl)
  (when (and dl *clm-delay-lines*)
    (setf (aref *clm-delay-lines* (dly-id dl)) nil)
    (if *clm-zdelay-lines* (setf (aref *clm-zdelay-lines* (dly-id dl)) nil))
    (setf (dly-id dl) nil)
    (if (not (eq *clm-language* :c)) (c-free-array (dly-pline dl)))
    (setf (dly-pline dl) nil)))

; due to lisp gc dumbness, and other unfortunate problems, we've decided to manage these delay lines 
; by hand in c (alongside the IO buffers)

(defun tap (d &optional (offset 0))
  ;; if offset is not zero, it should be greater than 0 -- we will subtract it from the current
  ;; location, assuming it is a tap on a delay line offset from 0 delay -- if it is less than 0,
  ;; there's a wrap-around effect because we don't fixup the delay length to take this into account.
  (if (zerop offset) 
      (fix-to-real (caref (dly-pline d) (dly-loc d)))
    (fix-to-real (caref (dly-pline d) (mod (- (dly-loc d) offset) (dly-size d))))))
	  
(defun Delay (d input)
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (tap d)
    (saref (dly-pline d) (dly-loc d) (real-to-fix input))
    (incf (dly-loc d))
    (if (<= (dly-size d) (dly-loc d)) (setf (dly-loc d) 0))))


;;; ZDelay -- zdelay is an "interpolating" delay line.  That is, the delay length is some
;;;     non-integral amount, and it can change by an arbitrary amount.  In this implementation,
;;;     we try to leave plenty of slack.  The original version did fm on the delay length,
;;;     but I find that hard to visualize, so the current version can be viewed as pm --
;;;     the pm-input is the difference between the nominal delay length and the actual length.
;;;     If it is positive, we are looking further back in time.

(defstruct zdly del phase)

(defun make-zdelay (length &key initial-contents initial-element true-length)
  (let ((tlength (floor (or true-length 
			    (and (< length 512) 
				 (max 64 (* length 2)))
			    (+ length 512)))))
    (make-zdly :phase (- tlength length)
	       :del (make-delay tlength :initial-contents initial-contents :initial-element initial-element))))

(defun zdelay-interp (pline true-length phase)
  (declare (optimize (speed 3) (safety 1)))
  (let ((low-index (floor phase)))
      (if (or (minusp low-index)
	      (>= low-index true-length))
	  (setf low-index (mod low-index true-length)))
      (if (= (floor phase) phase)
	  (fix-to-real (caref pline low-index))
	(let ((high-index (1+ low-index)))
	  (if (>= high-index true-length)
	      (setf high-index 0))
	  (let ((y0 (fix-to-real (caref pline low-index)))
		(y1 (fix-to-real (caref pline high-index)))
		(frac (- phase (floor phase))))
	    (+ y0 (* frac (- y1 y0))))))))

(defun ztap (z &optional (pm-input 0.0))
  (let ((size (dly-size (zdly-del z))))
    (zdelay-interp (dly-pline (zdly-del z)) size (- (zdly-phase z) pm-input))))

(defun zdelay (z input &optional (pm-input 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (let ((size (dly-size (zdly-del z))))
    (prog1
	(ztap z pm-input)
      ;; the pm-input is subtracted because we want a positive pm-input to correspond to
      ;; a lengthening of the delay
      (incf (zdly-phase z))
      (if (>= (zdly-phase z) size)	;> here for various error situations
	  (setf (zdly-phase z) 0))
      (delay (zdly-del z) input))))

(defun remember-zdelay (dl)
  (if (null *clm-zdelay-lines*)
      (setf *clm-zdelay-lines* (make-array *clm-max-delay-lines* :initial-element nil)))
  (let ((ind (dly-ext-id (zdly-del dl))))
    (if ind (setf (aref *clm-zdelay-lines* ind) dl))))



;;; Comb filter (a delay line with a scaler on the feedback term)
;;;
;;;    in filter parlance, y(n) <= x(n-D-1) + scaler * y(n-D)
;;;    As a rule of thumb, the decay time of the feedback part is 7*(delay)/(1-scaler) samples,
;;;    so to get a decay of DUR seconds, scaler <= 1-7*D/(DUR*Srate).  (D=delay length here).
;;;    The peak gain is 1/(1-(abs scaler)).
;;;
;;;    See Julius Smith's "An Introduction to Digital Filter Theory" in Strawn "Digital 
;;;    Audio Signal Processing"

(defstruct cmbflt dly-unit scaler)

(defun make-comb (mlt length)
  (make-cmbflt :dly-unit (make-delay length)
	       :scaler mlt))

(defun comb (cflt input)
  (declare (optimize (speed 3) (safety 1)))
  (let ((d (cmbflt-dly-unit cflt)))
    (delay d (+ input (* (cmbflt-scaler cflt) (tap d))))))


;;; Notch filter (a delay line with a feedforward term)
;;; see Julius Smith's "Music Applications of Digital Waveguides" for a brief discussion

(defun make-notch (mlt length)
  (make-comb mlt length))

(defmacro notch (cflt input)
  `(let ((val ,input))
     (+ (* val (cmbflt-scaler ,cflt))
	(delay (cmbflt-dly-unit ,cflt) val))))


;;; All-pass or "moving average comb" filter
;;;
;;;  just like comb-filter but with added feed-forward term with scaler 
;;;  (if feedback scaler = 0, we get the moving average comb)
;;;  (if both scale terms = 0, we get a pure delay line)
;;;  In filter parlance, y(n) <= feedforward*x(n-1) + x(n-D-1) + feedback*y(n-D) -- i.e. DM+G0(IN+DM*G1),DM:=(IN+DM*G1)

(Defstruct allpassflt dly-unit feedback feedforward)

(defun make-all-pass (back-mlt forward-mlt length)
  (make-allpassflt :dly-unit (make-delay length)
		   :feedback back-mlt
		   :feedforward forward-mlt))

(defun all-pass (f input)
  (declare (optimize (speed 3) (safety 1)))
  (let* ((d (allpassflt-dly-unit f))
	 (d-in (+ input (* (allpassflt-feedback f) (tap d)))))
    (+ (delay d d-in)
       (* (allpassflt-feedforward f) d-in))))

#|
;;; from the formula above, one might think all-pass should be:

(defun all-pass (f input)
  (declare (optimize (speed 3) (safety 1)))
  (let* ((d (allpassflt-dly-unit f))
	 (d-in (* (allpassflt-feedback f) (fix-to-real (caref (dly-pline d) (dly-loc d))))))
    (+ (delay d (+ input d-in))
       (* (allpassflt-feedforward f) input)
       d-in)))

;;; but that wrecks all the Mus-10 and Samson box instruments that use scale the
;;; resultant y(n-D) term by both scale factors before returning the result.
|#



;;; Direct, Lattice, and Ladder filters
;;;     by Markel and Gray (see "Linear Prediction of Speech" and latpak.f4)
;;;     Most of this code translated from Fortran, hence the odd indexing in places
;;;     See d2l.lisp for test procedures (also from Markel and Gray)

(defun d2l (m a p al)			;convert from direct form to two-multiplier lattice form
  ;; m=order, a=denominator, p=numerator
  ;; returns p=tap parameters, a=k parameters
  (let ((b (make-array 30 :element-type 'short-float :initial-element 0.0))
	(sum 0.0)
	(alpha 1.0)
	(nstab nil))
    (dotimes (j m)
      (let* ((mr (- m j))
	     (dd (- 1.0 (* (aref a mr) (aref a mr)))))
	(setf nstab (or nstab (>= (aref a mr) 1.0)))
	(incf sum (* alpha (aref p mr) (aref p mr)))
	(setf alpha (/ alpha dd))
	(setf (aref al (1- mr)) alpha)
	(dotimes (k mr)
	  (setf (aref b k) (aref a (- mr k))))
	(dotimes (k mr)
	  (decf (aref p k) (* (aref p mr) (aref b k)))
	  (setf (aref a k) (/ (- (aref a k) (* (aref a mr) (aref b k))) dd)))))
    (setf (aref a 0) (+ sum (* (aref p 0) (aref p 0) alpha)))
    (if nstab (setf (aref a 0) (- (abs (aref a 0)))))))

(defun l2nl (m a p al)			;convert from lattice form to normalized ladder form
  (let* ((mp (1+ m))
	 (vmax 0.0)
	 (so 0.0)
	 (vgain (aref a 0))
	 (q (sqrt vgain)))
    (dotimes (j m)
      (setf (aref p j) (* (aref p j) (sqrt (aref al j)))))
    (dotimes (j mp)
      (setf vmax (max vmax (aref p j))))
    (setf so vmax)
    (if (>= vmax q) 
	(setf so (/ vmax .999))
      (setf so q))
    (dotimes (j mp)
      (setf (aref p j) (/ (aref p j) so)))
    so))

(defun nrmlat (m a p c)			;convert from direct form to normalized ladder form
  (d2l m a p c)
  (let* ((so (l2nl m a p c))
	 (vgain (aref a 0)))
    (dotimes (j m)
      (setf (aref a j) (aref a (1+ j)))
      (setf (aref c j) (sqrt (- 1.0 (* (aref a j) (aref a j))))))
    (setf (aref a m) vgain)
    so))

(defconstant direct-form 0)
(defconstant lattice-form 1)
(defconstant ladder-form 2)

(defstruct flt m a b c d (so 1.0) (typ direct-form))		
					;names taken from Markel and Gray
					;except b="p" because flt-p is built-in type checker
					;"m" as passed by caller is m+1 from M&G point of view

(defun make-filter (&key (order nil) (m nil)
			 (a nil) (p nil) 
			 (x-coeffs nil) (y-coeffs nil) 
			 (type direct-form) 
			 (sn nil) (rc nil) (tap nil) (k nil)
			 (cn nil))
  (case type 
    (0					;direct-form
     (make-direct-filter :a (or y-coeffs a)
			 :p (or x-coeffs p)
			 :m (or order m (max (length (or x-coeffs p)) (length (or y-coeffs a))))))
    (1					;lattice-form
     (make-lattice-filter :k (or k rc) :tap tap :p (or p x-coeffs) :a (or a y-coeffs) :m m))
    (2					;ladder-form
     (make-ladder-filter :sn sn :cn cn :a (or a y-coeffs) :p (or p x-coeffs) :tap tap :m m))
    (t (error "unimplemented filter type: ~A" type))))

(defun float-list (l)
  (mapcar #'float l))

(defun pad (len lis)
  (if (> len (length lis))
      (dotimes (i (- len (length lis)))
	(setf lis (append lis '(0.0)))))
  (float-list lis))

(defun make-direct-filter (&key a p m)
  (make-flt :a (if a (make-array m :element-type 'short-float :initial-contents (pad m a))
		 (make-array m :element-type 'short-float :initial-element 0.0))
	    :b (if p (make-array m :element-type 'short-float :initial-contents (pad m p))
		 (make-array m :element-type 'short-float :initial-element 0.0))
	    :typ direct-form
	    :d (make-array m :element-type 'short-float :initial-element 0.0)
	    :m (1- m)))

(defun make-lattice-filter (&key (k nil) (rc nil) (tap nil) (p nil) (a nil) (m nil))
  (let* ((order (or m (max (length (or k rc a)) (length (or tap p)))))
	 (a1 (make-array order :element-type 'short-float :initial-contents (pad order (or rc k a))))
	 (p1 (make-array order :element-type 'short-float :initial-contents (pad order (or tap p)))))
    (if (or a p)
	(if (or rc k tap)
	    (error "inconsistent data")
	  (let ((c (make-array order :element-type 'short-float :initial-element 0.0)))
	    (d2l (1- order) a1 p1 c)
	    (dotimes (i (1- order)) (setf (aref a1 i) (aref a1 (1+ i)))))))
    (make-flt :a a1 
	      :b p1 
	      :d (make-array order :element-type 'short-float :initial-element 0.0)
	      :typ lattice-form 
	      :m (1- order))))
	
(defun make-ladder-filter (&key (sn nil) (cn nil) (a nil) (p nil) (tap nil) (so nil) (m nil))
  (let* ((order (or m (max (length (or sn a)) (length (or tap p)) (length cn))))
	 (so1 so)
	 (a1 (make-array order :element-type 'short-float :initial-contents (pad order (or sn a))))
	 (p1 (make-array order :element-type 'short-float :initial-contents (pad order (or tap p))))
	 (c1 (if cn (make-array order :element-type 'short-float :initial-contents (pad order cn))
	       (make-array order :element-type 'short-float :initial-element 0.0))))
    (if (or a p)
	(if (or sn cn tap)
	    (error "inconsistent data")
	  (setf so1 (nrmlat (1- order) a1 p1 c1))))
    (make-flt :a a1 
	      :b p1 
	      :c c1
	      :d (make-array order :element-type 'short-float :initial-element 0.0)
	      :typ ladder-form 
	      :so so1 
	      :m (1- order))))

(defun direct-filter (fl inp)
  (let ((xout 0.0))
    (setf (aref (flt-d fl) 0) inp)
    (loop for j from (flt-m fl) downto 1 do
      (incf xout (* (aref (flt-d fl) j) 
		    (aref (flt-b fl) j)))
      (decf (aref (flt-d fl) 0) (* (aref (flt-a fl) j) 
				   (aref (flt-d fl) j)))
      (setf (aref (flt-d fl) j) (aref (flt-d fl) (1- j))))
    (+ xout (* (aref (flt-d fl) 0) (aref (flt-b fl) 0)))))

(defun lattice-filter (fl inp)
  (let ((xout 0.0))
    (loop for i from (1- (flt-m fl)) downto 0 and j from (flt-m fl) by -1 do
      (decf inp (* (aref (flt-a fl) i) (aref (flt-d fl) i)))
      (setf (aref (flt-d fl) j) (+ (aref (flt-d fl) i) (* (aref (flt-a fl) i) inp)))
      (incf xout (* (aref (flt-d fl) j) 
		    (aref (flt-b fl) j))))
    (setf (aref (flt-d fl) 0) inp)
    (+ xout (* inp (aref (flt-b fl) 0)))))

(defun ladder-filter (fl inp)
  (let ((xout 0.0))
    (loop for i from (1- (flt-m fl)) downto 0 do
      (setf (aref (flt-d fl) (1+ i)) (+ (* inp (aref (flt-a fl) i)) 
					(* (aref (flt-d fl) i) (aref (flt-c fl) i))))
      (setf inp (- (* inp (aref (flt-c fl) i)) 
		   (* (aref (flt-d fl) i) 
		      (aref (flt-a fl) i))))
      (incf xout (* (aref (flt-d fl) (1+ i)) (aref (flt-b fl) (1+ i)))))
    (setf (aref (flt-d fl) 0) inp)
    (* (flt-so fl) (+ xout (* (aref (flt-d fl) 0) (aref (flt-b fl) 0))))))
 
(defun filter (fl inp)
  (if (= (flt-typ fl) direct-form)
      (direct-filter fl inp)
    (if (= (flt-typ fl) lattice-form)
	(lattice-filter fl inp)
      (if (= (flt-typ fl) ladder-form)
	  (ladder-filter fl inp)
	(error "unimplemented filter type: ~A" (flt-typ fl))))))

					
;;; see fltdes.lisp for a function compatible with filter above that provides the coefficients
;;; for Butterworth and Chebychev low pass filters.


;;; the following are common special cases of the filter given above

(defstruct smpflt a0 a1 a2 b1 b2 (x1 0.0) (x2 0.0) (y1 0.0) (y2 0.0)) 
					;JOS article and Dodge book use a(n) on input, b(n) output, and subtract output
					;this is different from "Digital Signal Processing" convention

;;; one zero  y(n) = a0 x(n) + a1 x(n-1)
(defun make-one-zero (a0 a1) 
  (make-smpflt :a0 a0 :a1 a1))

(defun one-zero (f input) 
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (+ (* (smpflt-a0 f) input) (* (smpflt-a1 f) (smpflt-x1 f)))
    (setf (smpflt-x1 f) input)))


;;; one-pole  y(n) = a0 x(n) - b1 y(n-1)
(defun make-one-pole (a0 b1)
  (make-smpflt :a0 a0 :b1 b1))

(defun one-pole (f input)
  (declare (optimize (speed 3) (safety 1)))
  (setf (smpflt-y1 f) 
    (- (* (smpflt-a0 f) input) (* (smpflt-b1 f) (smpflt-y1 f)))))


;;; two-pole  y(n) = a0 x(n) - b1 y(n-1) - b2 y(n-2)
(defun make-two-pole (a0 b1 b2)
  (if (or (>= (abs b1) 2.0)
	  (>= (abs b2) 1.0)
	  (and (>= (- (* b1 b1) (* b2 4.0)) 0.0)
	       (or (>= (+ b1 b2) 1.0)
		   (>= (- b2 b1) 1.0))))
      (print "unstable filter..."))
  (make-smpflt :a0 a0 :b1 b1 :b2 b2))

(defun two-pole (f input)
  (declare (optimize (speed 3) (safety 1)))
  (let ((y0 (- (* (smpflt-a0 f) input) 
	       (* (smpflt-b1 f) (smpflt-y1 f)) 
	       (* (smpflt-b2 f) (smpflt-y2 f)))))
    (setf (smpflt-y2 f) (smpflt-y1 f))
    (setf (smpflt-y1 f) y0)
    y0))


;;; two-zero  y(n) = a0 x(n) + a1 x(n-1) + a2 x(n-2)
(defun make-two-zero (a0 a1 a2)
  (make-smpflt :a0 a0 :a1 a1 :a2 a2))

(defun two-zero (f input)
  (declare (optimize (speed 3) (safety 1)))
  (let ((y0 (+ (* (smpflt-a0 f) input)
	       (* (smpflt-a1 f) (smpflt-x1 f))
	       (* (smpflt-a2 f) (smpflt-x2 f)))))
    (setf (smpflt-x2 f) (smpflt-x1 f))
    (setf (smpflt-x1 f) input)
    y0))

;;; These can have time-varying coefficients -- for example:
;;; (setf (smpflt-a1 f) (env e))

;;; The following are cases JOS thought were nice to provide in the Sambox world:

(defun make-ppolar (R freq) 
  (make-two-pole 1.0 
		 (- (* 2.0 R (cos (* freq frequency-mag)))) 
		 (* R R)))
(defmacro ppolar (f input) 
  `(two-pole ,f ,input))

(defun make-zpolar (R freq)		;was two-pole until 15-May-92, by accident
  (make-two-zero 1.0 
		 (- (* 2.0 R (cos (* freq frequency-mag))))
		 (* R R)))
(defmacro zpolar (f input) 
  `(two-zero ,f ,input))

(defstruct frmnt tz tp G)

(defun make-formnt (R freq &optional (G 1.0)) 
  (if (minusp R) (error "R=~F is meaningless" R))
  (make-frmnt :G (* G (- 1.0 R))
	      :tp (make-ppolar R freq)
	      :tz (make-two-zero 1.0 0.0 (- R))))

(defmacro formnt (f input) 
  `(ppolar (frmnt-tp ,f) (two-zero (frmnt-tz ,f) (* (frmnt-G ,f) ,input))))


;;; RandH and RandI
;;;
;;;    randH latches its output random number (between 0 and 1.0), getting a new number 
;;;    every srate/freq samples -- internally we pretend that our cycle is between 0 and
;;;    two-pi so that the caller can use frequency-mag without confusion.  This way, 
;;;    frequency calculations look the same between oscil and randh and so on.
;;;
;;;    randI interpolates between successive random numbers (also between 0.0 and 1.0)


(defun ran (lo hi)			;returns random numbers between lo and hi
  (if (= hi lo) 
      0.0
    (+ lo (random (- hi lo)))))

(defun clm-random (n)
  (if (zerop n) 0
    (random n)))


(defstruct (noi 
	    (:print-function
	     (lambda (d s k)
	       (declare (ignore k))
	       (format s "<Noi: freq: ~A (~A Hz), phase: ~A (~A deg), base: ~A, output: ~A, incr: ~A, op: ~A>"
		       (Noi-freq d) (round (/ (Noi-freq d) frequency-mag))
		       (Noi-phase d) (round (* (Noi-phase d) (/ 360.0 two-pi)))
		       (Noi-base d) (Noi-output d) (Noi-incr d) (Noi-ran-op d)))))
  freq base phase output incr ran-op)

(defun make-randh (&key (frequency 440.0) (amplitude 1.0) (increment 0.0) (type 'clm-random))
  (make-noi :freq (* frequency frequency-mag)
	    :base amplitude
	    :phase 0.0
	    :incr increment
	    :ran-op type
	    :output 0.0))

(defun randh (r &optional (sweep 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (progn
    (if (>= (noi-phase r) two-pi)
	(progn
	  (loop while (>= (noi-phase r) two-pi) do (decf (noi-phase r) two-pi))
	  (setf (noi-output r) (funcall (noi-ran-op r) (noi-base r)))))
    (incf (noi-phase r) (+ (noi-freq r) sweep))
    (loop while (minusp (noi-phase r)) do (incf (noi-phase r) two-pi))
    (noi-output r)))

(defun make-randi (&key (frequency 440.0) (amplitude 1.0))
  (make-randh :frequency frequency
	      :amplitude amplitude 
	      :increment (if (zerop amplitude) 
			     0.0 
			   (* (random amplitude) (/ frequency sampling-rate)))))

(defun randi (r &optional (sweep 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (incf (noi-output r) (noi-incr r))
    (when (>= (noi-phase r) two-pi)
      (loop while (>= (noi-phase r) two-pi) do (decf (noi-phase r) two-pi))
      (setf (noi-incr r) (* (- (funcall (noi-ran-op r) (noi-base r)) 
			       (noi-output r)) 
			    (/ (+ (noi-freq r) sweep) two-pi))))
    ;; the (+ freq sweep) is obviously just a wild guess at the current "frequency"

    (incf (noi-phase r) (+ (noi-freq r) sweep))
    (loop while (minusp (noi-phase r)) do (incf (noi-phase r) two-pi))))

;;; to get arbitrary kinds of noise, set up a function of one argument returning the noise 
;;; values between 0.0 and that argument, then set the noi-ran-op field to the name of that function.




;;; Envelopes
;;;
;;; the first group of procedures provide all the usual lisp functions applied to envelopes
;;; (and later the same for arrays, and in dpysnd, the same for sound files -- if lisp made its
;;; underlying functions generic, we could just add methods here rather than all new functions).

;;; List Interpolation -- assume a list of x y pairs (i.e. envelope breakpoints or synth tables)

(defun list-interp (x fn &optional (base 0)) ;order of args is like NTH
  (declare (optimize (speed 3) (safety 1)))
  (cond ((null fn) 0.0)			;no data -- return 0.0
	((or (<= x (first fn))		;we're sitting on x val (or if < we blew it)
	     (null (third fn)))		;or we're at the end of the list
	 (second fn))			;so return current y value
	((> (third fn) x)		;x <= next fn x axis value
	 (if (= (second fn) (fourth fn))
	     (second fn)		;y1=y0, so just return y0 (avoid endless calculations below)
	   (if (or (= 0 base) 
		   (= 1 base))		;linear envelope
	       (+ (second fn)		;y0+(x-x0)*(y1-y0)/(x1-x0)
		  (* (- x (first fn))
		     (/ (- (fourth fn) (second fn))
			(- (third fn) (first fn)))))
	     (+ (second fn)		;y0+[[[y1-y0] [-1.0 + [base ^ [x-x0]/[x1-x0]]] / [base-1]
		(* (/ (- (fourth fn) (second fn))
		      (- base 1.0))	;scaled y1-y0
		   (- (expt base (/ (- x (first fn)) 
				    (- (third fn) (first fn)))) 1.0))))))
	(t (list-interp x (cddr fn) base))))	;go on looking for x segment


;;; See env.lisp for the envelope handlers used in the generic functions in dpysnd.


(defun env-last-x (env)
  (if (null env) 0.0 (nth (- (length env) 2) env)))

;;; The following functions do various useful operations on a single envelope

(defun max-envelope (env &optional (cur-max 0.0))
  (max cur-max (if env (loop for y in (cdr env) by #'cddr maximize y))))

(defun scale-envelope (env scale &optional (offset 0.0))
  (if (endp env) nil
      (append (list (car env) (+ offset (* scale (cadr env))))
	      (scale-envelope (cddr env) scale offset))))
;;; version using nth and a loop was a little slower than this despite all the appends

(defun normalize-envelope (env &optional (new-max 1.0))
  (scale-envelope env (/ new-max (max-envelope env))))


;;; magify-seg takes an envelope, a starting time in samples, the envelope duration in samples, 
;;; and a y scaler.  It returns another envelope-like list (i.e. a list of time-value pairs), 
;;; where the times are pass numbers, and the values are increments to be added on each pass to 
;;; get to the next y-value.   For very large envelopes, (say more than 50 segments), we should 
;;; simply load the thing into an array and use Table-lookup to read it out.

(defun magify-seg (envelope start-in-samples duration-in-samples scaler &optional (stepit nil))
  (declare (optimize (speed 3) (safety 1)))
  (let* ((lim (- (length envelope) 2))
	 (x0 0.0) (y0 0.0) (cur-x 0.0) (y-incr 0.0)
	 (x1 (car envelope))
	 (result nil)
	 (cur-pass start-in-samples)
	 (x-diff (- (nth lim envelope) x1))) ; x1 is really x0 here
    (if (zerop x-diff) (warn "envelope repeats x axis values: ~A" envelope))
    (let* ((x-mag (/ (1- duration-in-samples) x-diff)))
      (if (zerop x-mag) (warn "envelope duration is 0 samples: ~D at ~1,3F for ~A" 
			      duration-in-samples 
			      (/ start-in-samples sampling-rate)
			      envelope))
      (let* ((inv-x-mag (/ 1.0 x-mag))
	     (y1 (cadr envelope)))
	(loop for i from 0 by 2 below lim and n2 in (cddr envelope) by #'cddr and n3 in (cdddr envelope) by #'cddr do
	  (setf x0 x1)
	  (setf x1 n2)
	  (setf y0 y1)
	  (setf y1 n3)
	  (setf cur-x (max 1 (round (* x-mag (- x1 x0)))))
	  (setf x1 (+ x0 (* inv-x-mag cur-x)))
	  (push cur-pass result)
	  (if (not stepit)
	      (if (= y0 y1)		;no change in y on this segment
		  (setf y-incr 0)    
		(setf y-incr (* scaler (/ (- y1 y0) cur-x))))
	    (setf y-incr (* scaler y0)))
	  (push y-incr result)
	  (incf cur-pass cur-x))
	(nreverse result)))))


(defun fix-up-exp-env (e off scl base)
  (declare (optimize (speed 3) (safety 1)))
  (if e
      (let* ((min-y (+ off (* scl (cadr e))))
	     (max-y min-y)
	     (val 0.0)
	     (tmp 0.0)
	     (nb (and base (not (zerop base)) (/= 1.0 base)))
	     (b (if nb (/ 1.0 (log base))))
	     (b-1 (if nb (- base 1.0)))
	     (result nil)
	     (flat nil)
	     (len (length e)))
	(loop for i from 1 below len by 2 and ni in (cdr e) by #'cddr and ni-1 in e by #'cddr do
	  ;;(setf val (+ off (* scl (nth i e))))
	  (setf val (+ off (* scl ni)))
	  (setf min-y (min min-y val))
	  (setf max-y (max max-y val))
	  
	  ;;(push (nth (- i 1) e) result)
	  (push ni-1 result)
	  (push val result))
	(setf result (nreverse result))
	(setf flat (= min-y max-y))
	(if (not flat) (setf val (/ 1.0 (- max-y min-y))))
	(loop for i from 1 below len by 2 do
	  (if (not flat)
	      (setf tmp (* val (- (nth i result) min-y)))
	    (setf tmp 1.0))
	  ;; tmp is now a number between 0 and 1, we need the power that will give us that number given base...
	  (if nb 
	      (setf (nth i result) (* (log (+ 1.0 (* tmp b-1))) b))
	    (setf (nth i result) tmp)))
	;; that is -- ((base^x)-1)/(base-1) solved for x in terms of base.
	(values result min-y max-y))
    (values nil 0)))
	     

(defstruct envelope current-value rate data pass base offset scaler power op end restart type)

(defun make-env (&key envelope
		      start-time
		      start
		      duration
		      end
		      (offset 0.0)
		      (scaler 1.0)
		      base
		      op
		      restartable)
  (let* ((start-in-samples (or start 
			       (and start-time 
				    (floor (* start-time sampling-rate)))
			       0))
	 (dur-in-samples (or (and start 
				  end 
				  (1+ (- end start)))
			     (and duration 
				  (floor (* duration sampling-rate)))
			     0))
	 (end-in-samples (or end (+ start-in-samples dur-in-samples)))
	 (checked-envelope (if (not envelope) 
			       (list 0 0 1 0)
			     (if (= (length envelope) 2)
				 (list (car envelope) (cadr envelope) (1+ (car envelope)) (cadr envelope))
			       envelope)))
	 (y0 (cadr checked-envelope))
	 (init-y (+ offset (* scaler y0))))
    (if (zerop dur-in-samples) (error "envelope duration = 0?"))
    (when (> *clm-safety* 0)
      (let ((x0 (first checked-envelope)))
	(loop for x1 in (cddr checked-envelope) by #'cddr do
	  (if (< x1 x0) (warn "X axis values out of order in: '~A going from ~A to ~A" envelope x0 x1))
	  (setf x0 x1))))
    (if (or (null base)
	    (= base 1) 
	    (= base 0))
	(let ((data (magify-seg checked-envelope start-in-samples dur-in-samples scaler (and (not op) (numberp base) (zerop base)))))
	  (make-envelope :current-value init-y
			 :rate 0.0
			 :base base
			 :end end-in-samples
			 :pass start-in-samples
			 :data data
			 :type (if op :map :seg)
			 :op op
			 :restart (if restartable (list init-y 
							(loop for x in data by #'cddr and y in (cdr data) by #'cddr
							 collect (- x start-in-samples)
							 collect y)
							(- end-in-samples start-in-samples)))))
      (multiple-value-bind 
	  (new-e min-y max-y)
	  (fix-up-exp-env checked-envelope offset scaler base)
	(let ((data (magify-seg new-e start-in-samples dur-in-samples 1.0)))
	  (make-envelope :current-value init-y
			 :power (cadr new-e)
			 :base base
			 :pass start-in-samples
			 :end end-in-samples
			 :offset min-y
			 :scaler (if op 
				     (- max-y min-y) 
				   (/ (- max-y min-y) (- base 1.0)))
			 :rate 0.0
			 :op op
			 :type :exp
			 :data data
			 :restart (if restartable (list init-y 
							(loop for x in data by #'cddr and y in (cdr data) by #'cddr
							 collect (- x start-in-samples)
							 collect y)
							(- end-in-samples start-in-samples)
							(cadr new-e)))))))))

(defun make-func (&rest args) (apply #'make-env args))

(defun restart-env (e)
  ;; return all state to start-up state except update all pass values in the data list to reflect passage of time
  (if e
      (if (not (envelope-restart e))
	  (error "attempt to restart an unrestartable envelope")
	(let ((restart-data (second (envelope-restart e)))
	      (pass (envelope-pass e)))
	  (setf (envelope-current-value e) (first (envelope-restart e)))
	  (setf (envelope-end e) (+ pass (third (envelope-restart e))))
	  (setf (envelope-data e) (loop for x in restart-data by #'cddr and 
				   y in (cdr restart-data) by #'cddr
				   collect (+ pass x) collect y))
	  (when (eq (envelope-type e) :exp)
	    (setf (envelope-power e) (fourth (envelope-restart e))))
	  (setf (envelope-rate e) 0.0)	;probably not necessary
	  e))))


(defun env (e)
  (declare (optimize (speed 3) (safety 1)))
  (if (eq (envelope-type e) :seg)
      (prog1
	  (envelope-current-value e)
	(when (and (envelope-data e)	;are there any segments queued up?
		   (>= (envelope-pass e) (car (envelope-data e))))
	  (setf (envelope-rate e) (cadr (envelope-data e)))
	  (setf (envelope-data e) (cddr (envelope-data e))))
	(incf (envelope-pass e))
	(if (or (null (envelope-base e))
		(not (zerop (envelope-base e))))
	    (if (and (/= 0.0 (envelope-rate e))
		     (<= (envelope-pass e) (envelope-end e)))
		(incf (envelope-current-value e) (envelope-rate e)))
	  (setf (envelope-current-value e) (envelope-rate e))))
    (if (eq (envelope-type e) :map)	;map between un-normalized data and new, via OP
					;here is where generic functions would be very nice
	(prog1
	    (funcall (envelope-op e) (envelope-current-value e))
	  (when (and (envelope-data e)	;are there any segments queued up?
		     (>= (envelope-pass e) (car (envelope-data e))))
	    (setf (envelope-rate e) (cadr (envelope-data e)))
	    (setf (envelope-data e) (cddr (envelope-data e))))
	  (incf (envelope-pass e))
	  (if (/= 0.0 (envelope-rate e))
	      (incf (envelope-current-value e) (envelope-rate e))))
      (if (eq (envelope-type e) :exp)	;exponential interpolation between break-points
	  (prog1
	      (envelope-current-value e)
	    (when (and (envelope-data e)
		       (>= (envelope-pass e) (car (envelope-data e))))
	      (setf (envelope-rate e) (cadr (envelope-data e)))
	      (setf (envelope-data e) (cddr (envelope-data e))))
	    (incf (envelope-pass e))
	    (when (and (/= 0.0 (envelope-rate e))
		       (<= (envelope-pass e) (envelope-end e)))
	      (incf (envelope-power e) (envelope-rate e))
	      (setf (envelope-current-value e) 
		(+ (envelope-offset e)
		   (* (envelope-scaler e) 
		      (if (envelope-op e)
			  (funcall (envelope-op e) (envelope-power e) (envelope-base e))
			(- (expt (envelope-base e) (envelope-power e)) 1.0)))))))
	(error "unknown envelope type: ~A" (envelope-type e))))))

;;; the "OP" slot allows the caller to call any function he likes to interpolate between
;;; breakpoints (i.e. SIN or LOG and so on).  The function should take an argument between
;;; 0 and 1, and return a value between 0 and 1.
;;;      (Defun sin-interp (x y) (sin (* x half-pi))) 
;;; gives sinusoidal connection between points.
;;; We then connect it into ENV with 
;;;      (make-env ... :op 'sin-interp)
;;; The second argument is the "BASE" field of the envelope structure -- if it is not null, 0, or 1,
;;; you get a fixup so that the exponential function will produce the original breakpoints.  If BASE
;;; is null, and OP is not null, you get a map between the un-fixed-up values and the new operation


;;; FIX ENVELOPES TO FIT INTO n BIT VALUES --------------------------------------------------------
(defvar ramp-bits 24)
(defvar ramp-max (expt 2 ramp-bits))
(defvar min-unfixed-ramp (/ 10.0 ramp-max))

;;; we need 16 bit envelopes in fasmix, so changed these from constants to variables

(defun trouble-in-env (e)
  (and e
       (loop for n in (cdr e) by #'cddr 
	if (and (not (zerop n))
		(< (abs n) min-unfixed-ramp))
	return t)))

(defun fix-one-ramp (e end)
  (if e					;since we have decided to use 24 bit fractions here (on the 56k),
					;we can easily end up with slopes that fall in the cracks.
					;I.e if we want to ramp (in integer-land) from 0 to 1 in 3 samps,
					;we need a slope of .333, but because we're doing integer arithmetic,
					;there is no such slope.  So, we divide the ramp into 2 pieces,
					;one for 1 sample at slope 1, and another slope 0 for 2 samples.
					;With 4 bits less on SAM, very few users noticed this was happening,
					;so we have high hopes we can squeeze by here, and save endless programming.
      (if (or (zerop (cadr e))
	      (> (abs (cadr e)) min-unfixed-ramp))
	  (append (list (car e) (cadr e)) 
		  (fix-one-ramp (cddr e) end))
	(multiple-value-bind 
	    (int frac) 
	    (floor (* (cadr e) ramp-max))
	  (if (< (abs frac) .001)	;we're close enough to integer slope, so don't mess with it
	      (append (list (car e) (cadr e)) 
		      (fix-one-ramp (cddr e) end))
	    ;; now figure out how many passes we have, and split between the two closest available ramps.
	    (let* ((ramp-dur (if (cddr e) 
				 (- (third e) (first e)) 
			       (- end (first e))))
		   (ramp0 (scale-float (float int) (- ramp-bits)))
		   (ramp1 (scale-float (float (+ int 1)) (- ramp-bits)))
		   (e1 (list (first e) ramp0)) ; still starting at same time (car=start of current ramp)
		   (e2 (list (+ (first e) (round (* ramp-dur (- 1.0 frac)))) ramp1)))
	      (append e1 e2 (fix-one-ramp (cddr e) end))))))))
  
(defun fix-envelope (e end)		; E=magified data as (pass-ctr rate ...), END=pass ctr at end
  ;; since re-consing the whole thing twice may be expensive, and shouldn't be needed except
  ;; in highly unusual cases, we first run through the data and check for trouble spots.
  (if (trouble-in-env e) (fix-one-ramp e end) e))




;;; DIVSEG replacement.
;;; return a new envelope taking into account the attack and decay times given
;;; for a version with like the old divseg (with up to 5 segments) see divenv

(defun divseg (fn old-att new-att &optional old-dec new-dec)
  (declare (optimize (speed 3) (safety 1)))
  (if (and old-dec (not new-dec)) (error "incorrect number of arguments to divseg"))
  (when fn
    (let* ((x0 (car fn))
	   (new-x x0)
	   (last-x (env-last-x fn))
	   (y0 (cadr fn))
	   (new-fn (list y0 x0))
	   (scl (/ (- new-att x0) (max .0001 (- old-att x0)))))
      (loop for x1 in (cddr fn) by #'cddr and
	        y1 in (cdddr fn) by #'cddr do
	(when (and (< x0 old-att)
		   (>= x1 old-att))
	  (if (= x1 old-att)
	      (setf y0 y1)
	    (setf y0 (float (+ y0 (* (- y1 y0) (/ (- old-att x0) (- x1 x0)))))))
	  (setf x0 old-att)
	  (setf new-x new-att)
	  (push new-x new-fn)
	  (push y0 new-fn)
	  (setf scl (if old-dec 
			(/ (- new-dec new-att) (- old-dec old-att))
		      (/ (- last-x new-att) (- last-x old-att)))))
	(when (and old-dec
		   (< x0 old-dec)
		   (>= x1 old-dec))
	  (if (= x1 old-dec)
	      (setf y0 y1)
	    (setf y0 (float (+ y0 (* (- y1 y0) (/ (- old-dec x0) (- x1 x0)))))))
	    (setf x0 old-dec)
	    (setf new-x new-dec)
	    (push new-x new-fn)
	    (push y0 new-fn)
	    (setf scl (/ (- last-x new-dec) (- last-x old-dec))))
	(when (/= x0 x1)
	  (incf new-x (float (* scl (- x1 x0))))
	  (push new-x new-fn)
	  (push y1 new-fn)
	  (setf x0 x1)
	  (setf y0 y1)))
      (nreverse new-fn))))
		
(defun divenv (env dur &optional p1 t1 p2 t2 p3 t3 p4 t4 p5 t5)
  (declare (optimize (speed 3) (safety 1)))
  (flet ((div-checkPt (att dur &optional (last-x 100.0))
	   (if att
	       (if (or (zerop att)
		       (minusp att))
		   (* last-x (/ .001 dur))
		 (if (< att dur)
		     (* last-x (/ att dur))
		   last-x))
	     last-x)))
    (flet ((div-insert-point (res env px)
	     (let ((newres nil)
		   (len (length res)))
	       (loop for x in res by #'cddr and
		         y in (cdr res) by #'cddr and
                         i from 2 by 2 do
		 (if (= px x) (return-from div-insert-point res)
		   (progn
		     (setf newres (append newres (list x y)))
		     (if (and (> px x)
			      (< i len)
			      (< px (nth i res)))
			 (setf newres (append newres (list px (float (list-interp px env)))))))))
	       newres)))
      (flet ((div-insert-points (env &optional p1 p2 p3 p4 p5)
	       (let ((result (copy-list env)))
		 (when p1
		   (setf result (div-insert-point result env p1))
		   (when p2
		     (setf result (div-insert-point result env p2))
		     (when p3
		       (setf result (div-insert-point result env p3))
		       (when p4
			 (setf result (div-insert-point result env p4))
			 (when p5
			   (setf result (div-insert-point result env p5)))))))
		 result)))
	(if (or (and p1 (not t1)) (and p2 (not t2)) (and p3 (not t3)) (and p4 (not t4)) (and p5 (not t5)))
	    (error "incorrect number of arguments to divenv"))
	(let ((last-X (env-last-x env)))
	  (when (and (< p1 last-X)
		     (not p2))
	    (setf t2 (max (- dur t1) 0.01))
	    (setf p2 last-X))
	  (when (and p2 
		     (< p2 last-X)
		     (not p3))
	    (setf t3 (max (- dur (+ t1 t2)) 0.01))
	    (setf p3 last-X))
	  (when (and p3
		     (< p3 last-X)
		     (not p4))
	    (setf t4 (max (- dur (+ t1 t2 t3)) 0.01))
	    (setf p4 last-X))
	  (when (and p4
		     (< p4 last-X)
		     (not p5))
	    (setf t5 (max (- dur (+ t1 t2 t3 t4)) 0.01))
	    (setf p5 last-X))
	  (let* ((dur-1 (max dur (+ (if t1 (max t1 .001) 0.0) 
				    (if t2 (max t2 .001) 0.0)
				    (if t3 (max t3 .001) 0.0)
				    (if t4 (max t4 .001) 0.0)
				    (if t5 (max t5 .001) 0.0))))
		 (np1 (if t1 (div-checkPt t1 Dur-1 last-X)))
		 (np2 (if (and t1 t2) (+ np1 (div-checkPt t2 Dur-1 last-X))))
		 (np3 (if (and t2 t3) (+ np2 (div-checkPt t3 Dur-1 last-X))))
		 (np4 (if (and t3 t4) (+ np3 (div-checkPt t4 Dur-1 last-X))))
		 (np5 (if (and t4 t5) (+ np4 (div-checkPt t5 Dur-1 last-X))))
		 (sc1 (if np1 (/ np1 p1)))
		 (sc2 (if np2 (/ (- np2 np1) (- p2 p1))))
		 (sc3 (if np3 (/ (- np3 np2) (- p3 p2))))
		 (sc4 (if np4 (/ (- np4 np3) (- p4 p3))))
		 (sc5 (if np5 (/ (- np5 np4) (- p5 p4))))
		 (totalenv (div-insert-points env p1 p2 p3 p4 p5))
		 (result (list (car env) (cadr env)))
		 (nX (car env)))
	    (if p1
		(progn
		  (loop for x in (cddr totalenv) by #'cddr and 
		            y in (cdddr totalenv) by #'cddr do
		    (setf nx (if (< x p1) (* x sc1)
			       (if (and p2 (< x p2)) (+ np1 (* (- x p1) sc2))
				 (if (and p3 (< x p3)) (+ np2 (* (- x p2) sc3))
				   (if (and p4 (< x p4)) (+ np3 (* (- x p3) sc4))
				     (if p5 (+ np4 (* (- x p4) sc5))
				       last-X))))))
		    (setf result (append result (list nx y)))))
	      (setf result (copy-list env)))
	    result))))))

(defun op-y-env (fn op)
  (do ((i 0 (+ i 2))
       (lim (length fn))
       (result (copy-list fn)))
      ((>= i lim) result)
      (setf (nth (+ i 1) result) (funcall op (nth (+ i 1) result)))))
	
(defun op-x-env (fn op)
  (do ((i 0 (+ i 2))
       (lim (length fn))
       (result (copy-list fn)))
      ((>= i lim) result)
      (setf (nth i result) (funcall op (nth i result)))))



;;; various simple waveforms (triangle needed for vibrato)
;;; the "fm" argument is scaled just like oscil or randh -- it assumes a period length of two-pi.
	
(defstruct (sw 
	    (:print-function
	     (lambda (d s k)
	       (declare (ignore k))
	       (format s "<Sw: freq: ~A (~A Hz), phase: ~A (~A deg), base: ~A, current-value: ~A>"
		       (Sw-freq d) (round (/ (Sw-freq d) frequency-mag))
		       (Sw-phase d) (round (* (Sw-phase d) (/ 360.0 two-pi)))
		       (Sw-base d) (Sw-current-value d)))))
  current-value freq phase base)
	
(defun fix-up-phase (s)
  (if (plusp (sw-phase s))
      (loop while (>= (sw-phase s) two-pi) do (decf (sw-phase s) two-pi))
    (loop while (minusp (sw-phase s)) do (incf (sw-phase s) two-pi))))

(defun make-triangle-wave (&key frequency (amplitude 1.0) (initial-phase nil))
  (setf initial-phase (check-initial-phase initial-phase))
  (make-sw :current-value (/ (tri-val amplitude (or initial-phase 0.0)) half-pi)
	   :base (/ (* 2 amplitude) one-pi)
	   :phase (or initial-phase 0.0)
	   :freq (in-Hz frequency)))
	
(defun triangle-wave (s &optional (fm 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (sw-current-value s)
    (incf (sw-phase s) (+ (sw-freq s) fm))
    (if (or (minusp (sw-phase s))
	    (>= (sw-phase s) two-pi))
	(fix-up-phase s))
    (setf (sw-current-value s) (tri-val (sw-base s) (sw-phase s)))))

(defun tri-val (amplitude phase)
  (declare (optimize (speed 3) (safety 1)))
  (* amplitude (if (< phase half-pi) phase 
		 (if (< phase three-half-pi) (- one-pi phase)
		   (- phase two-pi)))))


;;; old method using increments tended to wander (and was off by a factor of two)
	

(defun make-square-wave (&key frequency (amplitude 1.0) (initial-phase nil))
  (setf initial-phase (check-initial-phase initial-phase))
  (make-sw :current-value (if (not initial-phase) 0.0 (if (< initial-phase one-pi) 0.0 amplitude))
	   :base amplitude
	   :phase (or initial-phase 0.0)
	   :freq (in-Hz frequency)))

(defun square-wave (s &optional (fm 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (sw-current-value s)
    (incf (sw-phase s) (+ (sw-freq s) fm))
    (if (or (minusp (sw-phase s))
	    (>= (sw-phase s) two-pi)) 
	(fix-up-phase s))
    (setf (sw-current-value s) (if (< (sw-phase s) one-pi) (sw-base s) 0.0))))
	

(defun make-sawtooth-wave (&key frequency (amplitude 1.0) (initial-phase nil))
  (setf initial-phase (check-initial-phase initial-phase))
  (make-sw :current-value (if (not initial-phase) 0.0 (* amplitude (/ (- initial-phase one-pi) one-pi)))
	   :base (/ amplitude one-pi)
	   :phase (or initial-phase one-pi)
	   :freq (in-Hz frequency)))
	
(defun sawtooth-wave (s &optional (fm 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (sw-current-value s)
    (incf (sw-phase s) (+ (sw-freq s) fm))
    (if (or (minusp (sw-phase s))
	    (>= (sw-phase s) two-pi))
	(fix-up-phase s))
    (setf (sw-current-value s) (* (sw-base s) (- (sw-phase s) one-pi)))))


(defun make-pulse-train (&key frequency (amplitude 1.0) (initial-phase nil))
  (setf initial-phase (check-initial-phase initial-phase))
  (make-sw :current-value 0.0
	   :base amplitude
	   :phase (or initial-phase two-pi)
					; this will give us an immediate pulse
	   :freq (in-Hz frequency)))

(defun pulse-train (s &optional (fm 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (if (>= (abs (sw-phase s)) two-pi)
	  (progn
	    (fix-up-phase s)
	    (sw-base s))
	0.0)
    (incf (sw-phase s) (+ (sw-freq s) fm))))


;;; Waveshaping (just a matter of loading an array with the polynomial, then using Osc output 
;;;    with offset to drive array-interp)
;;;    see "Digital Waveshaping Synthesis" by Marc Le Brun in JAES 1979 April, vol 27, no 4, p250

(defun signify (Harm-amps)		;taken very directly from MLB's Mus10 code.  
					;Here we assume Harm-amps is ordered by partial number.
  (let ((lastpt (array-dimension Harm-amps 0)))
    (do ((i 2 (+ i di))
	 (di 1 (- 4 di)))		;successively 1 and 3, getting the pattern + + - - + + ...
	((>= i LastPt) Harm-amps)
      (setf (aref Harm-amps i) (- (aref Harm-amps i))))))


; T(n+1) <= 2xT(n)-T(n-1) gives the Chebychev polynomials of the first kind 
; (T0 = 1, T1 = X to get recursion going)

; we take the array of signified partial amplitudes (Harm-amps) and use them to weight the 
; associated Chebychev polynomial

(defvar waveshaping-table-size (+ default-table-size 1))

(defun Make-waveshape-table (Harm-amps &optional (norm t) (TblSiz waveshaping-table-size))
  (declare (optimize (speed 3) (safety 1)))
  (let ((F (make-table tblsiz)))
    (do* ((MaxI (- tblsiz 1))
	  (MaxHarm (array-dimension Harm-amps 0))
	  (MaxI-2 (/ 2.0 MaxI))
	  (i 0 (+ i 1)))
	((> i MaxI) F)
      (do* ((Hnum 0 (+ Hnum 1))		; current harmonic number
	    (sum 0.0)			; collects all contributions to current F[i]
	    (temp 0.0)
	    (x (- (* i MaxI-2) 1))	; -1 <= x <= 1 -- fill up F with this interval of the summed polynomial
	    (Tn 1.0)			; now set up Chebychev recursion
	    (Tn1 x))
	  ((= Hnum MaxHarm) (setf (aref F i) sum))
	(if (/= 0.0 (aref Harm-amps Hnum))
	    (incf sum (* Tn (aref Harm-amps Hnum))))
	;; sums the current contribution of the Hnum-th partial to this point in the table
	(setf temp Tn1)			; now crank the recursion one step
	(setf Tn1 (- (* 2.0 Tn1 x) Tn))
	(setf Tn temp)))
    (if norm (normalize F))
					;    (if norm (c-normalize (array-dimension F 0) F))
    F))

;;; assume we're using synth-data that looks like additive synthesis tables 
;;; (i.e. a list of partial-amp pairs).  That means we have to prepare it for 
;;; the preceding two procedures by loading it into an array.

(defun normalize-partials (partials)
  (declare (optimize (speed 3) (safety 1)))
  (let ((sum 0.0))
    (loop for i in (cdr partials) by #'cddr do (incf sum (abs i)))
    (if (zerop sum) (warn "all partials have 0.0 amplitude: ~A" partials))
    (setf sum (/ 1.0 sum))
    (do ((i 1 (+ i 2)))
	((>= i (length partials)) partials)
      (setf (nth i partials) (* (nth i partials) sum)))))

(defun highest-partial (data)
  (declare (optimize (speed 3) (safety 1)))
  (if (endp data) 0.0
    (max (car data) (highest-partial (cddr data)))))

(defun massage-partials (data)
  (declare (optimize (speed 3) (safety 1)))
  (do* ((i 0 (+ i 2))
	(lim (length data))
	(maxH (highest-partial data))
	(hamps (make-table (+ maxH 1))))
      ((>= i lim) hamps)
    (setf (aref hamps (nth i data)) (float (nth (+ i 1) data)))))

(defstruct ws tab os offset)

(defun make-waveshape (&key (frequency 440.0) (partials '(1 1)))
  (make-ws :tab (make-waveshape-table 
		 (signify 
		  (massage-partials 
		   (normalize-partials partials))))
	   :offset (floor (* default-table-size .5))
	   :os (make-oscil :frequency frequency)))

(defun table-interp (fn x &optional (size 0)) (array-interp fn x size))

(defmacro waveshape (w &optional (index 1.0) (fm 0.0))
  `(table-interp (ws-tab ,w) (* (ws-offset ,w) (+ 1.0 (* ,index (oscil (ws-os ,w) ,fm))))))


;;; phase-quadrature waveshaping involves Fx and Fy tables, then 
;;;    (+ (array-interp Fx index*COS) (* index*SIN (array-interp Fy index*COS)))

(defun make-phase-quad-table (Harm-amps phases &optional (tblsiz default-table-size))
  (let* ((Fx (make-table tblsiz))
	 (Fy (make-table tblsiz))
	 (x 0.0) (Tk 0.0) (Uk 0.0) (Tk1 0.0) (Uk1 0.0) (Tk2 0.0) (Uk2 0.0)
	 (lim (array-dimension Harm-amps 0))
	 (Cphi (make-table lim))
	 (Sphi (make-table lim)))
    (loop for i below lim do 
      (setf (aref CPhi i) (* (aref harm-amps i) (cos (aref phases i))))
      (setf (aref SPhi i) (* (aref harm-amps i) (sin (aref phases i)))))
    (loop for i below tblsiz do
      (setf x (- (* 2 (/ i tblsiz)) 1))	;map into signed unit interval as before
      (setf Tk 1.0)			;initialize Chebychev recursion of first and second kind
      (setf Uk 0.0)
      (setf Tk1 x)
      (setf Uk1 -1.0)
      (loop for k below lim do		;sum up contributions for both tables at this point (i)
	(incf (aref Fx i) (* Tk (aref CPhi k)))
	(incf (aref Fy i) (* Uk (aref SPhi k)))
	(setf Tk2 Tk1)			;crank recursion 
	(setf Tk1 Tk)
	(setf Tk (- (* 2 x Tk1) Tk2))
	(setf Uk2 Uk1)
	(setf Uk1 Uk)
	(setf Uk (- (* 2 x Uk1) Uk2))))
    (values Fx Fy)))			;return both new tables

;;; on the 56000 chip, it is faster to load down the polynomial coefficients and let the chip evaluate
;;; the polynomial, rather than do a table lookup (the table load time is prohibitive).

(defun partial-amp (n partials)
  (loop for i on partials by #'cddr do
    (if (= n (car i)) (return (cadr i)))))

(defun get-chebychev-coefficients (partials &optional (kind 1))
  ;;assume that partials are normalized and signified already (neither is vital)
  (declare (optimize (speed 3) (safety 1)))
  (let* ((top (floor (highest-partial partials)))
	 (size (+ top 1))
	 (T0 (make-array size :element-type 'integer :initial-element 0))
	 (T1 (make-array size :element-type 'integer :initial-element 0))
	 (Tn (make-array size :element-type 'integer :initial-element 0))
	 (Cc (make-array size :element-type 'short-float :initial-element 0.0))
	 (amp 0.0))
    (setf (aref T0 0) kind)
    (setf (aref T1 1) 1)		;initialize Tn recursion (0 in T0 is Un)
    (loop for i from 1 to top do	;linear combination of polynomials weighted by harmonic amplitude
      (setf amp (or (partial-amp i partials) 0.0))
      (when (/= 0.0 amp)
	(if (= kind 1)
	    (loop for k from 0 to i do (incf (aref Cc k) (* amp (aref T1 k))))
	  (loop for k from 1 to i do (incf (aref Cc (- k 1)) (* amp (aref T1 k))))))
      (when (/= i top)
	(loop for k from (+ i 1) downto 1 do
	  (setf (aref Tn k) (- (* 2 (aref T1 (- k 1))) (aref T0 k))))
	(setf (aref Tn 0) (- (aref T0 0)))
	(loop for k from (+ i 1) downto 0 do
	  (setf (aref T0 k) (aref T1 k))
	  (setf (aref T1 k) (aref Tn k)))))
    Cc))

(defun evaluate-polynomial (x coeffs) 
  (polynomial coeffs x))

(defun polynomial (coeffs x)
  (declare (optimize (speed 3) (safety 1)))
  (let* ((top (- (array-dimension coeffs 0) 1))
	 (sum (aref coeffs top)))
    (loop for i from (- top 1) downto 0 do
      (setf sum (+ (* x sum) (aref coeffs i))))
    sum))
	 
;;; this version of waveshaping is:
;;; (evaluate-polynomial (* index (coscil os)) (coeff-table))
;;; similarly, we can get the phase quadrature business by setting "kind" to 0,
;;; (+ (evaluate-polynomial (* index (coscil os)) (first-kind-table))
;;;    (* (oscil os) (evaluate-polynomial (* index (coscil os)) (second-kind-table))))
;;; see ins.lisp for examples



;;; Sum of cosines (a Sambox mode)
;;;   this is a bit of a kludge to generate band-limited pulses using the formula
;;;   1+2(cos(x)+cos(2x)+...cos(nx)) = sin((n+.5)x)/sin(x/2)

(defstruct cosp cosines scaler phase freq)

(defun make-sum-of-cosines (&key (cosines 1) (frequency 440.0) (initial-phase 0.0))
  (let ((cs (make-cosp :cosines cosines
		       :freq (* frequency frequency-mag)
		       :phase (check-initial-phase initial-phase))))
    (if (zerop cosines) (warn "sum-of-cosines with 0 cosines?"))
    (setf (cosp-scaler cs) (/ 1.0 (+ 1 (* 2 cosines))))
    cs))

(defun sum-of-cosines (cs &optional (fm 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1 
      (if (or (= 0.0 (cosp-phase cs)) (= two-pi (cosp-phase cs)))
	  1.0
	(/ (* (cosp-scaler cs) (sin (* (cosp-phase cs) (+ (cosp-cosines cs) .5)))) (sin (* (cosp-phase cs) .5))))
    (incf (cosp-phase cs) (+ (cosp-freq cs) fm))
    (if (> (cosp-phase cs) two-pi) (decf (cosp-phase cs) two-pi))
    (if (< (cosp-phase cs) (- two-pi)) (incf (cosp-phase cs) two-pi))))


;;; There are lots of formulas like the one used above.  Among those that have been explored are J A Moorer's
;;; sine summation formulas, extended by Palamin et al to be closer to FM, and waveshaping of Arfib and Le Brun.


;;; Sine summation synthesis.

(defstruct sss phase freq fm a N B aN a2)
;;; if N=nil assume infinity

(defun make-sine-summation (&key (N 1) (a .5) (B-ratio 1.0) (frequency 440.0) (initial-phase 0.0))
  (make-sss :freq (* frequency frequency-mag)
	    :phase (check-initial-phase initial-phase)
	    :aN (if N (expt a (+ N 1)) 0.0)
	    :a2 (+ 1 (* a a))
	    :a a :N N :B B-ratio))

(defun sine-summation (s &optional (fm 0.0))
  (let* ((th (sss-phase s))		;for readability in the formulas below
	 (a (sss-a s))
	 (N (sss-N s))
	 (B (* (sss-B s) th))
	 (thB (- th B))
	 (divisor (- (sss-a2 s) (* 2 a (cos B)))))
    (prog1
	(if (not N)
	    (/ (- (sin th) (* a (sin thB))) divisor)
	  (/ (- (sin th) (* a (sin thB)) (* (sss-aN s) (- (sin (+ th (* (+ N 1) B))) (* a (sin (+ th (* N B))))))) divisor))
      (incf (sss-phase s) (+ (sss-freq s) fm))
      (if (> (sss-phase s) two-pi) (decf (sss-phase s) two-pi))
      (if (< (sss-phase s) (- two-pi)) (incf (sss-phase s) two-pi)))))


;;; Moorer also suggests use of the following formula:
;;; (* (exp (* a (cos b))) (sin (+ th (* a (sin b))))) to get a weighted sum of sines (see "Signal Processing
;;;     Aspects of Computer Music", but this formula has more recently been extended by Palamin and Palamin,
;;;     "A Method of Generating and Controlling Asymmetrical Spectra" JAES vol 36, no 9, Sept 88, p671-685:

(defstruct asymfm r freq ratio phase cosr sinr)

(defun make-asymmetric-fm (&key (r 1.0) (ratio 1.0) (frequency 440.0) (initial-phase 0.0))
  (make-asymfm :r r
	       :freq (* frequency frequency-mag)
	       :ratio ratio
	       :phase (check-initial-phase initial-phase)
	       :cosr (* .5 (- r (/ 1.0 r)))
	       :sinr (* .5 (+ r (/ 1.0 r)))))
			
(defun asymmetric-fm (af index fm)
  (let* ((th (asymfm-phase af))
	 (mth (* (asymfm-ratio af) th))
	 (cr (asymfm-cosr af))
	 (sr (asymfm-sinr af))
	 (result (* (exp (* index cr (cos mth))) (sin (+ th (* sr (sin mth)))))))
    (incf (asymfm-phase af) (+ (asymfm-freq af) fm))
    (if (> (asymfm-phase af) lotsa-pi) (decf (asymfm-phase af) lotsa-pi))
    (if (< (asymfm-phase af) (- lotsa-pi)) (incf (asymfm-phase af) lotsa-pi))
    result))

;;; amplitude normalization for this is complicated.
;;; say we have bessi0 = modified Bessel (see num.lisp), then a normalized version of this asymmetric fm is:
;;;
;;;	 (result (* (exp (- (* index cr (cos mth))
;;;                         (* .5 (log (bessi0 (* index cr))))))      ;this is the normalization                    
;;;                 (sin (+ th (* sr (sin mth)))))))
;;;
;;; If there's interest in this, I will implement it on the DSP chip (see I0-load in lib56.lisp for 56000 code)

;;; There are lots more such formulas -- MLB in "Digital Waveshaping Synthesis" JAES vol 27 no 4 Apr 79 suggests:
;;; (* (exp ax) (sin ay)) ; in context of Chebychev stuff (for ax and ay).  See the phase quadrature
;;; waveshaping instrument in ins.lisp for asymmetric spectra using waveshaping (more general, and perhaps
;;; cheaper than the asymmetric fm stuff above).

#+MCL(defun bessi0 (x) (declare (ignore x)) (error "fix me"))
#-MCL (defun bessi0 (x)			;I0(x) from "Numerical Recipes" by Press et al
  (if (< (abs x) 3.75)
      (let* ((y (expt (/ x 3.75) 2)))
	(+ 1.0
	   (* y (+ 3.5156229
		   (* y (+ 3.0899424
			   (* y (+ 1.2067492
				   (* y (+ 0.2659732
					   (* y (+ 0.360768e-1
						   (* y 0.45813e-2)))))))))))))
    (let* ((ax (abs x))
	   (y (/ 3.75 ax)))
      (* (/ (exp ax) (sqrt ax)) 
	 (+ 0.39894228
	    (* y (+ 0.1328592e-1
		    (* y (+ 0.225319e-2
			    (* y (+ -0.157565e-2
				    (* y (+ 0.916281e-2
					    (* y (+ -0.2057706e-1
						    (* y (+ 0.2635537e-1
							    (* y (+ -0.1647633e-1
								    (* y 0.392377e-2))))))))))))))))))))


;;; LOCSIG
;;; "placement" in speakers (i.e. take degree and distance and pretend to conjure up some amplitudes
;;; before sending the signal out the speakers.  This (despite its name) gives you a very diffuse
;;; apparent source, and under normal conditions, that is exactly the right thing.

(defstruct locs deg dis pc-rev ascl bscl cscl dscl rscl revname)

(defun make-locsig (&key (degree 0.0) (distance 1.0) (revscale 0.01) (revin nil))
  (if (not (quad))
      (let* ((frac (if (mono) 
		       0.0 
		     (/ (min 90.0 (max 0.0 degree)) 90.0)))
	     (dist (/ 1.0 (max distance 1.0)))
	     (sdist (/ 1.0 (sqrt (max distance 1.0))))
	     (ascl (* dist (- 1.0 frac)))
	     (bscl (* dist frac))
	     (rscl (* revscale sdist)))
	(make-locs :deg degree
		   :dis distance
		   :pc-rev revscale
		   :ascl ascl
		   :bscl bscl
		   :rscl rscl
		   :revname (or revin *reverb*)))
    (make-quadsig :degree degree :distance distance :revscale revscale :revin revin)))

(defun make-quadsig (&key (degree 0.0) (distance 1.0) (revscale 0.01) (revin nil))
  (let* ((dist (/ 1.0 (max distance 1.0)))
	 (sdist (/ 1.0 (sqrt (max distance 1.0))))
	 (ascl (if (<= 0 degree 90)
		   (* dist (/ (- 90 degree) 90.0))
		 (if (<= 270 degree 360)
		     (* dist (/ (- degree 270) 90))
		   0)))
	 (bscl (if (<= 90 degree 180)
		   (* dist (/ (- 180 degree) 90.0))
		 (if (<= 0 degree 90)
		     (* dist (/ degree 90))
		   0)))
	 (cscl (if (<= 180 degree 270)
		   (* dist (/ (- 270 degree) 90.0))
		 (if (<= 90 degree 180)
		     (* dist (/ (- degree 90) 90))
		   0)))
	 (dscl (if (<= 270 degree 360)
		   (* dist (/ (- 360 degree) 90.0))
		 (if (<= 180 degree 270)
		     (* dist (/ (- degree 180) 90))
		   0)))
	 (rscl (* revscale sdist)))
    (make-locs :deg degree
	       :dis distance
	       :pc-rev revscale
	       :ascl ascl
	       :bscl bscl
	       :cscl cscl
	       :dscl dscl
	       :rscl rscl
	       :revname (or revin *reverb*))))

(defun locsig (l i in-sig)
  (declare (optimize (speed 3) (safety 1)))
  (outa i (* in-sig (locs-ascl l)))	;scale for distance even if one channel
  (if (or (stereo) (quad)) 
      (outb i (* in-sig (locs-bscl l))))
  (when (quad)
    (outc i (* in-sig (locs-cscl l)))
    (outd i (* in-sig (locs-dscl l))))
  (if (locs-revname l) (outa i (* in-sig (locs-rscl l)) (locs-revname l))))


;;; ROOMSIG -- attempt to more accurately imitate the initial echos.  Here we assume a rectangular room.
;;; see "Extension of the Image Model to Arbitrary Polyhedra" by J. Borish  JASA 75(6) June 84.

(defvar speed-of-sound 344)		;in air in meters per second (under normal conditions)
(defvar width-of-head .15)		;meters -- close to width of my head, I believe
(defvar half-head .075)
(defvar reflection-coefficient .95)
;(defvar water-speed-of-sound 1460)	;meters/sec at 20 C (goes up to around 1500 at 27 C)

(defvar sound-mag 0)

(defun set-sound-mag ()			;conversion factor for meters/sec to samples/meter
  (setf sound-mag (/ sampling-rate speed-of-sound)))

(defun in-samples (x)			;x = distance in meters, converts to samples
  (* x sound-mag))

(defun attenuation (distS distVS refl)	;distS=dist of source, distVS=dist of virtual source, refl=number of reflections
  (* (/ distS distVS) (expt reflection-coefficient refl)))

(defun distance (x y)			;pythagorean theorem
  (sqrt (+ (* x x) (* y y))))

(defun describe-VS (x y refl distS)	;(x,y)=position relative to listener's nose
  (let ((left-ear-dist (distance (if (minusp x) (- (abs x) half-head) (+ x half-head)) y))
	(right-ear-dist (distance (if (minusp x) (+ (abs x) half-head) (- x half-head)) y)))
    (values (in-samples left-ear-dist)
	    (in-samples right-ear-dist)
	    (attenuation distS left-ear-dist refl)
	    (attenuation distS right-ear-dist refl)
	    (if (minusp x)
		(if (minusp y)
		    "LB" "LF")
	      (if (minusp y)
		  "RB" "RF")))))

(defun max-room (time W L)		;given room size and time of last desired echo, get max of virtual rooms
  (let* ((radius (* time speed-of-sound))
	 (maxW (floor radius W))
	 (maxL (floor radius L)))
    (values maxW maxL radius)))

(defun get-all-VS-blocks (rm Nw Nl)	;march through all the virtual rooms
  (loop for i from 0 to Nw do
    (loop for j from 0 to Nl do
	(get-one-VS-block rm i j))))

(defun get-one-VS-block (rm w l)
  (get-VS rm w l)
  (if (/= 0 w) (get-VS rm (- w) l))
  (if (/= 0 l) (get-VS rm w (- l)))
  (if (and (/= 0 w) (/= 0 l)) (get-VS rm (- w) (- l))))

(defstruct rmloc W L uW uL rdiff ldiff fdiff bdiff data distS sxdiff sydiff minR)

(defun get-VS (rm w l)			;add info for this VS to the data array in rm
  (flet ((get-x-s (rm w)
	   (if (zerop w) 0
	     (if (oddp w)
		 (if (minusp w)
		     (- (+ (* 2 (rmloc-ldiff rm)) (* (rmloc-W rm) (1- (abs w)))))
		   (+ (* 2 (rmloc-rdiff rm)) (* (rmloc-W rm) (1- w))))
	       (* w (rmloc-W rm)))))
	 (get-y-S (rm l)
	   (if (zerop l) 0
	     (if (oddp l)
		 (if (minusp l)
		     (- (+ (* 2 (rmloc-bdiff rm)) (* (rmloc-L rm) (1- (abs l)))))
		   (+ (* 2 (rmloc-fdiff rm)) (* (rmloc-L rm) (1- l))))
	       (* l (rmloc-L rm))))))
    (flet ((get-xx (rm w) (+ (get-x-S rm w) (rmloc-sxdiff rm)))
	   (get-yy (rm l) (+ (get-y-S rm l) (rmloc-sydiff rm)))
	   (get-refl (rm w l) (declare (ignore rm)) (+ (abs w) (abs l)))
	   (add-VS (rm l r vl vr quad)
	     (if (>= (max vl vr) (rmloc-MinR rm))
		 (push (list l r vl vr quad) (rmloc-data rm)))))
      (multiple-value-bind
	  (l r vl vr quad) (describe-VS (get-xx rm w) 
					(get-yy rm l) 
					(get-refl rm w l) 
					(rmloc-distS rm))
	(add-VS rm l r vl vr quad)))))

;; 2*diff+W*(w-1) or W*w (ldiff rdiff fdiff bdiff)

(defun make-roomsig (W L uW uL sW sL time) ;width of room, length of room, 
					;user-X user-Y (from lower left corner), 
					;source-X source-Y, seconds of echos to compute

  (if (or (minusp W) (minusp L)) (error "impossible room dimensions")
    (if (or (> uW W) (> uL L)) (error "listener is outside the room")
      (if (or (minusp uW) (minusp uL)) (error "impossible listener position")
	(if (or (> sW W) (> sL L)) (error "source is outside room")
	  (if (or (minusp sW) (minusp sL)) (error "impossible source position")
	    (if (and (= sW uW) (= sL uL)) (error "source and listener cannot be at same location")))))))

  (set-sound-mag)
  (multiple-value-bind 
      (mW mL rad) (max-room time W L)
    (let ((rm (make-rmloc :W W
			  :L L
			  :uW uW
			  :uL uL
			  :rdiff (- W sW)
			  :ldiff sW
			  :fdiff (- L sL)
			  :bdiff sL
			  :distS (distance (- sW uW) (- sL uL))
			  :sxdiff (- sW uW)
			  :sydiff (- sL uL)
			  :data nil)))
      (setf (rmloc-minR rm) (min 0.95 (/ (rmloc-DistS rm) rad)))
      (get-all-VS-blocks rm mW mL)
      rm)))

;;; for some examples of instruments that use ROOMSIG, see room.ins.

;;; arbitrary shapes are slightly harder.  The following code provides the support for handling them (in 2 dimensions)

(defun intersection-of-line-with-perpendicular-from-point (x0 y0 x1 y1 sx sy)
  ;; find the point of intersection with the line between (x0 y0) and (x1 y1) and a perpendicular to it dropped from (sx sy)
  ;; this gives the first part of the calculation of the virtual source (i.e. reflection) given any current source (sx sy) and any line
  ;; the line equation is y = (d-b)/(c-a) x + (bc-ad)/(c-a) 
  ;; the perpendicular to it that passes through (e f) is y = f + (c-a)/(d-b) e - (c-a)/(d-b) x
  ;; we can solve these two equations for x, then plug in that value to get y (to get the point of intersection)
  ;; x = (f + (c-a)/(d-b) e - (bc-ad)/(c-a)) / ((d-b)/(c-a) + (c-a)/(d-b))
  ;; all this algebra is just an explicit solution of the vector equations given in Borish's article.
  (if (= x0 x1)
      (values x0 sy)
    (if (= y0 y1)
	(values sx y0)
      (let* ((c-a (- x1 x0))
	     (d-b (- y1 y0))
	     (s1 (/ c-a d-b))
	     (s2 (/ d-b c-a))
	     (bc (* y0 x1))
	     (ad (* x0 y1))
	     (offset (/ (- bc ad) c-a))
	     (inX (/ (+ (* s1 sx) sy (- offset)) (+ s1 s2)))
	     (inY (+ (* inX s2) offset)))
	(values inX inY)))))

(defun reflection-of-point-through-line (x0 y0 x1 y1 sx sy)
  (multiple-value-bind
      (inX inY) (intersection-of-line-with-perpendicular-from-point x0 y0 x1 y1 sx sy)
    (values (- (* 2 inX) sx)
	    (- (* 2 inY) sy))))

;;; this function can also be used to get the new virtual room vertices -- just reflect each original vertex through the current side.



;;; SIGNAL PROCESSING (as opposed to synthesis stuff above)

;;; "read data" -- sound file reader -- the user must provide us with INA and INB to do input from a file,
;;; just like OUTA and OUTB to output to a file.  Once read-in (we assume we get a short-float from INA)
;;; we can treat it in any way desired.


(defstruct smp fil sr x lst nxt i chn)

(defun clm-get-run-time-resample-index (rd)
  (if (and rd
	   (smp-fil rd)
	   (or (and (= (smp-chn rd) 0) (IO-dat-a (smp-fil rd)))
	       (and (= (smp-chn rd) 1) (IO-dat-b (smp-fil rd)))
	       (and (= (smp-chn rd) 2) (IO-dat-c (smp-fil rd)))
	       (and (= (smp-chn rd) 3) (IO-dat-d (smp-fil rd)))))
      (+ (ash (smp-chn rd) 8)
	 (ash (clm-get-run-time-file-index (smp-fil rd) (smp-chn rd)) 16))
    -1))

(defun make-resample (&key file (srate 1.0) start-time start input-file-start input-file-start-time (channel :A))
  (let* ((x (or input-file-start start 
		(and (or input-file-start-time start-time) (* (or input-file-start-time start-time) sampling-rate)) 
		0))
	 (i (floor x))
	 (s (make-smp  :sr srate
		       :i i
		       :x x
		       :lst 0.0
		       :nxt 0.0
		       :chn (if (or (eq channel :A) (eq channel 'A)) 0
			      (if (or (eq channel :B) (eq channel 'B)) 1
				(if (or (eq channel :C) (eq channel 'C)) 2 3)))
		       :fil file)))
    (if (and (> (smp-chn s) 2)
	     (not (quad (smp-fil s))))
	(error "~S is ~A" file (if (stereo (smp-fil s)) "stereo" "mono")))
    (if (and (not (zerop (smp-chn s)))
	     (not (stereo (smp-fil s))))
	(error "~S is mono." file))
    (if (zerop (smp-chn s))
	(progn
	  (setf (smp-lst s) (IN-A i (smp-fil s)))
	  (setf (smp-nxt s) (IN-A (+ i 1) (smp-fil s))))
      (if (= (smp-chn s) 1)
	  (progn
	    (setf (smp-lst s) (IN-B i (smp-fil s)))
	    (setf (smp-nxt s) (IN-B (+ i 1) (smp-fil s))))
	(if (= (smp-chn s) 2)
	    (progn
	      (setf (smp-lst s) (IN-C i (smp-fil s)))
	      (setf (smp-nxt s) (IN-C (+ i 1) (smp-fil s))))
	  (if (= (smp-chn s) 3)
	      (progn
		(setf (smp-lst s) (IN-D i (smp-fil s)))
		(setf (smp-nxt s) (IN-D (+ i 1) (smp-fil s))))))))
    s))
	
(defun resample (s &optional (sr-change 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (prog1
      (+ (smp-lst s) (* (- (smp-x s) (smp-i s))
			(- (smp-nxt s) (smp-lst s))))
    (incf (smp-x s) (+ (smp-sr s) sr-change))
    ;; that is, "smp-sr" is the steady amount added to the index on each sample (analogous to frequency in an oscil)
    ;; and "sr-change" is like the fm input to an oscil
    (when (/= (smp-i s) (floor (smp-x s)))
      (setf (smp-i s) (floor (smp-x s)))
      (if (zerop (smp-chn s))
	  (progn
	    (setf (smp-lst s) (IN-A (smp-i s) (smp-fil s)))
	    (setf (smp-nxt s) (IN-A (+ (smp-i s) 1) (smp-fil s))))
	(if (= (smp-chn s) 1)
	    (progn
	      (setf (smp-lst s) (IN-B (smp-i s) (smp-fil s)))
	      (setf (smp-nxt s) (IN-B (+ (smp-i s) 1) (smp-fil s))))
	  (if (= (smp-chn s) 2)
	      (progn
		(setf (smp-lst s) (IN-C (smp-i s) (smp-fil s)))
		(setf (smp-nxt s) (IN-C (+ (smp-i s) 1) (smp-fil s))))
	    (if (= (smp-chn s) 3)
		(progn
		  (setf (smp-lst s) (IN-D (smp-i s) (smp-fil s)))
		  (setf (smp-nxt s) (IN-D (+ (smp-i s) 1) (smp-fil s)))))))))))

    
;;; "real" sampling rate conversion should probably do the interpolation-filter-decimation
;;; shuffle using some FIR filter to do the whole operation (see "Multirate Digital Signal
;;; Processing" by Crochiere and Rabiner), The following code is based on kindly advice of
;;; Perry Cook (all errors are my fault).  We convolve the original with a sinc function to
;;; get the "ideal" low pass filter, then resample at the new sampling rate.  If we are
;;; going down in srate, we also have to low pass the original to half the new rate (this is
;;; also a low pass filter -- it could be convolved into the conversion filter to save computation).

(defun sinc (x) (if (zerop x) 1.0 (let ((pi-x (* pi x))) (/ (sin pi-x) pi-x))))

(defstruct sr rd x incr data (width 5) (left 0) (right -1) (filt nil))

(defun make-src (&key file
		      (srate 1.0)
		      (channel :A)
		      start-time input-file-start-time
		      start input-file-start
		      (width 5))
  (make-sr :rd (make-readin :file file :start-time (or input-file-start-time start-time) :start (or input-file-start start) :channel channel)
	   :x (or start (and (or input-file-start-time start-time) (* (or input-file-start-time start-time) sampling-rate)) 0)
	   :incr srate
	   :width width
	   :filt (if (> srate 1.0) (let ((cs (ceiling srate))) (make-table (if (oddp cs) cs (1+ cs)))))
	   ;;simplest low-pass is probably modified least-squares
	   ;; i.e (1/2n [sum of 2n+1 points where end points are at .5] -- zeros at pi/n)
	   :data (make-array (1+ (* width 2)) :element-type 'short-float :initial-element 0.0)))

(defun src (s &optional (sr-change 0.0))
  ;; get data window lined up right, convolve with sinc (and lp-filter too if needed)
  (declare (optimize (speed 3) (safety 1)))
  (flet ((smoothed-table (d input)
	   (let ((size (array-dimension d 0)))
	     (if (= size 1)
		 input
	       (let ((top (1- size)))
		 (loop for i from 0 below top do
		   (setf (aref d i) (aref d (1+ i))))
		 (setf (aref d top) input)
		 (let* ((sum (* .5 (+ (aref d 0)
				      (aref d top)))))
		   (loop for i from 1 below top do
		     (incf sum (aref d i)))
		   (/ sum top)))))))
    (let* ((sum 0.0)
	   (loc 0)
	   (lim (* 2 (sr-width s)))
	   (start-x (- (sr-x s) (sr-width s)))
	   (fsx (max 0 (floor start-x))))
      (when (or (< (sr-left s) fsx)
		(> (+ fsx lim) (sr-right s)))
	;; align data correctly, readin as needed (with optional filter), reset bounds
	(if (<= fsx (sr-right s))
	    (progn
	      (loop for i from fsx to (sr-right s) and k from (- lim (- (sr-right s) fsx)) do
		(setf (aref (sr-data s) loc) (aref (sr-data s) k))
		(incf loc)))
	  (if (/= (sr-right s) -1)
	      (loop for i from (sr-right s) below fsx do (readin (sr-rd s)))))
	(loop for i from loc to lim do
	  (setf (aref (sr-data s) i) 
	    (if (sr-filt s) 
		(smoothed-table (sr-filt s) (readin (sr-rd s)))
	      (readin (sr-rd s)))))
	(setf (sr-left s) fsx)
	(setf (sr-right s) (+ fsx lim)))
      ;; now dot-product with sinc
      (loop for i from 0 to lim and j from (- (sr-left s) (sr-x s)) do 
	(incf sum (* (aref (sr-data s) i) (sinc j))))
      (incf (sr-x s) (+ (sr-incr s) sr-change))
      sum)))
  

;;; contrast enhancement

(defmacro contrast-enhancement (in-samp &optional (fm-index 1.0))
  `(let ((var ,in-samp))		;don't evaluate in-samp twice (might be expression with side-effects)
     (sin (+ (* var 1.5707964)
	   (* ,fm-index (sin (* var 6.2831855)))))))


;;; readin

(defstruct rdin i chn fil inc)

(defun clm-get-run-time-readin-index (rd)
  (if (and rd
	   (rdin-fil rd)
	   (or (and (= (rdin-chn rd) 0) (IO-dat-a (rdin-fil rd)))
	       (and (= (rdin-chn rd) 1) (IO-dat-b (rdin-fil rd)))
	       (and (= (rdin-chn rd) 2) (IO-dat-c (rdin-fil rd)))
	       (and (= (rdin-chn rd) 3) (IO-dat-d (rdin-fil rd)))))
      (+ (ash (rdin-chn rd) 8)
	 (ash (clm-get-run-time-file-index (rdin-fil rd) (rdin-chn rd)) 16))
    -1))

(defun make-readin (&key file start-time start input-file-start-time input-file-start (channel :A))
  (if file (clm-check-file file))
  (make-rdin :fil file
	     :inc 1
	     :i (or input-file-start start 
		    (and (or input-file-start-time start-time) 
			 (floor (* (or input-file-start-time start-time) sampling-rate))) 0)
	     :chn (if (or (eq channel :A) (eq channel 'A)) 0
		    (if (or (eq channel :B) (eq channel 'B)) 1
		      (if (or (eq channel :C) (eq channel 'C)) 2 3)))))

(defun readin (rd)
  (declare (optimize (speed 3) (safety 1)))
  (if (rdin-fil rd)
      (prog1
	  (case (rdin-chn rd)
	    (0 (in-a (rdin-i rd) (rdin-fil rd)))
	    (1 (in-b (rdin-i rd) (rdin-fil rd)))
	    (2 (in-c (rdin-i rd) (rdin-fil rd)))
	    (3 (in-d (rdin-i rd) (rdin-fil rd))))
	(incf (rdin-i rd) (rdin-inc rd)))
    (- 1.0 (random 2.0))))

(defun make-reverse (&key file start-time start input-file-start-time input-file-start (channel :A))
  (clm-check-file file)
  (make-rdin :fil file
	     :i (or input-file-start start 
		    (and (or input-file-start-time start-time)
			 (floor (* (or input-file-start-time start-time) sampling-rate))) 
		    (1- (clm-get-samples file)))
	     :chn (if (or (eq channel :A) (eq channel 'A)) 0
		    (if (or (eq channel :B) (eq channel 'B)) 1
		      (if (or (eq channel :C) (eq channel 'C)) 2 3)))
	     :inc -1))

(defun readin-reverse (rd)
  (readin rd))



;;; BLOCK PROCESSING -- fft-based filtering, "expansion", wave-trains, etc
;;;   In these, we actually move forward by blocks, so to fit the style of
;;;   work developed above, each such process keeps its current look-ahead
;;;   and sends out a sample at a time, like the others.  Whenever needed,
;;;   it processes another block, saves the look-ahead, and goes back to sample
;;;   at a time mode.

(defstruct rblk buf siz loc ctr)

(defun make-block (&key size (trigger 0))
  (make-rblk :siz size
	     :buf (make-table size)	;initializes buf to 0.0
	     ;; this should be c-make-array for external run-blocks
	     :loc 0
	     :ctr trigger))

(defun run-block (b)
  ;; this should use c-arrays
  (declare (optimize (speed 3) (safety 1)))
  (let ((val (if (< (rblk-loc b) (rblk-siz b))
		 (aref (rblk-buf b) (rblk-loc b))
	       0.0)))			;i.e. return 0 if beyond end of block's data
    (incf (rblk-loc b))
    (if (>= (rblk-loc b) (rblk-ctr b))
	(progn
	  (if (< (rblk-loc b) (rblk-siz b))
	      (progn
		(loop for i from 0 below (rblk-loc b) do
		  (setf (aref (rblk-buf b) i) 0.0))
		(loop for j from (rblk-loc b) below (rblk-siz b) and i from 0 do
		  (setf (aref (rblk-buf b) i) (aref (rblk-buf b) j))
		  (setf (aref (rblk-buf b) j) 0.0)))
	    (loop for i from 0 below (rblk-siz b) do
	      (setf (aref (rblk-buf b) i) 0.0)))
	  (decf (rblk-ctr b) (rblk-loc b))
;	  (if (< (rblk-ctr b) 0) (setf (rblk-ctr b) 0))
	  (setf (rblk-loc b) 0)))
    val))


(defstruct wt wave wsiz freq b internal-mag phase)

(defun make-wave-train (&key wave (frequency 440.0) (initial-phase 0.0))
  (let ((wave-size (array-dimension wave 0)))
    (setf initial-phase (check-initial-phase initial-phase))
    (make-wt :wave wave
	     :wsiz wave-size
	     :internal-mag (/ 1.0 frequency-mag)
	     :b (make-block :size wave-size
			    :trigger 0)
	     :phase (if (not (zerop initial-phase))
			(* wave-size (/ initial-phase two-pi))
		      0.0)
	     :freq frequency)))

(defun wave-train (w &optional (fm 0.0))
  (declare (optimize (speed 3) (safety 1)))
  (when (zerop (rblk-loc (wt-b w)))
    (loop for i from 0 below (wt-wsiz w) do
      (incf (aref (rblk-buf (wt-b w)) i) (array-interp (wt-wave w) (+ i (wt-phase w)) (wt-wsiz w))))
    (incf (rblk-ctr (wt-b w)) (/ sampling-rate (+ (wt-freq w) (* fm (wt-internal-mag w))))))
  (run-block (wt-b w)))

;;; may want to add table-read rate (rather than always 1) -- probably best way is to
;;; add another layer that changes the wave table (table size changes which is a headache)
;;; also see add-one-segment below for amplitude envelopes.


;;; So fft-filtering is : make block given fft size
;;;                       get first input buffer, fft, multiply by spectrum, un-fft
;;;                       put un-fft data in buffer, set trigger to hop-size
;;;                       each call returns first value, and if second is not nil,
;;;                          run fft on next block, and add into current buffer
;;;
;;; since this process has to be able to look ahead in its input, we can't just provide an
;;; input slot as with things like oscil or filter -- we have to have the data in a file (or stream).

;;; .basic-fft (lib56) does not pre-shift the data so that all results are still fractional, so
;;; we have to do that here (so that make-fft-filter need not be different for the two cases)

(defstruct fftflt env siz hop rd b datar datai half-siz)

(defun make-fft-filter (&key filter file start-time start input-file-start-time input-file-start (channel :A) (fft-size 512))
  (let* ((fft1 (expt 2 (ceiling (log fft-size 2))))
	 (fft2 (floor fft1 2)))
    (make-fftflt :env (if (listp filter)
			  (let ((arr (make-array fft2 :element-type 'short-float))
				(arr-env (make-env :envelope filter :scaler 1.0 :start 0 :end fft2)))
			    (loop for i from 0 below fft2 do
			      (setf (aref arr i) (env arr-env)))
			    arr)
			filter)		;needs to be stretched by (/ fft1 fft-size)
		 :siz fft1
		 :half-siz fft2
		 :b (make-block :size fft1 :trigger 0)
		 :hop -1		;signal that we are at the start (to throw away "pre-ring")
		 :datar (make-table fft1)
		 :datai (make-table fft1)
		 :rd (make-readin :start-time (or input-file-start-time start-time) 
				  :start (or input-file-start start) 
				  :channel channel 
				  :file file))))
  
;;; fft from "Numerical Recipes in Pascal" translated to lisp and so on -- see comments in lib56.lisp
;;; and num.lisp.

(defun _fft (xdata ydata n &optional (isign 1))
  #-mcl (c-fft xdata ydata n isign (floor (log n 2)))
  #+mcl (__fft xdata ydata n isign)
  )

(defun __fft (xdata ydata n &optional (isign 1))
  (declare (optimize (speed 3) (safety 1)))
  (let ((mmax 0) (j 0)
	(ipow (floor (log n 2)))
	(pow 0)	(wtemp 0.0) (wr 0.0) (wpr 0.0) 
	(prev 0) (wpi 0.0) (wi 0.0) (theta 0.0)
	(tempr 0.0) (tempi 0.0) (wrs 0.0) (wis 0.0))
    (dotimes (i n)			;bit reversal section starts here
      (when (> j i)
	(setf tempr (aref xdata j))	;swap (as complex) data[j] and data[i]
	(setf tempi (aref ydata j))
	(setf (aref xdata j) (aref xdata i))
	(setf (aref ydata j) (aref ydata i))
	(setf (aref xdata i) tempr)
	(setf (aref ydata i) tempi))
      (let ((m (floor n 2)))
	(do () 
	    ((or (< m 2) (< j m)))
	  (decf j m)
	  (setf m (floor m 2)))
	(incf j m)))
    (setf prev 1)
    (setf mmax 2)			;now the Danielson-Lanczos section
    (setf pow (floor n 2))
    (setf theta (* 6.28318530717959 isign .5))
    (dotimes (lg ipow)
      (setf wpr (cos theta))
      (setf wpi (sin theta))
      (setf wr 1.0)
      (setf wi 0.0)
      (dotimes (ii prev)
	(setf wrs wr)
	(setf wis wi)
	(do* ((jj 0 (+ jj 1))
	      (i ii (+ i mmax))
	      (j (+ i prev) (+ j mmax)))
	    ((>= jj pow))
	  (setf tempr (- (* wrs (aref xdata j)) (* wis (aref ydata j))))
	  (setf tempi (+ (* wrs (aref ydata j)) (* wis (aref xdata j))))
	  (setf (aref xdata j) (- (aref xdata i) tempr))
	  (setf (aref ydata j) (- (aref ydata i) tempi))
	  (incf (aref xdata i) tempr)
	  (incf (aref ydata i) tempi))
	(setf wtemp wr)
	(setf wr (- (* wr wpr) (* wi wpi)))
	(setf wi (+ (* wi wpr) (* wtemp wpi))))
      (setf pow (* pow .5))
      (setf prev mmax)
      (setf theta (* theta .5))
      (setf mmax (* mmax 2)))))

(defun _inverse-fft (xdata ydata n) (_fft xdata ydata n -1))

;;; for fft's in the Run loop we need to be able to recognize the parallel real/imaginary arrays
;;; and allocate them in the correct memory type/location, so we need a struct to pass around as the argument.

(defstruct fft-data real imaginary size)

(defun make-fft-data-arrays (size)
  (make-fft-data :real (make-array size :element-type 'short-float :initial-element 0.0)
		 :imaginary (make-array size :element-type 'short-float :initial-element 0.0)
		 :size size))

(defun fft (data &optional (direction 1)) (_fft (fft-data-real data) (fft-data-imaginary data) (fft-data-size data) direction))
(defun inverse-fft (data) (_inverse-fft (fft-data-real data) (fft-data-imaginary data) (fft-data-size data)))


(defun dot-product (in1 in2 &optional (start 0))
  ;; also known as scalar product, and in orthogonal coordinate systems the same as inner product
  (declare (optimize (speed 3) (safety 1)))
  (let ((lim (min (array-dimension in2 0)
		  (- (array-dimension in1 0) start)))
	(sum 0.0))
    (loop for i from 0 below lim and j from start do
      (incf sum (* (aref in1 j) (aref in2 i))))
    sum))


(defun basic-convolve (out in1 in2 &optional fftsiz fftscl in2-real in2-imag)
  (declare (optimize (speed 3) (safety 1)))
  (let* ((in1-lim (array-dimension in1 0))
	 (in2-lim (array-dimension in2 0))
	 (lim (min in1-lim in2-lim)))
    (if (or fftsiz (> lim 32))
	(let* ((fft-size (or fftsiz (expt 2 (ceiling (/ (log (+ in1-lim in2-lim)) (log 2))))))
	       (fft-scale (or fftscl (float (/ 1.0 fft-size))))
	       (real1 (make-array fft-size :element-type 'short-float :initial-element 0.0))
	       (real2 (or in2-real (make-array fft-size :element-type 'short-float :initial-element 0.0)))
	       (imag1 (make-array fft-size :element-type 'short-float :initial-element 0.0))
	       (imag2 (or in2-imag (make-array fft-size :element-type 'short-float :initial-element 0.0))))
	  (loop for i from 0 below in1-lim do
	    (setf (aref real1 i) (aref in1 i)))
	  (when (or (not in2-real) (not in2-imag))
	    (loop for i from 0 below in2-lim do
	      (setf (aref real2 i) (aref in2 i)))
	    (_fft real2 imag2 fft-size))
	  (_fft real1 imag1 fft-size)	    
	  (loop for i from 0 below fft-size do
	    (let ((newr (* fft-scale (- (* (aref real1 i) (aref real2 i)) (* (aref imag1 i) (aref imag2 i)))))
		  (newi (* fft-scale (+ (* (aref real1 i) (aref imag2 i)) (* (aref real2 i) (aref imag1 i))))))
	      (setf (aref out i) newr)
	      (setf (aref imag1 i) newi)))
	  (_inverse-fft out imag1 fft-size))
      (let* ((tab (make-table in2-lim)))
	  (loop for i from 0 below (+ in1-lim in2-lim) do
	    (if (< i in1-lim) 
		(setf (aref tab 0) (aref in1 i))
	      (setf (aref tab 0) 0.0))
	    (setf (aref out i) (dot-product tab in2))
	    (loop for j from (1- in2-lim) downto 1 do
	      (setf (aref tab j) (aref tab (1- j)))))))))


(defun readin-data (ff)
  (declare (optimize (speed 3) (safety 1)))
  (let ((data-start (floor (fftflt-siz ff) 4))
	(data-amount (fftflt-half-siz ff)))
    (loop for i from 0 below data-start do
      (setf (aref (fftflt-datar ff) i) 0.0)
      (setf (aref (fftflt-datai ff) i) 0.0))
    (loop for k from 0 below data-amount and i from data-start do
      (setf (aref (fftflt-datar ff) i) (/ (readin (fftflt-rd ff)) (fftflt-siz ff)))
      (setf (aref (fftflt-datai ff) i) 0.0))
    (loop for k from 0 below data-start and i from (+ data-start data-amount) do
      (setf (aref (fftflt-datar ff) i) 0.0)
      (setf (aref (fftflt-datai ff) i) 0.0))))

(defun readout-first-data (ff)
  (declare (optimize (speed 3) (safety 1)))
  (let* ((data-start (floor (fftflt-siz ff) 4))
	 (data-amount (* data-start 3)))
    (loop for i from 0 below data-amount and k from data-start do
      (setf (aref (rblk-buf (fftflt-b ff)) i) (aref (fftflt-datar ff) k)))))

(defun readout-data (ff)
  (declare (optimize (speed 3) (safety 1)))
  (let ((data-stop (fftflt-hop ff)))
    (loop for i from 0 below data-stop do
      (incf (aref (rblk-buf (fftflt-b ff)) i) (aref (fftflt-datar ff) i)))
    (loop for i from 0 below data-stop and j from data-stop do
      (setf (aref (rblk-buf (fftflt-b ff)) j) (aref (fftflt-datar ff) j)))))

(defun apply-envelope-and-shift (ff)
  (declare (optimize (speed 3) (safety 1)))
  (let ((env-val 0.0))
    (loop for i from 0 below (fftflt-half-siz ff) and 
              j from (1- (fftflt-siz ff)) by -1 do
      (setf env-val (aref (fftflt-env ff) i))
      (setf (aref (fftflt-datar ff) i) (* (aref (fftflt-datar ff) i) env-val))
      (setf (aref (fftflt-datar ff) j) (* (aref (fftflt-datar ff) j) env-val))
      (setf (aref (fftflt-datai ff) i) (* (aref (fftflt-datai ff) i) env-val))
      (setf (aref (fftflt-datai ff) j) (* (aref (fftflt-datai ff) j) env-val)))))
      
(defun fft-filter (ff)
  (declare (optimize (speed 3) (safety 1)))
  (when (zerop (rblk-loc (fftflt-b ff)))
    ;; clear data, from fft-size/4 to 3*fft-size/4 load up next window of data, move window by hop
    ;; clear imag part, fft, multiply by envelope symmetric about midpoint, ifft, overlap add
    ;; starting in data and b from 0 except the first time when we start in data at fft-size/4
    (readin-data ff)
    (_fft (fftflt-datar ff) (fftflt-datai ff) (fftflt-siz ff))
    (apply-envelope-and-shift ff)
    (_inverse-fft (fftflt-datar ff) (fftflt-datai ff) (fftflt-siz ff))
    (if (minusp (fftflt-hop ff))
	(progn
	  (setf (fftflt-hop ff) (fftflt-half-siz ff))
	  (setf (rblk-ctr (fftflt-b ff)) (floor (fftflt-siz ff) 4))
	  (readout-first-data ff))
      (progn
	(readout-data ff)
	(setf (rblk-ctr (fftflt-b ff)) (fftflt-hop ff)))))
  (run-block (fftflt-b ff)))


;;; convolution

(defstruct conv fftf filtr filti)

(defun make-convolve (&key filter file start-time start input-file-start-time input-file-start (channel :A) (fft-size 512))
  (if (io-p filter)			;that is we're convolving two sound files
      (make-conv :filtr nil :filti nil 
		 :fftf (make-fftflt :rd (make-readin :start-time 0 
						     :channel :A 
						     :file (big-convolve filter file 
									 :start-time (or input-file-start-time start-time)
									 :start (or input-file-start start)
									 :channel channel))))
    (let* ((fft1 (expt 2 (ceiling (/ (log fft-size) (log 2)))))
	   (fft2 (floor fft1 2))
	   (fltr (make-table fft1))
	   (flti (make-table fft1)))
      (loop for i from 0 below (array-total-size filter) do
	(setf (aref fltr i) (aref filter i)))
      (_fft fltr flti fft1)
      (make-conv :filtr fltr
		 :filti flti
		 :fftf (make-fftflt :siz fft1 ;no env -> used as convolution attribute in pass-fftflt
				    :hop -1
				    :half-siz fft2
				    :datar (make-table fft1)
				    :datai (make-table fft1)
				    :b (make-block :size fft1 :trigger 0)
				    :rd (make-readin :start-time (or input-file-start-time start-time)
						     :start (or input-file-start start)
						     :channel channel 
						     :file file))))))
      
;;; deconvolve divides where convolve multiplies (deconvolution is unhappy if spectrum is ever 0)

(defun convolve (ff)
  (declare (optimize (speed 3) (safety 1)))
  (if (conv-filtr ff)
      (progn
	(when (zerop (rblk-loc (fftflt-b (conv-fftf ff)))) ;this is starting to look like SAIL code!
	  
	  ;; clear data, from fft-size/4 to 3*fft-size/4 load up next window of data, move window by hop
	  ;; clear imag part, fft, convolve, ifft, overlap add
	  ;; starting in data and b from 0 except the first time when we start in data at fft-size/4
	  
	  (readin-data (conv-fftf ff))
	  (basic-convolve (fftflt-datar (conv-fftf ff)) 
			  (fftflt-datai (conv-fftf ff)) ;uh... was datar here (changed 31-Jan-93)
			  (conv-filtr ff)
			  (fftflt-siz (conv-fftf ff))
			  1.0
			  (conv-filtr ff)
			  (conv-filti ff))
	  (if (minusp (fftflt-hop (conv-fftf ff)))
	      (progn
		(setf (fftflt-hop (conv-fftf ff)) (fftflt-half-siz (conv-fftf ff)))
		(setf (rblk-ctr (fftflt-b (conv-fftf ff))) (floor (fftflt-siz (conv-fftf ff)) 4))
		(readout-first-data (conv-fftf ff)))
	    (progn
	      (readout-data (conv-fftf ff))
	      (setf (rblk-ctr (fftflt-b (conv-fftf ff))) (fftflt-hop (conv-fftf ff))))))
	(run-block (fftflt-b (conv-fftf ff))))
    (readin (fftflt-rd (conv-fftf ff)))))

(defun big-convolve (filter file &key start-time start input-file-start-time input-file-start channel)
  ;; open temp file, do entire convolution, close, reopen as clm-input file
  ;; return io struct of new file (and add to hidden file list under original file's index)

  (let* ((st (or input-file-start start (floor (* (or input-file-start-time start-time) sampling-rate))))
	 (samps (- (io-siz file) st))
	 (filter-2 (ceiling (log (io-siz filter)) (log 2)))
	 (file-2 (ceiling (log samps) (log 2)))
	 (all-2 (ceiling (log (+ samps (io-siz filter))) (log 2)))
	 (fname (if (eq channel :B) "/zap/temporary-clm-convolution-output-B.snd" "/zap/temporary-clm-convolution-output-A.snd"))
	 (ipow (if (or (< all-2 20) (< (- file-2 filter-2) 3))
		   all-2 
		 (1+ filter-2)))
	 (size (expt 2 ipow))
	 (fullsize (expt 2 all-2)))
    (c-convolve fname 
		(io-fil file) (io-hdr-end file) (io-siz file)
		(io-fil filter) (io-hdr-end filter) (io-siz filter)
		st size ipow all-2 samps fullsize (if (io-dat-b file) 2 1) (if (eq channel :B) 1 0))
    (let ((tmp-file (clm-open-input fname)))
      (push (list (io-open-index file) (io-open-index tmp-file)) *clm-hidden-open-files*)
      tmp-file)))


(defconstant rectangular-window 0)
(defconstant hanning-window 1)
(defconstant welch-window 2)
(defconstant parzen-window 3)
(defconstant bartlett-window 4)
(defconstant hamming-window 5)
(defconstant order-0-window 6)
(defconstant order-1-window 7)
(defconstant order-2-window 8)
(defconstant order-3-window 9)
(defconstant order-4-window 10)
(defconstant exponential-window 11)
(defconstant kaiser-window 12)

(defmacro window (this that) (declare (ignore this that)) (error "window has changed to fft-window"))

(defun fft-window (fftdat wind)
  ;; this exists mostly for compatibility between Run as no-op and Run as DSP compiler
  (loop for i from 0 below (fft-data-size fftdat) do
    (setf (aref (fft-data-real fftdat) i) (* (aref (fft-data-real fftdat) i) (aref wind i)))))

(defun clear-block (block)
  (loop for i from 0 below (length block) do (setf (aref block i) 0.0)))

(defun sqr (x) (* x x))

(defun apply-window (data n i val)
  (progn
    (setf (aref data i) (* (aref data i) val))
    (setf (aref data (- n i 1)) (* (aref data (- n i 1)) val))))

(defun spectrum (data &optional (window parzen-window) (beta 2.5))
					;data is real array, windows, FFTs, returns magnitude spectrum (in place)
  (let* ((n (array-dimension data 0))
	 (n2 (ceiling (/ (log n) (log 2))))
	 (midn (floor n 2))
	 (fftn (expt 2 n2))
	 (freq (/ two-pi n))
	 (rate (/ 1.0 midn))
	 (angle 0.0)
	 (expn (+ 1.0 (/ (log 2) midn)))
	 (expsum 1.0)
	 (ffti (make-array (+ fftn 1) :element-type 'float :initial-element 0.0)))
    (if (and (/= window rectangular-window)
	     (/= window order-0-window))
	(loop for i from 0 to midn do	;all these windows are symmetric around the midpoint
	  (if (or (= window hamming-window)
		  (= window order-1-window))
	      (progn
		(apply-window data n i (- 0.54 (* 0.46 (cos angle))))
		(incf angle freq))
	    (if (= window bartlett-window) ;triangular window
		(progn
		  (apply-window data n i angle)
		  (incf angle rate))
	      (if (= window hanning-window) ;cosine that starts and ends at 0
		  (progn
		    (apply-window data n i (- 0.5 (* 0.5 (cos angle))))
		    (incf angle freq))
		(if (= window parzen-window) ;triangle with small offset 
		    (apply-window data n i (- 1.0 (abs (/ (- i (* 0.5 (- n 1))) (* 0.5 (+ n 1))))))
		  (if (= window welch-window) ;parzen squared
		      (apply-window data n i (- 1.0 (sqr (/ (- i (* 0.5 (- n 1))) (* 0.5 (+ n 1))))))
		    (if (= window order-2-window)
			(progn
			  (apply-window data n i (+ 0.42323 
						    (* -0.49755 (cos angle)) 
						    (* 0.07922 (cos (* angle 2)))))
			  (incf angle freq))
		      (if (= window order-3-window)
			  (progn
			    (apply-window data n i (+ 0.35875 
						      (* -0.48829 (cos angle)) 
						      (* 0.14128 (cos (* angle 2)))
						      (* -0.01168 (cos (* angle 3)))))
			    (incf angle freq))
			(if (= window order-4-window)
			    (progn
			      (apply-window data n i (+ 0.287333
							(* -0.4471689 (cos angle))
							(* 0.2084454 (cos (* angle 2)))
							(* -0.0519053 (cos (* angle 3)))
							(* 0.00514933 (cos (* angle 4)))))
			      (incf angle freq))
			  (if (= window kaiser-window)
			      (apply-window data n i (/ (bessi0 (* beta (sqrt (- 1.0 (sqr (/ (- n (* 2 i)) n)))))) (bessi0 beta)))
			    (if (= window exponential-window)
				(progn
				  (apply-window data n i (- expsum 1.0))
				  (setf expsum (* expsum expn)))))))))))))))
    ;;now DATA is windowed and ready to be transformed
    (_fft data ffti fftn)
    ;;now turn this into a polar spectrum
    (setf expn 0.0)
    (loop for i from 0 below n do
      (setf expsum (sqrt (+ (sqr (aref data i)) (sqr (aref ffti i)))))
      (setf expn (max expn expsum))
      (setf (aref data i) expsum))
    ;;now normalize the power spectrum (this is a dubious kludge...)
    (loop for i below n do (setf (aref data i) (/ (aref data i) expn)))))

(defun log-magnitude (data)		;assumes data normalized
  (let ((20log10 (/ 20 (log 10)))
	(lowest 1.0e-6))
    (loop for i from 0 below (array-total-size data) do
      (setf (aref data i) (* 20log10 (log (max (aref data i) lowest)))))))



;;; expansion is almost identical to fft-filter but has amp-env, sliding pointer into input, etc
;;; Expand was originally called SpeedFile (in Mixer.Sai)
;;;   Another version of SpeedFile alternated between forward and backward segments.

(defstruct spd rd len rmp amp in-spd out-spd (cur-in 0) cur-out b s20 s50 (ctr 0))

(defun add-one-segment (e)
  ;; e -> spd structure, with cur-in (and rdin-i) set up at correct starting point, and buffer (b)
  ;;   ready to accept the next segment.
  (let ((amp 0.0)
	(steady-time (floor (- (spd-len e) (* 2 (spd-rmp e)))))
	(k 0)
	(incr (/ (spd-amp e) (spd-rmp e))))
    ;; ramp up
    (loop for i from 0 below (spd-rmp e) do
      (incf (aref (spd-b e) k) (* amp (readin (spd-rd e))))
      (incf k)
      (incf amp incr))
    ;; now the "steady-state"
    (loop for i from 0 below steady-time do
      (incf (aref (spd-b e) k) (* amp (readin (spd-rd e))))
      (incf k))
    ;; now ramp down
    (loop for i from 0 below (spd-rmp e) do
      (incf (aref (spd-b e) k) (* amp (readin (spd-rd e))))
      (incf k)
      (decf amp incr))))

(defun set-expansion-triggers (e)
  (decf (spd-ctr e) (spd-cur-out e))
  (setf (spd-cur-out e) (+ (spd-out-spd e) (random (spd-s50 e))))
  (setf (rdin-i (spd-rd e)) (+ (spd-cur-in e) (random (spd-s20 e))))
  (incf (spd-cur-in e) (spd-in-spd e)))

(defun shift-block-back (e)		;some redundancy here with run-block 
  (let* ((start (floor (spd-cur-out e)))
	 (end (- (spd-len e) start)))
    (loop for i from 0 below end and
              j from start do
      (setf (aref (spd-b e) i) (aref (spd-b e) j)))
    (loop for i from end below (spd-len e) do
      (setf (aref (spd-b e) i) 0.0))))
              
;;; much time was wasted here trying to change resample to use readin and then
;;;   add a resample option to expand, but this is more bother than it appears
;;;   first because readin steps by 1, whereas resample wants to be free to
;;;   step by any amount (i.e. it is not stupid that resample calls INA rather
;;;   than readin, and to extend readin to take arbitrary increments makes it
;;;   almost the same as resample.  Second, if resample is called here, it is
;;;   non-trivial (both here and in the 56000 world) to reset the resampler's
;;;   notion of where it is in the file as we backtrack overlapping successive
;;;   slices of the input.

(defun make-expand (fil &key start-time input-file-start-time
			     start input-file-start
			     (channel :A)
			     (segment-length .15)
			     (segment-scaler .6)
			     (expansion-amount 1.0)
			     (output-hop .05)    ;hop size in output
			     (ramp-time .4))     ;amount of segment spent sloping up or down (envelope)
  (let ((val (make-spd :cur-out 0
		       :rd (make-readin :file fil 
					:start-time (or input-file-start-time start-time) 
					:start (or input-file-start start) 
					:channel channel)
		       :cur-in (or input-file-start start (and (or input-file-start-time start-time)
							       (floor (* (or input-file-start-time start-time) sampling-rate))) 0)
		       :len (ceiling (* segment-length sampling-rate))
		       :rmp (floor (* ramp-time segment-length sampling-rate))
		       :amp segment-scaler
		       :in-spd (floor (/ (* output-hop .5 sampling-rate) expansion-amount))
		       :out-spd (floor (* output-hop sampling-rate))
		       :b nil
		       :s20 (floor (/ sampling-rate 20))
		       :s50 (floor (/ sampling-rate 50))
		       :ctr 0)))
    (setf (spd-b val) (make-table (spd-len val)))
    ;; now load up first segment, set trigger for next segment (and readin pointer)
    (add-one-segment val)
    (set-expansion-triggers val)
    val))

(defun expand (e)			;return one sample from expansion, get next block, if needed
  (let ((cur-val (aref (spd-b e) (floor (spd-ctr e)))))
    (incf (spd-ctr e))
    (when (>= (spd-ctr e) (spd-cur-out e))
      (shift-block-back e)
      (add-one-segment e)
      (set-expansion-triggers e))
    cur-val))


(defun clm-end-run ()
  (when *clm-hidden-open-files*
    (loop for pair in *clm-hidden-open-files* do
      (clm-close-input (aref *clm-current-open-files* (second pair))))
    (setf *clm-hidden-open-files* nil))
  (when *clm-delay-lines*
    (loop for i from 0 to *clm-max-delay* do
      (forget-delay (aref *clm-delay-lines* i)))))


(defun read-dir (rd dir)
  (when rd
    (if (rdin-p rd) (setf (rdin-inc rd) dir)
      (if (sr-p rd) (setf (rdin-inc (sr-rd rd)) dir)
	(if (fftflt-p rd) (setf (rdin-inc (fftflt-rd rd)) dir)
	  (if (conv-p rd) (setf (rdin-inc (fftflt-rd (conv-fftf rd))) dir)
	    (if (spd-p rd) (setf (rdin-inc (spd-rd rd)) dir)
	      (error "cannot set increment of ~A to ~A" rd dir))))))))

(defun read-forward (rd) (read-dir rd 1))
(defun read-backward (rd) (read-dir rd -1))

(defun read-position (rd)
  (if rd
    (if (rdin-p rd) (rdin-i rd)
      (if (sr-p rd) (rdin-i (sr-rd rd))
	(if (fftflt-p rd) (rdin-i (fftflt-rd rd))
	  (if (conv-p rd) (rdin-i (fftflt-rd (conv-fftf rd)))
	    (if (spd-p rd) (rdin-i (spd-rd rd))
	      (error "cannot get current sample of ~A" rd))))))
    0))



;;; support procedures for LPC
;;;
;;; translated from "Numerical Recipes in Pascal" by Press, Flannery, Teukolsky, and Vetterling

(defun memcof (data n m pm cof)
  (let* ((wk1 (make-array (1+ n) :element-type 'float :initial-element 0.0))
	 (wk2 (make-array (1+ n) :element-type 'float :initial-element 0.0))
	 (wkm (make-array (1+ m) :element-type 'float :initial-element 0.0))
	 (num 0.0) 
	 (p 0.0) 
	 (denom 0.0))
    (loop for j from 1 to n do
      (incf p (sqr (aref data j))))
    (setf pm (/ p n))
    (setf (aref wk1 1) (aref data 1))
    (setf (aref wk2 (- n 1)) (aref data n))
    (loop for j from 2 to (- n 1) do
      (setf (aref wk1 j) (aref data j))
      (setf (aref wk2 (- j 1)) (aref data j)))
    (loop for k from 1 to m do
      (setf num 0.0)
      (setf denom 0.0)
      (loop for j from 1 to (- n k) do
	(incf num (* (aref wk1 j) (aref wk2 j)))
	(incf denom (+ (sqr (aref wk1 j)) (sqr (aref wk2 j)))))
      (if (zerop denom)			;probably should return an error (attempt to evaluate 0/0)
	  (setf (aref cof k) 0.0)
	(setf (aref cof k) (* 2.0 (/ num denom))))
      (setf pm (* pm (- 1.0 (sqr (aref cof k)))))
      (loop for i from 1 to (- k 1) do
	(setf (aref cof i) (- (aref wkm i) (* (aref cof k) (aref wkm (- k i))))))
      (when (/= k m)
	(loop for i from 1 to k do 
	  (setf (aref wkm i) (aref cof i)))
	(loop for j from 1 to (- n k 1) do
	  (decf (aref wk1 j) (* (aref wkm k) (aref wk2 j)))
	  (setf (aref wk2 j) (- (aref wk2 (+ j 1)) (* (aref wkm k) (aref wk1 (+ j 1))))))))
    pm))

(defun laguer (a m x eps polish)
  (let ((epss 6.0e-8)
	(mixt 100))
	;(dxold (abs x))
    (loop for iter from 1 to mixt do
      (let* ((b (aref a (1+ m)))
	     (err (abs b))
	     (d #C(0 0))
	     (f #C(0 0))
	     (g #C(0 0))
	     (g2 #C(0 0))
	     (gp #C(0 0))
	     (gm #C(0 0))
	     (h #C(0 0))
	     (cdum #C(0 0))
	     (sq #C(0 0))
	     (dx #C(0 0))
	     (cdx #C(0 0))
	     (x1 #C(0 0))
	     (abx (abs x)))
	(loop for j from m downto 1 do
	  (let ((dum (realpart f)))
	    (setf f (complex (+ (* (realpart x) (realpart f))
				(realpart d)
				(- (* (imagpart x) (imagpart f))))
			     (+ (* (realpart x) (imagpart f))
				(imagpart d)
				(* (imagpart x) dum))))
	    (setf dum (realpart d))
	    (setf d (complex (+ (* (realpart x) (realpart d))
				(realpart b)
				(- (* (imagpart x) (imagpart d))))
			     (+ (* (realpart x) (imagpart d))
				(imagpart b)
				(* (imagpart x) dum))))
	    (setf dum (realpart b))
	    (setf b (complex (+ (* (realpart x) (realpart b))
				(realpart (aref a j))
				(- (* (imagpart x) (imagpart b))))
			     (+ (* (realpart x) (imagpart b))
				(imagpart (aref a j))
				(* (imagpart x) dum))))
	    (setf err (+ (abs b) (* abx err)))))
	(setf err (* epss err))
	(if (<= (abs b) err) (return-from laguer x))
	(setf g (/ d b))
	(setf g2 (complex (- (sqr (realpart g)) (sqr (imagpart g)))
			  (* 2.0 (realpart g) (imagpart g))))
	(setf cdum (/ f b))
	(setf h (complex (- (realpart g2) (* 2.0 (realpart cdum)))
			 (- (imagpart g2) (* 2.0 (imagpart cdum)))))
	(setf cdum (complex (* (1- m) (- (* m (realpart h)) (realpart g2)))
			    (* (1- m) (- (* m (imagpart h)) (imagpart g2)))))
	(setf sq (sqrt cdum))
	(setf gp (+ g sq))
	(setf gm (- g sq))
	(if (< (abs gp) (abs gm)) (setf gp gm))
	(setf cdum (complex m 0))
	(setf dx (/ cdum gp))
	(setf x1 (- x dx))
	(if (= x x1) (return-from laguer x))
	(setf x x1)
	(setf cdx (abs dx))
	;(setf dxold cdx)
	(if (and (not polish)
		 (<= cdx (* eps (abs x))))
	    (return-from laguer x))))))
	
(defun zroots (a m roots polish)
  (let ((eps 2.0e-6)
	(ad (make-array (+ m 2) :element-type 'complex))
	(b #C(0 0))
	(c #C(0 0))
	(dum 0.0))
    (loop for j from 1 to (1+ m) do
      (setf (aref ad j) (aref a j)))
    (loop for j from m downto 1 do
      (let ((x (laguer ad j #C(0 0) eps nil)))
	(if (<= (abs (imagpart x))
		(* 2.0 (sqr eps) (abs (realpart x))))
	    (setf x (complex (realpart x) 0.0)))
	(setf (aref roots j) x)
	(setf b (aref ad (1+ j)))
	(loop for jj from j downto 1 do
	  (setf c (aref ad jj))
	  (setf (aref ad jj) b)
	  (setf dum (realpart b))
	  (setf b (complex (+ (* (realpart b) (realpart x))
			      (realpart c)
			      (- (* (imagpart b) (imagpart x))))
			   (+ (* dum (imagpart x))
			      (imagpart c)
			      (* (imagpart b) (realpart x))))))))
    (if polish
	(loop for j from 1 to m do
	  (setf (aref roots j) (laguer a m (aref roots j) eps t))))
    (loop for j from 2 to m do
      (let ((x (aref roots j))
	    (i (1- j)))
	(loop while (and (>= i 1) 
			 (> (realpart (aref roots i)) (realpart x))) do
	  (setf (aref roots (1+ i)) (aref roots i))
	  (decf i))
	(setf (aref roots (1+ i)) x)))))

(defun fixrts (d npoles)
  (let ((a (make-array (+ npoles 2) :element-type 'complex))
	(roots (make-array (+ npoles 2) :element-type 'complex))
	(dum 0.0))
    (setf (aref a (1+ npoles)) #C(1.0 0.0))
    (loop for j from npoles downto 1 do
      (setf (aref a j) (complex (- (aref d (+ npoles 1 (- j))))
				0.0)))
    (zroots a npoles roots t)
    (loop for j from 1 to npoles do
      (let ((size (+ (sqr (realpart (aref roots j))) 
		     (sqr (imagpart (aref roots j))))))
	(if (> size 1.0)
	    (setf (aref roots j) (complex (/ (realpart (aref roots j)) size)
					  (/ (imagpart (aref roots j)) size))))))
    (setf (aref a 1) (- (aref roots 1)))
    (setf (aref a 2) #C(1.0 0.0))
    (loop for j from 2 to npoles do
      (setf (aref a (1+ j)) #C(1.0 0.0))
      (loop for i from j downto 2 do
	(setf dum (realpart (aref a i)))
	(setf (aref a i) (complex (+ (realpart (aref a (1- i)))
				     (- (* (realpart (aref a i)) (realpart (aref roots j))))
				     (* (imagpart (aref a i)) (imagpart (aref roots j))))
				  (- (imagpart (aref a (1- i)))
				     (* dum (imagpart (aref roots j)))
				     (* (imagpart (aref a i)) (realpart (aref roots j)))))))
      (setf dum (realpart (aref a 1)))
      (setf (aref a 1) (complex (- (* (imagpart (aref a 1)) (imagpart (aref roots j)))
				   (* (realpart (aref a 1)) (realpart (aref roots j))))
				(- (+ (* dum (imagpart (aref roots j)))
				      (* (imagpart (aref a 1)) (realpart (aref roots j))))))))
    (loop for j from 1 to npoles do
      (setf (aref d (+ npoles 1 (- j))) (- (realpart (aref a j)))))))

(defun predic (data ndata d npoles future nfut)
  (let ((reg (make-array (1+ npoles) :element-type 'float)))
    (loop for j from 1 to npoles do
      (setf (aref reg j) (aref data (+ ndata 1 (- j)))))
    (loop for j from 1 to nfut do
      (let (				;(discrp 0.0)
	    (sum 0.0))
	(loop for k from 1 to npoles do
	  (incf sum (* (aref d k) (aref reg k))))
	(loop for k from npoles downto 2 do
	  (setf (aref reg k) (aref reg (1- k))))
	(setf (aref reg 1) sum)
	(setf (aref future j) sum)))))


;;; poor man's display function -- old line printer style data display

(defun show-data (data-1 &optional lim0 lim1 (y-extent 20))
  (let* ((size (or (and lim1 lim0 (- lim1 lim0)) lim0 (array-total-size data-1)))
	 (offset (or (and lim1 lim0) 0))
	 (data (make-array size :element-type 'float)))
    (loop for i from 0 below size do 
      (setf (aref data i) (float (aref data-1 (+ offset i)))))
    (let* ((miny (aref data 0))
	   (maxy miny)
	   (scale 0.0)
	   (ytop maxy)
	   (minx 0)
	   (maxx size))
      (loop for i from minx below maxx do
	(setf maxy (max maxy (aref data i)))
	(setf miny (min miny (aref data i))))
      (setf ytop (max maxy (abs miny)))
      (if (= maxy miny)
	  (format nil "data = ~F" miny)
	(progn
	  (setf scale (/ (- maxy miny) y-extent))
	  (loop for y from maxy downto miny by scale and i from 0 and
	            y0 from (- maxy scale) by (- scale) do
	    (if (zerop (mod i 5))
		(if (> ytop 100)
		    (princ (format nil "~9,1F |" y))
		  (if (> ytop .001)
		      (princ (format nil "~9,3F |" y))
		    (princ (format nil "~9,6F |" y))))
	      (princ "          |"))
	    (loop for x from minx below maxx do
	      (if (<= y0 (aref data x) y)
		  (princ "*")
		(princ " ")))
	    (terpri)))))))

(defun show-log-data (data &optional (x-extent 128))
  (let* ((size (floor (array-total-size data) 2))
	 (dumb-number -1000.0)
	 (log-data (make-array x-extent :element-type 'float :initial-element dumb-number))
	 (xscl (/ x-extent (ceiling (log size 2.0))))
	 (scale 0.0)
	 (val 0.0))
    (setf (aref log-data 0) (aref data 0))
    (loop for i from 1 below size do
      (let ((li (min (floor (* xscl (log (float i) 2.0)))
		     (1- x-extent))))
	(setf (aref log-data li) (max (aref log-data li) (aref data i)))))
    (setf val (aref log-data 0))
    (loop for i from 1 below x-extent do
      (if (= (aref log-data i) dumb-number) 
	  (setf (aref log-data i) val)
	(setf val (aref log-data i))))
    (let* ((miny (aref log-data 0))
	   (maxy miny))
      (loop for i from 0 below x-extent do
	(setf maxy (max maxy (aref log-data i)))
	(setf miny (min miny (aref log-data i))))
      (if (= maxy miny)
	  (format nil "data = ~F" miny)
	(progn
	  (setf scale (/ (- maxy miny) 20))
	  (loop for y from maxy downto miny by scale and i from 0 and
	            y0 from (- maxy scale) by (- scale) do
	    (loop for x from 0 below x-extent do
	      (if (zerop (mod i 5))
		  (princ (format nil "~9,3F |" y))
		(princ "          |"))
	      (if (<= y0 (aref log-data x) y)
		  (princ "*")
		(princ " ")))
	    (terpri)))))))

(defun show-env (e &optional (y-extent 20))
  (when e
    (let* ((miny (second e))
	   (maxy miny)
	   (scale 0.0)
	   (ytop maxy)
	   (minx (first e))
	   (maxx (env-last-x e))
	   (step (/ (- maxx minx) 100)))
      (loop for y in (cdr e) by #'cddr do
	(setf maxy (max maxy y))
	(setf miny (min miny y)))
      (setf ytop (max maxy (abs miny)))
      (if (= maxy miny)
	  (format nil "data = ~F" miny)
	(progn
	  (setf scale (/ (- maxy miny) y-extent))
	  (loop for y from maxy downto miny by scale and i from 0 and
	   y0 from (- maxy scale) by (- scale) do
	    (if (zerop (mod i 5))
		(if (> ytop 100)
		    (princ (format nil "~9,1F |" y))
		  (if (> ytop .001)
		      (princ (format nil "~9,3F |" y))
		    (princ (format nil "~9,6F |" y))))
	      (princ "          |"))
	    (loop for x from minx below maxx by step do
	      (if (<= y0 (list-interp x e) y)
		  (princ "*")
		(princ " ")))
	    (terpri))
	  (princ (format nil "~9,3F -----" minx))
	  (loop for x from (+ minx (* step 25)) to maxx by (* step 25) do
	    (princ (format nil "--------------- ~8,3F " x))))))))

(defun print-hash (tab) (maphash #'(lambda (a b) (print (format nil "~A ~A" a b))) tab))

(defun clm-print (fstr &rest args) (princ (apply #'format nil fstr args)))
