;;; -*- Syntax: Common-Lisp; Package: clm; Base: 10; Mode: Lisp -*-

(in-package :clm)

;;;
;;; Procedural interface to C for sound io on the Mac. Sound
;;; buffers in Coral Lisp must be allocated from the Mac heap, rather
;;; than from static or malloc'ed C memory. The function c-make-array
;;; calls make-buffer, which allocates space on the Mac heap and returns 
;;; a handle to the buffer.  All of these fuctions manipulate a buffer 
;;; through its handle by first locking the buffer down and passing its
;;; dereferenced handle (ie the actual pointer) to the C routines.
;;;

(eval-when (compile load eval)
  (defvar *clm-check-args* t)
  ;; in this version of clm (where there aren't very many calls down to C),
  ;; turning off arg-checking saved only about 2% compute time.
)

;;;
;;; Routines for implementing buffers on the mac heap.
;;;

(defun make-buffer (size)
  (let ((hd (#_NewHandle :check-error 0)))
    (unless (ccl:handlep hd)
      (error "#_newHandle failed with ~A." hd))
    (#_SetHandleSize :check-error hd (* size 4))  ; buffers are 32bit longs
    hd))

(defun free-buffer (handle)
  (unless (ccl:handlep handle)
    (error "~S is not a valid handle." handle))
  (#_DisposHandle :check-error handle))

(defun buffer-size (handle)
  (unless (ccl:handlep handle)
    (error "~S is not a valid buffer handle." handle)
  (#_GetHandleSize handle)))

;;;
;;; setf-able buffer accessors for byte, word and long values.
;;;

(defun bbyte (handle index)
  (ccl:with-dereferenced-handles ((p handle))
    (ccl:%get-byte p index)))

(defun bword (handle index)
  (declare (fixnum index))
  (ccl:with-dereferenced-handles ((p handle))
    (ccl:%get-word p (* index 2))))

(defun blong (handle index)
  (declare (fixnum index))
  (ccl:with-dereferenced-handles ((p handle))
    (ccl:%get-long p (* index 4))))

(defun set-buffer-byte (handle index value)
  (ccl:with-dereferenced-handles ((p handle))
    (ccl:%put-byte p value index))
  value)

(defun set-buffer-word (handle index value)
  (declare (fixnum index))
  (ccl:with-dereferenced-handles ((p handle))
    (ccl:%put-word p value (* index 2)))
  value)

(defun set-buffer-long (handle index value)
  (declare (fixnum index))
  (ccl:with-dereferenced-handles ((p handle))
    (ccl:%put-long p value (* index 4)))
  value)

(defsetf bbyte set-buffer-byte "Sets buffer[index] to byte value.")

(defsetf bword set-buffer-word "Sets buffer[index] to word value.")

(defsetf blong set-buffer-long "Sets buffer[index] to long value.")


;;;
;;; Array routines
;;;

(defun c-clear-array-1 (begin end array)
  (ccl:with-dereferenced-handles ((p array))
    (c-clear-array-aux begin end p)))

(ccl:deffcfun (c-clear-array-aux "cleararray1" :check-args *clm-check-args*)
  ((fixnum :long) (fixnum :long) (t :ptr))
 :novalue)

(defun c-abs-max-array (begin end array)
  (ccl:with-dereferenced-handles ((p array))
    (c-abs-max-array-aux begin end p)))

(ccl:deffcfun (c-abs-max-array-aux "absmaxarr" :check-args *clm-check-args*)
  ((fixnum :long) (fixnum :long) (t :ptr))
 :long)

(ccl:deffcfun (c-last-timed-max "get_last_time" :check-args *clm-check-args*) () :long)

(defun c-timed-abs-max-array (begin end array)
  (ccl:with-dereferenced-handles ((p array))
    (c-timed-abs-max-array-aux begin end p)))

(ccl:deffcfun (c-timed-abs-max-array-aux "timedabsmaxarr" :check-args *clm-check-args*)
  ((fixnum :long) (fixnum :long) (t :ptr))
 :long)

#|
;;; this can't be used because we can't pass float arrays reliably to C
(defun c-normalize (length array)
  (ccl:with-dereferenced-handles ((p array))
    (c-normalize-aux length p)))

(ccl:deffcfun (c-normalize-aux "normarray") 
  ((fixnum :long) (t :ptr))
  :novalue)
|#

(defun c-arrblt-1 (begin end new-begin array)
  (ccl:with-dereferenced-handles ((p array))
    (c-arrblt-1-aux begin end new-begin p)))

(ccl:deffcfun (c-arrblt-1-aux "arrblt" :check-args *clm-check-args*) 
  ((fixnum :long) (fixnum :long) (fixnum :long) (t :ptr))
  :novalue)
 
(defun c-make-array (size)
  (let ((arr (make-buffer size)))
  ;; added by BIL 19-Oct
  (c-clear-array-1 0 (1- size) arr)
  arr))

(defun c-free-array (array)
  (free-buffer array))

(defun c-setf-aref (array index value)
  (setf (blong array index) value))

(defun c-getf-aref (array index)
  (blong array index))

(defun c-incf-aref (array index value)
  (let ((old (blong array index)))
    (setf (blong array index)
          (+ old value))))

(defun c-incf2 (array index value)
  (ccl:with-dereferenced-handles ((p array))
    (c-incf-aux p index value)))

(ccl:deffcfun (c-incf-aux "incarray" :check-args *clm-check-args*) 
  ((t :ptr) (fixnum :long) (fixnum :long))
  :long)

;;;
;;; File routines
;;;

(ccl:deffcfun (c-close "clm_close" :check-args *clm-check-args*) 
  ((fixnum :long))
  :novalue)

(defun c-open-input-file (pathname)
  (c-open-input-file-aux (namestring (translate-logical-pathname pathname))))

(ccl:deffcfun (c-open-input-file-aux "clm_open_read" :check-args *clm-check-args*) 
  (string)
  :long)

(defun c-open-output-file (pathname)
  (c-open-output-file-aux (namestring (translate-logical-pathname pathname))))

(ccl:deffcfun (c-open-output-file-aux "clm_open_write" :check-args *clm-check-args*) 
  (string) 
  :long)

(ccl:deffcfun (c-open-clm-file-descriptors "open_clm_file_descriptors" :check-args *clm-check-args*)
  ((fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long))
  :novalue)

(ccl:deffcfun (c-close-clm-file-descriptors "close_clm_file_descriptors" :check-args *clm-check-args*)
  ((fixnum :long))
  :novalue)

(ccl:deffcfun (c-write-zeros "cwritez" :check-args *clm-check-args*)
  ((fixnum :long) (fixnum :long))
  :novalue)

(ccl:deffcfun (c-true-file-length "clm_true_file_length" :check-args *clm-check-args*)
  (string)
  :long)

(ccl:deffcfun (c-file-position "clm_seek" :check-args *clm-check-args*)
  ((fixnum :long) (fixnum :long) (fixnum :long))
  :long)

(defun c-read-mono (fd beg end buf)
  (ccl:with-dereferenced-handles ((p buf))
    (c-read-mono-aux fd beg end p)))

(ccl:deffcfun (c-read-mono-aux "crdmono" :check-args *clm-check-args*) 
  ((fixnum :long) (fixnum :long) (fixnum :long) (t :ptr))
  :novalue)

(defun c-read-stereo (fd beg end bufa bufb)
  (ccl:with-dereferenced-handles ((p1 bufa)
				  (p2 bufb))
    (c-read-stereo-aux fd beg end p1 p2)))

(ccl:deffcfun (c-read-stereo-aux "crdstereo" :check-args *clm-check-args*)
  ((fixnum :long)  (fixnum :long) (fixnum :long) 
   (t :ptr) (t :ptr))
  :novalue)

(defun c-read-quad (fd beg end bufa bufb bufc bufd)
  (ccl:with-dereferenced-handles ((p1 bufa)
				  (p2 bufb)
				  (p3 bufc)
				  (p4 bufd))
    (c-read-quad-aux fd beg end p1 p2 p3 p4)))

(ccl:deffcfun (c-read-quad-aux "crdquad" :check-args *clm-check-args*)
  ((fixnum :long)  (fixnum :long) (fixnum :long) 
   (t :ptr) (t :ptr) (t :ptr) (t :ptr))
  :novalue)



(defun c-write-mono (fd beg end buf)
  (ccl:with-dereferenced-handles ((p buf))
    (c-write-mono-aux fd beg end p)))

(ccl:deffcfun (c-write-mono-aux "cwrtmono" :check-args *clm-check-args*) 
  ((fixnum :long) (fixnum :long) (fixnum :long) (t :ptr))
  :novalue)

(defun c-write-stereo (fd beg end bufa bufb)
  (ccl:with-dereferenced-handles ((p1 bufa)
                              (p2 bufb))
    (c-write-stereo-aux fd beg end p1 p2)))

(ccl:deffcfun (c-write-stereo-aux "cwrtstereo" :check-args *clm-check-args*) 
  ((fixnum :long)  (fixnum :long) (fixnum :long)
   (t :ptr) (t :ptr))
  :novalue)


(defun c-write-quad (fd beg end bufa bufb bufc bufd)
  (ccl:with-dereferenced-handles ((p1 bufa)
				  (p2 bufb)
				  (p3 bufc)
				  (p4 bufd))
   (c-write-quad-aux fd beg end p1 p2 p3 p4)))

(ccl:deffcfun (c-write-quad-aux "cwrtquad" :check-args *clm-check-args*) 
  ((fixnum :long)  (fixnum :long) (fixnum :long)
   (t :ptr) (t :ptr) (t :ptr) (t :ptr))
  :novalue)


(ccl:deffcfun (c-convolve "c_convolve" :check-args *clm-check-args*)
  ((string) 
   (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) 
   (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) 
   (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long))
  :novalue)


(ccl:deffcfun (c-mix-compatible-sounds-aux "mix_compatible_sounds" :check-args *clm-check-args*)
  ((string) (fixnum :long) (string) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (t :ptr))
  :long)

(defun c-mix-compatible-sounds (mainf mains mergef merges samps scl use env)
  (ccl:with-dereferenced-handles ((p1 env)) 
    (c-mix-compatible-sounds-aux mainf mains mergef merges samps scl use p1)))


(ccl:deffcfun (c-mix-mono-to-stereo-sounds-aux "mix_mono_to_stereo_sounds" :check-args *clm-check-args*)
  ((string) (fixnum :long) (string) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (t :ptr))
  :long)

(defun c-mix-mono-to-stereo-sounds (mainf mains mergef merges samps sclA sclB chan use env)
  (ccl:with-dereferenced-handles ((p1 env)) 
    (c-mix-mono-to-stereo-sounds-aux mainf mains mergef merges samps sclA sclB chan use p1)))


(ccl:deffcfun (c-mix-stereo-to-mono-sounds-aux "mix_stereo_to_mono_sounds" :check-args *clm-check-args*)
  ((string) (fixnum :long) (string) (fixnum :long) (fixnum :long) (fixnum :long) 
   (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (t :ptr))
  :long)

(defun c-mix-stereo-to-mono-sounds (mainf mains mergef merges samps sclA sclB chan use init env)
  (ccl:with-dereferenced-handles ((p1 env)) 
    (c-mix-stereo-to-mono-sounds-aux mainf mains mergef merges samps sclA sclB chan use init p1)))


(ccl:deffcfun (c-mix-stereo-to-stereo-sounds "mix_stereo_to_stereo_sounds" :check-args *clm-check-args*)
  ((string) (fixnum :long) (string) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long))
  :long)


(ccl:deffcfun (c-mix-quad-to-quad-sounds "mix_quad_to_quad_sounds" :check-args *clm-check-args*)
  ((string) (fixnum :long) (string) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) 
   (fixnum :long) (fixnum :long) (t :ptr))
  :long)
