;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;

(in-package :clm)

;;;
;;; I/O routines 
;;;
;;;     buffered I/O, packing and unpacking, open/close files, OUTA, OUTB, INA, INB
;;;
;;; from the user's point of view, we are reading/writing short-floats
;;; from the DAC's point of view (and in .SND files) we use 16 bit two's-complement bytes packed
;;;    two to a word.
;;;
;;; Needed by these routines:
;;;
;;; (read-header file-ptr) returns ptr to header, data-offset, file size (in samples), chans
;;; (write-header file hdr) writes hdr, returns ptr to hdr, offset, chans
;;; (update-header file hdr new-file-size) updates that portion of the header (file size is in samples)
;;;

(defvar clm-statistics nil)
(defvar clm-notehook nil)

(defvar *clm-interrupted* 0)

(defconstant in-f 0)			;open for input
(defconstant out-f 1)			;open for output

(defvar *clm-save-output-array* nil)
(defvar *clm-saved-output-array* nil)	;these are for programs like dpysnd that want the "instrument" output directly

(defun caref (arr i)
  (if #-mcl (integerp arr)
      #+mcl (ccl:handlep arr)
      (c-getf-aref arr i)
    (aref arr i)))

(defun saref (arr i val)
  (if #-mcl (integerp arr)
      #+mcl (ccl:handlep arr)
      (c-setf-aref arr i val)
    (setf (aref arr i) val)))

(defun iaref (arr i val)
  (if #-mcl (integerp arr)
      #+mcl (ccl:handlep arr)
      (c-incf-aref arr i val)
    (incf (aref arr i) val)))

(defun array-contents (arr arrsiz)
  (let ((top (min *clm-array-print-length* arrsiz)))
    (loop for i from 0 below top collect (caref arr i))))


(defstruct (IO

  (:print-function			;lisp prints the entire goddam array in trace!
   (lambda (f s k)			;struct stream depth
     (declare (ignore k))
     (format s "~&#<IO: \"~A\", ~A, ~A, IO-chn: ~A,
     beg: ~A, end: ~A, siz: ~A, hdr-end: ~A, data-start: ~A, data-end: ~A, index: ~A~A, 
     header: ~A, 
     DatA: (~{~A ~}~A [~A, ~A]~A~A~A>"
     (IO-nam f) 
     (if (= (IO-dir f) in-f) "read" "write")
     (if (IO-dat-d f) "quad" (if (IO-dat-b f) "stereo" "mono"))
     (IO-fil f) (IO-beg f) (IO-end f) (IO-siz f) (IO-hdr-end f) (IO-data-start f) (IO-data-end f)
     (IO-open-index f) 
     (if (IO-external f) ", external" "")
     (IO-hdr f) 
     (if (IO-dat-a f) (array-contents (IO-dat-a f) (IO-bufsiz f)) '("EMPTY!"))
     (if (> (IO-bufsiz f) *clm-array-print-length*) "...)"  ")")
     (IO-dat-a f) (IO-bufsiz f)
     (if (IO-dat-b f) 
	 (format nil "~%     DatB: (~{~A ~}~A)"
		 (array-contents (IO-dat-b f) (IO-bufsiz f))
		 (if (> (IO-bufsiz f) *clm-array-print-length*) "..."  ""))
       "")
     (if (IO-dat-c f) 
	 (format nil "~%     DatC: (~{~A ~}~A)"
		 (array-contents (IO-dat-c f) (IO-bufsiz f))
		 (if (> (IO-bufsiz f) *clm-array-print-length*) "..."  ""))
       "")
     (if (IO-dat-d f) 
	 (format nil "~%     DatD: (~{~A ~}~A)"
		 (array-contents (IO-dat-d f) (IO-bufsiz f))
		 (if (> (IO-bufsiz f) *clm-array-print-length*) "..."  ""))
       "")))))
  fil					;the stream associated with this record
  nam					;its pathname
  dat-a					;its currently in-core data (channel A) (may be c array, not lisp array)
					;  on SGI, this is a bignum
  (bufsiz file-buffer-size)		;how big are these buffers (needed for c calls)
  (dat-b nil)				;channel B
  (dat-c nil)
  (dat-d nil)
  beg					;number of first in-core sample (samples start at 0 no matter where the header ends)
  end					;and the last (inclusive here)
  hdr					;pointer to header data, if any
  hdr-end				;where real data starts
  data-end				;where real data ends in DAT (so we don't output endless 0's)
  (dir out-f)				;are we output, input, or both
  open-index
  siz					;number of samples in file (i.e. "frames" -- bytes = siz * 2 *channels)
  (data-start 0)
  external)				;are we following a non-loop-sample-counter index


(defvar *current-output-file* nil)	;pointer to IO record holding an open output file
(defvar *current-input-file* nil)
(defvar *clm-current-open-files* nil)
(defvar *clm-hidden-open-files* nil)
(defvar *reverb* nil)

(defconstant fixnum-size (integer-length most-positive-fixnum))

(defvar last-buffer-index (- file-buffer-size 1))
(defvar maximum-arrblt-size (* file-buffer-size .9))
(defvar read-backwards-size (* file-buffer-size .75))

(defun change-file-buffer-size (new-size)
  (setf file-buffer-size new-size)
  (setf last-buffer-index (- file-buffer-size 1))
  (setf maximum-arrblt-size (* file-buffer-size .9))
  (setf read-backwards-size (* file-buffer-size .75)))

(defconstant snd-sample-size 15)	;number of significant bits in a sample as stored in a file

(defconstant high-byte-mask (ash -1 (+ snd-sample-size 1)))
(defconstant low-byte-mask (lognot high-byte-mask))

(defconstant snd-max (- (expt 2 snd-sample-size) 1))
(defconstant snd-min (- (expt 2 snd-sample-size)))
(defconstant snd-top (expt 2 (+ snd-sample-size 1)))

(defconstant fil-sample-type 'fixnum)
(defconstant fil-sample-zero 0)


#|
;;; these versions are pretty, but horrendously slow
(defun real-to-fix (x) 
  (min (max snd-min (round (scale-float (float x) snd-sample-size))) snd-max))

(defun fix-to-real (i)
  (scale-float (float i) (- snd-sample-size)))  
|#

(defun real-to-fix (x)
  (declare (optimize (safety 0) (speed 3))) 
  (declare (type single-float x)) 
  (round (* 32768 x)))

(defconstant flt_scale (/ 1.0 32768.0))

(defun fix-to-real (i)
  (declare (optimize (safety 0) (speed 3))) 
  (declare (type fixnum i)) 
  (float (* i flt_scale)))
  
(defun check-stream-indices (o-stream)
  (if (or (minusp (IO-data-start o-stream)) (minusp (IO-data-end o-stream)))
      (cerror "Try to write anyway?!?" 
	      "File buffer index for ~A is negative: ~D" 
	      o-stream
	      (if (minusp (IO-data-start o-stream))
		  (IO-data-start o-stream)
		(IO-data-end o-stream)))))

(defun put-data (o-stream)
  ;; Lisp really ought to have block IO.
  (when (IO-fil o-stream)
    (check-stream-indices o-stream)
    (if (IO-dat-b o-stream)
	(if (and (IO-dat-c o-stream)
		 (IO-dat-d o-stream))
	    (c-write-quad (IO-fil o-stream)
			  (IO-data-start o-stream) 
			  (IO-data-end o-stream) 
			  (IO-dat-a o-stream) 
			  (IO-dat-b o-stream)
			  (IO-dat-c o-stream)
			  (IO-dat-d o-stream))
	  (c-write-stereo (IO-fil o-stream) 
			  (IO-data-start o-stream) 
			  (IO-data-end o-stream) 
			  (IO-dat-a o-stream) 
			  (IO-dat-b o-stream)))
      (c-write-mono (IO-fil o-stream) 
		    (IO-data-start o-stream) 
		    (IO-data-end o-stream) 
		    (IO-dat-a o-stream)))))

(defun put-partial-data (o-stream end)
  (when (IO-fil o-stream)
    (check-stream-indices o-stream)
    (if (IO-dat-b o-stream)
	(if (and (IO-dat-c o-stream)
		 (IO-dat-d o-stream))
	    (c-write-quad (IO-fil o-stream)
			  (IO-data-start o-stream) 
			  end
			  (IO-dat-a o-stream) 
			  (IO-dat-b o-stream)
			  (IO-dat-c o-stream)
			  (IO-dat-d o-stream))
	  (c-write-stereo (IO-fil o-stream) 
			  (IO-data-start o-stream) 
			  end
			  (IO-dat-a o-stream) 
			  (IO-dat-b o-stream)))
      (c-write-mono (IO-fil o-stream) 
		    (IO-data-start o-stream) 
		    end
		    (IO-dat-a o-stream)))))


(defun get-data (i-stream i-size)
  (if (> i-size (IO-bufsiz i-stream))
      (error "input request is too big: ~D for ~S" i-size i-stream))
  (if (zerop i-size) (error "no-op in get-data"))
  (check-stream-indices i-stream)
  (if (IO-dat-b i-stream)
      (if (and (IO-dat-c i-stream)
	       (IO-dat-d i-stream))
	  (c-read-quad (IO-fil i-stream) 0 (- i-size 1) (IO-dat-a i-stream) (IO-dat-b i-stream) (IO-dat-c i-stream) (IO-dat-d i-stream))  
	(c-read-stereo (IO-fil i-stream) 0 (- i-size 1) (IO-dat-a i-stream) (IO-dat-b i-stream)))
    (c-read-mono (IO-fil i-stream) 0 (- i-size 1) (IO-dat-a i-stream))))


(defun sample-position (io-stream loc)	;replacement for FILE-POSITION (sort of -- 16 bit bytes here)
					;in original, this was 32 bit bytes, and hdr-end was 32 as well
					;7-Mar-94 changed to straight byte-wise position (funny-length headers)
					;so it's sort of sample-position given header size in bytes
  (if (IO-fil io-stream)
      (if (IO-dat-b io-stream)
	  (if (and (IO-dat-c io-stream)
		   (IO-dat-d io-stream))
	      (c-file-position (IO-fil io-stream) (+ (IO-hdr-end io-stream) (floor (* 8 loc))) 0)
	    (c-file-position (IO-fil io-stream) (+ (IO-hdr-end io-stream) (floor (* 4 loc))) 0))
	(c-file-position (IO-fil io-stream) (+ (IO-hdr-end io-stream) (floor (* 2 loc))) 0)))
  loc)

(defun flush-snd (o-stream)		;if anything needs to be written out, do it
  (when (and (IO-fil o-stream)
	     (/= in-f (IO-dir o-stream)) 
	     (plusp (IO-data-end o-stream))
	     (/= (IO-data-end o-stream) (IO-data-start o-stream)))
    (if (>= (IO-data-end o-stream) (IO-bufsiz o-stream))
	(progn
	  (cerror "continuation will discard this buffer" "data-end is too big: ~S" o-stream)
	  (setf (io-data-end o-stream) (io-data-start o-stream)))
      (progn
	(sample-position o-stream (+ (IO-beg o-stream) (IO-data-start o-stream)))
	(put-data o-stream)
	(setf (IO-data-start o-stream) (IO-data-end o-stream))
	(setf (IO-siz o-stream) 
	  (max (IO-siz o-stream) 
	       (+ (IO-beg o-stream) (IO-data-end o-stream) 1)))))))
               ;; + 1 here because data-end is 0 based, but datasize is 1 based

(defvar min-flush-amount 128)
(defvar max-flush-amount 512)

(defun flush-partial-snd (o-stream)
  (when (and (IO-fil o-stream)
	     (plusp (IO-data-end o-stream))
	     (> (IO-data-end o-stream) (+ (IO-data-start o-stream) min-flush-amount)))
    (let ((end (min (+ (IO-data-start o-stream) max-flush-amount)
		    (IO-data-end o-stream))))
      (sample-position o-stream (+ (IO-beg o-stream) (IO-data-start o-stream)))
      (put-partial-data o-stream end)
      (setf (IO-data-start o-stream) end)
      (setf (IO-siz o-stream) 
	(max (IO-siz o-stream) 
	     (+ (IO-beg o-stream) end 1))))))

(defvar *clm-current-max-file-index* -1)

(defun find-open-slot (arr)		;used also in remember-delay in mus.lisp
  (declare (optimize (speed 3) (safety 0)))
  (do ((i 0 (+ i 1))
       (lim (array-total-size arr)))
      ((or (>= i lim) (null (aref arr i))) i)))

(defun add-file (fil)
  (declare (optimize (speed 3) (safety 0)))
  (if (null *clm-current-open-files*)
      (setf *clm-current-open-files* (make-array *available-IO-channels* :initial-element nil)))
  (let ((ind (find-open-slot *clm-current-open-files*)))
    (if (< ind *available-IO-channels*)
	(progn
	  (setf (aref *clm-current-open-files* ind) fil)
	  (setf *clm-current-max-file-index* (max ind *clm-current-max-file-index*)))
      (error "clm has run out of file IO channels"))
    ind))

(defun remove-file (fil)
  (declare (optimize (speed 3) (safety 0)))
  (flet ((get-new-max-index ()
	   (let ((temp -1))
	     (loop for i from 0 below (array-total-size *clm-current-open-files*) do
               (if (aref *clm-current-open-files* i) (setf temp i)))
	     temp)))
    (when *clm-current-open-files*
      (setf (aref *clm-current-open-files* (IO-open-index fil)) nil)
      (if (= *clm-current-max-file-index* (IO-open-index fil))
	  (setf *clm-current-max-file-index* (get-new-max-index))))
    nil))
		   
(defun clm-open-input (&optional (name default-sound-file-name) 
				 (reopen nil)) 
  ;;open .snd file and read in a buffer's worth, return IO pointer
  (declare (optimize (speed 3) (safety 0)))
  (let ((real-name (namestring (full-merge-pathnames name default-sound-file-name))))
    (multiple-value-bind
	(head head-end file-size chans)	;head-end can be anything as of 6-Mar-94
	(read-header real-name)
      (cond ((or (minusp chans) (zerop chans))
	     (cerror "Set channel number to 1" "Weird number of channels: ~D." chans)
	     (setf chans 1)))
      (if (> chans 4)
	  (error "We can only handle mono, stereo, or quad, not ~D channels." chans))
      (let* ((min-size (min file-size file-buffer-size))
	     (buf-size (if reopen file-buffer-size min-size))
	     (ifile (make-IO :beg 0
			     :end (- buf-size 1)
			     :bufsiz buf-size
			     :data-end min-size
			     :nam real-name
			     :dir (if reopen out-f in-f)
			     :hdr head
			     :hdr-end head-end
			     :siz file-size
			     :dat-a (c-make-array buf-size)
			     :dat-b (if (> chans 1) (c-make-array buf-size))
			     :dat-c (if (= chans 4) (c-make-array buf-size))
			     :dat-d (if (= chans 4) (c-make-array buf-size))
			     :fil (if reopen 
				      (c-open-output-file real-name)
				    (c-open-input-file real-name)))))
	(c-open-clm-file-descriptors (io-fil ifile) (c-snd-header-format) (c-snd-header-datum-size) (c-snd-header-data-location))
	(if (and reopen (null *current-output-file*)) 
	    (setf *current-output-file* ifile)
	  (setf *current-input-file* ifile))
	(sample-position ifile 0)	;start at first data sample
	(if (plusp min-size) (get-data ifile min-size))
	(setf (IO-open-index ifile) (add-file ifile))
	ifile))))

(defun clm-write-header (name header output)
  (declare (optimize (speed 3) (safety 0)))
  (if output
      (write-header name header)
    (values nil 7 1)))

(defun clm-open-output (&optional (name default-sound-file-name) (header nil) (out-file-p t) (buf-size nil))
  (declare (optimize (speed 3) (safety 0)))
  (let ((real-name (namestring (full-merge-pathnames name default-sound-file-name)))
	(true-buffer-size (or buf-size file-buffer-size)))
    (multiple-value-bind
	(head head-end chans)
	(clm-write-header real-name header out-file-p)
      (if (zerop chans) (setf chans 1))
      (cond ((> chans 4)
	     (cerror "Will set channels to 2"
		     "We can only handle mono, stereo, or quad currently, not ~D channels" chans)
	     (setf chans 2)))
      (let ((ofile (make-IO :beg 0
			    :end (- true-buffer-size 1)
			    :bufsiz true-buffer-size
			    :data-end 0
			    :siz 0
			    :nam real-name
			    :dir out-f
			    :hdr head
			    :hdr-end (* 4 head-end)
			    :dat-a (c-make-array true-buffer-size)
			    :dat-b (if (> chans 1) (c-make-array true-buffer-size))
			    :dat-c (if (= chans 4) (c-make-array true-buffer-size))
			    :dat-d (if (= chans 4) (c-make-array true-buffer-size))
			    :fil (if out-file-p (c-open-output-file real-name)))))
	(c-open-clm-file-descriptors (io-fil ofile) snd-16-linear 2 (* 4 head-end))
	(if (null *current-output-file*) (setf *current-output-file* ofile))
	(setf (IO-open-index ofile) (add-file ofile))
	ofile))))
#|
  ;;if no /zap area, should use current working dir:
  (when (and (equal (second (pathname-directory default-sound-file-name")) "zap")
        (not (directory "/zap"))
    (setf default-sound-file-name "test.snd")
    (setf default-reverb-file-name "reverb.snd"))
|#

(defun clm-reopen-output (&optional (name default-sound-file-name))
  (declare (optimize (speed 3) (safety 0)))
  (let ((fil (clm-open-input name t)))
    (if fil (setf (IO-data-end fil) 0))		;i.e nothing needs to be written yet
    fil))

(defun bufclr (arr arrsiz &optional (beg 0))
  (declare (optimize (speed 3) (safety 0)))
  (c-clear-array-1 beg (floor (- arrsiz 1)) arr))

(defun IO-bufclr (f &optional (beg 0))
  (declare (optimize (speed 3) (safety 0)))
  (let ((arrend (floor (- (IO-bufsiz f) 1))))
    (if (not (IO-dat-a f)) (error "~A: attempt to clear non-existent buffer" (IO-nam f)))
    (c-clear-array-1 beg arrend (IO-dat-a f))
    (when (IO-dat-b f) 
      (c-clear-array-1 beg arrend (IO-dat-b f))
      (if (IO-dat-c f) (c-clear-array-1 beg arrend (IO-dat-c f)))
      (if (IO-dat-d f) (c-clear-array-1 beg arrend (IO-dat-d f))))))

(defun read-in (i-stream loc &optional (num 0) (clear-rest t)) ;get a buffer's worth from this stream starting at LOC
  (let* ((file-end (IO-siz i-stream))	;"loc" of last sample in file
	 (bytes (if (/= 0 num) num (min file-buffer-size (- file-end loc)))))
    (if (minusp bytes)			;tried to access beyond current end of file
	(progn
	  (if (= (IO-dir i-stream) in-f)
	      (error "Attempt to read ~A past end of data (sample ~D)." 
		     (IO-nam i-stream) loc))
	  (IO-bufclr i-stream)
	  (setf bytes file-buffer-size)
	  (setf (IO-data-end i-stream) 0)
	  (c-file-position (IO-fil i-stream) 0 2) ;=>lseek(fd,0L,2)=>go to end of file
	  ;;now write zeros until we reach LOC, then set BEG and END based on word boundaries
	  (if (IO-dat-d i-stream)
	      (progn
		(c-write-zeros (IO-fil i-stream) (* 2 (floor (- loc file-end))))
		(setf (IO-beg i-stream) loc))
	    (if (IO-dat-b i-stream)
		(progn
		  (c-write-zeros (IO-fil i-stream) (floor (- loc file-end)))
		  (setf (IO-beg i-stream) loc))
	      (progn
		(c-write-zeros (IO-fil i-stream) (floor (- loc file-end) 2))
		(setf (IO-beg i-stream) (if (evenp loc) loc (- loc 1)))))))
      (progn
	(setf (IO-beg i-stream) (sample-position i-stream loc))
	(if (plusp bytes) (get-data i-stream bytes))
	(if (and clear-rest 
		 (< bytes last-buffer-index)) 
	    (IO-bufclr i-stream bytes))
	(if (= (io-dir i-stream) in-f)
	    (setf (IO-data-end i-stream) (max 0 (min (floor (- file-end (IO-beg i-stream) 1)) last-buffer-index)))
	  (setf (io-data-end i-stream) 0))))
    (setf (IO-end i-stream) (+ (IO-beg i-stream) last-buffer-index)))
  (setf (IO-data-start i-stream) (max 0 (IO-data-end i-stream))))

(defun free-IO-buffers (i-stream)
  (declare (optimize (speed 3) (safety 0)))
  (if (null (IO-dat-a i-stream))
      (error "attempt to release buffers twice: ~A" (io-nam i-stream)))
  (setf (IO-bufsiz i-stream) -1)
  (if (and *clm-save-output-array*
	   (null (IO-fil i-stream)))
      (progn
	(when (and *clm-saved-output-array* (not (zerop *clm-saved-output-array*)))
	  (print "why are we freeing the left over output array?")
	  (c-free-array *clm-saved-output-array*))
	(setf *clm-saved-output-array* (IO-dat-a i-stream)))
    (c-free-array (IO-dat-a i-stream)))
  (setf (IO-dat-a i-stream) nil)
  (when (IO-dat-b i-stream)
    (c-free-array (IO-dat-b i-stream))
    (setf (IO-dat-b i-stream) nil)
    (when (IO-dat-c i-stream)
      (c-free-array (IO-dat-c i-stream))
      (setf (IO-dat-c i-stream) nil))
    (when (IO-dat-d i-stream)
      (c-free-array (IO-dat-d i-stream))
      (setf (IO-dat-d i-stream) nil))))

(defun cleanup-possible-duplications (i-stream)
  (declare (optimize (speed 3) (safety 0)))
  (if (eq i-stream *current-output-file*)
      (setf *current-output-file* nil))
  (if (eq i-stream *reverb*)
      (setf *reverb* nil))
  (if (eq i-stream *current-input-file*)
      (setf *current-input-file* nil)))
	
(defun clm-close-input (&optional (i-stream *current-input-file*))
  (declare (optimize (speed 3) (safety 0)))
  (when i-stream
    (if *clm-hidden-open-files*
	(let ((index (io-open-index i-stream)))
	  (loop for fil-pair in *clm-hidden-open-files* do
	    (if (= (first fil-pair) index)
		(clm-close-input (aref *clm-current-open-files* (second fil-pair)))))
	  (setf *clm-hidden-open-files* (remove index *clm-hidden-open-files* :key #'first))))
    (c-close (IO-fil i-stream))
    (remove-file i-stream)
    (free-IO-buffers i-stream)
    (cleanup-possible-duplications i-stream)))

(defun clm-close-output (&optional (o-stream *current-output-file*))
  (declare (optimize (speed 3) (safety 0)))
  (when o-stream
    (flush-snd o-stream)
    (if (IO-fil o-stream) (c-close (IO-fil o-stream)))
    (remove-file o-stream)
    (free-IO-buffers o-stream)
    (if (IO-fil o-stream)
	(update-header (IO-nam o-stream) 
		       (IO-hdr o-stream)
		       (IO-siz o-stream)))
    (cleanup-possible-duplications o-stream)))

(defun file-check (loc io-stream default-stream dir-type default-name &optional (bufsiz 0))
  (when (null io-stream)		;output file not opened yet
    (let ((typ (if (= dir-type in-f) "input" "output")))
      (if (null default-stream)
	  (progn
	    (cerror "Open ~S and go on" "No ~S file open" default-name typ)
	    (if (= dir-type in-f)
		(setf *current-input-file* (clm-open-input default-name))
	      (setf *current-output-file* (clm-open-output default-name))))
	(progn
	  (cerror "Use current default ~S ~S instead"
		  "Desired ~S file ~S is not open"
		  typ default-name typ io-stream)
	  (setf io-stream default-stream)))))
  (when (not (<= (IO-beg io-stream) loc (+ loc bufsiz) (IO-end io-stream)))
    (if (/= dir-type in-f) 
	(flush-snd io-stream))		;current in-core buffer does not accomodate LOC
    (if (and (< loc (IO-beg io-stream)) 
	     (> (+ loc maximum-arrblt-size) (IO-beg io-stream)))
	(let* ((loc1 (if (>= (+ loc 10) (IO-beg io-stream))
			 (max 0 (- loc read-backwards-size))
		       loc))
	       (true-new-loc (if (IO-dat-b io-stream) loc1 (floor (* 2 (floor loc1 2)))))
	       (newbytes (floor (- (IO-beg io-stream) true-new-loc)))
	       (newend (min last-buffer-index (floor (+ newbytes (IO-data-end io-stream)))))
	       (newstart (floor (- newend newbytes))))
	  
	  ;; data ends at IO-data-end, buffer relative.
	  ;; we want to make room for newbytes more samples at the front of the buffer.
	  
	  (c-arrblt-1 newstart 0 newend (IO-dat-a io-stream))
	  (when (IO-dat-b io-stream) 
	    (c-arrblt-1 newstart 0 newend (IO-dat-b io-stream))
	    (if (IO-dat-c io-stream) (c-arrblt-1 newstart 0 newend (IO-dat-c io-stream)))
	    (if (IO-dat-d io-stream) (c-arrblt-1 newstart 0 newend (IO-dat-d io-stream))))
	  (read-in io-stream true-new-loc newbytes nil))
      (read-in io-stream loc))))

(defun file-reset (loc f)		;force f to start at loc (assume data-end set already)
  (when (/= loc (IO-beg f))
    (if (= out-f (IO-dir f)) (flush-snd f))
    (if (or (= out-f (IO-dir f))
	    (< loc (IO-siz f)))
	(read-in f loc))))

(defun out-sample (loc data o-stream buf)
  (declare (optimize (safety 0) (speed 3)))
  (declare (type io o-stream))
  (when (/= 0.0 data)
    (let ((index (- loc (IO-beg o-stream))))
      (when (not (<= (IO-beg o-stream) loc (IO-end o-stream)))
	(file-check loc o-stream *current-output-file* out-f "Test.snd")
	(setf index (- loc (io-beg o-stream))))
      (c-incf-aref buf index (real-to-fix data)) ;was iaref
      (setf (IO-data-end o-stream) (max (IO-data-end o-stream) index)))))

(defun setf-out-sample (loc data o-stream buf)
  (when (/= 0.0 data)
    (file-check loc o-stream *current-output-file*  out-f "Test.snd")
    (saref buf (- loc (IO-beg o-stream)) (real-to-fix data))
    (setf (IO-data-end o-stream) (max (IO-data-end o-stream) (- loc (IO-beg o-stream))))))

(defun outa (loc data &optional (o-stream *current-output-file*))
  (if o-stream
      (out-sample loc data o-stream (IO-dat-a o-stream))
    (error "attempt to do output without any output file open")))

(defun outb (loc data &optional (o-stream *current-output-file*))
  (if o-stream
      (if (IO-dat-b o-stream)
	  (out-sample loc data o-stream (IO-dat-b o-stream))
	(cerror "Will flush it" "Attempt to set channel 2 of mono file: ~A?" (IO-nam o-stream)))
    (error "attempt to do output without any output file open")))

(defun outc (loc data &optional (o-stream *current-output-file*))
  (if o-stream
      (if (IO-dat-c o-stream)
	  (out-sample loc data o-stream (IO-dat-c o-stream))
	(error "Attempt to set channel 3 of ~A file: ~A?" (if (IO-dat-b o-stream) "stereo" "mono") (IO-nam o-stream)))
    (error "attempt to do output without any output file open")))

(defun outd (loc data &optional (o-stream *current-output-file*))
  (if o-stream
      (if (IO-dat-d o-stream)
	  (out-sample loc data o-stream (IO-dat-d o-stream))
	(error "Attempt to set channel 4 of ~A file: ~A?" (if (IO-dat-b o-stream) "stereo" "mono") (IO-nam o-stream)))
    (error "attempt to do output without any output file open")))

(defun quad (&optional (o-stream *current-output-file*)) (and o-stream (IO-dat-d o-stream)))
(defun stereo (&optional (o-stream *current-output-file*)) (and o-stream (IO-dat-b o-stream) (not (IO-dat-d o-stream))))
(defun mono (&optional (o-stream *current-output-file*)) (and o-stream (not (IO-dat-b o-stream))))

(defun clm-input-check (loc i-stream bufsiz)
  (file-check (floor loc) i-stream *current-input-file* in-f "test.snd" bufsiz))

(defun in-sample (loc i-stream buf)
  (if (and (> loc (IO-siz i-stream))
	   (or (not (eq *current-output-file* i-stream))
	       (> loc (io-end i-stream))))
	 0.0				;attempt to read past end of file?
    (progn
      (clm-input-check loc i-stream 0)
      (fix-to-real (caref buf (floor (- loc (IO-beg i-stream))))))))
;      (fix-to-real (aref buf (- (floor loc) (IO-beg i-stream)))))))

(defun ina (loc &optional (i-stream *current-input-file*))
  (in-sample loc i-stream (IO-dat-a i-stream)))

(defun inb (loc &optional (i-stream *current-input-file*))
  (if (IO-dat-b i-stream)
      (in-sample loc i-stream (IO-dat-b i-stream))
    (error "Attempt to read channel B of mono file: ~A" (IO-nam i-stream))))

(defun in-a (loc &optional (i-stream *current-input-file*))
  (in-sample loc i-stream (IO-dat-a i-stream)))

(defun in-b (loc &optional (i-stream *current-input-file*))
  (if (IO-dat-b i-stream)
      (in-sample loc i-stream (IO-dat-b i-stream))
    (error "Attempt to read channel B of mono file: ~A" (IO-nam i-stream))))

(defun in-c (loc &optional (i-stream *current-input-file*))
  (if (IO-dat-c i-stream)
      (in-sample loc i-stream (IO-dat-c i-stream))
    (error "Attempt to read channel C of ~A file: ~A" (if (IO-dat-b i-stream) "stereo" "mono") (IO-nam i-stream))))

(defun in-d (loc &optional (i-stream *current-input-file*))
  (if (IO-dat-d i-stream)
      (in-sample loc i-stream (IO-dat-d i-stream))
    (error "Attempt to read channel D of ~A file: ~A" (if (IO-dat-b i-stream) "stereo" "mono") (IO-nam i-stream))))

    
;;; the following is mostly to gussy up our "user interface" a little

(defun month-name (month) (nth (- month 1) '("Jan" "Feb" "Mar" "Apr" "May" "Jun" "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")))
(defun day-name (day) (nth day '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun")))

(defun whos-to-blame () 
  (declare (optimize (speed 3) (safety 0)))
  (let (#-mcl (site (or (long-site-name) (short-site-name)))
	#+mcl (site nil)
	#-mcl (user (first (last (pathname-directory (user-homedir-pathname))))) ;can be (:ABSOLUTE "Net" ...)
	#+mcl (user (ccl::with-macptrs ((h (mcl-get-string -16096)))
		      (if (not (or (ccl::%null-ptr-p h) 
				   (ccl::%null-ptr-p (ccl::%get-ptr h)) 
				   (eql 0 (ccl::%get-byte (ccl::%get-ptr h)))))
			  (ccl::%get-string h))))
	#-mcl (machine (machine-type))
	#+mcl (machine (format nil "~A (~A/system ~A with ~DMB)" (mac-machine-type) (mac-processor) (mac-os-name) (mac-ram-in-MB)))
	(lisp (lisp-implementation-type)))
    (if (or user site machine lisp)
	(format nil "~A~A~A~A~A~A~A~A~A"
		(if user "by " "")
		(if user user "")
		(if site " at " "")
		(if site site "")
		(if machine " (" "")
		(if machine machine "")
		(if machine ")" "")
		(if lisp " using " "")
		(if lisp lisp "")))))

(defun get-time-zone (time-zone daylight)

  ;; this is an incredible can of worms -- here I thought "time zones have names"
  ;; live and learn.  Some of these names are via Dave Mellinger and /etc/zoneinfo/sources/europe.
  ;; also, if you od -a /etc/zoneinfo/<name> the end will give the abbreviations for the time zone (i.e. PST)
  ;;          and od -x /etc/zoneinfo/<name> just before the abbreviations are the time zone offsets in seconds
  ;;     (these numbers are backwards from 24 from our point of view, and can be off by fractional hour amounts)
  ;; these time zone files are described in /usr/include/{bsd}/tzfile.h

  ;; and MCL apparently ignores time zones (MCL 2.0.1 returns 0 here in sunny CA)
  (declare (optimize (speed 3) (safety 0)))
  (case time-zone
    (0 (if daylight "BST " "GMT "))	;British Summer Time and Greenwich Mean Time

    (4 (if daylight "ADT " "AST "))	;Atlantic (?)
    (5 (if daylight "EDT " "EST "))	;Eastern (Daylight/Standard) Time
    (6 (if daylight "CDT " "CST "))	;Central
    (7 (if daylight "MDT " "MST "))	;Mountain
    (8 (if daylight "PDT " "PST "))	;Pacific
    (9 (if daylight "YDT " "YST "))	;Yukon

    ;; apparently some of Autralia's zones are off by a half hour
    ;;     North[CST] = 15.5
    ;;     South[CDT,CST] same 
    (12 (if daylight "NZDT " "NZST "))	;New Zealand? -- etc/zoneinfo/NZ
    (14 (if daylight "EDT " "EST "))	;/etc/zoneinfo/Australia/NSW and Queensland
    (15 "JST ")				;Japan time via Fernando
    (16 "WST ")				;Australia/West/Singapore?

    (21 (if daylight "EET DST " "EET ")) ;Eastern Europe Time (Turkey)
    (22 (if daylight "MET DST " "MET ")) ;Middle Europe Time (Poland)
    (23 (if daylight "WET DST " "WET ")) ;Western Europe?
    (t " ")))

(defun clm-get-default-header ()

  ;; get-decoded-time was returning completely silly times, but decode-universal-time
  ;; doesn't know about daylight savings time (it always returns nil)
  ;; Time zone can be between -24 and 24 (exclusive?)
  (declare (optimize (speed 3) (safety 0)))
  (multiple-value-bind (second minute hour date month year day daylight-saving-p time-zone)
      (get-decoded-time)
    #-(or kcl mcl) (declare (ignore second))
    #+(or kcl mcl) (declare (ignore second time-zone daylight-saving-p))
    (format nil "~&;Written on ~A ~D-~A-~D at ~D:~2,'0D ~A~A and clm of ~A" 
	    (day-name day) date (month-name month) (- year 1900) hour minute 
	    #-(or kcl mcl) (get-time-zone (if (not (minusp time-zone)) 
					      (floor time-zone) 
					    (+ (floor time-zone) 24)) 
					  daylight-saving-p)
            #+(or kcl mcl) " "
	    (whos-to-blame) 
	    *clm-date*)))


(defmacro revin (i)
  `(ina ,i *current-input-file*))

(defmacro revout (i val)
  `(outa ,i ,val *reverb*))

(defun clm-cleanup (&optional paranoid)
  (declare (optimize (speed 3) (safety 0)))
  (if *clm-current-open-files*
      (loop for i from 0 below *available-IO-channels* do
	(let ((fil (aref *clm-current-open-files* i)))
	  (if fil
	      (progn
		(if (= in-f (IO-dir fil))
		    (clm-close-input fil)
		  (progn
		    (when paranoid
		      (if (minusp (io-beg fil)) (setf (io-beg fil) 0))
		      (if (minusp (io-end fil)) (setf (io-end fil) 0)))
		    (clm-close-output fil)))
		(setf (aref *clm-current-open-files* i) nil))))))
  (setf clm-statistics nil)
  (setf clm-notehook nil)
  (setf *clm-interrupted* 0)
  (setf *current-output-file* nil)
  (setf *current-input-file* nil)
  (setf *reverb* nil))

(defun get-io-size (f)       (IO-siz f))
(defun get-io-hdr (f)        (IO-hdr f))
(defun clm-io-nam (n)        (IO-nam (aref *clm-current-open-files* n)))
(defun clm-io-dat-a (n)      (IO-dat-a (aref *clm-current-open-files* n)))
(defun clm-io-bufsiz (n)     (IO-bufsiz (aref *clm-current-open-files* n)))
(defun clm-io-dat-b (n)      (IO-dat-b (aref *clm-current-open-files* n)))
(defun clm-io-dat-c (n)      (IO-dat-c (aref *clm-current-open-files* n)))
(defun clm-io-dat-d (n)      (IO-dat-d (aref *clm-current-open-files* n)))
(defun clm-io-beg (n)        (IO-beg (aref *clm-current-open-files* n)))
(defun clm-io-end (n)        (IO-end (aref *clm-current-open-files* n)))
(defun clm-io-hdr (n)        (IO-hdr (aref *clm-current-open-files* n)))
(defun clm-io-hdr-end (n)    (IO-hdr-end (aref *clm-current-open-files* n)))
(defun clm-io-data-end (n)   (IO-data-end (aref *clm-current-open-files* n)))
(defun clm-io-dir (n)        (IO-dir (aref *clm-current-open-files* n)))
(defun clm-io-open-index (n) (IO-open-index (aref *clm-current-open-files* n)))
(defun clm-io-siz (n)        (IO-siz (aref *clm-current-open-files* n)))
(defun clm-mark-external-stream (n) (if n (setf (io-external n) t)))

(defun clm-get-run-time-file-index (fil chan)
  (declare (optimize (speed 3) (safety 0)))
  (if (and fil
	   (or (and (= chan 0) (IO-dat-a fil))
	       (and (= chan 1) (IO-dat-b fil))
	       (and (= chan 2) (IO-dat-c fil))
	       (and (= chan 3) (IO-dat-d fil))))
      (IO-open-index fil)
    -1))

(defun clm-io-output-bufsiz () (IO-bufsiz *current-output-file*))

(defun clm-reset-all-files (beg curend &optional (flush t))
  ;; flush current contents, move to start at beg, set data-end to curend
  (if *clm-current-open-files*
      (loop for i from 0 to *clm-current-max-file-index* do
	(let ((fil (aref *clm-current-open-files* i)))
	  (when fil
	    (if (and (not (IO-external fil)) 
		     (= out-f (IO-dir fil)))
		(setf (IO-data-end fil) (max curend (IO-data-end fil))))
	    (if flush (file-reset beg fil)))))))


(defun clm-flush-all-files (&optional curbeg curend)
  (if *clm-current-open-files*
      (loop for i from 0 to *clm-current-max-file-index* do
	(let ((fil (aref *clm-current-open-files* i)))
	  (when (and fil
		     (= out-f (IO-dir fil)))
	    (if (and curbeg curend (not (IO-external fil)))
		(progn
		  (setf (IO-data-end fil) (max curend (IO-data-end fil)))
		  (setf (IO-data-start fil) (min curbeg (IO-data-start fil))))
	      (flush-partial-snd fil)))))))


(defun clm-count-active-output-buffers ()
  (let ((bufs 0))
    (if *clm-current-open-files*
	(loop for i from 0 to *clm-current-max-file-index* do
	  (let ((fil (aref *clm-current-open-files* i)))
	    (if (and fil
		     (= out-f (IO-dir fil)))
		(incf bufs (if (quad fil) 4 (if (stereo fil) 2 1)))))))
    bufs))

(defun clm-collect-output-buffer-addresses (n &optional no-output-here)
  (if (or (null n) (zerop n) (minusp n))
      (if (not no-output-here)
	  (error "No output channels open!"))
    (let ((bufs (make-array n :element-type 'fixnum :initial-element 0))
	  (j 0))
      (if *clm-current-open-files*
	  (loop for i from 0 to *clm-current-max-file-index* do
	    (let ((fil (aref *clm-current-open-files* i)))
	      (when (and fil
			 (= out-f (IO-dir fil)))
		(setf (aref bufs j) (IO-dat-a fil))
		(if (stereo fil) (setf (aref bufs (1+ j)) (IO-dat-b fil)))
		(when (quad fil) 
		  (setf (aref bufs (+ j 1)) (IO-dat-b fil))
		  (setf (aref bufs (+ j 2)) (IO-dat-c fil))
		  (setf (aref bufs (+ j 3)) (IO-dat-d fil)))
		(incf j (if (quad fil) 4 (if (stereo fil) 2 1)))))))
      bufs)))


(defun get-beg-end (start-time duration)
  (declare (optimize (speed 3) (safety 0)))
  (values (floor (* start-time sampling-rate))
	  (floor (* (+ start-time duration) sampling-rate))))

(defun check-f-type (f) (if (not (io-p f)) (error "argument ~A must be a file pointer, not a ~(~A~)" f (type-of f))))
;;; this check is needed in kcl because otherwise we die with a bus error!
(defun clm-get-duration (f) (check-f-type f) (float (/ (- (get-IO-size f) 1) sampling-rate)))
(defun clm-get-samples (f) (check-f-type f) (get-IO-size f))
(defun clm-get-channels (f) (check-f-type f) (if (quad f) 4 (if (stereo f) 2 1)))
(defun clm-get-sampling-rate (f) (check-f-type f) (snd-header-srate (IO-hdr f)))
(defun clm-check-file (f) (check-f-type f) (if (not (IO-dat-a f)) (error "input file ~A appears to have been prematurely closed." (io-nam f))))

(defun clm-get-max-amp (f)
  (declare (optimize (speed 3) (safety 0)))
  (let ((maxampA 0)
	(maxampB (if (or (stereo f) (quad f)) 0 nil))
	(maxampC (if (quad f) 0 nil))
	(maxampD (if (quad f) 0 nil))
	(end (1- (clm-get-samples f))))
    (loop for i from 0 to end by file-buffer-size do
      (let ((samps (min (1- file-buffer-size) (- end i))))
	(file-reset i f)
	(setf maxampA (max maxampA (c-abs-max-array 0 samps (IO-dat-A f))))
	(if (or (stereo f) (quad f)) (setf maxampB (max maxampB (c-abs-max-array 0 samps (IO-dat-B f)))))
	(when (quad f)
	  (setf maxampC (max maxampC (c-abs-max-array 0 samps (IO-dat-C f))))
	  (setf maxampD (max maxampD (c-abs-max-array 0 samps (IO-dat-D f)))))))
    (values (fix-to-real maxampA)
	    (and maxampB (fix-to-real maxampB))
	    (and maxampC (fix-to-real maxampC))
	    (and maxampD (fix-to-real maxampD)))))

(defun clm-get-timed-max-amp (f)
  (declare (optimize (speed 3) (safety 0)))
  (let ((maxampA 0)
	(maxampB (if (or (stereo f) (quad f)) 0 nil))
	(maxampC (if (quad f) 0 nil))
	(maxampD (if (quad f) 0 nil))
	(timeA 0)
	(timeB 0)
	(timeC 0)
	(timeD 0)
	(end (1- (clm-get-samples f))))
    (loop for i from 0 to end by file-buffer-size do
      (let ((samps (min (1- file-buffer-size) (- end i))))
	(file-reset i f)
	(let ((curmax (c-timed-abs-max-array 0 samps (IO-dat-A f)))
	      (curtime (c-last-timed-max)))
	  (when (> curmax maxAmpA)
	    (setf maxampA curmax)
	    (setf timeA (+ curtime i))))
	(when (or (stereo f) (quad f)) 
	  (let ((curmax (c-timed-abs-max-array 0 samps (IO-dat-B f)))
		(curtime (c-last-timed-max)))
	    (when (> curmax maxAmpB)
	      (setf maxampB curmax)
	      (setf timeB (+ curtime i))))
	  (when (quad f)
	    (let ((curmax (c-timed-abs-max-array 0 samps (IO-dat-C f)))
		  (curtime (c-last-timed-max)))
	      (when (> curmax maxAmpC)
		(setf maxampC curmax)
		(setf timeC (+ curtime i))))
	    (let ((curmax (c-timed-abs-max-array 0 samps (IO-dat-D f)))
		  (curtime (c-last-timed-max)))
	      (when (> curmax maxAmpD)
		(setf maxampD curmax)
		(setf timeD (+ curtime i))))))))
    (values (fix-to-real maxampA)
	    (and maxampB (fix-to-real maxampB))
	    (and maxampC (fix-to-real maxampC))
	    (and maxampD (fix-to-real maxampD))
	    timeA timeB timeC timeD)))

(defun clm-get-timed-max-amp-mono (f)
  (declare (optimize (speed 3) (safety 0)))
  (if f
      (let ((maxampA 0)
	    (timeA 0)
	    (end (1- (clm-get-samples f))))
	(loop for i from 0 to end by file-buffer-size do
	  (let ((samps (min (1- file-buffer-size) (- end i))))
	    (file-reset i f)
	    (let ((curmax (c-timed-abs-max-array 0 samps (IO-dat-A f)))
		  (curtime (c-last-timed-max)))
	      (when (> curmax maxAmpA)
		(setf maxampA curmax)
		(setf timeA (+ curtime i))))))
	(values (fix-to-real maxampA) timeA))
    (values nil nil)))


(defvar clm-start-time nil)
(defvar clm-total-duration nil)
(defvar clm-last-begin-time nil)
(defvar clm-outfile-name nil)
(defvar clm-revfile-name nil)

(defun clm-print-statistics (stats) 
  (declare (optimize (speed 3) (safety 0)))
  (when stats
    (flet ((convert-samples-to-seconds (samp) (if samp (float (/ samp sampling-rate)) 0.0)))
      (let* ((total-time (float (/ (- (get-internal-real-time) clm-start-time) internal-time-units-per-second)))
	     (outf (clm-open-input clm-outfile-name))
	     (clm-last-end-time (get-IO-size outf))
	     (revf (and clm-revfile-name (clm-open-input clm-revfile-name))))
	(multiple-value-bind
	    (clm-max-outA-amp clm-max-outB-amp clm-max-outC-amp clm-max-outD-amp
	     clm-max-outA-amp-time clm-max-outB-amp-time clm-max-outC-amp-time clm-max-outD-amp-time)
	    (clm-get-timed-max-amp outf)
	  (clm-close-input outf)
	  (multiple-value-bind 
	      (clm-max-revA-amp clm-max-revA-time)
	      (clm-get-timed-max-amp-mono revf)
	    (format t "~A~A~A~A~A~A~A~A"
		    (format nil "~A: ~%  Duration: ~,4F, Last begin time: ~,4F~%" 
			    (namestring clm-outfile-name)
			    (convert-samples-to-seconds clm-last-end-time)
			    (convert-samples-to-seconds clm-last-begin-time))
		    (format nil "  Compute time: ~,3F, Compute ratios: ~,2F (~,2F)~%" 
			    total-time
			    (if (not (zerop clm-last-end-time))
				(/ total-time (convert-samples-to-seconds clm-last-end-time))
			      0.0)
			    (if (not (zerop clm-total-duration))
				(/ total-time (convert-samples-to-seconds clm-total-duration))
			      0.0))
		    (if (> total-time 3600)
			(let* ((days (floor total-time (* 24 60 60)))
			       (notdays (- total-time (* days 24 60 60)))
			       (hours (floor notdays (* 60 60)))
			       (nothours (- notdays (* hours 60 60)))
			       (minutes (floor nothours 60))
			       (seconds (- nothours (* minutes 60))))
			  (multiple-value-bind (second minute hour date month year day daylight-saving-p time-zone)
			      (get-decoded-time)
			    (declare (ignore daylight-saving-p time-zone second))
			    (format nil "    (~A~A~A~,3F seconds, finished~A on ~A ~D-~A-~D at ~D:~2,'0D)~%"
				    (if (plusp days) (format nil "~D day~P, " days days) "")
				    (if (plusp hours) (format nil "~D hour~P, " hours hours) "")
				    (if (plusp minutes) (format nil "~D minute~P, " minutes minutes) "")
				    seconds
				    (if (plusp days) " (Laus Deo)" "")
				    (day-name day) date (month-name month) (- year 1900) hour minute)))
		      "")
		    (format nil "  OutA max amp: ~,3F (near ~,3F secs)~%" 
			    clm-max-outA-amp
			    (convert-samples-to-seconds clm-max-outA-amp-time))
		    (if clm-max-outB-amp
			(format nil "  OutB max amp: ~,3F (near ~,3F secs)~%" 
				clm-max-outB-amp
				(convert-samples-to-seconds clm-max-outB-amp-time))
		      "")
		    (if clm-max-outC-amp
			(format nil "  OutC max amp: ~,3F (near ~,3F secs)~%" 
				clm-max-outC-amp 
				(convert-samples-to-seconds clm-max-outC-amp-time))
		      "")
		    (if clm-max-outD-amp
			(format nil "  OutD max amp: ~,3F (near ~,3F secs)~%" 
				clm-max-outD-amp
				(convert-samples-to-seconds clm-max-outD-amp-time))
		      "")
		    (if clm-max-revA-amp
			(format nil "  RevA max amp: ~,3F (near ~,3F secs)~%" 
				clm-max-revA-amp
				(convert-samples-to-seconds clm-max-revA-time))
		      ""))))))))

(defun clm-initialize-statistics (stats ofile &optional rfile) 
  (declare (optimize (speed 3) (safety 0)))
  (setf clm-statistics stats)
  (setf clm-start-time (get-internal-real-time))
  (setf clm-total-duration 0)
  (setf clm-last-begin-time 0)
  (setf clm-outfile-name ofile)
  (setf clm-revfile-name rfile))

(defun clm-initialize-notehook (hook) 
  (declare (optimize (speed 3) (safety 0)))
  (setf clm-notehook hook))

