;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-

(in-package :clm)

;;; continuation from code56.lisp (file got too big)

(defvar dsp-version-number 8)		;changes whenever non-backwards-compatible changes are made
					;1 on 10-Dec-90 for new QP monitor HC commands
					;2 on 31-Dec-90 for new step envelopes (pass-env-data has to be in sync with .ENV)
					;3 on 9-Feb-91 for system 2.0 Next QP support (need a safe global location for DRAM size)
					;4 on 12-Aug-91 -- added OUT-HC to monitor, changed envelope structures etc.
					;5 on 15-Mar-92 -- needed to provide escape to lisp from Run and external envelopes
					;6 on 22-May-92 for better external function communication
					;7 on 25-Sep-93 for merging various DSP monitors (preparatory to 486 work and so on)
					;8 on 3-Apr-94 moving everything in to the clm package
(defvar *dsp-version-date* "3-Apr-94")


;;; SET UP LOAD AND RUN SEQUENCE

(defmacro pass-zero (addr) 
  `(setf-xy-mem ,addr 0 0))

(defmacro pass-nil (addr) 
  `(setf-xy-mem ,addr ,<nil> 0))

(defun pass-real (nx addr)
  (declare (optimize (speed 3) (safety 0)))
  
  (let (
	#-Excl (x (float nx))
	#+Excl (x (if (excl::exceptional-floating-point-number-p nx) 0.0 (float nx)))
	;; this is ridiculous -- in excl, excl::*nan-single* can be placed in an uninitialized short-float array,
	;;  and that NAN is a number, is a float, but has no value in any numeric comparison operation, but
	;;  you cannot call floor on it!!  So without this trap, we die with an exponent overflow complaint.
	;;  If Lisp weren't chock-full of numeric stupidities, I'd call it a bug.
	)
    (if (< (abs x) smallest-dsp-number)
	(setf-xy-mem addr 0 0)
      (if (plusp x)
	  (if (> x biggest-dsp-number)
	      (setf-xy-mem addr biggest-dsp-number 0)
	    (multiple-value-bind (ix fx) (floor x)
	      (setf-xy-mem addr ix (floor (scale-float fx 24)))))
	(if (< x (- biggest-dsp-number))
	    (setf-xy-mem addr #x800001 0) ;(logand (- biggest-dsp-number) #xFFFFFF)
	  (multiple-value-bind (ix fx) (floor x)
	    (setf-xy-mem addr (logand ix #xFFFFFF) (floor (scale-float fx 24)))))))))

(defun pass-fraction (x addr)
  (if (numberp x)
      (setf-xy-mem addr 0 (mac-fraction x))
    (if x
	(pass-zero addr)
      (pass-nil addr))))

(defun pass-long-int (i addr)		;long integer stored X-high Y-low 
  (let ((ri (floor i)))
    (setf-xy-mem addr (ash ri -24) (logand ri #xFFFFFF))))


(defun pass-user-real (x addr)
  (if (numberp x)
      (pass-real x addr)
    (if x
	(pass-zero addr)		;i.e. t=0 nil=#x10000 on chip
      (pass-nil addr))))


(defvar max-external-envelopes 256)

(defun current-max-env-size ()
  (let ((maxptr (max ex-ptr ey-ptr)))
    (if (< maxptr 1024) 255
      (if (< maxptr 2048) 31
	7))))

(defun pass-env-data (e cur-val-loc)
  (declare (optimize (speed 3) (safety 0)))
  (let* ((end (envelope-end e))
	 (expenv (eq (envelope-type e) :exp))
	 (restartable (envelope-restart e))
	 (new-data (cdr (fix-envelope (envelope-data e) end)))
					;envelope-data is a list of (pass rate) values.
					;we skip the first pass value, since it is useless
					;in this context.
	 (data-len (length new-data))
	 (max-data-len (current-max-env-size))
	 (data-size (max 2 (+ (min max-data-len data-len) 1)))
	 (env-size (+ 1 data-size (if (not expenv) 0 5))) ;see also offsets in dm-env
	 (restart-base (if restartable (get-L-mem 5) 0))
	 (data-base (get-L-mem env-size))
	 (true-data-base (if (not expenv) data-base (+ data-base 5)))
	 (dummy-pointer-loc (+ cur-val-loc 1)))

    (setf-x-mem dummy-pointer-loc true-data-base) ;patch pointer to dynamically allocated array
    (setf-y-mem dummy-pointer-loc (+ data-size (ash true-data-base 8)))
    (setf-x-mem (+ cur-val-loc 3) restart-base)

    (when restartable
      (setf-x-mem restart-base true-data-base)
      (setf-y-mem restart-base (floor data-size 2))
      (setf-x-mem (+ restart-base 1) (if expenv (+ data-base 3) 0))
      (setf-y-mem (+ restart-base 1) 0)
      (if expenv
	  (pass-real (envelope-power e) (+ restart-base 2))
	(pass-real (envelope-current-value e) (+ restart-base 2)))
      (pass-long-int (envelope-pass e) (+ restart-base 3))
      (if expenv
	  (pass-real (envelope-current-value e) (+ restart-base 4))))

    (if (or (not expenv)
	    (= (envelope-base e) 1.0)
	    (= (envelope-base e) 0.0))
	(progn
	  (if (or (not (envelope-base e))
		  (not (zerop (envelope-base e))))
	      (progn
		(pass-real (envelope-current-value e) cur-val-loc)
		(pass-real 1.0 (+ cur-val-loc 2)))
	    (progn
	      (pass-zero cur-val-loc)
	      (pass-real (envelope-base e) (+ cur-val-loc 2)))))
      (progn
	(setf-x-mem (+ cur-val-loc 2) data-base)
	(setf-y-mem (+ cur-val-loc 2) 0)
	(pass-real (envelope-power e) cur-val-loc)
	(pass-real (envelope-base e) data-base)
	(pass-real (envelope-scaler e) (+ data-base 1))
	(pass-real (envelope-offset e) (+ data-base 2))
	(pass-real (envelope-current-value e) (+ data-base 3))
	(pass-real (log (abs (envelope-base e))) (+ data-base 4))))
    
    (if (not new-data)
	(progn
	  (pass-long-int (+ end 1) true-data-base)
	  (pass-zero (+ true-data-base 1)))
      (if (>= max-data-len data-len)
	  (loop for dat on new-data by #'cddr and i from true-data-base by 2 do 
	    ;;here we see pass then rate (or nil at end)
	    (if (cdr dat)
		(pass-long-int (cadr dat) i)
	      (progn
		(pass-long-int (+ end 1) i) ; need block-end pointer 
		(pass-long-int -1 (+ i 2))))
	    (if (zerop (car dat))
		(pass-zero (+ i 1))
	      (pass-real (car dat) (+ i 1))))
	(progn
	  (if (null external-envelopes)
	      (setf external-envelopes (make-array max-external-envelopes :initial-element nil)))
	  (let ((index (position-if #'null external-envelopes))
		(segs (floor data-size 2))
		(last-addr (1- (+ data-base env-size))))
	    ;; pass data-size/2 segments
	    ;; then (- index) as X side int (segs in data-size in Y)
	    ;; .env in lib56 will see -1 -1 as true env end (stick point)
	    ;;                        -ind siz as ext env signal
	    (loop for k from 1 to segs and i from true-data-base by 2 do
	      (let* ((rate (pop new-data))
		     (pctr (or (pop new-data) (1+ end))))
		(pass-long-int pctr i)
		(pass-real rate (1+ i))))
	    (setf (aref external-envelopes index) (list end true-data-base new-data))
	    (setf-x-mem last-addr (logand (- (+ index 1)) #xFFFFFF))
	    (setf-y-mem last-addr segs)))))))

(defun pass-envelope-data (e cur-val-loc)
  (if (and e (not (eq e t)))
      (pass-env-data e cur-val-loc)
    (progn
      (pass-nil cur-val-loc)
      (pass-zero (+ cur-val-loc 1)))))

(defun pass-osc (s addr)
  (if (osc-p s)
      (progn
	(pass-real (- (osc-phase s) (osc-freq s)) addr)
	;; oscil in lib56 is ahead one pass in a sense -- this almost never matters.
	(pass-real (osc-freq s) (+ addr 1))
	(pass-zero (+ addr 2)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to oscil" s)) ;I need a struct-p function
      (pass-nil addr))))

(defun pass-sw (s addr)
  (if (sw-p s)
      (progn
	(pass-real (sw-current-value s) addr)
	(pass-real (sw-phase s) (+ addr 1))
	(pass-real (sw-freq s) (+ addr 2))
	(pass-zero (+ addr 3))
	(if (sw-base s)
	    (pass-real (sw-base s) (+ addr 4))
	  (pass-zero (+ addr 4))))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to square-wave (or friend)" s))
      (pass-nil addr))))

(defun pass-noi (s addr)
  ;; need to pass ran-op too for run-time type decision		
  (if (noi-p s)
      (progn
	(pass-real (noi-phase s) addr)
	(pass-real (noi-freq s) (+ addr 1))
	(pass-zero (+ addr 2))

	(if (eq (noi-ran-op s) 'Sambox-random)
	    (progn
	      (setf-x-mem (+ addr 6) 1)
	      (setf-y-mem (+ addr 6) (ash #o2336545 2)))
	  (progn
	    (setf-x-mem (+ addr 6) 0)
	    (setf-y-mem (+ addr 6) 0)))

	(pass-real (noi-output s) (+ addr 3))
	(pass-zero (+ addr 5))
	(pass-real (noi-base s) (+ addr 4)))
     (if (numberp s)
	 (pass-real s addr)
       (progn
	 (if (and s (not (constantp s))) (warn "~A passed to randh" s))
	 (pass-nil addr)))))
  
(defun pass-randi (s addr)
  (if (noi-p s)
      (progn
	(pass-noi s addr)
	(pass-real (noi-incr s) (+ addr 5))
	(setf-x-mem (+ addr 6) 0)
	(setf-y-mem (+ addr 6) (make-fraction (/ (noi-freq s) two-pi))))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to randi" s))
      (pass-nil addr))))

(defun pass-tbl (s addr)		;table-lookup
  (if (tbl-p s)
      (pass-table s addr)
    (pass-nil addr)))

(defun pass-table (s addr)		;table-lookup, not table
#+Excl  (if (not (eq 'single-float (array-element-type (tbl-table s)))) 
           (error "c cannot handle this array: ~A -- :element-type has to be 'short-float." (tbl-table s)))
#+Kcl   (if (not (eq 'short-float (array-element-type (tbl-table s)))) 
           (error "c cannot handle this array: ~A -- :element-type has to be 'short-float." (tbl-table s)))
  (multiple-value-bind
      (buffer-start x-side)
      (get-a-block-of-x-or-y-memory (tbl-table-size s))
    (if (plusp buffer-start)
	(progn  
	  (pass-real (tbl-phase s) addr)
	  (pass-real (tbl-freq s) (+ addr 1))
	  (pass-zero (+ addr 2))
	  (setf-x-mem (+ addr 3) (tbl-table-size s))
	  (setf-y-mem (+ addr 3) buffer-start)
	  (pass-real (tbl-internal-mag s) (+ addr 4))
	  (setf-y-mem (+ addr 5) (if x-side 0 1))
	  (c-load-fractional-array (tbl-table s)
				   (tbl-table-size s) 
				   (if (or (zerop external-mem-size) (< buffer-start ext-X-offset))
				       buffer-start 
				     (+ (- buffer-start ext-X-offset)
					(if x-side 0 External-Y-from-X-offset)))
				   (if (or (zerop external-mem-size) (< buffer-start ext-X-offset))
				       (if x-side internal-x-memory internal-y-memory)
				     external-memory)))
      (error "cannot accomodate this table on chip: ~S" s))))

(defun pass-user-table (s addr)		;i.e. a table, not a table-lookup
  (if s
      (let ((size (array-dimension s 0)))
#+Excl
      (if (not (eq 'single-float (array-element-type s))) 
	    (error "c cannot handle this array: ~A -- :element-type has to be short-float." s))
#+Kcl (if (not (eq 'short-float (array-element-type s))) 
	    (error "c cannot handle this array: ~A -- :element-type has to be short-float." s))
	(multiple-value-bind 
	    (buffer-start x-side)
	    (get-a-block-of-x-or-y-memory size)
	  (if (plusp buffer-start)
	      (progn
		(setf-x-mem addr buffer-start)
		(setf-y-mem addr size)
		(setf-x-mem (+ addr 1) (1- buffer-start))
		(setf-y-mem (+ addr 1) (if x-side 0 1))
		(c-load-fractional-array s size 
					 (if (or (zerop external-mem-size) (< buffer-start ext-X-offset))
					     buffer-start 
					   (+ (- buffer-start ext-X-offset)
					      (if x-side 0 External-Y-from-X-offset)))
					 (if (or (zerop external-mem-size) (< buffer-start ext-X-offset))
					     (if x-side internal-x-memory internal-y-memory)
					   external-memory)))
	    (error "cannot accomodate this table on chip: ~S" s))))
    (progn
      (pass-nil addr)
      (pass-zero (+ addr 1)))))

(defun pass-blk (s addr)
  (if (rblk-p s)
      (progn
	(pass-real (rblk-loc s) addr)
	(pass-real (rblk-ctr s) (+ addr 1))
	(pass-user-table (rblk-buf s) (+ addr 2)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to run-block" s))
      (pass-nil addr)
      (pass-zero (+ addr 1)))))

(defun pass-wt (s addr)
  (if (wt-p s)
      (let* ((old-freq (float (wt-freq s)))
	     (old-ctr (float (/ sampling-rate old-freq))))
					;	(setf (rblk-ctr (wt-b s)) old-ctr) ; 25-Feb-92
	(pass-blk (wt-b s) addr)
	(pass-user-table (wt-wave s) (+ addr 4))
	(pass-real (wt-phase s) (+ addr 6))
	(pass-real (wt-freq s) (+ addr 7))
	(pass-zero (+ addr 8))		;fm
	(pass-real old-freq (+ addr 9))
	(pass-real old-ctr (+ addr 10))
	(pass-real (wt-internal-mag s) (+ addr 11))
	(pass-real sampling-rate (+ addr 12)))
    (progn 
      (if (and s (not (constantp s))) (warn "~A passed to wave-train" s))
      (pass-nil addr)
      (loop for i from (+ addr 1) to (+ addr 12) do (pass-zero i)))))

(defun pass-nil-dly (base)
  (progn
     (pass-zero base)
     (setf-x-mem (+ base 1) 0)
     (setf-y-mem (+ base 1) 3)
     (pass-zero (+ base 2))))

(defvar DRAM-loc 0)			;for delay buffers in QP DRAM
(defvar DRAM-list nil)

(defun pass-dly (s base)
  (if (dly-p s)
      (pass-delay s base)
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to delay" s))
      (pass-nil-dly base))))

(defun pass-delay (s base)
  (multiple-value-bind
      (buffer-start x-side)
      (if (or (<= dsps 1) (< (dly-size s) 2048))
	  (get-a-block-of-x-or-y-memory (dly-size s) nil 1024) 
					; "nil" here means "don't complain if buffer too big"
					; the 1024 means leave at least that much space for others
	(values -1 nil))
    (if (plusp buffer-start)
	(progn
	  (setf-x-mem base buffer-start)
	  (setf-y-mem base (+ buffer-start (dly-size s) -1))
	  (setf-x-mem (+ base 1) buffer-start) ;i.e. current pointer starts at base
	  (setf-y-mem (+ base 1) (+ (if x-side 0 1) (if (dly-ext-id s) 8 0)))
	  (if (dly-ext-id s)
	      (loop for i from 0 below (dly-size s) and j from buffer-start do
		(if x-side
		    (setf-x-mem j (ash (caref (dly-pline s) i) 8))
		  (setf-y-mem j (ash (caref (dly-pline s) i) 8)))))
	  (setf (dly-ext-id s) nil)	;run-time indication that this buffer lives on chip
	  (setf-x-mem (+ base 2) (dly-size s)))
      (progn				;set up for either external (68000) memory or QP DRAM
	(setf-x-mem base DRAM-loc)
	(setf-y-mem base (+ DRAM-loc (dly-size s) -1))
	(setf-x-mem (+ base 1) DRAM-loc)
	(setf-y-mem (+ base 1) (+ 2 (if (dly-ext-id s) 8 0)))
	(setf (dly-ext-id s) (dly-id s))
	;;external or DRAM buffer (assume external for now)
	(push (+ base 1) DRAM-list)
	(incf DRAM-loc (dly-size s))
	(setf-y-mem (+ base 2) (dly-id s))
	(setf-x-mem (+ base 2) (dly-size s))))))
  
(defun pass-table-header (addr buffer-start size x-side)
  (setf-x-mem addr buffer-start)
  (setf-y-mem addr size)
  (setf-x-mem (+ addr 1) (1- buffer-start))
  (setf-y-mem (+ addr 1) (if x-side 0 1)))

(defun pass-filter (fl base)
  (if (flt-p fl)
					;R2->flt struct: type   (end c)  (start d)  so
					;                order  (end p)  (end a)  (end d) (d has extra end word of 0)
      (let* ((size (1+ (flt-m fl)))
	     (c-base (if (= (flt-typ fl) ladder-form)
			 (get-x-memory size)
		       0))
	     (p-base (get-y-memory size))
	     (a-base (get-y-memory size))
	     (d-base (get-x-memory (1+ size))))
	(setf-x-mem base (if (= (flt-typ fl) direct-form) 0
			    (if (= (flt-typ fl) lattice-form) 1
			      (if (= (flt-typ fl) ladder-form) 2
				(error "invalid filter type: ~A" (flt-typ fl))))))
	(setf-y-mem base (flt-m fl))
	(if (/= 0 c-base)
	    (progn
	      (setf-x-mem (+ base 1) (+ c-base (flt-m fl)))
	      (loop for i from 0 below size and j from c-base do
		(setf-x-mem j (make-fraction-1 (aref (flt-c fl) i)))))
	  (setf-x-mem (+ base 1) 0))
	(setf-y-mem (+ base 1) (+ p-base (flt-m fl)))
	(loop for i from 0 below size and j from p-base do
	  (setf-y-mem j (make-fraction-1 (aref (flt-b fl) i))))
	(setf-x-mem (+ base 2) d-base)
	(setf-y-mem (+ base 2) (+ a-base (flt-m fl)))
	(loop for i from 0 below size and j from a-base do
	  (setf-y-mem j (make-fraction-1 (aref (flt-a fl) i))))
	(setf-x-mem (+ base 3) (make-fraction-1 (flt-so fl)))
	(setf-y-mem (+ base 3) (+ d-base (flt-m fl)))
	(loop for i from 0 to size and j from d-base do 
	  (setf-x-mem j 0))
	(pass-table-header (+ base 4) a-base size nil)
	(pass-table-header (+ base 6) p-base size nil) ;p=b
	(pass-table-header (+ base 8) c-base (if (/= 0 c-base) size 0) t)
	(pass-table-header (+ base 10) d-base (1+ size) t))
    (progn
      (if (and fl (not (constantp fl))) (warn "~A passed to filter" fl))
      (pass-nil base)
      (loop for i from (+ base 1) to (+ base 3) do
	(pass-zero i)))))
  
(defun check-for-overflow (x str &optional two) 
  (if (> (abs x) 1.0) (warn "~A: (abs ~,2F) > ~A -- too big for filter coefficient" str (if two (* 2 x) x) (if two "2.0" "1.0")))
  x)

(defun pass-cmbflt (c base)
  (if (cmbflt-p c)
      (progn
	(pass-dly (cmbflt-dly-unit c) base)
	(setf-x-mem (+ base 3) 0)
	(setf-y-mem (+ base 3) (make-fraction (check-for-overflow (cmbflt-scaler c) "comb scaler"))))
    (progn
      (if (and c (not (constantp c))) (warn "~A passed to comb or notch" c))
      (pass-nil-dly base))))

(defun pass-allpassflt (c base)
  (if (allpassflt-p c)
      (progn
	(pass-dly (allpassflt-dly-unit c) base)
	(setf-y-mem (+ base 3) (make-fraction (check-for-overflow (allpassflt-feedback c) "all-pass feedback")))
	(setf-x-mem (+ base 3) (make-fraction (check-for-overflow (allpassflt-feedforward c) "all-pass feedforward"))))
    (progn
      (if (and c (not (constantp c))) (warn "~A passed to all-pass" c))
      (pass-nil-dly base))))

(defun pass-one-pole (s loc1)
  (if (smpflt-p s)
      (progn
	(setf-x-mem loc1 (make-fraction (check-for-overflow (* (smpflt-a0 s) .5) "one-pole a0" t)))
	(setf-y-mem loc1 (make-fraction (check-for-overflow (* (smpflt-b1 s) .5) "one-pole b1" t)))
	(pass-zero (+ loc1 1)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to one-pole" s))
      (pass-nil loc1))))

(defun pass-one-zero (s loc1)
  (if (smpflt-p s)
      (progn
	(setf-x-mem loc1 (make-fraction (check-for-overflow (* (smpflt-a0 s) .5) "one-zero a0" t)))
	(setf-y-mem loc1 0)
	(setf-x-mem (+ loc1 1) (make-fraction (check-for-overflow (* (smpflt-a1 s) .5) "one-zero a1" t))))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to one-zero" s))
      (pass-nil loc1))))

(defun pass-two-pole (s loc1)
  (if (smpflt-p s)
      (progn
	(setf-x-mem loc1 (make-fraction (check-for-overflow (* (smpflt-a0 s) .5) "two-pole a0" t)))
	(setf-y-mem loc1 (make-fraction (check-for-overflow (* (smpflt-b1 s) .5) "two-pole b1" t)))
	(setf-x-mem (+ loc1 1) 0)
	(setf-y-mem (+ loc1 1) (make-fraction (check-for-overflow (* (smpflt-b2 s) .5) "two-pole b2" t)))
	(pass-zero (+ loc1 2)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to two-pole" s))
      (pass-nil loc1))))

(defun pass-two-zero (s loc1)
  (if (smpflt-p s)
      (progn
	(setf-x-mem loc1 (make-fraction (check-for-overflow (* (smpflt-a0 s) .5) "two-zero a0" t)))
	(setf-y-mem loc1 0)
	(setf-x-mem (+ loc1 1) (make-fraction (check-for-overflow (* (smpflt-a1 s) .5) "two-zero a1" t)))
	(setf-y-mem (+ loc1 1) 0)
	(setf-x-mem (+ loc1 2) (make-fraction (check-for-overflow (* (smpflt-a2 s) .5) "two-zero a2" t)))
	(setf-y-mem (+ loc1 2) 0))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to two-zero" s))
      (pass-nil loc1))))

(defun pass-frmnt (s loc1)
  (if (frmnt-p s)
      (progn
	(pass-two-zero (frmnt-tz s) (+ loc1 1))
	(pass-two-pole (frmnt-tp s) (+ loc1 4))
	(pass-real (frmnt-G s) loc1))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to formnt" s))
      (pass-nil loc1))))

(defun pass-smp (s loc1)
  (if (smp-p s)
      (progn
	(setf-x-mem loc1 (make-fraction (smp-lst s)))
	(setf-y-mem loc1 (make-fraction (smp-nxt s)))
	(pass-zero (+ loc1 1))
	(pass-real (smp-sr s) (+ loc1 2))
	(pass-long-int (smp-i s) (+ loc1 3)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to resample" s))
      (pass-nil loc1))))

(defun pass-rdin (s loc1)
  (if (rdin-p s)
      (progn
	(pass-long-int (rdin-i s) loc1)
	(pass-long-int (rdin-inc s) (+ loc1 1))) ;this order assumed in <read-position> and others
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to readin" s))
      (pass-nil loc1))))

(defun pass-ws (s loc1)
  (if (ws-p s)
      (progn
	(pass-user-table (ws-tab s) loc1)
	(setf-x-mem (+ loc1 2) (floor (ws-offset s)))
	(pass-osc (ws-os s) (+ loc1 3)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to waveshape" s))
      (pass-nil loc1))))

(defun pass-zdly (s loc1)
  (if (zdly-p s)
      (progn
	(pass-zero (+ loc1 3))
	(pass-real (float (zdly-phase s)) (+ loc1 4))
	(pass-real (float (dly-size (zdly-del s))) (+ loc1 5))
	(pass-dly (zdly-del s) loc1)
	(if (dly-ext-id (zdly-del s))	;possible trouble -- save back pointer to zdly struct
	    (remember-zdelay s)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to zdelay" s))
      (pass-nil-dly loc1))))		;needed to avoid trying to clear unallocated memory

(defun pass-cosp (s loc1)
  (if (cosp-p s)
      (progn
	(pass-real (cosp-phase s) loc1)
	(pass-real (cosp-freq s) (+ loc1 1))
	(pass-zero (+ loc1 2))
	(setf-x-mem (+ loc1 3) (make-fraction (cosp-scaler s)))
	(pass-real (+ (cosp-cosines s) .5) (+ loc1 4)))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to sum-of-cosines" s))
      (pass-nil loc1))))

(defun pass-locs (l loc1)
  (if (locs-p l)
      (progn
	(setf-x-mem loc1 (make-fraction (check-for-overflow (locs-ascl l) "locsig ascl")))
	(setf-y-mem loc1 (make-fraction (check-for-overflow (locs-bscl l) "locsig bscl")))
	(if (locs-cscl l) (setf-x-mem (+ loc1 2) (make-fraction (check-for-overflow (locs-cscl l) "locsig cscl"))))
	(if (locs-dscl l) (setf-y-mem (+ loc1 2) (make-fraction (check-for-overflow (locs-dscl l) "locsig dscl"))))
	(setf-x-mem (+ loc1 1) (make-fraction (check-for-overflow (locs-rscl l) "locsig rscl")))
	(setf-y-mem (+ loc1 1) (+ (if (or (stereo) (quad)) 1 0) 
				  (if (locs-revname l) 2 0)
				  (if (quad) 4 0))))
    (progn
      (if (and l (not (constantp l))) (warn "~A passed to locsig" l))
      (pass-nil loc1))))

(defun pass-spd (e addr)
  (if (spd-p e)
      (multiple-value-bind
	  (buffer-start x-side)
	  (get-a-block-of-x-or-y-memory (spd-len e))
#+Excl	(if (not (eq 'single-float (array-element-type (spd-b e)))) (error "c cannot handle this array: ~A" (spd-b e)))
#+Kcl	(if (not (eq 'short-float (array-element-type (spd-b e)))) (error "c cannot handle this array: ~A" (spd-b e)))
	(if (plusp buffer-start)
	    (progn  
	      (setf-x-mem addr 0)	                  ;cur-val
	      (setf-y-mem addr (if x-side 0 1))           ;        : x-or-y
	      (setf-x-mem (+ addr 1) 0)                   ;ctr
	      (setf-y-mem (+ addr 1) buffer-start)        ;        : ptr-to-buf
	      (setf-x-mem (+ addr 2) (spd-cur-out e))     ;cur-out
	      (setf-y-mem (+ addr 2) (1- (spd-len e)))    ;        : (len - 1)
	      (setf-x-mem (+ addr 3) (floor (- (spd-len e)
					       (* 2 (spd-rmp e)))))
	      (setf-y-mem (+ addr 3) (spd-rmp e))         ;steady  : rmp
	      (setf-x-mem (+ addr 4) (make-fraction (/ (spd-amp e) (spd-rmp e))))
	      (setf-y-mem (+ addr 4) 0)                   ;incr    : temp (for amp in env)
	      (pass-long-int (rdin-i (spd-rd e)) (+ addr 5))
	      (pass-long-int (rdin-inc (spd-rd e)) (+ addr 6))
	      ;; 5 words for rdin structure here
	      (setf-x-mem (+ addr 12) (spd-out-spd e))    ;out-spd
	      (setf-y-mem (+ addr 12) (spd-s50 e))        ;        : s50
	      (pass-long-int (spd-cur-in e) (+ addr 13))  ;cur-in
	      (setf-x-mem (+ addr 14) (spd-in-spd e))     ;in-spd
	      (setf-y-mem (+ addr 14) (spd-s20 e))        ;        : s20
	      (c-load-fractional-array (spd-b e)
				       (spd-len e)
				       (if (or (zerop external-mem-size) (< buffer-start ext-X-offset))
					   buffer-start 
					 (+ (- buffer-start ext-X-offset)
					    (if x-side 0 External-Y-from-X-offset)))
				       (if (or (zerop external-mem-size) (< buffer-start ext-X-offset))
					   (if x-side internal-x-memory internal-y-memory)
					 external-memory)))
	  (error "cannot accomodate this table on chip: ~S" e)))
    (progn
      (if (and e (not (constantp e))) (warn "~A passed to expand" e))
      (pass-nil addr))))


(defvar clm-src-samples-per-zero-crossing 5)
(defvar clm-src-width 5)

(defun fill-sinc-table (&optional (width clm-src-width))
  (when (or (zerop width)
	    (/= width clm-src-width))
    (setf clm-src-width (if (zerop width) 5 width)))
  (let* ((size (* clm-src-width clm-src-samples-per-zero-crossing))
	 (sinc-table (make-table size))
	 (win-freq (/ one-pi size))
	 (sinc-freq (/ 1.0 clm-src-samples-per-zero-crossing)))
    (setf (aref sinc-table 0) 1.0)
    (loop for i from 1 below size do
      (setf (aref sinc-table i) 
	(coerce (* (+ 0.5 (* 0.5 (cos (* i win-freq)))) (sinc (* i sinc-freq))) 
		'short-float)))
    sinc-table))
      
(defun pass-sr (s addr)
  (if (sr-p s)
      (let ((sinc-table (fill-sinc-table (sr-width s))))
	(setf-x-mem addr (sr-left s))
	(setf-y-mem addr (sr-right s))
	(setf-x-mem (1+ addr) (sr-width s))
	(setf-y-mem (1+ addr) (max 1 clm-src-samples-per-zero-crossing))
	(pass-real (sr-x s) (+ addr 2))
	(pass-real (sr-incr s) (+ addr 3))

	;; pass rdin, x incr width filt if needed, data, and sinc table, also room for left, right
	;;      (7)   (1) (1) (.5) (2+table)       (2+table) (2+table)                (.5) (.5)
	;; = 17 words of static space + 3 tables

	(pass-user-table (sr-data s) (+ addr 4))
	(pass-rdin (sr-rd s) (+ addr 6)) ;assumed 6 in <read-position> and others
	(pass-user-table (sr-filt s) (+ addr 13))
	(pass-user-table sinc-table (+ addr 15))
	(if (sr-filt s)
	    (setf-x-mem (+ addr 17) 
			(make-fraction (float (/ 1.0 (1- (array-total-size (sr-filt s)))))))
	  (pass-nil (+ addr 17))))
    (progn
      (if (and s (not (constantp s))) (warn "~A passed to src" s))
      (pass-nil addr)
      (pass-nil (+ addr 6)))))
  
(defun pass-fftflt (ff addr)
  (if (fftflt-p ff)
      (let ((buffer-start (get-L-mem (fftflt-siz ff))))
	(if (not (plusp buffer-start)) (error "fft data arrays are too big to fit in dsp memory"))
	(pass-blk (fftflt-b ff) addr)
	(setf-x-mem (+ addr 4) (fftflt-siz ff))
	(setf-y-mem (+ addr 4) (fftflt-hop ff))
	(setf-x-mem (+ addr 5) (fftflt-half-siz ff))
	(setf-y-mem (+ addr 5) (floor (fftflt-siz ff) 4))
	(setf-x-mem (+ addr 6) buffer-start)
	(setf-y-mem (+ addr 6) (fftflt-siz ff))
	(setf-x-mem (+ addr 7) (1- buffer-start))
	(setf-y-mem (+ addr 7) 0)	;x-side normally (here we fake it)
	(setf-x-mem (+ addr 10) 0)	;in case file=nil for white noise input (this is a horrible kludge for sansy's benefit)
	(setf-y-mem (+ addr 10) 0)
	(pass-rdin (fftflt-rd ff) (+ addr 8)) ;assumed 8 in <read-position> and others
	(if (fftflt-env ff)
	    (progn
	      (pass-user-table (fftflt-env ff) (+ addr 15))
	      (setf-x-mem (+ addr 17) 0))
	  (setf-x-mem (+ addr 17) 2)) ;i.e. convolution flag on
	(setf-y-mem (+ addr 17) (make-fraction (float (/ 1.0 (fftflt-siz ff))))))
    (progn
      (if (and ff (not (constantp ff))) (warn "~A passed to fft-filter" ff))
      (pass-nil addr))))

(defun pass-fft-data (ff addr)
  (if (fft-data-p ff)
      (let ((buffer-start (get-L-mem (fft-data-size ff))))
	(if (not (plusp buffer-start)) (error "fft data arrays are too big to fit in dsp memory"))
	;; now make the two arrays look like normal dsp tables, real = X side, etc.
	;; layout will be addr: real-table [2] imaginary-table (structs) [2], so 4 words needed
	;; tables are laid-out as start:size unused:x-or-y[1]
	;; so <fft> needs to set R3 to X addr (can assume x-or-y is informational)
	(pass-table-header addr buffer-start (fft-data-size ff) t)
	(pass-table-header (+ addr 2) buffer-start (fft-data-size ff) nil))

;;; may need to load the tables here

    (progn
      (if (and ff (not (constantp ff))) (warn "~A passed to fft-data" ff))
      (pass-nil addr))))

(defun pass-conv (ff addr)
  (if (conv-p ff)			;fftflt-p??? (11-Mar-93)
      (if (conv-filtr ff)
	  (let* ((size (fftflt-siz (conv-fftf ff)))
		 (scl 1.0)
		 (sclint 1)
		 (filt-buf-start (get-L-mem size)))
	    (if (not (plusp filt-buf-start)) (error "convolve filter fft arrays are too big to fit in dsp memory"))
	    (pass-fftflt (conv-fftf ff) addr)
	    (loop for i from 0 below size do
	      (setf scl (max scl (abs (aref (conv-filtr ff) i)) (abs (aref (conv-filti ff) i)))))
	    (setf sclint (ceiling scl))
	    (setf scl (/ 1.0 (float sclint)))
	    (loop for i from 0 below size do
	      (setf (aref (conv-filtr ff) i) (* scl (aref (conv-filtr ff) i)))
	      (setf (aref (conv-filti ff) i) (* scl (aref (conv-filti ff) i))))
	    (pass-table-header (+ addr 15) filt-buf-start size nil)
	    (setf-y-mem (+ addr 16) sclint)
	    (c-load-fractional-array (conv-filtr ff) 
				     size 
				     (if (< filt-buf-start ext-X-offset) filt-buf-start (- filt-buf-start ext-X-offset))
				     (if (or (zerop external-mem-size) (< filt-buf-start ext-X-offset)) internal-x-memory external-memory))
	    (c-load-fractional-array (conv-filti ff) 
				     size 
				     (if (or (zerop external-mem-size) (< filt-buf-start ext-X-offset))
					 filt-buf-start 
				       (+ External-Y-from-X-offset (- filt-buf-start ext-X-offset)))
				     (if (or (zerop external-mem-size) (< filt-buf-start ext-X-offset))
					 internal-y-memory 
				       external-memory)))
	(progn
	  (setf-x-mem (+ addr 17) 4)
	  (pass-rdin (fftflt-rd (conv-fftf ff)) (+ addr 8))))
    (progn
      (if (and ff (not (constantp ff))) (warn "~A passed to convolve" ff))
      (pass-nil addr))))


(defun pass-IO (loc1 bufstart bufend x-side)
  (setf-x-mem loc1 (+ ext-L-offset bufstart))
  (setf-y-mem loc1 (+ ext-L-offset bufend))
  (setf-x-mem (+ loc1 1) (+ ext-L-offset bufstart))
  (setf-y-mem (+ loc1 1) (if x-side 0 1))
  (pass-zero (+ loc1 2)))

(defun pass-insig-IO (loc ptr siz x-side chn) 
  (let ((index (logand (ash chn -16) #xFF)))
    (pass-long-int (clm-io-siz index) loc)
    (pass-long-int -1 (+ loc 1))
    (pass-long-int 0 (+ loc 2))
    (setf-x-mem (+ loc 3) (+ ext-L-offset ptr))
    (setf-y-mem (+ loc 3) (+ chn (if x-side 0 1)))
    (setf-x-mem (+ loc 4) 0)	
    (setf-y-mem (+ loc 4) siz)))

(defvar biggest-structure 18)

(defun structure-size_56 (typ)
  (case typ
    ((integer real fraction long-int) 1)
    (tbl 6)
    (ws 6)
    (cmbflt 4)
    (allpassflt 4)
    (dly 3)
    (zdly 6)
    (flt 12)
    (flt-one-pole 2)
    (flt-one-zero 2)
    (flt-two-pole 3)
    (flt-two-zero 3)
    (frmnt 7)
    (noi 7)
    (randi 7)
    (sw 5)
    (smp 9)
    (osc 3)
    (rdin 7)
    (spd 15)
    (sr 18)
    ((conv fftflt) 18)
    (fft-data 4)
    (cosp 5)
    (locs 3)
    (array 2)
    ((table x-table y-table) 2)
    (rblk 4)
    (wt 13)
    (envelope 4)			;current-value, pointers, 0 or exp pointers, restart pointers
    (smpflt 1)				;actually a temporarily stored address
    (t (or (user-struct-size_56 typ)
	   (error "odd type: ~A" typ)))))

(defun fpass-var (addr var type)
  (case type
    (integer      (setf-x-mem addr (if (numberp var) var 0)))
    (long-int     (pass-long-int var addr))
    (real         (pass-user-real var addr))
    (fraction     (pass-fraction var addr))
    (tbl  	  (pass-tbl var addr))
    (dly	  (pass-dly var addr))
    (zdly         (pass-zdly var addr))
    (cmbflt       (pass-cmbflt var addr))
    (allpassflt   (pass-allpassflt var addr))
    (flt          (pass-filter var addr))
    (flt-one-pole (pass-one-pole var addr))
    (flt-two-pole (pass-two-pole var addr))
    (flt-one-zero (pass-one-zero var addr))
    (flt-two-zero (pass-two-zero var addr))
    (frmnt        (pass-frmnt var addr))
    (noi 	  (pass-noi var addr))
    (randi 	  (pass-randi var addr))
    (smp          (pass-smp var addr))
    (rdin         (pass-rdin var addr))
    (spd          (pass-spd var addr))
    (sr           (pass-sr var addr))
    (fftflt       (pass-fftflt var addr))
    (fft-data     (pass-fft-data var addr))
    (conv         (pass-conv var addr))
    (cosp         (pass-cosp var addr))
    (locs         (pass-locs var addr))
    ((table x-table y-table) (pass-user-table var addr))
    (array        (pass-frac-array var addr))
    ;; the idea here is that an array of arrays (as opposed to a multidimensional array) will only be
    ;; an array of fractional arrays (either tables or polynomial coefficients), and we'll be called
    ;; with "type" = 'array only in this kind of recursive pass-var of an array of arrays.  pqwvox.ins
    ;; is an example.  I suppose we could use (array-element-type var) here and do this correctly.
    (sw           (pass-sw var addr))
    (ws           (pass-ws var addr))
    (rblk         (pass-blk var addr))
    (wt           (pass-wt var addr))
    (osc	  (pass-osc var addr))
    (envelope     (pass-envelope-data var addr))
    (t (pass-user-struct_56 var addr type))))


(defun pass-frac-array (arr loc1)
  (if arr
      (let* ((ar-siz (array-total-size arr))
	     (base (get-L-mem ar-siz)))
	(setf-x-mem loc1 (* base 2))	;*2 for 56000 "fractional" arithmetic
	(setf-y-mem loc1 1)
	(setf-x-mem (+ loc1 1) (- ar-siz 1))
	(setf-y-mem (+ loc1 1) (+ base ar-siz -1))
	(loop for i from 0 below ar-siz and j from base by 1 do
	  (pass-fraction (aref arr i) j)))
    (pass-nil loc1)))

(defmacro pass-var (addr var type)
  (case type
    (integer      `(setf-x-mem ,addr (if (numberp ,var) ,var 0)))
    (long-int     `(pass-long-int ,var  ,addr))
    (real         `(pass-real ,var ,addr))
    (fraction     `(pass-fraction ,var ,addr))
    (tbl  	  `(pass-tbl ,var ,addr))
    (dly	  `(pass-dly ,var ,addr))
    (zdly         `(pass-zdly ,var ,addr))
    (cmbflt       `(pass-cmbflt ,var ,addr))
    (allpassflt   `(pass-allpassflt ,var ,addr))
    (flt          `(pass-filter ,var ,addr))
    (flt-one-pole `(pass-one-pole ,var ,addr))
    (flt-two-pole `(pass-two-pole ,var ,addr))
    (flt-one-zero `(pass-one-zero ,var ,addr))
    (flt-two-zero `(pass-two-zero ,var ,addr))
    (frmnt        `(pass-frmnt ,var ,addr))
    (noi 	  `(pass-noi ,var ,addr))
    (randi 	  `(pass-randi ,var ,addr))
    (smp          `(pass-smp ,var ,addr))
    (rdin         `(pass-rdin ,var ,addr))
    (spd          `(pass-spd ,var ,addr))
    (sr           `(pass-sr ,var ,addr))
    (fftflt       `(pass-fftflt ,var ,addr))
    (fft-data     `(pass-fft-data ,var ,addr))
    (conv         `(pass-conv ,var ,addr))
    (cosp         `(pass-cosp ,var ,addr))
    (locs         `(pass-locs ,var ,addr))
    ((table x-table y-table) `(pass-user-table ,var ,addr))
    (array        `(pass-frac-array ,var ,addr))
    (sw           `(pass-sw ,var ,addr))
    (ws           `(pass-ws ,var ,addr))
    (rblk         `(pass-blk ,var ,addr))
    (wt           `(pass-wt ,var ,addr))
    (osc	  `(pass-osc ,var ,addr))
    (envelope     `(pass-envelope-data ,var ,addr))
    (t            `(pass-user-struct_56 ,var ,addr ',type))))

#|
(defmacro pass-array (arr loc1 typ)
  (let ((aname (gensym)))
    `(let ((el-siz (max 1 (structure-size_56 ',typ))))
       (if ,arr
	   (if (subtypep (type-of ,arr) 'ARRAY)
	       (let* ((ar-siz (array-total-size ,arr))
		      (size (* ar-siz el-siz))
		      (base (get-L-mem size)))
		 (setf-x-mem ,loc1 (* base 2)) ;*2 for 56000 "fractional" arithmetic
		 (setf-y-mem ,loc1 el-siz)
		 (setf-x-mem (1+ ,loc1) (- size 1))
		 (setf-y-mem (1+ ,loc1) (+ base size -1))
		 (if (= (array-rank ,arr) 1)
		     (loop for i from 0 below ar-siz and j from base by el-siz do
			   (pass-var j (aref ,arr i) ,typ))
		   (let ((,aname (make-array ar-siz :displaced-to ,arr :element-type (array-element-type ,arr))))
		     (loop for i from 0 below ar-siz and j from base by el-siz do
			   (pass-var j (aref ,aname i) ,typ)))))
	     (error "~A is of type ~A, not an array." ,arr (type-of ,arr)))
	 (pass-nil ,loc1)))))
|#
(defun pass-array (arr loc1 typ)
  (let ((el-siz (max 1 (structure-size_56 typ))))
    (if arr
	(if (subtypep (type-of arr) 'ARRAY)
	    (let* ((ar-siz (array-total-size arr))
		   (size (* ar-siz el-siz))
		   (base (get-L-mem size)))
	      (setf-x-mem loc1 (* base 2)) ;*2 for 56000 "fractional" arithmetic
	      (setf-y-mem loc1 el-siz)
	      (setf-x-mem (1+ loc1) (- size 1))
	      (setf-y-mem (1+ loc1) (+ base size -1))
	      (if (= (array-rank arr) 1)
		  (loop for i from 0 below ar-siz and j from base by el-siz do
			(fpass-var j (aref arr i) typ))
		(let ((aname (make-array ar-siz :displaced-to arr :element-type (array-element-type arr))))
		  (loop for i from 0 below ar-siz and j from base by el-siz do
			(fpass-var j (aref aname i) typ)))))
	  (error "~A is of type ~A, not an array." arr (type-of arr)))
      (pass-nil loc1))))


;;; DSP DEBUGGING AIDS 

(defvar last-loaded-ld nil)
(defun active-dsp () last-loaded-ld)

(defstruct (dar 
	    (:print-function
	     (lambda (d s k)
	       (declare (ignore k))
	       (if (or (< (dar-siz d) 32) 
		       (and (numberp (dar-el-siz d))
			    (> (dar-el-siz d) 2)))
		   (format s "~AArray[~A, (~A:~A by ~A~A)]: ~{~&    [~D] ~A~}~%" 
			   (if (dar-err d) "Bogus " "")
			   (dar-siz d) (dar-bas d) (dar-top d) (dar-el-siz d)
			   (if (dar-err d) (dar-err d) "")
			   (loop for j from 0 below (dar-siz d)
			    collect j 
			    collect (aref (dar-ray d) j)))
		 (format s "~AArray[~A, (~A:~A by ~A~A)]: ~{~A ~} ... ~{~A ~}~%"
			   (if (dar-err d) "Bogus " "")
			   (dar-siz d) (dar-bas d) (dar-top d) (dar-el-siz d)
			   (if (dar-err d) (dar-err d) "")
			   (loop for j from 0 below 16
			    collect (aref (dar-ray d) j))
			   (loop for j from (- (dar-siz d) 16) below (dar-siz d)
			    collect (aref (dar-ray d) j)))))))
  ray siz bas top el-siz err)

(defvar dsp-print-it t)

(defun dm-x (i) (dsp-mem 'X i 'integer))
(defun dm-y (i) (dsp-mem 'Y i 'integer))
(defun dm-real (i) (real56 0 (dm-x i) (dm-y i)))
(defun dm-frac (i) (frac24 (dm-y i)))
(defun dm-int (i) (dm-x i))
(defun dm-lint (i) 
  (let ((bigi (+ (ash (dm-x i) 24) (dm-y i))))
    (if (>= bigi (expt 2 47)) (- bigi (expt 2 48)) bigi)))
(defun not-nil (addr) (or (/= <nil> (dm-x addr)) (/= 0 (dm-y addr))))

(defun dm-tbl (addr)
  (if (not-nil addr)
      (let ((size (dm-x (+ addr 3)))
	    (base (dm-y (+ addr 3)))
	    (x-side (= 0 (dm-y (+ addr 5)))))
	(make-tbl :phase (dm-real addr) 
		  :freq (dm-real (1+ addr)) 
		  :table-size size
		  :internal-mag (dm-real (+ addr 4))
		  :table (if dsp-print-it
			     (format nil "~A ~A" 
				     base
				     (if x-side "X" "Y"))
			   (let ((new-array (make-array size :element-type 'short-float :initial-element 0.0)))
			     (loop for i from 0 below size and j from base do 
			       (setf (aref new-array i) (if x-side (frac24 (dm-x j)) (frac24 (dm-y j)))))))))))

;;; display envelope -- there's currently no attempt to make a
;;; real envelope here -- we're interested in readable debugging info.

(defun dm-env (addr)
  (if (not-nil addr)
      (if (not dsp-print-it) 
	  (print "we can't handle on-chip envelopes in phrase-value yet")
	(let* ((dummy-pointer (dm-x (+ addr 1))) ;pointer to dynamically allocated data
	       (useful-info (dm-y (+ addr 1)))   ;data-size in low 8 bits, data-base (start of dynamic space) in high byes
					         ; -- there may be 3 words of envelope info below this location
	       (restart-info (dm-x (+ addr 3)))
	       (data-size (logand useful-info #xFF)) ;there are data-size/2 points in dsp memory
	       (data-points (/ data-size 2))
	       (true-data-base (ash (logand useful-info #xFFFFFF) -8)) 
	       (current-value (dm-real addr))    ;if not exp-func, else current power
	       (base (dm-real (+ addr 2)))       ;if exp-func a pointer to data block, else = 1.0 (0.0 -> step func)
	       (exp-time (and (not (zerop base)) (/= base 1.0)))
	       (run-beg (dm-lint 0))             ;*run-beg* can always be found here -- we want to subtract out the start point
	       (in-memory-data (loop for i from 0 below data-points and j from true-data-base by 2
				collect (- (dm-lint j) run-beg)
				collect (dm-real (+ j 1))))
	       (base-ok (or (< addr ext-L-offset) (not (zerop useful-info)) (not (zerop (dm-y addr)))))
	       (y-mem-ok (or (< true-data-base ext-L-offset)
			     (loop for j from true-data-base and i from 0 below data-size
			      if (not (zerop (dm-y j))) return t)))
	       (end-marker (dm-lint (+ true-data-base data-size)))
	       (maxx (loop for x in in-memory-data by #'cddr maximize x))
	       (new-data (append 
			  (list 0 0)
			  (loop for x in in-memory-data by #'cddr 
			   and y in (cdr in-memory-data) by #'cddr 
			   and lstx = 0 then x
			   sum (* (- x lstx) y) into lasty
			   collect (* 100.0 (if (zerop maxx) 1.0 (/ (- x run-beg) maxx)))
			   collect lasty)))
	       (miny1 (loop for i in (cdr new-data) by #'cddr minimize i))
	       (miny (if (< (abs miny1) .0005) 0.0 miny1))
	       (maxy (loop for i in (cdr new-data) by #'cddr maximize i))
	       (scaley (if (= maxy miny) 1.0 (/ 1.0 (- maxy miny))))
	       (miny2 (if (not (plusp maxy)) (- 1.0 (* maxy scaley)) miny))
	       (scaled-data (loop for x in new-data by #'cddr and y in (cdr new-data) by #'cddr
			     collect (* .01 (round (* 100 x)))
			     collect (* .001 (round (* 1000 (+ miny2 (* scaley y)))))))
	       (trouble (or (< dummy-pointer true-data-base)
			    (> dummy-pointer (+ true-data-base data-size 2))))
	       (end-state (if (= end-marker -1) 
			      "" 
			    (format nil " :ext-env ~D for ~D" 
				    (1- (abs (- (dm-x (+ true-data-base data-size)) (expt 2 24))))
				    (dm-y (+ true-data-base data-size)))))
	       (scaler (and exp-time (dm-real (- true-data-base 4))))
	       (offset (and exp-time (dm-real (- true-data-base 3))))
	       (exp-base (and exp-time (dm-real (- true-data-base 5))))
	       (exp-cur-val (and exp-time (dm-real (- true-data-base 2))))
	       (exp-constant-base (and exp-time (dm-real (- true-data-base 1)))))
	  (format nil "#<~A-envelope[~D for ~D~A] :current-value ~,3F~A~A~A~A~%  ~
                           :raw-data[~A~A] '(~{~D ~,3E ~})~%  :massaged-data '(~{~,2F ~,2F ~})>"
		  (if (= base 1.0) "Seg" 
		    (if (= base 0.0) "Step"
		      "Exp"))
		  true-data-base data-points
		  (if (not base-ok)
		      " (envelope header clobbered by dac)"
		    (if (not y-mem-ok)
			" (envelope data clobbered by dac)"
		      ""))
		  (if exp-time exp-cur-val current-value)
		  (if exp-time (format nil " :base ~,3F (via ~,3F)" exp-base exp-constant-base) "")
		  (if exp-time (format nil " :scaler ~,3F" scaler) "")
		  (if exp-time (format nil " :offset ~,3F" offset) "")
		  (if (plusp restart-info) " :restartable t" "")
		  (if trouble "bogus pointer" (format nil "->~D" (floor (- dummy-pointer true-data-base) 2)))
		  end-state
		  in-memory-data
		  scaled-data)))))

(defun dm-dly (addr)
  (let ((type (dm-y (+ addr 1))))
    (if (or (= 2 type) (= 10 type))
	(let* ((id (dm-y (+ addr 2)))
	       (lim (if (active-dsp) (length (ld-sigbas (active-dsp))) 0))
	       (lid (or (and (active-dsp)
			     (loop for i from 0 below lim
			      if (and (= (aref (ld-sigbas (active-dsp)) i) id) 
				      (or (= (aref (ld-sigops (active-dsp)) i) 2)
					  (= (aref (ld-sigops (active-dsp)) i) 4)))
			      return i))
			0))
	       (bd (and *clm-delay-lines* (aref *clm-delay-lines* id))))
	  (if bd
	      (make-dly :size (dly-size bd)
			:loc (if (active-dsp) (aref (ld-sigptr (active-dsp)) lid) "no ptr found")
			:pline (dly-pline bd)
			:ext-id (if (= type 2) "External" "External preset")
			:id id)
	    (make-dly :size (dm-x (+ addr 2))
		      :loc (dm-x (+ addr 1))
		      :id "External delay -- flushed")))
      (make-dly :size (dm-x (+ addr 2)) 
		:loc  (dm-x (+ addr 1))
		:id (format nil "~A ~A" 
			    (dm-x addr) 
			    (if (= 1 type) "Y" 
			      (if (= 0 type) "X" 
				(if (= 3 type) "unused delay" 
				  (if (= 4 type) "QP DRAM"
				    (if (= 8 type) "X preset"
				      (if (= 9 type) "Y preset"
					"illegal type!")))))))))))
      
(defun dm-flt (addr)
  (if (not-nil addr)
      (let* ((m1 (dm-y addr))
	     (size (1+ m1))
	     (a-base (- (dm-y (+ addr 2)) m1))
	     (p-base (- (dm-y (+ addr 1)) m1))
	     (d-base (dm-x (+ addr 2)))
	     (c-base-1 (dm-x (+ addr 1)))
	     (c-base (- c-base-1 m1))
	     (a1 (make-array size :element-type 'short-float))
	     (p1 (make-array size :element-type 'short-float))
	     (c1 (if (not (zerop c-base-1)) (make-array size :element-type 'short-float :initial-element 0.0)))
	     (d1 (make-array (1+ size) :element-type 'short-float :initial-element 0.0)))
	(loop for i from 0 to m1 and ja from a-base and
	          jp from p-base and jc from c-base and jd from d-base do
	  (setf (aref a1 i) (frac24 (dm-y ja)))
	  (setf (aref p1 i) (frac24 (dm-y jp)))
	  (setf (aref d1 i) (frac24 (dm-x jd)))
	  (if (not (zerop c-base-1)) (setf (aref c1 i) (frac24 (dm-x jc)))))
	(make-flt :m (dm-y addr)
		  :typ (case (dm-x addr) (0 direct-form) (1 lattice-form) (2 ladder-form) (t "invalid"))
		  :so (frac24 (dm-x (+ addr 3)))
		  :a a1 :b p1 :c c1 :d d1))))
					;R2->flt struct: type   (end c)  (start d)  so
					;                order  (end p)  (end a)  (end d) (d has extra end word of 0)
(defun dm-rdin (addr)
  (if (not dsp-print-it) (print "we can't handle on-chip readin structures in phrase-value yet"))
  (if (not-nil addr)
      (format nil "Readin: i: ~A, inc: ~A, fil: ~A, chan: ~A, ~
                           bufstart: ~A, bufend: ~A, file-end: ~A, buf: ~A:~A [~A]"
	      (dm-lint addr) (dm-lint (+ addr 1))
	      (logand (ash (dm-y (+ addr 5)) -16) #xFF)
	      (logand (ash (dm-y (+ addr 5)) -8) #xFF)
	      (dm-lint (+ addr 4)) 
	      (if (or (/= (dm-x (+ addr 3)) #xFFFFFF)
		      (/= (dm-y (+ addr 3)) #xFFFFFF))
		  (dm-lint (+ addr 3))
		"un-opened")
	      (dm-lint (+ addr 2))
	      (if (zerop (logand (dm-y (+ addr 5)) 1)) "X" "Y")
	      (dm-x (+ addr 5)) (dm-lint (+ addr 6)))))

(defun dm-spd (addr)
  (if (not dsp-print-it) (print "we can't handle on-chip expansion structures in phrase-value yet"))
  (if (not-nil addr)
      (format nil "Expand: cur-val: ~A, ctr: ~A, b: ~A:~A [~A], ~
                           cur-out: ~A, steady: ~A, rmp: ~A, incr: ~A, temp: ~A, ~
		           Readin: i: ~A, inc: ~A, fil: ~A, chan: ~A, ~
                           bufstart: ~A, bufend: ~A, file-end: ~A, buf: ~A:~A [~A], ~
                           out-spd: ~A, s50: ~A, cur-in: ~A, in-spd: ~A, s20: ~A"

	      (frac24 (dm-x addr))
	      (dm-x (+ addr 1))
	      (if (zerop (logand (dm-y addr) 1)) "X" "Y")
	      (dm-y (+ addr 1))
	      (+ (dm-y (+ addr 2)) 1)
	      (dm-x (+ addr 2))
	      (dm-x (+ addr 3))
	      (dm-y (+ addr 3))
	      (frac24 (dm-x (+ addr 4)))
	      (frac24 (dm-y (+ addr 4)))
	      (dm-lint (+ addr 5)) (dm-lint (+ addr 6))
	      (logand (ash (dm-y (+ addr 10)) -16) #xFF)
	      (logand (ash (dm-y (+ addr 10)) -8) #xFF)
	      (dm-lint (+ addr 9)) 
	      (if (or (/= (dm-x (+ addr 8)) #xFFFFFF)
		      (/= (dm-y (+ addr 8)) #xFFFFFF))
		  (dm-lint (+ addr 8))
		"un-opened")
	      (dm-lint (+ addr 7))
	      (if (zerop (logand (dm-y (+ addr 10)) 1)) "X" "Y")
	      (dm-x (+ addr 10)) (dm-lint (+ addr 11))
	      (dm-x (+ addr 12))
	      (dm-y (+ addr 12))
	      (dm-lint (+ addr 13))
	      (dm-x (+ addr 14))
	      (dm-y (+ addr 14)))))

(defun dm-sr (addr)
  (if (not-nil addr)
      (make-sr :rd (dm-rdin (+ addr 6))
	       :x (dm-real (+ addr 2))
	       :incr (dm-real (+ addr 3))
	       :left (dm-x addr)
	       :right (dm-y addr)
	       :width (dm-x (+ addr 1))
	       :data (dm-table (+ addr 4))
	       :filt (dm-table (+ addr 13)))))

(defun dm-fftflt (addr)
  (if (not-nil addr)
      (make-fftflt :siz (dm-x (+ addr 4))
		   :half-siz (dm-x (+ addr 5))
		   :hop (dm-y (+ addr 4))
		   :b (dm-blk addr)
		   :rd (dm-rdin (+ addr 8))
		   :datar (dm-table (+ addr 6) 0)
		   :datai (dm-table (+ addr 6) 1)
		   :env (if (zerop (dm-x (+ addr 17)))
			    (dm-table (+ addr 15))
			  (if (/= 2 (dm-x (+ addr 17)))
			      (get-dsp-var (+ addr 15) 'envelope))))))

(defun dm-fft-data (addr)
  (if (not-nil addr)
      (make-fft-data :size (dm-y addr)
		     :real (dm-table addr 0)
		     :imaginary (dm-table (+ addr 2) 1))))

(defun dm-conv (addr)
  (if (not-nil addr)
      (if (/= 4 (dm-x (+ addr 17)))
	  (make-conv :fftf (dm-fftflt addr)
		     :filtr (dm-table (+ addr 15) 0)
		     :filti (dm-table (+ addr 15) 1))
	(format nil "Conv: external"))))

(defun dm-sw (addr)
  (if (not-nil addr)
      (make-sw :freq (dm-real (+ addr 2))
	       :phase (dm-real (+ addr 1))
	       :current-value (dm-real addr)
	       :base (dm-real (+ addr 4)))))

(defun dm-noi (addr)
  (if (not-nil addr)
      (make-noi :freq (dm-real (1+ addr))
		:phase (dm-real addr)
		:output (dm-real (+ addr 3))
		:ran-op (if (= 0 (dm-x (+ addr 6))) 'random 'sambox-random)
		:base (dm-real (+ addr 4)))))

(defun dm-smp (addr)
  (if (not dsp-print-it) (print "we can't handle on-chip srate conversion structures in phrase-value yet"))
  (if (not-nil addr)
      (format nil "~A, fil: ~A, chan: ~A, bufstart: ~A, bufend: ~A, ~
                       file-end: ~A, buf: ~A:~A [~A]"   
	      (make-smp :lst (frac24 (dm-x addr))
			:nxt (frac24 (dm-y addr))
			:x (frac24 (dm-x (+ addr 1)))
			:i (dm-lint (+ addr 3))
			:sr (dm-real (+ addr 2)))
	      (logand (ash (dm-y (+ addr 7)) -16) #xFF)
	      (logand (ash (dm-y (+ addr 7)) -8) #xFF)
	      (dm-lint (+ addr 6)) (dm-lint (+ addr 5)) (dm-lint (+ addr 4))
	      (if (zerop (logand (dm-y (+ addr 7)) 1)) "X" "Y")
	      (dm-x (+ addr 7)) 
	      (dm-lint (+ addr 8)))))

(defun dm-table (addr &optional side)
  (if (not-nil addr)
      (let* ((base (dm-x addr))
	     (size (dm-y addr))
	     (x-side (if side (zerop side) (zerop (dm-y (+ addr 1)))))
	     (d (make-dar :bas base
			  :siz size
			  :el-siz (format nil "1 ~A" (if x-side " (X)" " (Y)"))
			  :err nil
			  :top (+ base size -1)
			  :ray (make-array size))))
	(loop for j from 0 below size and i from base do
	  (setf (aref (dar-ray d) j) (frac24 (if x-side (dm-x i) (dm-y i)))))
	(if dsp-print-it
	    d
	  (dar-ray d)))))


(defun dm-blk (addr)
  (if (not-nil addr)
      (make-blk :loc (dm-real addr)
		:ctr (dm-real (+ addr 1))
		:siz (dm-y (+ addr 2))
		:buf (dm-table (+ addr 2)))))

(defun dm-wt (addr)
  (if (not-nil addr)
      (if dsp-print-it
	  (format nil "b: ~A~%  wave: ~A~%  freq: ~A, phase: ~A, fm: ~A, internal-mag: ~A, srate: ~A, old-freq: ~A, old-ctr: ~A"
		  (dm-blk addr)
		  (dm-table (+ addr 4))
		  (dm-real (+ addr 7))
		  (dm-real (+ addr 6))
		  (dm-real (+ addr 8))
		  (dm-real (+ addr 11))
		  (dm-real (+ addr 12))
		  (dm-real (+ addr 9))
		  (dm-real (+ addr 10)))
	(make-wt :b (dm-blk addr)
		 :wave (dm-table (+ addr 4))
		 :freq (dm-real (+ addr 7))
		 :phase (dm-real (+ addr 6))
		 :internal-mag (dm-real (+ addr 11))))))

(defun dm-array (addr element-type)
  (if (not-nil addr)
      (let ((base (/ (dm-x addr) 2))
	    (top (dm-y (1+ addr)))
	    (el-size (dm-y addr))
	    (warning nil))
	(when (or (< base 6) (and (plusp external-mem-size) (> base (+ ext-L-offset (* 2 External-Mem-Size)))))
	  (setf base (/ (aref internal-x-memory addr) 2))
	  (setf warning " -- bad base"))
	(when (or (< top 6) (and (plusp external-mem-size) (> top (+ ext-L-offset External-Mem-size))))
	  (setf top (aref internal-y-memory (1+ addr)))
	  (setf warning (if warning " bad base, top" " -- bad top")))
	(when (or (< el-size 1) (> el-size biggest-structure))
	  (setf el-size (aref internal-y-memory addr))
	  (setf warning (format nil "~A~A" 
				(if warning warning "") 
				(if warning ", element size" " -- bad element size"))))
	(if (or (zerop base)
		(zerop top)
		(zerop el-size)
		(minusp (- top base -1))
		(> (/ (- top base) el-size) 5000))
	    (list "probably bogus array -- can't fix it up:" base top el-size)
	  (let ((d (make-dar :bas base 
			     :siz (/ (- top base -1) el-size) 
			     :el-siz el-size 
			     :err nil
			     :top top 
			     :ray (make-array (/ (- top base -1) el-size)))))
	    (loop for j from 0 and i from base to top by el-size do
	      (setf (aref (dar-ray d) j) (get-dsp-var i element-type)))
	    (if warning
		(setf (dar-err d) warning))					
	    (if dsp-print-it
		d
	      (dar-ray d)))))))

(defun dm-one-pole (addr)
  (if (not-nil addr)
      (make-smpflt :a0 (* 2 (frac24 (dm-x addr)))
		   :b1 (* 2 (frac24 (dm-y addr)))
		   :y1 (frac24 (dm-x (1+ addr))))))

(defun dm-one-zero (addr)
  (if (not-nil addr)
      (make-smpflt :a0 (* 2 (frac24 (dm-x addr)))
		   :a1 (* 2 (frac24 (dm-x (1+ addr))))
		   :x1 (frac24 (dm-y addr)))))

(defun dm-two-zero (addr)
  (if (not-nil addr)
      (make-smpflt :a0 (* 2 (frac24 (dm-x addr)))
		   :a1 (* 2 (frac24 (dm-x (1+ addr))))
		   :a2 (* 2 (frac24 (dm-x (+ addr 2))))
		   :x1 (frac24 (dm-y addr))
		   :x2 (frac24 (dm-y (+ addr 2))))))

(defun dm-two-pole (addr)
  (if (not-nil addr)
      (make-smpflt :a0 (* 2 (frac24 (dm-x addr)))
		   :b1 (* 2 (frac24 (dm-y addr)))
		   :b2 (* 2 (frac24 (dm-y (1+ addr))))
		   :y1 (frac24 (dm-x (1+ addr)))
		   :y2 (frac24 (dm-y (+ addr 2))))))

(defun dm-randi (addr)
  (if (not-nil addr)
      (let ((n (get-dsp-var addr 'noi)))
	(setf (noi-incr n) (dm-real (+ addr 5)))
	(setf (noi-ran-op n) (frac24 (dm-y (+ addr 6))))
	n)))

(defun dm-cosp (addr)
  (if (not-nil addr)
      (make-cosp :phase (dm-real addr) 
		 :freq (dm-real (1+ addr)) 
		 :scaler (frac24 (dm-x (+ addr 3)))
		 :cosines (- (dm-real (+ addr 4)) .5))))

(defun dm-locs (addr)
  (if (not-nil addr)
      (make-locs :ascl (frac24 (dm-x addr))
		 :bscl (frac24 (dm-y addr))
		 :cscl (frac24 (dm-x (+ addr 2)))
		 :dscl (frac24 (dm-y (+ addr 2)))
		 :rscl (frac24 (dm-x (1+ addr)))
		 :revname (dm-y (1+ addr)))))

(defun dm-osc (addr)
  (if (not-nil addr)
      (make-osc :freq (dm-real (1+ addr)) :phase (dm-real addr))))

(defun dm-allpassflt (addr)
  (if (not-nil addr)
      (make-allpassflt :feedback (frac24 (dm-y (+ addr 3)))
		       :feedforward (frac24 (dm-x (+ addr 3)))
		       :dly-unit (get-dsp-var addr 'dly))))

(defun dm-cmbflt (addr)
  (if (not-nil addr)
      (make-cmbflt :scaler (frac24 (dm-y (+ addr 3)))
		   :dly-unit (get-dsp-var addr 'dly))))

(defun dm-ws (addr)
  (if (not-nil addr)
      (make-ws :tab (get-dsp-var addr 'tbl)
	       :offset (dm-real (+ addr 6))
	       :os (get-dsp-var (+ addr 7) 'osc))))

(defun dm-zdly (addr)
  (if (not-nil addr)
      (make-zdly :phase (dm-real (+ addr 4))
		 :del (get-dsp-var addr 'dly))))

(defun dm-formnt (addr)
  (if (not-nil addr)
      (if dsp-print-it
	  (format nil "Frmnt: G: ~A, ~A, ~A" 
		  (dm-real addr)
		  (dm-two-zero (+ addr 1))
		  (dm-two-pole (+ addr 4)))
	(make-frmnt :G (dm-real addr)
		    :tz (dm-two-zero (+ addr 1))
		    :tp (dm-two-pole (+ addr 4))))))

(defun get-dsp-var (addr &optional (type 'real) (element-type 'real))
  (case type
    (integer      (dm-x addr))
    (long-int     (dm-lint addr))
    (real         (let ((val (dm-real addr)))
		    (if (= val 65536.0) (if dsp-print-it 
					    (format nil "~A (or nil)" val) 
					  nil)
		      (if dsp-print-it
			  (format nil "~,4F" val)
			val))))
    (fraction     (dm-frac addr))
    (tbl  	  (dm-tbl addr))
    (dly	  (dm-dly addr))
    (flt          (dm-flt addr))
    (noi 	  (dm-noi addr))
    (sw           (dm-sw addr))
    (rdin         (dm-rdin addr))
    (spd          (dm-spd addr))
    (sr           (dm-sr addr))
    (fftflt       (dm-fftflt addr))
    (fft-data     (dm-fft-data addr))
    (conv         (dm-conv addr))
    (smp          (dm-smp addr))
    ((table x-table y-table) (dm-table addr))
    (array        (dm-array addr element-type))
    (cmbflt       (dm-cmbflt addr))
    (allpassflt   (dm-allpassflt addr))
    (flt-one-pole (dm-one-pole addr))
    (flt-two-pole (dm-two-pole addr))
    (flt-one-zero (dm-one-zero addr))
    (flt-two-zero (dm-two-zero addr))
    (frmnt        (dm-formnt addr))
    (randi        (dm-randi addr))	  
    (cosp         (dm-cosp addr))
    (locs         (dm-locs addr))
    (ws           (dm-ws addr))
    (rblk         (dm-blk addr))
    (wt           (dm-wt addr))
    (zdly         (dm-zdly addr))
    (osc	  (dm-osc addr))
    (envelope     (dm-env addr))
    (t (display-user-struct_56 type addr))))

(defun get-dsp-value (addr &optional (type 'real) (element-type 'real))
  (let ((dsp-print-it nil))
    (get-dsp-var addr type element-type)))

(defun dm-io (input inbase inlen)
  (let ((insigs 0))
    (when (plusp inlen)
      (loop for i from inbase below (+ inbase inlen) do
	(if (plusp (dm-x (dm-x i))) (incf insigs)))
      (princ (format nil "There ~A currently ~A ~A stream~P active:~&~{        ~A:~A~A to ~A [->~A~A] (~A structure ~A at ~A)~&~}"
		     (if (= insigs 1) "is" "are")
		     insigs
		     (if input "input" "output")
		     insigs
		     (loop for i from inbase below (+ inbase inlen) and j from 0 by 1
		      when (plusp (dm-x (dm-x i)))
		      append (list 
			      (if (logbitp 0 (dm-y (+ (dm-x i) 1))) "Y" "X")
			      (- (dm-x (dm-x i)) ext-L-offset) 
			      (if (not (zerop ext-L-offset)) (format nil " (with external offset ~D)" ext-L-offset) "")
			      (if (not (zerop (dm-y (dm-x i))))
				  (- (dm-y (dm-x i)) ext-L-offset)
				"?? (index clobbered by dac)")
			      (- (dm-x (+ (dm-x i) 1)) ext-L-offset)
			      (if (logbitp 1 (dm-y (+ (dm-x i) 1))) " (touched this pass)" "")
			      (if input "input" "output")
			      j (dm-x i))))))))

(defun get-io-stuff (input output inlen)
  (if (and (not input) (not output) (zerop inlen))
      (princ "no IO channels")
    (let ((inbase (dm-x input))		;base of array of pointers
	  (outbase (dm-x output))
	  ;;	(inlen (dm-y input))		;length of that array -- not anymore (zero'd at end to flush final output)
	  (outlen (dm-y output)))
      (princ (format nil "~%We have provision for ~A input stream~P and ~A output stream~P.~&" 
		     (if (plusp inlen) inlen "no") inlen
		     (if (plusp outlen) outlen "no") outlen))
      (if (or (> inlen 16)
	      (> outlen 16))
	  (princ "which must be an error...")
	(progn
	  (dm-io t inbase inlen)
	  (dm-io nil outbase outlen))))))


(defvar <input> nil)
(defvar <output> nil)
(defvar <clm-ins-tag> nil)
(defvar <clm-tag-addr> nil)
;(defvar saved-pup nil)
(defvar saved-pp nil)

(defun c56-current-slot () 
  (dsp-set (max 0 current-slot) (max 0 current-dsp))
  (max 0 current-slot))

(defun c56-current-dsp () 
  (max 0 current-dsp))

(defun dsp-print-var (key val)
  ;; get-dsp-var address type element-type
  (princ (format nil "~%~A ~A" key (get-dsp-var (second (first val)) (second val) (third val)))))

(defun display-instrument-state ()
  (let* ((ydat (get *current-instrument-name* :dsp-int-y-data))
	 (yloc (get *current-instrument-name* :dsp-<input>))
	 (inlen (or (and ydat yloc (aref ydat yloc)) 0)))
    (dsp-debug (c56-current-slot) (c56-current-dsp))
    (maphash 'dsp-print-var (get *current-instrument-name* :ins-vars))
    (get-io-stuff (get *current-instrument-name* :dsp-<input>)
		  (get *current-instrument-name* :dsp-<output>)
		  inlen)
    (dsp-close (c56-current-slot) (c56-current-dsp))))

(defun find-a-home-for (var presumptive-type &optional (element-type nil))

  ;; need to get i a place (L), and load up initial values, or get entire structure read in.
  ;; this has to happen at run-time -- will need to store env data in heap, I guess.
  ;; So, at compile-time (from RUN point of view), we allocate statically until all vars
  ;; accounted for, then pass back current heap-pointer.  On each call, we run through envelope list,
  ;; loading current data (with fix-ups), and back-patching data pointer location (a load, not an
  ;; instruction change, since everything indirects through the "dummy pointer").  A similar situation
  ;; occurs with TABLE-LOOKUP tables (including wave-shaping), delay lines, and filter coefficient tables.

  (let ((addr (if (not (eq var loop-var)) 
		  (get-L-mem (structure-size_56 presumptive-type))
		(cadr (get-home-address '<pass-counter>)))))
    
    (DEBUGGING (push `(COMMENT ,var ,addr ,presumptive-type ,element-type) pp))

    (add-var var presumptive-type 
	     (if (eq presumptive-type 'integer)
		 (list 'X addr)
	       (if (eq presumptive-type 'fraction)
		   (list 'Y addr)
		 (list 'L addr)))
	     nil nil
	     element-type)
    (if (or (temp-sig var) (car-member var typed-user-sig))
	(push `(pass-zero ,addr) pup)
      (case presumptive-type
	(integer      (push `(setf-x-mem ,addr ,var) pup))
	(long-int     (if (not (eq var loop-var)) (push `(pass-long-int ,var ,addr) pup)))
	(real         (if (or (not element-type)
			      (not (listp element-type)))
			  (push `(pass-user-real ,var ,addr) pup)
			(push `(pass-user-real ,element-type ,addr) pup)))
	(fraction     (push `(pass-fraction ,var ,addr) pup))
	(tbl  	      (push `(pass-tbl  ,var ,addr) pup))
	(dly	      (push `(pass-dly ,var ,addr) pup))
	(zdly         (push `(pass-zdly ,var ,addr) pup))
	(cmbflt       (push `(pass-cmbflt ,var ,addr) pup))
	(allpassflt   (push `(pass-allpassflt ,var ,addr) pup))
	(flt          (push `(pass-filter ,var ,addr) pup))
	(flt-one-pole (push `(pass-one-pole ,var ,addr) pup))
	(flt-two-pole (push `(pass-two-pole ,var ,addr) pup))
	(flt-one-zero (push `(pass-one-zero ,var ,addr) pup))
	(flt-two-zero (push `(pass-two-zero ,var ,addr) pup))
	(frmnt        (push `(pass-frmnt ,var ,addr) pup))
	(noi 	      (push `(pass-noi ,var ,addr) pup))
	(randi 	      (push `(pass-randi ,var ,addr) pup))
	(array        (push `(pass-array ,var ,addr ',element-type) pup))
	(ws           (push `(pass-ws ,var ,addr) pup))
	(rblk         (push `(pass-blk ,var ,addr) pup))
	(wt           (push `(pass-wt ,var ,addr) pup))
	(smp          (push `(pass-smp ,var ,addr) pup))
	(rdin         (push `(pass-rdin ,var ,addr) pup))
	(spd          (push `(pass-spd ,var ,addr) pup))
        (sr           (push `(pass-sr ,var ,addr) pup))
	(fftflt       (push `(pass-fftflt ,var ,addr) pup))
	(fft-data     (push `(pass-fft-data ,var ,addr) pup))
	(conv         (push `(pass-conv ,var ,addr) pup))
	(cosp         (push `(pass-cosp ,var ,addr) pup))
	(locs         (push `(pass-locs ,var ,addr) pup))
	((table x-table y-table) (push `(pass-user-table ,var ,addr) pup))
	(sw    	      (push `(pass-sw ,var ,addr) pup))
	(osc	      (push `(pass-osc ,var ,addr) pup))
	(envelope     (push `(pass-envelope-data ,var ,addr) pup))
	(t            (push `(pass-user-struct_56 ,var ,addr ',presumptive-type) pup))))))

(defun initialize-everything_56 ()
  ;; here we clear out everything in the code-generator and the assembler.
  (DEBUGGING (setf emit-prog nil))
  (initialize-chip-memory)		;sets ix-pc and friends to initial values
  (free-all-regs)
  (setf alias-list nil)
  (setf pp nil)
  (setf pup nil)
  (setf pip nil)
  (setf p-init nil)			;from lib56 (library initialization code)
  (setf pp-init-sample nil)
  (setf insigs nil)
  (setf outsigs nil)
  ;;  (setf run-time-stack nil)
  (clrhash vars)
  (init-emit)
  (setf correct-pipeline-silently t)	;just in case it got clobbered (see dsp56.lisp insert-nop)
  (loop for i from 0 to 7 do (setf (aref N-values i) -1))
  (setf library-load-list nil))

(defmacro <start_56> (beg end)
  (initialize-everything_56)
  (if beg (push `(setf *run-beg* (floor ,beg)) pup))
  (if end (push `(setf *run-end* (floor ,end)) pup))
  (let ((beg-addr (get-L-mem))
	(end-addr (get-L-mem))
	(out-addr (get-L-mem 2))
	(in-addr  (get-L-mem)))
    (setf <clm-tag-addr> (get-X-memory))
    (setf <clm-ins-tag> (random (expt 2 23)))
    (when beg 
      (push `(pass-long-int *run-beg* 0) pup)
      (push `(pass-long-int *run-beg* ,beg-addr) pup))
    (add-var '<pass-counter> 'long-int (list 'L beg-addr) nil)
    (push `(DEFINE <input> ,in-addr) pp)
    (add-var '<input> 'integer (list 'X in-addr) nil)
    (setf <input> in-addr)
    (push `(DEFINE <output> ,out-addr) pp)
    (add-var '<output> 'integer (list 'X out-addr) nil)
    (setf <output> out-addr)
    (add-var '<output+1> 'integer (list 'L (1+ out-addr)) nil)
    (add-var '*RUN-BEG* 'long-int (list 'L beg-addr) nil)
    (if end (push `(pass-long-int *run-end* ,end-addr) pup))
    (add-var '<run-end> 'long-int (list 'L end-addr) nil)
    (add-var '*RUN-END* 'long-int (list 'L end-addr) nil)
    (push `(DEFINE <run-end> ,end-addr) pp))
  (loop for (var typ el) in typed-user-var do
    (find-a-home-for var typ el))
  (loop for (var typ) in typed-user-sig do
    (find-a-home-for var typ))
  (loop for i in user-var do
    (when (not (gethash i vars))
      (DEBUGGING (push `(COMMENT assume ,i is real) pp))
      (find-a-home-for i 'real)))
  (loop for i in true-user-var do
    (when (not (gethash i vars))
      (DEBUGGING (push `(COMMENT assume ,i is real) pp))
      (find-a-home-for i 'real)))
  nil)

(defun make-initial-contents (data len &optional (offset 0))
  (when (plusp len)
    (let ((vect nil))
      (loop for i from (- len 1) downto 0 do 
	(push (aref data (+ i offset)) vect))
      `(make-array ,len :element-type 'fixnum :initial-contents ',vect))))

;;; can't use vector here because in this lisp a simple array is not the same as a vector, causing
;;; no end of confusion at the C end (but, of course, no error messages anywhere...)

(defun set-initial-contents (name ixp iyp exp eyp hp)
  (if (plusp ixp) (c-array-transfer 0 (- ixp 1) (get name :dsp-int-x-data) 0 internal-x-memory))
  (if (plusp iyp) (c-array-transfer 0 (- iyp 1) (get name :dsp-int-y-data) 0 internal-y-memory))
  (if (plusp exp) (c-array-transfer 0 (- exp 1) (get name :dsp-ext-x-data) 0 external-memory))
  (if (plusp eyp) (c-array-transfer 0 (- eyp 1) (get name :dsp-ext-y-data) External-Y-from-X-offset external-memory))
  (setf heap-ptr hp)
  (setf DRAM-loc 0)
  (setf DRAM-list nil)
  (setf ix-ptr ixp)
  (setf iy-ptr iyp)
  (setf ex-ptr exp)
  (setf ey-ptr eyp))

(defun set-final-allocation-pointers ()
  (values heap-ptr ix-ptr iy-ptr ex-ptr ey-ptr))


;;; INSTRUMENT RUN-TIME 

(defun get-addresses (sglst sgadr)
  (let ((sigs 0))
    (when (and sglst sgadr)
      (loop for (op str chn loc) in sglst and j from 0 do
	(setf (aref sgadr j) 
	  (if (= str -1) 
	      0
	    (case op
	      ((0 1) 
	       (incf sigs)
	       (if (= chn 0) 
		   (clm-IO-dat-a str)
		 (if (= chn 1)
		     (clm-IO-dat-b str)
		   (if (= chn 2)
		       (clm-IO-dat-c str)
		     (clm-IO-dat-d str)))))
	      (3 (if (/= chn -1) (incf sigs)) 0)
	      ((2 4) (dly-pline (aref *clm-delay-lines* str))))))))
    sigs))


(defun pass-xy-IO (loc ptr siz x-side) 
  (if (plusp siz) 
      (pass-IO loc ptr (+ ptr siz -1) x-side)
    (progn
      (pass-zero loc)
      (pass-zero (+ loc 1))
      (pass-zero (+ loc 2))))
  siz)
;;; I believe this is not right -- if siz = 1, we should use the 3rd word of the IO structure
;;; (if siz = 0, it's an inactive stream)


;;; sgops indices are: 0 = out-n (outa or outb)
;;;                    1 = in-n (ina or inb)
;;;                    2 = external delay line (normal delay)
;;;                    3 = random access input (clloadinput in next56.lisp) (if chn /= -1 => inactive readin?)
;;;                   -1 = inactive 

(defun get-ops (sglst sgops sgszs sgbas sgtop sgptr bufsiz ex-pc ey-pc sram-locs x-or-ys)
  #-QP (declare (ignore sram-locs x-or-ys))
  (let ((ex-cur ex-pc)
	(ey-cur ey-pc)
	(dlys 0)
	(ins 0))
    (loop for (op str chn loc) in sglst and j from 0 do
      (if (/= str -1)
	  (progn
	    (when (or (= 0 op) (= 1 op))
	      (if (<= ex-cur ey-cur)
		  (progn
		    #+QP (QP-save-IO-data (+ ex-cur ext-L-offset) 0 j sram-locs x-or-ys)
		    (incf ex-cur (pass-xy-IO loc ex-cur bufsiz t)))
		(progn
		  #+QP (QP-save-IO-data (+ ey-cur ext-L-offset) 1 j sram-locs x-or-ys)
		  (incf ey-cur (pass-xy-IO loc ey-cur bufsiz nil)))))
	    (when (and (= op 3)		;random access input
		       (/= chn -1))
	      (if (<= ex-cur ey-cur)
		  (incf ex-cur (pass-insig-IO loc ex-cur bufsiz t chn))
		(incf ey-cur (pass-insig-IO loc ey-cur bufsiz nil chn))))
	    (if (= op 1) 
		(progn
		  (incf ins)
		  (setf (aref sgszs j) (clm-IO-siz str))
		  (setf (aref sgptr j) 0) ;was str
		  (setf (aref sgbas j) (clm-IO-beg str))
		  (setf (aref sgtop j) (clm-IO-end str)))
	      (if (= op 2)
		  (progn
		    (incf dlys)
		    (setf (aref sgszs j) (dly-size (aref *clm-delay-lines* str)))
		    (setf (aref sgbas j) str)
		    (setf (aref sgtop j) (- (dly-size (aref *clm-delay-lines* str)) 1)))))
	    (setf (aref sgops j) op))
	(progn
	  (if (or (= op 0) (= op 1)) (pass-xy-IO loc 0 0 nil))
	  (setf (aref sgops j) -1))))
    (values dlys ins)))


(defun add-dsp-data (dsp beg end bufsiz len sigops sigadr sigptr sigbas sigtop sigsiz dly-ops in-ops srams xs)
  (make-ld :dsp dsp
	   :beg beg
	   :curbeg beg
	   :curend (min (+ beg bufsiz -1) end)
	   :end end
	   :bufsiz bufsiz
	   :len len
	   :dly-ops dly-ops
	   :in-ops in-ops
	   :sigops sigops
	   :sigadr sigadr
	   :sigptr sigptr
	   :sigbas sigbas
	   :sigtop sigtop
	   :sigsiz sigsiz
	   :sram-locs srams
	   :x-or-ys xs))

(defun set-buf-siz (ex-pc ey-pc bufs &optional (ext-L-size external-L-size) (no-output-here nil))
  ;; returns maximum possible IO buffer size given external heap sizes and current IO needs
  (if no-output-here
      0
    (progn
      (if (not (plusp bufs)) 
	  (error "~A sound buffers open?~A" 
		 (if (zerop bufs) "no" bufs) 
		 (if (zerop bufs) " (instrument call outside with-sound?)" "")))
      (let* ((bx (min (- ext-L-size ex-pc) file-buffer-size))
	     (by (min (- ext-L-size ey-pc) file-buffer-size))
	     (bmin (min bx by))
	     (bmax (max bx by)))
	(if (and (zerop bx) (zerop by))	;both are full
	    1				;so use word 3 of the allocated IO structure
	  (let ((fbmax (floor bmax bufs)))
	    (if (or (zerop bmin)	;one or other is full
		    (> fbmax bmin))
		fbmax
	      (if (= bx by)		;this is the normal case
		  (floor bx (ceiling bufs 2))
		(if (< (floor bmax (- bufs 1)) bmin)
		    (floor bmax (- bufs 1))
		  fbmax)))))))))

(defun set-up-basic-IO-arrays (ins outs &optional (no-output-here nil))
  (if ins
      (let* ((len (length ins))
	     (base (get-X-memory len)))
	(setf-y-mem <input> len)
	(setf-x-mem <input> base)
	(loop for i in ins and j from base do
	  (setf-x-mem j i)))
    (progn
      (setf-x-mem <input> 0)		;not pass-zero here to avoid process-check confusion
					;(if running compiler and dsp at same time in same job)
      (setf-y-mem <input> 0)))
  (if outs
      (let* ((len (length outs))
	     (base (get-X-memory len)))
	(setf-y-mem <output> len)
	(setf-x-mem <output> base)
	(loop for i in outs and j from base do
	  (setf-x-mem j i)))
    (if (not no-output-here)
	(error "currently clm has to have at least one call on outa..d that uses the outer loop index as its sample number."))))

(defvar monitor-end 0)			;QP needs to know where to patch up the monitor
(defvar label-list nil)			;for stack trace on chip
(defun save-labels (name loc)
  (push (list name loc) label-list))

(defvar c56-load-chip-data-break nil)
(defvar c56-user-start-break nil)
(defvar c56-user-start-1-break nil)
(defvar c56-pass-start-break nil)
(defvar c56-run-expansions 0)

#|
(defun load-pathname ()
  #+(and excl cltl2) *load-pathname*
  #+(and excl (not cltl2)) tpl::*last-file-loaded*
  #+kcl system:*load-pathname*
  )
|#

(defun memory-map-name (map-offset)
  (if (= map-offset 4096) "8K external, P overlays X"
    (if (= map-offset 16384) "32K external, P overlays X"
      (if (/= map-offset 0) "unknown dsp memory map"
	#+(and NeXT arielpc56d) "128K external, P separate from X"
	#+(and NeXT (not arielpc56d)) "192K external, P separate from X"
	#+Mac "2K external, P separate from X"
	))))


(defmacro <end_56> (&optional no-output-here) ;finish up program, load and run it -- here pp is prog list
  (let ((inited nil))
    ;;(if (plusp c56-run-expansions) (warn "clm cannot yet handle more than one Run within an instrument"))
    ;;(incf c56-run-expansions)
    (DEBUGGING (pprint new-prog))
    (setf label-hook 'save-labels)
    (setf label-list nil)
    (spill-ALU-registers)		;prepare for loop end and so on
    (setf monitor-end (- (get-dsp-monitor) 4)) ;set up our "monitor" ready to load
    (emit `(DEFINE <pass-counter> ,(cadr (get-home-address '<pass-counter>))))
    (emit `(DEFINE <output+1> ,(cadr (get-home-address '<output+1>))))
    #+QP (QP-initialize-temps)
    (emit '(CLR A) '(LOAD B 0 SHORT))	;make sure pipeline settles (on ANDI that ends monitor -- this was a NOP)
    (emit '(.user-start))
    (if c56-user-start-break (emit '(JSR .break)))
    (load-chip-data)
    (if c56-load-chip-data-break (emit '(JSR .break)))
    ;; this has to happen before we try to load the library routines because otherwise .user-start
    ;; may end up in external memory, and the jump there (to load external memory) will go off into space.
    ;; We return to this point at the start of the next note, if there are two notes from the same instrument.
    (emit '(JMP .user-start-1))		;USER-START-1 is after library routines somewhere
    (load-flush-buffers <input> <output>)
    (if insigs (load-fill-buffers <input>))
    #+QP (if (or (member '.QP-initialize-DRAM library-load-list) 
		 (member '.delay library-load-list))
	     (QP-initialize-DRAM-load))	;must be in internal P memory (turns off ext mem while initing DRAM)
    (when (member '.env library-load-list)
      (setf library-load-list (delete '.env library-load-list))
      (if c56-need-to-load-expt
	  (progn
	    (expt-init)			;get Orig-A defined
	    (setf inited '(.expt-init))
	    (env-load t)
	    (need-loaded '.expt '.real-mpy))
	(env-load)))
    (load-library-routines library-load-list inited) ; get library loaded, data in x-memory and y-memory (from dsp56)
    (emit '(.user-start-1))
    (if c56-user-start-1-break (emit '(JSR .break)))
    (if insigs 
	(progn
	  (emit '(BSET M-HF2 X-IO M-HCR))
	  (emit '(JSR .fill-buffers))
	  (emit '(BCLR M-HF2 X-IO M-HCR))))
    
    #-(and excl (not Allegro-v3.1)) (DEBUGGING (pprint (append (list 'progn) (reverse p-init))))
    #+(and excl (not Allegro-v3.1)) (DEBUGGING (print (format nil "~{  ~A~%~}" (reverse p-init))))

    (if p-init				;some library routine needs to be initialized
	(call-emit (nreverse p-init)))

    (emit '(.pass-start))		;each pass after the first starts here
    (if c56-pass-start-break (emit '(JSR .break)))
    (emit '(JSSET 0 X <output+1> .flush-buffers))

    (if pp-init-sample (call-emit pp-init-sample))
    (DEBUGGING (setf saved-pp (reverse pp)))
    (call-emit (nreverse (optimize-parallel-moves (nreverse pp))))
    ;; moving this (the user-code portion of the per-sample computation) back into internal P memory
    ;; made no ascertainable difference in the compute times.

    (emit '(CLR B) '(LOAD A L <pass-counter>))
    (emit '(LOAD B0 1 SHORT))		;want long int add of 1, so have to CLR B above first
    (emit '(ADD B A) '(LOAD B L <run-end>))
    (emit '(STORE A L <pass-counter>))
    (emit '(CMP B A))

    (emil '(JLE .pass-start))		;end of dsp program
    (emit '(.pass-end))
    (emil '(JSR .flush-with-no-fill))	;flush last buffers
    (if (not insigs)			;undefined label floating around -- make it an error, in effect
	(emit '(.fill-buffers)))
    (emil '(JSR .break))
    (emit '(BCLR M-HF3 X-IO M-HCR))	;needed occasionally after RTS to restart instrument (JSR .break sets flag)
    (emit '(CLR A) '(LOAD B 0 SHORT))	;prepare for second note from this instrument (if any)
    (emit '(JMP .user-start))		;i.e. execute RTS in debugger starts next note 

    ;; now QP-DRAM cache management code.
    #+QP (QP-instrument-monitor-load <output>)
    (R-T-load)				;real-time controls, if ever desired (rt.lisp and rt.c)

    (setf-x-mem <clm-tag-addr> <clm-ins-tag>)
    (push `(setf-y-mem ,(1+ <output>) (count-if #'(lambda (n) (and (zerop (first n)) (/= (second n) -1))) *sigarr*)) pup)
    (set-up-basic-IO-arrays (nreverse insigs) (nreverse outsigs) no-output-here)

    (check-for-undefined-labels)
    (if (gethash 'random-seed names) (push `(setf-y-mem ,(gethash 'random-seed names) ran-seed) pup))

    (DEBUGGING (setf emit-prog (nreverse emit-prog)))
    (DEBUGGING (pprint (append '(case) emit-prog)))
    (DEBUGGING (pprint (append (list 'progn) (reverse pup))))
    (DEBUGGING (print (format nil "~S" (reverse pip))))

					;here is the new user-function body
    `(let* ((*run-beg* 0) 
	    (*run-end* 0)
	    (*data-memory-pc* ,heap-ptr)
	    (*version-number* (get *current-instrument-name* :dsp-version-number))
	    (*memory-map* (get *current-instrument-name* :dsp-memory-map))
	    (*ix-pc* ,ix-ptr) 
	    (*iy-pc* ,iy-ptr) 
	    (*ex-pc* ,ex-ptr) 
	    (*ey-pc* ,ey-ptr)
	    (*sigarr* ',pip))
       (c56-initialization-check t)
       (if (or (null *version-number*)
	       (/= dsp-version-number *version-number*))
	   (print (format nil "This instrument, ~A, needs to be recompiled (and reloaded) because ~
                               clm was changed on ~A in a way that is incompatible with it."
			  *current-instrument-name* *dsp-version-date*)))
       (if (/= external-Y-from-X-offset *memory-map*)
	   (progn
	     (print (format nil "~A needs to be recompiled and reloaded because its dsp memory map (~A) does not match the current map (~A)."
			    *current-instrument-name* (memory-map-name *memory-map*) (memory-map-name external-Y-from-X-offset)))
	     (force-output))
	 (progn
	   (set-initial-contents *current-instrument-name* *ix-pc* *iy-pc* *ex-pc* *ey-pc* *data-memory-pc*)
	   (locally (declare (optimize (speed 3) (safety 0)))
	     ,@(nreverse pup))
	   (multiple-value-setq 
	       (*data-memory-pc* *ix-pc* *iy-pc* *ex-pc* *ey-pc*)
	     (set-final-allocation-pointers))
	   (set-up-program-and-data *current-instrument-name*
				    *ix-pc* *iy-pc* *ex-pc* *ey-pc*
				    *run-beg* *run-end* (nreverse *sigarr*) 
				    ,<clm-ins-tag> ,<clm-tag-addr> ,monitor-end ,no-output-here))))))


(defun look-for-obvious-trouble (&optional name)
  (let* ((insname (or name *current-instrument-name*)))
    (if (not insname)
	(print "why ask for trouble?")
      (let* ((proglen (get name :dsp-compiled-program-length))
	     (version (get name :dsp-version-number)))
	(if (or (< proglen 100)
		(> proglen 511))
	    (print (format nil "program length ~A is impossible") proglen)
	  (if (/= version dsp-version-number)
	      (print (format nil "out of date dsp version -- try recompiling ~A" name))
#+Next	    (let* ((prog (get name :dsp-compiled-program)))
	      (if (> proglen 505)
		  (let ((jump-addr (loop for i from 505 below 511 
				     if (= (aref prog i) #xAF080)
				     return i)))
		    (if (not jump-addr)
			(print "no jump to external memory!")
		      (let* ((ext-addr (aref prog (1+ jump-addr))))
			(if (or (< ext-addr ext-P-offset)
				(> ext-addr (+ ext-P-offset external-L-size)))
			    (print "inserted jump to external memory jumps to ~A" ext-addr)))))))
           ))))))
			


(defun user-struct-field-type_56 (n)
  (or (and (listp n) (second n)) 'real))

(defun user-struct-field-size_56 (n)
  (structure-size_56 (user-struct-field-type_56 n)))

(defun user-struct-size_56 (x)
  (let ((desc (gethash x user-structs)))
    (loop for n in desc sum (user-struct-field-size_56 n))))

(defun user-struct-offset_56 (fld desc)
  (let ((sum 0))
    (loop for n in desc do
      (if (eq (if (listp n) (first n) n) fld)
	  (return-from user-struct-offset_56 sum)
	(incf sum (user-struct-field-size_56 n))))))

(defun pass-user-struct_56 (var addr type)
  (let ((desc (gethash (first var) user-structs))
	(adr addr))
    (if desc
	(loop for n in desc and i from 1 do
	  (let ((fld-typ (user-struct-field-type_56 n)))
	    (if (and (listp n) (eq (second n) 'array))
		(pass-array (nth i var) adr (or (third n) 'real))
	      (fpass-var adr (nth i var) fld-typ)))
	  (incf adr (user-struct-field-size_56 n)))
      (error "can't handle ~A of type ~A" var type))))

(defun display-user-struct_56 (type addr)
  ;; assume type makes sense here!
  (let ((desc (gethash type user-structs))
	(adr addr))
    (if desc
	(let ((new-struct (funcall (find-symbol (concatenate 'string "MAKE-" (symbol-name type))))))
	  (loop for n in desc and i from 1 do
	    (setf (nth i new-struct)
	      (get-dsp-var adr (user-struct-field-type_56 n) (and (listp n) (eq (second n) 'array) (or (third n) 'real))))
	    (incf adr (user-struct-field-size_56 n)))
	  new-struct)
      (warn "unknown type: ~A" type))))
