CLM Instruments (a sort of tutorial on building instruments)

This file contains a variety of "instruments" plus pointers to other
sources of information.  See clm.wn for an introduction to clm.  See
README.clm for some help with getting clm running.  I assume the
reader has a passing familiarity with lisp.

The instruments that follow are organized by the signal processing or
synthesis technique they use or refer to.  Some of these categories
are historical artifacts.  In brief, they are:

       additive synthesis       (sine + sine)
       waveshaping              (Chebychev polynomial driven with a sine)
       frequency modulation     (sine of sine)
         (amplitude modulation = sine*sine has not been found to be useful)
       reverberation            (using various filters and delays)
       subtractive synthesis    (filtering)
       karplus-strong synthesis (can be viewed in several ways = physical modelling)
       various signal processing tricks
       wave trains              (wave + wave -- a sort of extended additive synthesis)
          ("granular" or "wavetable" synthesis)
       physical modelling       (filters+delays+driving functions to mimic nature)
       spectral modelling       (additive synthesis after spectral analysis of natural sounds)
       spectral fusion          (using global vibrato streams to fuse voices)
       various examples of special case clm stuff
       sine summation synthesis, asymmetrical fm
       calling arbitrary Lisp or C functions
       known bugs and work-arounds

Many of the instruments in this document are translations of either
Mus-10 or Samson box instruments collected over the last 15 years
here at ccrma.  Instruments can be compiled to run either on a 56000
or on the main processor.  In this document, "[DSP]" signals that the
information applies only to the 56000 case, whereas "[C]" applies
only to the main processor case.
    


Common Lisp Music (we need a catchier name!) is a package of lisp, C,
and (currently) 56000 code that implements most of the common
synthesis and signal processing functions that have proven useful in
computer music.  The basic premise is that the composer writes normal
lisp code using our underlying functions to whatever degree he wants
-- this code then runs without further modification on any machine
that provides C and Common Lisp.  In the simplest case, we can just
calculate our new value, and add it into the current output:


(defun Simp (start-time duration frequency amplitude)
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate)))))
    (loop for i from beg to end and j from 0 by 1 do
      (outa i (* amplitude (sin (* j two-pi (/ frequency sampling-rate))))))))


Now to hear our sine wave:


 (with-sound () (simp 0 0.25 440.0 0.2))


This puts a sine-wave at 440.0 Hz, 0.2 amplitude, between times 0 and
0.25 seconds.  (OUTA i x) adds the value "x" into channel 1 of the
current output stream at location i.  The default output is a sound
file named "/zap/test.snd" (on the Nexts at ccrma).  SAMPLING-RATE
defaults to 22050.0 Hz on the Next.

WITH-SOUND opens the output sound file, runs the various instruments
called in its body, and plays the result.  It has a variety of options
documented in clm.wn.  Any lisp expression is legal within the body of
WITH-SOUND.  To hear the last sound again,

      (dac)

or, to hear some arbitrary sound file give the file's name as a string
as dac's argument.

      (dac "/zap/test.snd")

If you get an error (some NeXT programs don't obey their own sound
file protocols!), you can try to force the file to the DACs by setting
the optional second argument to nil.

      (dac "messed-up.snd" nil)

If you want to play the file several times, or start at some arbitrary
place in the file use DAC-N.  It takes the keyword arguments

      :FILE (defaults to the default output file)
      :START (defaults to 0.0)
      :WORRY (defaults to T -- if NIL, we try to ignore errors)
      :TIMES (defaults to 1 -- how many times to play the file)


Lisp compilers generate unbelievably bad numerical code, so we always
wrap up the "run-time" portion of our instrument in the macro RUN.  If
it can, it expands into code that either gets an available signal procesor
to speed up the work, or creates C code for the same purpose.
To take care of various system bookkeeping functions during this
process, we use DefInstrument, rather than Defun to define the 
instrument function. Its syntax can be considered to be identical to Defun.


(definstrument Simp (start-time duration frequency amplitude)
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let ((s (make-oscil :frequency frequency)))
      (Run				; this is our DSP compiler
       (loop for i from beg to end do 
	 (outa i (* amplitude (oscil s))))))))


OSCIL (from mus.lisp) keeps track of its current phase for you
(dependent on frequency) and returns a sine wave.  We then multiply it
by our desired amplitude.  We can use our previous note list here.
Let's put a real amplitude envelope on SIMP.


(definstrument Simp (start-time duration frequency amplitude 
		      &optional (amp-env '(0 0  50 1  100 0)))
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let ((s (make-oscil :frequency frequency))
	  (amp (make-env :envelope amp-env :scaler amplitude 
			 :start-time start-time :duration duration)))
      (Run 
       (loop for i from beg to end do
	 (outa i (* (env amp) (oscil s))))))))


Our amplitude envelope is a list of (x y) break-point pairs.  The
x-axis bounds are arbitrary, but it is conventional (here at ccrma) to
go from 0 to 100.  The y-axis values are normally between -1.0 and
1.0, so that it is easy to figure out how to apply the envelope in
various different situations.  In this case, our envelope is a ramp up
to the middle of the note, then a ramp down to 0.  The MAKE-ENV call
packages up the envelope information, pre-computing increments and so
on.  The envelope is then actually applied by calling ENV.

To get exponential connecting segments, rather than linear, just add
the desired exponent base to the MAKE-ENV call: 	 

(amp (make-env :envelope amp-env :scaler amplitude 
	       :start-time start-time :duration duration 
	       :base 3.0))) 

The higher the base, the more exaggerated the curve -- a base > 1.0
gives concave curves, < 1.0 convex.  For old Samson box users, that
synthesizer's exponential envelopes can be imitated by setting base to
.03125.  If base=0, you get a step function.  For exaggerated
exponential curves, draw the basic shape first with line segments,
then use the exponential envelope to connect the dots (the problem
here is that in 16 bits we quickly run out of room to make extreme
exponential curves).

What can we do with just a sine wave?  Well, we can imitate a
touch-tone telephone:


(setf touch-tab-1 '(0 697 697 697 770 770 770 852 852 852 941 941 941))
(setf touch-tab-2 '(0 1209 1336 1477 1209 1336 1477 1209 1336 1477 1209 1336 1477))
 
(definstrument touch-tone-telephone (telephone-number)
  (loop for k in telephone-number and 
            beg from 0 by (floor (* .4 sampling-rate)) do
    (let* ((i (if (integerp k)
		  (if (/= 0 k) k 11)
		(if (eq k '*) 10 12)))
	   (frq1 (make-oscil :frequency (nth i touch-tab-1)))
	   (frq2 (make-oscil :frequency (nth i touch-tab-2)))
	   (end (floor (+ beg (* .3 sampling-rate)))))
      (Run
       (loop for j from beg to end do
	 (outa j (* 0.25 (+ (oscil frq1) (oscil frq2)))))))))


Now (touch-tone-telephone '(7 2 3 4 9 7 1)) calls CCRMA.  I believe
that a dial tone is made up of sine waves at 349 and 440 Hz, and a
busy signal is a .8 second combination of 501 and 631 Hz.  The * key
is 1209 and 941Hz and # is 1633 and 941Hz.


---------------- ADDITIVE SYNTHESIS ---------------- 

We are now officially doing "additive synthesis" -- we are adding
together sine waves.  Additive synthesis is usually a last resort (the
mathematicians tell us nearly anything can be synthesized this way,
but don't bet the family farm on actually doing it).  A more
interesting example of sine-waves and additive synthesis is the
following bird song:


(definstrument bird (start-time duration frequency 
		     freq-skew amplitude Freq-envelope Amp-envelope)
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let* ((amp-env (make-env :envelope amp-envelope :scaler amplitude 
			      :start-time start-time :duration duration))
	   (gls-env (make-env :envelope freq-envelope
			      :scaler (* two-pi (/ freq-skew sampling-rate))
			      :start-time start-time :duration duration))
	   (s (make-oscil :frequency frequency)))
      (Run
       (loop for i from beg to end do
	 (outa i (* (env amp-env) (oscil s (env gls-env)))))))))


This is a sine wave with an amplitude envelope and a frequency
envelope (for glissando).  An example call might be:


(defun solitary-vireo (begin-time)
  (bird begin-time 0.4 1800 1200 .02 
	'(0.0 0.2  3.0 0.3
	  6.0 0.1  10.0 0.5
	  13.0 0.4  16.0 0.8
	  19.0 0.5  22.0 0.9
	  25.0 0.6  28.0 1.0
	  31.0 0.6  34.0 1.0		 
	  37.0 0.5  41.0 0.9
	  45.0 0.4  49.0 0.8
	  51.0 0.4  54.0 0.75
	  57.0 0.35  60.0 0.7
	  63.0 0.3  66.0 0.6
	  69.0 0.25  72.0 0.5
	  75.0 0.2  78.0 0.3
	  82.0 0.1  85.0 0.3
	  88.0 0.05 91.0 0.3
	  94.0 0.0  95.0 0.3
	  99.0 0.0  100.0 0.1)
	'(0 0  25 1  75 1  100 0)))

(with-sound () (solitary-vireo 0))


The file bird.clm has a variety of bird calls, mostly implemented with
sine waves.  The other instrument used there, named BigBird, also uses
additive synthesis.  If our basic wave form is the same throughout the
note, as is the case in BigBird, it is normally cheaper to load up a
table with that waveform and read it with Table-Lookup, rather than
calling SIN via OSCIL.


(definstrument BigBird (start-time duration frequency freq-skew
			amplitude Freq-envelope Amp-envelope Partials)
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let* ((waveform (load-synthesis-table partials (make-table)))
	   (s (make-table-lookup :frequency frequency 
				 :wave-table waveform))
	   (amp-env (make-env :envelope amp-envelope :scaler amplitude 
			      :start-time start-time :duration duration))
	   (gls-env (make-env :envelope freq-envelope
			      :scaler (in-Hz freq-skew) ;interpret amplitude as a frequency
			      :start-time start-time :duration duration)))
      (Run
       (loop for i from beg to end do
	 (outa i (* (env amp-env) (table-lookup s (env gls-env)))))))))


On the NeXT, it takes a lot of time to load the table, so you should
either save it and re-use it many times, or use the following version
of BigBird -- it does the same thing as the previous one, but produces
the waveform by setting up an array of oscils running on the chip:
	
 
(definstrument BigBird-1 (start-time duration frequency freq-skew amplitude 
			  &optional (Freq-env '(0 0 100 1)) 
			            (Ampl-env '(0 0 25 1 75 1 100 0)) 
			            (Partials '(1 .5 2 .25 3 .2)))
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let* ((size (/ (length partials) 2))
	   (amp-arr (make-array size :element-type 'short-float))
	   (frq-arr (make-array size :element-type 'short-float))
	   (osc-arr (make-array size :element-type 'osc))
	   (amp-env (make-env :envelope ampl-env :scaler amplitude 
			      :start-time start-time :duration duration))
	   (gls-env (make-env :envelope freq-env :scaler (in-Hz freq-skew)
			      :start-time start-time :duration duration))
	   (sum 0.0))
      (loop for i in (cdr partials) by #'cddr do (incf sum i))
      (loop for i from 0 below size do
	(setf (aref frq-arr i) (float (pop partials)))
	(setf (aref amp-arr i) (/ (float (pop partials)) sum))
	(setf (aref osc-arr i) (make-oscil :frequency (* frequency (aref frq-arr i)))))
      (Run
       (loop for i from beg to end do
	 (let ((sum 0.0)
	       (frqskw (env gls-env)))
	   (dotimes (k size)
	     (incf sum (* (aref amp-arr k) 
			  (oscil (aref osc-arr k) 
				 (* frqskw (aref frq-arr k))))))
	   (outa i (* (env amp-env) sum))))))))


We use parallel arrays here to pass data down to the chip because the
RUN macro doesn't support list operations like NTH -- maybe someday it
will.  We have to be sure the total amplitude turned out by the
individual partials does not exceed 1.0 (the SUM loop before the
instrument is run), because otherwise the location collecting the
partial outputs could overflow, giving us bogus results.

Yet another way to do this, perhaps the fastest in some cases, is to
use waveshaping.  In the instrument below, we evaluate the Chebychev
polynomial on the fly, rather than load it into a table (as in the
software version in mus.lisp).


(defun normalize-to-power-of-two (coeffs &optional (headroom 3))
  (let ((maxval (abs (aref coeffs 0)))
	(arrsiz (array-total-size coeffs))
	(scaler 0.0))
    (loop for i from 1 below arrsiz do
      (setf maxval (max maxval (abs (aref coeffs i)))))
    (setf maxval (expt 2 (+ headroom (ceiling (/ (log maxval) (log 2))))))
    (setf scaler (/ 1.0 maxval))
    (loop for i from 0 below arrsiz do
      (setf (aref coeffs i) (* scaler (aref coeffs i))))
    maxval))

(definstrument BigBird-2 (start duration frequency freqskew amplitude
		          &optional (Freq-env '(0 0 100 1)) 
			            (Ampl-env '(0 0 25 1 75 1 100 0)) 
			            (Partials '(1 .5 2 .25 3 .2)))
  (multiple-value-bind (beg end) (get-beg-end start duration)
    (let* ((gls-env (make-env :envelope freq-env :scaler (in-Hz freqskew)
			      :start-time start :duration duration))
	   (os (make-oscil :frequency frequency))
	   (coeffs (get-chebychev-coefficients (normalize-partials partials)))
	   (norm (normalize-to-power-of-two coeffs))  
	   (amp-env (make-env :envelope ampl-env :scaler (* norm amplitude) 
			      :start-time start :duration duration)))
      (Run
       (loop for i from beg to end do
	 (outa i (* (env amp-env) (polynomial coeffs (oscil os (env gls-env))))))))))


Of course, in waveshaping we can produce changing spectra by adding an
"index" envelope to scale the OSCIL output.  See below for another
waveshaping example.

Now we can call this using any of the BigBird calls in bird.lisp,
for example:


(defun scissor-tailed-flycatcher (begin-time)
  (let ((scissor '(0 0  40 1  60 1  100 0)))
    (bigbird-1 begin-time 0.05 1800 1800 .2 scissor 
	     '(0 0  25 1  75 1  100 0) 
	     '(1 .5  2 1  3 .5  4 .1  5 .01))))


To get distance effects from the birds, don't reverberate them -- it
will sound like you are in an aviary.  Instead, low-pass filter the
output (see below).

James A Moorer obtained spectra for a large range of notes from many
instruments.  spectr.clm has this information (these are steady-state
spectra, and therefore not likely to be convincing for instruments
whose spectra change during a note -- perhaps the piano spectra are
the best bet, but even here, a convincing piano simulation needs the
noise of the piano action and so on).

It probably helps many of these static spectra if we add vibrato --
this helps the spectra "fuse", sounding more natural.  Simple steady
vibrato (say with a sine wave) sounds silly, so we want to combine a
more or less periodic signal with some "jitter" to get a somewhat more
convincing vibrato.  So in the next instrument we use table-lookup to
read the spectrum, then randi and triangle-wave to create the vibrato
(there is an on-going discussion about what is the "best" periodic
vibrato, where it should be centered in terms of the base pitch, and
so on -- it appears that singers use a sinusoidal periodic vibrato,
but string players use one closer to a triangle.  In the examples
below, we place the vibrato so that it is centered on the base pitch).

(Both waveshaping and table-lookup assume the spectrum to be produced
is made up of partials that are an integer multiple of the
fundamental.  If you ask for something else, you'll probably get a
buzz.  Use "complete-add" given below, or some variant thereof, to get
arbitrary spectra).


(definstrument Spectrum (start-time duration frequency amplitude
		         &optional (Partials '(1 1 2 0.5))
			           (Amp-envelope '(0 0 50 1 100 0))
			           (vibrato-amplitude 0.005)
			           (vibrato-speed 5.0)
			           (degree 0.0)
			           (distance 1.0)
			           (reverb-amount 0.005))
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let* ((waveform (load-synthesis-table partials (make-table)))
	   (freq-in-Hz (in-Hz frequency))
	   (s (make-table-lookup :frequency frequency :wave-table waveform))
	   (amp-env (make-env :envelope amp-envelope :scaler amplitude 
			      :start-time start-time :duration duration))
	   (per-vib (make-triangle-wave :frequency vibrato-speed
					:amplitude (* vibrato-amplitude freq-in-Hz)))
	   (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	   (ran-vib (make-randi :frequency (+ vibrato-speed 1.0)
				:amplitude (* vibrato-amplitude freq-in-Hz))))
      (Run 
       (loop for i from beg to end do
	 (locsig loc i (* (env amp-env) 
			  (table-lookup s (+ (triangle-wave per-vib)
					     (randi ran-vib))))))))))


An example call:  

       (Spectrum 0 1 440.0 .1 p-a4   ;this is a piano at 440Hz taken from spectr.clm
         '(0.0 0.0 1.0 1.0 5.0 0.9 12.0 0.5 25.0 0.25 100.0 0.0))


The LOCSIG "unit generator" handles both signal placement (in mono or
stereo) and reverb input (if any).  In the default case, it is
equivalent to OUTA.  It is easy to add panning between channels and so
on -- simply put an envelope on the LOCSIG variables that scale the
amount going to outa (ascl), outb (bscl), and the reverberator (rscl).
A fancier (and far more convincing) illusion of movement would fix up
the initial phase delays, and use a stereo reverberator with correct
initial echoes.

The next obvious step is to interpolate between two tables (most
natural sounds involve changing spectra).  In our next instrument we
just add another table, and a function to control the interpolation.


(definstrument Two-tab (start-time duration frequency amplitude
		        &optional (Partial-1 '(1.0 1.0 2.0 0.5))
			          (partial-2 '(1.0 0.0 3.0 1.0))
			          (Amp-envelope '(0 0 50 1 100 0))
			          (interp-func '(0 1 100 0))
			          (vibrato-amplitude 0.005)
			          (vibrato-speed 5.0)
			          (degree 0.0)
			          (distance 1.0)
			          (reverb-amount 0.005))
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let* ((waveform-1 (load-synthesis-table partial-1 (make-table)))
	   (waveform-2 (load-synthesis-table partial-2 (make-table)))
	   (freq-in-Hz (in-Hz frequency))
	   (s-1 (make-table-lookup :frequency frequency :wave-table waveform-1))
	   (s-2 (make-table-lookup :frequency frequency :wave-table waveform-2))
	   (amp-env (make-env :envelope amp-envelope :scaler amplitude 
			      :start-time start-time :duration duration))
	   (interp-env (make-env :envelope interp-func :scaler 1.0 
				 :start-time start-time :duration duration))
	   (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	   (per-vib (make-triangle-wave :frequency vibrato-speed
					:amplitude (* vibrato-amplitude freq-in-Hz)))
	   (ran-vib (make-randi :frequency (+ vibrato-speed 1.0)
				:amplitude (* vibrato-amplitude freq-in-Hz))))
      (Run
       (loop for i from beg to end do
	 (let ((vib (+ (triangle-wave per-vib) 
		       (randi ran-vib)))
	       (intrp (env interp-env)))
	   (locsig loc i (* (env amp-env) 
			    (+ (* intrp (table-lookup s-1 vib))
			       (* (- 1.0 intrp) 
				  (table-lookup s-2 vib)))))))))))
						

We could use this to mimic moving formants:


(defvar formants 
    '((I 390 1990 2550)  (E 530 1840 2480)  (AE 660 1720 2410)
      (UH 520 1190 2390) (A 730 1090 2440)  (OW 570 840 2410)
      (U 440 1020 2240)  (OO 300 870 2240)  (ER 490 1350 1690)
      (W 300 610 2200)   (LL 380 880 2575)  (R 420 1300 1600)
      (Y 300 2200 3065)  (EE 260 3500 3800) (LH 280 1450 1600)
      (L 300 1300 3000)  (I2 350 2300 3340) (B 200 800 1750)
      (D 300 1700 2600)  (G 250 1350 2000)  (M 280 900 2200)
      (N 280 1700 2600)  (NG 280 2300 2750) (P 300 800 1750)
      (T 200 1700 2600)  (K 350 1350 2000)  (F 175 900 4400)
      (TH 200 1400 2200) (S 200 1300 2500)  (SH 200 1800 2000)
      (V 175 1100 2400)  (THE 200 1600 2200)(Z 200 1300 2500)
      (ZH 175 1800 2000) (ZZ 900 2400 3800) (VV 565 1045 2400)))
                   ;;formant center frequencies for a male speaker

(defun get-phoneme-data (phoneme form)
  (if (eql phoneme (caar form)) (cdar form)
    (get-phoneme-data phoneme (cdr form))))

(defun formant-table (center-freq amplitude carrier-freq)
  (let* ((harm (round center-freq carrier-freq))
	 (amps '(0.05  0.2  0.7  0.2  0.1))
	 (tab nil)
	 (partial 0))
    (loop for i from 0 below (length amps) do
      (setf partial (+ harm -2 i))
      (if (plusp partial)
	  (setf tab (append tab (list partial 
				      (* amplitude (nth i amps)))))))
    tab))

(defun make-formant-list (phoneme frequency)
  (let* ((phon (get-phoneme-data phoneme formants))
	 (amps '(0.8 0.15 0.05))
	 (tab nil))
    (loop for i in phon and j in amps do
      (setf tab (append tab (formant-table i j frequency))))
    tab))

(defun make-phoneme (begin-time duration frequency amplitude 
		     phoneme-1 phoneme-2
		     &optional (Amp-envelope '(0 0 50 1 100 0))
			       (interp-func '(0 1 100 0))
			       (vibrato-amplitude 0.025)
			       (vibrato-speed 6.0))
  (two-tab begin-time duration frequency amplitude
	   (make-formant-list phoneme-1 frequency)
	   (make-formant-list phoneme-2 frequency)
	   amp-envelope interp-func vibrato-amplitude vibrato-speed))


an example call: (make-phoneme 0 1.0 150 0.5 'OO 'ER)

More complicated additive synthesis involves separate amplitude
envelopes on all the partials.  The phase vocoder is based on this
approach.

The data here is taken from the cello tone analyzed by J.A. Moorer in
"Signal Processing Aspects of Computer Music: A Survey", Proceedings
of the IEEE vol 65 no 8 Aug 77.


(defvar amp-env-array (make-array 14))
(defvar frq-env-array (make-array 14))

(setf (aref amp-env-array 0) '(0 0 .038 0.126 .081 0.581 .122 0.675 .224 0.158 .344 0))
(setf (aref amp-env-array 1) '(0 0 .041 .041 .067 .115 .101 .143 .199 .015 .277 0 .344 0))
(setf (aref amp-env-array 2) '(0 0 .005 0 .066 .026 .083 .021 .106 .018 .127 .006 .212 0 .344 0))
(setf (aref amp-env-array 3) '(0 0 .008 0 .031 .01 .058 .018 .091 .019 .141 .005 .183 0 .344 0))
(setf (aref amp-env-array 4) '(0 0 .015 0 .052 .022 .072 .03 .088 .025 .126 .004 .138 0 .344 0))
(setf (aref amp-env-array 5) '(0 0 .023 0 .033 .006 .047 .007 .067 .006 .084 .003 .094 0 .344 0))
(setf (aref amp-env-array 6) '(0 0 .002 0 .018 .007 .038 .0001 .049 .009 .079 .004 .12 0 .344 0))
(setf (aref amp-env-array 7) '(0 0 .009 0 .029 .003 .063 .005 .097 .002 .117 .001 .133 0 .344 0))
(setf (aref amp-env-array 8) '(0 0 .013 0 .027 .0035 .044 .004 .061 .008 .101 .001 .109 0 .344 0))
(setf (aref amp-env-array 9) '(0 0 .012 0 .022 .0035 .055 .005 .074 .002 .094 .002 .101 0 .344 0))
(setf (aref amp-env-array 10) '(0 0 .019 0 .023 .001 .040 .002 .056 .002 .067 .001 .074 0 .344 0))
(setf (aref amp-env-array 11) '(0 0 .022 0 .029 .001 .038 .0015 .051 .001 .063 .0015 .072 0 .344 0))
(setf (aref amp-env-array 12) '(0 0 .019 0 .022 .0015 .024 .002 .038 .0005 .052 .0015 .058 0 .344 0))
(setf (aref amp-env-array 13) '(0 0 .02 0 .024 .0005 .033 .0005 .040 .001 .049 .0005 .056 0 .344 0))

(setf (aref frq-env-array 0) '(0 95 .006 299 .043 314 .086 312 .274 313 .344 314))
(setf (aref frq-env-array 1) '(0 339 .009 605 .036 629 .119 627 .244 629 .277 626 .278 0 .344 0))
(setf (aref frq-env-array 2) '(0 0 .004  0 .005 540 .009 915 .029 938 .083 949 .183 950 .212 946 .213 0 .344 0))
(setf (aref frq-env-array 3) '(0 0 .006 0 .008 780 .013 1225 .040 1258 .115 1259 .167 1265 .183 1255 .184 0 .344 0))
(setf (aref frq-env-array 4) '(0 0 .013 0 .015 1342 .019 1589 .023 1554 .099 1583 .133 1581 .138 1501 .140 0 .344 0))
(setf (aref frq-env-array 5) '(0 0 .022 0 .023 1344 .031 1874 .062 1903 .072 1903 .081 1899 .094 1771 .095 0 .344 0))
(setf (aref frq-env-array 6) '(0 0 .002 1470 .006 2121 .020 2188 .074 2193 .101 2214 .120 2193 .122 0 .344 0))
(setf (aref frq-env-array 7) '(0 0 .008 0 .009 1655 .013 2459 .026 2513 .049 2510 .080 2530 .133 2525 .134 0 .344 0))
(setf (aref frq-env-array 8) '(0 0 .012 0 .013 2464 .015 2704 .022 2815 .072 2845 .095 2841 .109 2803 .110 0 .344 0))
(setf (aref frq-env-array 9) '(0 0 .011 0 .012 2483 .015 3107 .045 3151 .079 3169 .095 3169 .101 3158 .102 0 .344 0))
(setf (aref frq-env-array 10) '(0 0 .018 0 .019 3131 .022 3463 .029 3438 .038 3459 .052 3467 .074 3451 .076 0 .344 0))
(setf (aref frq-env-array 11) '(0 0 .020 0 .022 3156 .024 3778 .029 3741 .034 3743 .051 3775 .072 3753 .073 0 .344 0))
(setf (aref frq-env-array 12) '(0 0 .018 0 .019 3278 .023 4078 .030 4063 .037 4087 .048 4092 .058 4079 .059 0 .344 0))
(setf (aref frq-env-array 13) '(0 0 .019 0 .020 3573 .022 4069 .029 4362 .040 4377 .047 4342 .056 4358 .058 0 .344 0))


(definstrument complete-add (begin-time duration amplitude 
			     amp-arr frq-arr)
  (let* ((beg (floor (* begin-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
  	 (siz (min (array-dimension amp-arr 0) 
		   (array-dimension frq-arr 0)))
	 (s (make-array siz :element-type 'osc))
	 (ae (make-array siz :element-type 'envelope))
	 (fe (make-array siz :element-type 'envelope)))
    (loop for i below siz do
      (setf (aref s i) (make-oscil :frequency 0.0)) ;dealt with directly by the frequency function
      (setf (aref ae i) (make-env :envelope (aref amp-arr i) :scaler amplitude 
				  :start-time begin-time :duration duration))
      (setf (aref fe i) (make-env :envelope (aref frq-arr i) :scaler (in-Hz 1.0) 
				  :start-time begin-time :duration duration)))
    (Run
     (loop for i from beg to end do
       (let ((sum 0.0))
	 (dotimes (j siz)
	   (incf sum (* (env (aref ae j)) (oscil (aref s j) (env (aref fe j))))))
	 (outa i sum))))))
				   

   
---------------- AMPLITUDE MODULATION ----------------

In amplitude modulation, as it is usually defined we add some constant
to one signal then multiply the result by another signal on a
sample-by-sample basis.  This is a slight improvement over "ring
modulation" which simply multiplies the two signals together but I
can't find an instrument using either synthesis method that any
composer has liked for any length of time.  The functions
Ring-modulation and Amplitude-modulation are provided in mus.lisp but
until someone can find a good use for them, we'll go on.


---------------- FREQUENCY MODULATION ----------------

In Frequency modulation (or its close cousin phase modulation) we
change ("modulate") the frequency so rapidly that the side-bands move
into the audio range and are perceived as harmonics.  (Vibrato is very
slow frequency modulation, and as you speed up the vibrato speed, you
eventually hear a complex tone rather than a warbling tone).  See
fm.wn for an introduction.


(definstrument Simple-FM (start-time duration frequency amplitude
		          &key (modulator-ratio 1.0)
		               (fm-index 1.0)
		               (fm-index-envelope '(0 0  50 1  100 0))
		               (amplitude-envelope '(0 0  50 1  100 0))
		               (degree 0.0) (distance 1.0) (reverb-amount 0.005))
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (carrier (make-oscil :frequency frequency))
	 (modulator (make-oscil :frequency (* modulator-ratio frequency)))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (amp-env (make-env :envelope amplitude-envelope :scaler amplitude 
			    :start-time start-time :duration duration))
	 (fm-env (make-env :envelope fm-index-envelope
			   :scaler (in-Hz (* fm-index modulator-ratio frequency))
			   :start-time start-time :duration duration)))
    (Run
     (loop for i from beg to end do
       (locsig loc i (* (env amp-env)
			(oscil carrier (* (env fm-env)
					  (oscil modulator)))))))))


John Chowning's article on FM gives a number of examples feeding data
to this instrument.  For example, a first pass at a brass sound:


    (with-sound ()
      (let ((brass-env '(0 0 1 1 2 .75 5 .6 6 0)))
	(simple-fm 0 .6 440 .25 
	         :fm-index 5.0 
	         :fm-index-envelope brass-env 
   	         :amplitude-envelope brass-env)))


All kinds of FM instruments have been developed over the years.  The
following are chosen because 1) they are known to have been used by
composers, 2) I happen to know about them, 3) they illustrate some of
the many possibilities.  First an FM-Insect (used for the frog-like
and cicada-like raspy sounds in the fifth movement of "Colony" -- in
the actual piece it was low pass filtered too):


(definstrument FM-insect (start-time duration frequency amplitude 
			  amp-env mod-freq mod-skew mod-freq-env 
			  mod-index mod-index-env fm-index fm-ratio
			  &key (degree 0.0) (distance 1.0) (reverb-amount 0.005))
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (carrier (make-oscil :frequency frequency))
	 (fm1-osc (make-oscil :frequency mod-freq))
	 (fm2-osc (make-oscil :frequency (* fm-ratio frequency)))
	 (ampf (make-env :envelope amp-env :scaler amplitude 
			 :start-time start-time :duration duration))
	 (indf (make-env :envelope mod-index-env :scaler (in-Hz mod-index)
			 :start-time start-time :duration duration))
	 (modfrqf (make-env :envelope mod-freq-env :scaler (in-Hz mod-skew)
			    :start-time start-time :duration duration))
	 (fm2-amp (in-Hz (* fm-index fm-ratio frequency))))
    (Run
     (loop for i from beg to end do
       (let* ((garble-in (* (env indf) (oscil fm1-osc (env modfrqf))))
	      (garble-out (* fm2-amp (oscil fm2-osc garble-in))))
	 (locsig loc i (* (env ampf) (oscil carrier (+ garble-out garble-in)))))))))


In the actual Fm-insect code, the &key parameter default values are
actually set according to the current values of the variables
Fm-insect-degree, Fm-insect-distance, and fm-insect-reverb-amount.
All &key parameters of the instruments provided on the clm area follow
this convention, but we have used the initial (default) variable
values in the code in this file to save space and make things easier
to follow.


(with-sound ()
  (let ((locust '(0 0 40 1 95 1 100 .5))  
	(Bug_hi '(0 1 25 .7 75 .78 100 1))  
	(amp '(0 0 25 1 75 .7 100 0)))
    (FM-Insect 0 1.699 4142.627 .015 amp 60 -16.707 locust 500.866 bug_hi .346 .500)
    (FM-Insect 0.195 .233 4126.284 .030 amp 60 -12.142 locust 649.490 bug_hi .407 .500)
    (FM-Insect 0.217 2.057 3930.258 .045 amp 60 -3.011 locust 562.087 bug_hi .591 .500)
    (FM-Insect 2.100 1.500 900.627 .06 amp 40 -16.707 locust 300.866 bug_hi .346 .500)
    (FM-Insect 3.000 1.500 900.627 .06 amp 40 -16.707 locust 300.866 bug_hi .046 .500)
    (FM-Insect 3.450 1.500 900.627 .09 amp 40 -16.707 locust 300.866 bug_hi .006 .500)
    (FM-Insect 3.950 1.500 900.627 .12 amp 40 -10.707 locust 300.866 bug_hi .346 .500)
    (FM-Insect 4.300 1.500 900.627 .09 amp 40 -20.707 locust 300.866 bug_hi .246 .500)))


Michael McNabb's FM-bell:


(definstrument FM-bell (start-time duration frequency amplitude amp-env index-env index
		        &key (degree 0.0) (distance 1.0) (reverb-amount 0.005))
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (fmInd1 (in-Hz (* 32.0 frequency)))
	 (fmInd2 (in-Hz (* 4.0 (- 8.0 (/ frequency 50.0)))))
	 (fmInd3 (* fmInd2 0.705 (- 1.4 (/ frequency 250.0))))  
	 ;;(no in-Hz because included in fmInd2)
	 (fmInd4 (in-Hz (* 32.0 (- 20 (/ frequency 20)))))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (mod1 (make-oscil :frequency (* frequency 2)))
	 (mod2 (make-oscil :frequency (* frequency 1.41)))
	 (mod3 (make-oscil :frequency (* frequency 2.82)))
	 (mod4 (make-oscil :frequency (* frequency 2.4)))
	 (car1 (make-oscil :frequency frequency))
	 (car2 (make-oscil :frequency frequency))
	 (car3 (make-oscil :frequency (* frequency 2.4)))
	 (indf (make-env :envelope index-env :scaler index 
			 :start-time start-time :duration duration))
	 (ampf (make-env :envelope amp-env :scaler amplitude 
			 :start-time start-time :duration duration)))
    (Run
     (loop for i from beg to end do
       (let ((fmenv (env indf)))
	 (locsig loc i 
		 (* (env ampf)
		    (+ (oscil car1 (* fmenv fmInd1 (oscil mod1)))
		       (* .15 (oscil car2 (* fmenv 
					     (+ (* fmInd2 (oscil mod2))
						(* fmInd3 (oscil mod3))))))
		       (* .15 (oscil car3 (* fmenv fmInd4 (oscil mod4))))))))))))


(with-sound ()
  (let ((fbell '(0 1 2 1.1 25 .75 75 .5 100 .2))
	(abell '(0 0 .1 1 10 .6 25 .3 50 .15 90 .1 100 0)))
    (FM-bell 0 5.000 233.046 .028 abell fbell .750)
    (FM-bell 5.912 2.000 205.641 .019 abell fbell .650)
    (FM-bell 6.085 5.000 207.152 .017 abell fbell .750)
    (FM-bell 6.785 7.000 205.641 .010 abell fbell .650)
    (FM-bell 15.000 .500  880    .060 abell fbell .500)
    (FM-bell 15.006 6.500 293.66 .1 abell fbell 0.500)
    (FM-bell 15.007 7.000 146.5  .1 abell fbell 1.000)
    (FM-bell 15.008 6.000 110    .1 abell fbell 1.000)
    (FM-bell 15.010 10.00 73.415 .028 abell fbell 0.500)
    (FM-bell 15.014 4.000 698.46 .068 abell fbell 0.500)))


Dexter Morrill's FM-trumpet (from CMJ feb 77 p51):


(definstrument FM-trumpet (start-time duration
			   &key (frq1 250) (frq2 1500) 
				(amp1 .5) (amp2 .1)
				(ampatt1 .03) (ampdec1 .35) (ampatt2 .03) (ampdec2 .3)
				(modfrq1 250) (modind11 0) 
				(modind12 2.66) (modfrq2 250) 
				(modind21 0) (modind22 1.8)
				(rvibamp .007) (rvibfrq 125) 
				(vibamp .007) (vibfrq 7)
				(vibatt .6) (vibdec .2) 
				(frqskw .03) (frqatt .06)
				(ampenv1 '(0 0  25 1  75 .9  100 0))
				(ampenv2 '(0 0  25 1  75 .9  100 0))
				(indenv1 '(0 0  25 1  75 .9  100 0))
				(indenv2 '(0 0  25 1  75 .9  100 0))
				(degree 0.0) (distance 1.0) (reverb-amount 0.005))
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (per-vib-f (make-env :envelope 
			       (divseg '(0 1  25 .1  75 0  100 0)
				       25 (* 100 (/ vibatt duration))
				       75 (* 100 (- 1.0 (/ vibdec duration))))
			      :scaler vibamp :start-time start-time :duration duration))
	 (ran-vib (make-randi :frequency rvibfrq :amplitude rvibamp))
	 (per-vib (make-oscil :frequency vibfrq))
	 (dec-01 (* 100 (- 1.0 (/ .01 duration))))
	 (frq-f (make-env :envelope (divseg '(0 0  25 1  75 1  100 0)
					     25 (* 100 (/ frqatt duration))
					     75 dec-01)
			  :scaler frqskw :start-time start-time :duration duration))
	 (ampattpt1 (* 100 (/ ampatt1 duration)))
	 (ampdecpt1 (* 100 (- 1.0 (/ ampdec1 duration))))
	 (ampattpt2 (* 100 (/ ampatt2 duration)))
	 (ampdecpt2 (* 100 (- 1.0 (/ ampdec2 duration))))
	 (frq-change 0.0)
	 (mod1-f (make-env :envelope (divseg indenv1 25 ampattpt1 75 dec-01)
			   :scaler (* modfrq1 (- modind12 modind11))
			   :start-time start-time :duration duration))
	 (mod1 (make-oscil :frequency 0.0))
	 (car1 (make-oscil :frequency 0.0)) ; set frequency to zero here because it is handled multiplicatively below
	 (car1-f (make-env :envelope (divseg ampenv1 25 ampattpt1 75 ampdecpt1)
			   :scaler amp1 :start-time start-time :duration duration))
	 (mod2-f (make-env :envelope (divseg indenv2 25 ampattpt2 75 dec-01)
			   :scaler (* modfrq2 (- modind22 modind21))
			   :start-time start-time :duration duration))
	 (mod2 (make-oscil :frequency 0.0))
	 (car2 (make-oscil :frequency 0.0))
	 (car2-f (make-env :envelope (divseg ampenv2 25 ampattpt2 75 ampdecpt2)
			   :scaler amp2 :start-time start-time :duration duration)))
    (Run
     (loop for i from beg to end do
       (setf frq-change (in-Hz (* (+ 1.0 (randi ran-vib))
				  (+ 1.0 (* (env per-vib-f) 
					    (oscil per-vib)))
				  (+ 1.0 (env frq-f)))))
       (locsig loc i 
	       (+ (* (env car1-f) 
		     (oscil car1 (* frq-change (+ frq1 (* (env mod1-f) 
							  (oscil mod1 (* modfrq1 frq-change)))))))
		  (* (env car2-f) 
		     (oscil car2 (* frq-change (+ frq2 (* (env mod2-f) 
							  (oscil mod2 (* modfrq2 frq-change)))))))))))))


DIVSEG scales breakpoint x-axis values to return a new envelope that
produces the desired "attack" and "decay" times.  Decay times aren't
usually vital. but attack times are.


The "noise" instrument (useful for Oceanic Music):


(defun attack-point (duration attack decay &optional (total-x 100.0))
  (* total-x (/ (if (= 0.0 attack)
		    (if (= 0.0 decay)
			(/ duration 4)
		      (/ (- duration decay) 4))
		  attack)
		duration)))

(definstrument FM-Noise (start-time duration freq0 amp ampfun 
			 ampat ampdc
			 freq1 glissfun freqat freqdc 
			 rfreq0 rfreq1 rfreqfun rfreqat rfreqdc
			 dev0 dev1 devfun devat devdc
			 &key (degree 0.0) (distance 1.0) (reverb-amount 0.005))
  
  ;; ampat = amp envelope attack time, and so on -- this instrument assumes
  ;; your envelopes go from 0 to 100 on the x-axis, and that the "attack"
  ;; portion ends at 25, the "decay" portion starts at 75.  "rfreq" is the
  ;; frequency of the random number generator -- if below about 25 Hz you
  ;; get automatic composition, above that you start to get noise.  Well,
  ;; you get a different kind of noise.  "dev" is the bandwidth of the
  ;; noise -- very narrow gives a whistle, very broad more of a whoosh.
  ;; This is "simple FM", but the modulating signal is white noise.
  
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (carrier (make-oscil :frequency freq0))
	 (modulator (make-randh :frequency rfreq0 :amplitude 1.0))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (dev-0 (in-Hz dev0))
	 
	 ;; next fix-up troubles in attack and decay times (there are lots of ways
	 ;; to handle this -- the basic problem is that these durned instruments
	 ;; end up having way too many parameters).  
	 
	 (amp-attack (attack-point duration ampat ampdc))
	 (amp-decay (- 100.0 (attack-point duration ampdc ampat)))
	 (freq-attack (attack-point duration freqat freqdc))
	 (freq-decay (- 100.0 (attack-point duration freqdc freqat)))
	 (dev-attack (attack-point duration devat devdc))
	 (dev-decay (- 100.0 (attack-point duration devdc devat)))
	 (rfreq-attack (attack-point duration rfreqat rfreqdc))
	 (rfreq-decay (- 100.0 (attack-point duration rfreqdc rfreqat)))
	 
	 ;; now make the actual envelopes -- these all assume we are thinking in
	 ;; terms of the "value when the envelope is 1" (i.e. dev1 and friends),
	 ;; and the "value when the envelope is 0" (i.e. dev0 and friends) -- over
	 ;; the years this seemed to make people happier than various other
	 ;; ways of describing the y-axis behaviour of the envelope.  All this
	 ;; boiler-plate for envelopes might seem overly elaborate when our basic
	 ;; instrument is really simple, but in most cases, and this one in
	 ;; particular, nearly all the musical interest comes from the envelopes,
	 ;; not the somewhat dull spectrum generated by the basic patch.
	 
	 (dev-f (make-env :envelope (divseg devfun 25 dev-attack 75 dev-decay) 
			  :scaler (in-Hz (- dev1 dev0)) :start-time start-time :duration duration))
	 (amp-f (make-env :envelope (divseg ampfun 25 amp-attack 75 amp-decay)
			  :scaler amp :start-time start-time :duration duration))
	 (freq-f (make-env :envelope (divseg glissfun 25 freq-attack 75 freq-decay)
			   :scaler (in-Hz (- freq1 freq0)) :start-time start-time :duration duration))
	 (rfreq-f (make-env :envelope (divseg rfreqfun 25 rfreq-attack 75 rfreq-decay)
			    :scaler (in-Hz (- rfreq1 rfreq0)) :start-time start-time :duration duration)))
    (Run
     (loop for i from beg to end do
       (locsig loc i (* (env amp-f)
			(oscil carrier (+ (env freq-f)
					  (* (+ dev-0 (env dev-f)) 
					     (randh modulator (env rfreq-f)))))))))))


(with-sound () (fm-noise 0 2.0 500 .25 
                         '(0 0 25 1 75 1 100 0) .1 .1 1000 
			 '(0 0 100 1) .1 .1 10 1000 
			 '(0 0 100 1) 0 0 100 500 '(0 0 100 1) 0 0))


Paul Wieneke's FM Gong:


(definstrument Gong (start-time duration frequency amplitude
		     &key (degree 0.0) (distance 1.0) (reverb-amount 0.005))
  (let* ((mfq1 (* frequency 1.16))
	 (mfq2 (* frequency 3.14))
	 (mfq3 (* frequency 1.005))
	 (indx01 (in-Hz (* .01 mfq1)))
	 (indx11 (in-Hz (* .30 mfq1)))
	 (indx02 (in-Hz (* .01 mfq2)))
	 (indx12 (in-Hz (* .38 mfq2)))
	 (indx03 (in-Hz (* .01 mfq3)))
	 (indx13 (in-Hz (* .50 mfq3)))
	 (atpt 5)
	 (atdur (* 100 (/ .002 duration)))
	 (EXPF '(0 0  3 1  15 .5  27 .25  50 .1  100 0))  
	 (RISE '(0 0  15 .3  30 1.0  75 .5  100 0))
	 (FMUP '(0 0  75 1.0  98 1.0  100 0))
	 (FMDWN '(0 0  2 1.0  100 0))
	 (ampfun (make-env :envelope (divseg EXPF atpt atdur)
			   :start-time start-time :duration duration :scaler Amplitude))
	 (indxfun1 (make-env :envelope FMUP
			     :start-time start-time :duration duration 
			     :scaler (- indx11 indx01) :offset indx01))
	 (indxfun2 (make-env :envelope FMDWN
			     :start-time start-time :duration duration 
			     :scaler (- indx12 indx02) :offset indx02))
	 (indxfun3 (make-env :envelope RISE
			     :start-time start-time :duration duration 
			     :scaler (- indx13 indx03) :offset indx03))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (carrier (make-oscil :frequency frequency))
	 (mod1 (make-oscil :frequency mfq1))
	 (mod2 (make-oscil :frequency mfq2))
	 (mod3 (make-oscil :frequency mfq3))
	 (beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate)))))
    (Run
     (loop for i from beg to end do
       (locsig loc i (* (env ampfun) 
			(oscil carrier (+ (* (env indxfun1) (oscil mod1))
					  (* (env indxfun2) (oscil mod2))
					  (* (env indxfun3) (oscil mod3))))))))))


(with-sound () (Gong 0 3 261.61 .6))


Jan Mattox's FM-drum (using "cascade fm"):


Cascade FM is a bit tricky if you are trying to get pitched sounds
because any 0Hz component in the upper FM pair becomes a constant
frequency offset in the carrier, creating inharmonic spectra or other
problems.

	
(definstrument FM-Drum (start-time duration frequency amplitude index 
			&optional (high nil) (degree 0.0) (distance 1.0) (reverb-amount 0.01))
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 ;; many of the following variables were originally passed as arguments
	 (casrat (if high 8.525 3.515))
	 (fmrat (if high 3.414 1.414))
	 (GLSFUN '(0 0  25 0  75 1  100 1))
	 (glsf (make-env :envelope GLSFUN 
			 :scaler (if high (in-Hz 66) 0.0) 
			 :start-time start-time :duration duration))
	 (AMPFUN '(0 0  3 .05  5 .2  7 .8  8 .95  10 1.0  12 .95  20 .3  30 .1  100 0))
	 (atdrpt (* 100 (/ (if high .01 .015) duration)))
	 (ampf (make-env :envelope (divseg AMPFUN 
					   10 atdrpt 
					   15 (max (+ atdrpt 1) 
						   (- 100 (* 100 (/ (- duration .2) duration)))))
		  :scaler amplitude :start-time start-time :duration duration))
	 (INDXFUN '(0  0     5  .014  10 .033  15 .061  20 .099  
		    25 .153  30 .228  35 .332  40 .477  
		    45 .681  50 .964  55 .681  60 .478  65 .332  
		    70 .228  75 .153  80 .099  85 .061  
		    90 .033  95 .0141 100 0))
	 (indxpt (- 100 (* 100 (/ (- duration .1) duration))))
	 (divindxf (divseg INDXFUN 50 atdrpt 65 indxpt))
	 (indxf (make-env :envelope divindxf 
			  :scaler (min (in-Hz (* index fmrat frequency)) pi)
			  :start-time start-time :duration duration))
	 (mindxf (make-env :envelope divindxf 
			   :scaler (min (in-Hz (* index casrat frequency)) pi)
			   :start-time start-time :duration duration))
	 (devf (make-env :envelope (divseg AMPFUN 
					   10 atdrpt 
					   90 (max (+ atdrpt 1) 
						   (- 100 (* 100 (/ (- duration .05) duration)))))
			 :scaler (min pi (in-Hz 7000)) 
			 :start-time start-time :duration duration))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (rn (make-randh :frequency 7000 :amplitude 1.0))
	 (carrier (make-oscil :frequency frequency))
	 (fmosc (make-oscil :frequency (* frequency fmrat)))
	 (cascade (make-oscil :frequency (* frequency casrat))))
    (Run
     (loop for i from beg to end do
       (let ((gls (env glsf)))
	 (locsig loc i (* (env ampf) 
			  (oscil carrier 
				 (+ gls 
				    (* (env indxf)
				       (oscil fmosc 
					      (+ (* gls fmrat)
						 (* (env mindxf) (oscil cascade 
									(+ (* gls casrat)
									   (* (env devf) (randh rn)))))))))))))))))
			
(with-sound ()
  (FM-Drum 0 1.5 55 .3 5 nil)
  (FM-Drum 2 1.5 66 .3 4 t))


John Chowning's FM-voice :


(definstrument Reson (start-time duration pitch amp numformants 
		      indxFun skewFun pcSkew SkewAt SkewDc
		      vibfreq vibpc ranvibfreq ranvibpc degree 
		      distance reverb-amount &rest data)
  ;; data is a list of lists of form '(ampf resonfrq resonamp ampat ampdc dev0 dev1 indxat indxdc)
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (carriers (make-array numFormants :element-type 'osc))
	 (modulator (make-oscil :frequency pitch))
	 (ampfs (make-array numFormants :element-type 'envelope))
	 (indfs (make-array numFormants :element-type 'envelope))
	 (c-rats (make-array numFormants :element-type 'fixnum))
	 (frqf (make-env :envelope (divseg skewfun 25 
					   (* 100 (/ skewat duration)) 
					   75 (- 100 (* 100 (/ Skewdc duration))))
		:scaler (in-Hz (* pcSkew pitch))
		:start-time start-time :duration duration))
	 (vib 0.0)
	 (outsum 0.0)
	 (totalAmp 0.0)
	 (modsig 0.0)
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (pervib (make-triangle-wave :frequency vibfreq :amplitude (in-Hz (* vibpc pitch))))
	 (ranvib (make-randi :frequency ranvibfreq :amplitude (in-Hz (* ranvibpc pitch)))))

    ;; initialize the "formant" generators
    (loop for i from 0 below numFormants do 
      (incf totalAmp (nth 2 (nth i data))))
    (loop for i from 0 below numFormants do
      (let* ((frmdat (nth i data))
	     (Freq (nth 1 frmdat))
	     (ampf (nth 0 frmdat))
	     (rfamp  (nth 2 frmdat))
	     (ampat (* 100 (/ (nth 3 frmdat) duration)))
	     (ampdc (- 100 (* 100 (/ (nth 4 frmdat) duration))))
	     (dev0 (in-Hz (* (nth 5 frmdat) freq)))
	     (dev1 (in-Hz (* (nth 6 frmdat) freq)))
	     (indxat (* 100 (/ (nth 7 frmdat) duration)))
	     (indxdc (- 100 (* 100 (/ (nth 8 frmdat) duration))))
	     (harm (round Freq pitch))
	     (rsamp (- 1.0 (abs (- harm (/ freq pitch)))))
	     (cfq (* pitch harm)))
	(if (zerop ampat) (setf ampat 25))
	(if (zerop ampdc) (setf ampdc 75))
	(if (zerop indxat) (setf indxat 25))
	(if (zerop indxdc) (setf indxdc 75))
	(setf (aref indfs i) 
	  (make-env :envelope (divseg indxFun 25 indxat 75 indxdc)
		    :scaler (- dev1 dev0) :offset dev0
		    :start-time start-time :duration duration))
	(setf (aref ampfs i) 
	  (make-env :envelope (divseg ampF 25 ampat 75 ampdc)
		    :scaler (* rsamp amp (/ rfamp totalAmp))
		    :start-time start-time :duration duration))
	(setf (aref c-rats i) harm)
	(setf (aref carriers i) (make-oscil :frequency cfq))))
    (Run
     (loop for i from beg to end do 
       (setf outsum 0.0)
       (setf vib (+ (triangle-wave pervib) (randi ranvib) (env frqf)))
       (setf modsig (oscil modulator vib))
       (dotimes (k numFormants)
	 (incf outsum (* (env (aref ampfs k)) 
			 (oscil (aref carriers k) 
				(+ (* vib (aref c-rats k))
				   (* (env (aref indfs k)) modsig))))))
       (locsig loc i outsum)))))


(with-sound () 
  (reson 0 1.0 440 .1 2 '(0 0 100 1) 
	 '(0 0 100 1) .1 .1 .1 5 .01 5 .01 0 1.0 0.01
         '((0 0 50 1 100 0) 1200 .5 .1 .1 0 1.0 .1 .1)
         '((0 0 50 1 100 0) 2400 .5 .1 .1 0 1.0 .1 .1)))


A more sophisticated version of Reson that tries to imitate a singing
voice can be found in jcvoi.lisp.  The following instrument uses a
similar idea, but adds moving formants.  We use the "formants" data
given above.  This instrument originally used waveshaping, rather than
FM (it was built by MLB in 1977 and used in a variety of pieces --
"SandCastle", "The New Music Liberation Army" etc).  See mlbvoi.lisp
for a slightly simpler version.


(defun find-phoneme (phoneme form)
  (loop for i in form do
    (if (eq phoneme (first i))
	(return-from find-phoneme (rest i)))))

(defun vox-fun (phons which)
  (let ((phdata nil)
	(f1 nil))
    (loop for x in phons by #'cddr and
              phoneme in (cdr phons) by #'cddr do
      (setf phdata (find-phoneme phoneme formants))
      ;; if you want more than 3 formants, remake the formants list
      (push x f1)
      (push (nth which phdata) f1))
    (nreverse f1)))

(definstrument vox (beg dur freq amp ampfun freqfun freqscl 
		    phonemes formant-amps formant-indices 
		    &optional (vibscl .1) (deg 0) (pcrev 0))
  (let* ((start (floor (* sampling-rate beg)))
	 (end (+ start (floor (* sampling-rate dur))))
	 (car-os (make-oscil :frequency 0))
	 (fs (length formant-amps))
	 (evens (make-array fs :element-type 'osc))
	 (odds (make-array fs :element-type 'osc))
	 (amps (make-array fs :element-type 'short-float))
	 (indices (make-array fs :element-type 'short-float))
	 (frmfs (make-array fs :element-type 'envelope))
	 (ampf (make-env :envelope ampfun 
			 :scaler amp :start-time beg :duration dur))
	 (freqf (make-env :envelope freqfun
			  :scaler (* freqscl freq) :offset freq
			  :start-time beg :duration dur))
	 (frq 0.0)
	 (car 0.0)
	 (frm 0.0)
	 (frm-int 0)
	 (frm0 0.0)
	 (even-amp 0.0)
	 (odd-amp 0.0)
	 (even-freq 0.0)
	 (odd-freq 0.0)
	 (sum 0.0)
	 (loc (make-locsig :degree deg :distance 1.0 :revscale pcrev))
	 (per-vib (make-triangle-wave :frequency 6 :amplitude (* freq vibscl)))
	 (ran-vib (make-randi :frequency 20 :amplitude (* freq .5 vibscl))))
    (loop for i from 0 below fs and
              amp in formant-amps and
              index in formant-indices do
      (setf (aref evens i) (make-oscil :frequency 0))
      (setf (aref odds i) (make-oscil :frequency 0))
      (setf (aref amps i) amp)
      (setf (aref indices i) index)
      (setf (aref frmfs i) 
	(make-env :envelope (vox-fun phonemes i)
		  :scaler 1.0 :offset 0.0 :start-time beg :duration dur)))
    (Run
     (loop for i from start to end do
       (setf frq (+ (env freqf) 
		    (triangle-wave per-vib) 
		    (randi ran-vib)))
       (setf car (oscil car-os (in-Hz frq)))
       (setf sum 0.0)
       (dotimes (k fs)
	 (setf frm (env (aref frmfs k)))
	 (setf frm0 (/ frm frq))
	 (setf frm-int (floor frm0))
	 (if (evenp frm-int)
	     (progn
	       (setf even-freq (in-Hz (* frm-int frq)))
	       (setf odd-freq (in-Hz (* (+ frm-int 1) frq)))
	       (setf odd-amp (- frm0 frm-int))
	       (setf even-amp (- 1.0 odd-amp)))
	   (progn
	     (setf odd-freq (in-Hz (* frm-int frq)))
	     (setf even-freq (in-Hz (* (+ frm-int 1) frq)))
	     (setf even-amp (- frm0 frm-int))
	     (setf odd-amp (- 1.0 even-amp))))
	 (incf sum (+ (* (aref amps k) 
			 (+ (* even-amp 
			       (oscil (aref evens k) 
				      (+ even-freq 
					 (* (aref indices k) car))))
			    (* odd-amp 
			       (oscil (aref odds k) 
				      (+ odd-freq 
					 (* (aref indices k) car)))))))))
       (locsig loc i (* (env ampf) sum))))))


(with-sound () 
  (vox 0 2 110 .4 '(0 0 25 1 75 1 100 0) 
                  '(0 0 5 .5 10 0 100 1) .1 
                  '(0 UH 25 UH 35 ER 65 ER 75 UH 100 UH) 
                  '(.8 .15 .05) 
		  '(.005 .0125 .025))
  (vox 2 2 160 .4 '(0 0 25 1 75 1 100 0) 
                  '(0 0 5 .5 10 0 100 1) .1 
		  '(0 UH 25 UH 35 ER 65 ER 75 UH 100 UH) 
		  '(.85 .10 .05) 
		  '(.005 .0125 .07))
  (vox 4 2 460 .4 '(0 0 25 1 75 1 100 0) 
                  '(0 0 5 .5 10 0 100 1) .1 
                  '(0 UH 25 UH 35 ER 65 ER 75 UH 100 UH) 
                  '(.85 .1 .05) 
                  '(.005 .0125 .02))
  (vox 6 2 560 .4 '(0 0 25 1 75 1 100 0) 
                  '(0 0 5 .5 10 0 100 1) .1 
                  '(0 OW 25 OW 35 OO 65 OO 75 OW 100 OO) 
                  '(.25 .7 .05) 
                  '(.1 .0125 .02)))


(see also pqwvox)


Fm Violin:


(definstrument FM-violin 
	       (start-time duration frequency amplitude
	       &key (fm-index 1.0)
		    (amp-env '(0 0  25 1  75 1  100 0))
		    (periodic-vibrato-rate 5.0)
		    (random-vibrato-rate 16.0)
		    (periodic-vibrato-amplitude 0.005)
		    (random-vibrato-amplitude 0.005)
		    (noise-amount 0.0)
		    (noise-freq 1000.0)
		    (gliss-env '(0 0  100 0))
		    (glissando-amount 0.0)
		    (fm1-env '(0 1  25 .4  75 .6  100 0))
		    (fm2-env fm1-env)
		    (fm3-env fm1-env)
		    (fm1-rat 1.0)
		    (fm2-rat 3.0)
		    (fm3-rat 4.0)
		    (base nil)
		    (reverb-amount 0.01)
		    (noise-type 'Sambox-random)
		    (index-type 'violin)
		    (degree 0.0)
		    (distance 1.0)
		&allow-other-keys)
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (frq-scl (in-Hz frequency))
	 (maxdev (* frq-scl fm-index))
	 (vln (not (eq index-type 'cello)))
	 (easy-case (and (zerop noise-amount)
			 (eq fm1-env fm2-env)
			 (eq fm1-env fm3-env)
			 (zerop (- fm1-rat (floor fm1-rat)))
			 (zerop (- fm2-rat (floor fm2-rat)))
			 (zerop (- fm3-rat (floor fm3-rat)))))
	 (index1 (min one-pi (* maxdev (/ (if vln 5.0 7.5) (log frequency)))))
	 (index2 (min one-pi 
		      (* maxdev 3.0 
			 (if vln 
			     (/ (- 8.5 (log frequency)) (+ 3.0 (/ frequency 1000)))
			   (/ 15.0 (sqrt frequency))))))
	 (index3 (min one-pi (* maxdev (/ (if vln 4.0 8.0) (sqrt frequency)))))
	 (coeffs (and easy-case 
		      (get-chebychev-coefficients 
		       (list fm1-rat index1 fm2-rat index2 fm3-rat index3))))
	 (norm (or (and easy-case 
			(normalize-to-power-of-two coeffs 2)) 
		   index1))
	 (carrier (make-oscil :frequency frequency))
	 (fmosc1  (make-oscil :frequency (* fm1-rat frequency)))
	 (fmosc2  (or easy-case (make-oscil :frequency (* fm2-rat frequency))))
	 (fmosc3  (or easy-case (make-oscil :frequency (* fm3-rat frequency))))
	 (ampf  (make-env :envelope amp-env :base base
			  :start-time start-time :duration duration 
			  :scaler amplitude))
	 (indf1 (make-env :envelope fm1-env 
			  :start-time start-time :duration duration 
			  :scaler norm))
	 (indf2 (or easy-case
		    (make-env :envelope fm2-env 
			      :start-time start-time :duration duration 
			      :scaler index2)))
	 (indf3 (or easy-case 
		    (make-env :envelope fm3-env 
			      :start-time start-time :duration duration 
			      :scaler index3)))
	 (frqf (make-env :envelope gliss-env
			 :start-time start-time :duration duration 
			 :scaler (* glissando-amount frq-scl)))
	 (pervib (make-triangle-wave 
		  :frequency periodic-vibrato-rate 
		  :amplitude (* periodic-vibrato-amplitude frq-scl)))
	 (ranvib (make-randi 
		  :frequency random-vibrato-rate 
		  :amplitude (* random-vibrato-amplitude frq-scl)))
	 (fm-noi (if (/= 0.0 noise-amount)
		     (make-randh :frequency noise-freq
				 :amplitude (* one-pi noise-amount)
				 :type noise-type)
		   0.0))
	 (vib 0.0) 
	 (modulation 0.0)
	 (loc (make-locsig :degree degree :revscale reverb-amount :distance distance))
	 (fuzz 0.0))
    (Run
     (loop for i from beg to end do
       (if (/= 0.0 noise-amount)
	   (setf fuzz (randh fm-noi)))
       (setf vib (+ (env frqf) (triangle-wave pervib) (randi ranvib)))
       (if easy-case
	   (setf modulation
	     (* (env indf1) (polynomial coeffs (oscil fmosc1 vib))))
	 (setf modulation
	   (+ (* (env indf1) (oscil fmosc1 (+ (* fm1-rat vib) fuzz)))
	      (* (env indf2) (oscil fmosc2 (+ (* fm2-rat vib) fuzz)))
	      (* (env indf3) (oscil fmosc3 (+ (* fm3-rat vib) fuzz))))))
       (locsig loc i
	     (* (env ampf) (oscil carrier (+ vib modulation))))))))


fmviolin.clm contains a large number of examples of the fm violin in action.  


See also canter.lisp and drone.lisp in conjunction with bag.clm
for Peter Common's bagpipe simulation.



---------------- REVERBERATION ----------------

All reverberators run much faster in C than on the DSP, even when you have
plenty of DSP memory.


JC-style reverb:


(definstrument JCreverb (start-time duration)	
  (let* ((allpass1 (make-all-pass 0.700 -0.700 1051))
	 (allpass2 (make-all-pass 0.700 -0.700  337))
	 (allpass3 (make-all-pass 0.700 -0.700  113))
	 (comb1 (make-comb 0.742 4799))
	 (comb2 (make-comb 0.733 4999))
	 (comb3 (make-comb 0.715 5399))
	 (comb4 (make-comb 0.697 5801))
	 (outdel1 (make-delay (* 0.013 sampling-rate)))
	 (outdel2 (if (stereo) (make-delay (* 0.011 sampling-rate))))
	 (allpass-sum 0.0)
	 (comb-sum 0.0)
	 (chan2 (stereo))
	 (beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate)))))
    (Run
     (loop for i from beg to end do
       (setf allpass-sum (all-pass allpass3
				   (all-pass allpass2
					     (all-pass allpass1 
						       (revin i)))))
       (setf comb-sum (+ (comb comb1 allpass-sum)
			 (comb comb2 allpass-sum)
			 (comb comb3 allpass-sum)
			 (comb comb4 allpass-sum)))
       ;; In the version I use, I low pass filter the output here
       (outa i (delay outdel1 comb-sum))
       (if chan2 (outb i (delay outdel2 comb-sum)))))
    (End-Run)))
 
 
[DSP] This instrument ends with a call to END-RUN (after the loop has
finished) -- it is best if all instruments include this, but only
those that allocate delay lines must have it (these include any that
use comb filters or all pass filters, as well as straight delay
lines).
[C] End-run is a no-op in the C case.


NREV (the most popular Samson box reverb):


(defun prime (val)
  (or (= val 2)
      (and (oddp val)
	   (do ((i 3 (+ i 2))
		(lim (sqrt val)))
	       ((or (= 0 (mod val i)) (> i lim))
		(> i lim))))))
       
(definstrument nrev (start-time duration
		     &key (reverb-factor 1.09) (lp-coeff .7) (output-scale 1.0))
  (let* ((srscale (/ sampling-rate 25641))
	 (val 0)
	 (dly-len (make-array 15 :element-type 'fixnum 
			      :initial-contents '(19 29 37 43 53 59 37 113 347 2399 2251 2053 1867 1601 1433))))
    (loop for i below 15 do
      (setf val (floor (* srscale (aref dly-len i))))
      (if (= 0 (mod val 2)) (incf val))
      (loop while (not (prime val)) do (incf val 2))
      (setf (aref dly-len i) val))
    (let* ((comb1 (make-comb (* .822 reverb-factor) (aref dly-len 0)))
	   (comb2 (make-comb (* .802 reverb-factor) (aref dly-len 1)))
	   (comb3 (make-comb (* .773 reverb-factor) (aref dly-len 2)))
	   (comb4 (make-comb (* .753 reverb-factor) (aref dly-len 3)))
	   (comb5 (make-comb (* .753 reverb-factor) (aref dly-len 4)))
	   (comb6 (make-comb (* .733 reverb-factor) (aref dly-len 5)))
	   (low (make-one-pole lp-coeff (- lp-coeff 1.0)))
	   (allpass1 (make-all-pass 0.700 -0.700 (aref dly-len 6)))
	   (allpass2 (make-all-pass 0.700 -0.700 (aref dly-len 7)))
	   (allpass3 (make-all-pass 0.700 -0.700 (aref dly-len 8)))
	   (allpass4 (make-all-pass 0.700 -0.700 (aref dly-len 9)))
	   (allpass6 (make-all-pass 0.700 -0.700 (aref dly-len 11)))
	   (revin 0.0)
	   (ina-val 0.0)
	   (beg (floor (* start-time sampling-rate)))
	   (end (+ beg (floor (* duration sampling-rate)))))
      (Run
       (loop for i from beg to end do
	 (setf ina-val (ina i))
	 (setf revin (* ina-val .1))
	 (outa i (+ ina-val
	       (* output-scale 
		  (all-pass allpass6 
			    (all-pass allpass4
				      (one-pole low
						(all-pass allpass3
							  (all-pass allpass2
								    (all-pass allpass1
									      (+ (comb comb1 revin)
										 (comb comb2 revin)
										 (comb comb3 revin)
										 (comb comb4 revin)
										 (comb comb5 revin)
										 (comb comb6 revin)))))))))))))
      (end-run))))


KIPREV (Kip Sheeline's reverberator):

Originally intended to run at sampling rate of 25600 -- I don't know
how much that matters.  On the Next, this reverberator is somewhat
slow (ca. 10:1 compute ratio) because 18 of the delay lines are
handled in 68000 memory, shuffling data through the host interface,
and that is a major bottleneck on the 56000. If an Ariel QP card is
available, the run-time system automatically uses the QP master and
its DRAM, so in that case, the reverberators run a lot faster.


(definstrument KipRev (start-time duration 
		       &optional (DGain .95) (Lpfbg .45) (atten .7) (outscale 10.0))
  (let* ((beg (floor (* start-time sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (combfactors (make-array 17 :element-type 'short-float 
				  :initial-contents '(.841 .504 .491 .379 
						      .380 .346 .289 .272 .192
						      .193 .217 .181 .180 .181 
						      .176 .142 .167)))
	 (combdelays (make-array 17 :element-type 'fixnum
				 :initial-contents '(109 547 577 683 691 761 
						     1171 1237 1459
						     1499 1523 1567 1811 
						     1861 1901 1931 2039)))
	 (combs (make-array 17 :element-type 'cmbflt))
	 (tmpr (make-array 22 :element-type 'short-float :initial-element 0.0))
	 (syncit (make-delay 652))
	 (fbfact (make-array 6 :element-type 'short-float
			     :initial-contents '(.05797 .051759 .04751 
						 .042625 .040257 .037160)))
	 (delays (make-array 6 :element-type 'dly))
	 (delaysizes (make-array 6 :element-type 'fixnum
				 :initial-contents '(1279 1433 1559 1741 1847 1997)))
	 (filts (make-array 6 :element-type 'smpflt))
	 (allpasses (make-array 4 :element-type 'allpassflt))
	 (allpassdlys (make-array 4 :element-type 'fixnum :initial-contents '(53 41 37 23)))
	 (loopg (- 1.0 (abs Lpfbg)))
	 (tapout 0.0)
	 (k1 0) (k2 0) (k3 0) (k4 0)
	 (chan2 (stereo))
	 (outval 0.0)
	 (cmbin 0.0)
	 (outsm 0.0))
    ;; initialization of filters, delays, and so on
    (loop for i from 0 to 16 do 
      (setf (aref combs i) (make-comb (aref combfactors i) (aref combdelays i))))
    (loop for i from 0 to 5 do 
      (setf (aref delays i) (make-delay (aref delaysizes i)))
      (setf (aref filts i) (make-one-pole 1.0 lpfbg)))
    (loop for i from 0 to 3 do 
      (setf (aref allpasses i) (make-all-pass .7 -.7 (aref allpassdlys i))))
    (run
     (loop for i from beg to end do
       (setf outval (revin i))
       (setf tapout 0.0)
       (dotimes (k 17) 
	 (setf k1 k) 
	 (incf tapout (comb (aref combs k) outval)))
       (setf cmbin (delay syncit tapout))
       (do ((k 0 (+ k 1))
	    (j 0 (+ j 3)))
	   ((> k 5))
	 (setf k2 k) (setf k3 j)
	 (setf (aref tmpr j) 
	   (+ (* loopg dgain (aref tmpr (+ j 2))) 
	      (* cmbin atten (aref fbfact k))))
	 (setf (aref tmpr (+ j 1)) 
	   (delay (aref delays k) (aref tmpr j)))
	 (setf (aref tmpr (+ j 2)) 
	   (one-pole (aref filts k) (aref tmpr (+ j 1)))))
       (setf outsm 0.0)
       (do ((k 1 (+ k 3)))
	   ((> k 16))
	 (setf k4 k)
	 (incf outsm (aref tmpr k)))
       (dotimes (k 4) 
	 (setf outsm (all-pass (aref allpasses k) outsm)))
       (setf outval (* outsm outscale))
       (outa i outval)
       (if chan2 (outb i outval))))
    (end-run)))



---------------- WAVESHAPING ----------------

The simplest waveshaping instrument is Bigbird-2 given above.  The
simplest extension of BigBird-2 would be to add an index function (a
scaler ranging between 0 and 1) on the amplitude of the cosine (i.e.
the OSCIL) used to calculate the argument to polynomial.  You can move
the spectrum up to some arbitrary "carrier" by multiplying the output
of polynomial by another oscil.  More interesting is the following
instrument that uses "phase quadrature waveshaping" to create
arbitrary spectral shapes (i.e. shapes that are not reflected around
the "carrier" as in FM).  We use the normalization procedure from
BigBird-2 -- this is only needed by the DSP version, although it
doesn't hurt the software version.


(definstrument Pqw (start duration spacing-freq carrier-freq 
		    amplitude ampfun indexfun partials
		    &key (degree 0.0) (distance 1.0) (reverb-amount 0.005))
  ;; phase-quadrature waveshaping used to create asymmetric 
  ;; (i.e. single side-band) spectra. The basic idea here is a variant of 
  ;; sin x sin y - cos x cos y = cos (x + y)
  (let* ((normalized-partials (normalize-partials partials))
	 (spacing-cos (make-oscil :frequency spacing-freq :initial-phase half-pi))
	 (spacing-sin (make-oscil :frequency spacing-freq))
	 (carrier-cos (make-oscil :frequency carrier-freq :initial-phase half-pi))
	 (carrier-sin (make-oscil :frequency carrier-freq))
	 (sin-coeffs (get-chebychev-coefficients normalized-partials 0))
	 (cos-coeffs (get-chebychev-coefficients normalized-partials 1))
	 (cos-norm (normalize-to-power-of-two cos-coeffs))
	 (sin-norm (normalize-to-power-of-two sin-coeffs))
	 (amp-env (make-env :envelope ampfun :scaler amplitude 
			    :start-time start :duration duration))
	 (ind-env (make-env :envelope indexfun :scaler 1.0 
			    :start-time start :duration duration))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (vib 0.0)
	 (ax 0.0)
	 (fax 0.0)
	 (yfax 0.0)
	 (r (/ carrier-freq spacing-freq))
	 (tr (make-triangle-wave :frequency 5 
				 :amplitude (in-Hz (* .005 spacing-freq))))
	 (rn (make-randi :frequency 6 
			 :amplitude (in-Hz (* .005 spacing-freq))))
	 (beg (floor (* start sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate)))))
    (Run
     (loop for i from beg to end do
       (setf vib (+ (triangle-wave tr) (randi rn)))
       (setf ax (* (env ind-env) (oscil spacing-cos vib)))
       (setf fax (* cos-norm (polynomial cos-coeffs ax)))
       (setf yfax (* (oscil spacing-sin vib) 
		     sin-norm 
		     (polynomial sin-coeffs ax)))
       (locsig loc i 
	       (* (env amp-env)
		  (- (* (oscil carrier-sin (* vib r)) yfax) 
		     (* (oscil carrier-cos (* vib r)) fax))))))))


(with-sound ()
  (pqw 0 .5 200 1000 .2 '(0 0 25 1 100 0) '(0 1 100 0) '(2 .1 3 .3 6 .5)))


To see the asymmetric spectrum most clearly, set the index function
above to '(0 1 100 1).  pqwvox.lisp uses this technique to create
moving formant regions of arbitrary shape.


---------------- FILTERING ----------------

RESFLT and ADDFLTS:

These produce resonances using filters (resflt has stable formants,
and addflt has moving formants).


(definstrument resflt (start duration driver ranfreq noiamp noifun cosamp 
		       cosfreq1 cosfreq0 cosnum ampcosfun freqcosfun 
		       frq1 r1 g1 frq2 r2 g2 frq3 r3 g3
		       &key (degree 0.0) (distance 1.0) (reverb-amount 0.005))
  (let* ((beg (floor (* start sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (f1 (make-ppolar r1 frq1))
	 (f2 (make-ppolar r2 frq2))
	 (f3 (make-ppolar r3 frq3))
	 (input1 0.0)
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (frqf (if (/= driver 1) 
		   (make-env :envelope freqcosfun :scaler (in-Hz (- cosfreq1 cosfreq0))
			     :start-time start :duration duration)))
	 (ampf (if (= driver 1)
		   (make-env :envelope noifun :scaler noiamp 
			     :start-time start :duration duration)
		 (make-env :envelope ampcosfun :scaler cosamp 
			   :start-time start :duration duration)))
	 (rn (if (= driver 1) (make-randh :frequency ranfreq)))
	 (cn (if (= driver 0)
		 (make-sum-of-cosines :frequency cosfreq0 :cosines cosnum))))
    (Run
     (loop for i from beg to end do
       (if (= driver 1)
	   (setf input1 (* (env ampf) (randh rn)))
	 (setf input1 (* (env ampf) (sum-of-cosines cn (env frqf)))))
       (locsig loc i (+ (ppolar f1 (* input1 g1))
			(ppolar f2 (* input1 g2))
			(ppolar f3 (* input1 g3))))))))


(with-sound ()
  (resflt 0 1.0 0 0 0 nil .1 200 230 10 '(0 0 50 1 100 0) 
          '(0 0 100 1) 500 .995 .1 1000 .995 .1 2000 .995 .1)
  (resflt 1.0 1.0 1 10000 .01 '(0 0 50 1 100 0) 0 0 0 0 nil 
          nil 500 .995 .1 1000 .995 .1 2000 .995 .1))


Sum-of-cosines can turn out a band-limited pulse train made up of any
number of cosines (the cost of the unit-generator is independent of
the number of cosines). 

The next instrument is an extension of resflt -- here we add various
functions on the coefficients which gives the effect of moving formant
regions.


(defun ppolar-B1 (R freq) (- (* 2.0 R (cos (in-Hz freq)))))
(defun ppolar-B2 (R) (* R R))

(definstrument addflts (start duration numfilts driver 
			freqfun freq0 freq1 ampfun amp cosnum 
			inloc degree distance reverb-amount &rest filts)
  ;; numfilts: number of second order pole sections used.
  ;; driver: 1=white noise, 2=sum-of-cosines, 3=inloc.
  ;; filts: list of lists of filter data: 
  ;;   '(freq-fun R-fun frq0 frq1 R0 R1 scaler)
  (let* ((beg (floor (* start sampling-rate)))
	 (end (+ beg (floor (* duration sampling-rate))))
	 (rfs (make-array numFilts :element-type 'envelope))
	 (ffs (make-array numFilts :element-type 'envelope))
	 (pps (make-array numFilts :element-type 'smpflt))
	 (scs (make-array numFilts :element-type 'short-float))
	 (loc (make-locsig :degree degree :distance distance :revscale reverb-amount))
	 (rn (if (= driver 1) (make-randh :frequency freq0)))
	 (cn (if (= driver 2)
		 (make-sum-of-cosines :frequency freq0 :cosines cosnum)))
	 (insig 0.0)
	 (outsig 0.0)
	 (ampf (make-env :envelope ampfun :scaler amp 
			 :start-time start :duration duration))
	 (frqf (make-env :envelope freqfun :scaler (in-Hz (- freq1 freq0)) 
			 :start-time start :duration duration)))
    (loop for i in filts and k from 0 below numfilts do
      (let* ((frqfun (nth 0 i))
	     (R0 (nth 4 i))
	     (R1 (nth 5 i))
	     (G (nth 6 i))
	     (F0 (nth 2 i))
	     (F1 (nth 3 i))
	     (B1-bot (ppolar-B1 R0 F0))
	     (B1-top (ppolar-B1 R1 F1))
	     (B2-bot (ppolar-B2 R0))
	     (B2-top (ppolar-B2 R1))
	     (Rfun (nth 1 i)))
	(setf (aref pps k) (make-ppolar R0 F0))
	(setf (aref ffs k) 
	  (make-env :envelope frqfun 
		    :offset B1-bot :scaler (- B1-top B1-bot)
		    :start-time start :duration duration))
	(setf (aref rfs k) 
	  (make-env :envelope rfun 
		    :offset B2-bot :scaler (- B2-top B2-bot)
		    :start-time start :duration duration))
	(setf (aref scs k) G)))
    (Run
    (loop for i from beg to end do
      (setf outsig 0.0)
      (setf insig (* (env ampf) (if (= driver 1) 
				    (randh rn (env frqf))
				  (if (= driver 2)
				      (sum-of-cosines cn (env frqf))
				    inloc))))
      (dotimes (k numFilts)
	(incf outsig (ppolar (aref pps k) (* (aref scs k) insig)))
	(setf (smpflt-b1 (aref pps k)) (env (aref ffs k)))
	(setf (smpflt-b2 (aref pps k)) (env (aref rfs k))))
      (locsig loc i outsig)))))

(with-sound ()
  (addflts 0 1.0 2 2 '(0 0 100 1) 100 120 '(0 0 50 1 100 0) 
           .25 30 0 0 1.0 0.01
	   '((0 0 100 1) (0 0 100 1) 1200 2400 .9 .995 .005)
	   '((0 0 100 1) (0 0 100 1) 600 1200 .995 .995 .01)))


The next instrument was suggested by C Penrose.  It has moving formant
regions sliding around in white noise.

;;; Formnt:    y(n) = x(n) - R*x(n-2) + 2*R*cos(2*pi*Freq/Srate)*y(n-1) - R*R*y(n-2)

(defmacro formant (R freq G x0 x2 y0 y1 y2)
  `(let ((tG (* ,G (- 1.0 ,R))))
     (setf ,y2 ,y1)
     (setf ,y1 ,y0)
     (setf ,y0 (- (+ (* tG ,x0) 
		     (* 2 ,R (cos (in-Hz ,freq)) ,y1)) 
		  (+ (* tG ,R ,x2) 
		     (* ,R ,R ,y2))))))

(definstrument noi-attempt (beg dur amp R freq)
  (let* ((a0 1.)
	 (a1 .4)
	 (a2 .25)
	 (a3 .1)
	 (noi (make-randh :frequency 11025 :amplitude amp))
	 (vib0 (make-randh :frequency 5. :amplitude .0183))
	 (vib1 (make-randh :frequency 7.07 :amplitude .013))
	 (vib2 (make-randh :frequency 3.14 :amplitude .02))
	 (vib3 (make-randh :frequency 8.33 :amplitude .015))
	 (x2 0.0)
	 (x1 0.0)
	 (x0 0.0)
	 (st (floor (* sampling-rate beg)))
	 (nd (+ st (floor (* sampling-rate dur))))
	 (f0y2 0.0) (f0y1 0.0) (f0y0 0.0)
	 (f1y2 0.0) (f1y1 0.0) (f1y0 0.0)
	 (f2y2 0.0) (f2y1 0.0) (f2y0 0.0)
	 (f3y2 0.0) (f3y1 0.0) (f3y0 0.0))
    (Run
      (loop for i from st to nd do
	(setf x2 x1)
	(setf x1 x0)
	(setf x0 (randh noi))
	(outa i (* amp
		   (+ (formant R (* freq (+ 1. (randh vib0))) a0 x0 x2 f0y0 f0y1 f0y2)
		      (formant R (* freq 3.006 (+ 1. (randh vib1))) a1 x0 x2 f1y0 f1y1 f1y2)
		      (formant R (* freq 3.984 (+ 1. (randh vib2))) a2 x0 x2 f2y0 f2y1 f2y2)
		      (formant R (* freq 5. (+ 1. (randh vib3))) a3 x0 x2 f3y0 f3y1 f3y2))))))))

(with-sound () (noi-attempt 0 2 .3 .99 1200))


Here's an example of using fft-filter with a nil file (to get white
noise as the input to the filter):


(definstrument fftins (beg dur file fftenv)
  (let* ((start (floor (* beg sampling-rate)))
         (end (+ start (floor (* dur sampling-rate))))
         (fil (if file (open-input file)))
         (ff (make-fft-filter :file fil
                              :start-time beg
                              :fft-size 256
                              :filter fftenv)))
    (Run
      (loop for i from start to end do
        (outa i (fft-filter ff))))
    (if file (close-input fil))))

(with-sound () (fftins 0 2 nil '(0 0 .1 0 .11 1 1.0 1)))

(with-sound () (fftins 0 2 nil '(0 1 .1 1 .11 0 1.0 0)))



---------------- KARPLUS-STRONG ----------------

PLUCK:

The Karplus-Strong algorithm as extended by David Jaffe and Julius Smith.


(defun getOptimumC (S o p)
  (let* ((pa (* (/ 1.0 o) (atan (* S (sin o)) 
				(+ (- 1.0 S) (* S (cos o))))))
	 (tmpInt (floor (- p pa)))
	 (pc (- p pa tmpInt)))
    (loop while (< pc .1) do 
      (decf tmpInt)
      (incf pc))
    (values tmpInt (/ (- (sin o) (sin (* o pc))) 
		      (sin (+ o (* o pc)))))))

(defun tuneIt (f s1)
  (let* ((p (/ sampling-rate f))	 ;period as float
	 (s (if (zerop s1) 0.5 s1))
	 (o (in-Hz f)))
    (multiple-value-bind
	(T1 C1)
	(getOptimumC s o p)
      (multiple-value-bind
	  (T2 C2)
	  (getOptimumC (- 1.0 s) o p)
	(if (and (/= s .5)
		 (< (abs C1) (abs C2)))
	    (values (- 1.0 s) C1 T1)
	  (values s C2 T2))))))

(definstrument Pluck (start duration freq amp weighting 
		      lossfact decaytime attacktime)
  ;; DAJ explains weighting and lossfact as follows: weighting is the ratio
  ;; of the once-delayed to the twice-delayed samples.  It defaults to
  ;; .5=shortest decay.  anything other than .5 = longer decay.  Must be
  ;; between 0 and less than 1.0.  lossfact can be used to shorten decays.
  ;; Most useful values are between .8 and 1.0.
  (multiple-value-bind
      (wt0 c dlen)
      (tuneIt freq weighting)
    (let* ((beg (floor (* start sampling-rate)))
	   (end (+ beg (floor (* duration sampling-rate))))
	   (lf (if (zerop lossfact) 1.0 (min 1.0 lossfact)))
	   (wt (if (zerop wt0) 0.5 (min 1.0 wt0)))
	   (tab (make-table dlen))
	   ;; get initial waveform in "tab" -- here we can introduce 0's to simulate
	   ;; different pick positions, and so on -- see the CMJ article for
	   ;; numerous extensions.  The normal case is to load it with white noise
	   ;; (between -1 and 1).
	   (val 0.0)
	   (allp (make-one-zero (* lf (- 1.0 wt)) (* lf wt)))
	   (feedb (make-one-zero c 1.0))
	   (ctr 0))
      (loop for i from 0 below dlen do 
	(setf (aref tab i) (- 1.0 (random 2.0))))
      (Run
       (loop for i from beg to end do
	 (setf val (aref tab ctr))	 ;current output value
	 (setf (aref tab ctr) 
	   (* (- 1.0 c) 
	      (one-zero feedb (one-zero allp val))))
	 (incf ctr)
	 (if (>= ctr dlen) (setf ctr 0))
	 (outa i (* amp val)))))))



---------------- SOUND PROCESSING ----------------

One of the original ideas behind the clm system was that there is no
necessary distinction between sound synthesis and signal processing --
ideally the two should be able to fit together seamlessly.  The
"Contrast-enhancement" function, for example, uses a sound file as the
input to fm.

The basic sound file access mechanism is through INA and INB.  These
are packaged up in READIN, and READIN-REVERSE.  Once the data is
available, you can treat it just like any other data (say from OSCIL,
for example).  Some examples are given below.  See also RESAMPLE,
CONTRAST-ENHANCEMENT, FILTER, EXPAND, and so on.

Functions that open files should probably wrap an "unwind-protect"
around the file handling -- if you end up in the debugger for some
reason, then :reset back to the top level, you'll end up with open
files lying around and may get confusing results on subsequent file
processing.

As a simple example, the following reverberates a sound file
using jc-reverb (from jcrev.ins, not the version given above):


(defun reverberate (input-name 
		    &optional (output-name default-sound-file-name) 
			      (rev-amount .1))
  (let ((inf (open-input input-name)))
    (unwind-protect
	(let* ((true-input-name (IO-nam inf))
	       (dur (clm-get-duration inf)))
	  (open-output output-name
		       (make-header :channels (clm-get-channels inf)
				    :sampling-rate (set-srate (clm-get-sampling-rate inf))
				    :info (format nil "reverberate ~A ~A" 
						  input-name 
						  (clm-get-default-header))))
	  ;; here we are making sure the output file agrees
	  ;; with the input file. "clm-get-default-header" provides a time stamp
	  
	  (jc-reverb 0 (+ dur 3) rev-amount)  
	  ;; "dur+3" to let the reverb sing for awhile at the end

	  (close-input)			; close reverb input
	  (fasmix true-input-name)
	  (close-output)
	  (dac output-name))		; play the result
      (clu-reset))))


Reverse a sound file:

(definstrument reverse-sound (file &optional (start-time-in-output 0.0) 
					     (start-time-in-input -1.0) 
					     (duration nil))
  (let* ((f (open-input file))
	 (len-f (clm-get-samples f))
	 (insamp (if (plusp start-time-in-input) 
		     (min len-f (* start-time-in-input sampling-rate))
		   len-f))
	 (st (floor (* start-time-in-output sampling-rate)))
	 (nd (floor (if duration 
			(+ st (* duration sampling-rate))
		      (+ st insamp))))
	 (rdA (make-reverse :file f 
			    :start insamp))
	 (rdB (if (and (stereo f) (stereo *current-output-file*))
		  (make-reverse :file f 
				:channel :B 
				:start insamp))))
    (Run
     (loop for i from st to nd do 
       (outa i (readin-reverse rdA))
       (if rdB (outb i (readin-reverse rdB)))))
    (close-input f)))



Panning or swapping channels is equally simple.  Here's an instrument
that reverberates a sound while moving it around in 2 channels, adding
both amplitude and distance envelopes:

(definstrument space (file onset duration &key (distance-env '(0 1 100 10))
					       (amplitude-env '(0 1 100 1))
					       (degree-env '(0 45 50 0 100 90))
					       (reverb-amount .05))
  (let ((f (open-input file)))
    (unwind-protect
	(let* ((beg (floor (* onset sampling-rate)))
	       (end (+ beg (floor (* sampling-rate duration))))
	       (loc (make-locsig :degree 0
				 :distance 1
				 :revscale reverb-amount))
	       (rdA (make-readin :file f))
	       (dist-env (make-env :envelope distance-env :start-time onset :duration duration))
	       (amp-env (make-env :envelope amplitude-env :start-time onset :duration duration))
	       (deg-env (make-env :start-time onset :duration duration
				  :envelope (loop for x in degree-env by #'cddr and
						  y in (cdr degree-env) by #'cddr 
						collect x collect (/ y 90))))
	       (dist-scaler 0.0))
	  (Run
	   (loop for i from beg to end do
	     (let ((rdval (* (readin rdA) (env amp-env)))
		   (degval (env deg-env))
		   (distval (env dist-env)))
	       (setf dist-scaler (/ 1.0 distval))
	       (setf (locs-ascl loc) (* (- 1.0 degval) dist-scaler))
	       (setf (locs-bscl loc) (* degval dist-scaler))
	       (setf (locs-rscl loc) (* reverb-amount (sqrt dist-scaler)))
	       (locsig loc i rdval)))))
      (close-input f))))


To filter a sound:


(definstrument filter-sound (file beg 
			     &optional (dur -1.0) 
				       (orig-beg 0.0) 
				       (x-coeffs '(.5 .2 .1))
				       (y-coeffs nil)
				       (filter-type direct-form))
  (let* ((f (open-input file))
	 (st (floor (* beg sampling-rate)))
	 (new-dur (if (plusp dur) 
		      dur 
		    (- (clm-get-duration f) orig-beg)))
	 (flA (make-filter :type filter-type
			   :x-coeffs x-coeffs
			   :Y-coeffs y-coeffs))
	 (rdA (make-readin :file f :start-time orig-beg))
	 (two-chans (and (stereo f) (stereo *current-output-file*)))
	 (rdB (if two-chans 
		  (make-readin :file f :start-time orig-beg :channel :B)))
	 (flB (if two-chans 
		  (make-filter :type filter-type
			       :x-coeffs x-coeffs
			       :y-coeffs y-coeffs)))
	 (nd (+ st (floor (* sampling-rate new-dur)))))
    (Run
     (loop for i from st to nd do
       (outa i (filter flA (readin rdA)))
       (if two-chans (outb i (filter flB (readin rdB))))))
    (close-input f)))


Other currently supported types are lattice-form (Markel and Gray's
"two multiplier lattice") and ladder-form (Markel and Gray's
"normalized ladder").  These can be of any order.  If you already know
the reflection (k) coefficients, you can call the underlying
functions, make-lattice-filter and lattice-filter (in place of
make-filter and filter), or make-ladder-filter and ladder-filter, or
make-direct-filter and direct-filter.  If you pass the direct form
coefficients but ask for a lattice or ladder filter, the make-filter
function automatically converts the coefficients to the correct form.
Following Markel and Gray, the initial y coefficient is assumed to be
1.0 (that is, the first number in the y-coeff list should always be
1.0).  [DSP] Currently, all coefficients must be less than or equal to 1.0
(in magnitude).  A filter design package for both FIR and IIR filters
of all the "normal" types can be found in fltdes.lisp.


(definstrument expand-sound (file beg 
			     &optional (dur -1.0) 
				       (orig-beg 0.0) 
				       (exp-amt 1.0))
  (let* ((f (open-input file))
	 (st (floor (* beg sampling-rate)))
	 (new-dur (if (plusp dur) 
		      dur 
		    (- (clm-get-duration f) orig-beg)))
	 (two-chans (and (stereo f) (stereo *current-output-file*)))
	 (exA (make-expand f :start-time orig-beg :expansion-amount exp-amt))
	 (exB (if two-chans 
		  (make-expand f :start-time orig-beg :expansion-amount exp-amt :channel :B)))
	 (nd (+ st (floor (* sampling-rate new-dur)))))
    (Run
     (loop for i from st to nd do
       (outa i (expand exA))
       (if two-chans (outb i (expand exB)))))
    (close-input f)))


A very similar instrument is the following:


(definstrument pvins (beg dur file amp tscl &optional (size 128))
  (let* ((start (floor (* beg sampling-rate)))
	 (end (+ start (floor (* dur sampling-rate))))
	 (fil (open-input file))
	 (size-1 (1- size))
	 (wtb (make-block :size size :trigger 0))
	 (window (make-block :size size))
	 (freq-inc (/ size 2))
	 (rate (/ 1.0 freq-inc))
	 (filptr 0)
	 (oldfilptr 0)
	 (inj (floor (/ freq-inc tscl))))
    (loop for i from 0 below freq-inc and j from size-1 by -1 and angle from 0.0 by rate do
      (setf (aref (rblk-buf window) i) angle)
      (setf (aref (rblk-buf window) j) angle))
    (Run
     (loop for i from start to end do
       (when (zerop (rblk-loc wtb))
	 (setf filptr oldfilptr)
	 (dotimes (k size)
	   (incf (aref (rblk-buf wtb) k) (* (aref (rblk-buf window) k) (ina filptr fil)))
	   (incf filptr))
	 (incf oldfilptr inj)
	 (incf (rblk-ctr wtb) freq-inc))
       (outa i (* amp (run-block wtb)))))))



---------------- WAVE TRAIN ----------------
    
The wave-train "generator" has a stored wave form like additive
synthesis, but here we want to control the repetition rate of the wave
form, not the rate at which the table is read.  Since the successive
waves may overlap, we can't use zdelay.


(definstrument wtins (beg dur frq amp fm-env phase)
  (let* ((start (floor (* beg sampling-rate)))
	 (end (+ start (floor (* dur sampling-rate))))
	 (hi (make-array 128 :element-type 'short-float :initial-element 0.0))
	 (fm (make-env :envelope fm-env :scaler (in-Hz frq) 
		       :start-time beg :duration dur))
	 (ampf (make-env :envelope '(0 0 25 1 75 1 100 0) :start-time beg :duration dur :scaler amp))
	 (wt0 (make-wave-train :wave hi :initial-phase phase :frequency frq)))
    (loop for i from 0 to 127 do 
      (setf (aref hi i) (float (/ (- 127 i) 1000))))
    (Run
     (loop for i from start to end do
       (outa i (* (env ampf) (wave-train wt0 (env fm))))))))


(with-sound () (wtins 0 1 100 .5 '(0 0 100 1) 0.0)) 

The waves start with appended zeros, then gradually repeat faster 
until they overlap (if srate=22050)


Another example:


(definstrument gran-synth (start-time duration audio-freq grain-dur grain-interval amp)
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let* ((grain-env (make-env :envelope '(0 0 25 1 75 1 100 0)
				:start-time 0 :duration grain-dur))
	   (carrier (make-oscil :frequency audio-freq))
	   (grain-size (ceiling (* (max grain-dur grain-interval) sampling-rate)))
	   (grain (make-array grain-size :element-type 'short-float :initial-element 0.0))
	   (grains (make-wave-train :wave grain :frequency (/ 1.0 grain-interval))))
      (loop for i from 0 below grain-size do
	(setf (aref grain i) (* (env grain-env) (oscil carrier))))
      (Run
       (loop for i from beg to end do
	 (outa i (* amp (wave-train grains))))))))
	 
(with-sound () (gran-synth 0 2 100 .0189 .02 .4))




---------------- PHYSICAL MODELLING ----------------

The block diagrams published in various articles are not enough to get
an actual model to "speak", and working code seems to be plastered
over with copyright and patent notices, so the following is only a
weak stab at the clarinet discussed by J O Smith in various
publications, including "Music Applications of Digital Waveguides".
My first n attempts to follow Smith's diagrams failed to produce any
sound...  (As these instruments leak into the public domain, or as I
get the time, I'll add more to this section, but it's a bother to have
to worry about the lawyers).


(definstrument clarinet (beg dur freq amp att)
  (let* ((start (floor (* beg sampling-rate)))
	 (finish (+ start (floor (* dur sampling-rate))))
	 (len (/ sampling-rate freq))
	 (mouth-function (make-env :envelope (list 0 0 att 1 (- dur att) 1 dur 0) 
				   :scaler 1.0 :start-time beg :duration dur))
	 (bore-right (make-delay len))
	 (bore-left (make-delay (* 2 len)))
	 (noise (make-randh :frequency 1000 :amplitude .001))
	 (reed 0.0)
	 (reed-back 0.0)
	 (filtered 0.0)
	 (la 0.0) (lb 0.0) (lc 0.0)
	 (lowpass (make-one-pole .7 .3)))
    (Run
     (loop for i from start to finish do
       (setf reed-back (+ (* (env mouth-function) (1+ (randh noise))) (* .9 reed)))
       (setf filtered (one-pole lowpass (+ (* reed-back reed-back (+ .3 (* reed-back -.8))))))
       (setf reed (* .45 (+ (delay bore-right filtered) (delay bore-left filtered))))
       (setf lc lb)			;a "modified least-squares" low pass filter
       (setf lb la)
       (setf la (* amp filtered))
       (outa i (+ (* .5 lb) (* .25 (* (+ la lc)))))))
    (end-run)))

(with-sound () (clarinet 0 2 220 2 .1) (clarinet 2 2 440 2 .1) (clarinet 4 2 880 2 .1))


---------------- SPECTRAL MODELLING ----------------

The following instrument analyzes a sound into its main component
spectral peaks and then resynthesizes it using oscils.  See san.ins
for the latest version and discussion of sound transformations.  This
version assumes its input sound is relatively stable and noise-free.
Xavier Serra's SMS program takes the next step of modelling the noise
as well -- an exercise for the reader!  [DSP] (On the NeXT where we have 8K
external memory, this is about as big as an instrument can get because
the entire program has to fit into internal P memory (512 words) and
external X memory (4K words), and we're up to about 3800 words with
the basic instrument given here).

(definstrument pins (beg dur file amp 
		     &key (transposition 1.0) ; this can be used to transpose the sound
			  (time-scaler 1.0)   ; this can make things happen faster (< 1.0)/slower (> 1.0) in the output
			  (fftsize 256)       ; should be a power of 2
			  ;; at 22050 srate, this is ok for sounds above 300Hz or so, below that you need 512 or 1024,
			  ;; at 44100, probably best to double these sizes -- it takes some searching sometimes.
			  (highest-bin 128)   ; how high in fft data should we search for peaks
			  (max-peaks 16)      ; how many spectral peaks to track at the maximum
			  printit)	      ; whether to print out the envelopes as they are formed
  ;; do the sliding fft shuffle, translate to polar coordinates, find spectral peaks,
  ;;   match with current, do some interesting transformation, resynthesize using oscils
  ;;   All the envelopes are created on the fly.  max-peaks is how many of these peaks
  ;;   we are willing to track at any given time.
  (let* ((start (floor (* beg sampling-rate)))
	 (end (+ start (floor (* dur sampling-rate))))
	 (fil (open-input file))
	 (fftsize-1 (1- fftsize))
	 (fftdata (make-fft-data-arrays fftsize))
	 (window (make-table fftsize))
	 (fftamps (make-table fftsize))
	 (max-oscils (* 2 max-peaks))
	 (current-peak-freqs (make-array max-oscils :element-type 'short-float :initial-element 0.0))
	 (last-peak-freqs (make-array max-oscils :element-type 'short-float :initial-element 0.0))
	 (current-peak-amps (make-array max-oscils :element-type 'short-float :initial-element 0.0))
	 (last-peak-amps (make-array max-oscils :element-type 'short-float :initial-element 0.0))
	 (peak-amps (make-array max-peaks :element-type 'short-float :initial-element 0.0))
	 (peak-freqs (make-array max-peaks :element-type 'short-float :initial-element 0.0))
	 (resynth-oscils (make-array max-oscils :element-type 'osc))
	 (amps (make-table max-oscils))	;run-time generated amplitude and frequency envelopes
	 (rates (make-table max-oscils))
	 (freqs (make-table max-oscils))
	 (sweeps (make-table max-oscils))
	 (lowest-magnitude .001)
	 (freq-inc (/ fftsize 2))
	 (hop (floor fftsize 4))
	 (outhop (* time-scaler hop))
	 (ifreq (/ 1.0 outhop))
	 (ihifreq (in-Hz ifreq))
	 (rate (/ two-pi fftsize))
	 (fftscale (/ 1.0 (* fftsize .42323))) ;integrate Blackman-Harris window = .42323*window width and shift by fftsize
	 (fft-mag (float (/ sampling-rate fftsize)))
	 (furthest-away-accepted .1)
	 (filptr 0)
	 (trigger 0)
	 (peaks 0)
	 (sum 0.0))
    (loop for i from 0 below freq-inc and j from fftsize-1 by -1 and angle from 0.0 by rate do
      (let* ((cosx (cos angle))
	     (val (* (+ .34401 (* cosx (+ -.49755 (* cosx .15844)))) fftscale)))
	;; Chebychev polynomial version of 2nd order Blackman-Harris window 
	;;   (+ .43232 (* -.49755 (cos angle)) (* .07922 (cos (* 2 angle))))
	(setf (aref window i) val)
	(setf (aref window j) val)))
    (loop for i from 0 below max-oscils do
      (setf (aref resynth-oscils i) (make-oscil :frequency 0)))
    (Run
     (loop for i from start to end do
       (when (>= trigger outhop)
	 ;; get next block of data and apply window to it
	 (setf trigger 0)
	 (dotimes (k fftsize)
	   (setf (aref (fft-data-real fftdata) k) (* (aref window k) (ina filptr fil)))
	   (incf filptr))
	 (clear-block (fft-data-imaginary fftdata))
	 (decf filptr (- fftsize hop))
	 ;; get the fft 
	 (fft fftdata)
	 ;; change to polar coordinates (ignoring phases)
	 (dotimes (k highest-bin)	;no need to paw through the upper half (so (<= highest-bin (floor fft-size 2)))
	   (let ((x (aref (fft-data-real fftdata) k))
		 (y (aref (fft-data-imaginary fftdata) k)))
	     (setf (aref fftamps k) (* 2 (expt (+ (* x x) (* y y)) .5)))))
	 (dotimes (k max-oscils)
	   (setf (aref last-peak-freqs k) (aref current-peak-freqs k))
	   (setf (aref last-peak-amps k) (aref current-peak-amps k))
	   (setf (aref current-peak-amps k) 0.0))
	 (dotimes (k max-peaks)
	   (setf (aref peak-amps k) 0.0))
	 (let ((ra (aref fftamps 0))
	       (la 0.0)
	       (ca (aref fftamps 0)))
	   ;; search for current peaks following Xavier Serra's recommendations in
	   ;; "A System for Sound Analysis/Transformation/Synthesis 
	   ;;      Based on a Deterministic Plus Stochastic Decomposition"
	   (setf peaks 0)		;how many peaks found so far
	   (dotimes (k highest-bin)
	     (setf la ca)
	     (setf ca ra)
	     (setf ra (aref fftamps k))
	     (if (and (> ca lowest-magnitude)
		      (> ca ra)
		      (> ca la))
		 ;; found a local maximum above the current threshold (its bin number is k-1)
		 (let* ((logla (log la 10))
			(logca (log ca 10)) 
			(logra (log ra 10))
			(offset (/ (* .5 (- logla logra)) (+ logla (* -2 logca) logra)))
			(amp (expt 10.0 (- logca (* .25 (- logla logra) offset))))
			(freq (* fft-mag (+ k offset -1))))
		   (if (= peaks max-peaks)
		       ;; gotta either flush this peak, or find current lowest and flush him
		       (let ((minp 0)
			     (minpeak (aref peak-amps 0)))
			 (loop for j from 1 below max-peaks do
			   (when (< (aref peak-amps j) minpeak)
			     (setf minp j)
			     (setf minpeak (aref peak-amps j))))
			 (when (> amp minpeak)
			   (setf (aref peak-freqs minp) freq)
			   (setf (aref peak-amps minp) amp)))
		     (progn
		       (setf (aref peak-freqs peaks) freq)
		       (setf (aref peak-amps peaks) amp)
		       (incf peaks)))))))
	 ;; now we have the current peaks -- match them to the previous set and do something interesting with the result
	 ;; the end results are reflected in the updated values in the rates and sweeps arrays.
	 ;; search for fits between last and current, set rates/sweeps for those found
	 ;;   try to go by largest amp first 
	 (dotimes (k peaks)
	   (let ((maxp 0)
		 (maxpk (aref peak-amps 0)))
	     (loop for j from 1 below max-peaks do
	       (when (> (aref peak-amps j) maxpk)
		 (setf maxp j)
		 (setf maxpk (aref peak-amps j))))
	     ;; now maxp points to next largest unmatched peak
	     (when (> maxpk 0.0)
	       (let* ((closestp -1)
		      (closestamp 10)
		      (current-freq (aref peak-freqs maxp))
		      (icf (/ 1.0 current-freq)))
		 (loop for j from 0 below max-peaks do
		   (if (> (aref last-peak-amps j) 0.0)
		       (let ((closeness (* icf (abs (- (aref last-peak-freqs j) current-freq)))))
			 (when (< closeness closestamp)
			   (setf closestamp closeness)
			   (setf closestp j)))))
		 (when (< closestamp furthest-away-accepted)
		   ;; peak-amp is transferred to appropriate current-amp and zeroed,
		   (setf (aref current-peak-amps closestp) (aref peak-amps maxp))
		   (setf (aref peak-amps maxp) 0.0)
		   (setf (aref current-peak-freqs closestp) current-freq))))))
	   (loop for k from 0 below max-peaks do
	     (if (> (aref peak-amps k) 0.0)
	       ;; find a place for a new oscil and start it up
	       (let ((new-place (loop for j from 0 below max-oscils
				 if (and (zerop (aref last-peak-amps j)) (zerop (aref current-peak-amps j)))
				 return j)))
		 (setf (aref current-peak-amps new-place) (aref peak-amps k))
		 (setf (aref peak-amps k) 0.0)
		 (setf (aref current-peak-freqs new-place) (aref peak-freqs k))
		 (setf (aref last-peak-freqs new-place) (aref peak-freqs k))
		 (setf (osc-freq (aref resynth-oscils new-place)) (* transposition (in-Hz (aref peak-freqs k)))))))

	 (dotimes (k max-oscils)
	   (setf (aref rates k) (* ifreq (- (aref current-peak-amps k) (aref last-peak-amps k))))
	   (setf (aref sweeps k) (* ihifreq transposition (- (aref current-peak-freqs k) (aref last-peak-freqs k)))))

	 (when printit
	   ;; this block prints out the envelope data as it is generated
	   (print filptr)
	   (dotimes (k max-oscils) 
	     (when (or (> (aref rates k) 0.0) 
		       (> (aref amps k) 0.0))
	       (print " ") 
	       (princ k) (princ " ") 
	       (princ (aref current-peak-freqs k)) (princ " ")
	       (princ (aref current-peak-amps k))))))
       ;; run oscils, update envelopes
       (incf trigger)
       (setf sum 0.0)
       (dotimes (k max-oscils)
	 (when (or (/= (aref amps k) 0.0) (/= (aref rates k) 0.0))
	   (incf sum (* (aref amps k) (oscil (aref resynth-oscils k) (aref freqs k))))
	   (incf (aref amps k) (aref rates k))
	   (incf (aref freqs k) (aref sweeps k))))
       (outa i (* amp sum))))))



---------------- VARIOUS OTHER EXAMPLES ----------------

Here's an example of using run-block explicitly.  This instrument is
an opened-up wave-train (see wtins above) but without the
initial-phase or fm possibilities.


(definstrument wtins-1 (beg dur frq amp)
  (let* ((start (floor (* beg sampling-rate)))
	 (end (+ start (floor (* dur sampling-rate))))
	 (hi (make-array 128 :element-type 'short-float :initial-element 0.0))
	 (ampf (make-env :envelope '(0 0 25 1 75 1 100 0) :start-time beg :duration dur :scaler amp))
	 (wtb (make-block :size 128 :trigger 0))
	 (freq-inc (/ sampling-rate frq)))
    (loop for i from 0 to 127 do 
      (setf (aref hi i) (float (/ (- 127 i) 1000))))
    (Run
     (loop for i from start to end do
       (when (zerop (rblk-loc wtb))
	 (dotimes (k 128)
	   (incf (aref (rblk-buf wtb) k) (aref hi k)))
	 (incf (rblk-ctr wtb) freq-inc))
       (outa i (* (env ampf) (run-block wtb)))))))


Here we change the pitch of the sound without changing its length
(this is equivalent to a phase vocoder analysis, straight shift, and
resynthesis, all done in the run loop -- of course this will make a
harmonic sound inharmonic because we are simply shifting bins by an
integer amount).


(definstrument pvins (beg dur file amp shift &optional (fftsize 128)) ;assume shift is a positive integer
  (let* ((start (floor (* beg sampling-rate)))
	 (end (+ start (floor (* dur sampling-rate))))
	 (fil (open-input file))
	 (fftsize-1 (1- fftsize))
	 (fftdata (make-fft-data-arrays fftsize))
	 (wtb (make-block :size fftsize :trigger 0))
	 (window (make-block :size fftsize))
	 (freq-inc (/ fftsize 2))
	 (freq-inc-1 (1- freq-inc))
	 (rate (/ 1.0 (* freq-inc fftsize)))
	 (filptr 0)
	 (shift-top (- fftsize shift 1)))
    (loop for i from 0 below freq-inc and j from fftsize-1 by -1 and angle from 0.0 by rate do
      (setf (aref (rblk-buf window) i) angle)
      (setf (aref (rblk-buf window) j) angle))
    (Run
     (loop for i from start to end do
       (when (zerop (rblk-loc wtb))
	 (dotimes (k fftsize)
	   (setf (aref (fft-data-real fftdata) k) (* (aref (rblk-buf window) k) (ina filptr fil)))
	   (incf filptr))
	 (clear-block (fft-data-imaginary fftdata))
	 (decf filptr freq-inc)
	 (fft fftdata)
	 ;; now shift the positive frequencies up (in the fft data array) and the negative frequencies down
	 (loop for k from freq-inc-1 downto shift and j from freq-inc to shift-top do
	   (setf (aref (fft-data-real fftdata) k) (aref (fft-data-real fftdata) (- k shift)))
	   (setf (aref (fft-data-imaginary fftdata) k) (aref (fft-data-imaginary fftdata) (- k shift)))
	   (setf (aref (fft-data-real fftdata) j) (aref (fft-data-real fftdata) (+ j shift)))
	   (setf (aref (fft-data-imaginary fftdata) j) (aref (fft-data-imaginary fftdata) (+ j shift))))
	 (dotimes (k shift)
	   (setf (aref (fft-data-real fftdata) k) 0.0)
	   (setf (aref (fft-data-imaginary fftdata) k) 0.0)
	   (setf (aref (fft-data-real fftdata) (- fftsize-1 k)) 0.0)
	   (setf (aref (fft-data-imaginary fftdata) (- fftsize-1 k)) 0.0))
	 (inverse-fft fftdata)
	 (dotimes (k fftsize)
	   (incf (aref (rblk-buf wtb) k) (aref (fft-data-real fftdata) k)))
	 (incf (rblk-ctr wtb) freq-inc))
       (outa i (* amp (run-block wtb)))))))


Here's a phase modulation instrument:


(definstrument pm (beg end freq amp mc-ratio index &optional (index-env '(0 1 100 1)))
  (let* ((cr (make-oscil :frequency freq))
         (md (make-oscil :frequency (* freq mc-ratio)))
         (ampf (make-env :envelope index-env :scaler amp :start beg :end end))
         (indf (make-env :envelope index-env :scaler index :start beg :end end)))
    (Run
      (loop for i from beg to end do
        (outa i (* (env ampf) (oscil cr 0.0 (* (env indf) (oscil md)))))))))


An example of "spectral fusion" (see also fusion.ins for a more sophisticated example):


(definstrument Fusion (start-time duration frequencies amplitudes ampf fusef)
  (multiple-value-bind (beg end) (get-beg-end start-time duration)
    (let* ((notes (length frequencies))
	   (oscs (make-array notes :element-type 'osc))
	   (amp-env (make-env :envelope ampf :scaler 1.0 :start-time start-time :duration duration))
	   (amps (make-array notes :element-type 'short-float))
	   (fuse-env (make-env :envelope fusef :scaler 1.0 :start-time start-time :duration duration))
	   (global-vib (make-oscil :frequency 5.0))
	   (global-vib-val 0.0)
	   (vib-scalers (make-array notes :element-type 'short-float))
	   (sum 0.0)
	   (min-freq (loop for x in frequencies minimize x))
	   (global-vib-amount 0.0))
      (loop for i from 0 below notes and frequency in frequencies and amplitude in amplitudes do
	(setf (aref oscs i) (make-oscil :frequency frequency))
	(setf (aref vib-scalers i) (float (/ frequency min-freq)))
	(setf (aref amps i) amplitude))
      (Run				
       (loop for i from beg to end do 
	 (setf sum 0.0)
	 (setf global-vib-amount (env fuse-env))
	 (setf global-vib-val (* .005 (oscil global-vib)))
	 (loop for j from 0 below notes do
	   (incf sum (* (aref amps j) 
			(oscil (aref oscs j) 
			       (* global-vib-amount (aref vib-scalers j) global-vib-val)))))
	 (outa i (* (env amp-env) sum)))))))
			       
(with-sound () (fusion 0 10 '(440 1760 2200 2640) '(.1 .05 .025 .015) '(0 0 .1 1 5 .5 25 .25 100 0) '(0 0 25 0 75 1 100 1)))


Sometimes we want to create a file of temporary sound data as a way to
communicate between two instruments, caring only about the overall
result.  Here's an example "instrument" that expands, then resamples a
file.


(defun expsrc (beg dur input-file exp-ratio src-ratio)

  ;; expand input-file by exp-ratio, then resample the result at src-ratio,
  ;; using sound-let to hold the intermediate results.
  ;; We assume we are called within with-sound.
  ;; The two instruments called need not be local to the function (we could
  ;; have defined them separately and so on), but it doesn't hurt either.

  (instrument-let 
    ((expand-sound ()
       (let ((f (open-input input-file)))
	 (unwind-protect
	     (let* ((st 0)
		    (two-chans (and (stereo f) (stereo *current-output-file*)))
		    (exA (make-expand f :start-time 0.0 :expansion-amount exp-ratio))
		    (exB (if two-chans (make-expand f :start-time 0.0 :expansion-amount exp-ratio :channel :B)))
		    (nd (+ st (floor (* sampling-rate dur src-ratio)))))
	       (Run
		(loop for i from st to nd do
		  (outa i (expand exA))
		  (if two-chans (outb i (expand exB))))))
	   (close-input f))))
     (resample-sound (file beg)
       (let ((f (open-input file)))
	 (unwind-protect
	     (let* ((st (floor (* beg sampling-rate)))
		    (two-chans (and (stereo f) (stereo *current-output-file*)))
		    (srcA (make-resample :file f :start-time 0.0 :srate src-ratio))
		    (srcB (if two-chans (make-resample :file f :start-time 0.0 :srate src-ratio :channel :B)))
		    (nd (+ st (floor (* sampling-rate dur)))))
	       (Run
		(loop for i from st to nd do
		  (outa i (resample srcA))
		  (if two-chans (outb i (resample srcB))))))
	   (close-input f)))))

    (sound-let ((temp-sound () (expand-sound)))
      (resample-sound temp-sound beg))))

(with-sound () (fm-violin 0 .5 440 .1) (expsrc .25 4 "/me/snd/howl1.snd" 3 2) (fm-violin 1 .5 660 .1))


A simpler example of sound-let: say we have the fm-violin 
and the expins instruments loaded:


(defun one-sound ()
  (sound-let ((temp-1 () (fm-violin 0 1 440 .1))
	      (temp-2 () (fm-violin 0 2 660 .1) 
		         (fm-violin .125 .5 880 .1)))
    (expand-sound temp-1 0 2 0 2)
    (expand-sound temp-2 1 1 0 2)))

(with-sound () (one-sound))


Restartable envelopes.

Here's an example of an instrument that uses "restartable" envelopes --
in this case we use it simply to repeat the same envelope over and over:


(definstrument restartable-simp (beg dur env-dur)
  (let* ((os (make-oscil))
	 (en (make-env :envelope '(0 0 50 1 100 0) 
		       :start beg
		       :end (+ beg env-dur)
		       :scaler .1 
		       :restartable t))
	 (j beg)
	 (env-stop (+ beg env-dur)))
    (Run
     (loop for i from beg to (+ beg dur) do 
       (let ((val (* (env en) (oscil os))))
	 (incf j)
	 (when (> j env-stop)
	   (incf env-stop env-dur)
	   (restart-env en))
	 (outa i val))))))


[DSP] The only things to notice are the ":restartable t" option passed to
make-env (so it saves the necessary start-state information), and
the use of :start and :end -- within Run, these are sample counters.
[C] All envelopes are automatically restartable, so the :restartable
option is unneeded.


Here's an instrument that can run either in or out of with-sound:


(definstrument auto (dur)
  (let* ((outf (and (not *current-output-file*)
		    (open-output "/zap/test.snd" 
				 (make-header :channels 1 
					      :sampling-rate 22050))))
	 (os (make-oscil :frequency 440))
	 (end (floor (* dur 22050))))
    (Run
     (loop for i from 0 to end do
       (outa i (* .1 (oscil os)))))
    (when outf
      (close-output)
      (clm-cleanup)
      (dac "/zap/test.snd"))))
	

*current-output-file* will be open if auto is called within with-sound
or clm-load.  Otherwise, auto opens a clm output file (/zap/test.snd).
It then writes out its little toot, and if it opened the output file,
it goes ahead and closes it, and plays it.

The following instrument reads a sound file forwards and backwards
until the duration of the instrument call is used up.  [DSP] We declare the
sample counters explicitly to be "bignums" so that they can't
accidentally (i.e. incorrectly, but ...) end up as "reals" -- on the
56000, a "real" has 23 bits in the integer part, which means a real as
a sample counter can only handle a 3 minute sound file (at 44100 Hz),
so it's worth some trouble to avoid it.


(definstrument backandforth (onset duration file)
  ;; read file forwards and backwards until dur is used up
  (let* ((f (open-input file))
	 (last-sample (floor (* sampling-rate (clm-get-duration f))))
	 (beg (floor (* sampling-rate onset)))
	 (end (+ beg (floor (* sampling-rate duration))))
	 (rd (make-readin :file f :start 0))
	 (cs 0))
    (Run
     (loop for i from beg to end do
       (declare (type bignum last-sample) (type bignum cs))
       (setf cs (read-position rd))
       (if (>= cs last-sample) (read-backward rd))
       (if (<= cs 0) (read-forward rd))
       (outa i (readin rd))))
    (close-input f)))


Similarly (next we use the src generator in place of readin):


(definstrument backandforth (onset duration file src-ratio)
  ;; read file forwards and backwards until dur is used up
  (let* ((f (open-input file))
	 (last-sample (floor (* sampling-rate (clm-get-duration f))))
	 (beg (floor (* sampling-rate onset)))
	 (end (+ beg (floor (* sampling-rate duration))))
	 (rd (make-src :file f :start 0 :srate src-ratio))
	 (cs 0))
    (run
     (loop for i from beg to end do
       (declare (type bignum last-sample) (type bignum cs))
       (setf cs (read-position rd))
       (if (>= cs last-sample) (read-backward rd))
       (if (<= cs 0) (read-forward rd))
       (outa i (src rd))))
    (close-input f)))



---------------- ASYMMETRIC FM ----------------

These two instruments should be equivalent:


(defun asy0 (beg dur freq amp index &optional (r 1.0) (ratio 1.0))
  (let* ((st (floor (* beg sampling-rate)))
	 (nd (+ st (floor (* dur sampling-rate))))
	 (asyf (make-asymmetric-fm :r r :ratio ratio :frequency freq)))
    (loop for i from st to nd do
      (outa i (* amp (asymmetric-fm asyf index 0.0))))))

(definstrument asy (beg dur freq amp index &optional (r 1.0) (ratio 1.0))
  (let* ((st (floor (* beg sampling-rate)))
	 (nd (+ st (floor (* dur sampling-rate))))
	 (modosc (make-oscil :frequency (* ratio freq)))
	 (cososc (make-oscil :frequency (* ratio freq) :initial-phase (* .5 pi)))
	 (carosc (make-oscil :frequency freq))
	 (sr (* .5 (+ r (/ 1.0 r))))
	 (cr (* .5 (- r (/ 1.0 r)))))
    (Run
     (loop for i from st to nd do
       (outa i (* amp 
		  (exp (* index cr (oscil cososc))) 
		  (oscil carosc 0.0 (* sr (oscil modosc)))))))))

(with-sound () (asy 0 1 440 .1 1.0) (asy0 1.25 1 440 .1 1.0))



---------------- SINE SUMMATION ----------------

These two instruments should be equivalent:


(defun ss0 (beg dur freq amp &optional (N 1) (a .5) (B-ratio 1.0))
  (let* ((st (floor (* sampling-rate beg)))
	 (nd (+ st (floor (* sampling-rate dur))))
	 (sgen (make-sine-summation :N N :a a :B-ratio B-ratio :frequency freq)))
    (loop for i from st to nd do
      (outa i (* amp (sine-summation sgen))))))

(definstrument ss (beg dur freq amp &optional (N 1) (a .5) (B-ratio 1.0))
  (let* ((st (floor (* sampling-rate beg)))
	 (nd (+ st (floor (* sampling-rate dur))))
	 (aN (expt a (1+ N)))
	 (a2 (1+ (* a a)))
	 (thosc (make-oscil :frequency freq))
	 (thBosc (make-oscil :frequency (* freq (- 1.0 B-ratio))))
	 (thN1Bosc (make-oscil :frequency (* freq (+ 1 (* (+ N 1) B-ratio)))))
	 (thNBosc (make-oscil :frequency (* freq (+ 1 (* N B-ratio)))))
	 (cosB (make-oscil :frequency (* freq B-ratio) :initial-phase (* .5 pi))))
    (Run
     (loop for i from st to nd do
       (outa i (* amp
		  (/ (- (oscil thosc) 
			(* a (oscil thBosc))
			(* aN (- (oscil thN1Bosc)
				 (* a (oscil thNBosc)))))
		     (- a2 (* 2 a (oscil cosB))))))))))



---------------- ARBITRARY LISP AND C FUNCTIONS IN RUN ----------------

[C] Here's an example of the foreign function connection to C -- it can
be invoked in any instrument. 

#|
/* the C code is: */
float c_average (float a, float b) {return((a+b)*.5);}
|#

#|
;;; now in a separate file (which we compile/load before the instrument):

(in-package :clm)

(export '(AVERAGE))

(def-dsp-fun 'average #'(lambda (var x) (package-op '<average> var x 'real)))

(defmacro <average> (result-name &rest args)
  (format *c-file* "  ~A = c_average(~A,~A);~%" (lc result-name) (lc (first args)) (lc (second args)))
  nil)

;;; the def-dsp-fun tells the Run macro how to handle the function average (it walks the
;;; argument list and so on -- see run.lisp for other examples).  The macro <average> 
;;; sends out the associated c code when that point is reached in the translation.
;;; see cmus.lisp for more examples -- the "lc" function translates from lisp-style
;;; names, if any, to C style.  You can also include type checks and whatnot.
|#

;;; and the instrument file:

(definstrument (call-c :c-include-file "cins.c" :language :c) (beg dur freq amp)
  (let ((os1 (make-oscil :frequency freq))
	(os2 (make-oscil :frequency (* freq 3/2)))
	(bg (floor (* sampling-rate beg)))
	(nd (+ beg (floor (* sampling-rate dur)))))
    (Run
     (loop for i from bg to nd do
       (outa i (* amp (average (oscil os1) (oscil os2))))))))


If you have the C code already compiled and want to access its
contents through "extern" and so on, use the :c-include-file
option to definstrument.




---------------- CLM BUGS AND WORK-AROUNDS ----------------

Currently I know of the following bugs in CLM:

1) [DSP]
   If an instrument does not happen to call an output function
   (outa or locsig, for example) on some samples, the CLM/DSP
   communication interface gets confused and hangs.  For example,

   (definstrument gad ()
     (Run
      (loop for i from 0 to 10000 do
        (if (< 500 i 750) (outa i .1)))))

   (with-sound () (gad))

   dies with the error:

   "hit a break-point in checkHF3" 
   "pc: 71 -- can't see what's wrong" Break: call to the `break' function.

   
   The work-around for this is to rewrite the Run section of the
   instrument so that it always outputs something, even if only
   a 0:

   (definstrument gad ()
     (Run
      (loop for i from 0 to 10000 do
        (let ((val 0.0))
          (if (< 500 i 750) (setf val .1))
          (outa i val)))))

   Of course, if you're not using "Run", there is no problem.


2) The dac function's :end option doesn't work -- this is
   a reflection of a bug in NeXT's SoundStartPlaying.

3) [DSP]
   Basic-filter, polynomial, and a few others cannot handle
   coefficients greater than 1.0 (absolute value), and currently
   there's no error message associated with this.  Similarly,
   sinh, asinh, and associated hyperbolics assume their argument
   is fractional.  Delay lines assume they are delaying sound
   samples -- you can set the clm-safety variable to 1 to get
   warnings about overflows of this kind:

     (Run (loop for i from 0 to 4 do (declare (optimize (safety 1))) ...))

4) [DSP]
   If you're running on a NeXT, and have some non-standard DSP
   memory expansion module (either NeXT's 32K or SFSU's 192K), 
   the clm compiler doesn't notice the difference until you've
   run with-sound.  This means you have to compile twice sometimes.
   To deal with this by hand, add this line to your clm-init file:

    (set-up-dsp-memory-map)

5. [C]
   If you compile and load CLM, then later load a different version
   of walk.lisp, a subsequent call on definstrument to generate a
   C intermediate file may die with 

   Error: write bytes 4 resulted in error: Bad file number 

   The fix is to reload clm'c walk.fasl.
