;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;

(in-package :clm)

(defvar *headers-call-direct* t)
(defvar *headers-check-args* nil)
(defvar *headers-prototyped* t)   ; Should be nil only if C implementation is not


;;; the rest of CLM doesn't need to know what kind of headers it is read/writing,
;;; so here we provide the calls to get/set the needed information. 

(defun full-merge-pathnames (pathname &optional defaults default-version) 
  (declare (optimize (speed 3) (safety 0)))
  #-mcl (merge-pathnames pathname defaults default-version)
  #+mcl (ccl:full-pathname (merge-pathnames pathname defaults default-version))
  )


;;; these constants need to be kept in sync with sound_types.h
(defconstant snd-16-linear 1)
(defconstant snd-8-mulaw 2)
(defconstant snd-8-linear 3)
(defconstant snd-32-float 4)
(defconstant snd-32-linear 5)
(defconstant snd-8-alaw 6)
(defconstant snd-8-unsigned 7)
(defconstant snd-24-linear 8)
(defconstant snd-64-double 9)

(defun snd-bytes (frm siz)
  (declare (optimize (speed 3) (safety 0)))
  (if (= frm snd-16-linear) siz
    (if (or (= frm snd-32-float) (= frm snd-32-linear)) (* siz 2)
      (if (= frm snd-24-linear) (+ siz (ash siz -1))
	(if (= frm snd-64-double) (* siz 4)
	  (floor siz 2))))))

(defun snd-16-sample-bytes (frm siz)
  (declare (optimize (speed 3) (safety 0)))
  (if (= frm snd-16-linear) siz
    (if (or (= frm snd-32-float) (= frm snd-32-linear)) (floor siz 2)
      (if (= frm snd-24-linear) (+ (ash siz -2) (ash siz -1))
	(if (= frm snd-64-double) (floor siz 4)
	  (* siz 2))))))

(defun snd-header-format-name (n)
  (declare (optimize (speed 3) (safety 0)))
  (cond ((= (snd-header-format n) snd-16-linear) "16-bit linear")
	((= (snd-header-format n) snd-8-mulaw) "8-bit mulaw")
	((= (snd-header-format n) snd-8-linear) "8-bit linear")
	((= (snd-header-format n) snd-32-float) "32-bit float")
	((= (snd-header-format n) snd-32-linear) "32-bit linear")
	((= (snd-header-format n) snd-8-alaw) "8-bit alaw")
	((= (snd-header-format n) snd-8-unsigned) "8-bit unsigned")
	((= (snd-header-format n) snd-24-linear) "24-bit linear")
	((= (snd-header-format n) snd-64-double) "64-bit double")
	(t (format nil "unknown (originally ~A)" 
		   (if (= (snd-header-type n) NeXT-sound-file)
		       (next-snd-data-format-name (c-snd-header-original-format))
		     (if (= (snd-header-type n) RIFF-sound-file)
			 (riff-snd-data-format-name (c-snd-header-original-format))
		       (c-snd-header-original-format)))))))

(defun snd-format-name (n)
  (declare (optimize (speed 3) (safety 0)))
  (cond ((= n snd-16-linear) "16-bit linear")
	((= n snd-8-mulaw) "8-bit mulaw")
	((= n snd-8-linear) "8-bit linear")
	((= n snd-32-float) "32-bit float")
	((= n snd-32-linear) "32-bit linear")
	((= n snd-8-alaw) "8-bit alaw")
	((= n snd-8-unsigned) "8-bit unsigned")
	((= n snd-24-linear) "24-bit linear")
	((= n snd-64-double) "64-bit double")
	(t "unknown")))

(defun snd-header-type-name (n)
  (declare (optimize (speed 3) (safety 0)))
  (cond ((= n NeXT-sound-file) "NeXT")
	((= n AIFF-sound-file) "AIFF")
	((= n RIFF-sound-file) "RIFF")
	((= n BICSF-sound-file) "BICSF")
	((= n NIST-sound-file) "NIST")
	((= n INRS-sound-file) "INRS")
	((= n ESPS-sound-file) "ESPS")
	((= n SVX-sound-file) "8SVX")
	((= n VOC-sound-file) "VOC")
	((= n SNDT-sound-file) "SndTools")
	((= n raw-sound-file) "raw")
	((= n SMP-sound-file) "SMP")
	((= n SD2-sound-file) "SoundDesigner 2")
	(t "unknown")))

(defun next-snd-data-format-name (n)
  (cond ((= n 0) "unspecified")
	((= n 1) "mulaw_8")
	((= n 2) "linear_8")
	((= n 3) "linear_16")
	((= n 4) "linear_24")
	((= n 5) "linear_32")
	((= n 6) "float")
	((= n 7) "double")
	((= n 8) "indirect")
	((= n 9) "nested")
	((= n 10) "dsp_core")
	((= n 11) "dsp_data_8")
	((= n 12) "dsp_data_16")
	((= n 13) "dsp_data_24")
	((= n 14) "dsp_data_32")
	((= n 16) "display")
	((= n 17) "mulaw_squelch")
	((= n 18) "emphasized")
	((= n 19) "compressed")
	((= n 20) "compressed_emphasized")
	((= n 21) "dsp_commands")
	((= n 22) "dsp_commands_samples")
	((= n 23) "adpcm_g721")
	((= n 24) "adpcm_g722")
	((= n 25) "adpcm_g723")
	((= n 26) "adpcm_g723_5")
	((= n 27) "alaw_8")
	(t (format nil "unknown: ~D" n))))

(defun riff-snd-data-format-name (n)
  (cond ((= n 1) "pcm")			;16-bit linear, I believe
	((= n 2) "adpcm")
	((= n 6) "alaw")
	((= n 7) "mulaw")
	;; oki-mulaw?
	((= n #x15) "digistd")
	((= n #x16) "digifix")
	((= n #x101) "ibm-mulaw")
	((= n #x102) "ibm-alaw")
	;; ibm-adpcm?
	(t (format nil "unknown: ~D" n))))


(defstruct (snd-header 
	    (:print-function
	     (lambda (n s k)
	       (declare (ignore k))
	       (format s "#<~A-snd: chans: ~A, srate: ~A, size: ~A, data location: ~A, format: ~A, comment: \"~A\">"
		       (snd-header-type-name (snd-header-type n))
		       (snd-header-chans n)
		       (snd-header-srate n)
		       (snd-bytes (snd-header-format n) (snd-header-size n))
		       (snd-header-location n)
		       (snd-header-format-name n)
		       (let ((len (length (snd-header-comment n))))
			 (if (zerop len) ""
			   (if (> len 4096)
			       (format nil "~A ... ~A" 
				       (subseq (snd-header-comment n) 0 256) 
				       (subseq (snd-header-comment n) (- len 256)))
			     (snd-header-comment n))))))))
  type srate chans location size format comment sample-size)


(defun make-space-for-comment (type n)
  (if (= type NeXT-sound-file)
      (+ 24 (* 4 (ceiling (length n) 4)))
    0))


(defun make-header (&key (channels 1) 
			 (format snd-16-linear)
			 (sampling-rate sampling-rate)
			 (info "    ")
			 data-location
			 (type default-sound-file-type))
  (declare (optimize (speed 3) (safety 0)))
  (make-snd-header :chans channels :format format :size 0 
		   :location (or data-location (make-space-for-comment type info))
		   :type type :srate sampling-rate :comment info :sample-size 2))


(defun get-snd-header-comment (name start end)
  ;; done in Lisp since we want a lisp string which can be problematic as a returned value from C in
  ;;   some cases, and we need this value only in Lisp (i.e. the C header calls won't read/write the comment)
  (declare (optimize (speed 3) (safety 0)))
  (if (and (/= start 0)
	   (/= start end))
      (let* ((len (1+ (- end start)))
	     (str (make-string len)))
	(with-open-file (fil name)
	  (file-position fil start)
	  (loop for i from 0 below len do
	    (setf (schar str i) (read-char fil))))
	(string-left-trim (list (code-char 0)) 
			  (string-right-trim (list (code-char 0)) 
					     str)) ;can't use #\null because KCL doesn't know about it
	)))

(defun AFsp-pairs (str &optional (pairs nil))
  ;; return a list of paired strings consisting of a name and a value
  ;; one of these might someday be "CLM " followed by the CLM header string.
  ;; The standard fields proposed by Kabal are "date", "user", "program", "text", "speaker", 
  ;;    "recording_conditions", "database", ":", "sample_rate", and I'd like to add "clm".
  (if (zerop (length str))
      pairs
    (let* ((nstr (if pairs str (subseq str 4))) ;toss "AFsp" at start
	   (len (length nstr))
	   (lpos (or (position (code-char 0) nstr) len))
	   (lp (subseq nstr 0 lpos))
	   (ns (if (< lpos len) (subseq nstr (1+ lpos))))
	   (rpos (position (code-char 58) lp))) ;58=#\:
      (AFsp-pairs (if ns (string-left-trim (list (code-char 0)) ns))
		  (append pairs (list (list (subseq lp 0 rpos) (subseq lp (1+ rpos)))))))))

(defun AFsp-comment (pairs)
  ;; take a list of lists of paired AFsp style comments and return a string with 
  ;;  these pairs separated by ":" with null between subsequent pairs.
  (format nil (concatenate 'string "AFsp~:{~A:~A" (make-string 1 :initial-element (code-char 0)) "~}") pairs))


(defun dac*-read-header (name)
  (let ((true-name (namestring (full-merge-pathnames name default-sound-file-name))))
    (if (= (c-read-header true-name) -1)
	(warn "can't read ~S's header" name)
      (values (c-snd-header-chans) (c-snd-header-srate)))))

(defvar no-header-sticky-srate 44100)
(defvar no-header-sticky-chans 2)
(defvar no-header-sticky-format snd-16-linear)

(defun read-header (name &key ok-formats ok-srates ok-chans)
  (declare (optimize (speed 3) (safety 0)))
  (let ((true-name (namestring (full-merge-pathnames name default-sound-file-name))))
    (if (= (c-read-header true-name) -1)
	(warn "can't open ~S" name)
      (let ((header-type (c-snd-header-type)))
        (when (and (= header-type raw-sound-file)
	           (not (y-or-n-p "no header found for ~S: defaults are ~D Hz, ~A, ~D (~A).  Are these ok? "
				  name 
				  no-header-sticky-srate 
				  (if (= no-header-sticky-chans 4) "quad"
				    (if (= no-header-sticky-chans 2) "stereo"
				      "mono"))
				  no-header-sticky-format
				  (snd-format-name no-header-sticky-format))))
	  ;; now CLM's ultra-fancy user interface springs into action!!
	  (let* ((srate (progn 
		          (print (format nil "srate (<cr>=~D):" no-header-sticky-srate))
			  (setf no-header-sticky-srate (or (read-from-string (read-line) nil nil) no-header-sticky-srate))))
		 (chans (progn 
			  (print (format nil "chans (<cr>=~D):" no-header-sticky-chans))
			  (setf no-header-sticky-chans (or (read-from-string (read-line) nil nil) no-header-sticky-chans))))
		 (data-format (progn 
				(print (format nil "data format (<cr>=~D (~A)):" 
					       no-header-sticky-format
					       (snd-format-name no-header-sticky-format)))
				(setf no-header-sticky-format (or (read-from-string (read-line) nil nil) no-header-sticky-format)))))
	    (c-set-snd-header srate chans data-format)))

        (let* ((chans (c-snd-header-chans))
	       (srate (c-snd-header-srate))
	       (data-location (c-snd-header-data-location))
	       (data-size (c-snd-header-data-size))
	       (data-format (c-snd-header-format))
	       (distributed-header (/= 0 (c-snd-header-distributed)))
	       (bytes-per-sample (c-snd-header-datum-size))
	       (header-comment (get-snd-header-comment name (c-snd-header-comment-start) (c-snd-header-comment-end))))
	  (if (and (not *ignore-header-errors*)
		   (not distributed-header))
	      (let ((len (c-true-file-length true-name)))
		(if (/= len (+ (snd-bytes data-format data-size) data-location))
		    (if *fix-header-errors*
			(setf data-size (snd-16-sample-bytes data-format (- len data-location)))
		      (warn "~S has ~D bytes but its header claims it has ~D bytes.~%" 
			    name len (+ (snd-bytes data-format data-size) data-location))))))
	  (if (and ok-formats
		   (not (member data-format ok-formats)))
	      (warn "~S format is not supported.~%" (snd-header-format-name data-format)))
	  (if (and ok-srates
		   (not (member srate ok-srates)))
	      (warn "~D is not a supported sampling rate.~%" srate))
	  (if (and ok-chans
		   (not (member chans ok-chans)))
	      (warn "~D channels is not supported.~%"))
	  (values (make-snd-header :chans chans :srate srate :format data-format :location data-location :sample-size bytes-per-sample
				   :size data-size :type (c-snd-header-type) :comment header-comment)
		  data-location		;can't assume word boundary here (.au files don't guarantee it)
		  (floor data-size (max 1 (* 2 chans)))
		  chans))))))


(defun write-header (name &optional header)
  (declare (optimize (speed 3) (safety 0)))
  (let ((true-name (namestring (full-merge-pathnames name default-sound-file-name)))
	(hdr (or header (make-header))))
    (if (= (c-write-header true-name 
			   (snd-header-type hdr) (snd-header-srate hdr) (snd-header-chans hdr) 
			   (snd-header-location hdr) (snd-header-size hdr) (snd-header-format hdr) 
			   (snd-header-comment hdr) (length (snd-header-comment hdr)))
	   -1)
	(warn "can't write ~S's header" name)
      (progn
	(if (or (= (snd-header-type hdr) AIFF-sound-file)
		(= (snd-header-type hdr) RIFF-sound-file))
	    (setf (snd-header-location hdr) (c-snd-header-data-location)))
	(values hdr (floor (snd-header-location hdr) 4) (snd-header-chans hdr))))))


(defun update-header (name header new-size)
  (declare (optimize (speed 3) (safety 0)))
  (let ((true-name (namestring (full-merge-pathnames name default-sound-file-name))))
    (setf (snd-header-size header) 
      (max (snd-header-size header)
	   (* new-size 2 (snd-header-chans header))))
    (if (= (c-update-header true-name (snd-header-type header) (snd-header-size header) 0 0)
	   -1)
	(warn "can't update ~S's header" name))))
    

(defun edit-header (name &key channels datasize datalocation offset dataformat srate end)
  (let* ((true-name (namestring (full-merge-pathnames name default-sound-file-name)))
	 (old-hdr (read-header name))
	 (old-size (snd-header-size old-hdr))
	 (old-loc (snd-header-location old-hdr)))
    (if channels (setf (snd-header-chans old-hdr) channels))
    (if datasize (setf (snd-header-size old-hdr) datasize))
    (if datalocation (setf (snd-header-location old-hdr) datalocation))
    (if dataformat (setf (snd-header-format old-hdr) dataformat))
    (if srate (setf (snd-header-srate old-hdr) srate))
    (when offset
      (let ((change (floor (* offset (snd-header-chans old-hdr) (snd-header-srate old-hdr) (snd-header-sample-size old-hdr)))))
	(incf (snd-header-location old-hdr) change)
	(decf (snd-header-size old-hdr) change)))
    (when end
      (setf (snd-header-size old-hdr)
	(min (snd-header-size old-hdr)
	     (floor (* (- end (or offset 0.0)) (snd-header-chans old-hdr) (snd-header-srate old-hdr) (snd-header-sample-size old-hdr))))))
    (if (= (c-update-header true-name (snd-header-type old-hdr) (snd-header-size old-hdr) (or (and srate (floor srate)) 0) (or dataformat 0))
	   -1)
	(warn "can't edit ~S's header" name))
    (values old-loc old-size)))


#+(and Excl NeXT)
    (progn
      (ff:defforeign 'c-read-header :entry-point "_c_read_header" :arguments '(string) :return-type :integer)
      (ff:defforeign 'c-snd-header-chans :entry-point "_c_snd_header_chans" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-srate :entry-point "_c_snd_header_srate" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-data-size :entry-point "_c_snd_header_data_size" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-datum-size :entry-point "_c_snd_header_datum_size" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-data-location :entry-point "_c_snd_header_data_location" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-format :entry-point "_c_snd_header_format" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-original-format :entry-point "_c_snd_header_original_format" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-type :entry-point "_c_snd_header_type" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-distributed :entry-point "_c_snd_header_distributed" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-comment-start :entry-point "_c_snd_header_comment_start" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-comment-end :entry-point "_c_snd_header_comment_end" :arguments nil :return-type :integer)
      (ff:defforeign 'c-snd-header-type-specifier :entry-point "_c_snd_header_type_specifier" :arguments nil :return-type :integer)
      (ff:defforeign 'c-set-snd-header :entry-point "_c_set_snd_header" :arguments '(fixnum fixnum fixnum) :return-type :void)
      (ff:defforeign 'c-write-header :entry-point "_c_write_header" 
		     :arguments '(string fixnum fixnum fixnum fixnum fixnum fixnum string fixnum)
		     :return-type :integer)
      (ff:defforeign 'c-update-header :entry-point "_c_update_header" 
		     :arguments '(string fixnum fixnum fixnum fixnum)
		     :return-type :integer)
      )

#+(and Excl SGI)
    (ff:defforeign-list
     `((c-read-header :entry-point "c_read_header" 
		      :arguments (string) 
		      :return-type :integer
		      :call-direct  ,*headers-call-direct*
		      :arg-checking ,*headers-check-args*
		      :prototype    ,*headers-prototyped*)
       (c-snd-header-chans :entry-point "c_snd_header_chans" 
			   :arguments nil 
			   :return-type :integer
			   :call-direct  ,*headers-call-direct*
			   :arg-checking ,*headers-check-args*
			   :prototype    ,*headers-prototyped*)
       (c-snd-header-srate :entry-point "c_snd_header_srate" 
			   :arguments nil 
			   :return-type :integer
			   :call-direct  ,*headers-call-direct*
			   :arg-checking ,*headers-check-args*
			   :prototype    ,*headers-prototyped*)			 
       (c-snd-header-data-size :entry-point "c_snd_header_data_size" 
			       :arguments nil 
			       :return-type :integer
			       :call-direct  ,*headers-call-direct*
			       :arg-checking ,*headers-check-args*
			       :prototype    ,*headers-prototyped*)
       (c-snd-header-datum-size :entry-point "c_snd_header_datum_size" 
				:arguments nil 
				:return-type :integer
				:call-direct  ,*headers-call-direct*
				:arg-checking ,*headers-check-args*
				:prototype    ,*headers-prototyped*)
       (c-snd-header-data-location :entry-point "c_snd_header_data_location" 
				   :arguments nil 
				   :return-type :integer
				   :call-direct  ,*headers-call-direct*
				   :arg-checking ,*headers-check-args*
				   :prototype    ,*headers-prototyped*)
       (c-snd-header-format :entry-point "c_snd_header_format" 
			    :arguments nil 
			    :return-type :integer
			    :call-direct  ,*headers-call-direct*
			    :arg-checking ,*headers-check-args*
			    :prototype    ,*headers-prototyped*)
       (c-snd-header-original-format :entry-point "c_snd_header_original_format" 
				     :arguments nil 
				     :return-type :integer
				     :call-direct  ,*headers-call-direct*
				     :arg-checking ,*headers-check-args*
				     :prototype    ,*headers-prototyped*)
       (c-snd-header-type :entry-point "c_snd_header_type" 
			  :arguments nil 
			  :return-type :integer
			  :call-direct  ,*headers-call-direct*
			  :arg-checking ,*headers-check-args*
			  :prototype    ,*headers-prototyped*)
       (c-snd-header-distributed :entry-point "c_snd_header_distributed" 
				 :arguments nil 
				 :return-type :integer
				 :call-direct  ,*headers-call-direct*
				 :arg-checking ,*headers-check-args*
				 :prototype    ,*headers-prototyped*)
       (c-snd-header-comment-start :entry-point "c_snd_header_comment_start" 
				   :arguments nil 
				   :return-type :integer
				   :call-direct  ,*headers-call-direct*
				   :arg-checking ,*headers-check-args*
				   :prototype    ,*headers-prototyped*)
       (c-snd-header-comment-end :entry-point "c_snd_header_comment_end" 
				 :arguments nil 
				 :return-type :integer
				 :call-direct  ,*headers-call-direct*
				 :arg-checking ,*headers-check-args*
				 :prototype    ,*headers-prototyped*)
       (c-snd-header-type-specifier :entry-point "c_snd_header_type_specifier" 
				    :arguments nil 
				    :return-type :integer
				    :call-direct  ,*headers-call-direct*
				    :arg-checking ,*headers-check-args*
				    :prototype    ,*headers-prototyped*)
       (c-set-snd-header :entry-point  "c_set_snd_header" 
			 :arguments    (fixnum fixnum fixnum) 
			 :return-type  :void
			 :call-direct  ,*headers-call-direct*
			 :arg-checking ,*headers-check-args*
			 :prototype    ,*headers-prototyped*)
       (c-write-header :entry-point "c_write_header" 
		       :arguments (string integer integer integer integer integer integer string integer)
		       :return-type :integer
		       :call-direct  ,*headers-call-direct*
		       :arg-checking ,*headers-check-args*
		       :prototype    ,*headers-prototyped*)
       (c-update-header :entry-point "c_update_header" 
			:arguments (string integer integer integer integer)
			:return-type :integer
			:call-direct  ,*headers-call-direct*
			:arg-checking ,*headers-check-args*
			:prototype    ,*headers-prototyped*)
       ))

#+MCL
    (progn
      (ccl:deffcfun (c-read-header "c_read_header") (string) :long)
      (ccl:deffcfun (c-snd-header-chans "c_snd_header_chans") () :long)
      (ccl:deffcfun (c-snd-header-srate "c_snd_header_srate") () :long)
      (ccl:deffcfun (c-snd-header-data-size "c_snd_header_data_size") () :long)
      (ccl:deffcfun (c-snd-header-datum-size "c_snd_header_datum_size") () :long)
      (ccl:deffcfun (c-snd-header-data-location "c_snd_header_data_location") () :long)
      (ccl:deffcfun (c-snd-header-format "c_snd_header_format") () :long)
      (ccl:deffcfun (c-snd-header-original-format "c_snd_header_original_format") () :long)
      (ccl:deffcfun (c-snd-header-type "c_snd_header_type") () :long)
      (ccl:deffcfun (c-snd-header-distributed "c_snd_header_distributed") () :long)
      (ccl:deffcfun (c-snd-header-comment-start "c_snd_header_comment_start") () :long)
      (ccl:deffcfun (c-snd-header-comment-end "c_snd_header_comment_end") () :long)
      (ccl:deffcfun (c-snd-header-type-specifier "c_snd_header_type_specifier") () :long)
      (ccl:deffcfun (c-set-snd-header "c_set_snd_header") ((fixnum :long) (fixnum :long) (fixnum :long)) :novalue)
      (ccl:deffcfun (c-write-header "c_write_header" )
		     ((string) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long) 
		      (fixnum :long) (fixnum :long) (string) (fixnum :long))
		     :long)
      (ccl:deffcfun (c-update-header "c_update_header") 
		     ((string) (fixnum :long) (fixnum :long) (fixnum :long) (fixnum :long))
		     :long)

      (defun read-sound-designer-resource-info (name)
	(ccl::with-open-resource-file (f name)
	  (values (ccl::get-string "sample-rate")
		  (ccl::get-string "channels")
		  (ccl::get-string "sample-size"))))
      ;; there is also the file comment as a pascal string (str255) as 4th field (10 bytes offset) of "sdDD" id 1000
      )



;;; ------------------------------------------------------------------------
;;; error checks for headers

#-I386 (defconstant Next-type-specifier #x2e736e64) ; ".snd" 
#+I386 (defconstant Next-type-specifier #x646e732e) ; "dns." -- from 386 point of view it's swapped

(defun Next-sound-file-error-checks (nam)
  ;;check for header problems and what-not since Next's errors are uninformative, and
  ;;the SndFree routine appears to hang onto the dsp if there is an error (causing endless pain).
  (let* ((hdr (read-header nam))
	 (mag (c-snd-header-type-specifier)))
    (if (/= Next-type-specifier mag)
	(warn "~A does not start with the Next '.snd' type indication: ~X /= ~X" nam mag Next-type-specifier)
      (if (not (plusp (snd-header-size hdr)))
	  (warn "header claims we have ~D bytes of information here" (snd-header-size hdr))
	(if (and (not *sound-player*)	;if user-specified dac routine, all bets are off here
		 (/= 1 (snd-header-chans hdr)) 
		 (/= 2 (snd-header-chans hdr)))
	    (warn "built-in DAC routine can only handle mono and stereo files, not ~D channels" (snd-header-chans hdr))
	  (if (and (/= snd-16-linear (snd-header-format hdr))
		   (/= snd-8-mulaw (snd-header-format hdr)))
	      (warn "Unsupported format for built-in DAC routine: ~A" (snd-header-format-name hdr))
	    (if (and (/= 22050 (snd-header-srate hdr))
		     (/= 44100 (snd-header-srate hdr))
		     (/= 8012 (snd-header-srate hdr)))
		(warn "unsupported sampling-rate for built-in DAC routine: ~D" (snd-header-srate hdr))
	      ;; at one time I thought "hey, why not play it anyway?" and thereupon hung the NeXT so
	      ;; thoroughly that even the non-maskable interrupt was masked!  Had to pull the power plug
	      ;; and sweat through the fsck disk check -- "not worth it" I said to myself...
	      (let* ((len (c-true-file-length nam)))
		(if (/= len (+ (snd-header-location hdr) (snd-bytes (snd-header-format hdr) (snd-header-size hdr))))
		    (warn "header info and data size /= actual file total size: ~D + ~D /= ~D" 
			  (snd-header-location hdr) (snd-header-size hdr) len)
		  t)))))))))

