;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;

(in-package :clm)

;;;
;;; translation of most of the FORTRAN programs given in "Digital Filter Design" by Parks and Burrus
;;;
#|
   Documentation:

   The programs in "Digital Filter Design" are designated below as
   design-FIR-n and design-IIR-n where n is the P&B program number.
   Thus, design-FIR-1 is their program 1.  See P&B for full details.
   What follows here is my attempt to try these guys out.  All frequency
   specifications assume a sampling rate of 1.0.  All arrays start at
   1 (as per FORTRAN) -- ignore the 0-th element.  P&B program 7 calls
   a subroutine named CHEB which is omitted, and I can't find any such
   routine in the standard FORTRAN libraries.


   ;design-FIR-1: linear phase filter design using frequency sampling.
   (setf x (design-FIR-1))      ;4-th order lowpass cutoff at .25
   => -0.1035 0.603             ;half the symmetric impulse response
   (freq x 4 25)                ;show response at 25 points
                                ;creates a list of x-coefficients for filter in mus.lisp
   (setf x (design-FIR-1 :order 21))
   ; this is P&B example 3.1 -- coefficients should match those on pp39-40
   (freq x 21 50)               ;should match figure 3.2a
   (make-filter-coefficients 21 x)
   (setf x (design-FIR-1 :order 20))
   ; P&B example 3.2 --  coefficients on pp 41-42
   (freq x 20 50)               ;should match fig 3.3a


   ;design-FIR-2: linear phase low-pass FIR filter design using a discrete least squared error criterion
   (setf x (design-FIR-2 :order 21 :frequency-samples 81 :sample-at-dc t))
   ; P&B example 3.3 -- coeffs on p49


   ;design-FIR-3: linear phase low-pass FIR filter using LS error criterion and a transition region
   (setf x (design-FIR-3 :order 21 :passband-edge .2 :stopband-edge .3 :transition-type 1))
   ; P&B example 3.9 p 67


   ;design-FIR-4: linear phase low-pass FIR using LS error criterion and optional windows
   (setf x (design-FIR-4 :order 21))
   ; P&B example 3.6 pp57-58
   (setf x (design-FIR-4 :order 21 :window Hanning))
   ; P&B example 3.10 pp73-77
   (setf x (design-FIR-4 :order 21 :window kaiser :beta 4))
   ; P&B example 3.11 p77


   ;design-FIR-5: linear phase FIR differentiator using LS error criterion
   (setf x (design-FIR-5 :order 21))
   ; P&B example 3.7 p63 (note that they have printed -h(0) etc)
   (setf x (design-FIR-5 :order 20))
   ; P&B example 3.8 p63 (note sign reversal)


   ;design-FIR-6: linear phase remez exchange FIR design
   (setf x (design-FIR-6 :order 21 
                         :type multiple-band 
                         :bands 2 
                         :band-edges (make-array 5 :element-type 'float 
                                                   :initial-contents '(0.0 0.0 0.33 0.37 0.5)) 
                         :functions (make-array 3 :element-type 'float 
                                                  :initial-contents '(0.0 1.0 0.0)) 
                         :weights (make-array 3 :element-type 'float 
                                                :initial-contents '(0.0 1.0 1.0))))
   ;P&B example on p290 (low h values seem off a little, but higher are ok)
   ;P&B example 3.15 tries out even order and p101 tries 55th order -- same observation as above
   ;there may be problems with more than 2 bands (numbers seem to be off)

   (setf x (design-FIR-6 :order 21 
                         :type multiple-band 
                         :bands 3 
                         :band-edges (make-array 7 :element-type 'float 
                                                   :initial-contents '(0.0 0.0 0.14 0.18 0.33 0.37 0.5)) 
                         :functions (make-array 4 :element-type 'float 
                                                  :initial-contents '(0.0 0.0 1.0 0.0)) 
                         :weights (make-array 4 :element-type 'float 
                                                :initial-contents '(0.0 1.0 1.0 1.0))))

   ;design-FIR-8: FIR least-square design program
   ;P&B example from p303
    (setf x (design-FIR-8 :order 21 
                         :type multiple-band 
                         :bands 2 
                         :group-delay 8.0 
                         :group-delay-error-weight 1.0 
                         :phase-error-weight 1.0
                         :band-edges (make-array 5 :element-type 'float 
                                                   :initial-contents '(0.0 0.0 0.33 0.37 0.50)) 
                         :functions (make-array 3 :element-type 'float 
                                                  :initial-contents '(0.0 1.0 0.0)) 
                         :weights (make-array 3 :element-type 'float 
                                                :initial-contents '(0.0 1.0 1.0))))
   ;this is so slow that I C-c'd out of it.


   ;These next two run, but I haven't checked the results at all.
   ;design-IIR-9: IIR filter design using Butterworth, Chebychev, and Elliptic function approximations
   (setf x (design-IIR-9 :class Butterworth :type lowpass :order 4 :band-edge .3)) 


   ;design-IIR-10: IIR filter design using LS error criterion and frequency sampling
   (setf x (design-IIR-10 :numerator-order 6 
			  :denominator-order 6 
			  :frequency-samples 41 
                          :real-samples (loop for i from 0 to 41 collect (if (< i 9) .2 0.0))
                          :imaginary-samples (loop for i from 0 to 41 collect (random 1.0))))


  ;given this instrument:

    (definstrument filter-noise (beg dur amp &key (x-coeffs '(.5 .2 .1)) (y-coeffs nil) (filter-type direct-form))
      (let* ((st (floor (* beg sampling-rate)))
	     (noi (make-randh :frequency (* .5 sampling-rate) :amplitude amp))
	     (flA (make-filter :type filter-type
		  	       :x-coeffs x-coeffs
			       :y-coeffs y-coeffs))
	     (nd (+ st (floor (* sampling-rate dur)))))
        (Run
         (loop for i from st to nd do
           (outa i (filter flA (randh noi)))))))

  ;we can filter noise with:
   (with-sound () 
     (filter-noise 0 1 .2 
		   :x-coeffs (make-filter-coefficients 
			      12 
			      (design-FIR-with-env 
			       :order 12 
			       :envelope '(0 0.0 .125 0.5 .200 0.0 .300 1.0 .500 0.0 1.000 0.0))))) 



|#

(defun sqr (x) (* x x))


(defun make-filter-coefficients (n h)
  (let ((len (floor n 2)))
    (append 
     (loop for i from 1 to len collect (coerce (aref h i) 'single-float))
     (if (oddp n) 
	 (loop for i from (1+ len) downto 1 collect (coerce (aref h i) 'single-float))
       (loop for i from len downto 1 collect (coerce (aref h i) 'single-float))))))

(defun F4-div (a b) (if (zerop b) a (/ a b)))


;;; program 1 -- FIR filters using frequency sampling or interpolation

(defun design-FIR-1 (&key (order 4)
			  (band-edge .25)
			  (dc 0))
  (let* ((x (make-array 102 :element-type 'float :initial-element 0.0))
	 (a (make-array 102 :element-type 'float :initial-element 0.0))
	 (n order)
	 (fp band-edge)
	 (m (floor (1+ n) 2))
	 (am (/ (+ n 1.0) 2.0))
	 (m1 (floor (+ (/ n 2) 1)))
	 (q (/ (* 2 pi) n))
	 (n2 (floor n 2))
	 (np (floor (+ (* n fp) 1.0))))
    (if (not (zerop dc)) (setf np (floor (+ (* n fp) .5))))
    (loop for j from 1 to np do
      (setf (aref a j) 1.0))
    (loop for j from (1+ np) to m1 do
      (setf (aref a j) 0.0))
    (if (zerop dc)
	(progn
	  (loop for j from 1 to m do
	    (let ((xt (/ (aref a 1) 2)))
	      (loop for i from 2 to m do
		(incf xt (* (aref a i) (cos (* q (- am j) (- i 1))))))
	      (setf (aref x j) (/ (* 2.0 xt) n)))))
      (progn
	(loop for j from 1 to m do
	  (let ((xt 0))
	    (loop for i from 1 to n2 do
	      (incf xt (* (aref a i) (cos (* q (- am j) (- i 0.5))))))
	    (if (/= am m)
		(incf xt (* (aref a m) .5 (cos (* pi (- am j))))))
	    (setf (aref x j) (/ (* 2 xt) n))))))
    x))


(defun design-FIR-from-env (&key (order 4)
				 (envelope nil)
				 (dc 0))
  (let* ((x (make-array 102 :element-type 'float :initial-element 0.0))
	 (a (make-array 102 :element-type 'float :initial-element 0.0))
	 (n order)
	 (last-x (env-last-x envelope))
	 (freq-scl (/ last-x order))
	 (m (floor (1+ n) 2))
	 (am (/ (+ n 1.0) 2.0))
	 (m1 (floor (+ (/ n 2) 1)))
	 (q (/ (* 2 pi) n))
	 (n2 (floor n 2)))
    (loop for j from 1 to m1 and fr from 0 by freq-scl do
      (setf (aref a j) (eref envelope fr)))
    (if (zerop dc)
	(progn
	  (loop for j from 1 to m do
	    (let ((xt (/ (aref a 1) 2)))
	      (loop for i from 2 to m do
		(incf xt (* (aref a i) (cos (* q (- am j) (- i 1))))))
	      (setf (aref x j) (/ (* 2.0 xt) n)))))
      (progn
	(loop for j from 1 to m do
	  (let ((xt 0))
	    (loop for i from 1 to n2 do
	      (incf xt (* (aref a i) (cos (* q (- am j) (- i 0.5))))))
	    (if (/= am m)
		(incf xt (* (aref a m) .5 (cos (* pi (- am j))))))
	    (setf (aref x j) (/ (* 2 xt) n))))))
    x))

(defun freq (x n k)
  (let* ((a (make-array 102 :element-type 'float :initial-element 0.0))
	 (q (/ pi k))
	 (am (* (1+ n) 0.5))
	 (m (floor (1+ n) 2))
	 (n2 (floor n 2)))
    (loop for j from 1 to (1+ k) do
      (let* ((at 0.0))
	(if (= am m) (setf at (* 0.5 (aref x m))))
	(loop for i from 1 to n2 do
	  (incf at (* (aref x i) (cos (* q (- am i) (1- j))))))
	(setf (aref a j) (* 2 at))))
    (loop for j from 1 to (1+ k) do
      (print (format t "~F: ~F" (/ (* 0.5 (1- j)) k) (abs (coerce (aref a j) 'single-float)))))))


;;; program 2 uses SQRDC and SQRSL from LINPACK translated below
;;; FORTRAN is column major order, but Common Lisp is row major order
(defstruct (bigarr 
	    (:print-function
	     (lambda (d s k)
	       (format s "<[~A:~A] (~{~A ~} ...)>"
		       (bigarr-a d) (bigarr-b d) 
		       (loop for j from 0 below 8
			collect (aref (bigarr-arr d) j))))))
  arr a b)				;goddam lisp prints the entire array in stack trace and I can't turn it off
(defun make-bigarray (a b)
  (make-bigarr :a a :b b :arr (make-array (1+ (* a b)) :element-type 'float :initial-element 0.0)))
(defun faref (x a &optional b)
  (if b
      (aref (bigarr-arr x) (+ a (* (1- b) (bigarr-a x))))
    (aref x a)))
(defun setf-faref (x a b val)
  (setf (aref (bigarr-arr x) (+ a (* (1- b) (bigarr-a x)))) val))
(defsetf faref setf-faref)
(defun paref (x a &optional b)
  (if b
      (let* ((total-size (array-total-size (bigarr-arr x)))
	     (offset (+ a (* (1- b) (bigarr-a x)) -1))
	     (fake-size (if (> (+ (bigarr-a x) offset) total-size)
			    (- total-size offset 1)
			  (bigarr-a x))))
	(make-array fake-size
		    :displaced-to (bigarr-arr x)
		    :displaced-index-offset offset))
    (let* ((total-size (array-total-size x))
	   (offset (1- a))
	   (fake-size (- total-size offset 1)))
      (make-array fake-size
		  :displaced-to x
		  :displaced-index-offset offset))))



(defun design-FIR-2 (&key (order 4)
			  (band-edge .25)
			  (sample-at-dc nil)
			  (frequency-samples 100))
  (let* ((x (make-array 102 :element-type 'float :initial-element 0.0))
	 (a (make-array 502 :element-type 'float :initial-element 0.0))
	 (f (make-bigarray 501 102))
	 (qax (make-array 102 :element-type 'float :initial-element 0.0)) 
	 (ldx 501)
	 (n order)
	 (l (max frequency-samples order))
	 (fp band-edge)
	 (dc (if sample-at-dc 0 1))
	 (m (floor (1+ n) 2))
	 (am (float (/ (1+ n) 2.0)))
	 (lm (floor (1+ l) 2))
	 (l2 (1+ (floor l 2)))
	 (lp (+ 1.0 (* fp l)))
	 (qj 1.0)
	 (dum 0.0)
	 (info 0.0))
    (if (/= dc 0)
	(progn
	  (setf l2 lm)
	  (setf lp (+ 0.5 (* fp l)))
	  (setf qj 0.5)
	  (loop for j from 1 to l do
	    (setf (aref a j) 0.0))
	  (if (oddp n)
	      (loop for j from 1 to lp do
		(setf (aref a j) 1.0)
		(setf (aref a (1+ (- l j))) 1.0))
	    (loop for j from 1 to lp do
		(setf (aref a j) 1.0)
		(setf (aref a (1+ (- l j))) -1.0))))
      (progn				;dc 0
	(loop for j from 1 to l do
	  (setf (aref a j) 0.0))
	(if (oddp n)
	    (progn
	      (setf (aref a 1) 1.0)
	      (loop for j from 2 to lp do
		(setf (aref a j) 1.0)
		(setf (aref a (+ 2 (- l j))) 1.0)))
	  (progn
	    (setf (aref a 1) 1.0)
	    (loop for j from 2 to lp do
	      (setf (aref a j) 1.0)
	      (setf (aref a (+ 2 (- l j))) -1.0))))))
    (let* ((q (/ (* 2.0 pi) l))
	   (q1 0.0))
      (loop for i from 1 to m do
	(setf q1 (* q (- am i)))
	(loop for j from 1 to l do
	  (setf (faref f j i) (* 2 (cos (* q1 (- j qj))))))))
    (sqrdc f ldx l m qax dum dum 0)
    (sqrsl f ldx l m qax a dum a x dum dum 100 info)
    (if (oddp n) (setf (aref x m) (* 2.0 (aref x m))))
    x))
			  

;;; program 3 -- Linear phase low-pass FIR filter using least squared error criterion and a transition region

(defun ls (x n fp fq)
  (declare (ignore fq))
  (let* ((m (floor (1+ n) 2))
	 (am (/ (1+ n) 2.0))
	 (n2 (floor n 2)))
    (if (= m am) (setf (aref x m) fp))
    (loop for j from 1 to n2 do
      (let ((q (* pi (- j am))))
	(setf (aref x j) (/ (sin (* fp q)) q))))))

(defun wgt (x n tp fp fq)
  (declare (ignore fp))
  (when (not (zerop fq))
    (let* ((am (/ (1+ n) 2.0))
	   (n2 (floor n 2))
	   (q (* pi fq)))
      (if (zerop tp)
	  (progn
	    (loop for j from 1 to n2 do
	      (let ((wt (cos (* q (- j am)))))
		(if (> (abs wt) 1.0e-6)
		    (setf wt (/ wt (- 1 (sqr (* 2 fq (- j am))))))
		  (setf wt (/ pi 4.0)))
		(setf (aref x j) (* wt (aref x j))))))
	(progn
	  (loop for j from 1 to n2 do
	    (let* ((q1 (/ (* q (- j am)) tp))
		   (wt (expt (/ (sin q1) q1) tp)))
	      (setf (aref x j) (* (aref x j) wt)))))))))

;;; transition type: 0 raised-cosine, 1 linear, 2 2nd order, etc.  if fp=fs, no transition
(defun design-FIR-3 (&key (order 4) 
			  (passband-edge .25)
			  (stopband-edge .5)
			  (transition-type 0))
  (let* ((x (make-array 102 :element-type 'float :initial-element 0.0))
	 (tp transition-type)
	 (n order)
	 (fp passband-edge)
	 (fs stopband-edge)
;	 (m (floor (1+ n) 2))
	 (fq (- fs fp))
	 (fr (+ fs fp)))
    (ls x n fr fq)
    (wgt x n tp fr fq)
    x))
	 

;;; program 4 -- FIR linear phase lowpass using least squared error and windows

(defconstant Rectangular 0)
(defconstant Bartlett 1)
(defconstant Hanning 2)
(defconstant Hamming 3)
(defconstant Blackman 4)
(defconstant Kaiser 5)

(defun ls2 (x n fp)
  (let* ((m (floor (1+ n) 2))
	 (am (/ (1+ n) 2.0))
	 (n2 (floor n 2)))
    (if (= m am) (setf (aref x m) (* 2.0 fp)))
    (loop for j from 1 to n2 do
      (let* ((q (* pi (- j am))))
	(setf (aref x j) (/ (sin (* fp 2 q)) q))))))

(defun FI0 (z)
  (let* ((y (/ z 2.0))
	 (e 1.0)
	 (d 1.0)
	 (d2 0.0))
    (loop for j from 1 to 25 do
      (setf d (/ (* d y) j))
      (setf d2 (sqr d))
      (incf e d2)
      (if (< d2 (* e 1.0e-7)) (return-from FI0 e)))
    (print "I0 failed to converge")
    e))
	
(defun wind (x n tp beta)
  (when (not (= tp Rectangular))
    (let* ((am (/ (1+ n) 2.0))
	   (q (/ pi am))
	   (q1 (/ pi (1- am))))
      (if (= tp Bartlett)
	  (loop for j from 1 to am do (setf (aref x j) (* (aref x j) (/ j am))))
	(if (= tp Hanning)
	    (loop for j from 1 to am do (setf (aref x j) (* (aref x j) (- 0.5 (* 0.5 (cos (* j q)))))))
	  (if (= tp Hamming)
	      (loop for j from 1 to am do (setf (aref x j) (* (aref x j) (- 0.54 (* 0.46 (cos (* (1- j) q1)))))))
	    (if (= tp Blackman)
		(loop for j from 1 to am do (setf (aref x j) (* (aref x j) (+ (- 0.42 (* 0.5 (cos (* j q)))) (* 0.08 (cos (* j q 2)))))))
	      (if (/= tp Kaiser)
		  (error "unknown window: ~A" tp)
		(let ((fio1 (FI0 beta)))
		  (loop for j from 1 to am do 
		    (setf (aref x j) 
		      (* (aref x j) (/ (FI0 (* beta (sqrt (- 1.0 (sqr (/ (- am j) (- am 1))))))) fio1)))))))))))))

(defun design-FIR-4 (&key (order 4)
			  (band-edge .25)
			  (window Rectangular)
			  (beta 3))
  (let* ((tp Window)
	 (n order)
	 (fp band-edge)
	 (x (make-array 101 :element-type 'float :initial-element 0.0)))
    (ls2 x n fp)
    (wind x n tp beta)
    x))


;;; program 5 -- FIR differentiator using least squared error criterion

(defun design-FIR-5 (&key (order 4))
  (let* ((x (make-array 102 :element-type 'float :initial-element 0.0))
	 (n order)
	 (m (floor (1+ n) 2))
	 (am (/ (1+ n) 2.0))
	 (n2 (floor n 2)))
    (if (/= m am) 
	(loop for j from 1 to n2 do
	  (let ((q (* pi (- j am))))
	    (setf (aref x j) (/ (sin q) (* q (- j am))))))
      (progn
	(loop for j from 1 to (1- m) do
	  (setf (aref x j) (/ (cos (* pi (- m j))) (- m j))))
	(setf (aref x m) 0.0)))
    x))



;;; program 6 -- FIR linear phase design program

(defconstant multiple-band 1)
(defconstant differentiator 2)
(defconstant Hilbert-transformer 3)


;;; gee -- evaluate frequency response using Lagrange interpolation formula in barycentric form
(defun gee (k n grid ad x y)
  (let* ((pi2 (* 2.0 pi))
	 (p 0.0)
	 (xf (cos (* pi2 (aref grid k))))
	 (d 0.0))
    (loop for j from 1 to n do
      (let* ((c (/ (aref ad j) (- xf (aref x j)))))
	(incf d c)
	(incf p (* c (aref y j)))))
    (/ p d)))

;;; d -- calculate Lagrange interpolation coefficients for gee (above)
(defun d (k n m x)
  (let* ((d1 1.0)
	 (q (aref x k)))
    (loop for l from 1 to m do
      (loop for j from l to n by m do
	(if (not (zerop (- j k)))
	    (setf d1 (* 2.0 d1 (- q (aref x j)))))))
    (/ 1.0 d1)))

;;; eff -- calculate desired magnitude frequency response
(defun eff (freq fx wtx lband jtype)
  (declare (ignore wtx))
  (if (= jtype 2)
      (* freq (aref fx lband))
    (aref fx lband)))

;;; wate -- calculate weight function
(defun wate (freq fx wtx lband jtype)
  (if (or (/= jtype 2)
	  (>= (aref fx lband) 0.0001))
      (F4-div (aref wtx lband) freq)
    (aref wtx lband)))
    
;;; remes -- Remes exchange algorithm for weighted Chebychev approximation of continuous
;;;   functions with a sum of cosines.

(defvar p6-dev 0.0)

(defun remes (ad x y grid des wt alpha iext nfcns ngrid)
  (let* ((pi2 (* 2.0 pi))
	 (itrmax 25)
	 (devl -1.0)
	 (nz (1+ nfcns))
	 (nzz (+ nfcns 2))
	 (ynz 0.0)
	 (comp 0.0)
	 (dtemp 0.0)
	 (dnum 0.0)
	 (dden 0.0)
	 (jet 0)
	 (k 0)
	 (jchnge 0)
	 (k1 0)
	 (knz 0)
	 (klow 0)
	 (kup 0)
	 (l 0)
	 (y1 0.0)
	 (nut 0)
	 (nu 0)
	 (j 0)
	 (cn 0.0)
	 (delf 0.0)
	 (kkk 0)
	 (xe 0.0)
	 (xt 0.0)
	 (xt1 0.0)
	 (aa 0.0)
	 (bb 0.0)
	 (nm1 0)
	 (jm1 0)
	 (jp1 0)
	 (nf1j 0)
	 (err 0)
	 (nut1 0)
	 (kn 0)
	 (ft 0.0)
	 (fsh 0.0)
	 (gtemp 0.0)
	 (luck 0)
	 (alpha-size (array-dimension alpha 0))
	 (a (make-array alpha-size :element-type 'float :initial-element 0.0))
	 (p (make-array alpha-size :element-type 'float :initial-element 0.0))
	 (q (make-array alpha-size :element-type 'float :initial-element 0.0))
	 (niter 0))
    (tagbody
     100 
      (setf (aref iext nzz) (1+ ngrid))
      (incf niter)
      (if (> niter itrmax) (go 400))
      (loop for j from 1 to nz do
	(setf (aref x j) (cos (* pi2 (aref grid (aref iext j))))))
      (setf jet (floor (1+ (/ (1- nfcns) 15))))
      (loop for j from 1 to nz do
	(setf (aref ad j) (d j nz jet x)))
      (setf dnum 0.0
	    dden 0.0
	    k 1)
      (loop for j from 1 to nz do
	(setf l (aref iext j))
	(setf dtemp (* (aref ad j) (aref des l)))
	(incf dnum dtemp)
	(setf dtemp (/ (* (float k) (aref ad j)) (aref wt l)))
	(incf dden dtemp)
	(setf k (- k)))
      (setf p6-dev (/ dnum dden))
      (setf nu (if (> p6-dev 0.0) -1 1))
      (setf p6-dev (- (* nu p6-dev)))
      (setf k nu)
      (loop for j from 1 to nz do
	(setf l (aref iext j))
	(setf dtemp (/ (* k p6-dev) (aref wt l)))
	(setf (aref y j) (+ dtemp (aref des l)))
	(setf k (- k)))
      (if (<= p6-dev devl)
	  (progn
	    (error "failure to converge")
	    (go 400)))
      (setf devl p6-dev)
      (setf jchnge 0
	    k1 (aref iext 1)
	    knz (aref iext nz)
	    klow 0
	    nut (- nu)
	    j 1)
     200
      (if (= j nzz) (setf ynz comp))
      (if (>= j nzz) (go 300))
      (setf kup (aref iext (1+ j)))
      (setf l (1+ (aref iext j)))
      (setf nut (- nut))
      (if (= j 2) (setf y1 comp))
      (setf comp p6-dev)
      (if (>= l kup) (go 220))
      (setf err (* (- (gee l nz grid ad x y) 
		      (aref des l)) 
		   (aref wt l)))
      (setf dtemp (- (* (float nut) err) comp))
      (if (<= dtemp 0.0) (go 220))
      (setf comp (* (float nut) err))
     210
      (incf l)
      (if (>= l kup) (go 215))
      (setf err (* (- (gee l nz grid ad x y) 
		      (aref des l)) 
		   (aref wt l)))
      (setf dtemp (- (* (float nut) err) comp))
      (if (<= dtemp 0.0) (go 215))
      (setf comp (* (float nut) err))
      (go 210)
     215
      (setf (aref iext j) (1- l))
      (incf j)
      (setf klow (1- l))
      (incf jchnge)
      (go 200)
     220
      (decf l)
     225
      (decf l)
      (if (<= l klow) (go 250))
      (setf err (* (- (gee l nz grid ad x y) 
		      (aref des l)) 
		   (aref wt l)))
      (setf dtemp (- (* (float nut) err) comp))
      (if (> dtemp 0.0) (go 230))
      (if (<= jchnge 0) (go 225))
      (go 260)
     230
      (setf comp (* (float nut) err))
     235
      (decf l)
      (if (<= l klow) (go 240))
      (setf err (* (- (gee l nz grid ad x y) 
		      (aref des l)) 
		   (aref wt l)))
      (setf dtemp (- (* (float nut) err) comp))
      (if (<= dtemp 0.0) (go 240))
      (setf comp (* (float nut) err))
      (go 235)
     240
      (setf klow (aref iext j))
      (setf (aref iext j) (1+ l))
      (incf j)
      (incf jchnge)
      (go 200)
     250
      (setf l (1+ (aref iext j)))
      (if (> jchnge 0) (go 215))
     255
      (incf l)
      (if (>= l kup) (go 260))
      (setf err (* (- (gee l nz grid ad x y) 
		      (aref des l)) 
		   (aref wt l)))
      (setf dtemp (- (* (float nut) err) comp))
      (if (<= dtemp 0.0) (go 255))
      (setf comp (* (float nut) err))
      (go 210)
     260
      (setf klow (aref iext j))
      (incf j)
      (go 200)
     300
      (if (> j nzz) (go 320))
      (setf k1 (min k1 (aref iext 1)))
      (setf knz (max knz (aref iext nz)))
      (setf nut1 nut)
      (setf nut (- nu))
      (setf l 0)
      (setf kup k1)
      (setf comp (* ynz 1.00001))
      (setf luck 1)
     310
      (incf l)
      (if (>= l kup) (go 315))
      (setf err (* (- (gee l nz grid ad x y) 
		      (aref des l)) 
		   (aref wt l)))
      (setf dtemp (- (* (float nut) err) comp))
      (if (<= dtemp 0.0) (go 310))
      (setf comp (* (float nut) err))
      (setf j nzz)
      (go 210)
     315
      (setf luck 6)
      (go 325)
     320
      (if (> luck 9) (go 350))
      (setf y1 (max y1 comp))
      (setf k1 (aref iext nzz))
     325
      (setf l (1+ ngrid))
      (setf klow knz)
      (setf nut (- nut1))
      (setf comp (* y1 1.00001))
     330
      (decf l)
      (if (<= l klow) (go 340))
      (setf err (* (- (gee l nz grid ad x y) 
		      (aref des l)) 
		   (aref wt l)))
      (setf dtemp (- (* (float nut) err) comp))
      (if (<= dtemp 0.0) (go 330))
      (setf comp (* (float nut) err))
      (setf j nzz)
      (incf luck 10)
      (go 235)
     340
      (if (= luck 6) (go 370))
      (loop for j from 1 to nfcns do
	(let* ((nzzmj (- nzz j))
	       (nzmj (- nz j)))
	  (setf (aref iext nzzmj) (aref iext nzmj))))
      (setf (aref iext 1) k1)
      (go 100)
     350
      (setf kn (aref iext nzz))
      (loop for j from 1 to nfcns do
	(setf (aref iext j) (aref iext (1+ j))))
      (setf (aref iext nz) kn)
      (go 100)
     370
      (if (> jchnge 0) (go 100))
     400
      (setf fsh 1.0e-6
	    gtemp (aref grid 1))
      (setf nm1 (1- nfcns))
      (setf (aref x nzz) -2.0)
      (setf cn (1- (* 2 nfcns)))
      (setf delf (/ 1.0 cn))
      (setf l 1)
      (setf kkk 0)
      (if (and (< (aref grid 1) 0.01)
	       (> (aref grid ngrid) 0.49))
	  (setf kkk 1))
      (if (<= nfcns 3) (setf kkk 1))
      (when (/= kkk 1)
	(setf dtemp (cos (* pi2 (aref grid 1))))
	(setf dnum (cos (* pi2 (aref grid ngrid))))
	(setf aa (/ 2.0 (- dtemp dnum)))
	(setf bb (- (/ (+ dtemp dnum) (- dtemp dnum)))))
      (loop for j from 1 to nfcns do
	(tagbody
	  (setf ft (* delf (1- j)))
	  (setf xt (cos (* pi2 ft)))
	  (when (/= kkk 1)
	    (setf xt (/ (- xt bb) aa))
	    (setf xt1 (sqrt (- 1.0 (sqr xt))))
	    (setf ft (/ (atan xt1 xt) pi2))) ;ATAN with 2 args = ATAN2
	 410
	  (setf xe (aref x l))
	  (if (> xt xe) (go 420))
	  (if (< (- xe xt) fsh) (go 415))
	  (incf l)
	  (go 410)
	 415
	  (setf (aref a j) (aref y l))
	  (go 425)
	 420
	  (if (< (- xt xe) fsh) (go 415))
	  (setf (aref grid 1) ft)
	  (setf (aref a j) (gee 1 nz grid ad x y))
	 425
	  (if (> l 1) (decf l))))
      (setf (aref grid 1) gtemp)
      (setf dden (/ pi2 cn))
      (loop for j from 1 to nfcns do
	(setf dtemp 0.0)
	(setf dnum (* dden (1- j)))
	(when (>= nm1 1)
	  (loop for k from 1 to nm1 do
	    (let* ((dak (aref a (1+ k)))
		   (dk k))
	      (incf dtemp (* dak (cos (* dk dnum)))))))
	(setf dtemp (+ (* 2.0 dtemp) (aref a 1)))
	(setf (aref alpha j) dtemp))
      (loop for j from 2 to nfcns do
	(setf (aref alpha j) (/ (* 2.0 (aref alpha j)) cn)))
      (setf (aref alpha 1) (/ (aref alpha 1) cn))
      (when (/= kkk 1)
	(setf (aref p 1) (+ (* 2.0 (aref alpha nfcns) bb) (aref alpha nm1)))
	(setf (aref p 2) (* 2.0 (aref alpha nfcns) aa))
	(setf (aref q 1) (- (aref alpha (- nfcns 2)) (aref alpha nfcns)))
	(loop for j from 2 to nm1 do
	  (when (>= j nm1)
	    (setf aa (* .5 aa))
	    (setf bb (* .5 bb)))
	  (setf (aref p (1+ j)) 0.0)
	  (loop for k from 1 to j do
	    (setf (aref a k) (aref p k))
	    (setf (aref p k) (* 2.0 bb (aref a k))))
	  (incf (aref p 2) (* (aref a 1) 2.0 aa))
	  (setf jm1 (1- j))
	  (loop for k from 1 to jm1 do
	    (incf (aref p k) (+ (aref q k) (* aa (aref a (1+ k))))))
	  (setf jp1 (1+ j))
	  (loop for k from 3 to jp1 do
	    (incf (aref p k) (* aa (aref a (1- k)))))
	  (when (/= j nm1)
	    (loop for k from 1 to j do
	      (setf (aref q k) (- (aref a k))))
	    (setf nf1j (- nfcns 1 j))
	    (incf (aref q 1) (aref alpha nf1j))))
	(loop for j from 1 to nfcns do
	  (setf (aref alpha j) (aref p j))))
      (when (<= nfcns 3)
	(setf (aref alpha (+ nfcns 1)) 0.0)
	(setf (aref alpha (+ nfcns 2)) 0.0)))))


(defun design-FIR-6 (&key (length nil)
			  (order nil)
			  (type multiple-band)
			  (bands 2)
			  (band-edges nil)
			  (functions nil)
			  (weights nil))
  (let* ((nfilt (or length order 8))
	 (nfmax (max nfilt 128))
;	 (edge-size (max (* 2 bands) 20))
;	 (fx-size (max bands 10))
	 (jtype type)
	 (nbands bands)
	 (alpha-size (+ 2 (floor nfmax 2)))
;	 (hh-size (* 2 alpha-size))
	 (grid-size (* 16 alpha-size))
	 (iext (make-array (1+ alpha-size) :element-type 'fixnum :initial-element 0))
	 (ad (make-array (1+ alpha-size) :element-type 'float :initial-element 0.0))
	 (alpha (make-array (1+ alpha-size) :element-type 'float :initial-element 0.0))
	 (x (make-array (1+ alpha-size) :element-type 'float :initial-element 0.0))
	 (y (make-array (1+ alpha-size) :element-type 'float :initial-element 0.0))
	 (h (make-array (1+ alpha-size) :element-type 'float :initial-element 0.0))
;	 (hh (make-array (1+ hh-size) :element-type 'float :initial-element 0.0))
	 (des (make-array (1+ grid-size) :element-type 'float :initial-element 0.0))
	 (grid (make-array (1+ grid-size) :element-type 'float :initial-element 0.0))
	 (wt (make-array (1+ grid-size) :element-type 'float :initial-element 0.0))
;	 (deviat (make-array (1+ fx-size) :element-type 'float :initial-element 0.0))

	 (wtx weights)
	 (edge band-edges)
	 (fx functions)

	 (pi2 (* 2.0 pi))
	 (lgrid 16)
;	 (jb (* 2 bands))
	 (neg (if (= jtype 1) 0 1))
	 (nodd (if (oddp nfilt) 1 0))
	 (nfcns (floor nfilt 2))
	 (delf 0.0) (j 0) (l 0) (lband 0) (fup 0.0) (temp 0.0) (ngrid 0) (change 0.0)
	 (xt 0.0) (nm1 0) (nz 0) (nzmj 0) (nf2j 0) (nf3j 0))
    (tagbody
      (if (and (= nodd 1) (= neg 0)) (incf nfcns))
      (setf (aref grid 1) (aref edge 1))
      (setf delf (* lgrid nfcns))
      (setf delf (/ 0.5 delf))
      (if (/= neg 0)
	  (if (< (aref edge 1) delf) 
	      (setf (aref grid 1) delf)))
      (setf j 1)
      (setf l 1)
      (setf lband 1)
     140
      (setf fup (aref edge (1+ l)))
     145
      (setf temp (aref grid j))
      (setf (aref des j) (eff temp fx wtx lband jtype))
      (setf (aref wt j) (wate temp fx wtx lband jtype))
      (incf j)
      (setf (aref grid j) (+ temp delf))
      (if (<= (aref grid j) fup) (go 145))
      (setf (aref grid (1- j)) fup)
      (setf (aref des (1- j)) (eff fup fx wtx lband jtype))
      (setf (aref wt (1- j)) (wate fup fx wtx lband jtype))
      (incf lband)
      (incf l 2)
      (if (> lband nbands) (go 160))
      (setf (aref grid j) (aref edge l))
      (go 140)
     160
      (setf ngrid (1- j))
      (if (= neg nodd)
	  (if (> (aref grid ngrid) (- 0.5 delf))
	      (decf ngrid)))
      (if (plusp neg) (go 180))
      (if (= nodd 1) (go 200))
      (loop for j from 1 to ngrid do
	(setf change (cos (* pi (aref grid j))))
	(setf (aref des j) (/ (aref des j) change))
	(setf (aref wt j) (* (aref wt j) change)))
      (go 200)
     180
      (if (= nodd 1) (go 190))
      (loop for j from 1 to ngrid do
	(setf change (sin (* pi (aref grid j))))
	(setf (aref des j) (/ (aref des j) change))
	(setf (aref wt j) (* (aref wt j) change)))
      (go 200)
     190
      (loop for j from 1 to ngrid do
	(setf change (sin (* pi2 (aref grid j))))
	(setf (aref des j) (/ (aref des j) change))
	(setf (aref wt j) (* (aref wt j) change)))
     200
      (setf temp (/ (float (1- ngrid)) (float nfcns)))
      (loop for j from 1 to nfcns do
	(setf xt (1- j))
	(setf (aref iext j) (floor (+ 1.0 (* temp xt)))))
      (setf (aref iext (1+ nfcns)) ngrid)
      (setf nm1 (1- nfcns))
      (setf nz (1+ nfcns))
      (remes ad x y grid des wt alpha iext nfcns ngrid)
      (if (plusp neg) (go 320))
      (if (= nodd 0) (go 310))
      (loop for j from 1 to nm1 do
	(setf nzmj (- nz j))
	(setf (aref h j) (* 0.5 (aref alpha nzmj))))
      (setf (aref h nfcns) (aref alpha 1))
      (go 350)
     310
      (setf (aref h 1) (* 0.25 (aref alpha nfcns)))
      (loop for j from 2 to nm1 do
	(setf nzmj (- nz j))
	(setf nf2j (+ 2 (- nfcns j)))
	(setf (aref h j) (* 0.25 (+ (aref alpha nzmj) (aref alpha nf2j)))))
      (setf (aref h nfcns) (+ (* 0.5 (aref alpha 1)) (* 0.25 (aref alpha 2))))
      (go 350)
     320
      (if (= nodd 0) (go 330))
      (setf (aref h 1) (* 0.25 (aref alpha nfcns)))
      (setf (aref h 2) (* 0.25 (aref alpha nm1)))
      (loop for j from 3 to nm1 do
	(setf nzmj (- nz j))
	(setf nf3j (+ 3 (- nfcns j)))
	(setf (aref h j) (* 0.25 (- (aref alpha nzmj) (aref alpha nf3j)))))
      (setf (aref h nfcns) (- (* 0.5 (aref alpha 1)) (* 0.25 (aref alpha 3))))
      (setf (aref h nz) 0.0)
      (go 350)
     330
      (setf (aref h 1) (* 0.25 (aref alpha nfcns)))
      (loop for j from 2 to nm1 do
	(setf nzmj (- nz j))
	(setf nf2j (+ 2 (- nfcns j)))
	(setf (aref h j) (* 0.25 (- (aref alpha nzmj) (aref alpha nf2j)))))
      (setf (aref h nfcns) (- (* 0.5 (aref alpha 1)) (* 0.25 (aref alpha 2))))
     350
      )
    h))

;;; h is first half of impulse response
;;; for bandpass even order        h(nfcns+i) = h(nfcns-i+1)  [neg=0 and nodd=0, nfcns=nfilt/2]
;;;     bandpass odd order         h(nfcns+i) = h(nfcns-i)    [neg=0 and bodd=1, nfcns=nfilt/2+1]
;;;     diff or hilbert even order h(nfcns+i) = -h(nfcns-i+1) [neg=1 and nodd=0]
;;;     diif or hilbert odd order  h(nfcns+i+1) = -h(nfcns-i+1) [neg=1 and nodd=1]
      
      
;;; program 7 uses the unavailable procedure CHEB


;;; program 8 uses the LINPACK procedures DQRDC and DQRSL which appear to be identical
;;;   to SQRDC and SQRSL except for type declarations (real=>double precision).

(defun wate-8 (temp fx wtx lband jtype)
  (if (or (= jtype 2)
	  (< (aref fx lband) 0.0001))
      (aref wtx lband)
    (F4-div (aref wtx lband) temp)))

(defun eff-8 (temp slope fx wtx lband jtype j des u v)
  (declare (ignore wtx))
  (let ((t4 0.0) (rdes 0.0))
    (if (= jtype 1)
	(progn
	  (setf t4 (- (* temp slope)))
	  (setf rdes (aref fx lband))
	  (setf (aref des j) rdes)
	  (setf (aref u j) (* rdes (cos t4)))
	  (setf (aref v j) (* rdes (sin t4))))
      (if (= jtype 2)
	  (progn
	  (setf t4 (* temp slope))
	  (setf rdes (* (aref fx lband) temp 2.0 pi))
	  (setf (aref des j) rdes)
	  (setf (aref u j) (* rdes (sin t4)))
	  (setf (aref v j) (* rdes (cos t4))))
	(if (= jtype 3)
	    (progn
	      (setf rdes (aref fx lband))
	      (setf (aref des j) rdes)
	      (if (>= rdes 1.0e-5)
		  (progn
		    (setf t4 (- (* temp slope)))
		    (setf (aref u j) (* rdes (sin t4)))
		    (setf (aref v j) (- (* rdes (cos t4)))))
		(progn
		  (setf (aref u j) 0.0)
		  (setf (aref v j) 0.0)))))))))


(defun design-FIR-8 (&key (length nil)
			  (order nil)
			  (type multiple-band)
			  (bands 2)
			  (grid-density 16)
			  (group-delay nil)
			  (group-delay-error-weight nil)
			  (phase-error-weight nil)
			  (band-edges nil)
			  (functions nil)
			  (weights nil))
  (let* ((nfilt (or length order 8))
	 (nfmax (max nfilt 64))
;	 (edge-size (max (* 2 bands) 20))
;	 (fx-size (max bands 10))
	 (jtype type)
	 (nbands bands)
	 (pi2 (* 2.0 pi))
	 (lgrid grid-density)
	 (slope (if group-delay (* pi2 group-delay) 0))
	 (edge band-edges)
	 (fx functions)
	 (wtx weights)
	 (igd (if group-delay-error-weight 1 0))
	 (wgd group-delay-error-weight)
	 (iph (if phase-error-weight 1 0))
	 (wph phase-error-weight)
	 (fup 0.0) (lband 0) (delf 0.0) (temp 0.0) (j 0) (l 0) (klk 0) (info 0) (ngrid 0)
	 (theta 0.0) (tc 0.0) (ts 0.0) (mt 0) (delay 0.0) (ms 0)
	 (a (make-bigarray 4201 65))
	 (b (make-array 4201 :element-type 'float :initial-element 0.0))
	 (work (make-array 4201 :element-type 'float :initial-element 0.0))
	 (qy (make-array 4201 :element-type 'float :initial-element 0.0))
	 (qty(make-array 4201 :element-type 'float :initial-element 0.0))
	 (eor (make-array 4201 :element-type 'float :initial-element 0.0))
	 (ah (make-array 4201 :element-type 'float :initial-element 0.0))
	 (grid (make-array 1026 :element-type 'float :initial-element 0.0))
	 (wt (make-array 1026 :element-type 'float :initial-element 0.0))
	 (u (make-array 1026 :element-type 'float :initial-element 0.0))
	 (v (make-array 1026 :element-type 'float :initial-element 0.0))
	 (des (make-array 1026 :element-type 'float :initial-element 0.0))
	 (h (make-array 65 :element-type 'float :initial-element 0.0))
	 (qraux (make-array 65 :element-type 'float :initial-element 0.0))
	 (jpvt (make-array 65 :element-type 'fixnum :initial-element 0))
	 )
    (tagbody 
      (if (or (zerop jtype) (> nfilt nfmax) (< nfilt 3)) (error "invalid parameters"))
      (setf (aref grid 1) (aref edge 1))
      (setf delf (/ 0.5 (* nfilt lgrid)))
      (if (< (aref edge 1) delf) (setf (aref grid 1) delf))
      (setf j 1)
      (setf l 1)
      (setf lband 1)
     140
      (setf fup (aref edge (1+ l)))
     145
      (setf temp (aref grid j))
      (eff-8 temp slope fx wtx lband jtype j des u v) ;notice reference args!
      (setf (aref wt j) (wate-8 temp fx wtx lband jtype))
      (incf j)
      (setf (aref grid j) (+ temp delf))
      (if (<= (aref grid j) fup) (go 145))
      (setf klk (1- j))
      (setf (aref grid klk) fup)
      (eff-8 fup slope fx wtx lband jtype klk des u v)
      (setf (aref wt (1- j)) (wate-8 fup fx wtx lband jtype))
      (incf lband)
      (incf l 2)
      (if (> lband nbands) (go 160))
      (setf (aref grid j) (aref edge l))
      (go 140)
     160
      (setf ngrid (1- j))
      (if (> (aref grid ngrid) (- 0.5 delf)) (decf ngrid))
      (loop for j from 1 to 2 do
	(setf theta (/ (* pi (1- j)) 2.0)) 
	(setf tc (cos theta))
	(setf ts (sin theta))
	(loop for i from 1 to ngrid do
	  (setf mt (+ i (* (1- j) ngrid)))
	  (setf (aref b mt) (* (aref wt i) (- (* (aref u i) tc) (* (aref v i) ts))))
	  (loop for k from 1 to nfilt do
	    (setf temp (- (* pi2 (1- k) (aref grid i)) theta))
	    (setf (faref a mt k) (* (aref wt i) (cos temp))))))
      (setf delay (/ slope pi2))
      (setf ms (* ngrid 2))
      (when (/= 0 igd)
	(loop for i from 1 to ngrid do
	  (when (>= (aref des i) 1.0e-5)
	    (incf ms)
	    (setf (aref b ms) 0.0)
	    (setf theta (- (* slope (aref grid i))))
	    (if (= jtype 2) (incf theta (/ pi 2.0)))
	    (if (= jtype 3) (decf theta (/ pi 2.0)))
	    (loop for k from 1 to nfilt do
	      (setf temp (+ (* pi2 (1- k) (aref grid i)) theta))
	      (setf (faref a ms k) (* wgd (- k 1.0 delay) (cos temp)))))))
      (when (/= 0 iph)
	(setf theta (- (/ pi 2.0)))
	(loop for i from 1 to ngrid do
	  (when (>= (aref des i) 1.0e-5)
	    (incf ms)
	    (setf (aref b ms) 0.0)
	    (loop for k from 1 to nfilt do
	      (setf temp (+ (* pi2 (1- k) (aref grid i)) theta))
	      (setf (faref a ms k) (* wph (sin temp)))))))
      (if (> ms 4699) (error "trouble with ms"))
      (sqrdc a 4200 ms nfilt qraux jpvt work 0)
      (setf info 1)
      (sqrsl a 4200 ms nfilt qraux b qy qty h eor ah 110 info)
      )
    h))
	  
		  

;;; program 9 -- IIR filter design

;;; modulus from ratio of k/k'

(defun fk (u)
  (let* ((q (exp (- (* u pi))))
	 (a 1.0)
	 (b 1.0)
	 (c 1.0)
	 (d q))
    (loop for i from 1 to 15 while (>= c 1.0e-7) do
      (incf a (* 2 c d))
      (setf c (* c d d))
      (incf b c)
      (setf d (* d q)))
    (if (>= c 1.0e-7) (print "fk failed to converge"))
    (* 4 (sqrt q) (sqr (/ b a)))))

;;; Arc Elliptic Tangent

(defun Arcsc-val (y l a)
  (/ (+ (atan (/ a y)) (* pi l)) a))

(defun Arcsc (u kc)
  (let* ((a 1.0)
	 (b kc)
	 (y (/ 1.0 u))
	 (l 0))
    (loop for j from 1 to 15 do
      (let* ((bt (* a b)))
	(incf a b)
	(setf b (* 2.0 (sqrt bt)))
	(decf y (/ bt y))
	(if (zerop y) (setf y (* (sqrt bt) 1.0e-10)))
	(if (< (abs (- a b)) (* a 1.2e-7)) 
	    (return-from Arcsc (arcsc-val y (if (minusp y) (1+ l) l) a)))
	(incf l l)
	(if (minusp y) (incf l))))
    (print "Arcsc failed to converge")
    (arcsc-val y l a)))

;;; Elliptic functions

(defvar elp1-sn 0.0)			;ELP1 COMMON
(defvar elp1-cn 0.0)
(defvar elp1-dn 0.0)

(defun elp (x kc)
  (if (zerop x) 
      (progn
	(setf elp1-sn 0.0)
	(setf elp1-cn 1.0)
	(setf elp1-dn 1.0))
    (let* ((aa (make-array 17 :element-type 'float :initial-element 0.0))
	   (bb (make-array 17 :element-type 'float :initial-element 0.0))
	   (a 1.0)
	   (i 1)
	   (happy nil)
	   (b kc))
      (loop until (or happy (> i 15)) do
	(setf (aref aa i) a)
	(setf (aref bb i) b)
	(let ((at (/ (+ a b) 2)))
	  (setf b (sqrt (* a b)))
	  (setf a at)
	  (setf happy (< (/ (- a b) a) 1.3e-7))
	  (if (not happy) (incf i))))
      (if (not happy) (print "elp failed to converge"))
      (let* ((c (/ a (tan (* x a))))
	     (d 1.0))
	(loop until (zerop i) do
	  (let* ((e (/ (sqr c) a)))
	    (setf c (* c d))
	    (setf a (aref aa i))
	    (setf d (/ (+ e (aref bb i)) (+ e a)))
	    (decf i)))
	(let ((sn (/ 1.0 (sqrt (+ 1 (sqr c))))))
	  (setf elp1-sn sn)
	  (setf elp1-cn (* sn c))
	  (setf elp1-dn d))))))

;;; complete elliptic integral

(defun cei (kc)
  (let* ((a 1.0)
	 (b kc))
    (loop for j from 1 to 20 do
      (let* ((at (/ (+ a b) 2)))
	(setf b (sqrt (* a b)))
	(setf a at)
	(if (< (/ (- a b) a) 1.2e-7) (return-from cei (/ pi (* 2 a))))))
    (print "cei failed to converge")
    (/ pi (* 2 a))))

;;; Frequency transformation

(defvar parm-r1 0.0)			;PARM COMMON
(defvar parm-r2 0.0)
(defvar parm-wp 0.0)
(defvar parm-ws 0.0)
(defvar parm-n2 0.0)
(defvar parm-n 0.0)
(defvar parm-sr 0.0)
(defvar parm-ka 0.0)
(defvar parm-kad 0.0)
(defvar parm-kod 0.0)
(defvar parm-kf 0.0)

(defun freqxfm (w0 plr pli)
  ;; plr and pli are arrays
  (let* ((pc #C(0 0))
	 (sc #C(0 0))
	 (nt (1+ (* 2 parm-n2))))
    (if (< parm-kf 3)			;i.e. if lowpass or highpass
	(progn
	  (loop for j from 1 to parm-n2 do
	    (if (> (aref pli j) 1.0e15)
		(progn
		  (setf (aref plr j) 0.0)
		  (setf (aref pli j) 0.0))
	      (progn
		(setf pc (complex (aref plr j) (aref pli j)))
		(setf sc (/ 1.0 pc))
		(setf (aref plr j) (- (abs (realpart sc))))
		(setf (aref pli j) (abs (imagpart sc)))))))
      (progn
	(loop for j from 1 to parm-n2 do
	  (if (> (aref pli j) 1.0e15)
	      (progn
		(setf (aref plr j) 0.0)
		(setf (aref plr (- nt j)) 0.0)
		(setf (aref pli j) 1.0e17)
		(setf (aref pli (- nt j)) 0.0)
		(when (= parm-kf 4)	;bandreject
		  (setf (aref plr j) w0)
		  (setf (aref pli (- nt j)) w0)))
	    (progn
	      (setf pc (complex (aref plr j) (aref pli j)))
	      (if (= parm-kf 4) (setf pc (/ 1.0 pc)))
	      (setf sc (/ (- pc (sqrt (- (sqr pc) (* 4 (sqr w0))))) 2.0))
	      (setf (aref plr j) (- (abs (realpart sc))))
	      (setf (aref pli j) (abs (imagpart sc)))
	      (setf sc (/ (+ pc (sqrt (- (sqr pc) (* 4 (sqr w0))))) 2.0))
	      (setf (aref plr (- nt j)) (- (abs (realpart sc))))
	      (setf (aref pli (- nt j)) (abs (imagpart sc))))))))))
	      

;;; Digital Bilinear Transformation

(defun Blt (n2 sr r i)
  (let* ((a (* 2 sr)))
    (loop for j from 1 to (1+ n2) do
      (let* ((tr (aref r j))
	     (ti (aref i j)))
	(if (or (> (abs ti) 1.0e15)
		(> (abs tr) 1.0e15))
	    (progn
	      (setf (aref r j) -1.0)
	      (setf (aref i j) 0.0))
	  (let* ((td (+ (sqr (- a tr)) (sqr ti))))
	    (setf (aref r j) (/ (- (sqr a) (sqr tr) (sqr ti)) td))
	    (setf (aref i j) (/ (* 2.0 a ti) td))))))))

;;; Prewarp of Freqs before Blt

(defun prewrp (ww)
  (if (= parm-kad 1)			;analog??
      ww
    (* 2.0 parm-sr (tan (/ ww (* 2.0 parm-sr))))))

(defvar Root-pr nil)			;ROOT COMMON
(defvar Root-pi nil)
(defvar Root-zr nil)
(defvar Root-zi nil)

;;; Elliptic Filter Pole and Zero Locations

(defun Roots2 ()
  (when (null Root-pr)
    (setf Root-pr (make-array 21 :element-type 'float :initial-element 0.0))
    (setf Root-pi (make-array 21 :element-type 'float :initial-element 0.0))
    (setf Root-zr (make-array 21 :element-type 'float :initial-element 0.0))
    (setf Root-zi (make-array 21 :element-type 'float :initial-element 0.0)))
  (let* ((e (sqrt (- (expt 10.0 (* 0.1 parm-r1)) 1.0)))
	 (k (/ parm-wp parm-ws))
	 (kc (sqrt (- 1 (sqr k))))
	 (k1 (/ e (sqrt (- (expt 10.0 (* 0.1 parm-r2)) 1.0))))
	 (k1c (sqrt (- 1.0 (sqr k1))))
	 (kk (cei kc))
	 (kkc (cei k))
	 (kk1 (cei k1c))
	 (kk1c (cei k1))
	 (xn (/ (* kk kk1c) (* kk1 kkc))))
    (setf parm-n (floor (1+ xn)))
    (setf k1 (fk (/ (* parm-n kkc) kk)))
    (setf k1c (sqrt (- 1.0 (sqr k1))))
    (setf kk1 (cei k1c))
    (let ((l 0))
      (setf parm-n2 (/ (1+ parm-n) 2))
      (setf parm-kod 1)
      (if (evenp parm-n) (setf parm-kod 0))
      (if (zerop parm-kod) (setf l 1))
      (elp (* (/ kk (* kk1 parm-n)) (Arcsc (/ 1.0 e) k1)) k)
      (let* ((sm elp1-sn)
	     (cm elp1-cn)
	     (dm elp1-dn))
	(setf (aref Root-zi 1) 1.0e25)
	(loop for j from 1 to parm-n2 do
	  (elp (/ (* kk l) parm-n) kc)
	  (setf (aref Root-zr j) 0.0)
	  (if (not (zerop l)) (setf (aref Root-zi j) (/ parm-ws elp1-sn)))
	  (setf (aref Root-pr j) (- (/ (* Parm-wp sm cm elp1-cn elp1-dn)
				       (- 1.0 (sqr (elp1-dn sm))))))
	  (setf (aref Root-pi j) (/ (* Parm-wp dm elp1-sn)
				    (- 1.0 (sqr (elp1-dn sm)))))
	  (incf l 2))))))

;;; Butterworth, Chebychev I and II Pole and Zero locations

(defun Roots1 ()
  (when (null Root-pr)
    (setf Root-pr (make-array 21 :element-type 'float :initial-element 0.0))
    (setf Root-pi (make-array 21 :element-type 'float :initial-element 0.0))
    (setf Root-zr (make-array 21 :element-type 'float :initial-element 0.0))
    (setf Root-zi (make-array 21 :element-type 'float :initial-element 0.0)))
  (let* ((e (sqrt (- (expt 10.0 (* 0.1 parm-r1)) 1.0))))
    (if (= parm-ka 3) (setf e (/ 1.0 e))) ;Chebychev2
    (let ((l 0))
      (setf parm-n2 (/ (1+ parm-n) 2))
      (setf parm-kod 1)
      (if (evenp parm-n) (setf parm-kod 0))
      (if (zerop parm-kod) (setf l 1))
      (let ((sm 1.0)
	    (cm 1.0))
	(if (/= parm-ka 1)		;not Butterworth
	    (let ((v0 (/ (asinh (/ 1.0 e)) parm-n)))
	      ;; assume ASINH = ALOG(X+SQRT(X*X+1))
	      (setf sm (sinh v0))
	      (setf cm (cosh v0))))
	(loop for j from 1 to parm-n2 do
	  (let* ((arg (/ (* pi l) (* 2 parm-n)))
		 (tr (- (* sm (cos arg))))
		 (ti (* cm (sin arg))))
	    (setf (aref root-zr j) 0.0)
	    (setf (aref root-zi j) 1.0e25)
	    (if (= parm-ka 3)		;Chebychev2
		(progn
		  (if (not (zerop l)) (setf (aref root-zi j) (/ parm-ws (sin arg))))
		  (setf (aref root-pr j) (/ (* parm-ws tr) (+ (sqr tr) (sqr ti))))
		  (setf (aref root-pi j) (/ (* parm-ws ti) (+ (sqr tr) (sqr ti)))))
	      (progn
		(setf (aref root-pr j) (* parm-wp tr))
		(setf (aref root-pi j) (* parm-wp ti))))
	    (incf l 2)))))))

;;; srate = 1.0
;;; positive dB used in ripple and rejection

(defconstant Butterworth 1)
(defconstant Chebychev1 2)
(defconstant Chebychev2 3)
(defconstant Elliptic 4)

(defconstant Lowpass 1)
(defconstant Highpass 2)
(defconstant Bandpass 3)
(defconstant BandReject 4)

(defun design-IIR-9 (&key (class Butterworth)
		       (type Lowpass)
		       (srate 1.0)
		       (order 4)
		       (band-edge .2)
		       (ripple 6.0)
		       (lower-band-edge .2)
		       (upper-band-edge .3)
		       (f1 .1)
		       (f2 .2)
		       (f3 .3)
		       (f4 .4)
		       (r1 6.0)
		       (r2 6.0)
		       )
  (let ((tp (* 2 pi))
	(w0 0.0))
    (setf parm-ka class)		;Butterworth et al
    (setf parm-kf type)			;Lowpass et al
    (setf parm-kad 2)			;1=analog?
    (setf parm-sr srate)
    (setf parm-n order)
    (if (/= class Elliptic)
	(progn
	  (setf parm-r1 ripple)
	  (if (or (= type lowpass)
		  (= type highpass))
	      (progn
		(setf parm-wp (prewrp (* tp band-edge)))
		(if (= type Highpass) (setf parm-wp (/ 1.0 parm-wp)))
		(if (= class Chebychev2) (setf parm-ws parm-wp)))
	    (progn
	      (let* ((w1 (prewrp (* tp lower-band-edge)))
		     (w2 (prewrp (* tp upper-band-edge))))
		(setf w0 (sqrt (* w1 w2)))
		(setf parm-wp (/ (- (sqr w2) (sqr w0)) w2))
		(if (= type BandReject) (setf parm-wp (/ 1.0 parm-wp)))
		(if (= class Chebychev2) (setf parm-ws parm-wp)))))
	  (Roots1))
      (progn
	(if (or (= type lowpass)
		(= type highpass))
	    (progn
	      (setf parm-wp (prewrp (* tp lower-band-edge)))
	      (setf parm-ws (prewrp (* tp upper-band-edge)))
	      (when (= type highpass)
		(setf parm-wp (/ 1.0 parm-wp))
		(setf parm-ws (/ 1.0 parm-ws))))
	  (progn
	    (let* ((w1 (prewrp (* tp f1)))
		   (w2 (prewrp (* tp f2)))
		   (w3 (prewrp (* tp f3)))
		   (w4 (prewrp (* tp f4))))
	      (setf w0 (sqrt (* w3 w2)))
	      (setf parm-wp (/ (- (sqr w3) (sqr w0)) w3))
	      (setf parm-ws (/ (- (sqr w4) (sqr w0)) w4))
	      (let* ((wst (/ (- (sqr w0) (sqr w1)) w1)))
		(if (< wst parm-ws) (setf parm-ws wst))
		(when (/= type Bandpass)
		  (setf w0 (sqrt (* w1 w4)))
		  (setf parm-wp (/ w1 (- (sqr w0) (sqr w1))))
		  (setf parm-ws (/ w2 (- (sqr w0) (sqr w2))))
		  (setf wst (/ w3 (- (sqr w3) (sqr w0))))
		  (if (< wst parm-ws) (setf parm-ws wst)))))))
	(setf parm-r1 r1)
	(setf parm-r2 r2)
	(Roots2)))
    (when (/= type lowpass)
      (freqxfm w0 root-pr root-pi)
      (freqxfm w0 root-zr root-zi)
      (when (/= type Highpass)
	(setf parm-n2 parm-n)
	(setf parm-n (* 2 parm-n))
	(setf parm-kod 0)))
    (when (/= parm-kad 1)
      (blt parm-n2 parm-sr root-pr root-pi)
      (blt parm-n2 parm-sr root-zr root-zi))))

(defun dfr-9 (kk b1 b2 a1 a2 fm)
  (let* ((q (/ pi kk)))
    (loop for j from 1 to (1+ kk) do
      (let* ((w (* q (1- j)))
	     (w2 (* 2.0 w))
	     (br 1.0)
	     (bi 0.0)
	     (ar 1.0)
	     (ai 0.0)
	     (i0 1))
	(when (not (zerop parm-kod))
	  (setf br (+ (cos w) (aref b1 1)))
	  (setf bi (sin w))
	  (setf ar (+ (cos w) (aref a1 1)))
	  (setf ai (sin w))
	  (setf i0 2))
	(loop for i from i0 to parm-n2 do
	  (let* ((brt (+ (cos w2) (* (aref b1 i) (cos w)) (aref b2 i)))
		 (bit (+ (sin w2) (* (aref b1 i) (sin w))))
		 (art (+ (cos w2) (* (aref a1 i) (cos w)) (aref a2 i)))
		 (ait (+ (sin w2) (* (aref a1 i) (sin w))))
		 (brs (- (* br brt) (* bi bit)))
		 (ars (- (* ar art) (* ai ait))))
	    (setf bi (+ (* br bit) (* bi brt)))
	    (setf br brs)
	    (setf ai (+ (* ar ait) (* ai art)))
	    (setf ar ars)))
	(setf (aref fm j) (sqrt (/ (+ (sqr br) (sqr bi)) (+ (sqr ar) (sqr ai)))))))))

(defun cascad-9 (plr pli zr zi b1 b2 a1 a2)
  ;; (z*z + B1 z + B2) / (z*z + A1 z + A2)
  (let* ((k 0)
	 (j0 0))
    (if (and (odd parm-n2) (= parm-kf 3)) (setf k 1))
    (setf j0 1)
    (when (/= 0 parm-kod)
      (setf (aref b1 1) 1.0)
      (if (= parm-kf 2) (setf (aref b1 1) -1.0))
      (setf (aref b2 1) 0.0)
      (setf (aref a1 1) (- (aref plr 1)))
      (setf (aref a2 1) 0.0)
      (setf j0 2))
    (loop for j from j0 to parm-n2 do
      (setf (aref b1 j) (* -2.0 (aref zr j)))
      (setf (aref b2 j) (+ (sqr (aref zr j)) (sqr (aref zi j))))
      (when (and (= j 1) (= k 1)) 
	(setf (aref b1 j) 0.0)
	(setf (aref b2 j) -1.0))
      (setf (aref a1 j) (* -2.0 (aref plr j)))
      (setf (aref a2 j) (+ (sqr (aref plr j)) (sqr (aref pli j))))
      (when (= 0.0 (aref pli j))
	(setf (aref a1 1) (- (+ (aref plr 1) (aref plr (1+ parm-n2)))))
	(setf (aref a2 1) (* (aref plr 1) (aref plr (1+ parm-n2))))))))


;;; program 10 uses SQRDC and SQRSL translated below


(defun idft (n c d h)
  (let* ((q (/ (* 2.0 pi) n))
	 (m (floor (1+ n) 2))
	 (ht 0.0) (qq 0.0))
    (loop for j from 1 to n do
      (setf ht 0.0)
      (loop for k from 2 to m do
	(setf qq (* q (1- j) (1- k)))
	(incf ht (- (* (aref c k) (cos qq)) (* (aref d k) (sin qq)))))
      (setf ht (+ (aref c 1) (* 2.0 ht)))
      (if (evenp n) (incf ht (* (aref c (1+ m)) (cos (* pi (1- j))))))
      (setf (aref h j) (/ ht n)))))

(defun dfr (n m kk a b fm)
  (let* ((q (/ pi kk))
	 (br 0.0) (bi 0.0) (ar 0.0) (ai 0.0) (qq 0.0))
    (loop for j from 1 to (1+ kk) do
      (setf br (aref b (1+ m)))
      (setf bi 0.0)
      (setf qq (* q (1- j)))
      (loop for i from 1 to m do
	(incf br (* (aref b (1+ (- m i))) (cos (* qq i))))
	(incf bi (* (aref b (1+ (- m i))) (sin (* qq i)))))
      (setf ar (aref a (1+ n)))
      (setf ai 0.0)
      (loop for i from 1 to n do
	(incf ar (* (aref a (1+ (- n i))) (cos (* qq i))))
	(incf ai (* (aref a (1+ (- n i))) (sin (* qq i)))))
      (setf (aref fm j) (sqrt (/ (+ (sqr br) (sqr bi)) (+ (sqr ar) (sqr ai))))))))
      
(defun design-IIR-10 (&key (numerator-order 4)
			   (denominator-order 4)
			   (frequency-samples 50)
			   (real-samples nil)
			   (imaginary-samples nil))
  (let* (
;	 (kk number-of-frequencies)
	 (m numerator-order)
	 (n denominator-order)
	 (l1 frequency-samples)
	 (a (make-array 52 :element-type 'float :initial-element 0.0))
	 (b (make-array 52 :element-type 'float :initial-element 0.0))
	 (c (make-array 502 :element-type 'float :initial-element 0.0))
	 (d (make-array 502 :element-type 'float :initial-element 0.0))
	 (h (make-array 502 :element-type 'float :initial-element 0.0))
	 (h0 (make-array 502 :element-type 'float :initial-element 0.0))
	 (qax (make-array 52 :element-type 'float :initial-element 0.0))
;	 (fm (make-array 532 :element-type 'float :initial-element 0.0))
;	 (c (make-array 502 :element-type 'float :initial-element 0.0))
	 (h1 (make-bigarray 51 51))
	 (h2 (make-bigarray 511 51))
	 (l (1- l1))
	 (lm (- l m))
	 (m1 (1+ m))
	 (n1 (1+ n))
	 (ml (floor (1+ l) 2))
	 (ldx 501)
	 (i 0) (i0 0) (dum 0.0) (info 0) (bt 0.0))
    (loop for j from 1 to ml and x in real-samples and y in imaginary-samples do
      (setf (aref c j) x)
      (setf (aref d j) y))
    (idft l1 c d h)
    (loop for j from 1 to m1 do
      (setf i j)
      (loop for k from 1 to n1 do
	(if (< i 1) (setf i l1))
	(setf (faref h1 j k) (aref h i))
	(decf i)))
    (setf i0 m1)
    (loop for j from 1 to lm do
      (setf (aref h0 j) (- (aref h (1+ i0))))
      (setf i i0)
      (loop for k from 1 to n do
	(if (< i 1) (setf i l1))
	(setf (faref h2 j k) (aref h i))
	(decf i))
      (incf i0))
    (sqrdc h2 ldx lm n qax dum dum 0)
    (sqrsl h2 ldx lm n qax h0 dum h0 a dum dum 100 info)
    (loop for j from 1 to n do
      (setf (aref a (+ 2 (- n j))) (aref a (1+ (- n j)))))
    (setf (aref a 1) 1.0)
    (loop for j from 1 to m1 do
      (setf bt 0.0)
      (loop for k from 1 to n1 do
	(incf bt (* (faref h1 j k) (aref a k))))
      (setf (aref b j) bt))
    (values a b)))
  
    

;;; -------------- LINPACK routines ----------------

;    (sqrdc f ldx l m qax dum dum 0)

(defun sqrdc (x ldx n p qraux jpvt work job)
  (declare (ignore ldx))
  ;; SQRDC
  ;;*** from netlib, Fri Feb  8 14:57:09 EST 1991 ***
  ;;      subroutine sqrdc(x,ldx,n,p,qraux,jpvt,work,job)
  ;;      integer ldx,n,p,job
  ;;      integer jpvt(1)
  ;;      real x(ldx,1),qraux(1),work(1)
  ;;c
  ;;c     sqrdc uses householder transformations to compute the qr
  ;;c     factorization of an n by p matrix x.  column pivoting
  ;;c     based on the 2-norms of the reduced columns may be
  ;;c     performed at the users option.
  ;;c
  ;;c     on entry
  ;;c
  ;;c        x       real(ldx,p), where ldx .ge. n.
  ;;c                x contains the matrix whose decomposition is to be
  ;;c                computed.
  ;;c
  ;;c        ldx     integer.
  ;;c                ldx is the leading dimension of the array x.
  ;;c
  ;;c        n       integer.
  ;;c                n is the number of rows of the matrix x.
  ;;c
  ;;c        p       integer.
  ;;c                p is the number of columns of the matrix x.
  ;;c
  ;;c        jpvt    integer(p).
  ;;c                jpvt contains integers that control the selection
  ;;c                of the pivot columns.  the k-th column x(k) of x
  ;;c                is placed in one of three classes according to the
  ;;c                value of jpvt(k).
  ;;c
  ;;c                   if jpvt(k) .gt. 0, then x(k) is an initial
  ;;c                                      column.
  ;;c
  ;;c                   if jpvt(k) .eq. 0, then x(k) is a free column.
  ;;c
  ;;c                   if jpvt(k) .lt. 0, then x(k) is a final column.
  ;;c
  ;;c                before the decomposition is computed, initial columns
  ;;c                are moved to the beginning of the array x and final
  ;;c                columns to the end.  both initial and final columns
  ;;c                are frozen in place during the computation and only
  ;;c                free columns are moved.  at the k-th stage of the
  ;;c                reduction, if x(k) is occupied by a free column
  ;;c                it is interchanged with the free column of largest
  ;;c                reduced norm.  jpvt is not referenced if
  ;;c                job .eq. 0.
  ;;c
  ;;c        work    real(p).
  ;;c                work is a work array.  work is not referenced if
  ;;c                job .eq. 0.
  ;;c
  ;;c        job     integer.
  ;;c                job is an integer that initiates column pivoting.
  ;;c                if job .eq. 0, no pivoting is done.
  ;;c                if job .ne. 0, pivoting is done.
  ;;c
  ;;c     on return
  ;;c
  ;;c        x       x contains in its upper triangle the upper
  ;;c                triangular matrix r of the qr factorization.
  ;;c                below its diagonal x contains information from
  ;;c                which the orthogonal part of the decomposition
  ;;c                can be recovered.  note that if pivoting has
  ;;c                been requested, the decomposition is not that
  ;;c                of the original matrix x but that of x
  ;;c                with its columns permuted as described by jpvt.
  ;;c
  ;;c        qraux   real(p).
  ;;c                qraux contains further information required to recover
  ;;c                the orthogonal part of the decomposition.
  ;;c
  ;;c        jpvt    jpvt(k) contains the index of the column of the
  ;;c                original matrix that has been interchanged into
  ;;c                the k-th column, if pivoting was requested.
  ;;c
  ;;c     linpack. this version dated 08/14/78 .
  ;;c     g.w. stewart, university of maryland, argonne national lab.
  ;;c
  ;;c     sqrdc uses the following functions and subprograms.
  ;;c
  ;;c     blas saxpy,sdot,sscal,sswap,snrm2
  ;;c     fortran abs,amax1,min0,sqrt
  ;;c
  ;;c     internal variables
  ;;c
					;      integer j,jp,l,lp1,lup,maxj,pl,pu
					;      real maxnrm,snrm2,tt
					;      real sdot,nrmxl,t
					;      logical negj,swapj
  (let* ((j 0)
	 (jp 0)
					;	 (l 0)
	 (lp1 0)
	 (lup 0)
	 (maxj 0)
	 (pl 0)
	 (pu 0)
	 (maxnrm 0.0)
	 (tt 0.0)
	 (nrmxl 0.0)
	 (t4 0.0)
	 (negj nil)
	 (swapj nil))
					;c
					;      pl = 1
    (setf pl 1)
					;      pu = 0
    (setf pu 0)
					;      if (job .eq. 0) go to 60
    (when (not (zerop job))
					;c
					;c        pivoting has been requested.  rearrange the columns
					;c        according to jpvt.
					;c
					;         do 20 j = 1, p
      (loop for j from 1 to p do
					;            swapj = jpvt(j) .gt. 0
	(setf swapj (plusp (aref jpvt j)))
					;            negj = jpvt(j) .lt. 0
	(setf negj (minusp (aref jpvt j)))
					;            jpvt(j) = j
					;            if (negj) jpvt(j) = -j
	(setf (aref jpvt j) (if negj (- j) j))
					;            if (.not.swapj) go to 10
	(when swapj
					;               if (j .ne. pl) call sswap(n,x(1,pl),1,x(1,j),1)
	  (if (/= j pl) (sswap n (paref x 1 pl) 1 (paref x 1 j) 1))
					;               jpvt(j) = jpvt(pl)
	  (setf (aref jpvt j) (aref jpvt pl))
					;               jpvt(pl) = j
	  (setf (aref jpvt pl) j)
					;               pl = pl + 1
	  (incf pl)))
					;   10       continue
					;   20    continue
					;         pu = p
      (setf pu p)
					;         do 50 jj = 1, p
      (loop for jj from 1 to p do
					;            j = p - jj + 1
	(setf j (1+ (- p jj)))
					;            if (jpvt(j) .ge. 0) go to 40
	(when (minusp (aref jpvt j))
					;               jpvt(j) = -jpvt(j)
	  (setf (aref jpvt j) (- (aref jpvt j)))
					;               if (j .eq. pu) go to 30
	  (when (/= j pu)
					;                  call sswap(n,x(1,pu),1,x(1,j),1)
	    (sswap n (paref x 1 pu) 1 (paref x 1 j) 1)
					;                  jp = jpvt(pu)
	    (setf jp (aref jpvt pu))
					;                  jpvt(pu) = jpvt(j)
	    (setf (aref jpvt pu) (aref jpvt j))
					;                  jpvt(j) = jp
	    (setf (aref jpvt j) jp))
					;   30          continue
					;               pu = pu - 1
	  (decf pu))))
					;   40       continue
					;   50    continue
					;   60 continue
					;c
					;c     compute the norms of the free columns.
					;c
					;      if (pu .lt. pl) go to 80
    (when (>= pu pl)
					;      do 70 j = pl, pu
      (loop for j from pl to pu do
					;         qraux(j) = snrm2(n,x(1,j),1)
	(setf (aref qraux j) (snrm2 n (paref x 1 j) 1))
					;         work(j) = qraux(j)
	(setf (aref work j) (aref qraux j))))
					;   70 continue
					;   80 continue
					;c
					;c     perform the householder reduction of x.
					;c
					;      lup = min0(n,p)
    (setf lup (min n p))
					;      do 200 l = 1, lup
    (loop for l from 1 to lup do
					;         if (l .lt. pl .or. l .ge. pu) go to 120
      (when (and (>= l pl)
		 (< l pu))
					;c
					;c           locate the column of largest norm and bring it
					;c           into the pivot position.
					;c
					;            maxnrm = 0.0e0
	(setf maxnrm 0.0e0)
					;            maxj = l
	(setf maxj l)
					;            do 100 j = l, pu
	(loop for j from l to pu do
					;               if (qraux(j) .le. maxnrm) go to 90
	  (when (> (aref qraux j) maxnrm)
					;                  maxnrm = qraux(j)
	    (setf maxnrm (aref qraux j))
					;                  maxj = j
	    (setf maxj j)))
					;   90          continue
					;  100       continue
					;            if (maxj .eq. l) go to 110
	(when (/= maxj l)
					;               call sswap(n,x(1,l),1,x(1,maxj),1)
	  (sswap n (paref x 1 l) 1 (paref x 1 maxj) 1)
					;               qraux(maxj) = qraux(l)
	  (setf (aref qraux maxj) (aref qraux l))
					;               work(maxj) = work(l)
	  (setf (aref work maxj) (aref work l))
					;               jp = jpvt(maxj)
	  (setf jp (aref jpvt maxj))
					;               jpvt(maxj) = jpvt(l)
	  (setf (aref jpvt maxj) (aref jpvt l))
					;               jpvt(l) = jp
	  (setf (aref jpvt l) jp)))
					;  110       continue
					;  120    continue
					;         qraux(l) = 0.0e0
      (setf (aref qraux l) 0.0e0)
					;         if (l .eq. n) go to 190
      (when (/= l n)
					;c
					;c           compute the householder transformation for column l.
					;c
					;            nrmxl = snrm2(n-l+1,x(l,l),1)
	(setf nrmxl (snrm2 (1+ (- n l)) (paref x l l) 1))
					;            if (nrmxl .eq. 0.0e0) go to 180
	(when (/= nrmxl 0.0e0)
					;               if (x(l,l) .ne. 0.0e0) nrmxl = sign(nrmxl,x(l,l))
	  (if (/= (faref x l l) 0.0e0) (setf nrmxl (* nrmxl (signum (faref x l l)))))
	  ;; assume SIGN x y = (* x (signum y))
					;               call sscal(n-l+1,1.0e0/nrmxl,x(l,l),1)
	  (sscal (1+ (- n l)) (/ 1.0e0 nrmxl) (paref x l l) 1)
					;               x(l,l) = 1.0e0 + x(l,l)
	  (setf (faref x l l) (+ 1.0e0 (faref x l l)))
					;c
					;c              apply the transformation to the remaining columns,
					;c              updating the norms.
					;c
					;               lp1 = l + 1
	  (setf lp1 (1+ l))
					;               if (p .lt. lp1) go to 170
	  (when (>= p lp1)
					;               do 160 j = lp1, p
	    (loop for j from lp1 to p do
					;                  t = -sdot(n-l+1,x(l,l),1,x(l,j),1)/x(l,l)
	      (setf t4 (- (/ (sdot (1+ (- n l)) (paref x l l) 1 (paref x l j) 1) 
			     (faref x l l))))
					;                  call saxpy(n-l+1,t,x(l,l),1,x(l,j),1)
	      (saxpy (1+ (- n l)) t4 (paref x l l) 1 (paref x l j) 1)
					;                  if (j .lt. pl .or. j .gt. pu) go to 150
	      (when (and (>= j pl) 
			 (<= j pu))
					;                  if (qraux(j) .eq. 0.0e0) go to 150
		(when (/= (aref qraux j) 0.0e0)
					;                     tt = 1.0e0 - (abs(x(l,j))/qraux(j))**2
		  (setf tt (- 1.0e0 (sqr (/ (abs (faref x l j)) (aref qraux j)))))
					;                     tt = amax1(tt,0.0e0)
		  (setf tt (max tt 0.0e0))
					;                     t = tt
		  (setf t4 tt)
					;                     tt = 1.0e0 + 0.05e0*tt*(qraux(j)/work(j))**2
		  (setf tt (+ 1.0e0 (* 0.05e0 tt (sqr (/ (aref qraux j) (aref work j))))))
					;                     if (tt .eq. 1.0e0) go to 130
		  (if (/= tt 1.0e0)
					;                        qraux(j) = qraux(j)*sqrt(t)
		      (setf (aref qraux j) (* (aref qraux j) (sqrt t4)))
					;                     go to 140
					;  130                continue
					;                        qraux(j) = snrm2(n-l,x(l+1,j),1)
		    (progn
		      (setf (aref qraux j) (snrm2 (- n l) (paref x (1+ l) j) 1))
					;                        work(j) = qraux(j)
		      (setf (aref work j) (aref qraux j))))))))
					;  140                continue
					;  150             continue
					;  160          continue
					;  170          continue
					;c
					;c              save the transformation.
					;c
					;               qraux(l) = x(l,l)
	  (setf (aref qraux l) (faref x l l))
					;               x(l,l) = -nrmxl
	  (setf (faref x l l) (- nrmxl)))))))
					;  180       continue
					;  190    continue
					;  200 continue
					;      return
					;      end

(defun sscal (n sa sx incx)
  ;;      subroutine sscal(n,sa,sx,incx)
  ;;c
  ;;c     scales a vector by a constant.
  ;;c     uses unrolled loops for increment equal to 1.
  ;;c     jack dongarra, linpack, 3/11/78.
  ;;c
					;      real sa,sx(1)
					;      integer i,incx,m,mp1,n,nincx
  (let (				;(i 0)
	(m 0)
	(mp1 0)
	(nincx 0))
					;c
					;      if(n.le.0)return
    (when (plusp n)
					;      if(incx.eq.1)go to 20
      (if (/= incx 1)
					;c
					;c        code for increment not equal to 1
					;c
					;      nincx = n*incx
	  (progn
	    (setf nincx (* n incx))
					;      do 10 i = 1,nincx,incx
	    (loop for i from 1 to nincx by incx do
					;        sx(i) = sa*sx(i)
	      (setf (aref sx i) (* sa (aref sx i)))))
					;   10 continue
					;      return
					;c
					;c        code for increment equal to 1
					;c
					;c
					;c        clean-up loop
					;c
					;   20 m = mod(n,5)
	(progn
	  (setf m (mod n 5))
					;      if( m .eq. 0 ) go to 40
	  (when (/= m 0)
					;      do 30 i = 1,m
	    (loop for i from 1 to m do
					;        sx(i) = sa*sx(i)
	      (setf (aref sx i) (* sa (aref sx i))))
					;   30 continue
					;      if( n .lt. 5 ) return
	    (if (< n 5) (return-from sscal)))
					;   40 mp1 = m + 1
	  (setf mp1 (1+ m))
					;      do 50 i = mp1,n,5
	  (loop for i from mp1 to n by 5 do
					;        sx(i) = sa*sx(i)
	    (setf (aref sx i) (* sa (aref sx i)))
					;        sx(i + 1) = sa*sx(i + 1)
	    (setf (aref sx (+ i 1)) (* sa (aref sx (+ i 1))))
					;        sx(i + 2) = sa*sx(i + 2)
	    (setf (aref sx (+ i 2)) (* sa (aref sx (+ i 2))))
					;        sx(i + 3) = sa*sx(i + 3)
	    (setf (aref sx (+ i 3)) (* sa (aref sx (+ i 3))))
					;        sx(i + 4) = sa*sx(i + 4)
	    (setf (aref sx (+ i 4)) (* sa (aref sx (+ i 4))))))))))
					;   50 continue
					;      return
					;      end

(defun sswap (n sx incx sy incy)
  ;;      subroutine sswap (n,sx,incx,sy,incy)
  ;;c
  ;;c     interchanges two vectors.
  ;;c     uses unrolled loops for increments equal to 1.
  ;;c     jack dongarra, linpack, 3/11/78.
					;c
					;      real sx(1),sy(1),stemp
					;      integer i,incx,incy,ix,iy,m,mp1,n
  (let ((stemp 0.0)
					;	(i 0)
	(ix 0)
	(iy 0)
	(m 0)
	(mp1 0))
					;c
					;      if(n.le.0)return
    (when (plusp n)
					;      if(incx.eq.1.and.incy.eq.1)go to 20
      (if (or (/= incx 1)
	      (/= incy 1))
	  (progn
					;c
					;c       code for unequal increments or equal increments not equal
					;c         to 1
					;c
					;      ix = 1
	    (setf ix 1)
					;      iy = 1
	    (setf iy 1)
					;      if(incx.lt.0)ix = (-n+1)*incx + 1
	    (if (minusp incx) (setf ix (1+ (* (- 1 n) incx))))
					;      if(incy.lt.0)iy = (-n+1)*incy + 1
	    (if (minusp incy) (setf iy (1+ (* (- 1 n) incy))))
					;      do 10 i = 1,n
	    (loop for i from 1 to n do
					;        stemp = sx(ix)
	      (setf stemp (aref sx ix))
					;        sx(ix) = sy(iy)
	      (setf (aref sx ix) (aref sy iy))
					;        sy(iy) = stemp
	      (setf (aref sy iy) stemp)
					;        ix = ix + incx
	      (incf ix incx)
					;        iy = iy + incy
	      (incf iy incy)))
					;   10 continue
					;      return
					;c
					;c       code for both increments equal to 1
					;c
					;c
					;c       clean-up loop
					;c
	(progn
					;   20 m = mod(n,3)
	  (setf m (mod n 3))
					;      if( m .eq. 0 ) go to 40
	  (when (/= m 0)
					;      do 30 i = 1,m
	    (loop for i from 1 to m do
					;        stemp = sx(i)
	      (setf stemp (aref sx i))
					;        sx(i) = sy(i)
	      (setf (aref sx i) (aref sy i))
					;        sy(i) = stemp
	      (setf (aref sy i) stemp))
					;   30 continue
					;      if( n .lt. 3 ) return
	    (if (< n 3) (return-from sswap)))
					;   40 mp1 = m + 1
	  (setf mp1 (1+ m))
					;      do 50 i = mp1,n,3
	  (loop for i from mp1 to n by 3 do
					;        stemp = sx(i)
	    (setf stemp (aref sx i))
					;        sx(i) = sy(i)
	    (setf (aref sx i) (aref sy i))
					;        sy(i) = stemp
	    (setf (aref sy i) stemp)
					;        stemp = sx(i + 1)
	    (setf stemp (aref sx (1+ i)))
					;        sx(i + 1) = sy(i + 1)
	    (setf (aref sx (1+ i)) (aref sy (1+ i)))
					;        sy(i + 1) = stemp
	    (setf (aref sy (1+ i)) stemp)
					;        stemp = sx(i + 2)
	    (setf stemp (aref sx (+ i 2)))
					;        sx(i + 2) = sy(i + 2)
	    (setf (aref sx (+ i 2)) (aref sy (+ i 2)))
					;        sy(i + 2) = stemp
	    (setf (aref sy (+ i 2)) stemp)))))))
					;   50 continue
					;      return
					;      end

(defun saxpy (n sa sx incx sy incy)
  ;;      subroutine saxpy(n,sa,sx,incx,sy,incy)
  ;;c
  ;;c     constant times a vector plus a vector.
  ;;c     uses unrolled loop for increments equal to one.
  ;;c     jack dongarra, linpack, 3/11/78.
  ;;c
					;      real sx(1),sy(1),sa
					;      integer i,incx,incy,ix,iy,m,mp1,n
  (let ((ix 0)
	(iy 0)
	(m 0)
	(mp1 0))
					;c
					;      if(n.le.0)return
    (when (and (plusp n)
	       (/= sa 0.0))
					;      if (sa .eq. 0.0) return
					;      if(incx.eq.1.and.incy.eq.1)go to 20
      (if (or (/= incx 1)
	      (/= incy 1))
	  (progn
					;c
					;c        code for unequal increments or equal increments
					;c          not equal to 1
					;c
					;      ix = 1
	    (setf ix 1)
					;      iy = 1
	    (setf iy 1)
					;      if(incx.lt.0)ix = (-n+1)*incx + 1
	    (if (minusp incx) (setf ix (1+ (* (- 1 n) incx))))
					;      if(incy.lt.0)iy = (-n+1)*incy + 1
	    (if (minusp incy) (setf iy (1+ (* (- 1 n) incy))))
					;      do 10 i = 1,n
	    (loop for i from 1 to n do
					;        sy(iy) = sy(iy) + sa*sx(ix)
	      (incf (aref sy iy) (* sa (aref sx ix)))
					;        ix = ix + incx
	      (incf ix incx)
					;        iy = iy + incy
	      (incf iy incy)))
					;   10 continue
					;      return
					;c
					;c        code for both increments equal to 1
					;c
					;c
					;c        clean-up loop
					;c
					;   20 m = mod(n,4)
	(progn
	  (setf m (mod n 4))
					;      if( m .eq. 0 ) go to 40
	  (when (/= m 0)
					;      do 30 i = 1,m
	    (loop for i from 1 to m do
					;        sy(i) = sy(i) + sa*sx(i)
	      (incf (aref sy i) (* sa (aref sx i))))
					;   30 continue
					;      if( n .lt. 4 ) return
	    (if (< n 4) (return-from saxpy)))
					;   40 mp1 = m + 1
	  (setf mp1 (1+ m))
					;      do 50 i = mp1,n,4
	  (loop for i from mp1 to n by 4 do
					;        sy(i) = sy(i) + sa*sx(i)
	    (incf (aref sy i) (* sa (aref sx i)))
					;        sy(i + 1) = sy(i + 1) + sa*sx(i + 1)
	    (incf (aref sy (+ i 1)) (* sa (aref sx (+ i 1))))
					;        sy(i + 2) = sy(i + 2) + sa*sx(i + 2)
	    (incf (aref sy (+ i 2)) (* sa (aref sx (+ i 2))))
					;        sy(i + 3) = sy(i + 3) + sa*sx(i + 3)
	    (incf (aref sy (+ i 3)) (* sa (aref sx (+ i 3))))))))))
					;   50 continue
					;      return
					;      end

(defun sdot (n sx incx sy incy)
  ;;      real function sdot(n,sx,incx,sy,incy)
  ;;c
  ;;c     forms the dot product of two vectors.
  ;;c     uses unrolled loops for increments equal to one.
  ;;c     jack dongarra, linpack, 3/11/78.
  ;;c
					;      real sx(1),sy(1),stemp
					;      integer i,incx,incy,ix,iy,m,mp1,n
  (let ((stemp 0.0e0)
					;	(i 0)
	(ix 0)
	(iy 0)
	(m 0)
	(mp1 0))
					;c
					;      stemp = 0.0e0
					;      sdot = 0.0e0
					;      if(n.le.0)return
    (if (<= n 0) (return-from sdot 0.0))
					;      if(incx.eq.1.and.incy.eq.1)go to 20
    (if (or (/= incx 1)
	    (/= incy 1))
	(progn
					;c
					;c        code for unequal increments or equal increments
					;c          not equal to 1
					;c
					;      ix = 1
	  (setf ix 1)
					;      iy = 1
	  (setf iy 1)
					;      if(incx.lt.0)ix = (-n+1)*incx + 1
	  (if (minusp incx) (setf ix (1+ (* (- 1 n) incx))))
					;      if(incy.lt.0)iy = (-n+1)*incy + 1
	  (if (minusp incy) (setf iy (1+ (* (- 1 n) incy))))
					;      do 10 i = 1,n
	  (loop for i from 1 to n do
					;        stemp = stemp + sx(ix)*sy(iy)
	    (incf stemp (* (aref sx ix) (aref sy iy)))
					;        ix = ix + incx
	    (incf ix incx)
					;        iy = iy + incy
	    (incf iy incy))
	  stemp)
					;   10 continue
					;      sdot = stemp
					;      return
					;c
					;c        code for both increments equal to 1
					;c
					;c
					;c        clean-up loop
					;c
      (progn
					;   20 m = mod(n,5)
	(setf m (mod n 5))
					;      if( m .eq. 0 ) go to 40
	(when (/= m 0)
					;      do 30 i = 1,m
	  (loop for i from 1 to m do
					;        stemp = stemp + sx(i)*sy(i)
	    (incf stemp (* (aref sx i) (aref sy i))))
					;   30 continue
					;      if( n .lt. 5 ) go to 60
	  (if (< n 5) (return-from sdot stemp)))
					;   40 mp1 = m + 1
	(setf mp1 (1+ m))
					;      do 50 i = mp1,n,5
	(loop for i from mp1 to n by 5 do
					;        stemp = stemp + sx(i)*sy(i) + sx(i + 1)*sy(i + 1) +
					;     *   sx(i + 2)*sy(i + 2) + sx(i + 3)*sy(i + 3) + sx(i + 4)*sy(i + 4)
	  (incf stemp (+ (* (aref sx i) (aref sy i))
			 (* (aref sx (+ i 1)) (aref sy (+ i 1)))
			 (* (aref sx (+ i 2)) (aref sy (+ i 2)))
			 (* (aref sx (+ i 3)) (aref sy (+ i 3)))
			 (* (aref sx (+ i 4)) (aref sy (+ i 4))))))
	stemp))))
					;   50 continue
					;   60 sdot = stemp
					;      return
					;      end

(defun snrm2 (n sx incx)
  ;;      real function snrm2 ( n, sx, incx)
  ;;      integer          next
  ;;      real   sx(1),  cutlo, cuthi, hitest, sum, xmax, zero, one
  ;;      data   zero, one /0.0e0, 1.0e0/
  (let ((zero 0.0e0)
	(one 1.0e0)
	(cutlo 0.0)
	(cuthi 0.0)
	(hitest 0.0)
	(sum 0.0)
	(xmax 0.0)
	(nn 0)
	(i 0)
	(j 0)
	(next 0))
    ;;c
    ;;c     euclidean norm of the n-vector stored in sx() with storage
    ;;c     increment incx .
    ;;c     if    n .le. 0 return with result = 0.
    ;;c     if n .ge. 1 then incx must be .ge. 1
    ;;c
    ;;c           c.l.lawson, 1978 jan 08
    ;;c
    ;;c     four phase method     using two built-in constants that are
    ;;c     hopefully applicable to all machines.
    ;;c         cutlo = maximum of  sqrt(u/eps)  over all known machines.
    ;;c         cuthi = minimum of  sqrt(v)      over all known machines.
    ;;c     where
    ;;c         eps = smallest no. such that eps + 1. .gt. 1.
    ;;c         u   = smallest positive no.   (underflow limit)
    ;;c         v   = largest  no.            (overflow  limit)
    ;;c
    ;;c     brief outline of algorithm..
    ;;c
    ;;c     phase 1    scans zero components.
    ;;c     move to phase 2 when a component is nonzero and .le. cutlo
    ;;c     move to phase 3 when a component is .gt. cutlo
    ;;c     move to phase 4 when a component is .ge. cuthi/m
    ;;c     where m = n for x() real and m = 2*n for complex.
    ;;c
    ;;c     values for cutlo and cuthi..
    ;;c     from the environmental parameters listed in the imsl converter
    ;;c     document the limiting values are as follows..
    ;;c     cutlo, s.p.   u/eps = 2**(-102) for  honeywell.  close seconds are
    ;;c                   univac and dec at 2**(-103)
    ;;c                   thus cutlo = 2**(-51) = 4.44089e-16
    ;;c     cuthi, s.p.   v = 2**127 for univac, honeywell, and dec.
    ;;c                   thus cuthi = 2**(63.5) = 1.30438e19
    ;;c     cutlo, d.p.   u/eps = 2**(-67) for honeywell and dec.
    ;;c                   thus cutlo = 2**(-33.5) = 8.23181d-11
    ;;c     cuthi, d.p.   same as s.p.  cuthi = 1.30438d19
    ;;c     data cutlo, cuthi / 8.232d-11,  1.304d19 /
    ;;c     data cutlo, cuthi / 4.441e-16,  1.304e19 /
    ;;      data cutlo, cuthi / 4.441e-16,  1.304e19 /
    (setf cutlo 4.441e-16)
    (setf cuthi 1.304e19)
    ;;c
					;      if(n .gt. 0) go to 10
    (if (<= n 0) 
	zero
      (tagbody
					;         snrm2  = zero
					;         go to 300
					;c
					;   10 assign 30 to next
	(setf next 30)
					;      sum = zero
	(setf sum zero)
					;      nn = n * incx
	(setf nn (* n incx))
					;c                                                 begin main loop
					;      i = 1
	(setf i 1)
					;   20    go to next,(30, 50, 70, 110)
       20
	(if (= next 30) (go 30)
	  (if (= next 50) (go 50)
	    (if (= next 70) (go 70)
	      (if (= next 110) (go 110)
		(error "assign and goto confusion")))))
					;   30 if( abs(sx(i)) .gt. cutlo) go to 85
       30
	(if (> (abs (aref sx i)) cutlo) (go 85))
					;      assign 50 to next
	(setf next 50)
					;      xmax = zero
	(setf xmax zero)
					;c
					;c                        phase 1.  sum is zero
					;c
					;   50 if( sx(i) .eq. zero) go to 200
       50
	(if (= (aref sx i) zero) (go 200))
					;      if( abs(sx(i)) .gt. cutlo) go to 85
	(if (> (abs (aref sx i)) cutlo) (go 85))
					;c
					;c                                prepare for phase 2.
					;      assign 70 to next
	(setf next 70)
					;      go to 105
	(go 105)
					;c
					;c                                prepare for phase 4.
					;c
					;  100 i = j
       100
	(setf i j)
					;      assign 110 to next
	(setf next 110)
					;      sum = (sum / sx(i)) / sx(i)
	(setf sum (/ sum (sqr (aref sx i))))
					;  105 xmax = abs(sx(i))
       105
	(setf xmax (abs (aref sx i)))
					;      go to 115
	(go 115)
					;c
					;c                   phase 2.  sum is small.
					;c                             scale to avoid destructive underflow.
					;c
					;   70 if( abs(sx(i)) .gt. cutlo ) go to 75
       70
	(if (> (abs (aref sx i)) cutlo) (go 75))
					;c
					;c                     common code for phases 2 and 4.
					;c                     in phase 4 sum is large.  scale to avoid overflow.
					;c
					;  110 if( abs(sx(i)) .le. xmax ) go to 115
       110
	(if (<= (abs (aref sx i)) xmax) (go 115))
					;         sum = one + sum * (xmax / sx(i))**2
	(setf sum (+ one (* sum (sqr (/ xmax (aref sx i))))))
					;         xmax = abs(sx(i))
	(setf xmax (abs (aref sx i)))
					;         go to 200
	(go 200)
					;c
					;  115 sum = sum + (sx(i)/xmax)**2
       115
	(incf sum (sqr (/ (aref sx i) xmax)))
					;      go to 200
	(go 200)
					;c
					;c
					;c                  prepare for phase 3.
					;c
					;   75 sum = (sum * xmax) * xmax
       75
	(setf sum (* sum xmax xmax))
					;c
					;c
					;c     for real or d.p. set hitest = cuthi/n
					;c     for complex      set hitest = cuthi/(2*n)
					;c
					;   85 hitest = cuthi/float( n )
       85
	(setf hitest (float (/ cuthi (float n))))
					;c
					;c                   phase 3.  sum is mid-range.  no scaling.
					;c
					;      do 95 j =i,nn,incx
	(loop for j from 1 to nn by incx do
					;      if(abs(sx(j)) .ge. hitest) go to 100
	  (if (>= (abs (aref sx j)) hitest) (go 100))
					;   95    sum = sum + sx(j)**2
	  (incf sum (sqr (aref sx j))))
					;      snrm2 = sqrt( sum )
	(return-from snrm2 (sqrt sum))
	
					;      go to 300
					;c
					;  200 continue
       200
					;      i = i + incx
	(incf i incx)
					;      if ( i .le. nn ) go to 20
	(if (<= i nn) (go 20))
					;c
					;c              end of main loop.
					;c
					;c              compute square root and adjust for scaling.
					;c
					;      snrm2 = xmax * sqrt(sum)
	(* xmax (sqrt sum)) ))))
					;  300 continue
					;      return
					;      end
					;
;    (sqrsl f ldx l m qax a dum a x dum dum 100 info)

(defun sqrsl (x ldx n k qraux y qy qty b rsd xb job info)
  (declare (ignore ldx))
  ;; SQRSL
  ;;*** from netlib, Fri Feb  8 15:06:17 EST 1991 ***
  ;;      subroutine sqrsl(x,ldx,n,k,qraux,y,qy,qty,b,rsd,xb,job,info)
  ;;      integer ldx,n,k,job,info
  ;;      real x(ldx,1),qraux(1),y(1),qy(1),qty(1),b(1),rsd(1),xb(1)
  ;;c
  ;;c     sqrsl applies the output of sqrdc to compute coordinate
  ;;c     transformations, projections, and least squares solutions.
  ;;c     for k .le. min(n,p), let xk be the matrix
  ;;c
  ;;c            xk = (x(jpvt(1)),x(jpvt(2)), ... ,x(jpvt(k)))
  ;;c
  ;;c     formed from columnns jpvt(1), ... ,jpvt(k) of the original
  ;;c     n x p matrix x that was input to sqrdc (if no pivoting was
  ;;c     done, xk consists of the first k columns of x in their
  ;;c     original order).  sqrdc produces a factored orthogonal matrix q
  ;;c     and an upper triangular matrix r such that
  ;;c
  ;;c              xk = q * (r)
  ;;c                       (0)
  ;;c
  ;;c     this information is contained in coded form in the arrays
  ;;c     x and qraux.
  ;;c
  ;;c     on entry
  ;;c
  ;;c        x      real(ldx,p).
  ;;c               x contains the output of sqrdc.
  ;;c
  ;;c        ldx    integer.
  ;;c               ldx is the leading dimension of the array x.
  ;;c
  ;;c        n      integer.
  ;;c               n is the number of rows of the matrix xk.  it must
  ;;c               have the same value as n in sqrdc.
  ;;c
  ;;c        k      integer.
  ;;c               k is the number of columns of the matrix xk.  k
  ;;c               must nnot be greater than min(n,p), where p is the
  ;;c               same as in the calling sequence to sqrdc.
  ;;c
  ;;c        qraux  real(p).
  ;;c               qraux contains the auxiliary output from sqrdc.
  ;;c
  ;;c        y      real(n)
  ;;c               y contains an n-vector that is to be manipulated
  ;;c               by sqrsl.
  ;;c
  ;;c        job    integer.
  ;;c               job specifies what is to be computed.  job has
  ;;c               the decimal expansion abcde, with the following
  ;;c               meaning.
  ;;c
  ;;c                    if a.ne.0, compute qy.
  ;;c                    if b,c,d, or e .ne. 0, compute qty.
  ;;c                    if c.ne.0, compute b.
  ;;c                    if d.ne.0, compute rsd.
  ;;c                    if e.ne.0, compute xb.
  ;;c
  ;;c               note that a request to compute b, rsd, or xb
  ;;c               automatically triggers the computation of qty, for
  ;;c               which an array must be provided in the calling
  ;;c               sequence.
  ;;c
  ;;c     on return
  ;;c
  ;;c        qy     real(n).
  ;;c               qy conntains q*y, if its computation has been
  ;;c               requested.
  ;;c
  ;;c        qty    real(n).
  ;;c               qty contains trans(q)*y, if its computation has
  ;;c               been requested.  here trans(q) is the
  ;;c               transpose of the matrix q.
  ;;c
  ;;c        b      real(k)
  ;;c               b contains the solution of the least squares problem
  ;;c
  ;;c                    minimize norm2(y - xk*b),
  ;;c
  ;;c               if its computation has been requested.  (note that
  ;;c               if pivoting was requested in sqrdc, the j-th
  ;;c               component of b will be associated with column jpvt(j)
  ;;c               of the original matrix x that was input into sqrdc.)
  ;;c
  ;;c        rsd    real(n).
  ;;c               rsd contains the least squares residual y - xk*b,
  ;;c               if its computation has been requested.  rsd is
  ;;c               also the orthogonal projection of y onto the
  ;;c               orthogonal complement of the column space of xk.
  ;;c
  ;;c        xb     real(n).
  ;;c               xb contains the least squares approximation xk*b,
  ;;c               if its computation has been requested.  xb is also
  ;;c               the orthogonal projection of y onto the column space
  ;;c               of x.
  ;;c
  ;;c        info   integer.
  ;;c               info is zero unless the computation of b has
  ;;c               been requested and r is exactly singular.  in
  ;;c               this case, info is the index of the first zero
  ;;c               diagonal element of r and b is left unaltered.
  ;;c
  ;;c     the parameters qy, qty, b, rsd, and xb are not referenced
  ;;c     if their computation is not requested and in this case
  ;;c     can be replaced by dummy variables in the calling program.
  ;;c     to save storage, the user may in some cases use the same
  ;;c     array for different parameters in the calling sequence.  a
  ;;c     frequently occuring example is when one wishes to compute
  ;;c     any of b, rsd, or xb and does not need y or qty.  in this
  ;;c     case one may identify y, qty, and one of b, rsd, or xb, while
  ;;c     providing separate arrays for anything else that is to be
  ;;c     computed.  thus the calling sequence
  ;;c
  ;;c          call sqrsl(x,ldx,n,k,qraux,y,dum,y,b,y,dum,110,info)
  ;;c
  ;;c     will result in the computation of b and rsd, with rsd
  ;;c     overwriting y.  more generally, each item in the following
  ;;c     list contains groups of permissible identifications for
  ;;c     a single callinng sequence.
  ;;c
  ;;c          1. (y,qty,b) (rsd) (xb) (qy)
  ;;c
  ;;c          2. (y,qty,rsd) (b) (xb) (qy)
  ;;c
  ;;c          3. (y,qty,xb) (b) (rsd) (qy)
  ;;c
  ;;c          4. (y,qy) (qty,b) (rsd) (xb)
  ;;c
  ;;c          5. (y,qy) (qty,rsd) (b) (xb)
  ;;c
  ;;c          6. (y,qy) (qty,xb) (b) (rsd)
  ;;c
  ;;c     in any group the value returned in the array allocated to
  ;;c     the group corresponds to the last member of the group.
  ;;c
  ;;c     linpack. this version dated 08/14/78 .
  ;;c     g.w. stewart, university of maryland, argonne national lab.
  ;;c
  ;;c     sqrsl uses the following functions and subprograms.
  ;;c
  ;;c     blas saxpy,scopy,sdot
  ;;c     fortran abs,min0,mod
  ;;c
  ;;c     internal variables
					;c
					;      integer i,j,jj,ju,kp1
					;      real sdot,t,temp
					;      logical cb,cqy,cqty,cr,cxb
  (let (				;(i 0)
	(j 0)
					;(jj 0)
	(ju 0)
	(kp1 0)
	(t4 0.0)
	(temp 0.0)
	(cb nil)
	(cqy nil)
	(cqty nil)
	(cr nil)
	(cxb nil))
					;c
					;c
					;c     set info flag.
					;c
					;      info = 0
    (setf info 0)
					;c
					;c     determine what is to be computed.
					;c
					;      cqy = job/10000 .ne. 0
    (setf cqy (/= 0 (floor (/ job 10000))))
					;      cqty = mod(job,10000) .ne. 0
    (setf cqty (/= 0 (floor (mod job 10000))))
					;      cb = mod(job,1000)/100 .ne. 0
    (setf cb (/= 0 (floor (/ (floor (mod job 1000)) 100))))
					;      cr = mod(job,100)/10 .ne. 0
    (setf cr (/= 0 (floor (/ (floor (mod job 100)) 10))))
					;      cxb = mod(job,10) .ne. 0
    (setf cxb (/= 0 (floor (mod job 10))))
					;      ju = min0(k,n-1)
    (setf ju (min k (1- n)))
					;c
					;c     special action when n=1.
					;c
					;      if (ju .ne. 0) go to 40
    (if (zerop ju)
	(progn
					;         if (cqy) qy(1) = y(1)
	  (if cqy (setf (aref qy 1) (aref y 1)))
					;         if (cqty) qty(1) = y(1)
	  (if cqty (setf (aref qty 1) (aref y 1)))
					;         if (cxb) xb(1) = y(1)
	  (if cxb (setf (aref xb 1) (aref y 1)))
					;         if (.not.cb) go to 30
	  (when cb
					;            if (x(1,1) .ne. 0.0e0) go to 10
	    (if (= (faref x 1 1) 0.0e0)
					;               info = 1
		(setf info 1)
					;            go to 20
					;   10       continue
					;               b(1) = y(1)/x(1,1)
	      (setf (aref b 1) (/ (aref y 1) (faref x 1 1)))))
					;   20       continue
					;   30    continue
					;         if (cr) rsd(1) = 0.0e0
	  (if cr (setf (aref rsd 1) 0.0e0)))
					;      go to 250
					;   40 continue
					;c
					;c        set up to compute qy or qty.
					;c
      (progn
					;         if (cqy) call scopy(n,y,1,qy,1)
	(if cqy (scopy n y 1 qy 1))
					;         if (cqty) call scopy(n,y,1,qty,1)
	(if cqty (scopy n y 1 qty 1))
					;         if (.not.cqy) go to 70
	(when cqy
					;c
					;c           compute qy.
					;c
					;            do 60 jj = 1, ju
	  (loop for jj from 1 to ju do
					;               j = ju - jj + 1
	    (setf j (1+ (- ju jj)))
					;               if (qraux(j) .eq. 0.0e0) go to 50
	    (when (/= (aref qraux j) 0.0e0)
					;                  temp = x(j,j)
	      (setf temp (faref x j j))
					;                  x(j,j) = qraux(j)
	      (setf (faref x j j) (aref qraux j))
					;                  t = -sdot(n-j+1,x(j,j),1,qy(j),1)/x(j,j)
	      (setf t4 (- (/ (sdot (1+ (- n j)) (paref x j j) 1 (paref qy 1) 1) (faref x j j))))
					;                  call saxpy(n-j+1,t,x(j,j),1,qy(j),1)
	      (saxpy (1+ (- n j)) t4 (paref x j j) 1 (paref qy j) 1)
					;                  x(j,j) = temp
	      (setf (faref x j j) temp))))
					;   50          continue
					;   60       continue
					;   70    continue
					;         if (.not.cqty) go to 100
	(when cqty
					;c
					;c           compute trans(q)*y.
					;c
					;            do 90 j = 1, ju
	  (loop for j from 1 to ju do
					;               if (qraux(j) .eq. 0.0e0) go to 80
	    (when (/= (aref qraux j) 0.0e0)
					;                  temp = x(j,j)
	      (setf temp (faref x j j))
					;                  x(j,j) = qraux(j)
	      (setf (faref x j j) (aref qraux j))
					;                  t = -sdot(n-j+1,x(j,j),1,qty(j),1)/x(j,j)
	      (setf t4 (- (/ (sdot (1+ (- n j)) (paref x j j) 1 (paref qty j) 1) (faref x j j))))
					;                  call saxpy(n-j+1,t,x(j,j),1,qty(j),1)
	      (saxpy (1+ (- n j)) t4 (paref x j j) 1 (paref qty j) 1)
					;                  x(j,j) = temp
	      (setf (faref x j j) temp))))
					;   80          continue
					;   90       continue
					;  100    continue
					;c
					;c        set up to compute b, rsd, or xb.
					;c
					;         if (cb) call scopy(k,qty,1,b,1)
	(if cb (scopy k qty 1 b 1))
					;         kp1 = k + 1
	(setf kp1 (1+ k))
					;         if (cxb) call scopy(k,qty,1,xb,1)
	(if cxb (scopy k qty 1 xb 1))
					;         if (cr .and. k .lt. n) call scopy(n-k,qty(kp1),1,rsd(kp1),1)
	(if (and cr (< k n))
	    (scopy (- n k) (paref qty kp1) 1 (paref rsd kp1) 1))
					;         if (.not.cxb .or. kp1 .gt. n) go to 120
	(if (and cxb (<= kp1 n))
					;            do 110 i = kp1, n
	    (loop for i from kp1 to n do
					;               xb(i) = 0.0e0
	      (setf (aref xb i) 0.0e0)))
					;  110       continue
					;  120    continue
					;         if (.not.cr) go to 140
	(if cr
					;            do 130 i = 1, k
	    (loop for i from 1 to k do
					;               rsd(i) = 0.0e0
	      (setf (aref rsd i) 0.0e0)))
					;  130       continue
					;  140    continue
					;         if (.not.cb) go to 190
	(when cb
					;c
					;c           compute b.
					;c
					;            do 170 jj = 1, k
	  (let ((happy t))
	    (loop for jj from 1 to k while happy do
					;               j = k - jj + 1
	      (setf j (1+ (- k jj)))
					;               if (x(j,j) .ne. 0.0e0) go to 150
	      (if (zerop (faref x j j))
		  (progn
		    (setf info j)
		    (setf happy nil))
		(progn
					;                  info = j
					;c           ......exit
					;                  go to 180
					;  150          continue
					;               b(j) = b(j)/x(j,j)
		  (setf (aref b j) (/ (aref b j) (faref x j j)))
					;               if (j .eq. 1) go to 160
		  (when (/= j 1)
					;                  t = -b(j)
		    (setf t4 (- (aref b j)))
					;                  call saxpy(j-1,t,x(1,j),1,b,1)
		    (saxpy (1- j) t4 (paref x 1 j) 1 b 1)))))))
					;  160          continue
					;  170       continue
					;  180       continue
					;  190    continue
					;         if (.not.cr .and. .not.cxb) go to 240
	(when (or cr cxb)
					;c
					;c           compute rsd or xb as required.
					;c
					;            do 230 jj = 1, ju
	  (loop for jj from 1 to ju do
					;               j = ju - jj + 1
	    (setf j (1+ (- ju jj)))
					;               if (qraux(j) .eq. 0.0e0) go to 220
	    (when (/= (aref qraux j) 0.0e0)
					;                  temp = x(j,j)
	      (setf temp (faref x j j))
					;                  x(j,j) = qraux(j)
	      (setf (faref x j j) (aref qraux j))
					;                  if (.not.cr) go to 200
	      (when cr
					;                     t = -sdot(n-j+1,x(j,j),1,rsd(j),1)/x(j,j)
		(setf t4 (- (/ (sdot (1+ (- n j)) (paref x j j) 1 (paref rsd j) 1) (faref x j j))))
					;                     call saxpy(n-j+1,t,x(j,j),1,rsd(j),1)
		(saxpy (1+ (- n j)) t4 (paref x j j) 1 (paref rsd j) 1))
					;  200             continue
					;                  if (.not.cxb) go to 210
	      (when cxb
					;                     t = -sdot(n-j+1,x(j,j),1,xb(j),1)/x(j,j)
		(setf t4 (- (/ (sdot (1+ (- n j)) (paref x j j) 1 (paref xb j) 1) (faref x j j))))
					;                     call saxpy(n-j+1,t,x(j,j),1,xb(j),1)
		(saxpy (1+ (- n j)) t4 (paref x j j) 1 (paref xb j) 1))
					;  210             continue
					;                  x(j,j) = temp
	      (setf (faref x j j) temp))))))))
					;  220          continue
					;  230       continue
					;  240    continue
					;  250 continue
					;      return
					;      end
    
(defun scopy (n sx incx sy incy)
  ;;      subroutine scopy(n,sx,incx,sy,incy)
  ;;c
  ;;c     copies a vector, x, to a vector, y.
  ;;c     uses unrolled loops for increments equal to 1.
  ;;c     jack dongarra, linpack, 3/11/78.
  ;;c
					;      real sx(1),sy(1)
					;      integer i,incx,incy,ix,iy,m,mp1,n
  (let (				;(i 0)
	(ix 0)
	(iy 0)
	(m 0)
	(mp1 0))
					;c
					;      if(n.le.0)return
    (when (plusp n)
					;      if(incx.eq.1.and.incy.eq.1)go to 20
      (if (or (/= incx 1)
	      (/= incy 1))
	  (progn
					;c
					;c        code for unequal increments or equal increments
					;c          not equal to 1
					;c
					;      ix = 1
	    (setf ix 1)
					;      iy = 1
	    (setf iy 1)
					;      if(incx.lt.0)ix = (-n+1)*incx + 1
	    (if (minusp incx) (setf ix (1+ (* (- 1 n) incx))))
					;      if(incy.lt.0)iy = (-n+1)*incy + 1
	    (if (minusp incy) (setf iy (1+ (* (- 1 n) incy))))
					;      do 10 i = 1,n
	    (loop for i from 1 to n do
					;        sy(iy) = sx(ix)
	      (setf (aref sy iy) (aref sx ix))
					;        ix = ix + incx
	      (incf ix incx)
					;        iy = iy + incy
	      (incf iy incy)))
					;   10 continue
					;      return
					;c
					;c        code for both increments equal to 1
					;c
					;c
					;c        clean-up loop
					;c
	(progn
					;   20 m = mod(n,7)
	  (setf m (mod n 7))
					;      if( m .eq. 0 ) go to 40
	  (when (/= m 0) 
					;      do 30 i = 1,m
	    (loop for i from 1 to m do
					;        sy(i) = sx(i)
	      (setf (aref sy i) (aref sx i)))
					;   30 continue
					;      if( n .lt. 7 ) return
	    (if (< n 7) (return-from scopy)))
					;   40 mp1 = m + 1
	  (setf mp1 (1+ m))
					;      do 50 i = mp1,n,7
	  (loop for i from mp1 to n by 7 do
					;        sy(i) = sx(i)
	    (setf (aref sy i) (aref sx i))
					;        sy(i + 1) = sx(i + 1)
	    (setf (aref sy (+ i 1)) (aref sx (+ i 1)))
					;        sy(i + 2) = sx(i + 2)
	    (setf (aref sy (+ i 2)) (aref sx (+ i 2)))
					;        sy(i + 3) = sx(i + 3)
	    (setf (aref sy (+ i 3)) (aref sx (+ i 3)))
					;        sy(i + 4) = sx(i + 4)
	    (setf (aref sy (+ i 4)) (aref sx (+ i 4)))
					;        sy(i + 5) = sx(i + 5)
	    (setf (aref sy (+ i 5)) (aref sx (+ i 5)))
					;        sy(i + 6) = sx(i + 6)
	    (setf (aref sy (+ i 6)) (aref sx (+ i 6)))))))))
					;   50 continue
					;      return
					;      end
