;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;
;;; Envelope handlers
;;;
;;; many of these functions were originally intended for an extension of lisp operators to envelopes.
;;; The actually useful functions are probably:
;;;   env+ envs...                             add all envs together
;;;   env* envs...                             multiply envs together 
;;;   env-reverse env                          turn env backwards
;;;   merge-envelopes envs...                  merge all envs into one envelope
;;;   env-simplify env &optional ygrid xgrid   simplify env
;;;   fft-env-simplify env &optional cutoff    same but use fft filtering


(in-package :clm)

#-cltl2 (export '(EREF ENV-SUBSEQ ENV-COPY-SEQ ENV-LENGTH
	  ENV-REVERSE ENV-NREVERSE ENV-CONCATENATE
	  ENV+ ENV- ENV* ENV/ ENV-MIN ENV-MAX
	  ENV-OP ENV-OP-1 ENV-OP-1-REV ENV-OP-2 ENV-ALWAYS-OP ENV-ALWAYS-OP-2
	  ENV-FUNCALL ENV-APPLY ENV-MAP ENV-FUNCALL-1 MAP-ACROSS-ENVELOPES
	  ENV-FILL ENV-REPLACE ENV-FIND ENV-POSITION ADD-OR-EDIT-BREAKPOINT
	  ENV-SIMPLIFY FFT-ENV-SIMPLIFY
	  REDUCE-AMPLITUDE-QUANTIZATION-NOISE
	  MELD-ENVELOPES
	  ))

;;; I've made functional versions of all these (rather than just setting up methods
;;; dispatching on class basic-envelope) so that they could be used in other settings
;;; where the snd functions are not in use.  Since they aren't usually needed in the
;;; clm synthesis setting, I took them out of mus.lisp (so they need to be explicitly
;;; loaded)


(defun eref (env x) 
  (list-interp x env))			;like aref but for an envelope

(defun env-subseq (e start &optional end) 
  (subseq e (* 2 start) (if end (* 2 end))))

(defun env-copy-seq (e) 
  (copy-seq e))

(defun env-length (e) 
  (floor (length e) 2))

(defun env-ur-reverse (e &optional (nr nil))
  (if (or (null e)
	  (= (length e) 2))
      e
    (let* ((first-x (car e))
	   (lim (1- (length e)))
	   (last-x (env-last-x e))
	   (new-e (append (list first-x) 
			  (if nr 
			      (nreverse (cdr e)) 
			    (reverse (cdr e))))))
      (if (= lim 3)
	  new-e
	(progn
	  (loop for i from 2 by 2 and k from (- lim 3) by 2 downto 0 do
	    (setf (nth i new-e) (+ first-x (- last-x (nth k e)))))
	  new-e)))))

(defun env-reverse (e) (env-ur-reverse e))
(defun env-nreverse (e) (env-ur-reverse e t))

(defun env-concatenate (result-type &rest envs) 
  (declare (ignore result-type)) 
  (let* ((ur-env (apply #'concatenate (append (list 'list) envs)))
	 (lim (length ur-env))
	 (offset 0.0)
	 (x0 (if ur-env (first ur-env) 0.0))
	 (x1 (if (> lim 2) (third ur-env) x0)))
    (loop for i from 2 to (- lim 2) by 2 do
      (if (< x1 x0)
	  (incf offset (+ (- x0 x1) .01)))
      (setf x0 x1)
      (setf (nth i ur-env) (+ x0 offset))
      (if (< i lim) (setf x1 (nth (+ i 2) ur-env))))
    ur-env))

(defun merge-breakpoints (&rest breakpoints)
  (if breakpoints
      (apply #'append 
	     (sort (remove-duplicates breakpoints 
				      :test #'= 
				      :key #'car) 
		   #'< :key #'car))))

(defun env-funcall (function envelope)
  (loop for x in envelope by #'cddr and
            y in (cdr envelope) by #'cddr
    collect (funcall function x y)))

(defun env-apply (function envelope)
  (loop for x in envelope by #'cddr and
            y in (cdr envelope) by #'cddr
    collect (apply function (list x y))))

(defun env-map (result-type function envelope &rest more-envelopes)
  (declare (ignore result-type))
  (let ((first-group (env-funcall function envelope))
	(second-group (loop for i in more-envelopes 
		       append (env-funcall function i))))
    (apply #'merge-breakpoints (append first-group second-group))))


;;; what we actually want here is to run through all the envelopes in parallel (as per mapcar, I think)
;;; using list-interp to get y at the current x and then call the function on that list of y's:

(defun funcall-breakpoint (function x y)
  (list x (funcall function x y)))

(defun map-across-envelopes (function &rest envelopes)
  (let ((new-e nil)
	(current-x (loop for x in envelopes minimize (car x)))
	(last-x (loop for x in envelopes maximize (env-last-x x))))
    (setf new-e 
      (funcall-breakpoint function 
			  current-x 
			  (loop for x in envelopes 
			   collect (list-interp current-x x))))
    (loop until (>= current-x last-x) do
      (setf current-x (loop for x in envelopes 
		       minimize (or (loop for y in x by #'cddr 
				     if (> y current-x) 
				     return y) 
				    last-x)))
      (setf new-e (append new-e 
			  (funcall-breakpoint function 
					      current-x 
					      (loop for x in envelopes 
					       collect (list-interp current-x x))))))
    new-e))
      
(defun env+ (&rest envelopes) 
  (apply #'map-across-envelopes 
	 (append (list #'(lambda (x y) 
			   (declare (ignore x)) 
			   (apply #'+ y))) 
		 envelopes)))

(defun env- (&rest envelopes) 
  (apply #'map-across-envelopes 
	 (append (list #'(lambda (x y) 
			   (declare (ignore x)) 
			   (apply #'- y))) 
		 envelopes)))

(defun env* (&rest envelopes) 
  (apply #'map-across-envelopes 
	 (append (list #'(lambda (x y) 
			   (declare (ignore x)) 
			   (apply #'* y))) 
		 envelopes)))

(defun env/ (&rest envelopes) 
  (apply #'map-across-envelopes 
	 (append (list #'(lambda (x y) 
			   (declare (ignore x)) 
			   (apply #'/ y)))
		 envelopes)))

(defun env-max (&rest envelopes) 
  (apply #'map-across-envelopes 
	 (append (list #'(lambda (x y) 
			   (declare (ignore x)) 
			   (apply #'max y))) 
		 envelopes)))

(defun env-min (&rest envelopes) 
  (apply #'map-across-envelopes 
	 (append (list #'(lambda (x y) 
			   (declare (ignore x)) 
			   (apply #'min y))) 
		 envelopes)))

;;; see basic-data.lisp for more along these lines

;;; now a bunch of numerical operations that apply to the y-axis only 

(defun env-op (function envelope)
  (loop for x in envelope by #'cddr and
            y in (cdr envelope) by #'cddr
    append (list x (funcall function y))))

(defun env-op-1-rev (function data envelope) ;same as above but pass data as arg too
  (loop for x in envelope by #'cddr and
            y in (cdr envelope) by #'cddr
    append (list x (funcall function data y))))
 
(defun env-op-1 (function envelope data) ;reversed arg order to function (for basic-data methods primarily)
  (loop for x in envelope by #'cddr and
            y in (cdr envelope) by #'cddr
    append (list x (funcall function y data))))

(defun env-op-2 (function data envelope) ;same again but now pass the x axis point to the function as well as data and y
  (loop for x in envelope by #'cddr and
            y in (cdr envelope) by #'cddr
    append (list x (funcall function x y data))))

(defun env-always-op (function envelope)
  (loop for x in envelope by #'cddr and
            y in (cdr envelope) by #'cddr
    always (funcall function x y)))

(defun merge-envelopes (&rest envelopes)
  (apply #'env-map (append (list 'list #'(lambda (x y) (list x y))) envelopes)))

;;; env-funcall-1 expects its function to take x0 y0 x1 y1 x-first x-last as arguments
;;; and return a partial (or full) envelope as its result.  All these partial results
;;; are then merged in env-map.

(defun env-funcall-1 (function envelope)
  (if envelope
      (let ((x0 (first envelope))
	    (y0 (second envelope)))
	(if (= (env-length envelope) 1)
	    (merge-envelopes (funcall function x0 y0 x0 y0 x0 x0))
	  (let ((new-brks nil)
		(first-x x0)
		(last-x (env-last-x envelope)))
	    (loop for x1 in (cddr envelope) by #'cddr and
	              y1 in (cdddr envelope) by #'cddr do
	      (push (funcall function x0 y0 x1 y1 first-x last-x) new-brks)
	      (setf x0 x1)
	      (setf y0 y1))
	    (apply #'merge-envelopes new-brks))))))

(defun env-always-op-2 (function envelope data)
  (if envelope
      (let ((x0 (first envelope))
	    (y0 (second envelope)))
	(if (= (env-length envelope) 1)
	    (funcall function data x0 y0 x0 y0)
	  (progn
	    (loop for x1 in (cddr envelope) by #'cddr and
	              y1 in (cdddr envelope) by #'cddr do
	      (if (null (funcall function data x0 y0 x1 y1))
		  (return-from env-always-op-2 nil))
	      (setf x0 x1)
	      (setf y0 y1))
	    t)))))

(defun sample-envelope (e &optional (sample-density 2))
  (env-funcall-1 #'(lambda (x0 y0 x1 y1 fx lx)
		     (declare (ignore lx))
		     (let ((new-env (if (= x0 fx) (list y0 x0)))
			   (xinc (/ (- x1 x0) sample-density))
			   (yinc (/ (- y1 y0) sample-density)))

#-Kcl		       (loop for i from 0 below sample-density and
			         x from (+ x0 xinc) by xinc and
			         y from (+ y0 yinc) by yinc do
			 (push x new-env)
			 (push y new-env))
#+kcl		       (let ((x (+ x0 xinc))
			     (y (+ y0 yinc)))
			 (loop for i from 0 below sample-density do
			   (push x new-env)
			   (push y new-env)
			   (incf x xinc)
			   (incf y yinc)))

		       (nreverse new-env)))
		 e))

;;; We need sample-envelope for operators like env* to
;;; avoid unexpected 0's (consider multiplying '(0 0 100 1) 
;;; and (0 1 100 0) on a break-point by breakpoint basis -- 
;;; we'd get '(0 0 100 0) which is surely not what is wanted).
;;; The caller will have to handle this special case for now.

(defun add-or-edit-breakpoint (fn x y)
  (if (null fn)
      (list x y)
    (let ((lim (length fn)))
      (loop for px in fn by #'cddr and
                i from 0 by 2 do
	(if (= px x)
	    (if (= i 0)
		(return-from add-or-edit-breakpoint (append (list x y) (subseq fn 2 lim)))
	      (if (>= (+ i 2) lim)
		  (return-from add-or-edit-breakpoint (append (subseq fn 0 i) (list x y)))
		(return-from add-or-edit-breakpoint (append (subseq fn 0 i) (list x y) (subseq fn (+ i 2) lim)))))
	  (if (> px x)
	      (if (= i 0)
		  (return-from add-or-edit-breakpoint (append (list x y) fn))
		(return-from add-or-edit-breakpoint (append (subseq fn 0 i) (list x y) (subseq fn i lim)))))))
      (append fn (list x y)))))

(defun remove-breakpoint (fn x)
  (if fn
      (let ((lim (length fn)))
	(if (= (first fn) x)
	    (cddr fn)
	  (if (= (env-last-x fn) x)
	      (subseq fn 0 (- lim 2))
	    (if (> lim 2)
		(progn
		  (loop for x0 in (cddr fn) by #'cddr and i from 2 by 2 do
		    (if (= x0 x)
			(return-from remove-breakpoint (append (subseq fn 0 i) (subseq fn (+ i 2) lim)))))
		  (error "cannot find breakpoint at ~A in ~A" x fn))))))))



;;; now some of the lisp sequence functions as applied to envelopes

(defun env-with-start-end (e &optional start end)
  (if (and (not start) (not end)) 
      (env-copy-seq e)
    (let ((new-e (add-or-edit-breakpoint e (or start end) (eref e (or start end)))))
      (if (and start end)
	  (add-or-edit-breakpoint new-e end (eref new-e end))
	new-e))))
    
(defun env-fill (sequence item &key start end)

  ;; assume we're talking about y values here and start and end are x-axis values.
  ;; As usual, we have to add a breakpoint if none exists (at start or end)
  ;; We ignore CLtL p398 -- this does not destroy the original envelope.

  (let* ((new-e (env-with-start-end sequence start end))
	 (x0 (or start 0.0))
	 (x1 (or end (env-last-x sequence))))
    (loop for x in new-e by #'cddr and i from 1 by 2 do
      (if (<= x0 x x1) (setf (nth i new-e) item)))
    new-e))

(defun efind (e x)
  (loop for x0 in e by #'cddr and i from 0 by 1 do
    (if (>= x0 x) (return-from efind i))))

(defun env-section (e &optional start end)
  (if (or (= 1 (env-length e))
	  (and (not start) (not end)))
      (env-copy-seq e)
    (let ((x0 (or start (first e)))
	  (x1 (or end (env-last-x e))))
      (env-subseq e (efind e x0) (1+ (efind e x1))))))

(defun scaled-env-section (e x0 x1 new-x0 new-x1)
  (let ((new-e (env-section e x0 x1))
	(newx (/ (- new-x1 new-x0) (- x1 x0))))
    (loop for x in new-e by #'cddr and i from 0 by 2 do
      (setf (nth i new-e) (+ new-x0 (* newx (- x x0)))))
    new-e))
	
(defun env-replace (env1 env2 &key start1 end1 start2 end2)

  ;; again, start and end values refer to the x-axis.  env2 from start2 to end2
  ;; is inserted into env1 from start1 to end1 (if needed, the x-axis of env2
  ;; is changed to fit into start1 to end1 -- this involves scaling as well
  ;; as offset).  This is slightly different than the lisp replace function.

  (let* ((e1-x0 (or start1 (first env1)))
	 (e1-x1 (or end1 (env-last-x env1)))
	 (e2-x0 (or start2 (first env2)))
	 (e2-x1 (or end2 (env-last-x env2)))
	 (new-e1 (env-with-start-end env1 e1-x0))
	 (new-e2 (env-with-start-end env2 e2-x0 e2-x1)))
    (merge-envelopes (env-section new-e1 (first env1) e1-x0) 
		     (scaled-env-section new-e2 e2-x0 e2-x1 e1-x0 e1-x1)
		     (env-section new-e1 e1-x1 (env-last-x env1)))))

(defun scan-env (envelope function)
  (if envelope
      (let ((x0 (first envelope))
	    (y0 (second envelope)))
	(if (= (env-length envelope) 1)
	    (funcall function x0 y0 x0 y0)
	  (let ((pt nil))
	    (loop for x1 in (cddr envelope) by #'cddr and
	              y1 in (cdddr envelope) by #'cddr do
	      (if (setf pt (funcall function x0 y0 x1 y1))
		  (return-from scan-env pt))
	      (setf x0 x1)
	      (setf y0 y1))
	    pt)))))

(defun scan-env-backwards (envelope function)
  (if envelope
      (let ((x1 (env-last-x envelope))
	    (y1 (nth (1- (length envelope)) envelope))
	    (x0 nil)
	    (y0 nil))
	(if (= (env-length envelope) 1)
	    (funcall function x1 y1 x1 y1)
	  (let ((pt nil))
	    (loop for i from (- (length envelope) 4) downto 0 do
	      (setf x0 (nth i envelope))
	      (setf y0 (nth (1+ i) envelope))
	      (if (setf pt (funcall function x0 y0 x1 y1))
		  (return-from scan-env-backwards pt))
	      (setf x1 x0)
	      (setf y1 y0))
	    pt)))))

(defun env-find (item env &key from-end test start end key) ;key option doesn't make any sense here
  (declare (ignore key))
  (if env
      (let ((act-test (or test #'(lambda (x0 y0 x1 y1) 
				   (if (or (= item y0) 
					   (= item y1) 
					   (<= y0 item y1) 
					   (>= y0 item y1))
				       (list (+ x0 (* (- x1 x0) 
						      (/ (- item y0) 
							 (- y1 y0)))) 
					     item)))))
	    (act-env (env-section (env-with-start-end env start end) start end)))
	(if from-end
	    (scan-env-backwards act-env act-test)
	  (scan-env act-env act-test)))))
				  
(defun env-position (item env &key from-end test start end key)
  (declare (ignore key))
  (let ((pt (env-find item env :from-end from-end :start start :end end :test test)))
    (and pt (car pt))))

;;; env-remove could use remove-breakpoint and add-or-edit-breakpoint to clobber portions of the envelope


(defun point-on-line-p (px py qx qy tx ty)

  ;; is point tx ty on line defined by px py and qx qy --
  ;; nil if no, :before if on ray from p, :after if on ray from q, :within if between p and q
  ;; (these are looking at the "line" as a fat vector drawn on a grid)
  ;; taken from "Graphics Gems" by Glassner, code by A Paeth

  (if (or (= py qy ty) (= px qx tx))
      :within
    (if (< (abs (- (* (- qy py) (- tx px))
		   (* (- ty py) (- qx px))))
	   (max (abs (- qx px))
		(abs (- qy py))))
	(if (or (and (< qx px) (< px tx))
		(and (< qy py) (< py ty)))
	    :before
	  (if (or (and (< tx px) (< px qx))
		  (and (< ty py) (< py qy)))
	      :before
	    (if (or (and (< px qx) (< qx tx))
		    (and (< py qy) (< qy ty)))
		:after
	      (if (or (and (< tx qx) (< qx px))
		      (and (< ty qy) (< qy py)))
		  :after
		:within)))))))

(defun env-simplify (env &optional (ygrid 10) (xgrid 100))

  ;; grid = how fine a fluctuation we will allow.
  ;; the smaller the grid, the less likely a given bump will get through
  ;; original x and y values are not changed, just sometimes omitted.

  (if (and env
	   (> (env-length env) 2))
      (let* ((new-env (list (second env) (first env)))
	     (ymax (loop for y in (cdr env) by #'cddr maximize y))
	     (ymin (loop for y in (cdr env) by #'cddr minimize y))
	     (xmax (env-last-x env))
	     (xmin (first env)))
	(if (= ymin ymax)
	    (list xmin ymin xmax ymax)
	  (let* ((y-scl (/ ygrid (- ymax ymin)))
		 (x-scl (/ (or xgrid ygrid) (- xmax xmin)))
		 (px nil) (py nil)
		 (qx nil) (qy nil) 
		 (tx nil) (ty nil) 
		 (qtx nil) (qty nil))
	    (loop for ttx in env by #'cddr and
	              tty in (cdr env) by #'cddr do
	      (setf tx (round (* ttx x-scl)))
	      (setf ty (round (* tty y-scl)))
	      (if px
		  (if (not (point-on-line-p px py qx qy tx ty))
		      (progn
			(push qtx new-env)
			(push qty new-env)
			(setf px qx
			      py qy)))
		(setf px qx
		      py qy))
	      (setf qx tx
		    qy ty
		    qtx ttx
		    qty tty))
	    (push qtx new-env)
	    (push qty new-env)
	    (nreverse new-env))))
    (env-copy-seq env)))
	       


(defun reduce-amplitude-quantization-noise (e dur amp &optional (ramp-dur .5) (low-amp .005))
  ;; at very low amplitude (0.0 to .005), a ramp can cause an irritating sweeping buzz.
  ;; We will try to check for that case here and reduce its noticeability.
  ;; My previous theory about this noise seems incorrect to me now -- if I simply
  ;; multiply the "bad" sound by some scaler (say 100), the badness disappears,
  ;; so what we're actually protecting against here is the waveshaping non-linearities
  ;; in run-of-the-mill amplifiers and headphones.
  (let ((new-e nil))
    (when (and e
	       (> (length e) 2)
	       (> dur ramp-dur)
	       (plusp amp))
      (let* ((last-x (env-last-x e))
	     (first-x (car e))
	     ;; for now assume e is 0..1 on y axis
	     (x-dur (* ramp-dur (/ (- last-x first-x) dur)))
	     (y-val (/ low-amp amp))
	     (x0 first-x)
	     (y0 (cadr e)))
	(loop for x1 in (cddr e) by #'cddr and y1 in (cdddr e) by #'cddr do
	  (if (and (> (- x1 x0) x-dur)	;segment is longer than ramp-dur
		   (/= y0 y1))		;only ramp gives this buzz (otherwise "normal" noise)
	      (if (< y0 y-val)		;left value is below low-amp
		  (if (< y0 y1)		;going up => make it go up sooner
		      (setf new-e (add-or-edit-breakpoint (or new-e (env-copy-seq e))
							  (+ x0 x-dur) (min y1 (max y-val (eref e (+ x0 x-dur))))))
					;going down => delay down-slope as long as possible
		    (setf new-e (add-or-edit-breakpoint (or new-e (env-copy-seq e))
							  (- x1 x-dur) (min y0 (max y-val (eref e (- x1 x-dur)))))))
		(if (< y1 y-val)	;y0>y-val here, so we are going down and need not check for both y0<y-val and y1<y-val
		    (setf new-e (add-or-edit-breakpoint (or new-e (env-copy-seq e))
							(- x1 x-dur) (max y-val (eref e (- x1 x-dur))))))))
	  (setf x0 x1)
	  (setf y0 y1))))
    (or new-e e)))


(defun meld-envelopes (e0 e1)
  (if (not e0) 
      e1
    (if (not e1)
	e0
      (let* ((e0-last-x (env-last-x e0))
	     (x-scl (/ 1.0 e0-last-x)))
	(map-across-envelopes #'(lambda (x y)
				  (if (>= x e0-last-x)
				      (cadr y)
				    (+ (car y) (* (- (cadr y) (car y))
						  (* x x-scl)))))
			      e0 e1)))))


;;; fft-env-simplify uses fft filtering to smooth an envelope -- aimed at Sansy envelopes -- there
;;; are lots of variations on this idea depending on the application.  In "Numerical Recipes" they
;;; remove the linear trend of the data, but I have found that causes confusion when the underlying
;;; envelope is really a randomly jittered sine wave without any true linear trend.  Also, here I
;;; believe the min and max values and rescale the result to match the ambitus of the input -- 
;;; cave canem or whatever.

(defun fft-env-simplify (e &optional cutoff)
  (let* ((min-step (loop for x0 in e by #'cddr and 
		             x1 in (cddr e) by #'cddr 
		    minimize (- x1 x0)))
	 (x-extent (- (env-last-x e) (first e)))
	 (x0 (first e))
	 (pts (floor (/ x-extent min-step)))
	 (n (expt 2 (ceiling (log (1+ pts) 2.0))))
	 (cut-n (floor (* n (or cutoff .5))))
	 (datar (make-array n :element-type 'float :initial-element 0.0))
	 (datai (make-array n :element-type 'float :initial-element 0.0)))
    (loop for x from x0 by min-step and i from 0 to pts do
      (setf (aref datar i) (list-interp x e)))
    (let* ((min-y0 (loop for i from 0 to pts minimize (aref datar i)))
	   (max-y0 (loop for i from 0 to pts maximize (aref datar i)))
	   (midpt (* .5 (+ min-y0 max-y0))))
      (loop for i from 0 to pts do
	(decf (aref datar i) midpt))
      (_fft datar datai n 1)    
      (loop for i from cut-n below n do
	(setf (aref datar i) 0.0)
	(setf (aref datai i) 0.0))
      (_fft datar datai n -1)
      (let* ((min-y1 (loop for i from 0 to pts minimize (aref datar i)))
	     (max-y1 (loop for i from 0 to pts maximize (aref datar i)))
	     (inv-n (/ (- max-y0 min-y0) (- max-y1 min-y1)))
	     (y-off (- (* inv-n min-y1) min-y0)))
	(loop for i from 0 to pts and x from 0.0 by min-step 
	 collect x 
	 collect (- (* inv-n (aref datar i)) y-off))))))
