;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;
(in-package :clm)
;;;
;;; 56000 assembler
;;;
;;; -----------------------------------------------------------------------------
;;;
;;;     We try to stay relatively close to Motorola assembler syntax (for examples, see the end of this file or lib56.lisp).
;;;
;;;     However, I find their way of dividing up MOVE to be confusing, so
;;;     here's the method I use:
;;; There are four kinds of move: register to memory -- STORE,
;;;                               register to register -- COPY (also TRANSFER, LUA, and UPDATE in a sense)
;;;                               memory to register -- LOAD
;;;                               memory to memory -- MOVE (only for peripheral data to memory)
;;; On this chip, there are four kinds of memory: X, Y, P (the latter for program memory), and L (concatenation of X and Y)
;;; We would try to ignore this difference except that the parallel moves take advantage of it.  Also, for
;;; "peripheral" access, use X-IO and Y-IO as the memory names.
;;;
;;; There are the following registers, or concatenations of registers:
;;;    A B AB BA A0 A1 A2 A10 B0 B1 B2 B10 (the accumulators)
;;;    X Y X0 X1 Y0 Y1 (input registers)
;;;    R0 to R7 (address registers), N0 to N7 (associated offsets), M0..M7 (associated modifiers)
;;;    PC MR OMR CCR SR LA LC SP SS SSH SSL
;;; There are various sizes of data and addresses (short and long, peripheral IO, etc)
;;; There are the following addressing modes: immediate (short and long), absolute (i.e. the
;;;    address itself is given -- short or long), register (this includes all registers --
;;;    in particular N0..N7 and M0..M7 can be used as temporaries, register indirect (with
;;;    the following indirection modes: postdecrement by 1 or by N, predecrement by 1, postincrement
;;;    by 1 or N, R and N used together with no update, and R used without update).
;;; Here are my names for the register indirection modes: R-N R+N R-1 R+1 R RN 1-R. 
;;; Motorola uses syntax like a colon (in "X:") to distinguish X the register from X the memory.
;;; I use LOAD, STORE, and COPY, where we can be guaranteed what X is at each point.  Immediate
;;; data is told from absolute addresses because the latter have to have some memory partition
;;; indication: (LOAD A 32) as opposed to (LOAD A X 32).
;;; To force a short command, append the word SHORT: (LOAD A 32 SHORT).  Most such cases are automatically
;;;    made SHORT by the assembler, but SHORT is not assumed everywhere (as in Motorola).
;;; Address registers are accessed: (LOAD A X R3 RN).
;;;
;;; It is also possible to simply update an address register in a parallel "move",
;;;     so, we include UPDATE n m: update reg n according to mode m
;;;
;;; Rev C of the 56000 apparently fixes the problems in BSET #n,D and friends (see warnings below), and
;;;       removes the restriction on LA-1 for ENDDO, adds the following parallel moves:
;;;       MOVE A,X:<ea> X0,A
;;;       MOVE B,X:<ea> X0,B
;;;       MOVE Y0,A A,Y:<ea>
;;;       MOVE Y0,B B,Y:<ea>
;;;       also added are some SCI and SSI control bits, and stack error is sticky.
;;; For now, we'll ignore Rev C.


(defun operation (x) (car x))		;(LOAD ...
(defun store-source (x) (cadr x))	;(store A ...) 
(defun store-destination (x) (cddr x))	;(store n X R3 RN)
(defun load-source (x) (cddr x))	;(load n X 1234)
(defun load-destination (x) (cadr x))	;(load B ...)
(defun copy-source (x) (cadr x))	;(copy n M7)
(defun copy-destination (x) (caddr x))	;(copy SP ...)
(defun update-register (x) (cadr x))	;(update R4 ...)
(defun memory-side (x) (car x))		;(X ...) (returned by one of the above as a list = CDR in effect
(defun abs-addr (x) (cadr x))		;(x 1234)
(defun update-mode (x) (caddr x))	;(x r4 R+N)

(defconstant absolute-address-mode #x30)
(defconstant immediate-data-mode #x34)

(defun Reg-num (x)
  (case x 
    ((X0 Y0 A0 B0 R0 N0 M0) 0) 
    ((X1 Y1 A1 B1 R1 N1 M1) 1)
    ((A2 B2 R2 N2 M2) 2)
    ((R3 N3 M3) 3)
    ((R4 N4 M4) 4)
    ((R5 N5 M5) 5)
    ((R6 N6 M6) 6)
    ((R7 N7 M7) 7)
    (t (error "~S is not a valid numbered register:" x))))

(defconstant data-registers '(X Y X1 X0 Y1 Y0 A0 A1 A2 A10 B0 B1 B2 B10 AB BA))
(defun data-register (x) (member x data-registers))

(defconstant address-registers '(R0 R1 R2 R3 R4 R5 R6 R7 
				 N0 N1 N2 N3 N4 N5 N6 N7
				 M0 M1 M2 M3 M4 M5 M6 M7))
(defun address-register (x) (member x address-registers))

(defconstant R-and-N-registers '(R0 R1 R2 R3 R4 R5 R6 R7
				 N0 N1 N2 N3 N4 N5 N6 N7))
(defun R-and-N-register (x) (member x R-and-N-registers))

(defconstant modifier-registers '(M0 M1 M2 M3 M4 M5 M6 M7))
(defun modifier-register (x) (member x modifier-registers))

(defconstant R-registers '(R0 R1 R2 R3 R4 R5 R6 R7))
(defun R-register (x) (member x R-registers))
 
(defconstant control-registers '(PC MR CCR SR OMR LA LC SP SS SSH SSL))
(defun control-register (x) (member x control-registers))

(defconstant standard-registers '(X0 X1 Y0 Y1 A0 B0 A2 B2 A1 B1 A B
				  R0 R1 R2 R3 R4 R5 R6 R7
				  N0 N1 N2 N3 N4 N5 N6 N7))
(defun standard-register (x) (member x Standard-registers))
				  
(defun immediate-short-data (x) (= 0 (logand x (ash -1 8))))
(defun immediate-short-address (x) (= 0 (logand x (ash -1 6))))
(defun normal-data (x) (= 0 (logand x (ash -1 24))))
(defun normal-address (x) (= 0 (logand x (ash -1 16))))
(defun acceptable-bit (x) (< x 24))
(defun short-jump-address (x) (= 0 (logand x (ash -1 12))))
;;; (defun IO-short-address (x) (= 0 (logand x (ash -1 6)))) ;;; this is wrong because it's assumed to be a number like #xFFEB
(defun IO-address (x) (logand x #x3F))

(defun encode-register (x)
  (case x
    (SR  #b111001) 
    (OMR #b111010) 
    (SP  #b111011) 
    (SSH #b111100)
    (SSL #b111101) 
    (LA  #b111110) 
    (LC  #b111111)
    (A0  #b001000)
    (B0  #b001001)
    (A2  #b001010)
    (B2  #b001011)
    (A1  #b001100)
    (B1  #b001101)
    (A   #b001110)
    (B   #b001111)
    (X0  #b000100)
    (X1  #b000101)
    (Y0  #b000110)
    (Y1  #b000111)
    ((R0 R1 R2 R3 R4 R5 R6 R7) (logior (reg-num x) #b010000))
    ((N0 N1 N2 N3 N4 N5 N6 N7) (logior (reg-num x) #b011000))
    ((M0 M1 M2 M3 M4 M5 M6 M7) (logior (reg-num x) #b100000))
    (t (error "invalid register: ~S" x))))

(defun encode-standard-register-for-XY-load (x)
  (case x				;insert a 0 just for confusion's sake
    (A0  #b010000)
    (B0  #b010001)
    (A2  #b010010)
    (B2  #b010011)
    (A1  #b010100)
    (B1  #b010101)
    (A   #b010110)
    (B   #b010111)
    (X0  #b000100)
    (X1  #b000101)
    (Y0  #b000110)
    (Y1  #b000111)
    ((R0 R1 R2 R3 R4 R5 R6 R7) (logior (reg-num x) #b100000))
    ((N0 N1 N2 N3 N4 N5 N6 N7) (logior (reg-num x) #b110000))
    (t (error "invalid register: ~S" x))))

;;; register addressing modes
(defun addr-mode (m)
  (case m (R-N 0) (R+N 1) (R-1 2) (R+1 3) (R 4) (RN 5) (1-R 7) (t (error "invalid address mode: ~A" m))))

(defvar current-addr-mode nil)
(defvar current-source-register nil)

(defun eff-addr-mode (r m)
  (setf current-addr-mode m)
  (setf current-source-register (nth r r-registers))
  (logior r (ash (addr-mode m) 3)))

(defun uses-address-modifiers (m)
  (member m '(R-N R+N RN)))

(defun current-register-is-a-pointer ()
  current-addr-mode)

;;; We worry about these "insane instructions" because we can burn up the durned chip!
;;; Later that year... CC tells me this bug has been fixed, and you cannot burn up the chip anymore.

(defun check-for-insane-instruction (x)
  x)					;apparently no longer a problem
#|
  (if (not (member (logand x #xFF) '(0 #b100 #b100 #b1100)))
      (cond 
       ;; "Insane X: and Y: Parallel Move"
       ((= (logand x #xCF8000) #xCF8000) (error "X:ea,B Y:ea,B"))
       ((= (logand x #xCF8000) #xCA8000) (error "X:ea,A Y:ea,A"))
       ((= (logand x #x8C8008) #x8C8008) (error "X:ea B AC,B"))
       ((= (logand x #xC30008) #xC30008) (error "Y:ea,B AC,B"))
       ((= (logand x #x8C8008) #x888000) (error "X:ea,A AC A"))
       ((= (logand x #xC30008) #xC20000) (error "Y:ea,A AC,A"))
       ;; "Insane X: or Y: Parallel Move"
       ((= (logand x #xF18008) #x518008) (error "X:xx,B or Y:xx,B and AC,B"))
       ((= (logand x #xF18008) #x508000) (error "X:xx,A or Y:xx,A and AC,A"))
       ;; "Insane L: Parallel Move"
       ((= (logand x #xFF8008) #x418008) (error "L:xx,B10 AC,B"))
       ((= (logand x #xFF8008) #x408000) (error "L:xx,A10 AC,A"))
       ((= (logand x #xFF8008) #x498008) (error "L:xx,B AC,B"))
       ((= (logand x #xFF8008) #x488000) (error "L:xx,A AC,A"))
       ((= (logand x #xFF8000) #x4B8000) (error "L:xx,BA AC,B or A"))
       ((= (logand x #xFF8000) #x4A8000) (error "L:xx,AB AC,A or B"))
       ;; "Insane I: Parallel Move"
       ((= (logand x #xF90008) #x290008) (error "#xx,B AC,B"))
       ((= (logand x #xF90008) #x280000) (error "#xx,A AC,A"))
       ;; "Insane R: Parallel Move"
       ((= (logand x #xFC1908) #x200908) (error "reg,B AC,B"))
       ((= (logand x #xFC1908) #x200800) (error "reg,A AC,A"))
       ;; "Insane R:Y Parallel Move"
       ((= (logand x #xF3C008) #x13C008) (error "Y:ea,B AC,B"))
       ((= (logand x #xF3C008) #x12C000) (error "Y:ea,A AC,A"))
       ;; "Insane X:R Parallel Move"
       ((= (logand x #xFCC008) #x1C8008) (error "X:ea,B AC,B"))
       ((= (logand x #xFCC008) #x188000) (error "X:ea,A AC,A"))
       ;; "Insane R:Y and X:R Parallel Move"
       ((= (logand x #xFFC008) #x080000) (error "A,X:ea X0,A AC,A"))
       ((= (logand x #xFFC008) #x090008) (error "B,X:ea X0,B AC,B"))
       ((= (logand x #xFFC008) #x088000) (error "Y0,A A,Y:ea AC,A"))
       ((= (logand x #xFFC008) #x098008) (error "Y0,B B,Y:ea AC,B"))
       ;; "Reserved Operation Codes"
       ((or (= (logand x #xFFE000) #x206000)
	    (and (= (logand x #xFFC000) #x200000) (/= (logand x #x003F00) 0))
	    (= (logand x #xFF4080) #x074000)
	    (= (logand x #xFF4080) #x070080)
	    (= (logand x #xFFC0A0) #x068020)
	    (= (logand x #xFFC0A0) #x068000)
	    (= (logand x #xFF00A0) #x040020)
	    (= (logand x #xFE0080) #x020080)
	    (= (logand x #xFF0080) #x010080) ;there's an unexplained "D" at bit 3
	    (= (logand x #xFF8080) #x010000) ;ditto
	    (= (logand x #xFF00A0) #x000020))
	(error "reserved opcode"))
       (t x))))
|#


;;; the actual outgoing dsp code 
;;; Here we have to deal with the complicated memory map on the nExt -- 512 internal P, 256 internal X|Y,
;;;  then 8K external memory accessible in two images, with various kinds of overlays.
;;;  So, for P, use dsp-program until it fills up, then jump to external using $2000 addressing (this
;;;  gives us the full 8K if needed, whereas image 2 gives only 4K).  Similarly, fill up 256 X|Y first,
;;;  then start filling external -- if code still being sent, include jump around data.  X|Y|L external
;;;  should use image 2 ($A000) so that L mem refs work transparently.

(defvar internal-p-memory (make-array 8192 :element-type 'fixnum :initial-element 0))

(defvar x-pc 0)
(defvar internal-x-memory (make-array 8192 :element-type 'fixnum :initial-element 0))

(defvar y-pc 0)
(defvar internal-y-memory (make-array 8192 :element-type 'fixnum :initial-element 0))

(defvar ext-pc 0)
(defvar external-memory (make-array 65536 :element-type 'fixnum :initial-element 0))

(defconstant initial-x-ptr 5)
(defconstant initial-y-ptr 5)
(defconstant initial-heap-ptr 5)

(defvar ex-ptr 0)
(defvar ix-ptr initial-x-ptr)
(defvar ey-ptr 0)
(defvar iy-ptr initial-y-ptr)
(defvar heap-ptr initial-heap-ptr)

(defun initialize-chip-memory ()
  (setf ex-ptr 0
	ey-ptr 0
	ix-ptr initial-x-ptr
	iy-ptr initial-y-ptr
	heap-ptr initial-heap-ptr))

(defun get-heap-ptr () 
  (if (< (max ix-ptr iy-ptr) internal-L-size)
      (setf heap-ptr (max ix-ptr iy-ptr))
    (setf heap-ptr (+ (max ex-ptr ey-ptr) ext-L-offset))))

(defun get-x-memory (&optional (block-size 1))
  (if (minusp block-size) (error "illegal memory request: (get-X-memory ~D)" block-size))
  (if (< (+ ix-ptr block-size) internal-L-size)
      (prog1
	  ix-ptr
	(incf ix-ptr block-size))
    (if (< (+ ex-ptr block-size) external-L-size)
	(prog1
	    (+ ex-ptr ext-X-offset)
	  (incf ex-ptr block-size))
      nil)))

(defun get-y-memory (&optional (block-size 1))
  (if (minusp block-size) (error "illegal memory request: (get-Y-memory ~D)" block-size))
  (if (< (+ iy-ptr block-size) internal-L-size)
      (prog1
	  iy-ptr
	(incf iy-ptr block-size))
    (if (< (+ ey-ptr block-size) external-L-size)
	(prog1
	    (+ ey-ptr ext-Y-offset)
	  (incf ey-ptr block-size))
      nil)))

(defun get-a-block-of-x-or-y-memory (block-size &optional (moan t) (left-over 0))
  (if (and (plusp left-over)
	   (plusp external-L-size)
	   (< (- external-L-size (min ex-ptr ey-ptr)) left-over))
      (progn 
	(if moan (error "ran out of memory"))
	(values -1 nil))
    (let ((base (if (or (< ix-ptr iy-ptr) (< ex-ptr ey-ptr))
		    (get-x-memory block-size)
		  nil)))
      (if base 
	  (values base t)
	(if (setf base (get-y-memory block-size))
	    (values base nil)
	  (if (setf base (get-x-memory block-size))
	      (values base t)
	    (progn 
	      (if moan (error "ran out of memory"))
	      (values -1 nil))))))))

(defun get-L-mem (&optional (block-size 1))
  (if (minusp block-size) (error "illegal memory request: (get-L-mem ~D)" block-size))
  (if (< (+ (get-heap-ptr) block-size) internal-L-size)
      (prog1
	  heap-ptr
	(incf heap-ptr block-size)
	(setf ix-ptr heap-ptr)
	(setf iy-ptr heap-ptr))
    (progn
      (if (< heap-ptr ext-L-offset)	;we haven't had to jump into external memory until now
	  (if (< block-size external-L-size)
	      (setf heap-ptr (+ (max ex-ptr ey-ptr) ext-L-offset))
	    (error "memory request too large: ~D" block-size)))
      (if (< (+ (- heap-ptr ext-L-offset) block-size) external-L-size)
	  (prog1
	      heap-ptr
	    (incf heap-ptr block-size)
	    (setf ex-ptr (- heap-ptr ext-L-offset))
	    (setf ey-ptr (- heap-ptr ext-L-offset)))
	(error "ran out of memory")))))

(defmacro get-L-memory (&optional (block-size 1)) `(get-L-mem ,block-size))

(defun new-heap-address () (list 'L (get-L-mem)))

(defun get-MOD-mem (block-size)
  (if (minusp block-size) (error "illegal memory request: (get-MOD-mem ~D)" block-size))
  (if (= ex-ptr 0) 
      (progn
	(incf ex-ptr block-size) 
	(values ext-X-offset t))
    (if (= ey-ptr 0) 
	(progn
	  (incf ey-ptr block-size) 
	  (values ext-Y-offset nil))
      (let* ((pow2 (ceiling (/ (log block-size) (log 2))))
	     (boundary (expt 2 pow2))
	     (base-x (* boundary (ceiling (/ ex-ptr boundary))))
	     (base-y (* boundary (ceiling (/ ey-ptr boundary)))))
	(if (< (+ (min base-x base-y) block-size) external-L-size)
	    (if (< base-x base-y) 
		(progn
		  (setf ex-ptr (+ base-x block-size))
		  (values (+ base-x ext-X-offset) t))
	      (progn
		(setf ey-ptr (+ base-y block-size))
		(values (+ base-y ext-Y-offset) nil)))
	  (let* ((ibase-x (* boundary (ceiling (/ ix-ptr boundary))))
		 (ibase-y (* boundary (ceiling (/ iy-ptr boundary)))))
	    (if (< (+ (min ibase-x ibase-y) block-size) internal-L-size)
		(if (< ibase-x ibase-y) 
		    (progn
		      (setf ix-ptr (+ ibase-x block-size))
		      (values ibase-x t))
		  (progn
		    (setf iy-ptr (+ ibase-y block-size))
		    (values ibase-y nil)))
	      (error "can't find room for MOD-based block"))))))))
	      

(defun edit-dsp-program (loc op)
  (if (< loc Internal-P-size)
      (setf (aref internal-p-memory loc) op)
    (setf (aref external-memory (- loc ext-P-offset)) op)))

(defvar last-dsp-pc -1)

;;; next stuff is needed in pipeline error checks (set as side-effects to avoid endless redundant instruction translation)
(defvar current-class -1)
(defvar current-register nil)

(defun report-inserted-jump (jump-pc) 
  (DEBUGGING (let ((old-op (pop emit-prog))
		   (new-pc jump-pc))
	       (push (append (list (first old-op)) (list (list 'JMP new-pc))) emit-prog)
	       (setf (first old-op) new-pc)
	       (push old-op emit-prog))))


(defun check-for-inserted-jump-into-external-memory ()
  (when (< (- internal-p-size 6) dsp-pc internal-p-size)
					;need two words for JMP to ext mem, can't interrupt subsequent 2 word instr
    (edit-dsp-program dsp-pc #xAF080)	;JMP ext-mem -- i.e. insert a long jmp to external memory, image 1
    (if (= ex-ptr 0)			;i.e. ext P mem not allocated already as X mem
	(progn
	  (report-inserted-jump ext-P-offset)	;a debugging procedure (advised later)
	  (edit-dsp-program (+ dsp-pc 1) ext-P-offset)
	  (setf dsp-pc ext-P-offset))
      (let ((new-pc (+ ext-P-offset ex-ptr)))
	(report-inserted-jump new-pc)
	(edit-dsp-program (+ dsp-pc 1) new-pc)
	(setf dsp-pc new-pc)))
    (setf current-register nil)		;tell pipe checks note to worry
    (setf current-class -1)
    (tick-pipe)))

(defun set-dsp-program (op)
  (edit-dsp-program dsp-pc op)
  (incf dsp-pc)
  (if (and (not (zerop ext-P-offset)) 
	   (> dsp-pc ext-P-offset))
      (setf ex-ptr (- dsp-pc ext-P-offset)))
  (if (> dsp-pc top-P-loc) (error "Ran out of memory")))
			
(defun get-dsp-program (loc)
  (if (< loc Internal-P-size)
      (aref internal-p-memory loc)
    (if (< loc top-P-loc)
	(aref external-memory (- loc ext-P-offset))
      (error "~D is not a legal P address" loc))))

(defvar two-word-instruction nil)

(defun dsp-command (&rest x)
  (set-dsp-program (apply #'logior x)))

(defun ext-command (o)
  (set-dsp-program o))

(defun setf-x-mem (loc val)
  (declare (optimize (speed 3) (safety 0)))
  (if (< loc internal-L-size)
      (setf (aref internal-x-memory loc) val)
    (if (< loc top-L-loc)
	(setf (aref external-memory (- loc ext-X-offset)) val)
      (error "~X is an illegal external X memory address" loc))))

(defun setf-y-mem (loc val)
  (declare (optimize (speed 3) (safety 0)))
  (if (< loc internal-L-size)
      (setf (aref internal-y-memory loc) val)
    (if (< loc top-L-loc)
	(setf (aref external-memory (+ (- loc ext-L-offset) External-Y-from-X-offset)) val)
      (error "~X is an illegal external Y memory address" loc))))

(defun setf-xy-mem (loc xval yval)
  (declare (optimize (speed 3) (safety 0)))
  (if (< loc internal-L-size)
      (progn
	(setf (aref internal-x-memory loc) xval)
	(setf (aref internal-y-memory loc) yval))
    (if (< loc top-L-loc)
	(progn
	  (setf (aref external-memory (- loc ext-X-offset)) xval)
	  (setf (aref external-memory (+ (- loc ext-L-offset) External-Y-from-X-offset)) yval))
      (error "~X is an illegal external L memory address" loc))))

(defun getf-x-mem (loc)
  (if (< loc internal-L-size)
      (aref internal-x-memory loc)
    (if (< loc top-L-loc)
	(aref external-memory (- loc ext-X-offset))
      (error "~X is an illegal external X memory address" loc))))

(defun getf-y-mem (loc)
  (if (< loc internal-L-size)
      (aref internal-y-memory loc)
    (if (< loc top-L-loc)
	(aref external-memory (+ (- loc ext-L-offset) External-Y-from-X-offset))
      (error "~X is an illegal external X memory address" loc))))

;;; we have to subtract one from the DO end label (LA)

(defvar do-fixup-stack nil)

(defun ext-do-command (o)
  (if (and (integerp o)
	   (> o 0))
      (set-dsp-program (- o 1))
    (progn 
      (push dsp-pc do-fixup-stack)
      (set-dsp-program o)))
  (setf two-word-instruction t))


;;; LABELS
;;;    These are declared for use with "name".  They are defined as (name) (as a pseudo-op).
;;;    They are used by referring to the name (as a jump address and so on).  We assume all program
;;;    addresses fit in the 12 bit jump address field of commands like JMP (as opposed to the 16 bit
;;;    address extension word).  To fix up locations once the label's definition has been find, we
;;;    build a fixup chain (using the 12 bit field), and make one last paranoid check for illegal
;;;    bit pattern after fixing up the address.  So, if we come upon JMP L-1234 and haven't ever
;;;    seen L-1234 before, we put it on the label list with fixup pc, return 0 (as end of fixup
;;;    chain).  Each reference until it is defined gets current fixup, resets fixup to current pc.
;;;    Once defined, it's just a number, offset from program start (i.e. 0 in most cases).

(defvar relocation-bit 0)
(defun make-relocatable-program () (setf relocation-bit #x1000000))
(defun make-absolute-program () (setf relocation-bit 0))
(defun relocate (program program-length start-addr)
  (loop for i from 0 below program-length do
    (if (/= 0 (logand (aref program i) #x1000000))
	(setf (aref program i) (+ (logand (aref program i) #xFFFFFF) start-addr)))))
;;; i.e. to make a relocatable module, first call MAKE-RELOCATABLE-PROGRAM, set dsp-pc to 0
;;; (i.e. P-ORG 0 I believe), then emit the code, and when ready to load, call RELOCATE
;;; to fix up all the jump addresses.  Currently, we don't fix up anything but jumps.


(defvar Labels (make-hash-table))	;table of current known labels
(defvar last-label nil)

(defun get-label-value (name &optional (offset 0)) ;name => (def-loc fix-loc) (def-loc undefined until set)
  (multiple-value-bind (val exists) (gethash name labels)
    (if exists 
	(if (not (eq 'UNDEFINED (car val)))
	    (+ (car val) relocation-bit) ;label is known, defined
	  (let ((curval (cadr val)))	;label is known, undefined, so continue fixup chain
	    (setf (cadr val) (+ dsp-pc offset))
	    (setf (gethash name labels) val)
	    curval))			;return old fixup loc
      (progn				;label is not known yet
	(setf (gethash name labels) (list 'UNDEFINED (+ dsp-pc offset)))
	(setf last-label name)
	0))))

(defun fixup-moved-labels (loc)
  (if last-label
      (let* ((lab (gethash last-label labels))
	     (addr (cadr lab)))
	(if (eq 'UNDEFINED (car lab))
	    (if (or (= loc addr)
		    (= (1+ loc) addr))
		(setf (gethash last-label labels) (list 'UNDEFINED (1+ addr))))))))
      
(defun get-fixup-chain-address (loc name) ;returns fixup-loc and program word without chain info
  (let ((full-op (get-dsp-program loc)))
    (if (= 0 (logand full-op #xFF0000))	;normal case for two-word jump instruction
	(values full-op 0)		;so entire 2nd word is next 16-bit fixup loc
      (if (> (logand (ash full-op -16) #xF) #xB)
	  ;; short Jcc = E, short JMP = C, short JScc = F, short JSR = D
	  ;; long Jcc = A, long JMP = A, long JScc = B, long JSR = B
	  (values (logand full-op #xFFF) (logand full-op #xFFF000))
	(error "confusion in fixup-chain (probably multiple definitions of a label): #x~X at ~D for ~A" full-op loc name)))))

(defvar label-hook nil)

(defun fixup-label (name)
  (if label-hook (funcall label-hook name dsp-pc))
  (multiple-value-bind (val exists) (gethash name labels)
    (if exists
	(let ((p-loc dsp-pc))
	  (if (not (eq (car val) 'UNDEFINED))
	      (print (format nil "Multiple definitions of ~A?" name)))
	  (setf (car (gethash name labels)) p-loc)
	  (do ((loc (cadr val))		;top of fixup chain
	       (next-loc 0))
	      ((= 0 loc))
	    (if (< 0 loc dsp-pc)
		(multiple-value-bind
		    (fixup-loc instruction) (get-fixup-chain-address loc name)
		  (setf next-loc fixup-loc)
		  (if (and (not (short-jump-address p-loc))
			   (/= 0 instruction))
		      (error "warning: we appear to be trying to force a long jump address into a short jump: ~A at ~A" name loc))
		  (if (and do-fixup-stack
			   (= loc (car do-fixup-stack)))
		      (progn
			(pop do-fixup-stack)
			(edit-dsp-program loc (logior instruction relocation-bit (- p-loc 1))))
		    (edit-dsp-program loc (logior instruction relocation-bit p-loc)))
		  (setf loc next-loc))
	      (error "illegal fixup location: ~D for ~A" loc name))))
      (setf (gethash name labels) (list dsp-pc 0)))))

(defun check-for-undefined-labels ()
  (maphash #'(lambda (key val)
	       (if (eq (car val) 'UNDEFINED)
		   (error "undefined label: ~S" key))) labels))


;;; DEFINED NAMES

(defvar Names (make-hash-table))
(defun get-name-value (name) (gethash name names))
(defvar we-may-have-names nil)


(defconstant PIPE-BC-SP 0)		;BCHG|BCLR|BSET to SP
(defconstant PIPE-MOVE-from-SSH 1)	;MOVEc|m|p from SSH
(defconstant PIPE-MOVE-from-SSL 2)	;MOVEc|m|p from SSL
(defconstant PIPE-J-SSL 3)		;jump #n,SSL,xxxx
(defconstant PIPE-MOVE-SP 4)		;MOVEc|m|p SP
(defconstant PIPE-J-SSH 5)		;JCLR JSET JSCLR JSSET SSH
(defconstant PIPE-at-LA 6)		;Jcc JMP JScc JSR RESET RTI RTS STOP WAIT
(defconstant PIPE-after-REP 7)		;JSET JCLR JSCLR JSSET SWI
(defconstant PIPE-before-RTS 8)		;BCn SSH SSL, MOVEn SR SSH SSL
(defconstant PIPE-before-RTI 9)		;BCn SR, MOVEn SR, ANDI ORI MR
(defconstant PIPE-ANDI-CCR 10)		;ANDI ORI CCR
(defconstant PIPE-ENDDO 11)		;BCn LA LC, MOVEn LA LC
(defconstant PIPE-J-LA 12)		;JSR JScc JSCLR JSSET LA 
(defconstant PIPE-LA-3 13)		;BTST SSH
(defconstant PIPE-DO 14)		;DO
(defconstant PIPE-REP 15)		;REP


;;; moves and PARALLEL MOVES (90% of the assembler code)


;;; NO MOVE -----------------------------------------------------------------------------

(defun no-parallel-data (o)
  (dsp-command o #x200000))


;;; IMMEDIATE SHORT DATA MOVE -----------------------------------------------------------

(defun Immediate-short-load (o dest data)
  (if (and (standard-register dest) 
	   (immediate-short-data data))
      (progn
	(if (address-register dest) (setf current-register dest))
	(dsp-command o #x200000 (ash data 8) (ash (encode-register dest) 16)))
    (error "invalid operands for immediate short data move: ~S ~S" dest data)))


;;; REGISTER TO REGISTER MOVE -----------------------------------------------------------

(defun register-to-register-copy (o S D)
  (if (and (standard-register S) 
	   (standard-register D))
      (progn				;pipeline error check too
	(if (address-register D) (setf current-register D))
	(if (address-register S) (setf current-source-register S))
	(dsp-command o #x200000 (ash (encode-register S) 13) (ash (encode-register D) 8)))
    (error "invalid operands for register to register move: ~S ~S" S D)))


;;; REGISTER UPDATE  ; (UPDATE R3 R+1) for example ---------------------------------------

(defun ok-mode (m) (member m '(R-N R+N R-1 R+1)))

(defun address-register-update (o r m)
  (if (and (R-register r)
	   (ok-mode m))
      (dsp-command o #x204000 (ash (eff-addr-mode (reg-num r) m) 8))
    ;; I believe this command is not pipeline dependent.
    (error "invalid ~A for register update: ~S ~S" 
	   (if (not (R-register r)) "source register" "update mode") r m)))


;;; X MEMORY DATA MOVE AND Y DATA MEMORY MOVE --------------------------------------------

(defun ok-addr-mode (x) (member x '(R+N R-N R-1 R+1 R RN 1-R)))

(defun XY-load-or-store (o move)	;we get the whole move command

  ;; can be (LOAD d num) (LOAD d num SHORT) (LOAD d X num) (LOAD d Y num)
  ;;        (LOAD d X Rn mode) (LOAD d Y Rn mode) (LOAD d X num SHORT) and (LOAD d Y num SHORT)
  ;; both X and Y sides support their own (LOAD d Num) calls, which is a bit of redundancy
  ;; also (STORE D X|Y ) and so on, but no immediates here, of course

  (if (standard-register (load-destination move))
      (let* ((D (load-destination move))
	     (store (eq (car move) 'STORE))
	     (code-D (ash (encode-standard-register-for-XY-load D) 16))
	     (addr (load-source move)))	;a list (cddr move)
	(if (and (not store) (address-register D)) (setf current-register D))
	(if (or (eq 'X (car addr)) (eq 'Y (car addr)))
    ;;; it's a memory reference of some sort (not immediate data) -- (X num) (X R3 R+1) (X num SHORT)
    ;;; We try to use SHORT addressing wherever possible, even if not explicitly requested.
	    (let ((mem (memory-side addr)))
	      (if (integerp (cadr addr))
		  (let ((absolute-address (cadr addr)))
		    (if (immediate-short-address absolute-address)
			(dsp-command o code-D 
				     (ash absolute-address 8) 
				     #x400000
				     (if store 0 #x8000)
				     (if (eq 'Y mem) #x80000 0))
		      (if (normal-address absolute-address)
			  (progn
			    (dsp-command o code-D
					 #x404000
					 (if store 0 #x8000)
					 (if (eq 'Y mem) #x80000 0)
					 (ash absolute-address-mode 8))
			    (ext-command absolute-address))
			(error "invalid address: ~S" absolute-address))))
    ;;; register reference Rn mode
		(if (and (R-register (cadr addr))
			 (ok-addr-mode (caddr addr)))
		    (dsp-command o code-D
				 (ash (eff-addr-mode (reg-num (cadr addr)) (caddr addr)) 8)
				 #x404000
				 (if store 0 #x8000)
				 (if (eq 'Y mem) #x80000 0))
		  (error "invalid address: ~S" addr))))
    ;;; here we have immediate data, only 24 bits (LOAD A 32) -- addr will be (32)

	  (if (not store)
	      (if (cdr addr)		;cannot be SHORT here	
		  (error "invalid data syntax: ~S" addr)
		(if (normal-data (car addr))
		    (progn
		      (dsp-command o code-D
				   (ash immediate-data-mode 8)
				  ;;; (if (eq 'Y mem) #x80000 0) -- apparently I fell asleep typing
				   #x8000 ; write D
				   #x404000)
		      (ext-command (car addr)))
		  (error "invalid data: ~S" addr)))
	    (error "cannot STORE immediate! ~S" move))))
    (error "invalid destination for X or Y ~S: ~S" (car move) (cadr move))))
	
#|
(defun X-or-Y-load (o move) (X-or-Y-load-or-store o move))
(defun X-or-Y-store (o move) (X-or-Y-load-or-store o move))
|#

;;; X or Y MEMORY AND REGISTER DATA MOVE -------------------------------------------------
;;;    lots of possibilities here, but must be one COPY and one either LOAD or STORE

(defun X-R-S1-D1 (x) (case x (X0 0) (X1 1) (A 2) (B 3) (t (error "illegal S1/D1 for X-R: ~S" x))))
(defun Y-R-S2-D2 (x) (case x (Y0 0) (Y1 1) (A 2) (B 3) (t (error "illegal S2/D2 for R-Y: ~S" x))))
    
(defun X-or-Y-and-R-normal-load-or-store
    (o other copy load-or-store-addr load-or-store-reg write)
  (if (or (eq (copy-source copy) 'A) (eq (copy-source copy) 'B))
      (let ((x-side (or (eq (copy-destination copy) 'Y0) (eq (copy-destination copy) 'Y1))))
	(if (or (and x-side (eq (car load-or-store-addr) 'Y))
		(and (not x-side) (eq (car load-or-store-addr) 'X)))
	    (error "This ~S move isn't allowed: ~S ~S" (if x-side 'R-Y 'X-R) other copy))
	(if (integerp (cadr load-or-store-addr))
	    (if (and x-side (eq (car load-or-store-addr) 'X))
		(let ((act-addr (cadr load-or-store-addr)))
		  (dsp-command o
			       (ash absolute-address-mode 8)
			       #x100000
			       (if write #x8000 0)
			       (if (eq (copy-destination copy) 'Y0) 0 #x10000)
			       (if (eq (copy-source copy) 'A) 0 #x20000)
			       (ash (X-R-S1-D1 load-or-store-reg) 18))
		  (ext-command act-addr))
	      (if (eq (car load-or-store-addr) 'Y)
		  (let ((act-addr (cadr load-or-store-addr)))
		    (dsp-command o
				 (ash absolute-address-mode 8)
				 #x104000
				 (if write #x8000 0)
				 (if (eq (copy-destination copy) 'X0) 0 #x40000)
				 (if (eq (copy-source copy) 'A) 0 #x80000)
				 (ash (Y-R-S2-D2 load-or-store-reg) 16))
		    (ext-command act-addr))
		(error "invalid memory type: ~S" other)))
	;;reference to R reg class 1
	  (progn
	    (if (and x-side (eq (car load-or-store-addr) 'X))
		(dsp-command o
			     (ash (eff-addr-mode 
				   (reg-num (cadr load-or-store-addr)) 
				   (caddr load-or-store-addr)) 8)
			     #x100000
			     (if write #x8000 0)
			     (if (eq (copy-destination copy) 'Y0) 0 #x10000)
			     (if (eq (copy-source copy) 'A) 0 #x20000)
			     (ash (X-R-S1-D1 load-or-store-reg) 18))
	      (if (eq (car load-or-store-addr) 'Y)
		  (dsp-command o
			       (ash (eff-addr-mode (reg-num (cadr load-or-store-addr))
						   (caddr load-or-store-addr)) 8)
			       #x104000
			       (if write #x8000 0)
			       (if (eq (copy-destination copy) 'X0) 0 #x40000)
			       (if (eq (copy-source copy) 'A) 0 #x80000)
			       (ash (Y-R-S2-D2 load-or-store-reg) 16))
		(error "invalid memory type: ~S" other))))))
    (error "invalid copy-source: ~S" copy)))

(defun X-or-Y-and-R-load-or-store (o x-move y-move)
  (let* ((copy (if (eq (operation x-move) 'COPY) x-move
		 (if (eq (operation y-move) 'COPY) y-move
		   (error "at least one of these moves should be register to register: ~S" x-move y-move))))
	 (other (if (or (eq (operation x-move) 'LOAD) (eq (operation x-move) 'STORE)) x-move
		  (if (or (eq (operation y-move) 'LOAD) (eq (operation y-move) 'STORE)) y-move
		    (error "At least one of these moves should be a LOAD or STORE: ~S ~S" x-move y-move))))
	 (store (eq (operation other) 'STORE))
	 (load-or-store-reg (cadr other))
	 (load-or-store-addr (cddr other)))
    (if (integerp (car load-or-store-addr)) ;it's immediate data load + COPY
	(if store 
	    (error "attempt to store immediate! ~S" other)
	  (if (normal-data (car load-or-store-addr))
	      (if (and (member (copy-destination copy) '(Y0 Y1))
		       (member load-or-store-reg '(A B X0 X1))
		       (member (copy-source copy) '(A B)))
		  (progn
		    (dsp-command o 
				 (ash immediate-data-mode 8)
				 #x8000	;definitely a "write" of D
				 (if (eq (copy-destination copy) 'Y0) 0 #x10000)
				 (if (eq (copy-source copy) 'A) 0 #x20000)
				 (ash (X-R-S1-D1 load-or-store-reg) 18)
				 #x100000)
		    (ext-command (car load-or-store-addr)))
		(if (and (member (copy-destination copy) '(X0 X1))
			 (member load-or-store-reg '(A B Y0 Y1))
			 (member (copy-source copy) '(A B)))
		    (progn
		      (dsp-command o
				   (ash immediate-data-mode 8)
				   #x8000 ;a write from DSP viewpoint
				   (if (eq (copy-destination copy) 'X0) 0 #x40000)
				   (if (eq (copy-source copy) 'A) 0 #x80000)
				   (ash (Y-R-S2-D2 load-or-store-reg) 16)
				   #x104000)
		      (ext-command (car load-or-store-addr)))
		  (error "invalid collection of registers: ~S ~S" x-move y-move)))
	    (error "invalid immediate data: ~S" load-or-store-addr)))
      ;; here we have an address, not immediate load
      (if store				;the hard case -- three possibilities -- two class 2, one class 1
	  (if (or (eq (copy-source copy) 'X0) (eq (copy-source copy) 'Y0))
	      (let ((x-side (eq (copy-source copy) 'X0)))
		(if (integerp (cadr load-or-store-addr))
		    (if (or (eq (copy-destination copy) 'A) (eq (copy-destination copy) 'B))
			(if (or (eq load-or-store-reg 'A) (eq load-or-store-reg 'B))
			    (progn
			      (dsp-command o
					   (if x-side #x80000 #x88000)
					   (if (eq (copy-destination copy) 'A) 0 #x10000)
					   (ash absolute-address-mode 8))
			      (ext-command (cadr load-or-store-addr)))
			  (error "invalid ~S register for ~S-R move: ~S" (operation other) (if x-side 'X 'Y) other))
		      (error "invalid copy destination: ~S" copy))
		  ;; here we have a R reference in a class 2 command
		  (if (R-register (cadr load-or-store-addr))
		      (if (or (eq (copy-destination copy) 'A) (eq (copy-destination copy) 'B))
			  (if (or (eq load-or-store-reg 'A) (eq load-or-store-reg 'B))
			      (dsp-command o
					   (if x-side #x80000 #x88000)
					   (if (eq (copy-destination copy) 'A) 0 #x10000)
					   (ash (eff-addr-mode (reg-num (cadr load-or-store-addr)) 
							       (caddr load-or-store-addr)) 8))
			    (error "invalid ~S register for ~S-R move: ~S" (operation other) (if x-side 'X 'Y) other))
			(error "invalid copy destination: ~S" copy))
		    (error "invalid address: ~S" other))))
	    ;; here we are in class 1 where the COPY source is A or B
	    (X-or-Y-and-R-normal-load-or-store o other copy load-or-store-addr load-or-store-reg nil))

	;; here we are doing a class 1 load and copy
        (X-or-Y-and-R-normal-load-or-store o other copy load-or-store-addr load-or-store-reg t)))))

(defun X-R-Load-Copy (o x-move y-move)  (X-or-Y-And-R-load-or-store o x-move y-move))
(defun X-R-Store-Copy (o x-move y-move) (X-or-Y-And-R-load-or-store o x-move y-move))
(defun Y-R-Load-Copy (o x-move y-move)  (X-or-Y-And-R-load-or-store o x-move y-move))
(defun Y-R-Store-Copy (o x-move y-move) (X-or-Y-And-R-load-or-store o x-move y-move))
      
      
;;; LONG MEMORY DATA MOVE ----------------------------------------------------------------
;;;     (LOAD d L num) (LOAD d L R3 RN) (LOAD d L num SHORT) (also STORE)

(defun long-register (x) (member x '(A10 B10 X Y A B AB BA)))
(defun encode-long-register (x) 
  (case x
    (A10 0) (B10 #b0001) (X #b0010) (Y #b0011) (A #b1000) (B #b1001) (AB #b1010) (BA #b1011)))

(defun Long-memory-load-or-store (o move)
  (if (long-register (cadr move))
      (let* ((code-D (encode-long-register (cadr move)))
	     (store (eq (car move) 'STORE))
	     (addr (load-source move)))	;a list (cddr move)
	(if (integerp (cadr addr))
	    (let ((absolute-address (cadr addr)))
	      (if (immediate-short-address absolute-address)
		  (dsp-command o 
			       (ash code-D 16)
			       (ash absolute-address 8) 
			       #x400000
			       (if store 0 #x8000))
		(if (normal-address absolute-address)
		    (progn
		      (dsp-command o 
				   (ash code-D 16)
				   #x404000
				   (if store 0 #x8000)
				   (ash absolute-address-mode 8))
		      (ext-command absolute-address))
		  (error "invalid address: ~S" absolute-address))))
	  (if (and (R-register (cadr addr))
		   (ok-addr-mode (caddr addr)))
	      (dsp-command o 
			   (ash code-D 16)
			   (ash (eff-addr-mode (reg-num (cadr addr)) (caddr addr)) 8)
			   #x404000
			   (if store 0 #x8000))
	    (error "invalid address: ~S" addr))))
    (error "invalid ~A register for long move: ~S" (if (eq (car move) 'STORE) "source" "destination") move)))

(defun L-load (o move) (long-memory-load-or-store o move))
(defun L-store (o move) (long-memory-load-or-store o move))


;;; XY MEMORY DATA MOVE ------------------------------------------------------------------
;;;     here we have four combinations of load and store either x or y
;;;     we will assume the caller has put the moves in the correct order
;;;     special case here is that only R references are accepted, and if
;;;     one side uses R0..3, the other has to use R4..7 (LOAD A X R3 R)

(defun ok-XY-mode (m) (member m '(R+N R-1 R+1 R)))
(defun XY-addr-mode (x) (case x (R+N 1) (R-1 2) (R+1 3) (R 0)))
(defun ok-XY-X-S1-or-D1 (x) (member x '(X0 X1 A B)))
(defun ok-XY-Y-S2-or-D2 (x) (member x '(Y0 Y1 A B)))
(defun ok-XY-R-split (x y)
  (or (and (member x '(R0 R1 R2 R3)) (member y '(R4 R5 R6 R7)))
      (and (member x '(R4 R5 R6 R7)) (member y '(R0 R1 R2 R3)))))
(defun XY-S2-or-D2 (x) (case x (Y0 0) (Y1 1) (A 2) (B 3)))
(defun XY-S1-or-D1 (x) (case x (X0 0) (X1 1) (A 2) (B 3)))

(defun X-and-Y-load-or-store (o x-move y-move)
  (if (ok-XY-X-S1-or-D1 (cadr x-move))
      (if (ok-XY-Y-S2-or-D2 (cadr y-move))
	  (if (ok-XY-R-split (fourth x-move) (fourth y-move))
	      (if (ok-XY-mode (fifth x-move))
		  (if (ok-XY-mode (fifth y-move))
		      (dsp-command o
				   (ash (reg-num (fourth x-move)) 8)                ;X side R register
				   (ash (XY-addr-mode (fifth x-move)) 11)           ;X side addressing mode 
				   (ash (logand (reg-num (fourth y-move)) #b11) 13) ;low 2 bits of Y side R
				   (if (eq (car x-move) 'STORE) 0 #x8000)           ;is X read or write
				   (ash (XY-S2-or-D2 (second y-move)) 16)           ;Y side data register
				   (ash (XY-S1-or-D1 (second x-move)) 18)           ;X side data register
				   (ash (XY-addr-mode (fifth y-move)) 20)           ;Y side addressing mode
				   (if (eq (car y-move) 'STORE) 0 #x400000)         ;is Y side read or write
				   #x800000)                                        ;XY opcode 		
		    (error "invalid Y side address mode: ~S" y-move))
		(error "invalid X side address mode: ~S" x-move))
	    (error "XY memory data move requires split address registers: ~S ~S" x-move y-move))
	(error "invalid Y side data register in X:Y move: ~S" y-move))
    (error "invalid X side data register in X:Y move: ~S" x-move)))

		    
;;; MOVEC --------------------------------------------------------------------------------
;;; 5 different command classes -- MOVEC SSH,SSH is illegal

(defun movec-register (x) (or (control-register x) (modifier-register x)))

(defun X-or-Y-load-or-store-control-register (movec)
  ;; here we have these basic cases -- X or Y mem + SHORT, and immediate + SHORT
  ;; immediate looks like: (LOAD D num [SHORT])
  ;; memory ref: (LOAD|STORE D X|Y num|R.. [SHORT])
  (let ((D (encode-register (second movec)))
	(store (eq 'STORE (car movec)))
	(short (eq (car (last movec)) 'SHORT)))
    (if (not store)			;check pipeline errors
	(if (eq (second movec) 'SP) 
	    (setf current-class PIPE-MOVE-SP)
	  (if (or (eq (second movec) 'SR) (eq (second movec) 'SSH) (eq (second movec) 'SSL))
	      (setf current-class PIPE-before-RTS))))
    (if (integerp (third movec))
	(if (not store)
	    (if short
		(if (immediate-short-data (third movec))
		    (dsp-command (ash (third movec) 8) #x050080 D)
		  (error "~S is not a short datum" (third movec)))
	      (if (normal-data (third movec))
		  (progn		;8 bit = Write D1, 20 bit in encode-register
		    (dsp-command #x05c000 D (ash immediate-data-mode 8))
		    (ext-command (third movec)))
		(error "invalid immediate datum: ~S" (third movec))))
	  (error "Attempt to STORE immediate: ~S" movec))
      ;; here we have an address in X or Y memory, possibly short
      (if (not (or (eq (third movec) 'X) (eq (third movec) 'Y)))
	  (error "invalid memory type for movec: ~S" movec)
	(let ((x-mem (eq (third movec) 'X))
	      (abs-case (integerp (fourth movec))))
	  (if short
	      (if (immediate-short-address (fourth movec))
		  (dsp-command #x050000 D
			       (if store 0 #x8000)
			       (if x-mem 0 #x40)
			       (ash (fourth movec) 8))
		(Error "~S is not a short absolute address" (fourth movec)))
	    (progn
	      (dsp-command #x054000 D
			   (if store 0 #x8000)
			   (if x-mem 0 #x40)
			   (ash (if abs-case 
				    absolute-address-mode 
				  (eff-addr-mode (reg-num (fourth movec)) (fifth movec)) ) 8))
	      (if abs-case (ext-command (fourth movec))))))))))
	
(defun copy-control-register (movec)
  (let ((src (copy-source movec))
	(dest (copy-destination movec)))
    (if (eq src 'SSH) 
	(if (eq dest 'SSH)
	    (error "illegal instruction: ~S" movec)
	  (setf current-class PIPE-MOVE-from-SSH))
      (if (eq src 'SSL) 
	  (setf current-class PIPE-MOVE-from-SSL)
	(if (eq dest 'SP)
	    (setf current-class PIPE-MOVE-SP)
	  (if (or (eq dest 'SR) (eq dest 'SSH) (eq dest 'SSL))
	      (setf current-class PIPE-before-RTS)
	    (if (or (eq dest 'LA) (eq dest 'LC))
		(setf current-class PIPE-ENDDO))))))
    (if (movec-register src)		;S1,D2 case
	(progn
	  (if (address-register dest) (setf current-register dest))
	  (dsp-command #x44080		;#x440A0 but the high order bit is in encode-register
		       (encode-register src)
		       (ash (encode-register dest) 8)))
      (if (movec-register dest)
	  (dsp-command #x4C080
		       (encode-register dest)
		       (ash (encode-register src) 8))
	(error "invalid registers: ~S" movec)))))



;;; MOVEM --------------------------------------------------------------------------------
;;; P ea SHORT to D or S to P ea SHORT, S|D can be any register
;;; (LOAD s P ea SHORT) (STORE d P ea SHORT)
;;; because this is P memory, labels are legal as addresses

(defun P-memory-load-or-store (movem)
  (if (not (eq 'P (third movem)))
      (error "P-memory-move called on ~S memory?" (third movem))
    (let* ((D-or-S (encode-register (second movem)))
	   (short (eq (car (last movem)) 'SHORT))
	   (abs-case (and (not short)	;might be R R-1 | 2345 | P-mem-label
			  (or (integerp (fourth movem))
			      (not (R-register (fourth movem)))))))
      (if (and (control-register (second movem)) ;pipeline checks
	       (eq (car movem) 'LOAD))
	  (if (eq (second movem) 'SP)
	      (setf current-class PIPE-MOVE-SP)
	    (if (or (eq (second movem) 'SR) (eq (second movem) 'SSH) (eq (second movem) 'SSL))
		(setf current-class PIPE-before-RTS)
	      (if (or (eq (second movem) 'LA) (eq (second movem) 'LC))
		  (setf current-class PIPE-ENDDO)))))
      (if (address-register (second movem))
	  (if (eq (car movem) 'LOAD)
	      (setf current-register (second movem))
	    (if (eq (car movem) 'STORE)
		(setf current-source-register (second movem)))))
      (if (and short (not (immediate-short-address (fourth movem))))
	  (error "invalid address for short P memory ~S: ~S" (car movem) movem))
      (dsp-command (if short #x070000 #x074080)
		   (if (eq (car movem) 'STORE) 0 #x8000)
		   D-or-S
		   (ash (if short (fourth movem)
			  (if abs-case absolute-address-mode
			    (eff-addr-mode (reg-num (fourth movem)) (fifth movem)))) 8))
      (if abs-case (ext-command (if (integerp (fourth movem))
				    (fourth movem)
				  (get-label-value (fourth movem) 0)))))))
					;0 for offset because we already sent the first MOVE command


;;; MOVEP --------------------------------------------------------------------------------
;;; this one is special because it includes memory to memory moves, so names are different
;;; X-IO is X side IO memory, Y-IO is y-side.  Luckily, X the register unadorned is not legal.	       
;;; There are three basic classes: X or Y memory reference, P memory reference, and register reference
;;; (LOAD|STORE D X-IO|Y-IO pp) (STORE num X-IO|Y-IO pp) (MOVE P|X|Y|X-IO|Y-IO  pp|ea  P|X|Y|X-IO|Y-IO  pp|ea)
;;; In P mem refs, a label is a legal address.

(defun X-IO-or-Y-IO-load-or-store (movep)
  (let ((store (eq (car movep) 'STORE))
	(pp (IO-address (fourth movep)))
	(x-mem (eq (third movep) 'X-IO)))
    (if (integerp (second movep))
	(if store 
	    (if (and (normal-data (second movep))
		     (immediate-short-address pp))
		(progn
		  (dsp-command #x084080
			       (ash immediate-data-mode 8)
			       (if x-mem 0 #x010000)
			       #x8000	;write peripheral
			       pp)
		  (ext-command (second movep)))
	      (error "invalid operands for ~S to ~S ~S:" (car movep) (third movep) (fourth movep)))
	  (error "Attempt to load immediate with redundant peripheral reference: ~S" movep))
      ;; here we have a register to load/store from IO memory
      (progn				;first do pipeline checks
	(let ((dest (cadr movep)))
	  (if (eq dest 'SP) (setf current-class PIPE-MOVE-SP)
	    (if (or (eq dest 'SR) (eq dest 'SSH) (eq dest 'SSL)) 
		(setf current-class PIPE-before-RTS)
	      (if (or (eq dest 'LA) (eq dest 'LC))
		  (setf current-class PIPE-ENDDO))))
	  (if (address-register (second movep)) 
	      (if store 
		  (setf current-source-register (second movep))
		(setf current-register (second movep))))
	  (dsp-command #x084000
		       pp
		       (if store #x8000 0)
		       (if x-mem 0 #x10000)
		       (ash (encode-register (second movep)) 8)))))))
				
(defun X-IO-or-Y-IO-move (movep)
  (let* ((source-mem (second movep))
	 (source-addr (third movep))
	 (source-abs (not (R-register source-addr)))
	 (source-reg (if source-abs nil (third movep)))
	 (source-mode (if source-abs 
			  absolute-address-mode 
			(eff-addr-mode (reg-num source-reg) (fourth movep))))
	 (dest-mem (if source-abs (fourth movep) (fifth movep)))
	 (dest-addr (if source-abs (fifth movep) (sixth movep)))
	 (dest-abs (not (R-register dest-addr)))
	 (dest-reg (if dest-abs nil dest-addr))
	 (dest-mode (if dest-abs 
			absolute-address-mode 
		      (eff-addr-mode (reg-num dest-reg) 
				     (if source-abs (sixth movep) (seventh movep))))))
    (if (eq source-mem 'P)		;P,X:pp  P,Y:pp (A-164)
	(progn
	  (dsp-command #x084040
		       (IO-address dest-addr)
		       (ash source-mode 8)
		       #x8000		;write peripheral
		       (if (eq dest-mem 'X-IO) 0 #x10000))
	  (if source-abs (ext-command (if (integerp source-addr)
					  source-addr
					(get-label-value source-addr 0)))))
      (if (eq dest-mem 'P)		;X:pp,P  Y:pp,P (labels ok here)
	  (progn
	    (dsp-command #x084040
			 (IO-address source-addr)
			 (ash dest-mode 8)
			 (if (eq source-mem 'X-IO) 0 #x10000))
	    (if dest-abs (ext-command (if (integerp dest-addr)
					  dest-addr
					(get-label-value dest-addr 0)))))
	(if (or (eq source-mem 'X-IO) (eq source-mem 'Y-IO))
	    (progn			;X:pp,X:ea|Y:ea (to P handled above, to D handled in LOAD-STORE)
	      (dsp-command #x084080
			   (if (eq source-mem 'X-IO) 0 #x10000)
			   (IO-address source-addr)
			   (if (eq dest-mem 'X) 0 #x40)
			   (ash dest-mode 8))
	      (if dest-abs (ext-command dest-addr)))
	  (progn			;X:ea,X:pp (immediate handled as STORE)
	    (dsp-command #x084080
			 (if (eq dest-mem 'X-IO) 0 #x10000)
			 (IO-address dest-addr)
			 (if (eq source-mem 'X) 0 #x40)
			 (ash source-mode 8)
			 #x8000)	;write peripheral
	    (if source-abs (ext-command source-addr))))))))

;;; End of move possibilities.


(defun too-big (x &optional (siz 24))
  (/= 0 (logand x (ash -1 siz))))


(defun ABXorY (op src dest)
  (if (eq dest 'A)
     (case src 
       (B  0)
       (X0 #b01000000)
       (X1 #b01100000)
       (Y0 #b01010000)
       (Y1 #b01110000)
       (t (error "invalid operands for ~S: ~S ~S" op src dest)))
   (if (eq dest 'B)
       (case src 
	 (A  #b00001000)
	 (X0 #b01001000)
	 (X1 #b01101000)
	 (Y0 #b01011000)
	 (Y1 #b01111000)
	 (t (error "invalid operands for ~S: ~S ~S" op src dest)))
     (error "invalid operands for ~S: ~S ~S" op src dest))))

(defun XYtoAB (op src dest)
  (if (eq dest 'A)
      (case src 
	(X0 0)
	(X1 #b100000)
	(Y0 #b010000)
	(Y1 #b110000)
	(t (error "invalid operands for ~S: ~S to ~S" op src dest)))
    (if (eq dest 'B)
	(case src 
	  (X0 #b001000)
	  (X1 #b101000)
	  (Y0 #b011000)
	  (Y1 #b111000)
	  (t (error "invalid operands for ~S: ~S to ~S" op src dest)))
      (error "invalid source for ~S: ~S" op src))))

(defun ABXYtoAB (op src dest)
  (if (eq dest 'A)
      (case src
	(B 0) (X0 #b1000) (X1 #b1100) (Y0 #b1010) (Y1 #b1110)
	(t (error "invalid source for ~S: ~S" op src)))
    (if (eq dest 'B)
	(case src
	  (A #b0001) (X0 #b1001) (X1 #b1101) (Y0 #b1011) (Y1 #b1111)
	  (t (error "invalid source for ~S: ~S" op src)))
      (error "invalid destination for ~S: ~S" op dest))))

(defun encode-condition (x)
  (case x 
    ((JCC JSCC TCC JHS JSHS THS) 0) 
    ((JGE JSGE TGE) 1) ((JNE JSNE TNE) 2) ((JPL JSPL TPL) 3) ((JNN JSNN TNN) 4) 
    ((JEC JSEC TEC) 5) ((JLC JSLC TLC) 6) ((JGT JSGT TGT) 7) 
    ((JCS JSCS TCS JLO JSLO TLO) 8) 
    ((JLT JSLT TLT) 9) ((JEQ JSEQ TEQ) 10) ((JMI JSMI TMI) 11) ((JNR JSNR TNR) 12) 
    ((JES JSES TES) 13) ((JLS JSLS TLS) 14) ((JLE LSLE TLE) 15)
    (t (error "invalid condition: ~S" x))))


(defun l-reg (x)
  (case x 
    (A10 0) (B10 1) (X 2) (Y 3) (A 4) (B 5) (AB 6) (BA 7)
    (t (error "invalid L move register: ~S" x))))


(defun xy-reg-member (x)
  (member x '(X0 X1 Y0 Y1 A0 B0 A2 B2 A1 B1 A B X Y AB BA A10 B10)))

(defconstant DSP56-move-read-bit #x8000)

(defun both-members (x y l)
  (and (member x l) (member y l)))

(defun check-duplicate-destinations (main-dest move-dest)
  (if (or 
       (eq main-dest move-dest)		;most obvious case!
       (and				;are both A B X Y members
	(atom main-dest) 
	(atom move-dest)
	(or (both-members main-dest move-dest '(A A0 A1 A2 AB BA A10))
	    (both-members main-dest move-dest '(B B0 B1 B2 AB BA B10))
	    (both-members main-dest move-dest '(X X0 X1))
	    (both-members main-dest move-dest '(Y Y0 Y1)))))
      (error "duplicate destinations: ~S ~S" main-dest move-dest)))

(defun can-move (op d &optional (dest nil) (move-1 nil) (move-2 nil))
					;here we have possibly 0, one or two parallel moves packed with OP
					;DEST needed to check for accidental illegal duplicate destinations
					;OP can be 0 if we're doing a MOVE no-op (i.e. (LOAD A X 34) or something).
  (if (or (and move-1 (not (listp move-1)))
	  (and move-2 (not (listp move-2))))
      (error "invalid arguments to move decoder: ~S ~S" move-1 move-2))
  (let ((o (logior op d)))
    (if (not (or move-1 move-2))	;no moves -- just send op with explicit no-move code
	(no-parallel-data o)
      (if (not (and move-1 move-2))	;i.e. we got one move, not two
	  (let* ((mov (if move-1 move-1 move-2))
		 (op (car mov))
		 (mov-D (if (or (eq op 'LOAD) (eq op 'UPDATE)) (cadr mov) 
			  (if (eq op 'COPY) (caddr mov) 'L))))
	    (check-duplicate-destinations dest mov-D)
	    ;; in the one move case we can have COPY, UPDATE, LOAD, STORE
	    (case op
	      (COPY (register-to-register-copy o (cadr mov) (caddr mov)))
	      (UPDATE (address-register-update o (cadr mov) (caddr mov)))
	      ((LOAD STORE)
	       (if (and (eq op 'LOAD) (integerp (caddr mov)))
		   (if (eq (fourth mov) 'SHORT)
		       (immediate-short-load o (cadr mov) (caddr mov))
		     (XY-load-or-store o mov))
		 (if (eq (caddr mov) 'L) ;long memory reference
		     (long-memory-load-or-store o mov)
		   (XY-load-or-store o mov))))))
	;; here we have two parallel moves -- cannot be UPDATE, but others ok
	(cond ((and (eq (car move-1) 'COPY) (eq (car move-2) 'COPY))
	       (error "cannot do two COPY's in parallel: ~S ~S" move-1 move-2))
	      ((or (eq (caddr move-1) 'L) (eq (caddr move-2) 'L))
	       (error "cannot move L memory in parallel with some other move: ~S ~S" move-1 move-2))
	      ((eq (car move-2) 'COPY)
	       (check-duplicate-destinations dest (caddr move-2))
	       (if (eq (car move-1) 'LOAD) 
		   (check-duplicate-destinations dest (cadr move-1)))
	       (if (eq (caddr move-1) 'X)
		   (X-or-Y-and-R-load-or-store o move-1 move-2)
		 (X-or-Y-and-R-load-or-store o move-2 move-1)))
	      ((eq (car move-1) 'COPY)
	       (check-duplicate-destinations dest (caddr move-1))
	       (if (eq (car move-2) 'LOAD)
		   (check-duplicate-destinations dest (cadr move-2)))
	       (if (eq (caddr move-2) 'X)
		   (X-or-Y-and-R-load-or-store o move-2 move-1)
		 (X-or-Y-and-R-load-or-store o move-1 move-2)))
	      (t 
	       (if (eq (car move-1) 'LOAD) (check-duplicate-destinations dest (cadr move-1)))
	       (if (eq (car move-2) 'LOAD) (check-duplicate-destinations dest (cadr move-2)))
	       (if (and (eq (car move-1) 'LOAD) (eq (car move-2) 'LOAD))
		   (check-duplicate-destinations (cadr move-1) (cadr move-2)))
	       (if (and (eq (third move-1) 'X) (eq (third move-2) 'Y))
		     (X-and-Y-load-or-store o move-1 move-2)
		   (if (and (eq (third move-2) 'X) (eq (third move-1) 'Y))
		       (X-and-Y-load-or-store o move-2 move-1)
		     (error "cannot accomodate this parallel move request: ~S ~S" move-1 move-2)))))))))
		   
;;; op-codes
;;; the following set only the constant 1 bits in the command.

(defconstant DSP56-ABS     #x26     "abs -- absolute value"                   )
(defconstant DSP56-ADC     #x21     "adc -- add long with carry"              )
(defconstant DSP56-ADD     0        "add"                                     )
(defconstant DSP56-ADDL    #x12     "addl -- shift left and add"              )
(defconstant DSP56-ADDR    #x02     "addr -- shift right and add"             )
(defconstant DSP56-AND     #x46     "and -- logical and"                      )
(defconstant DSP56-ANDI    #xB8     "andi -- and immediate with CCR"          )
(defconstant DSP56-ASL     #x32     "asl -- arithmetic shift left"            )
(defconstant DSP56-ASR     #x22     "asr -- arithmetic shift right"           )
(defconstant DSP56-BCHG    #x0B0000 "bchg -- bit test and change"             )
(defconstant DSP56-BCLR    #x0A0000 "bclr -- bit test and clear"              )
(defconstant DSP56-BSET    #x0A0020 "bset -- bit test and set"                )
(defconstant DSP56-BTST    #x0B0020 "btst -- bit test on memory"              )
(defconstant DSP56-CLR     #x13     "clr -- clear accumulator"                )
(defconstant DSP56-CMP     #x05     "cmp -- compare"                          )
(defconstant DSP56-CMPM    #x07     "cmpm -- compare magnitude"               )
(defconstant DSP56-DIV     #x018040 "div -- divide iteration"                 )
(defconstant DSP56-DO      #x060000 "do -- start hardware loop"               )
(defconstant DSP56-ENDDO   #x8C     "enddo -- end current do loop"            )
(defconstant DSP56-EOR     #x43     "eor -- logical exclusive or"             )
(defconstant DSP56-ILLEGAL #x05     "illegal -- illegal instruction!"         )
(defconstant DSP56-Jcc     #x0A0000 "jcc -- jump conditionally"               )
(defconstant DSP56-JCLR    #x0A0000 "jclr -- jump if bit clear"               )
(defconstant DSP56-JMP     #x080000 "jmp -- jump"                             )
(defconstant DSP56-JScc    #x0B0000 "jscc -- jump to subroutine conditionally")
(defconstant DSP56-JSCLR   #x0B0000 "jsclr -- jump to subroutine if bit clear")
(defconstant DSP56-JSET    #x0A0020 "jset -- jump if bit set"                 )
(defconstant DSP56-JSR     #x090000 "jsr -- jump to subroutine"               )
(defconstant DSP56-JSSET   #x0A0020 "jsset -- jump to subroutine if bit set"  )
(defconstant DSP56-LSL     #x33     "logical shift left"                      )
(defconstant DSP56-LSR     #x23     "logical shift right"                     )
(defconstant DSP56-LUA     #x044010 "load updated address"                    )
(defconstant DSP56-MAC     #x82     "signed multiply-accumulate"              )
(defconstant DSP56-MACR    #x83     "signed multiply-accumulate and round"    )
(defconstant DSP56-MOVE    0        "move data"                               )
(defconstant DSP56-MOVEC   #x040020 "movec -- move control register"          )
(defconstant DSP56-MOVEM   #x070000 "movem -- move program memory"            )
(defconstant DSP56-MOVEP   #x084000 "movep -- move peripheral data"           )
(defconstant DSP56-MPY     #x80     "mpy -- signed multiply"                  )
(defconstant DSP56-MPYR    #x81     "mpyr -- signed multiply and round"       )
(defconstant DSP56-NEG     #x36     "neg -- negate accumulator"               )
(defconstant DSP56-NOP     0        "nop -- no operation"                     )
(defconstant DSP56-NORM    #x01D815 "norm -- normalize accumulator iteration" )
(defconstant DSP56-NOT     #x17     "not -- logical complement"               )
(defconstant DSP56-OR      #x42     "or -- logical inclusive or"              )
(defconstant DSP56-ORI     #xF8     "ori -- OR immediate with CCR"            )
(defconstant DSP56-REP     #x060020 "rep -- repeat next instruction"          )
(defconstant DSP56-RESET   #x84     "reset -- reset peripheral devices"       )
(defconstant DSP56-RND     #x11     "rnd -- round accumulator"                )
(defconstant DSP56-ROL     #x37     "rol -- rotate left"                      )
(defconstant DSP56-ROR     #x27     "ror -- rotate right"                     )
(defconstant DSP56-RTI     #x04     "rti -- return from interrupt"            )
(defconstant DSP56-RTS     #x0C     "rts -- return from subroutine"           )
(defconstant DSP56-SBC     #x25     "sbc -- subtract long with carry"         )
(defconstant DSP56-STOP    #x87     "stop -- stop processing"                 )
(defconstant DSP56-SUB     #x04     "sub -- subtract"                         )
(defconstant DSP56-SUBL    #x16     "subl -- shift left and subtract"         )
(defconstant DSP56-SUBR    #x06     "subr -- shift right and subtract"        )
(defconstant DSP56-SWI     #x06     "swi -- software interrupt"               )
(defconstant DSP56-Tcc     #x020000 "tcc -- transfer conditionally"           )
(defconstant DSP56-TFR     #x01     "tfr -- transfer data ALU register"       )
(defconstant DSP56-TST     #x03     "tst -- test accumulator"                 )
(defconstant DSP56-WAIT    #x86     "wait -- wait for interrupt"              )

(defun no-ac-case (op op-code src)
  (if src (error "~S takes no arguments: ~S" op src))
  (dsp-command op-code))

(defun single-ac-case (op op-code src dest x-move y-move)
  (if dest (error "too many arguments for ~S: ~S ~S" op src dest))
  (if (eq src 'A)   (can-move op-code 0 src x-move y-move) 
    (if (eq src 'B) (can-move op-code #b1000 src x-move y-move)
      (error "invalid operand for ~S: ~S" op src))))

(defun double-ac-case (op op-code src dest x-move y-move)
  (if   (and (eq src 'B) (eq dest 'A)) (can-move op-code 0 dest x-move y-move)
    (if (and (eq src 'A) (eq dest 'B)) (can-move op-code #b1000 dest x-move y-move)
	(error "invalid operands for ~S: ~S to ~S" op src dest))))

(defun check-for-defined-names (instruction)
  (if (and instruction
	   (not (member (car instruction) '(DEFINE UNDEFINE))))
      (loop for i below (length instruction) do
	(multiple-value-bind (value exists) (gethash (nth i instruction) names)
	  (if exists 
	      (setf (nth i instruction) value))))))

(defun check-for-fractions (op)
  (loop for k below (length op) do	;need this (rather than loop for i in op) because of setf below
    (let ((i (nth k op)))
      (if (numberp i)
	  (if (and (integerp i)		;handle negative integers (by tossing high byte)
		   (minusp i))
	      (if (> (abs i) #x7fffff)
		  (error "number too large: ~F" i)
		(setf (nth k op) (logand i #xFFFFFF)))
	    (if (not (integerp i))	;handle fractions (positive or negative)
		(if (> (abs i) 1.0)
		    (if (> (abs i) #x7fffff)
			(error "number too large: ~F" i)
		      (if (/= (floor i) i)
			  (error "cannot handle ~F in this way" i)
			(setf (nth k op) (floor i))))
		        ;; this is ambiguous -- 1.0 = 1, not #x7fffff
		  (setf (nth k op) (if (= i 1.0) 
				       #x7FFFFF 
				     (logand (floor (scale-float (float i) 23)) #xFFFFFF))))))))))

;;; next stuff helps in pipeline error checks
(defvar loop-flag 0)			;LF simulator
(defvar end-of-loop nil)
(defvar loop-address nil)		;LA simulator
(defvar pipe-1 -1)			;previous pipeline instruction class
(defvar pipe-2 -1)			;ditto but two back
(defvar pipe-3 -1)			;ditto but 3 back (need that much at end of DO loops)
(defvar last-reg nil)  
(defvar after-Do-reg nil)

;;; the following checks are made:
;;;   1. if an address register was referenced, make sure it wasn't explicitly set in the previous instruction
;;;   2. At LA (end Do loop) check last 3 for illegal instructions
;;;   3. disallow DO SSH,xxxx
;;;   4. disallow JSR,LA and friends when loop-flag t
;;;   5. at DO, ENDDO, RTI, RTS, check previous in various ways
;;;   6. check for SSH and SP troubles
;;;   7. check for previous REP
;;; for now we'll ignore the "fast interrupt routine" restrictions

(defconstant at-loop-end-class (list PIPE-BC-SP PIPE-MOVE-from-SSH PIPE-J-SSH PIPE-LA-3 PIPE-DO
				 PIPE-before-RTS PIPE-before-RTI PIPE-ENDDO))
(defconstant at-do-class (list PIPE-BC-SP PIPE-MOVE-from-SSH PIPE-MOVE-SP PIPE-before-RTS PIPE-ENDDO))
(defconstant at-enddo-class (list PIPE-BC-SP PIPE-MOVE-from-SSH  PIPE-MOVE-SP PIPE-before-RTS PIPE-before-RTI PIPE-ENDDO))
(defconstant at-rti-class (list PIPE-BC-SP PIPE-MOVE-from-SSH PIPE-MOVE-SP PIPE-before-RTS PIPE-before-RTI PIPE-ANDI-CCR))
(defconstant at-rts-class (list PIPE-BC-SP PIPE-MOVE-from-SSH PIPE-MOVE-SP PIPE-before-RTS))
(defconstant after-rep-class (list PIPE-DO PIPE-J-SSL PIPE-J-SSH PIPE-after-REP PIPE-REP PIPE-at-LA))

(defun in-trouble (n l)
  (member n l))

(defun tick-pipe ()
  (when (/= dsp-pc last-dsp-pc)
    (setf pipe-3 pipe-2)		;now run the pipe one instruction
    (setf pipe-2 pipe-1)
    (setf pipe-1 current-class)
    (setf last-reg current-register)
    (setf last-dsp-pc dsp-pc)))

(defvar correct-pipeline-silently t)	;default is to insert NOP's for R N and M pipeline troubles

(defun insert-nop ()			;handle simplest cases automatically, if requested by caller
  (let ((old-op-1 (get-dsp-program last-dsp-pc))
	(old-op-2 (get-dsp-program (+ last-dsp-pc 1))))
    (DEBUGGING (let ((old-op (pop emit-prog)))
		 (push (append (list (first old-op)) '((nop))) emit-prog)
		 (setf (first old-op) dsp-pc)
		 (push old-op emit-prog)))
    (fixup-moved-labels last-dsp-pc)
    (edit-dsp-program last-dsp-pc DSP56-NOP)
    (edit-dsp-program (+ last-dsp-pc 1) old-op-1)
    (if (= dsp-pc (+ last-dsp-pc 2)) (edit-dsp-program dsp-pc old-op-2))
    (setf last-dsp-pc dsp-pc)
    (incf dsp-pc)))

(defun check-for-pipeline-trouble (op arg1)
  (if (and last-reg			;R N M registers after explicit move require a pass to settle
					;if used as a pointer
	   current-source-register
	   (or (eq last-reg current-source-register)
	       (and (= (reg-num last-reg) (reg-num current-source-register))
		    (uses-address-modifiers current-addr-mode))))
      (if correct-pipeline-silently
	  (insert-nop)
	(error "address register ~S hasn't had time to settle: ~A" last-reg arg1)))
  (if end-of-loop
      (progn
	(setf end-of-loop nil)
	(decf loop-flag)
	(if (and current-register (eq (pop after-do-reg) current-register))
	    (error "address register ~S won't have time to settle upon DO iteration: ~A" current-register arg1))
	(if (= current-class PIPE-at-LA)
	    (error "illegal instruction at loop end: ~A in ~A" op arg1))
	(if (or (in-trouble pipe-1 at-loop-end-class)
		(in-trouble pipe-2 at-loop-end-class)
		(in-trouble pipe-3 at-loop-end-class))
	    (error "illegal instruction sequence within last three instructions before end of loop (found at ~A)" arg1))
	(if (or (in-trouble pipe-1 (list PIPE-at-LA PIPE-after-REP PIPE-REP PIPE-J-LA))
		two-word-instruction)
	    (error "illegal instruction sequence just before end of loop (near ~A)" arg1))))
  (case op
    (DO (if (in-trouble pipe-1 at-do-class)
	    (error "illegal instruction just before DO: ~A" arg1)))
    (ENDDO (if (in-trouble pipe-1 at-enddo-class)
	       (error "illegal instruction just before ENDDO: ~A" arg1)))
    (RTI (if (in-trouble pipe-1 at-rti-class)
	     (error "illegal instruction just before RTI: ~A" arg1)))
    (RTS (if (in-trouble pipe-1 at-rts-class)
	     (error "illegal instruction just before RTS: ~A" arg1))))
  (if (and (= pipe-1 PIPE-REP)
	   (in-trouble current-class after-rep-class))
      (error "illegal instruction after REP"))
  (if (and (or (= current-class PIPE-MOVE-from-SSH)
	       (= current-class PIPE-MOVE-from-SSL))
	   (or (= pipe-1 PIPE-BC-SP)
	       (= pipe-1 PIPE-MOVE-SP)))
      (error "illegal instruction sequence involving MOVEn SSn and BCnn SP: ~A" arg1))
  (if (and (or (= current-class PIPE-J-SSL) (= current-class PIPE-J-SSH))
	   (or (= pipe-1 PIPE-BC-SP) (= pipe-1 PIPE-MOVE-SP)))
      (error "illegal instruction sequence involving Jnnn SSH or SSL after BCn SP or MOVEn SP: ~A" arg1))

  (if (= pipe-1 PIPE-DO)
      (push current-register after-do-reg))
  (tick-pipe)
  
  (setf two-word-instruction nil)
  (setf current-class -1)		;the "all's well" pipeline state
  (setf current-register nil)
  (setf current-source-register nil)
  (setf current-addr-mode nil))

(defun emit (arg1 &optional arg2 arg3)
  (let ((op nil) (src nil) (dest nil) (pseudo-time nil)
	(instruction (copy-list arg1)) 
	(x-move (copy-list arg2)) 
	(y-move (copy-list arg3)))

    (DEBUGGING (push (append (list dsp-pc) 
			     (if arg3 (list arg1 arg2 arg3) 
			       (if arg2 (list arg1 arg2) (list arg1))))
		     emit-prog))

    (if we-may-have-names
	(progn
	  (check-for-defined-names instruction)
	  (if x-move (check-for-defined-names x-move))
	  (if y-move (check-for-defined-names y-move))))
    (if (eq (car instruction) 'LOAD) (check-for-fractions instruction))
    (if (and x-move (eq (car x-move) 'LOAD)) (check-for-fractions x-move))
    (if (and y-move (eq (car y-move) 'LOAD)) (check-for-fractions y-move))

    (if (and (not (eq pipe-1 PIPE-REP)) 
	     (not end-of-loop))
	(check-for-inserted-jump-into-external-memory))
    ;; can't insert JMP at LA and REP JMP would be a disaster.

;    (setf two-word-instruction nil)	;pipeline error check
;    (setf current-class -1)		;the "all's well" pipeline state
;    (setf current-register nil)
;    (setf current-source-register nil)
;    (setf current-addr-mode nil)

    (setf op (car instruction))
    (setf src (cadr instruction))
    (setf dest (caddr instruction))

    (case op
    
      ;; easy cases -- first those that take no arguments, and cannot involve parallel moves
      (ENDDO				;(ENDDO)
       (no-ac-case 'ENDDO DSP56-ENDDO src))
      (ILLEGAL				;(ILLEGAL)
       (no-ac-case 'ILLEGAL DSP56-ILLEGAL src))
      (NOP				;(NOP)
       (no-ac-case 'NOP DSP56-NOP src))
      (RESET				;(RESET)
       (setf current-class PIPE-at-LA)
       (no-ac-case 'RESET DSP56-RESET src))
      (RTI				;(RTI)
       (setf current-class PIPE-at-LA)
       (no-ac-case 'RTI DSP56-RTI src))
      (RTS				;(RTS)
       (setf current-class PIPE-at-LA)
       (no-ac-case 'RTS DSP56-RTS src))
      (STOP				;(STOP)
       (setf current-class PIPE-at-LA)
       (no-ac-case 'STOP DSP56-STOP src))
      (SWI				;(SWI)
       (setf current-class PIPE-after-REP)
       (no-ac-case 'SWI DSP56-SWI src))
      (WAIT				;(WAIT)
       (setf current-class PIPE-at-LA)
       (no-ac-case 'WAIT DSP56-WAIT src))
      
      ;; more easy cases -- those that take a single argument, allow parallel moves, and
      ;;     don't involve special case error checks
      (ABS				;((ABS ac) (move) (move))
       (single-ac-case 'ABS DSP56-ABS src dest x-move y-move))
      (ASL				;((ASL ac) (move) (move))
       (single-ac-case 'ASL DSP56-ASL src dest x-move y-move))
      (ASR				;((ASR ac) (move) (move))
       (single-ac-case 'ASR DSP56-ASR src dest x-move y-move))
      (CLR				;((CLR ac) (move) (move))
       (single-ac-case 'CLR DSP56-CLR src dest x-move y-move))
      (LSL				;((LSL ac) (move) (move))
       (single-ac-case 'LSL DSP56-LSL src dest x-move y-move))
      (LSR				;((LSR ac) (move) (move))
       (single-ac-case 'LSR DSP56-LSR src dest x-move y-move))
      (NEG				;((NEG ac) (move) (move))
       (single-ac-case 'NEG DSP56-NEG src dest x-move y-move))
      (NOT				;((NOT ac) (move) (move))
       (single-ac-case 'NOT DSP56-NOT src dest x-move y-move))
      (RND				;((RND ac) (move) (move))
       (single-ac-case 'RND DSP56-RND src dest x-move y-move))
      (ROL				;((ROL ac) (move) (move))
       (single-ac-case 'ROL DSP56-ROL src dest x-move y-move))
      (ROR				;((ROR ac) (move) (move))
       (single-ac-case 'ROR DSP56-ROR src dest x-move y-move))
      (TST				;((TST ac) (move) (move))
       (single-ac-case 'TST DSP56-TST src dest x-move y-move))
    
      ;; still more easy cases -- these take two ac args, allow moves, involve normal error checks
      (ADDL				;((ADDL ac ac) (move) (move))
       (double-ac-case 'ADDL DSP56-ADDL src dest x-move y-move))
      (ADDR				;((ADDR ac ac) (move) (move))
       (double-ac-case 'ADDR DSP56-ADDR src dest x-move y-move))
      (SUBL				;((SUBL ac ac) (move) (move))
       (double-ac-case 'SUBL DSP56-SUBL src dest x-move y-move))
      (SUBR				;((SUBR ac ac) (move) (move))
       (double-ac-case 'SUBR DSP56-SUBR src dest x-move y-move))

      ;; some un-complicated cases 
      (CMP				;((CMP ra ra) (move) (move))
       (can-move DSP56-CMP (ABXorY op src dest) dest x-move y-move))
      (CMPM				;((CMPM ra ra) (move) (move))
       (can-move DSP56-CMPM (ABXorY op src dest) dest x-move y-move))
      (DIV				;(DIV S D)
       (if x-move (error "DIV cannot accomodate parallel moves: ~S ~S" x-move y-move))
       (dsp-command DSP56-DIV (XYtoAB op src dest)))
      (EOR				;((EOR ra ra) (move) (move))
       (can-move DSP56-EOR (XYtoAB op src dest) dest x-move y-move))
      (OR				;((OR xy ac) (move) (move))
       (can-move DSP56-OR (XYtoAB op src dest) dest x-move y-move))
      ((TFR TRANSFER)			;((TFR S D) (move) (move))
       (can-move DSP56-TFR (ash (ABXYtoAB op src dest) 3) dest x-move y-move))
 
      ;; the rest involve some special handling
      ((ADC SBC)			;((ADC rg ac) (move) (move))
       (can-move (if (eq op 'ADC) DSP56-ADC DSP56-SBC)
		 (if (eq src 'X)	
		     (if (eq dest 'B) #b1000
		       (if (eq dest 'A) 0
			 (error "invalid operands for ~S: ~S to ~S" op src dest)))
		   (if (eq src 'Y)
		       (if (eq dest 'A) #b10000
			 (if (eq dest 'B) #b11000
			   (error "invalid operands for ~S: ~S to ~S" op src dest)))
		     (error "invalid operand for ~S: ~S" op src)))
		 dest x-move y-move))
      
      ((ADD SUB)			;((ADD ac ac) (move) (move))
       (can-move (if (eq op 'ADD) DSP56-ADD DSP56-SUB)
		 (cond ((and (eq src 'B) (eq dest 'A))  #b0010000)
		       ((and (eq src 'A) (eq dest 'B))  #b0011000)
		       ((and (eq src 'X) (eq dest 'A))  #b0100000)
		       ((and (eq src 'X) (eq dest 'B))  #b0101000)
		       ((and (eq src 'Y) (eq dest 'A))  #b0110000)
		       ((and (eq src 'Y) (eq dest 'B))  #b0111000)
		       ((and (eq src 'X0) (eq dest 'A)) #b1000000)
		       ((and (eq src 'X0) (eq dest 'B)) #b1001000)
		       ((and (eq src 'Y0) (eq dest 'A)) #b1010000)
		       ((and (eq src 'Y0) (eq dest 'B)) #b1011000)
		       ((and (eq src 'X1) (eq dest 'A)) #b1100000)
		       ((and (eq src 'X1) (eq dest 'B)) #b1101000)
		       ((and (eq src 'Y1) (eq dest 'A)) #b1110000)
		       ((and (eq src 'Y1) (eq dest 'B)) #b1111000)
		       (t (error "illegal operandS for ~S: ~S to ~S" op src dest)))
		 dest x-move y-move))
      
      (AND				;((AND rg ac) (move) (move))
       (if (or (eq dest 'MR) (eq dest 'CCR) (eq dest 'OMR))
	   (error "we don't support AND as alias for ANDI: ~S ~S ~S" op src dest))
       (can-move DSP56-AND (XYtoAB op src dest) dest x-move y-move))

      ((ANDI ORI)			;(ANDI n crg)
       (if (too-big src 8) (error "~S argument is too big: ~D" op src))
       (if x-move (error "too many arguments for ~S: ~S ~S ~S ~S" op src dest x-move y-move))
       (if (eq dest 'CCR) (setf current-class PIPE-ANDI-CCR)
	 (if (eq dest 'MR) (setf current-class PIPE-before-RTI)))
       (dsp-command (if (eq op 'ANDI) DSP56-ANDI DSP56-ORI)
		    (case dest (MR 0) (CCR 1) (OMR 2)
			  (t (error "invalid destination for ~S: ~S" op dest)))
		    (ash src 8)))
 
      ((BCHG BCLR BTST BSET)		;(BCHG n {D or {X|Y|X-IO|Y-IO addr [SHORT]}})
       (if (> src 23) (error "invalid bit number for ~S: ~S" op src))
       (let* ((b-op (case op (BCHG DSP56-BCHG) (BTST DSP56-BTST) (BSET DSP56-BSET) (BCLR DSP56-BCLR)))
	      (addr (fourth instruction))
	      (x-or-y (member dest '(X Y X-IO Y-IO)))
	      (short (if (r-register addr) (sixth instruction) (fifth instruction))))
	 (if (not x-or-y)		;i.e. BCLR n D (which appears to be broken ok our 56000's)
	     (progn			;first pipeline error checks
	       (if (control-register dest)
		   (if (eq op 'BTST) 
		       (if (eq dest 'SSH) (setf current-class PIPE-LA-3))
		     (cond ((eq dest 'SP) (setf current-class PIPE-BC-SP))
			   ((or (eq dest 'SSH) (eq dest 'SSL)) (setf current-class PIPE-before-RTS))
			   ((eq dest 'SR) (setf current-class PIPE-before-RTI))
			   ((or (eq dest 'LA) (eq dest 'LC)) (setf current-class PIPE-ENDDO)))))
	       (if (R-register dest) (setf current-register dest))
	       (print (format t "warning: this instruction does not work on CCRMA's NeXT's: ~S ~S ~S" op src dest))
	       (dsp-command b-op src	;BCHG #n,D (or whatever)
			    #xC040
			    (ash (encode-register dest) 8)))
	   (let ((mem (if (or (eq 'X dest) (eq 'X-IO dest)) 0 #x40))
		 (io (or (eq 'X-IO dest) (eq 'Y-IO dest))))
	     (if io
		 (dsp-command b-op src mem
			      #x8000
			      (ash (IO-address addr) 8))
	       (if short
		   (if (eq 'SHORT short)
		       (dsp-command b-op src mem
				    (ash addr 8))
		     (error "unknown address qualifier: ~S" short))
		 (if (r-register addr)
		     (dsp-command b-op src mem #x4000
				  (ash (eff-addr-mode (reg-num addr) (fifth instruction)) 8))
		   (if (normal-address addr)
		       (progn		;two word command
			 (dsp-command b-op ;must be X or Y abs-addr
				      #x4000 src mem
				      (ash absolute-address-mode 8))
			 (ext-command addr))
		     (error "~S is not a valid address" addr)))))))))
     
      ((JCC JSCC JHS JSHS JGE JSGE JNE JSNE JPL JSPL JNN JSNN JEC JSEC JLC JSLC JGT JSGT
	JCS JSCS JLO JSLO JLT JSLT JEQ JSEQ JMI JSMI JNR JSNR JES JSES JLS JSLS JLE JSLE)
					;(JCC addr [SHORT]) -- addr can be a label
       (if x-move (error "~S can not accomodate a parallel move: ~S" op x-move))
       (if (and dest (not (eq 'SHORT dest)) (not (R-register src)))
	   (error "too many arguments to ~S: ~S" op dest))
       (setf current-class PIPE-at-LA)
       (let ((J-op (if (member op '(JCC JHS JGE JNE JPL JNN JEC JLC JGT JCS JLO JLT JEQ JMI JNR JES JLS JLE))
		       DSP56-Jcc DSP56-JScc))
	     (addr (if (or (R-register src) (integerp src)) src
		     (get-label-value src (if (eq dest 'SHORT) 0 1)))))
	 (if (eq 'SHORT dest)		;can't be (Jcc R3 RN SHORT)
	     (if (short-jump-address addr)
		 (dsp-command J-op
			      #b1000000000000000000
			      (ash (encode-condition op) 12)
			      addr)
	       (error "invalid short jump address: ~S" src))
	   (if (R-register src)		;register reference
	       (dsp-command J-op
			    #b1100000010100000
			    (encode-condition op)
			    (ash (eff-addr-mode (reg-num src) dest) 8))
	     (progn			;two word command
	       (if (and (= j-op DSP56-JScc) 
			(member src loop-address))
		   (error "attempt to jump to LA: ~A" src))
	       (dsp-command J-op
			    #b1100000010100000
			    (encode-condition op)
			    (ash absolute-address-mode 8))
	       (ext-command addr))))))

      ((JMP JSR)			;(JSR xxx SHORT) or (JSR R3 R-1) or (maybe) (JSR addr), (JMP name SHORT)
       (if x-move (error "~S cannot accomodate a parallel move: ~S" op x-move))
       (let ((op-jsr (eq 'JSR op))
	     (addr (if (or (integerp src) (R-register src)) src
		     (get-label-value src (if (eq dest 'SHORT) 0 1)))))
	 (if (and op-jsr (member src loop-address)) 
	     (error "attempt to JSR to LA: ~A" src)
	   (setf current-class PIPE-at-LA))
	 (if (and dest (eq dest 'SHORT))
	     (if (short-jump-address addr)
		 (dsp-command addr (if op-jsr #x0D0000 #x0C0000))
	       (error "~S is an invalid short jump address" addr))
	   (if (R-register src)
	       (dsp-command (ash (eff-addr-mode (reg-num src) dest) 8)
			    (if op-jsr #x0BC080 #x0AC080))
	       ;; here we could look for short jump addresses:
	     (multiple-value-bind (val exists) (gethash src labels)
	       (if (and exists 
			(not (eq 'UNDEFINED (car val)))
			(short-jump-address (car val)))
		   (dsp-command (car val) (if op-jsr #x0D0000 #x0C0000))
		 (progn			;absolute address in p mem?
		   (dsp-command (if op-jsr #x0BC080 #x0AC080)
				(ash absolute-address-mode 8))
		   (ext-command addr))))))))
      
      (LUA				;(LUA R3 R4 R+N)
					;that is, it is like LOAD, not COPY (and different from Motorola)
					;syntax changed 12-Oct-90
					;destination register can be an N register
       (if x-move (error "LUA cannot accomodate a parallel move: ~S" x-move))
       (if (and (R-and-N-register src)
		(ok-mode (fourth instruction))
		(R-register dest))
	   (progn
	     (setf current-register src)
	     (dsp-command DSP56-LUA
			  (ash (eff-addr-mode (reg-num dest) (fourth instruction)) 8)
			  (reg-num src)))
	 (error "invalid ~A for ~S: ~S" 
		(if (not (R-register dest)) "source register"
		  (if (not (ok-mode (fourth instruction))) "update mode" "destination register"))
		op (cdr instruction))))

      ((MAC MACR MPY MPYR		;((MAC S1 S2 D) (move) (move))
	SMAC SMACR SMPY SMPYR)		;the subtracting forms
       (let ((D (fourth instruction)))
	 (if (and (not (eq D 'A)) (not (eq D 'B))) 
	     (error "invalid destination for ~S: ~S" op D))
	 (let ((minus (member op '(SMAC SMACR SMPY SMPYR)))
	       (mqqq (cond ((and (eq src 'X0) (eq dest 'X0)) 0)
			   ((and (eq src 'Y0) (eq dest 'Y0)) 1)
			   ((or (and (eq src 'X1) (eq dest 'X0))
				(and (eq src 'X0) (eq dest 'X1))) 2)
			   ((or (and (eq src 'Y1) (eq dest 'Y0))
				(and (eq src 'Y0) (eq dest 'Y1))) 3)
			   ((or (and (eq src 'X0) (eq dest 'Y1))
				(and (eq src 'Y1) (eq dest 'X0))) 4)
			   ((or (and (eq src 'Y0) (eq dest 'X0))
				(and (eq src 'X0) (eq dest 'Y0))) 5)
			   ((or (and (eq src 'X1) (eq dest 'Y0))
				(and (eq src 'Y0) (eq dest 'X1))) 6)
			   ((or (and (eq src 'Y1) (eq dest 'X1))
				(and (eq src 'X1) (eq dest 'Y1))) 7)
			   (t (error "invalid arguments for multiply: ~S ~S" src dest)))))
	   (can-move (case op 
		       ((MAC SMAC) DSP56-MAC)
		       ((MACR SMACR) DSP56-MACR)
		       ((MPY SMPY) DSP56-MPY)
		       ((MPYR SMPYR) DSP56-MPYR))
		     (logior (if minus #b100 0)
			     (if (eq D 'A) 0 #x8)
			     (ash mqqq 4))
		     D x-move y-move)))) 
      
      (NORM				;(NORM R3 A)
       (if x-move (error "too many arguments for NORM: ~S" x-move))
       (if (R-register src)
	   (dsp-command DSP56-NORM
			(ash (reg-num src) 8)
			(if (eq dest 'A) 0
			  (if (eq dest 'B) #x8
			    (error "invalid destination for NORM: ~S" dest))))
	 (error "invalid source for NORM: ~S" src)))
      
      (REP				;(REP num) (REP A) (REP X addr SHORT)
       ;; can addr be a label?
       (if x-move (error "REP cannot accomodate a parallel move: ~S" x-move))
       (setf current-class PIPE-REP)
       (if (integerp src)		;(REP num)
	   (if (too-big src 12)
	       (error "REP counter is too big: ~S" src)
	     (dsp-command #x0600A0
			  (ash (logand src #xFF) 8)
			  (ash (logand src #xF00) -8)))
	 (if dest			;(REP X addr SHORT)
	     (let ((short (fourth instruction)))
	       (if (and (integerp dest) (eq short 'SHORT))
		   (if (immediate-short-address dest)
		       (dsp-command #x060020
				    (if (eq src 'X) 0 #x40)
				    (ash dest 8))
		     (error "invalid short address: ~S" dest))
		 (dsp-command #x064020
			      (if (eq src 'X) 0 #x40)
			      (ash (eff-addr-mode (reg-num dest) short) 8))))
	   (dsp-command #x06C020	;(REP A)
			(ash (encode-register src) 8)))))
     
      ((TCC THS TGE TNE TPL TNN TEC TLC TGT TCS TLO TLT TEQ TMI TNR TES TLS TLE)
					;(TCC A B) (TCC A B R3 R4)
       (if x-move (error "~S cannot accomodate a parallel move: ~S" op x-move))
       (let ((R-source (fourth instruction))
	     (R-dest (fifth instruction)))
	 (if R-source
	     (if (and (R-register R-source) (R-register R-dest))
		 (progn
		   (setf current-register R-dest)
		   (setf current-source-register R-source)
		   (dsp-command #x030000
				(reg-num R-dest)
				(ash (reg-num R-source) 8)
				(ash (encode-condition op) 12)
				(ash (ABXYtoAB op src dest) 3)))
	       (error "invalid operands for ~S: ~S ~S ~S ~S" op src dest R-source R-dest))
	   (dsp-command #x020000
			(ash (ABXYtoAB op src dest) 3)
			(ash (encode-condition op) 12)))))
      
      ((JCLR JSCLR JSSET JSET)		;(JSCLR n ea jaddr)
       
       ;; cases are (JSCLR n D 16-bit-jump-address)
       ;;           (JSCLR n ea-in-IO-mem 16-bit-addr) -- (JSCLR n X-IO 23 label)
       ;;           (JSCLR n ea-SHORT 16-bit-addr)
       ;;           (JSCLR n ea 16-bit-addr)
       ;; we always use ext word as jump addr -- the ea part is where to test bit n
       ;; I assume jaddr can be a label. "Dest" can be a register name, X or Y.
       ;; There's one insidious hidden special case here -- JSSET has the #x10000 bit
       ;; on in all but one case (JSSET n S xxx).
       
       (if x-move (error "~S cannot accomodate a parallel move: ~S" op x-move))
       (if (> src 23) (error "invalid bit number for ~S: ~S" op src))
       (if (integerp dest) (error "invalid test location for ~S: ~S (not a memory specifier)" op dest))
       (setf current-class PIPE-after-REP) ; may need to be narrowed below
       (let* ((j-op (case op (JCLR DSP56-JCLR) (JSCLR DSP56-JSCLR) (JSSET DSP56-JSSET) (JSET DSP56-JSET)))
	      (j-num (car (last instruction)))
	      (j-addr (if (integerp j-num) j-num
			(get-label-value j-num 1)))) ;"1" here because we are always using the extension word
	 (if (member dest '(X Y X-IO Y-IO))
	     (let* ((x-mem (or (eq dest 'X) (eq dest 'X-IO)))
		    (test-addr (fourth instruction))
		    (good-grief (if (eq op 'JSSET) #x10000 0))
		    (reg (if (R-register test-addr) test-addr nil))
		    (mode (if reg (fifth instruction) nil))
		    (io (member dest '(X-IO Y-IO)))
		    (short (if reg nil (or (eq (fifth instruction) 'SHORT) 
					   (and (immediate-short-address test-addr)
						(not io))))))
	       (if (or short reg io)
		   (dsp-command j-op	;only top byte set except JSSET JSET set #x20 (sigh)
				src
				#x80	;unset only for register reference
				good-grief
				(if x-mem 0 #x40)
				(if short 0 (if io #x8000 #x4000))
				(ash (if reg (eff-addr-mode (reg-num reg) mode)
				       (if io (IO-address test-addr)
					 (if (Immediate-Short-Address test-addr) test-addr
					   (error "invalid short address: ~S" test-addr)))) 8))
		 (error "invalid test location for ~S: ~S ~S (test address must fit in 6 bits)" 
			op dest test-addr)))
	   (progn			;need more pipeline checks
	     (if (eq dest 'SSH) (setf current-class PIPE-J-SSH)
	       (if (eq dest 'SSL) (setf current-class PIPE-J-SSL)))
	     (print (format t "warning: this instruction probably does not work on CCRMA's NeXT's: ~S ~S ~S" op src dest))
	     (dsp-command j-op
			  src
			  (ash (encode-register dest) 8)
			  #xC000)))	;register case
	 (ext-command j-addr)))

      (DO				;(DO X R3 RN expr) (DO n expr) (DO S expr) (DO X short-addr expr)
					;src=number of iterations, dest=end of loop location (can be label of course)
					;src=0 is same as 2^16-1!
					;need to set LA, LF somewhere
					;thankfully, LA is always in the extension word, so we can easily find it later
					;DO SSH,xxxx is illegal
       ;; A-64 56000 manual: "assembler calculates LA as expr-1 to accommodate two-word instr" -- but what
       ;; about the one word case?  
       (if x-move (error "DO cannot accomodate a parallel move: ~S" x-move))
       (incf loop-flag)
       (setf current-class PIPE-DO)
       (let* ((j-num (car (last instruction)))
	      (j-addr (if (integerp j-num) j-num
			(get-label-value j-num 1))))
	 (push j-num loop-address)	;was J-ADDR (21-4-90)
	 (if (integerp src)		;(DO n expr)
	     (if (too-big src 12)	;called "immediate short data" but a 12 bit quantity
		 (error "invalid loop counter value: ~S" src)
	       (dsp-command DSP56-DO
			    #x80
			    (ash (logand src #xFF) 8)
			    (ash (logand src #xF00) -8)))
	   (if (or (eq src 'X) (eq src 'Y)) ;register X not valid here, so we'll march blithely on
	       (let ((x-mem (eq src 'X)))
		 (if (integerp dest)
		     (if (too-big dest 6)
			 (error "invalid short address: ~S" dest)
		       (dsp-command DSP56-DO
				    (if x-mem 0 #x40)
				    (ash dest 8)))
		   (dsp-command DSP56-DO
				(if x-mem 0 #x40)
				#x4000
				(ash (eff-addr-mode (reg-num dest) (fourth instruction)) 8))))
	     (if (eq src 'SSH) 
		 (error "illegal instruction: ~S" instruction)
	       (dsp-command DSP56-DO
			    #xC000
			    (ash (encode-register src) 8)))))
	 (ext-do-command j-addr)))
	
      (UPDATE				;(UPDATE R3 RN) -- can only be parallel move no-op + register update
       (if x-move (error "address register update cannot accomodate another parallel move: ~S" x-move))
       (address-register-update 0 src dest))

      (COPY				;(COPY A B) or (COPY SP A) or (COPY A X0) with (LOAD d Y ...)
					;(COPY A B) is very wierd -- B0 <- 0, B1 is "limited", etc
       (if (and (not x-move) (not y-move))
	   (if (or (movec-register src) (movec-register dest))
	       (copy-control-register instruction)
	     (if (and (or (eq dest 'A) (eq dest 'B))
		      (or (eq src 'B) (eq src 'A)))
		 (can-move DSP56-TFR (ash (ABXYtoAB op src dest) 3) dest x-move y-move)
	       (register-to-register-copy 0 src dest)))
	 (if (and x-move y-move)
	     (if (and (or (eq dest 'A) (eq dest 'B))
		      (member src '(A B X0 X1 Y0 Y1)))
		 (can-move DSP56-TFR (ash (ABXYtoAB op src dest) 3) dest x-move y-move)
	       (error "attempt to do three incompatible parallel moves: ~S ~S ~S" instruction x-move y-move))
	   (let ((mov (if x-move x-move y-move)))
	     ;; either R:Y X:R class I or II
	     (if (and (not (eq (caddr mov) 'L)) ;has to be X Y or immediate
		      (or (and (member (cadr mov) '(X0 X1 A B)) ;X:R class I
			       (member src '(A B))
			       (member dest '(Y0 Y1)))
			  (and (eq src 'X0)                     ;X:R class II
			       (member dest '(A B))
			       (member (cadr mov) '(A B X0 Y0)))
			  (and (eq src 'Y0)                     ;R:Y class II
			       (member dest '(A B))
			       (member (cadr mov) '(X0 Y0 A B)))
			  (and (member (cadr mov) '(Y0 Y1 A B)) ;R:Y class I
			       (member src '(A B))
			       (member dest '(X0 X1)))))
		 (can-move 0 0 nil instruction mov)
	       (if (and (or (eq dest 'A) (eq dest 'B))
			(member src '(A B X0 X1 Y0 Y1)))
		   (can-move DSP56-TFR (ash (ABXYtoAB op src dest) 3) dest x-move y-move)
		 (error "invalid parallel move request: ~S ~S" instruction mov)))))))
      
      (MOVE				;(MOVE X-IO to Y)  -- only memory to memory here
       (if x-move (error "MOVE(P) cannot accomodate parallel moves: ~S" x-move))
       (X-IO-or-Y-IO-move instruction))

      ((LOAD STORE)			;(lotsa choices)
       
       ;; here we sort out references to control and modifier registers, peripheral IO stuff,
       ;; program memory references, etc.  If the command received is parallel move compatible,
       ;; we vector off to CAN-MOVE with an op-code of 0 (the MOVE no-op).  Otherwise, we figure
       ;; out who to call here.
       
       (if x-move
	   (if y-move
	       (if (and (eq (car x-move) 'COPY)
			(or (eq (caddr x-move) 'A) (eq (caddr x-move) 'B))
			(member (cadr x-move) '(A B X0 X1 Y0 Y1)))
		   (can-move DSP56-TFR (ash (ABXYtoAB 'COPY (cadr x-move) (caddr x-move)) 3) (caddr x-move) instruction y-move)
		 (if (and (eq (car y-move) 'COPY)
			  (or (eq (caddr y-move) 'A) (eq (caddr y-move) 'B))
			  (member (cadr y-move) '(A B X0 X1 Y0 Y1)))
		     (can-move DSP56-TFR (ash (ABXYtoAB 'COPY (cadr y-move) (caddr y-move)) 3) (caddr y-move) instruction x-move)
		   (error "attempt to do three parallel moves: ~S ~S ~S" instruction x-move y-move)))
	     (can-move 0 0 nil instruction x-move))
	 (if y-move
	     (can-move 0 0 nil instruction y-move)
	   
	   ;; it's a single LOAD or STORE -- sort out all the possibilities -- we can dispatch off the
	   ;; memory type -- X Y P X-IO Y-IO L
	   (if (or (integerp dest)
		   (and (not (member dest '(X Y L P X-IO Y-IO)))
			(gethash dest names)))
	       (if (movec-register src)
		   (X-or-Y-load-or-store-control-register instruction)
		 (if (eq (fourth instruction) 'SHORT)
		     (if (eq op 'STORE)
			 (error "cannot store immediate: ~S" instruction)
		       (immediate-short-load 0 src dest))
		   (XY-load-or-store 0 instruction)))
	     (case dest
	       ((X Y)
		(if (movec-register src)
		    (X-or-Y-load-or-store-control-register instruction)
		  (XY-load-or-store 0 instruction)))
	       (L (long-memory-load-or-store 0 instruction))
	       (P (P-memory-load-or-store instruction))
	       ((X-IO Y-IO) (X-IO-or-Y-IO-load-or-store instruction))
	       (t (error "invalid memory type for ~S: ~S in ~S" op dest instruction)))))))
 
      (COMMENT           (setf pseudo-time t)) ;(COMMENT ...)
      (----------------> (setf pseudo-time t)) ;compiler generated comment
      (ORG               
       (setf pseudo-time t)
       (setf dsp-pc src))
      (X-ORG             
       (setf pseudo-time t)
       (setf x-pc src))
      (Y-ORG 
       (setf pseudo-time t)
       (setf y-pc src))
      (SET-DATA				;(SET-DATA X|Y loc val)
       (setf pseudo-time t)
       (let ((val (fourth instruction)))
	 (if (eq src 'X)
	     (setf-x-mem dest val)
	   (setf-y-mem dest val))))
      ((X-DATA Y-DATA)
       (setf pseudo-time t)
       (let ((x-mem (eq op 'X-DATA)))
	 (loop for i in (cdr instruction) do
	   (if (listp i)		;it's an implicit define (i.e. a label for some memory location)
	       (progn
		 (setf we-may-have-names t)
		 (setf (gethash (car i) names) (if x-mem x-pc y-pc)))
	     (progn
	       (if x-mem 
		   (progn
		     (setf-x-mem x-pc i)
		     (incf x-pc))
		 (progn
		   (setf-y-mem y-pc i)
		   (incf y-pc))))))))
      (HIDE				;(HIDE label label...)
       (setf pseudo-time t)
       (loop for i in (cdr instruction) do
	 (let ((lab (gethash i labels)))
	   (if lab			;label does exist
	       (if (eq (car lab) 'UNDEFINED)
		   (cerror "will ignore it" "~S is being hidden before it was defined!" i)
		 (remhash i labels))
	     (cerror "will ignore request" "HIDE ~S, but ~S doesn't exist!" i i)))))
      (UNDEFINE				;(UNDEFINE name name...)
       (setf pseudo-time t)
       (loop for i in (cdr instruction) do
	 (let ((nam (gethash i names)))
	   (if nam
	       (remhash i names)
	     (cerror "will ignore request" "UNDEFINE ~S, but ~S doesn't exist!" i i)))))
      (DEFINE				;(DEFINE name val)
       (setf pseudo-time t)
       (setf we-may-have-names t)
       (setf (gethash src names) dest))
      (t (if (and (cdr instruction)	;(lab) or (lab LOCAL) = definition of label LAB
		  (not (eq (cadr instruction) 'LOCAL)))
	     (error "unknown instruction: ~S" instruction)
	   (progn
	     (when (eq (car instruction) (car loop-address))
	       (setf loop-address (cdr loop-address))
	       (setf end-of-loop t))
	     (fixup-label (car instruction))
	     (setf pseudo-time t)
	     (if (eq (cadr instruction) 'LOCAL) (remhash (car instruction) labels))))))

  (if (not pseudo-time) (check-for-pipeline-trouble op arg1))))

	
(defun init-emit ()			;get us a clean assembly environment
  (clrhash labels)
  (clrhash names)
  (setf dsp-pc 0)
  (setf last-dsp-pc -1)
  (setf x-pc 0)
  (setf y-pc 0)
  (setf do-fixup-stack nil)
  (setf loop-flag 0)			;LF simulator
  (setf end-of-loop nil)
  (setf loop-address nil)		;LA simulator
  (setf two-word-instruction nil)
  (setf current-class -1)		;the "all's well" pipeline state
  (setf current-register nil)
  (setf current-source-register nil)
  (setf current-addr-mode nil)
  (setf pipe-1 -1)			;previous pipeline instruction class
  (setf pipe-2 -1)			;ditto but two back
  (setf pipe-3 -1)			;ditto but 3 back (need that much at end of DO loops)
  (setf last-reg nil)  
  (setf after-Do-reg nil))


;;; End of Assembler ----------------------------------------------------------------------------------------
;;;
#|

;;; debugging  and testing apparatus

(defun xemit (a &optional b c) (let ((i dsp-pc)) (emit a b c) (format nil "~X" (get-dsp-program i))))

(defun test-emit (x a &optional b c)
  (setf dsp-pc 0)
  (emit a b c)
  (if (/= x (get-dsp-program 0))
      (error "failed on: ~S ~S ~S" a b c)))

(test-emit #x200026 '(ABS A))
(test-emit #x20002E '(ABS B))
(test-emit #x8c     '(ENDDO))
(test-emit #x44f426 '(ABS A) '(LOAD X0 32))
(test-emit #x46f426 '(ABS A) '(LOAD Y0 0))
(test-emit #x200021 '(ADC X A))
(test-emit #x200039 '(ADC Y B))
(test-emit #x565b29 '(ADC X B) '(STORE A X R3 R+1))
(test-emit #x200010 '(ADD B A))
(test-emit #x200070 '(ADD Y1 A))
(test-emit #x21ee68 '(ADD X1 B) '(COPY B A))
(test-emit #x57a040 '(ADD X0 A) '(LOAD B X 32)) ;#x57f040 if not optimzed to SHORT move
(test-emit #x20001a '(ADDL A B))
(test-emit #x572012 '(ADDL B A) '(STORE B X 32)) ;#x577012 if not optimized
(test-emit #x20000a '(ADDR A B))
(test-emit #x204b02 '(ADDR B A) '(UPDATE R3 R+N))
(test-emit #x200046 '(AND X0 A))
(test-emit #x5de746 '(AND X0 A) '(LOAD B1 Y R7 R))
(test-emit #x59407e '(AND Y1 B) '(STORE B0 Y R0 R-N))
(test-emit #x3b8    '(ANDI 3 MR))
(test-emit #x20b9   '(ANDI 32 CCR))
(test-emit #x12ba   '(ANDI #x12 OMR))
(test-emit #x200032 '(ASL A))
(test-emit #x262032 '(ASL A) '(LOAD Y0 32 SHORT))
(test-emit #x20002a '(ASR B))
(test-emit #x49202a '(ASR B) '(STORE B L 32)) ;as above, #x49702a
(test-emit #xbfb52  '(BCHG #x12 SP))
(test-emit #xbd440  '(BCHG 0 R4))
(test-emit #xb4801  '(BCHG 1 X R0 R+N))
(test-emit #xb7043  '(BCHG 3 Y 32))
(test-emit #xb0310  '(BCHG #x10 X 3 SHORT))
(test-emit #xb8340  '(BCHG 0 Y-IO 3))
(test-emit #xa6c17  '(BCLR 23 X R4 RN))
(test-emit #xa1202  '(BCLR 2 X #x12 SHORT))
(test-emit #xa8041  '(BCLR 1 Y-IO 0))
(test-emit #xae743  '(BCLR 3 M7))
(test-emit #xa5367  '(BSET #b111 Y R3 R-1))
(test-emit #xa3821  '(BSET 1 X #b111000 SHORT))
(test-emit #xa8060  '(BSET 0 Y-IO 0))
(test-emit #xafc61  '(BSET 1 SSH))
(test-emit #xb7f22  '(BTST 2 X r7 1-R))
(test-emit #xb1f77  '(BTST 23 Y #x1F SHORT))
(test-emit #xb8f28  '(BTST 8 X-IO #xF))
(test-emit #xbce61  '(BTST 1 A))
(test-emit #x200013 '(CLR A))
(test-emit #x42d81b '(CLR B) '(LOAD X L R0 R+1))
(test-emit #x200005 '(CMP B A))
(test-emit #x418075 '(CMP Y1 A) '(LOAD B10 L 0 SHORT))
(test-emit #x20005f '(CMPM Y0 B))
(test-emit #x1b3077 '(CMPM Y1 A) '(STORE A X 0) '(COPY B Y1))
(test-emit #x18040  '(DIV X0 A))
(test-emit #x18078  '(DIV Y1 B))
(test-emit #x64000  '(DO X R0 R-N))
(test-emit 0        '(NOP))
(test-emit #x8c     '(ENDDO))
(test-emit #x6c400  '(DO X0 Do-Loop))
(test-emit #x61700  '(DO X 23 SHORT))
(test-emit #x62381  '(DO #x123 #xff))
(test-emit #x20006b '(EOR X1 B))
(test-emit #x510143 '(EOR X0 A) '(STORE B0 X 1 SHORT))
(test-emit #x5      '(ILLEGAL))
(test-emit #xaf0a0  '(JCC 1))
(test-emit #xad4ad  '(JES R4 R-1))
(test-emit #xe9017  '(JLT 23 SHORT))
(test-emit #xaf0af  '(JLE 0))
(test-emit #xa6c81  '(JCLR 1 X R4 Rn #x1234))
(test-emit #xa01c7  '(JCLR #b111 Y 1 SHORT 1000))
(test-emit #xaa280  '(JCLR 0 X-IO 34 3))
(test-emit #xaca01  '(JCLR 1 A2 1000))
(test-emit #xaf080  '(JMP #xff))
(test-emit #xad280  '(JMP R2 R-1))
(test-emit #xc00ff  '(JMP #xff SHORT))
(test-emit #xfa020  '(JSEQ 32 SHORT))
(test-emit #xbcfa4  '(JSNN R7 R+N))
(test-emit #xbf0ac  '(JSNR 1000))
(test-emit #xb60c1  '(JSCLR 1 Y R0 R))
(test-emit #xba08a  '(JSCLR #xA X-IO #x20 1000))
(test-emit #xbfa03  '(JSCLR 3 OMR 1000))
(test-emit #xa68a1  '(JSET 1 X R0 RN 23))
(test-emit #xa20b7  '(JSET 23 X 32 SHORT 23))
(test-emit #xa81e1  '(JSET 1 Y-IO 1 23))
(test-emit #xafe37  '(JSET 23 LA 123))
(test-emit #xd0002  '(JSR 2 SHORT))
(test-emit #xbf080  '(JSR 1))
(test-emit #xbd780  '(JSR R7 R-1))
(test-emit #xd001f  '(JSR 31 SHORT))
(test-emit #xb61f7  '(JSSET #x17 Y R1 R #xfff))
(test-emit #xb1fe1  '(JSSET 1 Y 31 SHORT #xf))
(test-emit #xb83a2  '(JSSET 2 X-IO 3 4))
(test-emit #xaff24  '(JSSET 4 LC 32))
(test-emit #x200033 '(LSL A))
(test-emit #x1d9433 '(LSL A) '(LOAD B X R4 R-1) '(COPY A Y1))
(test-emit #x20002b '(LSR B))
(test-emit #x117023 '(LSR A) '(COPY A X0) '(STORE Y1 Y #x1234))
(test-emit #x44116  '(LUA R6 R1 R-N))
(test-emit #x45710  '(LUA R0 R7 R-1))
(test-emit #x200082 '(MAC X0 X0 A))
(test-emit #x20feea '(MAC X1 Y0 B) '(COPY Y1 N6))
(test-emit #x23e6f6 '(SMAC Y1 X1 A) '(COPY N7 Y0))
(test-emit #x2effbe '(SMAC Y1 Y0 B) '(LOAD A #xff SHORT))
(test-emit #x2000d3 '(MACR Y0 X0 A))
(test-emit #x2000d3 '(MACR X0 Y0 A))
(test-emit #xd098fb '(MACR X1 Y1 B) '(LOAD X0 X R0 R+1) '(LOAD Y0 Y R4 R+N))
(test-emit #x9018eb '(MACR X1 Y0 B) '(STORE X0 X R0 R+1) '(STORE Y0 Y R4 R+N))
(test-emit #xd018ef '(SMACR X1 Y0 B) '(STORE X0 X R0 R+1) '(LOAD Y0 Y R4 R+N))
(test-emit #x9098ef '(SMACR X1 Y0 B) '(LOAD X0 X R0 R+1) '(STORE Y0 Y R4 R+N))
(test-emit #x2e3200 '(LOAD A #x32 SHORT))
(test-emit #x2b0300 '(LOAD B2 #x3 SHORT))
(test-emit #x227f00 '(COPY R3 N7))
(test-emit #x209200 '(COPY X0 R2))
(test-emit #x205b00 '(UPDATE R3 R+1))
(test-emit #x205700 '(UPDATE R7 R-1))
(test-emit #x76db00 '(LOAD N6 X R3 R+1))
(test-emit #x63f400 '(LOAD R3 #x23))
(test-emit #x53f400 '(LOAD B2 0))
(test-emit #x45f000 '(LOAD X1 X #x1234))
(test-emit #x44c000 '(LOAD X0 X R0 R-N))
(test-emit #x762000 '(STORE N6 X 32))	; #x767000 as above
(test-emit #x537900 '(STORE B2 X R1 1-R))
(test-emit #x77a000 '(LOAD N7 X 32 SHORT))
(test-emit #x572000 '(STORE B X 32 SHORT))
(test-emit #x109b00 '(LOAD X0 X R3 R+1) '(COPY A Y0))
(test-emit #x1b3000 '(STORE A X 123) '(COPY B Y1))
(test-emit #x1db400 '(LOAD B 23) '(COPY A Y1))
(test-emit #x83000  '(STORE A X 123) '(COPY X0 A))
(test-emit #x91400  '(STORE B X R4 R-1) '(COPY X0 B))
(test-emit #x5df000 '(LOAD B1 Y 123))
(test-emit #x584000 '(STORE A0 Y R0 R-N))
(test-emit #x798c00 '(LOAD N1 Y 12 SHORT))
(test-emit #x590100 '(STORE B0 Y 1 SHORT))
(test-emit #x10f000 '(COPY A X0) '(LOAD Y0 Y 12))
(test-emit #x1f5700 '(COPY B X1) '(STORE B Y R7 R-1))
(test-emit #x15f400 '(LOAD Y1 #x123) '(COPY A X1))
(test-emit #x8b900  '(STORE A Y R1 1-R) '(COPY Y0 A))
(test-emit #x9b000  '(COPY Y0 B) '(STORE A Y 123))
(test-emit #x40dc00 '(LOAD A10 L R4 R+1))
(test-emit #x428300 '(LOAD X L 3))	;#x42f000 if not optimized
(test-emit #x480100 '(STORE A L 1))	;#x487000 ditto
(test-emit #x4b7f00 '(STORE BA L R7 1-R)) ; Motorola manual A-146 gives 1-R mode 6, which can't be right
(test-emit #x4a8200 '(LOAD AB L 2 SHORT))
(test-emit #x430200 '(STORE Y L 2 SHORT))
(test-emit #xebb900 '(LOAD A X R1 R+1) '(LOAD B Y R5 R-1))
(test-emit #xabb900 '(LOAD A X R1 R+1) '(STORE B Y R5 R-1))
(test-emit #xef3900 '(STORE B X R1 R+1) '(LOAD B Y R5 R-1))
(test-emit #xac3900 '(STORE B X R1 R+1) '(STORE Y0 Y R5 R-1))
(test-emit #xa57e00 '(STORE X1 X R6 R+1) '(STORE Y1 Y R3 R-1))
(test-emit #x5f023  '(LOAD M3 X 123))
(test-emit #x5583b  '(STORE SP X R0 R+1))
(test-emit #x5f07a  '(LOAD OMR Y 0))
(test-emit #x54f7f  '(STORE LC Y R7 R+N))
(test-emit #x5f427  '(LOAD M7 #xff))
(test-emit #x5a03c  '(LOAD SSH X 32 SHORT))
(test-emit #x50c39  '(STORE SR X 12 SHORT))
(test-emit #x59f63  '(LOAD M3 Y #x1f SHORT))
(test-emit #x5007d  '(STORE SSL Y 0 SHORT))
(test-emit #x462a1  '(COPY M1 M2))
(test-emit #x44ebb  '(COPY SP A))
(test-emit #x4cebb  '(COPY A SP))
(test-emit #x5ffa7  '(LOAD M7 #xff SHORT))
(test-emit #x7d384  '(LOAD X0 P R3 R-1))
(test-emit #x770a6  '(STORE M6 P 23))
(test-emit #x7830e  '(LOAD A P 3 SHORT))
(test-emit #x70c3a  '(STORE OMR P 12 SHORT))
(test-emit #x8f083  '(MOVE X 123 X-IO 3))
(test-emit #x8e3c1  '(MOVE Y R3 R X-IO 1))
(test-emit #x8f483  '(STORE 123 X-IO 3))
(test-emit #x84083  '(MOVE X-IO 3 X R0 R-N))
(test-emit #x870c1  '(MOVE X-IO 1 Y 123))
(test-emit #x9d183  '(MOVE X R1 R-1 Y-IO 3))
(test-emit #x9f0c0  '(MOVE Y 1 Y-IO 0))
(test-emit #x9f482  '(STORE 1 Y-IO 2))
(test-emit #x94ac3  '(MOVE Y-IO 3 Y R2 R+N))
(test-emit #x97092  '(MOVE Y-IO #x12 X 123))
(test-emit #x8f043  '(MOVE P 1 X-IO 3))
(test-emit #x84140  '(MOVE X-IO 0 P R1 R-N))
(test-emit #x9f041  '(MOVE P 3 Y-IO 1))
(test-emit #x97f4f  '(MOVE Y-IO #b1111 P R7 1-R))
(test-emit #x85d03  '(LOAD N5 X-IO 3))
(test-emit #x8ca22  '(STORE A2 X-IO #x22))
(test-emit #x9d420  '(STORE R4 Y-IO 32))
(test-emit #x94901  '(LOAD B0 Y-IO 1))
(test-emit #x2000c0 '(MPY Y1 X0 A))
(test-emit #x56f48c '(SMPY X0 X0 B) '(LOAD A 23)) ;since we just exhaustively tested moves, we'll use this LOAD for the rest
(test-emit #x200099 '(MPYR Y0 Y0 B))
(test-emit #x2000e5 '(SMPYR X1 Y0 A))
(test-emit #x200036 '(NEG A))
(test-emit #x1de15  '(NORM R6 A))
(test-emit #x1d81d  '(NORM R0 B))
(test-emit #x200017 '(NOT A))
(test-emit #x56f41f '(NOT B) '(LOAD A 23))
(test-emit #x200042 '(OR X0 A))
(test-emit #x56f47a '(OR Y1 B) '(LOAD A 23))
(test-emit #x2fa    '(ORI 2 OMR))
(test-emit #x65c20  '(REP X R4 R+1))   (test-emit 0        '(NOP)) ; just clearing the pipeline
(test-emit #x67860  '(REP Y R0 1-R))   (test-emit 0        '(NOP))
(test-emit #x60120  '(REP X 1 SHORT))  (test-emit 0        '(NOP))
(test-emit #x620a0  '(REP 32))         (test-emit 0        '(NOP))
(test-emit #x6ce20  '(REP A))          (test-emit 0        '(NOP))
(test-emit #x6e520  '(REP M5))         (test-emit 0        '(NOP))
(test-emit #x6ffa1  '(REP #x1ff))      (test-emit 0        '(NOP))
(test-emit #x84     '(RESET))
(test-emit #x200011 '(RND A))
(test-emit #x56f419 '(RND B) '(LOAD A 23))
(test-emit #x200037 '(ROL A))
(test-emit #x56f42f '(ROR B) '(LOAD A 23))
(test-emit #x4      '(RTI))
(test-emit #xc      '(RTS))
(test-emit #x200025 '(SBC X A))
(test-emit #x56f43d '(SBC Y B) '(LOAD A 23))
(test-emit #x87     '(STOP))
(test-emit #x200014 '(SUB B A))
(test-emit #x56f45c '(SUB Y0 B) '(LOAD A 23))
(test-emit #x200016 '(SUBL B A))
(test-emit #x56f41e '(SUBL A B) '(LOAD A 23))
(test-emit #x20000e '(SUBR A B))
(test-emit #x6      '(SWI))
(test-emit #x2b040  '(TMI X0 A))
(test-emit #x3c374  '(TNR Y1 A R3 R4))
(test-emit #x200001 '(TFR B A))
(test-emit #x56f479 '(TFR Y1 B) '(LOAD A 23))
(test-emit #x200003 '(TST A))
(test-emit #x56f40b '(TST B) '(LOAD A 23))
(test-emit #x86     '(WAIT))


|#
