;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;
;;; definstrument, instrument-let, compile-ins, various instrument debugging functions
;;;
;;; used to be in sound.lisp, but split out to make debugging the new versions simpler.
;;; Definstrument was straightforward until version 3 of cmus -- in its new incarnation
;;; it has to know how to write/compile/load c modules on each system.
;;; 

(in-package :clm)

#+56-mus-gens
(defun make-var-list_56 (var-table)
  (let ((new-hash-list nil))
    (maphash #'(lambda (key val)
		 ;; key = variable name, val = '(type address work indirect element-type)
		 (if (and (second val) 
			  (not (temp-sig key)))
		     (push (list (symbol-name key) (second val) (first val) (fifth val)) new-hash-list)))
	     var-table)
    new-hash-list))

(defun make-var-list_c (var-table)
  (let ((new-hash-list nil))
    (maphash #'(lambda (key val)
		 (if (not (member key new-sig))
		     (push (append (list (symbol-name key)) val) new-hash-list)))
	     var-table)
    new-hash-list))

(defun make-var-hash (var-list)
  (let ((new-vars (make-hash-table :test #'equal)))
    ;; that test should be equalp -- all this to try to get rid of the goddamn useless package names!
    (loop for i in var-list do
      (setf (gethash (first i) new-vars) (rest i)))
    new-vars))

;;; can't store a hash table in a fasl file, at least in excl
;;; hkt says this should work:
;;; separate file hash.lisp:
;;;  (in-package :clm)
;;;  (setq vars '#.vars)

#-56-mus-gens
(defmacro set-instrument-properties (name)
  `(progn
     (setf (get ,name :ins-vars) (clm::make-var-hash ',(make-var-list_c vars)))
     (setf (get ,name :phrase-symbols) (copy-list clm::*current-phrase-symbols*))))

#+56-mus-gens
(defmacro set-instrument-properties (name)
  (when (or (not (zerop dsp-pc)) (eq *clm-language* :c))
    (let ((proglen (min (+ dsp-pc 1) (- internal-P-size 2))))
      (if (eq *clm-language* :c)	; i.e. we have an explicit C ins in a c/56k version of clm
	  `(progn
	     (setf (get ,name :ins-vars) (clm::make-var-hash ',(make-var-list_c vars)))
	     (setf (get ,name :phrase-symbols) (copy-list clm::*current-phrase-symbols*))
	     (setf (get ,name :language) :c))
	`(progn
	   (setf (get ,name :ins-vars) (clm::make-var-hash ',(make-var-list_56 vars)))
	   ;; we put these properties in the keyword package (by using ":" rather than "'") --
	   ;; otherwise, we get endless package-name confusion.
	   (setf (get ,name :language) :56k)
	   (setf (get ,name :dsp-compiled-program-length) ,proglen)
	   (setf (get ,name :dsp-<input>) ,<input>)
	   (setf (get ,name :dsp-<output>) ,<output>)
	   (setf (get ,name :dsp-compiled-program) ,(make-initial-contents internal-P-memory (1+ proglen)))
	   (setf (get ,name :dsp-version-number) ,dsp-version-number)
	   (if c56-need-to-save-labels 
	       (setf (get ,name :dsp-labels) ',(copy-tree label-list))
	     (setf (get ,name :dsp-labels) nil))
	   (setf (get ,name :dsp-memory-map) ,external-Y-from-X-offset)
	   (setf (get ,name :dsp-ext-x-data) ,(make-initial-contents external-memory ex-ptr))
	   (setf (get ,name :dsp-ext-y-data) ,(make-initial-contents external-memory ey-ptr External-Y-from-X-offset))
	   (setf (get ,name :dsp-int-x-data) ,(make-initial-contents internal-x-memory ix-ptr))
	   (setf (get ,name :dsp-int-y-data) ,(make-initial-contents internal-y-memory iy-ptr)))))))

#+56-mus-gens
(defun display-dsp-state (&optional (slot nil) (dsp nil))
  (if (zerop dsps)
      (print "DSP uninitialized, so I can't say what it's state might be.")
    (progn
      (if slot
	  (progn
	    (setf *current-instrument-name* (or (dsp-data-ins-name (find-dsp slot dsp)) *current-instrument-name*))
	    (dsp-set slot dsp))
	(progn
	  (if (not (eq *current-instrument-name* (dsp-data-ins-name (find-dsp (c56-current-slot) (c56-current-dsp)))))
	      (if (not (find-named-dsp *current-instrument-name*))
		  (print "Last instrument name and dsp names don't match")))))
      (if *current-instrument-name*
	  (if (symbol-plist *current-instrument-name*)
	      (if (get *current-instrument-name* :ins-vars)
		  (progn
		    (princ (format nil "~&We think the last instrument to run was ~A~A:~&" 
				   *current-instrument-name*
				   (if (= 1 dsps) 
				       " " 
				     (format nil " (slot: ~A, dsp: ~A)" 
					     (c56-current-slot) (c56-current-dsp)))))
		    (display-instrument-state))
		(error "Odd--can't find ~A's display function." *current-instrument-name*))
	    (error "Strange--~A's property list is nil." *current-instrument-name*))
	(print "I don't know who ran last")))))

#+56-mus-gens (defun describe-dsp-state (&optional (slot nil) (dsp nil)) (display-dsp-state slot dsp))

(defun ins-language ()
  #-56-mus-gens :c
  #+56-mus-gens (or (and *current-instrument-name*
			 (symbol-plist *current-instrument-name*)
			 (get *current-instrument-name* :language))
		    :56k)
  )

(defun describe-c-state ()
  (let* ((vars (and c-debug
		    *current-instrument-name*
		    (symbol-plist *current-instrument-name*)
		    (get *current-instrument-name* :ins-vars))))
    (if vars
	(maphash #'(lambda (a b)
		     (if (eq (first b) :pass)
			 (print (format nil "~A: ~A" a (dm-any_c b debug-datai debug-datar)))))
		 vars))))

(defun describe-ins-state ()
  #+56-mus-gens (if (eq (ins-language) :c) (describe-c-state) (describe-dsp-state))
  #-56-mus-gens (describe-c-state)
  )

(defun noopfun (x y z) 
  (declare (ignore y z)) 
  (if (and x (listp x) (eq *clm-language* :c) (> (length x) 1) (eq (car x) 'END-RUN))
      (setf *current-phrase-symbols* (cddr (second x))))
  x)

#-mcl
(defmacro definstrument (ins-name (&rest args) &body body &environment env)
  (let* ((*header-info* nil)
	 (*c-file-name* nil)
	 #+(and excl NeXT) (*c-compiler-options* "-c -O")
	 #+(and excl SGI) (*c-compiler-options* "-c -ansi -G 0 -O")
	 (silly-name (gensym))
	 (name (if (listp ins-name)
		   (apply #'(lambda (nam &key (language #+56-mus-gens :56k #-56-mus-gens :c)
					      (c-file nil)
					      (c-include-file nil)
					      (exp-env t) 
					      #+(and excl NeXT) (c-options "-c -O")
					      #+(and excl SGI) (c-options "-c -ansi -G 0 -O")
					      (save-labels nil))
				 (prog1 nam
				   (setf c56-need-to-save-labels save-labels)
				   (setf c56-need-to-load-expt exp-env)
				   (setf *c-file-name* c-file)
				   (setf *header-info* c-include-file)
				   #+excl (setf *c-compiler-options* c-options)
				   (setf *clm-language* language)))
			     ins-name)
		 (progn
		   (setf *clm-language* #+56-mus-gens :56k #-56-mus-gens :c)
		   ins-name))))
    (if (eq *clm-language* :56k)
	(progn
	  (setf <t> 0)
	  (setf <nil> #x10000)
	  `(progn
	     (eval-when (eval) (warn "A clm instrument intended for the DSP has almost no chance of working unless it is compiled."))
	     (defun ,name ,args 
	       (setf *current-instrument-name* ',name)
	       (progn
		 ,@body))
	     (defun ,silly-name ()
	       (pushnew ',name *clm-instruments*)
	       (set-instrument-properties ',name))
	     (eval-when (load) (,silly-name))))
      (let* ((c-ff (gentemp "CLM_C_"))
	     #+(or kcl mcl) (c1-ff (gentemp "CLM_C_"))
	     #+kcl (c2-ff (gentemp "clm_c_"))
	     #+mcl (mcl-name (intern (concatenate 'string "clm_" (lc name))))
	     #+excl (c-ff-name (symbol-name c-ff))
	     #+(and excl NeXT) (excl-c-ff-name (concatenate 'string "_" c-ff-name))
	     #+(and excl SGI) (excl-c-ff-name c-ff-name)
	     
	     #+kcl (c-file-name (or *c-file-name* 
				    (namestring 
				     (merge-pathnames 
				      (concatenate 'string "clm_" (symbol-name name) ".c")
				      si:*load-pathname*))))
	     
	     #+mcl (c-file-name (or *c-file-name* 
				    (namestring 
				     (merge-pathnames 
				      (concatenate 'string "clm:" "clm_" (lc name) ".c")
				      ccl:*loading-file-source-file*))))
	     
	     #+excl (c-file-name (or *c-file-name* 
				     (namestring
				      (merge-pathnames 
				       (concatenate 'string c-ff-name ".c")
				       excl:*source-pathname*))))
	     #+excl (l-file-name (concatenate 'string (subseq c-file-name 0 (- (length c-file-name) 2))  ".o")))
	(setf *c-file* (open c-file-name :direction :output :if-exists :supersede :if-does-not-exist :create))
	(princ (format nil "; Writing ~S~%" c-file-name))
	(multiple-value-bind (second minute hour date month year day daylight-saving-p time-zone)
	    (get-decoded-time)
	  (declare (ignore time-zone daylight-saving-p second))
	  (flet ((month-name (month) (nth (- month 1) '("Jan" "Feb" "Mar" "Apr" "May" "Jun" "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")))
		 (day-name (day) (nth day '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))))
		(format *c-file* "/* ~A C file generated by CLM ~A ~D-~A-~D at ~D:~2,'0D */~%~
                                  #include <stdio.h>~%~
                                  #include <signal.h>~%~
                                  #include <math.h>~%~
                                  #define CLM_C_INSTRUMENT 1~%~
                                  #include ~S~%~%~A~%~
                                  static int got_sigint = 0; /* catch C-C if hung */~%~
                                  static void sig_err(int sig) {got_sigint = sig;}~%~%~
                                  void ~A (int _beg_, int _end_, float *_datar_, int *_datai_)~%{~%"                         
			name (day-name day) date (month-name month) (- year 1900) hour minute 
			(concatenate 'string cmus.h-directory "cmus.h")
			(or *header-info* "") 
			c-ff)))
	;; to get this to work on the Mac, wrap #ifndef MAC #endif around the #include <signal.h> here and in instrument-let below
	(setf <nil> 0) 
	(setf <t> 1)
	#+excl (setf *c-proc* c-ff)
	#+(or kcl mcl) (setf *c-proc* c1-ff)
	(let ((ins-code (walk-form `(progn ,@body) env 'noopfun)))
	  ;; the extra lambda is needed by walk-form
	  
	  ;; in akcl, we have to pass the integer and float arrays as objects to the c function,
	  ;; then dereference them in C, hence the following extra step:
	  #+kcl (format *c-file* "void ~A (int beg, int end, object datr, object dati)~%{~%  ~
                                    ~A (beg,end,(datr->sfa.sfa_self),(dati->fixa.fixa_self));}~%"
			c2-ff c-ff)

	  ;; in mcl, we not only have to put the actual arrays on the heap, we have to pass mcl floats
	  ;; down to MPW C, where we have no guarantee the two floating point layouts will agree.
	  ;; So, the integer array is allocated via a handle and passed through after dereferencing,
	  ;; but the floats are first turned into 64 bit fixed point integer+fraction, 8 bytes per
	  ;; float, passed as an integer array to C, and then unwrapped in C into whatever floating
	  ;; point format MPW likes.  This requires an extra argument telling how many such floats
	  ;; are being passed.
	  #+mcl (format *c-file* "void ~A (int beg, int end, int *bufa, int *bufb, int len)~%{~%  ~
                                    float datar[1024];~%  ~
                                    int i,j;~%  ~
                                    for (j=0,i=0;i<len;i+=2,j++)~%    ~
                                      { datar[j] = (float)(bufa[i]+(float)(bufa[i+1]/32768.0)); }~%  ~
                                    ~A (beg,end,datar,bufb);~%  ~
                                    }~%"
			mcl-name c-ff)
	  (close *c-file*)
	  `(progn
	     #+excl (if (not (probe-file ,l-file-name)) 
			(excl:shell (format nil "cc ~A ~A -o ~A~%" ,c-file-name ,*c-compiler-options* ,l-file-name)))
	     #+excl (load ,l-file-name)
	     #+(and excl Next) (ff:defforeign ',c-ff :entry-point ,excl-c-ff-name 
					      :arguments '(fixnum fixnum array array) :return-type :void)
	     #+(and excl SGI) (ff:defforeign ',c-ff :entry-point ,excl-c-ff-name 
					     :arguments '(integer integer array array) :return-type :void)
	     
	     #+kcl (Clines ,(format nil "~%#include ~S~%" c-file-name))
	     #+kcl (defentry ,c1-ff (int int object object) (void ,c2-ff))

	     #+mcl (ccl:deffcfun (,c-ff ,(symbol-name mcl-name) :check-args t) 
				 ((fixnum :long) (fixnum :long) (t :ptr) (t :ptr) (fixnum :long)) 
				 :novalue)
	     #+mcl (defun ,c1-ff (beg end bufa bufb) 
		     (let ((hd1 (make-handle-f bufa)) 
			   (hd2 (make-handle-i bufb)))
		       (ccl:with-dereferenced-handles ((p1 hd1) (p2 hd2))
			 (,c-ff beg end p1 p2 (length bufb)))
		       (free-handle-f hd1) 
		       (free-handle-i hd2 bufb)))
	     (eval-when (eval load) (pushnew ',name *clm-instruments*))
	     (defun ,name ,args
	       (setf *current-instrument-name* ',name)
	       (if (zerop *clm-interrupted*)
		 ,ins-code))
	     (defun ,silly-name ()
	       (pushnew ',name *clm-instruments*)
	       (set-instrument-properties ',name))
	     (eval-when (eval load) (,silly-name))))))))

#+mcl
(defmacro definstrument (ins-name (&rest args) &body body &environment env)
  ;; if this is ever deleted (i.e. if MCL and C output get together), remember the c-make-array call in make-delay (mus.lisp).
  (declare (ignore env))
  (let ((name (if (listp ins-name) (car ins-name) ins-name)))
    `(defun ,name ,args ,@body)))

#-(and excl cltl2)
(defmacro defcinstrument (ins-name (&rest args) &body body &environment env)
  (if (listp ins-name)
      `(definstrument ,(append ins-name (list :language :c)) ,args ,@body ,env)
    `(definstrument (,ins-name :language :c) ,args ,@body ,env)))

#+(and excl cltl2)
(defmacro defcinstrument (ins-name (&rest args) &body body)
  (if (listp ins-name)
      `(definstrument ,(append ins-name (list :language :c)) ,args ,@body)
    `(definstrument (,ins-name :language :c) ,args ,@body)))

#+56-mus-gens
(defun compile-ins (name)    ;only needed for :56k case
  (compile name)
  (pushnew name *clm-instruments*)
  (set-instrument-properties name)
  name)

#+mcl
(defmacro instrument-let (ins &body body)
  (walk-form
   (if ins
       (let ((instr (car ins)))
	 `(flet ((,(first instr) ,(second instr)
	           (progn
		     ,(third instr))))
	    (instrument-let ,(cdr ins) ,@body)))
    `(progn ,@body))))

;;; now the usual two-step to avoid double expansion -- in this case, it can cause the instruments
;;; after the first to return garbage.

#-(or mcl (and excl cltl2))
(defmacro instrument-let (ins &body body &environment env)
  `(macroexpand-1 (instrument-let-1 ,ins ,@body ,env)))

#+(and excl cltl2)
(defmacro instrument-let (ins &body body)
  `(macroexpand-1 (instrument-let-1 ,ins ,@body)))

#-mcl
(defmacro instrument-let-1 (ins &body body &environment env)
  (walk-form
   (if ins
       (let ((ins-name (first (first ins))))
	 (let* ((*header-info* nil)
		(*c-file-name* nil)
		#+(and excl NeXT) (*c-compiler-options* "-c -O")
		#+(and excl SGI) (*c-compiler-options* "-c -ansi -G 0 -O")
		(name (if (listp ins-name)
			  (apply #'(lambda (nam &key (language #+56-mus-gens :56k #-56-mus-gens :c)
						     (c-file nil)
						     (c-include-file nil)
						     (exp-env t) 
						     #+(and excl NeXT) (c-options "-c -O")
						     #+(and excl SGI) (c-options "-c -ansi -G 0 -O")
						     (save-labels nil))
				     (prog1 nam
				       (setf c56-need-to-save-labels save-labels)
				       (setf c56-need-to-load-expt exp-env)
				       (setf *c-file-name* c-file)
				       (setf *header-info* c-include-file)
				       #+excl (setf *c-compiler-options* c-options)
				       (setf *clm-language* language)))
				 ins-name)
			(progn
			  (setf *clm-language* #+56-mus-gens :56k #-56-mus-gens :c)
			  ins-name))))
	   (if (eq *clm-language* :56k)
	       (progn
		 (setf <t> 0)
		 (setf <nil> #x10000)
		 `(flet ((,name ,(second (first ins))
			   (setf *current-instrument-name* ',name)
			   (if (zerop *clm-interrupted*)
			       ,(third (first ins)))))
		    (set-instrument-properties ',name)
		    (instrument-let-1 ,(cdr ins) ,@body)))
	     (let* ((c-ff (gentemp "CLM_C_"))
		    #+(or kcl mcl) (c1-ff (gentemp "CLM_C_"))
		    #+kcl (c2-ff (gentemp "clm_c_"))
		    #+mcl (mcl-name (intern (concatenate 'string "clm_" (lc name))))
		    #+excl (c-ff-name (symbol-name c-ff))
		    #+(and excl NeXT) (excl-c-ff-name (concatenate 'string "_" c-ff-name))
		    #+(and excl SGI) (excl-c-ff-name c-ff-name)
		    
		    #+kcl (c-file-name (or *c-file-name* 
					   (namestring
					    (merge-pathnames 
					     (concatenate 'string "clm_" (symbol-name name) ".c")
					     si:*load-pathname*))))
		    #+mcl (c-file-name (or *c-file-name* 
					   (namestring 
					    (merge-pathnames 
					     (concatenate 'string "clm:" "clm_" (lc name) ".c")
					     ccl:*loading-file-source-file*))))
		    #+excl (c-file-name (or *c-file-name* 
					    (namestring
					     (merge-pathnames 
					      (concatenate 'string c-ff-name ".c")
					      excl:*source-pathname*))))
		    #+excl (l-file-name (concatenate 'string (subseq c-file-name 0 (- (length c-file-name) 2))  ".o"))
		    )
	
	       (setf *c-file* (open c-file-name :direction :output :if-exists :supersede :if-does-not-exist :create))
	       (princ (format nil "; Writing ~S~%" c-file-name))
	       (multiple-value-bind (second minute hour date month year day daylight-saving-p time-zone)
				    (get-decoded-time)
		 (declare (ignore time-zone daylight-saving-p second))
		 (flet ((month-name (month) (nth (- month 1) '("Jan" "Feb" "Mar" "Apr" "May" "Jun" "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")))
			(day-name (day) (nth day '("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun"))))
		   (format *c-file* "/* ~A C file generated by CLM ~A ~D-~A-~D at ~D:~2,'0D */~%~
                                         #include <stdio.h>~%~
                                         #include <signal.h>~%~
                                         #include <math.h>~%~
                                         #define CLM_C_INSTRUMENT 1~%~
                                         #include ~S~%~%~A~%~
                                         static int got_sigint = 0; /* catch C-C if hung */~%~
                                         static void sig_err(int sig) {got_sigint = sig;}~%~%~
                                         void ~A (int _beg_, int _end_, float *_datar_, int *_datai_)~%{~%"                         
			       name (day-name day) date (month-name month) (- year 1900) hour minute
			       (concatenate 'string cmus.h-directory "cmus.h")
			       (or *header-info* "") 
			       c-ff)))
	       (setf <nil> 0) 
	       (setf <t> 1)
	       #+excl (setf *c-proc* c-ff)
	       #+(or kcl mcl) (setf *c-proc* c1-ff)
	       (let ((ins-code (walk-form `(progn ,(third (first ins))) env 'noopfun))
		     (silly-name (gensym)))
		 #+kcl (format *c-file* "void ~A (int beg, int end, object datr, object dati)~%{~%  ~
                                        ~A (beg,end,(datr->sfa.sfa_self),(dati->fixa.fixa_self));}~%"
			       c2-ff c-ff)
		 #+mcl (format *c-file* "void ~A (int beg, int end, int *bufa, int *bufb, int len)~%{~%  ~
                                         float datar[128];~%  ~
                                         int i,j;~%  ~
                                         for (j=0,i=0;i<len;i+=2,j++)~%    ~
                                           { datar[i] = (float)(bufa[i]+(float)(bufa[i+1]/32768.0)); }~%  ~
                                         ~A (beg,end,datar,bufb);~%  ~
                                         }~%"
			       mcl-name c-ff)
		 (close *c-file*)
		 `(progn
		    (when (not (member ',silly-name *clm-instruments*))
		      (push ',silly-name *clm-instruments*)
		      #+excl (if (not (probe-file ,l-file-name)) 
				 (excl:shell (format nil "cc ~A ~A -o ~A~%" ,c-file-name ,*c-compiler-options* ,l-file-name)))
		      #+excl (load ,l-file-name)
		      #+(and excl Next) (ff:defforeign ',c-ff :entry-point ,excl-c-ff-name 
						       :arguments '(fixnum fixnum array array) :return-type :void)
		      #+(and excl SGI) (ff:defforeign ',c-ff :entry-point ,excl-c-ff-name 
						      :arguments '(integer integer array array) :return-type :void)
		      
		      #+kcl (Clines ,(format nil "~%#include ~S~%" c-file-name))
		      #+kcl (defentry ,c1-ff (int int object object) (void ,c2-ff))
		      ;; unfortunately, this can't work because defentry expands to nil when not at top level

		      #+mcl (ccl:deffcfun (,c-ff ,(symbol-name mcl-name) :check-args t) 
			      ((fixnum :long) (fixnum :long) (t :ptr) (t :ptr) (fixnum :long)) 
			      :novalue)
		      #+mcl (defun ,c1-ff (beg end bufa bufb) 
			      (let ((hd1 (make-handle-f bufa)) 
				    (hd2 (make-handle-i bufb)))
				(ccl:with-dereferenced-handles ((p1 hd1) (p2 hd2))
				  (,c-ff beg end p1 p2 (length bufb)))
				(free-handle-f hd1) 
				(free-handle-i hd2 bufb)))
		      )
		    (flet ((,name ,(second (first ins))
			     (setf *current-instrument-name* ',name)
			     (if (zerop *clm-interrupted*)
				 ,ins-code)))
		      (set-instrument-properties ',name)
		      (instrument-let-1 ,(cdr ins) ,@body))))))))
     `(progn ,@body))))



;;; PHRASING
;;;
;;; 56k version in sched.lisp, c version in cmus.lisp


(defun end-run (&optional phrase)
  #+56-mus-gens (if (eq (ins-language) :56k) (end-run_56 phrase) (end-run_c phrase))
  #-56-mus-gens (end-run_c phrase)
  )

(defun make-phrase (&optional arg) 
  #+56-mus-gens (if (eq (ins-language) :56k) (make-phrase_56 arg) (make-phrase_c arg))
  #-56-mus-gens (make-phrase_c arg)
  )

(defun wait-for-phrase (&rest phrases) 
  #+56-mus-gens (if (eq (ins-language) :56k) (apply #'wait-for-phrase_56 phrases) (apply #'wait-for-phrase_c phrases))
  #-56-mus-gens (apply #'wait-for-phrase_c phrases)
  )

(defun phrase-value (phrase var) 
  #+56-mus-gens (if (eq (ins-language) :56k) (phrase-value_56 phrase var) (phrase-value_c phrase var))
  #-56-mus-gens (phrase-value_c phrase var)
  )

(defun setf-phrase-value (phrase var val) 
  #+56-mus-gens (if (eq (ins-language) :56k) (setf-phrase-value_56 phrase var val) (setf-phrase-value_c phrase var val))
  #-56-mus-gens (setf-phrase-value_c phrase var val)
  )

(defsetf phrase-value setf-phrase-value)

(defun phrase (p &rest vars) 
  #+56-mus-gens (if (eq (ins-language) :56k) (apply #'phrase_56 p vars) (apply #'phrase_c p vars))
  #-56-mus-gens (apply #'phrase_c p vars)
  )




