;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;
;;; this file tries to pull together the various DAC-related functions, all of
;;; which exist in C (sgi.c, next.c, etc).
;;;
;;; the main functions here are start-dac, stop-dac, wait-for-dac, volume.
;;; Volume is an function of no arguments that returns a number between 0 and 1.

(in-package :clm)

(defvar *dac-call-direct* t)
(defvar *dac-check-args* nil)
(defvar *dac-prototyped* t)   ; Should be nil only if C implementation is not



;;; ------------------------------------------------------------------------
;;; NeXT 
;;;

#+(and Excl NeXT)
  (progn					;kcl version is in kcl-clm.lisp
    (ff:defforeign 'get-dac-filter :entry-point "_getdacfilter" :arguments nil       :return-type :integer)
    (ff:defforeign 'set-dac-filter :entry-point "_setdacfilter" :arguments '(fixnum) :return-type :integer)
    (ff:defforeign 'get-dac-zero-fill :entry-point "_getdaczerofill" :arguments nil  :return-type :integer)
    (ff:defforeign 'set-dac-zero-fill :entry-point "_setdaczerofill" :arguments '(fixnum) :return-type :integer)
    (ff:defforeign 'ur-get-volume  :entry-point "_getvolume"    :arguments '(array)  :return-type :integer)
    (ff:defforeign 'ur-set-volume  :entry-point "_setvolume"    :arguments '(array)  :return-type :integer)
    (ff:defforeign 'start-dac      :entry-point "_startdac"     :prototype t
		   :arguments '(string fixnum single-float single-float) :return-type :integer)
    (ff:defforeign 'stop-dac       :entry-point "_abortdac"     :arguments nil       :return-type :void)
    (ff:defforeign 'abort-dac      :entry-point "_abortdac"     :arguments nil       :return-type :void)
    (ff:defforeign 'wait-for-dac   :entry-point "_waitfordac"   :arguments nil       :return-type :void)
    (ff:defforeign 'dac-is-in-use  :entry-point "_dacisrunning" :arguments nil       :return-type :integer)
    (ff:defforeign 'set-dac-ramp   :entry-point "_setramp"      :arguments '(fixnum) :return-type :integer)
    (ff:defforeign 'set-dac-host   :entry-point "_sethost"      :arguments '(string) :return-type :integer)

    ;(ff:defforeign 'start-dac* :entry-point "_start_dac" :arguments '(string fixnum fixnum fixnum fixnum fixnum array) :return-type :integer)
    ;(ff:defforeign 'abort-dac* :entry-point "_abort_dac" :arguments nil :return-type :void)
    )


#+NeXT 
  (progn
    (defun DSP-who ()
      (with-open-file (who "/tmp/dsp.who" :if-does-not-exist nil) 
	(if who 
	    (read-line who)
	  (format nil "DSP not in use")))
      ;; here we could also call (excl:shell "ps xxxx") where xxxx = the process number returned by dsp.who
      ;; (both as strings) -- the 2nd line returned (if any) from ps is the name and status of that process
      ;; in kcl, it's (lisp:system "ps xxxx")
      )

    (defconstant volume-min 0)
    (defconstant volume-max #x2b)		;/usr/include/nextdev/snd_snd.h

    (defun get-volume ()
      (let* ((arr (make-array 3 :element-type 'fixnum :initial-element 0))
	     (err (ur-get-volume arr)))
	(values (aref arr 0) (aref arr 1) (aref arr 2) err)))

    (defun set-volume (on left right)
      (let ((arr (make-array 3 :element-type 'fixnum))
	    (err 0))
	(setf (aref arr 0) on)
	(setf (aref arr 1) (max volume-min (min left volume-max)))
	(setf (aref arr 2) (max volume-min (min right volume-max)))
	(setf err (ur-set-volume arr))
	err))

    (defun volume ()
      (multiple-value-bind
	  (on left right)
	  (get-volume)
	(if (= 0 on) 0.0
	  (float (/ (* .5 (+ left right)) volume-max)))))

    (defun setf-volume (val) 
      (if (/= val 0.0)
	  (let ((tval (floor (* val volume-max))))
	    (set-volume 1 tval tval))
	(set-volume 0 0 0)))

    (defsetf volume setf-volume)


    (defun dac-filter () (get-dac-filter))
    (defun setf-dac-filter (val) (set-dac-filter val))
    (defsetf dac-filter setf-dac-filter)


    ;; the error returned by the sound-start-playing functions is defined in /usr/include/sound/sounderror.h
    (defun sound-error (i)
      (case i 
	(0 nil)
	(1 "not sound")	(2 "bad format") (3 "bad sampling rate") (4 "bad channel") (5 "bad size")
	(6 "bad filename") (7 "cannot open") (8 "cannot write") (9 "cannot read") (10 "cannot alloc")
	(11 "cannot free") (12 "cannot copy") (13 "cannot reserve") (14 "not reserved") (15 "cannot record")
	(16 "already recording") (17 "not recording") (18 "cannot play") (19 "already playing") (20 "not playing")
	(21 "not implemented") (22 "connot find") (23 "cannot edit") (24 "bad space, man") (25 "kernel error")
	(26 "bad configuration (maybe /usr/lib/sound trouble)") (27 "cannot configure (check that /usr/lib/sound is ok)")
	(28 "under-run") (29 "aborted") (30 "bad tag")
	(31 "cannot access") (32 "timeout") (33 "busy") (34 "cannot abort") (35 "too much info")
	(otherwise "unknown error")))
    )



;;; ------------------------------------------------------------------------
;;; SGI


#+(and excl SGI) 
  (progn
    (defun dsp-who () "can't say")

     (ff:defforeign-list
       `((start-dac      :entry-point  "urdac"        
			  :arguments    (string) 
			  :return-type  :integer
			  :call-direct  ,*dac-call-direct*
			  :arg-checking ,*dac-check-args*
			  :prototype    ,*dac-prototyped*)	
	 (ur-get-volume  :entry-point  "getvolume"    
			 :arguments    nil  
			 :return-type  :integer
			 :call-direct  ,*dac-call-direct*
			 :arg-checking ,*dac-check-args*
			 :prototype    ,*dac-prototyped*)	
	 (ur-set-volume  :entry-point  "setvolume"    
			 :arguments    (integer)  
			 :return-type  :integer
			 :call-direct  ,*dac-call-direct*
			 :arg-checking ,*dac-check-args*
			 :prototype    ,*dac-prototyped*)
	 ))

     (defconstant volume-min 0)
     (defconstant volume-max 255)

     (defun get-volume ()
       (let ((vol (ur-get-volume)))
	 (values 1 vol vol 0)))

     (defun set-volume (on left &optional (right 0))
       (declare (ignore on))
       (ur-set-volume (max volume-min (min (max left right) volume-max)))
       0)

     (defun volume () (float (/ (ur-get-volume) volume-max)))
     (defun setf-volume (val) (set-volume t (floor (* volume-max val))))
     (defsetf volume setf-volume)

     (defun stop-dac () (warn "can't stop the dac yet..."))
     (defun abort-dac () (stop-dac))
     (defun dac-is-in-use () 0)
     (defun wait-for-dac () nil)
     (defun set-dac-ramp () )
     (defun set-dac-host () )
     (defun get-dac-filter () 0)
     (defun set-dac-filter (filt) (declare (ignore filt)))
     (defun dac-filter () 0)
     (defsetf dac-filter set-dac-filter)
     (defun get-dac-zero-fill () 0)
     (defun set-dac-zero-fill (zero) (declare (ignore zero)))
     (defun sound-error (i) (declare (ignore i)))
     )



;;; ------------------------------------------------------------------------
;;; Mac


#+MCL 
  (progn

    ;(ccl:deffcfun (c-get-volume "get_volume") () :long)
    ;(ccl:deffcfun (c-set-volume "set_volume") ((fixnum :long) (fixnum :long)) :long)
    ;;(ccl:deffcfun (start-dac "start_dac") ((string)) :long)
    ;; doesn't work yet 
    (defun start-dac (name) (declare (ignore name)))
    (defun dsp-who () "unknown")

    (defun wait-for-dac () )
    (defun dac-is-in-use () 0)

    (defun stop-dac () (warn "can't stop the dac yet..."))
    (defun abort-dac () (stop-dac))

    (defun get-dac-filter () 0)
    (defun set-dac-filter (filt) (declare (ignore filt)))

    (defconstant volume-min 0)
    (defconstant volume-max 245)	;using new Sound Manager 3.0 controls

    ;(defun volume () (float (/ (logand (c-get-volume) #xffff) volume-max)))
    ;(defun setf-volume (val) (let ((vol (floor (* volume-max val)))) (c-set-volume vol vol)))
    ;(defsetf volume setf-volume)

    (defun dac-filter () 0)
    ;(defsetf dac-filter set-dac-filter)

    (defun sound-error (i) (declare (ignore i)))
    )
