;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-
;;;
(in-package :clm)
;;;
;;; Code generator for 56000, using the assembler in dsp56.lisp and the tree walker in walk.lisp, the
;;; intermediate code generator in run.lisp (where the tree is linearized and temporaries identified),
;;; the "monitor" and library in lib56.lisp, and various NeXT related stuff in next.lisp and next56.c.
;;;
;;; The 56000 really wants fractional data (i.e. -1.0<=n<1.0). However,
;;; we can't assume that all intermediate signals are fractional.  For example, a phase increment
;;; can easily be between -pi and pi (i.e. the Nyquist limit), so if limit ourselves to -1 to 1, we
;;; can't get frequencies over srate/two-pi.  Worse, in FM, there is no upper limit to the "index" --
;;; although you get fold-over if the phase increment is over pi, this is not necessarily a dumb
;;; thing to do.  So, we might naively choose a mixed-number format of sign-bit, 7 bits of integer, and
;;; 16 of fraction.  Unfortunately, for low frequencies, or very slow envelopes, 16 bits of fractional
;;; part is woefully inadequate.  And table lookup tables can be large -- up to 4096 words on this chip, and
;;; any size in principle, and we can expect FM of the table-lookup pointer, so once again we're
;;; dealing with arbitrarily large numbers.  Can't win with integers in this business.  But floats are
;;; way too slow.  If we use mixed numbers, we end up with an explosion of numerical operations.  So,
;;; special cases (like slow and very long ramps) are handled before being passed to the chip, and we
;;; actually use only 3 types on chip: 24 bit integers, 24 bit fractions, and 48 bit "reals" -- i.e. high word is signed integer,
;;; and low is unsigned fraction. (there are other types that pop up -- 48 bit integer as the pass counter 
;;; (rather than keep track of wrap-around), for real*real, a 48 bit number consisting of signed 24 bit integer 
;;; and signed 24 bit fraction, etc)
;;;
;;; Have to keep in mind the variety of hidden shifts
;;; built into MOVE commands, and when sign-extension and zero-filling occurs -- this chip is a bit of a mess.
;;; LSH is not full width, but ASH is; MAC MACR MPY MPYR can't handle X1 X1 or Y1 Y1; and so on...
;;;
;;; we have the accumulators A and B, split-able into 6 parts.
;;;         the operand registers X and Y, split-able 4 ways.
;;;         the address registers R0..7, N0..7, M0..7
;;;             These can be used as general data, but I can't see any advantage over memory
;;;             because there are no wait states, and load/store can be done in parallel (i.e. no cost)
;;; for each of these, we need to know who is in it, and its status (dead/alive, saved/not-saved)
;;; need register spill, allocation, update (for "aliasing")
;;;      temporary memory allocation and release (for stored temps)
;;;      permanent memory allocation (for user vars like envelope data, etc)
;;;           load sequence decisions, etc  type conversions if mixed
;;;      operations like * on mixed numbers of both sizes and integers

(defvar Library-Load-List nil)

(defun need-loaded (&rest args) 
  (loop for i in args do
    (if (not (member i Library-Load-List)) 
	(push i Library-Load-List))))



;;; REGISTER HANDLING -----------------------------------------------------------------------

(defun A-acc (a1) (member a1 '(A A2 A0 A1 BA A10 AB)))
(defun B-acc (a1) (member a1 '(B B2 B0 B1 BA B10 AB)))
(defun X-reg (a1) (member a1 '(X X0 X1)))
(defun Y-reg (a1) (member a1 '(Y Y0 Y1)))
(defun Any-ALU-reg (a1) (data-register a1))    ; in dsp56.lisp (the assembler)
(defun Any-AGU-reg (a1) (address-register a1)) ; "address generation unit"
(defun Any-Control-reg (a1) (control-register a1))
(defun high-side (a1) (member a1 '(X1 Y1 A1 B1)))
(defun low-side (a1) (member a1 '(X0 Y0 A0 B0)))
(defun any-side (a1) (member a1 '(X Y A B)))
(defun ext-side (a1) (member a1 '(A2 B2)))
(defun get-N-reg (r) (case r (R0 'N0) (R1 'N1) (R2 'N2) (R3 'N3) (R4 'N4) (R5 'N5) (R6 'N6) (R7 'N7)))
(defun get-M-reg (r) (case r (R0 'M0) (R1 'M1) (R2 'M2) (R3 'M3) (R4 'M4) (R5 'M5) (R6 'M6) (R7 'M7)))

(defun get-high-side (a1)
  (cond ((A-acc a1) 'A1) 
	((B-acc a1) 'B1) 
	((X-reg a1) 'X1) 
	((Y-reg a1) 'Y1) 
	((listp a1) (if (eq (car a1) 'L) (append (list 'X) (cdr a1)) a1))
	(t a1)))

(defun get-low-side (a1)
  (cond ((A-acc a1) 'A0) 
	((B-acc a1) 'B0) 
	((X-reg a1) 'X0) 
	((Y-reg a1) 'Y0) 
	((listp a1) (if (eq (car a1) 'L) (append (list 'Y) (cdr a1)) a1)) 
	(t a1)))
	
(defun get-ext-side (a1)
  (if (A-acc a1) 'A2 
    (if (B-acc a1) 'B2 
      nil)))

(defun get-full-reg (a1)
  (cond ((A-acc a1) 'A) 
	((B-acc a1) 'B) 
	((X-reg a1) 'X) 
	((Y-reg a1) 'Y) 
	((listp a1) (append (list 'L) (cdr a1))) 
	(t a1)))

(defvar N-values (make-array 8 :element-type 'fixnum :initial-element -1))
(defun getf-N-reg (reg) (aref N-values (reg-num reg)))
(defun setf-N-reg (reg val) (setf (aref N-values (reg-num reg)) val))

(defun init-N-regs ()
  (loop for i from 0 to 7 do (setf (aref N-values i) -1)))


(defvar stored-regs nil)

(defun init-stored-regs () 
  (setf stored-regs nil))

(defun add-stored-reg (val) 
  (push val stored-regs))

(defun check-stored-regs () 
  (loop while stored-regs do (restore-stored-reg (pop stored-regs))))

(defun restore-stored-reg (data)
  (let* ((reg (first data))
	 (n-reg (get-n-reg reg))
	 (addr (second data))
	 (var (third data)))
    (DEBUGGING (push `(----------------> restore locally stored ,reg from ,addr for ,var (at ,(fourth data))) pp))
    (if (plusp (reg-cost reg)) (error "~A has been allocated to ~A but we need him for ~A" reg (reg-list reg) var))
    (if (not (r-register reg)) (error "~A stored locally?!?" reg))
    (get-register reg)
    (push `(LOAD ,reg X ,addr) pp)
    (push `(LOAD ,n-reg Y ,addr) pp)
    (setf-n-reg n-reg -1)
    (update-var-and-reg var nil (get-type_56 var) 'L 'S reg)))

(defun remove-stored-reg-at (addr)
  (when stored-regs 
    (let ((val (find-if #'(lambda (data) (= (second data) addr)) stored-regs)))
      (setf stored-regs (remove-if #'(lambda (data) 
				       (= (second data) addr)) 
				   stored-regs))
      (fourth val))))


;;; temporaries and user-vars

(defun temp-sig (x) (member x new-sig))

(defun make-temp-sig ()
  (let ((var (new-signal)))
    (push var new-sig)
    var))

(defun add-var (name type address work &optional indirect element-type hidden-scale) 
  (setf (gethash name vars) (list type address (if work (list work) nil) indirect element-type hidden-scale)))

(defun sym-type (n) (first (gethash n vars)))
(defun sym-home (n) (second (gethash n vars)))
(defun sym-work (n) (third (gethash n vars)))
(defun sym-indirect (n) (fourth (gethash n vars)))
(defun sym-element-type (n) (fifth (gethash n vars)))
(defun sym-scale (n) (sixth (gethash n vars)))
(defun syml-type (n) (first n))
(defun syml-home (n) (second n))
(defun syml-work (n) (third n))
(defun syml-indirect (n) (fourth n))
(defun syml-element-type (n) (fifth n))
(defun syml-scale (n) (sixth n))

(defun get-appropriate-r-reg (typ)
  (case typ
    ((osc smp rdin flt spd sr fftflt conv wt rblk) (get-register 'R2))
    ((sw tbl noi randi table x-table y-table array) (get-register 'R3))
    ((dly zdly cmbflt allpassflt) (get-register 'R4))
    (envelope (get-register 'R7))
    (t (get-temporary-r-register))))

(defun check-for-stored-vars (home reg &optional (cur-type 'real) reg-if-indirect)
  (if (and home 
	   (not (numberp home))
	   (or (null (third home)) 
	       (and (R-register (second home)) 
		    (eq cur-type 'real))))
      home
    (if (R-register reg)
	(if (not (numberp home))
	    (if (and reg-if-indirect
		     (not (eq reg reg-if-indirect)))
		(progn
		  (get-register reg-if-indirect)
		  (push `(COPY ,reg ,reg-if-indirect) pp)
		  (list 'L reg-if-indirect 'R))
	      (list 'L reg 'R))
	  (let ((nreg (get-temporary-r-register (if reg-if-indirect (list reg-if-indirect)))))
	    (DEBUGGING (push `(----------------> get stored address at ,home) pp))
	    (let ((mem (or (remove-stored-reg-at home) 'L)))
	      ;; we need to use the stored memory type, if any, because pre-defined (mus.lisp) structures
	      ;;  with integer and fraction fields, sometimes use Y memory (rather than X), and if we
	      ;;  return L or nil here, we get X as the default later (upon <setf> for example)
	      (push `(LOAD ,(get-N-reg nreg) Y ,home) pp)
	      (setf-N-reg (get-N-reg nreg) -1)
	      (push `(LOAD ,nreg X ,home) pp)
	      (list mem nreg 'RN))))
      (if (and (null home) (null reg))
	  nil
	(let* ((reg-loc (or reg (second home)))
	       (r-reg (if (r-register reg-loc) ;can only happen if home = '(mem Rn RN), since reg /= r-register
			  (get-appropriate-r-reg cur-type)
			(get-temporary-r-register))))
	  (DEBUGGING (push `(----------------> restore addr ,reg-loc to ,r-reg) pp))
	  (if (r-register reg-loc)
	      (if (eq r-reg reg-loc)
		  (push `(UPDATE ,r-reg R+N) pp)
		(push `(LUA ,r-reg ,reg-loc R+N) pp))
	    ;; that is, if we have a pointer to a structure that is computed (via AREF for example)
	    ;; and we need the address directly (via Rn), not indirectly (via Rn+Nn), we reduce
	    ;; the expression and leave it in the register desired. If r-reg = reg-loc we use UPDATE.
	    (if (not (listp reg-loc))
		(push `(LOAD ,r-reg X ,reg-loc) pp)
	      (return-from check-for-stored-vars (list 'tref reg-loc))))
	  (if (and (null reg) (not (r-register reg-loc)))
	      (progn
		(push `(LOAD ,(get-N-reg r-reg) Y ,reg-loc) pp)
		(setf-N-reg (get-N-reg r-reg) -1) ;no idea what it might be -- play it safe...
		(list (first home) r-reg (third home)))
	    (list 'L r-reg 'R)))))))

(defun get-home-address (name &optional (cur-type 'real) reg-if-indirect) 
  ;; if reg-if-indirect is given, get-home-address must return the address in that register (or in a home address)
  (let* ((data (gethash name vars))
	 (addr (check-for-stored-vars (syml-home data) (syml-indirect data) cur-type reg-if-indirect))
	 (type (syml-type data)))
    (if (and (eq (car addr) 'L)
	     (member type '(fraction integer)))
	(append '(X) (cdr addr))
      addr)))

(defun get-work-addresses (name) (sym-work name))

(defun get-work-address (name) (car (sym-work name)))

(defun get-element-type (name) (sym-element-type name))

(defun get-any-address (name)
  (if (numberp name)
      name
    (or (get-work-address name) 
	(get-home-address name)
	(error "~S doesn't have a home!" name))))

(defun operand-reg (addr)	
  (or (A-acc addr) (B-acc addr) (X-reg addr) (Y-reg addr)))

(defun Reg-addr (addr)
  (and addr
      (or (operand-reg (car addr))
	  (reg-addr (cdr addr)))))

(defun in-register (var)
  (Reg-addr (get-work-addresses var)))
 
(defun in-AB-already (var)
  (let ((wrk (get-work-addresses var)))
    (or (member 'A wrk)
	(member 'B wrk))))

(defun in-AB-or-get-AB (var)
  (let ((wrk (get-work-addresses var)))
    (if (member 'A wrk) (get-register 'A)
      (if (member 'B wrk) (get-register 'B)
	(get-temporary-register '(A B))))))

(defun add-work-address (name new-addr)
  (push new-addr (third (gethash name vars))))

(defun remove-work-address (name addr)
  (if (sym-work name)
      (let ((l-addr (if (A-acc addr) '(A A0 A1 A2 A10)
		      (if (B-acc addr) '(B B0 B1 B2 B10)
			(if (eq addr 'X) '(X X0 X1)
			  (if (eq addr 'Y) '(Y Y0 Y1)
			    (if (X-reg addr) (list 'X addr)
			      (if (Y-reg addr) (list 'Y addr)
				(list addr)))))))))
	(setf (third (gethash name vars)) (set-difference (sym-work name) l-addr)))))

(defun get-type_56 (name) 
  (if name
      (if (listp name)
	  (first name)			;user struct, possibly -- otherwise an error since we can't handle normal lists
	(if (not (constantp name))
	    (sym-type name)
	  (if (integerp name) 'integer
	    (if (and (/= name 0.0) (< (abs name) 1.0)) 'fraction 'real))))))

(defun remove-var (name) 
  (remhash name vars))


;;; register allocation
;;;   each register keeps a list of what it currently houses:
;;;   (name [its address findable through emit tables] live/dead save/not-to-be-saved)
;;;   if this list is nil, the register is completely idle
;;;                if has data, but all are dead, not-to-be-saved, is idle
;;;                if needed, save any that need saving, mark as idle, return to caller
;;;   reg-cost = how many stores will be needed to free this guy up, and how many loads later to restore
;;;     that is dead-not saved costs 1, live-not-saved costs 2, live-saved costs 1, dead-saved costs 0.

(defvar <A> nil)
(defvar <B> nil)
(defvar <X> (make-array 3 :initial-element nil))
(defvar <Y> (make-array 3 :initial-element nil))
(defvar <R> (make-array 8 :initial-element nil))
(defvar <N> (make-array 8 :initial-element nil))
(defvar <M> (make-array 8 :initial-element nil))

(defun free-reg (r)			;set R free
  (case r
    ((A A0 A1 A2 A10) (setf <A> nil))
    ((B B0 B1 B2 B10) (setf <B> nil))
    (X (loop for i from 0 to 2 do (setf (aref <X> i) nil)))
    (Y (loop for i from 0 to 2 do (setf (aref <Y> i) nil)))
    ((X0 X1) (setf (aref <X> (+ 1 (reg-num r))) nil))
    ((Y0 Y1) (setf (aref <Y> (+ 1 (reg-num r))) nil))
    ((R0 R1 R2 R3 R4 R5 R6 R7) (setf (aref <R> (reg-num r)) nil))
    ((N0 N1 N2 N3 N4 N5 N6 N7) (setf (aref <N> (reg-num r)) nil))
    ((M0 M1 M2 M3 M4 M5 M6 M7) (setf (aref <M> (reg-num r)) nil)))
  r)

(defun free-all-regs ()
  (free-reg 'A) (free-reg 'B) (free-reg 'X) (free-reg 'Y)
  (init-stored-regs)
  (loop for i from 0 to 7 do
    (setf (aref <R> i) nil)
    (setf (aref <N> i) nil)
    (setf (aref <M> i) nil)))

(defun mark-reg (r l)			;mark R as housing L
  (case r
    ((A A0 A1 A2 A10) (push l <A>))
    ((B B0 B1 B2 B10) (push l <B>))
    (X (push l (aref <X> 0)))
    (Y (push l (aref <Y> 0)))
    ((X0 X1) (push l (aref <X> (+ 1 (reg-num r)))))
    ((Y0 Y1) (push l (aref <Y> (+ 1 (reg-num r)))))
    ((R0 R1 R2 R3 R4 R5 R6 R7) (push l (aref <R> (reg-num r))))
    ((N0 N1 N2 N3 N4 N5 N6 N7) (push l (aref <N> (reg-num r))))
    ((M0 M1 M2 M3 M4 M5 M6 M7) (push l (aref <M> (reg-num r))))))

(defun reg-list (r)			;return current use-list of R
  (case r
    ((A A0 A1 A2 A10) <A>)
    ((B B0 B1 B2 B10) <B>)
    (X (append (aref <X> 0) (aref <X> 1) (aref <X> 2)))
    (X0 (append (aref <X> 1) (aref <X> 0)))
    (X1 (append (aref <X> 2) (aref <X> 0)))
    (Y (append (aref <Y> 0) (aref <Y> 1) (aref <Y> 2)))
    (Y0 (append (aref <Y> 1) (aref <Y> 0)))
    (Y1 (append (aref <Y> 2) (aref <Y> 0)))
    ((R0 R1 R2 R3 R4 R5 R6 R7) (aref <R> (reg-num r)))
    ((N0 N1 N2 N3 N4 N5 N6 N7) (aref <N> (reg-num r)))
    ((M0 M1 M2 M3 M4 M5 M6 M7) (aref <M> (reg-num r)))))

(defun update-var-and-reg (var reg typ &optional (live nil) (store 'NS) (indirect nil))
  (if (gethash var vars)
      (if indirect
	  (progn
	    (remhash var vars)
	    (add-var var typ nil reg indirect))
	(if reg
	    (push reg (third (gethash var vars)))
	  (setf (third (gethash var vars)) nil)))
    (add-var var typ nil reg indirect))
  (let* ((true-reg (or reg indirect))
	 (regl (reg-list true-reg)))
    (if regl
	(let ((regl1 (remove var regl :key #'first)))
	  (free-reg true-reg)
	  (if regl1 (mark-reg true-reg regl1))))
    (if live
	(mark-reg true-reg (list var live store))
      (if (member var true-user-var)
	  (mark-reg true-reg (list var 'L store))
	(mark-reg true-reg (list var 'D store))))))
  
(defun change-mark (reg sig live store)
  (let ((reglist (reg-list reg)))
    (loop for i in reglist do
      (if (eq (car i) sig)
	  (progn
	    (setf (second i) live)
	    (setf (third i) store))))))

(defun kill-temp (sig)
  ;; if kill-temp is called, and followed by either spill-ALU-registers or get-register,
  ;; the sig can be left without any acertainable address -- be sure either to save the
  ;; current address before spilling, or move the killed sig before any other action.
  ;; (of course, if sig is not needed anymore, this warning doesn't apply).
  ;; Also, if the only address is in a register that is about to be spilled, and
  ;; kill-temp is called beforehand, even saving the address won't help -- in other
  ;; words, kill-temp is dangerous and should really be called only when sig is no
  ;; longer needed, but I don't always follow that discipline below.
  (let ((reglist (get-work-addresses sig))
	(ih (second (gethash sig vars)))
	(ir (fourth (gethash sig vars))))
    (if (not (numberp ih))
	(progn
	  (if ir (change-mark ir sig 'D 'NS))
	  (if (and ih (r-register (second ih))) (change-mark (second ih) sig 'L 'NS))))
    (loop for i in reglist do
      (change-mark i sig 'D 'NS))))

;;; r-list -> ((nam L/D S/NS) ...)
(defun reg-cost-l (l &optional (cost 0))
  (if (null l) 
      cost
    (reg-cost-l (cdr l) (+ cost
			   (if (eq 'L (second (car l))) 1 0)
			   (if (eq 'S (third (car l))) 1 0)))))

(defun reg-cost (r)			;how much will it cost to grab R
  (reg-cost-l (reg-list r)))

(defun free-one (r l)
  ;; there are lots of possibilities here -- a value can have a home address of the form
  ;; (mem addr) (mem is one of L X Y, addr is the actual numerical address)
  ;; (mem Rn typ) (Rn contains the address of the variable, typ is the indexing offset type (RN or R)
  ;; nil --  In this case, (fourth (gethash)) can have:
  ;;    Rn -- Rn contains the variable's address, or
  ;;    addr -- addr contains the address (i.e. Rn shadowed in a temporary storage location).
  ;; (mem addr typ) is the most complicated case -- here we have an indexed offset from location whose address
  ;;   is in addr -- that is the variable lives at memory location (value (+ typ (value addr))) -- double
  ;;   indirection (or is it triple?) arises when the reference is to a field of a structure in an array of
  ;;   structures, and, due to expression complexity, we are forced to save the field address for some period.
  ;;   In this case (X addr) = R value, (Y addr) = N value
  (if (eq (third l) 'S)			;store value
      (let ((addr (get-home-address (car l))))
	(DEBUGGING (push `(----------------> free up ,r by storing ,l (at ,addr) ,@(gethash (car l) vars)) pp))
	(if (or (null addr) (R-register (second addr)))
	    (let ((new-addr (get-L-mem))
		  (indr (or (and (null addr) (fourth (gethash (car l) vars)))
			    (and addr (R-register (second addr)) (second addr))))
		  (l-type (get-type_56 (car l))))
	      (if indr
		  (progn
		    (if (null addr)
			(setf (fourth (gethash (car l) vars)) new-addr)
		      (progn
			(push `(STORE ,(get-N-reg indr) Y ,new-addr) pp)
			(add-stored-reg (list indr new-addr (car l) (if (and (listp addr) (member (first addr) '(L X Y))) (first addr))))
			(DEBUGGING (push `(----------------> remember ,indr locally ,new-addr for ,(car l)) pp))
			(setf (second (gethash (car l) vars)) new-addr)))
		    (setf addr (list 'X new-addr)))
		(progn
		  (if (member l-type '(fraction integer))
		      (setf addr (list 'X new-addr))
		    (setf addr (list 'L new-addr)))
		  (setf (second (gethash (car l) vars)) addr)))))
	(if (and (null (fourth (gethash (car l) vars)))
		 (member (get-type_56 (car l)) '(integer fraction)))
	    (if (null (get-work-addresses (car l))) 
		(error "inconsistent addresses: ~A is holding ~A, but ~A does not know about it (~A)"
		       r (reg-list r) (car l) (gethash (car l) vars))
	      (push `(STORE ,(get-work-address (car l)) ,@addr) pp))
	  (push `(STORE ,r ,@addr) pp))))
  (remove-work-address (car l) r))	;tell var that he's no longer in this register

(defun get-register (r)
  (let ((r-l (reg-list r)))
    (loop while r-l do (free-one r (pop r-l)))
    (free-reg r)))

(defun spill-ALU-registers (&optional (regs '(A B X Y)))
  (loop for i in regs do
    (get-register i)))

(defun get-temporary-register (&optional (l '(A B X Y)) (d nil))
  (do* ((reg nil)
	(lcost 100)
	(cost 100)
	(r-l l (cdr r-l))
	(i (car r-l) (car r-l)))
      ((or (= lcost 0) (null i))
       (if (or (null reg) (member reg d))
	   (if (equal l '(A))
	       (get-register 'A)
	     (error "impossible register request: ~S ~S => ~S" l d reg))
	 (get-register reg)))
    (if (or (null d)
	    (not (member i d)))
	(progn
	  (setf lcost (reg-cost i))
	  (if (< lcost cost)
	      (progn
		(setf reg i)
		(setf cost lcost)))))))

(defun get-temporary-R-register (&optional l d)
  (get-temporary-register (or l r-registers) d))

(defun get-result-r-register (result)
  (find result r-registers :test #'(lambda (nam r) (if (eq nam (caar (reg-list r))) r))))
    


;;; JUMP OPTIMIZATION -------------------------------------------------------------------------------
;;;
;;; 12 bit jump addresses can be squeezed into one instruction (rather than use the extension word).
;;; Because an undefined label is building a fixup chain until it is defined, we can't play fancy
;;; games in the assembler, and since NeXT has mapped out most of the low address memory 
;;; (they go from $1FF directly to $2000), and since we have no way of knowing whether we are going
;;; to fit into low memory, we try to catch as many short jumps here.   This involves short look
;;; aheads (since we use jumps in lieu of a skip instruction).

(defconstant op-jumps-short '(JMP JSR 
			      JCC JSCC JHS JSHS JGE JSGE JNE JSNE JPL JSPL JNN JSNN JEC JSEC JLC JSLC JGT JSGT
			      JCS JSCS JLO JSLO JLT JSLT JEQ JSEQ JMI JSMI JNR JSNR JES JSES JLS JSLS JLE JSLE))

(defun last-word (op)
  (nth (- (length op) 1) op))

(defun loop-eq (lim lab prog)
  (do ((i 0 (+ 1 i)))
      ((or (>= i lim) (eq (caar (nth i prog)) lab))
       (< i lim))))

(defun found-label-within-low-memory (lab prog)
  (and (< dsp-pc (- Internal-P-size 20))
       (loop-eq 10 lab prog)))

(defun append-short (op)
  (list (append (car op) '(SHORT))))

(defun optimize-short-jumps (op prog)
  (if (and (member (caar op) op-jumps-short)
	   (not (eq 'SHORT (last-word (car op)))))
      (let* ((lab (last-word (car op)))
	     (loc (car (or (gethash lab labels) '(UNDEFINED)))))
	(if (not (eq 'UNDEFINED loc))
	    (if (< loc (expt 2 12))	;short jump address field is 12 bits wide
		(append-short op)
	      op)
	  (if (found-label-within-low-memory lab prog)
	      (append-short op)
	    op)))
    op))


;;; another case we should optimize is (ADD A B) (TRANSFER B A) and its relatives

(defun optimize-mpy-add (op ur-prog)
  ;; this is called at the very last minute, just before actually issuing the machine code
  ;; the only subsequent change is the assembler's insertion of NOP for pipeline settling
  (if (and (eq (first (first op)) 'JMP)
	   (eq (second (first op)) (first (first (first ur-prog)))))
      ;; (JMP label) (label) comments out the JMP
      (list (list '----------------> 'optimize (first op)))
    (let* ((prog (loop for next-op in ur-prog 
		  if (not (member (first (first next-op)) '(COMMENT ---------------->)))
		  return next-op))
	   (no-parallel-moves (and (null (second op)) 
				   (null (second prog))))
	   (prog-op (first (first prog)))
	   (mpy-add (and no-parallel-moves
			 (eq (first (first op)) 'MPY) 
			 ;;forget about MPYR and SMPYR for now [not generated by the compiler]
			 (member prog-op '(ADD SUB))
			 ;; this is safe because type conversions will separate the multiply and the add in all
			 ;; incompatible cases -> MPY X1 X1 A, convert to real, ADD A B
   	                 ;; MPY x y [A or B] and ADD x [A or B], so here we're asking if ADD's x = MPY's A or B
			 (eq (fourth (first op)) (second (first prog)))))
	   (store-load (and no-parallel-moves
			    (not mpy-add)
			    (eq (first (first op)) 'STORE)
			    (eq prog-op 'LOAD)
			    (eq (second (first op)) (second (first prog)))
			    (eq (third (first op)) (third (first prog)))
			    (eq (fourth (first op)) (fourth (first prog))))))
	(when mpy-add
	  (setf (first (first op)) (if (eq prog-op 'ADD) 'MAC 'SMAC))
	  (setf (fourth (first op)) (third (first prog))))
        (when (or mpy-add store-load)
	  (setf (first prog) `(----------------> optimize out ,(copy-list (first prog)))))
        op)))


;;; PARALLEL MOVE OPTIMIZATION --------------------------------------------------------------------------
;;;
;;; in many cases LOAD STORE COPY and UPDATE can be done in parallel with arithmetic operations.
;;; We ignore that possibility until the very last moment, just before assembling the program (via EMIT in dsp56)

(defun combine-two-ops (l1 l2) 
  (append l1 (list l2)))

(defun combine-three-ops (l1 l3) 
  (append l1 (list l3)))

(defun call-emit (prog)
  (if (null prog) (error "user program is empty!"))
  (loop for i = (pop prog) while i do
    (apply #'emit (optimize-mpy-add (optimize-short-jumps i prog) prog))))

(defconstant op-can-move '(ABS ADC ADD ADDL ADDR ASL ASR CLR CMP CMPM MAC SMAC MACR SMACR
			   MPY SMPY MPYR SMPYR NEG RND SBC SUB SUBL SUBR TFR TST AND EOR
			   LSL LSR NOT OR ROL ROR LOAD STORE COPY TRANSFER UPDATE))

(defconstant op-jumps-memory '(JCLR JSET JSCLR JSSET))
	
(defun op-makes-new-block (op)
  (or (member op op-jumps-short)
      (member op op-jumps-memory)
      (member op '(DO REP RESET RTS RTI STOP SWI WAIT))))

(defconstant pseudo-op '(HIDE UNDEFINE DEFINE SET-DATA X-DATA Y-DATA X-ORG Y-ORG ORG COMMENT ---------------->))

(defun op-code-p (op)
  (or (member op op-can-move)
      (op-makes-new-block op)
      (member op pseudo-op)
      (member op '(DIV NORM TCC THS TGE TNE TPL TNN TEC TLC TGT TCS TLO TLT TEQ TMI TNR TES TLS TLE
		   ANDI ORI BCLR BSET BCHG BTST ENDDO LUA MOVE NOP))))

(defun true-op (op)
  (and (not (member op pseudo-op))
       (op-code-p op)))

(defun op-can-be-moved (op)		;incoming op (a list, not a list of lists)
  (and (member (car op) '(LOAD STORE COPY UPDATE))
       (not (any-control-reg (second op)))
       (not (any-control-reg (third op)))
       (not (member (third op) '(X-IO Y-IO P)))))

(defun ok-move-op (op)
  (and (not (eq (third op) 'L))
       (not (address-register (second op)))
       (not (numberp (third op)))
       (or (null (fourth op)) (not (numberp (fourth op))))))

(defun op-can-accommodate-move (op)	;op here is a list of lists (in opt-prog, not old prog)
  (and (true-op (caar op))
       (or
	(and (null (second op))		;just one instruction -- no parallel moves here
	     (member (caar op) op-can-move)
	     ;; another case we don't take advantage of is (NOP) used for pipeline delays followed
	     ;;  by some command that doesn't use the R regs (or whatever) -- too hard to make sure
	     ;;  clobbering the NOP is safe.
	     (not (eq (caar op) 'UPDATE))
	     (or (not (eq (caar op) 'COPY)) ;that is, it can be turned into TFR if needed (L mem in 2nd for example)
		 (and (member (third (first op)) '(A B))
		      (member (second (first op)) '(A B X0 X1 Y0 Y1))))
	     (or (not (member (caar op) '(LOAD STORE)))
		 (ok-move-op (car op))))
	(and (null (third op))		;already one field taken -- check for I U L
	     (second op)
	     (ok-move-op (second op))))))
    
(defun new-basic-block-p (op)
  (or (op-makes-new-block (car op))
      (not (op-code-p (car op)))))	;i.e. it's a label, so all bets are off

(defun memory-type (mem)
  (if (numberp mem) 
      mem
    (case mem
      (X 'x-memory)
      (Y 'y-memory)
      (L 'l-memory)
      (P 'p-memory)
      ((X-IO Y-IO) 'io-memory)
      (t (error "invalid memory for parallelism: ~S" mem)))))

(defun full-name (dest) (if (data-register dest) (get-full-reg dest) dest))

(defun analyze-move-op (instruction)	;LOAD etc -- return (dest source)
  (let ((op (car instruction)))
    (case op
      (STORE (values op (memory-type (third instruction)) 
		     (second instruction) 
		     (if (R-register (fourth instruction)) (fourth instruction) nil)))
      (LOAD (values op (full-name (second instruction))
		    (memory-type (third instruction))
		    (if (R-register (fourth instruction)) (fourth instruction) nil)))
      ((TFR TRANSFER COPY UPDATE) (values op (full-name (third instruction)) (second instruction) nil))
      (t (error "not a legal parallel move possibility: ~S" instruction)))))

(defconstant single-ac-ops '(ABS ASL ASR CLR NEG NORM RND TST LSL LSR NOT ROL ROR UPDATE LUA))
(defconstant double-ac-ops '(ADC ADD ADDL ADDR CMP CMPM DIV SBC SUB SUBL SUBR TFR TRANSFER COPY
			     AND ANDI EOR OR ORI BCLR BSET BCHG 
			     TCC THS TGE TNE TPL TNN TEC TLC TGT TCS TLO TLT TEQ TMI TNR TES TLS TLE))
(defconstant triple-ops '(MPY MPYR MAC MACR SMPY SMPYR SMAC SMACR))
		
(defun dest-of (op)
  (if (member (car op) single-ac-ops) 
      (second op)
    (if (member (car op) double-ac-ops)
	(third op)
      (if (member (car op) triple-ops)
	  (fourth op)
	(if (member (car op) '(LOAD STORE))
	    (multiple-value-bind
		(oper dest src) (analyze-move-op op)
	      (declare (ignore oper src))
	      dest)
	  nil)))))

(defun affected-reg (reg)
  (case reg
    ((A A1 A0 A2 A10) '(A0 A1 A2 A A10))
    ((B B1 B0 B2 B10) '(B0 B1 B2 B B10))
    (X '(X0 X1 X))
    (Y '(Y0 Y1 Y))
    ((X-MEMORY Y-MEMORY L-MEMORY P-MEMORY IO-MEMORY) nil)
    ((R0 N0 M0) '(R0 N0 M0))
    ((R1 N1 M1) '(R1 N1 M1))
    ((R2 N2 M2) '(R2 N2 M2))
    ((R3 N3 M3) '(R3 N3 M3))
    ((R4 N4 M4) '(R4 N4 M4))
    ((R5 N5 M5) '(R5 N5 M5))
    ((R6 N6 M6) '(R6 N6 M6))
    ((R7 N7 M7) '(R7 N7 M7))
    (t (list reg))))

(defun reg-intersection (reg1 reg2)
  (intersection (affected-reg reg1) (affected-reg reg2)))

(defun op-member (ref instruction)
  (if (member (car instruction) '(LOAD STORE))
      (multiple-value-bind
	  (op dest src r-reg) (analyze-move-op instruction)
	(declare (ignore op))
	(or (reg-intersection ref dest)
	    (and r-reg (eq ref r-reg))
	    (reg-intersection ref src)))
    (intersection (affected-reg ref) instruction)))

(defun any-part-of (opt-op ref)
  (or (op-member ref (first opt-op))
      (and (second opt-op)
	   (op-member ref (second opt-op)))
      (and (third opt-op)
	   (op-member ref (third opt-op)))))

(defun any-dest-of (opt-op ref)
  (let ((full-ref (affected-reg ref)))
    (or (member (dest-of (first opt-op)) full-ref)
	(and (second opt-op)
	     (member (dest-of (second opt-op)) full-ref))
	(and (third opt-op)
	     (member (dest-of (third opt-op)) full-ref)))))

(defun named-ref (n)
  (if (integerp n) 
      n
    (multiple-value-bind 
	(val exists) (gethash n names)
      (if exists val -1))))

(defun eq-named-ref (n1 n2)
  (let ((val1 (named-ref n1))
	(val2 (named-ref n2)))
    (and (/= val1 -1)
	 (= val1 val2))))

(defun mem-dest (op1 op2)		;here we trap LOAD A X 64, STORE B L 64, and so on (including R reg ref)
  (and (member (car op1) '(LOAD STORE))
       (member (car op2) '(LOAD STORE))
       (or (eq (car op1) 'STORE)	        ;if both are LOADs, order is not important
	   (eq (car op2) 'STORE))
       (member (third op1) '(X Y L X-IO Y-IO))	;both must be memory refs, not immediate
       (member (third op2) '(X Y L X-IO Y-IO))
       (or (eq (third op1) 'L)		        ;if one is X, other Y, no collision
	   (eq (third op2) 'L) 
	   (eq (third op1) (third op2)))
       (or (eq (fourth op1) (fourth op2))       ;if they reference different locations, no collision
	   (R-register (fourth op1))	        ;if either is indirect, we assume the worst
	   (R-register (fourth op2))
	   (eq-named-ref (fourth op1) (fourth op2)))))

(defun any-mem-dest-of (opt-op par-op)
  (or (mem-dest (first opt-op) par-op)
      (and (second opt-op) 
	   (mem-dest (second opt-op) par-op))))

(defun legal-Y-R-1 (op1 op2 op1-op op1-src op1-dest op2-op op2-src op2-dest)
  (declare (ignore op2-op op1))
  (and (eq op1-op 'COPY)
       (or (and (member (second op2) '(A B Y0 Y1))
		(or (eq (third op2) 'Y) (numberp (third op2)))
		(member op1-src '(A B))
		(member op1-dest '(X0 X1)))
	   (and (member op2-src '(A B))
		(eq op2-dest 'y-memory)
		(member op1-dest '(A B))
		(eq op2-src op1-dest)
		(eq op1-src 'Y0)))))

(defun legal-X-R-1 (op1 op2 op1-op op1-src op1-dest op2-op op2-src op2-dest)
  (declare (ignore op1-op op2))
  (and (eq op2-op 'COPY)
       (or (and (member (second op1) '(A B X0 X1))
		(or (eq (third op1) 'X) (numberp (third op1)))
		(member op2-src '(A B))
		(member op2-dest '(Y0 Y1)))
	   (and (member op1-src '(A B))
		(eq op1-dest 'x-memory)
		(member op2-dest '(A B))
		(eq op1-src op2-dest)
		(eq op2-src 'X0)))))

(defun split-address-registers (r1 r2)
  (and r1 r2
       (R-register r1)
       (R-register r2)
       (or (and (<= 0 (reg-num r1) 3) (<= 4 (reg-num r2) 7))
	   (and (<= 0 (reg-num r2) 3) (<= 4 (reg-num r1) 7)))))

(defun split-address-modes (m1 m2)
  (and m1 m2
       (member m1 '(R-1 R+1 R R+N))
       (member m2 '(R-1 R+1 R R+N))))

(defun legal-X-Y-1 (op1 op2 op1-op op1-src op1-dest op2-op op2-src op2-dest)
  (declare (ignore op1-src op2-src))
  (and (member op1-op '(LOAD STORE))
       (member op2-op '(LOAD STORE))
       (eq (third op1) 'X)
       (eq (third op2) 'Y)
       (member (second op1) '(X0 X1 A B))
       (member (second op2) '(Y0 Y1 A B))
       (not (eq op1-dest op2-dest))
       (split-address-registers (fourth op1) (fourth op2))
       (split-address-modes (fifth op1) (fifth op2))))

(defun legal-X-R (move-op cur-op m-op m-src m-dest op src dest)
  (or (legal-X-R-1 move-op cur-op m-op m-src m-dest op src dest)
      (legal-X-R-1 cur-op move-op op src dest m-op m-src m-dest)))

(defun legal-Y-R (move-op cur-op m-op m-src m-dest op src dest)
  (or (legal-Y-R-1 move-op cur-op m-op m-src m-dest op src dest)
      (legal-Y-R-1 cur-op move-op op src dest m-op m-src m-dest)))

(defun legal-X-Y (move-op cur-op m-op m-src m-dest op src dest)
  (if (or (eq m-src 'x-memory) (eq m-dest 'x-memory))
      (legal-X-Y-1 move-op cur-op m-op m-src m-dest op src dest)
    (legal-X-Y-1 cur-op move-op op src dest m-op m-src m-dest)))

(defun incompatible-moves (move-op cur-op op dest src)
  ;; question here is: can a double parallel move be accomplished with move-op and (op dest src) as moves
  (or (eq (car move-op) 'UPDATE)
      (eq op 'UPDATE)
      (eq (third move-op) 'L)
      (eq dest 'l-memory)
      (eq src 'l-memory)
      (and (numberp (third move-op)) (numberp src)) ;that is, are both Immediate moves
      (and (eq op 'COPY) (eq (car move-op) 'COPY))
      (multiple-value-bind
	  (m-op m-dest m-src) (analyze-move-op move-op)
	(or (reg-intersection dest m-dest)
	    (and (or (eq src m-src) (eq src m-dest))
		 (member src '(x-memory y-memory)))
	    (and (eq dest m-src) 
		 (member dest '(x-memory y-memory)))
	    (and 
	     (not (legal-X-R move-op cur-op m-op m-src m-dest op src dest))
	     (not (legal-Y-R move-op cur-op m-op m-src m-dest op src dest))
	     (not (legal-X-Y move-op cur-op m-op m-src m-dest op src dest)))))))

(defun compatible-moves (move-op cur-op op dest src)
  (not (incompatible-moves move-op cur-op op dest src)))

(defun op-blocks-back-scan (opt-op op dest src r-reg)
  (and (true-op (caar opt-op))
       (or (any-part-of opt-op dest)
	   (any-dest-of opt-op src)
	   (and r-reg (any-dest-of opt-op r-reg))
	   (any-mem-dest-of opt-op op))))

(defun parallel-move-fits (opt-op cur-op op dest src)
  (and (op-can-accommodate-move opt-op)
       (or (and (null (second opt-op))
		(or (not (member (caar opt-op) '(LOAD STORE COPY)))
		    (compatible-moves (first opt-op) cur-op op dest src)))
	   (and (second opt-op)
		(not (member (caar opt-op) '(LOAD STORE COPY)))
		(compatible-moves (second opt-op) cur-op op dest src)))))

(defun optimize-parallel-moves (prog)
  
  ;; PROG is our first pass at the 56000 code -- a list of instructions (in order, not reversed).
  ;; here we run down it trying where-ever possible to take advantage of the parallel moves.

  (let ((opt-prog nil)
	(cur-op nil)
	(stop 0))
    (loop while prog do
      (setf cur-op (pop prog))
      (cond ((new-basic-block-p cur-op)	         ;can't back up over basic block boundary
	     (setf stop 0)
	     (push (list cur-op) opt-prog))
	    ((or (not (op-can-be-moved cur-op))
		 (member (car cur-op) pseudo-op)
		 (= stop 0))
	     (push (list cur-op) opt-prog)
	     (incf stop))
	    (t
	     (let ((ok1-list nil)
		   (ok2-list nil))
	       (multiple-value-bind
		   (parallel-op parallel-dest parallel-source parallel-r-reg) (analyze-move-op cur-op)
		 (do* ((i 0 (1+ i))
		       (opt-op (car opt-prog) (nth i opt-prog)))
		     ;;opt-op is current op being checked for compatibility with parallel move request
		     ((or (>= i stop)	;stop marks start of basic block
			  (op-blocks-back-scan opt-op cur-op parallel-dest parallel-source parallel-r-reg))
		      ;;result of DO has to check for blocked-back-scan upon source used as dest in parallel move etc
		      (if (and (or (not parallel-r-reg) (not (any-dest-of opt-op parallel-r-reg)))
			       (not (any-dest-of opt-op parallel-source))
			       (not (any-dest-of opt-op parallel-dest))
			       (parallel-move-fits opt-op cur-op parallel-op parallel-dest parallel-source))
			  (if (cdr opt-op)
			      (push i ok2-list)
			    (push i ok1-list))
			(if (and parallel-r-reg 
				 (any-dest-of opt-op parallel-r-reg))
			    ;; that is, direct setting of R reg takes 2 instructions to settle in pipeline
			    (if (and ok1-list (= (car ok1-list) (- i 1)))
				(pop ok1-list)
			      (if (and ok2-list (= (car ok2-list) (- i 1)))
				  (pop ok2-list))))))
		   ;;this is the DO body (keeps 2 lists of optimization possibilities)
		   (if (parallel-move-fits opt-op cur-op parallel-op parallel-dest parallel-source)
		       (if (cdr opt-op)
			   (push i ok2-list)
			 (push i ok1-list)))))
	       (if ok2-list
		   (setf (nth (car ok2-list) opt-prog) 
		     (combine-three-ops (nth (car ok2-list) opt-prog) cur-op))
		 (if ok1-list
		     (setf (nth (car ok1-list) opt-prog) 
		       (combine-two-ops (nth (car ok1-list) opt-prog) cur-op))
		   (progn
		     (push (list cur-op) opt-prog)
		     (incf stop))))))))
    opt-prog))


;;; BASIC MOVE OPERATIONS ------------------------------------------------------------------------------
;;; remember that A -> X1 moves A1, that A <- L-mem gets sign right but X <- L-mem doesn't, and so on

(defun clear (addr)
  (DEBUGGING (push `(----------------> clear ,addr) pp))
  (if (or (eq addr 'A) (eq addr 'B))
      (push `(CLR ,addr) pp)
    (if (listp addr)			;(clear (X addr))
	(let ((reg (get-temporary-register)))
	  (clear reg)
	  (push `(STORE ,reg ,@addr) pp))
      (if (member addr '(X Y))
	  (progn
	    (push `(LOAD ,(get-high-side addr) 0) pp)
	    (push `(LOAD ,(get-low-side addr) 0) pp))
	(push `(LOAD ,addr 0) pp)))))


(defun move (a1 a2)
  (DEBUGGING (push `(----------------> move ,a1 ,a2) pp))
  (if (or (null a1) (null a2)) (error "address lost: ~S ~S" a1 a2))
  (if (not (equal a1 a2))
      (if (numberp a1)
	  (load-number a2 a1)
	(if (listp a1)			;a1 might be '(L 123) or '(X 12 SHORT) etc (or Y R3 R+N)
	    (if (listp a2)		;memory to memory
		(let ((reg (get-temporary-register (if (eq (car a1) 'L) '(A B) '(A B X0 X1 Y0 Y1)))))
		  (push `(LOAD ,reg ,@a1) pp)
		  (push `(STORE ,reg ,@a2) pp))
	      (push `(LOAD ,a2 ,@a1) pp)) ;memory to register
	  (if (listp a2)		;register to memory
	      (push `(STORE ,a1 ,@a2) pp)
	    ;; next register to register with check for A->X and friends (which have to be done in two steps)
	    (if (and (member a1 '(A B X Y))
		     (member a2 '(A B X Y))
		     (or (member a1 '(X Y)) (member a2 '(X Y))))
		(progn
		  (if (member a2 '(A B))
		      (push `(COPY ,(get-high-side a1) ,a2) pp)
		    (push `(COPY ,(get-high-side a1) ,(get-high-side a2)) pp))
		  (push `(COPY ,(get-low-side a1) ,(get-low-side a2)) pp))
	      ;; here we must protect against the incredible stupidity of the COPY A B case
	      (if (and (member a1 '(A B)) (member a2 '(A B)))
		  (push `(TRANSFER ,a1 ,a2) pp)
		(if (member a1 '(X Y))	;check for COPY X X1
		    (push `(COPY ,(get-high-side a1) ,a2) pp)
		  (push `(COPY ,a1 ,a2) pp)))))))))
                  ;; this lets through COPY A1 A, for example, because that isn't a no-op (sets/clears A2, clears A0)



;;; SHIFTS --------------------------------------------------------------------------------------------

(defun ash-acc (acc num)
  (if (not (zerop num))
      (progn
	(push (list 'REP (abs num)) pp)
	(if (plusp num)
	    (push (list 'ASL (get-full-reg acc)) pp)
	  (push (list 'ASR (get-full-reg acc)) pp)))))

(defun lsh-acc (acc num)
  (if (not (zerop num))
      (progn
	(push (list 'REP (abs num)) pp)
	(if (plusp num)
	    (push (list 'LSL (get-full-reg acc)) pp)
	  (push (list 'LSR (get-full-reg acc)) pp)))))

(defun ash-shift (reg num)
  (if (or (A-acc reg) (B-acc reg))
      (ash-acc reg num)
    (let ((acc (get-temporary-register '(A B))))
      (move reg acc)
      (ash-acc acc num)
      (move acc reg))))

(defun mpy-shift (reg num &optional (tmp-reg nil))
  ;; right shift n, constant = 2^(-n).  So 4 bit right shift of X0:
  ;; (LOAD X1 #x080000)			;5-th bit down is on (fraction) (SHORT if n < 8) (remember sign bit)
  ;; (MPY X0 X1 A)                      ;result in A1
  ;;
  ;; left shift constant is 2^(n-1) (fractional mpy introduces 1 bit lsh for free)
  ;; (LOAD X1 #x8)			;4-bit left shift = 2^3 = 8 (integer) (LONG move, not SHORT)
  ;; (MPY X0 X1 A)                      ;result in A0
  (DEBUGGING (push `(----------------> shift ,reg ,num) pp))
  (if (not (zerop num))
      (let ((acc (if (or (A-acc reg) (B-acc reg)) 
		     reg 
		   (get-temporary-register '(A B))))
	    (r-reg (if tmp-reg tmp-reg reg))
	    (t-reg (get-temporary-register '(X0 X1 Y0 Y1) (if tmp-reg (list reg tmp-reg) (list reg)))))
	(if (and (not tmp-reg) (not (X-reg reg)) (not (Y-reg reg)))
	    (progn
	      (setf r-reg (get-temporary-register '(X0 X1 Y0 Y1) (list reg t-reg)))
	      (move reg r-reg)))
	(if (plusp num)			;left shift
	    (let ((shf (expt 2 (- num 1))))
	      (push `(LOAD ,t-reg ,shf) pp)
	      (push `(MPY ,r-reg ,t-reg ,acc) pp)
	      (if (not (eq acc reg)) 
		  (move (get-low-side acc) reg)))
	  (let ((n (if (> (abs num) 7) 
		       (expt 2 (- 23 (abs num)))
		     (expt 2 (- 7 (abs num))))))
	    (if (> (abs num) 7)
		(push `(LOAD ,t-reg ,n) pp)
	      (push `(LOAD ,t-reg ,n SHORT) pp))
	    (push `(MPY ,t-reg ,r-reg ,acc) pp)
	    (if (not (eq acc reg))
		(move (get-high-side acc) reg)))))))

(defun shift (reg num &optional (tmp-reg nil))
  (if (not (zerop num))
      (if (> (abs num) 2)
	  (mpy-shift reg num tmp-reg)
	(ash-shift reg num))))

(defun shift-left-24 (acc)
  (push `(COPY ,(get-high-side acc) ,(get-ext-side acc)) pp)
  (push `(COPY ,(get-low-side acc) ,(get-high-side acc)) pp)
  (push `(LOAD ,(get-low-side acc) 0) pp))

(defun shift-right-24 (acc)
  (push `(COPY ,(get-high-side acc) ,(get-low-side acc)) pp)
  (push `(COPY ,(get-ext-side acc) ,(get-high-side acc)) pp))

(defun shift-left-23 (acc)
  (shift acc 23))

(defun shift-right-23 (acc &optional (tmp-reg nil))
  (shift acc -23 tmp-reg))



;;; TYPE CONVERSIONS -------------------------------------------------------------------------------------

(defun real-to-integer (a1 a2)		;this case is easy because sign is already there
  (move (get-high-side a1) a2))		;i.e real's integer part is in high reg 24 bits A1 X1 etc

(defun fraction-to-real (a1 a2)
  (let ((addr1 (if (or (A-acc a2) (B-acc a2))
		   (get-full-reg a2)
		 (if (or (A-acc a1) (B-acc a1))
		     (get-full-reg a1)
		   (get-temporary-register '(A B)))))
	(t-reg (if (or (X-reg a1) (Y-reg a1)) a1 (get-temporary-register '(X0 X1 Y0 Y1)))))
    (if (not (eq t-reg a1)) (move a1 t-reg))
    (shift-right-23 addr1 t-reg)	;we want sign extension
    (if (not (eq addr1 (get-full-reg a2))) (move addr1 a2))))

(defun real-to-fraction (a1 a2)
  (let ((addr1 (if (> run-safety 0) 
		   (get-temporary-register '(A))
		 (if (or (A-acc a2) (B-acc a2))
		     (get-full-reg a2)
		   (if (or (A-acc a1) (B-acc a1))
		       (get-full-reg a1)
		     (get-temporary-register '(A B))))))
	(aa1 (if (listp a1) (cdr a1) nil)))
    (if (or (not (listp a1))
	    (and (listp aa1)
	         (not (R-register (car aa1)))
		 (> (car aa1) 63)))	;JCLR only works on first 64 locations, or index through Rn
	(progn
	  (if (listp a1)		;real is in memory (not register), but its address must be too large
	      (progn			;so move it down to temp-loc
		(push `(LOAD ,addr1 L ,@aa1) pp)
		(push `(STORE ,addr1 L temp-loc) pp))
	    (push `(STORE ,a1 L temp-loc) pp))
					;real is in register, but JCLR n #D is broken, at least in our 56000's
	  (setf aa1 '(temp-loc))))
    ;;the following code has been through numerous revisions! -- so many bugs that more are likely

    (when (> run-safety 0) 
      (need-loaded '.overflow-warning)
      (push `(LOAD A L ,@aa1) pp)
      (push '(STORE B L temp-loc-2) pp)
      (get-register 'B)
      (push `(LOAD B 1) pp)
      (push `(CMPM A B) pp)
      (push `(JGT no-error) pp)
      (push `(LOAD B1 15) pp)
      (push `(JSR .overflow-warning) pp)
      (push `(no-error LOCAL) pp)
      (push '(LOAD B L temp-loc-2) pp))

    (push `(LOAD ,addr1 Y ,@aa1) pp)	;STORE L puts Y=frac part, X=int part (i.e. A1->X and A0->Y)
					;sets extension register, if high bit is on (i.e. frac part>.5)
    (push `(JCLR 23 X ,@aa1 pos) pp)	;if negative, we need only make room for sign bit (ASR below)
					;but must make sure that -1.0 => -1.0 real to fraction
    (push `(LOAD ,(get-ext-side addr1) #xff SHORT) pp)
    (push `(JMP ok) pp)			;now extension register is set even if -1.0 as real (this clips to -1.0)
    (push `(pos LOCAL) pp)
    (push `(LOAD ,(get-ext-side addr1) 0 SHORT) pp)
					;if positive, clobber extension, then shift over for sign bit
    (push `(ok LOCAL) pp)
    (push `(ASR ,addr1) pp)		;make it a fraction 
    (move addr1 a2)))


(defun long-to-integer (a1 a2)
  (move (get-low-side a1) a2))

(defun integer-to-long (a1 a2)
  (let ((addr1 (if (or (A-acc a2) (B-acc a2))
		   (get-full-reg a2)
		 (if (or (A-acc a1) (B-acc a1))
		     (get-full-reg a1)
		   (get-temporary-register '(A B))))))
    (move a1 addr1)
    (shift-right-24 addr1)		;we want sign extension
    (if (not (eq addr1 (get-full-reg a2))) 
	(move addr1 a2))))


(defun convert (addr1 type1 addr2 type2)
  ;; convert X which is of type TYPE1 and lives in ADDR1 to TYPE2 and put it in ADDR2
  ;; assume ADDR1 and ADDR2 are of proper size, and are free.
  ;; types are: Integer, Fraction, Real, Long-Int (48 bit integer)
  ;; bit patterns are: Integer sign+23 bits, masked to 0 if 2^24
  ;;                   fraction sign+23 bits fraction
  ;;                   real sign+23 bits integer+24 bits fraction
  ;; A few routines in effect use intermediate types (real multiply for example), but we'll treat
  ;; these as purely internal -- no attempt to optimize.
  ;;
  ;;   X -> A sign extended
  ;;   X1 -> A sign extended with 0 -> A0 (X1 ends up in A1)
  ;;   X1 -> A1 just affects A1 -- no sign extension, no zero fill in A0
  ;;   X1 -> A10 not sign extended, but A0=0
  ;;   X1 -> A2 LSB only, clipped
  ;;   A -> X1 clipped, A1 -> X1 not clipped, A2->X1 LSB with sign extension
  ;;   #xx -> An Bn (explicit n), Rn Nn puts xx in LSB, clears rest (unsigned integer)
  ;;   #xx -> Xn Yn A B puts xx into MSB, clears rest (signed fraction)
  ;;   X1 -> R1 uses 16 LSB of X1 and zero fills (unsigned int) (same in reverse)
  ;;   X1 -> MR CCR SP is 8 LSB zero filled both ways
  ;;   X1 -> LC LA SR SSH SSL 16 LSB zero filled both ways
  ;;   R1 -> A goes to A1, clears A0 (LSB of A1)

  ;;   #xxxxx -> A goes to A1, clears A0
  ;;   A B A10 B10 are assumed to be L quantity (one datum)
  ;;   AB BA X Y are assumed to be two associated signed fractional quantities (real -- imag)
  ;;   A B AB BA used as source clip and shift, used as destination sign-extend and zero-fill

  ;; LOAD A L n = put X n in A1, Y n in A0, sign extend
  ;; LOAD X L n = put X n in X1, Y n in X0, treat as two quantities

  (DEBUGGING (push `(----------------> convert ,addr1 ,type1 to ,addr2 ,type2) pp))
  (if (or (not (member type1 '(real integer long-int fraction table-header)))
	  (not (member type2 '(real integer long-int fraction))))
      (error "~S at ~S cannot be coerced to ~S at ~S:" type1 addr1 type2 addr2))
  (if (numberp addr1)
      (if (not (listp addr2))		; i.e. put a number in a register
	  (case type2 
	    (integer (load-integer addr2 (round addr1)))
	    (fraction (load-fraction addr2 (float addr1)))
	    (real (load-real addr2 (float addr1)))
	    (long-int (load-long-int addr2 (round addr1))))
	(let ((treg (if (and (member type1 '(integer fraction))
			     (eq type1 type2))
			(get-temporary-register '(X0 X1 Y0 Y1 A B))
		      (get-temporary-register '(X Y A B)))))
	  (case type2 
	    (integer (load-integer treg (round addr1)))
	    (fraction (load-fraction treg (float addr1)))
	    (real (load-real treg (float addr1)))
	    (long-int (load-long-int treg (round addr1))))
	  (move treg addr2)))
    (case type1 
      (integer
       (case type2
	 (integer   (if (and (listp addr1) 
			     (eq 'L (car addr1)) 
			     (not (listp addr2)) 
			     (not (member addr2 '(A B X Y))))
			(move (append '(Y) (cdr addr1)) addr2)
		      (if (and (listp addr2)
			       (eq 'L (car addr2)) 
			       (not (listp addr1))
			       (not (member addr1 '(A B X Y))))
			  (move addr1 (append '(Y) (cdr addr2)))
			(move addr1 addr2))))
	 (fraction  (clear addr2))
	 (real      (if (member addr2 '(X Y))
			(progn
			  (clear (get-low-side addr2))
			  (move addr1 (get-high-side addr2)))
		      (move addr1 addr2)))
	 (long-int  (integer-to-long addr1 addr2))))
      (fraction
       (case type2
	 (integer   (clear addr2))
	 (fraction  (move addr1 addr2))
	 (real      (fraction-to-real addr1 addr2))
	 (long-int  (clear addr2))))
      (real 
       (case type2
	 (integer   (real-to-integer addr1 addr2))
	 (fraction  (real-to-fraction addr1 addr2))
	 (real      (move addr1 addr2))
	 (long-int  (integer-to-long addr1 addr2)))) ; i.e. assume integer part is in high side
      (long-int
       (case type2 
	 (integer   (long-to-integer addr1 addr2))
	 (fraction  (clear addr2))
	 ;;       (real      (move (get-low-side addr1) (get-high-side addr2))) 
	 ;; does not clear low side of destination, or sign extension -- 17-Feb-92
	 ;; There should be an error indication upon overflow.
	 ;; But, a valid destination is X reg -- LOAD X Y 12 for example
	 (real      (if (and (listp addr2) (not (listp addr1)))
			(progn
			  (shift-left-24 addr1)
			  (move (get-full-reg addr1) addr2))
		      (if (member addr2 '(A B))
			  (move (get-low-side addr1) (get-full-reg addr2))
			(progn
			  (clear (get-low-side addr2))
			  (move (get-low-side addr1) (get-high-side addr2))))))
	 (long-int  (move addr1 addr2))))
      (table-header
       (let* ((fracloc (if (not (listp addr2)) (get-low-side addr2) (get-temporary-register))))
	 (load-table-value-into-reg fracloc (first (second addr1)))
	 (convert fracloc 'fraction addr2 type2))))))

(defun check-for-fractional-overflow (val addr op)
  (if (numberp val)
      (if (> (abs val) 1.0)
	  (warn "fractional overflow: ~A is ~1,4F" (decode-overflow-op op) val))
    (progn
      (need-loaded '.overflow-warning)
      (push '(STORE A L temp-loc-1) pp)
      (push '(STORE B L temp-loc-2) pp)
      (if (not (eq addr 'A))
	  (if (eq addr 'B)
	      (push '(TRANSFER B A) pp)
	    (progn
	      (get-register 'A)
	      (convert addr (get-type_56 val) 'A 'real))))
      (get-register 'B)
      (push `(LOAD B 1) pp)
      (push `(CMPM A B) pp)
      (push `(JGT no-error) pp)
      (if (/= op 1) (push `(LOAD B1 ,op) pp))
      (push `(JSR .overflow-warning) pp)
      (push `(no-error LOCAL) pp)
      (push '(LOAD A L temp-loc-1) pp)
      (push '(LOAD B L temp-loc-2) pp))))



;;; BASIC ARITHMETIC (add subtract negate multiply divide setf abs max min) ---------------------------------

(defun collapse-constants (lst op)
  (apply op lst))

(defvar alias-list nil)

(defun add-alias (nam var)
  (push nam alias-list)
  (set nam var)
  nil)

(defun alias (var) 
  (member var alias-list))

(defun sort-operands (lst)		;turn aliases into actual values, sort out constants from variables
  (let ((c-l nil) (v-l nil))
    (loop for j in lst do
      (let ((i (if (alias j) (eval j) j)))
	(if (or (temp-sig i) (car-member i typed-user-sig))
	    (push i v-l)
	  (if (or (constantp i)		;it's an explicit constant or ? [was numberp here]
		  (not (known-var i)))
	      (push i c-l)
	    (push i v-l)))))
    (values c-l v-l)))

(defun make-integer (i)			;take 32 (or 29) bit lisp-ish integer and make it a 24 bit 56000 integer
  (logand i #xFFFFFF))			;i.e. clipped -- numbers too big are checked elsewhere

(defmacro mac-integer (i)
  (if (constantp i)
      (make-integer i)
    `(logand ,i #xFFFFFF)))

(defun load-integer (opl c-val)		;load as integer (i.e. into LSB, not MSB)
  (if (and (<= 0 c-val (- (expt 2 8) 1)) 
	   (or (member opl '(A1 A0 B1 B0))
	       (address-register opl)))
      (push `(LOAD ,opl ,(make-integer c-val) SHORT) pp)
    (if (zerop c-val) 
	(if (member opl '(A B)) 
	    (push `(CLR ,opl) pp)
	  (if (member opl '(X0 X1 Y0 Y1))
	      (push `(LOAD ,opl 0 SHORT) pp)
	    (push `(LOAD ,opl 0) pp)))
      (push `(LOAD ,opl ,(make-integer c-val)) pp)))
  'integer)

(defun make-long-int (i)		;32 or 29 bit integer here -- going to 48 on chip
  (values (logand i #xFFFFFF) (logand (ash i -24) #xFFFFFF)))

(defun load-long-int (opl c-val)
  (if (or (not (zerop c-val))
	  (not (member opl '(A B))))
      (multiple-value-bind 
	  (low high) (make-long-int c-val)
	(push `(LOAD ,opl ,high) pp)
	(push `(LOAD ,(get-low-side opl) ,low) pp))
    (push `(CLR ,opl) pp))
  'long-int)

(defun make-fraction (x)		;24 bit fraction here (this is the data type the 56000 supports)
  (if (>= x 1.0) #x7FFFFF
    (if (zerop x) 0
      (logand (floor (scale-float x 23)) #xFFFFFF))))

(defun make-fraction-1 (x)
  (if (> (abs x) 1.0) (error "non-fractional value being truncated to fit: ~F" x))
  (make-fraction x))

(defmacro mac-fraction (x)
  (if (constantp x)
      (make-fraction x)
    `(if (>= ,x 1.0) #x7FFFFF 
       (logand (floor (scale-float ,x 23)) #xFFFFFF))))

(defun load-fraction (opl c-val)
  (let ((c-frac (make-fraction c-val)))
    (if (zerop c-val)
	(load-integer opl 0)
      (if (and (= 0 (logand c-frac #xFFFF))
	       (member opl '(A B X1 Y1 X0 Y0)))
	  (push `(LOAD ,opl ,(ldb (byte 8 16) c-frac) SHORT) pp)
	(push `(LOAD ,opl ,c-frac) pp))))
  'fraction)

(defun make-unsigned-fraction (x)
  (logand (floor (scale-float (float x) 24)) #xffffff))

(defmacro mac-unsigned-fraction (x)
  (if (constantp x)
      (make-unsigned-fraction x)
    `(logand (floor (scale-float (float ,x) 24)) #xffffff)))

(defconstant biggest-dsp-number (1- (expt 2 23)))
(defconstant smallest-dsp-number (expt 2 -24))

(defun checked-floor (x)
  (if (zerop x)
      (values 0 0.0)
    (if (plusp x)
	(if (< x biggest-dsp-number)
	    (floor x)
	  (values biggest-dsp-number 0.0))
      (if (< x (- biggest-dsp-number))
	  (values (- biggest-dsp-number) 0.0)
	(if (< (abs x) smallest-dsp-number)
	    (values 0 0.0)
	  (floor x))))))

(defun make-real (x)
  (multiple-value-bind 
      (int frac) (checked-floor x)
    (values (make-integer int) (make-unsigned-fraction frac))))

(defun load-real (opl c-val)
  (if (and (zerop c-val)
	   (member opl '(A B)))
      (push `(CLR ,opl) pp)
    (multiple-value-bind
	(int frac) (make-real c-val)
      (if (member opl '(A B))
	  (progn
	    (push `(LOAD ,(get-full-reg opl) ,int) pp) ;get sign extension in extension register if A or B
	    (if (not (zerop frac))
		;; this should be safe because both long and short immediate moves clear A0 and A2
		(push `(LOAD ,(get-low-side opl) ,frac) pp)))
	(progn
	  (push `(LOAD ,(get-high-side opl) ,int) pp) ;can't LOAD X immediate
	  (push `(LOAD ,(get-low-side opl) ,frac) pp)))))
  'real)

(defun load-number (res-reg num)
  (DEBUGGING (push `(----------------> load constant ,num in ,res-reg) pp))
  (cond ((= num 0)         (clear res-reg) 'integer)
	((integerp num)    (if (< (abs num) (ash 1 23)) 
			       (load-integer res-reg num)
			     (load-long-int res-reg num)))
	((< (abs num) 1.0) (load-fraction res-reg num))
	(t                 (load-real res-reg num))))

(defun find-var-in-reg (var-list reg)
  (if var-list
      (if (member (car var-list) (reg-list reg))
	  (car var-list)
	(find-var-in-reg (cdr var-list) reg))
    nil))

(defun find-var-not-in-XY (var-l)
  (if var-l
      (if (and (not (member (car var-l) (reg-list 'X)))
	       (not (member (car var-l) (reg-list 'Y))))
	  (car var-l)
	(find-var-not-in-XY (cdr var-l)))
    nil))

(defun correct-side (typ)
  (case typ
    ((integer fraction) '(A B X0 Y0 X1 Y1))
    ((real long-int) '(A B X Y))))

(defmacro <setf_56> (result-name arg)

  ;; arg is either a constant or a variable [expressions handled by tree-walker].
  ;; result-name is either a temp or user var -- nothing complicated here except type coercions.
  ;; if result-name a temp, we use the aliasing lists.  If a user-var, we don't necessarily store
  ;; anything -- just update its register as live and store, and wait to let it be stored later.
  ;; if types are the same, and val is already in a register, we just update our addresses.

  (DEBUGGING (push `(----------------> <setf> ,result-name ,arg ---- ,result-name ,@(gethash result-name vars)) pp))
  (when arg
    (let ((val (if (alias arg) (eval arg) arg)))
      (let* ((res-type (or (and (gethash result-name vars) 
				(get-type_56 result-name))
			   (and (symbolp val) 
				(get-type_56 val))
			   (and (numberp val)
				(zerop val) 
				(or (get-type_56 result-name)
				    'integer))
			   (and (integerp val)
				'integer)
			   (and (numberp val)
				(<= -1.0 val 1.0)
				'fraction)
			   (and (numberp val)
				'real)
			   (get-type_56 result-name) 
			   'real)))
	(if (eq res-type 'table-header) (setf res-type 'fraction))
	(let* ((res-reg (or (and (symbolp val)
				 (in-register val)
				 (eq res-type (get-type_56 val))
				 (get-work-address val))
			    (and (in-register result-name)
				 (get-work-address result-name))
			    (get-temporary-register (or (correct-side res-type)
							(error "can't setf ~(~A~) to ~(~A~)" res-type (get-type_56 val))))))
	       (cur-type nil))
	  (if (numberp val)		;var:=3 or whatever
	      (if (integerp val)
		  (if (or (null res-type) 
			  (eq res-type 'integer))
		      (setf cur-type (load-integer res-reg val))
		    (if (eq res-type 'long-int)
			(setf cur-type (load-long-int res-reg val))
		      (setf cur-type (load-real res-reg (float val)))))
		(if (and (eq res-type 'fraction)
			 (<= -1.0 val 1.0))
		    (setf cur-type (load-fraction res-reg (float val)))
		  (setf cur-type (load-real res-reg (float val)))))
	    (progn
	      (if (temp-sig val) 
		  (let* ((indl (gethash result-name vars))
			 (ind (if indl (sixth indl)))
			 (val-adr (get-any-address val)))
		    (kill-temp val)	;temp need not be saved before conversion
		    (when (and ind (numberp ind) (= 2 ind)) ;don't ask... (filter coeffs divided by 2, but user doesn't know that)
		      (when (not (member res-reg '(A B)))
			(setf res-reg (get-temporary-register '(A B))))
		      (move val-adr res-reg)
		      (DEBUGGING (push `(----------------> scaled by .5 automatically) pp))
		      (push `(ASR ,res-reg) pp)
		      (push res-reg (third (gethash val vars))))))
	      (convert (get-any-address val)
		       (setf cur-type (get-type_56 val))
		       res-reg
		       (or res-type cur-type))
	      (if (and (not (temp-sig val))
		       (eq (get-full-reg res-reg) 
			   (get-full-reg (get-work-address val))))
		  (remove-work-address val (get-work-address val)))))
	  
	  (if (not (gethash result-name vars))
	      (let* ((addr (get-L-mem))
		     (ltyp (or res-type cur-type))
		     (laddr (if (member ltyp '(integer fraction))
				(list 'X addr)
			      (list 'L addr))))
		(mark-reg res-reg (list result-name 'L 'NS))
		(push `(STORE ,res-reg ,@laddr) pp)
		(add-var result-name ltyp laddr res-reg))
	    (if (fourth (gethash result-name vars))
		(progn
		  (DEBUGGING (push `(----------------> indirect ,result-name ,res-type ,@(gethash result-name vars)) pp))
		  (if (eq 'table-header (get-type_56 result-name))
		      (let* ((ind-reg (or (get-result-r-register result-name) (get-temporary-r-register)))
			     (tab-mem (get-element-type result-name))
			     (home-reg (if (not tab-mem) (get-temporary-r-register r-registers (list ind-reg))))
			     (home (car (fourth (gethash result-name vars)))))
			(if (not tab-mem) (push `(LOAD ,home-reg X ,home) pp))

			(DEBUGGING (push `(----------------> ,ind-reg => ,(reg-list ind-reg)) pp))
			(if (not (eq result-name (caar (reg-list ind-reg)))) (push `(LOAD ,ind-reg Y ,home) pp) (free-reg ind-reg))
			(if (not tab-mem)
			    (progn
			      (push `(JCLR 0 Y ,home-reg R setf-x-side) pp)
			      (push `(STORE ,res-reg Y ,ind-reg R) pp)
			      (push `(JMP setf-got-data) pp)
			      (push `(setf-x-side LOCAL) pp)
			      (push `(STORE ,res-reg X ,ind-reg R) pp)
			      (push `(setf-got-data LOCAL) pp))
			  (push `(STORE ,res-reg ,tab-mem ,ind-reg R) pp)))
		    (move res-reg (get-home-address result-name)))
		  (if (temp-sig result-name) (kill-temp result-name)))
	      (let ((old-type (get-type_56 result-name))
		    (old-home (get-home-address result-name)))
		;; too many bugs trying to avoid storing the value until necessary -- will just do it.
		;; if this (or some) setf is the first reference to result-name, it is safe to accept
		;; the expression type, rather than force a type coercion.  Otherwise, we may have assumed
		;; somewhere earlier that it had a given type, and it has to be that type upon returning
		;; in the implicit loop.
		(if (eq res-type old-type)
		    (progn
		      (move res-reg old-home)
		      (mark-reg res-reg (list result-name 'L 'NS))
		      (setf (third (gethash result-name vars)) (list res-reg))) ;added 8-June-92
		  ;; this is a dangerous change -- if any place in code56 forgets to explicitly get-register,
		  ;; then the following use of result-name may think it is still in that register
		  (if (not (member result-name setf-first-var))
		      (progn
			(convert res-reg res-type old-home old-type)
			(setf (third (gethash result-name vars)) nil))
		    (progn
		      (if (and (member res-type '(integer fraction))
			       (eq old-type 'real))
			  (setf (second (gethash result-name vars)) (list 'X (cadr old-home)))
			(if (and (eq res-type 'real)
				 (member old-type '(integer fraction)))
			    (setf (second (gethash result-name vars)) (list 'L (cadr old-home)))))
		      (setf (first (gethash result-name vars)) res-type)
		      (DEBUGGING (push `(----------------> changed ,result-name to ,res-type) pp))
		      (mark-reg res-reg (list result-name 'L 'NS))
		      (move res-reg (get-home-address result-name))
		      (setf (third (gethash result-name vars)) (list res-reg))))))))
	  (setf setf-first-var (delete result-name setf-first-var))))))
  nil)

(defun single-op (result-name val op)
  ;; the operand here has to be in A or B (ABS and NEG ops)
  (let* ((res-type (or (get-type_56 result-name)
		       (get-type_56 val)))
	 (res-reg (or (and (in-register val)
			   (member (get-work-address val) '(A B))
			   (get-work-address val))
		      (get-temporary-register '(A B)))))
    (if (temp-sig val) (kill-temp val))
    ;; I believe it is true that a temporary variable can be used only once as an operand, and then it is done.
    (convert (get-any-address val) (get-type_56 val) res-reg res-type)
    (push `(,op ,res-reg) pp)
    (update-var-and-reg result-name res-reg res-type 'L 'S)))

(defmacro <negate_56> (result-name arg)
  ;; a relatively easy case -- very much like <setf>, but with added negation (no need to store)
  (DEBUGGING (push `(----------------> <negate> ,result-name ,arg) pp))
  (let ((val (if (alias arg) (eval arg) arg)))
    (if (numberp val)
	(add-alias result-name (- val))
      ;; if val is not already in A B, load it (with conversion), negate (remember floats), update addresses
      (single-op result-name val 'NEG)))
  nil)
	
(defmacro <abs_56> (result-name arg) 
  (DEBUGGING (push `(----------------> <abs> ,result-name ,arg) pp))
  (let ((val (if (alias arg) (eval arg) arg)))
    (if (numberp val)
	(add-alias result-name (abs val))
      (single-op result-name val 'ABS)))
  nil)

(defun round-up-type (t1 t2)
  (if (eq t1 t2) t1			;operands are of same type
    (if (or (and (eq t1 'integer) (eq t2 'long-int))
	    (and (eq t2 'integer) (eq t1 'long-int)))
	'long-int
      'real)))

(defun round-up-add-type (t1 t2)
  (if (and (eq t1 t2) 
	   (not (eq t1 'fraction))) 
      t1 
    (if (or (and (eq t1 'integer) (eq t2 'long-int))
	    (and (eq t2 'integer) (eq t1 'long-int)))
	'long-int
      'real)))

(defun accumulate-result (result-name op variables res-type res-reg c-val)
  (let ((cur-type nil))
    (mark-reg res-reg (list result-name 'L 'S)) ;if type conversions, protect this register
    (loop while variables do
      (let* ((cur-var (or (find-var-in-reg variables (if (eq res-reg 'A) 'B 'A))
			  (find-var-in-reg variables 'X)
			  (find-var-in-reg variables 'Y)
			  (car variables)))
	     (opl nil) (prev-type nil))
	(setf cur-type (get-type_56 cur-var))
	(setf variables (remove cur-var variables :count 1))
	(setf prev-type res-type)
	(setf res-type (round-up-add-type res-type cur-type))
	(if (not (eq prev-type res-type))
	    (convert res-reg prev-type res-reg res-type))
	(if (temp-sig cur-var) (kill-temp cur-var))
	(if (in-register cur-var)
	    (progn
	      (setf opl (get-work-address cur-var))
	      (if (not (eq cur-type res-type))
		  (convert opl cur-type opl res-type)))
	  (progn
	    (setf opl (get-temporary-register (correct-side cur-type) (list res-reg)))
	    (convert (get-home-address cur-var) cur-type opl res-type)
	    (update-var-and-reg cur-var opl res-type)))
	(if (null opl) (error "somehow we got an address of NIL"))
	(if (member opl '(A2 A1 A0)) (setf opl 'A)
	  (if (member opl '(B2 B1 B0)) (setf opl 'B)))
	(push `(,op ,opl ,res-reg) pp)))
    
    (if (/= 0 c-val)			;we still have the constant to deal with
	(let ((c-reg (if (and (not (member res-type '(real long-int)))
			      (or (integerp c-val)
				  (< (abs c-val) 1.0)))
			 (get-temporary-register '(A B X0 X1 Y0 Y1) (list res-reg))
		       (get-temporary-register '(A B) (list res-reg))))
	      (old-type res-type))
	  (if (eq res-type 'real)
	      (load-real c-reg (float c-val))
	    (if (eq res-type 'long-int)
		(load-long-int c-reg (round c-val))
	      (progn
		(setf cur-type (load-number c-reg c-val))
		(setf res-type (round-up-add-type res-type cur-type)) 
		(if (not (eq old-type res-type))
		    (convert res-reg old-type res-reg res-type))
		(if (not (eq cur-type res-type))
		    (convert c-reg cur-type c-reg res-type)))))
	  (push `(,op ,c-reg ,res-reg) pp)))
    ;;res-reg should now have the result
    (if (and (gethash result-name vars)
	     (not (eq (get-type_56 result-name) res-type)))
	(convert res-reg res-type (get-home-address result-name) (get-type_56 result-name))
      (update-var-and-reg result-name res-reg res-type 'L 'S))))

(defmacro <add_56> (result-name &rest args)

  ;; passed in IR as [<add> dsp-nnn arg1 arg2 ...] -- no optimization or anything has been done yet.
  ;; first fold constants (explicit and named) as arg handled at initialization, get result-type from
  ;; context, or from args, find if any args already in usable reg, remove from list if so,
  ;; run down rest of list coercing to type and pushing ADD addr1 result.
  ;; Each dsp-nnn should have accompanying type designation, and current address (either stored in
  ;; low memory or in a register).
  ;; ADD [X Y A B X0 X1 Y0 Y1] [A B].   X0 to A, appends with 24 LSB zeros and adds to A1, etc.
  ;;
  ;; fold constants, find if any in A or B, use if possible, else free1 A or B, store one there,
  ;; find if any in Xn Yn, if so ADD in, go down rest of list, if any, get free X[n] or Y[n]
  ;; and ADD into accumulating result.  Make new VAR ref for dsp-nnn, say "in A" -- will be
  ;; spilled into low mem or heap, if needed.
  (DEBUGGING (push `(----------------> <add> ,result-name ,@args) pp))
  (multiple-value-bind (constants variables)
      (sort-operands args)		;we're assuming associativity here--dangerous assumption,
					;but a legal one in Common Lisp (see CLtL)
    (let ((c-val (if constants (collapse-constants constants '+) 0)))
      (if (not (integerp c-val))	;might be a ratio
	  (setf c-val (float c-val)))
      (if (null variables)
	  ;; if no variables, SET dsp-nnn c-val -- <add> collapsed into simple constant.
	  (add-alias result-name c-val)
	(if (and (= c-val 0)
		 (= (length variables) 1))
	    ;; if just one VAR, and constant part = 0, <add> collapses into dsp-nnn = var-name
	    (add-alias result-name (car variables))
	  ;; try to figure out easiest way to do the adds -- if all of same type, or all compatible with straight add etc.
	  (let* ((res-type nil)		;current type of result
		 (res-reg nil)		;current location of result (A or B)
		 (A-var (find-var-in-reg variables 'A))
		 (B-var (find-var-in-reg variables 'B)))
	    (if A-var
		(if B-var
		    (setf res-reg (if (< (reg-cost 'B) (reg-cost 'A)) 'B 'A))
		  (setf res-reg 'A))
	      (if B-var (setf res-reg 'B)))
	    (if res-reg			;that is, we found someone already in a useful register
		(progn			;so remove him from VARIABLES, update all association lists, etc
		  (get-register res-reg) ;free it up
		  (setf res-type (get-type_56 (if (eq res-reg 'A) A-var B-var)))
		  (setf variables (remove (if (eq res-reg 'A) A-var B-var) variables :count 1)))
	      (progn
		(setf res-reg 
		  (get-temporary-register 
		   (if (or variables (/= 0 c-val)) '(A B) '(A))))

		(if (/= 0 c-val)	;no variables in A or B -- maybe a constant to load?
		    (progn		;yes -- get A or B, load up constant -- do type conversion now (saves lots of ops)
		      (if (or (and (integerp c-val)
				   variables
				   (member (get-type_56 (car variables)) '(integer long-int)))
			      (and (< (abs c-val) 1.0)
				   variables
				   (eq (get-type_56 (car variables)) 'fraction)))
			  (setf res-type (load-number res-reg c-val))
			(setf res-type (load-real res-reg (float c-val))))
		      (setf c-val 0))
		  ;; at this branch, we have no variables in A or B, and no constant needs loading
		  (let ((m-var (or (find-var-not-in-XY variables) ; maybe some normal var needs loading in any case
				   (car variables))))             ; or just use any normal var
		    (move (get-any-address m-var) res-reg)
		    (setf res-type (get-type_56 m-var))
		    (if variables 
			(progn
			  (setf variables (remove m-var variables :count 1))
			  (if (temp-sig m-var) (kill-temp m-var))))))))
	    ;; we have finally chosen res-reg, and have something in it ready for adds.
	    ;; now loop through variables looking for easy cases (same type, in X Y or (not res-reg))
	    ;; then take care of constant, if any
	    ;; then rest of variables, if any
	    (accumulate-result result-name 'ADD variables res-type res-reg c-val))))))
  nil)


(defmacro <subtract_56> (result-name val &rest args)
  ;; simpler than ADD because we know the first argument (val) has to be loaded into A or B, if args
  (DEBUGGING (push `(----------------> <subtract> ,result-name ,val ,@args) pp))
  (multiple-value-bind
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants '+) 0)))
      (if (not (integerp c-val)) 
	  (setf c-val (float c-val)))
      (if (and (numberp val) (null variables))
	  (add-alias result-name (- val c-val))
	(if (and (= c-val 0) (null variables))
	    (add-alias result-name val)
	  ;; now we have at least two arguments, not both constant
	  (if (and (numberp val)	;maybe res:=0-var
		   (= val 0)
		   (= c-val 0)
		   (= (length variables) 1))
	      `(<negate_56> ,result-name ,(car variables))
	    ;; from here we have at least two honest arguments
	    (let ((res-reg (get-temporary-register '(A B)))
		  (res-type nil))
	      (if (numberp val)
		  (if (or (and (integerp val)
			       variables
			       (member (get-type_56 (car variables)) '(integer long-int)))
			  (and (< (abs val) 1.0)
			       variables
			       (eq (get-type_56 (car variables)) 'fraction)))
		      (setf res-type (load-number res-reg val))
		    (setf res-type (load-real res-reg (float val))))
		(progn
		  (setf res-type (get-type_56 val))
		  (move (get-any-address val) res-reg)
		  (if (temp-sig val) (kill-temp val))))
	      ;; subtract is now ready to go (val is in res-reg)
	      (accumulate-result result-name 'SUB variables
				 res-type res-reg c-val) 
	      nil)))))))


;;; MULTIPLY
;;;
;;; integer mpy ends up with extra sign bit at left (sign extension)
;;; fractional mpy ends up with 0 bit at far right (because of built in shift left) -- 56000 mpy is fractional

(defun full-reg-list (lst)
  (loop for i from 0 below (length lst) do
    (setf (nth i lst) (get-full-reg (nth i lst))))
  lst)

(defun get-reg-and-type-for (var res-reg-list typ)
  (DEBUGGING (push `(----------------> find reg for ,var (,res-reg-list ,typ)) pp))
  (let ((opl nil) 
	(res-type nil))
    (if (numberp var)
	(progn
	  (setf opl (get-temporary-register
		     (if (or (integerp var) (< (abs var) 1.0))
			 (if (member typ '(real long-int)) '(X1) '(X0 X1 Y0 Y1))
		       '(A B))
		     res-reg-list))
	  (setf res-type (load-number opl var)))
      (let ((cur-reg (get-work-address var)))
	(setf res-type (get-type_56 var))
	(if (temp-sig var) (kill-temp var))
	(if (or (member cur-reg res-reg-list) ;3-Mar-93 for (expt (+ (* x x) (* y y))) where x in B gets lost in the shuffle
		(and (member res-type '(real long-int)) 
		     (not (member cur-reg '(A B))))
		(and (not (member res-type '(real long-int))) 
		     (not (member cur-reg '(X0 X1 Y0 Y1)))))
	    (progn
	      (setf opl (get-temporary-register
			 (if (member res-type '(real long-int)) 
			     (if (member typ '(real long-int)) '(A B) '(A))
			   (if (eq typ 'real) '(X1) '(X0 X1 Y0 Y1)))
			 (if (eq res-type 'real) (full-reg-list res-reg-list) res-reg-list)))
	      (convert (get-any-address var) res-type opl res-type))
	  (setf opl cur-reg))))
    (values opl res-type)))

(defun round-up-mpy-type (t1 t2)
  (if (eq t1 t2) t1			;operands are of same type
    (if (or (eq t2 'long-int)
	    (eq t1 'long-int))
	'long-int
      'real)))

(defmacro <multiply_56> (result-name &rest args)
  (DEBUGGING (push `(----------------> <multiply> ,result-name ,@args) pp))
  (multiple-value-bind 
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants '*) 1)))
      (if (not (integerp c-val))	;might be a ratio
	  (setf c-val (float c-val)))
      (if (null variables)
	  ;; if no variables, SET dsp-nnn c-val -- <multiply> collapsed into simple constant.
	  (add-alias result-name c-val)
	(if (and constants (= c-val 0))	; there were constants, and they = 0, so everything collapses
	    (add-alias result-name c-val)
	  (if (and (= c-val 1)
		   (= (length variables) 1))
	      ;; if just one VAR, and constant part = 1, <multiply> collapses into dsp-nnn = var-name
	      (add-alias result-name (car variables))
	    (if (and (= c-val -1)
		     (= (length variables) 1))
		;; if just one VAR, and constant part = -1, <multiply> collapses <negate>
		(single-op result-name (car variables) 'NEG)
		
	      ;; here we have a honest multiply -- MPY Xn Yn to AB, but no X1 X1 or Y1 Y1 
	      ;; it is significantly less expensive to do fraction and integer multiplies.
	      (let ((fractions nil)
		    (integers nil)
		    (long-ints nil)
		    (true-res-reg nil)
		    (res-type nil) (cur-type nil) (prev-type nil)
		    (opl nil) (opl-var nil)
		    (opr nil) (opr-var nil)
		    (other-vars nil)
		    (res-reg nil)
		    (reals nil))
		(if (/= 1 c-val)
		    (if (integerp c-val) 
			(push c-val integers)
		      (if (< (abs c-val) 1.0)
			  (push c-val fractions)
			(push c-val reals))))
		(loop for i in variables do
		  (case (get-type_56 i)
		    (integer (push i integers))
		    (fraction (push i fractions))
		    (long-int (push i long-ints))
		    (t (push i reals))))
		(setf other-vars (or integers fractions reals long-ints))
		(let ((var-list (append fractions integers reals long-ints))
		      (opl-load nil))
		  (setf opl-var (pop var-list))
		  (if (eq (get-type_56 (car var-list)) 'real)
		      (setf res-reg (get-temporary-register '(A)))
		    (setf res-reg (get-temporary-register '(A B))))
		  (multiple-value-setq 
		      (opl res-type)
		    (get-reg-and-type-for opl-var (list res-reg) (get-type_56 (car var-list))))
		  (if (temp-sig opl-var) (kill-temp opl-var))
		  ;;even if this is real * fractional-number or integer, we want the fraction or
		  ;; integer intact, because real-frac-mpy et al are cheaper than real-mpy
		  (DEBUGGING (push `(----------------> ,opl-var in ,opl (result in ,res-reg)) pp)) 
		  (setf prev-type res-type)

		  ;(mark-reg res-reg (list result-name 'L 'NS)) ;if type conversions, protect this register

		  (loop while var-list do
		    (setf opr-var (pop var-list))
		    (multiple-value-setq
			(opr cur-type)
		      (get-reg-and-type-for opr-var (if (eq prev-type 'real) (list opl) (list res-reg opl)) prev-type))
		    (DEBUGGING (push `(----------------> * ,opr-var -> ,prev-type ,res-reg ,opl ,opr ,cur-type) pp))
		    (setf res-type (round-up-mpy-type res-type cur-type))

		    (if (and true-res-reg (not (eq prev-type res-type)))
			(let ((new-opl (if (member prev-type '(fraction integer)) 'X1 (get-full-reg opl))))
			  (move true-res-reg new-opl)
			  (setf opl new-opl))
		      (if opl-load (move res-reg opl)))

		    (if opl-var (setf opl-var nil))
		    (case res-type
		      (integer  (push `(MPY ,opl ,opr ,res-reg) pp)
				(push `(ASR ,res-reg) pp)
				(setf true-res-reg (get-high-side res-reg)))
		      (fraction (push `(MPY ,opl ,opr ,res-reg) pp)
				(setf true-res-reg (get-high-side res-reg)))
		      (long-int 
		       (setf res-reg (get-register 'A))
		       (setf true-res-reg 'A)
		       (case prev-type
			 (integer
			  (need-loaded '.int-long-int-mpy)
			  (spill-ALU-registers (libinfo-uses .int-long-int-mpy))
			  (if (not (eq opl 'X)) (move opl 'X1))
			  (when (not (eq opr 'A)) (move opr 'A) (setf opr 'A))
			  (push `(JSR .int-long-int-mpy) pp))
			 (fraction
			  (need-loaded '.frac-long-int-mpy)
			  (spill-ALU-registers (libinfo-uses .frac-long-int-mpy))
			  (if (not (eq opl 'X)) (move opl 'X1))
			  (when (not (eq opr 'A)) (move opr 'A) (setf opr 'A))
			  (push '(JSR .frac-long-int-mpy) pp))
			 (long-int
			  (need-loaded '.long-int-mpy)
			  (spill-ALU-registers (libinfo-uses .long-int-mpy))
			  (if (not (eq opl 'B)) (move opl 'B))
			  (when (not (eq opr 'A)) (move opr 'A) (setf opr 'A))
			  (push `(JSR .long-int-mpy) pp))
			 (real
			  (need-loaded '.real-long-int-mpy)
			  (spill-ALU-registers (libinfo-uses .real-long-int-mpy))
			  (if (not (eq opl 'B)) (move opl 'B))
			  (when (not (eq opr 'A)) (move opr 'A) (setf opr 'A))
			  (push `(JSR .real-long-int-mpy) pp))
			 (t (error "long-int multiply type confusion: ~A" prev-type)))
		       (setf opl 'A))
		      
		      (real		;since real*real is very expensive, we try to take advantage of special cases
		       (case prev-type
			 (fraction
			  (if (eq cur-type 'integer)
			      (push `(MPY ,opl ,opr ,res-reg) pp)
			    (progn	;fraction*real
			      (need-loaded '.real-frac-mpy)
			      (spill-ALU-registers (libinfo-uses .real-frac-mpy))
			      (setf res-reg (get-register 'A))
			      ;; now make sure A=real, X1=fraction
			      
			      (if (not (eq opl 'X)) (move opl 'X1))
			      (when (not (eq opr 'A)) (move opr 'A) (setf opr 'A))
			      (setf opl 'A) ;needed upon loop for get-reg-and-type-for
			    
			      (push `(JSR .real-frac-mpy) pp))))
			 (integer
			  (need-loaded '.real-int-mpy)
			  (spill-ALU-registers (libinfo-uses .real-int-mpy))
			  (setf res-reg (get-register 'A))
			  
			  (if (not (eq opl 'X)) (move opl 'X1))
			  (when (not (eq opr 'A)) (move opr 'A) (setf opr 'A))
			  (setf opl 'A)
			
			  (push `(JSR .real-int-mpy) pp))
			 (real
			  (need-loaded '.real-mpy)
			  (spill-ALU-registers (libinfo-uses .real-mpy))
			  (setf res-reg (get-register 'A))
			  (setf opl 'A)
			  (push `(JSR .real-mpy) pp))
			 (t (error "real multiply type confusion: ~A" prev-type)))
		       (setf true-res-reg res-reg)))
		    (setf prev-type res-type)
		    (setf opl-load var-list)))
		;; now we have any non-floats multiplied together in res-reg -- go deal with floats
		(update-var-and-reg result-name true-res-reg res-type 'L 'S))))))))
  nil)

(defun shift-A-up-23 ()
  (push `(COPY A1 A2) pp)
  (push `(COPY A0 A1) pp)
  (push `(LOAD A0 0 SHORT) pp)
  (push `(ASR A) pp))

(defun shift-up-as-far-as-possible ()
  (need-loaded '.shift-AB-up)
  (push `(JSR .shift-AB-up) pp))

;;; DIVIDE
;;;
;;; divide is very expensive on this chip, for all number types, so we first use multiply to collapse all
;;; arguments down to one, then divide by that, if necessary.

(defmacro <divide_56> (result-name val &rest args)
  (DEBUGGING (push `(----------------> <divide> ,result-name ,val ,@args) pp))
  (let ((dividend (if (alias val) (eval val) val)))
    (if (and (numberp dividend) (= dividend 0)) ;divide collapses to 0
	(add-alias result-name dividend)
      (if (> (length args) 1)
	  (let ((newsig (make-temp-sig)))
	    `(progn (<multiply_56> ,newsig ,@args) 
		    (<divide-1_56> ,result-name ,dividend ,newsig)))
	`(<divide-1_56> ,result-name ,dividend ,@args)))))

(defmacro <divide-1_56> (result-name dividend divider)
  (let ((divisor (if (alias divider) (eval divider) divider)))
    (if (numberp divisor)		;hooray -- we can use multiply
	(if (or (= divisor 1) (zerop divisor))
	    (add-alias result-name dividend)
	  (if (numberp dividend)	;even better -- every argument was a constant
	      (add-alias result-name (/ dividend divisor))
	    `(<multiply_56> ,result-name ,dividend ,(/ 1.0 divisor))))
      ;; here we have either number/variable or variable/variable -- no way out but DIV
      ;; now choose how to proceed with either fdiv in fplib or basic-divide (which is A1/B1 as though integer)
      (let ((num-type (get-type_56 dividend))
	    (den-type (get-type_56 divisor))
	    (res-type 'real))
	(let ((inter-case (or (eq num-type 'real) (eq den-type 'real))))
	      (need-loaded '.basic-divide)
	      (spill-ALU-registers '(A B X Y))
	      (if (temp-sig divisor) (kill-temp divisor))
	      (if (temp-sig dividend) (kill-temp dividend))
	      (let ((dividend-addr (get-any-address dividend))
		    (divisor-addr (get-any-address divisor)))
		(if (numberp dividend)
		    (load-number 'A dividend)
		  (if (or (not inter-case) (not (eq num-type 'fraction)))
		      (move dividend-addr 'A)
		    (convert dividend-addr num-type 'A 'real)))
		(if (or (not inter-case) (not (eq den-type 'fraction)))
		    (move divisor-addr 'B)
		  (convert divisor-addr den-type 'B 'real))
		(if inter-case (shift-up-as-far-as-possible))
		(push '(JSR .basic-divide) pp)
		(if (and (eq num-type 'integer) (eq den-type 'fraction))
		    (shift-A-up-23)
		  (if (and (eq num-type 'fraction) (eq den-type 'integer))
		      (setf res-type 'fraction)))
		(update-var-and-reg result-name 'A res-type 'L 'S)))
	nil))))

	
(defmacro <add-1_56> (result-name arg1)
  `(<add_56> ,result-name ,arg1 1))
	
(defmacro <subtract-1_56> (result-name arg1)
  `(<subtract_56> ,result-name ,arg1 1))


(defmacro <incf_56> (result-name arg1 &optional (arg2 1))
  (DEBUGGING (push `(----------------> <incf> ,result-name ,arg1 ,arg2 ---- ,arg1 ,@(gethash arg1 vars)) pp))
  `(progn
     (<add_56> ,result-name ,arg1 ,arg2)
     (<setf_56> ,arg1 ,result-name)))

(defmacro <decf_56> (result-name arg1 &optional (arg2 1))
  `(progn
     (<subtract_56> ,result-name ,arg1 ,arg2)
     (<setf_56> ,arg1 ,result-name)))


;;; MAX
;;; (CMP X0 A)           ;A-X0
;;; (TLT X0 A)           ;if A<X0 (i.e. if previous CMP was negative) A:=X0 (TGT for MIN)
;;; .fcmp first for floats

(defmacro <max_56> (result-name &rest args) 
  (DEBUGGING (push `(----------------> <max> ,result-name ,@args) pp))
  `(<min-max_56> TLT ,result-name ,@args))

(defmacro <min_56> (result-name &rest args)
  (DEBUGGING (push `(----------------> <min> ,result-name ,@args) pp))
  `(<min-max_56> TGT ,result-name ,@args))

(defmacro <min-max_56> (op result-name &rest args)
  (multiple-value-bind
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants (if (eq op 'TLT) 'max 'min)) nil)))
      (if (null variables)
	  (add-alias result-name c-val)
	(if (and (null constants)
		 (= (length variables)1))
	    (add-alias result-name (car variables))
	  ;; here we have at least two arguments, at least one of which is not a constant.
	  ;; load up args one at a time and CMP, then TLT or TGT for transfer.
	  ;; fractions must be sign extended, and use the whole A B X Y register.
	  ;; That is, coerce to real before compare.
	  (let* ((res-reg (if variables 
			      (get-temporary-register '(A B))
			    (get-temporary-register '(A))))
		 (res-type 'real)
		 (opl nil)
		 (cur-var (car variables))
		 (cur-type nil))
	    (if constants
		(if (not (eq (setf cur-type (load-number res-reg c-val)) res-type))
		    (convert res-reg cur-type res-reg res-type))
	      (if variables
		  (progn
		    (if (temp-sig cur-var) (kill-temp cur-var))
		    (convert (get-any-address cur-var) (get-type_56 cur-var)
			     res-reg res-type)
		    (setf variables (remove cur-var variables)))))
	    (loop while variables do
	      (setf cur-var (car variables))
	      (setf variables (remove cur-var variables)) ;remove duplicates too
	      (setf cur-type (get-type_56 cur-var))
	      (setf opl	(get-temporary-register '(A B) (list res-reg)))
	      (if (temp-sig cur-var) (kill-temp cur-var))
	      (convert (get-any-address cur-var) cur-type opl 'real)
	      (push `(CMP ,opl ,res-reg) pp)
	      (push `(,op ,opl ,res-reg) pp))
	    (update-var-and-reg result-name res-reg res-type 'L 'S))))))
  nil)


;;; TIE INTO RUN-TIME LIBRARY ---------------------------------------------------------------------------

(defmacro <trig_56> (result-name arg op lib-op &optional (arg-reg 'A) (arg-type 'real) (extra-reg nil) (res-type 'real))
  (DEBUGGING (push `(----------------> < ,op > ,result-name ,arg) pp))
  (let ((val (if (alias arg) (eval arg) arg)))
    (if (numberp val)			;collapse constant expression
	(add-alias result-name (funcall op val))
      (let ((val-addr (get-any-address val)))
	(need-loaded lib-op)
	(get-temporary-register (list arg-reg))
	(if (temp-sig val) (kill-temp val))
	(spill-ALU-registers)
	(convert val-addr (get-type_56 val) arg-reg arg-type)
	(if extra-reg (get-temporary-register (list extra-reg)))
	(push `(JSR ,lib-op) pp)
	(update-var-and-reg result-name 'A res-type 'L 'S))))
  nil)

(defmacro <log_56> (result-name arg &optional base)   
  (setf-N-reg 'N4 -1)       
  (if (not base)
      `(<trig_56> ,result-name ,arg log .log A real)
    (if (numberp base)
	(let ((newsig (make-temp-sig)))
	  `(progn (<trig_56> ,newsig ,arg log .log A real)
		  (<multiply_56> ,result-name ,newsig ,(/ 1.0 (log base)))))
      (let ((new1 (make-temp-sig))
	    (new2 (make-temp-sig)))
	`(progn (<trig_56> ,new1 ,arg log .log A real)
		(<trig_56> ,new2 ,base log .log A real)
		(<divide-1_56> ,result-name ,new1 ,new2))))))

(defmacro <sin_56> (result-name arg)   `(<trig_56> ,result-name ,arg sin .trig-sine A real nil fraction))
(defmacro <cos_56> (result-name arg)   `(<trig_56> ,result-name ,arg cos .trig-cosine A real nil fraction))        
(defmacro <tan_56> (result-name arg)   `(<trig_56> ,result-name ,arg tan .tan A real))
(defmacro <asin_56> (result-name arg)  `(<trig_56> ,result-name ,arg asin .asin A fraction))
(defmacro <acos_56> (result-name arg)  `(<trig_56> ,result-name ,arg acos .acos A fraction))
(defmacro <atan_56> (result-name arg &optional arg2)  
  (if (not arg2)
      `(<trig_56> ,result-name ,arg atan .atan A real)
    `(<atan2_56> ,result-name ,arg ,arg2)))
(defmacro <sinh_56> (result-name arg)  `(<trig_56> ,result-name ,arg sinh .sinh A fraction))
(defmacro <cosh_56> (result-name arg)  `(<trig_56> ,result-name ,arg cosh .cosh X0 fraction))
(defmacro <tanh_56> (result-name arg)  `(<trig_56> ,result-name ,arg tanh .tanh X0 fraction nil fraction))
(defmacro <asinh_56> (result-name arg) `(<trig_56> ,result-name ,arg asinh .asinh A real))
(defmacro <acosh_56> (result-name arg) `(<trig_56> ,result-name ,arg acosh .acosh A real))
(defmacro <atanh_56> (result-name arg) `(<trig_56> ,result-name ,arg atanh .atanh A fraction))

(defmacro <atan2_56> (result-name argy argx)
  (DEBUGGING (push `(----------------> <atan2> ,result-name ,argy ,argx) pp))
  (let ((valy (if (alias argy) (eval argy) argy))
	(valx (if (alias argx) (eval argx) argx)))
    (if (and (numberp valy)		;collapse constant expression
	     (numberp valx))
	(add-alias result-name (atan argy argx))
      (progn
	(need-loaded '.atan2)
	(spill-ALU-registers)
	(if (temp-sig valy) (kill-temp valy))
	(if (temp-sig valx) (kill-temp valx))
	(let ((valy-addr (get-any-address valy))
	      (valx-addr (get-any-address valx)))
	  (convert valy-addr (get-type_56 valy) 'A 'real)
	  (mark-reg 'A (list valy 'L 'NS))
	  (convert valx-addr (get-type_56 valx) 'B 'real)
	  (push `(JSR .atan2) pp)
	  (update-var-and-reg result-name 'A 'real 'L 'S)))))
  nil)

(defmacro <unimplemented_56> (result-name op &rest args)
  (DEBUGGING (push `(----------------> ,op ,result-name ,@args) pp))
  (let ((arg-list nil))
    (loop for i in args do
      (if (alias i) (push (eval i) arg-list) (push i arg-list)))
    (loop for i in arg-list do
      (if (not (constantp i)) (error "~S currently can only handle constants (~S)" op args)))
    (add-alias result-name (funcall op arg-list)))
  nil)

(defmacro <signum_56> (result-name arg)
  (DEBUGGING (push `(----------------> <signum> ,result-name ,arg) pp))
  (let ((val (if (alias arg) (eval arg) arg)))
    (if (numberp val)			;collapse constant expression
	(add-alias result-name (signum val))
      (let ((res-reg (in-AB-or-get-AB val))
	    (val-adr (get-any-address val)))
	(if (temp-sig val) (kill-temp val))
	(move val-adr res-reg)
	(push `(TST ,res-reg) pp)
	(push `(JEQ lab-0) pp)		;signum(0) => 0
	(push `(JMI lab-m) pp)
	(push `(LOAD ,res-reg 1) pp)
	(push `(JMP lab-0) pp)
	(push `(lab-m LOCAL) pp)
	(push `(LOAD ,res-reg #xffffff) pp)
	(push `(lab-0 LOCAL) pp)
	(update-var-and-reg result-name res-reg 'integer 'L 'S))))
  nil)


(defmacro <random_56> (result-name argl)
  ;; (random n) returns a number of type n in the interval [0,n) (positive).
  (DEBUGGING (push `(----------------> <random> ,result-name ,argl) pp))

  (let* ((arg (if (alias argl) (eval argl) argl))
	 (res-type (if (numberp arg)
		       (if (integerp arg) 'integer
			 (if (< -1.0 arg 1.0) 'fraction 'real))
		     'real)))
    (need-loaded '.random)
    (spill-ALU-registers (libinfo-uses .random))
    (push '(JSR .random) pp)		; result is in A0 (interpreted as a fraction in all cases)
    (push '(COPY A0 X1) pp)
    (if (eq res-type 'real)
	(progn
	  (spill-ALU-registers (libinfo-uses .real-frac-mpy))
	  (need-loaded '.real-frac-mpy)
	  (if (numberp arg)
	      (load-number 'A arg)
	    (convert (get-any-address arg) (get-type_56 arg) 'A res-type))
	  (push `(JSR .real-frac-mpy) pp))
      (progn
	(if (numberp arg)
	    (load-number 'X0 arg)
	  (convert (get-any-address arg) (get-type_56 arg) 'X0 res-type))
	(push `(MPY X0 X1 A) pp)
	(if (eq res-type 'integer) (push `(ASR A) pp))))
    (push `(ABS A) pp)
    (if (temp-sig arg) (kill-temp arg))
    (update-var-and-reg result-name 'A res-type 'L 'S))
  nil)


(defmacro <expt_56> (result-name base arg) 
  (DEBUGGING (push `(----------------> <expt> ,result-name ,base ,arg) pp))
  (let ((pow (if (alias arg) (eval arg) arg))
	(bas (if (alias base) (eval base) base)))
    (if (and (numberp pow) (numberp bas)) ;collapse constant expression
	(add-alias result-name (expt bas pow))
      (if (and (integerp pow)
	       (<= 0 pow 2))
	  (if (zerop pow)		;n^0 = 1 (even 0^0)
	      (add-alias result-name 1.0)
	    (if (= pow 1)		;n^1 = n
		(add-alias result-name bas)
	      `(<multiply_56> ,result-name ,bas ,bas))) ;assume n*n is (a lot) faster than n^2
	(if (and (integerp bas)
		 (<= 0 bas 1))
	    (if (zerop bas)		;0^n = 0 (0^0 already handled above)
		(add-alias result-name 0.0)
	      (add-alias result-name 1.0)) ;1^n = 1
	  (let* ((constant-base-p (numberp bas))
		 (bas-addr (if (not constant-base-p) (get-any-address bas))))
	    (need-loaded '.expt)
	    (if (temp-sig bas) (kill-temp bas))
	    (spill-ALU-registers (libinfo-uses .expt))
	    (get-temporary-register '(A))
	    (if constant-base-p
		(progn			;here we can preload (log base) and save half the computing
		  (load-real 'A (float bas))
		  (push `(STORE A L Orig-A) pp)
		  (load-real 'A (log (abs (float bas)))))
	      (convert bas-addr (get-type_56 bas) 'A 'real))
	    (mark-reg 'A (list bas 'L 'NS))
	    (if (temp-sig pow) (kill-temp pow))
	    (get-temporary-register '(B))
	    (convert (get-any-address pow) (get-type_56 pow) 'B 'real)
	    (if constant-base-p
		(progn
		  (push `(STORE B L temp-loc-1) pp)
		  (push `(JSR .constant-base-expt) pp))
	      (push `(JSR .expt) pp))
	    (update-var-and-reg result-name 'A 'real 'L 'S)
	    nil))))))

(defmacro <exp_56> (result-name arg)
  (DEBUGGING (push `(----------------> <exp> ,result-name ,arg) pp))
  (let ((pow (if (alias arg) (eval arg) arg)))
    (if (numberp pow)			;collapse constant expression
	(add-alias result-name (exp pow))
      (let ((pow-addr (get-any-address pow)))
	(need-loaded '.exp)
	(if (temp-sig pow) (kill-temp pow))
	(spill-ALU-registers (libinfo-uses .exp))
	(get-temporary-register '(A))
	(convert pow-addr (get-type_56 pow) 'A 'real)
	(push `(JSR .exp) pp)
	(update-var-and-reg result-name 'A 'real 'L 'S))))
  nil)


(defmacro <float_56> (result-name arg) 
  (DEBUGGING (push `(----------------> <float> ,result-name ,arg) pp))
  (let ((val (if (alias arg) (eval arg) arg)))
    (if (numberp val)			;collapse constant expression
	(add-alias result-name (float val))
      (let ((res-reg (in-AB-or-get-AB val)))
	(if (temp-sig val) (kill-temp val))
	(convert (get-any-address val) (get-type_56 val) res-reg 'real)
	(update-var-and-reg result-name res-reg 'real 'L 'S))))
  nil)

(defmacro <floor-1_56> (result-name val)
  (let* ((res-reg (in-AB-or-get-AB val))
	 (val-typ (get-type_56 val))
	 (typ (if (member val-typ '(integer long-int)) val-typ 'integer)))
    (if (temp-sig val) (kill-temp val))
    (convert (get-any-address val) (get-type_56 val) res-reg typ)
    (update-var-and-reg result-name res-reg typ 'L 'S))
  nil)

(defmacro <ceiling-1_56> (result-name arg)
  (let ((one-reg (get-temporary-register '(B X0 X1 Y0 Y1))))
    (if (temp-sig arg) (kill-temp arg))
    (convert (get-any-address arg) (get-type_56 arg) (get-temporary-register '(A)) 'real)
    (push `(LOAD ,one-reg 1) pp)
    (push `(ADD ,one-reg A) pp)
    (convert 'A 'real 'A 'integer)
    (update-var-and-reg result-name 'A 'integer 'L 'S))
  nil)

(defmacro <round-1_56> (result-name arg)
  (let ((one-reg (get-temporary-register '(B X Y))))
    (if (temp-sig arg) (kill-temp arg))
    (convert (get-any-address arg) (get-type_56 arg) (get-temporary-register '(A)) 'real)
    (if (eq one-reg 'B)
	(push `(CLR ,one-reg) pp)
      (push `(LOAD ,(get-high-side one-reg) 0) pp))
    (push `(LOAD ,(get-low-side one-reg) .5) pp)
    (push `(ADD ,one-reg A) pp)
    (convert 'A 'real 'A 'integer)
    (update-var-and-reg result-name 'A 'integer 'L 'S))
  nil)

(defmacro <truncate-1_56> (result-name arg)
  (let ((one-reg (get-temporary-register '(B X0 X1 Y0 Y1))))
    (if (temp-sig arg) (kill-temp arg))
    (convert (get-any-address arg) (get-type_56 arg) (get-temporary-register '(A)) 'real)
    (push `(TST A) pp)
    (push `(LOAD A0 0 SHORT) pp)
    (push `(JPL done) pp)
    (push `(LOAD ,one-reg 1) pp)
    (push `(ADD ,one-reg A) pp)
    (push `(done LOCAL) pp)
    (update-var-and-reg result-name 'A 'integer 'L 'S))
  nil)

(defmacro <floor_56> (result-name arg &optional div-1) 
  (DEBUGGING (push `(----------------> <floor> ,result-name ,arg ,div-1) pp))
  (let ((val (if (alias arg) (eval arg) arg))
	(div (if (alias div-1) (eval div-1) div-1)))
    (if (and (numberp val)
	     (or (not div-1) (numberp div)))
	(add-alias result-name (if div-1 (floor val div) (floor val)))
      (if div-1
	  (let ((newsig (make-temp-sig)))
	    `(progn
	       (<divide_56> ,newsig ,val ,div)
	       (<floor-1_56> ,result-name ,newsig)))
	`(<floor-1_56> ,result-name ,val)))))

(defmacro <ceiling_56> (result-name argl &optional div-1)
  (DEBUGGING (push `(----------------> <ceiling> ,result-name ,argl ,div-1) pp))
  (let ((arg (if (alias argl) (eval argl) argl))
	(div (if (alias div-1) (eval div-1) div-1)))
    (if (and (numberp arg)
	     (or (not div-1) (numberp div)))
	(add-alias result-name (if div-1 (ceiling arg div-1) (ceiling arg)))
      (if div-1
	  (let ((newsig (make-temp-sig)))
	    `(progn
	       (<divide_56> ,newsig ,arg ,div)
	       (<ceiling-1_56> ,result-name ,newsig)))
	`(<ceiling-1_56> ,result-name ,arg)))))

(defmacro <round_56> (result-name argl &optional div-1)
  (DEBUGGING (push `(----------------> <round> ,result-name ,argl ,div-1) pp))
  (let ((arg (if (alias argl) (eval argl) argl))
	(div (if (alias div-1) (eval div-1) div-1)))
    (if (and (numberp arg)
	     (or (not div-1) (numberp div)))
	(add-alias result-name (if div-1 (round arg div) (round arg)))
      (if div-1
	  (let ((newsig (make-temp-sig)))
	    `(progn
	       (<divide_56> ,newsig ,arg ,div)
	       (<round-1_56> ,result-name ,newsig)))
	`(<round-1_56> ,result-name ,arg)))))
	
(defmacro <truncate_56> (result-name argl &optional div-1)
  (DEBUGGING (push `(----------------> <truncate> ,result-name ,argl ,div-1) pp))
  (let ((arg (if (alias argl) (eval argl) argl))
	(div (if (alias div-1) (eval div-1) div-1)))
    (if (and (numberp arg)
	     (or (not div-1) (numberp div)))
	(add-alias result-name (if div-1 (truncate arg div) (truncate arg)))
      (if div-1
	  (let ((newsig (make-temp-sig)))
	    `(progn
	       (<divide_56> ,newsig ,arg ,div)
	       (<truncate-1_56> ,result-name ,newsig)))
	`(<truncate-1_56> ,result-name ,arg)))))

(defmacro <numerator_56> (result-name arg) `(<unimplemented_56> ,result-name numerator ,arg))
(defmacro <denominator_56> (result-name arg) `(<unimplemented_56> ,result-name denominator ,arg))

(defmacro <sqrt_56> (result-name arg) 
  (DEBUGGING (push `(----------------> <sqrt> ,result-name ,arg) pp))
  (let ((val (if (alias arg) (eval arg) arg)))
    (if (numberp val)			;collapse constant expression
	(add-alias result-name (sqrt val))
      (let ((res-type (get-type_56 val))
	    (val-addr (get-any-address val)))
	(if (temp-sig val) (kill-temp val))
	(case res-type
	  (integer
	   (need-loaded '.int-sqrt)
	   (spill-ALU-registers (libinfo-uses .int-sqrt))
	   (convert val-addr res-type 'Y1 res-type)
	   (push '(JSR .int-sqrt) pp))
	  (fraction
	   (need-loaded '.frac-sqrt)
	   (spill-ALU-registers (libinfo-uses .frac-sqrt))
	   (convert val-addr res-type 'Y1 res-type)
	   (push '(JSR .frac-sqrt) pp))
	  ((real long-int)
	   (need-loaded '.real-sqrt)
	   (spill-ALU-registers (libinfo-uses .real-sqrt))
	   (convert val-addr res-type 'A res-type)
	   (push '(JSR .real-sqrt) pp)))
	(update-var-and-reg result-name 'A res-type 'L 'S))))
  nil)
	
(defmacro <ash_56> (result-name arg1 arg2)
  (let ((int (if (alias arg1) (eval arg1) arg1))
	(cnt (if (alias arg2) (eval arg2) arg2)))
    (if (constantp cnt)
	(if (constantp int)
	    (add-alias result-name (ash int cnt))
	  (if (zerop cnt)
	      (add-alias result-name int)
	    (progn
	      (move (get-any-address int) (get-temporary-register '(A)))
	      (shift 'A cnt)
	      (if (temp-sig int) (kill-temp int))
	      (update-var-and-reg result-name 'A 'integer 'L 'S))))
      (progn
	(spill-ALU-registers (libinfo-uses .ash))
	(need-loaded '.ash)
	(if (constantp int)
	    (load-number 'A int)
	  (progn
	    (move (get-any-address int) 'A)
	    (if (temp-sig int) (kill-temp int))))
	(move (get-any-address cnt) 'B)
	(if (temp-sig cnt) (kill-temp cnt))
	(push '(JSR .ash) pp)
	(update-var-and-reg result-name 'A 'integer 'L 'S))))
  nil)

(defmacro <mod_56> (result-name arg1 arg2 &optional rem-time)
  (DEBUGGING (push `(----------------> ,(if rem-time '<rem> '<mod>) ,result-name ,arg1 ,arg2) pp))
  (let ((a1 (if (alias arg1) (eval arg1) arg1))
	(a2 (if (alias arg2) (eval arg2) arg2)))
    (if (and (constantp a1) (constantp a2)) ;collapse constant expression
	(add-alias result-name (if rem-time (rem a1 a2) (mod a1 a2)))
      (let ((jsr-loc (if rem-time '.int-rem '.int-mod))
	    (res-type 'integer))
	(if (and (or (integerp a1) (eq (get-type_56 a1) 'integer))
		 (or (integerp a2) (eq (get-type_56 a2) 'integer)))
	    (spill-ALU-registers (libinfo-uses .int-mod))
	  (progn
	    (setf jsr-loc (if rem-time '.real-rem '.real-mod))
	    (setf res-type 'real)
	    (spill-ALU-registers (libinfo-uses .real-mod))))
	(need-loaded jsr-loc)
	(get-temporary-register '(A))
	(if (numberp a1)
	    (load-number 'A a1)
	  (move (get-any-address a1) 'A))
	(if (temp-sig a1) (kill-temp a1))
	(get-temporary-register '(B))
	(if (numberp a2)
	    (load-number 'B a2)
	  (move (get-any-address a2) 'B))
	(if (temp-sig a2) (kill-temp a2))
	(push `(JSR ,jsr-loc) pp)
	(update-var-and-reg result-name 'A res-type 'L 'S))))
  nil)

(defmacro <rem_56> (result-name arg1 arg2)
  `(<mod_56> ,result-name ,arg1 ,arg2 t))

(defmacro <basic-gcd_56> (its-lcm result-name &rest args)
  (DEBUGGING (push `(----------------> ,(if its-lcm '<lcm> '<gcd>) ,result-name ,@args) pp))
  (multiple-value-bind
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants (if its-lcm 'lcm 'gcd)) 0)))
      (if (null variables)
	  (add-alias result-name c-val)
	(let ((jsr-loc (if its-lcm '.lcm '.gcd)))
	  (need-loaded jsr-loc)
	  (spill-ALU-registers (libinfo-uses .gcd))
	  (if (/= 0 c-val) 
	      (load-integer 'A c-val) 
	    (move (get-any-address (pop variables)) 'A))
	  (move (get-any-address (pop variables)) 'B)
	  (push `(JSR ,jsr-loc) pp)
	  (loop for i in variables do
	    (move (get-any-address i) 'B)
	    (push `(JSR ,jsr-loc) pp))
	  (update-var-and-reg result-name 'A 'integer 'L 'S)))))
  nil)
	
(defmacro <gcd_56> (result-name &rest args)
  `(<basic-gcd_56> nil ,result-name ,@args))

(defmacro <lcm_56> (result-name &rest args)
  `(<basic-gcd_56> t ,result-name ,@args))


(defmacro <lognot_56> (result-name arg)
  (DEBUGGING (push `(----------------> <lognot> ,result-name ,arg) pp))
  (let ((val (if (alias arg) (eval arg) arg)))
    (if (numberp val)			;collapse constant expression
	(add-alias result-name (lognot val))
      (let ((res-reg (in-AB-or-get-AB val)))
	(if (not (in-AB-already val)) (move (get-any-address val) res-reg))
	(if (temp-sig val) (kill-temp val))
	(push `(NOT ,res-reg) pp)
	(update-var-and-reg result-name res-reg 'integer 'L 'S))))
  nil)

(defmacro <logical-op_56> (result-name op res-reg add-not &rest args)
  (DEBUGGING (push `(----------------> < ,op > ,result-name ,@args) pp))
  (loop while args do
    (let* ((cur-argl (pop args))
	   (cur-arg (if (alias cur-argl) (eval cur-argl) cur-argl))
	   (in-reg (member (get-work-address cur-arg) '(X0 X1 Y0 Y1)))
	   (cur-opl (if in-reg 
			(car in-reg)
		      (get-temporary-register '(X0 X1 Y0 Y1)))))
      (if (not in-reg) (convert (get-any-address cur-arg) (get-type_56 cur-arg) cur-opl 'integer))
      (if (temp-sig cur-arg) (kill-temp cur-arg))
      (push `(,op ,cur-opl ,res-reg) pp)))
  (if add-not (push `(NOT ,res-reg) pp))
  (update-var-and-reg result-name res-reg 'integer 'L 'S)
  nil)

(defmacro <logand_56> (result-name &rest args)
  (multiple-value-bind
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants 'logand) -1)))
      (if (null variables)
	  (add-alias result-name c-val)	;if no args at all, this gives us -1 (as per Lisp)
	(let ((res-reg (get-temporary-register '(A B))))
	  (if (/= -1 c-val)
	      (load-integer res-reg c-val)
	    (move (get-any-address (pop variables)) res-reg))
	  `(<logical-op_56> ,result-name AND ,res-reg nil ,@variables))))))


(defmacro <logior_56> (result-name &rest args)
  (multiple-value-bind
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants 'logior) 0)))
      (if (null variables)
	  (add-alias result-name c-val)	;if no args at all, this gives us 0 (as per Lisp)
	(let ((res-reg (get-temporary-register '(A B))))
	  (if (/= 0 c-val)
	      (load-integer res-reg c-val)
	    (move (get-any-address (pop variables)) res-reg))
	  `(<logical-op_56> ,result-name OR ,res-reg nil ,@variables))))))


(defmacro <logxor_56> (result-name &rest args)
  (multiple-value-bind
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants 'logxor) 0)))
      (if (null variables)
	  (add-alias result-name c-val)	;if no args at all, this gives us 0 (as per Lisp)
	(let ((res-reg (get-temporary-register '(A B))))
	  (if (/= 0 c-val)
	      (load-integer res-reg c-val)
	    (move (get-any-address (pop variables)) res-reg))
	  `(<logical-op_56> ,result-name EOR ,res-reg nil ,@variables))))))

(defmacro <logeqv_56> (result-name &rest args)
  (multiple-value-bind
      (constants variables)
      (sort-operands args)
    (let ((c-val (if constants (collapse-constants constants 'logeqv) -1)))
      (if (null variables)
	  (add-alias result-name c-val)	;if no args at all, this gives us -1 (as per Lisp)
	(let ((res-reg (get-temporary-register '(A B))))
	  (if (/= -1 c-val)
	      (load-integer res-reg c-val)
	    (move (get-any-address (pop variables)) res-reg))
	  `(<logical-op_56> ,result-name EOR ,res-reg t ,@variables))))))
  
(defmacro <log-op_56> (result-name a-i1 a-i2 lisp-op op1 op2)
  (let ((i1 (if (alias a-i1) (eval a-i1) a-i1))
	(i2 (if (alias a-i2) (eval a-i2) a-i2)))
    (if (and (constantp i1) (constantp i2))
	(add-alias result-name (funcall lisp-op i1 i2))
      (let ((res-reg (get-temporary-register '(A B)))
	    (opl-reg (get-temporary-register '(X0 X1 Y0 Y1))))
	(if (constantp i1)
	    (load-integer res-reg i1)
	  (move (get-any-address i1) res-reg))
	(if (constantp i2)
	    (load-integer opl-reg i2)
	  (convert (get-any-address i2) (get-type_56 i2) opl-reg 'integer))
	(if (member op1 '(AND OR))
	    (push `(,op1 ,opl-reg ,res-reg) pp)
	  (push `(NOT ,res-reg) pp))
	(if (member op2 '(AND OR))
	    (push `(,op2 ,opl-reg ,res-reg) pp)
	  (push `(NOT ,res-reg) pp))
	(if (temp-sig i1) (kill-temp i1))
	(if (temp-sig i2) (kill-temp i2))
	(update-var-and-reg result-name res-reg 'integer 'L 'S))))
  nil)

(defmacro <lognand_56> (result-name i1 i2)
  `(<log-op_56> ,result-name ,i1 ,i2 lognand AND NOT))

(defmacro <lognor_56> (result-name i1 i2)
  `(<log-op_56> ,result-name ,i1 ,i2 lognor OR NOT))

(defmacro <logandc1_56> (result-name i1 i2)
  `(<log-op_56> ,result-name ,i1 ,i2 logandc1 NOT AND))

(defmacro <logandc2_56> (result-name i1 i2)
  `(<log-op_56> ,result-name ,i2 ,i1 logandc1 NOT AND))

(defmacro <logorc1_56> (result-name i1 i2)
  `(<log-op_56> ,result-name ,i1 ,i2 logorc1 NOT OR))

(defmacro <logorc2_56> (result-name i1 i2)
  `(<log-op_56> ,result-name ,i2 ,i1 logorc1 NOT OR))


(defmacro <ur-break_56> (result-name ind brk no-args res &optional fstr &rest args)
  ;; set HF2&HF3, send signal that this is the break point handler
  ;; send length of string, then string (ascii)
  ;; send number of args, then each arg
  ;; drop into chip's break-point handler (assume Lisp has done something reasonable)
  (DEBUGGING (if args 
		 (push `(----------------> <ur-break> ,result-name ,ind ,brk ,no-args ,res ,fstr ,@args) pp)
	       (push `(----------------> <ur-break> ,result-name ,ind ,brk ,no-args ,res ,fstr) pp)))
  (let* ((blksiz (+ (if (not no-args) 3 2) (ceiling (length fstr) 3)))
	 (blkadr (get-Y-memory blksiz))
	 (newl (new-label))
	 (dol (new-label))
	 (r-reg (get-temporary-r-register)))
    (push `(Y-ORG ,blkadr) pp)
    (if fstr
	(if (not no-args)
	    (push `(Y-DATA ,(eval ind) ,(length fstr) ,@(make-dsp-string fstr) ,(length args)) pp)
	  (push `(Y-DATA ,(eval ind) ,(length fstr) ,@(make-dsp-string fstr)) pp))
      (push `(Y-DATA ,(eval ind) 0 ,(length args)) pp))
    (push `(BSET M-HF2 X-IO M-HCR) pp)	; set HF2
    (push `(BSET M-HF3 X-IO M-HCR) pp)	; set HF3 -- next56 now can tell we are awaiting data
    (push `(LOAD ,r-reg ,blkadr) pp)
    (push `(DO ,blksiz ,dol) pp)
    (push   `(,newl) pp)
    (push   `(JCLR M-HTDE X-IO M-HSR ,newl) pp)
    (push   `(MOVE Y ,r-reg R+1 X-IO M-HTX) pp)
    (push   `(NOP) pp)
    (push   `(,dol LOCAL) pp)
    ;; now pass back the arguments, if any
    (when (not no-args)
      (spill-ALU-registers)
      (loop for arg in args do
	(let* ((argl (if (alias arg) (eval arg) arg))
	       (argtyplab (new-label))
	       (arglab (new-label))
	       (arg-type (if (numberp argl) 
			     (if (integerp argl) 'integer
			       (if (< (abs argl) 1.0) 'fraction
				 'real))
			   (get-type_56 argl))))

	  (if (temp-sig argl) (kill-temp argl))
	  (push `(,argtyplab) pp)
	  (push `(JCLR M-HTDE X-IO M-HSR ,argtyplab) pp)

	  (if (eq arg-type 'integer)
	      (push `(STORE ,(eval %external-integer) X-IO M-HTX) pp)
	    (if (eq arg-type 'fraction)
		(push `(STORE ,(eval %external-fraction) X-IO M-HTX) pp)
	      (if (eq arg-type 'long-int)
		  (push `(STORE ,(eval %external-long-integer) X-IO M-HTX) pp)
		(if (eq arg-type 'real)
		    (push `(STORE ,(eval %external-real) X-IO M-HTX) pp)
		  (error "clm can't pass ~A arguments to lisp at run-time" arg-type)))))
	  
	  (push `(,arglab) pp)
	  (push `(JCLR M-HTDE X-IO M-HSR ,arglab) pp)

	  (if (numberp argl)
	      (if (eq arg-type 'integer)
		  (push `(STORE ,(make-integer argl) X-IO M-HTX) pp)
		(if (eq arg-type 'fraction)
		    (push `(STORE ,(make-fraction argl) X-IO M-HTX) pp)
		  (let ((high 0)
			(low 0)
			(arg2lab (new-label)))
		    (if (eq arg-type 'long-int)
			(multiple-value-setq
			    (high low) 
			  (make-long-int argl))
		      (multiple-value-setq 
			  (high low)
			(make-real argl)))
		    (push `(STORE ,high X-IO M-HTX) pp)
		    (push `(,arg2lab) pp)
		    (push `(JCLR M-HTDE X-IO M-HSR ,arg2lab) pp)
		    (push `(STORE ,low X-IO M-HTX) pp))))
	    (let ((argadr (get-any-address argl)))
	      (if (or (eq arg-type 'integer) (eq arg-type 'fraction))
		  (if (listp argadr)
		      (push `(MOVE ,@argadr X-IO M-HTX) pp)
		    (push `(STORE ,argadr X-IO M-HTX) pp))
		(let ((arg3lab (new-label)))
		  (if (listp argadr)
		      (push `(MOVE ,@(get-high-side argadr) X-IO M-HTX) pp)
		    (push `(STORE ,(get-high-side argadr) X-IO M-HTX) pp))
		  (push `(,arg3lab) pp)
		  (push `(JCLR M-HTDE X-IO M-HSR ,arg3lab) pp)
		  (if (listp argadr)
		      (push `(MOVE ,@(get-low-side argadr) X-IO M-HTX) pp)
		    (push `(STORE ,(get-low-side argadr) X-IO M-HTX) pp)))))))))
    (when res
      (let ((rlab1 (new-label))
	    (rlab2 (new-label)))
	(spill-ALU-registers '(A))
	(push `(,rlab1) pp)
	(push `(JCLR M-HRDF X-IO M-HSR ,rlab1) pp)
	(push `(LOAD A X-IO M-HRX) pp)
	(push `(,rlab2) pp)
	(push `(JCLR M-HRDF X-IO M-HSR ,rlab2) pp)
	(push `(LOAD A0 X-IO M-HRX) pp)
	(update-var-and-reg result-name 'A 'real 'L 'S)))
    (push `(BCLR M-HF3 X-IO M-HCR) pp)	; clear HF3 
    (push `(BCLR M-HF2 X-IO M-HCR) pp)	; clear HF2
    (if brk (push '(JSR .break) pp)))
  nil)

(defmacro <break_56> (result-name &optional fstr &rest args)
  (add-alias result-name <nil>)
  `(<ur-break_56> ,result-name %external-break t nil nil ,fstr ,@args))

(defmacro <warn_56> (result-name &optional fstr &rest args)
  (add-alias result-name <nil>)
  `(<ur-break_56> ,result-name %external-warn nil nil nil ,fstr ,@args))

(defmacro <error_56> (result-name &optional fstr &rest args)
  (add-alias result-name <nil>)
  `(<ur-break_56> ,result-name %external-error t nil nil ,fstr ,@args))

(defmacro <print_56> (result-name fstr)
  ;; this should take any object as fstr, and return it as the value of <print>
  (add-alias result-name <nil>)
  (if (stringp fstr)
      `(<ur-break_56> ,result-name %external-print nil t nil ,fstr)
    `(<ur-break_56> ,result-name %external-variable-print nil nil nil nil ,fstr)))

(defmacro <clm-print_56> (result-name &optional fstr &rest args)
  (add-alias result-name <nil>)
  `(<ur-break_56> ,result-name %external-clm-print nil nil nil ,fstr ,@args))

(defmacro <princ_56> (result-name fstr)
  ;; this should take any object as fstr, and return it as the value of <princ>
  (add-alias result-name <nil>)
  (if (stringp fstr)
      `(<ur-break_56> ,result-name %external-princ nil t nil ,fstr)
    `(<ur-break_56> ,result-name %external-variable-princ nil nil nil nil ,fstr)))

(defmacro <y-or-n-p_56> (result-name &optional fstr &rest args)
  `(<ur-break_56> ,result-name %external-y-or-n-p nil nil t ,fstr ,@args))

(defmacro <yes-or-no-p_56> (result-name &optional fstr &rest args)
  `(<ur-break_56> ,result-name %external-yes-or-no-p nil nil t ,fstr ,@args))

(defmacro <terpri_56> (result-name)
  `(<ur-break_56> ,result-name %external-terpri nil t nil))

(defmacro <apply_56> (result-name function-name &rest args)
  `(<ur-break_56> ,result-name %external-apply nil nil t ,function-name ,@args))

(defmacro <funcall_56> (result-name function-name &rest args)
  `(<ur-break_56> ,result-name %external-funcall nil nil t ,function-name ,@args))




;;; now support for run-time booleans, conditionals, etc

(defmacro <label_56> (arg)       (spill-ALU-registers) (check-stored-regs) (push `(,arg) pp) nil)
(defmacro <hide_56> (&rest args) (push `(HIDE ,@args) pp) nil)
(defmacro <jump_56> (arg)        (spill-ALU-registers) (check-stored-regs) (push `(JMP ,arg) pp) nil)
(defmacro <jump-true_56> (arg)   (spill-ALU-registers) (check-stored-regs) (push `(JNE ,arg) pp) nil) 
					;these come after comparison with <nil> via <null>
(defmacro <jump-false_56> (arg)  (spill-ALU-registers) (check-stored-regs) (push `(JEQ ,arg) pp) nil)

(defmacro <jump-l_56> (arg)   (spill-ALU-registers) (check-stored-regs) (push `(JLT ,arg) pp) nil)
(defmacro <jump-leq_56> (arg) (spill-ALU-registers) (check-stored-regs) (push `(JLE ,arg) pp) nil)
(defmacro <jump-neq_56> (arg) (spill-ALU-registers) (check-stored-regs) (push `(JNE ,arg) pp) nil)
(defmacro <jump-eq_56> (arg)  (spill-ALU-registers) (check-stored-regs) (push `(JEQ ,arg) pp) nil)
(defmacro <jump-geq_56> (arg) (spill-ALU-registers) (check-stored-regs) (push `(JGE ,arg) pp) nil)
(defmacro <jump-g_56> (arg)   (spill-ALU-registers) (check-stored-regs) (push `(JGT ,arg) pp) nil)

(defmacro <jump-mi_56> (arg)  (spill-ALU-registers) (check-stored-regs) (push `(JMI ,arg) pp) nil)
(defmacro <jump-pl_56> (arg)  (spill-ALU-registers) (check-stored-regs) (push `(JGT ,arg) pp) nil) 
					;JPL=not neg, but PLUSP=not neg and not 0

(defmacro <local-label_56> (arg) (spill-ALU-registers) (check-stored-regs) (init-N-regs) (push `(,arg LOCAL) pp) nil)

(defmacro <undefine_56> (&rest args)
  (loop for arg in args do (kill-temp arg)))

#|
;;; currently we use the DSP hardware stack and have no run-time memory allocation needs

; <push> (push `(STORE ,reg ,mem R7 R+1) pp))
; <pop>  (push `(LOAD ,reg ,mem R7 1-R) pp))
(defvar run-time-stack nil)

(defmacro <push_56> (result-name argl)
  (let* ((arg (if (alias argl) (eval argl) argl))
	 (res-typ (get-type_56 arg))
	 (res-reg (or (get-work-address arg)
		      (and (eq res-typ 'real)
		           (get-temporary-register '(A B X Y)))
		      (get-temporary-register '(A B X0 X1 Y0 Y1))))
	 (mem (if (eq res-typ 'real) 'L 'X))
	 (Rn (get-temporary-r-register)))
    (when (not run-time-stack)
      (setf run-time-stack (get-L-mem 16))
      (push `(setf-x-mem ,run-time-stack ,(+ 1 run-time-stack)) pup))
    (push `(LOAD ,Rn X ,run-time-stack) pp)
    (move (get-any-address arg) res-reg)
    (push `(STORE ,res-reg ,mem ,Rn R+1) pp)
    (push `(STORE ,Rn X ,run-time-stack) pp)
    (update-var-and-reg result-name res-reg res-typ 'L 'S))
  nil)

(defmacro <pop_56> (argl)
  (let* ((arg (if (alias argl) (eval argl) argl))
	 (res-typ (get-type_56 arg))
 	 (res-reg (or (and (eq res-typ 'real)
			   (get-temporary-register '(A B X Y)))
		      (get-temporary-register '(A B X0 X1 Y0 Y1))))
	 (mem (if (eq res-typ 'real) 'L 'X))
	 (Rn (get-temporary-R-register)))
    (when (not run-time-stack)
      (setf run-time-stack (get-L-mem 16))
      (push `(setf-x-mem ,run-time-stack ,(+ 1 run-time-stack)) pup))
    (push `(LOAD ,Rn X ,run-time-stack) pp)
    (push `(LOAD ,res-reg ,mem ,Rn 1-R) pp)
    (push `(STORE ,Rn X ,run-time-stack) pp)
    (update-var-and-reg arg res-reg res-typ 'L 'S))
  nil)

(defmacro <pop-and-toss_56> ()
  (if (null run-time-stack) (error "attempt to pop non-existent stack"))
  (let* ((rn (get-temporary-r-register)))
    (push `(LOAD ,Rn X ,run-time-stack) pp)
    (push `(UPDATE ,Rn R-1) pp)
    (push `(STORE ,Rn X ,run-time-stack) pp))
  nil)
|#

(defmacro <case_56> (index name-list label-list)
  (let ((t-case nil))
    (get-temporary-register '(B))
    (get-temporary-register '(A))
    (move (get-any-address index) 'A)
    (loop for name in name-list and address in label-list do
      (if (or (eq name 'T)
	      (eq name 'OTHERWISE))
	  (setf t-case address)
	(progn
	  (push `(LOAD B ,name) pp)
	  (push `(CMP B A) pp)
	  (push `(JEQ ,address) pp))))
    (if t-case				;fell through all other case branches
	(push `(JMP ,t-case) pp))
    (push `(HIDE ,@label-list) pp)
  nil))


(defmacro <logtest_56> (result-name a-i1 a-i2)
  (let ((i1 (if (alias a-i1) (eval a-i1) a-i1))
	(i2 (if (alias a-i2) (eval a-i2) a-i2)))
    (if (and (constantp i1) (constantp i2))
	(add-alias result-name (logtest i1 i2))
      (progn
	(if (constantp i1)
	    (load-number (get-temporary-register '(A)) i1)
	  (move (get-any-address i1) (get-temporary-register '(A))))
	(if (constantp i2)
	    (load-number (get-temporary-register '(X0)) i2)
	  (convert (get-any-address i2) (get-type_56 i2) (get-temporary-register '(X0)) 'integer))
	(push `(AND X0 A) pp)
	(push `(NOT A) pp)
	(if (temp-sig i1) (kill-temp i1))
	(if (temp-sig i2) (kill-temp i2))
	(update-var-and-reg result-name 'A 'integer 'L 'S))))
  nil)

(defmacro <test_56> (result-name argl)
  (declare (ignore result-name))
  (let* ((arg (if (alias argl) (eval argl) argl))
	 (res-reg (if (member arg (reg-list 'A)) 'A
		    (if (member arg (reg-list 'B)) 'B
		      (get-temporary-register '(A B))))))
    (if (temp-sig arg) (kill-temp arg))
    (move (get-any-address arg) res-reg)
    (push `(TST ,res-reg) pp))
  nil)

(defmacro <not-null_56> (argl)
  (let* ((arg (if (alias argl) (eval argl) argl))
	 (opl (get-temporary-register '(A B X0 X1 Y0 Y1)))
	 (res-reg (get-temporary-register '(A B) (list opl)))
	 (arg-adr (get-any-address arg)))
    (if (temp-sig arg) (kill-temp arg))
    (if (or (not (listp arg-adr))
	    (not (eq 'L (car arg-adr)))
	    (member opl '(A B)))
	(move arg-adr opl)
      (move (append (list 'X) (rest arg-adr)) opl))
    (push `(LOAD ,res-reg ,<nil>) pp)
    (push `(CMP ,opl ,res-reg) pp))
  nil)
	
(defmacro <logbitp_56> (result-name indl argl)
  ;; if alive, store, then use memory, not register -- if no home address, make one
  ;; necessary because BTST #n,D doesn't work
  (declare (ignore result-name))
  (let ((arg (if (alias argl) (eval argl) argl))
	(ind (if (alias indl) (eval indl) indl))
	(res-reg (get-temporary-register '(A B))))
    (if (temp-sig arg) (kill-temp arg))
    (if (temp-sig ind) (kill-temp ind))
    (if (constantp arg)
	(progn
	  (push `(LOAD ,res-reg ,(make-integer arg)) pp)
	  (push `(STORE ,res-reg X temp-loc) pp))
      (convert (get-any-address arg) (get-type_56 arg) '(X temp-loc) 'integer))
    (if (constantp ind)
	(progn
	  (push `(CLR ,res-reg) pp)
	  (push `(JCLR ,ind X temp-loc logit) pp)
	  (push `(LOAD ,res-reg 1) pp)
	  (push `(logit LOCAL) pp)
	  (push `(TST ,res-reg) pp))
      (let* ((other-reg (get-temporary-register '(X0 X1 Y0 Y1))))
	(convert (get-any-address ind) (get-type_56 ind) res-reg 'integer)
	(push `(TST ,res-reg) pp)	;protect against bit 0 case
	(push `(COPY ,res-reg ,other-reg) pp)
	(push `(LOAD ,res-reg 1) pp)
	(push `(JEQ zero-case) pp)
	(push `(REP ,other-reg) pp)
	(push `(ASL ,res-reg) pp)
	(push `(zero-case LOCAL) pp)
	(push `(AND ,other-reg ,res-reg) pp))))
  nil)

(defmacro <test-bit-0_56> (result-name arg)
  `(<logbitp_56> ,result-name 0 ,arg))
	
(defmacro <compare_56> (argl1 argl2)

  ;;someday we should optimize this to recognize arg1 or arg2 = 0 and use TST -- the problem is
  ;;that at this point we don't know what jumps are being generated after this comparison, so the
  ;;optimization has to happen either in DSP-IZE (where it's not easy to detect this case), or
  ;;in a separate pass over the intermediate representation -- <load reg 0> <load reg0 0> 
  ;;<compare> 0 name <jump-eq> could become <tst> name <jump-eq>, saving the two loads and so on.
  ;;The optimizer should also delete unreferenced labels, and <jump> label followed immediately by label.

  (DEBUGGING (push `(----------------> <compare> ,argl1 ,argl2) pp))
  (let* ((arg2 (if (alias argl1) (eval argl1) argl1))
	 (arg1 (if (alias argl2) (eval argl2) argl2))
	 ;; BEWARE!! FLIPPED ARGS!! (we think of <compare> opposite to the 56000's way of doing it)
	 (res-type (round-up-type (get-type_56 arg1) (get-type_56 arg2)))
	 (res-reg (get-temporary-register '(A B)))
	 (opr-reg (if (member res-type '(real long-int)) ;CMP X1 A, for example, only looks at A1, so it's not what we want for reals
		      (get-temporary-register '(A B) (list res-reg))
		    (get-temporary-register '(X0 X1 Y0 Y1 A B) (list res-reg)))))
    (when (constantp arg1)
      (case res-type 
	(integer (load-integer opr-reg arg1))
	(long-int (load-long-int opr-reg arg1))
	(fraction (load-fraction opr-reg arg1))
	(t (load-real opr-reg arg1)))
      (mark-reg opr-reg (list argl2 'L 'NS)))
    (when (constantp arg2)
      (case res-type 
	(integer (load-integer res-reg arg2))
	(long-int (load-long-int res-reg arg2))
	(fraction (load-fraction res-reg arg2))
	(t (load-real res-reg arg2)))
      (mark-reg res-reg (list argl1 'L 'NS)))
    (when (not (constantp arg1))
      (if (temp-sig arg1) (kill-temp arg1))
      (convert (get-any-address arg1) (get-type_56 arg1) opr-reg res-type)
      (mark-reg opr-reg (list arg1 'L 'NS)))
    (when (not (constantp arg2))
      (if (temp-sig arg2) (kill-temp arg2))
      (convert (get-any-address arg2) (get-type_56 arg2) res-reg res-type))
    (push `(CMP ,opr-reg ,res-reg) pp))
  nil)



;;; OSCIL
;;;
;;; each OSCIL structure needs 3 words of static space on chip, and these must be initialized.
;;; phase freq fm (home address points to phase)

(defmacro <oscil_56> (result-name s &optional fm-1 pm-1)
  (let ((fm (if (alias fm-1) (eval fm-1) fm-1))
	(pm (if (alias pm-1) (eval pm-1) pm-1))
	(pm-ok nil))
    (DEBUGGING (push `(----------------> <oscil> ,result-name ,s ,fm ,pm) pp))
    (need-loaded '.oscil)
    (let ((addr (cadr (get-home-address s 'osc 'R2))))
      (when (/= 2 (getf-N-reg 'N2)) 
	(push `(LOAD N2 2 SHORT) pp)
	(setf-N-reg 'N2 2))
      (if (temp-sig s) (kill-temp s))
      (when fm
	(if (temp-sig fm) (kill-temp fm))
	(if (eq addr 'R2)		;i.e. osc is indirect through R2 (aref), so fm addr must be too
	    (convert (get-any-address fm) (get-type_56 fm) '(L R2 RN) 'real)
	  (progn
	    (if (and (not pm) 
		     (eq (get-any-address fm) 'B))
		(progn
		  (get-register 'A)
		  (push '(CLR A) pp)
		  (setf pm-ok t)))
	    (convert (get-any-address fm) (get-type_56 fm) (list 'L (+ addr 2)) 'real))))
      (spill-ALU-registers (libinfo-uses .oscil))
      (if pm
	  (progn
	    (if (temp-sig pm) (kill-temp pm))
	    (convert (get-any-address pm) (get-type_56 pm) 'A 'real))
	(if (not pm-ok) (push `(CLR A) pp)))
      (when (not (eq addr 'R2))		;i.e. when R2 isn't already set via aref
	(get-register 'R2)
	(if (< addr 256)
	    (push `(LOAD R2 ,addr SHORT) pp) ;SHORT check needs to be explicit (somewhat dumb assembler)
	  (push `(LOAD R2 ,addr) pp))))
    ;; this order makes it more likely the optimizer can combine operations
    (push `(JSR .oscil) pp)
    (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)


;;; SQUARE-WAVE and friends (TRIANGLE, SAWTOOTH, PULSE)

(defmacro <simple-wave_56> (result-name s fm-1 lib-op typ offset)
  (let ((fm (if (alias fm-1) (eval fm-1) fm-1))
	(addr (cadr (get-home-address s 'sw 'R3))))
    (DEBUGGING (push `(----------------> < ,lib-op > ,result-name ,s ,fm) pp))
    (need-loaded lib-op)
    (if (not (eq addr 'R3))
	(progn
	  (get-register 'R3)
	  (if (< addr 255)
	      (push `(LOAD R3 ,(+ addr offset) SHORT) pp)
	    (push `(LOAD R3 ,(+ addr offset)) pp)))
      (push `(UPDATE R3 R+1) pp))

    (when fm
      (if (temp-sig fm) (kill-temp fm))
      (if (eq addr 'R3)
	  (progn			;set N3 to 2 because R3 already is addr+1
	    (when (/= (- 3 offset) (getf-N-reg 'N3))
	      (push `(LOAD N3 ,(- 3 offset) SHORT) pp)
	      (setf-N-reg 'N3 (- 3 offset)))
	    (convert (get-any-address fm) (get-type_56 fm) '(L R3 RN) 'real))
	(convert (get-any-address fm) (get-type_56 fm) (list 'L (+ addr 2 offset)) 'real)))

    (if (not (member lib-op '(.pulse-train .sum-of-cosines)))
	(when (/= 4 (getf-N-reg 'N3))
	  (push `(LOAD N3 4 SHORT) pp)
	  (setf-N-reg 'N3 4)))

    (if (temp-sig s) (kill-temp s))
    (spill-ALU-registers (libinfo-uses (eval lib-op)))
    (push `(JSR ,lib-op) pp)
    (update-var-and-reg result-name 'A typ 'L 'S))
  nil)

(defmacro <square-wave_56> (result-name s &optional fm)
  `(<simple-wave_56> ,result-name ,s ,fm .square-wave real 1))

(defmacro <pulse-train_56> (result-name s &optional fm)
  `(<simple-wave_56> ,result-name ,s ,fm .pulse-train real 1))

(defmacro <triangle-wave_56> (result-name s &optional fm)
  `(<simple-wave_56> ,result-name ,s ,fm .triangle-wave real 1))

(defmacro <sawtooth-wave_56> (result-name s &optional fm)
  `(<simple-wave_56> ,result-name ,s ,fm .sawtooth-wave real 1))

(defmacro <sum-of-cosines_56> (result-name s &optional fm)
  `(<simple-wave_56> ,result-name ,s ,fm .sum-of-cosines real 0))


(defmacro <ur-table-lookup_56> (result-name s &optional fm-1)
  (let ((fm (if (alias fm-1) (eval fm-1) fm-1))
	(addr (cadr (get-home-address s 'tbl 'R3))))
    (DEBUGGING (push `(----------------> <ur-table-lookup> ,result-name ,s ,fm) pp))
    (need-loaded '.ur-table-lookup)
    (when (not (eq addr 'R3))
      (get-register 'R3)
      (if (< addr 255)
	  (push `(LOAD R3 ,addr SHORT) pp)
	(push `(LOAD R3 ,addr) pp)))

    (when fm
      (if (temp-sig fm) (kill-temp fm))
      (if (eq addr 'R3)
	  (progn
	    (when (/= 2 (getf-N-reg 'N3))
	      (push `(LOAD N3 2 SHORT) pp)
	      (setf-N-reg 'N3 2))
	    (convert (get-any-address fm) (get-type_56 fm) '(L R3 RN) 'real))
	(convert (get-any-address fm) (get-type_56 fm) (list 'L (+ addr 2)) 'real)))

    (if (temp-sig s) (kill-temp s))
    (spill-ALU-registers (libinfo-uses .ur-table-lookup))
    (get-register 'Y1)
    (if (eq addr 'R3)
	(progn
	  (when (/= 5 (getf-N-reg 'N3))
	    (push `(LOAD N3 5 SHORT) pp)
	    (setf-N-reg 'N3 5))
	  (push `(LOAD Y1 Y R3 RN) pp))
      (push `(LOAD Y1 Y ,(+ addr 5)) pp))
    (push `(STORE Y1 Y temp-loc-1) pp)
    (setf-N-reg 'N3 3)			;set in lib56
    (push `(JSR .ur-table-lookup) pp)
    (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)
    

(defmacro <randx_56> (result-name s fm-1 lib-op typ)
  (let ((fm (if (alias fm-1) (eval fm-1) fm-1)))
    (DEBUGGING (push `(----------------> < ,lib-op > ,result-name ,s ,fm) pp))
    (need-loaded lib-op)
    (let ((addr (cadr (get-home-address s 'noi 'R3))))
      (when (not (eq addr 'R3))
	(get-register 'R3)
	(if (< addr 255)
	    (push `(LOAD R3 ,addr SHORT) pp)
	  (push `(LOAD R3 ,addr) pp)))
      (when (/= 2 (getf-N-reg 'N3))
	(push `(LOAD N3 2 SHORT) pp)
	(setf-N-reg 'N3 2))
      (when fm
	(if (temp-sig fm) (kill-temp fm))
	(if (eq addr 'R3)
	    (convert (get-any-address fm) (get-type_56 fm) '(L R3 RN) 'real)
	  (convert (get-any-address fm) (get-type_56 fm) (list 'L (+ addr 2)) 'real))))
    (if (temp-sig s) (kill-temp s))
    (spill-ALU-registers (libinfo-uses (eval lib-op)))
    (push `(JSR ,lib-op) pp)
    (update-var-and-reg result-name 'A typ 'L 'S))
  nil)

(defmacro <randh_56> (result-name s &optional fm)
  `(<randx_56> ,result-name ,s ,fm .randh real))

(defmacro <randi_56> (result-name s &optional fm)
  `(<randx_56> ,result-name ,s ,fm .randi real))

	
(defmacro <amplitude-modulate_56> (result-name cr-1 s1-1 s2-1)
  ;; s1*(cr+s2)
  (let* ((s1 (if (alias s1-1) (eval s1-1) s1-1))
	 (s2 (if (alias s2-1) (eval s2-1) s2-1))
	 (cr (if (alias cr-1) (eval cr-1) cr-1))
	 (s1-reg (get-work-address s1))
	 (s1-xy-reg (or (and s1-reg (or (X-reg s1-reg) (Y-reg s1-reg)))
			(get-temporary-register '(X0 X1 Y0 Y1))))
	 (s2-reg (get-work-address s2))
	 (s2-xy-reg (or (and s2-reg (or (X-reg s2-reg) (Y-reg s2-reg)))
			(get-temporary-register '(X0 X1 Y0 Y1))))
	 (cr-reg (get-work-address cr))
	 (cr-xy-reg (or (and cr-reg (or (X-reg cr-reg) (Y-reg cr-reg)))
			(get-temporary-register '(X0 X1 Y0 Y1))))
	 (res-reg (get-temporary-register '(A B))))
    (if (and (> run-safety 0) (not (eq (get-type_56 s1) 'fraction))) (check-for-fractional-overflow s1 (get-any-address s1) 2))
    (convert (get-any-address s1) (get-type_56 s1) s1-xy-reg 'fraction)
    (mark-reg s1-xy-reg (list s1 'L 'NS))
    (if (and (> run-safety 0) (not (eq (get-type_56 s2) 'fraction))) (check-for-fractional-overflow s2 (get-any-address s2) 3))
    (convert (get-any-address s2) (get-type_56 s2) s2-xy-reg 'fraction)
    (mark-reg s2-xy-reg (list 'L 'NS))
    (if (and (> run-safety 0) (not (eq (get-type_56 cr) 'fraction))) (check-for-fractional-overflow cr (get-any-address cr) 4))
    (convert (get-any-address cr) (get-type_56 cr) cr-xy-reg 'fraction)
    (push `(MPY ,s1-xy-reg ,s2-xy-reg ,res-reg) pp)
    (push `(MACR ,s1-xy-reg ,cr-xy-reg ,res-reg) pp)
    (if (temp-sig s1) (kill-temp s1))
    (if (temp-sig s2) (kill-temp s2))
    (if (temp-sig cr) (kill-temp cr))
    (update-var-and-reg result-name res-reg 'fraction 'L 'S))
  nil)

(defmacro <polynomial_56> (result-name tab x)
  (DEBUGGING (push `(----------------> <polynomial> ,result-name ,tab ,x) pp))
  (let* ((xin (get-any-address x))
	 (r-reg (get-temporary-r-register))
	 (end-label (new-label))
	 (res-reg (get-temporary-register '(A B)))
	 (opl (get-temporary-register '(X0 X1 Y0 Y1)))
	 (opr (get-temporary-register '(X0 X1 Y0 Y1) (list opl)))
	 (do-reg (get-temporary-register '(R0 R3 R4 R5 R6 R7 R1 A B X0 X1 Y0 Y1) (list opl opr res-reg r-reg)))
	 (tabin (cadr (get-home-address tab))))
    (if (not (R-register tabin))
	(progn
	  (push `(LOAD ,r-reg Y ,(1+ tabin)) pp) ;top element address
	  (push `(LOAD ,do-reg X ,(1+ tabin)) pp)) ;number of elements - 1
      (progn
	(push `(UPDATE ,tabin R+1) pp)
	(push `(NOP) pp)
	(push `(LOAD ,r-reg Y ,tabin R) pp)
	(push `(LOAD ,do-reg X ,tabin R) pp)))
    (push `(LOAD ,opr Y ,r-reg R-1) pp) ;get top coeff, decrement table address reg
    (if (and (> run-safety 0) (not (eq (get-type_56 x) 'fraction))) (check-for-fractional-overflow x xin 5))
    (convert xin (get-type_56 x) opl 'fraction)
    (if (temp-sig x) (kill-temp x))
    (if (temp-sig tab) (kill-temp tab))
    (push `(LOAD ,res-reg Y ,r-reg R-1) pp)
    (push `(DO ,do-reg ,end-label) pp)
    (push `(MAC ,opr ,opl ,res-reg) pp)
    (push `(COPY ,res-reg ,opr) pp)	;final time this is unnecessary
    (push `(LOAD ,res-reg Y ,r-reg R-1) pp)
    (push `(,end-label) pp)
    (push `(HIDE ,end-label) pp)
    (update-var-and-reg result-name opr 'fraction 'L 'S)
    nil))

(defmacro <dot-product_56> (result-name s1 s2)
  (DEBUGGING (push `(----------------> <dot-product> ,result-name ,s1 ,s2) pp))
  (need-loaded '.table-dot-product)
  (let ((addr1 (cadr (get-home-address s1 'table)))
	(addr2 (cadr (get-home-address s2 'table))))
    (get-register 'R3)
    (if (< addr1 256)
	(push `(LOAD R3 ,addr1 SHORT) pp)
      (push `(LOAD R3 ,addr1) pp))
    (get-register 'R4)
    (if (< addr2 256)
	(push `(LOAD R4 ,addr2 SHORT) pp)
      (push `(LOAD R4 ,addr2) pp))
    (if (temp-sig s1) (kill-temp s1))
    (if (temp-sig s2) (kill-temp s2))
    (spill-ALU-registers (libinfo-uses .table-dot-product))
    (push `(JSR .table-dot-product) pp)
    (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)

(defmacro <table-interp_56> (result-name s index)
  (let ((fm (if (alias index) (eval index) index)))
    (DEBUGGING (push `(----------------> <table-interp> ,result-name ,s ,fm) pp))
    (need-loaded '.table-interp)
    (let ((addr (cadr (get-home-address s 'table 'R3))))
      (when (not (eq addr 'R3))		;i.e. when R2 isn't already set via aref
	(get-register 'R3)
	(if (< addr 256)
	    (push `(LOAD R3 ,addr SHORT) pp)
	  (push `(LOAD R3 ,addr) pp)))
      (setf-N-reg 'N4 -1)
      (convert (get-any-address fm) (get-type_56 fm) 'A 'real)
      (if (temp-sig s) (kill-temp s))
      (if (temp-sig fm) (kill-temp fm))
      (spill-ALU-registers (libinfo-uses .table-interp))
      (push `(JSR .table-interp) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)


(defmacro <array-interp_56> (result-name s index)
  (let ((fm (if (alias index) (eval index) index)))
    (DEBUGGING (push `(----------------> <array-interp> ,result-name ,s ,fm) pp))
    (need-loaded '.array-interp)
    (let ((addr (cadr (get-home-address s 'array 'R3))))
      (when (not (eq addr 'R3))		;i.e. when R3 isn't already set via aref
	(get-register 'R3)
	(if (< addr 256)
	    (push `(LOAD R3 ,addr SHORT) pp)
	  (push `(LOAD R3 ,addr) pp)))
      (setf-N-reg 'N4 -1)
      (convert (get-any-address fm) (get-type_56 fm) 'A 'real)
      (if (temp-sig fm) (kill-temp fm))
      (if (temp-sig s) (kill-temp s))
      (spill-ALU-registers (libinfo-uses .array-interp))
      (push `(JSR .array-interp) pp)
      (update-var-and-reg result-name 'A 'real 'L 'S)))
  nil)


;;; SIMPLE FILTERS

(defmacro <one-pole_56> (result-name filt in-argl)
  ;; filter coeffs are divided by two when loaded, so that we can use fractional arithmetic here
  ;; stored X:base=a0, Y:base=b1, X:(base+1)=y(n-1)
  ;; y(n):=a0 * x(n) - b1 * y(n-1) (we are following JOS "Digital Filter Theory" and Dodge "Computer Music" here,
  ;;                                not Schaffer and Oppenheim "Digital Signal Processing" and the Motorola documentation)
  ;;                                The latter assume "a0" = 1.0, swap a and b as names, and use + rather than - for y terms.
  ;;                                (I think the sign is a free parameter and is used to optimize noise floors and what not).
  (DEBUGGING (push `(----------------> <one-pole> ,result-name ,filt ,in-argl) pp))
  (let* ((r-reg nil)
	 (f-addr (cadr (get-home-address filt 'one-pole))))
    (if (not (r-register f-addr))
	(progn
	  (setf r-reg (get-temporary-R-register))
	  (if (< f-addr 255)
	      (push `(LOAD ,r-reg ,f-addr SHORT) pp)
	    (push `(LOAD ,r-reg ,f-addr) pp)))
      (setf r-reg f-addr))
    (let* ((in-arg (if (alias in-argl) (eval in-argl) in-argl))
	   (res-reg (get-temporary-register '(A B)))
	   (arg-reg (if (or (member in-arg (reg-list 'X))
			    (member in-arg (reg-list 'Y)))
			(get-work-address in-arg)
		      (get-temporary-register '(X0 X1 Y0 Y1))))
	   (coeff-reg (get-temporary-register (if (eq (get-full-reg arg-reg) 'X) '(Y) '(X)))))

      (if (temp-sig in-arg) (kill-temp in-arg))
      (if (temp-sig filt) (kill-temp filt))
      (if (and (> run-safety 0) (not (eq (get-type_56 in-arg) 'fraction))) (check-for-fractional-overflow in-arg (get-any-address in-arg) 6))
      (convert (get-any-address in-arg) (get-type_56 in-arg) arg-reg 'fraction)
      (push `(LOAD ,coeff-reg L ,r-reg R+1) pp)                        ;a0 and b1
      (push `(MPY ,arg-reg ,(get-high-side coeff-reg) ,res-reg) pp)    ;X->X1, Y->X0
      (get-register arg-reg)
      (push `(LOAD ,arg-reg X ,r-reg R) pp)                            ;y(n-1)
      (push `(SMACR ,(get-low-side coeff-reg) ,arg-reg ,res-reg) pp)
      (push `(ASL ,res-reg) pp)		                               ;fixup for coeff/2 kludge
      (push `(STORE ,res-reg X ,r-reg R) pp)                           ;save output for next sample
      (update-var-and-reg result-name res-reg 'fraction 'L 'S))
    nil))

(defmacro <one-zero_56> (result-name filt in-argl)
  ;; filter coeffs are divided by two when loaded, so that we can use fractional arithmetic here
  ;; stored X:base=a0, Y:base=x(n-1), X:(base+1)=a1 
  ;; y(n):= a0 * x(n) + a1 * x(n-1)
  (DEBUGGING (push `(----------------> <one-zero> ,result-name ,filt ,in-argl) pp))
  (let* ((r-reg nil)
	 (f-addr (cadr (get-home-address filt 'one-zero))))
    (if (not (r-register f-addr))
	(progn
	  (setf r-reg (get-temporary-R-register))
	  (if (< f-addr 255)
	      (push `(LOAD ,r-reg ,f-addr SHORT) pp)
	    (push `(LOAD ,r-reg ,f-addr) pp)))
      (setf r-reg f-addr))
    (let* ((in-arg (if (alias in-argl) (eval in-argl) in-argl))
	   (res-reg (get-temporary-register '(A B)))
	   (arg-reg (if (or (member in-arg (reg-list 'X))
			    (member in-arg (reg-list 'Y)))
			(get-work-address in-arg)
		      (get-temporary-register '(X0 X1 Y0 Y1))))
	   (coeff-reg (get-temporary-register (if (eq (get-full-reg arg-reg) 'X) '(Y) '(X)))))

      (if (temp-sig in-arg) (kill-temp in-arg))
      (if (temp-sig filt) (kill-temp filt))
      (if (and (> run-safety 0) (not (eq (get-type_56 in-arg) 'fraction))) (check-for-fractional-overflow in-arg (get-any-address in-arg) 7))
      (convert (get-any-address in-arg) (get-type_56 in-arg) arg-reg 'fraction)
      ;; we hope that all that caused at least one instruction since the r-reg load.
      (push `(LOAD ,coeff-reg L ,r-reg R) pp)            ;a0 and x(n-1)
      (push `(MPY ,arg-reg ,(get-high-side coeff-reg) ,res-reg) pp)
      (push `(STORE ,arg-reg Y ,r-reg R+1) pp)
      (push `(LOAD ,(get-high-side coeff-reg) X ,r-reg R) pp)            ;a1
      (push `(MAC ,(get-high-side coeff-reg) ,(get-low-side coeff-reg) ,res-reg) pp)
      (push `(ASL ,res-reg) pp)		                 ;fixup for coeff/2 kludge
      (update-var-and-reg result-name res-reg 'fraction 'L 'S))
    nil))

(defmacro <two-pole_56> (result-name filt in-argl)
  ;; filter coeffs are divided by two when loaded, so that we can use fractional arithmetic here
  ;; stored X:base=a0, Y:base=b1, X:(base+1)=y(n-1), Y:(base+1)=b2, Y:(base+2)=y(n-2)
  ;; y(n):= a0 * x(n) - b1 * y(n-1) - b2 * y(n-2) 
  (DEBUGGING (push `(----------------> <two-pole> ,result-name ,filt ,in-argl) pp))
  (let* ((r-reg nil)
	 (f-addr (cadr (get-home-address filt 'two-pole))))
    (if (not (r-register f-addr))
	(progn
	  (setf r-reg (get-temporary-R-register))
	  (if (< f-addr 255)
	      (push `(LOAD ,r-reg ,f-addr SHORT) pp)
	    (push `(LOAD ,r-reg ,f-addr) pp)))
      (setf r-reg f-addr))
    ;; i.e. if we're already in an index register (via AREF), use that register if possible.
    (let* ((in-arg (if (alias in-argl) (eval in-argl) in-argl))
	   (res-reg (get-temporary-register '(A B)))
	   (arg-reg (if (or (member in-arg (reg-list 'X))
			    (member in-arg (reg-list 'Y)))
			(get-work-address in-arg)
		      (get-temporary-register '(X0 X1 Y0 Y1))))
	   (coeff-reg (get-temporary-register (if (eq (get-full-reg arg-reg) 'X) '(Y) '(X)))))

      (if (temp-sig in-arg) (kill-temp in-arg))
      (if (temp-sig filt) (kill-temp filt))
      (if (and (> run-safety 0) (not (eq (get-type_56 in-arg) 'fraction))) (check-for-fractional-overflow in-arg (get-any-address in-arg) 8))
      (convert (get-any-address in-arg) (get-type_56 in-arg) arg-reg 'fraction)
      ;; we hope that all that caused at least one instruction since the r-reg load.
      (push `(LOAD ,coeff-reg L ,r-reg R+1) pp)                        ;a0 and b1
      (push `(MPY ,arg-reg ,(get-high-side coeff-reg) ,res-reg) pp)
      (get-register arg-reg)
      (push `(LOAD ,arg-reg X ,r-reg R) pp)                            ;y(n-1)
      (push `(SMAC ,(get-low-side coeff-reg) ,arg-reg ,res-reg) pp)
      (push `(LOAD ,(get-low-side coeff-reg) Y ,r-reg R+1) pp)         ;b2
      (push `(LOAD ,(get-high-side coeff-reg) Y ,r-reg R) pp)          ;y(n-2)
      (push `(SMACR ,(get-low-side coeff-reg) ,(get-high-side coeff-reg) ,res-reg) pp)
      (push `(STORE ,arg-reg Y ,r-reg R-1) pp)                         ;y(n-2) <- y(n-1)
      (push `(ASL ,res-reg) pp)		                               ;fixup for coeff/2 kludge
      (push `(STORE ,res-reg X ,r-reg R) pp)                           ;y(n-1) <- y(n)
      (update-var-and-reg result-name res-reg 'fraction 'L 'S))
    nil))

(defmacro <two-zero_56> (result-name filt in-argl)
  ;; filter coeffs are divided by two when loaded, so that we can use fractional arithmetic here
  ;; stored X:base=a0, X:(base+1)=a1, Y:base=x(n-1), X:(base+2)=a2, Y:(base+2)=x(n-2)
  ;; y(n):= a0 * x(n) + a1 * x(n-1) + a2 * x(n-2)
  (DEBUGGING (push `(----------------> <two-zero> ,result-name ,filt ,in-argl) pp))
  (let* ((r-reg nil)
	 (f-addr (cadr (get-home-address filt 'two-zero))))
    (if (not (r-register f-addr))
	(progn
	  (setf r-reg (get-temporary-R-register))
	  (if (< f-addr 255)
	      (push `(LOAD ,r-reg ,f-addr SHORT) pp)
	    (push `(LOAD ,r-reg ,f-addr) pp)))
      (setf r-reg f-addr))
    (let ((n-reg (get-N-reg r-reg)))
      (when (/= 2 (getf-N-reg n-reg))
	(push `(LOAD ,n-reg 2 SHORT) pp)
	(setf-N-reg n-reg 2))
      (let* ((in-arg (if (alias in-argl) (eval in-argl) in-argl))
	     (res-reg (get-temporary-register '(A B)))
	     (arg-reg (if (or (member in-arg (reg-list 'X))
			      (member in-arg (reg-list 'Y)))
			  (get-full-reg (get-work-address in-arg))
			(get-temporary-register '(X Y))))
	     (in-arg-reg (if (eq arg-reg (get-full-reg (get-work-address in-arg)))
			     (get-work-address in-arg)
			   (get-high-side arg-reg)))
	     (coeff-reg (get-temporary-register (if (eq (get-full-reg arg-reg) 'X) '(Y) '(X)))))
	(if (temp-sig in-arg) (kill-temp in-arg))
	(if (temp-sig filt) (kill-temp filt))
	(if (and (> run-safety 0) (not (eq (get-type_56 in-arg) 'fraction))) (check-for-fractional-overflow in-arg (get-any-address in-arg) 9))
	(convert (get-any-address in-arg) (get-type_56 in-arg) in-arg-reg 'fraction)
	;; we hope that all that caused at least one instruction since the r-reg load.
	(push `(LOAD ,(get-high-side coeff-reg) X ,r-reg R) pp)            ;a0
	(push `(MPY ,in-arg-reg ,(get-high-side coeff-reg) ,res-reg) pp)
	;;slight inefficiency here in storage so that all 4 of these use compatible on-chip addresses,
	;;so that accessor functions in run.lisp can be straight-forward
	(push `(LOAD ,(get-low-side coeff-reg) Y ,r-reg R+1) pp)           ;x(n-1)
	(push `(LOAD ,(get-high-side coeff-reg) X ,r-reg R-1) pp)          ;a1
	(push `(STORE ,in-arg-reg Y ,r-reg R+N) pp)                        ;x(n-1) <- x(n)
	(push `(MAC ,(get-high-side coeff-reg) ,(get-low-side coeff-reg) ,res-reg) pp)
	(push `(LOAD ,arg-reg L ,r-reg R) pp)                              ;a2 and x(n-2)
	(push `(MAC ,(get-high-side arg-reg) ,(get-low-side arg-reg) ,res-reg) pp)
	(push `(STORE ,(get-low-side coeff-reg) Y ,r-reg R) pp)            ;x(n-2) <- x(n-1)
	(push `(ASL ,res-reg) pp)	;fixup for coeff/2 kludge
	(update-var-and-reg result-name res-reg 'fraction 'L 'S)))
    nil))


;;; basic-filter handles direct-form, lattice-form, and ladder-form filters of
;;; arbitrary order, FIR and/or IIR.  Coefficients currently must be fractional.

(defmacro <basic-filter_56> (result-name s in-argl libname)
  (let ((fm (if (alias in-argl) (eval in-argl) in-argl)))
    (DEBUGGING (push `(----------------> <filter> ,result-name ,s ,fm) pp))
    ;; might want to push coeff fractional check onto pup
    (need-loaded libname)
    (let ((addr (cadr (get-home-address s 'flt 'R2))))
      (when (not (eq addr 'R2))
	(get-register 'R2)
	(if (< addr 256)
	    (push `(LOAD R2 ,addr SHORT) pp)
	  (push `(LOAD R2 ,addr) pp)))
      (setf-N-reg 'N2 -1)
      (setf-N-reg 'N5 -1)
      (if (temp-sig fm) (kill-temp fm))
      (if (and (> run-safety 0) (not (eq (get-type_56 fm) 'fraction))) (check-for-fractional-overflow fm (get-any-address fm) 10))
      (convert (get-any-address fm) (get-type_56 fm) 'A 'fraction)
      (if (temp-sig s) (kill-temp s))
      (spill-ALU-registers (libinfo-uses (eval libname)))
      (push `(JSR ,libname) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)

(defmacro <filter_56> (result-name s in-argl)
  `(<basic-filter_56> ,result-name ,s ,in-argl .filter))

(defmacro <direct-filter_56> (result-name s in-argl)
  `(<basic-filter_56> ,result-name ,s ,in-argl .direct-filter))

(defmacro <lattice-filter_56> (result-name s in-argl)
  `(<basic-filter_56> ,result-name ,s ,in-argl .lattice-filter))

(defmacro <ladder-filter_56> (result-name s in-argl)
  `(<basic-filter_56> ,result-name ,s ,in-argl .ladder-filter))


(defvar checked-variables nil)

(defmacro <check_56> (ref &optional (id 2))
    (if (and (listp ref)
	     (eq (car ref) 'AREF))
	(when (not (member (cadr ref) checked-variables))
	  (progn
	    (push (cadr ref) checked-variables)
	    (push `(dotimes (i (array-total-size ,(cadr ref))) 
		     (if (get-dly-ext-id (aref ,(cadr ref) i))
			 (push (list ,id (get-dly-ext-id (aref ,(cadr ref) i)) 0) *sigarr*))) pup)
	    ;; we assume that if an array of delay lines is used, it is used sequentially and as a block
	    ;; that is -- each such array should handle one operation and not be split up in funny ways.
	    ;; If split, we'll get a loop for each use -- I suppose the DO and DOTIMES calls could be
	    ;; made smarter, but is it worth the trouble?
	    (if (member (get-element-type (cadr ref)) '(dly cmbflt allpassflt zdly))
		(progn
		  (push (list `(LOAD R5 ,(cadr (get-home-address (cadr ref))))) p-init)
		  (push (list `(JSR .R-array-clear-x-or-y-memory)) p-init)))))
      (when (not (member ref checked-variables))
	(push ref checked-variables)
	(push `(if (get-dly-ext-id ,ref) (push (list ,id (get-dly-ext-id ,ref) 0) *sigarr*)) pup)))
  nil)

(defun get-dly-ext-id (ref)
  (if ref
      (typecase ref
	(dly (dly-ext-id ref))
	(zdly (dly-ext-id (zdly-del ref)))
	(cmbflt (dly-ext-id (cmbflt-dly-unit ref)))
	(allpassflt (dly-ext-id (allpassflt-dly-unit ref)))
	(t (error "bad ref for dly chk: ~S" ref)))))

(defmacro <delay_56> (result-name del in-argl)
  (let ((in-arg (if (alias in-argl) (eval in-argl) in-argl)))
    (DEBUGGING (push `(----------------> <delay> ,result-name ,del ,in-arg) pp))
    (need-loaded '.delay)
    (let* ((d-addr (cadr (get-home-address del 'dly 'R4)))
	   (addr (if (eq d-addr 'R4) nil (+ d-addr 1))) ;.delay expects pointer to 2nd word
	   (arg-addr (get-any-address in-arg)))

      (if (not (eq d-addr 'R4))
	  (progn
	    (get-register 'R4)
	    (need-loaded '.R-clear-x-or-y-memory)
	    (if (< addr 255)
		(progn
		  (push `(LOAD R4 ,addr SHORT) pp)
		  (push (list `(LOAD R4 ,(- addr 1) SHORT)) p-init))
	      (progn
		(push `(LOAD R4 ,addr) pp)
		(push (list `(LOAD R4 ,(- addr 1))) p-init)))
	    (push (list `(JSR .R-clear-x-or-y-memory)) p-init))
	(progn
	  (need-loaded '.R-array-clear-x-or-y-memory)
	  ;; if eq R4, it's an array reference -- <check> will handle the p-init push (need array name)
	  (push `(UPDATE R4 R+1) pp)))

      (if (temp-sig in-arg) (kill-temp in-arg))
      (if (temp-sig del) (kill-temp del))
      (spill-ALU-registers (libinfo-uses .delay)) ;includes R5
      (if (and (> run-safety 0) (not (eq (get-type_56 in-arg) 'fraction))) (check-for-fractional-overflow in-arg arg-addr 1))
      (convert arg-addr (get-type_56 in-arg) 'X0 'fraction)
      (push `(JSR .delay) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)

(defmacro <tap_56> (result-name del &optional offset)
  (DEBUGGING (push `(----------------> <tap> ,result-name ,del) pp))
  (need-loaded '.tap)
  (let* ((d-addr (cadr (get-home-address del 'dly 'R4)))
	 (addr (if (eq d-addr 'R4) nil (+ d-addr 1))) ;.delay expects pointer to 2nd word
	 (arg-addr (if offset (get-any-address offset))))
      (if (not (eq d-addr 'R4))
	  (progn
	    (get-register 'R4)
	    (need-loaded '.R-clear-x-or-y-memory)
	    (if (< addr 255)
		(progn
		  (push `(LOAD R4 ,addr SHORT) pp)
		  (push (list `(LOAD R4 ,(- addr 1) SHORT)) p-init))
	      (progn
		(push `(LOAD R4 ,addr) pp)
		(push (list `(LOAD R4 ,(- addr 1))) p-init)))
	    (push (list `(JSR .R-clear-x-or-y-memory)) p-init))
	(progn
	  (need-loaded '.R-array-clear-x-or-y-memory)
	  ;; if eq R4, it's an array reference -- <check> will handle the p-init push (need array name)
	  (push `(UPDATE R4 R+1) pp)))
      (if (temp-sig del) (kill-temp del))
      (if (and offset (temp-sig offset)) (kill-temp offset))
      (spill-ALU-registers (libinfo-uses .tap)) ;includes R5
      (if offset 
	  (convert arg-addr (get-type_56 offset) 'X0 'integer)
	(push '(LOAD X0 0) pp))
      (push `(JSR .tap) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)

(defmacro <zdelay_56> (result-name del in-argl &optional pm)
  (let ((in-arg (if (alias in-argl) (eval in-argl) in-argl))
	(in-pm (if (alias pm) (eval pm) pm)))
    (DEBUGGING (push `(----------------> <zdelay> ,result-name ,del ,in-arg ,in-pm) pp))
    (need-loaded '.zdelay)
    (let* ((d-addr (cadr (get-home-address del 'zdly 'R4)))
	   (addr (if (eq d-addr 'R4) nil (+ d-addr 1)))
	   (arg-addr (get-any-address in-arg)))

      (get-register 'R5)
      (if (not (eq d-addr 'R4))
	  (progn
	    (get-register 'R4)
	    (need-loaded '.R-clear-x-or-y-memory)
	    (if (< addr 254)
		(progn
		  (push `(LOAD R4 ,addr SHORT) pp)
		  (push `(LOAD R5 ,(+ addr 2) SHORT) pp)
		  (push (list `(LOAD R4 ,(- addr 1) SHORT)) p-init))
	      (progn
		(push `(LOAD R4 ,addr) pp)
		(push `(LOAD R5 ,(+ addr 2)) pp)
		(push (list `(LOAD R4 ,(- addr 1))) p-init)))
	    (push (list `(JSR .R-clear-x-or-y-memory)) p-init))
	(need-loaded '.R-array-clear-x-or-y-memory))
        ;; <aref> no longer handles the index+1 if R4
	;; if eq R4, it's an array reference -- <check> will handle the p-init push (need array name)

      (setf-N-reg 'N5 -1)
      (if (eq d-addr 'R4)		;we assume this is set up for R5 calc below
	  (when (/= 2 (getf-N-reg 'N4)) 
	    (push `(LOAD N4 2 SHORT) pp)
	    (setf-N-reg 'N4 2)))

      (if (eq d-addr 'R4) (push `(UPDATE R4 R+1) pp))

      (when in-pm
	(if (temp-sig in-pm) (kill-temp in-pm))
	(if (eq d-addr 'R4)		;i.e. zdelay is indirect through R4 (aref), so pm addr must be too
	    (convert (get-any-address in-pm) (get-type_56 in-pm) '(L R4 RN) 'real)
	  (convert (get-any-address in-pm) (get-type_56 in-pm) (list 'L (+ d-addr 3)) 'real)))

      (if (temp-sig in-arg) (kill-temp in-arg))
      (if (temp-sig del) (kill-temp del))
      (spill-ALU-registers (libinfo-uses .zdelay)) ;includes R5
      (if (eq d-addr 'R4)		;.zdelay assumes R5=R4+2, here we can assume N4=2
	  (push `(LUA R5 R4 R+N) pp))
      (if (and (> run-safety 0) (not (eq (get-type_56 in-arg) 'fraction))) (check-for-fractional-overflow in-arg arg-addr 11))
      (convert arg-addr (get-type_56 in-arg) 'X0 'fraction)
      (push `(JSR .zdelay) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)

(defmacro <ztap_56> (result-name del &optional pm)
  (let ((in-pm (if (alias pm) (eval pm) pm)))
    (DEBUGGING (push `(----------------> <ztap> ,result-name ,del ,in-pm) pp))
    (need-loaded '.ztap)
    (let* ((d-addr (cadr (get-home-address del 'zdly 'R4)))
	   (addr (if (eq d-addr 'R4) nil (+ d-addr 1))))

      (get-register 'R5)
      (if (not (eq d-addr 'R4))
	  (progn
	    (get-register 'R4)
	    (need-loaded '.R-clear-x-or-y-memory)
	    (if (< addr 254)
		(progn
		  (push `(LOAD R4 ,addr SHORT) pp)
		  (push `(LOAD R5 ,(+ addr 2) SHORT) pp)
		  (push (list `(LOAD R4 ,(- addr 1) SHORT)) p-init))
	      (progn
		(push `(LOAD R4 ,addr) pp)
		(push `(LOAD R5 ,(+ addr 2)) pp)
		(push (list `(LOAD R4 ,(- addr 1))) p-init)))
	    (push (list `(JSR .R-clear-x-or-y-memory)) p-init))
	(need-loaded '.R-array-clear-x-or-y-memory))
        ;; <aref> handles the index+1 if R4
	;; if eq R4, it's an array reference -- <check> will handle the p-init push (need array name)

      (setf-N-reg 'N5 -1)
      (if (eq d-addr 'R4)		;we assume this is set up for R5 calc below
	  (when (/= 2 (getf-N-reg 'N4)) 
	    (push `(LOAD N4 2 SHORT) pp)
	    (setf-N-reg 'N4 2)))

      (when in-pm
	(if (temp-sig in-pm) (kill-temp in-pm))
	(if (eq d-addr 'R4)		;i.e. ztap is indirect through R4 (aref), so pm addr must be too
	    (convert (get-any-address in-pm) (get-type_56 in-pm) '(L R4 RN) 'real)
	  (convert (get-any-address in-pm) (get-type_56 in-pm) (list 'L (+ d-addr 3)) 'real)))

      (if (temp-sig del) (kill-temp del))
      (spill-ALU-registers (libinfo-uses .ztap)) ;includes R5
      (if (eq d-addr 'R4)		;.ztap assumes R5=R4+2, here we can assume N4=2
	  (push `(LUA R5 R4 R+N) pp))
      (push `(JSR .ztap) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)

(defmacro <all-comb_56> (result-name fl in-argl)
  (let ((in-arg (if (alias in-argl) (eval in-argl) in-argl)))
    (DEBUGGING (push `(----------------> <all-comb> ,result-name ,fl ,in-arg (,(gethash in-arg vars))) pp))
    (need-loaded '.all-comb)
    (let* ((a-addr (cadr (get-home-address fl 'cmbflt 'R4)))
	   (addr (if (eq a-addr 'R4) nil (+ a-addr 1)))
	   (arg-addr (get-any-address in-arg)))

      (if (not (eq a-addr 'R4))
	  (progn
	    (need-loaded '.R-clear-x-or-y-memory)
	    (get-register 'R4)
	    (get-register 'N4)
	    (if (< addr 255)
		(progn
		  (push `(LOAD R4 ,addr SHORT) pp)
		  (push (list `(LOAD R4 ,(- addr 1) SHORT)) p-init))
	      (progn
		(push `(LOAD R4 ,addr) pp)
		(push (list `(LOAD R4 ,(- addr 1))) p-init)))
	    (push (list `(JSR .R-clear-x-or-y-memory)) p-init))
	(need-loaded '.R-array-clear-x-or-y-memory))

      (when (/= 2 (getf-N-reg 'N4))
	(push `(LOAD N4 2 SHORT) pp)
	(setf-N-reg 'N4 2))
      (if (eq a-addr 'R4) (push `(UPDATE R4 R+1) pp))
      (if (temp-sig in-arg) (kill-temp in-arg))
      (if (temp-sig fl) (kill-temp fl))
      (spill-ALU-registers (libinfo-uses .all-comb))
      (if (and (> run-safety 0) (not (eq (get-type_56 in-arg) 'fraction))) (check-for-fractional-overflow in-arg arg-addr 12))
      (convert arg-addr (get-type_56 in-arg) 'A 'fraction)
      (push `(JSR .all-comb) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)

(defmacro <comb_56> (result-name cmbf in-arg)
  (DEBUGGING (push `(----------------> <comb> ,result-name ,cmbf ,in-arg) pp))
  `(<all-comb_56> ,result-name ,cmbf ,in-arg))

(defmacro <all-pass_56> (result-name filt in-arg)
  (DEBUGGING (push `(----------------> <all-pass> ,result-name ,filt ,in-arg) pp))
  `(<all-comb_56> ,result-name ,filt ,in-arg))

(defmacro <dsp-address_56> (result-name s mem offset &optional (typ 'real) (scale 1) in-setf)
  ;;explicit run-time reference to struct field, or whatever
  ;; if R-relative ref (i.e. array of structs)
  (DEBUGGING (push `(----------------> address ,result-name ,s ,mem ,offset ,typ ---- ,s ,@(get-home-address s)) pp))
  (let* ((addr (cadr (get-home-address s)))
	 (normal-case (or in-setf (/= scale 2)))
	 (res-reg (and (not normal-case) (get-temporary-register '(A B))))
	 (t-reg (and (not normal-case) (get-temporary-register '(X Y)))))
    (if (R-register addr)
	(let ((n-reg (get-n-reg addr)))
	  (when (/= offset (getf-N-reg n-reg))
	    (push `(LOAD ,n-reg ,offset SHORT) pp)
	    (setf-N-reg n-reg offset))
	  ;; change ownership of addr, put structure offset into N, use RN to address
	  (if (temp-sig s) (kill-temp s))
	  (if normal-case
	      (progn
		(mark-reg addr (list result-name 'L 'S))
		(add-var result-name typ (list mem addr 'RN) nil addr nil (if (/= scale 1) scale)))
	    (push `(LOAD ,(get-high-side t-reg) ,mem ,addr RN) pp)))
      (if normal-case
	  (add-var result-name typ (list mem (+ addr offset)) nil nil nil (if (/= scale 1) scale))
	(push `(LOAD ,(get-high-side t-reg) ,mem ,(+ addr offset)) pp)))
    (if (not normal-case)
	(progn
	  (push `(LOAD ,(get-low-side t-reg) 2) pp)
	  (push `(MPY ,(get-low-side t-reg) ,(get-high-side t-reg) ,res-reg) pp)
	  ;; this is a 22 bit right shift -- 23 for fraction-to-real, then back 1 for scale factor
	  (update-var-and-reg result-name res-reg 'real 'L 'S))))
  nil)



;;; ENVELOPES
;;;
;;; env data is stored on chip as current-value at home-address,
;;; then current location in list (pointer to a rate, starts set at someplace in heap)
;;; then a list of pairs (rate pass) -- add in rate until pass, then increment location by 2.
;;; So each env needs 2 words static space, and an unknown amount of heap space (pointer to data
;;; must be set on each call -- sort of like dynamically allocated array).

(defmacro <env_56> (result-name e &optional restart) ;R7 should point to env struct current-value
  (DEBUGGING (push `(----------------> <env> ,result-name ,e) pp))
  (let* ((home (cadr (get-home-address e 'envelope 'R7))))
    (if (not (eq home 'R7)) (get-register 'R7))
    (if (not restart)
	(when (/= 1 (getf-N-reg 'N7))
	  (push `(LOAD N7 1 SHORT) pp)
	  (setf-N-reg 'N7 1))
      (when (/= 3 (getf-N-reg 'N7))
	(push `(LOAD N7 3 SHORT) pp)
	(setf-N-reg 'N7 3)))
    (if (R-register home)
	(if (not (eq home 'R7))
	    (push `(COPY ,home R7) pp))
      (if (< home 255)
	  (push `(LOAD R7 ,home SHORT) pp)
	(push `(LOAD R7 ,home) pp)))
    (if (temp-sig e) (kill-temp e))
    (if (not restart)
	(progn
	  (need-loaded '.env)
	  (spill-ALU-registers (libinfo-uses .env))
	  (push `(JSR .env) pp)
	  (setf-N-reg 'N6 3)
	  (update-var-and-reg result-name 'A 'real 'L 'S))
      (progn
	(need-loaded '.restart-env)
	(spill-ALU-registers (libinfo-uses .restart-env))
	(push `(JSR .restart-env) pp)
	(setf-N-reg 'N6 2)
	(setf-N-reg 'N3 2)
	(add-alias result-name <nil>)))
    nil))
;;; JAM's upside-down exponential is: X0=target value, Y1=rate, A=current value
;;; MAC X0 Y1 A, COPY A X1 (old A), SMAC X1 Y1 A, X1=>out

(defmacro <restart-env_56> (result-name e)
  (DEBUGGING (push `(----------------> <restart-env> ,result-name ,e) pp))
  `(<env_56> ,result-name ,e t))

(defun collapse-indices (arr indices res-reg ind-reg mul-reg)
  ;; indices is the list of aref indices.  The various registers are just an optimization (they have been allocated).
  ;; Here we set up the row-major-index calculation and produce the dimension offset multiplies at initialization time.
  (let* ((dims (length indices))
	 (base (get-X-memory dims))
	 (index (new-signal))
	 (index-home (get-L-mem)))
    (add-var index 'integer (list 'X index-home) nil nil)
    (push `(let ((dimls (array-dimensions ,(if (temp-sig arr) 
					       (car-member arr full-array-names)
					     arr))))
	     (loop for dim on dimls by #'cdr and loc from ,base do
	       (setf-X-mem loc (apply #'* (cdr dim)))))
	  pup)
    (push `(CLR ,res-reg) pp)
    (loop for ind in indices and loc from base do
      (push `(LOAD ,ind-reg X ,loc) pp)
      (if (numberp ind)
	  (push `(LOAD ,mul-reg ,ind) pp)
	(convert (get-any-address ind) (get-type_56 ind) mul-reg 'integer))
      (push `(MAC ,ind-reg ,mul-reg ,res-reg) pp))
    (push `(ASR ,res-reg) pp)
    (push `(STORE ,(get-low-side res-reg) X ,index-home) pp)
    index))

(defvar inited-arrays nil)

(defmacro <aref_56> (result-name arr arr-type &rest indices)
  (if (null indices) (error "no array index for ~A?" arr))
  (if (null arr) (error "array ~A is nil" arr))
  (let* ((consind (and (= (length indices) 1) (numberp (first indices))))
	 (arr-addr (get-home-address arr))
	 (arr-loc (cadr arr-addr))
	 (r-reg (get-appropriate-r-reg arr-type))
	 (res-reg (and (not consind) (get-temporary-register '(A B))))
	 (ind-reg (and (not consind) (get-temporary-register '(X0 X1 Y0 Y1))))
	 (mul-reg (and (not consind) (get-temporary-register '(X0 X1 Y0 Y1) (list ind-reg))))

	 (ind (if (= (length indices) 1) (first indices) (collapse-indices arr indices res-reg ind-reg mul-reg)))
	 (index (if (alias ind) (eval ind) ind))
	 (ind-addr (if (numberp index) nil (get-any-address index))))

    (DEBUGGING (push `(----------------> <aref> ,result-name ,arr ,arr-type ,ind) pp))
    (if (temp-sig index) (kill-temp index))
    (if (temp-sig arr) (kill-temp arr))	;added 23-Aug-91
    (setf (fifth (gethash arr vars)) (or arr-type 'real))
    (if consind		; since index is a constant, we can precompute the address offset
	(let ((true-index (* index (structure-size_56 (or arr-type 'real))))
	      (true-base (or (car-member arr inited-arrays) (get-X-memory)))
	      (n-reg (get-n-reg r-reg)))
	  (when (not (zerop true-index))
	    (push `(LOAD ,n-reg ,true-index) pp)
	    (setf-n-reg n-reg true-index))
	  (push `(LOAD ,r-reg X ,true-base) pp)
	  (when (not (zerop true-index))
	    (push `(UPDATE ,r-reg R+N) pp))
	  (when (not (car-member arr inited-arrays))
	    (push (list arr true-base) inited-arrays)
	    (push (list `(LOAD A X ,arr-loc)) p-init) ; here we are adding a little load-time program to get the true base location
	    (push (list `(ASR A)) p-init)
	    (push (list `(STORE A X ,true-base)) p-init)))
      (progn
	(push `(CLR ,res-reg) pp)
	(push `(LOAD ,(get-low-side res-reg) X ,arr-loc) pp) ;SHORT handled automatically by assembler
	(if (null ind-addr)
	    (push `(LOAD ,ind-reg ,index) pp)
	  (convert ind-addr (get-type_56 index) ind-reg 'integer))
	(push `(LOAD ,mul-reg Y ,arr-loc) pp)
	(push `(MAC ,mul-reg ,ind-reg ,res-reg) pp)
	(push `(ASR ,res-reg) pp)
	(push `(COPY ,(get-low-side res-reg) ,r-reg) pp)))
    (update-var-and-reg result-name nil (or arr-type 'real) 'L 'S r-reg)

    ;; an array header has size-1 (true size in words) at X:[home+1] and top address (base+size-1) at Y:[home+1]
    ;; at this point, r-reg has the (true) element address, so it should be <= Y:[home+1]
    (when (> run-safety 1)
      (need-loaded '.array-index-warning) 
      (get-register 'A)
      (get-register 'B)
      (push `(COPY ,r-reg A) pp)
      (push `(LOAD B Y ,(1+ arr-loc)) pp)
      (push `(CMP A B) pp)
      (push `(JGE array-index-ok) pp)
      (push `(LOAD B0 Y ,arr-loc) pp)
      (if consind
	  (push `(LOAD A0 ,index) pp)
	(push `(COPY ,ind-reg A0) pp))
      (push `(JSR .array-index-warning) pp)
      (push `(array-index-ok LOCAL) pp))

    (DEBUGGING (push `(----------------> now ,result-name is ,@(gethash result-name vars)) pp))
    nil))
    
(defmacro <tref_56> (result-name tab tab-type ind &optional (store nil)) ;tab-type = table element type (kinda stupid, but aref's fault)
  (declare (ignore tab-type))
  (if (null ind) (error "table index is nil"))
  (if (null tab) (error "table is nil"))
  (let* ((index (if (alias ind) (eval ind) ind))
	 (ind-addr (if (numberp index) nil (get-any-address index)))
	 (tab-addr (get-home-address tab))
	 (tab-loc (cadr tab-addr))
	 (table-type (get-type_56 tab))
	 (tab-mem (if (eq table-type 'x-table) 'X (if (eq table-type 'y-table) 'Y)))
	 (ind-reg (get-temporary-r-register))
	 (N-ind-reg (get-N-reg ind-reg))
	 (home-reg (get-temporary-register r-registers (list ind-reg)))
	 (res-reg (get-temporary-register '(A B X0 X1 Y0 Y1))))
    (DEBUGGING (push `(----------------> tref ,result-name ,tab ,ind ,store) pp))
    (if (temp-sig index) (kill-temp index))
    (if (temp-sig tab) (kill-temp tab))	;added 23-Aug-91
    (setf-N-reg N-ind-reg -1)
    (if (or store (not tab-mem)) (push `(LOAD ,home-reg ,tab-loc) pp))
    (if (null ind-addr)
	(if (or (not store) (not (numberp index)) (not (zerop index)))
	    (push `(LOAD ,N-ind-reg ,index) pp))
      (convert ind-addr (get-type_56 index) N-ind-reg 'integer))
    (if (or store (not tab-mem))
	(push `(LOAD ,ind-reg X ,home-reg R+1) pp) ;size in Y:R if check wanted
      (push `(LOAD ,ind-reg X ,tab-loc) pp))
    (if store
	(let ((home-addr (cadr (get-home-address result-name))))
	  (if (not tab-mem) (push `(STORE ,home-reg X ,home-addr) pp))
	  (if (or (not (numberp index)) (not (zerop index))) (push `(UPDATE ,ind-reg R+N) pp))
	  (push `(STORE ,ind-reg Y ,home-addr) pp)
	  (mark-reg ind-reg (list result-name 'L 'NS))
	  (add-var result-name 'table-header nil nil (list home-addr) tab-mem))
      (progn
	(if (not tab-mem)
	    (progn
	      (push `(JCLR 0 Y ,home-reg R x-side) pp)
	      (push `(LOAD ,res-reg Y ,ind-reg RN) pp)
	      (push `(JMP got-data) pp)
	      (push `(x-side LOCAL) pp)
	      (push `(LOAD ,res-reg X ,ind-reg RN) pp)
	      (push `(got-data LOCAL) pp))
	  (push `(LOAD ,res-reg ,tab-mem ,ind-reg RN) pp))
	(update-var-and-reg result-name res-reg 'fraction 'L 'S)))
    nil))

(defun load-table-value-into-reg (reg home)
  (let* ((ind-reg (get-temporary-r-register))
	 (home-reg (get-temporary-r-register r-registers (list ind-reg))))
    (push `(LOAD ,home-reg X ,home) pp)
    (push `(LOAD ,ind-reg Y ,home) pp)
    (push `(JCLR 0 Y ,home-reg R x-side) pp)
    (push `(LOAD ,reg Y ,ind-reg RN) pp)
    (push `(JMP got-data) pp)
    (push `(x-side LOCAL) pp)
    (push `(LOAD ,reg X ,ind-reg RN) pp)
    (push `(got-data LOCAL) pp)))

(defmacro <setf-tref_56> (result-name tab tab-type ind)
  `(<tref_56>, result-name ,tab ,tab-type ,ind t))
	 

(defvar outsigs nil)
(defvar outloc-list nil)

(defun get-outloc (chan &optional stream-name)
  (when (or (not stream-name) (equal stream-name "NIL"))
    (setf stream-name "*CURRENT-OUTPUT-FILE*"))
  (let ((found-it (if (null outsigs)
		      (setf outloc-list nil)
		    (loop for i in outloc-list
		     if (and (string-equal (first i) stream-name)
			     (= (second i) chan))
		     return (third i)))))
    (if found-it
	(list found-it t)
      (let ((outloc (get-L-mem 3)))
	(push (list stream-name chan outloc) outloc-list)
	(push outloc outsigs)
	(push (list `(BCLR 1 Y ,(+ outloc 1))) pp-init-sample)
	(push `(push (list 0 (clm-get-run-time-file-index ,(find-symbol stream-name) ,chan) ,chan ,outloc) *sigarr*) pup)
	(list outloc nil)))))

(defmacro out-n (result-name val-1 chan &optional o-stream)
  ;;set up at run-time -- 3rd word for special case when all memory is full
  (let* ((outloc-again (get-outloc chan (string o-stream)))
	 (outloc (first outloc-again))
	 (val (if (alias val-1) (eval val-1) val-1))
	 (output-addr (if (numberp val) 
			  nil
			(if (eq val-1 'X0) 
			    'X0
			  (get-any-address val)))))
    (DEBUGGING (push `(----------------> <out-n> ,val -- struct at  ,outloc (,(gethash val vars))) pp))
    (if (and o-stream (temp-sig o-stream)) 
      (error "Run can't handle an expression as the output stream pointer in Out~A" 
	     (if (= chan 0) "A" (if (= chan 1) "B" (if (= chan 2) "C" "D")))))
     (if (not (second outloc-again))
	 (progn
	   (need-loaded '.out-n)
	   (if (and (not (eq val-1 'X0)) (not (eq 'X0 (get-work-address val)))) (get-register 'X0))
	   (get-register 'R4)
	   (if (< outloc 256)
	       (push `(LOAD R4 ,outloc SHORT) pp)
	     (push `(LOAD R4 ,outloc) pp))
	   (if (null output-addr) 
	       (load-fraction 'X0 val)
	     (progn
	       (if (temp-sig val) (kill-temp val))
	       (if (not (eq val-1 'X0))
		   (progn
		     (if (and (> run-safety 0) (not (eq (get-type_56 val) 'fraction))) (check-for-fractional-overflow val output-addr 13))
		     (convert output-addr (get-type_56 val) 'X0 'fraction)))))
	   (spill-ALU-registers (libinfo-uses .out-n))
	   (push '(JSR .out-n) pp)
	   (update-var-and-reg result-name 'X0 'fraction 'L 'S))
       (progn
	 (need-loaded '.out-n-again)
	 (if (and (not (eq val 'X0)) (not (eq 'X0 (get-work-address val)))) (get-register 'X0))
	 (get-register 'R4)
	 (if (< outloc 255)
	     (push `(LOAD R4 ,(+ outloc 1) SHORT) pp)
	   (push `(LOAD R4 ,(+ outloc 1)) pp))
	 (if (null output-addr) 
	     (load-fraction 'X0 val)
	   (progn
	     (if (temp-sig val) (kill-temp val))
	     (if (not (eq val-1 'X0))
		   (progn
		     (if (and (> run-safety 0) (not (eq (get-type_56 val) 'fraction))) (check-for-fractional-overflow val output-addr 13))
		     (convert output-addr (get-type_56 val) 'X0 'fraction)))))
	 (spill-ALU-registers (libinfo-uses .out-n-again))
	 (push '(JSR .out-n-again) pp)
	 (update-var-and-reg result-name 'X0 'fraction 'L 'S)))
     nil))

(defmacro outsig-n (result-name pass val chan o-stream)
  (let* ((loc (get-X-memory)))		;need someplace to save the file id and channel number
    (push `(setf-x-mem ,loc 
		       (+ ,(ash chan 8) 
			  (ash (clm-get-run-time-file-index (or ,(if (temp-sig o-stream) 
								     (car-member o-stream full-array-names)
								   o-stream)
								*current-output-file*) 
							    ,chan) 16)))
	  pup)				;this establishes our back pointers at initialization time
    (push `(clm-mark-external-stream (or ,(if (temp-sig o-stream) 
					      (car-member o-stream full-array-names)
					    o-stream)
					 *current-output-file*))
	  pup)
    (need-loaded '.outsig)
    (get-register 'X0)
    (if (temp-sig val) (kill-temp val))
    (if (temp-sig o-stream) (kill-temp o-stream))
    (if (eq (get-type_56 val) 'fraction)
	(if (numberp val)
	    (load-fraction 'X0 val)
	  (move (get-any-address val) 'X0))
      (progn
	(if (and (> run-safety 0) (not (eq (get-type_56 val) 'fraction))) (check-for-fractional-overflow val (get-any-address val) 13))
	(convert (get-any-address val) (get-type_56 val) 'X0 'fraction)))
    (mark-reg 'X0 (list val 'L 'NS))
    (when (or (not (eq (get-work-address pass) 'A))
	      (not (eq (get-type_56 pass) 'long-int)))
      (get-register 'A)
      (if (temp-sig pass) (kill-temp pass))
      (convert (get-any-address pass) (get-type_56 pass) 'A 'long-int))
    (get-register 'X1)
    (push `(LOAD X1 X ,loc) pp)
    (push '(JSR .outsig) pp)
    (update-var-and-reg result-name 'X0 'fraction 'L 'S)
    nil))

(defmacro <outa_56> (result-name pass val &optional o-stream)
  (if (not (eq pass loop-var))
      `(outsig-n ,result-name ,pass ,val ,0 ,o-stream)
    `(out-n ,result-name ,val 0 ,o-stream)))

(defmacro <outb_56> (result-name pass val &optional o-stream)
  (if (not (eq pass loop-var))
      `(outsig-n ,result-name ,pass ,val ,1 ,o-stream)
    `(out-n ,result-name ,val 1 ,o-stream)))

(defmacro <outc_56> (result-name pass val &optional o-stream)
  (if (not (eq pass loop-var))
      `(outsig-n ,result-name ,pass ,val ,2 ,o-stream)
    `(out-n ,result-name ,val 2 ,o-stream)))

(defmacro <outd_56> (result-name pass val &optional o-stream)
  (if (not (eq pass loop-var))
      `(outsig-n ,result-name ,pass ,val ,3 ,o-stream)
    `(out-n ,result-name ,val 3 ,o-stream)))


(defmacro locsig-n (result-name loc-id pass val)
  (let* ((loc (get-X-memory)))		;need someplace to save the loc id
    (push `(setf-x-mem ,loc (remember-locsig ,loc-id)) pup)
    (need-loaded '.locsig-n)
    (get-register 'X0)
    (if (temp-sig val) (kill-temp val))
    (if (eq (get-type_56 val) 'fraction)
	(if (numberp val)
	    (load-fraction 'X0 val)
	  (move (get-any-address val) 'X0))
      (progn
	(if (and (> run-safety 0) (not (eq (get-type_56 val) 'fraction))) (check-for-fractional-overflow val (get-any-address val) 14))
	(convert (get-any-address val) (get-type_56 val) 'X0 'fraction)))
    (mark-reg 'X0 (list val 'L 'NS))
    (when (or (not (eq (get-work-address pass) 'A))
	      (not (eq (get-type_56 pass) 'long-int)))
      (get-register 'A)
      (if (temp-sig pass) (kill-temp pass))
      (convert (get-any-address pass) (get-type_56 pass) 'A 'long-int))
    (get-register 'X1)
    (push `(LOAD X1 X ,loc) pp)
    (push '(JSR .locsig-n) pp)
    (update-var-and-reg result-name 'X0 'fraction 'L 'S)
    nil))

(defmacro <locsig_56> (result-name loc-1 pass val-1)	;ascl:bscl rscl:type [0=outa only, 1=outa+outb, 2=outa+rev, 3=all]
  ;; 0 bit = outb on, 1 bit = rev on, 2 bit = quad on
  ;; for quad add types 5=quad 7=quad+rev, need one extra word allocated for cscl and dscl (pass-locs)
  ;; half the mess here is macroexpansion confusion, and another half is for multiple locsigs
  (if (not (eq pass loop-var))
      `(locsig-n ,result-name ,loc-1 ,pass ,val-1)
    (let* ((loc (if (alias loc-1) (eval loc-1) loc-1))
	   (val (if (alias val-1) (eval val-1) val-1))
	   (res-reg (get-temporary-register '(A B)))
	   (val-loc (get-work-address val))
	   (outadat (get-outloc 0 nil))
	   (outbdat (get-outloc 1 nil))
	   (outcdat (get-outloc 2 nil))
	   (outddat (get-outloc 3 nil))
	   (outrdat (get-outloc 0 "*REVERB*"))
	   (outaloc (first outadat))
	   (outbloc (first outbdat))
	   (outcloc (first outcdat))
	   (outdloc (first outddat))
	   (outrloc (first outrdat))
	   (a-again (second outadat))
	   (b-again (second outbdat))
	   (c-again (second outcdat))
	   (d-again (second outddat))
	   (r-again (second outrdat))
	   (storage (get-X-memory))
	   (addr (cadr (get-home-address loc)))
	   (adr-reg (if (> addr 62) (get-register 'R7)))
	   (in-reg (if (member val-loc '(X1 Y0 Y1)) val-loc (get-temporary-register '(X1 Y0 Y1))))
	   (scl-reg (get-temporary-register '(X0 X1 Y0 Y1) (list in-reg))))
      
      (DEBUGGING (push `(----------------> <locsig> ,loc ,val) pp))
      (if (and (> run-safety 0) (not (eq (get-type_56 val) 'fraction))) (check-for-fractional-overflow val (get-any-address val) 14))
      (convert (get-any-address val) (get-type_56 val) in-reg 'fraction)
      (push `(STORE ,in-reg X ,storage) pp)
      (push `(LOAD ,scl-reg X ,addr) pp) ;ascl (in-sig in in-reg)
      (push `(MPY ,scl-reg ,in-reg ,res-reg) pp)
      (push `(COPY ,res-reg X0) pp)
      
      (if (not (and a-again b-again c-again d-again r-again))
	  (need-loaded '.out-n))
      (if (or a-again b-again c-again d-again r-again)
	  (need-loaded '.out-n-again))
      
      (get-register 'R4)
      (if (temp-sig val) (kill-temp val))
      
      (if adr-reg (push `(LOAD ,adr-reg ,(+ addr 1)) pp))
      (if (< outaloc 255)
	  (push `(LOAD R4 ,(+ outaloc (if a-again 1 0)) SHORT) pp)
	(push `(LOAD R4 ,(+ outaloc (if a-again 1 0))) pp))
      (spill-ALU-registers (libinfo-uses (if a-again .out-n-again .out-n)))
      (if a-again
	  (push '(JSR .out-n-again) pp)
	(push '(JSR .out-n) pp))
      
      (if (null adr-reg)
	  (push `(JCLR 0 Y ,(+ addr 1) no-B) pp)
	(push `(JCLR 0 Y ,adr-reg R no-B) pp))
      
      (push `(LOAD ,in-reg X ,storage) pp)
      (push `(LOAD ,scl-reg Y ,addr) pp) ;bscl
      (push `(MPY ,scl-reg ,in-reg ,res-reg) pp)
      (push `(COPY ,res-reg X0) pp)
      
      (if (< outbloc 255)
	  (push `(LOAD R4 ,(+ outbloc (if b-again 1 0)) SHORT) pp)
	(push `(LOAD R4 ,(+ outbloc (if b-again 1 0))) pp))
      (spill-ALU-registers (libinfo-uses (if b-again .out-n-again .out-n)))
      (if b-again
	  (push '(JSR .out-n-again) pp)
	(push '(JSR .out-n) pp))
      
      (if (null adr-reg)
	  (push `(JCLR 2 Y ,(+ addr 1) no-B) pp)
	(push `(JCLR 2 Y ,adr-reg R no-B) pp))
	
      (push `(LOAD ,in-reg X ,storage) pp)
      (push `(LOAD ,scl-reg X ,(+ addr 2)) pp) ;cscl
      (push `(MPY ,scl-reg ,in-reg ,res-reg) pp)
      (push `(COPY ,res-reg X0) pp)

      (if (< outcloc 255)
	  (push `(LOAD R4 ,(+ outcloc (if c-again 1 0)) SHORT) pp)
	(push `(LOAD R4 ,(+ outcloc (if c-again 1 0))) pp))
      (spill-ALU-registers (libinfo-uses (if c-again .out-n-again .out-n)))
      (if c-again
	  (push '(JSR .out-n-again) pp)
	(push '(JSR .out-n) pp))

      (push `(LOAD ,in-reg X ,storage) pp)
      (push `(LOAD ,scl-reg Y ,(+ addr 2)) pp) ;dscl
      (push `(MPY ,scl-reg ,in-reg ,res-reg) pp)
      (push `(COPY ,res-reg X0) pp)
      
      (if (< outdloc 255)
	  (push `(LOAD R4 ,(+ outdloc (if d-again 1 0)) SHORT) pp)
	(push `(LOAD R4 ,(+ outdloc (if d-again 1 0))) pp))
      (spill-ALU-registers (libinfo-uses (if d-again .out-n-again .out-n)))
      (if d-again
	  (push '(JSR .out-n-again) pp)
	(push '(JSR .out-n) pp))
    
      (push '(no-B LOCAL) pp)
      (if (null adr-reg)
	  (push `(JCLR 1 Y ,(+ addr 1) no-Rev) pp)
	(push `(JCLR 1 Y ,adr-reg R no-Rev) pp))
      (push `(LOAD ,in-reg X ,storage) pp)
      (push `(LOAD ,scl-reg X ,(+ addr 1)) pp) ;rscl
      (push `(MPY ,scl-reg ,in-reg ,res-reg) pp)
      (push `(COPY ,res-reg X0) pp)
      
      (if (< outrloc 255)
	  (push `(LOAD R4 ,(+ outrloc (if r-again 1 0)) SHORT) pp)
	(push `(LOAD R4 ,(+ outrloc (if r-again 1 0))) pp))
      (spill-ALU-registers (libinfo-uses (if r-again .out-n-again .out-n)))
      (if r-again
	  (push '(JSR .out-n-again) pp)
	(push '(JSR .out-n) pp))
      
      (push '(no-Rev LOCAL) pp)
      nil)))


(defvar insigs nil)

(defmacro <in-n_56> (result-name inloc)
  (DEBUGGING (push `(----------------> <in-n> -- struct at  ,inloc) pp))
  (need-loaded '.in-n)
  (get-register 'X0)
  (get-register 'R4)
  (if (< inloc 256)
      (push `(LOAD R4 ,inloc SHORT) pp)
    (push `(LOAD R4 ,inloc) pp))
  (spill-ALU-registers (libinfo-uses .in-n))
  (push '(JSR .in-n) pp)
  (update-var-and-reg result-name 'X0 'fraction 'L 'S)
  nil)

(defvar inloc-list nil)

(defun allocate-inloc (chan o-stream)
  (let ((inloc (get-L-mem 3)))
    (push (list o-stream chan inloc) inloc-list)
    (push inloc insigs)
    (if o-stream
	(push `(push (list 1 (clm-get-run-time-file-index ,o-stream ,chan) ,chan ,inloc) *sigarr*) pup)
      (push `(push (list 1 (clm-get-run-time-file-index *current-input-file* ,chan) ,chan ,inloc) *sigarr*) pup))
    inloc))

(defun get-inloc (chan o-stream)
  (if (null insigs)
      (progn
	(setf inloc-list nil)
	(allocate-inloc chan o-stream))
    (progn
      (loop for i in inloc-list do
	(if (and (eq (car i) o-stream)
		 (= (cadr i) chan))
	    (return-from get-inloc (caddr i))))
      (allocate-inloc chan o-stream))))

(defmacro in-n (result-name chan &optional (o-stream nil)) 
  `(<in-n_56> ,result-name ,(get-inloc chan o-stream)))

(defmacro <insig-n_56> (result-name pass inloc)
  (DEBUGGING (push `(----------------> insig-n -- struct at  ,inloc) pp))
  (when (not (member 'A (get-work-addresses pass)))
    (get-temporary-register '(A))
    (move (get-any-address pass) 'A))
  (need-loaded '.insig-n)
  (get-register 'R4)
  (if (< inloc 256)
      (push `(LOAD R4 ,inloc SHORT) pp)
    (push `(LOAD R4 ,inloc) pp))
  (setf-N-reg 'N4 1)
  (setf-N-reg 'N5 -1)
  (if (temp-sig pass) (kill-temp pass))
  (spill-ALU-registers (libinfo-uses .insig-n))
  (push '(JSR .insig-n) pp)
  (update-var-and-reg result-name 'A 'fraction 'L 'S)
  nil)

(defmacro insig-n (result-name pass chan &optional (o-stream nil) (addr nil)) 
  (let ((inloc (or addr (get-L-mem 5))))
    ;; L:inloc (io-siz fil)
    ;; L:+1   bufend
    ;; L:+2   bufstart
    ;; X:+3   bufloc     Y:+3 fil<<16 | chn<<8 | x-or-y as bit 0
    ;; L:+4   bufsize (as long-int => put size in Y)
    (if (and o-stream (temp-sig o-stream)) 
	(error "Run can't handle an expression as the input stream pointer in In~A" 
	       (if (= chan 0) "A" (if (= chan 1) "B" (if (= chan 2) "C" "D")))))
    (push `(push (list 3 0 (+ ,(ash chan 8) 
			      (ash (clm-get-run-time-file-index 
				    (or ,o-stream *current-input-file*) 
				    ,chan) 16))
		       ,inloc) 
		 *sigarr*) 
	  pup)
    `(<insig-n_56> ,result-name ,pass ,inloc)))


(defmacro <in-a_56> (result-name pass &optional (o-stream nil))
  (if (eq pass loop-var)
      `(in-n ,result-name 0 ,o-stream)
    `(insig-n ,result-name ,pass 0 ,o-stream)))

;;; IN-N is the simple (fastest) version -- it marches along with OUTA pass-counter
;;; and is intended primarily to speed up reverb.  INSIG is the more general form
;;; that implements more of INA's lisp-capability (i.e. arbitrary index, but we assume
;;; the stream stays the same across a given call).  IN-N's sig index is 1 (IN_N in c),
;;; INSIG's is 3 (RAN_IN_N).  Ins (counted at load time and passed to c-read-dsp-block)
;;; is the number of IN_N's, and RAN_IN_N is dealt with elsewhere (HF2andHF3).

(defmacro <in-b_56> (result-name pass &optional (o-stream nil))
  (if (eq pass loop-var)
      `(in-n ,result-name 1 ,o-stream)
    `(insig ,result-name ,pass 1 ,o-stream)))

(defmacro <in-c_56> (result-name pass &optional (o-stream nil))
  (if (eq pass loop-var)
      `(in-n ,result-name 2 ,o-stream)
    `(insig ,result-name ,pass 2 ,o-stream)))

(defmacro <in-d_56> (result-name pass &optional (o-stream nil))
  (if (eq pass loop-var)
      `(in-n ,result-name 3 ,o-stream)
    `(insig ,result-name ,pass 3 ,o-stream)))
 

(defmacro <quad_56> (result-name &optional fil)
  (let ((chans (get-X-memory))
	(reg (get-temporary-register '(X0 X1 Y0 Y1))))
    (if (string-equal (symbol-name fil) "NIL")
	(push `(setf-x-mem ,chans (if (quad *current-output-file*) ,<t> ,<nil>)) pup)
      (push `(setf-x-mem ,chans (if (quad ,fil) ,<t> ,<nil>)) pup))
    (push `(LOAD ,reg X ,chans) pp)
    (update-var-and-reg result-name reg 'integer 'L 'S))
  nil)

(defmacro <stereo_56> (result-name &optional fil)
  (let ((chans (get-X-memory))
	(reg (get-temporary-register '(X0 X1 Y0 Y1))))
    (if (string-equal (symbol-name fil) "NIL")
	(push `(setf-x-mem ,chans (if (stereo *current-output-file*) ,<t> ,<nil>)) pup)
      (push `(setf-x-mem ,chans (if (stereo ,fil) ,<t> ,<nil>)) pup))
    (push `(LOAD ,reg X ,chans) pp)
    (update-var-and-reg result-name reg 'integer 'L 'S))
  nil)

(defmacro <mono_56> (result-name &optional fil)
  (let ((chans (get-X-memory))
	(reg (get-temporary-register '(X0 X1 Y0 Y1))))
    (if (string-equal (symbol-name fil) "NIL")
	(push `(setf-x-mem ,chans (if (mono *current-output-file*) ,<t> ,<nil>)) pup)
      (push `(setf-x-mem ,chans (if (mono ,fil) ,<t> ,<nil>)) pup))
    (push `(LOAD ,reg X ,chans) pp)
    (update-var-and-reg result-name reg 'integer 'L 'S))
  nil)


(defmacro <resample_56> (result-name s &optional (sr nil))	
  (let ((fm (if (alias sr) (eval sr) sr)))
					;s=smp struct pointer  s1:s2  cur-x:cur-out incr[L] i[L] insig-block
    (DEBUGGING (push `(----------------> <resample> ,result-name ,s ,fm) pp))
    (need-loaded '.resample)
    (let ((addr (cadr (get-home-address s 'smp 'R2))))
      (get-register 'R4)
      (if (not (eq addr 'R2))
	  (progn
	    (if (not (temp-sig s)) (push `(push (list 3 0 (clm-get-run-time-resample-index ,s) ,(+ addr 4)) *sigarr*) pup))
	    (get-register 'R2)
	    (if (< addr 252)
		(progn
		  (push `(LOAD R2 ,addr SHORT) pp)
		  (push `(LOAD R4 ,(+ addr 4) SHORT) pp))
	      (progn
		(push `(LOAD R2 ,addr) pp)
		(push `(LOAD R4 ,(+ addr 4)) pp))))
	(progn
	  (push `(COPY R2 R4) pp)
	  (if (= (getf-N-reg 'N2) 4)
	      (push `(UPDATE R4 R+N) pp)
	    (progn
	      (push `(REP 4) pp)
	      (push `(UPDATE R4 R+1) pp)))))
      (when fm
	(if (temp-sig fm) (kill-temp fm))
	(if (eq addr 'R2)		;i.e. resample smp is indirect through R2 (aref), so sr-change addr must be too
	    (progn
	      (when (/= 2 (getf-N-reg 'N2)) 
		(push `(LOAD N2 2 SHORT) pp)
		(setf-N-reg 'N2 2))
	      (if (numberp fm)
		  (progn
		    (get-register 'A)
		    (load-real 'A fm)
		    (push `(STORE A L R2 RN) pp))
		(convert (get-any-address fm) (get-type_56 fm) '(L R2 RN) 'real)))
	  (if (numberp fm)
	      (progn
		(get-register 'A)
		(load-real 'A fm)
		(push `(STORE A L ,(+ addr 2)) pp))
	    (convert (get-any-address fm) (get-type_56 fm) (list 'L (+ addr 2)) 'real))))
      (setf-N-reg 'N4 1)
      (setf-N-reg 'N5 -1)
      (setf-N-reg 'N2 -1)
      (if (temp-sig s) (kill-temp s))
      (spill-ALU-registers (libinfo-uses .resample))
      (push `(JSR .resample) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)


(defmacro <src_56> (result-name s &optional fm-1)
  (let ((fm (if (alias fm-1) (eval fm-1) fm-1)))
    (DEBUGGING (push `(----------------> <src> ,result-name ,s ,fm) pp))
    (need-loaded '.src)
    (let ((addr (cadr (get-home-address s 'sr 'R2))))
      (when (not (eq addr 'R2)) 
	(get-register 'R2)
	(if (not (temp-sig s)) (push `(push (list 3 0 (clm-get-run-time-readin-index (and ,s (sr-rd ,s))) ,(+ addr 8)) *sigarr*) pup))
	(if (< addr 256)
	    (push `(LOAD R2 ,addr SHORT) pp)
	  (push `(LOAD R2 ,addr) pp)))
      (if (temp-sig s) (kill-temp s))
      (init-N-regs)
      (spill-ALU-registers (libinfo-uses .src))
      (if fm
	  (if (numberp fm)
	      (load-real 'A (float fm))
	    (progn
	      (if (temp-sig fm) (kill-temp fm))
	      (convert (get-any-address fm) (get-type_56 fm) 'A 'real)))
	(clear 'A))
      (push `(JSR .src) pp)
      (update-var-and-reg result-name 'X0 'fraction 'L 'S))
    nil))

(defun rdin-offset (typ)
  (case typ 
    (rdin 0)
    (sr 6)
    (fftflt 8)
    (conv 8)
    (spd 5)
    (otherwise (break "unknown type: ~A" typ))))

(defmacro <read-position_56> (result-name rd)
  (DEBUGGING (push `(----------------> <read-position> ,result-name ,rd) pp))
  (let* ((addr (cadr (get-home-address rd)))
	 (typ (get-type_56 rd))
	 (offset (rdin-offset typ)))
    (get-register 'A)
    (if (temp-sig rd) (kill-temp rd))
    (push `(LOAD A L ,(+ addr offset)) pp)
    (update-var-and-reg result-name 'A 'long-int 'L 'S))
  nil)

(defmacro <read-forward_56> (result-name rd)
  (DEBUGGING (push `(----------------> <read-forward> ,result-name ,rd) pp))
  (let* ((addr (cadr (get-home-address rd)))
	 (typ (get-type_56 rd))
	 (offset (1+ (rdin-offset typ))))
    (get-register 'A)
    (if (temp-sig rd) (kill-temp rd))
    (push '(CLR A) pp)
    (push '(LOAD A0 1) pp)
    (push `(STORE A L ,(+ addr offset)) pp)
    (update-var-and-reg result-name 'A 'long-int 'L 'S))
  nil)

(defmacro <read-backward_56> (result-name rd)
  (DEBUGGING (push `(----------------> <read-backward> ,result-name ,rd) pp))
  (let* ((addr (cadr (get-home-address rd)))
	 (typ (get-type_56 rd))
	 (offset (1+ (rdin-offset typ))))
    (get-register 'A)
    (if (temp-sig rd) (kill-temp rd))
    (push '(LOAD A #xFFFFFF) pp)
    (push '(LOAD A0 #xFFFFFF) pp)
    (push `(STORE A L ,(+ addr offset)) pp)
    (update-var-and-reg result-name 'A 'long-int 'L 'S))
  nil)

(defmacro <readin_56> (result-name rd)
  (DEBUGGING (push `(----------------> <readin> ,result-name ,rd) pp))
  (let ((addr (cadr (get-home-address rd 'rdin 'R2)))) ;rdin: L:i L:inc L5:insig
    (need-loaded '.readin)
    (get-register 'R4)
    (if (not (eq addr 'R2))
	(progn
	  (if (not (temp-sig rd)) (push `(push (list 3 0 (clm-get-run-time-readin-index ,rd) ,(+ addr 2)) *sigarr*) pup))
	  (get-register 'R2)
	  (if (< addr 254)
	      (progn
		(push `(LOAD R2 ,addr SHORT) pp)
		(push `(LOAD R4 ,(+ addr 2) SHORT) pp))
	    (progn
	      (push `(LOAD R2 ,addr) pp)
	      (push `(LOAD R4 ,(+ addr 2)) pp))))
      (progn
	(push `(COPY R2 R4) pp)
	(if (= (getf-N-reg 'N2) 2)
	    (push `(UPDATE R4 R+N) pp)
	  (progn
	    (push `(UPDATE R4 R+1) pp)
	    (push `(UPDATE R4 R+1) pp)))))
    (setf-N-reg 'N4 1)
    (setf-N-reg 'N5 -1)
    (if (temp-sig rd) (kill-temp rd))
    (spill-ALU-registers (libinfo-uses .readin))
    (push `(JSR .readin) pp)
    (update-var-and-reg result-name 'X0 'fraction 'L 'S))
  nil)

(defmacro <check-anybody_56> (rd offset size ref) ;initialize an array of whatever structure insig blocks
  (if (listp rd)
      (let ((addr (cadr (get-home-address (cadr rd))))
	    (type (get-type_56 (cadr rd))))
	(if (eq type 'array)
	    (push `(dotimes (i (array-total-size ,(cadr rd))) 
		     (if (aref ,(cadr rd) i)
			 (push 
			  (list 3 0 (clm-get-run-time-readin-index (,ref (aref ,(cadr rd) i)))
				(+ (floor (aref internal-x-memory ,addr) 2) ,offset (* i ,size))) *sigarr*))) pup)
	  ;; otherwise it must be a field of a user-defined structure, so we'll
	  ;;   use the exact form passed to us in the arg to clm-check.
	  (let* ((fld (first rd))
		 (fld-data (gethash fld user-structs))
		 (fld-offset (first fld-data))
		 (type (first (second fld-data))))
	    (if (not (eq type 'rdin)) (warn "~A is defined to be ~(~A~), but is being used as a rdin (readin) structure" rd type))
	    (push `(push (list 3 0 (clm-get-run-time-readin-index ,rd) ,(+ addr fld-offset offset)) *sigarr*) pup)))))
  nil)

;;; instead of constants here, we should call structure-size to get the correct array element size

(defmacro <check-readin_56> (rd)		;initialize an array of rdin structure insig blocks
  `(<check-anybody_56> ,rd 2 7 identity))

(defmacro <check-resample_56> (rd)		;initialize an array of smp structure insig blocks
  (if (listp rd)
      (let ((addr (cadr (get-home-address (cadr rd))))
	    (type (get-type_56 (cadr rd))))
	(if (eq type 'array)
	    (push `(dotimes (i (array-total-size ,(cadr rd))) 
		     (if (aref ,(cadr rd) i)
			 (push 
			  (list 3 0 (clm-get-run-time-resample-index (aref ,(cadr rd) i))
				(+ (floor (aref internal-x-memory ,addr) 2) 4 (* i 9))) *sigarr*))) pup)
	  ;; (see <check-anybody> above)
	  (let* ((fld (first rd))
		 (fld-data (gethash fld user-structs))
		 (fld-offset (first fld-data))
		 (type (first (second fld-data))))
	    (if (not (eq type 'smp)) (warn "~A is defined to be ~(~A~), but is being used as a smp (resample) structure" rd type))
	    (push `(push (list 3 0 (clm-get-run-time-readin-index ,rd) ,(+ addr fld-offset 4)) *sigarr*) pup)))))
  nil)

(defmacro <check-src_56> (rd)		;initialize an array of sr structure insig blocks
  `(<check-anybody_56> ,rd 8 18 sr-rd))

(defmacro <check-fftflt_56> (rd)		;initialize an array of fftflt structure insig blocks
  `(<check-anybody_56> ,rd 10 18 fftflt-rd))
    
(defmacro <check-convolve_56> (rd)		;initialize an array of conv structure insig blocks
  `(<check-anybody_56> ,rd 10 18 conv-rd))

(defun conv-rd (rd) (fftflt-rd (conv-fftf rd)))
    

(defmacro <expand_56> (result-name e)
  (DEBUGGING (push `(----------------> <expand> ,result-name ,e) pp))
  (need-loaded '.expand)
  (let ((addr (cadr (get-home-address e 'spd 'R2))))
    (when (not (eq addr 'R2))
      (get-register 'R2)
      (if (not (temp-sig e))
	  (push `(push (list 3 0 (clm-get-run-time-readin-index (and ,e (spd-rd ,e))) ,(+ addr expand-readin-offset)) *sigarr*) pup))
      (if (< addr 256)
	  (push `(LOAD R2 ,addr SHORT) pp)
	(push `(LOAD R2 ,addr) pp)))
    (when (/= 1 (getf-N-reg 'N2)) 
      (push `(LOAD N2 1 SHORT) pp)
      (setf-N-reg 'N2 1))
    (if (temp-sig e) (kill-temp e))
    (spill-ALU-registers (libinfo-uses .expand))
    (push `(JSR .expand) pp)
    (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)

(defmacro <check-expand_56> (rd)
  (declare (ignore rd))
  nil)

(defmacro <run-block_56> (result-name e)
  (DEBUGGING (push `(----------------> <run-block> ,result-name ,e) pp))
  (need-loaded '.run-block)
  (let ((addr (cadr (get-home-address e 'rblk 'R2))))
    (when (not (eq addr 'R2)) 
      (get-register 'R2)
      (if (< addr 256)
	  (push `(LOAD R2 ,addr SHORT) pp)
	(push `(LOAD R2 ,addr) pp)))
    (setf-N-reg 'N3 -1)
    (setf-N-reg 'N5 -1)
    (if (temp-sig e) (kill-temp e))
    (spill-ALU-registers (libinfo-uses .run-block))
    (push `(JSR .run-block) pp)
    (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)

(defmacro <fft_56> (result-name data &optional (fft-dir 1))
  (DEBUGGING (push `(----------------> <fft> ,result-name ,data) pp))
  (add-alias result-name 0)
  (need-loaded '.basic-fft)
  (let ((addr (cadr (get-home-address data 'fft-data 'R2))))
    (spill-ALU-registers (libinfo-uses .basic-fft))
    (get-register 'R3)
    (push `(LOAD B ,fft-dir) pp)	;allow R2 pipeline to settle, if necessary
    (if (not (eq addr 'R2))
	(push `(LOAD R3 X ,addr) pp)
      (push `(LOAD R3 X R2 R) pp))
    (push `(LOAD A Y ,addr) pp)
    (push `(JSR .basic-fft) pp))
  nil)

(defmacro <inverse-fft_56> (result-name data)
  `(<fft_56> ,result-name ,data -1))

(defmacro <clear-block_56> (result-name data)
  (DEBUGGING (push `(----------------> <clear-block> ,result-name ,data) pp))
  (add-alias result-name 0)
  (need-loaded '.clear-table)
  (let ((addr (cadr (get-home-address data 'table 'R3))))
    (if (not (eq addr 'R3))
	(push `(LOAD R3 ,addr) pp))
    (spill-ALU-registers (libinfo-uses .clear-table))
    (get-register 'X0)
    (push `(LOAD X0 Y R3 R) pp)
    (push `(JSR .clear-table) pp))
  nil)
  

(defmacro <loop-finish_56 ()
  (push '(JMP .pass-end) pp)
  nil)


#|
(defmacro <spectrum_56> (result-name freq phase size &optional (normalized t))
  (need-loaded '.spectrum)
  ;; need to track phase if phase not nil and need to scale by max is normalized
  ;; lib56 version doesn't do either -- sanfft does atan, but doesn't scale 
  ;;  -- sansy version deals with window weight at load time, so it knows scaling in advance

  ;; maybe what we want is convert-to-polar (would take fftdata struct and do the Pythagorean/atan two step)

|#

(defmacro <fft-window_56> (result-name data window)
  (DEBUGGING (push `(----------------> <window> ,result-name ,data ,window) pp))
  (add-alias result-name 0)
  (need-loaded '.fft-window)
  (let ((addr (cadr (get-home-address data 'fft-data 'R3)))
	(wind (cadr (get-home-address window 'table 'R5))))
    (spill-ALU-registers (libinfo-uses .fft-window))
    (if (not (eq addr 'R3)) (push `(LOAD R3 ,addr) pp))
    (if (not (eq addr 'R5)) (push `(LOAD R5 ,wind) pp))
    (push `(JSR .fft-window) pp))
  nil)


(defmacro <fft-filter_56> (result-name ff)
  (DEBUGGING (push `(----------------> <fft-filter> ,result-name ,ff) pp))
  (need-loaded '.fft-filter)
  (let ((addr (cadr (get-home-address ff 'fftflt 'R2))))
    (when (not (eq addr 'R2))
      (get-register 'R2)
      (if (not (temp-sig ff)) 
	  (push `(push (list 3 0 (clm-get-run-time-readin-index (and ,ff (fftflt-rd ,ff))) ,(+ addr 10)) *sigarr*) pup))
      (if (< addr 256)
	  (push `(LOAD R2 ,addr SHORT) pp)
	(push `(LOAD R2 ,addr) pp)))
    (if (temp-sig ff) (kill-temp ff))
    (init-N-regs)
    (spill-ALU-registers (libinfo-uses .fft-filter))
    (push `(JSR .fft-filter) pp)
    (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)

(defmacro <convolve_56> (result-name ff)
  (DEBUGGING (push `(----------------> <convolve> ,result-name ,ff) pp))
  (need-loaded '.convolve)
  (let ((addr (cadr (get-home-address ff 'conv 'R2))))
    (when (not (eq addr 'R2))
      (get-register 'R2)
      (if (not (temp-sig ff))
	  (push `(push (list 3 0 (clm-get-run-time-readin-index (and ,ff (conv-rd ,ff))) ,(+ addr 10)) *sigarr*) pup))
      (if (< addr 256)
	  (progn
	    (push `(LOAD R2 ,addr SHORT) pp)
	    (push `(LOAD R4 ,(+ addr 10) SHORT) pp))
	(progn
	  (push `(LOAD R2 ,addr) pp)
	  (push `(LOAD R4 ,(+ addr 10)) pp))))
    (push `(LOAD N2 17 SHORT) pp)
    (if (temp-sig ff) (kill-temp ff))
    (init-N-regs)
    (spill-ALU-registers (libinfo-uses .convolve))
    (push `(JSR .convolve) pp)
    (update-var-and-reg result-name 'A 'fraction 'L 'S))
  nil)
  
(defmacro <wave-train_56> (result-name e &optional in-argl)
  (let ((fm (if (alias in-argl) (eval in-argl) in-argl)))
    (DEBUGGING (push `(----------------> <wave-train> ,result-name ,e ,fm) pp))
    (need-loaded '.wave-train)
    (let ((addr (cadr (get-home-address e 'wt 'R2))))
      (when (not (eq addr 'R2))
	(get-register 'R2)
	(if (< addr 256)
	    (push `(LOAD R2 ,addr SHORT) pp)
	  (push `(LOAD R2 ,addr) pp)))
      (setf-N-reg 'N2 -1)
      (setf-N-reg 'N3 -1)
      (setf-N-reg 'N4 -1)
      (setf-N-reg 'N5 -1)
      (when fm
	(if (eq addr 'R2) (push '(LOAD N2 8) pp))
	(if (temp-sig fm) (kill-temp fm))
	(if (eq addr 'R2)		;i.e. wt is indirect through R2 (aref), so fm addr must be too
	    (convert (get-any-address fm) (get-type_56 fm) '(L R2 RN) 'real)
	  (convert (get-any-address fm) (get-type_56 fm) (list 'L (+ addr 8)) 'real)))
      (if (temp-sig e) (kill-temp e))
      (spill-ALU-registers (libinfo-uses .wave-train))
      (push `(JSR .wave-train) pp)
      (update-var-and-reg result-name 'A 'fraction 'L 'S)))
  nil)
  

;;; code continues in ins56.lisp
