;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-

(in-package :clm)

;;; cmus version 3, Mar-94.
;;;
;;; C code generator using run (run.lisp) -- this replaces ins56, code56, and sched.
;;; The C compiler replaces dsp56.  cmus.c replaces lib56.
;;;
;;; Previous versions of this code made heavy use of lisp's foreign function interface,
;;; calling each "unit generator" as a separate c function.  The first version, by hkt,
;;; used c structs and ACL's defcstruct to map values across, but that was incompatible
;;; with other lisps, so the second version used arrays instead.  Unfortunately, not
;;; only was that incompatible with MCL, it was excruciatingly slow, and the time was
;;; going into the lisp/C interface, not anything useful.  So, this version has only
;;; one foreign function call, and everything is done in C.

;;; several of the unit generators set up the field pointers at run time, but these
;;; field locations cannot change (i.e. envelope data array location), so they could
;;; be pulled out of the run-loop.  This means inventing a unique name, declare it,
;;; substitute it for all occurrences in the loop, and initialize it.  I tried this by
;;; hand in the fm-violin and didn't notice much of an improvement (actually, none at all).


(defvar *c-file* nil)
(defvar *c-proc* nil)

(defconstant _sndfix_ 32768.0)
(defconstant _sndflt_ 0.000030517578)

(defmacro make-short-float (x) `(coerce ,x 'short-float))

(defvar variable-name_c (make-hash-table))
(defvar new-user-names nil)
(defvar ok-chars_c nil)
(defvar ok-numbers_c nil)

(defun Lisp-to-C-variable-name (n)
  ;; make sure n is a legal C variable name, and if not translate it.
  ;; Allow a..z A..Z 0..9 _.  Change all other characters to _.
  ;; save these changed names so we only do this tedious translation once.
  ;; In case the new name already exists (i.e. user has variables "sin<" and "sin>"),
  ;; use "__" and so on.
  (if (null ok-chars_c)
      (setf ok-chars_c '(#\A #\a #\B #\b #\C #\c #\D #\d #\E #\e #\F #\f 
			 #\G #\g #\H #\h #\I #\i #\J #\j #\K #\k #\L #\l
			 #\M #\m #\N #\n #\O #\o #\P #\p #\Q #\q #\R #\r
			 #\S #\s #\T #\t #\U #\u #\V #\v #\W #\w #\X #\x
			 #\Y #\y #\Z #\z #\_)))
  (if (null ok-numbers_c)
      (setf ok-numbers_c '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
  (or (gethash n variable-name_c)
      (let ((new-name (make-string (length n))))
	(loop for i from 0 below (length n) do
	  (if (and (not (member (elt n i) ok-chars_c))
		   (not (member (elt n i) ok-numbers_c)))
	      (setf (elt new-name i) #\_)
	    (setf (elt new-name i) (elt n i))))
	(if (member (elt new-name 0) ok-numbers_c)
	    (setf (elt new-name 0) #\_))
	(loop while (member new-name new-user-names :test #'string-equal) do
	  (setf new-name (concatenate 'string new-name "_")))
	(setf (gethash n variable-name_c) new-name)
	(push new-name new-user-names)
	new-name)))

(defun lc (n)
  (if n
      (if (numberp n) n
	  (lisp-to-c-variable-name (symbol-name n)))))

(defun get-type_c (n)
  (if n
      (if (numberp n)
	  (if (integerp n) 'integer 'real)
	(if (constantp n) 'real
	  (let ((val (gethash n vars)))
	    (and val (second val)))))))

(defun int-p (arg) (member (get-type_c arg) '(integer long-int)))
(defun rl-p (arg) (member (get-type_c arg) '(fraction real))) ;not flt-p because that steps on the flt structure check

(defvar dat-r 0)
(defvar dat-i 0)
(defvar *run-beg* nil)
(defvar *run-end* nil)
(defvar hidden-scalers nil)
(defvar *io-incrs* nil)
(defvar *io-inits* nil)
(defvar *io-finals* nil)
(defvar *current-phrase-symbols* nil)	;kludge upon kludge!

(defvar c-test-Z "_clm_test_Z_")	;1 if test arg not 0, else 0
(defvar c-test-C "_clm_test_C_")	;1 if compare >, 0 =, -1 <

(defvar c-debug nil)

(defconstant cr_white_space "       ")

(defun initialize-everything_c ()
  (setf pup nil)			;run-time array loading
  (setf pip nil)			;run-time array allocation
  (setf dat-r 0)			;pointers into datar and datai arrays at compile time
  (setf dat-i 1)
  (setf hidden-scalers nil)
  (setf *io-incrs* nil)
  (setf *io-inits* nil)
  (setf *io-finals* nil)
  (setf *current-phrase-symbols* nil)
  (setf new-user-names nil)
  (clrhash vars)
  (clrhash variable-name_c))

;;; find-a-home-for_c needs to allocate the pointers into datar and datai, set up
;;; the lisp-side load code, pushed on pup, set up the c side initializations
;;; and add the variable to vars.
 
(defun find-a-home-for_c (var type storage &optional element-type)
  ;; in the vars table, each variable is keyed on lisp name, val='(:pass|:temp type r-addr i-addr el-type)
  (let ((sizes (structure-size_c type)))
    (setf (gethash var vars) (list storage type (if (zerop (first sizes)) -1 dat-r) (if (zerop (second sizes)) -1 dat-i) element-type))
    (when (not (eq storage :temp))
      (incf dat-r (first sizes))
      (incf dat-i (second sizes)))))

(defconstant aref_block 0)
(defconstant aref_type 1)
(defconstant aref_size 2)
(defconstant aref_element_size 3)
(defconstant aref_dims 4)
(defconstant aref_dim_list_adr 5)
(defconstant i_sizeof_aref_header 6)
(defconstant r_sizeof_aref_header 0)

(defconstant i_sizeof_io* 17)
(defconstant i_sizeof_rdin 4)
(defconstant i_sizeof_rblk (+ 3 i_sizeof_aref_header))


(defmacro <start_c> (beg end)
  (initialize-everything_c)
  ;; variables are in user-var, typed-user-var as (name type), typed-user-sig, true-user-var, new-sig (true temps).
  ;; need to find the parallel space in datar and datai (or whatever), set up the load sequence in pup (lisp),
  ;;  and send out the definitions, saving the initialization (c) for the end of the macro, so that the
  ;;  C code is ready to go with bare references to the variable names and offsets from them (for structs).
  ;; But, we don't know the types of the true temps (Run makes no effort to type them), so we'll call them
  ;; all floats for now.
  (loop for (var type el) in typed-user-var do
    (find-a-home-for_c var type (if (and (not (eq var loop-var)) (not (member var new-sig))) :pass :temp) el))
  (loop for (var typ) in typed-user-sig do
    (if (not (member typ '(io integer)))
	(find-a-home-for_c var typ :temp) ;these are locals--not defined at top level
      (find-a-home-for_c var typ (if (eq typ 'io) :allocate :initialize))))
  (loop for i in user-var do
    (when (not (gethash i vars))
      (find-a-home-for_c i 'real :pass)))
  (loop for i in true-user-var do
    (when (not (gethash i vars))
      (find-a-home-for_c i 'real (if (member i new-sig) :allocate :pass))))
  (loop for i in new-sig do
    (when (not (gethash i vars))
      (find-a-home-for_c i 'real :temp)))
  ;; now we know how big the datar and datai arrays need to be, so we can send out the lisp code
  ;; to allocate them, load them (pup), and the c code that declares and initializes references to them.
  ;; the C declarations will be via maphash over vars.
  (setf *run-beg* beg)
  (setf *run-end* end)
  ;; There is also loop-var. <t> <nil>
  (let ((ints nil)
	(floats nil)
	(int*s nil)
	(float*s nil))
    (maphash #'(lambda (key val)
		 (let* ((c-name (lc key))
			(stored (eq (first val) :pass))
			(address (find key addresses :key #'first))
			(lisp-type (if (and address (not (eq (second address) 'array))) (second address) (second val)))
			;; this ought to check (third address), but that seems to be nil at odd times
			(el-typ (fifth val))
			(c-type (if (member lisp-type '(integer long-int)) (if address :int* :int)
				  (if (member lisp-type '(real fraction)) (if address :float* :float)
				    (if (member lisp-type pure-float-structs) :float* 
				      ;; only structs with no int fields are float*
				      ;; all arrays (no matter the type) are int* => addr of block describing the array
				      ;;   0: addr of actual array 1: where (0=datar, 1=datai, 2=direct)
				      ;;   2: (total)size of array, 3: dims 4...list of dim sizes
				      :int*))))
			(r-loc (third val))
			(i-loc (fourth val)))
		   (when stored
		     (let ((old-lisp-type lisp-type)
			   (old-pup nil)
			   (old-pip nil)
			   (old-key key)
			   (old-i-loc i-loc))
		       (when (eq lisp-type 'array)
			 (setf old-pup pup)
			 (setf pup nil)
			 (setf old-pip pip)
			 (setf pip nil)
			 (setf lisp-type el-typ)
			 (setf key `(aref ,key *clm-i*))
			 (setf i-loc '*clm-i-loc*)
			 (setf r-loc '*clm-r-loc*))
		       (case lisp-type
			 ((integer long-int) (push `(setf (aref *clm-datai* ,i-loc) 
						      (if (numberp ,key) (floor ,key)
							(if ,key 1 0)))
						   pup))
			 ((real fraction)    (push `(setf (aref *clm-datar* ,r-loc) 
						      (if (numberp ,key) (make-short-float (float ,key))
							(if ,key 1.0 0.0)))
						   pup))

			 (array ;; assumed to be array of fractional arrays as in pqwvox
			  (progn
			    (push `(when ,key
				     (push (list ,key ,i-loc *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length ,key)))
				  pip)
			    (push `(if (arrayp ,key)
				       (pass-array_c ,key ,i-loc *clm-datai* *clm-datar* *clm-array-info*)
				     (setf (aref *clm-datai* ,i-loc) -1))
				  pup)))

			 (flt-one-pole       (push `(if (smpflt-p ,key) 
							(pass-one-pole_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1)) pup))
			 (flt-one-zero       (push `(if (smpflt-p ,key) 
							(pass-one-zero_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1)) pup))
			 (flt-two-pole       (push `(if (smpflt-p ,key) 
							(pass-two-pole_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1)) pup))
			 (flt-two-zero       (push `(if (smpflt-p ,key) 
							(pass-two-zero_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1)) pup))
			 (frmnt              (push `(if (frmnt-p ,key) 
							(pass-frmnt_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1)) pup))
			 (sw                 (push `(if (sw-p ,key) 
							(pass-sw_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1))
						   pup))
			 (cosp               (push `(if (cosp-p ,key) 
							(pass-cosp_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1)) pup))
			 ((noi randi)        (push `(if (noi-p ,key) 
							(pass-noi_c ,key ,r-loc *clm-datar*) 
						      (setf (aref *clm-datai* ,i-loc) -1)) pup))
			 (hloc )	;nothing needed here
			 (osc                
			  (progn
			    (push `(if (osc-p ,key) 
				       (pass-osc_c ,key ,r-loc *clm-datar*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (pushnew "_osc_phase_check_" ints)))
			 (rdin            
			  (progn
			    (push `(if (rdin-p ,key) 
				       (pass-rdin_c ,key ,i-loc *clm-datai* *clm-io-open*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (and (rdin-p ,key) (rdin-fil ,key))
				     (let ((cur-index (io-open-index (rdin-fil ,key))))
				       (when (zerop (aref *clm-io-open* cur-index)) 
					 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					 (incf *clm-datai-len* ,i_sizeof_io*))))
				  pip)))
			 (flt               
			  (progn
			    (push `(if (flt-p ,key) 
				       (pass-flt_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (flt-p ,key)
				     (push (list (flt-a ,key) (+ 3 ,i-loc) *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (flt-a ,key)))
				     (push (list (flt-b ,key) (+ 3 ,i_sizeof_aref_header ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (flt-b ,key)))
				     (when (flt-c ,key)
				       (push (list (flt-c ,key) (+ 3 (* 2 ,i_sizeof_aref_header) ,i-loc)
						   *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				       (incf *clm-datar-len* (length (flt-c ,key))))
				     (push (list (flt-d ,key) (+ 3 (* 3 ,i_sizeof_aref_header) ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (flt-d ,key))))
				  pip)))
			 (cmbflt             
			  (progn
			    (push `(if (cmbflt-p ,key) 
				       (pass-cmbflt_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (cmbflt-p ,key)
				     (push (list (cmbflt-dly-unit ,key) (+ 2 ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (dly-size (cmbflt-dly-unit ,key))))
				  pip)))
			 (allpassflt     
			  (progn
			    (push `(if (allpassflt-p ,key) 
				       (pass-allpassflt_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) 
				  pup)
			    (push `(when (allpassflt-p ,key)
				     (push (list (allpassflt-dly-unit ,key) (+ 2 ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (dly-size (allpassflt-dly-unit ,key))))
				  pip)))
			 (tbl                
			  (progn
			    (push `(if (tbl-p ,key) 
				       (pass-tbl_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (tbl-p ,key)
				     (push (list (tbl-table ,key) (+ 1 ,i-loc) *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (tbl-table-size ,key)))
				  pip)))
			 (dly                
			  (progn
			    (push `(if (dly-p ,key) 
				       (pass-dly_c ,key ,i-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (dly-p ,key)
				     (push (list ,key (+ 1 ,i-loc) *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (dly-size ,key)))
				  pip)))
			 (zdly               
			  (progn
			    (push `(if (zdly-p ,key) 
				       (pass-zdly_c ,key ,i-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (zdly-p ,key)
				     (push (list (zdly-del ,key) (+ 2 ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (dly-size (zdly-del ,key))))
				  pip)))
			 (ws                 
			  (progn
			    (push `(if (ws-p ,key) 
				       (pass-ws_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (ws-p ,key)
				     (push (list (ws-tab ,key) (+ 1 ,i-loc) *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (ws-tab ,key))))
				  pip)))
			 (locs           
			  (progn
			    (push `(if (locs-p ,key) 
				       (pass-locs_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-io-open*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when *current-output-file*
				     (let ((cur-index (io-open-index *current-output-file*)))
				       (when (zerop (aref *clm-io-open* cur-index)) 
					 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					 (incf *clm-datai-len* ,i_sizeof_io*))))
				  pip)
			    (push `(when *reverb*
				     (let ((cur-index (io-open-index *reverb*)))
				       (when (zerop (aref *clm-io-open* cur-index)) 
					 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					 (incf *clm-datai-len* ,i_sizeof_io*))))
				  pip)))
			 (smp                
			  (progn
			    (push `(if (smp-p ,key) 
				       (pass-smp_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-io-open*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup) 
			    (push `(when (and (smp-p ,key) (smp-fil ,key))
				     (let ((cur-index (io-open-index (smp-fil ,key))))
				       (when (zerop (aref *clm-io-open* cur-index)) 
					 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					 (incf *clm-datai-len* ,i_sizeof_io*))))
				  pip)))
			 (sr 
			  (progn
			    (push `(if (sr-p ,key) 
				       (pass-sr_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info* *clm-io-open*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup) 
			    (push `(when (sr-p ,key)
				     (push (list (sr-data ,key) (+ 6 ,i_sizeof_rdin ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (sr-data ,key)))
				     (when (sr-filt ,key)
				       (push (list (sr-filt ,key) (+ 6 ,i_sizeof_rdin ,i_sizeof_aref_header ,i-loc)
						   *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				       (incf *clm-datar-len* (length (sr-filt ,key))))
				     (when (rdin-fil (sr-rd ,key))
				       (let ((cur-index (io-open-index (rdin-fil (sr-rd ,key)))))
					 (when (zerop (aref *clm-io-open* cur-index)) 
					   (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					   (incf *clm-datai-len* ,i_sizeof_io*)))))
				  pip)))
			 (fft-data       
			  (progn
			    (push `(if (fft-data-p ,key) 
				       (pass-fft-data_c ,key ,i-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (fft-data-p ,key)
				     (push (list (fft-data-real ,key) (+ 1 ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (fft-data-size ,key))
				     (push (list (fft-data-imaginary ,key) (+ 1 ,i_sizeof_aref_header ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (fft-data-size ,key)))
				  pip)))
			 (wt                 
			  (progn
			    (push `(if (wt-p ,key) 
				       (pass-wt_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (wt-p ,key)
				     (push (list (wt-wave ,key) (+ 1 ,i_sizeof_rblk ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (wt-wsiz ,key))
				     (push (list (rblk-buf (wt-b ,key)) (+ 4 ,i-loc)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (rblk-siz (wt-b ,key))))
				  pip)))
			 (rblk  
			  (progn
			    (push `(if (rblk-p ,key) 
				       (pass-blk_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (rblk-p ,key)
				     (push (list (rblk-buf ,key) (+ 3 ,i-loc) *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (rblk-siz ,key)))
				  pip)))
			 (conv           
			  (progn
			    ;; may not have any arrays if two large files were convolved -- in that case the
			    ;; unit generator merely performs a readin on the temp file created in make-convolve.
			    (push `(if (conv-p ,key) 
				       (pass-conv_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info* *clm-io-open*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)

			    (push `(when (and (conv-p ,key) (rdin-fil (fftflt-rd (conv-fftf ,key))))
				     (let ((cur-index (io-open-index (rdin-fil (fftflt-rd (conv-fftf ,key))))))
				       (when (zerop (aref *clm-io-open* cur-index)) 
					 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					 (incf *clm-datai-len* ,i_sizeof_io*))))
				  pip)
			    (push `(when (and (conv-p ,key) (arrayp (conv-filtr ,key)))
				     (push (list (fftflt-datar (conv-fftf ,key)) (+ ,i-loc 4 ,i_sizeof_rdin ,i_sizeof_rblk)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (fftflt-siz (conv-fftf ,key)))
				     (push (list (fftflt-datai (conv-fftf ,key)) 
						 (+ ,i-loc 4 ,i_sizeof_rdin ,i_sizeof_rblk ,i_sizeof_aref_header)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (fftflt-siz (conv-fftf ,key)))
				     (push (list (rblk-buf (fftflt-b (conv-fftf ,key))) (+ ,i-loc 4 3 ,i_sizeof_rdin)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (rblk-siz (fftflt-b (conv-fftf ,key))))
				     (push (list (conv-filtr ,key)
						 (+ ,i-loc 4 ,i_sizeof_rdin ,i_sizeof_rblk (* 3 ,i_sizeof_aref_header))
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (conv-filtr ,key)))
				     (push (list (conv-filti ,key)
						 (+ ,i-loc 4 ,i_sizeof_rdin ,i_sizeof_rblk (* 4 ,i_sizeof_aref_header))
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (conv-filti ,key))))
				  pip)))
			 (fftflt           
			  (progn
			    (push `(if (fftflt-p ,key)
				       (pass-fftflt_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info* *clm-io-open*)
				      (setf (aref *clm-datai* ,i-loc) -1))
				  pup)
			    (push `(when (and (fftflt-p ,key) (rdin-fil (fftflt-rd ,key)))
				     (let ((cur-index (io-open-index (rdin-fil (fftflt-rd ,key)))))
				       (when (zerop (aref *clm-io-open* cur-index)) 
					 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					 (incf *clm-datai-len* ,i_sizeof_io*))))
				  pip)
			    (push `(when (fftflt-p ,key)
				     (push (list (fftflt-datar ,key) (+ ,i-loc 4 ,i_sizeof_rdin ,i_sizeof_rblk)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (fftflt-siz ,key))
				     (push (list (fftflt-datai ,key) (+ ,i-loc 4 ,i_sizeof_rdin ,i_sizeof_rblk ,i_sizeof_aref_header)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (fftflt-siz ,key))
				     (push (list (rblk-buf (fftflt-b ,key)) (+ ,i-loc 4 3 ,i_sizeof_rdin)
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (rblk-siz (fftflt-b ,key)))
				     (push (list (fftflt-env ,key) (+ ,i-loc 4 ,i_sizeof_rdin ,i_sizeof_rblk (* 2 ,i_sizeof_aref_header))
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (length (fftflt-env ,key))))
				  pip)))
			 (spd                
			  (progn
			    (push `(if (spd-p ,key) 
				       (pass-spd_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info* *clm-io-open*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (spd-p ,key)
				     (push (list (spd-b ,key) (+ 10 ,i_sizeof_rdin ,i-loc) 
						 *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				     (incf *clm-datar-len* (spd-len ,key)))
				  pip)
			    (push `(when (and (spd-p ,key) (rdin-fil (spd-rd ,key)))
				     (let ((cur-index (io-open-index (rdin-fil (spd-rd ,key)))))
				       (when (zerop (aref *clm-io-open* cur-index)) 
					 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
					 (incf *clm-datai-len* ,i_sizeof_io*))))
				  pip)))
			 (envelope
			  (progn
			    (push `(if (envelope-p ,key) 
				       (pass-env_c ,key ,i-loc ,r-loc *clm-datai* *clm-datar* *clm-array-info*) 
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when (envelope-p ,key)
				     (let* ((len (length (envelope-data ,key))))
				       (push (list ,key (+ 7 ,i-loc) *clm-datai-len* *clm-datar-len* '(1 0) 'real) *clm-array-info*)
				       (incf *clm-datar-len* len)))
				  pip)))
			 (otherwise
			  (progn
			    (push `(if ,key (pass-user-struct_c ,key ,i-loc ,r-loc  *clm-datai* *clm-datar* *clm-array-info* *clm-io-open*)
				     (setf (aref *clm-datai* ,i-loc) -1)) pup)
			    (push `(when ,key 
				     (let ((allocation-data (allocate-user-struct_c ,key ,i-loc ,r-loc
							      *clm-datai-len* *clm-datar-len* *clm-array-info* *clm-io-open*)))
				       (incf *clm-datar-len* (first allocation-data))
				       (incf *clm-datai-len* (second allocation-data))
				       (setf *clm-array-info* (third allocation-data))))
				  pip)))
			 )
		       (when (eq old-lisp-type 'array)
			 (push `(when (and ,old-key (arrayp ,old-key))
				  (let* ((el-sizes ',(structure-size_c el-typ))
					 (i-size (second el-sizes))
					 (r-size (first el-sizes))
					 (len (array-total-size ,old-key))
					 (array-i-loc 0)
					 (array-r-loc 0)
					 (dims-loc 0))
				    #+mcl (declare (ccl:ignore-if-unused array-r-loc array-i-loc))
				    (when (/= (array-rank ,old-key) 1)
				      (setf dims-loc *clm-datai-len*)
				      (incf *clm-datai-len* (array-rank ,old-key)))
				    (setf array-i-loc *clm-datai-len*)
				    (setf array-r-loc *clm-datar-len*)
				    (push (list ,old-key ,old-i-loc *clm-datai-len* *clm-datar-len* el-sizes ',el-typ dims-loc) 
					  *clm-array-info*)
				    (incf *clm-datai-len* (* len i-size))
				    (incf *clm-datar-len* (* len r-size))
				    ,(if pip `(let ((,old-key (if (= (array-rank ,old-key) 1) ,old-key
								 (make-array len 
									     :displaced-to ,old-key 
									     :element-type (array-element-type ,old-key)))))
						(loop for *clm-i* from 0 below len and
					                  *clm-i-loc* from array-i-loc by i-size and
					                  *clm-r-loc* from array-r-loc by r-size do
						  ,@pip)))))
			       old-pip)
			 (push `(when (and ,old-key (arrayp ,old-key))
				  (let* ((data (find ,old-key *clm-array-info* :key #'first))
					 (el-sizes (fifth data))
					 (i-size (second el-sizes))
					 (r-size (first el-sizes))
					 (iadr (third data))
					 (radr (fourth data))
					 (len (array-total-size ,old-key))
					 (floater (or (member ',el-typ '(real fraction)) (member ',el-typ pure-float-structs)))
					 (arr-base (if floater radr iadr))
					 (element-size (if floater r-size i-size)))
				    (setf (aref *clm-datai* ,(+ old-i-loc aref_size)) len)
				    (setf (aref *clm-datai* ,(+ old-i-loc aref_element_size)) element-size)
				    (setf (aref *clm-datai* ,(+ old-i-loc aref_type)) (array-type_c ',el-typ))
				    (setf (aref *clm-datai* ,(+ old-i-loc aref_block)) arr-base)
				    (setf (aref *clm-datai* ,(+ old-i-loc aref_dims)) (array-rank ,old-key))
				    (when (/= (array-rank ,old-key) 1)
				      (let ((dim-base (seventh data))
					    (dim-list (array-dimensions ,old-key)))
					(setf (aref *clm-datai* ,(+ old-i-loc aref_dim_list_adr)) dim-base)
					(loop for dim on dim-list by #'cdr and i from dim-base by 1 do
					  (setf (aref *clm-datai* i) (apply #'* (cdr dim))))))
				    (let ((,old-key (if (= (array-rank ,old-key) 1) ,old-key
						      (make-array len 
								  :displaced-to ,old-key 
								  :element-type (array-element-type ,old-key)))))
				      (loop for *clm-i* from 0 below len and 
					        *clm-i-loc* from iadr by i-size and
					        *clm-r-loc* from radr by r-size do
					,@pup))))
			       old-pup)
			 (setf pip old-pip)
			 (setf pup old-pup))
		       ))
		   (case c-type 
		     (:int (push c-name ints))
		     (:float (push c-name floats))
		     (:int* (push c-name int*s))
		     (:float* (push c-name float*s))
		     (:int** (push c-name int*s))
		     (:float** (push c-name float*s)))))
	     vars)
    ;; start of c-function has already gone out
    (format *c-file* "  /* declarations. The DSP_nnnn variables are compiler-generated temporaries */~%")
    #-mcl (format *c-file* "  void (*old_SIGINT)();~%")
    (format *c-file* "  int ~A,~A;~%" c-test-Z c-test-C)
    (if ints (format *c-file* "  int ~{~A~^,~};~%" ints))
    (if floats (format *c-file* "  float ~{~A~^,~};~%" floats))
    (if int*s (format *c-file* "  int ~{*~A~^,~};~%" int*s))  
    (if float*s (format *c-file* "  float ~{*~A~^,~};~%" float*s))
    #-mcl (format *c-file* "  got_sigint = 0; old_SIGINT = signal(SIGINT,sig_err); /* trap SIGINT */~%")
    (format *c-file* "  /* initializations. All Run loop entities are passed through _datai_ and _datar_ */~%")
    (maphash #'(lambda (key val)
		 (let* ((c-name (lc key))
			(stored (member (first val) '(:pass :initialize)))
			(address (find key addresses :key #'first))
			(lisp-type (if (and address (not (eq (second address) 'array))) (second address) (second val)))
			(c-type (if (member lisp-type '(integer long-int)) (if address :int* :int)
				  (if (member lisp-type '(real fraction)) (if address :float* :float)
				    (if (member lisp-type pure-float-structs) :float* 
				      :int*))))
			(r-loc (third val))
			(i-loc (fourth val)))
		   (if stored
		       (case c-type
			 (:int     (format *c-file* "  ~A = _datai_[~D];~A/* ~(~A~) */~%" c-name i-loc cr_white_space key))
			 (:float   (format *c-file* "  ~A = _datar_[~D];~A/* ~(~A~) */~%" c-name r-loc cr_white_space key))
			 (:int*    (format *c-file* "  ~A = (int *)(_datai_+~D);~A/* ~(~A~) */~%" c-name i-loc cr_white_space key))
			 (:float*  (format *c-file* "  ~A = (float *)(_datar_+~D);~A/* ~(~A~) */~%" c-name r-loc cr_white_space key))
			 (:int**   (format *c-file* "  ~A = (int *)(_datai_+_datai_[~D]);~A/* ~(~A~) */~%" 
					   c-name (+ i-loc aref_block) cr_white_space key))
			 (:float** (format *c-file* "  ~A = (float *)(_datar_+_datai[~D]);~A/* ~(~A~) */~%" 
					   c-name (+ i-loc aref_block) cr_white_space key))))))
	     vars)
    (format *c-file* "  if (_beg_ > _end_) return;~%")
    (format *c-file* "  c_init_sine();~%")
    (format *c-file* "  ~A = _beg_;~A/* pass counter */~%" loop-var cr_white_space)
    (format *c-file* "  goto SAMPLE_LOOP_INIT;~A/* pick up stuff we missed in the first pass */~%" cr_white_space)
    (format *c-file* "~%SAMPLE_LOOP_BEGIN:~%")
    #-mcl (format *c-file* "  if (got_sigint != 0) {_datai_[0]=got_sigint; goto RUN_ALL_DONE;}~%");
    (format *c-file* "  if (~A > _end_) goto RUN_ALL_DONE;~%" (lc loop-var))
    )
  nil)

(defvar debug-datai nil)
(defvar debug-datar nil)

(defmacro <end_c> (&optional no-output-here)
  (declare (ignore no-output-here))
  (let ((oscs nil))
    (maphash #'(lambda (a b) (if (and (eq (first b) :pass)
				      (or (eq (second b) 'osc)
					  (and (eq (second b) 'array) (eq (fifth b) 'osc))))
				 (push (append (list a) b) oscs)))
	     vars)
    (when oscs
      (format *c-file* "  _osc_phase_check_++;~%  if (_osc_phase_check_ > 10000) {~%    _osc_phase_check_ = 0;~%")
      (loop for os in oscs do
	(let ((var (lc (first os))))
	  (if (eq (third os) 'osc)
	      (format *c-file* "    ~A[1] = fmod(~A[1],two_pi);~%" var var)
	    (format *c-file* "    { int _i_;~%      ~
                                    for (_i_=0;_i_<~A[~D];_i_++)~%        ~
                                      _datar_[~A[0]+(_i_*2)+1] = fmod( _datar_[~A[0]+(_i_*2)+1],two_pi);}~%"
		    var aref_size var var))))
      (format *c-file* "    }~%")))
  (format *c-file* "  ~A++;~A/* increment pass counter and loop */~%~
                      ~{~^  ~A~}~%  ~
                      goto SAMPLE_LOOP_BEGIN;~%~%SAMPLE_LOOP_INIT:~%"
	  (lc loop-var) cr_white_space (nreverse *io-incrs*))
  (format *c-file* "  /* pick up initializations that were not noticed until the code was already out */~%")
  (format *c-file* "~{~^  ~A~}" (nreverse *io-inits*))
  (format *c-file* "  goto SAMPLE_LOOP_BEGIN;~%")
  (format *c-file* "~%  /* end of run loop -- update io data for lisp */~%")
  (format *c-file* "RUN_ALL_DONE:~%")
  #-mcl (format *c-file* "  signal(SIGINT,old_SIGINT);~%")
  (if *io-finals* (format *c-file* "~{~^  ~A~}" (nreverse *io-finals*)) (format *c-file* ";~%"))
  (format *c-file* "}~%")

  `(let ((*clm-datar-len* ,(1+ dat-r))
	 (*clm-datai-len* ,(1+ dat-i))
	 (*clm-io-open* (make-array *available-io-channels* :element-type 'fixnum :initial-element 0))
	 (*clm-array-info* nil))
     ,@(nreverse pip)
     (let ((*clm-datar* (make-array *clm-datar-len* :element-type 'short-float :initial-element 0.0))
	   (*clm-datai* (make-array *clm-datai-len* :element-type 'fixnum :initial-element 0)))
       ,@(nreverse pup)
       (initialize-io-records_c *clm-io-open* *clm-datai*)
       (when c-debug
	 (setf clm::debug-datai *clm-datai*)
	 (setf clm::debug-datar *clm-datar*))

       (when (or *clm-verbose* clm-statistics)
	 (setf clm-last-begin-time (max (or clm-last-begin-time 0) ,*run-beg*)))
       (when clm-statistics
	 (incf clm-total-duration (- ,*run-end* ,*run-beg*)))
       (if clm-notehook (eval clm-notehook))
       (when *clm-verbose* (princ (format nil "~(~A~) ~,3F " *current-instrument-name* (/ ,*run-beg* sampling-rate))) (force-output))

       (,*c-proc* (floor ,*run-beg*) (floor ,*run-end*) *clm-datar* *clm-datai*)
       (finalize-io-records_c *clm-io-open* *clm-datai*)
       (if *clm-phrases-active* (clm::clm-check-phrase *clm-datai* *clm-datar* *clm-array-info*))
       (setf *clm-interrupted* (aref *clm-datai* 0))
       (if (not (zerop *clm-interrupted*)) (print "clm received C-C interrupt"))
       )))

(defmacro <loop-finish_c ()
  (format *c-file* "  goto RUN_ALL_DONE;~%")
  nil)

(defun cast (arg1 arg2)
  (let ((type1 (get-type_c arg1))
	(type2 (get-type_c arg2)))
    (if (eq type1 type2)
	""
      (if (member type1 '(real fraction))
	  "(float)"
	(if (member type1 '(integer long-int))
	    "(int)"
	  (progn
	    (warn "can't cast ~(~A~) to ~(~A~)" type1 type2)
	    ""
	    ))))))

(defun rcast (arg1)
  ;; make sure arg1 is cast to float
  (if (member (get-type_c arg1) '(integer long-int))
      "(float)"
    ""))

(defun icast (arg1)
  ;; make sure arg1 is cast to int
  (if (member (get-type_c arg1) '(real fraction))
      "(int)"
    ""))

(defun both-int (arg1 arg2)
  (and (int-p arg1) (int-p arg2)))

(defun both-rl (arg1 arg2)
  (and (rl-p arg1) (rl-p arg2)))

(defun both-agree (arg1 arg2)
  (or (both-int arg1 arg2)
      (both-rl arg1 arg2)))

;;; some dsp-style variables are actually computed addresses for setf and friends --
;;; run obligingly makes a list of such folks in addresses.
;;; each element of the list is a list of the form (name field-type structure-type element-type (if array within struct))

(defmacro <setf_c> (result-name arg)
  (let* ((result-address (find result-name addresses :key #'first))
	 (result-address-type (and result-address (not (eq (second result-address) 'array)) (second result-address)))
	 (arg-address (find arg addresses :key #'first))
	 (arg-address-type (and arg-address (not (eq (second arg-address) 'array)) (second arg-address)))
	 (lcr (lc result-name))
	 (lca (lc arg)))
    (if result-address
	(if arg-address
	    (format *c-file* "  ~A = ~A;~%" lcr lca)
	  (format *c-file* "  (*~A) = ~A~A~A;~%" 
		  lcr
		  (if (and (or (and result-address-type (member result-address-type '(real fraction))) 
			       (rl-p result-name)) 
			   (rl-p arg)) ""
		    (if (int-p arg) "(float)" "(int)"))
		  lca
		  (if (member result-name hidden-scalers) " * _sndfix_" "")))
      (if arg-address
	  (format *c-file* "  ~A = ~A(*~A)~A;~%" 
		  lcr
		  (if (and (or (and arg-address-type (member arg-address-type '(real fraction))) 
			       (rl-p arg)) 
			   (rl-p result-name)) ""
		    (if (int-p result-name) "(int)" "(float)"))
		  lca
		  (if (member result-name hidden-scalers) " * _sndfix_" ""))
	;; neither is an address
	(if (and (constantp arg) (numberp arg))
	    (format *c-file* "  ~A = ~A;~%" 
		    lcr (if (both-agree result-name arg) lca (if (rl-p arg) (floor arg) (float arg))))
	  (format *c-file* "  ~A = ~A~A~A;~%" 
		  lcr (cast result-name arg) (lc arg)
		  (if (member result-name hidden-scalers) " * _sndfix_" ""))))))
    nil)

(defmacro <incf_c> (result-name arg1 &optional (arg2 1))
  (let* ((address (find result-name addresses :key #'first)))
    (format *c-file* "  ~A += ~A~A; ~A = ~A;~%" 
	    (lc arg1) (lc arg2) 
	    (if (member arg1 hidden-scalers) " * _sndfix_" "")
	    (if address 
		(format nil "(*~A)" (lc result-name))
	      (lc result-name))
	    (lc arg1))
    nil))

(defmacro <decf_c> (result-name arg1 &optional (arg2 1))
  (let* ((address (find result-name addresses :key #'first)))
    (format *c-file* "  ~A -= ~A~A; ~A = ~A;~%" 
	    (lc arg1) (lc arg2) 
	    (if (member arg1 hidden-scalers) " * _sndfix_" "")
	    (if address 
		(format nil "(*~A)" (lc result-name))
	      (lc result-name))
	    (lc arg1))
    nil))

(defmacro <negate_c> (result-name arg)                 
  (format *c-file* "  ~A = -~A;~%" (lc result-name) (lc arg)) 
  nil)

(defmacro <abs_c> (result-name arg)                    
  (format *c-file* "  if (~A~A > 0.0) ~A = ~A~A; else ~A = ~A(-~A);~%"
	  (if (int-p arg) "(float)" "") 
	  (lc arg)
	  (lc result-name) 
	  (if (both-agree result-name arg) "" (if (int-p result-name) "(int)" "(float)"))
	  (lc arg)
	  (lc result-name)
	  (if (both-agree result-name arg) "" (if (int-p result-name) "(int)" "(float)"))
	  (lc arg))
  nil)

(defmacro <add_c> (result-name &rest args)     
  (format *c-file* "  ~A = ~{~A~^+~};~%" (lc result-name) (loop for arg in args collect (lc arg)))
  nil)

(defmacro <add-1_c> (result-name arg1)
  (format *c-file* "  ~A = ~A+~A;~%" (lc result-name) (lc arg1) (if (both-int result-name arg1) "1" "1.0"))
  nil)

(defmacro <subtract-1_c> (result-name arg1)
  (format *c-file* "  ~A = ~A-~A;~%" (lc result-name) (lc arg1) (if (both-int result-name arg1) "1" "1.0"))
  nil)
  
(defmacro <subtract_c> (result-name val &rest args)
  (format *c-file* "  ~A = ~A-~{~A~^-~};~%" (lc result-name) (lc val) (loop for arg in args collect (lc arg)))
  nil)

(defmacro <multiply_c> (result-name &rest args)
  (format *c-file* "  ~A = ~{~A~^*~};~%" (lc result-name) (loop for arg in args collect (lc arg)))
  nil)

(defmacro <divide_c> (result-name val &rest args)
  (format *c-file* "  ~A = ~A/(~{~A~^*~});~%" (lc result-name) (lc val) (loop for arg in args collect (lc arg)))
  nil)

(defmacro <max_c> (result-name &rest args) 
  (format *c-file* "  ~A = ~A; " (lc result-name) (lc (first args)))
  (loop for arg in (cdr args) do
    (format *c-file* " if (~A<~A) ~A=~A;" (lc result-name) (lc arg) (lc result-name) (lc arg)))
  (format *c-file* "~%")
  nil)

(defmacro <min_c> (result-name &rest args)
  (format *c-file* "  ~A = ~A; " (lc result-name) (lc (first args)))
  (loop for arg in (cdr args) do
    (format *c-file* " if (~A>~A) ~A=~A;" (lc result-name) (lc arg) (lc result-name) (lc arg)))
  (format *c-file* "~%")
  nil)

(defmacro <log_c> (result-name arg &optional base)
  ;; assume base=e for "log" function
  (if base
      (if (= base 10)
	  (format *c-file* "  ~A = (float)log10(~A~A);~%" (lc result-name) (rcast arg) (lc arg))
	(format *c-file* "  ~A = (float)((log((double)~A) / log((double)~A)));~%" (lc result-name) (lc arg) (lc base)))
    (format *c-file* "  ~A = (float)log(~A~A);~%" (lc result-name) (rcast arg) (lc arg)))
  nil)

(defmacro <sin_c> (result-name arg) 
  (format *c-file* "  ~A = (float)sin((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <cos_c> (result-name arg) 
  (format *c-file* "  ~A = (float)cos((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <tan_c> (result-name arg) 
  (format *c-file* "  ~A = (float)tan((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <asin_c> (result-name arg)
  (format *c-file* "  ~A = (float)asin((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <acos_c> (result-name arg)
  (format *c-file* "  ~A = (float)acos((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <atan_c> (result-name arg &optional arg2)  
  (if arg2
      (format *c-file* "  ~A = (float)atan2((double)~A,(double)~A);~%" (lc result-name) (lc arg) (lc arg2))
    (format *c-file* "  ~A = (float)atan((double)~A);~%" (lc result-name) (lc arg)))
  nil)

(defmacro <sinh_c> (result-name arg)
  (format *c-file* "  ~A = (float)sinh((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <cosh_c> (result-name arg)
  (format *c-file* "  ~A = (float)cosh((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <tanh_c> (result-name arg)
  (format *c-file* "  ~A = (float)tanh((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <asinh_c> (result-name arg) 
  (format *c-file* "  ~A = (float)asinh((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <acosh_c> (result-name arg) 
  (format *c-file* "  ~A = (float)acosh((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <atanh_c> (result-name arg) 
  (format *c-file* "  ~A = (float)atanh((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <atan2_c> (result-name argy argx)
  (format *c-file* "  ~A = (float)atan2((double)~A,(double)~A);~%" (lc result-name) (lc argy) (lc argx))
  nil)

(defmacro <signum_c> (result-name arg) 
  (format *c-file* "  if (((int)~A) > 0) ~A=1; else {if (((int)~A) < 0) ~A=-1; else ~A=0;}~%" 
	  (lc arg) (lc result-name) (lc arg) (lc result-name) (lc result-name))
  nil)

(defmacro <random_c> (result-name arg) 
  (format *c-file* "  ~A = ~Ac_frandom(~A~A);~%" 
	  (lc result-name) (if (int-p result-name) "(int)" "")
	  (if (int-p arg) "(float)" "") (lc arg))
  nil)

(defmacro <expt_c> (result-name base arg)  
  ;; (expt a b) = a^b
  (if (and (numberp arg) (= arg .5))
      (format *c-file* "  ~A = ~Asqrt((double)~A);~%"
	      (lc result-name) (if (int-p result-name) "(int)" "(float)") (lc base))
    (format *c-file* "  ~A = ~Apow((double)~A,(double)~A);~%" 
	    (lc result-name) (if (int-p result-name) "(int)" "(float)") (lc base) (lc arg)))
  nil)

(defmacro <exp_c> (result-name arg) 
  ;; (exp a) = e^a
  (format *c-file* "  ~A = (float)~Aexp((double)~A);~%" 
	  (lc result-name)
	  #+(and akcl I386) "" #-(and akcl I386) "jv_"
	  (lc arg))
  nil)

(defmacro <float_c> (result-name arg)  
  (format *c-file* "  ~A = (float)~A;~%" (lc result-name) (lc arg))
  nil)

(defmacro <floor_c> (result-name arg &optional div-1)  
  (if (not div-1)
      (format *c-file* "  ~A = ~A((int)~A);~%" 
	      (lc result-name) (if (int-p result-name) "" "(float)") (lc arg))
    (format *c-file* "  ~A = ~Afloor((double)(~A/~A));~%" 
	    (lc result-name) (if (int-p result-name) "(int)" "(float)") (lc arg) (lc div-1)))
  nil)

(defmacro <ceiling_c> (result-name arg &optional div-1) 
  (format *c-file* "  ~A = ~Aceil((double)(~A/~A));~%" 
	  (lc result-name) (if (int-p result-name) "(int)" "(float)")
	  (lc arg) (or (lc div-1) 1.0))
  nil)

(defmacro <round_c> (result-name arg &optional div-1)
  (format *c-file* "  ~A = ~Afloor((double)((~A/~A)+.5));~%"
	  (lc result-name) (if (int-p result-name) "(int)" "(float)")
	  (lc arg) (or (lc div-1) 1.0))
  nil)

(defmacro <truncate_c> (result-name arg &optional div-1)
  (if (int-p result-name)
      (format *c-file* "  ~A = (int)floor((double)((~A/~A)+.5)); if (~A < 0) ~A++;~%" 
	      (lc result-name) (lc arg) (or (lc div-1) 1.0) (lc arg) (lc result-name))
    (format *c-file* "  ~A = (float)floor((double)((~A/~A)+.5)); if (~A < 0.0) ~A += 1.0;~%" 
	    (lc result-name) (lc arg) (or (lc div-1) 1.0) (lc arg) (lc result-name)))
  nil)

(defmacro <sqrt_c> (result-name arg)  
  (format *c-file* "  ~A = (float)sqrt((double)~A);~%" (lc result-name) (lc arg))
  nil)

(defmacro <ash_c> (result-name arg1 arg2) 
  (format *c-file* "  ~A = (float)(~A << ~A);~%" (lc result-name) (lc arg1) (lc arg2))
  nil)

(defmacro <mod_c> (result-name arg1 arg2 &optional rem-time) 
  (declare (ignore rem-time))
  (if (both-int arg1 arg2)
      (format *c-file* "  ~A = (float)(~A % ~A);~%" (lc result-name) (lc arg1) (lc arg2))
    (format *c-file* "  ~A = (float)fmod((double)~A,(double)~A);~%" (lc result-name) (lc arg1) (lc arg2)))
  nil)

(defmacro <rem_c> (result-name arg1 arg2) 
  (format *c-file* "  ~A = (float)drem((double)~A,(double)~A);~%" (lc result-name) (lc arg1) (lc arg2))
  nil)

(defmacro <gcd_c> (result-name &rest args) 
  (format nil "  ~A = (float)c_gcd((double)~A,(double)~A);~%" (lc result-name) (lc (first args)) (lc (second args)))
  (if (> (length args) 2)
      (loop for arg in (cddr args) do
	(format nil "  ~A = (float)c_gcd((double)~A,(double)~A);~%" (lc result-name) (lc result-name) (lc arg))))
  nil)

(defmacro <lcm_c> (result-name &rest args) 
  (format nil "  ~A = (float)c_lcm((double)~A,(double)~A);~%" (lc result-name) (lc (first args)) (lc (second args)))
  (if (> (length args) 2)
      (loop for arg in (cddr args) do
	(format nil "  ~A = (float)c_lcm((double)~A,(double)~A);~%" (lc result-name) (lc result-name) (lc arg))))
  nil)

(defmacro <lognot_c> (result-name arg)
  (format *c-file* "  ~A = ~~~A;~%" (lc result-name) (lc arg))
  nil)

(defmacro <logand_c> (result-name &rest args) 
  (if (null args)
      (format *c-file* "  ~A = -1;~%" (lc result-name))
    (format *c-file* "  ~A = ~A~{~A)~^&~};~%" 
	    (lc result-name) 
	    (make-string (length args) :initial-element #\() 
	    (loop for arg in args collect (lc arg))))
  nil)

(defmacro <logior_c> (result-name &rest args) 
  (if (null args)
      (format *c-file* "  ~A = 0;~%" (lc result-name))
    (format *c-file* "  ~A = ~A~{~A)~^|~};~%" 
	    (lc result-name) 
	    (make-string (length args) :initial-element #\() 
	    (loop for arg in args collect (lc arg))))
  nil)

(defmacro <logxor_c> (result-name &rest args) 
  (if (null args)
      (format *c-file* "  ~A = 0;~%" (lc result-name))
    (let ((arg1 (first args)))
      (loop for arg in (cdr args) do	;xor by hand!
	(format *c-file* "  ~A = ((~~(~A&~A))&(~A|~A));~%"
		(lc result-name) 
		(lc arg1) (lc arg) (lc arg1) (lc arg))
	(setf arg1 result-name))))
  nil)

(defmacro <logeqv_c> (result-name &rest args) 
  (if (null args)
      (format *c-file* "  ~A = -1;~%" (lc result-name))
    (let ((arg1 (first args)))
      (loop for arg in (cdr args) do	;xor by hand!
	(format *c-file* "  ~A = ((~~(~A&~A))&(~A|~A));~%"
		(lc result-name) 
		(lc arg1) (lc arg) (lc arg1) (lc arg))
	(setf arg1 result-name))
      (format *c-file* "  ~A = ~~~A;~%" (lc result-name) (lc result-name))))
  nil)

(defmacro <lognand_c> (result-name i1 i2) 
  (format *c-file* "  ~A = (~~(~A&~A));~%" (lc result-name) (lc i1) (lc i2))
  nil)

(defmacro <lognor_c> (result-name i1 i2) 
  (format *c-file* "  ~A = (~~(~A|~A));~%" (lc result-name) (lc i1) (lc i2))
  nil)

(defmacro <logandc1_c> (result-name i1 i2) 
  (format *c-file* "  ~A = ((~~~A)&~A);~%" (lc result-name) (lc i1) (lc i2))
  nil)

(defmacro <logandc2_c> (result-name i1 i2) 
  (format *c-file* "  ~A = (~A&(~~~A));~%" (lc result-name) (lc i1) (lc i2))
  nil)

(defmacro <logorc1_c> (result-name i1 i2) 
  (format *c-file* "  ~A = ((~~~A)|~A);~%" (lc result-name) (lc i1) (lc i2))
  nil)

(defmacro <logorc2_c> (result-name i1 i2) 
  (format *c-file* "  ~A = (~A|(~~~A));~%" (lc result-name) (lc i1) (lc i2))
  nil)

(defun lisp-to-C-print (name fstr &rest args)
  ;; translate lisp format to C printf, for the simplest cases
  (let ((ns nil)
	(fs -1)
	(argl args)
	(flen (length fstr)))
    (if (null ok-numbers_c)
	(setf ok-numbers_c '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
    (loop while (< fs (1- flen)) do
      (let ((nc (elt fstr (incf fs))))
	(if (not (char= nc #\~)) (push nc ns)
	  (progn			;now for fUn WiTh fOrMaT
	    (setf nc (elt fstr (incf fs)))
	    (if (member nc '(#\% #\&)) (progn (push #\space ns)) ;I give up -- \n cannot be printed in Lisp.
	      (progn
		(push #\% ns)
		(loop while (or (member nc ok-numbers_c) (char= nc #\,)) do (setf nc (elt fstr (incf fs))))
		(if (char= nc #\A)
		    (let ((arg (pop argl)))
		      (if (int-p arg) (push #\d ns) (push #\f ns)))
		  (if (char= nc #\D) (progn (push #\d ns) (pop argl))
		    (if (member nc '(#\F #\G #\E #\$)) (progn (push #\f ns) (pop argl))
		      (if (char= nc #\~) (push #\~ ns)
			(warn "clm's dumb format translator can't handle ~~~C in ~A" nc fstr)))))))))))
    (let ((newstr (make-string (+ (length ns) (length name)))))
      (setf ns (nreverse ns))
      (loop for i from 0 below (length name) do
	(setf (elt newstr i) (elt name i)))
      (loop for i from (length name) below (length newstr) do
	(setf (elt newstr i) (pop ns)))
      (format nil "  printf(~S~A~{~^~A~^,~}); fflush(stdout)" 
	      newstr (if args "," "") (loop for arg in args collect (lc arg))))))

(defmacro <break_c> (result-name &optional fstr &rest args) 
  (format *c-file* "  ~A=0.0; ~A; lisp_call(BREAK_C);~%" (lc result-name) (apply #'lisp-to-C-print "break:" fstr args))
  nil)

(defconstant BREAK_C 101)		;cmus.h
#+excl (ff:defun-c-callable c-break () (break "C break"))
#+excl (ff:register-function 'c-break BREAK_C)


(defmacro <warn_c> (result-name &optional fstr &rest args) 
  (format *c-file* "  ~A=0.0; ~A;~%" (lc result-name) (apply #'lisp-to-C-print "warning:" fstr args))
  nil)

(defmacro <error_c> (result-name &optional fstr &rest args) 
  (format *c-file* "  ~A=0.0; ~A; lisp_call(ERROR_C);~%" (lc result-name) (apply #'lisp-to-C-print "error:" fstr args))
  nil)

(defconstant ERROR_C 100)		;cmus.h
#+excl (ff:defun-c-callable c-error () (error "C error"))
#+excl (ff:register-function 'c-error ERROR_C)

(defmacro <print_c> (result-name fstr) 
  (declare (ignore result-name))
  (if (stringp fstr)
      (format *c-file* "  ~A; c_newline();~%" (apply #'lisp-to-C-print "" fstr nil))
    (format *c-file* "  ~A; c_newline();~%" (apply #'lisp-to-C-print "" "~A" fstr nil)))
  nil)

(defmacro <clm-print_c> (result-name &optional fstr &rest args) 
  (declare (ignore result-name))
  (format *c-file* "  ~A;~%" (apply #'lisp-to-C-print "" fstr args))
  nil)

(defmacro <princ_c> (result-name fstr) 
  (declare (ignore result-name))
  (if (stringp fstr)
      (format *c-file* "  ~A;~%" (apply #'lisp-to-C-print "" fstr nil))
    (format *c-file* "  ~A;~%" (apply #'lisp-to-C-print "" "~A" fstr nil)))
  nil)

(defmacro <y-or-n-p_c> (result-name &optional fstr &rest args)
  (format *c-file* "  ~A; ~A = c_y_or_n_p();~%" (apply #'lisp-to-C-print "" fstr args) (lc result-name))
  nil)

(defmacro <yes-or-no-p_c> (result-name &optional fstr &rest args)
  (format *c-file* "  ~A; ~A = c_yes_or_no_p();~%" (apply #'lisp-to-C-print "error:" fstr args) (lc result-name))
  nil)

(defmacro <terpri_c> (result-name)
  (declare (ignore result-name))
  (format *c-file* "  c_newline();~%")
  nil)


(defvar clm-c-funs nil)
(defun clm-c-fun (name argn) 
  (if (null clm-c-funs) (setf clm-c-funs (make-hash-table)))
  (setf (gethash name clm-c-funs) argn))
(defun c-function (name) (gethash name clm-c-funs))

(defmacro <apply_c> (result-name function-name &rest args) 
  (if (c-function function-name)
      (if (/= (length args) (c-function function-name))
	  (error "wrong number of arguments to ~A" function-name)
	(format *c-file* "  ~A = ~A(~{~^~A~^,~});~%" (lc result-name) function-name args))
    (format *c-file* "  lisp_call(FUNCTION_C);~%"))
  nil)

(defconstant FUNCTION_C 102)		;cmus.h
#+excl (ff:defun-c-callable lisp-from-c-function () (error "can't callback to a lisp function yet"))
#+excl (ff:register-function 'lisp-from-c-function FUNCTION_C)


(defmacro <funcall_c> (result-name function-name &rest args) 
  (if (c-function function-name)
      (if (/= (length args) (c-function function-name))
	  (error "wrong number of arguments to ~A" function-name)
	(format *c-file* "  ~A = ~A(~{~^~A~^,~});~%" (lc result-name) function-name args))
    (format *c-file* "  lisp_call(FUNCTION_C);~%"))
  nil)

(defmacro <label_c> (arg)
  (format *c-file* "~A:~%" (lc arg))
  nil)

(defmacro <hide_c> (&rest args)		;nothing to do here -- labels follow block structure in C
  (declare (ignore args))
  nil)

(defmacro <jump_c> (arg)
  (format *c-file* "  goto ~A;~%" (lc arg))
  nil)

(defmacro <jump-true_c> (arg) 
  (format *c-file* "  if (~A) goto ~A;~%" c-test-Z (lc arg))
  nil)

(defmacro <jump-false_c> (arg)  
  (format *c-file* "  if (!~A) goto ~A;~%" c-test-Z (lc arg))
  nil)

(defmacro <jump-l_c> (arg) 
  (format *c-file* "  if (~A == -1) goto ~A;~%" c-test-C (lc arg))
  nil)

(defmacro <jump-leq_c> (arg)  
  (format *c-file* "  if (~A != 1) goto ~A;~%" c-test-C (lc arg))
  nil)

(defmacro <jump-neq_c> (arg) 
  (format *c-file* "  if (~A) goto ~A;~%" c-test-Z (lc arg))
  nil)

(defmacro <jump-eq_c> (arg)   
  (format *c-file* "  if (~A == 0) goto ~A;~%" c-test-Z (lc arg))
  nil)

(defmacro <jump-geq_c> (arg)  
  (format *c-file* "  if (~A != -1) goto ~A;~%" c-test-C (lc arg))
  nil)

(defmacro <jump-g_c> (arg)    
  (format *c-file* "  if (~A == 1) goto ~A;~%" c-test-C (lc arg))
  nil)

(defmacro <jump-mi_c> (arg)   
  (format *c-file* "  if (~A == -1) goto ~A;~%" c-test-C (lc arg))
  nil)

(defmacro <jump-pl_c> (arg)   
  (format *c-file* "  if (~A == 1) goto ~A;~%" c-test-C (lc arg))
  nil)

(defmacro <local-label_c> (arg) 
  (format *c-file* "~A:~%" (lc arg))
  nil)

(defmacro <undefine_c> (&rest args)	;not needed, I hope!
  (declare (ignore args))
  nil)

#|
(defmacro <push_c> (result-name argl)	;no stack in use here
  nil)

(defmacro <pop_c> (argl)		;no stack in use
  nil)

(defmacro <pop-and-toss_c> ()		;ditto
  nil)
|#

(defmacro <case_c> (index name-list label-list) 
  (format *c-file* "  switch (~A){~%" (lc index))
  (loop for name in name-list and label in label-list do
	(format *c-file* "    ~A: goto ~A; break;~%" 
		(if (not (member name '(T otherwise))) (format nil "case ~A" (lc name)) "default") (lc label)))
  (format *c-file* "    }~%")
  nil)

(defmacro <logtest_c> (result-name a-i1 a-i2) ;set flag (c flag not needed)
  (format *c-file* "  ~A = ((~A~A~A&~A~A~A) != 0); ~A = ~A;~A/* (logtest ~A ~A) */~%"
	  (lc result-name)
	  (if (int-p a-i1) "" "((int)") (lc a-i1) (if (int-p a-i1) "" ")")
	  (if (int-p a-i2) "" "((int)") (lc a-i2) (if (int-p a-i2) "" ")")
	  c-test-Z (lc result-name)
	  cr_white_space a-i1 a-i2)
  nil)

(defmacro <test_c> (result-name arg) 
  (declare (ignore result-name))
  (if (int-p arg)
      (format *c-file* "  ~A = (~A != 0);~%  if (~A == 0) ~A = 0;~%  else if (~A > 0) ~A = 1; else ~A = -1;~%"
	      c-test-Z (lc arg) (lc arg) c-test-C (lc arg) c-test-C c-test-C)
    (format *c-file* "  ~A = (~A != 0.0);~%  if (~A == 0.0) ~A = 0;~%  else if (~A > 0) ~A = 1; else ~A = -1;~%"
	      c-test-Z (lc arg) (lc arg) c-test-C (lc arg) c-test-C c-test-C))
  nil)

(defmacro <not-null_c> (arg)
  ;; always followed by jump-true/false, so c test not needed
  (if (or (int-p arg) (rl-p arg))
      (format *c-file* "  ~A = (~A != ~A);~%" c-test-Z (lc arg) (if (int-p arg) "0" "0.0"))
    (format *c-file* "  ~A = (_datai_[~D] != -1);~%" c-test-Z (fourth (gethash arg vars))))
  nil)

(defmacro <logbitp_c> (result-name indl arg) ;set flags
  ;; result-name is not used directly
  (declare (ignore result-name))
  (format *c-file* "  ~A = ((~A~A~A&(1<<~A)) != 0);~A/* (logbitp ~A ~A) */~%"
	  c-test-Z (if (int-p arg) "" "((int)") (lc arg) (if (int-p arg) "" ")")
	  (lc indl) cr_white_space indl arg)
  nil)

(defmacro <test-bit-0_c> (result-name arg) ;set flags
  (declare (ignore result-name))
  (format *c-file* "  ~A = (~A~A~A&1);~%" c-test-Z (if (int-p arg) "" "((int)") (lc arg) (if (int-p arg) "" ")"))
  nil)

(defmacro <compare_c> (arg1 arg2 &optional dont-need-C) ;set flags after arg1-arg2
  ;;(argl1 > = < argl2) 
  ;; in special cases where only = matters, dont-need-C will be t
  (if dont-need-C
      (format *c-file* "  ~A = (~A~A~A != ~A~A~A);~%" 
	      c-test-Z 
	      (if (and (rl-p arg1) (int-p arg2)) "((int)" "")
	      (lc arg1)
	      (if (and (rl-p arg1) (int-p arg2)) ")" "")
	      (if (and (rl-p arg2) (int-p arg1)) "((int)" "")
	      (lc arg2)
	      (if (and (rl-p arg2) (int-p arg1)) ")" ""))
    (if (int-p arg1)
	(if (int-p arg2)
	    (format *c-file* "  if (~A == ~A) {~A=0; ~A=0;}~%  else {~A=1; if (~A > ~A) ~A=1; else ~A=-1;}~%"
		    (lc arg1) (lc arg2) c-test-Z c-test-C c-test-Z (lc arg1) (lc arg2) c-test-C c-test-C)
	  (format *c-file* "  if (~A == ((int)~A)) {~A=0; ~A=0;} else {~A=1; if (~A > ((int)~A)) ~A=1; else ~A=-1;}~%"
		  (lc arg1) (lc arg2) c-test-Z c-test-C c-test-Z (lc arg1) (lc arg2) c-test-C c-test-C))
      (if (int-p arg2)
	  (format *c-file* "  if (((int)~A) == ~A) {~A=0; ~A=0;}~%  else {~A=1; if (((int)~A) > ~A) ~A=1; else ~A=-1;}~%"
		  (lc arg1) (lc arg2) c-test-Z c-test-C c-test-Z (lc arg1) (lc arg2) c-test-C c-test-C)
	(format *c-file* "  if (~A == ~A) {~A=0; ~A=0;} else {~A=1; if (~A > ~A) ~A=1; else ~A=-1;}~%"
		(lc arg1) (lc arg2) c-test-Z c-test-C c-test-Z (lc arg1) (lc arg2) c-test-C c-test-C))))
  nil)

(defmacro package-c-address (result-name s accessor c-type addr in-setf &optional deref scaled)
  ;; if no deref, then it's at s[addr] -- if setf, return (ptr *)(s+addr)
  ;; if deref, it's at datar[deref] + addr] -- if setf (ptr *)(_datar_+s[deref]+addr)
  ;; if float* it's a pure float struct (result is), if int* all the other structs including io
  ;; if float** it's an array header pointer to a float array, if int** currently an error
  (let ((lcr (lc result-name))
	(lcs (lc s)))
    (if (member c-type '(:int :float))
	(if (not deref)
	    (if (not in-setf)
		(format *c-file* "  ~A = ~A[~D];~A/* (~A ~(~A~)) */~%" 
			lcr lcs addr cr_white_space accessor s)
	      (format *c-file* "  ~A = (~A *)(~A + ~D);~A/* (setf (~A ~(~A~)) ...) */~%" 
		      lcr (if (eq c-type :float) "float" "int") lcs addr cr_white_space accessor s))
	  (if (not in-setf)
	      (format *c-file* "  ~A = _datar_[~A[~D]+~D]~A;~A/* (~A ~(~A~)) */~%" 
		      lcr lcs deref addr (if scaled " * _sndflt_" "") cr_white_space accessor s)
	    (format *c-file* "  ~A = (float *)(_datar_+~A[~D]+~D);~A/* (setf (~A ~(~A~)) ...) */~%" 
		    lcr lcs deref addr cr_white_space accessor s)))
      (if (member c-type '(:int* :float*))
	  (if (not deref)
	      (format *c-file* "  ~A = (~A *)(~A+~D);~A/* (~A ~(~A~)) */~%" 
		      lcr (if (eq c-type :float*) "float" "int") lcs addr cr_white_space accessor s)
	    (format *c-file* "  ~A = (float *)(_datar_+~A[~D]+~D);~A/* (~A ~(~A~)) */~%" 
		    lcr lcs deref addr cr_white_space accessor s))
	(if (eq c-type :float**)
	    ;; reference to the float array whose header is at s[addr]
	    (format *c-file* "  ~A = (int *)(~A+~D);~A/* (~A ~(~A~)) */~%" 
		    lcr lcs addr cr_white_space accessor s)))))
  nil)


;;; ---- AREF ----
;;;
;;; all arrays are pointers to the array header block.
;;; see <start> for aref indices
;;; if dims > 1 then aref_dim_list_adr points to the list of dimensions
;;; in all cases, size = array-total-size

(defun array-type_c (lisp-type)
  (case lisp-type 
	((integer long-int) 0) ((real fraction) 1) (tbl 2) (ws 3)
	(cmbflt 4) (allpassflt 5) (dly 6) (zdly 7) (flt 8) (flt-one-pole 9)             
	(flt-one-zero 10) (flt-two-pole 11) (flt-two-zero 12) (frmnt 13) 
	(noi 14) (randi 15) (sw 16) (smp 17) (osc 18) (rdin 19) (spd 20)
	(sr 21) (fftflt 22) (fft-data 23) (cosp 24) (locs 25) 
	((array table x-table y-table) 26) (rblk 27) (wt 28) 
	(envelope 29) (smpflt 30) (conv 31)                  
	(t 32)))

(defun array-type_lisp (num)
  (case num (0 'integer) (1 'real) (2 'tbl) (3 'ws)
	(4 'cmbflt) (5 'allpassflt) (6 'dly) (7 'zdly) (8 'flt) (9 'flt-one-pole)             
	(10 'flt-one-zero) (11 'flt-two-pole) (12 'flt-two-zero) (13 'frmnt) 
	(14 'noi) (15 'randi) (16 'sw) (17 'smp) (18 'osc) (19 'rdin) (20 'spd)
	(21 'sr) (22 'fftflt) (23 'fft-data) (24 'cosp) (25 'locs) 
	(26 'array) (27 'rblk) (28 'wt) (29 'envelope) (30 'smpflt) (31 'conv)               
	(32 t)))

#+excl (progn
	 (ff:defforeign 'c-bltf :entry-point "_c_bltf" :prototype t :arguments '(array fixnum array fixnum) :return-type :void)
	 (ff:defforeign 'c-blti :entry-point "_c_blti" :prototype t :arguments '(array fixnum array fixnum) :return-type :void)
	 )

(defun pass-array_c (arr i datai datar info)
  (let* ((val (find arr info :key #'first))
	 (iadr (third val))
	 (radr (fourth val))
	 (el-siz (fifth val))
	 (el-typ (sixth val))
	 (len (if (dly-p arr) (dly-size arr) 
		(if (envelope-p arr) (length (envelope-data arr))
		  (length arr))))
	 (arr-base 0)
	 (element-size 0))
    (if (member el-typ '(real fraction))
	(progn
	  (setf arr-base radr)
	  (setf element-size (first el-siz)))
      (progn
	(setf arr-base iadr)
	(setf element-size (second el-siz))))
    (setf (aref datai (+ i aref_size)) len)
    (setf (aref datai (+ i aref_element_size)) element-size)
    (setf (aref datai (+ i aref_type)) (array-type_c el-typ))
    (setf (aref datai (+ i aref_block)) arr-base)
    (setf (aref datai (+ i aref_dims)) 1)
    (if (dly-p arr)
	(if (dly-ext-id arr)
	    (if (listp (dly-ext-id arr))
		(loop for k from 0 below len and x in (dly-ext-id arr) do
		  (setf (aref datar (+ arr-base k)) (make-short-float x)))
	      (dotimes (k len) (setf (aref datar (+ arr-base k)) (dly-ext-id arr)))))
      (if (envelope-p arr)
	  (loop for k from 0 below len and x in (envelope-data arr) do
	    (setf (aref datar (+ arr-base k)) (make-short-float x)))
	(if (member el-typ '(integer long-int))
	    #+excl (c-blti datai arr-base arr len)
	    #-excl (dotimes (k len) (setf (aref datai (+ arr-base k)) (aref arr k)))
	  (if (member el-typ '(real fraction))
	      #+excl (c-bltf datar arr-base arr len)
	      #-excl (dotimes (k len) (setf (aref datar (+ arr-base k)) (make-short-float (aref arr k))))
	    ))))))


(defmacro <aref_c> (result-name arr arr-type &rest indices)
  (let* ((lcr (lc result-name))
	 (lca (lc arr))
	 (lci (if (= (length indices) 1) 
		  (if (int-p (first indices))
		      (lc (first indices))
		    (format nil "((int)~A)" (lc (first indices))))
		(format nil "(~{(~A~A * _datai_[~A[~D]+~D])~^+~})"
			(loop for index in indices and i from 0 by 1 
			 collect (if (rl-p index) "(int)" "")
			 collect (lc index)
			 collect lca 
			 collect aref_dim_list_adr
			 collect i))))
	 (el-siz (structure-size_c arr-type))
	 (i-size (second el-siz))
	 (r-size (first el-siz))
	 (flt-ind (rl-p (first indices)))
	 (lcint (if flt-ind "(int)" "")))
    (if (not (find result-name addresses :key #'first))
	(if (member arr-type '(real fraction))
	    (format *c-file* "  ~A = _datar_[~A[~D]+~A];~%" lcr lca aref_block lci)
	  (if (member arr-type '(integer long-int))
	      (format *c-file* "  ~A = ~A(_datai_[~A[~D]+~A]);~%" lcr (if (rl-p result-name) "(float)" "") lca aref_block lci)
	    (if (member arr-type pure-float-structs)
		(format *c-file* "  ~A = (float *)(_datar_ + ~A[~D] + (~A * ~D));~%" 
			lcr lca aref_block lci r-size)
	      (format *c-file* "  ~A = (int *)(_datai_ + ~A[~D] + (~A * ~D));~%" 
		      lcr lca aref_block lci i-size))))
      (if (member arr-type '(real fraction))
	  (format *c-file* "  ~A = (_datar_+~A[0] + ~A);~%" lcr lca lci)
	(if (member arr-type pure-float-structs)
	    (format *c-file* "  ~A = (float *)(_datar_+~A[~D] + ~A~A~A~A);~%" 
		    lcr lca aref_block 
		    (if (/= r-size 1) "(" "") 
		    lci
		    (if (/= r-size 1) (format nil " * ~D" r-size) "")
		    (if (/= r-size 1) ")" ""))
	  (if (member arr-type '(integer long-int))
	      (format *c-file* "  ~A = (_datai_+~A[0] + ~A~A);~%" lcr lca lcint lci)
	    (format *c-file* "  ~A = (int *)(_datai_+~A[~D] + ~A~A~A~A);~%" 
		    lcr lca aref_block 
		    (if (/= i-size 1) "(" "") 
		    lci
		    (if (/= i-size 1) (format nil " * ~D" i-size) "")
		    (if (/= i-size 1) ")" "")))))))
  nil)

(defmacro <tref_c> (result-name tab tab-type ind &optional (store nil)) 
  (declare (ignore store))
  `(<aref_c> ,result-name ,tab ,tab-type ,ind))

(defmacro <setf-tref_c> (result-name tab tab-type ind) 
  `(<aref_c> ,result-name ,tab ,tab-type ,ind))


(defmacro <array-rank_c> (result-name arr) `(package-c-address ,result-name ,arr "array-rank" :int ,aref_dims nil))
(defmacro <array-total-size_c> (result-name arr) `(package-c-address ,result-name ,arr "array-total-size" :int ,aref_size nil))
(defmacro <array-rank-limit_c> (result-name arr) (declare (ignore arr)) (format *c-file* "  ~A = 2048;~%" (lc result-name)))

(defmacro <array-dimension-limit_c> (result-name arr) 
  (declare (ignore arr)) 
  (format *c-file* "  ~A = ~D;~%" (lc result-name) array-dimension-limit))

;(defmacro <array-element-type_c> (result-name arr) 
;(defmacro <array-dimension_c> (result-name arr axis)
;(defmacro <array-in-bounds-p_c> (result-name arr &rest subscripts)
;(defmacro <adjustable-array-p_c> (result-name arr) '<nil>)




;;; ---- OSCIL ---- 
;;; 0:freq 1:phase 

(defmacro <oscil_c> (result-name s &optional fm-1 pm-1)
  ;; really simple unit generators are expanded in-line
  (let* ((pm-in-use (and pm-1 (or (not (constantp pm-1)) (not (zerop pm-1)))))
	 (fm-in-use (and fm-1 (or (not (constantp fm-1)) (not (zerop fm-1)))))
	 (s-name (lc s))
	 (pf (if (and pm-in-use (int-p pm-1)) "(float)" ""))
	 (ff (if (and fm-in-use (int-p fm-1)) "(float)" "")))
    (format *c-file* "  ~A = c_sin_lookup(~A[1]~A);~A/* (oscil ~(~A~)~A~A) */~%" 
	    (lc result-name) s-name 
	    (if pm-in-use (format nil " + ~A~A" pf (lc pm-1)) "")
	    cr_white_space
	    s (if fm-1 (format nil " ~A" fm-1) "") (if pm-1 (format nil " ~A" pm-1) ""))
    (format *c-file* "  ~A[1] += ~A~A[0]~A;~%"
	    s-name (if fm-in-use "(" "") s-name 
	    (if fm-in-use (format nil " + ~A~A)" ff (lc fm-1)) ""))
    nil))

(defconstant r_sizeof_osc 2)
(defconstant i_sizeof_osc 1)

(defun pass-osc_c (osc addr datar)
  (setf (aref datar addr) (make-short-float (osc-freq osc)))
  (setf (aref datar (1+ addr)) (make-short-float (osc-phase osc))))

(defun dm-osc_c (addr datar)
  (make-osc :freq (aref datar addr) :phase (aref datar (1+ addr))))

(defmacro <osc-phase_c> (result-name osc in-setf) `(package-c-address ,result-name ,osc "osc-phrase" :float 1 ,in-setf))
(defmacro <osc-freq_c> (result-name osc in-setf) `(package-c-address ,result-name ,osc "osc-freq" :float 0 ,in-setf))



;;; ---- SAWTOOTH-WAVE, SQUARE-WAVE, PULSE-TRAIN, TRIANGLE-WAVE ----
;;; 0:freq 1:phase 2:value 3:base

(defconstant i_sizeof_sw 1)
(defconstant r_sizeof_sw 4)

(defmacro <sw-freq_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sw-freq" :float 0 ,in-setf))
(defmacro <sw-phase_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sw-phase" :float 1 ,in-setf))
(defmacro <sw-current-value_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sw-current-value" :float 2 ,in-setf))
(defmacro <sw-base_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sw-base" :float 3 ,in-setf))

(defun pass-sw_c (s adr datar)
  (setf (aref datar adr) (make-short-float (sw-freq s)))
  (setf (aref datar (+ adr 1)) (make-short-float (sw-phase s)))
  (setf (aref datar (+ adr 2)) (make-short-float (sw-current-value s)))
  (setf (aref datar (+ adr 3)) (make-short-float (sw-base s))))

(defun dm-sw_c (adr datar)
  (make-sw :freq (aref datar adr) :phase (aref datar (+ adr 1)) :current-value (aref datar (+ adr 2)) :base (aref datar (+ adr 3))))

(defmacro <square-wave_c> (result-name s &optional fm) 
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = ~A~A[2];~A/* (square-wave ~(~A~) ~A) */~%  ~
                        ~A[1] += ~A~A[0]~A;~%  ~
                        while (~A[1] >= two_pi) ~A[1] -= two_pi; while (~A[1] < 0.0) ~A[1] += two_pi;~%  ~
                        if (~A[1] < one_pi) ~A[2] = ~A[3]; else ~A[2] = 0.0;~%"
	    (lc result-name) (if (int-p result-name) "(int)" "") lc-s cr_white_space s fm
	    lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")
	    lc-s lc-s lc-s lc-s
	    lc-s lc-s lc-s lc-s))
  nil)

(defmacro <pulse-train_c> (result-name s &optional fm) 
  (let ((lc-s (lc s)))
    (format *c-file* "  if ((~A[1] > two_pi) || (~A[1] < minus_two_pi))~A/* (pulse-train ~(~A~) ~A) */~%    ~
                          { ~A = ~A[3];~%      ~
                            while (~A[1] >= two_pi) ~A[1] -= two_pi; while (~A[1] < 0.0) ~A[1] += two_pi;}~%  ~
                        else ~A = 0.0;~%  ~
                        ~A[1] += ~A~A[0]~A;~%"
	    lc-s lc-s cr_white_space s fm
	    (lc result-name) lc-s
	    lc-s lc-s lc-s lc-s
	    (lc result-name)
	    lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")))
  nil)

(defmacro <triangle-wave_c> (result-name s &optional fm) 
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = ~A~A[2];~A/* (triangle-wave ~(~A~) ~A) */~%  ~
                        ~A[1] += ~A~A[0]~A;~%  ~
                        while (~A[1] >= two_pi) ~A[1] -= two_pi; while (~A[1] < 0.0) ~A[1] += two_pi;~%  ~
                        if (~A[1] < half_pi) ~A[2] = (~A[3]*~A[1]);~%  ~
                        else if (~A[1] < three_half_pi) ~A[2] = ~A[3]*(one_pi - ~A[1]);~%    ~
                          else ~A[2] = ~A[3]*(~A[1] - two_pi);~%"
	    (lc result-name) (if (int-p result-name) "(int)" "") lc-s cr_white_space s fm
	    lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")
	    lc-s lc-s lc-s lc-s
	    lc-s lc-s lc-s lc-s
	    lc-s lc-s lc-s lc-s
	    lc-s lc-s lc-s))
  nil)

(defmacro <sawtooth-wave_c> (result-name s &optional fm) 
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = ~A~A[2];~A/* (sawtooth-wave ~(~A~) ~A) */~%  ~
                        ~A[1] += ~A~A[0]~A;~%  ~
                        while (~A[1] >= two_pi) ~A[1] -= two_pi; while (~A[1] < 0.0) ~A[1] += two_pi;~%  ~
                        ~A[2] = (~A[3] * (~A[1] - one_pi));~%"
	    (lc result-name) (if (int-p result-name) "(int)" "") lc-s cr_white_space s fm
	    lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")
	    lc-s lc-s lc-s lc-s
	    lc-s lc-s lc-s))
  nil)



;;; ---- SUM-OF-COSINES ----
;;; cosp struct: 0:freq 1:phase 2:scaler 3:cosines

(defconstant i_sizeof_cosp 1)
(defconstant r_sizeof_cosp 4)

(defmacro <cosp-freq_c> (result-name s in-setf) `(package-c-address ,result-name ,s "cosp-freq" :float 0 ,in-setf))
(defmacro <cosp-phase_c> (result-name s in-setf) `(package-c-address ,result-name ,s "cosp-phase" :float 1 ,in-setf))
(defmacro <cosp-scaler_c> (result-name s in-setf) `(package-c-address ,result-name ,s "cosp-scaler" :float 2 ,in-setf))
(defmacro <cosp-cosines_c> (result-name s in-setf) `(package-c-address ,result-name ,s "cosp-cosines" :float 3 ,in-setf))

(defun pass-cosp_c (s adr datar)
  (setf (aref datar adr) (make-short-float (cosp-freq s)))
  (setf (aref datar (+ adr 1)) (make-short-float (cosp-phase s)))
  (setf (aref datar (+ adr 2)) (make-short-float (cosp-scaler s)))
  (setf (aref datar (+ adr 3)) (make-short-float (float (cosp-cosines s)))))

(defun dm-cosp_c (adr datar)
  (make-cosp :freq (aref datar adr) :phase (aref datar (+ adr 1)) :scaler (aref datar (+ adr 2)) :cosines (floor (aref datar (+ adr 3)))))

(defmacro <sum-of-cosines_c> (result-name s &optional fm) 
  (let ((lc-s (lc s)))
    (format *c-file* "  if ((~A[1] == 0.0) || (~A[1] == two_pi)) ~A = 1.0;~A/* (sum-of-cosines ~(~A~) ~A) */~%  ~
                        else ~A = ((~A[2]*c_sin_lookup(~A[1]*(~A[3]+.5))) / (c_sin_lookup(~A[1]*.5)));~%  ~
                        ~A[1] += ~A~A[0]~A;~%  ~
                        while (~A[1] >= two_pi) ~A[1] -= two_pi; while (~A[1] < 0.0) ~A[1] += two_pi;~%"
	    lc-s lc-s (lc result-name) cr_white_space s fm
	    (lc result-name) lc-s lc-s lc-s lc-s
	    lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")
	    lc-s lc-s lc-s lc-s))
  nil)


;;; ---- RANDH, RANDI ----
;;; 0:freq 1:phase 2:incr 3:output 4:base

(defconstant i_sizeof_noi 1)
(defconstant r_sizeof_noi 5)

(defmacro <noi-freq_c> (result-name s in-setf) `(package-c-address ,result-name ,s "noi-freq" :float 0 ,in-setf))
(defmacro <noi-phase_c> (result-name s in-setf) `(package-c-address ,result-name ,s "noi-phase" :float 1 ,in-setf))
(defmacro <noi-incr_c> (result-name s in-setf) `(package-c-address ,result-name ,s "noi-incr" :float 2 ,in-setf))
(defmacro <noi-output_c> (result-name s in-setf) `(package-c-address ,result-name ,s "noi-output" :float 3 ,in-setf))
(defmacro <noi-base_c> (result-name s in-setf) `(package-c-address ,result-name ,s "noi-base" :float 4 ,in-setf))

(defun pass-noi_c (s adr datar)
  (setf (aref datar adr) (make-short-float (noi-freq s)))
  (setf (aref datar (+ adr 1)) (make-short-float (noi-phase s)))
  (setf (aref datar (+ adr 2)) (make-short-float (noi-incr s)))
  (setf (aref datar (+ adr 3)) (make-short-float (noi-output s)))
  (setf (aref datar (+ adr 4)) (make-short-float (noi-base s))))

(defun dm-noi_c (adr datar)
  (make-noi :freq (aref datar adr) :phase (aref datar (+ adr 1)) :incr (aref datar (+ adr 2)) 
	    :output (aref datar (+ adr 3)) :base (aref datar (+ adr 4))))

(defmacro <randh_c> (result-name s &optional fm) 
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = ~A~A[3];~A/* (randh ~(~A~) ~A) */~%  ~
                        if ((~A[1] >= two_pi) || (~A[1] < minus_two_pi)) {~%    ~
                           ~A[3] = c_frandom(~A[4]);~%    ~
                           while (~A[1] >= two_pi) ~A[1] -= two_pi; while (~A[1] < minus_two_pi) ~A[1] += two_pi;}~%  ~
                         ~A[1] += ~A~A[0]~A;~%"
	    (lc result-name) (if (int-p result-name) "(int)" "") lc-s cr_white_space s fm
	    lc-s lc-s
	    lc-s lc-s lc-s lc-s lc-s lc-s
	    lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")))
  nil)

(defmacro <randi_c> (result-name s &optional fm) 
  ;; moving c_frandom inline makes no difference in speed
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = ~A~A[3];~A/* (randi ~(~A~) ~A) */~%  ~
                        ~A[3] += ~A[2];~%  ~
                        if ((~A[1] >= two_pi) || (~A[1] < minus_two_pi)) {~%    ~
                            ~A[2] = (c_frandom(~A[4]) - ~A[3]) * (~A~A[0]~A * inverse_pi);~%    ~
                            while (~A[1] >= two_pi) ~A[1] -= two_pi; while (~A[1] < minus_two_pi) ~A[1] += two_pi;}~%  ~
                         ~A[1] += ~A~A[0]~A;~%"
	    (lc result-name) (if (int-p result-name) "(int)" "") lc-s cr_white_space s fm
	    lc-s lc-s
	    lc-s lc-s
	    lc-s lc-s lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")
	    lc-s lc-s lc-s lc-s
	    lc-s (if fm "(" "") lc-s (if fm (format nil "+~A)" (lc fm)) "")))
  nil)



;;; ---- AMPLITUDE-MODULATE ----
(defmacro <amplitude-modulate_c> (result-name cr-1 s1-1 s2-1) 
  (format *c-file* "  ~A = (~A~A + (~A~A * ~A~A));~A/* (amplitude-modulate ~(~A~) ~A ~A) */~%"
	  (lc result-name) (if (int-p s1-1) "(float)" "") (lc s1-1)
	  (if (int-p cr-1) "(float)" "") (lc cr-1)
	  (if (int-p s2-1) "(float)" "") (lc s2-1)
	  cr_white_space cr-1 s1-1 s2-1)
  nil)



;;; ---- TABLE-LOOKUP ----
;;; datai: 0:datar adr 1: array header (of table)
;;; datar: 0:freq 1:phase 2:internal-mag

(defconstant i_sizeof_tbl (+ 1 i_sizeof_aref_header))
(defconstant r_sizeof_tbl 3)

(defmacro <tbl-freq_c> (result-name tbl in-setf) `(package-c-address ,result-name ,tbl "tbl-freq" :float 0 ,in-setf 0))
(defmacro <tbl-phase_c> (result-name tbl in-setf) `(package-c-address ,result-name ,tbl "tbl-phase" :float 1 ,in-setf 0))
(defmacro <tbl-internal-mag_c> (result-name tbl in-setf) `(package-c-address ,result-name ,tbl "tbl-internal-mag" :float 2 ,in-setf 0))
(defmacro <tbl-table_c> (result-name tbl in-setf) `(package-c-address ,result-name ,tbl "tbl-table" :float** 1 ,in-setf))
(defmacro <tbl-table-size_c> (result-name tbl in-setf) `(package-c-address ,result-name ,tbl "tbl-table-size" :int ,(+ 1 aref_size) ,in-setf))

(defun pass-tbl_c (tbl i r datai datar info)
  (setf (aref datai i) r)
  (setf (aref datar r) (make-short-float (tbl-freq tbl)))
  (setf (aref datar (+ r 1)) (make-short-float (tbl-phase tbl)))
  (setf (aref datar (+ r 2)) (make-short-float (tbl-internal-mag tbl)))
  (pass-array_c (tbl-table tbl) (+ i 1) datai datar info))

(defun dm-tbl_c (adr datai datar)
  (let ((size (aref datai (+ adr 1 aref_size))))
    (make-tbl :freq (aref datar (aref datai adr)) 
	      :phase (aref datar (+ (aref datai adr) 1))
	      :internal-mag (aref datar (+ (aref datai adr) 2))
	      :table-size size
	      :table (if (< size *clm-array-print-length*)
			 (make-array size
				     :element-type (type-of (aref datar 0))
				     ;; believe it or not (make-array n :element-type 'short-float) in ACL creates an
				     ;; array whose elements are of type single-float, and you cannot subsequently
				     ;; displace a short-float array onto the previous one!!
				     :displaced-to datar
				     :displaced-index-offset (aref datai (+ adr 1 aref_block)))
		       (make-array 2 :initial-contents (list 0 (1- size)))))))

(defmacro <ur-table-lookup_c> (result-name s &optional fm-1) 
  ;; OPT: could be made much faster by avoiding function call (inline it)
  (format *c-file* "  ~A = c_table_lookup(~A,_datar_,~A);~%"
	  (lc result-name) (lc s) (if fm-1 (lc fm-1) 0.0))
  nil)

(defmacro <table-interp_c> (result-name s index) 
  `(<array-interp_c> ,result-name ,s ,index))

(defmacro <array-interp_c> (result-name s index) 
  ;; OPT: could be made much faster by avoiding function call (inline it)
  (if (int-p index)
      (format *c-file* "  ~A = _datar_[~A[0]+~A];~%" (lc result-name) (lc s) (lc index))
    (format *c-file* "  ~A = c_table_interp((float *)(_datar_+~A[0]),~A,~A[~D]);~%"
	    (lc result-name) (lc s) (lc index) (lc s) aref_size))
  nil)                                  ;was (+ 1 aref_size), but that's a confusion with the table layout above -- here we're
					;  dealing with the actual array, not an imbedded array.



;;; ---- POLYNOMIAL ----
;;;
;;; float arrays are aways pointers to the array header with the data in _datar_

(defmacro <polynomial_c> (result-name tab x) 
  (let ((lc-t (lc tab)))
    (format *c-file* "  ~A = _datar_[~A[0]+~A[~D]-1];~A/* (polynomial ~(~A~) ~A) */~%  ~
                        {int _i_; for (_i_=(~A[0]+~A[~D]-2);_i_>=~A[0];_i_--) ~A = ((~A * ~A) + _datar_[_i_]);}~%"
	     (lc result-name) lc-t lc-t aref_size cr_white_space tab x
	     lc-t lc-t aref_size lc-t (lc result-name) (lc x) (lc result-name)))
  nil)




;;; ---- DOT-PRODUCT ----
(defmacro <dot-product_c> (result-name s1 s2) 
  (format *c-file* "  ~A = 0.0;~%  ~
                      { int _i_,_lim_; float *_j_,*_k_;~A/* (dot-product ~(~A~) ~(~A~)) */~%    ~
                        _lim_ = ~A[~D]; if (~A[~D] < _lim_) _lim_ = ~A[~D];~%    ~
                        _j_ = (float *)(_datar_+~A[0]); _k_ = (float *)(_datar_+~A[0]);~%    ~
                        for (_i_=0;_i_<_lim_;_i_++,_j_++,_k_++) ~A += ((*_j_) * (*_k_)); }~%"
	  (lc result-name)
	  cr_white_space (lc s1) (lc s2)
	  (lc s1) aref_size (lc s2) aref_size (lc s2) aref_size
	  (lc s1) (lc s2)
	  (lc result-name))
  nil)



;;; ---- ONE-POLE, ONE-ZERO, TWO-POLE, TWO-ZERO ----
;;; all smpflt allocate the same storage for structure references
;;; 0:a0 1:a1 2:a2 3:b1 4:b2 5:x1 6:x2 7:y1 8:y2

(defconstant r_sizeof_smpflt 9)
(defconstant i_sizeof_smpflt 1)

(defmacro <one-pole_c> (result-name filt arg) 
  (let ((flt (lc filt))
	(val (lc result-name))
	(a0 0)
	(b1 3)
	(y1 7))
    (format *c-file* "  ~A = (~A * ~A[~D]) - (~A[~D] * ~A[~D]); ~A[~D] = ~A;~A/* (one-pole ~(~A~) ~A) */~%"
	    val (lc arg)  flt a0  flt b1  flt y1  flt y1  val 
	    cr_white_space filt arg))
  nil)

(defconstant r_sizeof_flt-one-pole 9)
(defconstant i_sizeof_flt-one-pole 1)

(defun pass-one-pole_c (flt addr datar)
  (setf (aref datar addr) (make-short-float (smpflt-a0 flt)))
  (setf (aref datar (+ addr 3)) (make-short-float (smpflt-b1 flt))))

(defun dm-one-pole_c (addr datar)
  (make-smpflt :a0 (aref datar addr) :b1 (aref datar (+ addr 3)) :y1 (aref datar (+ addr 7))))

(defmacro <one-zero_c> (result-name filt arg) 
  (let ((flt (lc filt))
	(val (lc result-name))
	(a0 0)
	(a1 1)
	(x1 5))
    (format *c-file* "  ~A = (~A * ~A[~D]) + (~A[~D] * ~A[~D]); ~A[~D] = ~A;~A/* (one-zero ~(~A~) ~A) */~%"
	    val (lc arg)  flt a0  flt a1  flt x1  flt x1  (lc arg) 
	    cr_white_space filt arg))
  nil)

(defconstant r_sizeof_flt-one-zero 9)
(defconstant i_sizeof_flt-one-zero 1)

(defun pass-one-zero_c (flt addr datar)
  (setf (aref datar addr) (make-short-float (smpflt-a0 flt)))
  (setf (aref datar (+ addr 1)) (make-short-float (smpflt-a1 flt))))

(defun dm-one-zero_c (addr datar)
  (make-smpflt :a0 (aref datar addr) :a1 (aref datar (+ addr 1)) :x1 (aref datar (+ addr 5))))


(defmacro <two-pole_c> (result-name filt arg) 
  (let ((flt (lc filt))
	(val (lc result-name))
	(a0 0)
	(b1 3)
	(b2 4)
	(y1 7)
	(y2 8))
    (format *c-file* "  ~A = (~A[~D] * ~A) - (~A[~D] * ~A[~D]) - (~A[~D] * ~A[~D]);~A/* (two-pole ~(~A~) ~A) */~%  ~
                        ~A[~D] = ~A[~D];~%  ~
                        ~A[~D] = ~A;~%"
	    val  flt a0  (lc arg)  flt b1  flt y1  flt b2   flt y2  cr_white_space filt arg 
	    flt y2  flt y1  flt y1  val))
  nil)

(defconstant r_sizeof_flt-two-pole 9)
(defconstant i_sizeof_flt-two-pole 1)

(defun pass-two-pole_c (flt addr datar)
  (setf (aref datar addr) (make-short-float (smpflt-a0 flt)))
  (setf (aref datar (+ addr 3)) (make-short-float (smpflt-b1 flt)))
  (setf (aref datar (+ addr 4)) (make-short-float (smpflt-b2 flt))))

(defun dm-two-pole_c (addr datar)
  (make-smpflt :a0 (aref datar addr) :b1 (aref datar (+ addr 3)) :b2 (aref datar (+ addr 4))
	       :y1 (aref datar (+ addr 7)) :y2 (aref datar (+ addr 8))))


(defmacro <two-zero_c> (result-name filt arg) 
  (let ((flt (lc filt))
	(val (lc result-name))
	(a0 0)
	(a1 1)
	(a2 2)
	(x1 5)
	(x2 6))
    (format *c-file* "  ~A = (~A[~D] * ~A) + (~A[~D] * ~A[~D]) + (~A[~D] * ~A[~D]);~A/* (two-zero ~(~A~) ~A) */~%  ~
                        ~A[~D] = ~A[~D];~%  ~
                        ~A[~D] = ~A;~%"
	    val flt a0  (lc arg)  flt a1  flt x1  flt a2  flt x2  cr_white_space filt arg 
	    flt x2  flt x1  flt x1  (lc arg)))
  nil)

(defconstant r_sizeof_flt-two-zero 9)
(defconstant i_sizeof_flt-two-zero 1)

(defun pass-two-zero_c (flt addr datar)
  (setf (aref datar addr) (make-short-float (smpflt-a0 flt)))
  (setf (aref datar (+ addr 1)) (make-short-float (smpflt-a1 flt)))
  (setf (aref datar (+ addr 2)) (make-short-float (smpflt-a2 flt))))

(defun dm-two-zero_c (addr datar)
  (make-smpflt :a0 (aref datar addr) :a1 (aref datar (+ addr 1)) :a2 (aref datar (+ addr 2))
	       :x1 (aref datar (+ addr 5)) :x2 (aref datar (+ addr 6))))

;;; 0:a0 1:a1 2:a2 3:b1 4:b2 5:x1 6:x2 7:y1 8:y2
(defmacro <smpflt-a0_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-a0" :float 0 ,in-setf))
(defmacro <smpflt-a1_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-a1" :float 1 ,in-setf))
(defmacro <smpflt-a2_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-a2" :float 2 ,in-setf))
(defmacro <smpflt-b1_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-b1" :float 3 ,in-setf))
(defmacro <smpflt-b2_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-b2" :float 4 ,in-setf))
(defmacro <smpflt-x1_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-x1" :float 5 ,in-setf))
(defmacro <smpflt-x2_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-x2" :float 6 ,in-setf))
(defmacro <smpflt-y1_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-y1" :float 7 ,in-setf))
(defmacro <smpflt-y2_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "smpflt-y2" :float 8 ,in-setf))



;;; ---- FILTER ----
;;; 0:adr of datar 1:m 2:type 3:a hdr 3+siz:b hdr 3+siz+siz:c hdr 3+siz+siz+siz:d hdr
;;; 0:so in datar

(defconstant i_sizeof_flt (+ 3 (* 4 i_sizeof_aref_header)))
(defconstant r_sizeof_flt 1)

(defmacro <flt-m_c> (result-name f in-setf) `(package-c-address ,result-name ,f "flt-m" :int 1 ,in-setf))
(defmacro <flt-typ_c> (result-name f in-setf) `(package-c-address ,result-name ,f "flt-m" :int 2 ,in-setf))
(defmacro <flt-so_c> (result-name f in-setf) `(package-c-address ,result-name ,f "flt-so" :float 0 ,in-setf 0))
(defmacro <flt-a_c> (result-name f in-setf) `(package-c-address ,result-name ,f "flt-a" :float** 3 ,in-setf))
(defmacro <flt-b_c> (result-name f in-setf) `(package-c-address ,result-name ,f "flt-a" :float** ,(+ 3 i_sizeof_aref_header) ,in-setf))
(defmacro <flt-c_c> (result-name f in-setf) `(package-c-address ,result-name ,f "flt-a" :float** ,(+ 3 (* 2 i_sizeof_aref_header)) ,in-setf))
(defmacro <flt-d_c> (result-name f in-setf) `(package-c-address ,result-name ,f "flt-a" :float** ,(+ 3 (* 3 i_sizeof_aref_header)) ,in-setf))

(defun pass-flt_c (flt i r datai datar info)
  (setf (aref datai i) r)
  (setf (aref datar r) (make-short-float (flt-so flt)))
  (setf (aref datai (+ i 1)) (flt-m flt))
  (setf (aref datai (+ i 2)) (flt-typ flt))
  (pass-array_c (flt-a flt) (+ i 3) datai datar info)
  (pass-array_c (flt-b flt) (+ i 3 i_sizeof_aref_header) datai datar info)
  (when (flt-c flt) (pass-array_c (flt-c flt) (+ i 3 (* 2 i_sizeof_aref_header)) datai datar info))
  (pass-array_c (flt-d flt) (+ i 3 (* 3 i_sizeof_aref_header)) datai datar info))

(defun dm-flt_c (adr datai datar)
  (make-flt :m (aref datai (+ adr 1))
	    :so (aref datar (aref datai adr))
	    :typ (aref datai (+ adr 2))
	    :a (make-array (aref datai (+ adr 3 aref_size))
			   :element-type (type-of (aref datar 0))
			   :displaced-to datar 
			   :displaced-index-offset (aref datai (+ adr 3 aref_block)))
	    :b (make-array (aref datai (+ adr 3 i_sizeof_aref_header aref_size))
			   :element-type (type-of (aref datar 0))
			   :displaced-to datar 
			   :displaced-index-offset (aref datai (+ adr 3 i_sizeof_aref_header aref_block)))
	    :c (make-array (aref datai (+ adr 3 (* 2 i_sizeof_aref_header) aref_size))
			   :element-type (type-of (aref datar 0))
			   :displaced-to datar 
			   :displaced-index-offset (aref datai (+ adr 3 (* 2 i_sizeof_aref_header) aref_block)))
	    :d (make-array (aref datai (+ adr 3 (* 3 i_sizeof_aref_header) aref_size))
			   :element-type (type-of (aref datar 0))
			   :displaced-to datar 
			   :displaced-index-offset (aref datai (+ adr 3 (* 3 i_sizeof_aref_header) aref_block)))))

(defmacro <filter_c> (result-name s in-argl) 
  (let ((lc-s (lc s)))
    (format *c-file* "  switch (~A[2]) {~A/* (filter ~(~A~)) */~%" lc-s cr_white_space s)
    (format *c-file* "    case 0: ~A=c_direct_flt(~A[1],~A~A,(float *)(_datar_+~A[3]),~%                      ~
                               (float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D])); break;~%"
	    (lc result-name) lc-s (if (int-p in-argl) "(float)" "") (lc in-argl)
	    lc-s lc-s (+ 3 i_sizeof_aref_header) lc-s (+ 3 (* 3 i_sizeof_aref_header)))
    (format *c-file* "    case 1: ~A=c_lattice_flt(~A[1],~A~A,(float *)(_datar_+~A[3]),~%                      ~
                               (float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D])); break;~%"
	    (lc result-name) lc-s (if (int-p in-argl) "(float)" "") (lc in-argl)
	    lc-s lc-s (+ 3 i_sizeof_aref_header) lc-s (+ 3 (* 3 i_sizeof_aref_header)))
    (format *c-file* "    case 2: ~A=c_ladder_flt(~A[1],~A~A,_datar_[~A[0]],~%                      ~
                               (float *)(_datar_+~A[3]),(float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]),~%                      ~
                               (float *)(_datar_+~A[~D])); break;}~%"
	    (lc result-name) lc-s (if (int-p in-argl) "(float)" "") (lc in-argl) lc-s
	    lc-s lc-s (+ 3 i_sizeof_aref_header) lc-s (+ 3 (* 2 i_sizeof_aref_header)) lc-s (+ 3 (* 3 i_sizeof_aref_header))))
  nil)

(defmacro <direct-filter_c> (result-name s in-argl)
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = c_direct_flt(~A[1],~A~A,(float *)(_datar_+~A[3]),(float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]));~%"
	    (lc result-name) lc-s (if (int-p in-argl) "(float)" "") (lc in-argl)
	    lc-s lc-s (+ 3 i_sizeof_aref_header) lc-s (+ 3 (* 3 i_sizeof_aref_header))))
  nil)

(defmacro <lattice-filter_c> (result-name s in-argl) 
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = c_lattice_flt(~A[1],~A~A,(float *)(_datar_+~A[3]),(float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]));~%"
	    (lc result-name) lc-s (if (int-p in-argl) "(float)" "") (lc in-argl)
	    lc-s lc-s (+ 3 i_sizeof_aref_header) lc-s (+ 3 (* 3 i_sizeof_aref_header))))
  nil)

(defmacro <ladder-filter_c> (result-name s in-argl) 
  (let ((lc-s (lc s)))
    (format *c-file* "  ~A = c_ladder_flt(~A[1],~A~A,_datar[~A[0]],~%                  ~
                               (float *)(_datar_+~A[3]),(float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]));~%"
	    (lc result-name) lc-s (if (int-p in-argl) "(float)" "") (lc in-argl) lc-s
	    lc-s lc-s (+ 3 i_sizeof_aref_header) lc-s (+ 3 (* 2 i_sizeof_aref_header)) lc-s (+ 3 (* 3 i_sizeof_aref_header))))
  nil)

(defmacro <check_c> (ref &optional (id 2)) 
  (declare (ignore ref id))
  ;; I don't think I need to worry here -- this is for explicit memory clearing on the 56k and fixing up external delays
  nil)



;;; ---- DELAY ---- 
;;; 0:loc 1:float array header

(defmacro <delay_c> (result-name del arg) 
  (let ((dnam (lc del))
	(val (lc result-name)))
    (format *c-file* "  ~A = ~A_datar_[~A[0]+~A[~D]];~A/* (delay ~(~A~) ~A) */~%  ~
                        _datar_[~A[0]+~A[~D]] = ~A~A;~%  ~
                        ~A[0]++;~%  ~
                        if (~A[~D] <= ~A[0]) ~A[0] = 0;~%"
	    val (if (int-p result-name) "(int)" "") dnam dnam (+ 1 aref_block) cr_white_space del arg
	    dnam dnam (+ 1 aref_block) (if (int-p arg) "(float)" "") (lc arg)
	    dnam
	    dnam (+ 1 aref_size) dnam dnam))
  nil)

(defmacro <tap_c> (result-name del &optional offset) 
  (let ((dnam (lc del))
	(val (lc result-name))
	(lco (lc offset))
	(rlo (if (rl-p offset) "(int)" ""))
	(rlv (if (int-p result-name) "(int)" "")))
    (if (or (not offset)
	    (and (numberp offset) (zerop offset)))
	(format *c-file* "  ~A = ~A_datar_[~A[0]+~A[~D]];~A/* (tap ~(~A~)) */~%"
		val (if (int-p result-name) "(int)" "") dnam dnam (+ 1 aref_block) cr_white_space del)
      (format *c-file* "  if (~A[0] >= ~A~A) {~A = ~A_datar_[~A[~D]+((~A[0]-~A~A)%(~A[~D]))];}~A/* (tap ~(~A~) ~A) */~%~
                          else {~A = ~A_datar_[~A[~D]+~A[~D]+((~A[0]-~A~A)%(~A[~D]))];}~%"
		;; we can't trust C's % operator because it will return results like -3%4=-3.
	      dnam rlo lco val rlv dnam (+ 1 aref_block) dnam rlo lco dnam (+ 1 aref_size) cr_white_space del offset
	      val rlv dnam (+ 1 aref_block) dnam (+ 1 aref_size) dnam rlo lco dnam (+ 1 aref_size))))
  nil)

(defconstant r_sizeof_dly 0)
(defconstant i_sizeof_dly (+ 1 i_sizeof_aref_header))

(defun pass-dly_c (dly addr datai datar info)
  (setf (aref datai addr) (dly-loc dly))
  (pass-array_c dly (+ addr 1) datai datar info))

(defun dm-dly_c (addr datai datar)
  (let ((len (aref datai (+ addr aref_size 1))))
    (make-dly :loc (aref datai addr) 
	      :size len 
	      :pline (if (< len *clm-array-print-length*)
			 (make-array len 
				     :displaced-to datar 
				     :element-type (type-of (aref datar 0))
				     :displaced-index-offset (aref datai (+ addr 1 aref_block)))
		       (make-array 2 :initial-contents (list 0 (1- len)))))))

(defmacro <dly-loc_c> (result-name dly in-setf) `(package-c-address ,result-name ,dly "dly-loc" :int 0 ,in-setf))
(defmacro <dly-size_c> (result-name dly in-setf) `(package-c-address ,result-name ,dly "dly-size" :int ,(+ 1 aref_size) ,in-setf))
(defmacro <dly-pline_c> (result-name dly in-setf) `(package-c-address ,result-name ,dly "dly-pline" :float** 1 ,in-setf))



;;; ---- ZDELAY ----
;;;
;;; 0:phase 1:dly {0:loc 1:aref}

(defconstant i_sizeof_zdly (+ 1 i_sizeof_dly))
(defconstant r_sizeof_zdly 0)

(defmacro <zdly-phase_c> (result-name zd in-setf) `(package-c-address ,result-name ,zd "zdly-phase" :int 0 ,in-setf))
(defmacro <zdly-del_c> (result-name zd in-setf) `(package-c-address ,result-name ,zd "zdly-del" :int* 1 ,in-setf))

(defun pass-zdly_c (zd i datai datar info)
  (setf (aref datai i) (round (zdly-phase zd)))
  (pass-dly_c (zdly-del zd) (+ i 1) datai datar info))

(defun dm-zdly_c (adr datai datar)
  (make-zdly :phase (aref datai adr)
	     :del (dm-dly_c (+ adr 1) datai datar)))

(defmacro <zdelay_c> (result-name del in-argl &optional pm) 
  ;; OPT: inline c_table_interp
  (let ((lz (lc del)))
    (format *c-file* "  ~A = c_table_interp((float *)(_datar_+~A[~D]),(((float)~A[0])~A~A~A),~A[~D]);~A/* (zdelay ~(~A~) ~A ~A) */~%  ~
                        ~A[0]++;~%  ~
                        if (~A[~D] <= ~A[0]) ~A[0] = 0;~%  ~
                        _datar_[~A[1]+~A[~D]] = ~A~A;~%  ~
                        ~A[1]++;~%  ~
                        if (~A[~D] <= ~A[1]) ~A[1] = 0;~%"
	    (lc result-name) lz (+ 2 aref_block) lz (if pm " - " "") (if (int-p pm) "(float)" "") (if pm (lc pm) "")
	    lz (+ 2 aref_size) cr_white_space del in-argl pm
	    lz 
	    lz (+ 2 aref_size) lz lz lz lz (+ 2 aref_block) (if (int-p in-argl) "(float)" "") (lc in-argl)
	    lz
	    lz (+ 2 aref_size) lz lz))
  nil)

(defmacro <ztap_c> (result-name del &optional pm)
  ;; OPT: inline c_table_interp
  (let ((lz (lc del)))
    (format *c-file* "  ~A = c_table_interp((float *)(_datar_+~A[~D]),(((float)~A[0])~A~A~A),~A[~D]);~A/* (ztap ~(~A~) ~A) */~%"
	    (lc result-name) lz (+ 2 aref_block) lz (if pm " - " "") (if (int-p pm) "(float)" "") (if pm (lc pm) "")
	    lz (+ 2 aref_size) cr_white_space del (or pm 0.0)))
  nil)



;;; ---- COMB ----
;;; 0:datar adr [1:delay unit]= 1:loc 2:line header
;;; 0:scaler

(defconstant i_sizeof_cmbflt (+ 1 i_sizeof_dly))
(defconstant r_sizeof_cmbflt (+ 1 r_sizeof_dly))

(defmacro <cmbflt-scaler_c> (result-name cmb in-setf) `(package-c-address ,result-name ,cmb "cmbflt-scaler" :float 0 ,in-setf 0))
(defmacro <cmbflt-dly-unit_c> (result-name cmb in-setf) `(package-c-address ,result-name ,cmb "cmbflt-dly-unit" :int* 1 ,in-setf))

(defun pass-cmbflt_c (cmb i r datai datar info)
  (setf (aref datai i) r)
  (setf (aref datar r) (make-short-float (cmbflt-scaler cmb)))
  (pass-dly_c (cmbflt-dly-unit cmb) (+ i 1) datai datar info))

(defun dm-cmbflt_c (adr datai datar)
  (make-cmbflt :scaler (aref datar (aref datai adr))
	       :dly-unit (dm-dly_c (+ adr 1) datai datar)))

(defmacro <comb_c> (result-name cmb in-arg) 
  (let ((cnam (lc cmb))
	(val (lc result-name)))
    (format *c-file* "  ~A = _datar_[~A[1]+~A[~D]];~A/* (comb ~(~A~) ~A) */~%  ~
                        _datar_[~A[1]+~A[~D]] = (~A~A + (~A * _datar_[~A[0]]));~%  ~
                        ~A[1]++;~%  ~
                        if (~A[~D] <= ~A[1]) ~A[1] = 0;~%"
	    val cnam cnam (+ 2 aref_block) cr_white_space cmb in-arg
	    cnam cnam (+ 2 aref_block) (if (int-p in-arg) "(float)" "") (lc in-arg) (lc result-name) cnam
	    cnam
	    cnam (+ 2 aref_size) cnam cnam))
  nil)



;;; ---- ALL-PASS ----
;;; 0:datar adr [1:delay unit]= 1:loc 2:line header
;;; 0:feedback-scaler 1:feedforward-scaler

(defconstant i_sizeof_allpassflt (+ 1 i_sizeof_dly))
(defconstant r_sizeof_allpassflt (+ 2 r_sizeof_dly))

(defmacro <allpassflt-feedforward_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "allpass-feedforward" :float 1 ,in-setf 0))
(defmacro <allpassflt-feedback_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "allpass-feedback" :float 0 ,in-setf 0))
(defmacro <allpassflt-dly-unit_c> (result-name flt in-setf) `(package-c-address ,result-name ,flt "allpassflt-dly-unit" :int* 1 ,in-setf))

(defun pass-allpassflt_c (flt i r datai datar info)
  (setf (aref datai i) r)
  (setf (aref datar r) (make-short-float (allpassflt-feedback flt)))
  (setf (aref datar (+ r 1)) (make-short-float (allpassflt-feedforward flt)))
  (pass-dly_c (allpassflt-dly-unit flt) (+ i 1) datai datar info))

(defun dm-allpassflt_c (adr datai datar)
  (make-allpassflt :feedback (aref datar (aref datai adr))
		   :feedforward (aref datar (+ (aref datai adr) 1))
		   :dly-unit (dm-dly_c (+ adr 1) datai datar)))

(defmacro <all-pass_c> (result-name allpassf in-arg) 
  (let ((cnam (lc allpassf))
	(val (lc result-name)))
    (format *c-file* "  { float _f_;~A/* (all-pass ~(~A~) ~A) */~%    ~
                          ~A = _datar_[~A[1]+~A[~D]];~%    ~
                          _f_ = (~A~A + (~A * _datar_[~A[0]]));~%    ~
                          _datar_[~A[1]+~A[~D]] = _f_;~%    ~
                          ~A += (_datar_[~A[0]+1] * _f_);~%    ~
                          ~A[1]++;~%    ~
                          if (~A[~D] <= ~A[1]) ~A[1] = 0;}~%"
	    cr_white_space allpassf in-arg
	    val cnam cnam (+ 2 aref_block) 
	    ;; result-name = current value of delay line
	    (if (int-p in-arg) "(float)" "") (lc in-arg) val cnam
	    ;; _f_ = in-arg + feedback*result-name
	    cnam cnam (+ 2 aref_block) 
	    ;; delay input <= _f_
	    val cnam 
	    ;; output = delay output + feedforward*delay input
	    cnam
	    cnam (+ 2 aref_size) cnam cnam))
  nil)




;;; ---- ENV ----
;;;
;;; 0:datar adr 1:type 2:index 3:top 4:pass 5:end 6:restart-pass 7:data aref header
;;; 0:val 1:rate 2:base 3:offset 4:scaler 5:power 6:init-y 7:power
;;; the order here is reflected in cmus.c

(defconstant i_sizeof_env (+ 7 i_sizeof_aref_header))
(defconstant r_sizeof_env 8)

(defun pass-env_c (e i r datai datar info)
  (setf (aref datai i) r)
  (setf (aref datar r) (make-short-float (envelope-current-value e)))
  (setf (aref datar (+ r 6)) (make-short-float (envelope-current-value e)))
  (setf (aref datar (+ r 1)) (make-short-float (envelope-rate e)))
  (if (eq (envelope-type e) :seg) 
      (setf (aref datar (+ r 2)) (make-short-float (or (envelope-base e) 1.0)))
    (setf (aref datar (+ r 2)) (make-short-float (log (float (envelope-base e))))))
  (setf (aref datar (+ r 3)) (make-short-float (or (envelope-offset e) 0.0)))
  (setf (aref datar (+ r 4)) (make-short-float (or (envelope-scaler e) 0.0)))
  (setf (aref datar (+ r 5)) (make-short-float (or (envelope-power e) 0.0)))
  (setf (aref datar (+ r 7)) (make-short-float (or (envelope-power e) 0.0)))
  (setf (aref datai (+ i 1)) (if (eq (envelope-type e) :seg) 0 1))
  (setf (aref datai (+ i 2)) 0)
  (setf (aref datai (+ i 3)) (length (envelope-data e)))
  (setf (aref datai (+ i 4)) (envelope-pass e))
  (setf (aref datai (+ i 5)) (envelope-end e))
  (pass-array_c e (+ i 7) datai datar info)
  (setf (aref datai (+ i 6)) (envelope-pass e)))

(defun dm-env_c (adr datai datar)
  (make-envelope :current-value (aref datar (aref datai adr))
		 :rate (aref datar (+ (aref datai adr) 1))
		 :base (aref datar (+ (aref datai adr) 2))
		 :offset (aref datar (+ (aref datai adr) 3))
		 :scaler (aref datar (+ (aref datai adr) 4))
		 :power (aref datar (+ (aref datai adr) 5))
		 :type (aref datai (+ adr 1))
		 :pass (aref datai (+ adr 4))
		 :end (aref datai (+ adr 5))
		 :data (make-array (aref datai (+ adr 7 aref_size))
				   :displaced-to datar 
				   :element-type (type-of (aref datar 0))
				   :displaced-index-offset (aref datai (+ adr 7 aref_block)))))

(defmacro <env_c> (result-name e)
  ;; this is a lot faster than the previous function call (where this code was not inline)
  (let ((lce (lc e))
	(lcr (lc result-name)))
    (format *c-file* "  {int _loc_; int *_ei_; float *_data_,*_ef_;~A/* (env ~(~A~)) */~%    ~
                         _ef_ = (float *)(_datar_+~A[0]); _ei_ = (int *)(~A+1);~%    ~
                         _data_ = (float *)(_datar_+~A[~D]); _loc_ = _ei_[1];~%    ~
                         ~A = _ef_[0];~%    ~
                         if ((_loc_ < _ei_[2]) && (_ei_[3] >= (int)_data_[_loc_])) {~%      ~
                           _ef_[1] = _data_[_loc_+1]; _ei_[1] += 2; }~%    ~
                         _ei_[3] += 1;~%    ~
                         if (_ei_[3]<=_ei_[4])  {~%      ~
                           if (_ei_[0] == 0) {~%        ~
                             if (_ef_[2] != 0.0) _ef_[0] += _ef_[1]; else _ef_[0] = _ef_[1];}~%      ~
                           else if (_ef_[1] != 0.0) {~%        ~
                             _ef_[5] += _ef_[1];~%        ~
                             _ef_[0] = (_ef_[3] + (_ef_[4] * (~Aexp(_ef_[2]*_ef_[5]) - 1.0)));}}}~%"
	    cr_white_space e
	    lce lce lce (+ 7 aref_block)
	    lcr 
	    #+(and akcl I386) "" #-(and akcl I386) "jv_"))
  nil)

(defmacro <restart-env_c> (result-name e) 
  (let ((lce (lc e)))
    (format *c-file* "  ~A = c_env_restart((float *)(_datar_+~A[0]),(int *)(~A+1),(float *)(_datar_+~A[~D]));~A/* (env-restart ~(~A~)) */~%"
	    (lc result-name) 
	    lce lce lce (+ 7 aref_block) cr_white_space e))
  nil)




;;; ---- OUTA --- 
;;;
;;; the IO info passed down and reflected back into lisp at the end includes:
;;; the address of the c array holding this channel's current frame
;;; io-beg, io-end, io-data-end, the file id, size, chans
;;; overall loops follow the output buffers boundaries, so no immediate checks are needed

;;; io* struct in datai:
;;; 0:clm file descriptor  1:chans  2:size  3:beg  4:end  5:bufsiz  6:data-start  7:data-end  8:open-index
;;; 9:dat-a  10:dat-b  11:dat-c  12:dat-d  13:dir  14:loc  15:hdr-end  16:incr-flag

;;; if these change, update cmus.h! 
;;;
;;; the path from an IO structure open in a CLM instrument to a C file descriptor ready for
;;; clm_read/write/seek here is circuitous.  The problems include the fact that the same
;;; file can be open at the same time by many unit generators, so if one causes the in-core
;;; frame to move, the others need to know instantly, outa (with its implicit
;;; file) can be called many times in a sample on the same output (adding as it goes),
;;; it is the value of the lisp variable, not its name that we care about (so Run can't
;;; help us much), and so on.  So, concentrating here on OUTn alone, Run.lisp via
;;; the dsp-function 'outa calls fpackage-op which notices *clm-language* is :c and
;;; puts the (otherwise ignored) result of outn (call it LRES) on typed-user-sig (normally the
;;; list of user-declared locals from let and let*), with the type 'io.  This
;;; causes find-a-home-for_c (above) to give it one word of integer (datai) space,
;;; from Lisp's point of view (call it LADR), and it is declared in C as int* (call
;;; it CRES).  In lisp, pretend we have (outa LPASS LVAL LSTR), 
;;; where LSTR defaults to *current-output-file*, we add the pre-array (pip) statement:
;;;
;;;   (when LSTR (let ((I (io-open-index LSTR))) ;this ensures all ugs will be tracking the same file descriptor
;;;     (when (zerop (aref *clm-io-open* I))     ;only allocate the file space once
;;;       (setf (aref *clm-io-open* I) *clm-datai-len*) ;address of file descriptor (Lisp side)
;;;       (incf *clm-datai-len* i_sizeof_io))))  ;make sure we'll allocate the space when *clm-datai* is created.
;;;
;;; Then, after array allocation time (pup):
;;;
;;;   (setf (aref *clm-datai* LADR)              ;this is LRES's address from C point of view
;;;     (if LSTR (aref *clm-io-open* (io-open-index LSTR)) -1))
;;;
;;; So, CRES can tell whether the file is open, and if so can find the (datai) address of its descriptor.
;;; On the C side, we generate the int *CRES declaration, then in *io-inits* add the statement,
;;;
;;;   if (_datai_[LADR] != -1) { CRES = (int *)(_datai_+_datai_[LADR]);
;;;
;;; this makes CRES a pointer to the file descriptor. The next line:
;;;
;;;   if ((_beg_ < CRES[beg]) || (_beg_ >= CRES[end]) clm_file_reset(_beg_,CRES);
;;;
;;; makes sure we can write to the in-core frame.  We then initialize the location:
;;;
;;;   CRES[loc] = (_beg_ - CRES[beg]); } else CRES == NULL;
;;;
;;; When the outa call is encountered, we generate:
;;;
;;;   if (CRES) { (((int *)(CRES[dat_A+chan]))[CRES[loc]) += (int)(_sndfix_ * CVAL); CRES[incr]=1;}
;;;
;;; The first statement finds the address of the (c) buffer tracking our in-core 
;;; view of the given channel, gets our current offset into that array (loc) and adds
;;; in the current (c) notion of LVAL, turning it into a 16-bit integer. Then it sets
;;; the "my location needs to be incremented" flag (which we have to make sure happens
;;; only once per sample, no matter how many times that file is accessed).  Then at
;;; loop bottom, we add:
;;;
;;;   if (CRES) {if (CRES[incr] != 0) {CRES[loc]++; CRES[incr]=0; if (CRES[loc] >= CRES[bufsiz]) clm_file_reset((int)CPASS,CRES);}}
;;;
;;; This increments the location once, and checks to see if we've hit the end of the in-core
;;; frame.  CPASS is the C view on the pass-counter (LPASS).  If it is not the same as the
;;; actual loop counter, we also do a bounds check before adding in CVAL, and we use the
;;; actual value of CPASS, not depending on CRES[loc] (the output location could in theory be
;;; jumping around in some arbitrary way).  Then, after the loop is done, on *io-finals*:
;;; 
;;;   if (CRES) {if (CRES[data_end] < (CPASS - CRES[beg])) CRES[data_end] = (CPASS - CRES[beg]);}
;;;
;;; This is for lisp's benefit -- it needs to know how much data is left in the buffer
;;; when the instrument returns to lisp.  The actual (lisp-side) loading and unloading
;;; of the file descriptor data takes place in initialize-io-records_c and finalize-io-records_c.
;;; These are wrapped around the C function call in definstrument.
;;;
;;; 
(defconstant io*_fd 0)
(defconstant io*_chans 1)
(defconstant io*_size 2)
(defconstant io*_beg 3)
(defconstant io*_end 4)
(defconstant io*_bufsiz 5)
(defconstant io*_data_start 6)
(defconstant io*_data_end 7)
(defconstant io*_open_index 8)
(defconstant io*_dat_A 9)		;channels should be in order (we use chan+dat_A as offset)
(defconstant io*_dat_B 10)
(defconstant io*_dat_C 11)
(defconstant io*_dat_D 12)
(defconstant io*_dir 13)
(defconstant io*_loc 14)
(defconstant io*_hdr_end 15)
(defconstant io*_incr 16)
					;(defconstant i_sizeof_io* 17) -- defined at top for store's benefit

(defmacro <io-fil_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-fil" :int ,io*_fd ,in-setf))
(defmacro <io-siz_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-siz" :int ,io*_size ,in-setf))
(defmacro <io-beg_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-beg" :int ,io*_beg ,in-setf))
(defmacro <io-end_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-end" :int ,io*_end ,in-setf))
(defmacro <io-bufsiz_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-bufsiz" :int ,io*_bufsiz ,in-setf))
(defmacro <io-data-start_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-data-start" :int ,io*_data_start ,in-setf))
(defmacro <io-data-end_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-data-end" :int ,io*_data_end ,in-setf))
(defmacro <io-open-index_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-open-index" :int ,io*_open_index ,in-setf))
(defmacro <io-dat-A_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-dat-A" :int* ,io*_dat_A ,in-setf))
(defmacro <io-dat-B_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-dat-B" :int* ,io*_dat_B ,in-setf))
(defmacro <io-dat-C_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-dat-C" :int* ,io*_dat_C ,in-setf))
(defmacro <io-dat-D_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-dat-D" :int* ,io*_dat_D ,in-setf))
(defmacro <io-dir_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-dir" :int ,io*_dir ,in-setf))
(defmacro <io-hdr-end_c> (result-name io in-setf) `(package-c-address ,result-name ,io "io-dir" :int ,io*_hdr_end ,in-setf))
;;; io-nam not implemented yet

(defun pass-io_c (fil adr datai)
  (if (io-fil fil)
      (progn
	(setf (aref datai (+ adr io*_fd)) (io-fil fil))
	(setf (aref datai (+ adr io*_chans)) (clm-get-channels fil))
	(setf (aref datai (+ adr io*_size)) (io-siz fil))
	(setf (aref datai (+ adr io*_beg)) (io-beg fil))
	(setf (aref datai (+ adr io*_end)) (io-end fil))
	(setf (aref datai (+ adr io*_bufsiz)) (io-bufsiz fil))
	(setf (aref datai (+ adr io*_data_start)) (io-data-start fil))
	(setf (aref datai (+ adr io*_data_end)) (io-data-end fil))
	(setf (aref datai (+ adr io*_open_index)) (io-open-index fil))
	(setf (aref datai (+ adr io*_dir)) (io-dir fil))
	(setf (aref datai (+ adr io*_hdr_end)) (io-hdr-end fil))
	#-mcl (progn
		(setf (aref datai (+ adr io*_dat_A)) (io-dat-A fil))
		(setf (aref datai (+ adr io*_dat_B)) (or (io-dat-B fil) 0))
		(setf (aref datai (+ adr io*_dat_C)) (or (io-dat-C fil) 0))
		(setf (aref datai (+ adr io*_dat_D)) (or (io-dat-D fil) 0)))
	#+mcl (progn
		(ccl:with-dereferenced-handles 
		 ((p (io-dat-A fil))) (setf (aref datai (+ adr io*_dat_A)) (ccl:%ptr-to-int p)))
		(if (io-dat-B fil) (ccl:with-dereferenced-handles 
				    ((p (io-dat-B fil))) (setf (aref datai (+ adr io*_dat_B)) (ccl:%ptr-to-int p))))
		(if (io-dat-C fil) (ccl:with-dereferenced-handles 
				    ((p (io-dat-C fil))) (setf (aref datai (+ adr io*_dat_C)) (ccl:%ptr-to-int p))))
		(if (io-dat-D fil) (ccl:with-dereferenced-handles 
				    ((p (io-dat-D fil))) (setf (aref datai (+ adr io*_dat_D)) (ccl:%ptr-to-int p)))))
	)
    (setf (aref datai (+ adr io*_fd)) -1)))

(defun dm-io_c (adr datai)
  (if (/= (aref datai (+ adr io*_fd)) -1)
      (make-io :fil (aref datai (+ adr io*_fd))
	       :siz (aref datai (+ adr io*_size))
	       :beg (aref datai (+ adr io*_beg))
	       :end (aref datai (+ adr io*_end))
	       :bufsiz (aref datai (+ adr io*_bufsiz))
	       :data-start (aref datai (+ adr io*_data_start))
	       :data-end (aref datai (+ adr io*_data_end))
	       :dir (aref datai (+ adr io*_dir))
	       :hdr-end (aref datai (+ adr io*_hdr_end))
	       :open-index (aref datai (+ adr io*_open_index))
	       :dat-A (aref datai (+ adr io*_dat_A))
	       :dat-B (and (plusp (aref datai (+ adr io*_dat_B))) (aref datai (+ adr io*_dat_B)))
	       :dat-C (and (plusp (aref datai (+ adr io*_dat_C))) (aref datai (+ adr io*_dat_C)))
	       :dat-D (and (plusp (aref datai (+ adr io*_dat_D))) (aref datai (+ adr io*_dat_D))))))

(defun dm-io_c* (adr datai) (if (/= (aref datai adr) -1) (dm-io_c (aref datai adr) datai)))

(defun initialize-io-records_c (io-open datai) 
  (loop for i from 0 below *available-io-channels* do
    (let ((io (aref io-open i)))
      (if (plusp io)
	  (pass-io_c (aref *clm-current-open-files* i) io datai)))))

(defun finalize-io-records_c (io-open datai)  
  (loop for i from 0 below *available-io-channels* do
    (let ((fp (aref io-open i)))
      (if (plusp fp)
	  (let ((file (aref *clm-current-open-files* i)))
	    (setf (io-data-end file) (aref datai (+ fp io*_data_end)))
	    (setf (io-data-start file) (aref datai (+ fp io*_data_start)))
	    ;(setf (io-siz file) (+ (aref datai (+ fp io*_beg)) (aref datai (+ fp io*_data_end))))
	    (setf (io-siz file) (aref datai (+ fp io*_size)))
	    ;(print (format nil "io-siz: ~D start: ~D  end: ~D " (io-siz file) (io-data-start file) (io-data-end file)))
	    (setf (io-beg file) (aref datai (+ fp io*_beg)))
	    (setf (io-end file) (aref datai (+ fp io*_end)))
	    )))))

;;; io-end is bufsiz-1+io-beg, so if checking bufsiz use >=, else >

(defmacro <outn_c> (result-name pass val chan &optional o-stream)
  (let ((lcr (lc result-name))
	(ioadr (fourth (gethash result-name vars)))
	(str (if (null o-stream) '*current-output-file* o-stream)))
    ;; if str is a temp sig, we're in big trouble
    (push `(setf (aref *clm-datai* ,ioadr) (if ,str (aref *clm-io-open* (io-open-index ,str)) -1)) pup)
    (push (format nil "if (_datai_[~D] != -1) {~%    ~A = (int *)(_datai_+_datai_[~D]);~%" ioadr lcr ioadr) *io-inits*)
    (push (format nil "  if ((_beg_ < ~A[~D]) || (_beg_ > ~A[~D])) clm_file_reset(_beg_,~A);~%"
	      lcr io*_beg lcr io*_end lcr) *io-inits*)
    (push (format nil "  ~A[~D] = (_beg_ - ~A[~D]);}~%  else ~A = NULL;~%" lcr io*_loc lcr io*_beg lcr) *io-inits*)

    (push (format nil "if (~A) {if (~A[~D] < (~A - ~A[~D])) ~A[~D] = (~A - ~A[~D]);}~%" 
		  lcr lcr io*_data_end (lc pass) lcr io*_beg lcr io*_data_end (lc pass) lcr io*_beg) *io-finals*)
    
    (push `(when ,str
	     (let ((cur-index (io-open-index ,str)))
	       (when (zerop (aref *clm-io-open* cur-index)) 
		 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
		 (incf *clm-datai-len* ,i_sizeof_io*))))
	  pip)
    ;; if pass /= loop-var, calculate actual output loc and so on.
    (if (eq pass loop-var)
	(progn
	  (push (format nil "if (~A) {if (~A[~D] != 0) {~A[~D]++; ~A[~D]=0; if (~A[~D] >= ~A[~D]) clm_full_file_reset(~A~A,~A);}}~%" 
			lcr lcr io*_incr lcr io*_loc lcr io*_incr lcr io*_loc lcr io*_bufsiz 
			(if (rl-p pass) "(int)" "") (lc pass) lcr) *io-incrs*)
	  (format *c-file* "  if (~A) {(((int *)(~A[~D]))[~A[~D]]) += (int)(_sndfix_ * ~A); ~A[~D]=1;} ~A/* (out~A ~(~A~) ~A~A~A) */~%" 
		  lcr lcr (+ io*_dat_A chan) lcr io*_loc (lc val) lcr io*_incr
		  cr_white_space (if (= chan 0) "a" (if (= chan 1) "b" (if (= chan 2) "c" "d")))
		  pass val (if o-stream " " "") (or o-stream "")))
	(format *c-file* "  if (~A) {~%    ~
                              if ((~A < ~A[~D]) || (~A > ~A[~D])) clm_full_file_reset(~A~A,~A);~%    ~
                              (((int *)(~A[~D]))[~A - ~A[~D]]) += (int)(_sndfix_ * ~A);}~A/* (out~A ~(~A~) ~A~A~A) */~%" 
	      lcr (lc pass) lcr io*_beg (lc pass) lcr io*_end 
	      (if (rl-p pass) "(int)" "") (lc pass) lcr
	      lcr (+ io*_dat_A chan) (lc pass) lcr io*_beg
	      (lc val) cr_white_space (if (= chan 0) "a" (if (= chan 1) "b" (if (= chan 2) "c" "d")))
	      pass val (if o-stream " " "") (or o-stream ""))))
  nil)

(defmacro <outa_c> (result-name pass val &optional o-stream) 
  `(<outn_c> ,result-name ,pass ,val 0 ,o-stream))

(defmacro <outb_c> (result-name pass val &optional o-stream) 
  `(<outn_c> ,result-name ,pass ,val 1 ,o-stream))

(defmacro <outc_c> (result-name pass val &optional o-stream) 
  `(<outn_c> ,result-name ,pass ,val 2 ,o-stream))

(defmacro <outd_c> (result-name pass val &optional o-stream)
  `(<outn_c> ,result-name ,pass ,val 3 ,o-stream))



;;; ---- LOCSIG ----
;;; 
;;; 0:datar adr 1:out-index 2:rev-index
;;; 0:ascl 1:bscl 2:cscl 3:dscl 4:rscl 5:deg 6:dis 7:pc-rev

(defconstant i_sizeof_locs 3)
(defconstant r_sizeof_locs 8)

(defmacro package-loc-address (result-name s accessor type addr in-setf &optional de-ref)
  (pushnew result-name hidden-scalers)
  `(package-c-address ,result-name ,s ,accessor ,type ,addr ,in-setf ,de-ref t))

(defmacro <locs-ascl_c> (result-name io in-setf) `(package-loc-address ,result-name ,io "locs-ascl" :float 0 ,in-setf 0))
(defmacro <locs-bscl_c> (result-name io in-setf) `(package-loc-address ,result-name ,io "locs-bscl" :float 1 ,in-setf 0))
(defmacro <locs-cscl_c> (result-name io in-setf) `(package-loc-address ,result-name ,io "locs-cscl" :float 2 ,in-setf 0))
(defmacro <locs-dscl_c> (result-name io in-setf) `(package-loc-address ,result-name ,io "locs-dscl" :float 3 ,in-setf 0))
(defmacro <locs-rscl_c> (result-name io in-setf) `(package-loc-address ,result-name ,io "locs-rscl" :float 4 ,in-setf 0))
(defmacro <locs-deg_c> (result-name io in-setf) `(package-c-address ,result-name ,io "locs-deg" :float 5 ,in-setf 0))
(defmacro <locs-dis_c> (result-name io in-setf) `(package-c-address ,result-name ,io "locs-dis" :float 6 ,in-setf 0))
(defmacro <locs-pc-rev_c> (result-name io in-setf) `(package-c-address ,result-name ,io "locs-pc-rev" :float 7 ,in-setf 0))
(defmacro <locs-revname_c> (result-name io in-setf) `(package-c-address ,result-name ,io "locs-revname" :int 2 ,in-setf))

(defun pass-locs_c (loc i r datai datar io-data)
  (setf (aref datai i) r)
  (setf (aref datai (+ i 1)) (if *current-output-file* (aref io-data (io-open-index *current-output-file*)) -1))
  (setf (aref datai (+ i 2)) (if *reverb* (aref io-data (io-open-index *reverb*)) -1))
  (setf (aref datar r) (* _sndfix_ (float (or (locs-ascl loc) 0.0))))
  (setf (aref datar (+ r 1)) (* _sndfix_ (float (or (locs-bscl loc) 0.0))))
  (setf (aref datar (+ r 2)) (* _sndfix_ (float (or (locs-cscl loc) 0.0))))
  (setf (aref datar (+ r 3)) (* _sndfix_ (float (or (locs-dscl loc) 0.0))))
  (setf (aref datar (+ r 4)) (* _sndfix_ (float (or (locs-rscl loc) 0.0))))
  (setf (aref datar (+ r 5)) (float (or (locs-deg loc) 0.0)))
  (setf (aref datar (+ r 6)) (float (or (locs-dis loc) 0.0)))
  (setf (aref datar (+ r 7)) (float (or (locs-pc-rev loc) 0.0))))

(defun dm-locs_c (addr datai datar)
  (let ((adr (aref datai addr)))
    (make-locs :ascl (* _sndflt_ (aref datar adr))
	       :bscl (* _sndflt_ (aref datar (+ adr 1)))
	       :cscl (* _sndflt_ (aref datar (+ adr 2)))
	       :dscl (* _sndflt_ (aref datar (+ adr 3)))
	       :rscl (* _sndflt_ (aref datar (+ adr 4)))
	       :deg (aref datar (+ adr 5))
	       :dis (aref datar (+ adr 6))
	       :pc-rev (aref datar (+ adr 7))
	       :revname (plusp (aref datai (+ addr 2))))))

(defvar locsig-accessor nil)

(defmacro <check-locsig_c> (var)
  (setf locsig-accessor var)
  nil)

(defmacro <locsig_c> (result-name loc pass val) 
  ;; more straightforward than OUTn (where any io stream can be in use)--just *REVERB* and *CURRENT-OUTPUT-FILE*
  ;; and since we have a structure allocated at compile time, we don't really need to mess with result-name,
  ;; but by setting up the (float *) pointer at initialization time, we can save a lot of headaches.
  
  (let* ((vardat (gethash loc vars))
	 (fladr (third vardat))
	 (lcp (lc pass))
	 (lcl (lc loc))
	 (lcv (lc val))
	 (lcr (lc result-name)))
    
    (if (member loc new-sig)
      ;; field of user-defined struct or something -- make sure it's initialized -- this code won't work if
      ;; the locs expression is not a field of a user-defined structure or if the structure is not global to
      ;; the run loop.
	(let* ((fld-desc (gethash (first locsig-accessor) user-structs))
	       (fld-name (and fld-desc (fourth fld-desc)))
	       (desc (and fld-desc (gethash (third fld-desc) user-structs)))
	       (offset (and fld-desc (user-struct-offset_c fld-name desc))))
	  (if fld-desc
	      (let ((slcl (lc (second locsig-accessor)))
		    (sld (second offset)))
		(push (format nil "~A = (int *)(~A+~D);~%" lcl slcl sld) *io-inits*)
		(push (format nil "~A = (float *)(_datar_+~A[0]);~%" lcr lcl) *io-inits*))
	    (warn "locsig expression: ~A is about to cause a segmentation violation..." locsig-accessor)))
    ;; set up the scaler pointer at initialization time:
      (push (format nil "~A = (float *)(_datar_+~D);~%" lcr fladr) *io-inits*))
      
    (push (format nil "if ((_beg_ < _datai_[~A[1]+~D]) || (_beg_ > _datai_[~A[1]+~D])) clm_file_reset(_beg_,(int *)(_datai_+~A[1]));~%"
		  lcl io*_beg lcl io*_end lcl) *io-inits*)
    (push (format nil "if (~A[2] != -1)~%    ~
                         { if ((_beg_ < _datai_[~A[2]+~D]) || (_beg_ > _datai_[~A[2]+~D])) clm_file_reset(_beg_,(int *)(_datai_+~A[2])); }~%"
		  lcl lcl io*_beg lcl io*_end lcl) *io-inits*)

    (push (format nil "_datai_[~A[1]+~D] = (~A - _datai_[~A[1]+~D]);~%" lcl io*_data_end (lc pass) lcl io*_beg) *io-finals*)
    (push (format nil "if (~A[2] != -1) _datai_[~A[2]+~D] = (~A - _datai_[~A[2]+~D]);~%" 
		  lcl lcl io*_data_end (lc pass) lcl io*_beg) *io-finals*)

    ;; now generate the C code for locsig -- there might be more than one locsig call per sample
    (format *c-file* "  /* (locsig ~(~A~) ~A ~A) */~%" loc pass val)
    (format *c-file* "  { int *_outn_,*_reverb_; int _oloc_,_rloc_;~%    ~
                          _outn_ = (int *)(_datai_+~A[1]);~%    ~
                          if (~A[2] != -1) _reverb_ = (int *)(_datai_+~A[2]); else _reverb_ = NULL;~%" 
	    lcl lcl lcl)
    (format *c-file* "    if ((~A < _outn_[~D]) || (~A > _outn_[~D])) clm_full_file_reset(~A~A,_outn_);~%    ~
                          _oloc_ = (int)(~A - _outn_[~D]);~%    ~
                          if (_reverb_ != NULL) {~%      ~
                            if ((~A < _reverb_[~D]) || (~A > _reverb_[~D])) clm_full_file_reset(~A~A,_reverb_);~%      ~
                            _rloc_ = (int)(~A - _reverb_[~D]);}~%"
	    lcp io*_beg lcp io*_end (if (rl-p pass) "(int)" "") lcp
	    lcp io*_beg
	    lcp io*_beg lcp io*_end (if (rl-p pass) "(int)" "") lcp
	    lcp io*_beg)

    ;; now the file buffers are lined up ready for scaled output.  
    
    (format *c-file* "    (((int *)(_outn_[~D]))[_oloc_]) += (int)(~A * ~A[0]);~%    ~
                          if (_outn_[~D] > 1) {~%      ~
                            (((int *)(_outn_[~D]))[_oloc_]) += (int)(~A * ~A[1]);~%      ~
                            if (_outn_[~D] > 2) {~%        ~
                              (((int *)(_outn_[~D]))[_oloc_]) += (int)(~A * ~A[2]);~%        ~
                              (((int *)(_outn_[~D]))[_oloc_]) += (int)(~A * ~A[3]);}}~%    ~
                          if (_reverb_ != NULL)~%      ~
                            (((int *)(_reverb_[~D]))[_rloc_]) += (int)(~A * ~A[4]);}~%"
	    io*_dat_A lcv lcr
	    io*_chans
	    io*_dat_B lcv lcr
	    io*_chans
	    io*_dat_C lcv lcr
	    io*_dat_D lcv lcr
	    io*_dat_A lcv lcr)
    nil))



;;; ---- IN-A IN-B IN-C IN-D
;;;
;;; very similar to OUTA

(defmacro <in-n_c> (result-name pass chan &optional i-stream)
  (let* ((lcr0 (lc result-name))
	 (io-result (first (find result-name typed-user-sig :key #'third)))
	 (lcr (lc io-result))
	 (ioadr (fourth (gethash io-result vars)))
	 (str (if (null i-stream) '*current-input-file* i-stream)))
    (push `(setf (aref *clm-datai* ,ioadr) (if ,str (aref *clm-io-open* (io-open-index ,str)) -1)) pup)
    (push (format nil "if (_datai_[~D] != -1) {~%  ~A = (int *)(_datai_+_datai_[~D]);~%" ioadr lcr ioadr) *io-inits*)
    (push (format nil "  if (((_beg_ < ~A[~D]) || (_beg_ > ~A[~D])) && (_beg_ <= ~A[~D])) clm_file_reset(_beg_,~A);~%"
	      lcr io*_beg lcr io*_end lcr io*_size lcr) *io-inits*)
    (push (format nil "  ~A[~D] = (_beg_ - ~A[~D]);}~%else ~A = NULL;~%" lcr io*_loc lcr io*_beg lcr) *io-inits*)
    (push `(when ,str
	     (let ((cur-index (io-open-index ,str)))
	       (when (zerop (aref *clm-io-open* cur-index)) 
		 (setf (aref *clm-io-open* cur-index) *clm-datai-len*)
		 (incf *clm-datai-len* ,i_sizeof_io*))))
	  pip)
    ;; if pass /= loop-var, calculate actual input loc and so on.
    (if (eq pass loop-var)
	(progn
	  (push (format nil "if (~A) {~%  if ((~A[~D] != 0) && (~A~A < ~A[~D]))~%    ~
                               {~A[~D]++; ~A[~D]=0; if (~A[~D] >= ~A[~D]) clm_file_reset(~A~A,~A);}}~%" 
			lcr lcr io*_incr (if (rl-p pass) "(int)" "") (lc pass) lcr io*_size 
			lcr io*_loc lcr io*_incr lcr io*_loc lcr io*_bufsiz 
			(if (rl-p pass) "(int)" "") (lc pass) lcr) *io-incrs*)
	  (format *c-file* "  if (~A) {~%    if (~A~A <= ~A[~D])~A/* (in-~A ~(~A~) ~A) */~%      ~
                                {~A = (float)(_sndflt_ * (((int *)(~A[~D]))[~A[~D]])); ~A[~D]=1;}~%    ~
                              else ~A = 0.0;}~%  else ~A = 0.0;~%"
		  lcr (if (rl-p pass) "(int)" "") (lc pass) lcr io*_size
		  cr_white_space (if (= chan 0) "a" (if (= chan 1) "b" (if (= chan 2) "c" "d"))) pass (or i-stream "")
		  lcr0 lcr (+ io*_dat_A chan) lcr io*_loc lcr io*_incr
		  lcr0 lcr0))
      (format *c-file* "  if (~A) {~%    if (~A~A <= ~A[~D]) {~A/* (in-~A ~(~A~) ~A) */~%      ~
                            if ((~A < ~A[~D]) || (~A > ~A[~D])) clm_file_reset(~A~A,~A);~%      ~
                            ~A = (float)(_sndflt_ * (((int *)(~A[~D]))[~A~A - ~A[~D]]));}~%    ~
                          else ~A = 0.0;}~%  else ~A = 0.0;"
	      lcr (if (rl-p pass) "(int)" "") (lc pass) lcr io*_size
	      cr_white_space (if (= chan 0) "a" (if (= chan 1) "b" (if (= chan 2) "c" "d"))) pass (or i-stream "")
	      (lc pass) lcr io*_beg (lc pass) lcr io*_end (if (rl-p pass) "(int)" "") (lc pass) lcr
	      lcr0 lcr (+ io*_dat_A chan) (if (rl-p pass) "(int)" "") (lc pass) lcr io*_beg
	      lcr0 lcr0)))
  nil)

(defmacro <in-a_c> (result-name pass &optional (i-stream nil))
  `(<in-n_c> ,result-name ,pass 0 ,i-stream))

(defmacro <in-b_c> (result-name pass &optional (i-stream nil)) 
  `(<in-n_c> ,result-name ,pass 1 ,i-stream))

(defmacro <in-c_c> (result-name pass &optional (i-stream nil)) 
  `(<in-n_c> ,result-name ,pass 2 ,i-stream))

(defmacro <in-d_c> (result-name pass &optional (i-stream nil)) 
  `(<in-n_c> ,result-name ,pass 3 ,i-stream))



;;; ---- QUAD, STEREO, MONO ----
(defmacro <quad_c> (result-name &optional fil) 
  ;; result-name is declared an integer via fpackage-op, space is allocated in datai, nothing is implicitly passed
  (let ((ioadr (fourth (gethash result-name vars)))
	(str (if (null fil) '*current-output-file* fil)))
    (push `(setf (aref *clm-datai* ,ioadr) (if (and ,str (io-dat-D ,str)) 1 0)) pup)
    nil))

(defmacro <stereo_c> (result-name &optional fil) 
  (let ((ioadr (fourth (gethash result-name vars)))
	(str (if (null fil) '*current-output-file* fil)))
    (push `(setf (aref *clm-datai* ,ioadr) (if (and ,str (io-dat-B ,str) (null (io-dat-D ,str))) 1 0)) pup)
    nil))

(defmacro <mono_c> (result-name &optional fil) 
  (let ((ioadr (fourth (gethash result-name vars)))
	(str (if (null fil) '*current-output-file* fil)))
    (push `(setf (aref *clm-datai* ,ioadr) (if (and ,str (io-dat-A ,str) (null (io-dat-B ,str))) 1 0)) pup)
    nil))



;;; ---- READIN ----
;;; rdin layout is: 0:i 1:inc 2:chan 3:io-index

(defun rdin-offset_c (typ)
  (case typ 
    (rdin 0)
    (sr 6)
    (fftflt 4)
    (conv 4)
    (spd 9)
    (otherwise (break "unknown type: ~A" typ))))

;(defconstant i_sizeof_rdin 4) -- defined above for <start>
(defconstant r_sizeof_rdin 0)

(defmacro <rdin-i_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rdin-i" :int 0 ,in-setf))
(defmacro <rdin-inc_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rdin-inc" :int 1 ,in-setf))
(defmacro <rdin-chan_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rdin-chan" :int 2 ,in-setf))
(defmacro <rdin-fil_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rdin-fil" :int* 3 ,in-setf))

(defmacro <read-position_c> (result-name rd) 
  (format *c-file* "  ~A = ~A[~D];~A/* (read-position ~(~A~)) */~%" 
	  (lc result-name) (lc rd) (rdin-offset_c (get-type_c rd)) cr_white_space rd)
  nil)

(defmacro <read-forward_c> (result-name rd) 
  (declare (ignore result-name))
  (format *c-file* "  ~A[~D] = 1;~A/* (read-forward ~(~A~)) */~%" (lc rd) (+ 1 (rdin-offset_c (get-type_c rd))) cr_white_space rd)
  nil)

(defmacro <read-backward_c> (result-name rd)
  (declare (ignore result-name))
  (format *c-file* "  ~A[~D] = -1;~A/* (read-backward ~(~A~)) */~%" (lc rd) (+ 1 (rdin-offset_c (get-type_c rd))) cr_white_space rd)
  nil)

(defmacro <readin_c> (result-name rd)
  (let ((lcrd (lc rd))
	(lcr (lc result-name)))
    (format *c-file* "  {int *_io_; _io_ = (int *)(_datai_+~A[3]);~A/* (readin ~(~A~)) */~%    ~
                         if ((~A[0] >= 0) && (~A[0] < _io_[2])) {~%      ~
                           if ((~A[0] < _io_[3]) || (~A[0] > _io_[4])) clm_file_reset(~A[0],_io_);~%      ~
                           ~A = (float)(_sndflt_ * (((int *)(_io_[9+~A[2]]))[~A[0] - _io_[3]]));}~%    ~
                         else ~A = 0.0;~%    ~
                         ~A[0] += ~A[1];}~%"
	    lcrd cr_white_space rd
	    lcrd lcrd
	    lcrd lcrd lcrd
	    lcr lcrd lcrd
	    lcr
	    lcrd lcrd))
  nil)

(defun pass-rdin_c (rd adr datai io-data)
  (setf (aref datai adr) (rdin-i rd))
  (setf (aref datai (+ adr 1)) (rdin-inc rd))
  (setf (aref datai (+ adr 2)) (rdin-chn rd))
  (setf (aref datai (+ adr 3)) (if (rdin-fil rd) (aref io-data (io-open-index (rdin-fil rd))) -1)))

(defun dm-rdin_c (adr datai)
  (make-rdin :i (aref datai adr) :inc (aref datai (+ adr 1)) :chn (aref datai (+ adr 2)) :fil (dm-io_c* (+ adr 3) datai)))

(defmacro <check-readin_c> (rd)	 ;"rd" here is the caller's full name which might be an array of structures
  (declare (ignore rd))
  nil)



;;; ---- RESAMPLE ----
;;;
;;; 0:datar adr 1:chan 2:io-index
;;; 0:x 1:sr 2:lst 3:nxt

(defconstant i_sizeof_smp 3)
(defconstant r_sizeof_smp 4)

(defun pass-smp_c (s i r datai datar io-data)
  (setf (aref datar r) (make-short-float (smp-x s)))
  (setf (aref datar (+ r 1)) (make-short-float (smp-sr s)))
  (setf (aref datar (+ r 2)) (make-short-float (smp-lst s)))
  (setf (aref datar (+ r 3)) (make-short-float (smp-nxt s)))
  (setf (aref datai i) r)
  (setf (aref datai (+ i 1)) (smp-chn s))
  (setf (aref datai (+ i 2)) (if (smp-fil s) (aref io-data (io-open-index (smp-fil s))) -1)))

(defun dm-smp_c (adr datai datar)
  (make-smp :x (aref datar (aref datai adr))
	    :sr (aref datar (+ (aref datai adr) 1))
	    :lst (aref datar (+ (aref datai adr) 2))
	    :nxt (aref datar (+ (aref datai adr) 3))
	    :chn (aref datai (+ adr 1))
	    :fil (dm-io_c* (+ adr 2) datai)))

(defmacro <smp-chn_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "smp-chn" :int 1 ,in-setf))
(defmacro <smp-fil_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "smp-fil" :int* 2 ,in-setf))
(defmacro <smp-x_c> (result-name s in-setf) `(package-c-address ,result-name ,s "smp-x" :float 0 ,in-setf 0))
(defmacro <smp-sr_c> (result-name s in-setf) `(package-c-address ,result-name ,s "smp-sr" :float 1 ,in-setf 0))
(defmacro <smp-lst_c> (result-name s in-setf) `(package-c-address ,result-name ,s "smp-lst" :float 2 ,in-setf 0))
(defmacro <smp-nxt_c> (result-name s in-setf) `(package-c-address ,result-name ,s "smp-nxt" :float 3 ,in-setf 0))

(defmacro <resample_c> (result-name s &optional (sr nil)) 
  (let ((lcrs (lc s))
	(lcr (lc result-name)))
  (format *c-file* "  { int* _rio_; int _chn_; float *_rs_; int _loc_,_loc0_;~A/* (resample ~(~A~)) */~%    ~
                        _rs_ = (float *)(_datar_+~A[0]); _rio_ = (int *)(_datai_+~A[2]); _chn_ = ~A[1];~%    ~
                        _loc_ = (int)_rs_[0];~%    ~
                        ~A = _rs_[2] + (_rs_[0] - _loc_)*(_rs_[3] - _rs_[2]);~%    ~
                        _rs_[0] += ~A_rs_[1]~A;~%    ~
                        _loc0_ = (int)_rs_[0];~%    ~
                        if ((_loc0_ <= _rio_[2]) && (_loc0_ > 0)) {~%      ~
                          if (_loc_ != _loc0_) {~%        ~
	                    if ((_loc0_ < _rio_[3]) || ((_loc0_+1) >= _rio_[4])) clm_file_reset(_loc0_,_rio_);~%        ~
	                    _rs_[2] = (float)(_sndflt_ * (((int *)(_rio_[9+_chn_]))[_loc0_-_rio_[3]]));~%        ~
  	                    _rs_[3] = (float)(_sndflt_ * (((int *)(_rio_[9+_chn_]))[_loc0_+1-_rio_[3]]));}}~%    ~
                        else ~A=0.0;}~%"
	  cr_white_space s
	  lcrs lcrs lcrs
	  lcr
	  (if sr "(" "") (if sr (format nil " + ~A)" (lc sr)) "")
	  lcr))
  nil)

(defmacro <check-resample_c> (rd)	;rd here /= rd in resample call => for array checks
  (declare (ignore rd))
  nil)




;;; ---- SRC ----
;;; 0:datar adr 1:left 2:right 3:width 4:use-filter 5:filter-size 6:rd* n:data nn:filt
;;; 0:x 1:incr

(defconstant i_sizeof_sr (+ 6 i_sizeof_rdin (* 2 i_sizeof_aref_header)))
(defconstant r_sizeof_sr 2)

(defun pass-sr_c (s i r datai datar info io-data)
  (setf (aref datar r) (make-short-float (sr-x s)))
  (setf (aref datar (+ r 1)) (make-short-float (sr-incr s)))
  (setf (aref datai i) r)
  (setf (aref datai (+ i 1)) (sr-left s))
  (setf (aref datai (+ i 2)) (sr-right s))
  (setf (aref datai (+ i 3)) (sr-width s))
  (setf (aref datai (+ i 4)) (if (sr-filt s) 1 0))
  (setf (aref datai (+ i 5)) (if (sr-filt s) (length (sr-filt s)) 0))
  (pass-rdin_c (sr-rd s) (+ i 6) datai io-data)
  (pass-array_c (sr-data s) (+ i 6 i_sizeof_rdin) datai datar info)
  (if (sr-filt s) (pass-array_c (sr-filt s) (+ i 6 i_sizeof_rdin i_sizeof_aref_header) datai datar info)))

(defun dm-sr_c (adr datai datar)
  (make-sr :x (aref datar (aref datai adr))
	   :incr (aref datar (+ (aref datai adr) 1))
	   :left (aref datai (+ adr 1))
	   :right (aref datai (+ adr 2))
	   :width (aref datai (+ adr 3))
	   :rd (dm-rdin_c (+ adr 6) datai)
	   :filt (if (plusp (aref datai (+ adr 4)))
		     (make-array (aref datai (+ adr 5)) 
				 :displaced-to datar 
				 :element-type (type-of (aref datar 0))
				 :displaced-index-offset (aref datai (+ adr 6 i_sizeof_rdin i_sizeof_aref_header aref_block))))
	   :data (make-array (aref datai (+ adr 6 i_sizeof_rdin aref_size))
				   :displaced-to datar 
				   :element-type (type-of (aref datar 0))
				   :displaced-index-offset (aref datai (+ adr 6 i_sizeof_rdin aref_block)))))

(defmacro <sr-left_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sr-left" :int 1 ,in-setf))
(defmacro <sr-right_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sr-right" :int 2 ,in-setf))
(defmacro <sr-width_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sr-width" :int 3 ,in-setf))
(defmacro <sr-rd_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sr-rd" :int* 6 ,in-setf))

(defmacro <sr-data_c> (result-name s in-setf) 
  `(package-c-address ,result-name ,s "sr-data" :float** ,(+ 6 i_sizeof_rdin aref_block) ,in-setf))

(defmacro <sr-filt_c> (result-name s in-setf)
  `(package-c-address ,result-name ,s "sr-filt" :float** ,(+ 6 i_sizeof_rdin i_sizeof_aref_header aref_block) ,in-setf))

(defmacro <sr-x_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sr-x" :float 0 ,in-setf 0))
(defmacro <sr-incr_c> (result-name s in-setf) `(package-c-address ,result-name ,s "sr-incr" :float 1 ,in-setf 0))

(defmacro <src_c> (result-name s &optional fm-1) 
  (let ((lcs (lc s)))
    (format *c-file* "  ~A = c_src((int *)(~A+6),(int *)(_datai_+~A[9]),(float *)(_datar_+~A[0]),(int *)(~A+1),~%            ~
                                   (float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]),~A~A);~%"
	    (lc result-name) lcs lcs lcs lcs
	    lcs (+ 6 i_sizeof_rdin aref_block) lcs (+ 6 i_sizeof_rdin i_sizeof_aref_header aref_block) 
	    (if (and fm-1 (int-p fm-1)) "(float)" "") (if fm-1 (lc fm-1) 0.0)))
  nil)

(defmacro <check-src_c> (rd)
  (declare (ignore rd))
  nil)



;;; ---- RUN-BLOCK ----
;;; 
;;; 0:datar 1:loc 2:siz 3:buf header
;;; 0:ctr

(defmacro <rblk-loc_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rblk-loc" :int 1 ,in-setf))
(defmacro <rblk-ctr_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rblk-ctr" :float 0 ,in-setf 0))
(defmacro <rblk-siz_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rblk-siz" :int 2 ,in-setf))
(defmacro <rblk-buf_c> (result-name rd in-setf) `(package-c-address ,result-name ,rd "rblk-buf" :float** 3 ,in-setf))

;(defconstant i_sizeof_rblk (+ 3 i_sizeof_aref_header)) -- defined above for <start>
(defconstant r_sizeof_rblk 1)

(defun pass-blk_c (b i r datai datar info)
  (setf (aref datai i) r)
  (setf (aref datar r) (make-short-float (rblk-ctr b)))
  (setf (aref datai (+ i 1)) (rblk-loc b))
  (setf (aref datai (+ i 2)) (rblk-siz b))
  (pass-array_c (rblk-buf b) (+ i 3) datai datar info))

(defun dm-blk_c (adr datai datar)
  (let ((size (aref datai (+ adr 3 aref_size))))
    (make-rblk :loc (aref datai (+ adr 1))
	       :ctr (aref datar (aref datai adr))
	       :siz (aref datai (+ adr 2))
	       :buf (if (< size *clm-array-print-length*)
			(make-array size
				    :element-type (type-of (aref datar 0))
				    :displaced-to datar 
				    :displaced-index-offset (aref datai (+ adr 3 aref_block)))
		      (make-array 2 :initial-contents (list 0 (1- size)))))))

(defmacro <run-block_c> (result-name e)
  ;; OPT: this could be sped up by splitting out the loc check and calling c_base_run_block as in wave-train
  (format *c-file* "  ~A = c_run_block((int *)(~A+1),(float *)(_datar_+~A[0]),(float *)(_datar_+~A[3]));~%" 
	  (lc result-name) (lc e) (lc e) (lc e))
  nil)



;;; ---- FFT ----
;;; 0:size 1:rdat 1+aref:idat

(defconstant i_sizeof_fft-data (+ 1 (* 2 i_sizeof_aref_header)))
(defconstant r_sizeof_fft-data 0)

(defmacro <fft-data-real_c> (result-name f in-setf) 
  `(package-c-address ,result-name ,f "fft-data-real" :float** 1 ,in-setf))

(defmacro <fft-data-imaginary_c> (result-name f in-setf) 
  `(package-c-address ,result-name ,f "fft-data-imaginary" :float** ,(+ 1 i_sizeof_aref_header) ,in-setf))

(defmacro <fft-data-size_c> (result-name f in-setf) `(package-c-address ,result-name ,f "fft-data-size" :int 0 ,in-setf))

(defun pass-fft-data_c (f i datai datar info)
  (setf (aref datai i) (fft-data-size f))
  (pass-array_c (fft-data-real f) (+ i 1) datai datar info)
  (pass-array_c (fft-data-imaginary f) (+ i 1 i_sizeof_aref_header) datai datar info))

(defun dm-fft-data_c (i datai datar)
  (let ((size (aref datai i)))
    (make-fft-data :size size
		   :real (if (< size *clm-array-print-length*) 
			     (make-array size
					 :displaced-to datar 
					 :element-type (type-of (aref datar 0))
					 :displaced-index-offset (aref datai (+ i 1 aref_block)))
			   (make-array 2 :initial-contents (list 0 (1- size))))
		   :imaginary (if (< size *clm-array-print-length*)
				  (make-array size
					      :displaced-to datar 
					      :element-type (type-of (aref datar 0))
					      :displaced-index-offset (aref datai (+ i 1 i_sizeof_aref_header aref_block)))
				(make-array 2 :initial-contents (list 0 (1- size)))))))

(defmacro <fft_c> (result-name data &optional (fft-dir 1))
  (declare (ignore result-name))
  (format *c-file* "  c_fft((float *)(_datar_+~A[1]),(float *)(_datar_+~A[~D]),~A[0],~D,(int)ceil((log(~A[0])/log(2))));~%"
	  (lc data) (lc data) (+ 1 i_sizeof_aref_header) (lc data) (lc fft-dir) (lc data))
  nil)

(defmacro <inverse-fft_c> (result-name data)
  `(<fft_c> ,result-name ,data -1))

  

;;; ---- CLEAR-BLOCK ----
(defmacro <clear-block_c> (result-name data)
  (declare (ignore result-name))
  (format *c-file* "  c_clear_block((float *)(_datar_+~A[~D]),~A[~D]);~%" (lc data) aref_block (lc data) aref_size)
  nil)

(defmacro <spectrum_c> (result-name freq phase size &optional (normalized t))
  (declare (ignore result-name freq phase size normalized))
  nil)


;;; ---- FFT-WINDOW ----
(defmacro <fft-window_c> (result-name data window)
  (declare (ignore result-name))
  ;; data and window are array headers
  (format *c-file* "  c_fft_window((float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]),~A[~D]);~%"
	  (lc data) aref_block (lc window) aref_block (lc window) aref_size)
  nil)


;;; ---- FFT-FILTER ----
;;;
;;; 0:siz 1:hop 2:half_siz 3:{pow} 4:rd n:blk nn:data-real nnn:data-imag nnnn:env

(defconstant i_sizeof_fftflt (+ 4 i_sizeof_rdin i_sizeof_rblk (* 3 i_sizeof_aref_header)))
(defconstant r_sizeof_fftflt r_sizeof_rblk)

(defmacro <fftflt-siz_c> (result-name s in-setf) `(package-c-address ,result-name ,s "fftflt-siz" :int 0 ,in-setf))
(defmacro <fftflt-hop_c> (result-name s in-setf) `(package-c-address ,result-name ,s "fftflt-hop" :int 1 ,in-setf))
(defmacro <fftflt-half-siz_c> (result-name s in-setf) `(package-c-address ,result-name ,s "fftflt-half-siz" :int 2 ,in-setf))
(defmacro <fftflt-rd_c> (result-name s in-setf) `(package-c-address ,result-name ,s "fftflt-rd" :int* 4 ,in-setf))
(defmacro <fftflt-b_c> (result-name s in-setf) `(package-c-address ,result-name ,s "fftflt-b" :int* ,(+ 4 i_sizeof_rdin) ,in-setf))

(defmacro <fftflt-datar_c> (result-name s in-setf) 
  `(package-c-address ,result-name ,s "fftflt-datar" :float** ,(+ 4 i_sizeof_rdin i_sizeof_rblk) ,in-setf))

(defmacro <fftflt-datai_c> (result-name s in-setf) 
  `(package-c-address ,result-name ,s "fftflt-datai" :float** ,(+ 4 i_sizeof_rdin i_sizeof_rblk i_sizeof_aref_header) ,in-setf))

(defmacro <fftflt-env_c> (result-name s in-setf) 
  `(package-c-address ,result-name ,s "fftflt-env" :float** ,(+ 4 i_sizeof_rdin i_sizeof_rblk (* 2 i_sizeof_aref_header)) ,in-setf))

(defun pass-fftflt_c (f i r datai datar info io-data)
  (let ((blk-adr (+ i 4 i_sizeof_rdin))
	(radr (+ i 4 i_sizeof_rdin i_sizeof_rblk))
	(iadr (+ i 4 i_sizeof_rdin i_sizeof_rblk i_sizeof_aref_header))
	(env-adr (+ i 4 i_sizeof_rdin i_sizeof_rblk (* 2 i_sizeof_aref_header))))
    (setf (aref datai i) (or (fftflt-siz f) 0))
    (setf (aref datai (+ i 1)) (or (fftflt-hop f) 0))
    (setf (aref datai (+ i 2)) (or (fftflt-half-siz f) 0))
    (setf (aref datai (+ i 3)) (if (plusp (or (fftflt-siz f) 0)) (ceiling (log (fftflt-siz f) 2)) 0))
    (pass-rdin_c (fftflt-rd f) (+ i 4) datai io-data)
    (if (fftflt-b f)
	(pass-blk_c (fftflt-b f) blk-adr r datai datar info)
      (setf (aref datai blk-adr) -1))
    (if (fftflt-datar f)
	(pass-array_c (fftflt-datar f) radr datai datar info)
      (setf (aref datai radr) -1))
    (if (fftflt-datai f)
	(pass-array_c (fftflt-datai f) iadr datai datar info)
      (setf (aref datai iadr) -1))
    (if (fftflt-env f)
	(pass-array_c (fftflt-env f) env-adr datai datar info)
      (setf (aref datai env-adr) -1))))

(defun dm-fftflt_c (adr datai datar)
  (let ((blk-adr (+ adr 4 i_sizeof_rdin))
	(iadr (+ adr 4 i_sizeof_rdin i_sizeof_rblk i_sizeof_aref_header))
	(radr (+ adr 4 i_sizeof_rdin i_sizeof_rblk))
	(env-adr (+ adr 4 i_sizeof_rdin i_sizeof_rblk (* 2 i_sizeof_aref_header))))
    (make-fftflt :siz (aref datai adr)
		 :hop (aref datai (+ adr 1))
		 :half-siz (aref datai (+ adr 2))
		 :rd (dm-rdin_c (+ adr 4) datai)
		 :b (and (/= (aref datai blk-adr) -1) (dm-blk_c (aref datai blk-adr) datai datar))
		 :datar (and (/= (aref datai (+ radr aref_block)) -1)
			     (if (< (aref datai adr) *clm-array-print-length*)
				 (make-array (aref datai (+ radr aref_size))
					     :displaced-to datar 
					     :element-type (type-of (aref datar 0))
					     :displaced-index-offset (aref datai (+ radr aref_block)))
			       (make-array 2 :initial-contents (list 0 (1- (aref datai adr))))))
		 :datai (and (/= (aref datai (+ iadr aref_block)) -1)
			     (if (< (aref datai adr) *clm-array-print-length*)
				 (make-array (aref datai (+ iadr aref_size))
					     :displaced-to datar 
					     :element-type (type-of (aref datar 0))
					     :displaced-index-offset (aref datai (+ iadr aref_block)))
			       (make-array 2 :initial-contents (list 0 (1- (aref datai adr))))))
		 :env (and (/= (aref datai (+ env-adr aref_block)) -1)
			   (if (< (aref datai adr) *clm-array-print-length*)
			       (make-array (aref datai (+ env-adr aref_size))
					   :displaced-to datar 
					   :element-type (type-of (aref datar 0))
					   :displaced-index-offset 
					   (aref datai (+ env-adr aref_block)))
			     (make-array 2 :initial-contents (list 0 (1- (aref datai adr)))))))))

(defmacro <fft-filter_c> (result-name s)
  (let ((lcs (lc s))
	(blk-adr (+ 4 i_sizeof_rdin))
	(iadr (+ 4 i_sizeof_rdin i_sizeof_rblk i_sizeof_aref_header))
	(radr (+ 4 i_sizeof_rdin i_sizeof_rblk))
	(env-adr (+ 4 i_sizeof_rdin i_sizeof_rblk (* 2 i_sizeof_aref_header))))
    (format *c-file* "  { int *_io_; if (~A[7] != -1) _io_ = (int *)(_datai_+~A[7]); else _io_=NULL;~%    ~
                          ~A = c_fft_filter((int *)(~A+4),_io_,~%                 ~
                                     (int *)(~A+~D),(float *)(_datar_+~A[~D]),~A,(float *)(_datar_+~A[~D]),~%                 ~
                                     (float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]));}~%"
	    lcs lcs
	    (lc result-name) lcs 
	    lcs (+ blk-adr 1) lcs blk-adr lcs lcs (+ blk-adr 3 aref_block)
	    lcs (+ radr aref_block) 
	    lcs (+ iadr aref_block)
	    lcs (+ env-adr aref_block)))
  nil)

(defmacro <check-fftflt_c> (rd)
  (declare (ignore rd))
  nil)



;;; ---- CONVOLVE ----
;;;
;;; 0:fftflt n:filtr nn:filti

(defconstant i_sizeof_conv (+ i_sizeof_fftflt (* 2 i_sizeof_aref_header)))
(defconstant r_sizeof_conv r_sizeof_fftflt)

(defmacro <conv-fftf_c> (result-name s in-setf) `(package-c-address ,result-name ,s "conv-fftf" :int* 0 ,in-setf))
(defmacro <conv-filtr_c> (result-name s in-setf) `(package-c-address ,result-name ,s "conv-filtr" :float** ,i_sizeof_fftflt ,in-setf))
(defmacro <conv-filti_c> (result-name s in-setf) 
  `(package-c-address ,result-name ,s "conv-filti" :float** ,(+ i_sizeof_fftflt i_sizeof_aref_header) ,in-setf))

(defun pass-conv_c (c i r datai datar info io-data)
  (let ((radr (+ i i_sizeof_fftflt))
	(iadr (+ i i_sizeof_fftflt i_sizeof_aref_header)))
    (pass-fftflt_c (conv-fftf c) i r datai datar info io-data)
    (if (conv-filtr c)
	(pass-array_c (conv-filtr c) radr datai datar info)
      (setf (aref datai radr) -1))
    (if (conv-filti c)
	(pass-array_c (conv-filti c) iadr datai datar info)
      (setf (aref datai iadr) -1))))

(defun dm-conv_c (i datai datar)
  (let ((radr (+ i i_sizeof_fftflt))
	(iadr (+ i i_sizeof_fftflt i_sizeof_aref_header)))
    (make-conv :fftf (dm-fftflt_c i datai datar)
	       :filtr (and (/= (aref datai radr) -1)
			   (make-array (aref datai (+ radr aref_size)
					     :displaced-to datar 
					     :element-type (type-of (aref datar 0))
					     :displaced-index-offset (aref datai (+ radr aref_block)))))
	       :filti (make-array (and (/= (aref datai iadr) -1)
				       (aref datai (+ iadr aref_size)
					     :displaced-to datar 
					     :element-type (type-of (aref datar 0))
					     :displaced-index-offset (aref datai (+ iadr aref_block))))))))

(defmacro <convolve_c> (result-name ff) ;remember rdin-offset
  (let ((lcf (lc ff))
	(blk-adr (+ 4 i_sizeof_rdin))
	(radr (+ 4 i_sizeof_rdin i_sizeof_rblk))
	(iadr (+ 4 i_sizeof_rdin i_sizeof_rblk i_sizeof_aref_header)))
    (format *c-file* "  { int *_io_; float *_filtr_; /* conv-filtr null -> use readin, not convolve */~%    ~
                          if (~A[7] != -1) _io_ = (int *)(_datai_+~A[7]); else _io_=NULL;~%    ~
                          if (~A[~D] != -1) _filtr_ = (float *)(_datar_+~A[~D]); else _filtr_=NULL;~%    ~
                          ~A = c_convolve_or_readin((int *)(~A+4),_io_,~%                 ~
                                     (int *)(~A+~D),(float *)(_datar_+~A[~D]),~A,(float *)(_datar_+~A[~D]),~%                 ~
                                     (float *)(_datar_+~A[~D]),(float *)(_datar_+~A[~D]),~%                 ~
                                     _filtr_,(float *)(_datar_+~A[~D]));}~%"
	    lcf lcf
	    lcf (+ i_sizeof_fftflt aref_block) lcf (+ i_sizeof_fftflt aref_block) 
	    (lc result-name) lcf 
	    lcf (+ blk-adr 1) lcf blk-adr lcf lcf (+ blk-adr 3 aref_block)
	    lcf (+ radr aref_block) 
	    lcf (+ iadr aref_block)
	    lcf (+ i_sizeof_fftflt i_sizeof_aref_header aref_block)))
  nil)

(defmacro <check-convolve_c> (rd)
  (declare (ignore rd))
  nil)




;;; ---- EXPAND ----
;;;
;;; 0:datar adr 1:len 2:rmp 3:in_spd 4:out_spd 5:cur_in 6:cur_out 7:s20 8:s50 9:ctr 10:rdin n:b (float array -- len elements)
;;; 0:amp
;;; 
;;; order matters!  This marches in step with c_spd in cmus.c

(defconstant i_sizeof_spd (+ 10 i_sizeof_rdin i_sizeof_aref_header))
(defconstant r_sizeof_spd 1)

(defmacro <spd-amp_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-amp" :float 0 ,in-setf 0))
(defmacro <spd-len_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-len" :int 1 ,in-setf))
(defmacro <spd-rmp_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-rmp" :int 2 ,in-setf))
(defmacro <spd-in-spd_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-in-spd" :int 3 ,in-setf))
(defmacro <spd-out-spd_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-out-spd" :int 4 ,in-setf))
(defmacro <spd-cur-in_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-cur-out" :int 5 ,in-setf))
(defmacro <spd-cur-out_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-cur-out" :int 6 ,in-setf))
(defmacro <spd-s20_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-s20" :int 7 ,in-setf))
(defmacro <spd-s50_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-s50" :int 8 ,in-setf))
(defmacro <spd-ctr_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-ctr" :int 9 ,in-setf))
(defmacro <spd-rd_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-rd" :int* 10 ,in-setf))
(defmacro <spd-b_c> (result-name s in-setf) `(package-c-address ,result-name ,s "spd-b" :float** ,(+ 10 i_sizeof_rdin aref_block) ,in-setf))

(defun pass-spd_c (s i r datai datar info io-data)
  (setf (aref datai i) r)
  (setf (aref datar r) (make-short-float (spd-amp s)))
  (setf (aref datai (+ i 1)) (spd-len s))
  (setf (aref datai (+ i 2)) (spd-rmp s))
  (setf (aref datai (+ i 3)) (spd-in-spd s))
  (setf (aref datai (+ i 4)) (spd-out-spd s))
  (setf (aref datai (+ i 5)) (spd-cur-in s))
  (setf (aref datai (+ i 6)) (spd-cur-out s))
  (setf (aref datai (+ i 7)) (spd-s20 s))
  (setf (aref datai (+ i 8)) (spd-s50 s))
  (setf (aref datai (+ i 9)) (spd-ctr s))
  (pass-rdin_c (spd-rd s) (+ i 10) datai io-data)
  (pass-array_c (spd-b s) (+ i 10 i_sizeof_rdin) datai datar info))

(defun dm-spd_c (adr datai datar)
  (let ((size (aref datai (+ adr 10 i_sizeof_rdin aref_size))))
    (make-spd :amp (aref datar (aref datai adr))
	      :len (aref datai (+ adr 1))
	      :rmp (aref datai (+ adr 2))
	      :in-spd (aref datai (+ adr 3))
	      :out-spd (aref datai (+ adr 4))
	      :cur-in (aref datai (+ adr 5))
	      :cur-out (aref datai (+ adr 6))
	      :s20 (aref datai (+ adr 7))
	      :s50 (aref datai (+ adr 8))
	      :ctr (aref datai (+ adr 9))
	      :rd (dm-rdin_c (+ adr 10) datai)
	      :b (if (< size *clm-array-print-length*)
		     (make-array size
				 :displaced-to datar 
				 :element-type (type-of (aref datar 0))
				 :displaced-index-offset (aref datai (+ adr 10 i_sizeof_rdin aref_block)))
		   (make-array 2 :initial-contents (list 0 (1- size)))))))

(defmacro <expand_c> (result-name s) ;remember rdin-offset
  (let ((lcs (lc s)))
    (format *c-file* "  ~A = c_spd((int *)(~A+10),(int *)(~A+1),(float *)(_datar_+~A[0]),~%            ~
                                 (int *)(_datai_+~A[13]),(float *)(_datar_+~A[~D]));~%"
	    (lc result-name) lcs lcs lcs
	    lcs lcs (+ 10 i_sizeof_rdin aref_block)))
  nil)

(defmacro <check-expand_c> (rd)
  (declare (ignore rd))
  nil)



;;; ---- WAVE-TRAIN ----
;;;
;;; 0:datar adr 1:blk (imbedded =  0:adr 1:loc 2:siz 3:buf header) n:wave header
;;; 0:freq 1:phase 2:mag 3:srate [4:ctr]

(defconstant i_sizeof_wt (+ 1 i_sizeof_rblk i_sizeof_aref_header))
(defconstant r_sizeof_wt (+ 4 r_sizeof_rblk))

(defmacro <wt-wsiz_c> (result-name wt in-setf) 
  `(package-c-address ,result-name ,wt "wt-siz" :int 3 ,in-setf))

(defmacro <wt-b_c> (result-name wt in-setf) 
  `(package-c-address ,result-name ,wt "wt-b" :int* 1 ,in-setf))

(defmacro <wt-wave_c> (result-name wt in-setf) 
  `(package-c-address ,result-name ,wt "wt-wave" :float** ,(+ 1 i_sizeof_rblk) ,in-setf))

(defmacro <wt-freq_c> (result-name wt in-setf) `(package-c-address ,result-name ,wt "wt-freq" :float 0 ,in-setf 0))
(defmacro <wt-phase_c> (result-name wt in-setf) `(package-c-address ,result-name ,wt "wt-phase" :float 1 ,in-setf 0))
(defmacro <wt-internal-mag_c> (result-name wt in-setf) `(package-c-address ,result-name ,wt "wt-internal-mag" :float 2 ,in-setf 0))

(defun pass-wt_c (w i r datai datar info)
  (setf (aref datai i) r)
  (pass-blk_c (wt-b w) (+ i 1) (+ r 4) datai datar info)
  (pass-array_c (wt-wave w) (+ i 1 i_sizeof_rblk) datai datar info)
  (setf (aref datar r) (make-short-float (wt-freq w)))
  (setf (aref datar (+ r 1)) (make-short-float (wt-phase w)))
  (setf (aref datar (+ r 2)) (make-short-float (wt-internal-mag w)))
  (setf (aref datar (+ r 3)) (make-short-float sampling-rate)))

(defun dm-wt_c (adr datai datar)
  (let ((size (aref datai (+ adr 3)))
	(radr (aref datai adr)))
    (make-wt :freq (aref datar radr)
	     :phase (aref datar (+ radr 1))
	     :internal-mag (aref datar (+ radr 2))
	     :b (dm-blk_c (+ adr 1) datai datar)
	     :wsiz size
	     :wave (if (< size *clm-array-print-length*)
		       (make-array size
				   :displaced-to datar 
				   :element-type (type-of (aref datar 0))
				   :displaced-index-offset (aref datai (+ adr 1 i_sizeof_rblk aref_block)))
		     (make-array 2 :initial-contents (list 0 (1- size)))))))

(defmacro <wave-train_c> (result-name e &optional fm)
  (let ((lce (lc e)))
    ;; OPT: this could be sped up if phase /= 0.0 by moving the c_table_interp calls into one function
    (format *c-file* "  {float *_wt_,*_wave_; int *_blk_; float *_fblk_,*_blkbuf_;~A/* (wave-train ~(~A~) ~A) */~%    ~
                         _blk_ = (int *)(~A+2); _fblk_ = (float *)(_datar_+~A[1]); _blkbuf_ = (float *)(_datar_+~A[~D]);~%    ~
                         if (_blk_[0] == 0) {int _i_; float _f_;~%      ~
                           _wt_ = (float *)(_datar_+~A[0]); _wave_ = (float *)(_datar_+~A[~D]);~%      ~
                           if (_wt_[1] == 0.0) {for (_i_=0;_i_<~A[3];_i_++) _blkbuf_[_i_]+=_wave_[_i_];}~%      ~
                           else {for (_f_=_wt_[1],_i_=0;_i_<~A[3];_i_++,_f_+=1.0) _blkbuf_[_i_]+=c_table_interp(_wave_,_f_,~A[3]);}~%      ~
                           _fblk_[0] += (_wt_[3]/(_wt_[0]~A));}~%    ~
                         if (_blk_[0] < _blk_[1]) ~A = _blkbuf_[_blk_[0]]; else ~A = 0.0;~%    ~
                         _blk_[0]++;~%    ~
                         if (_blk_[0] >= _fblk_[0]) c_base_run_block(_blk_,_fblk_,_blkbuf_);}~%"
	    cr_white_space e fm
	    lce lce lce (+ 4 aref_block) 
	    lce lce (+ 1 i_sizeof_rblk aref_block) 
	    lce 
	    lce lce
	    (if fm (format nil "+(~A*_wt_[2])" (lc fm)) "")
	    (lc result-name) (lc result-name)))
  nil)



;;; ---- FORMNT ----
;;;
;;; 0:g 1:tz n:tp (all float)

(defconstant i_sizeof_frmnt 1)
(defconstant r_sizeof_frmnt (+ 1 r_sizeof_flt-two-pole r_sizeof_flt-two-zero))

(defmacro <frmnt-g_c> (result-name fr in-setf) `(package-c-address ,result-name ,fr "frmnt-g" :float 0 ,in-setf))
(defmacro <frmnt-tp_c> (result-name fr in-setf) `(package-c-address ,result-name ,fr "frmnt-tp" :float* ,( + 1 r_sizeof_flt-two-zero) ,in-setf))
(defmacro <frmnt-tz_c> (result-name fr in-setf) `(package-c-address ,result-name ,fr "frmnt-tz" :float* 1 ,in-setf))

(defun pass-frmnt_c (fr r datar)
  (setf (aref datar r) (make-short-float (frmnt-g fr)))
  (pass-two-zero_c (frmnt-tz fr) (+ r 1) datar)
  (pass-two-pole_c (frmnt-tp fr) (+ r 1 r_sizeof_flt-two-zero) datar))

(defun dm-frmnt_c (adr datar)
  (make-frmnt :g (aref datar adr)
	      :tz (dm-one-pole_c (+ adr 1) datar)
	      :tp (dm-two-pole_c (+ adr 1 r_sizeof_flt-two-zero) datar)))


;;; ---- WS ----
;;;
;;; 0:datar 0:offset 1:table (float array)
;;; 0:osc

(defconstant i_sizeof_ws (+ 2 i_sizeof_aref_header))
(defconstant r_sizeof_ws r_sizeof_osc)

(defmacro <ws-offset_c> (result-name fr in-setf) `(package-c-address ,result-name ,fr "ws-offset" :int 1 ,in-setf))
(defmacro <ws-tab_c> (result-name fr in-setf) `(package-c-address ,result-name ,fr "ws-tab" :int* 2 ,in-setf))

(defmacro <ws-os_c> (result-name fr in-setf)
  `(package-c-address ,result-name ,fr "ws-os" :float* 0 ,in-setf 0))

(defun pass-ws_c (w i r datai datar info)
  (setf (aref datai i) r)
  (setf (aref datai (+ i 1)) (floor (ws-offset w)))
  (pass-array_c (ws-tab w) (+ i 2) datai datar info)
  (pass-osc_c (ws-os w) r datar))

(defun dm-ws_c (adr datai datar)
  (let ((size (aref datai (+ adr 2 aref_size))))
    (make-ws :os (dm-osc_c (aref datai adr) datar)
	     :offset (aref datai (+ adr 1))
	     :tab (if (< size *clm-array-print-length*)
		      (make-array size
				  :displaced-to datar 
				  :element-type (type-of (aref datar 0))
				  :displaced-index-offset (aref datai (+ adr 2 aref_block)))
		    (make-array 2 :initial-contents (list 0 (1- size)))))))



;;; ---- USER-STRUCT ----
;;;
;;; very similar to 56k case in ins56.lisp

(defun user-struct-p_c (n) (gethash (first n) user-structs)) ;:list option used to defstruct, so type is first of list

(defun user-struct-size_c (typ)
  ;; return (list r i)
  (let ((desc (gethash typ user-structs))
	(sizes (list 0 1)))
    (when desc
	  (loop for n in desc do
	    (let ((field-size (user-struct-field-size_c n)))
	      (incf (first sizes) (first field-size))
	      (incf (second sizes) (second field-size)))))
    sizes))
	      
(defun user-struct-field-type_c (n)
  (or (and (listp n) (second n)) 'real))

(defun user-struct-field-size_c (n)
  (structure-size_c (user-struct-field-type_c n)))

(defun user-struct-offset_c (fld desc)
  (let ((sizes (list 0 1)))
    (loop for n in desc do
      (if (or (eq n fld) (and (listp n) (eq (first n) fld))) 
	  (return-from user-struct-offset_c sizes)
	(let ((field-size (user-struct-field-size_c n)))
	  (incf (first sizes) (first field-size))
	  (incf (second sizes) (second field-size)))))))

(defun pass-user-struct_c (var i r datai datar info io-data)
  (let ((desc (gethash (first var) user-structs))
	(iadr (+ i 1))
	(radr r))
    (if desc
	(progn
	  (setf (aref datai i) r)
	  (loop for n in desc and k from 1 do
	    (let* ((fld-typ (user-struct-field-type_c n))
		   (sizes (user-struct-field-size_c n)))    
	      (pass-any_c (nth k var) fld-typ iadr radr datai datar info io-data)
	      (incf iadr (second sizes))
	      (incf radr (first sizes)))))
      (error "can't handle ~A of type ~A" var (first var)))))

(defun dm-user-struct_c (type i r datai datar)
  ;; assume type makes sense here!
  (let ((desc (gethash type user-structs))
	(iadr (+ i 1))
	(radr r))
    (if (/= (aref datai i) -1)
	(if desc
	    (let ((new-struct (funcall (find-symbol (concatenate 'string "MAKE-" (symbol-name type))))))
	      (loop for n in desc and k from 1 do
		(let* ((fld-typ (user-struct-field-type_c n))
		       (sizes (user-struct-field-size_c n)))    
		  (setf (nth k new-struct) (dm-any_c (list nil fld-typ radr iadr (and (listp n) (third n))) datai datar))
		  (incf iadr (second sizes))
		  (incf radr (first sizes))))
	      new-struct)
	  (warn "unknown type: ~A" type)))))

(defun allocate-user-struct_c (var i-loc r-loc datar-len datai-len array-info io-info)
  ;; return (list r-size i-size new-array-info)
  ;; allocate i_sizeof_io* if needed after io-info check
  ;; push array data on array-info
  (let* ((new-sizes (list 0 0))
	 (iadr i-loc)
	 (radr r-loc)
	 (desc (gethash (first var) user-structs))
	 (new-array-info array-info)
	 (i-end-loc datar-len)
	 (r-end-loc datai-len))
    (loop for n in desc and i from 1 do
      (let* ((fld-typ (user-struct-field-type_c n))
	     (lsizes (user-struct-field-size_c n))
	     (fvar (nth i var))
	     (res (allocate-any_c fvar new-sizes fld-typ (and (eq fld-typ 'array) (or (third n) 'real))
				  iadr radr i-end-loc r-end-loc new-array-info io-info)))
	(setf i-end-loc (second res))
	(setf r-end-loc (first res))
	(setf new-array-info (third res))
	(incf iadr (second lsizes))
	(incf radr (first lsizes))))
    (list (first new-sizes) (second new-sizes) new-array-info)))

	
(defun allocate-any_c (fvar sizes fld-typ arr-typ iadr radr i-end-loc r-end-loc array-info io-info)
  (declare (ignore radr))
  ;; if it's an array or has imbedded arrays push needed loading info on array-info and make room
  ;; if it's an io field or has imbedded io fields, check io-info and allocate/load pointers if needed
  ;; if an array of structs or tables, make base array and then loop allocating inner structs
  (let ((new-array-info array-info))
    (flet ((check-io (fil)
	     (if fil
		 (let ((cur-index (io-open-index fil)))
		   (when (zerop cur-index)
		     (setf (aref io-info cur-index) i-end-loc)
		     (incf i-end-loc i_sizeof_io*)
		     (incf (second sizes) i_sizeof_io*)))))
	   (check-array (arr offset size)
	     (when (and arr (not (find arr new-array-info :key #'first)))
	       (push (list arr (+ offset iadr) i-end-loc r-end-loc '(1 0) 'real) new-array-info)
	       (incf (first sizes) size)
	       (incf r-end-loc size))))
      (if fvar
	  (case fld-typ
	    (rdin            (check-io (rdin-fil fvar)))
	    (flt (progn      (check-array (flt-a fvar) 3 (length (flt-a fvar)))
			     (check-array (flt-b fvar) (+ 3 i_sizeof_aref_header) (length (flt-b fvar)))
			     (if (flt-c fvar)
				 (check-array (flt-c fvar) (+ 3 (* 2 i_sizeof_aref_header)) (length (flt-c fvar))))
			     (check-array (flt-d fvar) (+ 3 (* 3 i_sizeof_aref_header)) (length (flt-d fvar)))))
	    (cmbflt          (check-array (cmbflt-dly-unit fvar) 2 (dly-size (cmbflt-dly-unit fvar))))	    
	    (allpassflt      (check-array (allpassflt-dly-unit fvar) 2 (dly-size (allpassflt-dly-unit fvar))))	    
	    (tbl             (check-array (tbl-table fvar) 1 (tbl-table-size fvar)))
	    (dly             (check-array fvar 1 (dly-size fvar)))
	    (zdly            (check-array (zdly-del fvar) 2 (dly-size (zdly-del fvar))))
	    (ws              (check-array (ws-tab fvar) 1 (length (ws-tab fvar))))
	    (locs (progn     (check-io *current-output-file*)
			     (check-io *reverb*)))
	    (smp             (check-io (smp-fil fvar)))
	    (sr (progn       (check-io (rdin-fil (sr-rd fvar)))
			     (check-array (sr-data fvar) (+ 6 i_sizeof_rdin) (length (sr-data fvar)))
			     (if (sr-filt fvar) 
				 (check-array (sr-filt fvar) (+ 6 i_sizeof_rdin i_sizeof_aref_header) (length (sr-filt fvar))))))
	    (wt (progn       (check-array (wt-wave fvar) (+ 1 i_sizeof_rblk) (wt-wsiz fvar))
			     (check-array (rblk-buf (wt-b fvar)) 4 (rblk-siz (wt-b fvar)))))
	    (rblk            (check-array (rblk-buf fvar) 3 (rblk-siz fvar)))
	    (conv (progn     (check-io (rdin-fil (fftflt-rd (conv-fftf fvar))))
			     (check-array (fftflt-datar (conv-fftf fvar)) 
					  (+ 4 i_sizeof_rdin i_sizeof_rblk) 
					  (fftflt-siz (conv-fftf fvar)))
			     (check-array (fftflt-datai (conv-fftf fvar)) 
					  (+ 4 i_sizeof_rdin i_sizeof_rblk i_sizeof_aref_header) 
					  (fftflt-siz (conv-fftf fvar)))
			     (check-array (rblk-buf (fftflt-b (conv-fftf fvar)))
					  (+ 4 3 i_sizeof_rdin)
					  (rblk-siz (fftflt-b (conv-fftf fvar))))
			     (check-array (conv-filtr fvar) 
					  (+ 4 i_sizeof_rdin i_sizeof_rblk (* 3 i_sizeof_aref_header))
					  (length (conv-filtr fvar)))
			     (check-array (conv-filti fvar) 
					  (+ 4 i_sizeof_rdin i_sizeof_rblk (* 4 i_sizeof_aref_header))
					  (length (conv-filti fvar)))))
	    (fftflt (progn   (check-io (rdin-fil (fftflt-rd fvar)))
			     (check-array (fftflt-datar fvar) 
					  (+ 4 i_sizeof_rdin i_sizeof_rblk) 
					  (fftflt-siz fvar))
			     (check-array (fftflt-datai fvar) 
					  (+ 4 i_sizeof_rdin i_sizeof_rblk i_sizeof_aref_header) 
					  (fftflt-siz fvar))
			     (check-array (rblk-buf (fftflt-b fvar))
					  (+ 4 3 i_sizeof_rdin)
					  (rblk-siz (fftflt-b fvar)))
			     (check-array (fftflt-env fvar) 
					  (+ 4 i_sizeof_rdin i_sizeof_rblk (* 2 i_sizeof_aref_header) )
					  (length (fftflt-env fvar)))))
	    (spd (progn      (check-io (rdin-fil (spd-rd fvar)))
			     (check-array (spd-b fvar) (+ 10 i_sizeof_rdin) (spd-len fvar))))
	    (envelope        (check-array fvar 7 (length (envelope-data fvar))))
	    (array  
	     (when (not (find fvar new-array-info :key #'first))
   	       (if (member arr-typ '(real fraction))
		   (check-array fvar 0 (length fvar))
		 ;; iadr = aref_header location
		 (if (member arr-typ '(long-int integer))
		     (let ((len (length fvar)))
		       (push (list fvar iadr i-end-loc r-end-loc '(0 1) 'integer) new-array-info)
		       (incf i-end-loc len)
		       (incf (second sizes) len))
		   (let* ((el-sizes (structure-size_c arr-typ))
			  (i-size (second el-sizes))
			  (r-size (first el-sizes))
			  (len (array-total-size fvar))
			  (rlen (* len r-size))
			  (ilen (* len i-size))
			  (arr-i-base i-end-loc)
			  (arr-r-base r-end-loc)
			  (arr (if (= (array-rank fvar) 1) fvar
				 (make-array len 
					     :displaced-to fvar 
					     :element-type (array-element-type fvar)))))
		     (push (list fvar iadr i-end-loc r-end-loc el-sizes arr-typ) new-array-info)
		     (incf i-end-loc ilen)
		     (incf r-end-loc rlen)
		     (incf (first sizes) rlen)
		     (incf (second sizes) ilen)
		     (if (not (member arr-typ pure-float-structs))
			 (loop for k from 0 below len and 
			                  i-loc from arr-i-base by i-size and
		                          r-loc from arr-r-base by r-size do
			    (let ((res (allocate-any_c (aref arr k) sizes arr-typ nil i-loc r-loc i-end-loc r-end-loc new-array-info io-info)))
			      (setf i-end-loc (second res))
			      (setf r-end-loc (first res))
			      (setf new-array-info (third res))))))))))))
      (list r-end-loc i-end-loc new-array-info))))		  
			  

(defun pass-any_c (var type i r datai datar info io-data)
  (if (member type '(integer long-int)) (setf (aref datai i) (if (numberp var) (round var) (if var 1 0)))
    (if (member type '(real fraction)) (setf (aref datar r) (if (numberp var) (make-short-float var) (if var 1.0 0.0)))
      (if (not var)
	  (setf (aref datai i) -1)
	(case type
	  (io              (pass-io_c var i datai))
	  (tbl             (pass-tbl_c var i r datai datar info))
	  (ws              (pass-ws_c var i r datai datar info))
	  (cmbflt          (pass-cmbflt_c var i r datai datar info))
	  (allpassflt      (pass-allpassflt_c var i r datai datar info))
	  (dly             (pass-dly_c var i datai datar info))
	  (zdly            (pass-zdly_c var i datai datar info))
	  (flt             (pass-flt_c var i r datai datar info))
	  (flt-one-pole    (pass-one-pole_c var r datar))
	  (flt-one-zero    (pass-one-zero_c var r datar))
	  (flt-two-pole    (pass-two-pole_c var r datar))
	  (flt-two-zero    (pass-two-zero_c var r datar))
	  (frmnt           (pass-frmnt_c var r datar))
	  ((noi randi)     (pass-noi_c var r datar))
	  (sw              (pass-sw_c var r datar))
	  (smp             (pass-smp_c var i r datai datar io-data))
	  (osc             (pass-osc_c var r datar))
	  (rdin            (pass-rdin_c var i datai io-data))
	  (spd             (pass-spd_c var i r datai datar info io-data))
	  (sr              (pass-sr_c var i r datai datar info io-data))
	  (conv            (pass-conv_c var i r datai datar info io-data))
	  (fftflt          (pass-fftflt_c var i r datai datar info io-data))
	  (fft-data        (pass-fft-data_c var i datai datar info))
	  (cosp            (pass-cosp_c var r datar))
	  (rblk            (pass-blk_c var i r datai datar info))
	  (wt              (pass-wt_c var i r datai datar info))
	  (locs            (pass-locs_c var i r datai datar io-data))
	  (envelope        (pass-env_c var i r datai datar info))
	  (hloc            )
	  (smpflt          )
	  ((table x-table y-table) (pass-array_c var i datai datar info))
	  (array
	   (let* ((len (array-total-size var))
		  (data (find var info :key #'first))
		  (el-type (sixth data))
		  (el-sizes (fifth data))
		  (i-size (second el-sizes))
		  (r-size (first el-sizes))
		  (iadr (third data))
		  (radr (fourth data))
		  (floater (or (member el-type '(real fraction)) (member el-type pure-float-structs)))
		  (arr-base (if floater radr iadr))
		  (element-size (if floater r-size i-size)))
	     (setf (aref datai (+ i aref_size)) len)
	     (setf (aref datai (+ i aref_element_size)) element-size)
	     (setf (aref datai (+ i aref_type)) (array-type_c el-type))
	     (setf (aref datai (+ i aref_block)) arr-base)
	     (setf (aref datai (+ i aref_dims)) (array-rank var))
	     (when (/= (array-rank var) 1)
		   (let ((dim-base (seventh data))
			 (dim-list (array-dimensions var)))
		     (setf (aref datai (+ i aref_dim_list_adr)) dim-base)
		     (loop for dim on dim-list by #'cdr and i from dim-base by 1 do
			   (setf (aref datai i) (apply #'* (cdr dim))))))
	     (let ((arr (if (= (array-rank var) 1) var
			  (make-array len 
				      :displaced-to var 
				      :element-type (array-element-type var)))))
	       (if (member el-type '(real fraction))
		   (dotimes (k len) (setf (aref datar (+ radr k)) (make-short-float (aref arr k))))
		 (if (member el-type '(long-int integer))
		     (dotimes (k len) (setf (aref datai (+ iadr k)) (round (aref arr k))))
		   (loop for k from 0 below len and 
		             i-loc from iadr by i-size and
		             r-loc from radr by r-size do
		     (pass-any_c (aref arr k) el-type i-loc r-loc datai datar info io-data)))))))
	  (t (pass-user-struct_c var i r datai datar info io-data)))))))



(defun structure-size_c (typ)
  (case typ
    ((integer long-int) (list 0 1))
    ((real fraction)    (list 1 0))
    (io                 (list 0 1))
    (tbl                (list r_sizeof_tbl i_sizeof_tbl))
    (ws                 (list r_sizeof_ws i_sizeof_ws))
    (cmbflt             (list r_sizeof_cmbflt i_sizeof_cmbflt))
    (allpassflt         (list r_sizeof_allpassflt i_sizeof_allpassflt))
    (dly                (list r_sizeof_dly i_sizeof_dly))
    (zdly               (list r_sizeof_zdly i_sizeof_zdly))
    (flt                (list r_sizeof_flt i_sizeof_flt))
    (flt-one-pole       (list r_sizeof_flt-one-pole i_sizeof_flt-one-pole))
    (flt-one-zero       (list r_sizeof_flt-one-zero i_sizeof_flt-one-zero))
    (flt-two-pole       (list r_sizeof_flt-two-pole i_sizeof_flt-two-pole))
    (flt-two-zero       (list r_sizeof_flt-two-zero i_sizeof_flt-two-zero))
    (frmnt              (list r_sizeof_frmnt i_sizeof_frmnt))
    ((noi randi)        (list r_sizeof_noi i_sizeof_noi))
    (sw                 (list r_sizeof_sw i_sizeof_sw))
    (smp                (list r_sizeof_smp i_sizeof_smp))
    (osc                (list r_sizeof_osc i_sizeof_osc))
    (rdin               (list r_sizeof_rdin i_sizeof_rdin))
    (spd                (list r_sizeof_spd i_sizeof_spd))
    (sr                 (list r_sizeof_sr i_sizeof_sr))
    (conv               (list r_sizeof_conv i_sizeof_conv))
    (fftflt             (list r_sizeof_fftflt i_sizeof_fftflt))
    (fft-data           (list r_sizeof_fft-data i_sizeof_fft-data))
    (cosp               (list r_sizeof_cosp i_sizeof_cosp))
    (locs               (list r_sizeof_locs i_sizeof_locs))
    ((array table x-table y-table) (list r_sizeof_aref_header i_sizeof_aref_header))
    (rblk               (list r_sizeof_rblk i_sizeof_rblk ))
    (wt                 (list r_sizeof_wt i_sizeof_wt))
    (hloc               (list 0 0))
    (envelope           (list r_sizeof_env i_sizeof_env))
    (smpflt             (list r_sizeof_smpflt i_sizeof_smpflt))
    (t (or (user-struct-size_c typ)
	   (error "odd type: ~A" typ)))))

(defun dm-any_c (data datai datar)
  (let ((iadr (fourth data))
	(radr (third data))
	(type (second data)))
    (if (member type '(integer long-int))
	(if (/= iadr -1) (aref datai iadr))
      (if (member type '(real fraction))
	  (if (/= radr -1) (aref datar radr))
	(if (/= (aref datai iadr) -1)
	    (case type
	      (io              (dm-io_c iadr datai))
	      (tbl             (dm-tbl_c iadr datai datar))
	      (ws              (dm-ws_c iadr datai datar))
	      (cmbflt          (dm-cmbflt_c iadr datai datar))
	      (allpassflt      (dm-allpassflt_c iadr datai datar))
	      (dly             (dm-dly_c iadr datai datar))
	      (zdly            (dm-zdly_c iadr datai datar))
	      (flt             (dm-flt_c iadr datai datar))
	      (flt-one-pole    (dm-one-pole_c radr datar))
	      (flt-one-zero    (dm-one-zero_c radr datar))
	      (flt-two-pole    (dm-two-pole_c radr datar))
	      (flt-two-zero    (dm-two-zero_c radr datar))
	      (frmnt           (dm-frmnt_c radr datar))
	      ((noi randi)     (dm-noi_c radr datar))
	      (sw              (dm-sw_c radr datar))
	      (smp             (dm-smp_c iadr datai datar))
	      (osc             (dm-osc_c radr datar))
	      (rdin            (dm-rdin_c iadr datai))
	      (spd             (dm-spd_c iadr datai datar))
	      (sr              (dm-sr_c iadr datai datar))
	      (conv            (dm-conv_c iadr datai datar))
	      (fftflt          (dm-fftflt_c iadr datai datar))
	      (fft-data        (dm-fft-data_c iadr datai datar))
	      (cosp            (dm-cosp_c radr datar))
	      (rblk            (dm-blk_c iadr datai datar))
	      (wt              (dm-wt_c iadr datai datar))
	      (locs            (dm-locs_c iadr datai datar))
	      (envelope        (dm-env_c iadr datai datar))
	      (hloc            )
	      (smpflt          )
	      ((array table x-table y-table)
	       (let* ((len (aref datai (+ iadr aref_size)))
		      (el-typ (or (fifth data) 'real)) ;array of fractional arrays passed recursively = no el-type
		      (el-sizes (structure-size_c el-typ))
		      (i-size (second el-sizes))
		      (r-size (first el-sizes))
		      (floater (or (member el-typ '(real fraction)) (member el-typ pure-float-structs))))
		 (if floater (setf i-size r-size))
		 (make-array len 
			     :initial-contents
			       (loop for i from 0 below len and 
				         ii from (aref datai iadr) by i-size collect
				 (dm-any_c (list (first data) el-typ ii ii) datai datar)))))
	      (t (dm-user-struct_c type iadr radr datai datar))))))))



;;; PHRASING (56k version in sched.lisp, tied together in defins.lisp)

(defstruct phrc vars arg)
(defvar *clm-phrases-active* nil)

(defun make-phrase_c (&optional arg)
  (setf *clm-phrases-active* t)
  (make-phrc :arg arg))

(defun wait-for-phrase_c (&rest phrases) 
  (declare (ignore phrases))) ;we can assume there's no parallelism here

(defun phrase-value_c (phrase var) 
  (if phrase
      (or (and var
	       (phrc-vars phrase)
	       (cdr (find (symbol-name var) (phrc-vars phrase) :key #'first :test #'string-equal)))
	  (phrc-arg phrase))))

(defun setf-phrase-value_c (phrase var val) 
  (if var (print "bad setf phrase-value"))
  (setf (phrc-arg phrase) val))
  
(defun phrase_c (p &rest vars)
  ;; the vars have been saved by the hidden clm-check-phrase call
  (when p (setf (phrc-vars p) *clm-phrases-active*))
  p)
      
(defun end-run_c (&optional phrase)
  phrase)

(defun clm-check-phrase (datai datar info)
  (declare (ignore info))
  ;; run through current instruments var hash table looking for sixth = t and save those found in *clm-phrases-active*
  (let* ((vars (get *current-instrument-name* :ins-vars))
	 (syms (get *current-instrument-name* :phrase-symbols))
	 (vals nil)
	 (newsyms nil))
    (when (and vars syms)
      (loop for sym in syms do
	(let ((sym-name (if (symbolp sym) (symbol-name sym) (format nil "~A" (eval sym)))))
	  (push sym-name newsyms)
	  (push (dm-any_c (gethash sym-name vars) datai datar) vals))))
    (when vals
      (setf *clm-phrases-active* (pairlis newsyms vals)))))



;;;                This is the story, a sad tale but true 
;;;                Of a programmer who had far too little to do.
;;;                One day as he sat in his hut swilling stew, 
;;;                He cried "CLM takes forever, it's stuck in a slough!,
;;;                Its C code is slow, too slow by a few.
;;;                Why, with just a small effort, say one line or two,
;;;                It could outpace a no-op, you could scarcely say 'boo'"!
;;;                So he sat in his kitchen and worked like a dog.
;;;                He typed and he typed 'til his mind was a fog. 
;;;                Now 6000 lines later, what wonders we see!  
;;;                CLM is much faster, and faster still it will be!
;;;                In fact, for most cases, C beats the DSP!  
;;;                But bummed is our coder; he grumbles at night.  
;;;                That DSP code took him a year to write.  
;;;                He was paid many dollars, and spent them with glee,
;;;                But his employer might mutter, this result were he to see.

