;;; -*- Lisp -*-

;;; This file assumes it can find the files on the current directory.
;;; In MCL and all versions of ACL, you need to compile the relevant C files 
;;; before loading this file.  In KCL, you need to combine the C files into
;;; the libmus.a library, and load that into your version of KCL.
;;; See the file README.clm, or the load statements below.  In MCL, you'll
;;; need to fixup the three pathnames "clm", "clib", and "mpwlib" in the
;;; relevant block below.  In KCL, the pathname to the libmus.a library
;;; may need to be fixed.  Similarly, the libdsp.a library path in ACL 4.1.
;;; For situations where one clm image is being used by many users, it
;;; might simplify use of the "c instrument" option to set the variable
;;; cmus.h-directory to point to the main clm directory.  It defaults to
;;; a null string (see defaults.lisp).

#|
The various setup possibilities are handled via Lisp's *features* list --
we push something on that list to tell Lisp what to include in the current setup.
The CLM *features* are:

     :56-mus-gens    use the 56000 version of the unit generators, if possible.
                     This is the default on the original NeXT hardware.

     :NeXT-3.0       under NeXTStep, use 3.0 or later versions of the libraries
                     if you're running 2.0 or 2.1, don't push :Next-3.0 on *features*
                     :Next-3.0 is the default under NeXTStep.

     :QP             include code to take advantage of the Ariel QP.
                     This applies only to the original NeXT cube.
                     For historical reasons, this is the default for the original NeXTs.

     :ArielPC56D     include code to take advantage of the Ariel PC-56d.
                     This currently applies only to NeXTStep/Intel.

     :Ilink          include code to take advantage of the Ilink i56 board
                     under NeXTStep/Intel.  This switch expects to be used in
                     conjunction with the ArielPC56D feature (that is, the
                     two boards are sharing the same code in CLM except for
                     a couple small nits, so ArielPC56D is the basic feature)

     :MultiSound     Same as Ilink, except for the Turtle Beach MultiSound board.
                     (This doesn't work yet).

     :libaudio-G0    SGI switch telling which library is available.


The *features* that CLM adds itself are:

     :clm

The other *features* that CLM currently notices (leaving aside walk.lisp and loop.lisp) are:

     :NeXT           NeXTStep OS.
     :Mac            Macintosh.
     :SGI            SGI.
     :I386           386-style processor (includes 486 and Pentium)
     :kcl            Kyoto CL (akcl)
     :excl           Franz Inc's Lisp (Allegro CL)
     :mcl            Macintosh CL
     :allegro-v3.1   ACL version 3.1.20 (to distinguish it from very different later 4.n versions)
     :cltl2          Current lisp is a version 2 CL -- that is, it uses the later package syntax, etc
     :pcl            Current lisp uses walk.lisp, not a built-in code-walker

|#

#+(or excl EXCL Excl) (progn

  (when (not (eq excl:*current-case-mode* :case-insensitive-upper))
    (warn "changing case mode from ~(~A~) to case-insensitive-upper..." excl:*current-case-mode*)
    (excl:set-case-mode :case-insensitive-upper))
  )

#+NeXT (pushnew :NeXT-3.0 *features*)
;;; NeXT 1.0 is no longer supported.  NeXT 2.0 and 2.1 worked the last time I was able to check them.
;;; This switch applies to any 3.0 or later version of NeXTStep.


#+(and cltl2 (not mcl)) 
  (progn
    (if (not (probe-file "clm-packages.fasl"))
	(compile-file "clm-packages.lisp"))
    (load "clm-packages.fasl"))

;;; If you have a G0 version of libaudio.a from SGI
#+SGI (defvar *libaudio-G0-p* nil)

#+SGI 
  (when *libaudio-G0-p*
    (pushnew :libaudio-G0  *features*)
    (defvar  *G0-directory-pathname* (make-pathname :directory "~/lib/")) ; /usr/lib
    )

;;; (pushnew :Ilink *features*)
;;; uncomment if you're running NeXTStep/Intel with an Ilink i56 board

;;; (pushnew :Arielpc56d *features*)
;;; uncomment if you're running NeXTStep/Intel with an ArielPC56D board


#+(and next (or (not I386) ArielPc56d)) (pushnew :56-mus-gens *features*)

#+(and NeXT 56-mus-gens (not I386)) (pushnew :QP *features*)
;;; uncomment if you have an Ariel QP board on a NeXT Cube


#+excl (progn

  #-cltl2 (require :loop)
  (require :foreign)
  (setf excl:*global-gc-behavior* nil)	;turn off the gc tenure message

  (defun compile-and-load (name)
    (top-level:do-command "cload" (concatenate 'string name ".lisp")))

  (setq sys:*source-file-types* '("cl" "lisp" nil "ins" "cm" "clm" "cmn"))

  (setq sys:*load-search-list*		;don't want to compile note-lists
    (append sys:*load-search-list* (list (make-pathname :type "ins")
					 (make-pathname :type "cm")
					 (make-pathname :type "clm") 
					 (make-pathname :type "cmn") 
					 )))
  )

#+kcl (progn
  (defun compile-and-load (name) 
    (let ((cname (concatenate 'string name ".lisp"))
	  (lname (concatenate 'string name ".o")))
      (if (probe-file cname)
	  (if (or (not (probe-file lname))
		  (> (file-write-date cname) (file-write-date lname)))
	      (let ((kcl-c-name (concatenate 'string "ak" name ".c"))
		    (kcl-h-name (concatenate 'string "ak" name ".h"))
		    (kcl-data-name (concatenate 'string "ak" name ".data"))
		    (kcl-o-name (concatenate 'string "ak" name ".o")))
		(if (and (not (probe-file lname))
			 (probe-file kcl-o-name)
			 (> (file-write-date kcl-o-name) (file-write-date cname)))
		    (setf lname kcl-o-name)
		  (progn
		    (compile-file cname :c-file kcl-c-name :h-file kcl-h-name :o-file kcl-o-name :data-file kcl-data-name)
		    (setf lname kcl-o-name)
		    (delete-file kcl-c-name)
		    (delete-file kcl-data-name)
		    (delete-file kcl-h-name)))))
	;; these c-file etc arguments keep us from over-writing io.c and friends
	;;  all four are needed because KCL is remarkably stupid
	(let ((scname (concatenate 'string name ".ins")))
	  (if (probe-file scname)
	      (if (or (not (probe-file lname))
		      (> (file-write-date scname) (file-write-date lname)))
		  (compile-file scname)))))
      (load lname)))

  (if (not (probe-file "akloop.o")) (load "loop.lisp"))
  (compile-and-load "loop")
  ;; we have to load loop first to make kcl happy about loop-translate
  )

#+mcl (progn

	;; updated to MPW 3.3 and MPW C 3.3.1 11-Jan-94
	;; on the Mac we use the directories named below to find the clm sources, the MPW CLibraries, and the MCL interfaces
	;; -- fix these names for your system:
	;;   clm: the directory that has the clm sources
	;;   ccl: the MCL library (where the interfaces folder resides)
	;;   clib: the MPW CLibrary location
	;;   mpwlib: the MPW Library location

	(setf (logical-pathname-translations "clm") '(("**;*.*.*" "Macintosh HD:bil:clm:**:")))
	(setf (logical-pathname-translations "clib") '(("**;*.*.*" "Macintosh HD:MPW Folder:Libraries:CLibraries:**:")))
	(setf (logical-pathname-translations "mpwlib") '(("**;*.*.*" "Macintosh HD:MPW Folder:Libraries:Libraries:**:")))
	;; (setf (logical-pathname-translations "cclib") '(("**;*.*.*" "Macintosh HD:MCL 2.0:Library:**:")))

	(setq *autoload-traps* t)

	(require :loop)
	(require "FF")
	(require "interfaces")
	(require "resources")

	(defun mac-compile-file (cname)
	  (print (format nil "compiling ~A~%" cname))
	  (force-output)
	  (compile-file cname))

	(defun mac-load (lname)
	  (print (format nil "loading ~A~%" lname))
	  (force-output)
	  (load lname))

	(defun compile-and-load (name &optional (dir "clm:"))
	  (let ((cname (concatenate 'string dir name ".lisp"))
		(lname (concatenate 'string dir name ".fasl")))
	    (if (probe-file cname)
		(if (or (not (probe-file lname))
			(> (file-write-date cname) (file-write-date lname)))
		    (mac-compile-file cname))
	      (let ((scname (concatenate 'string "clm:" name ".ins")))
		(if (probe-file scname)
		    (if (or (not (probe-file lname))
			    (> (file-write-date scname) (file-write-date lname)))
			(mac-compile-file scname)))))
	    (mac-load lname)))
	)


#-(and excl (not Allegro-v3.1))
  (unless (find-package :walker)
    #-SGI (compile-and-load "walk")
    #+SGI (compile-and-load "sgi-walk")
    )

#+(and cltl2 mcl) (compile-and-load "clm-packages")

(compile-and-load "initmus")
(compile-and-load "defaults")

#+(and excl next) (load "io.o" 
			:foreign-files (list
					(merge-pathnames "headers.o" excl::*source-pathname*)
					(merge-pathnames "cmus.o" excl::*source-pathname*)
					(merge-pathnames "merge.o" excl::*source-pathname*)
					(merge-pathnames "next.o" excl::*source-pathname*)
					#+56-mus-gens (merge-pathnames "next56.o" excl::*source-pathname*)
					#+QP (merge-pathnames "qp.o" excl::*source-pathname*)
					#-Allegro-v3.1 "/usr/local/lib/libdsp.a"
					)
			#-NeXT-3.0 :system-libraries 
			#-NeXT-3.0 '("dsp_s" "sys_s" "m")
			)

#+(and excl SGI) (load "io.o" 
		       :foreign-files (list
				       (merge-pathnames "sgi.o" excl::*source-pathname*)
				       (merge-pathnames "headers.o" excl::*source-pathname*)
				       (merge-pathnames "cmus.o" excl::*source-pathname*)
				       (merge-pathnames "merge.o" excl::*source-pathname*)
				       #+libaudio-G0 (merge-pathnames  "libaudio_G0.a" *G0-directory-pathname*)
				       )
		       ;;:verbose t
		       ;;:print t
		       )

#+mcl (progn
	(ccl:ff-load '("clm:io.c.o" 
		       "clm:headers.c.o" 
		       "clm:merge.c.o" 
		       "clm:cmus.c.o" 
		       ;"clm:mac.c.o"
		       ;"clm:clm_fm_violin.c.o"
		       ;"clm:clm_jc_reverb.c.o"
		       ) 
		     ;; these should be compiled in MPW using the command C -d MAC headers.c<ENTER>
		     ;; except for cmus.c which needs C -d MAC -s CMUS cmus.c<ENTER>, and C -d MAC -s CLIO io.c<ENTER>
		     ;; 
		     ;; instruments should be placed in the INS (or whatever) segment:
		     ;; C -d MAC -s INS clm_fm_violin.c
		     ;; 
		     :ffenv-name 'mac-clm
		     :libraries '("clib:StdCLib.o"
				  "clib:CSANELib.o"
				  "clib:Math.o"
				  ;; used to require CRuntime.o but that library seems to be no longer needed in MPW 3.3
				  "mpwlib:Interface.o"
				  "mpwlib:Runtime.o" 
				  )) 

	;; the MPW command is Lib -d io.c.o headers.c.o merge.c.o cmus.c.o mac.c.o 
	;;                        "{CLibraries}StdClib.o" "{CLibraries}CSANELib.o"
	;;                        "{Libraries}Interface.o" "{Libraries}RunTime.o"<ENTER>
	;; but I can't seem to get MCL to accept:
	;;    (ccl:ff-load '("clm:Lib.Out.o") :ffenv-name 'mac-clm)
	;; even if a dummy "main" program is added, and the -model far C option is used, and the C code is split into segments.

	(compile-and-load "mac")
	(compile-and-load "macintosh-io")
	)

(compile-and-load "headers")
(compile-and-load "c-io")
(compile-and-load "dacs")
(compile-and-load "io")
#+(and kcl (or sgi next)) (compile-and-load "kcl-clm")
(compile-and-load "mus")


(compile-and-load "globals")

#+56-mus-gens
  (progn
    #+(and kcl next) (compile-and-load "kcl-c56")
    (compile-and-load "next56")		;has constants et al needed by everyone
    #+QP (compile-and-load "qp")
    #+next (compile-and-load "nxtmem")
    (compile-and-load "dsp56")
    (compile-and-load "lib56")
    (compile-and-load "code56")
    (compile-and-load "ins56")
    #-QP (compile-and-load "qpstubs")
    (compile-and-load "sched")
    #+QP (compile-and-load "qplib56")
    (compile-and-load "rt")
    )

(compile-and-load "tied")
(compile-and-load "run")

#+56-mus-gens  (compile-and-load "addons")

#-56-mus-gens (compile-and-load "c56-stubs")
#+mcl (compile-and-load "mac-56")
(compile-and-load "cmus")

(compile-and-load "sound")
(compile-and-load "defins")
#+QP (compile-and-load "qpins")
(compile-and-load "env")


(use-package "CLM")

#-Kcl
(let ((init-file (merge-pathnames (or *clm-init* "clm-init") 
				  #+Excl excl::*source-pathname*
				  #+mcl ccl:*loading-file-source-file*
				  )))
  (with-open-file (ifile init-file	;bug in KCL -- with-open-file of nonexistent file gets CLOSE error no matter what
		   :direction :input
		   :if-does-not-exist nil)
    (if (streamp ifile) 
	(load init-file)
      (progn
	(compile-and-load "v")
	(compile-and-load "jcrev")))))

#+Kcl
(let ((init-file (open (merge-pathnames (or clm:*clm-init* "clm-init") si:*load-pathname*) 
		       :direction :input 
		       :if-does-not-exist nil)))
  (if init-file
      (progn
	(load init-file)
	(close init-file))
      (progn
	(compile-and-load "v")
	(compile-and-load "jcrev"))))
  
;;; to load CMN with CLM, use:
;(load "cmn-all.lisp")

#+(or mcl kcl) (progn
	(compile-and-load "tl")
	;; optional top level interpreter
	)


