;;; -*- syntax: common-lisp; package: clm; base: 10; mode: lisp -*-

(in-package :clm)

#-cltl2 (export '(RUN-ONCE GET-SPECTRUM))

;;; various additions to the RUN macro
;;;
;;; These can serve as examples of how to customize the behaviour of RUN.
;;; The basic idea is that RUN needs to know what functions it is going to
;;; see, and what associated function to call when it is seen.  Macros
;;; are expanded when they are encountered, so only functions need special
;;; treatment.  Currently, recursion is not supported explicitly.

;;; As a first example, say we want RUN to handle ARRAY-RANK-LIMIT correctly.
;;; Since we don't have much DSP memory, we'll arbitrarily limit it to 2048.
;;; The function DEF-DSP-FUN adds a new function to RUN's table:

;;; ARRAY-RANK-LIMIT

(defun arrdim (var x) (declare (ignore var x)) 2048)
(def-dsp-fun 'array-rank-limit 'arrdim)

;;; Now, let's add support for ARRAY-TOTAL-SIZE.
;;; Here we have to take into account the fact that LIB56 uses two kinds
;;; of arrays.  First we define the 56000 code that we want called when
;;; ARRAY-TOTAL-SIZE is encountered.  LIBDECL defines the entry point(s)
;;; and tells the compiler what to call at load time, what registers
;;; need protection, and what other library functions we call explcitly:
  
(libdecl .array-size 'array-size-load '(A B X Y) '(.basic-divide))
(defun array-size-load ()
  (emit '(.array-size))			;R3->array or table, returned size in A
  (emit '(LOAD A Y R3 R+1))		;this is the size for a table
  (emit '(LOAD B Y R3 R))
  (emit '(LOAD X0 1))
  (emit '(CMP X0 B))
  (emit '(JLE all-done))		;if Y[loc+1]>1 then it's an array
  (emit '(TFR A B)    '(LOAD A X R3 R))	;actual size - 1 (not taking element size into account) (B=el siz now)
  (emit '(ADD X0 A))
  (emit '(CMP X0 B))			;maybe elements are one word long
  (emit '(JSNE .basic-divide))
  (emit '(all-done LOCAL))
  (emit '(RTS)))

;;; Now the code to tie ARRAY-TOTAL-SIZE into RUN

(def-dsp-fun 'array-total-size 'array-size)

(defun array-size (var x)
  (package-op '<array-total-size> var x))

;;; PACKAGE-OP is the simplest way to take care of arguments and so on.
;;;
;;; Now the compiler's code for setting up the jump to .array-size --
;;; It needs to be sure the A B X and Y registers are safe, get the
;;; address of the array into R3, and return the information that
;;; the integer result of ARRAY-TOTAL-SIZE is in register A.
;;; "pp" is the linearized program being built up by RUN.

(defmacro <array-total-size> (result-name arg)
  (DEBUGGING (push `(----------------> array-total-size ,result-name ,arg) pp))
  (need-loaded '.array-size)
  (let ((addr (cadr (get-home-address arg))))
    (when (not (eq addr 'R3))
      (get-register 'R3)
      (if (< addr 256)
	  (push `(LOAD R3 ,addr SHORT) pp)
	(push `(LOAD R3 ,addr) pp)))
    (if (temp-sig arg) (kill-temp arg))
    (spill-ALU-registers (libinfo-uses .array-size))
    (push `(JSR .array-size) pp)
    (update-var-and-reg result-name 'A 'integer 'L 'S))
  nil)


;;; ARRAY-DIMENSION-LIMIT

(defun arrlim (var x) (declare (ignore var x)) external-L-size)
(def-dsp-fun 'array-dimension-limit 'arrlim)


;;; ARRAY-TOTAL-SIZE-LIMIT

(def-dsp-fun 'array-total-size-limit 'arrlim)


;;; ARRAY-IN-BOUNDS-P

(def-dsp-fun 'array-in-bounds-p #'(lambda (var x) 
				    (package-rel-op var 
						    `(<= 0 ,(third x) (1- (array-total-size ,(second x))))
						    '<jump-g> '<jump-leq>)))


;;; ARRAY-RANK
;;;
;;; this is harder because I currently just pass down indices and dimensional offsets,
;;;  so the number of dimensions is always 1 from Run's point of view.  If the array
;;;  passed as the argument is not a variable within the Run loop, we can handle it
;;;  as follows:

(def-dsp-fun 'array-rank #'(lambda (var x) (push `(<array-rank> ,var ,(second x)) new-prog) (rem-var var)))

(defmacro <array-rank> (var x)
  (let ((rank-loc (get-x-memory)))
    (push `(setf-x-mem ,rank-loc (array-rank ,x)) pup)
    (add-var var 'integer `(X ,rank-loc) nil)
    nil))


;;; and similar code for ARRAY-DIMENSION (assuming constants as args throughout)

(def-dsp-fun 'array-dimension #'(lambda (var x) (push `(<array-dimension> ,var ,(second x) ,(third x)) new-prog) (rem-var var)))

(defmacro <array-dimension> (var x axis)
  (let ((rank-loc (get-x-memory)))
    (push `(setf-x-mem ,rank-loc (array-dimension ,x ,axis)) pup)
    (add-var var 'integer `(X ,rank-loc) nil)
    nil))



;;; SPECTRUM support -- this is slightly more complicated than the
;;; array support procedures above because it involves passing
;;; and receiving arrays and what not at "run-time".  This is an
;;; example of how to use all the clm stuff in an array-processing
;;; mode -- we load all the code needed to get spectra, then just
;;; leave the 56000 sitting in a loop awaiting data.


(defun put-56-array (data-reg)
  (emit `(DEFINE loc-size ,(get-L-mem)))
  (get-datum 'A)			;size of data 
  (get-datum data-reg)			;where to put it
  (emit '(STORE A Y loc-size))
  (emit `(STORE ,data-reg X loc-size))
  (emit '(LOAD X0 0))
  (emit '(DO A get-data))
  (         get-datum 'B)
  (emit   `(STORE B X ,data-reg R))
  (emit   `(STORE X0 Y ,data-reg R+1))
  (emit   '(get-data LOCAL))
  (emit '(NOP)))

(defun get-56-array (data-reg)
  (emit '(LOAD A Y loc-size))
  (emit `(LOAD ,data-reg X loc-size))
  (emit '(DO A put-data))
  (emit   `(LOAD B X ,data-reg R+1))
  (        put-datum 'B)
  (emit   '(NOP))
  (emit   '(put-data LOCAL))
  (emit '(NOP)))

(libdecl .spectrum-56 'spectrum-56-load '(A B X Y) '(.spectrum .log))
(defun spectrum-56-load ()
  (emit '(.spectrum-56))
  (put-56-array 'R3)
  (emit `(LOAD R3 X loc-size))
  (get-datum 'A)			;ask for A value (then B)
  (get-datum 'B)
  (emit `(DEFINE log-or-linear ,(get-L-mem)))
  (get-datum 'X0)
  (emit '(STORE X0 X log-or-linear))
  (emit '(JSR .spectrum))
  (emit '(LOAD A X log-or-linear))
  (emit '(TST A))
  (emit '(JEQ all-done))
  (emit '(LOAD N3 Y loc-size))
  (emit '(LOAD R3 X loc-size))
  (emit '(DO N3 log-time))
  (emit   '(CLR A)   '(LOAD B 0 SHORT))
  (emit   '(LOAD A0 X R3 R))
  (emit   '(ASR A)   '(LOAD B0 1))
  (emit   '(CMP B A))
  (emit   '(TLT B A))
  (emit   '(JSR .log))
  (emit   '(REP 16))			;return 24 bit with 8 of integer, 16 of fraction
  (emit     '(ASL A))
  (emit   '(STORE A X R3 R+1))
  (emit   '(log-time LOCAL))
  (emit '(all-done LOCAL))
  (get-56-array 'R3)
  (emit '(JMP .spectrum-56)))

(defmacro <spectrum> (result-name &rest args)
  (declare (ignore result-name args))
  (need-loaded '.spectrum-56)
  (push '(JSR .spectrum-56) pp)
  nil)

(defmacro Run-once (x)
  `(Run
    (loop for i from 0 to 0 do ,x)
    t))

(def-dsp-fun 'spectrum 'spectrum-56)

(defun spectrum-56 (var x)
  (package-op '<spectrum> var x))

(defun get-spectrum (data window log-time)
  (let* ((size (array-total-size data))
	 (d1 (make-array size :element-type 'fixnum :initial-element 0)))
    (c-load-fractional-array data size 0 d1)
    (dsp-put-one-word size)
    (dsp-put-one-word (+ 40960 2048))
    (dsp-put-array 0 (1- size) d1)
    (dsp-put-one-word size)		;number of points
    (dsp-put-one-word window)		;window type
    (dsp-put-one-word log-time)		;log or linear scale
    (loop until (/= 0 (logand (dsp-read-isr) 1)))
    (dsp-get-array 0 (1- size) d1)
    (if (zerop log-time)
	(loop for i from 0 below size do
	  (setf (aref data i) (frac24 (aref d1 i))))
      (let ((scl (/ 20.0 (* (log 10.0) (expt 2 16)))))
	(loop for i from 0 below size do
	  (let ((num (aref d1 i)))	;turn 24 bit mixed number into lisp float, then convert to dB scale
	    (if (> num (1- (expt 2 23))) (setf num (- num (expt 2 24))))
	    (setf (aref data i) (* scl num))))))))
  
;;; To set up the 56000 so that it is ready to accept arrays of data and return spectra thereof,
;;; separately compile and instrument like:
;;;
;;; (definstrument set-up-spectrum () (call (spectrum nil)))
;;; 
;;; load it, evaluate (set-up-spectrum), and now get-spectrum will do what it says.
;;;
;;; (it has to be in a separate file to force correct macro expansion, I think)
