

functor PQueue (E : DATUM) : 
    sig 
	include PQUEUE 

	(* position : find an element in the heap satisfying 
	   some predicate.  If more than one element may satisfy the
	   predicate, the element found is not necessarily the one 
	   with the higher priority.  Returns the element found and its 
	   index in the heap.  The Position exception is raised if
	   no element satisfies the predicate. *)

	exception Position   

	val position : queue -> (E.element -> bool) -> (int * E.element)

	(* repair pq i: use to incrementally reorder the prority queue
	 after any -- destructive -- action changing the priority of the 
	 element at position i.  *)

	val repair : queue -> int -> unit
    end =
struct
	structure A = Array  

	(* the first element of an ML array has index 0. *)

	structure E = E

	type heap = E.element A.array ref

	 
	datatype queue = 
	        Pq of (E.element -> E.element -> bool) 
		   *  int ref (* index of last element of the heap *)
                   *  heap
	
	exception EmptyQueue

	val chunk = 512

	fun empty_queue (prior : E.element -> E.element -> bool) =
	    Pq(prior,ref ~1, ref(A.array(chunk,E.bottom)))

	fun grow heap =
	let val new_size = A.length (!heap) + chunk
	    val new  = A.array(new_size,E.bottom)
	    fun copy 0 = A.update(new,0,A.sub(!heap,0))
	      | copy n = (A.update(new,n,A.sub(!heap,n)); copy (n-1))
	in
		copy (A.length(!heap)-1);
		heap := new
	end
   
	(* parent of an element at  a[i] is at a[i div 2] *)

	(* upheap: while not root and element is preferred to its parent,
	 switch element and parent *)

        fun upheap (Pq(prior,_,heap)) (n : int) : unit =
        let val i = ref n in
	    while !i > 0 andalso 
		(prior (A.sub(!heap,!i)) (A.sub(!heap,!i div 2)))
		do 
		let val temp = A.sub(!heap,!i)
		in
		    A.update(!heap,!i,A.sub(!heap,!i div 2));
		    A.update(!heap,!i div 2,temp);
		    i := !i div 2    (* go to parent *)
		end
	end	    

	fun enqueue (q as Pq(prior,last,heap)) e =
	    (last := !last+1;
	     if !last >= (A.length (!heap)) then grow heap else ();
	     A.update(!heap,!last,e);
	     upheap q (!last))

	fun empty (Pq (_,ref ~1,_)) = true
          | empty _ = false

	fun downheap (Pq(prior,ref last,heap)) k =
	let val i = ref k  (* !i is current position of old last element *)
	    val j = ref 0
	    val done = ref false
	in
		while not (!done) andalso (!i <= (last div 2)) do

			(* push old last element down tree *)

			(if (!i*2 = last) (* i has only one son *)
			   orelse prior (A.sub(!heap,!i*2)) 
				           (A.sub(!heap,!i*2+1)) then
					(* the first son is preferred  *)
				j := !i*2
			 else
				j := !i*2+1;
			(* j is now the preferred or only child of i *)
			 
			 if prior (A.sub(!heap,!j))
			          (A.sub(!heap,!i)) then
				(* exchange old last element, i,  with
				   prior child *)
				let val temp = A.sub(!heap,!i)
				in
					A.update(!heap,!i,A.sub(!heap,!j));
					A.update(!heap,!j,temp);
					i := !j
				end
			 else
				done := true)  (* cannot push further *)
	end

	fun dequeue (pq1 as Pq(_,last,heap)) =
		if empty pq1 then 
			raise EmptyQueue
		else
			let val result = A.sub(!heap,0)
			in
				A.update(!heap,0,A.sub(!heap,!last));
				last := !last - 1;
				downheap pq1 0;
				result
			end

        fun front (pq1 as Pq(_,last,heap)) =
	    if empty pq1 then
		raise EmptyQueue
	    else
		A.sub(!heap,0)

        (* cop-out put and format functions *)

	fun put os pq = output(os,"<A Priority Queue>")    
	fun format pq = Pretty.string "<Priority Queue>"
	
	fun replace (pq1 as Pq(prior,last,heap)) e =
		if prior e (A.sub(!heap,0)) then 
			e
		else
			let val result = A.sub(!heap,0)
			in
				A.update(!heap,0,e);
				downheap pq1 0;
				result
			end;

       exception Position

       fun position (pq1 as Pq(_,last,heap)) (f: E.element -> bool) =
	   if empty pq1 then 
	       raise Position 
	   else
	       let fun loop i =
		   if i > (!last) then
		       raise Position
		   else if f (A.sub(!heap,i)) then
		       (i,A.sub(!heap,i))
		   else
		       loop (i+1)
	       in
		   loop 0
	       end

       val repair = upheap
end;

