(* This is a consolidation of Ulli's implementation, eliminating
signatures and functors other than RMS and Atms.  The former functors
are now defined here as substructures.  I would have preferred to
leave Ulli's code organized as it was, but do not want to add his
other functors and signatures to the qwertz toolbox.  As they also use
other qwertz toolbox modules, I don't see any way to use these as a
SourceGroup library.  Finally, the RMS signature is commented out
because of a mysterious complier bug in SML of NJ version 75.  The 
circuit example confirms however that the structures generated do
indeed match the RMS signature. Tom G., 3 Feb 91 *)

functor Atms (structure
		  D : DATUM 
	      and S : SET   (* bitsets are recomended *)
	      sharing type D.element = S.E.element) (* : RMS *)  =
struct
    structure D : DATUM = D

    (* E : Environments are sets of assumptions *)

    structure E : 
	sig 
	    include SET
	    val inconsistent : set -> bool
	    val nogood : set -> unit 
            val content : set -> S.set
        end = 
	struct
	    structure E = D (* substructure required by SET signature *)

	    datatype ass = 
		ass of 
		S.set   	       (* actual set *)
		* int 		       (* size *)
		* bool ref	       (* consistent if false *)

	    type set = ass ref
	    type element = D.element

	    exception Choose = S.Choose
			
	    fun ass_set(x) = ref(ass(x, S.size(x), ref false))
	    fun content(ref(ass(x,_,_))) = x

	    fun empty(ref(ass(_,s,_))) = (s = 0)
	    fun singleton(x) = ass_set(S.singleton(x))
	    val empty_set = ass_set(S.empty_set)

	    fun member x (ref(ass(y,_,_))) = S.member x y 
	    fun eqsets (ref(ass(x,_,_))) (ref(ass(y,_,_))) = S.eqsets x y 
	    fun subset (ref(ass(x,_,_))) (ref(ass(y,_,_))) = S.subset x y
	
	    fun eq x y = (x=y);   (* reference comparison *)
	   
	    fun union (ref(ass(x,_,_))) (ref(ass(y,_,_))) = 
		ass_set(S.union x y)		   
	    fun intersection (ref(ass(x,_,_))) (ref(ass(y,_,_))) = 
		ass_set(S.intersection x y)		   
	    fun difference (ref(ass(x,_,_))) (ref(ass(y,_,_))) = 
		ass_set(S.difference x y)			       	

	    fun choose (ref(ass(x,_,_))) = 
		let val (a,b) = S.choose(x) in 
		    (a,ass_set b) 
		end
	    fun add x (ref(ass(y,_,_))) = ass_set(S.add x y)
	    fun remove x (ref(ass(y,_,_))) = ass_set(S.remove x y)
	    fun filter f (ref(ass(x,_,_))) = ass_set(S.filter f x)

	    fun reduce f x (ref(ass(y,_,_))) = S.reduce f x y

	    fun size (ref(ass(_,s,_))) = s

	    fun set [] = empty_set
	      | set (h::t) = add h (set t)

	    fun list s = 
		if empty s then []
		else
		    let val (element,rest) = choose s in 
			element::(list rest)
		    end

	    fun inconsistent(ref(ass(_,_,i))) = !i
	    fun nogood(ref(ass(_,_,i))) = (i := true) 
	    fun format(ref(ass(x,_,_))) = S.format x
	    fun put_set os (ref(ass(x,_,_))) =
		S.put_set os x; 

	end; (* E structure *)



 (* 
  Labels: these are represented by lists that are
    - ordered by size of their elements,
    - are minimal in the sense that if an element s is a member of
    the label, then all supersets of s are also members of the label, but
    only s is explicitly stored in the list  
 *)

    structure L = 
	struct
	    type element = E.set   (* i.e. an environment *)
	    type label = element list

	    val empty_label = ([] : element list)
	    fun label l  = l
	    fun singleton x = [x]

	    fun empty []  = true
	      | empty _ = false

	    fun minimize x [] = [] 
	      | minimize x (y::l) =
            	if E.subset x y then
		    minimize x l
                else
		    y::(minimize x l)

	    (* add' inserts an set to the label, making sure the 
	     the resulting set is minimal. *)

	    fun add' e [] = [e]
	      | add' e (h::t) =
		if (E.size h) > (E.size e) then
		    e::(minimize e (h::t))
		else if (E.subset h e) then
		    h::t
		     else
			 h::(add' e t)

	    (* add inserts a set into the label without minimizing the
	     result. Oddly, this is the operation which is exported in the
	     LABEL signature, rather that add'. add' would be the more
	     appropriate operation if we wanted to be consistent about viewing
	     labels as sets of sets of assumptions, but add rather than add'
	     happens to be needed in the Lms functor using this functor. *)

	    fun add x [] = [x] 
	      |  add x (y::l) =
		 if (E.size y) > (E.size x) then 
		     x::y::l
		 else 
		     y::(add x l)

	    fun member x [] = false
	      | member x (y::l) = 
		if (E.size y) > (E.size x) then  
		    (* thus y cannot be a subset of x *)
		    false
		else if E.subset y x then
		    true
		     else 
			 member x l

	    fun union [] s2 = s2 
	      | union s1 [] = s1
	      | union (h::t) s2 = 
		union t (add' h s2)
	

	    fun intersection [] s2 = []
	      | intersection (h::t) s2 = 
	    	if member h s2 then
		    add' h (intersection t s2)
	    	else
		    intersection t s2

	    fun difference [] s2 = []
	      | difference (h::t) s2 =
	    	if member h s2 then
		    difference t s2
	    	else
		    h::(difference t s2)

	    fun cross_singleton(_,[],result) = result 
	      | cross_singleton(x,y::lab,result) = 
		add' (E.union x y) (cross_singleton(x,lab,result))
		  
	    fun cross2([],_) = [] 
	      | cross2(x::l1,l2) = cross_singleton(x,l2, cross2(l1,l2))
	  
	    fun cross [] = [] 
	      | cross [lab] = lab 
	      | cross (lab::list) = cross2(lab,cross list)
	  
	    val map = map
	  
	    fun mapc fct [] = () 
	      | mapc fct (x::l) = (fct x; mapc fct l)
	  
	    fun map_supers(f,[],x,v) = v 
	      | map_supers(f,y::l,x,v) = 
		if (E.size y) < (E.size x) then 
		    map_supers(f,l,x,v) 
		else
		    if (E.subset x y) then 
			map_supers(f,l,x,f(y,v))
		    else 
			map_supers(f,l,x,v) 
	  
	    fun contains_nogoods []  = false 
	      | contains_nogoods (x::l) = 
		if E.inconsistent x then 
		    true 
		else 
		    contains_nogoods l
	  
	    fun purge' [] = [] 
	      | purge' (x::l) = 
		if E.inconsistent x then 
		    purge' l
		else 
		    x::(purge' l)
						   
	    fun purge l = if contains_nogoods l then purge' l else l

	end;  (* L structure for labels  *)

    (* Nodes *)

    structure N = 
	struct
	    datatype  node = 	
		form_node of D.element 
		* L.label ref
		*  node list ref  (* predecessors *)
		*  node list ref  (* successors *)
	      |   just_node of node list *  node 

	    structure T = AList(structure Key = D
				and Value = struct 
						type element = node
						fun put os e = () 
					        fun format _ = 
						    Pretty.string "<node>"
					    end)
						
	    val nodes = ref (T.table());

            fun reset () = nodes := T.table ()

	    exception Impossible of  unit

	    fun eq (form_node(x,_,_,_)) (form_node(y,_,_,_)) = D.eq x y
	      | eq _ _ = false   (* don't need to compare just_nodes *)
	  
	    fun formula_node x = 
		(T.get (!nodes) x) handle T.Get  =>
		    let val n = form_node(x, ref L.empty_label, 
					  ref [], ref [])
		    in
			T.put (!nodes) x n;
			n
		    end;
	  
	    val contradiction = formula_node D.bottom

	    fun link_consequent (form_node(_,_,p,_), j) = (p := j::(!p))
	      | link_consequent (_,_) = raise Impossible ();
		
	    fun link_antecedent (form_node(_,_,_,s), j) = (s := j::(!s))
	      | link_antecedent (_,_) = raise Impossible ()

	    fun link_antecedents (nl, j) = 
		map (fn n => link_antecedent (n,j)) nl;

	    fun justification (nl, n) =
		let val just = just_node (nl, n)
		in
		    link_consequent (n,just);
		    link_antecedents (nl,just); 
		    just
		end;
	  
	    fun is_justification (just_node(_,_)) = true 
	      |  is_justification _ = false
		  
	    fun is_contradiction n = eq n contradiction
		
	    fun label (form_node(_,l,_,_)) = l 
	      |  label  _ = raise Impossible ()
	  
	    fun datum (form_node(d,_,_,_)) = d 
	      |  datum  _ = raise Impossible ()
	 
	    fun pred (form_node(_,_,p,_)) = !p 
	      |  pred (just_node(p,_)) = p 
			
	    fun succ (form_node(_,_,_,s)) = !s 
	      |  succ (just_node(_,s)) = [s]
	
	    fun consequent (just_node(_,s)) = s
	      |  consequent _ = raise Impossible ()

	end;  (* N structure for nodes *)

    (* Label Management System *)

    structure Lms = 
	struct
	    structure DB : TABLE = 
		AList(structure
			  Key : EQ = 
			  struct 
			      type element = S.set
			      val put = S.put_set
			      val format = S.format
			      val eq = S.eqsets
			  end 
		      and
			  Value : ELEMENT =
			  struct
			      type element = E.set
			      val put = E.put_set
			      val format = E.format
			  end)

	    val ass_table = DB.table()
	    val ass_list = ref L.empty_label
	    val nogoods = ref L.empty_label
  
	    val datum_node  = N.formula_node
	 
	    fun formula_label n  = 
		( (N.label(n) := L.purge(!(N.label n))); 
		 !(N.label(n)) )
	  
	    fun ass_set set  =
		(DB.get ass_table (E.content set)) handle DB.Get =>
		    ((DB.put ass_table (E.content set) set);
                     ass_list := (L.add set (!ass_list));
		     if (L.member set (!nogoods)) then E.nogood set  else ();
			 set)			
			
	    fun presume() = (L.singleton(ass_set(E.empty_set)))
						  
	    fun assume(d) = (L.singleton(ass_set(E.singleton d)))
				  
	    fun extend(n,delta) =	      				  
		let val old_lab = formula_label(n) in
		    let val new_lab = (L.union (L.purge delta) old_lab) in
		        N.label n  := new_lab;
		        let val new_delta =  (L.difference new_lab old_lab) in
			    (new_delta, not (L.empty new_delta))
			end
		    end
		end
							 
	    fun select nodes = map formula_label nodes
		  
	    fun select' nodes n delta = 
		map (fn x => if (N.eq n x) then 
		     delta else formula_label x) nodes
	  	  
	    fun justification_label j = 
		(L.purge (L.map ass_set (L.cross (select (N.pred j)))))
	
	    fun justification_delta_label (j,n,delta) = 
		(L.purge (L.map ass_set 
			  (L.cross (select' (N.pred j) n delta))))
	  
	    fun add_nogoods delta  =
		(nogoods := (L.union delta (!nogoods));
		 L.mapc (fn nogood => L.map_supers 
			 ((fn (y,z) => E.nogood y), !ass_list, nogood, ()))
		 delta)
	end; (* Lms structure *)

    type datum = D.element
    type environment = E.set

    fun adjoin_item item [] = [item]
      | adjoin_item (n1,delta1) (q as ((n2,delta2)::queue)) = 
	if (L.empty delta1) then
	    q
	else	
	    if (N.eq n1 n2) then 
		(n1, (L.union delta1 delta2))::queue 
	    else 
		((n2,delta2)::(adjoin_item (n1,delta1) queue))

    fun adjoin_items([], queue) = queue
      |  adjoin_items((item::items), queue) = 
	 adjoin_items(items, adjoin_item item queue)
	
    val node = Lms.datum_node
	  
    fun update_formula (fnode,delta) = 
	if (N.is_contradiction fnode) then 
	    (Lms.add_nogoods delta; []) 
	else
	    let val (new_delta, changed) = Lms.extend(fnode,delta) in
		if changed then
		    map (fn just => 
			 (N.consequent just,
			  Lms.justification_delta_label(just,fnode,
							new_delta)))
		    (N.succ fnode)
		else []
	    end

    fun update [] = ()
      | update (item::queue) = 
	update(adjoin_items(update_formula item, queue))

    fun start_update(fnode, delta) = 
	if (L.empty delta) then () else update [(fnode, delta)] 

    val env = E.set
			
    fun assume (e : datum) =
	let val fnode = node e
	in
	    start_update(fnode, Lms.assume(N.datum fnode))
	end

    fun presume (e : datum) =
	start_update(node e, (Lms.presume ()))

    fun justify (e : environment) (concl : datum) =
	if E.empty e then 
	    presume concl
	else
	    let val anodes = E.reduce (fn x => fn l => (node x)::l) [] e
		val cnode = node concl
	    in
		start_update(cnode, 
			     Lms.justification_label(N.justification(anodes,
								     cnode)))
	    end

    val reset = N.reset
	
    fun inconsistent (env : environment) : bool =
	L.member env (!Lms.nogoods)

    fun derivable (env : environment) (e : datum) : bool =
	(L.member env (Lms.formula_label (node e)))

    structure S = ListStream ()

    fun nogoods () = S.list2stream (!Lms.nogoods)
    fun supports (d : datum) = S.list2stream (Lms.formula_label (node d))
	
end;  (* Atms functor *)


